# The LearningOnline Network with CAPA
# User Roles Screen
#
# $Id: lonroles.pm,v 1.256.2.5 2010/12/30 21:18:18 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
###

=pod

=head1 NAME

Apache::lonroles - User Roles Screen

=head1 SYNOPSIS

Invoked by /etc/httpd/conf/srm.conf:

 <Location /adm/roles>
 PerlAccessHandler       Apache::lonacc
 SetHandler perl-script
 PerlHandler Apache::lonroles
 ErrorDocument     403 /adm/login
 ErrorDocument	  500 /adm/errorhandler
 </Location>

=head1 OVERVIEW

=head2 Choosing Roles

C<lonroles> is a handler that allows a user to switch roles in
mid-session. LON-CAPA attempts to work with "No Role Specified", the
default role that a user has before selecting a role, as widely as
possible, but certain handlers for example need specification which
course they should act on, etc. Both in this scenario, and when the
handler determines via C<lonnet>'s C<&allowed> function that a certain
action is not allowed, C<lonroles> is used as error handler. This
allows the user to select another role which may have permission to do
what they were trying to do. C<lonroles> can also be accessed via the
B<CRS> button in the Remote Control. 

=begin latex

\begin{figure}
\begin{center}
\includegraphics[width=0.45\paperwidth,keepaspectratio]{Sample_Roles_Screen}
  \caption{\label{Sample_Roles_Screen}Sample Roles Screen} 
\end{center}
\end{figure}

=end latex

=head2 Role Initialization

The privileges for a user are established at login time and stored in the session environment. As a consequence, a new role does not become active till the next login. Handlers are able to query for privileges using C<lonnet>'s C<&allowed> function. When a user first logs in, their role is the "common" role, which means that they have the sum of all of their privileges. During a session it might become necessary to choose a particular role, which as a consequence also limits the user to only the privileges in that particular role.

=head1 INTRODUCTION

This module enables a user to select what role he wishes to
operate under (instructor, student, teaching assistant, course
coordinator, etc).  These roles are pre-established by the actions
of upper-level users.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 HANDLER SUBROUTINE

This routine is called by Apache and mod_perl.

=over 4

=item *

Roles Initialization (yes/no)

=item *

Get Error Message from Environment

=item *

Who is this?

=item *

Generate Page Output

=item *

Choice or no choice

=item *

Table

=item *

Privileges

=back

=cut


package Apache::lonroles;

use strict;
use Apache::lonnet;
use Apache::lonuserstate();
use Apache::Constants qw(:common);
use Apache::File();
use Apache::lonmenu;
use Apache::loncommon;
use Apache::lonhtmlcommon;
use Apache::lonannounce;
use Apache::lonlocal;
use Apache::lonpageflip();
use Apache::lonnavdisplay();
use Apache::loncoursequeueadmin;
use GDBM_File;
use LONCAPA qw(:DEFAULT :match);
use HTML::Entities;


sub redirect_user {
    my ($r,$title,$url,$msg,$launch_nav) = @_;
    $msg = $title if (! defined($msg));
    &Apache::loncommon::content_type($r,'text/html');
    &Apache::loncommon::no_cache($r);
    $r->send_http_header;
    my $swinfo=&Apache::lonmenu::rawconfig();
    my $navwindow;
    if ($launch_nav eq 'on') {
        $navwindow.=&Apache::lonnavdisplay::launch_win('now',undef,undef,
                                                       ($url =~ m-^/adm/whatsnew-));
    } else {
        $navwindow.=&Apache::lonnavmaps::close();
    }

    # Breadcrumbs
    my $brcrum = [{'href' => $url,
                   'text' => 'Switching Role'},];
    my $start_page = &Apache::loncommon::start_page('Switching Role',undef,
                                                    {'redirect' => [1,$url],
                                                     'bread_crumbs' => $brcrum,});
    my $end_page   = &Apache::loncommon::end_page();

# Note to style police: 
# This must only replace the spaces, nothing else, or it bombs elsewhere.
    $url=~s/ /\%20/g;
    $r->print(<<ENDREDIR);
$start_page
<script type="text/javascript">
// <![CDATA[
$swinfo
// ]]>
</script>
$navwindow
<p>$msg</p>
$end_page
ENDREDIR
    return;
}

sub error_page {
    my ($r,$error,$dest)=@_;
    &Apache::loncommon::content_type($r,'text/html');
    &Apache::loncommon::no_cache($r);
    $r->send_http_header;
    return OK if $r->header_only;
    # Breadcrumbs
    my $brcrum = [{'href' => $dest,
                   'text' => 'Problems during Course Initialization'},];
    $r->print(&Apache::loncommon::start_page('Problems during Course Initialization',
                                             undef,
                                             {'bread_crumbs' => $brcrum,})
    );
    $r->print(
        '<script type="text/javascript">'.
        '// <![CDATA['.
        &Apache::lonmenu::rawconfig().
        '// ]]>'.
        '</script>'.
	      '<p class="LC_error">'.&mt('The following problems occurred:').
          '<br />'.
	      $error.
	      '</p><br /><a href="'.$dest.'">'.&mt('Continue').'</a>'
    );
    $r->print(&Apache::loncommon::end_page());
}

sub handler {

    my $r = shift;

    my $now=time;
    my $then=$env{'user.login.time'};
    my $refresh=$env{'user.refresh.time'};
    if (!$refresh) {
        $refresh = $then;
    }
    my $envkey;
    my %dcroles = ();
    my $numdc = &check_fordc(\%dcroles,$then);
    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'});
    my $loncaparev = $Apache::lonnet::perlvar{'lonVersion'};

# ================================================================== Roles Init
    if ($env{'form.selectrole'}) {

        my $locknum=&Apache::lonnet::get_locks();
        if ($locknum) { return 409; }

        if ($env{'form.newrole'}) {
            $env{'form.'.$env{'form.newrole'}}=1;
	}
	if ($env{'request.course.id'}) {
            # Check if user is CC trying to select a course role
            if ($env{'form.switchrole'}) {
                my $switch_is_active;
                if (defined($env{'user.role.'.$env{'form.switchrole'}})) {
                    my ($start,$end) = split(/\./,$env{'user.role.'.$env{'form.switchrole'}});
                    if (!$end || $end > $now) {
                        if (!$start || $start < $refresh) {
                            $switch_is_active = 1;
                        }
                    }
                }
                unless ($switch_is_active) {
                    &adhoc_course_role($refresh,$then);
                }
            }
	    my %temp=('logout_'.$env{'request.course.id'} => time);
	    &Apache::lonnet::put('email_status',\%temp);
	    &Apache::lonnet::delenv('user.state.'.$env{'request.course.id'});
	}
	&Apache::lonnet::appenv({"request.course.id"   => '',
			 	 "request.course.fn"   => '',
				 "request.course.uri"  => '',
				 "request.course.sec"  => '',
				 "request.role"        => 'cm',
                                 "request.role.adv"    => $env{'user.adv'},
				 "request.role.domain" => $env{'user.domain'}});
# Check if user is a DC trying to enter a course or author space and needs privs to be created
        if ($numdc > 0) {
            foreach my $envkey (keys %env) {
# Is this an ad-hoc Coordinator role?
                if (my ($ccrole,$domain,$coursenum) =
		    ($envkey =~ m-^form\.(cc|co)\./($match_domain)/($match_courseid)$-)) {
                    if ($dcroles{$domain}) {
                        &Apache::lonnet::check_adhoc_privs($domain,$coursenum,
                                                           $then,$refresh,$now,$ccrole);
                    }
                    last;
                }
# Is this an ad-hoc CA-role?
                if (my ($domain,$user) =
		    ($envkey =~ m-^form\.ca\./($match_domain)/($match_username)$-)) {
                    if (($domain eq $env{'user.domain'}) && ($user eq $env{'user.name'})) {
                        delete($env{$envkey});
                        $env{'form.au./'.$domain.'/'} = 1;
                        my ($server_status,$home) = &check_author_homeserver($user,$domain);
                        if ($server_status eq 'switchserver') {
                            my $trolecode = 'au./'.$domain.'/';
                            my $switchserver = '/adm/switchserver?otherserver='.$home.'&amp;role='.$trolecode;
                            $r->internal_redirect($switchserver);
                        }
                        last;
                    }
                    if (my ($castart,$caend) = ($env{'user.role.ca./'.$domain.'/'.$user} =~ /^(\d*)\.(\d*)$/)) {
                        if (((($castart) && ($castart < $now)) || !$castart) && 
                            ((!$caend) || (($caend) && ($caend > $now)))) {
                            my ($server_status,$home) = &check_author_homeserver($user,$domain);
                            if ($server_status eq 'switchserver') {
                                my $trolecode = 'ca./'.$domain.'/'.$user;
                                my $switchserver = '/adm/switchserver?otherserver='.$home.'&amp;role='.$trolecode;
                                $r->internal_redirect($switchserver);
                            }
                            last;
                        }
                    }
                    # Check if author blocked ca-access
                    my %blocked=&Apache::lonnet::get('environment',['domcoord.author'],$domain,$user);
                    if ($blocked{'domcoord.author'} eq 'blocked') {
                        delete($env{$envkey});
                        $env{'user.error.msg'}=':::1:User '.$user.' in domain '.$domain.' blocked domain coordinator access';
                        last;
                    }
                    if ($dcroles{$domain}) {
                        my ($server_status,$home) = &check_author_homeserver($user,$domain);
                        if (($server_status eq 'ok') || ($server_status eq 'switchserver')) {
                            &Apache::lonnet::check_adhoc_privs($domain,$user,$then,
                                                               $refresh,$now,'ca');
                            if ($server_status eq 'switchserver') {
                                my $trolecode = 'ca./'.$domain.'/'.$user; 
                                my $switchserver = '/adm/switchserver?'
                                                  .'otherserver='.$home.'&amp;role='.$trolecode;
                                $r->internal_redirect($switchserver);
                            }
                        } else {
                            delete($env{$envkey});
                        }
                    } else {
                        delete($env{$envkey});
                    }
                    last;
                }
            }
        }

        foreach $envkey (keys %env) {
            next if ($envkey!~/^user\.role\./);
            my ($where,$trolecode,$role,$tstatus,$tend,$tstart);
            &Apache::lonnet::role_status($envkey,$then,$refresh,$now,\$role,\$where,
                                         \$trolecode,\$tstatus,\$tstart,\$tend);
            if ($env{'form.'.$trolecode}) {
		if ($tstatus eq 'is') {
		    $where=~s/^\///;
		    my ($cdom,$cnum,$csec)=split(/\//,$where);
                    if (($cnum) && ($role ne 'ca') && ($role ne 'aa')) {
                        my $home = $env{'course.'.$cdom.'_'.$cnum.'.home'};
                        my @ids = &Apache::lonnet::current_machine_ids();
                        unless ($loncaparev eq '' && $home && grep(/^\Q$home\E$/,@ids)) {
                            my %curr_reqd_hash = &Apache::lonnet::userenvironment($cdom,$cnum,'internal.releaserequired');
                            if ($curr_reqd_hash{'internal.releaserequired'} ne '') {
                                my ($switchserver,$switchwarning) =
                                    &check_release_required($loncaparev,$cdom.'_'.$cnum,$trolecode,$curr_reqd_hash{'internal.releaserequired'});
                                if ($switchwarning ne '' || $switchserver ne '') {
                                    &Apache::loncommon::content_type($r,'text/html');
                                    &Apache::loncommon::no_cache($r);
                                    $r->send_http_header;
                                    my $end_page=&Apache::loncommon::end_page();
                                    $r->print(&Apache::loncommon::start_page('Selected course unavailable on this server').
                                              '<p class="LC_warning">');
                                    if ($switchwarning) {
                                        $r->print($switchwarning.'<br /><a href="/adm/roles">');
                                        if (&Apache::loncommon::show_course()) {
                                            $r->print(&mt('Display courses'));
                                        } else {
                                            $r->print(&mt('Display roles'));
                                        }
                                        $r->print('</a>');
                                    } elsif ($switchserver) {
				        $r->print(&mt('This course requires a newer version of LON-CAPA than is installed on this server.').
                                                  '<br />'.
                                                  '<a href="/adm/switchserver?'.$switchserver.'">'.
                                                  &mt('Switch Server').
                                                  '</a>');
                                    }
                                    $r->print('</p>'.&Apache::loncommon::end_page());
                                    return OK;
                                }
                            }
                        }
                    }
# check for course groups
                    my %coursegroups = &Apache::lonnet::get_active_groups(
                          $env{'user.domain'},$env{'user.name'},$cdom, $cnum);
                    my $cgrps = join(':',keys(%coursegroups));

# store role if recent_role list being kept
                    if ($env{'environment.recentroles'}) {
                        my %frozen_roles =
                           &Apache::lonhtmlcommon::get_recent_frozen('roles',$env{'environment.recentrolesn'});
			&Apache::lonhtmlcommon::store_recent('roles',
							     $trolecode,' ',$frozen_roles{$trolecode});
                    }


# check for keyed access
		    if (($role eq 'st') && 
                       ($env{'course.'.$cdom.'_'.$cnum.'.keyaccess'} eq 'yes')) {
# who is key authority?
			my $authdom=$cdom;
			my $authnum=$cnum;
			if ($env{'course.'.$cdom.'_'.$cnum.'.keyauth'}) {
			    ($authnum,$authdom)=
				split(/:/,$env{'course.'.$cdom.'_'.$cnum.'.keyauth'});
			}
# check with key authority
			unless (&Apache::lonnet::validate_access_key(
				     $env{'environment.key.'.$cdom.'_'.$cnum},
					     $authdom,$authnum)) {
# there is no valid key
			     if ($env{'form.newkey'}) {
# student attempts to register a new key
				 &Apache::loncommon::content_type($r,'text/html');
				 &Apache::loncommon::no_cache($r);
				 $r->send_http_header;
				 my $swinfo=&Apache::lonmenu::rawconfig();
				 my $start_page=&Apache::loncommon::start_page
				    ('Verifying Access Key to Unlock this Course');
				 my $end_page=&Apache::loncommon::end_page();
				 my $buttontext=&mt('Enter Course');
				 my $message=&mt('Successfully registered key');
				 my $assignresult=
				     &Apache::lonnet::assign_access_key(
						     $env{'form.newkey'},
						     $authdom,$authnum,
						     $cdom,$cnum,
                                                     $env{'user.domain'},
						     $env{'user.name'},
                                                     &mt('Assigned from [_1] at [_2] for [_3]'
                                                        ,$ENV{'REMOTE_ADDR'}
                                                        ,&Apache::lonlocal::locallocaltime()
                                                        ,$trolecode)
                                                     );
				 unless ($assignresult eq 'ok') {
				     $assignresult=~s/^error\:\s*//;
				     $message=&mt($assignresult).
				     '<br /><a href="/adm/logout">'.
				     &mt('Logout').'</a>';
				     $buttontext=&mt('Re-Enter Key');
				 }
				 $r->print(<<ENDENTEREDKEY);
$start_page
<script type="text/javascript">
// <![CDATA[
$swinfo
// ]]>
</script>
<form action="" method="post">
<input type="hidden" name="selectrole" value="1" />
<input type="hidden" name="$trolecode" value="1" />
<span class="LC_fontsize_large">$message</span><br />
<input type="submit" value="$buttontext" />
</form>
$end_page
ENDENTEREDKEY
                                 return OK;
			     } else {
# print form to enter a new key
				 &Apache::loncommon::content_type($r,'text/html');
				 &Apache::loncommon::no_cache($r);
				 $r->send_http_header;
				 my $swinfo=&Apache::lonmenu::rawconfig();
				 my $start_page=&Apache::loncommon::start_page
				    ('Enter Access Key to Unlock this Course');
				 my $end_page=&Apache::loncommon::end_page();
				 $r->print(<<ENDENTERKEY);
$start_page
<script type="text/javascript">
// <![CDATA[
$swinfo
// ]]>
</script>
<form action="" method="post">
<input type="hidden" name="selectrole" value="1" />
<input type="hidden" name="$trolecode" value="1" />
<input type="text" size="20" name="newkey" value="$env{'form.newkey'}" />
<input type="submit" value="Enter key" />
</form>
$end_page
ENDENTERKEY
				 return OK;
			     }
			 }
		     }
		    &Apache::lonnet::log($env{'user.domain'},
					 $env{'user.name'},
					 $env{'user.home'},
					 "Role ".$trolecode);
		    
		    &Apache::lonnet::appenv(
					   {'request.role'        => $trolecode,
					    'request.role.domain' => $cdom,
					    'request.course.sec'  => $csec,
                                            'request.course.groups' => $cgrps});
                    my $tadv=0;

		    if (($cnum) && ($role ne 'ca') && ($role ne 'aa')) {
                        my $msg;
			my ($furl,$ferr)=
			    &Apache::lonuserstate::readmap($cdom.'/'.$cnum);
			if (($env{'form.orgurl'}) && 
			    ($env{'form.orgurl'}!~/^\/adm\/flip/)) {
			    my $dest=$env{'form.orgurl'};
                            if ($env{'form.symb'}) {
                                if ($dest =~ /\?/) {
                                    $dest .= '&';
                                } else {
                                    $dest .= '?'
                                }
                                $dest .= 'symb='.$env{'form.symb'};
                            }
			    if (&Apache::lonnet::allowed('adv') eq 'F') { $tadv=1; }
			    &Apache::lonnet::appenv({'request.role.adv'=>$tadv});
                            if (($ferr) && ($tadv)) {
				&error_page($r,$ferr,$dest);
			    } else {
                                if ($dest =~ m{^/adm/coursedocs\?folderpath}) {
                                    if ($env{'request.course.id'} eq $cdom.'_'.$cnum) { 
                                        my $chome = &Apache::lonnet::homeserver($cnum,$cdom);
                                        &update_content_constraints($cdom,$cnum,$chome,$cdom.'_'.$cnum);
                                    }
                                }
				$r->internal_redirect($dest);
			    }
			    return OK;
			} else {
			    if (!$env{'request.course.id'}) {
				&Apache::lonnet::appenv(
				      {"request.course.id"  => $cdom.'_'.$cnum});
				$furl='/adm/roles?tryagain=1';
                $msg='<p><span class="LC_error">'
                    .&mt('Could not initialize [_1] at this time.',
                         $env{'course.'.$cdom.'_'.$cnum.'.description'})
                    .'</span></p>'
                    .'<p>'.&mt('Please try again.').'</p>'
                    .'<p>'.$ferr.'</p>';
			    }
			    if (&Apache::lonnet::allowed('adv') eq 'F') { $tadv=1; }
			    &Apache::lonnet::appenv({'request.role.adv'=>$tadv});

			    if (($ferr) && ($tadv)) {
				&error_page($r,$ferr,$furl);
			    } else {
				# Check to see if the user is a CC entering a course 
				# for the first time
				my (undef, undef, $role, $courseid) = split(/\./, $envkey);
				if (substr($courseid, 0, 1) eq '/') {
				    $courseid = substr($courseid, 1);
				}
				$courseid =~ s/\//_/;
				if ((($role eq 'cc') || ($role eq 'co')) 
                                    && ($env{'course.' . $courseid .'.course.helper.not.run'})) { 
				    $furl = "/adm/helper/course.initialization.helper";
				    # Send the user to the course they selected
				} elsif ($env{'request.course.id'}) {
                                    if ($env{'form.destinationurl'}) {
                                        my $dest = $env{'form.destinationurl'};
                                        if ($env{'form.destsymb'} ne '') {
                                            my $esc_symb = &HTML::Entities::encode($env{'form.destsymb'},'"<>&');
                                            $dest .= '?symb='.$esc_symb;
                                        }
                                        &redirect_user($r, &mt('Entering [_1]',
                                                       $env{'course.'.$courseid.'.description'}),
                                                       $dest, $msg,
                                                       $env{'environment.remotenavmap'});
                                        return OK;
                                    }
				    if (&Apache::lonnet::allowed('whn',
								 $env{'request.course.id'})
					|| &Apache::lonnet::allowed('whn',
								    $env{'request.course.id'}.'/'
								    .$env{'request.course.sec'})
					) {
					my $startpage = &courseloadpage($courseid);
					unless ($startpage eq 'firstres') {         
					    $msg = &mt('Entering [_1] ...',
						       $env{'course.'.$courseid.'.description'});
                                            &redirect_user($r,&mt('New in course'),
                                                           '/adm/whatsnew?refpage=start',$msg,
                                                           $env{'environment.remotenavmap'});
					    return OK;
					}
				    }
				}
# Are we allowed to look at the first resource?
				if (($furl !~ m|^/adm/|) || 
                                    (($env{'environment.remotenavmap'} eq 'on') && 
                                     ($furl =~ m{^/adm/navmaps}))) {
# Guess not ...
				    $furl=&Apache::lonpageflip::first_accessible_resource();
				}
                                $msg = &mt('Entering [_1] ...',
					   $env{'course.'.$courseid.'.description'});
                                &redirect_user($r,&mt('Entering [_1]',
                                                      $env{'course.'.$courseid.'.description'}),
                                               $furl,$msg,
                                               $env{'environment.remotenavmap'});
			    }
			    return OK;
			}
		    }
                    #
                    # Send the user to the construction space they selected
                    if ($role =~ /^(au|ca|aa)$/) {
                        my $redirect_url = '/priv/';
                        if ($role eq 'au') {
                            $redirect_url.=$env{'user.name'};
                        } else {
                            $where =~ /\/(.*)$/;
                            $redirect_url .= $1;
                        }
                        $redirect_url .= '/';
                        &redirect_user($r,&mt('Entering Construction Space'),
                                       $redirect_url);
                        return OK;
                    }
                    if ($role eq 'dc') {
                        my $redirect_url = '/adm/menu/';
                        &redirect_user($r,&mt('Loading Domain Coordinator Menu'),
                                       $redirect_url);
                        return OK;
                    }
                    if ($role eq 'sc') {
                        my $redirect_url = '/adm/grades?command=scantronupload';
                        &redirect_user($r,&mt('Loading Data Upload Page'),
                                       $redirect_url);
                        return OK;
                    }
		}
            }
        }
    }


# =============================================================== No Roles Init

    &Apache::loncommon::content_type($r,'text/html');
    &Apache::loncommon::no_cache($r);
    $r->send_http_header;
    return OK if $r->header_only;

    my $crumbtext = 'User Roles';
    my $pagetitle = 'My Roles';
    my $recent = &mt('Recent Roles');
    my $show_course=&Apache::loncommon::show_course();
    if ($show_course) {
        $crumbtext = 'Courses';
        $pagetitle = 'My Courses';
        $recent = &mt('Recent Courses');
    }
    my $brcrum =[{href=>"/adm/roles",text=>$crumbtext}];
    my $swinfo=&Apache::lonmenu::rawconfig();
    my $start_page=&Apache::loncommon::start_page($pagetitle,undef,{bread_crumbs=>$brcrum});
    my $standby=&mt('Role selected. Please stand by.');
    $standby=~s/\n/\\n/g;
    my $noscript='<span class="LC_error">'.&mt('Use of LON-CAPA requires Javascript to be enabled in your web browser.').'<br />'.&mt('As this is not the case, most functionality in the system will be unavailable.').'</span><br />';

    $r->print(<<ENDHEADER);
$start_page
<br />
<noscript>
$noscript
</noscript>
<script type="text/javascript">
// <![CDATA[
$swinfo
window.focus();

active=true;

function enterrole (thisform,rolecode,buttonname) {
    if (active) {
	active=false;
        document.title='$standby';
        window.status='$standby';
	thisform.newrole.value=rolecode;
	thisform.submit();
    } else {
       alert('$standby');
    }   
}
// ]]>
</script>
ENDHEADER

# ------------------------------------------ Get Error Message from Environment

    my ($fn,$priv,$nochoose,$error,$msg)=split(/:/,$env{'user.error.msg'});
    if ($env{'user.error.msg'}) {
	$r->log_reason(
   "$msg for $env{'user.name'} domain $env{'user.domain'} access $priv",$fn);
    }

# ------------------------------------------------- Can this user re-init, etc?

    my $advanced=$env{'user.adv'};
    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},['tryagain']);
    my $tryagain=$env{'form.tryagain'};
    my $reinit=$env{'user.reinit'};
    delete $env{'user.reinit'};

# -------------------------------------------------------- Generate Page Output
# --------------------------------------------------------------- Error Header?
    if ($error) {
        $r->print("<h1>".&mt('LON-CAPA Access Control')."</h1>");
	$r->print("<!-- LONCAPAACCESSCONTROLERRORSCREEN --><hr /><pre>");
	if ($priv ne '') {
            $r->print(&mt('Access  : ').&Apache::lonnet::plaintext($priv)."\n");
	}
	if ($fn ne '') {
            $r->print(&mt('Resource: ').&Apache::lonenc::check_encrypt($fn)."\n");
	}
	if ($msg ne '') {
            $r->print(&mt('Action  : ').$msg."\n");
	}
	$r->print("</pre><hr />");
	my $url=$fn;
	my $last;
	if (tie(my %hash,'GDBM_File',$env{'request.course.fn'}.'_symb.db',
		&GDBM_READER(),0640)) {
	    $last=$hash{'last_known'};
	    untie(%hash);
	}
	if ($last) { $fn.='?symb='.&escape($last); }

	&Apache::londocs::changewarning($r,undef,'You have modified your course recently, [_1] may fix this access problem.',
					&Apache::lonenc::check_encrypt($fn));
    } else {
        if ($env{'user.error.msg'}) {
            if ($reinit) {
                $r->print(
 '<h3><span class="LC_error">'.
 &mt('As your session file for the course or community has expired, you will need to re-select it.').'</span></h3>');
            } else {
	        $r->print(
 '<h3><span class="LC_error">'.
 &mt('You need to choose another user role or enter a specific course or community for this function.').
 '</span></h3>');
	    }
        }
    }
# -------------------------------------------------------- Choice or no choice?
    if ($nochoose) {
	$r->print("<h2>".&mt('Sorry ...')."</h2>\n<span class='LC_error'>".
		  &mt('This action is currently not authorized.').'</span>'.
		  &Apache::loncommon::end_page());
	return OK;
    } else {
        if (($ENV{'REDIRECT_QUERY_STRING'}) && ($fn)) {
    	    $fn.='?'.$ENV{'REDIRECT_QUERY_STRING'};
        }
        $r->print('<form method="post" name="rolechoice" action="'.(($fn)?$fn:$r->uri).'">');
        $r->print('<input type="hidden" name="orgurl" value="'.$fn.'" />');
        $r->print('<input type="hidden" name="selectrole" value="1" />');
        $r->print('<input type="hidden" name="newrole" value="" />');
    }
    $r->rflush();

    my (%roletext,%sortrole,%roleclass,%futureroles,%timezones);
    my ($countactive,$countfuture,$inrole,$possiblerole) = 
        &gather_roles($then,$refresh,$now,$reinit,$nochoose,\%roletext,\%sortrole,\%roleclass,
                      \%futureroles,\%timezones,$loncaparev);

    $refresh = $now;
    &Apache::lonnet::appenv({'user.refresh.time'  => $refresh});
    if ($env{'user.adv'}) {
        $r->print('<p><label><input type="checkbox" name="showall"');
        if ($env{'form.showall'}) { $r->print(' checked="checked" '); }
        $r->print(' />'.&mt('Show all roles').'</label>'
                 .' <input type="submit" value="'.&mt('Update display').'" />'
                 .'</p>');
    } else {
        if ($countactive > 0) {
            $r->print(&Apache::loncoursequeueadmin::queued_selfenrollment());
            my $domdesc = &Apache::lonnet::domain($env{'user.domain'},'description');
            my $esc_dom = &HTML::Entities::encode($env{'user.domain'},'"<>&'); 
            $r->print(
                '<p>'
               .&mt('[_1]Visit the [_2]Course/Community Catalog[_3]'
                   .' to view all [_4] LON-CAPA courses and communities.'
                   ,'<b>'
                   ,'<a href="/adm/coursecatalog?showdom='.$esc_dom.'">'
                   ,'</a></b>',$domdesc)
               .'<br />'
               .&mt('If a course or community is [_1]not[_2] in your list of current courses and communities below,'
                   .' you may be able to enroll if self-enrollment is permitted.'
                   ,'<b>','</b>')
               .'</p>'
            );
        }
    }

# No active roles
    if ($countactive==0) {
	if ($inrole) {
	    $r->print('<h2>'.&mt('Currently no additional roles, courses or communities').'</h2>');
	} else {
	    $r->print('<h2>'.&mt('Currently no active roles, courses or communities').'</h2>');
	}
        &findcourse_advice($r);
        &requestcourse_advice($r); 
	$r->print('</form>');
        if ($countfuture) {
            $r->print(&mt('The following [quant,_1,role,roles] will become active in the future:',$countfuture));
            my $doheaders = &roletable_headers($r,\%roleclass,\%sortrole,
                                               $nochoose);
            &print_rolerows($r,$doheaders,\%roleclass,\%sortrole,\%dcroles,
                            \%roletext);
            my $tremark='';
            my $tbg;
            if ($env{'request.role'} eq 'cm') {
                $tbg="LC_roles_selected";
                $tremark=&mt('Currently selected.').' ';
            } else {
                $tbg="LC_roles_is";
            }
            $r->print(&Apache::loncommon::start_data_table_row()
                     .'<td class="'.$tbg.'">&nbsp;</td>'
                     .'<td colspan="3">'
                     .&mt('No role specified')
                     .'</td>'
                     .'<td>'.$tremark.'&nbsp;</td>'
                     .&Apache::loncommon::end_data_table_row()
            );

            $r->print(&Apache::loncommon::end_data_table());
        }
        $r->print(&Apache::loncommon::end_page());
	return OK;
    }
# ----------------------------------------------------------------------- Table

    if ($numdc > 0) {
        $r->print(&coursepick_jscript());
        $r->print(&Apache::loncommon::coursebrowser_javascript().
                  &Apache::loncommon::authorbrowser_javascript());
    }

    unless ((!&Apache::loncommon::show_course()) || ($nochoose) || ($countactive==1)) {
	$r->print("<h2>".&mt('Select a Course to Enter')."</h2>\n");
    }
    if ($env{'form.destinationurl'}) {
        $r->print('<input type="hidden" name="destinationurl" value="'.
                  $env{'form.destinationurl'}.'" />');
        if ($env{'form.destsymb'} ne '') {
            $r->print('<input type="hidden" name="destsymb" value="'.
                      $env{'form.destsymb'}.'" />');
        }
    }

    my $doheaders = &roletable_headers($r,\%roleclass,\%sortrole,$nochoose);
    if ($env{'environment.recentroles'}) {
        my %recent_roles =
               &Apache::lonhtmlcommon::get_recent('roles',$env{'environment.recentrolesn'});
	my $output='';
	foreach my $role (sort(keys(%recent_roles))) {
	    if (ref($roletext{'user.role.'.$role}) eq 'ARRAY') {
		$output.= &Apache::loncommon::start_data_table_row().
                          $roletext{'user.role.'.$role}->[0].
                          &Apache::loncommon::end_data_table_row();
                if ($roletext{'user.role.'.$role}->[1] ne '') {
                    $output .= &Apache::loncommon::continue_data_table_row().
                               $roletext{'user.role.'.$role}->[1].
                               &Apache::loncommon::end_data_table_row();
                }
                if ($role =~ m{dc\./($match_domain)/} 
		    && $dcroles{$1}) {
		    $output .= &adhoc_roles_row($1,'recent');
                }
	    } elsif ($numdc > 0) {
                unless ($role =~/^error\:/) {
                    my ($roletext,$role_text_end) = &display_cc_role('user.role.'.$role);
                    if ($roletext) {
                        $output.= &Apache::loncommon::start_data_table_row().
                                  $roletext.
                                  &Apache::loncommon::end_data_table_row();
                        if ($role_text_end) {
                            $output .= &Apache::loncommon::continue_data_table_row().
                                       $role_text_end.
                                       &Apache::loncommon::end_data_table_row();
                        }
                    }
                }
            }
	}
	if ($output) {
	    $r->print(&Apache::loncommon::start_data_table_empty_row()
                     .'<td align="center" colspan="5">'
                     .$recent
                     .'</td>'
                     .&Apache::loncommon::end_data_table_empty_row()
            );
	    $r->print($output);
            $doheaders ++;
	}
    }
    &print_rolerows($r,$doheaders,\%roleclass,\%sortrole,\%dcroles,\%roletext);
    if ($countactive > 1) {
        my $tremark='';
        my $tbg;
        if ($env{'request.role'} eq 'cm') {
            $tbg="LC_roles_selected";
            $tremark=&mt('Currently selected.').' ';
        } else {
                $tbg="LC_roles_is";
        }
        $r->print(&Apache::loncommon::start_data_table_row());
        unless ($nochoose) {
	    if ($env{'request.role'} ne 'cm') {
	        $r->print('<td class="'.$tbg.'"><input type="submit" value="'.
		          &mt('Select').'" name="cm" /></td>');
	    } else {
	        $r->print('<td class="'.$tbg.'">&nbsp;</td>');
	    }
        }
        $r->print('<td colspan="3">'
                 .&mt('No role specified')
                 .'</td>'
                 .'<td>'.$tremark.'&nbsp;</td>'
                 .&Apache::loncommon::end_data_table_row()
        );
    } 
    $r->print(&Apache::loncommon::end_data_table());
    unless ($nochoose) {
	$r->print("</form>\n");
    }
# ------------------------------------------------------------ Privileges Info
    if (($advanced) && (($env{'user.error.msg'}) || ($error))) {
	$r->print('<hr /><h2>'.&mt('Current Privileges').'</h2>');
	$r->print(&privileges_info());
    }
    $r->print(&Apache::lonnet::getannounce());
    if ($advanced) {
        my $esc_dom = &HTML::Entities::encode($env{'user.domain'},'"<>&');
        $r->print('<p><small><i>'
                 .&mt('This LON-CAPA server is version [_1]',$r->dir_config('lonVersion'))
                 .'</i><br />'
                 .'<a href="/adm/logout">'.&mt('Logout').'</a>&nbsp;&nbsp;'
                 .'<a href="/adm/coursecatalog?showdom='.$esc_dom.'">'
                 .&mt('Course/Community Catalog')
                 .'</a></small></p>');
    }
    $r->print(&Apache::loncommon::end_page());
    return OK;
}

sub gather_roles {
    my ($then,$refresh,$now,$reinit,$nochoose,$roletext,$sortrole,$roleclass,$futureroles,$timezones,$loncaparev) = @_;
    my ($countactive,$countfuture,$inrole,$possiblerole) = (0,0,0,'');
    my $advanced = $env{'user.adv'};
    my $tryagain = $env{'form.tryagain'};
    my @ids = &Apache::lonnet::current_machine_ids();
    foreach my $envkey (sort(keys(%env))) {
        my $button = 1;
        my $switchserver='';
        my $switchwarning;
        my ($role_text,$role_text_end,$sortkey);
        if ($envkey=~/^user\.role\./) {
            my ($role,$where,$trolecode,$tstart,$tend,$tremark,$tstatus,$tpstart,$tpend);
            &Apache::lonnet::role_status($envkey,$then,$refresh,$now,\$role,\$where,
                                         \$trolecode,\$tstatus,\$tstart,\$tend);
            next if (!defined($role) || $role eq '' || $role =~ /^gr/);
            $tremark='';
            $tpstart='&nbsp;';
            $tpend='&nbsp;';
            if ($env{'request.role'} eq $trolecode) {
                $tstatus='selected';
            }
            my $tbg;
            if (($tstatus eq 'is')
                || ($tstatus eq 'selected')
                || ($tstatus eq 'future')
                || ($env{'form.showall'})) {
                my $timezone = &role_timezone($where,$timezones);
                if ($tstart) {
                    $tpstart=&Apache::lonlocal::locallocaltime($tstart,$timezone);
                }
                if ($tend) {
                    $tpend=&Apache::lonlocal::locallocaltime($tend,$timezone);
                }
                if ($tstatus eq 'is') {
                    $tbg='LC_roles_is';
                    $possiblerole=$trolecode;
                    $countactive++;
                } elsif ($tstatus eq 'future') {
                    $tbg='LC_roles_future';
                    $button=0;
                    $futureroles->{$trolecode} = $tstart.':'.$tend;
                    $countfuture ++;
                } elsif ($tstatus eq 'expired') {
                    $tbg='LC_roles_expired';
                    $button=0;
                } elsif ($tstatus eq 'will_not') {
                    $tbg='LC_roles_will_not';
                    $tremark.=&mt('Expired after logout.').' ';
                } elsif ($tstatus eq 'selected') {
                    $tbg='LC_roles_selected';
                    $inrole=1;
                    $countactive++;
                    $tremark.=&mt('Currently selected.').' ';
                }
                my $trole;
                if ($role =~ /^cr\//) {
                    my ($rdummy,$rdomain,$rauthor,$rrole)=split(/\//,$role);
                    if ($tremark) { $tremark.='<br />'; }
                    $tremark.=&mt('Customrole defined by [_1].',$rauthor.':'.$rdomain);
                }
                $trole=Apache::lonnet::plaintext($role);
                my $ttype;
                my $twhere;
                my ($tdom,$trest,$tsection)=
                    split(/\//,Apache::lonnet::declutter($where));
                # First, Co-Authorship roles
                if (($role eq 'ca') || ($role eq 'aa')) {
                    my $home = &Apache::lonnet::homeserver($trest,$tdom);
                    my $allowed=0;
                    foreach my $id (@ids) { if ($id eq $home) { $allowed=1; } }
                    if (!$allowed) {
                        $button=0;
                        $switchserver='otherserver='.$home.'&amp;role='.$trolecode;
                    }
                    #next if ($home eq 'no_host');
                    $home = &Apache::lonnet::hostname($home);
                    $ttype='Construction Space';
                    $twhere=&mt('User').': '.$trest.'<br />'.&mt('Domain').
                        ': '.$tdom.'<br />'.
                        ' '.&mt('Server').':&nbsp;'.$home;
                    $env{'course.'.$tdom.'_'.$trest.'.description'}='ca';
                    $tremark.=&Apache::lonhtmlcommon::authorbombs('/res/'.$tdom.'/'.$trest.'/');
                    $sortkey=$role."$trest:$tdom";
                } elsif ($role eq 'au') {
                    # Authors
                    my $home = &Apache::lonnet::homeserver
                        ($env{'user.name'},$env{'user.domain'});
                    my $allowed=0;
                    foreach my $id (@ids) { if ($id eq $home) { $allowed=1; } }
                    if (!$allowed) {
                        $button=0;
                        $switchserver='otherserver='.$home.'&amp;role='.$trolecode;
                    }
                    #next if ($home eq 'no_host');
                    $home = &Apache::lonnet::hostname($home);
                    $ttype='Construction Space';
                    $twhere=&mt('Domain').': '.$tdom.'<br />'.&mt('Server').
                        ':&nbsp;'.$home;
                    $env{'course.'.$tdom.'_'.$trest.'.description'}='ca';
                    $tremark.=&Apache::lonhtmlcommon::authorbombs('/res/'.$tdom.'/'.$env{'user.name'}.'/');
                    $sortkey=$role;
                } elsif ($trest) {
                    my $tcourseid=$tdom.'_'.$trest;
                    $ttype = &Apache::loncommon::course_type($tcourseid);
                    $trole = &Apache::lonnet::plaintext($role,$ttype,$tcourseid);
                    if ($env{'course.'.$tcourseid.'.description'}) {
                        my $home=$env{'course.'.$tcourseid.'.home'};
                        $twhere=$env{'course.'.$tcourseid.'.description'};
                        $sortkey=$role."\0".$tdom."\0".$twhere."\0".$envkey;
                        $twhere = &HTML::Entities::encode($twhere,'"<>&');
                        unless ($twhere eq &mt('Currently not available')) {
                            $twhere.=' <span class="LC_fontsize_small">'.
        &Apache::loncommon::syllabuswrapper(&mt('Syllabus'),$trest,$tdom).
                                    '</span>';
                            unless ($home && grep(/^\Q$home\E$/,@ids) && $loncaparev eq '') {
                                my $required = $env{'course.'.$tcourseid.'.internal.releaserequired'};
                                if ($required ne '') {
                                    ($switchserver,$switchwarning) = 
                                        &check_release_required($loncaparev,$tcourseid,$trolecode,$required);
                                    if ($switchserver || $switchwarning) {
                                        $button = 0;
                                    }
                                }
                            }
                        }
                    } else {
                        my %newhash=&Apache::lonnet::coursedescription($tcourseid);
                        if (%newhash) {
                            $sortkey=$role."\0".$tdom."\0".$newhash{'description'}.
                                "\0".$envkey;
                            $twhere=&HTML::Entities::encode($newhash{'description'},'"<>&').
                                    ' <span class="LC_fontsize_small">'.
                                     &Apache::loncommon::syllabuswrapper(&mt('Syllabus'),$trest,$tdom).
                                    '</span>';
                            $ttype = $newhash{'type'};
                            $trole = &Apache::lonnet::plaintext($role,$ttype,$tcourseid);
                            my $home = $newhash{'home'};
                            unless ($home && grep(/^\Q$home\E$/,@ids) && $loncaparev eq '') {
                                my $required = $newhash{'internal.releaserequired'};
                                if ($required ne '') {
                                    ($switchserver,$switchwarning) =
                                        &check_release_required($loncaparev,$tcourseid,$trolecode,$required);
                                    if ($switchserver || $switchwarning) {
                                        $button = 0;
                                    }
                                }
                            }
                        } else {
                            $twhere=&mt('Currently not available');
                            $env{'course.'.$tcourseid.'.description'}=$twhere;
                            $sortkey=$role."\0".$tdom."\0".$twhere."\0".$envkey;
                            $ttype = 'Unavailable';
                        }
                    }
                    if ($tsection) {
                        $twhere.='<br />'.&mt('Section').': '.$tsection;
                    }
                    if ($role ne 'st') { $twhere.="<br />".&mt('Domain').":".$tdom; }
                } elsif ($tdom) {
                    $ttype='Domain';
                    $twhere=$tdom;
                    $sortkey=$role.$twhere;
                } else {
                    $ttype='System';
                    $twhere=&mt('system wide');
                    $sortkey=$role.$twhere;
                }
                ($role_text,$role_text_end) =
                    &build_roletext($trolecode,$tdom,$trest,$tstatus,$tryagain,
                                    $advanced,$tremark,$tbg,$trole,$twhere,$tpstart,
                                    $tpend,$nochoose,$button,$switchserver,$reinit,$switchwarning);
                $roletext->{$envkey}=[$role_text,$role_text_end];
                if (!$sortkey) {$sortkey=$twhere."\0".$envkey;}
                $sortrole->{$sortkey}=$envkey;
                $roleclass->{$envkey}=$ttype;
            }
        }
    }
    return ($countactive,$countfuture,$inrole,$possiblerole);
}

sub role_timezone {
    my ($where,$timezones) = @_;
    my $timezone;
    if (ref($timezones) eq 'HASH') { 
        if ($where =~ m{^/($match_domain)/($match_courseid)}) {
            my $cdom = $1;
            my $cnum = $2;
            if ($cdom && $cnum) {
                if (!exists($timezones->{$cdom.'_'.$cnum})) {
                    my $tz;
                    if ($env{'course.'.$cdom.'_'.$cnum.'.description'}) {
                        $tz = $env{'course.'.$cdom.'_'.$cnum.'.timezone'};
                    } else {
                        my %timehash =
                            &Apache::lonnet::get('environment',['timezone'],$cdom,$cnum);
                        $tz = $timehash{'timezone'};
                    }
                    if ($tz eq '') {
                        if (!exists($timezones->{$cdom})) {
                            my %domdefaults = 
                                &Apache::lonnet::get_domain_defaults($cdom);
                            if ($domdefaults{'timezone_def'} eq '') {
                                $timezones->{$cdom} = 'local';
                            } else {
                                $timezones->{$cdom} = $domdefaults{'timezone_def'};
                            }
                        }
                        $timezones->{$cdom.'_'.$cnum} = $timezones->{$cdom};
                    } else {
                        $timezones->{$cdom.'_'.$cnum} = 
                            &Apache::lonlocal::gettimezone($tz);
                    }
                }
                $timezone = $timezones->{$cdom.'_'.$cnum};
            }
        } else {
            my ($tdom) = ($where =~ m{^/($match_domain)});
            if ($tdom) {
                if (!exists($timezones->{$tdom})) {
                    my %domdefaults = &Apache::lonnet::get_domain_defaults($tdom);
                    if ($domdefaults{'timezone_def'} eq '') {
                        $timezones->{$tdom} = 'local';
                    } else {
                        $timezones->{$tdom} = $domdefaults{'timezone_def'};
                    }
                }
                $timezone = $timezones->{$tdom};
            }
        }
        if ($timezone eq 'local') {
            $timezone = undef;
        }
    }
    return $timezone;
}

sub roletable_headers {
    my ($r,$roleclass,$sortrole,$nochoose) = @_;
    my $doheaders;
    if ((ref($sortrole) eq 'HASH') && (ref($roleclass) eq 'HASH')) {
        $r->print('<br />'
                 .&Apache::loncommon::start_data_table()
                 .&Apache::loncommon::start_data_table_header_row()
        );
        if (!$nochoose) { $r->print('<th>&nbsp;</th>'); }
        $r->print('<th>'.&mt('User Role').'</th>'
                 .'<th>'.&mt('Extent').'</th>'
                 .'<th>'.&mt('Start').'</th>'
                 .'<th>'.&mt('End').'</th>'
                 .&Apache::loncommon::end_data_table_header_row()
        );
        $doheaders=-1;
        my @roletypes = &roletypes();
        foreach my $type (@roletypes) {
            my $haverole=0;
            foreach my $which (sort {uc($a) cmp uc($b)} (keys(%{$sortrole}))) {
                if ($roleclass->{$sortrole->{$which}} =~ /^\Q$type\E/) {
                    $haverole=1;
                }
            }
            if ($haverole) { $doheaders++; }
        }
    }
    return $doheaders;
}

sub roletypes {
    my @types = ('Domain','Construction Space','Course','Community','Unavailable','System');
    return @types; 
}

sub print_rolerows {
    my ($r,$doheaders,$roleclass,$sortrole,$dcroles,$roletext) = @_;
    if ((ref($roleclass) eq 'HASH') && (ref($sortrole) eq 'HASH')) {
        my @types = &roletypes();
        foreach my $type (@types) {
            my $output;
            foreach my $which (sort {uc($a) cmp uc($b)} (keys(%{$sortrole}))) {
                if ($roleclass->{$sortrole->{$which}} =~ /^\Q$type\E/) {
                    if (ref($roletext) eq 'HASH') {
                        if (ref($roletext->{$sortrole->{$which}}) eq 'ARRAY') {
                            $output.= &Apache::loncommon::start_data_table_row().
                                      $roletext->{$sortrole->{$which}}->[0].
                                      &Apache::loncommon::end_data_table_row();
                            if ($roletext->{$sortrole->{$which}}->[1] ne '') {
                                $output .= &Apache::loncommon::continue_data_table_row().
                                           $roletext->{$sortrole->{$which}}->[1].
                                           &Apache::loncommon::end_data_table_row();
                            }
                        }
                        if ($sortrole->{$which} =~ m-dc\./($match_domain)/-) {
                            if (ref($dcroles) eq 'HASH') {
                                if ($dcroles->{$1}) {
                                    $output .= &adhoc_roles_row($1,'');
                                }
                            }
                        }
                    }
                }
            }
            if ($output) {
                if ($doheaders > 0) {
                    $r->print(&Apache::loncommon::start_data_table_empty_row()
                             .'<td align="center" colspan="5">'
                             .&mt($type)
                             .'</td>'
                             .&Apache::loncommon::end_data_table_empty_row()
                    );
                }
                $r->print($output);
            }
        }
    }
}

sub findcourse_advice {
    my ($r) = @_;
    my $domdesc = &Apache::lonnet::domain($env{'user.domain'},'description');
    my $esc_dom = &HTML::Entities::encode($env{'user.domain'},'"<>&');
    if (&Apache::lonnet::auto_run(undef,$env{'user.domain'})) {
        $r->print(&mt('If you were expecting to see an active role listed for a particular course in the [_1] domain, it may be missing for one of the following reasons:',$domdesc).'
<ul>
 <li>'.&mt('The course has yet to be created.').'</li>
 <li>'.&mt('Automatic enrollment of registered students has not been enabled for the course.').'</li>
 <li>'.&mt('You are in a section of course for which automatic enrollment in the corresponding LON-CAPA course is not active.').'</li>
 <li>'.&mt('The start date for automated enrollment has yet to be reached.').'</li>
 <li>'.&mt('You registered for the course recently and there is a time lag between the time you register, and the time this information becomes available for the update of LON-CAPA course rosters.').'</li>
 </ul>');
    } else {
        $r->print(&mt('If you were expecting to see an active role listed for a particular course, that course may not have been created yet.').'<br />');
    }
    $r->print('<h3>'.&mt('Self-Enrollment').'</h3>'.
              '<p>'.&mt('The [_1]Course/Community Catalog[_2] provides information about all [_3] classes for which LON-CAPA courses have been created, as well as any communities in the domain.','<a href="/adm/coursecatalog?showdom='.$esc_dom.'">','</a>',$domdesc).'<br />');
    $r->print(&mt('You can search for courses and communities which permit self-enrollment, if you would like to enroll in one.').'</p>'.
              &Apache::loncoursequeueadmin::queued_selfenrollment());
    return;
}

sub requestcourse_advice {
    my ($r) = @_;
    my $domdesc = &Apache::lonnet::domain($env{'user.domain'},'description');
    my $esc_dom = &HTML::Entities::encode($env{'user.domain'},'"<>&');
    my (%can_request,%request_doms);
    &Apache::lonnet::check_can_request($env{'user.domain'},\%can_request,\%request_doms);
    if (keys(%request_doms) > 0) {
        my ($types,$typename) = &Apache::loncommon::course_types();
        if ((ref($types) eq 'ARRAY') && (ref($typename) eq 'HASH')) { 
            $r->print('<h3>'.&mt('Request creation of a course or community').'</h3>'.
                      '<p>'.&mt('You have rights to request the creation of courses and/or communities in the following domain(s):').'<ul>');
            my (@reqdoms,@reqtypes);
            foreach my $type (sort(keys(%request_doms))) {
                push(@reqtypes,$type); 
                if (ref($request_doms{$type}) eq 'ARRAY') {
                    my $domstr = join(', ',map { &Apache::lonnet::domain($_) } sort(@{$request_doms{$type}}));
                    $r->print(
                        '<li>'
                       .&mt('[_1]'.$typename->{$type}.'[_2] in domain: [_3]',
                            '<i>',
                            '</i>',
                            '<b>'.$domstr.'</b>')
                       .'</li>'
                    );
                    foreach my $dom (@{$request_doms{$type}}) {
                        unless (grep(/^\Q$dom\E/,@reqdoms)) {
                            push(@reqdoms,$dom);
                        }
                    }
                }
            }
            my @showtypes;
            foreach my $type (@{$types}) {
                if (grep(/^\Q$type\E$/,@reqtypes)) {
                    push(@showtypes,$type);
                }
            }
            my $requrl = '/adm/requestcourse';
            if (@reqdoms == 1) {
                $requrl .= '?showdom='.$reqdoms[0];
            }
            if (@showtypes > 0) {
                $requrl.=(($requrl=~/\?/)?'&':'?').'crstype='.$showtypes[0];
            }
            if (@reqdoms == 1 || @showtypes > 0) {
                $requrl .= '&state=crstype&action=new';
            } 
            $r->print('</ul>'.&mt('Use the [_1]request form[_2] to submit a request for creation of a new course or community.','<a href="'.$requrl.'">','</a>').'</p>');
        }
    }
    return;
}

sub privileges_info {
    my ($which) = @_;
    my $output;

    $which ||= $env{'request.role'};

    foreach my $envkey (sort(keys(%env))) {
	next if ($envkey!~/^user\.priv\.\Q$which\E\.(.*)/);

	my $where=$1;
	my $ttype;
	my $twhere;
	my (undef,$tdom,$trest,$tsec)=split(m{/},$where);
	if ($trest) {
	    if ($env{'course.'.$tdom.'_'.$trest.'.description'} eq 'ca') {
		$ttype='Construction Space';
		$twhere='User: '.$trest.', Domain: '.$tdom;
	    } else {
		$ttype= &Apache::loncommon::course_type($tdom.'_'.$trest);
		$twhere=$env{'course.'.$tdom.'_'.$trest.'.description'};
		if ($tsec) {
		    my $sec_type = 'Section';
		    if (exists($env{"user.role.gr.$where"})) {
			$sec_type = 'Group';
		    }
		    $twhere.=' ('.$sec_type.': '.$tsec.')';
		}
	    }
	} elsif ($tdom) {
	    $ttype='Domain';
	    $twhere=$tdom;
	} else {
	    $ttype='System';
	    $twhere='/';
	}
	$output .= "\n<h3>".&mt($ttype).': '.$twhere.'</h3>'."\n<ul>";
	foreach my $priv (sort(split(/:/,$env{$envkey}))) {
	    next if (!$priv);

	    my ($prv,$restr)=split(/\&/,$priv);
	    my $trestr='';
	    if ($restr ne 'F') {
		$trestr.=' ('.
		    join(', ',
			 map { &Apache::lonnet::plaintext($_) } 
			     (split('',$restr))).') ';
	    }
	    $output .= "\n\t".
		'<li>'.&Apache::lonnet::plaintext($prv).$trestr.'</li>';
	}
	$output .= "\n".'</ul>';
    }
    return $output;
}

sub build_roletext {
    my ($trolecode,$tdom,$trest,$tstatus,$tryagain,$advanced,$tremark,$tbg,$trole,$twhere,$tpstart,$tpend,$nochoose,$button,$switchserver,$reinit,$switchwarning) = @_;
    my ($roletext,$roletext_end);
    my $is_dc=($trolecode =~ m/^dc\./);
    my $rowspan=($is_dc) ? ''
                         : ' rowspan="2" ';

    unless ($nochoose) {
        my $buttonname=$trolecode;
        $buttonname=~s/\W//g;
        if (!$button) {
            if ($switchserver) {
                $roletext.='<td'.$rowspan.' class="'.$tbg.'">'
                          .'<a href="/adm/switchserver?'.$switchserver.'">'
                          .&mt('Switch Server')
                          .'</a></td>';
            } else {
                $roletext.=('<td'.$rowspan.' class="'.$tbg.'">&nbsp;</td>');
            }
            if ($switchwarning) {
                if ($tremark eq '') {
                    $tremark = $switchwarning;
                } else {
                    $tremark .= '<br />'.$switchwarning;
                }
            }
        } elsif ($tstatus eq 'is') {
            $roletext.='<td'.$rowspan.' class="'.$tbg.'">'.
                        '<input name="'.$buttonname.'" type="button" value="'.
                        &mt('Select').'" onclick="javascript:enterrole(this.form,\''.
                        $trolecode."','".$buttonname.'\');" /></td>';
        } elsif ($tryagain) {
            $roletext.=
                '<td'.$rowspan.' class="'.$tbg.'">'.
                '<input name="'.$buttonname.'" type="button" value="'.
                &mt('Try Selecting Again').'" onclick="javascript:enterrole(this.form,\''.
                        $trolecode."','".$buttonname.'\');" /></td>';
        } elsif ($advanced) {
            $roletext.=
                '<td'.$rowspan.' class="'.$tbg.'">'.
                '<input name="'.$buttonname.'" type="button" value="'.
                &mt('Re-Initialize').'" onclick="javascript:enterrole(this.form,\''.
                        $trolecode."','".$buttonname.'\');" /></td>';
        } elsif ($reinit) {
            $roletext.= 
                '<td'.$rowspan.' class="'.$tbg.'">'.
                '<input name="'.$buttonname.'" type="button" value="'.
                &mt('Re-Select').'" onclick="javascript:enterrole(this.form,\''.
                        $trolecode."','".$buttonname.'\');" /></td>';
        } else {
            $roletext.=
                '<td'.$rowspan.' class="'.$tbg.'">'.
                '<input name="'.$buttonname.'" type="button" value="'.
                &mt('Re-Select').'" onclick="javascript:enterrole(this.form,\''.
                        $trolecode."','".$buttonname.'\');" /></td>';
        }
    }
    if ($trolecode !~ m/^(dc|ca|au|aa)\./) {
	$tremark.=&Apache::lonannounce::showday(time,1,
			 &Apache::lonannounce::readcalendar($tdom.'_'.$trest));
    }
    $roletext.='<td>'.$trole.'</td>'
              .'<td>'.$twhere.'</td>'
              .'<td>'.$tpstart.'</td>'
              .'<td>'.$tpend.'</td>';
    if (!$is_dc) {
        $roletext_end = '<td colspan="4">'.
                        $tremark.'&nbsp;'.
                        '</td>';
    }
    return ($roletext,$roletext_end);
}

sub check_needs_switchserver {
    my ($possiblerole) = @_;
    my $needs_switchserver;
    my ($role,$where) = split(/\./,$possiblerole,2);
    my (undef,$tdom,$twho) = split(/\//,$where);
    my ($server_status,$home);
    if (($role eq 'ca') || ($role eq 'aa')) {
        ($server_status,$home) = &check_author_homeserver($twho,$tdom);
    } else {
        ($server_status,$home) = &check_author_homeserver($env{'user.name'},
                                                          $env{'user.domain'});
    }
    if ($server_status eq 'switchserver') {
        $needs_switchserver = 1;
    }
    return $needs_switchserver;
}

sub check_author_homeserver {
    my ($uname,$udom)=@_;
    if (($uname eq '') || ($udom eq '')) {
        return ('fail','');
    }
    my $home = &Apache::lonnet::homeserver($uname,$udom);
    if (&Apache::lonnet::host_domain($home) ne $udom) {
        return ('fail',$home);
    }
    my @ids=&Apache::lonnet::current_machine_ids();
    if (grep(/^\Q$home\E$/,@ids)) {
        return ('ok',$home);
    } else {
        return ('switchserver',$home);
    }
}

sub check_fordc {
    my ($dcroles,$then) = @_;
    my $numdc = 0;
    if ($env{'user.adv'}) {
        foreach my $envkey (sort keys %env) {
            if ($envkey=~/^user\.role\.dc\.\/($match_domain)\/$/) {
                my $dcdom = $1;
                my $livedc = 1;
                my ($tstart,$tend)=split(/\./,$env{$envkey});
                if ($tstart && $tstart>$then) { $livedc = 0; }
                if ($tend   && $tend  <$then) { $livedc = 0; }
                if ($livedc) {
                    $$dcroles{$dcdom} = $envkey;
                    $numdc++;
                }
            }
        }
    }
    return $numdc;
}

sub adhoc_course_role {
    my ($refresh,$then) = @_;
    my ($cdom,$cnum,$crstype);
    $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
    $crstype = &Apache::loncommon::course_type();
    if (&check_forcc($cdom,$cnum,$refresh,$then,$crstype)) {
        my $setprivs;
        if (!defined($env{'user.role.'.$env{'form.switchrole'}})) {
            $setprivs = 1;
        } else {
            my ($start,$end) = split(/\./,$env{'user.role.'.$env{'form.switchrole'}});
            if (($start && ($start>$refresh || $start == -1)) ||
                ($end && $end<$then)) {
                $setprivs = 1;
            }
        }
        if ($setprivs) {
            if ($env{'form.switchrole'} =~ m-^(in|ta|ep|ad|st|cr)([\w/]*)\./\Q$cdom\E/\Q$cnum\E/?(\w*)$-) {
                my $role = $1;
                my $custom_role = $2;
                my $usec = $3;
                if ($role eq 'cr') {
                    if ($custom_role =~ m-^/$match_domain/$match_username/\w+$-) {
                        $role .= $custom_role;
                    } else {
                        return;
                    }
                }
                my (%userroles,%newrole,%newgroups,%group_privs);
                my %cgroups =
                    &Apache::lonnet::get_active_groups($env{'user.domain'},
                                            $env{'user.name'},$cdom,$cnum);
                foreach my $group (keys(%cgroups)) {
                    $group_privs{$group} =
                        $env{'user.priv.cc./'.$cdom.'/'.$cnum.'./'.$cdom.'/'.$cnum.'/'.$group};
                }
                $newgroups{'/'.$cdom.'/'.$cnum} = \%group_privs;
                my $area = '/'.$cdom.'/'.$cnum;
                my $spec = $role.'.'.$area;
                if ($usec ne '') {
                    $spec .= '/'.$usec;
                    $area .= '/'.$usec;
                }
                &Apache::lonnet::standard_roleprivs(\%newrole,$role,$cdom,$spec,$cnum,$area);
                &Apache::lonnet::set_userprivs(\%userroles,\%newrole,\%newgroups);
                my $adhocstart = $refresh-1;
                $userroles{'user.role.'.$spec} = $adhocstart.'.';
                &Apache::lonnet::appenv(\%userroles,[$role,'cm']);
            }
        }
    }
    return;
}

sub check_forcc {
    my ($cdom,$cnum,$refresh,$then,$crstype) = @_;
    my ($is_cc,$ccrole);
    if ($crstype eq 'Community') {
        $ccrole = 'co';
    } else {
        $ccrole = 'cc';
    }
    if ($cdom ne '' && $cnum ne '') {
        if (&Apache::lonnet::is_course($cdom,$cnum)) {
            my $envkey = 'user.role.'.$ccrole.'./'.$cdom.'/'.$cnum;
            if (defined($env{$envkey})) {
                $is_cc = 1;
                my ($tstart,$tend)=split(/\./,$env{$envkey});
                if ($tstart && $tstart>$refresh) { $is_cc = 0; }
                if ($tend   && $tend  <$then) { $is_cc = 0; }
            }
        }
    }
    return $is_cc;
}

sub check_release_required {
    my ($loncaparev,$tcourseid,$trolecode,$required) = @_;
    my ($switchserver,$warning);
    if ($required ne '') {
        my ($reqdmajor,$reqdminor) = ($required =~ /^(\d+)\.(\d+)$/);
        my ($major,$minor) = ($loncaparev =~ /^\'?(\d+)\.(\d+)\.[\w.\-]+\'?$/);
        if ($reqdmajor ne '' && $reqdminor ne '') {
            my $otherserver;
            if (($major eq '' && $minor eq '') || 
                (($reqdmajor > $major) || (($reqdmajor == $major) && ($reqdminor > $minor)))) {
                my ($userdomserver) = &Apache::lonnet::choose_server($env{'user.domain'});
                my $switchlcrev = 
                    &Apache::lonnet::get_server_loncaparev($env{'user.domain'},
                                                           $userdomserver);
                my ($swmajor,$swminor) = ($switchlcrev =~ /^\'?(\d+)\.(\d+)\.[\w.\-]+\'?$/);
                if (($swmajor eq '' && $swminor eq '') || ($reqdmajor > $swmajor) || 
                    (($reqdmajor == $swmajor) && ($reqdminor > $swminor))) {
                    my $cdom = $env{'course.'.$tcourseid.'.domain'};
                    if ($cdom ne $env{'user.domain'}) {
                        my ($coursedomserver,$coursehostname) = &Apache::lonnet::choose_server($cdom); 
                        my $serverhomeID = &Apache::lonnet::get_server_homeID($coursehostname);
                        my $serverhomedom = &Apache::lonnet::host_domain($serverhomeID);
                        my %defdomdefaults = &Apache::lonnet::get_domain_defaults($serverhomedom);
                        my %udomdefaults = &Apache::lonnet::get_domain_defaults($env{'user.domain'});
                        my $remoterev = &Apache::lonnet::get_server_loncaparev($serverhomedom,$coursedomserver);
                        my $canhost =
                            &Apache::lonnet::can_host_session($env{'user.domain'},
                                                              $coursedomserver,
                                                              $remoterev,
                                                              $udomdefaults{'remotesessions'},
                                                              $defdomdefaults{'hostedsessions'});

                        if ($canhost) {
                            $otherserver = $coursedomserver;
                        } else {
                            $warning = &mt('Requires LON-CAPA version [_1].',$env{'course.'.$tcourseid.'.internal.releaserequired'}).'<br />'. &mt("No suitable server could be found amongst servers in either your own domain or in the course's domain.");
                        }
                    } else {
                        $warning = &mt('Requires LON-CAPA version [_1].',$env{'course.'.$tcourseid.'.internal.releaserequired'}).'<br />'.&mt("No suitable server could be found amongst servers in your own domain (which is also the course's domain).");
                    }
                } else {
                    $otherserver = $userdomserver;
                }
            }
            if ($otherserver ne '') {
                $switchserver = 'otherserver='.$otherserver.'&amp;role='.$trolecode;
            }
        }
    }
    return ($switchserver,$warning);
}

sub update_content_constraints {
    my ($cdom,$cnum,$chome,$cid) = @_;
    my %curr_reqd_hash = &Apache::lonnet::userenvironment($cdom,$cnum,'internal.releaserequired');
    my ($reqdmajor,$reqdminor) = split(/\./,$curr_reqd_hash{'internal.releaserequired'}); 
    my %checkresponsetypes;
    foreach my $key (keys(%Apache::lonnet::needsrelease)) {
        my ($item,$name,$value) = split(/:/,$key);
        if ($item eq 'resourcetag') {
            if ($name eq 'responsetype') {
                $checkresponsetypes{$value} = $Apache::lonnet::needsrelease{$key}
            }
        }
    }
    my $navmap = Apache::lonnavmaps::navmap->new();
    if (defined($navmap)) {
        my %allresponses;
        foreach my $res ($navmap->retrieveResources(undef,sub { $_[0]->is_problem() },1,0)) {
            my %responses = $res->responseTypes();
            foreach my $key (keys(%responses)) {
                next unless(exists($checkresponsetypes{$key}));
                $allresponses{$key} += $responses{$key};
            }
        }
        foreach my $key (keys(%allresponses)) {
            my ($major,$minor) = split(/\./,$checkresponsetypes{$key});
            if (($major > $reqdmajor) || ($major == $reqdmajor && $minor > $reqdminor)) { 
                ($reqdmajor,$reqdminor) = ($major,$minor);
            } 
        }
        undef($navmap);
    }
    unless (($reqdmajor eq '') && ($reqdminor eq '')) {
        &Apache::lonnet::update_released_required($reqdmajor.'.'.$reqdminor,$cdom,$cnum,$chome,$cid);
    }
    return;
}

sub courselink {
    my ($dcdom,$rowtype) = @_;
    my $courseform=&Apache::loncommon::selectcourse_link
                   ('rolechoice','dccourse'.$rowtype.'_'.$dcdom,
                    'dcdomain'.$rowtype.'_'.$dcdom,'coursedesc'.$rowtype.'_'.
                    $dcdom,$dcdom,undef,'Course/Community');
    my $hiddenitems = '<input type="hidden" name="dcdomain'.$rowtype.'_'.$dcdom.'" value="'.$dcdom.'" />'.
                      '<input type="hidden" name="origdom'.$rowtype.'_'.$dcdom.'" value="'.$dcdom.'" />'.
                      '<input type="hidden" name="dccourse'.$rowtype.'_'.$dcdom.'" value="" />'.
                      '<input type="hidden" name="coursedesc'.$rowtype.'_'.$dcdom.'" value="" />';
    return $courseform.$hiddenitems;
}

sub coursepick_jscript {
    my %lt = &Apache::lonlocal::texthash(
                  plsu => "Please use the 'Select Course/Community' link to open a separate pick course window where you may select the course or community you wish to enter.",
                  youc => 'You can only use this screen to select courses and communities in the current domain.',
             );
    my $verify_script = <<"END";
<script type="text/javascript">
// <![CDATA[
function verifyCoursePick(caller) {
    var numbutton = getIndex(caller)
    var pickedCourse = document.rolechoice.elements[numbutton+4].value
    var pickedDomain = document.rolechoice.elements[numbutton+2].value
    if (document.rolechoice.elements[numbutton+2].value == document.rolechoice.elements[numbutton+3].value) {
        if (pickedCourse != '') {
            if (numbutton != -1) {
                var courseTarget = "cc./"+pickedDomain+"/"+pickedCourse
                document.rolechoice.elements[numbutton+1].name = courseTarget
                document.rolechoice.submit()
            }
        }
        else {
            alert("$lt{'plsu'}");
        }
    }
    else {
        alert("$lt{'youc'}")
    }
}
function getIndex(caller) {
    for (var i=0;i<document.rolechoice.elements.length;i++) {
        if (document.rolechoice.elements[i] == caller) {
            return i;
        }
    }
    return -1;
}
// ]]>
</script>
END
    return $verify_script;
}

sub coauthorlink {
    my ($dcdom,$rowtype) = @_;
    my $coauthorform=&Apache::loncommon::selectauthor_link('rolechoice',$dcdom);
    my $hiddenitems = '<input type="hidden" name="adhoccauname'.$rowtype.'_'.$dcdom.'" value="" />';
    return $coauthorform.$hiddenitems;
}

sub display_cc_role {
    my $rolekey = shift;
    my ($roletext,$roletext_end);
    my $advanced = $env{'user.adv'};
    my $tryagain = $env{'form.tryagain'};
    unless ($rolekey =~/^error\:/) {
        if ($rolekey =~ m{^user\.role\.(cc|co)\./($match_domain)/($match_courseid)$}) {
            my $ccrole = $1;
            my $tdom = $2;
            my $trest = $3;
            my $tcourseid = $tdom.'_'.$trest;
            my $trolecode = $ccrole.'./'.$tdom.'/'.$trest;
            my $twhere;
            my $ttype;
            my $tbg='LC_roles_is';
            my %newhash=&Apache::lonnet::coursedescription($tcourseid);
            if (%newhash) {
                $twhere=$newhash{'description'}.
                        ' <span style="LC_fontsize_small">'.
                        &Apache::loncommon::syllabuswrapper(&mt('Syllabus'),$trest,$tdom).
                        '</span>';
                $ttype = $newhash{'type'};
            } else {
                $twhere=&mt('Currently not available');
                $env{'course.'.$tcourseid.'.description'}=$twhere;
            }
            my $trole = &Apache::lonnet::plaintext($ccrole,$ttype,$tcourseid);
            $twhere.="<br />".&mt('Domain').":".$tdom;
            ($roletext,$roletext_end) = &build_roletext($trolecode,$tdom,$trest,'is',$tryagain,$advanced,'',$tbg,$trole,$twhere,'','','',1,'');
        }
    }
    return ($roletext,$roletext_end);
}

sub adhoc_roles_row {
    my ($dcdom,$rowtype) = @_;
    my $output = &Apache::loncommon::continue_data_table_row()
                 .' <td colspan="5">'
                 .&mt('[_1]Ad hoc[_2] roles in domain [_3] --'
                     ,'<span class="LC_cusr_emph">','</span>',$dcdom)
                 .' ';
    my $selectcclink = &courselink($dcdom,$rowtype);
    my $ccrole = &Apache::lonnet::plaintext('co',undef,undef,1);
    my $carole = &Apache::lonnet::plaintext('ca');
    my $selectcalink = &coauthorlink($dcdom,$rowtype);
    $output.=$ccrole.': '.$selectcclink
            .' | '.$carole.': '.$selectcalink.'</td>'
            .&Apache::loncommon::end_data_table_row();
    return $output;
}

sub recent_filename {
    my $area=shift;
    return 'nohist_recent_'.&escape($area);
}

sub courseloadpage {
    my ($courseid) = @_;
    my $startpage;
    my %entry_settings = &Apache::lonnet::get('nohist_whatsnew',
					      [$courseid.':courseinit']);
    my ($tmp) = %entry_settings;
    unless ($tmp =~ /^error: 2 /) {
        $startpage = $entry_settings{$courseid.':courseinit'};
    }
    if ($startpage eq '') {
        if (exists($env{'environment.course_init_display'})) {
            $startpage = $env{'environment.course_init_display'};
        }
    }
    return $startpage;
}

1;
__END__

=head1 NAME

Apache::lonroles - User Roles Screen

=head1 SYNOPSIS

Invoked by /etc/httpd/conf/srm.conf:

 <Location /adm/roles>
 PerlAccessHandler       Apache::lonacc
 SetHandler perl-script
 PerlHandler Apache::lonroles
 ErrorDocument     403 /adm/login
 ErrorDocument	  500 /adm/errorhandler
 </Location>

=head1 OVERVIEW

=head2 Choosing Roles

C<lonroles> is a handler that allows a user to switch roles in
mid-session. LON-CAPA attempts to work with "No Role Specified", the
default role that a user has before selecting a role, as widely as
possible, but certain handlers for example need specification which
course they should act on, etc. Both in this scenario, and when the
handler determines via C<lonnet>'s C<&allowed> function that a certain
action is not allowed, C<lonroles> is used as error handler. This
allows the user to select another role which may have permission to do
what they were trying to do. C<lonroles> can also be accessed via the
B<CRS> button in the Remote Control.

=begin latex

\begin{figure}
\begin{center}
\includegraphics[width=0.45\paperwidth,keepaspectratio]{Sample_Roles_Screen}
  \caption{\label{Sample_Roles_Screen}Sample Roles Screen} 
\end{center}
\end{figure}

=end latex

=head2 Role Initialization

The privileges for a user are established at login time and stored in the session environment. As a consequence, a new role does not become active till the next login. Handlers are able to query for privileges using C<lonnet>'s C<&allowed> function. When a user first logs in, their role is the "common" role, which means that they have the sum of all of their privileges. During a session it might become necessary to choose a particular role, which as a consequence also limits the user to only the privileges in that particular role.

=head1 INTRODUCTION

This module enables a user to select what role he wishes to
operate under (instructor, student, teaching assistant, course
coordinator, etc).  These roles are pre-established by the actions
of upper-level users.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 HANDLER SUBROUTINE

This routine is called by Apache and mod_perl.

=over 4

=item *

Roles Initialization (yes/no)

=item *

Get Error Message from Environment

=item *

Who is this?

=item *

Generate Page Output

=item *

Choice or no choice

=item *

Table

=item *

Privileges

=back

=cut
