# LearningOnline Network with CAPA
# option list style responses
#
# $Id: functionplotresponse.pm,v 1.49 2010/12/03 00:52:59 www Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#

package Apache::functionplotresponse;
use strict;
use Apache::response();
use Apache::lonlocal;
use Apache::lonnet;
 
BEGIN {
  &Apache::lonxml::register('Apache::functionplotresponse',('functionplotresponse','backgroundplot','spline',
                                                            'functionplotrule','functionplotruleset',
                                                            'functionplotelements'));
}

#
# There can be a number of applets on a page, each called ggbApplet_$id, 
# where $id is the "_"-concatenated part and responseid
#

sub geogebra_startcode {
    my ($id)=@_;
    return (<<ENDSTARTCODE);
<applet name="ggbApplet_$id" code="geogebra.GeoGebraApplet" archive="geogebra.jar"
         codebase="/adm/geogebra/"  width="722" height="447" MAYSCRIPT>
       <param name="java_arguments" value="-Xmx512m -Djnlp.packEnabled=true"/>
ENDSTARTCODE
}

sub geogebra_endcode {
    return &Apache::lonhtmlcommon::java_not_enabled()."</applet>\n";
}

#
# This is the internal GeoGebra bytecode which defines the spline functions
#
sub geogebra_spline_program {
    return (<<ENDSPLINEPROGRAM);
<param name="ggbBase64" value="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" />
ENDSPLINEPROGRAM
}

#
# The standard set of parameters inside <applet>
#
sub geogebra_default_parameters {
   my ($id)=@_;
   return(<<ENDDEFAULTPARAMETERS);
        <param name="image" value="/adm/lonIcons/lonanim.gif"  />
        <param name="boxborder" value="false"  />
        <param name="centerimage" value="true"  />
	<param name="cache_archive" value="geogebra.jar, geogebra_main.jar, geogebra_gui.jar, geogebra_cas.jar, geogebra_algos.jar, geogebra_export.jar, geogebra_javascript.jar, jlatexmath.jar, jlm_greek.jar, jlm_cyrillic.jar, geogebra_properties.jar" />
	<param name="cache_version" value="3.9.115.0, 3.9.115.0, 3.9.115.0, 3.9.115.0, 3.9.115.0, 3.9.115.0, 3.9.115.0, 3.9.115.0, 3.9.115.0, 3.9.115.0, 3.9.115.0" />
        <param name="framePossible" value="false" />

        <param name="showResetIcon" value="false" />
        <param name="showAnimationButton" value="false" />
        <param name="enableRightClick" value="false" />
        <param name="errorDialogsActive" value="true" />
        <param name="enableLabelDrags" value="false" />
        <param name="showMenuBar" value="false" />
        <param name="showToolBar" value="false" />
        <param name="showToolBarHelp" value="false" />
        <param name="showAlgebraInput" value="false" />
        <param name="enableShiftDragZoom" value="false" />
        <param name="allowRescaling" value="false" />
        <param name="enableLabelDrags" value="false" />
        <param name="ggbOnInitParam" value="applet_$id" />
ENDDEFAULTPARAMETERS
}

#
# This subroutine is called by LON-CAPA at </problem>
# Each applet on the page will call function ggbOnInit when it is done loading
# This function in turn will call the respective function registered by start_init_script
# Which one of the registered functions is called is determined by ggbOnInitParam, which GeoGebra passes to ggbOnInit
#

sub init_script {
   if ($#Apache::functionplotresponse::callscripts>=0) {
      my $script='';
      foreach my $id (@Apache::functionplotresponse::callscripts) {
          $script.="if (param=='applet_$id') { loaded_$id=true; }\n";
      }
      $script.="if (".join(' && ',map { "loaded_$_" } (@Apache::functionplotresponse::callscripts)).
               ") { setTimeout('ggbInitAll()',20) }";
      my $calls=join("\n",map { "ggbInit_$_();" } (@Apache::functionplotresponse::callscripts)); 
      return (<<ENDGGBINIT);
<script type="text/javascript">
// <![CDATA[
// Function that each applet will call when loaded
// It will pass "its" parameter
// Set flags for when an applet is loaded, wait till all are loaded, and then some
function ggbOnInit(param) {
$script
}
function ggbInitAll() {
$calls
}
// ]]>
</script>
ENDGGBINIT
   }
}

#
# Each Geogebra applet is supposed to call this when parameters change
# Changes the hidden fields on the web page
#
sub update_script {
    my ($id)=@_;
    return (<<ENDUPDATESCRIPT);
<script type="text/javascript">
// <![CDATA[
function updatePointCoordinates_$id(coordinateName) {
            var x = document.ggbApplet_$id.getXcoord(coordinateName);
            var y = document.ggbApplet_$id.getYcoord(coordinateName);
            document.lonhomework.elements["HWVAL_$id\_" + coordinateName + "_x"].value = x;
            document.lonhomework.elements["HWVAL_$id\_" + coordinateName + "_y"].value = y;
        }
// ]]>
</script>
ENDUPDATESCRIPT
}

#
# Register the above update-handler for a variable
#

sub update_register {
   my ($id,$variable)=@_;
   return "document.ggbApplet_$id.registerObjectUpdateListener('$variable','updatePointCoordinates_$id');\n";
}

#
# Set a point coordinate variable
#
sub set_point_coordinate {
   my ($id,$variable,$x,$y,$fixed)=@_;
   my $mult=($fixed?'a*':'');
   return (<<ENDSETVARIABLE);
document.ggbApplet_$id.evalCommand("a=1");
document.ggbApplet_$id.evalCommand("$variable=$mult($x,$y)");
document.ggbApplet_$id.setLabelVisible("$variable",false);
ENDSETVARIABLE
}

#
# Set a slope coordinate variable
#
sub set_slope_coordinate {
   my ($id,$variable,$xrel,$yrel,$xmin,$xmax,$ymin,$ymax,$pointname,$fixed)=@_;
   my $xvariable=$variable.'x';
   my $yvariable=$variable.'y';
   my $domain=$xmax-$xmin;
   my $range=$ymax-$ymin;
   my $xinterval=$domain/100.;
   my $yinterval=$range/200.;
   my $mult=($fixed?'a*':'');
   return (<<ENDSETSVARIABLE);
document.ggbApplet_$id.evalCommand("a=1");
document.ggbApplet_$id.evalCommand("$xvariable=Slider[0,$domain,$xinterval]");
document.ggbApplet_$id.setVisible("$xvariable", false);
document.ggbApplet_$id.evalCommand("$xvariable=$xrel");
document.ggbApplet_$id.evalCommand("$yvariable=Slider[-$range,$range,$yinterval]");
document.ggbApplet_$id.setVisible("$yvariable", false);
document.ggbApplet_$id.evalCommand("$yvariable=$yrel");
document.ggbApplet_$id.evalCommand("$variable=$mult($xvariable+x($pointname),$yvariable+y($pointname))");
document.ggbApplet_$id.setLabelVisible("$variable", false);
ENDSETSVARIABLE
}

#
# Input field name for a coordinate variable
#

sub field_name {
    my ($id,$variable,$name)=@_;
    return "HWVAL_$id\_$variable\_$name";
}

#
# Generate an input field for a coordinate variable
#

sub generate_input_field {
    my ($id,$variable,$x,$y)=@_;
    $Apache::functionplotresponse::inputfields.=
       "<input type='hidden' name='".&field_name($id,$variable,'x')."' value='$x' />\n".
       "<input type='hidden' name='".&field_name($id,$variable,'y')."' value='$y' />\n";
}

#
# Initialize a new point coordinate variable at set a listener on it
#
sub new_point_coordinate {
    my ($id,$variable,$x,$y,$fixed)=@_;
    if (defined($Apache::functionplotresponse::previous{&field_name($id,$variable,'x')})) {
       $x=$Apache::functionplotresponse::previous{&field_name($id,$variable,'x')};
    }
    if (defined($Apache::functionplotresponse::previous{&field_name($id,$variable,'y')})) {
       $y=$Apache::functionplotresponse::previous{&field_name($id,$variable,'y')};
    }
    &generate_input_field($id,$variable,$x,$y);
    return &set_point_coordinate($id,$variable,$x,$y,$fixed).&update_register($id,$variable);
}

#
# Initialize a new slope coordinate variable at set a listener on it
#
sub new_slope_coordinate {
    my ($id,$variable,$x,$y,$pointname,$xp,$yp,$xmin,$xmax,$ymin,$ymax,$fixed)=@_;
#
# $variable: name of the slope point
# $x, $y: coordinates of the slope point
# $pointname: name of the associated point point
# $xp $yp: coordinates of the point point
#
    if (defined($Apache::functionplotresponse::previous{&field_name($id,$variable,'x')})) {
       $x=$Apache::functionplotresponse::previous{&field_name($id,$variable,'x')};
    }
    if (defined($Apache::functionplotresponse::previous{&field_name($id,$variable,'y')})) {
       $y=$Apache::functionplotresponse::previous{&field_name($id,$variable,'y')};
    }
    if (defined($Apache::functionplotresponse::previous{&field_name($id,$pointname,'x')})) {
       $xp=$Apache::functionplotresponse::previous{&field_name($id,$pointname,'x')};
    }
    if (defined($Apache::functionplotresponse::previous{&field_name($id,$pointname,'y')})) {
       $yp=$Apache::functionplotresponse::previous{&field_name($id,$pointname,'y')};
    }

    &generate_input_field($id,$variable,$x,$y);
    my $xrel=$x-$xp;
    my $yrel=$y-$yp;
    return &set_slope_coordinate($id,$variable,$xrel,$yrel,$xmin,$xmax,$ymin,$ymax,$pointname,$fixed).&update_register($id,$variable);
}

#
# This registers the init-function call for ggbOnInit, which LON-CAPA will place at </problem>
# It then starts the right headers
#
sub start_init_script {
    my ($id)=@_;
# Add id to the list of ggbInit_$id functions that need to be called
    push(@Apache::functionplotresponse::callscripts,$id);
# ... and open this function
    return (<<ENDSTARTINIT);
<script type="text/javascript">
// <![CDATA[
// variable that will eventually be passed back to the server
var coordinateMap_$id = [];
// flag for not loaded yet
var loaded_$id=false;
// Init-function for applet
function ggbInit_$id() {
ENDSTARTINIT
}

#
# This sets the axes inside ggbInit_$id
#

sub axes_script {
    my ($id,$xmin,$xmax,$ymin,$ymax,$xvisible,$yvisible,$gvisible)=@_;
    return (<<ENDAXESSCRIPT);
            // changes (xmin, xmax, ymin, ymax)
            document.ggbApplet_$id.setCoordSystem($xmin,$xmax,$ymin,$ymax);

            // makes the (x,y) axis (in)visible
            document.ggbApplet_$id.setAxesVisible($xvisible,$yvisible);
            // makes the grid (in)visible
            document.ggbApplet_$id.setGridVisible($gvisible);
ENDAXESSCRIPT
}

sub axes_label {
    my ($id,$xmin,$xmax,$ymin,$ymax,$xlabel,$ylabel)=@_;
    unless ($xlabel || $ylabel) { return ''; }
    my $return='document.ggbApplet_'.$id.'.evalCommand("topRight=Corner[3]");';
    if ($xlabel) {
      if (($ymin<0) && ($ymax>0)) {
       $return.=(<<ENDXAXISLABELSCRIPT);
document.ggbApplet_$id.evalCommand("Xlabel=(x(topRight)-AxisStepX[],AxisStepY[]/6)");
document.ggbApplet_$id.setVisible("Xlabel",false);
document.ggbApplet_$id.evalCommand("Text[\\"$xlabel\\", Xlabel]");
ENDXAXISLABELSCRIPT
      } else {
       $return.=(<<ENDXOFFAXISLABEL);
document.ggbApplet_$id.evalCommand("LowerRight=Corner[2]");
document.ggbApplet_$id.evalCommand("Text[\\"$xlabel\\", (x(LowerRight) - AxisStepX[], y(LowerRight) + AxisStepY[] / 2)]");
ENDXOFFAXISLABEL
      }
    }
    if ($ylabel) {
      if (($xmin<0) && ($xmax>0)) {
       $return.=(<<ENDYAXISLABELSCRIPT);
document.ggbApplet_$id.evalCommand("Ylabel=(AxisStepX[]/6,y(topRight)-AxisStepY[]/3)");
document.ggbApplet_$id.setVisible("Ylabel",false);
document.ggbApplet_$id.evalCommand("Text[\\"$ylabel\\", Ylabel]");
ENDYAXISLABELSCRIPT
      } else {
       $return.=(<<ENDYOFFAXISLABEL);
document.ggbApplet_$id.evalCommand("UpperLeft=Corner[4]");
document.ggbApplet_$id.evalCommand("Text[\\"$ylabel\\", (x(UpperLeft) + AxisStepX[] / 5, y(UpperLeft) - AxisStepY[] / 1.8)]");
ENDYOFFAXISLABEL
      }
    }
    return $return;
}

sub plot_script {
   my ($id,$function,$fixed,$label,$color,$xmin,$xmax,$thickness)=@_;
   $label=~s/\W//g;
   if (($label) && ($label!~/^[A-Za-z]/)) {
      $label='C'.$label;
   }
   my $visible=0;
   if ($label) {
      $visible="1";
   } else {
      $Apache::functionplotresponse::counter++;
      $label='C'.$Apache::functionplotresponse::counter;
   }
   my $rc=0;
   my $gc=0;
   my $bc=0;
   if ($color) {
      my ($rh,$gh,$bh)=($color=~/(..)(..)(..)/);
      $rc=hex($rh);
      $gc=hex($gh);
      $bc=hex($bh);
   }
   if ($fixed) {
      return "document.ggbApplet_$id.evalCommand('$label=Function[$function,$xmin,$xmax]');\n".
             ($visible?'':"document.ggbApplet_$id.setLabelVisible('$label', false);\n").
             ($color?"document.ggbApplet_$id.setColor('$label',$rc,$gc,$bc);\n":'').
             ($thickness?"document.ggbApplet_$id.setLineThickness('$label',$thickness);\n":'');
   } else {
       return "document.ggbApplet_$id.evalCommand('y=$function');\n";
   }
}

#
# Subroutine that generates code for spline $label based on stored information
#

sub generate_spline {
   my ($id,$label,$xmin,$xmax,$ymin,$ymax,$fixed)=@_;
   my $result='';
   my $order=$Apache::functionplotresponse::splineorder{$label};
   my $x=$Apache::functionplotresponse::splineinitx{$label};
   my $y=$Apache::functionplotresponse::splineinity{$label};
   my $sx=$Apache::functionplotresponse::splinescalex{$label};
   my $sy=$Apache::functionplotresponse::splinescaley{$label};
   my @coords=();
   foreach my $i (1..$order) {
       $result.=&new_point_coordinate($id,$label.'P'.$i,$x,$y,$fixed);
       my $xp=$x;
       $x+=$sx/(2.*($order-1));
       push(@coords,$label.'P'.$i);
       $result.=&new_slope_coordinate($id,$label.'S'.$i,$x,$y+$sy,$label.'P'.$i,$xp,$y,$xmin,$xmax,$ymin,$ymax,$fixed);
       $x+=$sx/(2.*($order-1));
       push(@coords,$label.'S'.$i);
   }
   $result.='document.ggbApplet_'.$id.'.evalCommand("Spline'.$order.'['.join(',',@coords).']");'."\n";
   return $result;
}
#
# <backgroundplot function="..." fixed="yes/no" />
#
sub start_backgroundplot {
   my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
   my $result='';
   my $internalid = $Apache::inputtags::part.'_'.$Apache::inputtags::response[-1];
   my $function=&Apache::lonxml::get_param('function',$parstack,$safeeval);
   my $xinitial=&Apache::lonxml::get_param('xinitial',$parstack,$safeeval);
   my $xfinal=&Apache::lonxml::get_param('xfinal',$parstack,$safeeval);
   my $label=&Apache::lonxml::get_param('label',$parstack,$safeeval);
   my $color=&Apache::lonxml::get_param('color',$parstack,$safeeval);
   $color=~s/[^a-fA-F0-9]//gs;
   unless (length($color)==6) { $color=''; }
   my $fixed=(&Apache::lonxml::get_param('fixed',$parstack,$safeeval)=~/on|true|yes|1/i?1:0);
 
   unless ($function) { $function="0"; }
   if ($target eq 'web') {
      my ($xmin,$xmax,$ymin,$ymax)=&boundaries($parstack,$safeeval,-3);
      unless (defined($xinitial)) { $xinitial=$xmin; }
      unless (defined($xfinal)) { $xfinal=$xmax; }
      $result.=&plot_script($internalid,$function,$fixed,$label,$color,$xinitial,$xfinal);
   } elsif ($target eq 'edit') {
        $result=&Apache::edit::tag_start($target,$token,'Background Function Plot').
             &Apache::edit::text_arg('Function:','function',
                                     $token,'16').
             &Apache::edit::text_arg('Initial x-value (optional):','xinitial',
                                     $token,'8').
             &Apache::edit::text_arg('Final x-value (optional):','xfinal',
                                     $token,'8').
             &Apache::edit::text_arg('Label on Plot:','label',
                                     $token,'8').
             &Apache::edit::text_arg('Color (hex code):','color',
                                     $token,'8').
             &Apache::edit::select_arg('Fixed location:','fixed',
                                  ['yes','no'],$token).
             &Apache::edit::end_row();
  } elsif ($target eq 'modified') {
    my $constructtag=&Apache::edit::get_new_args($token,$parstack,
                                                 $safeeval,'function','label','xinitial','xfinal','color','fixed');
    if ($constructtag) { $result=&Apache::edit::rebuild_tag($token); }
  }
  return $result;
}

sub end_backgroundplot {
   my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
   my $result='';
   if ($target eq 'edit') {
       $result=&Apache::edit::end_table();
   }
   return $result;
}

#
# <functionplotrule ... />
#
sub start_functionplotrule {
   my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
   my $result='';
   my $label=&Apache::lonxml::get_param('index',$parstack,$safeeval);
   $Apache::functionplotresponse::counter++;
   if ($label=~/\W/) {
      &Apache::lonxml::warning(&mt('Rule indices should only contain alphanumeric characters.'));
   }
   $label=~s/\W//gs;
   unless ($label) {
      $label='R'.$Apache::functionplotresponse::counter;
   } else {
      $label='R'.$label;
   }
   if ($Apache::functionplotresponse::splineorder{$label}) {
       &Apache::lonxml::error(&mt('Rule indices must be unique.'));
   }


   if ($target eq 'grade') {
# Simply remember - in order - for later
      my $beginninglabel=&Apache::lonxml::get_param('xinitiallabel',$parstack,$safeeval);
      my $endinglabel=&Apache::lonxml::get_param('xfinallabel',$parstack,$safeeval);
      if (($beginninglabel=~/\W/) || ($endinglabel=~/W/)) {
          &Apache::lonxml::warning(&mt('Rule labels must be alphanumeric.'));
      }
      $beginninglabel=~s/\W//gs;
      $endinglabel=~s/\W//gs;
      my $relationship=&Apache::lonxml::get_param('relationship',$parstack,$safeeval);
      $relationship=~s/\W//gs;
      $relationship=lc($relationship);
      unless ($relationship=~/^(eq|ge|gt|le|lt|ne)$/) {
          &Apache::lonxml::warning(&mt('Rule relationship not defined.'));
          $relationship='eq';
      }
      my $derivative=&Apache::lonxml::get_param('derivativeorder',$parstack,$safeeval);
      unless (($derivative==-1) || ($derivative==0) || ($derivative==1) || ($derivative==2)) {
         &Apache::lonxml::warning(&mt('Rule derivative not defined.'));
         $derivative=0;
      }
      push(@Apache::functionplotresponse::functionplotrules,join(':',(
           $label,
           $derivative,
           &Apache::lonxml::get_param('xinitial',$parstack,$safeeval),
           $beginninglabel,
           &Apache::lonxml::get_param('xfinal',$parstack,$safeeval),
           $endinglabel,
           &Apache::lonxml::get_param('minimumlength',$parstack,$safeeval),
           &Apache::lonxml::get_param('maximumlength',$parstack,$safeeval),
           $relationship,
           &Apache::lonxml::get_param('value',$parstack,$safeeval),
           &Apache::lonxml::get_param('percenterror',$parstack,$safeeval)
          )));
   } elsif ($target eq 'edit') {
        $result=&Apache::edit::tag_start($target,$token,'Function Plot Evaluation Rule').
             &Apache::edit::text_arg('Index/Name:','index',
                                     $token,'10').'&nbsp;'.
             &Apache::edit::select_arg(&mt('Function:'),'derivativeorder',
                                  [['0','Function itself'],
                                   ['1','First derivative'],
                                   ['2','Second derivative'],
                                   ['-1','Integral']],$token).'<br />'.
             &Apache::edit::text_arg('Initial x-value:','xinitial',
                                      $token,'8').
             &Apache::edit::select_or_text_arg('Initial x-value label:','xinitiallabel',
                                               [['start','Start of Plot'],
                                                ['end','End of Plot']],$token,'8').'<br />'.

             &Apache::edit::text_arg('Final x-value (optional):','xfinal',
                                      $token,'8').
             &Apache::edit::select_or_text_arg('Final x-value label (optional):','xfinallabel',
                                               [['end','End of Plot']],$token,'8').'<br />'.
             &Apache::edit::text_arg('Minimum length for range (optional):','minimumlength',
                                     $token,'8').
             &Apache::edit::text_arg('Maximum length for range (optional):','maximumlength',
                                     $token,'8').'<br />'.
             &Apache::edit::select_or_text_arg(&mt('Relationship:'),'relationship',
                                  [['eq','equal'],
                                   ['ne','not equal'],
                                   ['ge','greater than or equal'],
                                   ['gt','greater than'],
                                   ['lt','less than'],
                                   ['le','less than or equal']],$token).
             $result.= &Apache::edit::select_or_text_arg('Value:','value',
                                               [['undef','not defined']],$token,'8').
             &Apache::edit::text_arg('Percent error:','percenterror',
                                     $token,'8').
             &Apache::edit::end_row();
  } elsif ($target eq 'modified') {
    my $constructtag=&Apache::edit::get_new_args($token,$parstack,
                                                 $safeeval,'index','derivativeorder',
                                                           'xinitial','xinitiallabel','xfinal','xfinallabel',
                                                           'minimumlength','maximumlength',
                                                           'relationship','value','percenterror');
    if ($constructtag) { $result=&Apache::edit::rebuild_tag($token); }
   }
   return $result;
}

sub end_functionplotrule {
   my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
   my $result='';
   if ($target eq 'edit') {
       $result=&Apache::edit::end_table();
   }
   return $result;
}


#
# <spline index="..." order="1,2,3,4" initx="..." inity="..." scalex="..." scaley="..." />
#
# Unfortunately, GeoGebra seems to want all splines after everything else, so we need to store them
#
sub start_spline {
   my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
   my $result='';
   if ($target eq 'web') {
      my $label=&Apache::lonxml::get_param('index',$parstack,$safeeval);
      $Apache::functionplotresponse::counter++;
      if ($label=~/\W/) {
         &Apache::lonxml::warning(&mt('Spline indices should only contain alphanumeric characters.'));
      }
      $label=~s/\W//gs;
      unless ($label) { 
         $label='S'.$Apache::functionplotresponse::counter; 
      } else {
         $label='S'.$label;
      }
      if ($Apache::functionplotresponse::splineorder{$label}) {
         &Apache::lonxml::error(&mt('Spline indices must be unique.'));
      }

      my $order=&Apache::lonxml::get_param('order',$parstack,$safeeval);
      if ($order<2) { $order=2; }
      if ($order>8) { $order=8; }
      $Apache::functionplotresponse::splineorder{$label}=$order;

      my $x=&Apache::lonxml::get_param('initx',$parstack,$safeeval);
      unless ($x) { $x=0; }
      $Apache::functionplotresponse::splineinitx{$label}=$x;

      my $y=&Apache::lonxml::get_param('inity',$parstack,$safeeval);
      unless ($y) { $y=0; }
      $Apache::functionplotresponse::splineinity{$label}=$y;

      my $sx=&Apache::lonxml::get_param('scalex',$parstack,$safeeval);
      unless ($sx) { $sx=$order; }
      $Apache::functionplotresponse::splinescalex{$label}=$sx;

      my $sy=&Apache::lonxml::get_param('scaley',$parstack,$safeeval);
      unless ($sy) { $sy=2; }
      $Apache::functionplotresponse::splinescaley{$label}=$sy;
   } elsif ($target eq 'edit') {
        $result=&Apache::edit::tag_start($target,$token,'Spline').
             &Apache::edit::text_arg('Index:','index',
                                     $token,'4').'&nbsp;'.
             &Apache::edit::select_arg('Order:','order',
                                  ['2','3','4','5','6','7','8'],$token).'&nbsp;'.
             &Apache::edit::text_arg('Initial x-value:','initx',
                                     $token,'4').'&nbsp;'.
             &Apache::edit::text_arg('Initial y-value:','inity',
                                     $token,'4').'&nbsp;'.
             &Apache::edit::text_arg('Scale x:','scalex',
                                     $token,'4').'&nbsp;'.
             &Apache::edit::text_arg('Scale y:','scaley',
                                     $token,'4').
             &Apache::edit::end_row();
  } elsif ($target eq 'modified') {
    my $constructtag=&Apache::edit::get_new_args($token,$parstack,
                                                 $safeeval,'index','order','initx','inity',
                                                           'scalex','scaley');
    if ($constructtag) { $result=&Apache::edit::rebuild_tag($token); }
  }
  return $result;
}

sub end_spline {
   my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
   my $result='';
   if ($target eq 'edit') {
       $result=&Apache::edit::end_table();
   }
   return $result;
}
 
sub end_init_script {
    return (<<ENDENDINIT);
}
// ]]>
</script>
ENDENDINIT
}

#
# Storing and restoring spline coordinates from part answers
#
sub decode_previous_answer {
   my ($answer)=@_;
   foreach my $coordinate (split(/\,/,$answer)) {
      my ($key,$value)=split(/\=/,$coordinate);
      $Apache::functionplotresponse::previous{$key}=$value;
   }
}

sub get_answer_from_form_fields {
   my ($id)=@_;
   my $answer='';
   my %coords=();
   foreach my $field (keys(%env)) {
      if ($field=~/^form\.HWVAL\_$id/) {
         $field=~/^form\.(.*)$/;
         $coords{$1}=$env{$field};
      }
   }
   $answer=join(',',map { $_.'='.$coords{$_} } (sort(keys(%coords))));
   return ($answer,%coords);
}

#
# The following functions calculate the cubic-hermite splines server-side
#

sub cubic_hermite {
   my ($t,$p1,$s1,$p2,$s2)=@_;
   return (2.*$t*$t*$t-3.*$t*$t+1.)*$p1 + 3.*($t*$t*$t-2.*$t*$t+$t)*($s1-$p1)+
          (-2.*$t*$t*$t+3.*$t*$t)  *$p2 + 3.*($t*$t*$t-$t*$t)      *($s2-$p2);
}

#
# d/dt(...)
# 

sub ddt_cubic_hermite {
   my ($t,$p1,$s1,$p2,$s2)=@_;
   return (6.*$t*$t-6.*$t) *$p1 + 3.*(3.*$t*$t-4.*$t+1.)*($s1-$p1)+
          (-6.*$t*$t+6.*$t)*$p2 + 3.*(3.*$t*$t-2.*$t)   *($s2-$p2);
}

#
# d^2/dt^2(...)
#

sub d2dt2_cubic_hermite {
   my ($t,$p1,$s1,$p2,$s2)=@_;
   return (12.*$t-6.) *$p1 + 3.*(6.*$t-4.)*($s1-$p1)+
          (-12.*$t+6.)*$p2 + 3.*(6.*$t-2.)*($s2-$p2);
}

#
# Array index calculation
#
sub array_index {
   my ($xmin,$xmax,$x)=@_;
   if ($x ne '') {
      return int(($x-$xmin)/($xmax-$xmin)*400.+0.5);
   } else {
      return undef;
   }
}

#
# Populate the arrays
#

sub populate_arrays {
    my ($id,$xmin,$xmax)=@_;
    for (my $i=0; $i<=400; $i++) {
       $Apache::functionplotresponse::actualxval[$i]=undef;
       $Apache::functionplotresponse::func[$i]=undef;
       $Apache::functionplotresponse::dfuncdx[$i]=undef;
       $Apache::functionplotresponse::d2funcd2x[$i]=undef;
    }
    unless ($xmax>$xmin) { return 'no_func'; }
# Run over all splines in response
    foreach my $label (split(/\,/,$env{"form.HWVAL_AllSplines_$id"})) {
        my $xiold=-1;
# Run over all points in spline
        for (my $i=1; $i<$env{"form.HWVAL_SplineOrder_".$id."_".$label}; $i++) {
            my $ni=$i+1;
            my @xparms=($env{'form.HWVAL_'.$id.'_'.$label.'P'.$i.'_x'},
                        $env{'form.HWVAL_'.$id.'_'.$label.'S'.$i.'_x'},
                        $env{'form.HWVAL_'.$id.'_'.$label.'P'.$ni.'_x'},
                        $env{'form.HWVAL_'.$id.'_'.$label.'S'.$ni.'_x'});
            my @yparms=($env{'form.HWVAL_'.$id.'_'.$label.'P'.$i.'_y'},
                        $env{'form.HWVAL_'.$id.'_'.$label.'S'.$i.'_y'},
                        $env{'form.HWVAL_'.$id.'_'.$label.'P'.$ni.'_y'},
                        $env{'form.HWVAL_'.$id.'_'.$label.'S'.$ni.'_y'});
# Run in small steps over spline parameter
            for (my $t=0; $t<=1; $t+=0.0001) {
                my $xreal=&cubic_hermite($t,@xparms);
                my $xi=&array_index($xmin,$xmax,$xreal);
                if ($xi<$xiold) { return 'no_func'; }
                if (($xi>$xiold) && ($xi>=0) && ($xi<=400)) {
                   if (defined($Apache::functionplotresponse::func[$xi])) { return 'no_func'; }
                   $xiold=$xi;
                   $Apache::functionplotresponse::actualxval[$xi]=$xreal;
# Function value
                   my $funcval=&cubic_hermite($t,@yparms);
                   $Apache::functionplotresponse::func[$xi]=$funcval;
                   if (defined($funcval)) {
                      if ($xi<$Apache::functionplotresponse::functionplotrulelabels{'start'}) {
                         $Apache::functionplotresponse::functionplotrulelabels{'start'}=$xi;
                      }
                      if ($xi>$Apache::functionplotresponse::functionplotrulelabels{'end'}) {
                         $Apache::functionplotresponse::functionplotrulelabels{'end'}=$xi;
                      }
                   }
# Chain rule
# dy/dx=dy/dt/(dx/dt)
                   my $dxdt=&ddt_cubic_hermite($t,@xparms);
                   if ($dxdt) {
                      $Apache::functionplotresponse::dfuncdx[$xi]=&ddt_cubic_hermite($t,@yparms)/$dxdt;
# Second derivative
                      $Apache::functionplotresponse::d2funcdx2[$xi]=
                         ($dxdt*&d2dt2_cubic_hermite($t,@yparms)-&ddt_cubic_hermite($t,@yparms)*&d2dt2_cubic_hermite($t,@xparms))/
                         ($dxdt*$dxdt*$dxdt);
                   }
                }
            }
        }
    }
}

#
# Implementation of <functionplotresponse>
#

sub start_functionplotresponse {
  my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
  my $result='';
# To remember the splines - somehow, they need to come last
  undef %Apache::functionplotresponse::splineorder;
  undef %Apache::functionplotresponse::splineinitx;
  undef %Apache::functionplotresponse::splineinity;
  undef %Apache::functionplotresponse::splinescalex;
  undef %Apache::functionplotresponse::splinescaley;
# Remember input fields, etc
  undef %Apache::functionplotresponse::previous;
  $Apache::functionplotresponse::inputfields='';
  $Apache::functionplotresponse::counter=0;
# Remember rules
  undef @Apache::functionplotresponse::functionplotrules;
# Remember failed rules
  if ($target eq 'grade') {
     undef @Apache::functionplotresponse::failedrules;
  }
# Delete previous awards
  undef $Apache::functionplotresponse::awarddetail;
# Part and ID
  my $partid=$Apache::inputtags::part;
  my $id=&Apache::response::start_response($parstack,$safeeval);
# Internal ID to mark the applet and its coordinates
  my $internalid = $partid.'_'.$id;
# Previous answer
  &decode_previous_answer($Apache::lonhomework::history{"resource.$partid.$id.submission"});

# Parameters of <functionplotresponse>
  my ($xmin,$xmax,$ymin,$ymax)=&boundaries($parstack,$safeeval);
  my $xaxisvisible=(&Apache::lonxml::get_param('xaxisvisible',$parstack,$safeeval)=~/on|true|yes|1/i?'true':'false');
  my $yaxisvisible=(&Apache::lonxml::get_param('yaxisvisible',$parstack,$safeeval)=~/on|true|yes|1/i?'true':'false');
  my $gridvisible=(&Apache::lonxml::get_param('gridvisible',$parstack,$safeeval)=~/on|true|yes|1/i?'true':'false');
  my $xlabel=&Apache::lonxml::get_param('xlabel',$parstack,$safeeval);
  my $ylabel=&Apache::lonxml::get_param('ylabel',$parstack,$safeeval);
  if ($target eq 'edit') {
    $result.=&Apache::edit::start_table($token)
       .'<tr><td><span class="LC_nobreak">'.&mt('Function Plot Question').'</span></td>'
       .'<td><span class="LC_nobreak">'.&mt('Delete?').' '
       .&Apache::edit::deletelist($target,$token).'&nbsp;&nbsp;&nbsp;'
       .&Apache::edit::insertlist($target,$token)
       .'</span></td>'
       ."<td>&nbsp;"
       .&Apache::edit::end_row()
       .&Apache::edit::start_spanning_row()
       ."\n";
    $result.=&Apache::edit::text_arg('Label x-axis:','xlabel',
                                     $token,'6').'&nbsp;'.
             &Apache::edit::text_arg('Minimum x-value:','xmin',
                                     $token,'4').'&nbsp;'.
             &Apache::edit::text_arg('Maximum x-value:','xmax',
                                     $token,'4').'&nbsp;'.
             &Apache::edit::select_arg('x-axis visible:','xaxisvisible',
                                  ['yes','no'],$token).'<br />'.
             &Apache::edit::text_arg('Label y-axis:','ylabel',
                                     $token,'6').'&nbsp;'.
             &Apache::edit::text_arg('Minimum y-value:','ymin',
                                     $token,'4').'&nbsp;'.
             &Apache::edit::text_arg('Maximum y-value:','ymax',
                                     $token,'4').'&nbsp;'.
             &Apache::edit::select_arg('y-axis visible:','yaxisvisible',
                                  ['yes','no'],$token).'<br />'.
             &Apache::edit::select_arg('Grid visible:','gridvisible',
                                  ['yes','no'],$token).'<br />'.
             &Apache::edit::text_arg('Background plot(s) for answer (function:xmin:xmax,function:xmin:xmax,...):',
                                         'answerdisplay',$token,'50');

             &Apache::edit::end_row().&Apache::edit::start_spanning_row();
  } elsif ($target eq 'modified') {
    my $constructtag=&Apache::edit::get_new_args($token,$parstack,
                                                 $safeeval,'xlabel','xmin','xmax','ylabel','ymin','ymax',
                                                           'xaxisvisible','yaxisvisible','gridvisible','answerdisplay');
    if ($constructtag) { $result = &Apache::edit::rebuild_tag($token); }

  } elsif ($target eq 'meta') {
       $result=&Apache::response::meta_package_write('functionplotresponse');
  } elsif (($target eq 'answer') &&
            ($env{'form.answer_output_mode'} ne 'tex') &&
            ($Apache::lonhomework::viewgrades == 'F')) {
      my (undef,undef,$udom,$uname)=&Apache::lonnet::whichuser();
      my $windowopen=&Apache::lonhtmlcommon::javascript_docopen();
      my $start_page = &Apache::loncommon::start_page('Rules Log', undef,
                                               {'only_body' => 1,
                                                'bgcolor'   => '#FFFFFF',
                                                'js_ready'  => 1,});
      my $end_page = &Apache::loncommon::end_page({'js_ready' => 1,});
      $uname =~s/\W//g;
      $udom  =~s/\W//g;
      my $function_name =
                join('_','LONCAPA_scriptvars',$uname,$udom,
                     $env{'form.counter'},$Apache::lonxml::curdepth);
      my $rules_var ="<script type=\"text/javascript\">
// <![CDATA[
    function $function_name() {newWindow=open('','new_W','width=500,height=500,scrollbars=1,resizable=yes');newWindow.$windowopen;newWindow.document.writeln('$start_page<pre>".
    $Apache::functionplotresponse::ruleslog.
    "<\\/pre>$end_page');newWindow.document.close();newWindow.focus()}
// ]]>
</script><a href=\"javascript:$function_name();void(0);\">".&mt('Rules Log')."</a><br />";
            &Apache::lonxml::add_script_result($rules_var);
        }

  return $result;
}

sub compare_rel {
   my ($relationship,$value,$realval,$tol)=@_;
# is the real value undefined?
   unless (defined($realval)) {
# the real value is not defined
      if ($relationship eq 'eq') {
         if ($value eq 'undef') {
            return 1;
         } else {
            return 0;
         }
      } elsif ($relationship eq 'ne') {
         if ($value eq 'undef') {
            return 0;
         } else {
            return 1;
         }
      } else {
         return 0;
      }
   }

# is the expected value undefined?
   if ($value eq 'undef') {
# but by now we know that the real value is defined
      return 0;
   }

# both are defined.
   if ($relationship eq 'gt') {
      return ($realval>$value);
   } elsif ($relationship eq 'ge') {
      return ($realval>$value-$tol);
   } elsif ($relationship eq 'lt') {
      return ($realval<$value);
   } elsif ($relationship eq 'le') {
      return ($realval<$value+$tol);
   } elsif ($relationship eq 'ne') {
      return (abs($value-$realval)>$tol);
   } else {
      return (abs($value-$realval)<$tol);
   }
   return 0;
}

sub addlog {
   my ($text)=@_;
   $Apache::functionplotresponse::ruleslog.=$text.'<br />';
}

sub actualval {
   my ($i,$xmin,$xmax)=@_;
   return $xmin+$i/400.*($xmax-$xmin);
}
 
sub functionplotrulecheck {
   my ($rule,$xmin,$xmax,$ymin,$ymax)=@_;

   my ($label,$derivative,$xinitial,$xinitiallabel,$xfinal,$xfinallabel,$minimumlength,$maximumlength,$relationship,$value,$percent)
      =split(/\:/,$rule);
   $percent=($percent>0?$percent:5);
   &addlog("=================");
   &addlog("Rule $label for ".($derivative<0?'integral':('function itself','first derivative','second derivative')[$derivative])." $relationship $value");
   if ((defined($minimumlength)) || (defined($maximumlength))) {
      &addlog("Minimumlength $minimumlength Maximumlength $maximumlength");
   }
   my $li=0;
   my $lh=400;

# Special case: the upper boundary was not defined
# and needs to be set to the value where
# the condition is not true anymore => set flag

   my $findupper=0;
   if (($xfinal eq '')
    && (!defined($Apache::functionplotresponse::functionplotrulelabels{$xfinallabel}))
    && ($xfinallabel)) {
       $findupper=1;
   }

# if a hard value is set for the boundaries, it overrides the label
   if (($xinitial ne '') && ($xinitiallabel ne '') && ($xinitiallabel ne 'start')) {
      $li=&array_index($xmin,$xmax,$xinitial);
      $Apache::functionplotresponse::functionplotrulelabels{$xinitiallabel}=$li;
   }
   if (($xfinal ne '') && ($xfinallabel ne '') && ($xfinallabel ne 'end')) {
      $lh=&array_index($xmin,$xmax,$xfinal);
      $Apache::functionplotresponse::functionplotrulelabels{$xfinallabel}=$lh;
   }
# if the label is defined, use it
   if (defined($Apache::functionplotresponse::functionplotrulelabels{$xinitiallabel})) {
      &addlog("Using lower label $xinitiallabel");
      $li=$Apache::functionplotresponse::functionplotrulelabels{$xinitiallabel};
   } else {
      $li=&array_index($xmin,$xmax,$xinitial);
   }
   unless ($findupper) {
      if (defined($Apache::functionplotresponse::functionplotrulelabels{$xfinallabel})) {
         &addlog("Using upper label $xfinallabel");
         $lh=$Apache::functionplotresponse::functionplotrulelabels{$xfinallabel}-1;
      } else {
         $lh=&array_index($xmin,$xmax,$xfinal);
      }
   }
# Basic sanity checks
   if ($li<0) { $li=0; }
   if ($lh>400) { $lh=400; }
   if (($li>$lh) || (!defined($lh))) {
       $lh=$li;
   }

   &addlog("Boundaries: x=".&actualval($li,$xmin,$xmax)." (".$Apache::functionplotresponse::actualxval[$li]."; index $li)) to x=".
                            &actualval($lh,$xmin,$xmax)." (".$Apache::functionplotresponse::actualxval[$lh]."; index $lh))");
   if ($findupper) {
      &addlog("Looking for label $xfinallabel");
   }
   my $tol=$percent*($ymax-$ymin)/100;
   if ($xmax>$xmin) {
      if ($derivative==2) {
         $tol=4.*$tol/($xmax-$xmin);
      } elsif ($derivative==1) {
         $tol=2.*$tol/($xmax-$xmin);
      } elsif ($derivative==-1) {
         $tol=$tol*($xmax-$xmin)/2.;
      }
   }
   my $integral=0;
   my $binwidth=($xmax-$xmin)/400.;
   if (($derivative<0) && (!$findupper)) {
# definite integral, calculate over whole length
     &addlog("Calculating definite integral");
     for (my $i=$li; $i<=$lh; $i++) {
        $integral+=$Apache::functionplotresponse::func[$i]*$binwidth;
     }
     unless (&compare_rel($relationship,$value,$integral,$tol)) {
        &addlog("Actual integral ".(defined($integral)?$integral:'undef').", expected $value, tolerance $tol");
        &addlog("Rule $label failed.");
        my $hintlabel=$label;
        $hintlabel=~s/^R//;
        push(@Apache::functionplotresponse::failedrules,$hintlabel);
        &addlog("Set hint condition $hintlabel");
        return 0;
     } 
   } else {
     for (my $i=$li; $i<=$lh; $i++) {
        my $val;
        if ($derivative==2) {
           $val=$Apache::functionplotresponse::d2funcdx2[$i];
        } elsif ($derivative==1) {
           $val=$Apache::functionplotresponse::dfuncdx[$i];
        } elsif ($derivative==-1) {
           $integral+=$Apache::functionplotresponse::func[$i]*$binwidth;
           $val=$integral;      
        } else {
           $val=$Apache::functionplotresponse::func[$i];
        }
        unless (&compare_rel($relationship,$value,$val,$tol)) { 
           &addlog("Actual value ".(defined($val)?$val:'undef').", expected $value, tolerance $tol");
           &addlog("Condition not fulfilled at x=".&actualval($i,$xmin,$xmax)." (".$Apache::functionplotresponse::actualxval[$i]."; index $i)");
           if (($findupper) && ($i>$li)) {
# Check lengths
              unless (&checklength($i,$li,$minimumlength,$maximumlength,$xmin,$xmax,$label)) { return 0; }
# Successfully found a new label, set it
              $Apache::functionplotresponse::functionplotrulelabels{$xfinallabel}=$i;
              &addlog("Rule $label passed, setting label $xfinallabel");
              return 1;
           } else {
              &addlog("Rule $label failed.");
              &setfailed($label);
              return 0; 
           }
        }
     }
   }
# Corner case where this makes sense: using start or stop as defined labels
   unless (&checklength($lh,$li,$minimumlength,$maximumlength,$xmin,$xmax,$label)) { return 0; }
   &addlog("Rule $label passed.");
   return 1;
}

#
# check for minimum and maximum lengths
#

sub checklength {
    my ($i,$li,$minimumlength,$maximumlength,$xmin,$xmax,$label)=@_;
    unless (($minimumlength) || ($maximumlength)) { return 1; }
    my $length=&actualval($i,$xmin,$xmax)-&actualval($li,$xmin,$xmax);
    if ($minimumlength) {
       if ($length<$minimumlength) {
          &addlog("Rule $label failed, actual length $length, minimum length $minimumlength");
          &setfailed($label);
          return 0;
       }
    }
    if ($maximumlength) {
       if ($length>$maximumlength) {
          &addlog("Rule $label failed, actual length $length, maximum length $maximumlength");
          &setfailed($label);
          return 0;
       }
    }
    return 1;
}

sub setfailed {
   my ($label)=@_;
   my $hintlabel=$label;
   $hintlabel=~s/^R//;
   push(@Apache::functionplotresponse::failedrules,$hintlabel);
   &addlog("Set hint condition $hintlabel");
}

sub start_functionplotruleset {
   my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
   if ($target eq 'edit') {
      return &Apache::edit::start_table($token).
        '<tr><td><span class="LC_nobreak">'.&mt('Function Plot Rule Set').'</span></td>'
       .'<td><span class="LC_nobreak">'.&mt('Delete?').' '
       .&Apache::edit::deletelist($target,$token).'&nbsp;&nbsp;&nbsp;'.
        &Apache::edit::insertlist($target,$token)
       .'</span></td>'
       ."<td>&nbsp;"
       .&Apache::edit::end_row()
       .&Apache::edit::start_spanning_row()
       ."\n";
   }
}

sub end_functionplotruleset {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my $id=$Apache::inputtags::response[-1];
    my $partid=$Apache::inputtags::part;
    my $internalid = $partid.'_'.$id;

    if ($target eq 'edit' ) {
        return &Apache::edit::end_table();
    }  elsif ($target eq 'grade'
         && &Apache::response::submitted()
         && $Apache::lonhomework::type ne 'exam') {
#
# Actually grade
#
    my ($xmin,$xmax,$ymin,$ymax)=&boundaries($parstack,$safeeval,-2);

        my $ad='';
        undef  %Apache::functionplotresponse::functionplotrulelabels;
        $Apache::functionplotresponse::ruleslog='';
        $Apache::functionplotresponse::functionplotrulelabels{'start'}=400;
        $Apache::functionplotresponse::functionplotrulelabels{'end'}=0;
        if (&populate_arrays($internalid,$xmin,$xmax) eq 'no_func') {
           $ad='NOT_FUNCTION';
        } else {
           &addlog("Start of function ".&actualval($Apache::functionplotresponse::functionplotrulelabels{'start'},$xmin,$xmax)." (index ".
                                        $Apache::functionplotresponse::functionplotrulelabels{'start'}.")");
           &addlog("End of function ".&actualval($Apache::functionplotresponse::functionplotrulelabels{'end'},$xmin,$xmax)." (index ".
                                        $Apache::functionplotresponse::functionplotrulelabels{'end'}.")");

# We have a function that we can actually grade, go through the spline rules.
           foreach my $rule (@Apache::functionplotresponse::functionplotrules) {
              unless (&functionplotrulecheck($rule,$xmin,$xmax,$ymin,$ymax)) {
                 $ad='INCORRECT';
                 last;
              }
           }
# If it's not wrong, it's correct 
           unless ($ad) { $ad='EXACT_ANS' };
        }
        &addlog("Set hint conditions: ".join(",",@Apache::functionplotresponse::failedrules));
        &addlog("Assigned award detail: $ad");
# Store for later to be assigned at end_functionplotresponse
        $Apache::functionplotresponse::awarddetail=$ad;
     }
}


sub end_functionplotresponse {
  my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
  &Apache::response::end_response;

  my $result;
  my $id=$Apache::inputtags::response[-1];
  my $partid=$Apache::inputtags::part;
  my $internalid = $partid.'_'.$id;

    if ($target eq 'edit') { $result=&Apache::edit::end_table(); }
    if ($target eq 'grade'
         && &Apache::response::submitted()
         && $Apache::lonhomework::type eq 'exam') {

        &Apache::response::scored_response($partid,$id);

    } elsif ($target eq 'grade'
         && &Apache::response::submitted()
         && $Apache::lonhomework::type ne 'exam') {
        my ($response,%coords)=&get_answer_from_form_fields($internalid);
        $Apache::lonhomework::results{"resource.$partid.$id.submission"}=$response;
        my %previous=&Apache::response::check_for_previous($response,$partid,$id);
#
# Assign grade
#
        my $ad=$Apache::functionplotresponse::awarddetail;
#
# Store grading info
#
        $Apache::lonhomework::results{"resource.$partid.$id.awarddetail"}=$ad;
        &Apache::response::handle_previous(\%previous,$ad);
   } elsif ($target eq 'web') {
        undef @Apache::functionplotresponse::failedrules;
   }
   return $result;
}

sub end_functionplotelements {
  my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
  my $result='';
  my $id=$Apache::inputtags::response[-1];
  my $partid=$Apache::inputtags::part;
  my $internalid = $partid.'_'.$id;
  if ($target eq 'edit' ) {
     $result=&Apache::edit::end_table();
  } elsif ($target eq 'web') {
     my ($xmin,$xmax,$ymin,$ymax)=&boundaries($parstack,$safeeval,-2);

# Are we in show answer mode?
     my $showanswer=&Apache::response::show_answer();
     if ($showanswer) {
# Render answerdisplay
        my $answerdisplay=&Apache::lonxml::get_param('answerdisplay',$parstack,$safeeval,-2);
        if ($answerdisplay=~/\S/s) {
           foreach my $plot (split(/\s*\,\s*/,$answerdisplay)) {
              my ($func,$xl,$xh)=split(/\s*\:\s*/,$plot);
              if ((!defined($xl)) || ($xl eq '')) { $xl=$xmin; }
              if ((!defined($xh)) || ($xh eq '')) { $xh=$xmax; }
              $result.=&plot_script($internalid,$func,1,'','00aa00',$xl,$xh,6);
           }
        }
     }
     my $fixed=0;
     if (($showanswer) || (&Apache::response::check_status()>=2)) { $fixed=1; }
# Now is the time to render all of the stored splines
     foreach my $label (keys(%Apache::functionplotresponse::splineorder)) {
        $result.=&generate_spline($internalid,$label,$xmin,$xmax,$ymin,$ymax,$fixed);
     }
# close the init script
     $result.=&end_init_script();
# register all splines in this response 
     $result.='<input type="hidden" name="HWVAL_AllSplines_'.$internalid.'" value="'.
                 join(',',keys(%Apache::functionplotresponse::splineorder)).'" />'."\n";
     foreach my $label (keys(%Apache::functionplotresponse::splineorder)) {
        $result.='<input type="hidden" name="HWVAL_SplineOrder_'.$internalid.'_'.$label.'" value="'.
                 $Apache::functionplotresponse::splineorder{$label}.'" />'."\n";
     }
# generate the input fields
     $result.=$Apache::functionplotresponse::inputfields;
# actually start the <applet>-tag
     $result.=&geogebra_startcode($internalid);
# load the spline bytecode
     $result.=&geogebra_spline_program();
# set default parameters
     $result.=&geogebra_default_parameters($internalid);
# close the <applet>-tag
     $result.=&geogebra_endcode();
  }
  return $result;
}

sub boundaries {
   my ($parstack,$safeeval,$level)=@_;
   my $xmin=&Apache::lonxml::get_param('xmin',$parstack,$safeeval,$level);
   $xmin=(defined($xmin)?$xmin:-10);
   my $xmax=&Apache::lonxml::get_param('xmax',$parstack,$safeeval,$level);
   $xmax=(defined($xmax)?$xmax:10);
   my $ymin=&Apache::lonxml::get_param('ymin',$parstack,$safeeval,$level);
   $ymin=(defined($ymin)?$ymin:-10);
   my $ymax=&Apache::lonxml::get_param('ymax',$parstack,$safeeval,$level);
   $ymax=(defined($ymax)?$ymax:10);
   if ($xmax<=$xmin) {
      $xmax=$xmin+20;
   }
   if ($ymax<=$ymin) {
      $ymax=$ymin+20;
   }
   return ($xmin,$xmax,$ymin,$ymax);
}

sub start_functionplotelements {
   my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
   my $result='';
   my $id=$Apache::inputtags::response[-1];
   my $partid=$Apache::inputtags::part;
   my $internalid = $partid.'_'.$id;

   if ($target eq 'edit') {
      return &Apache::edit::start_table($token).
        '<tr><td><span class="LC_nobreak">'.&mt('Function Plot Elements').'</span></td>'
       .'<td><span class="LC_nobreak">'.&mt('Delete?').' '
       .&Apache::edit::deletelist($target,$token).'&nbsp;&nbsp;&nbsp;'.
        &Apache::edit::insertlist($target,$token)
       .'</span></td>'
       ."<td>&nbsp;"
       .&Apache::edit::end_row()
       .&Apache::edit::start_spanning_row()
       ."\n";
   } elsif ($target eq 'web') {
      my ($xmin,$xmax,$ymin,$ymax)=&boundaries($parstack,$safeeval,-2);
      my $xaxisvisible=(&Apache::lonxml::get_param('xaxisvisible',$parstack,$safeeval,-2)=~/on|true|yes|1/i?'true':'false');
      my $yaxisvisible=(&Apache::lonxml::get_param('yaxisvisible',$parstack,$safeeval,-2)=~/on|true|yes|1/i?'true':'false');
      my $gridvisible=(&Apache::lonxml::get_param('gridvisible',$parstack,$safeeval,-2)=~/on|true|yes|1/i?'true':'false');
      my $xlabel=&Apache::lonxml::get_param('xlabel',$parstack,$safeeval,-2);
      my $ylabel=&Apache::lonxml::get_param('ylabel',$parstack,$safeeval,-2);


# paste in the update routine to receive stuff back from the applet
     $result.=&update_script($internalid);
# start the initscript for this applet
     $result.=&start_init_script($internalid);
# put the axis commands inside
     $result.=&axes_script($internalid,$xmin,$xmax,$ymin,$ymax,$xaxisvisible,$yaxisvisible,$gridvisible);
     $result.=&axes_label($internalid,$xmin,$xmax,$ymin,$ymax,$xlabel,$ylabel);
# init script is left open
  }
  return $result;
}

1;

__END__
 
=head1 NAME

Apache::functionplotresponse.pm;

=head1 SYNOPSIS

Handles tags associated with accepting function plots.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 HANDLER SUBROUTINE

start_functionplotresponse()

=head1 OTHER SUBROUTINES

=over

=item end_functionplotresponse()

=back

=cut
