# The LearningOnline Network with CAPA
# Handler to resolve ambiguous file locations
#
# $Id: lonambiguous.pm,v 1.23 2008/11/20 15:19:05 jms Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#



package Apache::lonambiguous;

use strict;
use Apache::lonnet;
use Apache::Constants qw(:common REDIRECT);
use GDBM_File;
use Apache::loncommon;
use Apache::lonlocal;

my %bighash;

sub cleanup {
    if (tied(%bighash)){
	&Apache::lonnet::logthis('Cleanup ambiguous: bighash');
        unless (untie(%bighash)) {
	    &Apache::lonnet::logthis('Failed cleanup ambiguous: bighash');
        }
    }
    return OK;
}

# ----------------------------------------------------------- Could not resolve

sub getlost {
    my ($r,$errmsg)=@_;
    $errmsg=&mt($errmsg);
    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;
    $r->print(&Apache::loncommon::start_page('Could not handle ambiguous resource reference').
	      $errmsg.
	      &Apache::loncommon::end_page());
}

# ================================================================ Main Handler

sub make_symb {
    my ($id)=@_;
    my ($mapid,$resid)=split(/\./,$id);
    my $map=$bighash{'map_id_'.$mapid};
    my $res=$bighash{'src_'.$id};
    my $symb=&Apache::lonnet::encode_symb($map,$resid,$res);
    return $symb;
}

sub handler {
   my $r=shift;

   if ($r->header_only) {
      &Apache::loncommon::content_type($r,'text/html');
      $r->send_http_header;
      return OK;
   }

# ---------------------------------------------------------- Is this selecting?
 
   if ($env{'form.selecturl'}) {
       my $envkey;
       if (tie(%bighash,'GDBM_File',$env{'request.course.fn'}.'.db',
                    &GDBM_READER(),0640)) {
          foreach $envkey (keys %env) {
             if ($envkey=~/^form\.(\d+)\.(\d+)$/) {
# ---------------------------------------------------- Update symb and redirect
	         my $mapid=$1;
                 my $resid=$2;
                 my $resurl=$bighash{'src_'.$mapid.'.'.$resid};
                 &Apache::lonnet::symblist($bighash{'map_id_'.$mapid},
				           $resurl => [$resurl,$resid]);
                 untie(%bighash);
                 $r->header_out(Location => 
				&Apache::lonnet::absolute_url().$resurl);
                 return REDIRECT;
             }
	  }
          untie(%bighash);
       } else {
          &getlost($r,'Could not access course structure.');
          return OK;
       }
   }

# ---------------------------------------------------------- Do we have a case?

   my $thisfn;
   unless (($thisfn=$env{'request.ambiguous'})&&($env{'request.course.fn'})) {
       &getlost($r,'Could not find information on resource.');
       return OK;
   }
      
# ---------------------------------- Should this file have been part of a page?

    $thisfn=&Apache::lonnet::declutter($thisfn);
    my %hash;
    my $syval='';
    
    if (tie(%hash,'GDBM_File',$env{'request.course.fn'}.'_symb.db',
                  &GDBM_READER(),0640)) {
       $syval=$hash{$thisfn};
       untie(%hash);
    }

# ---------------------------------------------------------- There was an entry

    if ($syval) {
	my ($page,undef,$res)=&Apache::lonnet::decode_symb($syval);
	if ($res eq 'page') {
# ----------------------------------- Okay, this should have appeared on a page
	   &Apache::loncommon::content_type($r,'text/html');
           $r->header_out(Location => 
			  &Apache::lonnet::absolute_url().
			  &Apache::lonnet::clutter($page));
           return REDIRECT;
       } else {
#  There is not really a problem (???), but cannot go back without endless loop
           &getlost($r,'The nature of the problem is unclear');
           return OK;
       }
    }
# ------------------------------------Encrypted requests go straight to navmaps
   if ($env{'request.enc'}) {
       &Apache::loncommon::content_type($r,'text/html');
       $r->header_out(Location => 
		      &Apache::lonnet::absolute_url().'/adm/navmaps');
       return REDIRECT;
   }
# ------------------------------------------------ Would be standalone resource

   if (tie(%bighash,'GDBM_File',$env{'request.course.fn'}.'.db',
                    &GDBM_READER(),0640)) {
# ---------------------------------------------- Get ID(s) for current resource
      my $ids=$bighash{'ids_'.&Apache::lonnet::clutter($thisfn)};
      if ($ids) {
# ------------------------------------------------------------------- Has ID(s)
         my @possibilities=split(/\,/,$ids);
         my $couldbe='';
         foreach (@possibilities) {
             if ($bighash{'encrypted_'.$_}) { next; }
	     my $symb=&make_symb($_);
             if (&Apache::lonnet::allowed('bre',$bighash{'src_'.$_},$symb)) {
	         if ($couldbe) {
		     $couldbe.=','.$_;
                 } else {
                     $couldbe=$_;
                 }
             }
	 }
         if ($couldbe) {
            @possibilities=split(/\,/,$couldbe);
	    if ($#possibilities==0) {
		my $id=$possibilities[0];
		my $resurl=$bighash{'src_'.$id};
		my $mapurl=$bighash{'map_id_'.(split(/\./,$id))[0]};
		my $symb=&make_symb($id);
		&Apache::loncommon::content_type($r,'text/html');
		$r->header_out(Location => 
			       &Apache::lonnet::absolute_url().
			       $resurl.'?symb='.$symb);
		return REDIRECT;
	    }
            if ($#possibilities>0) {
# ----------------------------------------------- Okay, really multiple choices
	       &Apache::loncommon::content_type($r,'text/html');
               $r->send_http_header;
               my $start_page=
		   &Apache::loncommon::start_page('Pick Instance of Resource');
               $r->print(<<ENDSTART);
$start_page
The resource you had been accessing appears more than once in this course,
and LON-CAPA has insufficient session information to determine which instance
of the resource you meant.
<p>
Please click on the instance of the resource you intended to access:
</p>
<table border="2">
<tr><th>Title</th><th>Part of ...</th></tr>
ENDSTART
               map {
		   my $resurl=$bighash{'src_'.$_};
                   my $mapurl=$bighash{'map_id_'.(split(/\./,$_))[0]};
		   my $symb=&make_symb($_);
		   $r->print('<tr><td><a href="'.$resurl.'?symb='.$symb.'">'.
			     &Apache::lonnet::gettitle($symb).
			     '</a></td><td>'.
			     &Apache::lonnet::gettitle($mapurl).'&nbsp;'.
			     '</td></tr>');
               } @possibilities;
               $r->print('</table>'.&Apache::loncommon::end_page());
	       untie(%bighash);
               return OK;
            }
         }
      }
      untie(%bighash);
  }

# ------------------------------------ This handler should not have been called
   &getlost($r,'Invalid call of handler');
   return OK;
}

1;
__END__



=head1 NAME

Apache::lonambiguous

=head1 SYNOPSIS

Handler to resolve ambiguous file locations.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 HANDLER SUBROUTINE

make_symb()

and

handler()

=head1 OTHER SUBROUTINES

=over

=item *

cleanup()

=item *

getlost()

=back

=cut




