#
# The LearningOnline Network with CAPA
# image click response style
#
# $Id: imageresponse.pm,v 1.100 2010/12/19 01:08:11 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
#FIXME LATER assumes multiple possible submissions but only one is possible 
#currently


=head1 NAME

Apache::imageresponse

=head1 SYNOPSIS

Coordinates the response to clicking an image.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 SUBROUTINES

=over

=item start_imageresponse()

=item end_imageresponse()

=item start_foilgroup()

=item getfoilcounts()

=item whichfoils()

=item prep_image()

=item draw_image()

=item displayfoils()

=item format_prior_response()

=item display_answers()

=item clean_up_image()

=item gradefoils()

=item stringify_submission()

=item get_submission()

=item end_foilgroup()

=item start_conceptgroup()

=item end_conceptgroup()

=item insert_foil()

=item start_foil()

=item end_foil()

=item start_text()

=item end_text()

=item start_image()

=item end_image()

=item start_rectangle()

=item grade_rectangle()

=item end_rectangle()

=item start_polygon()

=item grade_polygon()

=item end_polygon()

=back

=cut


package Apache::imageresponse;
use strict;
use Image::Magick();
use Apache::randomlylabel();
use Apache::londefdef();
use Apache::Constants qw(:common :http);
use Apache::lonlocal;
use Apache::lonnet;
use lib '/home/httpd/lib/perl/';
use LONCAPA;
 

BEGIN {
    &Apache::lonxml::register('Apache::imageresponse',('imageresponse'));
}

sub start_imageresponse {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    #when in a radiobutton response use these
    &Apache::lonxml::register('Apache::imageresponse',
			      ('foilgroup','foil','text','image','rectangle',
			       'polygon','conceptgroup'));
    push (@Apache::lonxml::namespace,'imageresponse');
    my $id = &Apache::response::start_response($parstack,$safeeval);
    undef(%Apache::response::foilnames);
    if ($target eq 'meta') {
	$result=&Apache::response::meta_package_write('imageresponse');
    } elsif ($target eq 'analyze') {
	my $part_id="$Apache::inputtags::part.$id";
        $Apache::lonhomework::analyze{"$part_id.type"} = 'imageresponse';
	push (@{ $Apache::lonhomework::analyze{"parts"} },$part_id);
	push (@{ $Apache::lonhomework::analyze{"$part_id.bubble_lines"} },
	      1);
    } elsif ( $target eq 'edit' ) {
	$result .= &Apache::edit::tag_start($target,$token).
	    &Apache::edit::text_arg('Max Number Of Shown Foils:',
				    'max',$token,'4').
	    &Apache::edit::end_row().
	    &Apache::edit::start_spanning_row();
    } elsif ( $target eq 'modified' ) {
	my $constructtag=
	    &Apache::edit::get_new_args($token,$parstack,$safeeval,'max');
	if ($constructtag) { $result = &Apache::edit::rebuild_tag($token); }
    }
    return $result;
}

sub end_imageresponse {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;

    my $part_id     = $Apache::inputtags::part;
    my $response_id = $Apache::inputtags::response[-1]; 

    pop(@Apache::lonxml::namespace);
    &Apache::lonxml::deregister('Apache::imageresponse',('foilgroup','foil','text','image','rectangle','conceptgroup'));

    my $result;
    if ($target eq 'edit') {
	$result=&Apache::edit::end_table();
    } elsif ($target eq 'tex'
	     && $Apache::lonhomework::type eq 'exam') {
	$result=&Apache::inputtags::exam_score_line($target);
    }

    undef(%Apache::response::foilnames);
    
    if ($target eq 'grade' || $target eq 'web' || $target eq 'answer' || 
	$target eq 'tex' || $target eq 'analyze') {
	&Apache::lonxml::increment_counter(&Apache::response::repetition(), 
					   "$part_id.$response_id");
	if ($target eq 'analyze') {
	    &Apache::lonhomework::set_bubble_lines();
	}
	
    }
    &Apache::response::end_response();
    return $result;
}

%Apache::response::foilgroup=();
sub start_foilgroup {
    my ($target) = @_;
    %Apache::response::foilgroup=();
    $Apache::imageresponse::conceptgroup=0;
    &Apache::response::pushrandomnumber(undef,$target);
    return '';
}

sub getfoilcounts {
    my ($parstack,$safeeval)=@_;

    my $max = &Apache::lonxml::get_param('max',$parstack,$safeeval,'-2');
    # +1 since instructors will count from 1
    my $count = $#{ $Apache::response::foilgroup{'names'} }+1;
    if (&Apache::response::showallfoils()) { $max=$count; }
    return ($count,$max);
}

sub whichfoils {
    my ($max)=@_;
    return if (!defined(@{ $Apache::response::foilgroup{'names'} }));
    my @names = @{ $Apache::response::foilgroup{'names'} };
    my @whichopt;
    while ((($#whichopt+1) < $max) && ($#names > -1)) {
	&Apache::lonxml::debug("Have $#whichopt max is $max");
	my $aopt;
	if (&Apache::response::showallfoils()) {
	    $aopt=0;
	} else {
	    $aopt=int(&Math::Random::random_uniform() * ($#names+1));
	}
	&Apache::lonxml::debug("From $#names elms, picking $aopt");
	$aopt=splice(@names,$aopt,1);
	&Apache::lonxml::debug("Picked $aopt");
	push(@whichopt,$aopt);
    }
    return @whichopt;
}

sub prep_image {
    my ($image,$mode,$name)=@_;

    my ($x,$y)= &get_submission($name);
    &Apache::lonxml::debug("for $name drawing click at $x and $y");
    &draw_image($mode,$image,$x,$y,$Apache::response::foilgroup{"$name.area"});
}

sub draw_image {
    my ($mode,$image,$x,$y,$areas) = @_;

    my $id=&Apache::loncommon::get_cgi_id();

    my (%x,$i);
    $x{"cgi.$id.BGIMG"}=&escape($image);

    #draws 2 xs on the image at the clicked location
    #one in white and then one in red on top of the one in white

    if (defined($x)    && $x =~/\S/ 
	&& defined($y) && $y =~/\S/ 
	&& ($mode eq 'submission' || !&Apache::response::show_answer())
	&& $mode ne 'answeronly') {
	my $length = 6;
	my $width = 1;
	my $extrawidth = 2;
	my $xmin=($x-$length);
	my $xmax=($x+$length); 
	my $ymin=($y-$length);
	my $ymax=($y+$length);

	$x{"cgi.$id.OBJTYPE"}.='LINE:';
	$i=$x{"cgi.$id.OBJCOUNT"}++;
	$x{"cgi.$id.OBJ$i"}=join(':',(($xmin),($ymin),($xmax),($ymax),
				      "FFFFFF",($width+$extrawidth)));
	$x{"cgi.$id.OBJTYPE"}.='LINE:';
	$i=$x{"cgi.$id.OBJCOUNT"}++;
	$x{"cgi.$id.OBJ$i"}=join(':',(($xmin),($ymax),($xmax),($ymin),
				      "FFFFFF",($width+$extrawidth)));
	$x{"cgi.$id.OBJTYPE"}.='LINE:';
	$i=$x{"cgi.$id.OBJCOUNT"}++;
	$x{"cgi.$id.OBJ$i"}=join(':',(($xmin),($ymin),($xmax),($ymax),
				      "FF0000",($width)));
	$x{"cgi.$id.OBJTYPE"}.='LINE:';
	$i=$x{"cgi.$id.OBJCOUNT"}++;
	$x{"cgi.$id.OBJ$i"}=join(':',(($xmin),($ymax),($xmax),($ymin),
				      "FF0000",($width)));
    }
    if ($mode eq 'answer' || $mode eq 'answeronly') {
	my $width = 1;
	my $extrawidth = 2;
	foreach my $area (@{ $areas }) {
	    if ($area=~/^rectangle:/) {
		$x{"cgi.$id.OBJTYPE"}.='RECTANGLE:';
		$i=$x{"cgi.$id.OBJCOUNT"}++;
		my ($x1,$y1,$x2,$y2)=
		    ($area=~m/rectangle:\(([0-9]+),([0-9]+)\)\-\(([0-9]+),([0-9]+)\)/);
		$x{"cgi.$id.OBJ$i"}=join(':',($x1,$y1,$x2,$y2,"FFFFFF",
					      ($width+$extrawidth)));
		$x{"cgi.$id.OBJTYPE"}.='RECTANGLE:';
		$i=$x{"cgi.$id.OBJCOUNT"}++;
		$x{"cgi.$id.OBJ$i"}=join(':',($x1,$y1,$x2,$y2,"00FF00",$width));
	    } elsif ($area=~/^polygon:(.*)/) {
		$x{"cgi.$id.OBJTYPE"}.='POLYGON:';
		$i=$x{"cgi.$id.OBJCOUNT"}++;
		$x{"cgi.$id.OBJ$i"}=join(':',("FFFFFF",($width+$extrawidth)));
		$x{"cgi.$id.OBJEXTRA$i"}=$1;
		$x{"cgi.$id.OBJTYPE"}.='POLYGON:';
		$i=$x{"cgi.$id.OBJCOUNT"}++;
		$x{"cgi.$id.OBJ$i"}=join(':',("00FF00",$width));
		$x{"cgi.$id.OBJEXTRA$i"}=$1;
	    }
	}
    }
    &Apache::lonnet::appenv(\%x);
    return $id;
}

sub displayfoils {
    my ($target,@whichopt) = @_;
    my $result ='';
    my $temp=1;
    my @images;
    foreach my $name (@whichopt) {
	$result.=$Apache::response::foilgroup{"$name.text"};
	&Apache::lonxml::debug("Text is $result");
	if ($target eq 'tex') {$result.="\\vskip 0 mm \n";} else {$result.="<br />\n";}
	my $image=$Apache::response::foilgroup{"$name.image"};
	&Apache::lonxml::debug("image is $image");
	if ( ($target eq 'web' || $target eq 'answer') 
	     && $image !~ /^https?\:/ ) {
	    $image=&clean_up_image($image);
	}
	push(@images,$image);
	&Apache::lonxml::debug("image is $image");
	if ( &Apache::response::show_answer() ) {
	    if ($target eq 'tex') {
		$result.=$Apache::response::foilgroup{"$name.image"}."\\vskip 0 mm \n";
	    } else {
		my $token=&prep_image($image,'answer',$name);
		$result.="<img src=\"/adm/randomlabel.png?token=$token\" /><br />\n";
	    }
	} else {
	    if ($target eq 'tex') {
		$result.=$Apache::response::foilgroup{"$name.image"}."\\vskip 0 mm \n";
	    } else {
		my $respid=$Apache::inputtags::response['-1'];
		my $token=&prep_image($image,'submission',$name);
		my $input_id = "HWVAL_$respid:$temp";
		my $id = $env{'form.request.prefix'}.$input_id;
		$result.='<img onclick="image_response_click(\''.$id.'\',event);"
	                       src="/adm/randomlabel.png?token='.$token.'" 
                               id="'.$id.'_imageresponse"
                                />'.
			       '<br />'.
			       '<input type="hidden" name="'.$input_id.'_token" value="'.$token.'" />'.
			       '<input type="hidden" name="'.$input_id.'" value="'.
			       join(':',&get_submission($name)).'" />';
	    }
	}
	$temp++;
    }
    if ($target eq 'web') {
        &get_prior_options(\@images,\@whichopt);
    }
    return $result;
}

sub get_prior_options {
    my ($currimages,$curropt) = @_;
    return unless((ref($curropt) eq 'ARRAY') && 
                  (ref($currimages) eq 'ARRAY'));
    my $part = $Apache::inputtags::part;
    my $respid   = $Apache::inputtags::response[-1];
    foreach my $i (1..$Apache::lonhomework::history{'version'}) {
        my $partprefix = "$i:resource.$part";
        my $sub_key = "$partprefix.$respid.submission";
        next if (!exists($Apache::lonhomework::history{$sub_key}));
        my $type_key = "$partprefix.type";
        my @whichopt = ();
        my @images = ();
        if ($Apache::lonhomework::history{$type_key} eq 'randomizetry') {
            my $order_key = "$partprefix.$respid.foilorder";
            @whichopt = &Apache::lonnet::str2array($Apache::lonhomework::history{$order_key});
            if (@whichopt > 0) {
                foreach my $name (@whichopt) {
                    my $image=$Apache::response::foilgroup{"$name.image"};
                    if ($image !~ /^https?\:/ ) {
                        $image=&clean_up_image($image);
                    }
                    push(@images,$image);
                }
            }
        } else {
            @whichopt = @{$curropt};
            @images = @{$currimages};
        }
        my $submission = $Apache::lonhomework::history{$sub_key};
        my $output =  &format_prior_response('grade',$submission,
                                             [\@images,\@whichopt]);
        if (defined($output)) {
            $Apache::inputtags::submission_display{$sub_key} = $output;
        }
    }
}

sub format_prior_response {
    my ($mode,$answer,$other_data) = @_;
    
    my $result;

    # make a copy of the data in the refs
    my @images = @{ $other_data->[0] };
    my @foils = @{ $other_data->[1] };
    foreach my $name (@foils) {
	my $image = pop(@images);
	my ($x,$y) = &get_submission($name,$answer);
	my $token = &draw_image('submission',$image,$x,$y);
	$result .=
	    '<img class="LC_prior_image"
                  src="/adm/randomlabel.png?token='.$token.'" /><br />';
    }
    return $result;
}

sub display_answers {
    my ($target,$whichopt)=@_;

    my $result=&Apache::response::answer_header('imageresponse');
    foreach my $name (@$whichopt) {
	my $image=$Apache::response::foilgroup{"$name.image"};
	&Apache::lonxml::debug("image is $image");
	if ( ($target eq 'web' || $target eq 'answer')
	     && $image !~ /^https?\:/ ) {
	    $image = &clean_up_image($image);
	} 
	my $token=&prep_image($image,'answeronly',$name);

	$result.=&Apache::response::answer_part('imageresponse',"<img src=\"/adm/randomlabel.png?token=$token\" /><br />\n");
    }
    $result.=&Apache::response::answer_footer('imageresponse');
    return $result;
}

sub clean_up_image {
    my ($image)=@_;
    if ($image =~ /\s*<img\s*/) {
	($image) = ($image =~ /src\s*=\s*[\"\']([^\"\']+)[\"\']/i);
	if ($image !~ /^https?\:/) {
	    $image=&Apache::lonnet::hreflocation('',$image);
	}
	if (!$image) {
	    $image='/home/httpd/html/adm/lonKaputt/lonlogo_broken.gif';
	}
    } else {
	$image=&Apache::lonnet::filelocation($Apache::lonxml::pwd[-1],$image);
	&Apache::lonxml::debug("repcopying: $image");
	if (&Apache::lonnet::repcopy($image) ne 'ok') {
	    $image='/home/httpd/html/adm/lonKaputt/lonlogo_broken.gif';
	}
    }
    return $image;
}

sub gradefoils {
    my (@whichopt) = @_;

    my $partid = $Apache::inputtags::part;
    my $id     = $Apache::inputtags::response['-1'];

    if ($Apache::lonhomework::type eq 'exam') {
	&Apache::response::scored_response($partid,$id);
	return;
    }
    
    my @results;
    my $temp=1;
    my %response;
    foreach my $name (@whichopt) {
	my ($x,$y) = split(':',$env{"form.HWVAL_$id:$temp"});
	$response{$name} = $env{"form.HWVAL_$id:$temp"};
	&Apache::lonxml::debug("Got a x of $x and a y of $y for $name");
	if (defined($x) && defined($y) &&
	    defined(@{ $Apache::response::foilgroup{"$name.area"} })) {
	    my @areas = @{ $Apache::response::foilgroup{"$name.area"} };
	    my $grade="INCORRECT";
	    foreach my $area (@areas) {
		&Apache::lonxml::debug("Area is $area for $name");
		$area =~ m/([a-z]*):/;
		&Apache::lonxml::debug("Area of type $1");
		if ($1 eq 'rectangle') {
		    $grade=&grade_rectangle($area,$x,$y);
		} elsif ($1 eq 'polygon') {
		    $grade=&grade_polygon($area,$x,$y);
		} else {
		    &Apache::lonxml::error(&mt('Unknown area style [_1]',$area));
		}
		&Apache::lonxml::debug("Area said $grade");
		if ($grade eq 'APPROX_ANS') { last; }
	    }
	    &Apache::lonxml::debug("Foil was $grade");
	    push(@results, $grade)
	} else {
	    push(@results, 'MISSING_ANSWER')
	}
	$temp++;
    }
    if ($Apache::lonhomework::type eq 'randomizetry') {
        $Apache::lonhomework::results{"resource.$partid.$id.foilorder"} = &Apache::lonnet::array2str(@whichopt);
    }
    my ($result) = &Apache::inputtags::finalizeawards(\@results,[]);
    &Apache::lonxml::debug("Question is $result");

    my $part=$Apache::inputtags::part;
    my %previous=
	&Apache::response::check_for_previous(&stringify_submission(\%response),
					      $part,$id);
    if ($result) { 
	if ($Apache::lonhomework::type eq 'survey') { 
            $result='SUBMITTED';
        } elsif ($Apache::lonhomework::type eq 'surveycred') { 
            $result='SUBMITTED_CREDIT'; 
        } elsif ($Apache::lonhomework::type eq 'anonsurvey') { 
            $result='ANONYMOUS'; 
        } elsif ($Apache::lonhomework::type eq 'anonsurveycred') { 
            $result='ANONYMOUS_CREDIT'; 
        }
    }

    &Apache::response::handle_previous(\%previous,$result);
    $Apache::lonhomework::results{"resource.$part.$id.submission"}=
	&stringify_submission(\%response);
    $Apache::lonhomework::results{"resource.$part.$id.awarddetail"}=$result;
    return;
}

sub stringify_submission {
    my ($response) = @_;
    return &Apache::lonnet::hash2str(%{ $response });
}

sub get_submission {
    my ($name,$string) = @_;

    if (!defined($string)) {
	my $part=$Apache::inputtags::part;
	my $respid=$Apache::inputtags::response['-1'];
   	$string = 
	    $Apache::lonhomework::history{"resource.$part.$respid.submission"};
    }

    if ($string !~ /=/) {
	return split(':',$string);
    } else {
	my %response = &Apache::lonnet::str2hash($string);
	return split(':',$response{$name});
    }
}

sub end_foilgroup {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result='';
    my @whichopt;

    if ($target eq 'web' || $target eq 'grade' || $target eq 'tex' ||
	$target eq 'analyze' || $target eq 'answer') {

	my ($count,$max) = &getfoilcounts($parstack,$safeeval);
	if ($count>$max) { $count=$max }
	&Apache::lonxml::debug("Count is $count from $max");

	@whichopt = &whichfoils($max);

	if ($target eq 'web' || $target eq 'tex') {
	    $result=&displayfoils($target,@whichopt);
            if ($Apache::lonhomework::type eq 'randomizetry') {
                if ($target eq 'web') {
                    &get_prior_options($target,@whichopt);
                }
            }
	    $Apache::lonxml::post_evaluate=0;
	} elsif ($target eq 'grade') {
	    if ( defined $env{'form.submitted'}) { &gradefoils(@whichopt); }
	} elsif ( $target eq 'analyze') {
	    &Apache::response::analyze_store_foilgroup(\@whichopt,
						      ['text','image','area']);
	} elsif ($target eq 'answer'
		 && $env{'form.answer_output_mode'} ne 'tex') {
	    $result=&display_answers($target,\@whichopt);
	}

    } elsif ($target eq 'edit') {
	$result=&Apache::edit::end_table();
    }
    &Apache::response::poprandomnumber();
    return $result;
}

sub start_conceptgroup {
    $Apache::imageresponse::conceptgroup=1;
    %Apache::response::conceptgroup=();
    return '';
}

sub end_conceptgroup {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    $Apache::imageresponse::conceptgroup=0;
    my $result;
    if ($target eq 'web' || $target eq 'grade' || $target eq 'tex' ||
	$target eq 'analyze') {
	&Apache::response::pick_foil_for_concept($target,
						 ['area','text','image'],
						 \%Apache::hint::image,
						 $parstack,$safeeval);
    } elsif ($target eq 'edit') {
	$result=&Apache::edit::end_table();
    }
    return $result;
}

sub insert_foil {
    return '
       <foil>
           <image></image>
           <text></text>
           <rectangle></rectangle>
       </foil>
';
}

$Apache::imageresponse::curname='';
sub start_foil {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    if ($target eq 'web' || $target eq 'grade' || $target eq 'tex' ||
	$target eq 'analyze' || $target eq 'answer') {
	my $name = &Apache::lonxml::get_param('name',$parstack,$safeeval);
	if ($name eq "") {
	    &Apache::lonxml::warning(&mt('Foils without names exist. This can cause problems to malfunction.'));
	    $name=$Apache::lonxml::curdepth;
	}
	if (defined($Apache::response::foilnames{$name})) {
	    &Apache::lonxml::error(&mt("Foil name [_1] appears more than once. Foil names need to be unique."
                                      ,'<b><tt>'.$name.'</tt></b>'));
	}
	$Apache::response::foilnames{$name}++;
	if ( $Apache::imageresponse::conceptgroup
	     && !&Apache::response::showallfoils()
	     ) {
	    push(@{ $Apache::response::conceptgroup{'names'} }, $name);
	} else {
	    push(@{ $Apache::response::foilgroup{'names'} }, $name);
	}
	$Apache::imageresponse::curname=$name;
    } elsif ($target eq 'edit') {
	$result  = &Apache::edit::tag_start($target,$token);
	$result .= &Apache::edit::text_arg('Name:','name',$token);
	$result .= &Apache::edit::end_row().
	    &Apache::edit::start_spanning_row();
    } elsif ($target eq 'modified') {
	my $constructtag=&Apache::edit::get_new_args($token,$parstack,
						     $safeeval,'name');
	if ($constructtag) { $result = &Apache::edit::rebuild_tag($token); }
    } 
    return $result;;
}

sub end_foil {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    if ($target eq 'edit') {
	$result=&Apache::edit::end_table();
    }
    return $result;
}

sub start_text {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result='';
    if ($target eq 'web' || $target eq 'tex' || $target eq 'analyze'
	|| $target eq 'answer') { 
	&Apache::lonxml::startredirection; 
    } elsif ($target eq 'edit') {
	my $descr=&Apache::lonxml::get_all_text('/text',$parser,$style);
	$result=&Apache::edit::tag_start($target,$token,'Task Description').
	    &Apache::edit::editfield($token->[1],$descr,'Text',60,2).
	    &Apache::edit::end_row();
    } elsif ($target eq "modified") {
	$result=$token->[4].&Apache::edit::modifiedfield('/text',$parser);
    }
    return $result;
}

sub end_text {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    if ($target eq 'web' || $target eq 'tex' || $target eq 'analyze'
	|| $target eq 'answer') {
	my $name = $Apache::imageresponse::curname;
	if ( $Apache::imageresponse::conceptgroup
	     && !&Apache::response::showallfoils()
	     ) {
	    $Apache::response::conceptgroup{"$name.text"} = &Apache::lonxml::endredirection;
	} else {
	    $Apache::response::foilgroup{"$name.text"} = &Apache::lonxml::endredirection;
	}
    } elsif ($target eq 'edit') {
	$result=&Apache::edit::end_table();
    }
    return $result;
}

sub start_image {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result='';
    my $only = join(',',&Apache::loncommon::filecategorytypes('Pictures'));
    if ($target eq 'web' || $target eq 'tex' || $target eq 'analyze'
	|| $target eq 'answer') { 
	&Apache::lonxml::startredirection; 
    } elsif ($target eq 'edit') {
	my $bgimg=&Apache::lonxml::get_all_text('/image',$parser,$style);
	$Apache::edit::bgimgsrc=$bgimg;
	$Apache::edit::bgimgsrcdepth=$Apache::lonxml::curdepth;

	$result=&Apache::edit::tag_start($target,$token,'Clickable Image').
	    &Apache::edit::editline($token->[1],$bgimg,'Image Source File',40);
	$result.=&Apache::edit::browse(undef,'textnode',undef,$only).' ';
	$result.=&Apache::edit::search(undef,'textnode').
	    &Apache::edit::end_row();
    } elsif ($target eq "modified") {
	$result=$token->[4].&Apache::edit::modifiedfield('/image',$parser);
    }
    return $result;
}

sub end_image {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    my $name = $Apache::imageresponse::curname;
    if ($target eq 'web' || $target eq 'answer') {
	my $image = &Apache::lonxml::endredirection();
	&Apache::lonxml::debug("original image is $image");
	if ( $Apache::imageresponse::conceptgroup
	     && !&Apache::response::showallfoils()
	     ) {
	    $Apache::response::conceptgroup{"$name.image"} = $image;
	} else {
	    $Apache::response::foilgroup{"$name.image"} = $image;
	}
    } elsif ($target eq 'analyze') {
	my $image = &Apache::lonxml::endredirection();
	if ( $Apache::imageresponse::conceptgroup
	     && !&Apache::response::showallfoils()
	     ) {
	    $Apache::response::conceptgroup{"$name.image"} = $image;
	} else {
	    $Apache::response::foilgroup{"$name.image"} = $image;
	}
    } elsif ($target eq 'edit') {
	$result=&Apache::edit::end_table();
    } elsif ($target eq 'tex') {
	my $src = &Apache::lonxml::endredirection();

	#  There may be all sorts of whitespace on fore and aft:

	$src =~ s/\s+$//s;
	$src =~ s/^\s+//s;

	#
	#  Gnuplot e.g. just generates the latex to put inplace.
	#
	my $graphinclude;
	if ($src =~ /^%DYNAMICIMAGE/) {
	    # This is needed because the newline is not always passed -> tex.
	    # At present we don't care about the sizing info.

	    my ($commentline, $restofstuff) = split(/\n/, $src);
	    $graphinclude = $src;
	    $graphinclude =~ s/^$commentline//;
	} elsif (!($src =~ /\\/)) {
	    my ($path,$file) = &Apache::londefdef::get_eps_image($src);
	    my ($height_param,$width_param)=
		&Apache::londefdef::image_size($src,0.3,$parstack,$safeeval);
	    $graphinclude = '\graphicspath{{'.$path.'}}\includegraphics[width='.$width_param.' mm]{'.$file.'}';
	} else {
	    $graphinclude = $src;   # Already fully formed.
	}
	$Apache::response::foilgroup{"$name.image"} ='\vskip 0 mm \noindent '.$graphinclude;
    } 
    return $result;
}

sub start_rectangle {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result='';
    if ($target eq 'web' || $target eq 'grade' || $target eq 'tex' ||
	$target eq 'analyze' || $target eq 'answer') { 
	&Apache::lonxml::startredirection; 
    } elsif ($target eq 'edit') {
	my $coords=&Apache::lonxml::get_all_text('/rectangle',$parser,$style);
	$result=&Apache::edit::tag_start($target,$token,'Rectangle').
	    &Apache::edit::editline($token->[1],$coords,'Coordinate Pairs',40).
	    &Apache::edit::entercoord(undef,'textnode',undef,undef,'box').
	    &Apache::edit::end_row();
    } elsif ($target eq "modified") {
	&Apache::edit::deletecoorddata();
	$result=$token->[4].&Apache::edit::modifiedfield('/rectangle',$parser);
    }
    return $result;
}

sub grade_rectangle {
    my ($spec,$x,$y) = @_;
    &Apache::lonxml::debug("Spec is $spec");
    my ($x1,$y1,$x2,$y2)=($spec=~m/rectangle:\(([0-9]+),([0-9]+)\)\-\(([0-9]+),([0-9]+)\)/);
    &Apache::lonxml::debug("Point $x1,$y1,$x2,$y2");
    if ($x1 > $x2) { my $temp=$x1;$x1=$x2;$x2=$temp; }
    if ($y1 > $y2) { my $temp=$y1;$y1=$y2;$y2=$temp; }
    if (($x >= $x1) && ($x <= $x2) && ($y >= $y1) && ($y <= $y2)) {
	return 'APPROX_ANS';
    }
    return 'INCORRECT';
}

sub end_rectangle {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    if ($target eq 'web' || $target eq 'grade' || $target eq 'tex' ||
	$target eq 'analyze' || $target eq 'answer') {
	my $name = $Apache::imageresponse::curname;
	my $area = &Apache::lonxml::endredirection;
	$area=~s/\s//g;
	&Apache::lonxml::debug("out is $area for $name");
	if ( $Apache::imageresponse::conceptgroup
	     && !&Apache::response::showallfoils()
	     ) {
	    push @{ $Apache::response::conceptgroup{"$name.area"} },"rectangle:$area";
	} else {
	    push @{ $Apache::response::foilgroup{"$name.area"} },"rectangle:$area";
	}
    } elsif ($target eq 'edit') {
	$result=&Apache::edit::end_table();
    }
    return $result;
}

sub start_polygon {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result='';
    if ($target eq 'web' || $target eq 'grade' || $target eq 'tex' ||
	$target eq 'analyze' || $target eq 'answer') { 
	&Apache::lonxml::startredirection; 
    } elsif ($target eq 'edit') {
	my $coords=&Apache::lonxml::get_all_text('/polygon',$parser,$style);
	$result=&Apache::edit::tag_start($target,$token,'Polygon').
	    &Apache::edit::editline($token->[1],$coords,'Coordinate list',40).
	    &Apache::edit::entercoord(undef,'textnode',undef,undef,'polygon').
	    &Apache::edit::end_row();
    } elsif ($target eq "modified") {
	$result=$token->[4].&Apache::edit::modifiedfield('/polygon',$parser);
    }
    return $result;
}

sub grade_polygon {
    my ($spec,$x,$y) = @_;
    &Apache::lonxml::debug("Spec is $spec");
    $spec=~s/^polygon://;
    my @polygon;
    foreach my $coord (split('-',$spec)) {
	my ($x,$y)=($coord=~m/\(([0-9]+),([0-9]+)\)/);
	&Apache::lonxml::debug("x $x y $y");
	push @polygon, {'x'=>$x,'y'=>$y};
    }
    #make end point start point
    push @polygon, $polygon[0];
    # cribbed from
    # http://geometryalgorithms.com/Archive/algorithm_0103/algorithm_0103.htm
    my $crossing = 0;    # the crossing number counter

    # loop through all edges of the polygon
    for (my $i=0; $i<$#polygon; $i++) {    # edge from V[i] to V[i+1]
	if ((($polygon[$i]->{'y'} <= $y)
	     && ($polygon[$i+1]->{'y'} > $y))    # an upward crossing
	    || 
	    (($polygon[$i]->{'y'} > $y) 
	     && ($polygon[$i+1]->{'y'} <= $y))) { # a downward crossing
	    # compute the actual edge-ray intersect x-coordinate
            my $vt = ($y - $polygon[$i]->{'y'}) 
		/ ($polygon[$i+1]->{'y'} - $polygon[$i]->{'y'});
            if ($x < $polygon[$i]->{'x'} + $vt * 
		($polygon[$i+1]->{'x'} - $polygon[$i]->{'x'})) { # x<intersect
                $crossing++;   # a valid crossing of y=P.y right of P.x
	    }
	}
    }

    # 0 if even (out), and 1 if odd (in)
    if ($crossing%2) {
	return 'APPROX_ANS';
    } else {
	return 'INCORRECT';
    }
}

sub end_polygon {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    if ($target eq 'web' || $target eq 'grade' || $target eq 'tex' ||
	$target eq 'analyze' || $target eq 'answer') {
	my $name = $Apache::imageresponse::curname;
	my $area = &Apache::lonxml::endredirection;
	$area=~s/\s*//g;
	&Apache::lonxml::debug("out is $area for $name");
	if ( $Apache::imageresponse::conceptgroup
	     && !&Apache::response::showallfoils()
	     ) {
	    push @{ $Apache::response::conceptgroup{"$name.area"} },"polygon:$area";
	} else {
	    push @{ $Apache::response::foilgroup{"$name.area"} },"polygon:$area";
	}
    } elsif ($target eq 'edit') {
	$result=&Apache::edit::end_table();
    }
    return $result;
}
1;
__END__
 
