# The LearningOnline Network with CAPA
# displays the blocking status table
#
# $Id: lonblockingstatus.pm,v 1.7 2010/02/19 10:20:03 bisitz Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
#
package Apache::lonblockingstatus;

use strict;
use Apache::Constants qw(:common);
use Apache::loncommon();
use Apache::lonnet;
use Apache::lonlocal;

sub handler {
    my $r = shift;
    Apache::loncommon::no_cache($r);
    Apache::loncommon::content_type($r,'text/html');

    $r->send_http_header;
    return OK if $r->header_only;

    $r->print( Apache::loncommon::start_page(
                  'Communication Blocking Status Information',
                  undef, {'only_body' => 1}));

    $r->print(blockpage());
    
    $r->print(Apache::loncommon::end_page());

    return OK;
}


sub blockpage {

    # determine what kind of blocking we want details for
    Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},['activity']);
    my $activity = $env{'form.activity'};

    if ($activity !~ /^[a-z]+$/) { return 'Error'; }


    # in case of a portfolio block we need to determine the owner of the files
    # we're trying to look at. This information is passed via query string.
    my ($uname, $udom);

    if ($activity eq 'port') {
        Apache::loncommon::get_unprocessed_cgi(
            $ENV{'QUERY_STRING'}, ['udom', 'uname'] );

        ($uname, $udom) = ($env{'form.uname'}, $env{'form.udom'});
        #TODO sanitize input: $uname, $udom
    }


    # retrieve start/end of possible active blocking
    my %setters;
    my ($startblock, $endblock) = Apache::loncommon::blockcheck(
                                        \%setters, $activity, $uname, $udom);

    # nothing to do if there's no active blocking
    unless ($startblock && $endblock) { return ''; }

    # lookup $activity -> description
                   #possible activity          #corresponding description
    my $description = 
                   $activity eq 'boards'     ? 'Discussion posts in this course'
                 : $activity eq 'chat'       ? 'Chat Room'
                 : $activity eq 'msgdisplay' ? 'This message'
                 : $activity eq 'blogs'      ? 'Blogs'
                 : $activity eq 'groups'     ? 'Groups in this course'
                 : $activity eq 'port'       ? get_portfolio_category(
                                                  $uname, $udom, \%setters) 
                 :                             'Communication';

    my $showstart = Apache::lonlocal::locallocaltime($startblock);
    my $showend   = Apache::lonlocal::locallocaltime($endblock);

    my $output;
    
    if ( ref($description) ne 'ARRAY' ) { 
        #default: $description is one of the above descriptions
        $output = mt( $description
                      . ' will be inaccessible between [_1] and [_2] because'
                      . ' communication is being blocked.'
                     ,$showstart, $showend);  
    } else {
        # @$description is is the array returned from get_portfolio_category()
        # and contains the description (e.g. "Portfolio files belonging to [_1]"
        # and the value for [_1]
        $output = mt( $$description[0]
                      . ' will be inaccessible between [_2] and [_3] because'
                      . ' communication is being blocked.'
                     ,$$description[1], $showstart, $showend)  
    }

    $output = "<p class=\"LC_info\">$output</p>";

    # show a table containing details, except if user is trying to look 
    # at a different user's portfolio files
    if (   $activity ne 'port'                        # no portfolio
        || (   $uname eq $env{'user.name'}            # or own portfolio
            && $udom  eq $env{'user.domain'} ) 
        || Apache::lonnet::is_course($udom, $uname) ) # or portfolio of a course
    {
        $output .= build_block_table($startblock,$endblock,\%setters);
    }

    return $output;
}

sub build_block_table {
    my ($startblock,$endblock,$setters) = @_;
    my %lt = &Apache::lonlocal::texthash(
        'cacb' => 'Currently active communication blocks',
        'cour' => 'Course',
        'dura' => 'Duration',
        'blse' => 'Block set by'
    );
    my $output;
    $output  = Apache::loncommon::start_data_table()
             . Apache::loncommon::data_table_caption($lt{'cacb'})
             . Apache::loncommon::start_data_table_header_row()
             . "<th>$lt{'cour'}</th> <th>$lt{'dura'}</th> <th>$lt{'blse'}</th>"
             . Apache::loncommon::end_data_table_header_row();

    foreach my $course (keys(%{$setters})) {
        my %courseinfo=&Apache::lonnet::coursedescription($course);
        for (my $i=0; $i<@{$$setters{$course}{staff}}; $i++) {
            my ($uname,$udom) = @{$$setters{$course}{staff}[$i]};
            my $fullname = Apache::loncommon::plainname($uname,$udom);
            if (defined($env{'user.name'}) && defined($env{'user.domain'})
                && $env{'user.name'} ne 'public'
                && $env{'user.domain'} ne 'public') 
            {
                $fullname = Apache::loncommon::aboutmewrapper($fullname,$uname,$udom);
            }
            my ($openblock,$closeblock) = @{$$setters{$course}{times}[$i]};
            $openblock = &Apache::lonlocal::locallocaltime($openblock);
            $closeblock= &Apache::lonlocal::locallocaltime($closeblock);
            my $duration = mt('[_1] to [_2]', $openblock, $closeblock);
            $output .= Apache::loncommon::start_data_table_row()
                     . "<td>$courseinfo{'description'}</td>"
                     . "<td>$duration</td>"
                     . "<td>$fullname</td>"
                     . Apache::loncommon::end_data_table_row();
        }
    }
    $output .= Apache::loncommon::end_data_table();
}

sub get_portfolio_category {
    my ($uname, $udom, $setters) = @_;

    if ($uname eq $env{'user.name'} && $udom eq $env{'user.domain'}) {
        # user's portolfio files

        return 'Your portfolio files';

    } elsif (Apache::lonnet::is_course($udom, $uname)) {
        # group portfolio files

        my $coursedesc;

        foreach my $course (keys(%{$setters})) {
            my %courseinfo = Apache::lonnet::coursedescription($course);
            $coursedesc    = $courseinfo{'description'};
        }

        return ["Group portfolio in the course '[_1]'", $coursedesc];
        
    } else {
        # different user's portfolio files
        
        my $plainname = Apache::loncommon::plainname($uname, $udom);

        unless (   $env{'user.name'}   eq 'public' 
                && $env{'user.domain'} eq 'public' ) 
        {
            $plainname = Apache::loncommon::aboutmewrapper(
                            $plainname, $uname, $udom);
        }

        return ['Portfolio files belonging to [_1]', $plainname];
    }
}

1;
__END__
