# The LearningOnline Network with CAPA
# Handler to set parameters for assessments
#
# $Id: lonparmset.pm,v 1.522.2.13 2014/02/26 03:34:01 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
###################################################################
###################################################################

=pod

=head1 NAME

lonparmset - Handler to set parameters for assessments and course

=head1 SYNOPSIS

lonparmset provides an interface to setting course parameters. 

=head1 DESCRIPTION

This module sets coursewide and assessment parameters.

=head1 INTERNAL SUBROUTINES

=over

=item parmval()

Figure out a cascading parameter.

Inputs:  $what - a parameter spec (incluse part info and name I.E. 0.weight)
         $id   - a bighash Id number
         $def  - the resource's default value   'stupid emacs

Returns:  A list, the first item is the index into the remaining list of items of parm valuse that is the active one, the list consists of parm values at the 14 possible levels

14- General Course
13- Map or Folder level in course
12- resource default
11- map default
10- resource level in course
9 - General for section
8 - Map or Folder level for section
7 - resource level in section
6 - General for group
5 - Map or Folder level for group
4 - resource level in group
3 - General for specific student
2 - Map or Folder level for specific student
1 - resource level for specific student

=item parmval_by_symb()

=item reset_caches()

=item cacheparmhash() 

=item parmhash()

=item symbcache()

=item preset_defaults()

=item date_sanity_info()

=item storeparm()

Store a parameter by symb

    Takes
    - symb
    - name of parameter
    - level
    - new value
    - new type
    - username
    - userdomain

=item log_parmset()

=item storeparm_by_symb_inner()

=item valout()

Format a value for output.

Inputs:  $value, $type, $editable

Returns: $value, formatted for output.  If $type indicates it is a date,
localtime($value) is returned.
$editable will return an icon to click on

=item plink()

Produces a link anchor.

Inputs: $type,$dis,$value,$marker,$return,$call

Returns: scalar with html code for a link which will envoke the 
javascript function 'pjump'.

=item page_js()

=item startpage()

=item print_row()

=item print_td()

=item print_usergroups()

=item parm_control_group()

=item extractResourceInformation() : 

 extractResourceInformation extracts lots of information about all of the the course's resources into a variety of hashes.

Input: See list below

=over 4

=item * B<env{'user.name'}> : Current username

=item * B<env{'user.domain'}> : Domain of current user.

=item * B<env{"request.course.fn"}> : Course

=back

Outputs: See list below

=over 4

=item * B<ids> (out) : An array that will contain all of the ids in the course.

=item * B<typep>(out) : hash, id->type, where "type" contains the extension of the file, thus, I<problem exam quiz assess survey form>.

=item * B<keyp> (out) : hash, id->key list, will contain a comma separated list of the meta-data keys available for the given id

=item * B<allparms> (out) : hash, name of parameter->display value (what is the display value?)

=item * B<allparts> (out) : hash, part identification->text representation of part, where the text representation is "[Part $part]"

=item * B<allmaps> (out) : hash, ???

=item * B<mapp> : ??

=item * B<symbp> : hash, id->full sym?

=item * B<maptitles>

=item * B<uris>

=item * B<keyorder>

=item * B<defkeytype>

=back

=item isdateparm()

=item parmmenu()

=item partmenu()

=item usermenu()

=item displaymenu()

=item mapmenu()

=item levelmenu()

=item sectionmenu()

=item keysplit()

=item keysinorder()

=item keysinorder_bytype()

=item keysindisplayorder()

=item standardkeyorder()

=item assessparms() : 

Show assessment data and parameters.  This is a large routine that should
be simplified and shortened... someday.

Inputs: $r - the Apache request object.
  
Returns: nothing

Variables used (guessed by Jeremy):

=over

=item * B<pscat>: ParameterS CATegories? ends up a list of the types of parameters that exist, e.g., tol, weight, acc, opendate, duedate, answerdate, sig, maxtries, type.

=item * B<psprt>: ParameterS PaRTs? a list of the parts of a problem that we are displaying? Used to display only selected parts?

=item * B<@catmarker> contains list of all possible parameters including part #s

=item * B<$fullkeyp> contains the full part/id # for the extraction of proper parameters

=item * B<$tempkeyp> contains part 0 only (no ids - ie, subparts)
        When storing information, store as part 0
        When requesting information, request from full part

=back

=item tablestart()

=item tableend()

=item extractuser()

=item parse_listdata_key()

=item listdata()

=item date_interval_selector()

=item get_date_interval_from_form()

=item default_selector()

=item string_selector()

=item dateshift()

=item newoverview()

=item secgroup_lister()

=item overview()

=item clean_parameters()

=item date_shift_one()

=item date_shift_two()

=item parse_key()

=item header()

Output html header for page

=item print_main_menu()

=item output_row()

Set portfolio metadata

=item order_meta_fields()

=item addmetafield()

=item setrestrictmeta()

=item get_added_meta_fieldnames()

=item get_deleted_meta_fieldnames()

=item defaultsetter()

=item components()

=item load_parameter_names()

=item parm_change_log()

=item handler() : 

Main handler.  Calls &assessparms subroutine.

=back

=cut

###################################################################
###################################################################

package Apache::lonparmset;

use strict;
use Apache::lonnet;
use Apache::Constants qw(:common :http REDIRECT);
use Apache::lonhtmlcommon();
use Apache::loncommon;
use GDBM_File;
use Apache::lonhomework;
use Apache::lonxml;
use Apache::lonlocal;
use Apache::lonnavmaps;
use Apache::longroup;
use Apache::lonrss;
use HTML::Entities;
use LONCAPA qw(:DEFAULT :match);


sub parmval {
    my ($what,$id,$def,$uname,$udom,$csec,$cgroup,$courseopt)=@_;
    return &parmval_by_symb($what,&symbcache($id),$def,$uname,$udom,$csec,
                                                           $cgroup,$courseopt);
}

sub parmval_by_symb {
    my ($what,$symb,$def,$uname,$udom,$csec,$cgroup,$courseopt)=@_;

    my $useropt;
    if ($uname ne '' && $udom ne '') {
    $useropt = &Apache::lonnet::get_userresdata($uname,$udom);
    }

    my $result='';
    my @outpar=();
# ----------------------------------------------------- Cascading lookup scheme
    my $map=(&Apache::lonnet::decode_symb($symb))[0];
    $map = &Apache::lonnet::deversion($map);

    my $symbparm=$symb.'.'.$what;
    my $mapparm=$map.'___(all).'.$what;

    my $grplevel=$env{'request.course.id'}.'.['.$cgroup.'].'.$what;
    my $grplevelr=$env{'request.course.id'}.'.['.$cgroup.'].'.$symbparm;
    my $grplevelm=$env{'request.course.id'}.'.['.$cgroup.'].'.$mapparm;

    my $seclevel=$env{'request.course.id'}.'.['.$csec.'].'.$what;
    my $seclevelr=$env{'request.course.id'}.'.['.$csec.'].'.$symbparm;
    my $seclevelm=$env{'request.course.id'}.'.['.$csec.'].'.$mapparm;

    my $courselevel=$env{'request.course.id'}.'.'.$what;
    my $courselevelr=$env{'request.course.id'}.'.'.$symbparm;
    my $courselevelm=$env{'request.course.id'}.'.'.$mapparm;


# --------------------------------------------------------- first, check course

    if (defined($$courseopt{$courselevel})) {
    $outpar[14]=$$courseopt{$courselevel};
    $result=14;
    }

    if (defined($$courseopt{$courselevelm})) {
    $outpar[13]=$$courseopt{$courselevelm};
    $result=13;
    }

# ------------------------------------------------------- second, check default

    if (defined($def)) { $outpar[12]=$def; $result=12; }

# ------------------------------------------------------ third, check map parms

    my $thisparm=&parmhash($symbparm);
    if (defined($thisparm)) { $outpar[11]=$thisparm; $result=11; }

    if (defined($$courseopt{$courselevelr})) {
    $outpar[10]=$$courseopt{$courselevelr};
    $result=10;
    }

# ------------------------------------------------------ fourth, back to course
    if ($csec ne '') {
        if (defined($$courseopt{$seclevel})) {
        $outpar[9]=$$courseopt{$seclevel};
        $result=9;
    }
        if (defined($$courseopt{$seclevelm})) {
        $outpar[8]=$$courseopt{$seclevelm};
        $result=8;
    }

        if (defined($$courseopt{$seclevelr})) {
        $outpar[7]=$$courseopt{$seclevelr};
        $result=7;
    }
    }
# ------------------------------------------------------ fifth, check course group
    if ($cgroup ne '') {
        if (defined($$courseopt{$grplevel})) {
            $outpar[6]=$$courseopt{$grplevel};
            $result=6;
        }
        if (defined($$courseopt{$grplevelm})) {
            $outpar[5]=$$courseopt{$grplevelm};
            $result=5;
        }
        if (defined($$courseopt{$grplevelr})) {
            $outpar[4]=$$courseopt{$grplevelr};
            $result=4;
        }
    }

# ---------------------------------------------------------- fifth, check user

    if ($uname ne '') {
    if (defined($$useropt{$courselevel})) {
        $outpar[3]=$$useropt{$courselevel};
        $result=3;
    }

    if (defined($$useropt{$courselevelm})) {
        $outpar[2]=$$useropt{$courselevelm};
        $result=2;
    }

    if (defined($$useropt{$courselevelr})) {
        $outpar[1]=$$useropt{$courselevelr};
        $result=1;
    }
    }
    return ($result,@outpar);
}



# --- Caches local to lonparmset


sub reset_caches {
    &resetparmhash();
    &resetsymbcache();
    &resetrulescache();
}

{
    my $parmhashid;
    my %parmhash;
    sub resetparmhash {
    undef($parmhashid);
    undef(%parmhash);
    }

    sub cacheparmhash {
    if ($parmhashid eq  $env{'request.course.fn'}) { return; }
    my %parmhashfile;
    if (tie(%parmhashfile,'GDBM_File',
        $env{'request.course.fn'}.'_parms.db',&GDBM_READER(),0640)) {
        %parmhash=%parmhashfile;
        untie(%parmhashfile);
        $parmhashid=$env{'request.course.fn'};
    }
    }

    sub parmhash {
    my ($id) = @_;
    &cacheparmhash();
    return $parmhash{$id};
    }
 }

{
    my $symbsid;
    my %symbs;
    sub resetsymbcache {
    undef($symbsid);
    undef(%symbs);
    }

    sub symbcache {
    my $id=shift;
    if ($symbsid ne $env{'request.course.id'}) {
        undef(%symbs);
    }
    if (!$symbs{$id}) {
        my $navmap = Apache::lonnavmaps::navmap->new();
        if ($id=~/\./) {
        my $resource=$navmap->getById($id);
        $symbs{$id}=$resource->symb();
        } else {
        my $resource=$navmap->getByMapPc($id);
        $symbs{$id}=&Apache::lonnet::declutter($resource->src());
        }
        $symbsid=$env{'request.course.id'};
    }
    return $symbs{$id};
    }
 }

{
    my $rulesid;
    my %rules;
    sub resetrulescache {
    undef($rulesid);
    undef(%rules);
    }

    sub rulescache {
    my $id=shift;
    if ($rulesid ne $env{'request.course.id'}
        && !defined($rules{$id})) {
        my $dom = $env{'course.'.$env{'request.course.id'}.'.domain'};
        my $crs = $env{'course.'.$env{'request.course.id'}.'.num'};
        %rules=&Apache::lonnet::dump('parmdefactions',$dom,$crs);
        $rulesid=$env{'request.course.id'};
    }
    return $rules{$id};
    }
}



sub preset_defaults {
    my $type=shift;
    if (&rulescache($type.'_action') eq 'default') {
# yes, there is something
    return (&rulescache($type.'_hours'),
        &rulescache($type.'_min'),
        &rulescache($type.'_sec'),
        &rulescache($type.'_value'));
    } else {
# nothing there or something else
    return ('','','','','');
    }
}




sub date_sanity_info {
   my $checkdate=shift;
   unless ($checkdate) { return ''; }
   my $result='';
   my $crsprefix='course.'.$env{'request.course.id'}.'.';
   if ($env{$crsprefix.'default_enrollment_end_date'}) {
      if ($checkdate>$env{$crsprefix.'default_enrollment_end_date'}) {
         $result.='<div class="LC_warning">'
                 .&mt('After course enrollment end!')
                 .'</div>';
      }
   }
   if ($env{$crsprefix.'default_enrollment_start_date'}) {
      if ($checkdate<$env{$crsprefix.'default_enrollment_start_date'}) {
         $result.='<div class="LC_warning">'
                 .&mt('Before course enrollment start!')
                 .'</div>';
      }
   }
# Preparation for additional warnings about dates in the past/future.
# An improved, more context sensitive version is recommended,
# e.g. warn for due and answer dates which are defined before the corresponding open date, etc.
#   if ($checkdate<time) {
#      $result.='<div class="LC_info">'
#              .'('.&mt('in the past').')'
#              .'</div>';
#      }
#   if ($checkdate>time) {
#      $result.='<div class="LC_info">'
#              .'('.&mt('in the future').')'
#              .'</div>';
#      }
   return $result;
}
##################################################
##################################################
#
# Store a parameter by ID
#
# Takes
# - resource id
# - name of parameter
# - level
# - new value
# - new type
# - username
# - userdomain

sub storeparm {
    my ($sresid,$spnam,$snum,$nval,$ntype,$uname,$udom,$csec,$cgroup)=@_;
    &storeparm_by_symb(&symbcache($sresid),$spnam,$snum,$nval,$ntype,$uname,$udom,$csec,'',$cgroup);
}

my %recstack;
sub storeparm_by_symb {
    my ($symb,$spnam,$snum,$nval,$ntype,$uname,$udom,$csec,$recflag,$cgroup)=@_;
    unless ($recflag) {
# first time call
    %recstack=();
    $recflag=1;
    }
# store parameter
    &storeparm_by_symb_inner
    ($symb,$spnam,$snum,$nval,$ntype,$uname,$udom,$csec,$cgroup);
# don't do anything if parameter was reset
    unless ($nval) { return; }
    my ($prefix,$parm)=($spnam=~/^(.*[\_\.])([^\_\.]+)$/);
# remember that this was set
    $recstack{$parm}=1;
# what does this trigger?
    foreach my $triggered (split(/\:/,&rulescache($parm.'_triggers'))) {
# don't backfire
       unless ((!$triggered) || ($recstack{$triggered})) {
       my $action=&rulescache($triggered.'_action');
       my ($whichaction,$whichparm)=($action=~/^(.*\_)([^\_]+)$/);
# set triggered parameter on same level
       my $newspnam=$prefix.$triggered;
       my $newvalue='';
       my $active=1;
       if ($action=~/^when\_setting/) {
# are there restrictions?
           if (&rulescache($triggered.'_triggervalue')=~/\w/) {
           $active=0;
           foreach my $possiblevalue (split(/\s*\,\s*/,&rulescache($triggered.'_triggervalue'))) {
               if (lc($possiblevalue) eq lc($nval)) { $active=1; }
           }
           }
           $newvalue=&rulescache($triggered.'_value');
       } else {
           my $totalsecs=((&rulescache($triggered.'_days')*24+&rulescache($triggered.'_hours'))*60+&rulescache($triggered.'_min'))*60+&rulescache($triggered.'_sec');
           if ($action=~/^later\_than/) {
           $newvalue=$nval+$totalsecs;
           } else {
           $newvalue=$nval-$totalsecs;
           }
       }
       if ($active) {
           &storeparm_by_symb($symb,$newspnam,$snum,$newvalue,&rulescache($triggered.'_type'),
                   $uname,$udom,$csec,$recflag,$cgroup);
       }
       }
    }
    return '';
}

sub log_parmset {
    return &Apache::lonnet::write_log('course','parameterlog',@_);
}

sub storeparm_by_symb_inner {
# ---------------------------------------------------------- Get symb, map, etc
    my ($symb,$spnam,$snum,$nval,$ntype,$uname,$udom,$csec,$cgroup)=@_;
# ---------------------------------------------------------- Construct prefixes
    $spnam=~s/\_([^\_]+)$/\.$1/;
    my $map=(&Apache::lonnet::decode_symb($symb))[0];
    $map = &Apache::lonnet::deversion($map);

    my $symbparm=$symb.'.'.$spnam;
    my $mapparm=$map.'___(all).'.$spnam;

    my $grplevel=$env{'request.course.id'}.'.['.$cgroup.'].'.$spnam;
    my $grplevelr=$env{'request.course.id'}.'.['.$cgroup.'].'.$symbparm;
    my $grplevelm=$env{'request.course.id'}.'.['.$cgroup.'].'.$mapparm;

    my $seclevel=$env{'request.course.id'}.'.['.$csec.'].'.$spnam;
    my $seclevelr=$env{'request.course.id'}.'.['.$csec.'].'.$symbparm;
    my $seclevelm=$env{'request.course.id'}.'.['.$csec.'].'.$mapparm;

    my $courselevel=$env{'request.course.id'}.'.'.$spnam;
    my $courselevelr=$env{'request.course.id'}.'.'.$symbparm;
    my $courselevelm=$env{'request.course.id'}.'.'.$mapparm;

    my $storeunder='';
    if (($snum==14) || ($snum==3)) { $storeunder=$courselevel; }
    if (($snum==13) || ($snum==2)) { $storeunder=$courselevelm; }
    if (($snum==10) || ($snum==1)) { $storeunder=$courselevelr; }
    if ($snum==9) { $storeunder=$seclevel; }
    if ($snum==8) { $storeunder=$seclevelm; }
    if ($snum==7) { $storeunder=$seclevelr; }
    if ($snum==6) { $storeunder=$grplevel; }
    if ($snum==5) { $storeunder=$grplevelm; }
    if ($snum==4) { $storeunder=$grplevelr; }


    my $delete;
    if ($nval eq '') { $delete=1;}
    my %storecontent = ($storeunder         => $nval,
            $storeunder.'.type' => $ntype);
    my $reply='';
    if ($snum>3) {
# ---------------------------------------------------------------- Store Course
#
    my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
    my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
# Expire sheets
    &Apache::lonnet::expirespread('','','studentcalc');
    if (($snum==10) || ($snum==7) || ($snum==4)) {
        &Apache::lonnet::expirespread('','','assesscalc',$symb);
    } elsif (($snum==11) || ($snum==8) || ($snum==5)) {
        &Apache::lonnet::expirespread('','','assesscalc',$map);
    } else {
        &Apache::lonnet::expirespread('','','assesscalc');
    }
# Store parameter
    if ($delete) {
        $reply=&Apache::lonnet::del
        ('resourcedata',[keys(%storecontent)],$cdom,$cnum);
            &log_parmset(\%storecontent,1);
    } else {
        $reply=&Apache::lonnet::cput
        ('resourcedata',\%storecontent,$cdom,$cnum);
        &log_parmset(\%storecontent);
    }
    &Apache::lonnet::devalidatecourseresdata($cnum,$cdom);
    } else {
# ------------------------------------------------------------------ Store User
#
# Expire sheets
    &Apache::lonnet::expirespread($uname,$udom,'studentcalc');
    if ($snum==1) {
        &Apache::lonnet::expirespread
        ($uname,$udom,'assesscalc',$symb);
    } elsif ($snum==2) {
        &Apache::lonnet::expirespread
        ($uname,$udom,'assesscalc',$map);
    } else {
        &Apache::lonnet::expirespread($uname,$udom,'assesscalc');
    }
# Store parameter
    if ($delete) {
        $reply=&Apache::lonnet::del
        ('resourcedata',[keys(%storecontent)],$udom,$uname);
        &log_parmset(\%storecontent,1,$uname,$udom);
    } else {
        $reply=&Apache::lonnet::cput
        ('resourcedata',\%storecontent,$udom,$uname);
        &log_parmset(\%storecontent,0,$uname,$udom);
    }
    &Apache::lonnet::devalidateuserresdata($uname,$udom);
    }

    if ($reply=~/^error\:(.*)/) {
    return "<span class=\"LC_error\">Write Error: $1</span>";
    }
    return '';
}


sub valout {
    my ($value,$type,$editable)=@_;
    my $result = '';
    # Values of zero are valid.
    if (! $value && $value ne '0') {
        if ($editable) {
            $result =
                '<img src="/res/adm/pages/editmap.png"'
               .' alt="'.&mt('Change').'"'
               .' title="'.&mt('Change').'" style="border:0;" />';
        } else {
            $result='&nbsp;';
        }
    } else {
        if ($type eq 'date_interval') {
            my ($sec,$min,$hour,$mday,$mon,$year)=gmtime($value);
            my @timer;
            $year=$year-70;
            $mday--;
            if ($year) {
#               $result.=&mt('[quant,_1,yr]',$year).' ';
                push(@timer,&mt('[quant,_1,yr]',$year));
            }
            if ($mon) {
#               $result.=&mt('[quant,_1,mth]',$mon).' ';
                push(@timer,&mt('[quant,_1,mth]',$mon));
            }
            if ($mday) {
#               $result.=&mt('[quant,_1,day]',$mday).' ';
                push(@timer,&mt('[quant,_1,day]',$mday));
            }
            if ($hour) {
#               $result.=&mt('[quant,_1,hr]',$hour).' ';
                push(@timer,&mt('[quant,_1,hr]',$hour));
            }
            if ($min) {
#               $result.=&mt('[quant,_1,min]',$min).' ';
                push(@timer,&mt('[quant,_1,min]',$min));
            }
            if ($sec) {
#               $result.=&mt('[quant,_1,sec]',$sec).' ';
                push(@timer,&mt('[quant,_1,sec]',$sec));
            }
#           $result=~s/\s+$//;
            if (!@timer) { # Special case: all entries 0 -> display "0 secs" intead of empty field to keep this field editable
                push(@timer,&mt('[quant,_1,sec]',0));
            }
            $result.=join(", ",@timer);
        } elsif (&isdateparm($type)) {
            $result = &Apache::lonlocal::locallocaltime($value).
        &date_sanity_info($value);
        } else {
            $result = $value;
            $result=~s/\,/\, /gs;
        $result = &HTML::Entities::encode($result,'"<>&');
        }
    }
    return $result;
}


sub plink {
    my ($type,$dis,$value,$marker,$return,$call)=@_;
    my $winvalue=$value;
    unless ($winvalue) {
    if (&isdateparm($type)) {
            $winvalue=$env{'form.recent_'.$type};
        } else {
            $winvalue=$env{'form.recent_'.(split(/\_/,$type))[0]};
        }
    }
    my ($parmname)=((split(/\&/,$marker))[1]=~/\_([^\_]+)$/);
    my ($hour,$min,$sec,$val)=&preset_defaults($parmname);
    unless (defined($winvalue)) { $winvalue=$val; }
    my $valout = &valout($value,$type,1);
    my $unencmarker = $marker;
    foreach my $item (\$type, \$dis, \$winvalue, \$marker, \$return, \$call,
              \$hour, \$min, \$sec) {
    $$item = &HTML::Entities::encode($$item,'"<>&');
    $$item =~ s/\'/\\\'/g;
    }
    return '<table width="100%"><tr valign="top" align="right"><td><a name="'.$unencmarker.'" /></td></tr><tr><td align="center">'.
    '<a href="javascript:pjump('."'".$type."','".$dis."','".$winvalue."','"
        .$marker."','".$return."','".$call."','".$hour."','".$min."','".$sec."'".');">'.
        $valout.'</a></td></tr></table>';
}

sub page_js {

    my $selscript=&Apache::loncommon::studentbrowser_javascript();
    my $pjump_def = &Apache::lonhtmlcommon::pjump_javascript_definition();

    return(<<ENDJS);
<script type="text/javascript">
// <![CDATA[

    $pjump_def

    function psub() {
        if (document.parmform.pres_marker.value!='') {
            document.parmform.action+='#'+document.parmform.pres_marker.value;
            var typedef=new Array();
            typedef=document.parmform.pres_type.value.split('_');
           if (document.parmform.pres_type.value!='') {
            if (typedef[0]=='date') {
                eval('document.parmform.recent_'+
                     document.parmform.pres_type.value+
             '.value=document.parmform.pres_value.value;');
            } else {
                eval('document.parmform.recent_'+typedef[0]+
             '.value=document.parmform.pres_value.value;');
            }
       }
            document.parmform.submit();
        } else {
            document.parmform.pres_value.value='';
            document.parmform.pres_marker.value='';
        }
    }

    function openWindow(url, wdwName, w, h, toolbar,scrollbar) {
        var options = "width=" + w + ",height=" + h + ",";
        options += "resizable=yes,scrollbars="+scrollbar+",status=no,";
        options += "menubar=no,toolbar="+toolbar+",location=no,directories=no";
        var newWin = window.open(url, wdwName, options);
        newWin.focus();
    }
// ]]>
</script>
$selscript
ENDJS

}

sub showhide_js {
    return <<"COURSECONTENTSCRIPT";

function showHide_courseContent() {
    var parmlevValue=document.getElementById("parmlev").value;
    if (parmlevValue == 'general') {
        document.getElementById('mapmenu').style.display="none";
    } else {
        if ((parmlevValue == "full") || (parmlevValue == "map")) {
            document.getElementById('mapmenu').style.display ="";
        } else {
            document.getElementById('mapmenu').style.display="none";
        }
    }
    return;
}

COURSECONTENTSCRIPT
}

sub startpage {
    my ($r,$psymb) = @_;

    my %loaditems = (
                      'onload'   => "group_or_section('cgroup')",
                    );
    if (!$psymb) {
        $loaditems{'onload'} = "showHide_courseContent(); group_or_section('cgroup'); resize_scrollbox('mapmenuscroll','1','1');";
    }

    if ((($env{'form.command'} eq 'set') && ($env{'form.url'})
         && (!$env{'form.dis'})) || ($env{'form.symb'})) {
    &Apache::lonhtmlcommon::add_breadcrumb({help=>'Problem_Parameters',
        text=>"Problem Parameters"});
    } else {
    &Apache::lonhtmlcommon::add_breadcrumb({href=>'/adm/parmset?action=settable',
       text=>"Table Mode",
       help => 'Course_Setting_Parameters'});
    }
    my $js = &page_js().'
<script type="text/javascript">
// <![CDATA[
'.
            &Apache::lonhtmlcommon::resize_scrollbox_js('params').'
// ]]>
</script>
';
    my $start_page =
    &Apache::loncommon::start_page('Set/Modify Course Parameters',$js,
                       {'add_entries' => \%loaditems,});
    my $breadcrumbs =
    &Apache::lonhtmlcommon::breadcrumbs('Table Mode Parameter Setting','Table_Mode');
    my $escfilter=&Apache::lonhtmlcommon::entity_encode($env{'form.filter'});
    my $escpart=&Apache::lonhtmlcommon::entity_encode($env{'form.part'});
    $r->print($start_page.$breadcrumbs);
    $r->print(<<ENDHEAD);
<form method="post" action="/adm/parmset?action=settable" name="parmform">
<input type="hidden" value="" name="pres_value" />
<input type="hidden" value="" name="pres_type" />
<input type="hidden" value="" name="pres_marker" />
<input type="hidden" value="1" name="prevvisit" />
<input type="hidden" value="$escfilter" name="filter" />
<input type="hidden" value="$escpart" name="part" />
ENDHEAD
}


sub print_row {
    my ($r,$which,$part,$name,$symbp,$rid,$default,$defaulttype,$display,$defbgone,
    $defbgtwo,$defbgthree,$parmlev,$uname,$udom,$csec,$cgroup,$usersgroups)=@_;
    my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
    my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $courseopt=&Apache::lonnet::get_courseresdata($cnum,$cdom);
# get the values for the parameter in cascading order
# empty levels will remain empty
    my ($result,@outpar)=&parmval($$part{$which}.'.'.$$name{$which},
      $rid,$$default{$which},$uname,$udom,$csec,$cgroup,$courseopt);
# get the type for the parameters
# problem: these may not be set for all levels
    my ($typeresult,@typeoutpar)=&parmval($$part{$which}.'.'.
                                          $$name{$which}.'.type',$rid,
         $$defaulttype{$which},$uname,$udom,$csec,$cgroup,$courseopt);
# cascade down manually
    my $cascadetype=$$defaulttype{$which};
    for (my $i=14;$i>0;$i--) {
     if ($typeoutpar[$i]) {
            $cascadetype=$typeoutpar[$i];
    } else {
            $typeoutpar[$i]=$cascadetype;
        }
    }
    my $parm=$$display{$which};

    if ($parmlev eq 'full') {
        $r->print('<td style="background-color:'.$defbgtwo.';" align="center">'
                  .($$part{$which} eq '0'?'0 ('.&mt('default').')':$$part{$which}).'</td>');
    } else {
        $parm=~s|\[.*\]\s||g;
    }
    my $automatic=&rulescache(($which=~/\_([^\_]+)$/)[0].'_triggers');
    if ($automatic) {
    $parm.='<span class="LC_warning"><br />'.&mt('Automatically sets').' '.join(', ',split(/\:/,$automatic)).'</span>';
    }
    $r->print('<td>'.$parm.'</td>');

    my $thismarker=$which;
    $thismarker=~s/^parameter\_//;
    my $mprefix=$rid.'&'.$thismarker.'&';
    my $effective_parm = &valout($outpar[$result],$typeoutpar[$result]);
    my ($othergrp,$grp_parm,$controlgrp);

    if ($parmlev eq 'general') {

        if ($uname) {
            &print_td($r,3,$defbgone,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
        } elsif ($cgroup) {
            &print_td($r,6,$defbgthree,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
        } elsif ($csec) {
            &print_td($r,9,$defbgtwo,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
        } else {
            &print_td($r,14,$defbgone,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
        }
    } elsif ($parmlev eq 'map') {

        if ($uname) {
            &print_td($r,2,$defbgone,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
        } elsif ($cgroup) {
            &print_td($r,5,$defbgthree,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
        } elsif ($csec) {
            &print_td($r,8,$defbgtwo,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
        } else {
            &print_td($r,13,$defbgone,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
        }
    } else {
        if ($uname) {
            if (@{$usersgroups} > 1) {
                my ($coursereply,$grp_parm,$controlgrp);
                ($coursereply,$othergrp,$grp_parm,$controlgrp) =
                    &print_usergroups($r,$$part{$which}.'.'.$$name{$which},
                       $rid,$cgroup,$defbgone,$usersgroups,$result,$courseopt);
                if ($coursereply && $result > 3) {
                    if (defined($controlgrp)) {
                        if ($cgroup ne $controlgrp) {
                            $effective_parm = $grp_parm;
                            $result = 0;
                        }
                    }
                }
            }
        }

        &print_td($r,14,$defbgone,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);

    &print_td($r,13,$defbgone,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
    &print_td($r,12,'#FFDDDD',$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
    &print_td($r,11,'#FFDDDD',$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
    &print_td($r,10,$defbgone,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);

    if ($csec) {
        &print_td($r,9,$defbgtwo,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
        &print_td($r,8,$defbgtwo,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
        &print_td($r,7,$defbgtwo,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
    }

        if ($cgroup) {
            &print_td($r,6,$defbgthree,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
            &print_td($r,5,$defbgthree,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
            &print_td($r,4,$defbgthree,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
        }

    if ($uname) {
            if ($othergrp) {
                $r->print($othergrp);
            }
        &print_td($r,3,$defbgone,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
        &print_td($r,2,$defbgone,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
        &print_td($r,1,$defbgone,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
    }

    } # end of $parmlev if/else
    $r->print('<td style="background-color:#CCCCFF;" align="center">'.$effective_parm.'</td>');

    if ($parmlev eq 'full') {
        my $sessionval=&Apache::lonnet::EXT('resource.'.$$part{$which}.
                                        '.'.$$name{$which},$$symbp{$rid});
        my $sessionvaltype=$typeoutpar[$result];
        if (!defined($sessionvaltype)) { $sessionvaltype=$$defaulttype{$which}; }
        $r->print('<td style="background-color:#999999;" align="center"><font color="#FFFFFF">'.
                  &valout($sessionval,$sessionvaltype).'&nbsp;'.
                  '</font></td>');
    }
    $r->print('</tr>');
    $r->print("\n");
}

sub print_td {
    my ($r,$which,$defbg,$result,$outpar,$mprefix,$value,$typeoutpar,$display)=@_;
    $r->print('<td style="background-color:'.(($result==$which)?'#AAFFAA':$defbg).
              ';" align="center">');
    my $nolink = 0;
    if ($which == 11 || $which == 12) {
        $nolink = 1;
    } elsif ($mprefix =~ /availablestudent\&$/) {
        if ($which > 3) {
            $nolink = 1;
        }
    } elsif ($mprefix =~ /examcode\&$/) {
        unless ($which == 2) {
            $nolink = 1;
        }
    }
    if ($nolink) {
        $r->print(&valout($$outpar[$which],$$typeoutpar[$which]));
    } else {
        $r->print(&plink($$typeoutpar[$which],
                         $$display{$value},$$outpar[$which],
                         $mprefix."$which",'parmform.pres','psub'));
    }
    $r->print('</td>'."\n");
}

sub print_usergroups {
    my ($r,$what,$rid,$cgroup,$defbg,$usersgroups,$result,$courseopt) = @_;
    my $courseid = $env{'request.course.id'};
    my $output;
    my $symb = &symbcache($rid);
    my $symbparm=$symb.'.'.$what;
    my $map=(&Apache::lonnet::decode_symb($symb))[0];
    my $mapparm=$map.'___(all).'.$what;
    my ($coursereply,$resultitem,$resultgroup,$resultlevel,$resulttype) =
          &parm_control_group($courseid,$usersgroups,$symbparm,$mapparm,$what,
                                                                   $courseopt);
    my $bgcolor = $defbg;
    my $grp_parm;
    if (($coursereply) && ($cgroup ne $resultgroup)) {
        if ($result > 3) {
            $bgcolor = '#AAFFAA';
            $grp_parm = &valout($coursereply,$resulttype);
        }
        $grp_parm = &valout($coursereply,$resulttype);
        $output = '<td style="background-color:'.$bgcolor.';" align="center">';
        if ($resultgroup && $resultlevel) {
            $output .= '<small><b>'.$resultgroup.'</b> ('.$resultlevel.'): </small>'.$grp_parm;
        } else {
            $output .= '&nbsp;';
        }
        $output .= '</td>';
    } else {
        $output .= '<td style="background-color:'.$bgcolor.';">&nbsp;</td>';
    }
    return ($coursereply,$output,$grp_parm,$resultgroup);
}

sub parm_control_group {
    my ($courseid,$usersgroups,$symbparm,$mapparm,$what,$courseopt) = @_;
    my ($coursereply,$resultitem,$resultgroup,$resultlevel,$resulttype);
    my $grpfound = 0;
    my @levels = ($symbparm,$mapparm,$what);
    my @levelnames = ('resource','map/folder','general');
    foreach my $group (@{$usersgroups}) {
        if ($grpfound) { last; }
        for (my $i=0; $i<@levels; $i++) {
            my $item = $courseid.'.['.$group.'].'.$levels[$i];
            if (defined($$courseopt{$item})) {
                $coursereply = $$courseopt{$item};
                $resultitem = $item;
                $resultgroup = $group;
                $resultlevel = $levelnames[$i];
                $resulttype = $$courseopt{$item.'.type'};
                $grpfound = 1;
                last;
            }
        }
    }
    return($coursereply,$resultitem,$resultgroup,$resultlevel,$resulttype);
}



sub extractResourceInformation {
    my $ids = shift;
    my $typep = shift;
    my $keyp = shift;
    my $allparms = shift;
    my $allparts = shift;
    my $allmaps = shift;
    my $mapp = shift;
    my $symbp = shift;
    my $maptitles=shift;
    my $uris=shift;
    my $keyorder=shift;
    my $defkeytype=shift;

    my $keyordercnt=100;

    my $navmap = Apache::lonnavmaps::navmap->new();
    my @allres=$navmap->retrieveResources(undef,undef,1,undef,1);
    foreach my $resource (@allres) {
        my $id=$resource->id();
        my ($mapid,$resid)=split(/\./,$id);
        if ($mapid eq '0') { next; }
        $$ids[$#$ids+1]=$id;
        my $srcf=$resource->src();
        $srcf=~/\.(\w+)$/;
        $$typep{$id}=$1;
        $$keyp{$id}='';
        $$uris{$id}=$srcf;

        foreach my $key (split(/\,/,&Apache::lonnet::metadata($srcf,'allpossiblekeys'))) {
            next if ($key!~/^parameter_/);

# Hidden parameters
            next if (&Apache::lonnet::metadata($srcf,$key.'.hidden') eq 'parm');
#
# allparms is a hash of parameter names
#
            my $name=&Apache::lonnet::metadata($srcf,$key.'.name');
            if (!exists($$allparms{$name}) || $$allparms{$name} =~ m/^\s*$/ ) {
                my ($display,$parmdis);
                $display = &standard_parameter_names($name);
                if ($display eq '') {
                    $display= &Apache::lonnet::metadata($srcf,$key.'.display');
                    $parmdis = $display;
                    $parmdis =~ s/\s*\[Part.*$//g;
                } else {
                    $parmdis = &mt($display);
                }
                $$allparms{$name}=$parmdis;
                if (ref($defkeytype)) {
                    $$defkeytype{$name}=
                    &Apache::lonnet::metadata($srcf,$key.'.type');
                }
            }

#
# allparts is a hash of all parts
#
            my $part= &Apache::lonnet::metadata($srcf,$key.'.part');
            $$allparts{$part} = &mt('Part: [_1]',$part);
#
# Remember all keys going with this resource
#
            if ($$keyp{$id}) {
                $$keyp{$id}.=','.$key;
            } else {
                $$keyp{$id}=$key;
            }   
#
# Put in order
#
            unless ($$keyorder{$key}) {
                $$keyorder{$key}=$keyordercnt;
                $keyordercnt++;
            }
        }


        if (!exists($$mapp{$mapid})) {
            $$mapp{$id}=
            &Apache::lonnet::declutter($resource->enclosing_map_src());
            $$mapp{$mapid}=$$mapp{$id};
            $$allmaps{$mapid}=$$mapp{$id};
            if ($mapid eq '1') {
                $$maptitles{$mapid}=&mt('Main Content');
            } else {
                $$maptitles{$mapid}=&Apache::lonnet::gettitle($$mapp{$id});
            }
            $$maptitles{$$mapp{$id}}=$$maptitles{$mapid};
            $$symbp{$mapid}=$$mapp{$id}.'___(all)';
        } else {
            $$mapp{$id} = $$mapp{$mapid};
        }
        $$symbp{$id}=&Apache::lonnet::encode_symb($$mapp{$id},$resid,$srcf);
    }
}



sub isdateparm {
    my $type=shift;
    return (($type=~/^date/) && (!($type eq 'date_interval')));
}

#
# parmmenu displays a list of the selected parameters.
# It also offers a link to show/hide the complete parameter list
# from which you can select all desired parameters.
#
sub parmmenu {
    my ($r,$allparms,$pscat,$keyorder)=@_;
    my $tempkey;
    $r->print(<<ENDSCRIPT);
<script type="text/javascript">
// <![CDATA[
    function checkall(value, checkName) {

        var li = "_li";
        var displayOverview = "";
        
        if (value == false) {
            displayOverview = "none"
        }

    for (i=0; i<document.forms.parmform.elements.length; i++) {
            ele = document.forms.parmform.elements[i];
            if (ele.name == checkName) {
                document.forms.parmform.elements[i].checked=value;
            }
        }
    }

    function checkthis(thisvalue, checkName) {


    for (i=0; i<document.forms.parmform.elements.length; i++) {
            ele = document.forms.parmform.elements[i];
            if (ele.name == checkName) {
        if (ele.value == thisvalue) {
            document.forms.parmform.elements[i].checked=true;
        }
            }
        }
    }

    function checkdates() {
    checkthis('duedate','pscat');
     checkthis('opendate','pscat');
    checkthis('answerdate','pscat');
    }

    function checkdisset() {
     checkthis('discussend','pscat');
     checkthis('discusshide','pscat');
     checkthis('discussvote','pscat');
    }

    function checkcontdates() {
    checkthis('contentopen','pscat');
     checkthis('contentclose','pscat');
    }

    function checkvisi() {
    checkthis('hiddenresource','pscat');
     checkthis('encrypturl','pscat');
    checkthis('problemstatus','pscat');
    checkthis('contentopen','pscat');
    checkthis('opendate','pscat');
    }

    function checkparts() {
    checkthis('hiddenparts','pscat');
    checkthis('display','pscat');
    checkthis('ordered','pscat');
    }

    function checkstandard() {
        checkall(false,'pscat');
    checkdates();
    checkthis('weight','pscat');
    checkthis('maxtries','pscat');
    checkthis('type','pscat');
    checkthis('problemstatus','pscat');
    }

// ]]>
</script>
ENDSCRIPT

    $r->print('<hr />');
    &shortCuts($r,$allparms,$pscat,$keyorder);
    $r->print('<hr />');
}
# return a hash
sub categories {
    return ('time_settings' => 'Time Settings',
    'grading' => 'Grading',
    'tries' => 'Tries',
    'problem_appearance' => 'Problem Appearance',
    'behaviour_of_input_fields' => 'Behaviour of Input Fields',
    'hiding' => 'Hiding',
    'high_level_randomization' => 'High Level Randomization',
    'slots' => 'Slots',
    'file_submission' => 'File Submission',
    'misc' => 'Miscellaneous' ); 
}

# return a hash. Like a look-up table
sub lookUpTableParameter {
 
    return ( 
        'opendate' => 'time_settings',
        'duedate' => 'time_settings',
        'answerdate' => 'time_settings',
        'interval' => 'time_settings',
        'contentopen' => 'time_settings',
        'contentclose' => 'time_settings',
        'discussend' => 'time_settings',
	'printopendate' => 'time_settings',
	'printclosedate' => 'time_settings',
        'weight' => 'grading',
        'handgrade' => 'grading',
        'maxtries' => 'tries',
        'hinttries' => 'tries',
        'randomizeontries' => 'tries',
        'type' => 'problem_appearance',
        'problemstatus' => 'problem_appearance',
        'display' => 'problem_appearance',
        'ordered' => 'problem_appearance',
        'numbubbles' => 'problem_appearance',
        'tol' => 'behaviour_of_input_fields',
        'sig' => 'behaviour_of_input_fields',
        'turnoffunit' => 'behaviour_of_input_fields',
        'hiddenresource' => 'hiding',
        'hiddenparts' => 'hiding',
        'discusshide' => 'hiding',
        'buttonshide' => 'hiding',
        'turnoffeditor' => 'hiding',
        'encrypturl' => 'hiding',
        'randomorder' => 'high_level_randomization',
        'randompick' => 'high_level_randomization',
        'available' => 'slots',
        'useslots' => 'slots',
        'availablestudent' => 'slots',
        'uploadedfiletypes' => 'file_submission',
        'maxfilesize' => 'file_submission',
        'cssfile' => 'misc',
        'mapalias' => 'misc',
        'acc' => 'misc',
        'maxcollaborators' => 'misc',
        'scoreformat' => 'misc',
        'lenient' => 'grading',
        'retrypartial' => 'tries',
        'discussvote'  => 'misc',
        'examcode' => 'high_level_randomization',
    );    
}

sub whatIsMyCategory {
    my $name = shift;
    my $catList = shift;
    my @list;
    my %lookUpList = &lookUpTableParameter; #Initilize the lookupList
    my $cat = $lookUpList{$name};
    if (defined($cat)) {
        if (!defined($$catList{$cat})){
            push @list, ($name);
            $$catList{$cat} = \@list;
        } else {
            push @{${$catList}{$cat}}, ($name);     
        }
    } else {
        if (!defined($$catList{'misc'})){
            push @list, ($name);
            $$catList{'misc'} = \@list;
        } else {
            push @{${$catList}{'misc'}}, ($name);     
        }
    }        
}

sub keysindisplayorderCategory {
    my ($name,$keyorder)=@_;
    return sort {
        $$keyorder{'parameter_0_'.$a} <=> $$keyorder{'parameter_0_'.$b}; 
    } ( @{$name});
}

sub category_order {
    return (
        'time_settings' => 1,
        'grading' => 2,
        'tries' => 3,
        'problem_appearance' => 4,
        'hiding' => 5,
        'behaviour_of_input_fields' => 6,
        'high_level_randomization'  => 7,
        'slots' => 8,
        'file_submission' => 9,
        'misc' => 10
    );

}

sub parmboxes {
    my ($r,$allparms,$pscat,$keyorder)=@_;
    my $tempkey;
    my $tempparameter;
    my %categories = &categories;
    my %category_order = &category_order();
    my %categoryList = (
        'time_settings' => [],
        'grading' => [],
        'tries' => [],
        'problem_appearance' => [],
        'behaviour_of_input_fields' => [],
        'hiding' => [],
        'high_level_randomization' => [],
        'slots' => [],
        'file_submission' => [],
        'misc' => [],
    );

    foreach $tempparameter (keys %$allparms) {
        &whatIsMyCategory($tempparameter, \%categoryList);
    }
    #part to print the parm-list
    $r->print('<div class="LC_columnSection">'."\n");

    #Print parameters
    for my $key (sort { $category_order{$a} <=> $category_order{$b} } keys %categoryList) {
        next if(@{$categoryList{$key}} == 0);
        $r->print('<div class="LC_Box LC_400Box">'
                  .'<h4 class="LC_hcell">'
                  .&mt($categories{$key})
                  .'</h4>'."\n");
        foreach $tempkey (&keysindisplayorderCategory($categoryList{$key},$keyorder)) {
            $r->print('<span class="LC_nobreak">'
                     .'<label><input type="checkbox" name="pscat" '
                     .'value="'.$tempkey.'" ');
            if ($$pscat[0] eq "all" || grep $_ eq $tempkey, @{$pscat}) {
                $r->print(' checked="checked"');
            }
            $r->print(' />'.($$allparms{$tempkey}=~/\S/ ? $$allparms{$tempkey}
                                                      : $tempkey)
                     .'</label></span><br />'."\n");
        }
        $r->print("</div>\n");
    }

    $r->print("</div>\n");
}
#
# This function offers some links on the parameter section to get with one click a group a parameters
#
sub shortCuts {
    my ($r,$allparms,$pscat,$keyorder)=@_;

    # Parameter Selection
    $r->print(
        &Apache::lonhtmlcommon::start_funclist(&mt('Parameter Selection'))
       .&Apache::lonhtmlcommon::add_item_funclist(
            '<a href="javascript:checkall(true, \'pscat\')">'.&mt('Select All').'</a>')
       .&Apache::lonhtmlcommon::add_item_funclist(
            '<a href="javascript:checkstandard()">'.&mt('Select Common Only').'</a>')
       .&Apache::lonhtmlcommon::add_item_funclist(
            '<a href="javascript:checkall(false, \'pscat\')">'.&mt('Unselect All').'</a>')
       .&Apache::lonhtmlcommon::end_funclist()
    );

    # Add Selection for...
    $r->print(
        &Apache::lonhtmlcommon::start_funclist(&mt('Add Selection for...'))
       .&Apache::lonhtmlcommon::add_item_funclist(
            '<a href="javascript:checkdates()">'.&mt('Problem Dates').'</a>')
       .&Apache::lonhtmlcommon::add_item_funclist(
            '<a href="javascript:checkcontdates()">'.&mt('Content Dates').'</a>')
       .&Apache::lonhtmlcommon::add_item_funclist(
            '<a href="javascript:checkdisset()">'.&mt('Discussion Settings').'</a>')
       .&Apache::lonhtmlcommon::add_item_funclist(
            '<a href="javascript:checkvisi()">'.&mt('Visibilities').'</a>')
       .&Apache::lonhtmlcommon::add_item_funclist(
            '<a href="javascript:checkparts()">'.&mt('Part Parameters').'</a>')
       .&Apache::lonhtmlcommon::end_funclist()
    );
}

sub partmenu {
    my ($r,$allparts,$psprt)=@_;
    my $selsize = 1+scalar(keys(%{$allparts}));
    if ($selsize > 8) {
        $selsize = 8;
    }

    $r->print('<select multiple="multiple" name="psprt" size="'.$selsize.'">');
    $r->print('<option value="all"');
    $r->print(' selected="selected"') unless (@{$psprt});
    $r->print('>'.&mt('All Parts').'</option>');
    my %temphash=();
    foreach (@{$psprt}) { $temphash{$_}=1; }
    foreach my $tempkey (sort {
    if ($a==$b) { return ($a cmp $b) } else { return ($a <=> $b); }
    } keys(%{$allparts})) {
    unless ($tempkey =~ /\./) {
        $r->print('<option value="'.$tempkey.'"');
        if ($$psprt[0] eq "all" ||  $temphash{$tempkey}) {
        $r->print(' selected="selected"');
        }
        $r->print('>'.$$allparts{$tempkey}.'</option>');
    }
    }
    $r->print('</select>');
}

sub usermenu {
    my ($r,$uname,$id,$udom,$csec,$cgroup,$parmlev,$usersgroups)=@_;
    my $chooseopt=&Apache::loncommon::select_dom_form($udom,'udom').' '.
        &Apache::loncommon::selectstudent_link('parmform','uname','udom');
    my $selscript=&Apache::loncommon::studentbrowser_javascript();

    my $sections='';
    my %sectionhash = &Apache::loncommon::get_sections();

    my $groups;
    my %grouphash = &Apache::longroup::coursegroups();

    my $g_s_header='';
    my $g_s_footer='';

    if (%sectionhash) {
        $sections=&mt('Section:').' <select name="csec"';
        if (%grouphash && $parmlev ne 'full') {
            $sections .= qq| onchange="group_or_section('csec')" |;
        }
        $sections .= '>';
    foreach my $section ('',sort keys %sectionhash) {
        $sections.='<option value="'.$section.'" '.
        ($section eq $csec?'selected="selected"':'').'>'.$section.
                                                              '</option>';
        }
        $sections.='</select>';
    }

    if (%sectionhash && %grouphash && $parmlev ne 'full') {
        $sections .= '&nbsp;'.&mt('or').'&nbsp;';
        $sections .= qq|
<script type="text/javascript">
// <![CDATA[
function group_or_section(caller) {
   if (caller == "cgroup") {
       if (document.parmform.cgroup.selectedIndex != 0) {
           document.parmform.csec.selectedIndex = 0;
       }
   } else {
       if (document.parmform.csec.selectedIndex != 0) {
           document.parmform.cgroup.selectedIndex = 0;
       }
   }
}
// ]]>
</script>
|;
    } else {
        $sections .= qq|
<script type="text/javascript">
// <![CDATA[
function group_or_section(caller) {
    return;
}
// ]]>
</script>
|;
    }

    if (%grouphash) {
        $groups=&mt('Group:').' <select name="cgroup"';
        if (%sectionhash && $env{'form.action'} eq 'settable') {
            $groups .= qq| onchange="group_or_section('cgroup')" |;
        }
        $groups .= '>';
        foreach my $grp ('',sort keys %grouphash) {
            $groups.='<option value="'.$grp.'" ';
            if ($grp eq $cgroup) {
                unless ((defined($uname)) && ($grp eq '')) {
                    $groups .=  'selected="selected" ';
                }
            } elsif (!defined($cgroup)) {
                if (@{$usersgroups} == 1) {
                    if ($grp eq $$usersgroups[0]) {
                        $groups .=  'selected="selected" ';
                    }
                }
            }
            $groups .= '>'.$grp.'</option>';
        }
        $groups.='</select>';
    }

    if (%sectionhash || %grouphash) {
        $r->print(&Apache::lonhtmlcommon::row_title(&mt('Group/Section')));
        $r->print($sections.$groups);
        $r->print(&Apache::lonhtmlcommon::row_closure());
    }

    $r->print(&Apache::lonhtmlcommon::row_title(&mt('User')));
    $r->print(&mt('For User [_1] or Student/Employee ID [_2] at Domain [_3]'
                 ,'<input type="text" value="'.$uname.'" size="12" name="uname" />'
                 ,'<input type="text" value="'.$id.'" size="12" name="id" /> '
                 ,$chooseopt));
}

#
# This function shows on table Mode the available Parameters for the selected Resources
#
sub displaymenu {
    my ($r,$allparms,$pscat,$psprt,$keyorder,$divid)=@_;

    $r->print(&Apache::lonhtmlcommon::start_pick_box());
    $r->print(&Apache::lonhtmlcommon::row_title(&mt('Select Parameters to View')));

    &parmmenu($r,$allparms,$pscat,$keyorder);
    $r->print(&Apache::loncommon::start_scrollbox('480px','440px','200px',$divid));
    &parmboxes($r,$allparms,$pscat,$keyorder);
    $r->print(&Apache::loncommon::end_scrollbox());

    $r->print(&Apache::lonhtmlcommon::row_closure(1));
    $r->print(&Apache::lonhtmlcommon::end_pick_box());
 
}

sub mapmenu {
    my ($r,$allmaps,$pschp,$maptitles,$symbp)=@_;
    my %allmaps_inverted = reverse %$allmaps;
    my $navmap = Apache::lonnavmaps::navmap->new();
    my $tree=[];
    my $treeinfo={};
    if (defined($navmap)) {
        my $it=$navmap->getIterator(undef,undef,undef,1,1,undef);
        my $curRes;
        my $depth = 0;
        my %parent = ();
        my $startcount = 5;
        my $lastcontainer = $startcount;
# preparing what is to show ...
        while ($curRes = $it->next()) {
            if ($curRes == $it->BEGIN_MAP()) {
                $depth++;
                $parent{$depth}= $lastcontainer;
            }
            if ($curRes == $it->END_MAP()) {
                $depth--;
                $lastcontainer = $parent{$depth};
            }
            if (ref($curRes)) {
                my $symb = $curRes->symb();
                my $ressymb = $symb;
                if (($curRes->is_sequence()) || ($curRes->is_page())) {
                    my $type = 'sequence';
                    if ($curRes->is_page()) {
                        $type = 'page';
                    }
                    my $id= $curRes->id();
                    my $srcf = $curRes->src();
                    my $resource_name = &Apache::lonnet::gettitle($srcf);
                    if(!exists($treeinfo->{$id})) {
                        push(@$tree,$id);
                        my $enclosing_map_folder = &Apache::lonnet::declutter($curRes->enclosing_map_src());        
                        $treeinfo->{$id} = {
                                    depth => $depth,
                                    type  => $type,
                                    name  => $resource_name,
                                    enclosing_map_folder => $enclosing_map_folder,
                                    };
                    }
                }
            }
        }
    }
# Show it ...    
    $r->print(&Apache::lonhtmlcommon::row_title(&mt('Select Enclosing Map or Folder'),'','',' id="mapmenu"'));
    if ((ref($tree) eq 'ARRAY') && (ref($treeinfo) eq 'HASH')) {
        my $icon = '<img src="/adm/lonIcons/navmap.folder.open.gif" alt="" />';
        my $whitespace =
            '<img src="'
           .&Apache::loncommon::lonhttpdurl('/adm/lonIcons/whitespace_21.gif')
           .'" alt="" />';

        # Info about selectable folders/maps
        $r->print(
            '<div class="LC_info">'
           .&mt('You can only select maps and folders which have modifiable settings.')
           .' '.&Apache::loncommon::help_open_topic('Parameter_Set_Folder') 
           .'</div>'
        );

        $r->print(&Apache::loncommon::start_scrollbox('700px','680px','400px','mapmenuscroll'));
        $r->print(&Apache::loncommon::start_data_table(undef,'mapmenuinner'));

        # Display row: "All Maps or Folders"
        $r->print(
            &Apache::loncommon::start_data_table_row(undef,'picklevel')
           .'<td>'
           .'<label>'
           .'<input type="radio" name="pschp"'
        );
        $r->print(' checked="checked"') if ($pschp eq 'all' || !$pschp);
        $r->print(
            ' value="all" />&nbsp;'.$icon.'&nbsp;'
           .&mt('All Maps or Folders')
           .'</label>'
           .'<hr /></td>'
           .&Apache::loncommon::end_data_table_row()
        );

        # Display row: "Main Content"
        if (exists($$allmaps{1})) {
            $r->print(
                &Apache::loncommon::start_data_table_row()
               .'<td>'
               .'<label>'
               .'<input type="radio" name="pschp" value="1"'
            );
            $r->print(' checked="checked"') if ($pschp eq '1');
            $r->print(
                '/>&nbsp;'.$icon.'&nbsp;'
               .$$maptitles{1}
               .($$allmaps{1} !~/^uploaded/?' ['.$$allmaps{1}.']':'')
               .'</label>'
               .'</td>'
               .&Apache::loncommon::end_data_table_row()
            );
        }

        # Display rows for all course maps and folders
        foreach my $id (@{$tree}) {
            my ($mapid,$resid)=split(/\./,$id);
            # Indentation
            my $depth = $treeinfo->{$id}->{'depth'};
            my $indent;
            for (my $i = 0; $i < $depth; $i++) {
                $indent.= $whitespace;
            }
            $icon =  '<img src="/adm/lonIcons/navmap.folder.open.gif" alt="" />';
            if ($treeinfo->{$id}->{'type'} eq 'page') {
                $icon = '<img src="/adm/lonIcons/navmap.page.open.gif" alt="" />';
            }
            my $symb_name = $$symbp{$id};
            my ($front, $tail) = split (/___${resid}___/, $symb_name);
            $symb_name = $tail;
            $r->print(
                &Apache::loncommon::start_data_table_row()
               .'<td>'
               .'<label>'
            );
            # Only offer radio button for folders/maps which can be parameterized
            if ($allmaps_inverted{$symb_name}) {
                $r->print(
                    '<input type ="radio" name="pschp"'
                   .' value="'.$allmaps_inverted{$symb_name}.'"'
                );
                $r->print(' checked="checked"') if ($allmaps_inverted{$symb_name} eq $pschp);
                $r->print('/>');
            } else {
                $r->print($whitespace);
            }
            $r->print(
                $indent.$icon.'&nbsp;'
               .$treeinfo->{$id}->{name}
               .($$allmaps{$mapid}!~/^uploaded/?' ['.$$allmaps{$mapid}.']':'')
               .'</label>'
               .'</td>'
               .&Apache::loncommon::end_data_table_row()
            );
        }

        $r->print(&Apache::loncommon::end_data_table().
                  '<br style="line-height:2px;" />'.
                  &Apache::loncommon::end_scrollbox());
    }
}

# Build up the select Box to choose if your parameter specification should work for the resource, map/folder or the course level
# The value of default selection in the select box is set by the value that is given by the argument in $parmlev.
sub levelmenu {
    my ($r,$alllevs,$parmlev)=@_;

    $r->print(&Apache::lonhtmlcommon::row_title(&mt('Select Parameter Level').&Apache::loncommon::help_open_topic('Course_Parameter_Levels')));
    $r->print('<select id="parmlev" name="parmlev" onchange="showHide_courseContent()">');
    foreach (reverse sort keys %{$alllevs}) {
    $r->print('<option value="'.$$alllevs{$_}.'"');
    if ($parmlev eq $$alllevs{$_}) {
        $r->print(' selected="selected"');
    }
    $r->print('>'.&mt($_).'</option>');
    }
    $r->print("</select>");
}


sub sectionmenu {
    my ($r,$selectedsections)=@_;
    my %sectionhash = &Apache::loncommon::get_sections();
    return if (!%sectionhash);

    $r->print('<select name="Section" multiple="multiple" size="8">');
    foreach my $s ('all',sort keys %sectionhash) {
    $r->print('    <option value="'.$s.'"');
    foreach (@{$selectedsections}) {
        if ($s eq $_) {
        $r->print(' selected="selected"');
        last;
        }
    }
    $r->print('>'.$s."</option>\n");
    }
    $r->print("</select>\n");
}

sub groupmenu {
    my ($r,$selectedgroups)=@_;
    my %grouphash = &Apache::longroup::coursegroups();
    return if (!%grouphash);

    $r->print('<select name="Group" multiple="multiple" size="8">');
    foreach my $group (sort(keys(%grouphash))) {
    $r->print('    <option value="'.$group.'"');
    foreach (@{$selectedgroups}) {
        if ($group eq $_) {
        $r->print(' selected="selected"');
        last;
        }
    }
    $r->print('>'.$group."</option>\n");
    }
    $r->print("</select>\n");
}


sub keysplit {
    my $keyp=shift;
    return (split(/\,/,$keyp));
}

sub keysinorder {
    my ($name,$keyorder)=@_;
    return sort {
    $$keyorder{$a} <=> $$keyorder{$b};
    } (keys %{$name});
}

sub keysinorder_bytype {
    my ($name,$keyorder)=@_;
    return sort {
    my $ta=(split('_',$a))[-1];
    my $tb=(split('_',$b))[-1];
    if ($$keyorder{'parameter_0_'.$ta} == $$keyorder{'parameter_0_'.$tb}) {
        return ($a cmp $b);
    }
    $$keyorder{'parameter_0_'.$ta} <=> $$keyorder{'parameter_0_'.$tb};
    } (keys %{$name});
}

sub keysindisplayorder {
    my ($name,$keyorder)=@_;
    return sort {
    $$keyorder{'parameter_0_'.$a} <=> $$keyorder{'parameter_0_'.$b};
    } (keys %{$name});
}

sub sortmenu {
    my ($r,$sortorder)=@_;
    $r->print('<br /><label><input type="radio" name="sortorder" value="realmstudent"');
    if ($sortorder eq 'realmstudent') {
       $r->print(' checked="checked"');
    }
    $r->print(' />'.&mt('Sort by realm first, then student (group/section)'));
    $r->print('</label><br /><label><input type="radio" name="sortorder" value="studentrealm"');
    if ($sortorder eq 'studentrealm') {
       $r->print(' checked="checked"');
    }
    $r->print(' />'.&mt('Sort by student (group/section) first, then realm').
          '</label>');
}

sub standardkeyorder {
    return ('parameter_0_opendate' => 1,
        'parameter_0_duedate' => 2,
        'parameter_0_answerdate' => 3,
        'parameter_0_interval' => 4,
        'parameter_0_weight' => 5,
        'parameter_0_maxtries' => 6,
        'parameter_0_hinttries' => 7,
        'parameter_0_contentopen' => 8,
        'parameter_0_contentclose' => 9,
        'parameter_0_type' => 10,
        'parameter_0_problemstatus' => 11,
        'parameter_0_hiddenresource' => 12,
        'parameter_0_hiddenparts' => 13,
        'parameter_0_display' => 14,
        'parameter_0_ordered' => 15,
        'parameter_0_tol' => 16,
        'parameter_0_sig' => 17,
        'parameter_0_turnoffunit' => 18,
        'parameter_0_discussend' => 19,
        'parameter_0_discusshide' => 20,
        'parameter_0_discussvote' => 21,
	'parameter_0_printopendate'  =>  22,
	'parameter_0_printclosedate' =>  23);
}


sub assessparms {

    my $r=shift;



# -------------------------------------------------------- Variable declaration
    my @ids=();
    my %symbp=();
    my %mapp=();
    my %typep=();
    my %keyp=();
    my %uris=();
    my %maptitles=();
    my %allmaps=();
    my %alllevs=();

    my $uname;
    my $udom;
    my $uhome;
    my $csec;
    my $cgroup;
    my @usersgroups = ();

    my $coursename=$env{'course.'.$env{'request.course.id'}.'.description'};

    $alllevs{'Resource Level'}='full';
    $alllevs{'Map/Folder Level'}='map';
    $alllevs{'Course Level'}='general';

    my %allparms;
    my %allparts;
# ------------------------------------------------------------------------------

#
# Order in which these parameters will be displayed
#
    my %keyorder=&standardkeyorder();

#    @ids=();
#    %symbp=();       # These seem defined above already.
#    %typep=();

    my $message='';

    $csec=$env{'form.csec'};
    $cgroup=$env{'form.cgroup'};

    if      ($udom=$env{'form.udom'}) {
    } elsif ($udom=$env{'request.role.domain'}) {
    } elsif ($udom=$env{'user.domain'}) {
    } else {
        $udom=$r->dir_config('lonDefDomain');
    }
    

    my @pscat=&Apache::loncommon::get_env_multiple('form.pscat');
    my $pschp=$env{'form.pschp'};


    my @psprt=&Apache::loncommon::get_env_multiple('form.psprt');
    if (!@psprt) { $psprt[0]='0'; }
    if (($env{'form.part'}) && ($psprt[0] ne 'all')) { $psprt[0]=$env{'form.part'}; }

    my $pssymb='';
    my $parmlev='';

    unless ($env{'form.parmlev'}) {
        $parmlev = 'map';
    } else {
        $parmlev = $env{'form.parmlev'};
    }

# ----------------------------------------------- Was this started from grades?

    if (($env{'form.command'} eq 'set') && ($env{'form.url'})
    && (!$env{'form.dis'})) {
        my $url=$env{'form.url'};
        $url=~s-^http://($ENV{'SERVER_NAME'}|$ENV{'HTTP_HOST'})--;
        $pssymb=&Apache::lonnet::symbread($url);
        if (!@pscat) { @pscat=('all'); }
        $pschp='';
        $parmlev = 'full';
    } elsif ($env{'form.symb'}) {
        $pssymb=$env{'form.symb'};
        if (!@pscat) { @pscat=('all'); }
        $pschp='';
        $parmlev = 'full';
    } else {
        $env{'form.url'}='';
    }

    my $id=$env{'form.id'};
    if (($id) && ($udom)) {
        $uname=(&Apache::lonnet::idget($udom,$id))[1];
        if ($uname) {
            $id='';
        } else {
            $message=
            '<span class="LC_error">'.&mt("Unknown ID")." '$id' ".
            &mt('at domain')." '$udom'</span>";
        }
    } else {
        $uname=$env{'form.uname'};
    }
    unless ($udom) { $uname=''; }
    $uhome='';
    if ($uname) {
        $uhome=&Apache::lonnet::homeserver($uname,$udom);
        if ($uhome eq 'no_host') {
            $message=
            '<span class="LC_error">'.&mt("Unknown user")." '$uname' ".
            &mt("at domain")." '$udom'</span>";
            $uname='';
        } else {
            $csec=&Apache::lonnet::getsection($udom,$uname,
                          $env{'request.course.id'});
            if ($csec eq '-1') {
                $message='<span class="LC_error">'.
                &mt("User")." '$uname' ".&mt("at domain")." '$udom' ".
                &mt("not in this course")."</span>";
                $uname='';
                $csec=$env{'form.csec'};
                $cgroup=$env{'form.cgroup'};
            } else {
                my %name=&Apache::lonnet::userenvironment($udom,$uname,
                  ('firstname','middlename','lastname','generation','id'));
                $message="\n<p>\n".&mt("Full Name").": ".
                $name{'firstname'}.' '.$name{'middlename'}.' '
                .$name{'lastname'}.' '.$name{'generation'}.
                "<br />\n".&mt('Student/Employee ID').": ".$name{'id'}.'<p>';
            }
            @usersgroups = &Apache::lonnet::get_users_groups(
                                       $udom,$uname,$env{'request.course.id'});
            if (@usersgroups > 0) {
                unless (grep(/^\Q$cgroup\E$/,@usersgroups)) {
                    $cgroup = $usersgroups[0];
                }
            }
        }
    }

    unless ($csec) { $csec=''; }
    unless ($cgroup) { $cgroup=''; }

# --------------------------------------------------------- Get all assessments
    &extractResourceInformation(\@ids, \%typep,\%keyp, \%allparms, \%allparts, \%allmaps,
                \%mapp, \%symbp,\%maptitles,\%uris,
                \%keyorder);

    $mapp{'0.0'} = '';
    $symbp{'0.0'} = '';

# ---------------------------------------------------------- Anything to store?
    if ($env{'form.pres_marker'}) {
        my @markers=split(/\&\&\&/,$env{'form.pres_marker'});
        my @values=split(/\&\&\&/,$env{'form.pres_value'});
        my @types=split(/\&\&\&/,$env{'form.pres_type'});
        my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
        my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
        my $chome = $env{'course.'.$env{'request.course.id'}.'.home'};
        my ($got_chostname,$chostname,$cmajor,$cminor);
        my $totalstored = 0;


        for (my $i=0;$i<=$#markers;$i++) {
            my ($needsrelease,$needsnewer,$name);
            if ($markers[$i] =~ /^[\d.]+\&0_availablestudent\&(1|2|3)$/) {
                my (@ok_slots,@fail_slots,@del_slots);
                my $courseopt=&Apache::lonnet::get_courseresdata($cnum,$cdom);
                my ($level,@all) =
                    &parmval_by_symb('0.availablestudent',$pssymb,'',$uname,$udom,
                                     $csec,$cgroup,$courseopt);
                foreach my $slot_name (split(/:/,$values[$i])) {
                    next if ($slot_name eq '');
                    if (&update_slots($slot_name,$cdom,$cnum,$pssymb,$uname,$udom) eq 'ok') {
                        push(@ok_slots,$slot_name);

                    } else {
                        push(@fail_slots,$slot_name);
                    }
                }
                if (@ok_slots) {
                    $values[$i] = join(':',@ok_slots);
                } else {
                    $values[$i] = '';
                }
                if ($all[$level] ne '') {
                    my @existing = split(/:/,$all[$level]);
                    foreach my $slot_name (@existing) {
                        if (!grep(/^\Q$slot_name\E$/,split(/:/,$values[$i]))) {
                            if (&delete_slots($slot_name,$cdom,$cnum,$uname,$udom,$pssymb) eq 'ok') {
                                push(@del_slots,$slot_name);
                            }
                        }
                    }
                }
            } elsif ($markers[$i] =~ /_(type|lenient|retrypartial|discussvote|examcode)\&\d+$/) {
                $name = $1;
                my $val = $values[$i];
                if ($name eq 'examcode') {
                   if (&Apache::lonnet::validCODE($values[$i])) {
                       $val = 'valid';
                   }
                }
                $needsrelease =
                    $Apache::lonnet::needsrelease{"parameter:$name:$val"};
                if ($needsrelease) {
                    unless ($got_chostname) {
                        ($chostname,$cmajor,$cminor) = &parameter_release_vars();
                        $got_chostname = 1;
                    }
                    $needsnewer = &parameter_releasecheck($name,$val,
                                                          $needsrelease,
                                                          $chostname,
                                                          $cmajor,$cminor);
                }
            }
            if ($needsnewer) {
                $message .= &oldversion_warning($name,$values[$i],$chostname,$cmajor,
                                                $cminor,$needsrelease);
            } else {
                $message.=&storeparm(split(/\&/,$markers[$i]),
                                     $values[$i],
                                     $types[$i],
                                     $uname,$udom,$csec,$cgroup);
                $totalstored ++;
            }
        }
# ---------------------------------------------------------------- Done storing
        if ($totalstored) {
            $message.='<p class="LC_warning">'
                     .&mt('Changes can take up to 10 minutes before being active for all students.')
                     .&Apache::loncommon::help_open_topic('Caching')
                     .'</p>';
        }
    }
#----------------------------------------------- if all selected, fill in array
    if ($pscat[0] eq "all") {@pscat = (keys %allparms);}
    if (!@pscat) { @pscat=('duedate','opendate','answerdate','weight','maxtries','type','problemstatus') };
    if ($psprt[0] eq "all" || !@psprt) {@psprt = (keys %allparts);}
# ------------------------------------------------------------------ Start page

    &startpage($r,$pssymb);

    foreach ('tolerance','date_default','date_start','date_end',
        'date_interval','int','float','string') {
        $r->print('<input type="hidden" value="'.
          &HTML::Entities::encode($env{'form.recent_'.$_},'"&<>').
          '" name="recent_'.$_.'" />');
    }

    # ----- Start Parameter Selection

    # Hide parm selection?
    $r->print(<<ENDPARMSELSCRIPT);
<script type="text/javascript">
// <![CDATA[
function parmsel_show() {
  document.getElementById('parmsel').style.display = "";
  document.getElementById('parmsellink').style.display = "none";
}
// ]]>
</script>
ENDPARMSELSCRIPT
    
    if (!$pssymb) {
        my $parmselhiddenstyle=' style="display:none"';
        if($env{'form.hideparmsel'} eq 'hidden') {
           $r->print('<div id="parmsel"'.$parmselhiddenstyle.'>');
        } else  {
           $r->print('<div id="parmsel">');
        }

        # Step 1
        $r->print(&Apache::lonhtmlcommon::topic_bar(1,&mt('Resource Specification'),'parmstep1'));
        $r->print('
<script type="text/javascript">
// <![CDATA['.
                 &showhide_js().'
// ]]>
</script>
');
        $r->print(&Apache::lonhtmlcommon::start_pick_box(undef,'parmlevel'));
        &levelmenu($r,\%alllevs,$parmlev);
        $r->print(&Apache::lonhtmlcommon::row_closure());
        &mapmenu($r,\%allmaps,$pschp,\%maptitles, \%symbp);
        $r->print(&Apache::lonhtmlcommon::row_closure());
        $r->print(&Apache::lonhtmlcommon::row_title(&mt('Select Parts to View')));
        &partmenu($r,\%allparts,\@psprt);
        $r->print(&Apache::lonhtmlcommon::row_closure(1));
        $r->print(&Apache::lonhtmlcommon::end_pick_box());

        # Step 2
        $r->print(&Apache::lonhtmlcommon::topic_bar(2,&mt('Parameter Specification'),'parmstep2'));
        &displaymenu($r,\%allparms,\@pscat,\@psprt,\%keyorder,'parmmenuscroll');

        # Step 3
        $r->print(&Apache::lonhtmlcommon::topic_bar(3,&mt('User Specification (optional)'),'parmstep3'));
        $r->print(&Apache::lonhtmlcommon::start_pick_box());
        &usermenu($r,$uname,$id,$udom,$csec,$cgroup,$parmlev,\@usersgroups);
        $r->print(&Apache::lonhtmlcommon::row_closure(1));
        $r->print(&Apache::lonhtmlcommon::end_pick_box());

        # Update Display Button
        $r->print('<p>'
             .'<input type="submit" name="dis"'
             .' value="'.&mt('Update Display').'" />'
             .'<input type="hidden" name="hideparmsel" value="hidden" />'
             .'</p>');
        $r->print('</div>');

        # Offer link to display parameter selection again
        $r->print('<p id="parmsellink"');
        if ($env{'form.hideparmsel'} ne 'hidden') {
           $r->print($parmselhiddenstyle);
        }
        $r->print('>'
             .'<a href="javascript:parmsel_show()">'
             .&mt('Change Parameter Selection')
             .'</a>'
             .'</p>');
    } else {
        $r->print();
        # parameter screen for a single resource. 
        my ($map,$iid,$resource)=&Apache::lonnet::decode_symb($pssymb);
        my $title = &Apache::lonnet::gettitle($pssymb);
        $r->print(&mt('Specific Resource: [_1] ([_2])',
                         $title,'<span class="LC_filename">'.$resource.'</span>').
                '<input type="hidden" value="'.$pssymb.'" name="symb" />'.
                  '<br />');
        $r->print(&Apache::lonhtmlcommon::topic_bar('',&mt('Additional Display Specification (optional)')));
        $r->print(&Apache::lonhtmlcommon::start_pick_box());
        $r->print(&Apache::lonhtmlcommon::row_title(&mt('Select Parts to View')).
                  '<label>'.
                  '<input type="checkbox" name="psprt" value="all"'.
                  ($env{'form.psprt'}?' checked="checked"':'').' />'.
                  &mt('Show all parts').
                  '</label></td></tr>');
        &usermenu($r,$uname,$id,$udom,$csec,$cgroup,$parmlev,\@usersgroups);
        $r->print(&Apache::lonhtmlcommon::row_closure(1));
        $r->print(&Apache::lonhtmlcommon::end_pick_box());
        $r->print('<p>'
             .'<input type="submit" name="dis"'
             .' value="'.&mt('Update Display').'" />'
             .'<input type="hidden" name="hideparmsel" value="hidden" />'
             .'</p>');
    }
    
    # ----- End Parameter Selection

    # Display Messages
    $r->print('<div>'.$message.'</div>');


    my @temp_pscat;
    map {
        my $cat = $_;
        push(@temp_pscat, map { $_.'.'.$cat } @psprt);
    } @pscat;

    @pscat = @temp_pscat;

    if (($env{'form.prevvisit'}) || ($pschp) || ($pssymb)) {
# ----------------------------------------------------------------- Start Table
        my @catmarker=map { tr|.|_|; 'parameter_'.$_; } @pscat;
        my $csuname=$env{'user.name'};
        my $csudom=$env{'user.domain'};

        if ($parmlev eq 'full') {
#
# This produces the cascading table output of parameters
#
               my $coursespan=$csec?8:5;
               my $userspan=3;
               if ($cgroup ne '') {
                  $coursespan += 3;
               }

               $r->print(&Apache::loncommon::start_data_table());
#
# This produces the headers
#
               $r->print('<tr><td colspan="5"></td>');
               $r->print('<th colspan="'.($coursespan).'">'.&mt('Any User').'</th>');
               if ($uname) {
                if (@usersgroups > 1) {
                       $userspan ++;
                   }
                   $r->print('<th colspan="'.$userspan.'" rowspan="2">');
                   $r->print(&mt("User")." $uname ".&mt('at Domain')." $udom</th>");
               }
               my %lt=&Apache::lonlocal::texthash(
                'pie'    => "Parameter in Effect",
                'csv'    => "Current Session Value",
                'rl'     => "Resource Level",
                'ic'     => 'in Course',
                'aut'    => "Assessment URL and Title",
                'type'   => 'Type',
                'emof'   => "Enclosing Map or Folder",
                'part'   => 'Part',
                'pn'     => 'Parameter Name',
                'def'    => 'default',
                'femof'  => 'from Enclosing Map or Folder',
                'gen'    => 'general',
                'foremf' => 'for Enclosing Map or Folder',
                'fr'     => 'for Resource'
            );
               $r->print(<<ENDTABLETWO);
<th rowspan="3">$lt{'pie'}</th>
<th rowspan="3">$lt{'csv'}<br />($csuname:$csudom)</th>
</tr><tr><td colspan="5"></td><th colspan="2">$lt{'ic'}</th><th colspan="2">$lt{'rl'}</th>
<th colspan="1">$lt{'ic'}</th>

ENDTABLETWO
               if ($csec) {
                   $r->print('<th colspan="3">'.
                  &mt("in Section")." $csec</th>");
               }
               if ($cgroup) {
                $r->print('<th colspan="3">'.
                &mt("in Group")." $cgroup</th>");
               }
               $r->print(<<ENDTABLEHEADFOUR);
</tr><tr><th>$lt{'aut'}</th><th>$lt{'type'}</th>
<th>$lt{'emof'}</th><th>$lt{'part'}</th><th>$lt{'pn'}</th>
<th>$lt{'gen'}</th><th>$lt{'foremf'}</th>
<th>$lt{'def'}</th><th>$lt{'femof'}</th><th>$lt{'fr'}</th>
ENDTABLEHEADFOUR

               if ($csec) {
                   $r->print('<th>'.&mt('general').'</th><th>'.&mt('for Enclosing Map or Folder').'</th><th>'.&mt('for Resource').'</th>');
               }

               if ($cgroup) {
                $r->print('<th>'.&mt('general').'</th><th>'.&mt('for Enclosing Map or Folder').'</th><th>'.&mt('for Resource').'</th>');
               }

               if ($uname) {
                if (@usersgroups > 1) {
                    $r->print('<th>'.&mt('Control by other group?').'</th>');
                   }
                   $r->print('<th>'.&mt('general').'</th><th>'.&mt('for Enclosing Map or Folder').'</th><th>'.&mt('for Resource').'</th>');
               }

               $r->print('</tr>');
#
# Done with the headers
# 
               my $defbgone='';
               my $defbgtwo='';
               my $defbgthree = '';

               foreach (@ids) {

                my $rid=$_;
                my ($inmapid)=($rid=~/\.(\d+)$/);

                if ((!$pssymb &&
                 (($pschp eq 'all') || ($allmaps{$pschp} eq $mapp{$rid})))
                ||
                ($pssymb && $pssymb eq $symbp{$rid})) {
# ------------------------------------------------------ Entry for one resource
                    if ($defbgone eq '#E0E099') {
                        $defbgone='#E0E0DD';
                    } else {
                        $defbgone='#E0E099';
                    }
                    if ($defbgtwo eq '#FFFF99') {
                        $defbgtwo='#FFFFDD';
                    } else {
                        $defbgtwo='#FFFF99';
                    }
                    if ($defbgthree eq '#FFBB99') {
                        $defbgthree='#FFBBDD';
                    } else {
                        $defbgthree='#FFBB99';
                    }

                    my $thistitle='';
                    my %name=   ();
                    undef %name;
                    my %part=   ();
                    my %display=();
                    my %type=   ();
                    my %default=();
                    my $uri=&Apache::lonnet::declutter($uris{$rid});

                    my $filter=$env{'form.filter'};
                    foreach (&keysplit($keyp{$rid})) {
                        my $tempkeyp = $_;
                        if (grep $_ eq $tempkeyp, @catmarker) {
                          my $parmname=&Apache::lonnet::metadata($uri,$_.'.name');
# We may only want certain parameters listed
                          if ($filter) {
                             unless ($filter=~/\Q$parmname\E/) { next; }
                          }
                          $name{$_}=$parmname;
                          $part{$_}=&Apache::lonnet::metadata($uri,$_.'.part');

                          my $parmdis=&Apache::lonnet::metadata($uri,$_.'.display');
                          if ($allparms{$name{$_}} ne '') {
                              my $identifier;
                              if ($parmdis =~ /(\s*\[Part.*)$/) {
                                  $identifier = $1;
                              }
                              $display{$_} = $allparms{$name{$_}}.$identifier;
                          } else {
                              $display{$_} = $parmdis;
                          }
                          unless ($display{$_}) { $display{$_}=''; }
                          $display{$_}.=' ('.$name{$_}.')';
                          $default{$_}=&Apache::lonnet::metadata($uri,$_);
                          $type{$_}=&Apache::lonnet::metadata($uri,$_.'.type');
                          $thistitle=&Apache::lonnet::metadata($uri,$_.'.title');
                        }
                    }
                    my $totalparms=scalar keys %name;
                    if ($totalparms>0) {
                           my $firstrow=1;
                        my $title=&Apache::lonnet::gettitle($symbp{$rid});
                        $r->print('<tr><td style="background-color:'.$defbgone.';"'.
                             ' rowspan='.$totalparms.
                             '><tt><font size="-1">'.
                             join(' / ',split(/\//,$uri)).
                             '</font></tt><p><b>'.
                             "<a href=\"javascript:openWindow('".
                          &Apache::lonnet::clutter($uri).'?symb='.
                          &escape($symbp{$rid}).
                             "', 'metadatafile', '450', '500', 'no', 'yes');\"".
                             " target=\"_self\">$title");

                        if ($thistitle) {
                            $r->print(' ('.$thistitle.')');
                        }
                        $r->print('</a></b></td>');
                        $r->print('<td style="background-color:'.$defbgtwo.';"'.
                                      ' rowspan='.$totalparms.'>'.$typep{$rid}.
                                      '</td>');

                        $r->print('<td style="background-color:'.$defbgone.';"'.
                                      ' rowspan='.$totalparms.
                                      '>'.$maptitles{$mapp{$rid}}.'</td>');
                        foreach (&keysinorder_bytype(\%name,\%keyorder)) {

                            unless ($firstrow) {
                                $r->print('<tr>');
                            } else {
                                undef $firstrow;
                            }
                            &print_row($r,$_,\%part,\%name,\%symbp,$rid,\%default,
                                       \%type,\%display,$defbgone,$defbgtwo,
                                       $defbgthree,$parmlev,$uname,$udom,$csec,
                                                            $cgroup,\@usersgroups);
                        }
                    }
                }
            } # end foreach ids
# -------------------------------------------------- End entry for one resource
            $r->print(&Apache::loncommon::end_data_table);
        } # end of  full
#--------------------------------------------------- Entry for parm level map
        if ($parmlev eq 'map') {
            my $defbgone = '#E0E099';
            my $defbgtwo = '#FFFF99';
            my $defbgthree = '#FFBB99';

            my %maplist;

            if ($pschp eq 'all') {
                %maplist = %allmaps;
            } else {
                %maplist = ($pschp => $mapp{$pschp});
            }

#-------------------------------------------- for each map, gather information
            my $mapid;
               foreach $mapid (sort {$maplist{$a} cmp $maplist{$b}} keys %maplist) {
                my $maptitle = $maplist{$mapid};

#-----------------------  loop through ids and get all parameter types for map
#-----------------------------------------          and associated information
                my %name = ();
                my %part = ();
                my %display = ();
                my %type = ();
                my %default = ();
                my $map = 0;

#        $r->print("Catmarker: @catmarker<br />\n");

                foreach (@ids) {
                    ($map)=(/([\d]*?)\./);
                      my $rid = $_;

#                  $r->print("$mapid:$map:   $rid <br /> \n");

                     if ($map eq $mapid) {
                        my $uri=&Apache::lonnet::declutter($uris{$rid});
#                    $r->print("Keys: $keyp{$rid} <br />\n");

#--------------------------------------------------------------------
# @catmarker contains list of all possible parameters including part #s
# $fullkeyp contains the full part/id # for the extraction of proper parameters
# $tempkeyp contains part 0 only (no ids - ie, subparts)
# When storing information, store as part 0
# When requesting information, request from full part
#-------------------------------------------------------------------
                        foreach (&keysplit($keyp{$rid})) {
                             my $tempkeyp = $_;
                              my $fullkeyp = $tempkeyp;
                              $tempkeyp =~ s/_\w+_/_0_/;

                              if ((grep $_ eq $fullkeyp, @catmarker) &&(!$name{$tempkeyp})) {
                                $part{$tempkeyp}="0";
                                $name{$tempkeyp}=&Apache::lonnet::metadata($uri,$fullkeyp.'.name');
                                my $parmdis=&Apache::lonnet::metadata($uri,$fullkeyp.'.display');
                                if ($allparms{$name{$tempkeyp}} ne '') {
                                    my $identifier;
                                    if ($parmdis =~ /(\s*\[Part.*)$/) {
                                        $identifier = $1;
                                    }
                                    $display{$tempkeyp} = $allparms{$name{$tempkeyp}}.$identifier;
                                } else {
                                    $display{$tempkeyp} = $parmdis;
                                }
                                unless ($display{$tempkeyp}) { $display{$tempkeyp}=''; }
                                $display{$tempkeyp}.=' ('.$name{$tempkeyp}.')';
                                $display{$tempkeyp} =~ s/_\w+_/_0_/;
                                $default{$tempkeyp}=&Apache::lonnet::metadata($uri,$fullkeyp);
                                $type{$tempkeyp}=&Apache::lonnet::metadata($uri,$fullkeyp.'.type');
                              }
                        } # end loop through keys
                      }
                } # end loop through ids

#---------------------------------------------------- print header information
                my $foldermap=&mt($maptitle=~/^uploaded/?'Folder':'Map');
                my $showtitle=$maptitles{$maptitle}.($maptitle!~/^uploaded/?' ['.$maptitle.']':'');
                my $tmp="";
                if ($uname) {
                    my $person=&Apache::loncommon::plainname($uname,$udom);
                    $tmp.=&mt("User")." <font color=\"red\"><i>$uname \($person\) </i></font> ".
                        &mt('in')." \n";
                } else {
                    $tmp.="<font color=\"red\"><i>".&mt('all').'</i></font> '.&mt('users in')." \n";
                }
                if ($cgroup) {
                    $tmp.=&mt("Group")." <font color=\"red\"><i>$cgroup".
                              "</i></font> ".&mt('of')." \n";
                    $csec = '';
                } elsif ($csec) {
                    $tmp.=&mt("Section")." <font color=\"red\"><i>$csec".
                              "</i></font> ".&mt('of')." \n";
                }
                $r->print('<div align="center"><h4>'
                         .&mt('Set Defaults for All Resources in [_1]Specifically for [_2][_3]'
                             ,$foldermap.'<br /><font color="red"><i>'.$showtitle.'</i></font><br />'
                             ,$tmp
                             ,'<font color="red"><i>'.$coursename.'</i></font>'
                             )
                         ."<br /></h4>\n"
                );
#---------------------------------------------------------------- print table
                $r->print('<p>'.&Apache::loncommon::start_data_table()
                         .&Apache::loncommon::start_data_table_header_row()
                         .'<th>'.&mt('Parameter Name').'</th>'
                         .'<th>'.&mt('Default Value').'</th>'
                         .'<th>'.&mt('Parameter in Effect').'</th>'
                         .&Apache::loncommon::end_data_table_header_row()
                );

                foreach (&keysinorder(\%name,\%keyorder)) {
                    $r->print(&Apache::loncommon::start_data_table_row());
                    &print_row($r,$_,\%part,\%name,\%symbp,$mapid,\%default,
                           \%type,\%display,$defbgone,$defbgtwo,$defbgthree,
                           $parmlev,$uname,$udom,$csec,$cgroup);
                }
                $r->print(&Apache::loncommon::end_data_table().'</p>'
                         .'</div>'
                );
            } # end each map
        } # end of $parmlev eq map
#--------------------------------- Entry for parm level general (Course level)
        if ($parmlev eq 'general') {
            my $defbgone = '#E0E099';
            my $defbgtwo = '#FFFF99';
            my $defbgthree = '#FFBB99';

#-------------------------------------------- for each map, gather information
            my $mapid="0.0";
#-----------------------  loop through ids and get all parameter types for map
#-----------------------------------------          and associated information
            my %name = ();
            my %part = ();
            my %display = ();
            my %type = ();
            my %default = ();

            foreach (@ids) {
                my $rid = $_;

                my $uri=&Apache::lonnet::declutter($uris{$rid});

#--------------------------------------------------------------------
# @catmarker contains list of all possible parameters including part #s
# $fullkeyp contains the full part/id # for the extraction of proper parameters
# $tempkeyp contains part 0 only (no ids - ie, subparts)
# When storing information, store as part 0
# When requesting information, request from full part
#-------------------------------------------------------------------
                foreach (&keysplit($keyp{$rid})) {
                    my $tempkeyp = $_;
                      my $fullkeyp = $tempkeyp;
                      $tempkeyp =~ s/_\w+_/_0_/;
                      if ((grep $_ eq $fullkeyp, @catmarker) &&(!$name{$tempkeyp})) {
                        $part{$tempkeyp}="0";
                        $name{$tempkeyp}=&Apache::lonnet::metadata($uri,$fullkeyp.'.name');
                        my $parmdis=&Apache::lonnet::metadata($uri,$fullkeyp.'.display');
                        if ($allparms{$name{$tempkeyp}} ne '') {
                            my $identifier;
                            if ($parmdis =~ /(\s*\[Part.*)$/) {
                                $identifier = $1;
                            }
                            $display{$tempkeyp} = $allparms{$name{$tempkeyp}}.$identifier;
                        } else {
                            $display{$tempkeyp} = $parmdis;
                        }
                        unless ($display{$tempkeyp}) { $display{$tempkeyp}=''; }
                        $display{$tempkeyp}.=' ('.$name{$tempkeyp}.')';
                        $display{$tempkeyp} =~ s/_\w+_/_0_/;
                        $default{$tempkeyp}=&Apache::lonnet::metadata($uri,$fullkeyp);
                        $type{$tempkeyp}=&Apache::lonnet::metadata($uri,$fullkeyp.'.type');
                      }
                } # end loop through keys
            } # end loop through ids

#---------------------------------------------------- print header information
            my $setdef=&mt("Set Defaults for All Resources in Course");
            $r->print(<<ENDMAPONE);
<center>
<h4>$setdef
<font color="red"><i>$coursename</i></font><br />
ENDMAPONE
            if ($uname) {
                my $person=&Apache::loncommon::plainname($uname,$udom);
                $r->print(" ".&mt("User")."<font color=\"red\"> <i>$uname \($person\) </i></font> \n");
            } else {
                $r->print("<i><font color=\"red\"> ".&mt("ALL")."</i> ".&mt("USERS")."</font> \n");
            }

            if ($csec) {$r->print(&mt("Section")."<font color=\"red\"> <i>$csec</i></font>\n")};
            if ($cgroup) {$r->print(&mt("Group")."<font color=\"red\"> <i>$cgroup</i></font>\n")};
            $r->print("</h4>\n");
#---------------------------------------------------------------- print table
            $r->print('<p>'.&Apache::loncommon::start_data_table()
                     .&Apache::loncommon::start_data_table_header_row()
                     .'<th>'.&mt('Parameter Name').'</th>'
                     .'<th>'.&mt('Default Value').'</th>'
                     .'<th>'.&mt('Parameter in Effect').'</th>'
                     .&Apache::loncommon::end_data_table_header_row()
            );

            foreach (&keysinorder(\%name,\%keyorder)) {
                $r->print(&Apache::loncommon::start_data_table_row());
                &print_row($r,$_,\%part,\%name,\%symbp,$mapid,\%default,
                       \%type,\%display,$defbgone,$defbgtwo,$defbgthree,
                                   $parmlev,$uname,$udom,$csec,$cgroup);
            }
            $r->print(&Apache::loncommon::end_data_table()
                     .'</p>'
                     .'</center>'
            );
        } # end of $parmlev eq general
    }
    $r->print('</form>');
    $r->print(&Apache::loncommon::end_page());
} # end sub assessparms

##################################################
# Overview mode
##################################################
my $tableopen;

sub tablestart {
    if ($tableopen) {
    return '';
    } else {
    $tableopen=1;
    return &Apache::loncommon::start_data_table().'<tr><th>'.&mt('Parameter').'</th><th>'.
        &mt('Delete').'</th><th>'.&mt('Set to ...').'</th></tr>';
    }
}

sub tableend {
    if ($tableopen) {
    $tableopen=0;
    return &Apache::loncommon::end_data_table();
    } else {
    return'';
    }
}

sub readdata {
    my ($crs,$dom)=@_;
# Read coursedata
    my $resourcedata=&Apache::lonnet::get_courseresdata($crs,$dom);
# Read userdata

    my $classlist=&Apache::loncoursedata::get_classlist();
    foreach (keys %$classlist) {
        if ($_=~/^($match_username)\:($match_domain)$/) {
        my ($tuname,$tudom)=($1,$2);
        my $useropt=&Apache::lonnet::get_userresdata($tuname,$tudom);
            foreach my $userkey (keys %{$useropt}) {
        if ($userkey=~/^$env{'request.course.id'}/) {
                    my $newkey=$userkey;
            $newkey=~s/^($env{'request.course.id'}\.)/$1\[useropt\:$tuname\:$tudom\]\./;
            $$resourcedata{$newkey}=$$useropt{$userkey};
        }
        }
    }
    }
    return $resourcedata;
}


# Setting

sub storedata {
    my ($r,$crs,$dom)=@_;
# Set userlevel immediately
# Do an intermediate store of course level
    my $olddata=&readdata($crs,$dom);
    my %newdata=();
    undef %newdata;
    my @deldata=();
    undef @deldata;
    my ($got_chostname,$chostname,$cmajor,$cminor);
    foreach my $key (keys(%env)) { 
    if ($key =~ /^form\.([a-z]+)\_(.+)$/) {
        my $cmd=$1;
        my $thiskey=$2;
        my ($tuname,$tudom)=&extractuser($thiskey);
        my $tkey=$thiskey;
            if ($tuname) {
        $tkey=~s/\.\[useropt\:$tuname\:$tudom\]\./\./;
        }
        if ($cmd eq 'set' || $cmd eq 'datepointer' || $cmd eq 'dateinterval') {
        my ($data, $typeof, $text, $name);
        if ($cmd eq 'set') {
            $data=$env{$key};
            $typeof=$env{'form.typeof_'.$thiskey};
            $text = &mt('Saved modified parameter for');
            if ($typeof eq 'string_questiontype') {
                $name = 'type';
            } elsif ($typeof eq 'string_lenient') {
                $name = 'lenient';
            } elsif ($typeof eq 'string_discussvote') {
                $name = 'discussvote';
            } elsif ($typeof eq 'string_examcode') {
                $name = 'examcode';
            } elsif ($typeof eq 'string_yesno') {
                if ($thiskey =~ /\.retrypartial$/) {
                    $name = 'retrypartial';
                }
            }
            if ($name ne '') {
                my ($needsrelease,$needsnewer);
                $needsrelease = $Apache::lonnet::needsrelease{"parameter:$name:$data"};
                if ($needsrelease) {
                    unless ($got_chostname) {
                        ($chostname,$cmajor,$cminor)=&parameter_release_vars();
                        $got_chostname = 1;
                    }
                    $needsnewer = &parameter_releasecheck($name,$data,
                                                          $needsrelease,
                                                          $chostname,$cmajor,
                                                          $cminor);
                }
                if ($needsnewer) {
                    $r->print('<br />'.&oldversion_warning($name,$data,
                                                           $chostname,$cmajor,
                                                           $cminor,$needsrelease));
                    next;
                }
            }
        } elsif ($cmd eq 'datepointer') {
            $data=&Apache::lonhtmlcommon::get_date_from_form($env{$key});
            $typeof=$env{'form.typeof_'.$thiskey};
            $text = &mt('Saved modified date for');
        } elsif ($cmd eq 'dateinterval') {
            $data=&get_date_interval_from_form($thiskey);
            $typeof=$env{'form.typeof_'.$thiskey};
            $text = &mt('Saved modified date for');
        }
        if (defined($data) and $$olddata{$thiskey} ne $data) {
            if ($tuname) {
            if (&Apache::lonnet::put('resourcedata',{$tkey=>$data,
                                 $tkey.'.type' => $typeof},
                         $tudom,$tuname) eq 'ok') {
                &log_parmset({$tkey=>$data,$tkey.'.type' => $typeof},0,$tuname,$tudom);
                $r->print('<br />'.$text.' '.
                      &Apache::loncommon::plainname($tuname,$tudom));
            } else {
                $r->print('<div class="LC_error">'.
                      &mt('Error saving parameters').'</div>');
            }
            &Apache::lonnet::devalidateuserresdata($tuname,$tudom);
            } else {
            $newdata{$thiskey}=$data;
             $newdata{$thiskey.'.type'}=$typeof;
                   }
        }
        } elsif ($cmd eq 'del') {
        if ($tuname) {
            if (&Apache::lonnet::del('resourcedata',[$tkey],$tudom,$tuname) eq 'ok') {
                &log_parmset({$tkey=>''},1,$tuname,$tudom);
            $r->print('<br />'.&mt('Deleted parameter for').' '.&Apache::loncommon::plainname($tuname,$tudom));
            } else {
            $r->print('<div class="LC_error">'.
                  &mt('Error deleting parameters').'</div>');
            }
            &Apache::lonnet::devalidateuserresdata($tuname,$tudom);
        } else {
            push (@deldata,$thiskey,$thiskey.'.type');
        }
        }
    }
    }
# Store all course level
    my $delentries=$#deldata+1;
    my @newdatakeys=keys %newdata;
    my $putentries=$#newdatakeys+1;
    if ($delentries) {
    if (&Apache::lonnet::del('resourcedata',\@deldata,$dom,$crs) eq 'ok') {
        my %loghash=map { $_ => '' } @deldata;
        &log_parmset(\%loghash,1);
        $r->print('<h2>'.&mt('Deleted [_1] parameter(s)',$delentries).'</h2>');
    } else {
        $r->print('<div class="LC_error">'.
              &mt('Error deleting parameters').'</div>');
    }
    &Apache::lonnet::devalidatecourseresdata($crs,$dom);
    }
    if ($putentries) {
    if (&Apache::lonnet::put('resourcedata',\%newdata,$dom,$crs) eq 'ok') {
                &log_parmset(\%newdata,0);
        $r->print('<h3>'.&mt('Saved [_1] parameter(s)',$putentries/2).'</h3>');
    } else {
        $r->print('<div class="LC_error">'.
              &mt('Error saving parameters').'</div>');
    }
    &Apache::lonnet::devalidatecourseresdata($crs,$dom);
    }
}

sub extractuser {
    my $key=shift;
    return ($key=~/^$env{'request.course.id'}.\[useropt\:($match_username)\:($match_domain)\]\./);
}

sub parse_listdata_key {
    my ($key,$listdata) = @_;
    # split into student/section affected, and
    # the realm (folder/resource part and parameter
    my ($student,$realm) =
    ($key=~/^\Q$env{'request.course.id'}\E\.\[([^\.]+)\]\.(.+)$/);
    # if course wide student would be undefined
    if (!defined($student)) {
    ($realm)=($key=~/^\Q$env{'request.course.id'}\E\.(.+)$/);
    }
    # strip off the .type if it's not the Question type parameter
    if ($realm=~/\.type$/ && !exists($listdata->{$key.'.type'})) {
    $realm=~s/\.type//;
    }
    # split into resource+part and parameter name
    my ($res,    $parm) = ($realm=~/^(.*)\.(.*)$/);
       ($res, my $part) = ($res  =~/^(.*)\.(.*)$/);
    return ($student,$res,$part,$parm);
}

sub listdata {
    my ($r,$resourcedata,$listdata,$sortorder)=@_;
# Start list output

    my $oldsection='';
    my $oldrealm='';
    my $oldpart='';
    my $pointer=0;
    $tableopen=0;
    my $foundkeys=0;
    my %keyorder=&standardkeyorder();

    foreach my $thiskey (sort {
    my ($astudent,$ares,$apart,$aparm) = &parse_listdata_key($a,$listdata);
    my ($bstudent,$bres,$bpart,$bparm) = &parse_listdata_key($b,$listdata);

    # get the numerical order for the param
    $aparm=$keyorder{'parameter_0_'.$aparm};
    $bparm=$keyorder{'parameter_0_'.$bparm};

    my $result=0;

    if ($sortorder eq 'realmstudent') {
            if ($ares     ne $bres    ) {
        $result = ($ares     cmp $bres);
            } elsif ($astudent ne $bstudent) {
        $result = ($astudent cmp $bstudent);
        } elsif ($apart    ne $bpart   ) {
        $result = ($apart    cmp $bpart);
        }
    } else {
        if      ($astudent ne $bstudent) {
        $result = ($astudent cmp $bstudent);
        } elsif ($ares     ne $bres    ) {
        $result = ($ares     cmp $bres);
        } elsif ($apart    ne $bpart   ) {
        $result = ($apart    cmp $bpart);
        }
    }

    if (!$result) {
            if (defined($aparm) && defined($bparm)) {
        $result = ($aparm <=> $bparm);
            } elsif (defined($aparm)) {
        $result = -1;
            } elsif (defined($bparm)) {
        $result = 1;
        }
    }

    $result;
    } keys %{$listdata}) {

    if ($$listdata{$thiskey.'.type'}) {
            my $thistype=$$listdata{$thiskey.'.type'};
            if ($$resourcedata{$thiskey.'.type'}) {
        $thistype=$$resourcedata{$thiskey.'.type'};
        }
        my ($middle,$part,$name)=
        ($thiskey=~/^$env{'request.course.id'}\.(?:(.+)\.)*([\w\s]+)\.(\w+)$/);
        my $section=&mt('All Students');
        if ($middle=~/^\[(.*)\]/) {
        my $issection=$1;
        if ($issection=~/^useropt\:($match_username)\:($match_domain)/) {
            $section=&mt('User').": ".&Apache::loncommon::plainname($1,$2);
        } else {
            $section=&mt('Group/Section').': '.$issection;
        }
        $middle=~s/^\[(.*)\]//;
        }
        $middle=~s/\.+$//;
        $middle=~s/^\.+//;
        my $realm='<span class="LC_parm_scope_all">'.&mt('All Resources').'</span>';
        if ($middle=~/^(.+)\_\_\_\(all\)$/) {
        $realm='<span class="LC_parm_scope_folder">'.&mt('Folder/Map').': '.&Apache::lonnet::gettitle($1).' <br /><span class="LC_parm_folder">('.$1.')</span></span>';
        } elsif ($middle) {
        my ($map,$id,$url)=&Apache::lonnet::decode_symb($middle);
        $realm='<span class="LC_parm_scope_resource">'.&mt('Resource').': '.&Apache::lonnet::gettitle($middle).' <br /><span class="LC_parm_symb">('.$url.' in '.$map.' id: '.$id.')</span></span>';
        }
        if ($sortorder eq 'realmstudent') {
        if ($realm ne $oldrealm) {
            $r->print(&tableend()."\n<hr /><h1>$realm</h1>");
            $oldrealm=$realm;
            $oldsection='';
        }
        if ($section ne $oldsection) {
            $r->print(&tableend()."\n<h2>$section</h2>");
            $oldsection=$section;
            $oldpart='';
        }
        } else {
        if ($section ne $oldsection) {
            $r->print(&tableend()."\n<hr /><h1>$section</h1>");
            $oldsection=$section;
            $oldrealm='';
        }
        if ($realm ne $oldrealm) {
            $r->print(&tableend()."\n<h2>$realm</h2>");
            $oldrealm=$realm;
            $oldpart='';
        }
        }
        if ($part ne $oldpart) {
        $r->print(&tableend().
              "\n".'<span class="LC_parm_part">'.&mt('Part').": $part</span>");
        $oldpart=$part;
        }
#
# Ready to print
#
            my $parmitem = &standard_parameter_names($name);
        $r->print(&tablestart().
              &Apache::loncommon::start_data_table_row().
              '<td><b>'.&mt($parmitem).
              '</b></td><td><input type="checkbox" name="del_'.
              $thiskey.'" /></td><td>');
        $foundkeys++;
        if (&isdateparm($thistype)) {
        my $jskey='key_'.$pointer;
        $pointer++;
        $r->print(
              &Apache::lonhtmlcommon::date_setter('parmform',
                                  $jskey,
                              $$resourcedata{$thiskey},
                                  '',1,'','').
'<input type="hidden" name="datepointer_'.$thiskey.'" value="'.$jskey.'" />'.
(($$resourcedata{$thiskey}!=0)?'<span class="LC_nobreak"><a href="/adm/parmset?&action=dateshift1&timebase='.$$resourcedata{$thiskey}.'">'.
&mt('Shift all dates based on this date').'</a></span>':'').
&date_sanity_info($$resourcedata{$thiskey})
              );
        } elsif ($thistype eq 'date_interval') {
        $r->print(&date_interval_selector($thiskey,
                          $$resourcedata{$thiskey}));
        } elsif ($thistype =~ m/^string/) {
        $r->print(&string_selector($thistype,$thiskey,
                       $$resourcedata{$thiskey},$name));
        } else {
        $r->print(&default_selector($thiskey,$$resourcedata{$thiskey}));
        }
        $r->print('<input type="hidden" name="typeof_'.$thiskey.'" value="'.
              $thistype.'" />');
        $r->print('</td>'.&Apache::loncommon::end_data_table_row());
    }
    }
    return $foundkeys;
}


sub date_interval_selector {
    my ($thiskey, $showval) = @_;
    my $result;
    foreach my $which (['days', 86400, 31],
               ['hours', 3600, 23],
               ['minutes', 60, 59],
               ['seconds',  1, 59]) {
    my ($name, $factor, $max) = @{ $which };
    my $amount = int($showval/$factor);
    $showval  %= $factor;
    my %select = ((map {$_ => $_} (0..$max)),
              'select_form_order' => [0..$max]);
    $result .= &Apache::loncommon::select_form($amount,$name.'_'.$thiskey,
                           \%select);
    $result .= ' '.&mt($name);
    }
    $result .= '<input type="hidden" name="dateinterval_'.$thiskey.'" />';
    return $result;

}

sub get_date_interval_from_form {
    my ($key) = @_;
    my $seconds = 0;
    foreach my $which (['days', 86400],
               ['hours', 3600],
               ['minutes', 60],
               ['seconds',  1]) {
    my ($name, $factor) = @{ $which };
    if (defined($env{'form.'.$name.'_'.$key})) {
        $seconds += $env{'form.'.$name.'_'.$key} * $factor;
    }
    }
    return $seconds;
}


sub default_selector {
    my ($thiskey, $showval) = @_;
    return '<input type="text" name="set_'.$thiskey.'" value="'.$showval.'" />';
}

my %strings =
    (
     'string_yesno'
             => [[ 'yes', 'Yes' ],
         [ 'no', 'No' ]],
     'string_problemstatus'
             => [[ 'yes', 'Yes' ],
         [ 'answer', 'Yes, and show correct answer if they exceed the maximum number of tries.' ],
         [ 'no', 'No, don\'t show correct/incorrect feedback.' ],
         [ 'no_feedback_ever', 'No, show no feedback at all.' ]],
     'string_questiontype'
             => [[ 'problem', 'Standard Problem'],
                 [ 'survey', 'Survey'],
                 [ 'anonsurveycred', 'Anonymous Survey (credit for submission)'],
                 [ 'exam', 'Exam'],
                 [ 'anonsurvey', 'Anonymous Survey'],
                 [ 'randomizetry', 'New Randomization Each N Tries (default N=1)'],
                 [ 'practice', 'Practice'],
                 [ 'surveycred', 'Survey (credit for submission)']],
     'string_lenient'
             => [['yes', 'Yes' ],
                 [ 'no', 'No' ],
                 [ 'default', 'Default - only bubblesheet grading is lenient' ]],
     'string_discussvote'
             => [['yes','Yes'],
                 ['notended','Yes, unless discussion ended'],
                 ['no','No']],
     );

sub standard_string_options {
    my ($string_type) = @_;
    if (ref($strings{$string_type}) eq 'ARRAY') {
        return $strings{$string_type};
    }
    return;
}

sub string_selector {
    my ($thistype, $thiskey, $showval, $name) = @_;

    if (!exists($strings{$thistype})) {
    return &default_selector($thiskey,$showval);
    }

    my %skiptype;
    if (($thistype eq 'string_questiontype') || 
        ($thistype eq 'string_lenient') ||
        ($thistype eq 'string_discussvote') ||
        ($name eq 'retrypartial')) {
        my ($got_chostname,$chostname,$cmajor,$cminor); 
        foreach my $possibilities (@{ $strings{$thistype} }) {
            next unless (ref($possibilities) eq 'ARRAY');
            my ($parmval, $description) = @{ $possibilities };
            my $needsrelease=$Apache::lonnet::needsrelease{"parameter:$name:$parmval"};
            if ($needsrelease) {
                unless ($got_chostname) {
                    ($chostname,$cmajor,$cminor)=&parameter_release_vars();
                    $got_chostname = 1;
                }
                my $needsnewer=&parameter_releasecheck($name,$parmval,$needsrelease,
                                                       $chostname,$cmajor,
                                                       $cminor);
                if ($needsnewer) {
                    $skiptype{$parmval} = 1;
                }
            }
        }
    }
    

    my $result;
    my $numinrow = 3;
    if ($thistype eq 'string_problemstatus') {
        $numinrow = 2;
    } elsif ($thistype eq 'string_questiontype') {
        if (keys(%skiptype) > 0) {
             $numinrow = 4;
        }
    }
    my $rem;
    if (ref($strings{$thistype}) eq 'ARRAY') {
        my $i=0;
        foreach my $possibilities (@{ $strings{$thistype} }) {
            next unless (ref($possibilities) eq 'ARRAY');
            my ($name, $description) = @{ $possibilities };
            next if ($skiptype{$name}); 
            $rem = $i%($numinrow);
            if ($rem == 0) {
                if ($i > 0) {
                    $result .= '</tr>';
                }
                $result .= '<tr>';
            }
            $result .= '<td class="LC_left_item">'.
                       '<span class="LC_nobreak"><label>'.
                       '<input type="radio" name="set_'.$thiskey.
                       '" value="'.$name.'"';
            if ($showval eq $name) {
                $result .= ' checked="checked"';
            }
            $result .= ' />'.&mt($description).'</label></span></td>';
            $i++;
        }
        $rem = @{ $strings{$thistype} }%($numinrow);
        my $colsleft = $numinrow - $rem;
        if ($colsleft > 1 ) {
            $result .= '<td colspan="'.$colsleft.'" class="LC_left_item">'.
                       '&nbsp;</td>';
        } elsif ($colsleft == 1) {
            $result .= '<td class="LC_left_item">&nbsp;</td>';
        }
        $result .= '</tr>';
    }
    if ($result) {
        $result = '<table border="0">'.$result.'</table>';
    }
    return $result;
}

#
# Shift all start and end dates by $shift
#

sub dateshift {
    my ($shift)=@_;
    my $dom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $crs = $env{'course.'.$env{'request.course.id'}.'.num'};
    my %data=&Apache::lonnet::dump('resourcedata',$dom,$crs);
# ugly retro fix for broken version of types
    foreach my $key (keys %data) {
        if ($key=~/\wtype$/) {
            my $newkey=$key;
            $newkey=~s/type$/\.type/;
            $data{$newkey}=$data{$key};
            delete $data{$key};
        }
    }
    my %storecontent=();
# go through all parameters and look for dates
    foreach my $key (keys %data) {
       if ($data{$key.'.type'}=~/^date_(start|end)$/) {
          my $newdate=$data{$key}+$shift;
          $storecontent{$key}=$newdate;
       }
    }
    my $reply=&Apache::lonnet::cput
                ('resourcedata',\%storecontent,$dom,$crs);
    if ($reply eq 'ok') {
       &log_parmset(\%storecontent);
    }
    &Apache::lonnet::devalidatecourseresdata($crs,$dom);
    return $reply;
}

sub newoverview {
    my ($r) = @_;

    my $dom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $crs = $env{'course.'.$env{'request.course.id'}.'.num'};
    &Apache::lonhtmlcommon::add_breadcrumb({href=>'/adm/parmset?action=setoverview',
        text=>"Overview Mode"});

    my %loaditems = (
                      'onload'   => "showHide_courseContent(); resize_scrollbox('mapmenuscroll','1','1');",
                    );
    my $js = '
<script type="text/javascript">
// <![CDATA[
'.
            &Apache::lonhtmlcommon::resize_scrollbox_js('params')."\n".
            &showhide_js()."\n".
'// ]]>
</script>
';
    my $start_page = &Apache::loncommon::start_page('Set Parameters',$js,
                                                    {'add_entries' => \%loaditems,});
    my $breadcrumbs = &Apache::lonhtmlcommon::breadcrumbs('Overview');
    $r->print($start_page.$breadcrumbs);
    $r->print(<<ENDOVER);
<form method="post" action="/adm/parmset?action=newoverview" name="parmform">
ENDOVER
    my @ids=();
    my %typep=();
    my %keyp=();
    my %allparms=();
    my %allparts=();
    my %allmaps=();
    my %mapp=();
    my %symbp=();
    my %maptitles=();
    my %uris=();
    my %keyorder=&standardkeyorder();
    my %defkeytype=();

    my %alllevs=();
    $alllevs{'Resource Level'}='full';
    $alllevs{'Map/Folder Level'}='map';
    $alllevs{'Course Level'}='general';

    my $csec=$env{'form.csec'};
    my $cgroup=$env{'form.cgroup'};

    my @pscat=&Apache::loncommon::get_env_multiple('form.pscat');
    my $pschp=$env{'form.pschp'};

    my @psprt=&Apache::loncommon::get_env_multiple('form.psprt');
    if (!@psprt) { $psprt[0]='0'; }

    my @selected_sections =
    &Apache::loncommon::get_env_multiple('form.Section');
    @selected_sections = ('all') if (! @selected_sections);
    foreach my $sec (@selected_sections) {
        if ($sec eq 'all') {
            @selected_sections = ('all');
        }
    }
    my @selected_groups =
        &Apache::loncommon::get_env_multiple('form.Group');

    my $pssymb='';
    my $parmlev='';

    unless ($env{'form.parmlev'}) {
        $parmlev = 'map';
    } else {
        $parmlev = $env{'form.parmlev'};
    }

    &extractResourceInformation(\@ids, \%typep,\%keyp, \%allparms, \%allparts, \%allmaps,
                \%mapp, \%symbp,\%maptitles,\%uris,
                \%keyorder,\%defkeytype);

    if (grep {$_ eq 'all'} (@psprt)) {
        @psprt = keys(%allparts);
    }
# Menu to select levels, etc

    $r->print('<div class="LC_Box">');
    #$r->print('<h2 class="LC_hcell">Step 1</h2>');
    $r->print('<div>');
    $r->print(&Apache::lonhtmlcommon::start_pick_box(undef,'parmlevel'));
    &levelmenu($r,\%alllevs,$parmlev);
    if ($parmlev ne 'general') {
        $r->print(&Apache::lonhtmlcommon::row_closure());
        &mapmenu($r,\%allmaps,$pschp,\%maptitles,\%symbp);
    }
    $r->print(&Apache::lonhtmlcommon::row_closure(1));
    $r->print(&Apache::lonhtmlcommon::end_pick_box());
    $r->print('</div></div>');

    $r->print('<div class="LC_Box">');
    $r->print('<div>');
    &displaymenu($r,\%allparms,\@pscat,\%keyorder);
    $r->print(&Apache::lonhtmlcommon::start_pick_box());
    $r->print(&Apache::lonhtmlcommon::row_title(&mt('Select Parts to View')));
    $r->print('<table>'.
              '<tr><th>'.&mt('Parts').'</th><th>'.&mt('Section(s)').
              '</th><th>'.&mt('Group(s)').'</th></tr><tr><td>');
    &partmenu($r,\%allparts,\@psprt);
    $r->print('</td><td>');
    &sectionmenu($r,\@selected_sections);
    $r->print('</td><td>');
    &groupmenu($r,\@selected_groups);
    $r->print('</td></tr></table>');
    #$r->print('</td></tr></table>');
    $r->print(&Apache::lonhtmlcommon::row_closure(1));
    $r->print(&Apache::lonhtmlcommon::end_pick_box());
    $r->print('</div></div>');

    $r->print('<div class="LC_Box">');
    $r->print('<div>');
    my $sortorder=$env{'form.sortorder'};
    unless ($sortorder) { $sortorder='realmstudent'; }
    &sortmenu($r,$sortorder);
    $r->print('</div></div>');

    $r->print('<p><input type="submit" name="dis" value="'.&mt('Display').'" /></p>');

# Build the list data hash from the specified parms

    my $listdata;
    %{$listdata}=();

    foreach my $cat (@pscat) {
        &secgroup_lister($cat,$pschp,$parmlev,$listdata,\@psprt,\@selected_sections,\%defkeytype,\%allmaps,\@ids,\%symbp);
        &secgroup_lister($cat,$pschp,$parmlev,$listdata,\@psprt,\@selected_groups,\%defkeytype,\%allmaps,\@ids,\%symbp);
    }

    if (($env{'form.store'}) || ($env{'form.dis'})) {

        if ($env{'form.store'}) { &storedata($r,$crs,$dom); }

# Read modified data

        my $resourcedata=&readdata($crs,$dom);

# List data

        &listdata($r,$resourcedata,$listdata,$sortorder);
    }
    $r->print(&tableend().
         ((($env{'form.store'}) || ($env{'form.dis'}))?'<p><input type="submit" name="store" value="'.&mt('Save').'" /></p>':'').
          '</form>');
    $r->print(&Apache::loncommon::end_page());
}

sub secgroup_lister {
    my ($cat,$pschp,$parmlev,$listdata,$psprt,$selections,$defkeytype,$allmaps,$ids,$symbp) = @_;
    foreach my $item (@{$selections}) {
        foreach my $part (@{$psprt}) {
            my $rootparmkey=$env{'request.course.id'};
            if (($item ne 'all') && ($item ne 'none') && ($item)) {
                $rootparmkey.='.['.$item.']';
            }
            if ($parmlev eq 'general') {
# course-level parameter
                my $newparmkey=$rootparmkey.'.'.$part.'.'.$cat;
                $$listdata{$newparmkey}=1;
                $$listdata{$newparmkey.'.type'}=$$defkeytype{$cat};
            } elsif ($parmlev eq 'map') {
# map-level parameter
                foreach my $mapid (keys %{$allmaps}) {
                    if (($pschp ne 'all') && ($pschp ne $mapid)) { next; }
                    my $newparmkey=$rootparmkey.'.'.$$allmaps{$mapid}.'___(all).'.$part.'.'.$cat;
                    $$listdata{$newparmkey}=1;
                    $$listdata{$newparmkey.'.type'}=$$defkeytype{$cat};
                }
            } else {
# resource-level parameter
                foreach my $rid (@{$ids}) {
                    my ($map,$resid,$url)=&Apache::lonnet::decode_symb($$symbp{$rid});
                    if (($pschp ne 'all') && ($$allmaps{$pschp} ne $map)) { next; }
                    my $newparmkey=$rootparmkey.'.'.$$symbp{$rid}.'.'.$part.'.'.$cat;
                    $$listdata{$newparmkey}=1;
                    $$listdata{$newparmkey.'.type'}=$$defkeytype{$cat};
                }
            }
        }
    }
}

sub overview {
    my ($r) = @_;
    my $dom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $crs = $env{'course.'.$env{'request.course.id'}.'.num'};

    &Apache::lonhtmlcommon::add_breadcrumb({href=>'/adm/parmset?action=setoverview',
    text=>"Overview Mode"});
    my $start_page=&Apache::loncommon::start_page('Modify Parameters');
    my $breadcrumbs = &Apache::lonhtmlcommon::breadcrumbs('Overview');
    $r->print($start_page.$breadcrumbs);
    $r->print('<form method="post" action="/adm/parmset?action=setoverview" name="parmform">');

# Store modified

    &storedata($r,$crs,$dom);

# Read modified data

    my $resourcedata=&readdata($crs,$dom);


    my $sortorder=$env{'form.sortorder'};
    unless ($sortorder) { $sortorder='realmstudent'; }
    &sortmenu($r,$sortorder);

# List data

    my $foundkeys=&listdata($r,$resourcedata,$resourcedata,$sortorder);

    $r->print(&tableend().'<p>'.
    ($foundkeys?'<input type="submit" value="'.&mt('Save').'" />':'<span class="LC_info">'.&mt('There are no parameters.').'</span>').'</p></form>'.
          &Apache::loncommon::end_page());
}

sub clean_parameters {
    my ($r) = @_;
    my $dom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $crs = $env{'course.'.$env{'request.course.id'}.'.num'};

    &Apache::lonhtmlcommon::add_breadcrumb({href=>'/adm/parmset?action=cleanparameters',
        text=>"Clean Parameters"});
    my $start_page=&Apache::loncommon::start_page('Clean Parameters');
    my $breadcrumbs = &Apache::lonhtmlcommon::breadcrumbs('Clean');
    $r->print(<<ENDOVER);
$start_page
$breadcrumbs
<form method="post" action="/adm/parmset?action=cleanparameters" name="parmform">
ENDOVER
# Store modified

    &storedata($r,$crs,$dom);

# Read modified data

    my $resourcedata=&readdata($crs,$dom);

# List data

    $r->print('<h3>'.
          &mt('These parameters refer to resources that do not exist.').
          '</h3>'.
          '<input type="submit" value="'.&mt('Delete Selected').'" />'.'<br />'.
          '<br />');
    $r->print(&Apache::loncommon::start_data_table().
          '<tr>'.
          '<th>'.&mt('Delete').'</th>'.
          '<th>'.&mt('Parameter').'</th>'.
          '</tr>');
    foreach my $thiskey (sort(keys(%{$resourcedata}))) {
    next if (!exists($resourcedata->{$thiskey.'.type'})
         && $thiskey=~/\.type$/);
    my %data = &parse_key($thiskey);
    if (1) { #exists($data{'realm_exists'})
        #&& !$data{'realm_exists'}) {
        $r->print(&Apache::loncommon::start_data_table_row().
              '<tr>'.
              '<td><input type="checkbox" name="del_'.$thiskey.'" /></td>'              );

        $r->print('<td>');
        my $display_value = $resourcedata->{$thiskey};
        if (&isdateparm($resourcedata->{$thiskey.'.type'})) {
        $display_value =
            &Apache::lonlocal::locallocaltime($display_value);
        }
            my $parmitem = &standard_parameter_names($data{'parameter_name'});
            $parmitem = &mt($parmitem);
        $r->print(&mt('Parameter: "[_1]" with value: "[_2]"',
              $parmitem,$resourcedata->{$thiskey}));
        $r->print('<br />');
        if ($data{'scope_type'} eq 'all') {
        $r->print(&mt('All users'));
        } elsif ($data{'scope_type'} eq 'user') {
        $r->print(&mt('User: [_1]',join(':',@{$data{'scope'}})));
        } elsif ($data{'scope_type'} eq 'section') {
        $r->print(&mt('Section: [_1]',$data{'scope'}));
        } elsif ($data{'scope_type'} eq 'group') {
        $r->print(&mt('Group: [_1]',$data{'scope'}));
        }
        $r->print('<br />');
        if ($data{'realm_type'} eq 'all') {
        $r->print(&mt('All Resources'));
        } elsif ($data{'realm_type'} eq 'folder') {
        $r->print(&mt('Folder: [_1]'),$data{'realm'});
        } elsif ($data{'realm_type'} eq 'symb') {
        my ($map,$resid,$url) =
            &Apache::lonnet::decode_symb($data{'realm'});
        $r->print(&mt('Resource: [_1]with ID: [_2]in folder [_3]',
                  $url.' <br />&nbsp;&nbsp;&nbsp;',
                  $resid.' <br />&nbsp;&nbsp;&nbsp;',$map));
        }
        $r->print(' <br />&nbsp;&nbsp;&nbsp;'.&mt('Part: [_1]',$data{'parameter_part'}));
        $r->print('</td></tr>');

    }
    }
    $r->print(&Apache::loncommon::end_data_table().'<p>'.
          '<input type="submit" value="'.&mt('Delete Selected').'" />'.
          '</p></form>');
    $r->print(&Apache::loncommon::end_page());
}

sub date_shift_one {
    my ($r) = @_;
    my $dom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $crs = $env{'course.'.$env{'request.course.id'}.'.num'};

    &Apache::lonhtmlcommon::add_breadcrumb({href=>'/adm/parmset?action=dateshift1&timebase='.$env{'form.timebase'},
        text=>"Shifting Dates"});
    my $start_page=&Apache::loncommon::start_page('Shift Dates');
    my $breadcrumbs = &Apache::lonhtmlcommon::breadcrumbs('Shift');
    $r->print($start_page.$breadcrumbs);
    $r->print('<form name="shiftform" method="post" action="">'.
              '<table><tr><td>'.&mt('Currently set date:').'</td><td>'.
              &Apache::lonlocal::locallocaltime($env{'form.timebase'}).'</td></tr>'.
              '<tr><td>'.&mt('Shifted date:').'</td><td>'.
                    &Apache::lonhtmlcommon::date_setter('shiftform',
                                                        'timeshifted',
                                                        $env{'form.timebase'},,
                                                        '').
              '</td></tr></table>'.
              '<input type="hidden" name="action" value="dateshift2" />'.
              '<input type="hidden" name="timebase" value="'.$env{'form.timebase'}.'" />'.
              '<input type="submit" value="'.&mt('Shift all dates accordingly').'" /></form>');
    $r->print(&Apache::loncommon::end_page());
}

sub date_shift_two {
    my ($r) = @_;
    my $dom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $crs = $env{'course.'.$env{'request.course.id'}.'.num'};
    &Apache::lonhtmlcommon::add_breadcrumb({href=>'/adm/parmset?action=dateshift1&timebase='.$env{'form.timebase'},
        text=>"Shifting Dates"});
    my $start_page=&Apache::loncommon::start_page('Shift Dates');
    my $breadcrumbs = &Apache::lonhtmlcommon::breadcrumbs('Shift');
    $r->print($start_page.$breadcrumbs);
    my $timeshifted=&Apache::lonhtmlcommon::get_date_from_form('timeshifted');
    $r->print('<h2>'.&mt('Shift Dates').'</h2>'.
              '<p>'.&mt('Shifting all dates such that [_1] becomes [_2]',
              &Apache::lonlocal::locallocaltime($env{'form.timebase'}),
              &Apache::lonlocal::locallocaltime($timeshifted)).'</p>');
    my $delta=$timeshifted-$env{'form.timebase'};
    &dateshift($delta);
    $r->print(
        &Apache::lonhtmlcommon::confirm_success(&mt('Done')).
        '<br /><br />'.
        &Apache::lonhtmlcommon::actionbox(
            ['<a href="/adm/parmset">'.&mt('Content and Problem Settings').'</a>']));
    $r->print(&Apache::loncommon::end_page());
}

sub parse_key {
    my ($key) = @_;
    my %data;
    my ($middle,$part,$name)=
    ($key=~/^$env{'request.course.id'}\.(?:(.+)\.)*([\w\s]+)\.(\w+)$/);
    $data{'scope_type'} = 'all';
    if ($middle=~/^\[(.*)\]/) {
           $data{'scope'} = $1;
    if ($data{'scope'}=~/^useropt\:($match_username)\:($match_domain)/) {
        $data{'scope_type'} = 'user';
        $data{'scope'} = [$1,$2];
    } else {
        #FIXME check for group scope
        $data{'scope_type'} = 'section';
    }
    $middle=~s/^\[(.*)\]//;
    }
    $middle=~s/\.+$//;
    $middle=~s/^\.+//;
    $data{'realm_type'}='all';
    if ($middle=~/^(.+)\_\_\_\(all\)$/) {
    $data{'realm'} = $1;
    $data{'realm_type'} = 'folder';
    $data{'realm_title'} = &Apache::lonnet::gettitle($data{'realm'});
    ($data{'realm_exists'}) = &Apache::lonnet::is_on_map($data{'realm'});
    } elsif ($middle) {
    $data{'realm'} = $middle;
    $data{'realm_type'} = 'symb';
    $data{'realm_title'} = &Apache::lonnet::gettitle($data{'realm'});
    my ($map,$resid,$url) = &Apache::lonnet::decode_symb($data{'realm'});
    $data{'realm_exists'} = &Apache::lonnet::symbverify($data{'realm'},$url);
    }

    $data{'parameter_part'} = $part;
    $data{'parameter_name'} = $name;

    return %data;
}


sub header {
    return &Apache::loncommon::start_page('Content and Problem Settings');
}



sub print_main_menu {
    my ($r,$parm_permission)=@_;
    #
    $r->print(&header());
    $r->print(&Apache::lonhtmlcommon::breadcrumbs('Content and Problem Settings'));
    $r->print(<<ENDMAINFORMHEAD);
<form method="post" enctype="multipart/form-data"
      action="/adm/parmset" name="studentform">
ENDMAINFORMHEAD
#
    my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
    my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $vgr  = &Apache::lonnet::allowed('vgr',$env{'request.course.id'});
    my $mgr  = &Apache::lonnet::allowed('mgr',$env{'request.course.id'});
    my $dcm = &Apache::lonnet::allowed('dcm',$env{'request.course.id'});
    if ((!$dcm) && ($env{'request.course.sec'} ne '')) {
        $dcm = &Apache::lonnet::allowed('dcm',$env{'request.course.id'}.
                                        '/'.$env{'request.course.sec'});
    }

    my $crstype = &Apache::loncommon::course_type();
    my $lc_crstype = lc($crstype);

    my @menu =
        ( { categorytitle=>"Content Settings for this $crstype",
        items => [
          { linktext => 'Portfolio Metadata',
            url => '/adm/parmset?action=setrestrictmeta',
            permission => $parm_permission,
            linktitle => "Restrict metadata for this $lc_crstype." ,
            icon =>'contact-new.png'   ,
            },
          { linktext => 'Reset Student Access Times',
            url => '/adm/helper/resettimes.helper',
            permission => $mgr,
            linktitle => "Reset access times for folders/maps, resources or the $lc_crstype."  ,
            icon => 'start-here.png'  ,
            },
          { linktext => 'Blocking Communication/Resource Access',
            url => '/adm/setblock',
            permission => $dcm,
            linktitle => 'Configure blocking of communication/collaboration and access to resources during an exam',
            icon => 'comblock.png',
            },
          { linktext => 'Set Parameter Setting Default Actions',
            url => '/adm/parmset?action=setdefaults',
            permission => $parm_permission,
            linktitle =>'Set default actions for parameters.'  ,
            icon => 'folder-new.png'  ,
            }]},
      { categorytitle => 'New and Existing Parameter Settings for Resources',
        items => [
          { linktext => 'Edit Resource Parameters - Helper Mode',
            url => '/adm/helper/parameter.helper',
            permission => $parm_permission,
            linktitle =>'Set/Modify resource parameters in helper mode.'  ,
            icon => 'dialog-information.png'  ,
            #help => 'Parameter_Helper',
            },
          { linktext => 'Edit Resource Parameters - Overview Mode',
            url => '/adm/parmset?action=newoverview',
            permission => $parm_permission,
            linktitle =>'Set/Modify resource parameters in overview mode.'  ,
            icon => 'edit-find.png'  ,
            #help => 'Parameter_Overview',
            },
          { linktext => 'Edit Resource Parameters - Table Mode',
            url => '/adm/parmset?action=settable',
            permission => $parm_permission,
            linktitle =>'Set/Modify resource parameters in table mode.'  ,
            icon => 'edit-copy.png'  ,
            #help => 'Table_Mode',
            }]},
           { categorytitle => 'Existing Parameter Settings for Resources',
         items => [
          { linktext => 'Modify Resource Parameters - Overview Mode',
            url => '/adm/parmset?action=setoverview',
            permission => $parm_permission,
            linktitle =>'Set/Modify existing resource parameters in overview mode.'  ,
            icon => 'preferences-desktop-wallpaper.png'  ,
            #help => 'Parameter_Overview',
            },
          { linktext => 'Change Log',
            url => '/adm/parmset?action=parameterchangelog',
            permission => $parm_permission,
            linktitle =>"View parameter and $lc_crstype blog posting/user notification change log."  ,
            icon => 'document-properties.png',
            }]}
          );
    $r->print(&Apache::lonhtmlcommon::generate_menu(@menu));
    $r->print('</form>'.&Apache::loncommon::end_page());
    return;
}



sub output_row {
    my ($r, $field_name, $field_text, $added_flag) = @_;
    my $output;
    my $options=$env{'course.'.$env{'request.course.id'}.'.metadata.'.$field_name.'.options'};
    my $values=$env{'course.'.$env{'request.course.id'}.'.metadata.'.$field_name.'.values'};
    if (!defined($options)) {
        $options = 'active,stuadd';
        $values = '';
    }
    if (!($options =~ /deleted/)) {
        my @options= ( ['active', 'Show to student'],
                    ['stuadd', 'Provide text area for students to type metadata'],
                    ['choices','Provide choices for students to select from']);
#           ['onlyone','Student may select only one choice']);
        if ($added_flag) {
            push @options,['deleted', 'Delete Metadata Field'];
        }
       $output = &Apache::loncommon::start_data_table_row();
        $output .= '<td><strong>'.$field_text.':</strong></td>';
        $output .= &Apache::loncommon::end_data_table_row();
        foreach my $opt (@options) {
        my $checked = ($options =~ m/$opt->[0]/) ? ' checked="checked" ' : '' ;
        $output .= &Apache::loncommon::continue_data_table_row();
        $output .= '<td>'.('&nbsp;' x 5).'<label>
                   <input type="checkbox" name="'.
                   $field_name.'_'.$opt->[0].'" value="yes"'.$checked.' />'.
                   &mt($opt->[1]).'</label></td>';
        $output .= &Apache::loncommon::end_data_table_row();
    }
        $output .= &Apache::loncommon::continue_data_table_row();
        $output .= '<td>'.('&nbsp;' x 10).'<input name="'.$field_name.'_values" type="text" value="'.$values.'" size="80" /></td>';
        $output .= &Apache::loncommon::end_data_table_row();
        my $multiple_checked;
        my $single_checked;
        if ($options =~ m/onlyone/) {
            $multiple_checked = '';
            $single_checked = ' checked="checked"';
        } else {
            $multiple_checked = ' checked="checked"';
            $single_checked = '';
        }
    $output .= &Apache::loncommon::continue_data_table_row();
    $output .= '<td>'.('&nbsp;' x 10).'
                <input type="radio" name="'.$field_name.'_onlyone" value="multiple"'.$multiple_checked .' />
                '.&mt('Student may select multiple choices from list').'</td>';
    $output .= &Apache::loncommon::end_data_table_row();
    $output .= &Apache::loncommon::continue_data_table_row();
    $output .= '<td>'.('&nbsp;' x 10).'
                <input type="radio" name="'.$field_name.'_onlyone"  value="single"'.$single_checked.' />
                '.&mt('Student may select only one choice from list').'</td>';
    $output .= &Apache::loncommon::end_data_table_row();
    }
    return ($output);
}



sub order_meta_fields {
    my ($r)=@_;
    my $idx = 1;
    my $dom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $crs = $env{'course.'.$env{'request.course.id'}.'.num'};
    $r->print(&Apache::loncommon::start_page('Order Metadata Fields'));
    &Apache::lonhtmlcommon::add_breadcrumb({href=>'/adm/parmset?action=addmetadata',
        text=>"Add Metadata Field"});
    &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>"/adm/parmset?action=setrestrictmeta",
              text=>"Restrict Metadata"},
             {text=>"Order Metadata"});
    $r->print(&Apache::lonhtmlcommon::breadcrumbs('Order Metadata'));
    if ($env{'form.storeorder'}) {
        my $newpos = $env{'form.newpos'} - 1;
        my $currentpos = $env{'form.currentpos'} - 1;
        my @neworder = ();
        my @oldorder = split /,/,$env{'course.'.$env{'request.course.id'}.'.metadata.addedorder'};
        my $i;
        if ($newpos > $currentpos) {
        # moving stuff up
            for ($i=0;$i<$currentpos;$i++) {
            $neworder[$i]=$oldorder[$i];
            }
            for ($i=$currentpos;$i<$newpos;$i++) {
            $neworder[$i]=$oldorder[$i+1];
            }
            $neworder[$newpos]=$oldorder[$currentpos];
            for ($i=$newpos+1;$i<=$#oldorder;$i++) {
            $neworder[$i]=$oldorder[$i];
            }
        } else {
        # moving stuff down
            for ($i=0;$i<$newpos;$i++) {
                $neworder[$i]=$oldorder[$i];
            }
            $neworder[$newpos]=$oldorder[$currentpos];
            for ($i=$newpos+1;$i<$currentpos+1;$i++) {
                $neworder[$i]=$oldorder[$i-1];
            }
            for ($i=$currentpos+1;$i<=$#oldorder;$i++) {
                $neworder[$i]=$oldorder[$i];
            }
        }
    my $ordered_fields = join ",", @neworder;
        my $put_result = &Apache::lonnet::put('environment',
                           {'metadata.addedorder'=>$ordered_fields},$dom,$crs);
    &Apache::lonnet::appenv({'course.'.$env{'request.course.id'}.'.metadata.addedorder' => $ordered_fields});
    }
    my $fields = &get_added_meta_fieldnames($env{'request.course.id'});
    my $ordered_fields;
    my @fields_in_order = split /,/,$env{'course.'.$env{'request.course.id'}.'.metadata.addedorder'};
    if (!@fields_in_order) {
        # no order found, pick sorted order then create metadata.addedorder key.
        foreach my $key (sort keys %$fields) {
            push @fields_in_order, $key;
            $ordered_fields = join ",", @fields_in_order;
        }
        my $put_result = &Apache::lonnet::put('environment',
                            {'metadata.addedorder'=>$ordered_fields},$dom,$crs);
    }
    $r->print('<table>');
    my $num_fields = scalar(@fields_in_order);
    foreach my $key (@fields_in_order) {
        $r->print('<tr><td>');
        $r->print('<form method="post" action="">');
        $r->print('<select name="newpos" onchange="this.form.submit()">');
        for (my $i = 1;$i le $num_fields;$i ++) {
            if ($i eq $idx) {
                $r->print('<option value="'.$i.'"  SELECTED>('.$i.')</option>');
            } else {
                $r->print('<option value="'.$i.'">'.$i.'</option>');
            }
        }
        $r->print('</select></td><td>');
        $r->print('<input type="hidden" name="currentpos" value="'.$idx.'" />');
        $r->print('<input type="hidden" name="storeorder" value="true" />');
        $r->print('</form>');
        $r->print($$fields{$key}.'</td></tr>');
        $idx ++;
    }
    $r->print('</table>');
    return 'ok';
}


sub continue {
    my $output;
    $output .= '<form action="" method="post">';
    $output .= '<input type="hidden" name="action" value="setrestrictmeta" />';
    $output .= '<input type="submit" value="Continue" />';
    return ($output);
}


sub addmetafield {
    my ($r)=@_;
    &Apache::lonhtmlcommon::add_breadcrumb({href=>'/adm/parmset?action=addmetadata',
        text=>"Add Metadata Field"});
    $r->print(&Apache::loncommon::start_page('Add Metadata Field'));
    $r->print(&Apache::lonhtmlcommon::breadcrumbs('Add Metadata Field'));
    my $dom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $crs = $env{'course.'.$env{'request.course.id'}.'.num'};
    if (exists($env{'form.undelete'})) {
        my @meta_fields = &Apache::loncommon::get_env_multiple('form.undeletefield');
        foreach my $meta_field(@meta_fields) {
            my $options = $env{'course.'.$env{'request.course.id'}.'.metadata.'.$meta_field.'.options'};
            $options =~ s/deleted//;
            $options =~ s/,,/,/;
            my $put_result = &Apache::lonnet::put('environment',
                                        {'metadata.'.$meta_field.'.options'=>$options},$dom,$crs);

            $r->print('Undeleted Metadata Field <strong>'.$env{'course.'.$env{'request.course.id'}.'.metadata.'.$meta_field.'.added'}."</strong> with result ".$put_result.'<br />');
        }
        $r->print(&continue());
    } elsif (exists($env{'form.fieldname'})) {
        my $meta_field = $env{'form.fieldname'};
        my $display_field = $env{'form.fieldname'};
        $meta_field =~ s/\W/_/g;
        $meta_field =~ tr/A-Z/a-z/;
        my $put_result = &Apache::lonnet::put('environment',
                            {'metadata.'.$meta_field.'.values'=>"",
                             'metadata.'.$meta_field.'.added'=>"$display_field",
                             'metadata.'.$meta_field.'.options'=>""},$dom,$crs);
        $r->print('Added new Metadata Field <strong>'.$env{'form.fieldname'}."</strong> with result ".$put_result.'<br />');
        $r->print(&continue());
    } else {
        my $fields = &get_deleted_meta_fieldnames($env{'request.course.id'});
        if ($fields) {
            $r->print('You may undelete previously deleted fields.<br />Check those you wish to undelete and click Undelete.<br />');
            $r->print('<form method="post" action="">');
            foreach my $key(keys(%$fields)) {
                $r->print('<input type="checkbox" name="undeletefield" value="'.$key.'" />'.$$fields{$key}.'<br /');
            }
            $r->print('<input type="submit" name="undelete" value="Undelete" />');
            $r->print('</form>');
        }
        $r->print('<hr /><strong>Or</strong> you may enter a new metadata field name.<form method="post" action="/adm/parmset?action=addmetadata"');
        $r->print('<input type="text" name="fieldname" /><br />');
        $r->print('<input type="submit" value="Add Metadata Field" />');
    }
    $r->print('</form>');
}



sub setrestrictmeta {
    my ($r)=@_;
    my $next_meta;
    my $output;
    my $item_num;
    my $put_result;
    &Apache::lonhtmlcommon::add_breadcrumb({href=>'/adm/parmset?action=setrestrictmeta',
        text=>"Restrict Metadata"});
    $r->print(&Apache::loncommon::start_page('Restrict Metadata'));
    $r->print(&Apache::lonhtmlcommon::breadcrumbs('Restrict Metadata'));
    my $dom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $crs = $env{'course.'.$env{'request.course.id'}.'.num'};
    my $key_base = $env{'course.'.$env{'request.course.id'}.'.'};
    my $save_field = '';
    if ($env{'form.restrictmeta'}) {
        foreach my $field (sort(keys(%env))) {
            if ($field=~m/^form.(.+)_(.+)$/) {
                my $options;
                my $meta_field = $1;
                my $meta_key = $2;
                if ($save_field ne $meta_field) {
                    $save_field = $meta_field;
                    if ($env{'form.'.$meta_field.'_stuadd'}) {
                        $options.='stuadd,';
                    }
                    if ($env{'form.'.$meta_field.'_choices'}) {
                        $options.='choices,';
                    }
                    if ($env{'form.'.$meta_field.'_onlyone'} eq 'single') {
                        $options.='onlyone,';
                    }
                    if ($env{'form.'.$meta_field.'_active'}) {
                        $options.='active,';
                    }
                    if ($env{'form.'.$meta_field.'_deleted'}) {
                        $options.='deleted,';
                    }
                    my $name = $save_field;
                     $put_result = &Apache::lonnet::put('environment',
                                                  {'metadata.'.$meta_field.'.options'=>$options,
                                                   'metadata.'.$meta_field.'.values'=>$env{'form.'.$meta_field.'_values'},
                                                   },$dom,$crs);
                }
            }
        }
    }
    &Apache::lonnet::coursedescription($env{'request.course.id'},
                       {'freshen_cache' => 1});
    # Get the default metadata fields
    my %metadata_fields = &Apache::lonmeta::fieldnames('portfolio');
    # Now get possible added metadata fields
    my $added_metadata_fields = &get_added_meta_fieldnames($env{'request.course.id'});
    my $row_alt = 1;
    $output .= &Apache::loncommon::start_data_table();
    foreach my $field (sort(keys(%metadata_fields))) {
        if ($field ne 'courserestricted') {
            $row_alt = $row_alt ? 0 : 1;
        $output.= &output_row($r, $field, $metadata_fields{$field});
    }
    }
    my $buttons = (<<ENDButtons);
        <input type="submit" name="restrictmeta" value="Save" />
        </form><br />
        <form method="post" action="/adm/parmset?action=addmetadata" name="form1">
        <input type="submit" name="restrictmeta" value="Add a Metadata Field" />
        </form>
        <br />
        <form method="post" action="/adm/parmset?action=ordermetadata" name="form2">
        <input type="submit" name="restrictmeta" value="Order Metadata Fields" />
ENDButtons
    my $added_flag = 1;
    foreach my $field (sort(keys(%$added_metadata_fields))) {
        $row_alt = $row_alt ? 0 : 1;
        $output.= &output_row($r, $field, $$added_metadata_fields{$field},$added_flag, $row_alt);
    }
    $output .= &Apache::loncommon::end_data_table();
    $r->print(<<ENDenv);
        <form method="post" action="/adm/parmset?action=setrestrictmeta" name="form">
        $output
        $buttons
        </form>
ENDenv
    $r->print(&Apache::loncommon::end_page());
    return 'ok';
}



sub get_added_meta_fieldnames {
    my ($cid) = @_;
    my %fields;
    foreach my $key(%env) {
        if ($key =~ m/\Q$cid\E\.metadata\.(.+)\.added$/) {
            my $field_name = $1;
            my ($display_field_name) = $env{$key};
            $fields{$field_name} = $display_field_name;
        }
    }
    return \%fields;
}



sub get_deleted_meta_fieldnames {
    my ($cid) = @_;
    my %fields;
    foreach my $key(%env) {
        if ($key =~ m/\Q$cid\E\.metadata\.(.+)\.added$/) {
            my $field_name = $1;
            if ($env{'course.'.$env{'request.course.id'}.'.metadata.'.$field_name.'.options'} =~ m/deleted/) {
                my ($display_field_name) = $env{$key};
                $fields{$field_name} = $display_field_name;
            }
        }
    }
    return \%fields;
}
sub defaultsetter {
    my ($r) = @_;

    &Apache::lonhtmlcommon::add_breadcrumb({href=>'/adm/parmset?action=setdefaults',
        text=>"Set Defaults"});
    my $start_page =
    &Apache::loncommon::start_page('Parameter Setting Default Actions');
    my $breadcrumbs = &Apache::lonhtmlcommon::breadcrumbs('Defaults');
    $r->print($start_page.$breadcrumbs);
    $r->print('<form method="post" action="/adm/parmset?action=setdefaults" name="defaultform">');

    my $dom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $crs = $env{'course.'.$env{'request.course.id'}.'.num'};
    my @ids=();
    my %typep=();
    my %keyp=();
    my %allparms=();
    my %allparts=();
    my %allmaps=();
    my %mapp=();
    my %symbp=();
    my %maptitles=();
    my %uris=();
    my %keyorder=&standardkeyorder();
    my %defkeytype=();

    &extractResourceInformation(\@ids, \%typep,\%keyp, \%allparms, \%allparts, \%allmaps,
                \%mapp, \%symbp,\%maptitles,\%uris,
                \%keyorder,\%defkeytype);
    if ($env{'form.storerules'}) {
    my %newrules=();
    my @delrules=();
    my %triggers=();
    foreach my $key (keys(%env)) {
            if ($key=~/^form\.(\w+)\_action$/) {
        my $tempkey=$1;
        my $action=$env{$key};
                if ($action) {
            $newrules{$tempkey.'_action'}=$action;
            if ($action ne 'default') {
            my ($whichaction,$whichparm)=($action=~/^(.*\_)([^\_]+)$/);
            $triggers{$whichparm}.=$tempkey.':';
            }
            $newrules{$tempkey.'_type'}=$defkeytype{$tempkey};
            if (&isdateparm($defkeytype{$tempkey})) {
            $newrules{$tempkey.'_days'}=$env{'form.'.$tempkey.'_days'};
            $newrules{$tempkey.'_hours'}=$env{'form.'.$tempkey.'_hours'};
            $newrules{$tempkey.'_min'}=$env{'form.'.$tempkey.'_min'};
            $newrules{$tempkey.'_sec'}=$env{'form.'.$tempkey.'_sec'};
            } else {
            $newrules{$tempkey.'_value'}=$env{'form.'.$tempkey.'_value'};
            $newrules{$tempkey.'_triggervalue'}=$env{'form.'.$tempkey.'_triggervalue'};
            }
        } else {
            push(@delrules,$tempkey.'_action');
            push(@delrules,$tempkey.'_type');
            push(@delrules,$tempkey.'_hours');
            push(@delrules,$tempkey.'_min');
            push(@delrules,$tempkey.'_sec');
            push(@delrules,$tempkey.'_value');
        }
        }
    }
    foreach my $key (keys %allparms) {
        $newrules{$key.'_triggers'}=$triggers{$key};
    }
    &Apache::lonnet::put('parmdefactions',\%newrules,$dom,$crs);
    &Apache::lonnet::del('parmdefactions',\@delrules,$dom,$crs);
    &resetrulescache();
    }
    my %lt=&Apache::lonlocal::texthash('days' => 'Days',
                       'hours' => 'Hours',
                       'min' => 'Minutes',
                       'sec' => 'Seconds',
                       'yes' => 'Yes',
                       'no' => 'No');
    my @standardoptions=('','default');
    my @standarddisplay=('',&mt('Default value when manually setting'));
    my @dateoptions=('','default');
    my @datedisplay=('',&mt('Default value when manually setting'));
    foreach my $tempkey (&keysindisplayorder(\%allparms,\%keyorder)) {
    unless ($tempkey) { next; }
    push @standardoptions,'when_setting_'.$tempkey;
    push @standarddisplay,&mt('Automatically set when setting ').$tempkey;
    if (&isdateparm($defkeytype{$tempkey})) {
        push @dateoptions,'later_than_'.$tempkey;
        push @datedisplay,&mt('Automatically set later than ').$tempkey;
        push @dateoptions,'earlier_than_'.$tempkey;
        push @datedisplay,&mt('Automatically set earlier than ').$tempkey;
    }
    }
$r->print(&mt('Manual setting rules apply to all interfaces.').'<br />'.
      &mt('Automatic setting rules apply to table mode interfaces only.'));
    $r->print("\n".&Apache::loncommon::start_data_table().
          &Apache::loncommon::start_data_table_header_row().
          "<th>".&mt('Rule for parameter').'</th><th>'.
          &mt('Action').'</th><th>'.&mt('Value').'</th>'.
          &Apache::loncommon::end_data_table_header_row());
    foreach my $tempkey (&keysindisplayorder(\%allparms,\%keyorder)) {
    unless ($tempkey) { next; }
    $r->print("\n".&Apache::loncommon::start_data_table_row().
          "<td>".$allparms{$tempkey}."\n<br />(".$tempkey.')</td><td>');
    my $action=&rulescache($tempkey.'_action');
    $r->print('<select name="'.$tempkey.'_action">');
    if (&isdateparm($defkeytype{$tempkey})) {
        for (my $i=0;$i<=$#dateoptions;$i++) {
        if ($dateoptions[$i]=~/\_$tempkey$/) { next; }
        $r->print("\n<option value='$dateoptions[$i]'".
              ($dateoptions[$i] eq $action?' selected="selected"':'').
              ">$datedisplay[$i]</option>");
        }
    } else {
        for (my $i=0;$i<=$#standardoptions;$i++) {
        if ($standardoptions[$i]=~/\_$tempkey$/) { next; }
        $r->print("\n<option value='$standardoptions[$i]'".
              ($standardoptions[$i] eq $action?' selected="selected"':'').
              ">$standarddisplay[$i]</option>");
        }
    }
    $r->print('</select>');
    unless (&isdateparm($defkeytype{$tempkey})) {
        $r->print("\n<br />".&mt('Triggering value(s) of other parameter (optional, comma-separated):').
              '<input type="text" size="20" name="'.$tempkey.'_triggervalue" value="'.&rulescache($tempkey.'_triggervalue').'" />');
    }
    $r->print("\n</td><td>\n");

        if (&isdateparm($defkeytype{$tempkey})) {
        my $days=&rulescache($tempkey.'_days');
        my $hours=&rulescache($tempkey.'_hours');
        my $min=&rulescache($tempkey.'_min');
        my $sec=&rulescache($tempkey.'_sec');
        $r->print(<<ENDINPUTDATE);
<input name="$tempkey\_days" type="text" size="4" value="$days" />$lt{'days'}<br />
<input name="$tempkey\_hours" type="text" size="4" value="$hours" />$lt{'hours'}<br />
<input name="$tempkey\_min" type="text" size="4" value="$min" />$lt{'min'}<br />
<input name="$tempkey\_sec" type="text" size="4" value="$sec" />$lt{'sec'}
ENDINPUTDATE
    } elsif ($defkeytype{$tempkey} eq 'string_yesno') {
            my $yeschecked='';
            my $nochecked='';
            if (&rulescache($tempkey.'_value') eq 'yes') { $yeschecked=' checked="checked"'; }
            if (&rulescache($tempkey.'_value') eq 'no') { $nochecked=' checked="checked"'; }

        $r->print(<<ENDYESNO);
<label><input type="radio" name="$tempkey\_value" value="yes"$yeschecked /> $lt{'yes'}</label><br />
<label><input type="radio" name="$tempkey\_value" value="no"$nochecked /> $lt{'no'}</label>
ENDYESNO
        } else {
        $r->print('<input type="text" size="20" name="'.$tempkey.'_value" value="'.&rulescache($tempkey.'_value').'" />');
    }
        $r->print('</td>'.&Apache::loncommon::end_data_table_row());
    }
    $r->print(&Apache::loncommon::end_data_table().
          "\n".'<input type="submit" name="storerules" value="'.
          &mt('Save').'" /></form>'."\n");
    $r->print(&Apache::loncommon::end_page());
    return;
}

sub components {
    my ($key,$uname,$udom,$exeuser,$exedomain,$typeflag)=@_;

    if ($typeflag) {
    $key=~s/\.type$//;
    }

    my ($middle,$part,$name)=
    ($key=~/^$env{'request.course.id'}\.(?:(.+)\.)*([\w\s]+)\.(\w+)$/);
    my $issection;

    my $section=&mt('All Students');
    if ($middle=~/^\[(.*)\]/) {
    $issection=$1;
    $section=&mt('Group/Section').': '.$issection;
    $middle=~s/^\[(.*)\]//;
    }
    $middle=~s/\.+$//;
    $middle=~s/^\.+//;
    if ($uname) {
    $section=&mt('User').": ".&Apache::loncommon::plainname($uname,$udom);
    $issection='';
    }
    my $realm='<span class="LC_parm_scope_all">'.&mt('All Resources').'</span>';
    my $realmdescription=&mt('all resources');
    if ($middle=~/^(.+)\_\_\_\(all\)$/) {
    $realm='<span class="LC_parm_scope_folder">'.&mt('Folder/Map').': '.&Apache::lonnet::gettitle($1).' <span class="LC_parm_folder"><br />('.$1.')</span></span>';
     $realmdescription=&mt('folder').' '.&Apache::lonnet::gettitle($1);
   } elsif ($middle) {
    my ($map,$id,$url)=&Apache::lonnet::decode_symb($middle);
    $realm='<span class="LC_parm_scope_resource">'.&mt('Resource').': '.&Apache::lonnet::gettitle($middle).' <br /><span class="LC_parm_symb">('.$url.' in '.$map.' id: '.$id.')</span></span>';
    $realmdescription=&mt('resource').' '.&Apache::lonnet::gettitle($middle);
    }
    my $what=$part.'.'.$name;
    return ($realm,$section,$name,$part,
        $what,$middle,$uname,$udom,$issection,$realmdescription);
}

my %standard_parms;
my %standard_parms_types;

sub load_parameter_names {
    open(my $config,"<$Apache::lonnet::perlvar{'lonTabDir'}/packages.tab");
    while (my $configline=<$config>) {
    if ($configline !~ /\S/ || $configline=~/^\#/) { next; }
    chomp($configline);
    my ($short,$plain)=split(/:/,$configline);
    my (undef,$name,$type)=split(/\&/,$short,3);
    if ($type eq 'display') {
        $standard_parms{$name} = $plain;
        } elsif ($type eq 'type') {
            $standard_parms_types{$name} = $plain;
        }
    }
    close($config);
    $standard_parms{'int_pos'}      = 'Positive Integer';
    $standard_parms{'int_zero_pos'} = 'Positive Integer or Zero';
}

sub standard_parameter_names {
    my ($name)=@_;
    if (!%standard_parms) {
    &load_parameter_names();
    }
    if ($standard_parms{$name}) {
    return $standard_parms{$name};
    } else {
    return $name;
    }
}

sub standard_parameter_types {
    my ($name)=@_;
    if (!%standard_parms_types) {
        &load_parameter_names();
    }
    if ($standard_parms_types{$name}) {
        return $standard_parms_types{$name};
    }
    return;
}

sub parm_change_log {
    my ($r)=@_;
    &Apache::lonhtmlcommon::add_breadcrumb({href=>'/adm/parmset?action=settable',
    text=>"Parameter Change Log"});
    my $js = '<script type="text/javascript">'."\n".
             '// <![CDATA['."\n".
             &Apache::loncommon::display_filter_js('parmslog')."\n".
             '// ]]>'."\n".
             '</script>'."\n";
    $r->print(&Apache::loncommon::start_page('Parameter Change Log',$js));
    $r->print(&Apache::lonhtmlcommon::breadcrumbs('Parameter Change Log'));
    my %parmlog=&Apache::lonnet::dump('nohist_parameterlog',
                      $env{'course.'.$env{'request.course.id'}.'.domain'},
                      $env{'course.'.$env{'request.course.id'}.'.num'});

    if ((keys(%parmlog))[0]=~/^error\:/) { undef(%parmlog); }

    $r->print('<div class="LC_left_float">'.
              '<fieldset><legend>'.&mt('Display of Changes').'</legend>'.
              '<form action="/adm/parmset?action=parameterchangelog"
                     method="post" name="parameterlog">');

    my %saveable_parameters = ('show' => 'scalar',);
    &Apache::loncommon::store_course_settings('parameter_log',
                                              \%saveable_parameters);
    &Apache::loncommon::restore_course_settings('parameter_log',
                                                \%saveable_parameters);
    $r->print(&Apache::loncommon::display_filter('parmslog').'&nbsp;'."\n".
              '<input type="submit" value="'.&mt('Display').'" />'.
              '</form></fieldset></div><br clear="all" />');

    my $courseopt=&Apache::lonnet::get_courseresdata($env{'course.'.$env{'request.course.id'}.'.num'},
                             $env{'course.'.$env{'request.course.id'}.'.domain'});
    $r->print(&Apache::loncommon::start_data_table().&Apache::loncommon::start_data_table_header_row().
          '<th>'.&mt('Time').'</th><th>'.&mt('User').'</th><th>'.&mt('Extent').'</th><th>'.&mt('Users').'</th><th>'.
          &mt('Parameter').'</th><th>'.&mt('Part').'</th><th>'.&mt('New Value').'</th><th>'.&mt('Announce').'</th>'.
          &Apache::loncommon::end_data_table_header_row());
    my $shown=0;
    my $folder='';
    if ($env{'form.displayfilter'} eq 'currentfolder') {
    my $last='';
    if (tie(my %hash,'GDBM_File',$env{'request.course.fn'}.'_symb.db',
        &GDBM_READER(),0640)) {
        $last=$hash{'last_known'};
        untie(%hash);
    }
    if ($last) { ($folder) = &Apache::lonnet::decode_symb($last); }
    }
    foreach my $id (sort
            {
            if ($parmlog{$b}{'exe_time'} ne $parmlog{$a}{'exe_time'}) {
                return $parmlog{$b}{'exe_time'} <=>$parmlog{$a}{'exe_time'}
            }
            my $aid = (split('00000',$a))[-1];
            my $bid = (split('00000',$b))[-1];
            return $bid<=>$aid;
            } (keys(%parmlog))) {
        my @changes=keys(%{$parmlog{$id}{'logentry'}});
    my $count = 0;
    my $time =
        &Apache::lonlocal::locallocaltime($parmlog{$id}{'exe_time'});
    my $plainname =
        &Apache::loncommon::plainname($parmlog{$id}{'exe_uname'},
                      $parmlog{$id}{'exe_udom'});
    my $about_me_link =
        &Apache::loncommon::aboutmewrapper($plainname,
                           $parmlog{$id}{'exe_uname'},
                           $parmlog{$id}{'exe_udom'});
    my $send_msg_link='';
    if ((($parmlog{$id}{'exe_uname'} ne $env{'user.name'})
         || ($parmlog{$id}{'exe_udom'} ne $env{'user.domain'}))) {
        $send_msg_link ='<br />'.
        &Apache::loncommon::messagewrapper(&mt('Send message'),
                           $parmlog{$id}{'exe_uname'},
                           $parmlog{$id}{'exe_udom'});
    }
    my $row_start=&Apache::loncommon::start_data_table_row();
    my $makenewrow=0;
    my %istype=();
    my $output;
    foreach my $changed (reverse(sort(@changes))) {
            my $value=$parmlog{$id}{'logentry'}{$changed};
        my $typeflag = ($changed =~/\.type$/ &&
                !exists($parmlog{$id}{'logentry'}{$changed.'.type'}));
            my ($realm,$section,$parmname,$part,$what,$middle,$uname,$udom,$issection,$realmdescription)=
        &components($changed,$parmlog{$id}{'uname'},$parmlog{$id}{'udom'},undef,undef,$typeflag);
        if ($env{'form.displayfilter'} eq 'currentfolder') {
        if ($folder) {
            if ($middle!~/^\Q$folder\E/) { next; }
        }
        }
        if ($typeflag) {
        $istype{$parmname}=$value;
        if (!$env{'form.includetypes'}) { next; }
        }
        $count++;
        if ($makenewrow) {
        $output .= $row_start;
        } else {
        $makenewrow=1;
        }
            my $parmitem = &standard_parameter_names($parmname);
        $output .='<td>'.$realm.'</td><td>'.$section.'</td><td>'.
              &mt($parmitem).'</td><td>'.
              ($part?&mt('Part: [_1]',$part):&mt('All Parts')).'</td><td>';
        my $stillactive=0;
        if ($parmlog{$id}{'delflag'}) {
        $output .= &mt('Deleted');
        } else {
        if ($typeflag) {
                    my $parmitem = &standard_parameter_names($value); 
                    $parmitem = &mt($parmitem);
            $output .= &mt('Type: [_1]',$parmitem);
        } else {
            my ($level,@all)=&parmval_by_symb($what,$middle,&Apache::lonnet::metadata($middle,$what),
                              $uname,$udom,$issection,$issection,$courseopt);
                    my $showvalue = $value;
                    if ($istype{$parmname} eq '') {
                        my $type = &standard_parameter_types($parmname);
                        if ($type ne '') {
                            if (&isdateparm($type)) {
                                $showvalue =
                                    &Apache::lonlocal::locallocaltime($value);
                            }
                        }
                    } else {
                if (&isdateparm($istype{$parmname})) {
                $showvalue = 
                                &Apache::lonlocal::locallocaltime($value);
                }
                    }
                    $output .= $showvalue;
            if ($value ne $all[$level]) {
            $output .= '<br /><span class="LC_warning">'.&mt('Not active anymore').'</span>';
            } else {
            $stillactive=1;
            }
        }
        }
        $output .= '</td><td>';
            
        if ($stillactive) {
                my $parmitem = &standard_parameter_names($parmname);
                $parmitem = &mt($parmitem);
        my $title=&mt('Changed [_1]',$parmitem);
                my $description=&mt('Changed [_1] for [_2] to [_3]',
                                    $parmitem,$realmdescription,
                    (&isdateparm($istype{$parmname})?&Apache::lonlocal::locallocaltime($value):$value));
        if (($uname) && ($udom)) {
            $output .=
            &Apache::loncommon::messagewrapper('Notify User',
                               $uname,$udom,$title,
                               $description);
        } else {
            $output .=
            &Apache::lonrss::course_blog_link($id,$title,
                              $description);
        }
        }
        $output .= '</td>'.&Apache::loncommon::end_data_table_row();
    }
        if ($env{'form.displayfilter'} eq 'containing') {
        my $wholeentry=$about_me_link.':'.
        $parmlog{$id}{'exe_uname'}.':'.$parmlog{$id}{'exe_udom'}.':'.
        $output;
        if ($wholeentry!~/\Q$env{'form.containingphrase'}\E/i) { next; }
    }
        if ($count) {
        $r->print($row_start.'<td rowspan="'.$count.'">'.$time.'</td>
                       <td rowspan="'.$count.'">'.$about_me_link.
          '<br /><tt>'.$parmlog{$id}{'exe_uname'}.
                      ':'.$parmlog{$id}{'exe_udom'}.'</tt>'.
          $send_msg_link.'</td>'.$output);
        $shown++;
    }
    if (!($env{'form.show'} eq &mt('all')
          || $shown<=$env{'form.show'})) { last; }
    }
    $r->print(&Apache::loncommon::end_data_table());
    $r->print(&Apache::loncommon::end_page());
}

sub update_slots {
    my ($slot_name,$cdom,$cnum,$symb,$uname,$udom) = @_;
    my %slot=&Apache::lonnet::get_slot($slot_name);
    if (!keys(%slot)) {
        return 'error: slot does not exist';
    }
    my $max=$slot{'maxspace'};
    if (!defined($max)) { $max=99999; }

    my %consumed=&Apache::lonnet::dump('slot_reservations',$cdom,$cnum,
                                       "^$slot_name\0");
    my ($tmp)=%consumed;
    if ($tmp=~/^error: 2 / ) {
        return 'error: unable to determine current slot status';
    }
    my $last=0;
    foreach my $key (keys(%consumed)) {
        my $num=(split('\0',$key))[1];
        if ($num > $last) { $last=$num; }
        if ($consumed{$key}->{'name'} eq $uname.':'.$udom) {
            return 'ok';
        }
    }

    if (scalar(keys(%consumed)) >= $max) {
        return 'error: no space left in slot';
    }
    my $wanted=$last+1;

    my %reservation=('name'      => $uname.':'.$udom,
                     'timestamp' => time,
                     'symb'      => $symb);

    my $success=&Apache::lonnet::newput('slot_reservations',
                                        {"$slot_name\0$wanted" =>
                                             \%reservation},
                                        $cdom, $cnum);
    if ($success eq 'ok') {
        my %storehash = (
                          symb    => $symb,
                          slot    => $slot_name,
                          action  => 'reserve',
                          context => 'parameter',
                        );
        &Apache::lonnet::write_log('course','slotreservationslog',\%storehash,
                                        '',$uname,$udom,$cnum,$cdom);

        &Apache::lonnet::write_log('course',$cdom.'_'.$cnum.'_slotlog',\%storehash,
                                        '',$uname,$udom,$uname,$udom);
    }
    return $success;
}

sub delete_slots {
    my ($slot_name,$cdom,$cnum,$uname,$udom,$symb) = @_;
    my $delresult;
    my %consumed = &Apache::lonnet::dump('slot_reservations',$cdom,
                                         $cnum, "^$slot_name\0");
    if (&Apache::lonnet::error(%consumed)) {
        return 'error: unable to determine current slot status';
    }
    my ($tmp)=%consumed;
    if ($tmp=~/^error: 2 /) {
        return 'error: unable to determine current slot status';
    }
    foreach my $key (keys(%consumed)) {
        if ($consumed{$key}->{'name'} eq $uname.':'.$udom) {
            my $num=(split('\0',$key))[1];
            my $entry = $slot_name.'\0'.$num;
            $delresult = &Apache::lonnet::del('slot_reservations',[$entry],
                                              $cdom,$cnum);
            if ($delresult eq 'ok') {
                my %storehash = (
                                  symb    => $symb,
                                  slot    => $slot_name,
                                  action  => 'release',
                                  context => 'parameter',
                                );
                &Apache::lonnet::write_log('course','slotreservationslog',\%storehash,
                                                1,$uname,$udom,$cnum,$cdom);
                &Apache::lonnet::write_log('course',$cdom.'_'.$cnum.'_slotlog',\%storehash,
                                                1,$uname,$udom,$uname,$udom);
            }
        }
    }
    return $delresult;
}

sub check_for_course_info {
    my $navmap = Apache::lonnavmaps::navmap->new();
    return 1 if ($navmap);
    return 0;
}

sub parameter_release_vars { 
   my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
   my $chome = $env{'course.'.$env{'request.course.id'}.'.home'};
   my $chostname = &Apache::lonnet::hostname($chome);
   my ($cmajor,$cminor) = 
       split(/\./,&Apache::lonnet::get_server_loncaparev($cdom,$chome));
   return ($chostname,$cmajor,$cminor);
}

sub parameter_releasecheck {
    my ($name,$value,$needsrelease,$chostname,$cmajor,$cminor) = @_;
    my $needsnewer;
    my ($needsmajor,$needsminor) = split(/\./,$needsrelease);
    if (($cmajor < $needsmajor) || 
        ($cmajor == $needsmajor && $cminor < $needsminor)) {
        $needsnewer = 1;
    } else {
        &Apache::lonnet::update_released_required($Apache::lonnet::needsrelease{'parameter:'.$name.':'.$value});
    }
    return $needsnewer;
}

sub oldversion_warning {
    my ($name,$value,$chostname,$cmajor,$cminor,$needsrelease) = @_;
    my $desc;
    my %stringtypes = (
                        type         => 'string_questiontype',
                        lenient      => 'string_lenient',
                        retrypartial => 'string_yesno',
                        discussvote  => 'string_discussvote',
                        examcode     => 'string_examcode',
                      );
    if (exists($stringtypes{$name})) {
        if ($name eq 'examcode') {
            $desc = $value;
        } elsif (ref($strings{$stringtypes{$name}}) eq 'ARRAY') {
            foreach my $possibilities (@{ $strings{$stringtypes{$name}} }) {
                next unless (ref($possibilities) eq 'ARRAY');
                my ($parmval, $description) = @{ $possibilities };
                if ($parmval eq $value) {
                    $desc = $description;
                    last;
                }
            }
        }
    }
    my $standard_name = &standard_parameter_names($name);
    return '<p class="LC_warning">'.
           &mt('[_1] was [_2]not[_3] set to [_4].',
               $standard_name,'<b>','</b>','"'.$desc.'"').'<br />'.
           &mt('LON-CAPA version ([_1]) installed on home server ([_2]) does not meet version requirements ([_3] or newer).',
           $cmajor.'.'.$cminor,$chostname,
           $needsrelease).
           '</p>';
}

sub handler {
    my $r=shift;

    &reset_caches();

    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;
    return OK if $r->header_only;

    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},
                        ['action','state',
                                             'pres_marker',
                                             'pres_value',
                                             'pres_type',
                                             'filter','part',
                                             'udom','uname','symb','serial','timebase']);


    &Apache::lonhtmlcommon::clear_breadcrumbs();
    &Apache::lonhtmlcommon::add_breadcrumb({href=>"/adm/parmset",
                        text=>"Content and Problem Settings",
                        faq=>10,
                        bug=>'Instructor Interface',
                                            help =>
                                            'Parameter_Manager,Course_Environment,Parameter_Helper,Parameter_Overview,Table_Mode'});

# ----------------------------------------------------- Needs to be in a course
    my $parm_permission =
    (&Apache::lonnet::allowed('opa',$env{'request.course.id'}) ||
     &Apache::lonnet::allowed('opa',$env{'request.course.id'}.'/'.
                  $env{'request.course.sec'}));
    my $exists = &check_for_course_info();

    if ($env{'request.course.id'} &&  $parm_permission && $exists) {
        #
        # Main switch on form.action and form.state, as appropriate
        #
        # Check first if coming from someone else headed directly for
        #  the table mode
        if ((($env{'form.command'} eq 'set') && ($env{'form.url'})
         && (!$env{'form.dis'})) || ($env{'form.symb'})) {
        &assessparms($r);
        } elsif (! exists($env{'form.action'})) {
            &print_main_menu($r,$parm_permission);
        } elsif ($env{'form.action'} eq 'setoverview') {
        &overview($r);
    } elsif ($env{'form.action'} eq 'addmetadata') {
        &addmetafield($r);
    } elsif ($env{'form.action'} eq 'ordermetadata') {
        &order_meta_fields($r);
        } elsif ($env{'form.action'} eq 'setrestrictmeta') {
        &setrestrictmeta($r);
        } elsif ($env{'form.action'} eq 'newoverview') {
        &newoverview($r);
        } elsif ($env{'form.action'} eq 'setdefaults') {
        &defaultsetter($r);
    } elsif ($env{'form.action'} eq 'settable') {
        &assessparms($r);
        } elsif ($env{'form.action'} eq 'parameterchangelog') {
        &parm_change_log($r);
        } elsif ($env{'form.action'} eq 'cleanparameters') {
        &clean_parameters($r);
        } elsif ($env{'form.action'} eq 'dateshift1') {
            &date_shift_one($r);
        } elsif ($env{'form.action'} eq 'dateshift2') {
            &date_shift_two($r);
        }
    } else {
# ----------------------------- Not in a course, or not allowed to modify parms
    if ($exists) {
        $env{'user.error.msg'}=
        "/adm/parmset:opa:0:0:Cannot modify assessment parameters";
    } else {
        $env{'user.error.msg'}=
        "/adm/parmset::0:1:Course environment gone, reinitialize the course";
    }
    return HTTP_NOT_ACCEPTABLE;
    }
    &reset_caches();

    return OK;
}

1;
__END__


