#
# $Id: lonwhatsnew.pm,v 1.105.2.9 2013/12/14 02:08:30 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#


package Apache::lonwhatsnew;

use strict;
use lib qw(/home/httpd/lib/perl);
use Apache::lonnet;
use Apache::loncommon();
use Apache::lonhtmlcommon();
use Apache::lonlocal;
use Apache::loncoursedata();
use Apache::lonnavmaps();
use Apache::lonuserstate;
use Apache::lonuserutils; 
use Apache::Constants qw(:common :http);
use Time::Local;
use GDBM_File;
use lib '/home/httpd/lib/perl/';
use LONCAPA;
use HTML::Entities;

#----------------------------
# handler
#
#----------------------------

sub handler {
    my $r = shift;
    if ($r->header_only) {
        &Apache::loncommon::content_type($r,'text/html');
        $r->send_http_header;
        return OK;
    }
    &Apache::loncommon::get_unprocessed_cgi(
                                   $ENV{'QUERY_STRING'},['command','refpage']);

    my $command = $env{'form.command'};
    my $refpage = $env{'form.refpage'};

    my %checkallowed = ( coursenormalmail => 1,
			 coursecritmail => 1, );
    foreach my $perm_check (['whn','whatsnew',1],
			    ['pch','coursediscussion',1],
			    ['mgr','handgrading',1],
			    ['vgr','abovethreshold',1],
			    ['opa','haserrors',1],
			    ['mdc','versionchanges',0],
                            ['vcl','newroles',1],
                            ['vcl','oldroles',1],
                            ['whn','crslogin',1],
			    ) {
	my ($perm,$key,$check_section) = @{ $perm_check };
	my $scope = $env{'request.course.id'};
	if (!($checkallowed{$key} = &Apache::lonnet::allowed($perm,$scope))) {
	    $scope .= '/'.$env{'request.course.sec'};
	    if ( $check_section ) {
		$checkallowed{$key} = &Apache::lonnet::allowed($perm,$scope);
	    }
	    if ($checkallowed{$key}) {
		$checkallowed{$key.'_section'} = $env{'request.course.sec'};
	    }
	}
    }

    if ( ! $env{'request.course.fn'} || ! $checkallowed{'whatsnew'}) {
        # Not in a course, or no whn priv in course
        $env{'user.error.msg'}="/adm/whatsnew:whn:0:0:Cannot display what's new page";
        return HTTP_NOT_ACCEPTABLE;
    }

    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;

    $r->print(&display_header($command,\%checkallowed));

    &Apache::lonhtmlcommon::clear_breadcrumbs();
    &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>'/adm/whatsnew',
              text=>"What's New?"});
    if (($command eq 'chgthreshold') && $checkallowed{'abovethreshold'}) {
        &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>'/adm/whatsnew?command=chgthreshold&amp;refpage='.$refpage,
              text=>"Change thresholds"});
        $r->print(&Apache::lonhtmlcommon::breadcrumbs
            ("What's New?",#'Course_Action_Items_Thresholds'
	     ));
    } elsif (($command eq 'chginterval') && $checkallowed{'versionchanges'} ) {
        &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>'/adm/whatsnew?command=chginterval&amp;refpage='.$refpage,
              text=>"Change interval"});
        $r->print(&Apache::lonhtmlcommon::breadcrumbs
            ("What's New?",#'Course_Action_Items_Intervals'
	     ));
    } elsif (($command eq 'chgdisc') && $checkallowed{'coursediscussion'}) {
        &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>'/adm/whatsnew?command=chgdisc&amp;refpage='.$refpage,
              text=>"Change discussion display"});
        $r->print(&Apache::lonhtmlcommon::breadcrumbs
            ("What's New?",#'Course_Action_Items_Intervals'
	     ));
    } elsif ($command eq 'courseinit') {
        &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>'/adm/whatsnew?command=courseinit&amp;refpage='.$refpage,
              text=>"Course initialization preference"});
        $r->print(&Apache::lonhtmlcommon::breadcrumbs
            ("What's New?",#'Course_Action_Items_Initialization'
	     ));
    } elsif ($command eq 'chgoldroleinterval' && $checkallowed{'oldroles'}) {
        &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>'/adm/whatsnew?command=chgoldroleinterval&amp;refpage='.$refpage,
              text=>"Change interval"});
        $r->print(&Apache::lonhtmlcommon::breadcrumbs
            ("What's New?",#'Course_Action_Items_Intervals'
             ));
    } elsif ($command eq 'chgnewroleinterval' && $checkallowed{'newroles'}) {
        &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>'/adm/whatsnew?command=chgnewroleinterval&amp;refpage='.$refpage,
              text=>"Change interval"});
        $r->print(&Apache::lonhtmlcommon::breadcrumbs
            ("What's New?",#'Course_Action_Items_Intervals'
             ));
    } elsif ($command eq 'chgcrslogininterval' && $checkallowed{'crslogin'}) {
        &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>'/adm/whatsnew?command=chgcrslogininterval&amp;refpage='.$refpage,
              text=>"Change interval"});
        $r->print(&Apache::lonhtmlcommon::breadcrumbs
            ("What's New?",#'Course_Action_Items_Intervals'
             ));
    } else {
        $r->print(&Apache::lonhtmlcommon::breadcrumbs
            ("What's New?",#'Course_Action_Items_Display'
	     ));
    }
    &display_main_box($r,$command,$refpage,\%checkallowed);
    return OK;
}

#------------------------------
# display_main_box
#
# Display all the elements within the main box
#------------------------------
                                                                                
sub display_main_box {
    my ($r,$command,$refpage,$checkallowed) = @_;
    my $domain=&Apache::loncommon::determinedomain();
    my $function = &Apache::loncommon::get_users_function();
    my $lctype = lc(&Apache::loncommon::course_type());
    $r->print('<table width="100%" border="0" cellpadding="5" cellspacing="0"><tr><td width="100%">');

    my %threshold_titles = &Apache::lonlocal::texthash (
                         av_attempts => 'Average number of attempts',
                         degdiff => 'Degree of difficulty',
                         numstudents => 'Total number of students with submissions',
    );
    my %versions = (
                       -1 => "version changes since start of $lctype",
                  2592000 => 'version changes since last month',
                   604800 => 'version changes since last week',
                    86400 => 'version changes since yesterday',
                   );
    my %newroles = (
                       -1 => "roles which have become active since start of $lctype",
                  2592000 => 'roles which have become active since last month',
                   604800 => 'roles which have become active since last week',
                    86400 => 'roles which have become active since yesterday',
                   );
    my %oldroles = (
                       -1 => "roles which expired since start of $lctype",
                  2592000 => 'roles which expired since last month',
                   604800 => 'roles which expired since last week',
                    86400 => 'roles which expired since yesterday',
                   );
    my %crslogins = (
                       -1 => 'last logins for anyone who has ever logged in',
                  2592000 => 'last logins for users in last 30 days',
                   604800 => 'last logins for users in last 7 days',
                    86400 => 'last logins for users in last 24 hours',
                   );
    my %interval_titles = (
        versions => \%versions,
        newroles => \%newroles,
        oldroles => \%oldroles,
        crslogin => \%crslogins,
    );
    my %initpage = &Apache::lonlocal::texthash (
                     firstres => "first resource in the $lctype",
                     whatsnew => "What's New Page",
                     userpref => 'your general user preferences',
                     coursespecific => "specific setting for this $lctype",
                   );
    my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $crs = $env{'course.'.$env{'request.course.id'}.'.num'};

    if (($command eq 'chgthreshold') 
	&& $checkallowed->{'abovethreshold'}) {
        &display_threshold_config($r,$refpage,\%threshold_titles,
                                                                   $cdom,$crs);
    } elsif (($command eq 'chginterval') 
	     && $checkallowed->{'versionchanges'}) {
        &display_interval_config($r,$refpage,\%interval_titles,'versions');
    } elsif (($command eq 'chgdisc') 
	     && $checkallowed->{'coursediscussion'}) {
        &display_discussion_config($r,$refpage);
    } elsif ($command eq 'courseinit') {
        &courseinit_config($r,$refpage,\%initpage);
    } elsif (($command eq 'chgnewroleinterval')
             && $checkallowed->{'newroles'}) {
        &display_interval_config($r,$refpage,\%interval_titles,'newroles');
    } elsif (($command eq 'chgoldroleinterval')
             && $checkallowed->{'oldroles'}) {
        &display_interval_config($r,$refpage,\%interval_titles,'oldroles');
    } elsif (($command eq 'chgcrslogininterval')
             && $checkallowed->{'crslogin'}) {
        &display_interval_config($r,$refpage,\%interval_titles,'crslogin');
    } else {
        &display_actions_box($r,$command,$refpage,\%threshold_titles,
                        \%interval_titles,\%initpage,$cdom,$crs,$checkallowed);
    }
    my $end_page = &Apache::loncommon::end_page();
    $r->print(<<END_OF_BLOCK);
  </td>
 </tr>
</table><br />
$end_page
END_OF_BLOCK
}

#-------------------------------
# display_header
#
# Display the header information and set
# up the HTML
#-------------------------------

sub display_header {
    my ($command,$checkallowed) = @_;
    
    my $scripttag;
    unless ($command eq 'chgthreshold' || $command eq 'chginterval' || 
            $command eq 'chgoldroleinterval' ||
            $command eq 'chgnewroleinterval' || $command eq 'chgcrslogininterval') {
       $scripttag = <<"END";
<script type="text/javascript">
// <![CDATA[
function change_display(caller,change) {
    caller.value = change;
    document.visible.submit();
}

function changeAll(change) {
END
        foreach my $item (keys(%{$checkallowed})) {
	    if ($item =~ /_section$/) { next; }
            if ($$checkallowed{$item}) {
                $scripttag.='document.visible.display_'.$item.'.value=change'.
                            "\n";
            }
        }
        $scripttag.='document.visible.submit();
}

function thresholdreset() {
    document.visible.command.value="reset";
    document.visible.submit();
}

function togglelogins() {
    var total = document.visible.logincount.value;
    var sumrow = document.visible.loginrow.value;
    if (total == 0) {
        return;
    }
    var showlogindetails = 0;
    for (var i=0; i<document.visible.logindetails.length; i++) {
        if (document.visible.logindetails[i].checked) {
            showlogindetails = document.visible.logindetails[i].value;
        }
    }
    var detval = "none";
    var sumval = "";
    if (showlogindetails == 1) {
        detval = "";
        sumval = "none";
    }
    for (var j=0; j<total; j++) {
        var counter = j+1;
        var itemid = "logindet_"+counter;
        personele = document.getElementById(itemid);
        if (personele != null) {
            personele.style.display = detval;
        }
    }
    var detheaderele = document.getElementById("logintitledet");
    if (detheaderele != null) {
        detheaderele.style.display = detval;
    }
    for (var k=0; k<sumrow; k++) {
        var counter = k+1;
        var itemid = "loginsum_"+counter;
        logincatele = document.getElementById(itemid);
        if (logincatele != null) {
            logincatele.style.display = sumval;
        }
    }
    var sumheaderele = document.getElementById("logintitlesum");
    if (sumheaderele != null) {
        sumheaderele.style.display = sumval;
    }
    return;
}
// ]]>
</script>
';
    }
    my $course_type=&Apache::loncommon::course_type();
    return &Apache::loncommon::start_page("What's New?",
					  $scripttag);
}

#-------------------------------
# display_actions_box
#
# Display the action items
#
#-------------------------------
                                                                                
sub display_actions_box {
    my ($r,$command,$refpage,$threshold_titles,$interval_titles,$initpage,
        $cdom,$crs,$checkallowed) = @_;
    my $udom = $env{'user.domain'};
    my $uname = $env{'user.name'};
    my $cid = $env{'request.course.id'};
    my $crstype = &Apache::loncommon::course_type();
    my $lctype = lc($crstype);
    my %stulabel = (
                    'Course' => 'students',
                    'Community' => 'members',
                   );
    my %lt = &Apache::lonlocal::texthash(
                 'yacc' => 'You are accessing an invalid course',
                 'gtfr' => 'Go to first resource',
                 'hial' => 'Hide all',
                 'shal' => 'Show all',
    );

    my %unread = ();
    my %ungraded = ();
    my %bombed = ();
    my %triggered = ();
    my %changed = ();
    my @newmsgs = ();
    my @critmsgs = ();
    my @newdiscussions = ();
    my @tograde = ();
    my @bombs = ();
    my @warnings = ();
    my $msgcount = 0;
    my $critmsgcount = 0;
    my $expirecount;
    my %expired;
    my $activecount;
    my %activated;
    my %loggedin;
    my $logincount;
    my %res_title = ();
    my %show = ();
    my $needitems = 0;
    my $boxcount = 0;

    my $result;
    if ($command eq 'newcourseinit') {
        $result = &store_courseinit_setting($uname,$udom,$cid,$initpage);
    }

    my %threshold = ();
    my %pagedesc = &Apache::lonlocal::texthash (
                     firstres => 'First resource',
                     whatsnew => "What's New Page",
                     userpref => 'user preference',
                     coursespecific => $lctype.' only',
                     default => 'default',
                   );

    my ($initcontrol,$initdisp) = &curr_courseinit();
    my $currinit = $pagedesc{$initdisp}.' ('.$pagedesc{$initcontrol}.')';

    unless ($cid) {
        $r->print('<p><span style="text-align: center; font-weight: bold;">'.$lt{'yacc'}.'</span></p>');
        return;
    }

    my $header = '';
    if ($refpage eq 'start') {
        if (tie(my %bighash,'GDBM_File',$env{'request.course.fn'}.'.db',
            &GDBM_READER(),0640)) {
            my $furl=&HTML::Entities::encode($bighash{'first_url'},'"<>&');
            untie(%bighash);
            $header .= '<b><a href="'.$furl.'">'.$lt{'gtfr'}.
                  '</a></b><br />';
        }
    }
    $header .= &mt('Page set to be displayed after you have selected a role in this '.$lctype).'.'
              .' <span class="LC_nobreak">'
              .&mt('Currently: [_1].','<i>'.$currinit.'</i>')
              .'&nbsp;&nbsp;'
              .&mt('[_1]Change[_2] for just [_3]this '.$lctype.'[_4] or for [_5]all your courses/communities[_6].'
                  ,'<b>'
                  ,'</b>'
                  ,'<a href="/adm/whatsnew?command=courseinit&amp;refpage='.$refpage.'">'
                  ,'</a>'
                  ,'<a href="/adm/preferences?action=changecourseinit&amp;refpage='.$refpage.'">'
                  ,'</a>')
              .' </span>';

    $r->print(&Apache::loncommon::head_subbox($header));

    if ($command eq 'reset') {
        $result = &process_reset($cdom,$crs);
    } elsif ($command eq 'update') {
        $result = &process_update($uname,$udom,$threshold_titles);
    } elsif ($command eq 'newinterval') {
        $result = &store_interval_setting($uname,$udom,$cid,$interval_titles);
    } elsif ($command eq 'newdiscconf') {
        $result = &store_discussion_setting($uname,$udom,$cid);
    }

    my $store_result=&store_display_settings($uname,$udom,$cid,$checkallowed);

    unless ($store_result eq 'ok') { 
        &Apache::lonnet::logthis('Error saving whatsnew settings: '.
            $store_result.' for '.'user '.$uname.':'.$udom.' in '.$lctype.' '.$cid);
        $result .= '<span class="LC_error">'
                  .&mt('Unable to save visibility settings due to [_1]',
                       $store_result)
                  .'</span>';
    }

    if ($result) {
        $r->print($result.'<hr />');
    }
    $r->rflush();

    my (%timediff,%interval);
    my %display_settings = &get_display_settings($uname,$udom,$cid);
    $timediff{'versions'} = $display_settings{$cid.':interval'};
    unless (defined($timediff{'versions'})) { $timediff{'versions'} = 604800; } 
    $interval{'versions'} = $interval_titles->{'versions'}->{$timediff{'versions'}};

    my %headings = &Apache::lonlocal::texthash(
                coursediscussion => 'Unread '.$lctype.' discussion posts',
                handgrading      => 'Problems requiring handgrading',
                haserrors        => 'Problems with errors',
                coursenormalmail => 'New '.$lctype.' messages',
                coursecritmail   => 'New critical messages in '.$lctype,
    );

    if ($timediff{'versions'} == -1) {
        $headings{'versionchanges'} = &mt('Resources in '.$lctype.' with version changes since start of '.$lctype);
    } elsif ($timediff{'versions'} == 2592000) {
        $headings{'versionchanges'} = &mt('Resources in '.$lctype.' with version changes since last month');
    } elsif ($timediff{'versions'} == 604800) {
        $headings{'versionchanges'} = &mt('Resources in '.$lctype.' with version changes since last week');
    } elsif ($timediff{'versions'} == 86400) {
        $headings{'versionchanges'} = &mt('Resources in '.$lctype.' with version changes since yesterday');
    }

    $timediff{'oldroles'} = $display_settings{$cid.':oldroleinterval'};
    unless (defined($timediff{'oldroles'})) { $timediff{'oldroles'} = 604800; }
    $interval{'oldroles'} = $interval_titles->{'oldroles'}->{$timediff{'oldroles'}};

    if ($timediff{'oldroles'} == -1) {
        $headings{'oldroles'} = &mt('Roles for which access to '.$lctype.' has expired since start of '.$lctype);
    } elsif ($timediff{'oldroles'} == 2592000) {
        $headings{'oldroles'} = &mt('Roles for which access to '.$lctype.' has expired since last month');
    } elsif ($timediff{'oldroles'} == 604800) {
        $headings{'oldroles'} = &mt('Roles for which access to '.$lctype.' has expired since last week');
    } elsif ($timediff{'oldroles'} == 86400) {
        $headings{'oldroles'} = &mt('Roles for which access to '.$lctype.' has expired since yesterday');
    }

    $timediff{'newroles'} = $display_settings{$cid.':newroleinterval'};
    unless (defined($timediff{'newroles'})) { $timediff{'newroles'} = 604800; }
    $interval{'newroles'} = $interval_titles->{'newroles'}->{$timediff{'newroles'}};

    if ($timediff{'newroles'} == -1) {
        $headings{'newroles'} = &mt('Roles for which access to '.$lctype.' has become available since start of '.$lctype);
    } elsif ($timediff{'newroles'} == 2592000) {
        $headings{'newroles'} = &mt('Roles for which access to '.$lctype.' has become available since last month');
    } elsif ($timediff{'newroles'} == 604800) {
        $headings{'newroles'} = &mt('Roles for which access to '.$lctype.' has become available since last week');
    } elsif ($timediff{'newroles'} == 86400) {
        $headings{'newroles'} = &mt('Roles for which access to '.$lctype.' has become available since yesterday');
    }

    $timediff{'crslogin'} = $display_settings{$cid.':crslogininterval'};
    unless (defined($timediff{'crslogin'})) { $timediff{'crslogin'} = 604800; }
    $interval{'crslogin'} = $interval_titles->{'crslogin'}->{$timediff{'crslogin'}};

    if ($timediff{'crslogin'} == -1) {
        $headings{'crslogin'} = &mt('Last login for anyone who has ever logged in');
    } elsif ($timediff{'crslogin'} == 2592000) {
        $headings{'crslogin'} = &mt('Last login for users in last 30 days');
    } elsif ($timediff{'crslogin'} == 604800) {
        $headings{'crslogin'} = &mt('Last login for users in last 7 days');
    } elsif ($timediff{'crslogin'} == 86400) {
        $headings{'crslogin'} = &mt('Last login for users in last 24 hours');
    }

    my ($now,$starttime,$activatedstart,$expiredstart,$crsloginstart);
    $now = time;

    if ($timediff{'versions'} == -1) {
        $starttime = 0;
    } else {
        $starttime = $now - $timediff{'versions'};
    }

    if ($timediff{'newroles'} == -1) {
        $activatedstart = 0;
    } else {
        $activatedstart = $now - $timediff{'newroles'};
    }

    if ($timediff{'oldroles'} == -1) {
        $expiredstart = 0;
    } else {
        $expiredstart = $now - $timediff{'oldroles'};
    }

    if ($timediff{'crslogin'} == -1) {
        $crsloginstart = 0;
    } else {
        $crsloginstart = $now - $timediff{'crslogin'};
    }

    my $countunread = $display_settings{$cid.':countunread'};
    unless (defined($countunread)) {
        $countunread = 'on';
    }
    if ($$checkallowed{'abovethreshold'}) {
        &get_curr_thresholds(\%threshold,$uname,$udom,$cid,$cdom,$crs);
    }

    $headings{'abovethreshold'} = 
	&mt('(Problems with av. attempts &ge; [_1] or deg. difficulty &ge; [_2]) [_3] and total number of '.$stulabel{$crstype}.' with submissions &ge; [_4]',
	    $threshold{'av_attempts'},$threshold{'degdiff'},
	    '<br />',$threshold{'numstudents'});

    my @actionorder = ('handgrading','haserrors','abovethreshold','versionchanges','coursediscussion','coursenormalmail','coursecritmail','newroles','oldroles','crslogin');

    foreach my $key (keys(%{$checkallowed})) {
	if ($key =~ /_section$/) { next; }
        $show{$key} = 0;
        if ($$checkallowed{$key}) {
            unless ($display_settings{$cid.':'.$key} eq 'hide') {
                $show{$key} = 1;
            }
        }
    }

    foreach my $item (@actionorder) {
        unless ($item eq 'coursenormalmail' || $item eq 'coursecritmail' ||
                $item eq 'newroles' || $item eq 'oldroles') {
            if ($show{$item}) {
                $needitems = 1;
                last;
            }
        }
    }

    my $itemserror;
    if ($needitems) {
        $itemserror = &getitems(\%unread,\%ungraded,\%bombed,\%triggered,\%changed,\@newdiscussions,\@tograde,\@bombs,\@warnings,\%threshold,$cdom,$crs,\%res_title,\%show,$starttime,$countunread);
    }
    my $classlist;
    if ($show{'oldroles'} || $show{'newroles'} || $show{'crslogin'}) {
        $classlist = &Apache::loncoursedata::get_classlist();
    }
    if ($show{'coursenormalmail'}) {
        $msgcount = &getnormalmail(\@newmsgs);
    }
    if ($show{'coursecritmail'}) {
        $critmsgcount = &getcritmail(\@critmsgs);
    }
    if ($show{'oldroles'}) {
        $expirecount = &getexpired(\%expired,$expiredstart,'previous',$classlist);
    }
    if ($show{'newroles'}) {
        $activecount = &getactivated(\%activated,$activatedstart,'active',$classlist);
    }
    if ($show{'crslogin'}) {
        $logincount = &getloggedin($cdom,$crs,\%loggedin,$crsloginstart);
    }
    $r->print(qq|<a href="javascript:changeAll('hide');">$lt{'hial'}</a>
     &nbsp;&nbsp;<a href="javascript:changeAll('show');">$lt{'shal'}</a>
     <form method="post" name="visible" action="/adm/whatsnew">\n|);
    foreach my $item (keys(%{$checkallowed})) {
	if ($item =~ /_section$/) { next; }
        if ($$checkallowed{$item}) {
            $r->print('<input type="hidden" name="display_'.$item.'" />'."\n");
        }
    }

    $r->print('<input type="hidden" name="refpage" value="'.$refpage.'" /><table class="LC_double_column"><tr><td class="LC_left_col">');

    my $displayed = 0;
    my $totalboxes = 0;
    foreach my $key (keys(%{$checkallowed})) {
	if ($key =~ /_section$/) { next; }
	if ($key eq 'whatsnew' ) { next; } # whatsnew check creates no box
        if ($$checkallowed{$key}) {
            $totalboxes ++;
        }
    }
    my $halfway = 4;
#    my $halfway = int($totalboxes/2) + $totalboxes%2;
    foreach my $actionitem (@actionorder) {
        if ($$checkallowed{$actionitem}) {
            if ($displayed == $halfway) {
                $r->print('</td><td>&nbsp;</td><td class="LC_right_col" >');
            }
            &display_launcher($r,$actionitem,$refpage,$checkallowed,\%show,\%headings,\%res_title,\@tograde,\%ungraded,\@bombs,\%bombed,\%changed,\@warnings,\%triggered,\@newdiscussions,\%unread,$msgcount,\@newmsgs,$critmsgcount,\@critmsgs,\%interval,$countunread,\%expired,$expirecount,\%activated,$activecount,$crstype,$itemserror,\%loggedin,$logincount,$classlist);
            $displayed ++; 
        }
    }
    $r->print('
      </td>
    </tr>
   </table>
   </form>
');
}

#-------------------------------
# display_threshold_config
#
# Display the threshold setting screen 
#
#-------------------------------
                                                                                
sub display_threshold_config {
    my ($r,$refpage,$threshold_titles,$cdom,$crs) = @_;
    my $uname = $env{'user.name'};
    my $udom = $env{'user.dom'};
    my $cid = $env{'request.course.id'};
    my %threshold = ();
    my $rowColor1 = "#ffffff";
    my $rowColor2 = "#eeeeee";
    my $rowColor;

    my @thresholditems = ("av_attempts","degdiff","numstudents");
    my %threshold_titles = &Apache::lonlocal::texthash(
                         av_attempts => 'Average number of attempts',
                         degdiff => 'Degree of difficulty',
                         numstudents => 'Total number of students with submissions',
                         );
    &get_curr_thresholds(\%threshold,$uname,$udom,$cid,$cdom,$crs);

    $r->print('<br /><form name="thresholdform" method="post" action="/adm/whatsnew">'.
              &Apache::loncommon::start_data_table().
              &Apache::loncommon::start_data_table_header_row().
             '<th>'.&mt('Threshold Name').'</th>'."\n".
             '<th>'.&mt('Current value').'</th>'."\n".
             '<th>'.&mt('Change?').'</th>'."\n".
              &Apache::loncommon::end_data_table_header_row());
    foreach my $type (@thresholditems) {
        my $parameter = $env{'request.course.id'}.':threshold_'.$type;
# onchange is javascript to automatically check the 'Set' button.
        my $onchange = 'onfocus="javascript:window.document.forms'.
              "['thresholdform'].elements['".$parameter."_setparmval']".
              '.checked=true;"';
        $r->print(&Apache::loncommon::start_data_table_row()."\n".
                 '<td>'.$threshold_titles{$type}.'</td>'."\n".
                 '<td>'.&Apache::lonhtmlcommon::textbox($parameter.'_value',
                                            $threshold{$type},
                                            10,$onchange).'</td>'."\n".
                 '<td>'.
                 &Apache::lonhtmlcommon::checkbox($parameter.'_setparmval').
                 '</td>'."\n".
                 &Apache::loncommon::end_data_table_row());
    }
    $r->print(&Apache::loncommon::end_data_table()."\n".
          '<br /><input type="submit" name="threshold" value="'.&mt('Save').'" />
                 <input type="hidden" name="command" value="update" />
                 <input type="hidden" name="refpage" value="'.$refpage.'" />
               </form>');
}

#-------------------------------
# display_interval_config
#
# Display the interval setting screen
#
#-------------------------------
                                                                                   
sub display_interval_config {
    my ($r,$refpage,$interval_titles,$context) = @_;
    my $setting = 'interval';
    if ($context eq 'oldroles') {
        $setting = 'oldroleinterval';
    } elsif ($context eq 'newroles') {
        $setting = 'newroleinterval'; 
    }
    my $lctype = lc(&Apache::loncommon::course_type());
    my $current = &get_current($env{'user.name'},$env{'user.domain'},
                                $env{'request.course.id'},$setting);
    if ($context eq 'oldroles') {
        $r->print('<br />'.&mt('Choose the time window to use to display roles for which access to the '.$lctype.' expired.').'<br />');
    } elsif ($context eq 'newroles') {
        $r->print('<br />'.&mt('Choose the time window to use to display roles for which access to the '.$lctype.' became available.').'<br />');
    } elsif ($context eq 'crslogin') {
        $r->print('<br />'.&mt('Choose the time window to use to display the last login by a user in the '.$lctype).'<br />');
    } else {
        $r->print('<br />'.&mt('Choose the time window to use to display resources in the '.$lctype.' with version changes.').'<br />');
    }
    unless ($current eq '') {
        if (ref($interval_titles->{$context}) eq 'HASH') {
            $r->print(' '.&mt('Current value is "[_1]".','<b>'.
                      $interval_titles->{$context}->{$current}.'</b>').'<br />');
        }
    }
    $r->print('<br />
<form method="post" name="intervalswitch" action="/adm/whatsnew">
<input type="hidden" name="command" value="newinterval" />
<input type="hidden" name="intervaltype" value="'.$context.'" />
<input type="hidden" name="refpage" value="'.$refpage.'" />'.
&mt('Display:').' 
<select name="interval">
<option value="" selected="selected">'.&mt('Select').'</option>
');
    if (ref($interval_titles) eq 'HASH') {
        if (ref($interval_titles->{$context}) eq 'HASH') {
            foreach my $key (reverse sort ({$a cmp $b} (keys(%{$interval_titles->{$context}})))) {
                $r->print('<option value="'.$key.'">'.&mt($interval_titles->{$context}->{$key}).
                          '</option>'."\n");
            }
        }
    }
    $r->print('</select>&nbsp;&nbsp;
               <input type="submit" name="display" value="'.
               &mt('Save').'" /></form>');
    return;
}

#----------------------------------------------
# display_discussion_config
#
# Display the discussion display setting screen
#
#----------------------------------------------
                                                                                  
sub display_discussion_config {
    my ($r,$refpage) = @_;
    my $current = &get_current($env{'user.name'},$env{'user.domain'},
                                $env{'request.course.id'},'countunread');
    if ($current eq '') {
        $current = 'on';
    }
    my %opposite = ( 
                      'on' => 'off',
                      'off' => 'on',
                    );
    $r->print('<script type="text/javascript">
function toggle_countunread(choice) {
    if (choice == "unchanged") {
        document.discussionswitch.command.value = "";
    }
    document.discussionswitch.submit();
}
</script>');
    $r->print('<br />'
             .&mt('Choose whether or not to display a count of the number of new posts for each resource or discussion board which has unread posts.')
             .'<br />'
             .&mt("This can increase the time taken to gather data for the [_1]What's New Page[_2] by a few seconds.",'<i>','</i>')
             .'&nbsp;&nbsp;'
             .&mt('Currently set to [_1].','<b>'.$current.'</b>')
    );
    $r->print('<br /><br />
<form method="post" name="discussionswitch" action="/adm/whatsnew">
<input type="hidden" name="command" value="newdiscconf" />
<input type="hidden" name="refpage" value="'.$refpage.'" />
<input type="hidden" name="countunread" value="'.$opposite{$current}.'" />
');
    $r->print('<br/>
               <input type="button" name="display" value="'.
               &mt('Change to [_1]',&mt($opposite{$current})).'" 
               onclick="javascript:toggle_countunread('."'change'".')" />'.
               ('&nbsp;' x7).
               '<input type="button" name="nochange" value="'.
               &mt("No change").'" 
               onclick="javascript:toggle_countunread('."'unchanged'".')" />
               </form>');
    return;
}

#---------------------------------------------------
# courseinit_config
#
# Set page displayed when course loads after 
# selecting a role in the course from the roles page. 
#
#---------------------------------------------------

sub courseinit_config {
    my ($r,$refpage,$initpage) = @_;
    my ($control,$current) = &curr_courseinit();
    my @chgstate = ('userpref','coursespecific');
    my @chgentry = ('firstres','whatsnew');
    my $lctype = lc(&Apache::loncommon::course_type());
    my %lt = &Apache::lonlocal::texthash(
                             'chwp' => "Choose which page will be displayed when you enter this $lctype after selecting a role.",
                             'cuva' => 'Current value is determined by',
                             'anis' => 'and is set to display',
                             'padc' => 'Page display controlled by',
                             'chce' => 'Choose '.$lctype.' entry',
                             'moce' => 'Save',
    );
    $r->print(<<"END"); 
<br />$lt{'chwp'}
<br />$lt{'cuva'}: <b>
$$initpage{$control}</b> $lt{'anis'} <b>
$$initpage{$current}</b>.<br /><br />
<form method="post" name="courseinitswitch" action="/adm/whatsnew">
<input type="hidden" name="command" value="newcourseinit" />
<input type="hidden" name="refpage" value="$refpage" />
$lt{'padc'}:&nbsp;&nbsp;
END
    foreach my $choice (@chgstate) {
        my $chkstring;
        if ($choice eq $control) {
            $chkstring = ' checked="checked" ';
        }  
        $r->print('<span class="LC_nobreak"><label><input type="radio" name="courseinit_control" value="'.
                   $choice.'"'.$chkstring.'/>'.$$initpage{$choice}.
                   '&nbsp;&nbsp;</label></span>');
    }
    $r->print('<br /><br />'.&mt('If').' '.$$initpage{'coursespecific'}.
              ' - <br />'.$lt{'chce'}.": \n");
    foreach my $choice (@chgentry) {
        my $chkstring;
        if (($choice eq $current) && ($control eq 'coursespecific')) {
            $chkstring = ' checked="checked" ';
        }
        $r->print('<span class="LC_nobreak"><label><input type="radio" name="courseinit_page" value="'.
                  $choice.'"'.$chkstring.'/>'.$$initpage{$choice}.
                  '&nbsp;&nbsp;</label></span>');
    }
    $r->print('<br /><br /><input type="submit" name="display" value="'.
               $lt{'moce'}.'" /></form>');
    return;
}

sub curr_courseinit {
    my $current = &get_current($env{'user.name'},$env{'user.domain'},
                                $env{'request.course.id'},'courseinit');
    my $control;
    if ($current) {
        $control = 'coursespecific';
    } else {
        $control = 'userpref';
        my %userenv = &Apache::lonnet::get('environment',
                                                      ['course_init_display']);
        if (exists($userenv{'course_init_display'})) {
            $current = $userenv{'course_init_display'};
        }
        unless ($current) {
            $current = 'whatsnew';
        }
    }
    return ($control,$current);
}

sub display_launcher {
    my ($r,$action,$refpage,$checkallowed,$show,$headings,$res_title,
        $tograde,$ungraded,$bombs,$bombed,$changed,$warnings,$triggered,
        $newdiscussions,$unread,$msgcount,$newmsgs,$critmsgcount,$critmsgs,
        $interval,$countunread,$expired,$expirecount,$activated,$activecount,
        $crstype,$itemserror,$loggedin,$logincount,$classlist) = @_;

    if ($$checkallowed{$action}) {
        &start_box($r,$show,$headings,$action,$refpage);
        if ($$show{$action}) {
            if ($action eq 'handgrading') {    # UNGRADED ITEMS
                &display_handgrade($r,$tograde,$ungraded,$itemserror);
            } elsif ($action eq 'haserrors') { # BOMBS
                &display_haserrors($r,$bombs,$bombed,$res_title,$itemserror);
            } elsif ($action eq 'versionchanges') { # VERSION CHANGES
                &display_versionchanges($r,$changed,$res_title,$interval->{'versions'},$itemserror);
            } elsif ($action eq 'abovethreshold') { # DEGDIFF/AV. TRIES TRIGGERS
                &display_abovethreshold($r,$refpage,$warnings,$triggered,
					$res_title,$itemserror);
            } elsif ($action eq 'coursediscussion') { # UNREAD COURSE DISCUSSION
                &display_coursediscussion($r,$newdiscussions,$unread,
                                $countunread,$res_title,$itemserror);
            } elsif ($action eq 'coursenormalmail') { # NORMAL MESSAGES
                &display_coursenormalmail($r,$msgcount,$newmsgs);
            } elsif ($action eq 'coursecritmail') { # CRITICAL MESSAGES
                &display_coursecritmail($r,$critmsgcount,$critmsgs);
            } elsif ($action eq 'newroles') { # ACTIVATED ROLES
                &display_rolechanges($r,$activecount,$activated,$interval->{'newroles'},
                                     $crstype,$classlist);
            } elsif ($action eq 'oldroles') { # EXPIRED ROLES
                &display_rolechanges($r,$expirecount,$expired,$interval->{'oldroles'},
                                     $crstype,$classlist);
            } elsif ($action eq 'crslogin') { #LAST LOGIN
                &display_crslogins($r,$logincount,$loggedin,$interval->{'crslogin'},
                                   $crstype,$classlist);
            }
        }
        &end_box($r);
    }
    return;
}

sub getitems {
    my ($unread,$ungraded,$bombed,$triggered,$changed,$newdiscussions,
        $tograde,$bombs,$warnings,$threshold,$cdom,$crs,$res_title,$show,
        $starttime,$countunread) = @_;
    my $navmap = Apache::lonnavmaps::navmap->new();
    if (!defined($navmap)) {
        my $itemserror = '<span class="LC_warning">'.&mt('An error occurred retrieving information about the course.').'<br />'.&mt('It is recommended that you [_1]re-select the course[_2].','<a href="/adm/roles">','</a>').'</span>';
        return $itemserror;
    }
    # force retrieve Resource to seed the part id cache we'll need it later
    my @allres=$navmap->retrieveResources(undef,
                     sub {if ($_[0]->is_problem) { $_[0]->parts();} return 1;});
    my %resourcetracker;
    my $discussiontime;

# Resource version changes
    if ($$show{'versionchanges'}) {
        &checkversions($cdom,$crs,$navmap,$changed,$starttime);
    }

    if ($$show{'abovethreshold'}) {
        %resourcetracker =  &Apache::lonnet::dump('nohist_resourcetracker',
                                                                   $cdom,$crs);
    }

    foreach my $resource (@allres) {
        my $result = '';
        my $applies = 0;
        my $symb = $resource->symb();
        %{$$bombed{$symb}} = ();
        %{$$ungraded{$symb}} = ();
        %{$$triggered{$symb}} = ();
        $$triggered{$symb}{numparts} = 0;
        if ($resource->encrypted()) {
            $$triggered{$symb}{'enclink'} = $resource->link();
            $$triggered{$symb}{'encsymb'} = $resource->shown_symb();
        }
        my $title = $resource->compTitle();
        $$res_title{$symb} = $title;
        my $ressymb = $resource->wrap_symb();

# Check if there are unread discussion postings
        if ($$show{'coursediscussion'}) {
            &check_discussions($resource,$symb,$ressymb,$title,
			       $newdiscussions,$unread,$countunread);
        }

# Check for ungraded problems
        if ($resource->is_problem()) {
            if ($$show{'handgrading'}) {
                &check_handgraded($resource,$symb,$title,$cdom,$crs,$ungraded,
                                                                     $tograde);
            }
        }

# Check for bombs
        if ($$show{'haserrors'}) {
            &check_bombed($resource,$symb,$title,$bombs,$bombed);
        }

# Maxtries and degree of difficulty for problem parts, unless handgradeable
        if ($$show{'abovethreshold'}) {  
            &check_thresholds($resource,$symb,\%resourcetracker,
			      $triggered,$threshold,$warnings);
        }

    }
    return; 
}

sub check_discussions {
    my ($resource,$symb,$ressymb,$title,$newdiscussions,$unread,
	$countunread) = @_;

    if (!$resource->hasDiscussion()) { return; }

    %{$$unread{$ressymb}} = ();
    $$unread{$ressymb}{'title'} = $title;
    $$unread{$ressymb}{'symb'} = $symb;
    if ($resource->encrypted()) {
        $$unread{$ressymb}{'enclink'} = $resource->link();
        $$unread{$ressymb}{'encsymb'} = $resource->shown_symb();
    }
    push(@{$newdiscussions}, $ressymb);
    
    $$unread{$ressymb}{'lastpost'} = $resource->last_post_time();
    
    if ($countunread eq 'on') {
	$$unread{$ressymb}{'unreadcount'} = 
                            $resource->discussion_info('unread');
    }
}

sub check_handgraded {
    my ($resource,$symb,$title,$cdom,$cnum,$ungraded,$tograde) = @_;
    if ($resource->is_problem()) {
        my ($map,$ind,$url)=&Apache::lonnet::decode_symb($symb);
        my $partlist=$resource->parts();
        my $handgradeable;
        foreach my $part (@$partlist) {
            if ($resource->handgrade($part) eq 'yes') {
                $handgradeable=1; last;
            }
        }
        if ($handgradeable) {
            my @ungraded = &Apache::bridgetask::get_queue_symb_status(
                                             'gradingqueue',$symb,$cdom,$cnum);
            if (@ungraded > 0) {
                $$ungraded{$symb}{count} = scalar(@ungraded);
                $$ungraded{$symb}{title} = $title;
                if ($resource->encrypted()) {
                    $$ungraded{$symb}{'enclink'} = $resource->link();
                    $$ungraded{$symb}{'encsymb'} = $resource->shown_symb();
                }
                push(@{$tograde},$symb);
            }
        }
    }
}

sub check_bombed {
    my ($resource,$symb,$title,$bombs,$bombed) = @_;
    if ($resource->getErrors()) {
        my $errors = $resource->getErrors();
        $errors =~ s/^,//;
        my @bombs = split(/,/, $errors);
        my $errorcount = scalar(@bombs);
        my $errorlink = '<a href="/adm/email?display='.
                        &escape($bombs[0]).'">'.
                        $title.'</a>';
        $$bombed{$symb}{errorcount} = $errorcount;
        $$bombed{$symb}{errorlink} = $errorlink;
        push(@{$bombs}, $symb);
    }
}

sub check_thresholds {
    my ($resource,$symb,$resourcetracker,$triggered,$threshold,$warnings) = @_;
# Compile maxtries and degree of difficulty for problem parts, unless handgradeable
    my @parts = @{$resource->parts()};
    my %stats;
    my %lastreset = ();
    my $warning = 0;
    foreach my $part (@parts) {
        if ($resource->handgrade($part) eq 'yes') {
            next;
        }
        if ($resource->is_anonsurvey($part)) {
            next;
        }
        if ($resource->is_survey($part)) {
            next;
        }
        %{$stats{$part}} = ();
        my ($attempts,$users,$corrects,$degdiff,$av_attempts);
        if (exists($$resourcetracker{$symb."\0".$part."\0attempts"})) {
            $attempts = $$resourcetracker{$symb."\0".$part."\0attempts"};
        }
        if (exists($$resourcetracker{$symb."\0".$part."\0users"})) {
            $users = $$resourcetracker{$symb."\0".$part."\0users"};
        }
        if (exists($$resourcetracker{$symb."\0".$part."\0correct"})) {
            $corrects = $$resourcetracker{$symb."\0".$part."\0correct"};
        }
        if ($attempts > 0) {
            $degdiff =  1 - ($corrects/$attempts);
            $degdiff = sprintf("%.2f",$degdiff);
        }
        if ($users > 0) {
            $av_attempts = $attempts/$users;
            $av_attempts = sprintf("%.2f",$av_attempts);
        }
        &Apache::lonnet::statslog($symb,$part,$users,$av_attempts,$degdiff);
        if ((($degdiff ne '' && $degdiff >= $$threshold{'degdiff'}) || ($av_attempts ne '' && $av_attempts >= $$threshold{'av_attempts'})) && ($users >= $$threshold{'numstudents'})) {
            $stats{$part}{degdiff} = $degdiff;
            $stats{$part}{attempts} = $av_attempts;
            $stats{$part}{users} = $users;
            $lastreset{$part} = $$resourcetracker{$symb."\0".$part."\0resettime"};
            if ($lastreset{$part}) {
                $lastreset{$part} = &Apache::lonnavmaps::timeToHumanString($lastreset{$part});
            }
            $warning = 1;
        }
    }
    if ($warning) {
        $$triggered{$symb}{title} = $resource->title;
        my $partcount = 0;
        @{$$triggered{$symb}{text}} = ();
        foreach my $part (@parts) {
            if (exists($stats{$part}{users})) {
                my $resetname = 'reset_'.&escape($symb."\0".$part);
                my $resettitle = 'title_'.&escape($symb."\0".$part);
                if (@parts > 1) {
                    $$triggered{$symb}{text}[$partcount] = '
                     <td>'.&mt('part - ').$part.'</td>';
                } else {
                    $$triggered{$symb}{text}[$partcount] = '
                     <td>'.&mt('single part').'</td>';
                }
                $$triggered{$symb}{text}[$partcount] .= '
                     <td>'.$stats{$part}{users}.'</td>
                     <td>'.$stats{$part}{attempts}.'</td>
                     <td>'.$stats{$part}{degdiff}.'</td>
                     <td>'.$lastreset{$part}.'</td>
                     <td><input type="checkbox" name="'.$resetname.'" /><input type="hidden" name="'.$resettitle.'" value="'.&escape($$triggered{$symb}{title}).'" /></td>';
                $partcount ++;
            }
            $$triggered{$symb}{numparts} = $partcount;
        }
        push(@{$warnings},$symb);
    }
}


sub get_curr_thresholds {
    my ($threshold,$uname,$udom,$cid,$cdom,$crs) = @_;
    # set default values
    %$threshold = (av_attempts => 2,
                  degdiff => 0.5,
                  numstudents => 2
                  );
    my %thresholdsettings = &Apache::lonnet::dump('nohist_whatsnew',$udom,
                                                     $uname,$cid.':threshold');
    my $thresholdcount = 0;
    my ($tmp) = %thresholdsettings;
    unless ($tmp =~ /^(con_lost|error|no_such_host)/i) {
        foreach my $item (keys %{$threshold}) { 
            if (exists($thresholdsettings{$cid.':threshold_'.$item})) {
                $$threshold{$item} = 
                             $thresholdsettings{$cid.':threshold_'.$item};
                $thresholdcount ++;
            }
        }
    }
    if ($thresholdcount == 3) {
        return;
    }
    my %coursesettings = &Apache::lonnet::dump('environment',
                                              $cdom,$crs,'internal.threshold');
    my ($temp) = %coursesettings;
    unless ($temp =~ /^(con_lost|error|no_such_host)/i) {  
        foreach my $item (keys %{$threshold}) {
            unless (exists($thresholdsettings{$cid.':threshold_'.$item})) {
                if (exists($coursesettings{'internal.threshold_'.$item})) {
                    $$threshold{$item} = 
                             $coursesettings{'internal.threshold_'.$item};
                }
            }
        }
    }
    return;
}

sub get_current {
    my ($uname,$udom,$cid,$caller) = @_;
    my $currvalue;
    my %settings = &Apache::lonnet::dump('nohist_whatsnew',$udom,$uname,$cid.
                                                                ':'.$caller);
    my ($tmp) = %settings;
    unless ($tmp =~ /^(con_lost|error|no_such_host)/i) {
        $currvalue = $settings{$cid.':'.$caller};
    }
    return $currvalue;
}

sub process_reset {
    my ($dom,$crs) = @_;
    my $result = '<b>'.&mt('Counters reset for following problems (and parts):').
                           '</b><br />';
    my @agg_types = ('attempts','users','correct');
    my %agg_titles = &Apache::lonlocal::texthash (
                     attempts => 'Number of submissions',
                     users => 'Students with submissions',
                     correct => 'Number of correct submissions',
                     );
    my @resets = ();
    my %titles = ();
    foreach my $key (keys(%env)) {
        next if ($key !~ /^form\.reset_(.+)$/);
        my $title = &unescape($env{'form.title_'.$1});
        my $reset_item = &unescape($1);
        my %curr_aggregates = &Apache::lonnet::dump('nohist_resourcetracker',$dom,$crs,$reset_item);
        my %aggregates = ();
        my ($symb,$part) = split(/\0/,$reset_item);
        foreach my $type (@agg_types) {
            $aggregates{$reset_item."\0".$type} = 0;
        }  
	$aggregates{$reset_item."\0".'resettime'} = time;
        my $putresult = &Apache::lonnet::put('nohist_resourcetracker',\%aggregates,
                          $dom,$crs);
        if ($putresult eq 'ok') {
            $result .= $title.' -part '.$part.': ';
            my %new_aggregates = &Apache::lonnet::dump('nohist_resourcetracker',$dom,$crs,$reset_item);
            foreach my $type (@agg_types) {
                $result .= $agg_titles{$type}.' = '.$new_aggregates{$reset_item."\0".$type}.'; ';
            }
            $result =~ s/; $//;
            $result .= '<br />';
        } else {
            $result = $title.' -part '.$part.': '.&mt('Unable to reset counters to zero due to [_1]',$putresult).'.<br />'."\n";
        }
    }
    return $result;
}

sub process_update {
    my ($uname,$udom,$threshold_titles) = @_;
    my $setoutput = '<b>'.&mt('Changes to threshold(s) for problem tracking:').'</b><br/><br />';
    foreach (keys %env) {
        next if ($_!~/^form\.(.+)\_setparmval$/);
        my $name  = $1;
        my $value = $env{'form.'.$name.'_value'};
        if ($name && defined($value) && ($value ne '')) {
            my $put_result = &Apache::lonnet::put('nohist_whatsnew',
                                                  {$name=>$value},$udom,$uname);
           
            my ($shortname) = ($name =~ /^\Q$env{'request.course.id'}\E:threshold_(.+)$/);
            if ($put_result eq 'ok') {
                $setoutput.= &Apache::lonhtmlcommon::confirm_success(&mt('Set threshold for [_1] to [_2]',
				'<b>'.$$threshold_titles{$shortname}.'</b>',
				'<b>'.$value.'</b>').'<br />');
	    } else {
                $setoutput.= &Apache::lonhtmlcommon::confirm_success(&mt('Unable to set threshold for [_1] to [_2] due to [_3].',
				'<b>'.$name.'</b>','<b>'.$value.'</b>',
				'<tt>'.$put_result.'</tt>').'<br />',1);
            }
        }
    }
    return &Apache::loncommon::confirmwrapper($setoutput);
}

sub getnormalmail {
    my ($newmsgs) = @_;
# Check for unread mail in course
    my $msgcount = 0;

    my @messages = sort(&Apache::lonnet::getkeys('nohist_email'));
    foreach my $message (@messages) {
	my $msgid=&escape($message);
        my ($sendtime,$shortsubj,$fromname,$fromdom,$status,$fromcid)=
            &Apache::lonmsg::unpackmsgid($msgid);
        if (($fromcid) && ($fromcid eq $env{'request.course.id'})) {
            if (defined($sendtime) && $sendtime!~/error/) {
                if ($status eq 'new') {
                $sendtime = &Apache::lonlocal::locallocaltime($sendtime);
		    $msgcount ++;
                    if ($shortsubj eq '') {
                        $shortsubj = &mt('No subject');
                    }
                    push(@{$newmsgs}, {
                        msgid    => $msgid,
                        sendtime => $sendtime,
                        shortsub => $shortsubj,
                        from     => $fromname,
                        fromdom  => $fromdom
                        });
                }
            }
        }
    }
    return $msgcount;
}

sub getcritmail {
    my ($critmsgs) = @_; 
# Check for critical messages in course
    my %what=&Apache::lonnet::dump('critical');
    my $result = '';
    my $critmsgcount = 0;
    foreach my $msgid (sort(keys(%what))) {
        my ($sendtime,$shortsubj,$fromname,$fromdom,$status,$fromcid)=
            &Apache::lonmsg::unpackmsgid($msgid);
        if (($fromcid) && ($fromcid eq  $env{'request.course.id'})) {
            if (defined($sendtime) && $sendtime!~/error/) {
                $sendtime = &Apache::lonlocal::locallocaltime($sendtime);
                $critmsgcount ++;
                if ($shortsubj eq '') {
                    $shortsubj = &mt('No subject');
                }
                push(@{$critmsgs}, {
                        msgid    => $msgid,
                        sendtime => $sendtime,
                        shortsub => $shortsubj,
                        from     => $fromname,
                        fromdom  => $fromdom
                        });
            }
        }
    }
    return $critmsgcount;
}

sub getexpired {
    my ($rolechgs,$rolechgtime,$status,$classlist) = @_;
    my $expirecount = &getrolechanges($rolechgs,$rolechgtime,$status,$classlist);
    return $expirecount;
}

sub getactivated {
    my ($rolechgs,$rolechgtime,$status,$classlist) = @_;
    my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
    my $now = time();
    my $context = 'course';
    my ($permission,$allowed) =
        &Apache::lonuserutils::get_permission($context);
    my $viewablesec = &Apache::lonuserutils::viewable_section($permission);
    my %changes=&Apache::lonnet::dump('nohist_rolelog',$cdom,$cnum);
    my (%stucounted,%advcounted);
    my $activatedcount = 0;
    if (keys(%changes) > 0) {
        foreach my $chg (keys(%changes)) {
            if (ref($changes{$chg}) eq 'HASH') {
                next if ($changes{$chg}{'delflag'});
                if ($rolechgtime > 0) {
                    next if ($changes{$chg}{'exe_time'} < $rolechgtime);
                }
                if ($changes{$chg}{'exe_time'}) {
                    my $timestamp = $changes{$chg}{'exe_time'};
                    if (ref($changes{$chg}{'logentry'}) eq 'HASH') {
                        my $end = $changes{$chg}{'logentry'}{'end'};
                        next if ($end && $end <= $now);
                        my $start = $changes{$chg}{'logentry'}{'start'};
                        next if ($start >= $timestamp);
                        my $section = $changes{$chg}{'logentry'}{'section'};
                        if (($viewablesec ne '') && ($section ne '')) {
                            next if ($viewablesec ne $section);
                        }
                        my $role = $changes{$chg}{'logentry'}{'role'};
                        my $uname = $changes{$chg}{'uname'};
                        my $udom = $changes{$chg}{'udom'};
                        if ($role eq 'st') {
                            $stucounted{$uname.':'.$udom.':'.$section} = $start.':'.$end;
                        } else {
                            $advcounted{$uname.':'.$udom.':'.$role.':'.$section} = $start.':'.$end;
                        }
                        my %chginfo = (
                                       'section' => $section,
                                       'uname'   => $uname,
                                       'udom'    => $udom,
                                       'role'    => $role,
                                       'status'  => $status,
                        );
                        $activatedcount ++;
                        push (@{$rolechgs->{$timestamp}},\%chginfo);
                    }
                }
            }
        }
    }
    $activatedcount += &getrolechanges($rolechgs,$rolechgtime,$status,$classlist,\%stucounted,
                                       \%advcounted);
    return $activatedcount;
}

sub getrolechanges {
    my ($rolechgs,$rolechgtime,$status,$classlist,$stucountref,$advcountref) = @_;
    my (%stucounted,%advcounted);
    if (ref($stucountref) eq 'HASH') {
        %stucounted = %{$stucountref};
    }
    if (ref($advcountref) eq 'HASH') {
        %advcounted = %{$advcountref};
    }
    my $withsec = 1;
    my $hidepriv = 1;
    my $context = 'course';
    my @statuses = ($status);
    my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
    my $now = time();
    my ($permission,$allowed) =
        &Apache::lonuserutils::get_permission($context);
    my $viewablesec = &Apache::lonuserutils::viewable_section($permission);
    my $secidx = &Apache::loncoursedata::CL_SECTION();
    my $startidx = &Apache::loncoursedata::CL_START();
    my $endidx = &Apache::loncoursedata::CL_END();
    my $rolechgcount = 0;
    if (ref($classlist) eq 'HASH') {
        foreach my $key (keys(%{$classlist})) {
            my ($userstatus,$eventtime);   
            my $student = $classlist->{$key};
            if (ref($student) eq 'ARRAY') {
                my $start = $student->[$startidx];
                my $end = $student->[$endidx];
                my $sec = $student->[$secidx]; 
                my ($stuname,$studom) = split(/:/,$key);
                if ($status eq 'active') {
                    if (exists($stucounted{$key.':'.$sec})) {
                        next;
                    }
                }
                if (($end == 0) || ($end > $start)) {
                    if ($start <= $now) {
                        if ($end && $end < $now) {
                            if ($rolechgtime > 0) {
                                if ($end > $rolechgtime) {
                                    $userstatus = 'previous';
                                }
                            } else {
                                $userstatus = 'previous';
                            }
                        } else {
                            if ($rolechgtime > 0) {
                                if ($start >= $rolechgtime) {
                                    $userstatus = 'active';
                                }
                            } else {
                                $userstatus = 'active';
                            }
                        }
                    }
                }
                next if ($userstatus ne $status);
                if ($status eq 'active') {
                    $eventtime = $start;
                } else {
                    $eventtime = $end;
                }
                if (($viewablesec ne '') && ($sec ne '')) {
                    next if ($viewablesec ne $sec);
                }
                my %chginfo = (
                                'section' => $sec,
                                'uname'   => $stuname,
                                'udom'    => $studom,
                                'role'    => 'st', 
                                'status'  => $userstatus,
                              );
                $rolechgcount ++;
                push(@{$rolechgs->{$eventtime}},\%chginfo);
            }
        }
    }
    my %advrolehash = &Apache::lonnet::get_my_roles($cnum,$cdom,undef,
                              \@statuses,undef,undef,$withsec,$hidepriv);
    foreach my $item (keys(%advrolehash)) {
        my ($userstatus,$eventtime);
        my ($uname,$udom,$role,$section) = split(/:/,$item,-1);
        my ($start,$end) = split(/:/,$advrolehash{$item});
        if ($start eq '-1' && $end eq '-1') {
            next;
        } else {
            if ($status eq 'active') {
                if (exists($advcounted{$item})) {
                    next;
                }
            }
            if (($end == 0) || ($end > $start)) {
                if ($start <= $now) {
                    if ($end && $end < $now) {
                        if ($rolechgtime > 0) {
                            if ($end > $rolechgtime) {
                                $userstatus = 'previous';
                            }
                        } else {
                            $userstatus = 'previous';
                        }
                    } else {
                        if ($rolechgtime > 0) {
                            if ($start >= $rolechgtime) {
                                $userstatus = 'active';
                            }
                        } else {
                            $userstatus = 'active';
                        }
                    }
                }
            }
            next if ($userstatus ne $status);
            if ($status eq 'active') {
                $eventtime = $start;
            } else {
                $eventtime = $end;
            }
        }
        if (($viewablesec ne '') && ($section ne '')) {
            next if ($viewablesec ne $section);
        }
        my %chginfo = ( 
                        'section' => $section,
                        'uname'   => $uname,
                        'udom'    => $udom,
                        'role'    => $role,
                        'status'  => $userstatus,    
                      );
        $rolechgcount ++; 
        push (@{$rolechgs->{$eventtime}},\%chginfo); 
    }
    return $rolechgcount;
}

sub getloggedin {
    my ($cdom,$crs,$lastlogins,$starttime) = @_;
    my $context = 'course';
    my ($permission,$allowed) =
        &Apache::lonuserutils::get_permission($context);
    my $viewablesec = &Apache::lonuserutils::viewable_section($permission);
    my %crslogins=&Apache::lonnet::dump('nohist_crslastlogin',$cdom,$crs);
    my $logincount = 0;
    my ($tmp) = keys(%crslogins);
    unless ($tmp =~ /^(con_lost|error|no_such_host)/i) {
        if (keys(%crslogins) > 0) {
            foreach my $key (keys(%crslogins)) {
                my ($uname,$udom,$section,$role) = split(/:/,$key);
                my $eventtime = $crslogins{$key};
                if ($eventtime > $starttime) {
                    if (($viewablesec ne '') && ($section ne '')) {
                        next if ($viewablesec ne $section);
                    }
                    my %chginfo = (
                                   'section' => $section,
                                   'uname'   => $uname,
                                   'udom'    => $udom,
                                   'role'    => $role,
                    );
                    $logincount ++;
                    push (@{$lastlogins->{$eventtime}},\%chginfo);
                }
            }
        }
    }
    return $logincount;
}

sub checkversions {
    my ($cdom,$crs,$navmap,$changed,$starttime) = @_;
    my %changes=&Apache::lonnet::dump('versionupdate',$cdom,$crs);
    my ($tmp) = keys(%changes);
    unless ($tmp =~ /^(con_lost|error|no_such_host)/i) {
        if (keys(%changes) > 0) {
            foreach my $key (sort(keys(%changes))) {
                if ($changes{$key} > $starttime) {
                    my $version;
                    my ($root,$extension)=($key=~/^(.*)\.(\w+)$/);
                    my $currentversion=&Apache::lonnet::getversion($key);
                    my $revdate = 
                          &Apache::lonnet::metadata($root.'.'.$extension,
                                                     'lastrevisiondate');
                    $revdate =  &Apache::lonlocal::locallocaltime($revdate);
                    my $linkurl=&Apache::lonnet::clutter($key);
                    my $usedversion=$navmap->usedVersion($linkurl);
                    my @resources = $navmap->getResourceByUrl($linkurl,1);
                    if (($usedversion) && ($usedversion ne 'mostrecent')) {
                        $version = $usedversion;     
                    } else {
                        $version = $currentversion;
                    }
                    foreach my $res (@resources) {
                         if (ref($res) eq 'Apache::lonnavmaps::resource') { 
                            my $symb = $res->symb();
                            %{$$changed{$symb}} = (
                                                current => $currentversion,
                                                version => $version,
                                                revdate => $revdate,
                            );
                        }
                    }
                }
            }
        }
    }
    return;
}

sub display_handgrade {
    my ($r,$tograde,$ungraded,$itemserror) = @_;
    my %lt = &Apache::lonlocal::texthash(
                        'prna' => 'Problem Name',
                        'nmun' => 'Number ungraded',
                        'nopr' => 'No problems require handgrading',
    );
    if (@{$tograde} > 0) {
        $r->print('<tr class="LC_info_row"><td class="LC_left_item">'.
                  $lt{'prna'}.'</td><td class="LC_right_item">'.
                  $lt{'nmun'}.'</td></tr>');
        my $rowNum = 0;
        foreach my $res (@{$tograde}) {
            $rowNum ++;
            my $css_class = $rowNum%2?' class="LC_odd_row"':'';
            my ($map,$id,$url)=&Apache::lonnet::decode_symb($res);
            my $linkurl=&Apache::lonnet::clutter($url);
            $linkurl .= '?symb='.&escape($res);
            if ($$ungraded{$res}{'enclink'}) {
                $linkurl =
                    $$ungraded{$res}{'enclink'}.'?symb='.$$ungraded{$res}{'encsymb'};
            }
            $r->print('<tr'.$css_class.'><td><a href="'.$linkurl.'">'.$$ungraded{$res}{title}.'</a></td><td class="LC_right_item">'.$$ungraded{$res}{count}.'</td></tr>');
        }
    } elsif ($itemserror) {
        $r->print('<tr class="LC_empty_row"><td>'.$itemserror.'</td></tr>');
    } else {
        $r->print('<tr class="LC_empty_row"><td>'.$lt{'nopr'}.'</td></tr>');
    }
}

sub display_haserrors {
    my ($r,$bombs,$bombed,$res_title,$itemserror) = @_;
    my $bombnum = 0;
    my %lt = &Apache::lonlocal::texthash(
                                   reso => 'Resource',
                                   nmer => 'Number of errors',
                                   noer => 'No problems with errors',
    );
    if (@{$bombs} > 0) {
        $r->print('<tr class="LC_info_row"><td class="LC_left_item">'.
                  $lt{'reso'}.'</td><td class="LC_right_item">'.
                  $lt{'nmer'}.'</td></tr>');
        @{$bombs} = sort { &cmp_title($a,$b,$res_title) } @{$bombs};
        foreach my $bomb (@{$bombs}) {
            $bombnum ++;
            my $css_class = $bombnum%2?' class="LC_odd_row"':'';
            $r->print('<tr'.$css_class.'><td>'.$$bombed{$bomb}{errorlink}.
                      '</td><td class="LC_right_item">'.
                      $$bombed{$bomb}{errorcount}.'</td></tr>');
        }
    } elsif ($itemserror) {
        $r->print('<tr class="LC_empty_row"><td>'.$itemserror.'</td></tr>');
    } else {
        $r->print('<tr class="LC_empty_row"><td>'.$lt{'noer'}.'</td></tr>');
    }
    return;
}

sub display_abovethreshold {
    my ($r,$refpage,$warnings,$triggered,$res_title,$itemserror) = @_;
    my %lt = &Apache::lonlocal::texthash(
                 reso => 'Resource',
                 part => 'Part',
                 nust => 'Num. students',
                 avat => 'Av. Attempts',
                 dedi => 'Deg. Diff',
                 lare => 'Last Reset',
                 reco => 'Reset Count?',
                 rese => 'Reset counters to 0',
                 nopr => 'No problems satisfy threshold criteria',
    );
    if (@{$warnings} > 0) {
        @{$warnings} = sort { &cmp_title($a,$b,$res_title) } @{$warnings};
        $r->print('<tr class="LC_info_row">'.
		  '<td class="LC_left_item">'.$lt{'reso'}.
                  '<input type="hidden" name="command" value="" /></td>'.
		  '<td>'.$lt{'part'}.'</td><td>'.$lt{'nust'}.'</td>'.
		  '<td>'.$lt{'avat'}.'</td><td>'.$lt{'dedi'}.'</td>'.
		  '<td>'.$lt{'lare'}.'</td><td  class="LC_right_item">'.
                  $lt{'reco'}.'</td></tr>');
	my $row;
        foreach my $res (@{$warnings}) {
	    $row++;
            my ($map,$id,$url)=&Apache::lonnet::decode_symb($res);
            my $linkurl=&Apache::lonnet::clutter($url);
            my $rowspan;
            if ($$triggered{$res}{numparts} > 1) {
                $rowspan = 'rowspan="'.$$triggered{$res}{numparts}.'"';
            }
            $linkurl .= '?symb='.&escape($res);
            if ($$triggered{$res}{'enclink'}) {
                $linkurl = 
                  $$triggered{$res}{'enclink'}.'?symb='.$$triggered{$res}{'encsymb'};
            }
            my $css_class = $row%2?' class="LC_odd_row"':'';
            $r->print('<tr'.$css_class.'>'.
		      '<td class="LC_first_item" '.$rowspan.'><a href="'.$linkurl.'">'.
		      $$triggered{$res}{title}.'</a></td>');
                      if (ref($$triggered{$res}{text}) eq 'ARRAY') {
		          $r->print($$triggered{$res}{text}[0]);
                      }
                      $r->print('</tr>');
            if (ref($$triggered{$res}{text}) eq 'ARRAY') {
                if (@{$$triggered{$res}{text}} > 1) {
                    for (my $i=1; $i<@{$$triggered{$res}{text}}; $i++) {
                        $r->print('<tr '.$css_class.'>'.
                                  $$triggered{$res}{text}[$i].'</tr>');
                    }
                }
            }
        }
        $r->print('<tr class="LC_info_row"><td colspan="7" class="LC_right_item"><br /><input type="button" name="counters" value="'.$lt{'rese'}.'" onclick="javascript:thresholdreset();" /></td></tr>');
    } elsif ($itemserror) {
        $r->print('<tr class="LC_empty_row"><td>'.$itemserror.'</td></tr>');
    } else {
        $r->print('<tr class="LC_empty_row"><td>'.$lt{'nopr'}.'</td></tr>');
    }
}

sub display_versionchanges {
    my ($r,$changed,$res_title,$interval,$itemserror) = @_;
    my %lt = &Apache::lonlocal::texthash(
        'reso' => 'Resource',
        'revd' => 'Last revised',
        'newv' => 'New version',
        'veru' => 'Version used',
    );
    if (keys(%{$changed}) > 0) {
        $r->print('<tr class="LC_info_row"><td class="LC_left_item">'.
                  $lt{'reso'}.'</td><td>'.$lt{'revd'}.'</td><td>'.
                  $lt{'newv'}.'</td><td class="LC_right_item">'.
                  $lt{'veru'}.'</td></tr>');
        my @changes = sort { &cmp_title($a,$b,$res_title) } keys(%{$changed});
        my $changenum = 0;
        foreach my $item (@changes) {
            $changenum ++;
            my $css_class = $changenum%2?' class="LC_odd_row"':'';
            my ($map,$id,$url)=&Apache::lonnet::decode_symb($item);
            my $linkurl=&Apache::lonnet::clutter($url);
            $linkurl .= '?symb='.&escape($item);

            $r->print('<tr'.$css_class.'><td><a href="'.$linkurl.'">'.
                      $$res_title{$item}.'</a></td><td>'.
                      $$changed{$item}{'revdate'}.'</td><td>'.
                      $$changed{$item}{'current'}.'</td><td>'.
                      $$changed{$item}{'version'}.'</td></tr>');
        }
    } elsif ($itemserror) {
        $r->print('<tr class="LC_empty_row"><td>'.$itemserror.'</td></tr>');
    } else {
        $r->print('<tr class="LC_empty_row"><td>'
                 .&mt('No '.$interval).'</td></tr>');
    }
    return;
}

sub display_rolechanges {
    my ($r,$chgcount,$changed,$interval,$crstype,$classlist) = @_;
    my $now = time();
    my %lt = &Apache::lonlocal::texthash(
        'user'  => 'User',
        'tich'  => 'Time of change',
        'role'  => 'Role',
        'sec'   => 'Section',
        'status'  => 'Status',
    );
    if ($chgcount) {
        $r->print('<tr class="LC_info_row">'.
                  '<td class="LC_left_item">'.$lt{'tich'}.'</td>'.
                  '<td class="LC_left_item">'.$lt{'user'}.'</td>'.
                  '<td class="LC_left_item">'.$lt{'role'}.'</td>'.
                  '<td class="LC_left_item">'.$lt{'sec'}.'</td>'.
                  '<td class="LC_left_item">'.$lt{'status'}.'</td></tr>');
        if (ref($changed) eq 'HASH') {
            my @changes = sort { $b <=> $a } (keys(%{$changed}));
            my $changenum = 0;
            my $fullnameidx = &Apache::loncoursedata::CL_FULLNAME();
            foreach my $item (@changes) {
                if (ref($changed->{$item}) eq 'ARRAY') {
                    foreach my $chg (@{$changed->{$item}}) {
                        if (ref($chg) eq 'HASH') {
                            my $section;
                            my $role = 
                                &Apache::lonnet::plaintext($chg->{'role'},$crstype);
                            my $status = &mt($chg->{'status'});
                            if ($chg->{'section'} eq '') {
                                $section = &mt('none');
                            } else {
                                $section = $chg->{'section'};
                            }
                            my $uname = $chg->{'uname'};
                            my $udom = $chg->{'udom'};
                            $changenum ++;
                            my $css_class = $changenum%2?' class="LC_odd_row"':'';
                            my $fullname;
                            if (ref($classlist->{$uname.':'.$udom}) eq 'ARRAY') {
                                $fullname = $classlist->{$uname.':'.$udom}->[$fullnameidx];
                            } else {
                                $fullname = &Apache::loncommon::plainname($uname,$udom,'lastname');
                            }
                            my $link = 
                                &Apache::loncommon::aboutmewrapper($fullname,$uname,$udom);
                            $r->print('<tr'.$css_class.'>'.
                                      '<td>'.&Apache::lonlocal::locallocaltime($item).'</td>'.
                                      '<td>'.$link.'</td>'.
                                      '<td>'.$role.'</td>'.
                                      '<td>'.$section.'</td>'.
                                      '<td>'.$status.'</td></tr>');
                        }
                    }
                }
            }
        }
    } else {
        $r->print('<tr class="LC_empty_row"><td>'
                 .&mt('There are no '.$interval).'</td></tr>');
    }
    return;
}

sub display_crslogins {
    my ($r,$logincount,$loggedin,$interval,$crstype,$classlist) = @_;
    return unless (ref($classlist) eq 'HASH');
    my %lt = &Apache::lonlocal::texthash(
        'user'   => 'User',
        'role'   => 'Role',
        'sec'    => 'Section',
        'number' => 'Total number of logins',
    );
    if ($logincount) {

        my $hdr = '<tr class="LC_info_row" style="display:none" id="logintitledet">'.
                  '<td class="LC_left_item">'.$lt{'user'}.'</td>'.
                  '<td class="LC_left_item">'.$lt{'role'}.'</td>'.
                  '<td class="LC_left_item">'.$lt{'sec'}.'</td></tr>'.
                  '<tr class="LC_info_row" id="logintitlesum">'.
                  '<td class="LC_left_item">'.$lt{'number'}.'</td>'.
                  '<td class="LC_left_item">'.$lt{'role'}.'</td>'.
                  '<td class="LC_left_item">'.$lt{'sec'};
        my (%bylastname,%counts);
        if (ref($loggedin) eq 'HASH') {
            my @logins = sort { $b <=> $a } (keys(%{$loggedin}));
            my $numlogin = 0;
            my $fullnameidx = &Apache::loncoursedata::CL_FULLNAME();
            foreach my $item (@logins) {
                if (ref($loggedin->{$item}) eq 'ARRAY') {
                    foreach my $user (@{$loggedin->{$item}}) {
                        if (ref($user) eq 'HASH') {
                            my $section;
                            my $role =
                                &Apache::lonnet::plaintext($user->{'role'},$crstype);
                            my $status = &mt($user->{'status'});
                            if ($user->{'section'} eq '') {
                                $section = &mt('none');
                            } else {
                                $section = $user->{'section'};
                            }
                            $counts{$user->{'role'}}{$section} ++;
                            my $uname = $user->{'uname'};
                            my $udom = $user->{'udom'};
                            my $fullname;
                            if (ref($classlist->{$uname.':'.$udom}) eq 'ARRAY') {
                                $fullname = $classlist->{$uname.':'.$udom}->[$fullnameidx];
                            } else {
                                $fullname = &Apache::loncommon::plainname($uname,$udom,'lastname');
                            }
                            my $link =
                                &Apache::loncommon::aboutmewrapper($fullname,$uname,$udom);
                            push(@{$bylastname{$fullname}},
                                                          '<td>'.$link.'</td>'.
                                                          '<td>'.$role.'</td>'.
                                                          '<td>'.$section.'</td>');
                        }
                    }
                }
            }
            my $table;
            foreach my $person (sort(keys(%bylastname))) {
                if (ref($bylastname{$person}) eq 'ARRAY') {
                    foreach my $item (@{$bylastname{$person}}) {
                        $numlogin ++;
                        my $css_class = $numlogin%2?' class="LC_odd_row"':'';
                        $table .= '<tr'.$css_class.' style="display:none;"  id="logindet_'.$numlogin.'">'.$item.'</tr>';
                    }
                }
            }
            my $numrow = 0;
            foreach my $role (sort(keys(%counts))) {
                my $showrole = &Apache::lonnet::plaintext($role,$crstype);
                if (ref($counts{$role}) eq 'HASH') {
                    foreach my $sec (sort { $b <=> $a } (keys(%{$counts{$role}}))) {
                        $numrow ++;
                        my $css_class = $numrow%2?' class="LC_odd_row"':'';
                        $table .= '<tr '.$css_class.' id="loginsum_'.$numrow.'">'.
                                  '<td>'.$counts{$role}{$sec}.'</td>'.
                                  '<td>'.$showrole.'</td>'.
                                  '<td>'.$sec.'</td></tr>';
                   }
                }
            }
            $r->print($hdr.'<input type="hidden" name="logincount" value="'.$numlogin.
                      '" /><input type="hidden" name="loginrow" value="'.$numrow.
                      '" /></td></tr>'.$table);
        }
    } else {
        $r->print('<tr class="LC_empty_row"><td>'.
                  &mt('There are no '.$interval).
                  '<input type="hidden" name="logincount" value="'.$logincount.
                  '." /></td></tr>');
    }
    return;
}
 
sub display_coursediscussion {
    my ($r,$newdiscussions,$unread,$countunread,$res_title,$itemserror) = @_;
    my $lctype = lc(&Apache::loncommon::course_type());
    my %lt = &Apache::lonlocal::texthash(
                'loca' => 'Location',
                'type' => 'Type',
                'numn' => 'Number of new posts',
                'noun' => 'No unread posts in '.$lctype.' discussions',
                'tmlp' => 'Time of last post', 
    );
    if (@{$newdiscussions} > 0) {
        $r->print('<tr class="LC_info_row"><td class="LC_left_item">'.
                  $lt{'loca'}.'</td><td>'.
                  $lt{'type'}.'</td>');
        if ($countunread eq 'on') {
            $r->print('<td>'.$lt{'tmlp'}.'</td>'.
                      '<td class="LC_right_item">'.$lt{'numn'}.'</td>');
        } else {
            $r->print('<td class="LC_right_item">'.$lt{'tmlp'}.'</td>');
        }
        $r->print("</tr>\n");
        @{$newdiscussions} = sort { &cmp_title($a,$b,$res_title) }
                                                            @{$newdiscussions};
        my $rowNum = 0;
        foreach my $ressymb (@{$newdiscussions}) {
            $rowNum ++;
            my $forum_title = $$unread{$ressymb}{'title'};
            my $type = 'Resource';
            my $feedurl=&Apache::lonfeedback::get_feedurl($ressymb);
            my $disclink = $feedurl.'?symb='.&escape($$unread{$ressymb}{symb});
            if ($feedurl =~ /bulletinboard/) {
                $type = 'Discussion Board';
            }
            if ($$unread{$ressymb}{'enclink'}) {
                $disclink = $$unread{$ressymb}{'enclink'}.'?symb='.$$unread{$ressymb}{'encsymb'};
            }
            my $css_class = $rowNum%2?' class="LC_odd_row"':'';
            my $lastpost = &Apache::lonnavmaps::timeToHumanString(
                                               $$unread{$ressymb}{'lastpost'});
            $r->print('<tr'.$css_class.'><td><a href="'.$disclink.'">'.$forum_title.'</a>&nbsp;</td><td>'.&mt($type).'&nbsp;</td>');
            if ($countunread eq 'on') {
                my $unreadnum = $$unread{$ressymb}{'unreadcount'};
                $r->print('<td>'.$lastpost.'</td><td class="LC_right_item">'.
                          $unreadnum.'&nbsp;</td>');
            } else {
                $r->print('<td class="LC_right_item">'.$lastpost.'</td>');
            }
            $r->print("</tr>\n");
        }
    } elsif ($itemserror) {
        $r->print('<tr class="LC_empty_row"><td>'.$itemserror.'</td></tr>');
    } else {
        $r->print('<tr class="LC_empty_row"><td>'.$lt{'noun'}.'</td></tr>');
    }
}

sub display_coursenormalmail {
    my ($r,$msgcount,$newmsgs) = @_;
    my $lctype = lc(&Apache::loncommon::course_type());
    if ($msgcount > 0) {
        $r->print('<tr class="LC_info_row"><td class="LC_left_item">'.
                  &mt('Number').'</td><td>'.&mt('Subject').'</td><td>'.
                  &mt('Sender').'</td><td class="LC_right_item">'.
                  &mt('Date/Time').'</td></tr>');
        my $mailcount = 0;
        foreach my $msg (@{$newmsgs}) {
            $mailcount ++;
            my $css_class = $mailcount%2?' class="LC_odd_row"':'';
            $r->print('<tr'.$css_class.'><td>'.$mailcount
                      .'.&nbsp;</td><td><a href="/adm/email?display='
                      .$msg->{'msgid'}.'">'
                      .$msg->{'shortsub'}.'</a>&nbsp;&nbsp;</td><td>&nbsp;'
                      .$msg->{'from'}.':'.$msg->{'fromdom'}.'&nbsp;</td><td>'
                      .$msg->{'sendtime'}.'</td></tr>');
        }
    } else {
        $r->print('<tr class="LC_empty_row"><td>'.
                  &mt('No new '.$lctype.' messages').'</td></tr>');
    }
}

sub display_coursecritmail {
    my ($r,$critmsgcount,$critmsgs) = @_;
    my $lctype = lc(&Apache::loncommon::course_type());
    if ($critmsgcount > 0) {
        $r->print('<tr class="LC_info_row"><td class="LC_left_item">'.
                  &mt('Number').'</td><td>'.&mt('Subject').'</td><td>'.
                  &mt('Sender').'</td><td class="LC_right_item">'.
                  &mt('Date/Time').'</td></tr>');
        my $mailcount = 0;
        foreach my $msg (@{$critmsgs}) {
            $mailcount ++;
            my $css_class = $mailcount%2?' class="LC_odd_row"':'';
            $r->print('<tr'.$css_class.'><td>'.$mailcount.
                      '. &nbsp;</td><td><a href="/adm/email?folder=critical">'.
                      $msg->{'shortsub'}.'</a>&nbsp; &nbsp;</td><td>&nbsp;'.
                      $msg->{'from'}.':'.$msg->{'fromdom'}.'&nbsp;</td><td>'.
                      $msg->{'sendtime'}.'</td></tr>');
        }
    } else {
        $r->print('<tr class="LC_empty_row"><td>'.
                  &mt('No unread critical messages in '.$lctype).
                  '</td></tr>');
    }
}

sub cmp_title {
    my ($a,$b,$res_title) = @_;
    my ($atitle,$btitle) = (lc($$res_title{$a}),lc($$res_title{$b}));
    $atitle=~s/^\s*//;
    $btitle=~s/^\s*//;
    return $atitle cmp $btitle;
}

sub get_display_settings {
    my ($uname,$udom,$cid) = @_;
    my %settings = &Apache::lonnet::dump('nohist_whatsnew',$udom,$uname,$cid); 
    my ($tmp) = keys(%settings);
    if ($tmp=~ /^(con_lost|error|no_such_host)/i) {
        %settings = ();
        unless ($tmp =~ /^error: 2 /) {
	    my $lctype = lc(&Apache::loncommon::course_type());
            &Apache::lonnet::logthis('Error retrieving whatsnew settings: '.
            $tmp.' for '.$uname.':'.$udom.' for '.$lctype.': '.$cid);
        }
    }
    return %settings;
}

sub store_display_settings {
    my ($uname,$udom,$cid,$checkallowed) = @_;
    my %whatsnew_settings;
    my $result;
    foreach my $key (keys(%{$checkallowed})) {
	if ($key =~ /_section$/) { next; }
        if (exists($env{'form.display_'.$key})) {
            unless ($env{'form.display_'.$key} eq '') {
                $whatsnew_settings{$cid.':'.$key} = $env{'form.display_'.$key};
            }
        }
    }
    if (keys(%whatsnew_settings)) {
        $result = &Apache::lonnet::put('nohist_whatsnew',\%whatsnew_settings,
                                                                 $udom,$uname);
    } else {
        $result = 'ok';
    }
    return $result;
}

sub store_interval_setting {
    my ($uname,$udom,$cid,$interval_titles) = @_;
    my %interval_settings = ();
    my $result;
    my $context = $env{'form.intervaltype'};
    if ($env{'form.interval'} ne '') {
        if ($context eq 'oldroles') {
            $interval_settings{$cid.':oldroleinterval'} = $env{'form.interval'};
        } elsif ($context eq 'newroles') {
            $interval_settings{$cid.':newroleinterval'} = $env{'form.interval'};
        } elsif ($context eq 'crslogin') {
            $interval_settings{$cid.':crslogininterval'} = $env{'form.interval'};
        } else {
            $interval_settings{$cid.':interval'} = $env{'form.interval'};
        }
        my $outcome = &Apache::lonnet::put('nohist_whatsnew',
                                             \%interval_settings,$udom,$uname);
        if ($outcome eq 'ok') {
            if (ref($interval_titles->{$context}) eq 'HASH') {
                $result = &Apache::lonhtmlcommon::confirm_success(&mt('New filter setting: [_1].','<b>'. 
                              $interval_titles->{$context}->{$env{'form.interval'}}.'</b>').'<br />');
            }
        } else {
	    my $lctype = lc(&Apache::loncommon::course_type());
            &Apache::lonnet::logthis('Error saving whatsnew '.$context.' interval setting'.
                ' '.$outcome.' for '.$uname.':'.$udom.' in '.$lctype.' '.$cid);
            $result = &Apache::lonhtmlcommon::confirm_success(&mt('Unable to set interval to [_1] due to [_2].',
                         '<b>'.$interval_titles->{$context}->{$env{'form.interval'}}.'</b>',
                         '<tt>'.$outcome.'</tt>'),1);
        }
    }
    return &Apache::loncommon::confirmwrapper($result);
}

sub store_discussion_setting {
    my ($uname,$udom,$cid) = @_;
    my %discussion_settings;
    my $result;
    if (defined($env{'form.countunread'})) {
        $discussion_settings{$cid.':countunread'} = $env{'form.countunread'};
        my $outcome = &Apache::lonnet::put('nohist_whatsnew',
                                             \%discussion_settings,$udom,$uname);
        if ($outcome eq 'ok') {
            $result = &Apache::lonhtmlcommon::confirm_success(&mt('Count unread posts in discussions display set to [_1]',
                  '<b>'.&mt($env{'form.countunread'}).'</b>').'<br />');
                                                                                  
        } else {
	    my $lctype = lc(&Apache::loncommon::course_type());
            &Apache::lonnet::logthis('Error saving whatsnew countunread setting'.
                ' '.$outcome.' for '.$uname.':'.$udom.' in '.$lctype.' '.$cid);
            $result = &Apache::lonhtmlcommon::confirm_success(&mt('Unable to set "number unread posts display" to [_1]'.
                          ' due to [_2].',
                         '<b>'.&mt($env{'form.countunread'}).'</b>',
                         '<tt>'.$outcome.'</tt>'),1);
        }
    }
    return &Apache::loncommon::confirmwrapper($result);
}

sub store_courseinit_setting {
    my ($uname,$udom,$cid,$initpage) = @_;
    my %courseinit_settings;
    my $page_control;
    my $result;
    if (defined($env{'form.courseinit_control'})) {
        if ($env{'form.courseinit_control'} eq 'userpref') {
            $courseinit_settings{$cid.':courseinit'} = '';
            $page_control = 'global preferences';
        } else {
            if (defined($env{'form.courseinit_page'})) {
                $courseinit_settings{$cid.':courseinit'} = 
                                                  $env{'form.courseinit_page'};
                $page_control = 'course specific setting';
            }
        }
        if ($page_control) {
	    my $lctype = lc(&Apache::loncommon::course_type());
            my $outcome = &Apache::lonnet::put('nohist_whatsnew',
                                           \%courseinit_settings,$udom,$uname);
            if ($outcome eq 'ok') {
                if ($page_control eq 'global preferences') {
                    $result = &Apache::lonhtmlcommon::confirm_success(&mt("Page displayed after role selection in $lctype now set by [_1]user's global preferences[_2].",'<b>','</b>'));
                } else {
                    $result = &Apache::lonhtmlcommon::confirm_success(&mt('Page displayed after role selection in this '.$lctype.' set to [_1].'
                                 ,'<b>'.$$initpage{$env{'form.courseinit_page'}}.'</b>'));
                }
            } else {
                &Apache::lonnet::logthis('Error saving whatsnew courseinit '.
                                         'setting: '.$outcome.' for '.$uname.
                                         ':'.$udom.' in '.$lctype.' '.$cid);
                if ($page_control eq 'global preferences') {
                    $result = &Apache::lonhtmlcommon::confirm_success(&mt('Unable to set control of page display to [_1]'.
                          ' due to [_2].',
                         '<b>'.$page_control.'</b>',
                         '<tt>'.$outcome.'</tt>'),1);
                } else {
                    $result = &Apache::lonhtmlcommon::confirm_success(&mt('Unable to set page display, after role selection, for this '.$lctype.' to [_1] due to [_2].'
                                 ,'<b>'.$$initpage{$env{'form.courseinit_page'}}.'</b>'
                                 ,'<tt>'.$outcome.'</tt>'),1);
                }
            }
        }
    }
    return &Apache::loncommon::confirmwrapper($result);
}

sub start_box {
    my ($r,$show,$heading,$caller,$refpage) = @_;
    my %lt = &Apache::lonlocal::texthash( 
                       chth => 'Change thresholds?',
                       chin => 'Change interval?',
                       chop => 'Change options?',
    );
    my $showhide;
    if ($$show{$caller}) {
        $showhide = '<b><a href="javascript:change_display(document.visible.'.
                               'display_'.$caller.",'hide'".');">'.
                               &mt('Hide').'</a></b>';
   
    } else {
        $showhide = '<b><a href="javascript:change_display(document.visible.'.
                               'display_'.$caller.",'show'".');">'.
                               &mt('Show').'</a></b>';
    }
    
    $r->print('
         <table class="LC_nested_outer">
          <tr>
           <th class="LC_left_item">'.$$heading{$caller}.'</th>
           <th class="LC_right_item">'.$showhide.'</th>
          </tr>');
     if (($caller eq 'abovethreshold') && ($$show{$caller})) {
         if ($$show{$caller}) {
             $r->print('
              <tr>
                <td class="LC_subheader" colspan="2"><a href="/adm/whatsnew?command=chgthreshold&amp;refpage='.$refpage.'">'.$lt{'chth'}.'</a></td>
              </tr>');
         }
     } elsif (($caller eq 'versionchanges') && ($$show{$caller})) {
         if ($$show{$caller}) {
             $r->print('
              <tr>
                <td class="LC_subheader" colspan="2"><a href="/adm/whatsnew?command=chginterval&amp;refpage='.$refpage.'">'.$lt{'chin'}.'</a></td>
              </tr>');
         }
     } elsif ($caller eq 'coursediscussion') {
         if ($$show{$caller}) {
             $r->print('
              <tr>
                <td class="LC_subheader" colspan="2"><a href="/adm/whatsnew?command=chgdisc&amp;refpage='.$refpage.'">'.$lt{'chop'}.'</a></td>
              </tr>');
         }
     } elsif (($caller eq 'newroles') && ($$show{$caller})) {
         if ($$show{$caller}) {
             $r->print('
              <tr>
                <td class="LC_subheader" colspan="2"><a href="/adm/whatsnew?command=chgnewroleinterval&amp;refpage='.$refpage.'">'.$lt{'chin'}.'</a></td>
              </tr>');
         }
     } elsif (($caller eq 'oldroles') && ($$show{$caller})) {
         if ($$show{$caller}) {
             $r->print('
              <tr>
                <td class="LC_subheader" colspan="2"><a href="/adm/whatsnew?command=chgoldroleinterval&amp;refpage='.$refpage.'">'.$lt{'chin'}.'</a></td>
              </tr>');
         }
    } elsif (($caller eq 'crslogin') && ($$show{$caller})) {
        if ($$show{$caller}) {
            $r->print('
              <tr>
                <td class="LC_subheader" align="left"><span class="LC_nobreak"><label><input type="radio" name="logindetails" value="0" checked="checked" onclick="javascript:togglelogins();" />&nbsp;'.&mt('Summary').'</label><label><input type="radio" name="logindetails" value="1" onclick="javascript:togglelogins();" />&nbsp;'.&mt('Details').'</label></span></td><td class="LC_subheader"><a href="/adm/whatsnew?command=chgcrslogininterval&amp;refpage='.$refpage.'">'.$lt{'chin'}.'</a></td>
              </tr>');
        }
    }

    $r->print('
              <tr>
               <td colspan="2">
                <table class="LC_nested">
');
    return;
}

sub end_box {
    my ($r) = shift;
    $r->print('
   </table>
  </td>
 </tr>
</table><br />');
    return;
}

1;
