# The LearningOnline Network with CAPA
# TeX Conversion Module
#
# $Id: lontexconvert.pm,v 1.112.2.4 2014/06/18 06:48:47 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
# Copyright for TtHfunc and TtMfunc by Ian Hutchinson. 
# TtHfunc and TtMfunc (the "Code") may be compiled and linked into 
# binary executable programs or libraries distributed by the 
# Michigan State University (the "Licensee"), but any binaries so 
# distributed are hereby licensed only for use in the context
# of a program or computational system for which the Licensee is the 
# primary author or distributor, and which performs substantial 
# additional tasks beyond the translation of (La)TeX into HTML.
# The C source of the Code may not be distributed by the Licensee
# to any other parties under any circumstances.
#

package Apache::lontexconvert;

use strict;
use tth();
use vars qw($errorstring);
#use Apache::lonxml();
use Apache::lonlocal;
use Apache::lonnet;
use lib '/home/httpd/lib/perl/';
use LONCAPA;
use URI::Escape;
use IO::Socket::INET;


#
# Table of substitutions to unicode characters.
#

my %unicode_harpoons = (
                        '\rightleftharpoons'  => 0x21cc,
                      );

my %unicode_translations = (

# Brackets - unicode for browsers/OS which support it.

    '&#63726;'             => 0x23a1,
    '&#63727;'             => 0x23a2,
    '&#63728;'             => 0x23a3,
    '&#63737;'             => 0x23a4,
    '&#63738;'             => 0x23a5,
    '&#63739;'             => 0x23a6,

#  Parens - unicode for browsers/OS which support it

    '&#63723;'              => 0x239b,
    '&#63724;'              => 0x239c,
    '&#63725;'              => 0x239d,
    '&#63734;'              => 0x239e,
    '&#63735;'              => 0x239f,
    '&#63736;'              => 0x23a0,

);

my %ascii_8bit_translations = (

# Brackets - pure 8-bit ascii ugliness for browsers/OS which can't handle unicode

    '&#63726;'              => 0x5b,
    '&#63727;'              => 0x5b,    # '['
    '&#63728;'              => 0x5b,
    '&#63737;'              => 0x5d,    # ']'
    '&#63738;'              => 0x5d,
    '&#63739;'              => 0x5d,

# Parens - pure 8-bit ascii ugliness for browsers/OS which can't handle unicode

    '&#63723;'              => 0x28,
    '&#63724;'              => 0x28,      # '('
    '&#63725;'              => 0x28,
    '&#63734;'              => 0x29,
    '&#63735;'              => 0x29,      # '('
    '&#63736;'              => 0x29,

);

##
# Utility to convert elements of a string to unicode:
#
# @param input - Input string
# @param pattern - Pattern to convert
# @param unicode - Unicode to substitute for pattern.
#
# @return string - resulting string.
# 
sub unicode_subst {
    my ($input, $pattern, $unicode) = @_;
    
    my $char = pack('U', $unicode);

    $input =~ s/$pattern/$char/g;

    return $input;
}

# ====================================================================== Header

sub init_tth {
    my $options=$env{'course.'.$env{'request.course.id'}.'.tthoptions'};
    if ($options =~ /\S/) {
	$options = ' '.$options;
    } else {
	undef($options);
    }
    if ($env{'browser.mathml'}) {
	&tth::ttminit();
	if ($env{'browser.unicode'}) {
	    &tth::ttmoptions('-L -u1'.$options);
	} else {
	    &tth::ttmoptions('-L -u0'.$options);
	}
    } else {
	&tth::tthinit();
	if ($env{'browser.unicode'}) {
	    &tth::tthoptions('-L -u1'.$options);
	} else {
	    &tth::tthoptions('-L -u0'.$options);
	}
    }
}

# ================================================================== Conversion

$Apache::lontexconvert::messedup=0;


sub convert_real {
    my ($texstring)=@_;
    my ($xmlstring,$errorstring);
    local $SIG{SEGV}=sub { $Apache::lontexconvert::messedup=1; die; };
    local $SIG{ALRM}=sub { 
	&Apache::lonnet::logthis("ALRM");
	$xmlstring='['.&mt("TeX unconverted due to errors").']';
	$Apache::lontexconvert::messedup=1;
	die &mt("TeX unconverted due to errors");
    };
    &Apache::lonxml::start_alarm();
    if ($env{'browser.mathml'}) {
	$xmlstring=&tth::ttm($$texstring);
	$xmlstring=~s/\<math\>/\<math xmlns=\"\&mathns\;\"\>/g;
	$xmlstring=~s/\<br\>/\<br\/\>/g;
	$xmlstring=~s/\<p\>/\<p\>\<\/p\>/g;
	$errorstring.=&tth::ttmerror();
    } else {
	$xmlstring=&tth::tth($$texstring);
	$errorstring.=&tth::ttherror();
	$xmlstring=~s-</font(\s*)>-</font>-g;
    }
    $xmlstring=~s/^\s*\<br clear\=\"all\"/\<br/s;
    $xmlstring=~s/^\s*//;
    $xmlstring=~s/\s*$//;
    &Apache::lonxml::end_alarm();

    #
    # Several strings produced by tth require
    # transliteration -> unicode equivalents to render reliably
    # in browsers. %unicode_translations and %unicode_harpoons are tables of
    # string->substitution which we now apply. (%ascii_8bit_translations used
    # instead for Windows XP and mobile devices.

    my $use_ascii;
    if ($env{'browser.os'} eq 'win') {
        if (($env{'browser.osversion'}) && ($env{'browser.osversion'} < 6.0)) {
            $use_ascii = 1;
        }
    }
    if ($env{'browser.mobile'}) {
        $use_ascii = 1;
    }

    foreach my $pattern (keys(%unicode_translations)) {
	my $unicode = $unicode_translations{$pattern};
	if ($use_ascii) {
	    $unicode = $ascii_8bit_translations{$pattern};
	}
	$xmlstring = &unicode_subst($xmlstring, $pattern, $unicode);
    }

    foreach my $pattern (keys(%unicode_harpoons)) {
        $xmlstring = &unicode_subst($xmlstring, $pattern, $unicode_harpoons{$pattern});
    }

    return ($xmlstring,$errorstring);
}

sub tth_converted {
    my $texstring=shift;
    my $xmlstring='['.&mt('UNDISPLAYABLE').']';
    if ($Apache::lontexconvert::messedup) {
	return '['.&mt('TeX unconverted due to previous errors').']';
    }
    $$texstring ='\\documentstyle{article}'.$$texstring;

    eval(<<'ENDCONV');
    ($xmlstring,$errorstring)=&convert_real($texstring)
ENDCONV
    if ($@) {
	$errorstring.=&mt("Evaluation Error: ").$@;
	$Apache::lontexconvert::messedup=1;
    }
    if ($Apache::lontexconvert::messedup || &tth::tthmessedup() || 
	$errorstring) {
	&Apache::lonnet::logthis("Trying to kill myself");
	$Apache::lontexconvert::messedup=1;
	if (ref($Apache::lonxml::request)) {
	    $Apache::lonxml::request->child_terminate();
	} else {
	    my $request;
	    eval { $request=Apache->request; };
	    if (!$request) {
		eval { $request=Apache2::RequestUtil->request; };
	    }
	    if ($request) {
		$request->child_terminate();
	    } else {
		&Apache::lonnet::logthis("Unable to find a request to do child_terminate on");
	    }
	}
    }
    return $xmlstring;
}

sub clean_out_math_mode {
    my ($texstring)=@_;
    $$texstring=~s/(?<!\\)\$//g;
    $$texstring=~s/\\[\)\(\]\[]//g;
    $$texstring=~s/\\ensuremath//g;
    return '';
}

sub displaystyle {
    my ($texstring)=@_;
    #has a $$ or \[ or \displaystyle or eqnarray in it, guessinng it's display mode
    if ($$texstring=~/[^\\]\$\$/ ||
        $$texstring=~/\\\[/ ||
        $$texstring=~/\\displaystyle/ ||
        $$texstring=~/eqnarray/
       ) { return 1; }
    return 0;
}

sub MathJax_converted {
    my $texstring=shift;
    my $tag='math/tex;';
    if (&displaystyle($texstring)) { $tag='math/tex; mode=display'; }
    &clean_out_math_mode($texstring);
    return &MathJax_header().
      '<script type="'.$tag.'">'.$$texstring.'</script>';
}

{
    #Relies heavily on the previous jsMath installation
    my @MathJax_sent_header;
    sub MathJax_reset {
        undef(@MathJax_sent_header);
    }
    sub MathJax_push {
        push(@MathJax_sent_header,0);
    }
    sub MathJax_header {
        if (!@MathJax_sent_header) {
            &Apache::lonnet::logthis("mismatched calls of MathJax_header and MathJax_process");
            return '';
        }
        return '' if $MathJax_sent_header[-1];
        $MathJax_sent_header[-1]=1;
        return
          '<script type="text/javascript" src="/adm/MathJax/MathJax.js?config=TeX-AMS-MML_HTMLorMML"></script>'."\n";
    }
}


sub jsMath_converted {
    my $texstring=shift;
    my $tag='span';
    if (&displaystyle($texstring)) { $tag='div'; }
    &clean_out_math_mode($texstring);
    return &jsMath_header().
	'<'.$tag.' class="math">'.$$texstring.'</'.$tag.'>';
}

{
    my @jsMath_sent_header;
    sub jsMath_reset {
	undef(@jsMath_sent_header);
    }
    sub jsMath_push {
	push(@jsMath_sent_header,0);
    }
    sub jsMath_header {
	if (!@jsMath_sent_header) {
	    &Apache::lonnet::logthis("mismatched calls of jsMath_header and jsMath_process");
	    return '';
	}
	return '' if $jsMath_sent_header[-1];
	$jsMath_sent_header[-1]=1;
	return
            '<script type="text/javascript">
                     function NoFontMessage () {}
                     jsMath = {Parser: {prototype: {environments: {\'eqnarray*\' :[\'Array\',null,null,\'rcl\',[5/18,5/18],3,\'D\']}}}};
                   </script>'."\n".
	    '<script type="text/javascript" src="/adm/jsMath/jsMath.js"></script>'."\n";
    }
    sub jsMath_process {
	my $state = pop(@jsMath_sent_header);
	return '' if !$state;
	return "\n".
	    '<script type="text/javascript">jsMath.Process()</script>'."\n";
    }
    sub jsMath_state {
	my ($level) = @_;
	return $jsMath_sent_header[$level];
    }
}

sub tex_engine {
    if (exists($env{'form.texengine'})) {
	if ($env{'form.texengine'} ne '') {
            return $env{'form.texengine'};
        }
    }    
    if ($env{'request.course.id'}
	&& exists($env{'course.'.$env{'request.course.id'}.'.texengine'})) {
	return $env{'course.'.$env{'request.course.id'}.'.texengine'};
    }
    if (exists($env{'environment.texengine'})) {
	return $env{'environment.texengine'};
    }
    return 'tth';
}

sub init_math_support {
    my ($inherit_jsmath) = @_;
    &init_tth();
    &Apache::lontexconvert::jsMath_push();
    if (lc(&tex_engine()) eq 'jsmath' ||
	($inherit_jsmath && &jsMath_state(-2))) {
	return &Apache::lontexconvert::jsMath_header();
    }
    &Apache::lontexconvert::MathJax_push();
    if (lc(&tex_engine()) eq 'mathjax') { # ||
        #($inherit_jsmath && &jsMath_state(-2))) {
        return &Apache::lontexconvert::MathJax_header();
    }
    return;
}

sub mimetex_valign {
    my ($esc_texstring)=@_;
    my $valign = 0;
    my $path = '/cgi-bin/mimetex.cgi?'.$esc_texstring;
    my $socket;
    &Apache::lonxml::start_alarm();
    $socket = IO::Socket::INET->new(PeerAddr => 'localhost',
                                    PeerPort => 'http(80)',
                                    Proto    => 'tcp');
    if ($socket) {
        my $headreq = "HEAD $path HTTP/1.0\r\n\r\n";
        print $socket $headreq;
        while (<$socket>) {
            if (/Vertical\-Align\:\s*?([\-\d]+)/) {
                $valign = $1;
            }
        }
        $socket->close();
    }
    &Apache::lonxml::end_alarm();
    return $valign;
}

sub mimetex_converted {
    my $texstring=shift;

# Alt-Argument for screen readers
    my $alt_string=$$texstring;
    $alt_string=~s/\"/\'\'/g;

# Is this displaystyle?

    my $displaystyle=&displaystyle($texstring);

# Remove math environment delimiters

    &clean_out_math_mode($texstring);

    if ($displaystyle) {
	$$texstring='\\displaystyle \\Large '.$$texstring;
    }
    my $esc_texstring = &uri_escape($$texstring);
    my $valign = &mimetex_valign($esc_texstring);
    my $result='<img src="/cgi-bin/mimetex.cgi?'.$esc_texstring.'" style="vertical-align:'.$valign.'px" alt="'.$alt_string.'" />';
    if ($displaystyle) {
	$result='<div style="text-align:center">'.$result.'</div>';
    }
    return $result;
}

sub converted {
    my ($string,$mode)=@_;
    if ($mode eq '') { $mode = &tex_engine(); }
    if ($mode =~ /tth/i) {
	return &tth_converted($string);
    } elsif ($mode =~ /jsmath/i) {
	return &jsMath_converted($string);
    } elsif ($mode =~ /mathjax/i) {
        return &MathJax_converted($string);
    } elsif ($mode =~ /mimetex/i) {
	return &mimetex_converted($string);
    } elsif ($mode =~ /raw/i) {
        return $$string;
    }
    return &tth_converted($string);
}

# ------------------------------------------------------------ Message display

sub to_convert {
    my ($string) = @_;
    &init_tth();
    $string=~s/\<br\s*\/?\>/ /gs;
#    $string=~s/\s/ /gs;
    $string=&HTML::Entities::decode($string);
    return &converted(\$string);
}

sub smiley {
    my $expression=shift;
    my %smileys=(
    	 '\:\-*\)' => 'face-smile.png',
		 '8\-\)'  => 'face-cool.png',
		 '8\-(I|\|)'   => 'face-glasses.png',
		 '\:\-(I|\|)'   => 'face-plain.png',
		 '\:\-(o|O|\(\))' => 'face-surprise.png',
		 ':\-\('  => 'face-sad.png',
		 '\;\-\)' => 'face-wink.png',
		 '\:\-(P|p)'  => 'face-raspberry.png',
		 '\:\-(\\\|\\/)' => 'face-uncertain.png',
		 '\:\-D'  => 'face-smile-big.png',
		 '\:\-(C|\@)'  => 'face-angry.png',
		 '\:(\'|\`)\-*\(' => 'face-crying.png',
		 '\:\-(X|x|\#)' => 'face-quiet.png',
		 '\:\-(s|S)' => 'face-uncertain.png',
		 '\:\-\$' => 'face-embarrassed.png',
		 '\:\-\*' => 'face-kiss.png',
		 '\+O\(' => 'face-sick.png',
		 '(\&lt\;3|\(heart\))' => 'heart.png',
		 '\(rose\)' => 'rose.png',
		 '\(pizza\)' => 'food-pizza.png',
		 '\(cake\)' => 'food-cake.png',
		 '\(ninja\)' => 'face-ninja.png',
		 '\(pirate\)' => 'face-pirate.png',
		 '\((agree|yes)\)' => 'opinion-agree.png',
		 '\((disagree|nay)\)' => 'opinion-disagree.png',
		 '(o|O)\-\)' => 'face-angel.png',
		 );
    my $iconpath=$Apache::lonnet::perlvar{'lonIconsURL'};
    foreach my $smiley (keys(%smileys)) {
	$expression=~s/$smiley/\<img src="$iconpath\/$smileys{$smiley}" \/\>/gs; 
    }
    return $expression;
}

sub msgtexconverted {
    my ($message,$email) = @_;
    $errorstring='';
    my $outmessage='';
    my $tex=0;
    foreach my $fragment (split(/(?:\&lt\;|\<)\/*m\s*(?:\&gt\;|\>)/i,$message)) {
	if ($tex) {
	    if ($email) {
		$outmessage.='</pre><tt>'.&to_convert($fragment).'</tt><pre>';
		$tex=0;
	    } else {
		$outmessage.=&to_convert($fragment);
		$tex=0;
	    }
	} else {
            $outmessage.=&smiley($fragment);
	    $tex=1;
	}
    }
    $message=$outmessage; $outmessage=''; $tex=0;
    foreach my $fragment (split(/(?:\&lt\;|\<)\/*algebra\s*(?:\&gt\;|\>)/i,
				$message)) {
	if ($tex) {
        my $algebra = &algebra($fragment, 'web', undef, undef, undef, 'tth');
	    if ($email) {
		$outmessage.='</pre><tt>'.$algebra.'</tt><pre>';
		$tex=0;
	    } else {
		$outmessage.=$algebra;
		$tex=0;
	    }
	} else {
        $outmessage.=$fragment;
	    $tex=1;
	}
    }
    if (wantarray) {
	return ($outmessage,$errorstring);
    } else {
	return $outmessage.$errorstring;
    }
}

sub algebra {
    use AlgParser;
    my ($string,$target,$style,$parstack,$safeeval,$tth)=@_;
    my $parser = new AlgParserWithImplicitExpand;
    if ($tth eq 'tth') {&init_tth();}
    $string=&prepare_algebra($string);
    my $ret = $parser->parse($string);
    my $result='['.&mt('Algebra unconverted due to previous errors').']';
    if ( ref($ret) ) {
	#$parser->tostring();
	$parser->normalize();
	my $latex=$parser->tolatex();
	$latex=&postprocess_algebra($latex);
	if ($style eq 'display') {
	    $latex='$$'.$latex.'$$x';
	} else {
	    # style is 'inline'
	    $latex='\\ensuremath{'.$latex.'}';
	}
	if ($target eq 'web' || $target eq 'analyze') {
            my $display=&Apache::lonxml::get_param('display',$parstack,$safeeval);
            $result = &converted(\$latex,$display);
#	    $result = &converted(\$latex);
	} else {
	    $result = $latex;
	}
    } else {
	&Apache::lonxml::error($parser->{'htmlerror'});
    }
}

sub prepare_algebra {
    my ($string)=@_;

    # makes the decision about what is a minus sign easier supposedly
    $string =~ s/(\<\>|\<\=|\>\=[\=\>\<] *)-/$1 zeroplace -/g;

    return $string;
}

sub postprocess_algebra {
    my ($string)=@_;
    
    # moodle had these and I don't know why, ignoring them for now
    # $string =~s/\\fun/ /g;

    # sqrt(3,4) means the 4 root of 3
    $string =~s/\\sqrt{([^,]+),([^\}]+)}/\\sqrt[$2]{$1}/gs;

    # log(3,4) means the log base 4 of 3
    $string =~s/\\log\\left\((.+?),(.+?)\\right\)/\\log_{$2}\\left($1\\right)/gs;

    # log(3,4) means the log base 4 of 3
    $string =~s/\\((?:sin|cos|tan|sec|csc|cot)(?:h)?)\\left\((.+?),(.+?)\\right\)/\\$1^{$3}\\left($2\\right)/gs;

    # int(3,a,b) integral from a to b of 3
    $string =~s/\\int\\left\((.+?),(.+?),(.+?)\\right\)/\\int_{$2}^{$3}\\left($1\\right)/gs;

    # int( ... dx) -> ...
    $string =~s/\\int\\left\((.+?)d[a-z]\\right\)/$1/gs;

    # 
    $string =~s/\\lim\\left\((.+?),(.+?),(.+?)\\right\)/\\lim_{$2\\to $3}$1/gs;
    return $string;
}


1;
__END__


=pod

=head1 NAME

Apache::lontexconvert;

=head1 SYNOPSIS

Access to tth/ttm

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.


=head1 SUBROUTINES

=over

=item init_tth()

Header

=item convert_real()

 we need this routine because &converted can get called from inside
 of the safespace (through &xmlparse('<m>stuff</m>') which doesn't
 allow the opcode for alarm, so we need to compile this before we get
 into the safe space since opcode checks only occur at compile time

=item tth_converted()


=item clean_out_math_mode()


=item displaystyle()


=item jsMath_converted()

=item MathJax_converted()
        - Mimics the jsMath functionality

=item tex_engine()


=item init_math_support()

=item mimetex_valign()

 Makes a HEAD call to /cgi-bin/mimetex.cgi via IO:: to retrieve the 
 vertical alignment, before the subsequent call to mimetex_converted()
 which generates the <img> tag and the corresponding image.

 Input: 1.  $esc_texstring (escaped TeX to be rendered by mimetex).
 Output: 1. $valign - number of pixels: positive or negative integer 
            which will be included in <img> tag for mimetex image to
            support vertical alignment of image within a line of text.

 If a server is running SSL, and Apache rewrite rules are in place 
 to rewrite requests for http to https, modification will most likely 
 be needed for pass through for HEAD requests for /cgi-bin/mimetex.cgi. 

 Example rewrite rules which rewrite all http traffic to https, 
 except HEAD requests for /cgi-bin/mimetex.cgi are:

 <IfModule mod_rewrite.c>
     RewriteEngine On
     RewriteLogLevel 0

     RewriteCond %{HTTPS} off
     RewriteCond %{HTTP:Host} (.*)
     RewriteCond %{REQUEST_METHOD} !HEAD 
     RewriteRule ^/(.*) https://%1/$1 [R=301,L]

     RewriteCond %{HTTPS} off
     RewriteCond %{HTTP:Host} (.*)
     RewriteCond %{REQUEST_METHOD} HEAD
     RewriteCond %{REQUEST_URI} !^/cgi-bin/mimetex.cgi
     RewriteRule ^/(.*) https://%1/$1 [R=301,L]
 </IfModule>

=item mimetex_converted()


=item converted()


=item to_convert()

message display

=item smiley()

???

=item msgtexconverted()

=item algebra()

=item prepare_algebra()

=item postprocess_algebra()

=back

=cut



