#!/usr/bin/perl
#
# The LearningOnline Network
#
# Move Construction Spaces from /home/$user/public_html
# to /home/httpd/html/priv/$domain/$user and vice versa
#
# $Id: move_construction_spaces.pl,v 1.9 2012/06/08 13:08:13 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
#################################################

use strict;
use lib '/home/httpd/lib/perl/';
use LONCAPA::Configuration;
use LONCAPA qw(:DEFAULT :match);
use Apache::lonlocal;
use File::Copy;
use GDBM_File;
use DBI;

my ($lonusersdir,$londocroot,$londaemons,$lonsqlaccess);

BEGIN {
    my $perlvar=&LONCAPA::Configuration::read_conf();
    if (ref($perlvar) eq 'HASH') {
        $lonusersdir = $perlvar->{'lonUsersDir'};
        $londocroot = $perlvar->{'lonDocRoot'};
        $londaemons = $perlvar->{'lonDaemons'};
        $lonsqlaccess = $perlvar->{'lonSqlAccess'};
    }
    undef($perlvar);
}

my $lang = &Apache::lonlocal::choose_language();
&Apache::lonlocal::get_language_handle(undef,$lang);

if ($< != 0) {
    print &mt('You must be root in order to move Construction Spaces.')."\n".
          &mt('Stopping')."\n";
    exit;
}

if ($lonusersdir eq '') {
    print &mt('Could not determine location of [_1] directory.',"'lonUsersDir'")."\n".
          &mt('Stopping')."\n";
    exit;
}

if ($londocroot eq '') {
    print &mt('Could not determine location of [_1] directory.',"'lonDocRoot'")."\n".
          &mt('Stopping')."\n";
    exit;
}

my $distro;
if ($londaemons eq '') {
    print &mt('Could not determine location of [_1] directory.',"'lonDaemons'")."\n".
          &mt('Stopping')."\n";
    exit;
} else {
    if (-e "$londaemons/distprobe") {
        if (open(PIPE,"perl $londaemons/distprobe|")) {
            $distro = <PIPE>;
            close(PIPE);
        }
    }
}

if ($distro eq '') {
    print &mt('Could not determine Linux distro.')."\n".
          &mt('Stopping')."\n";
    exit;
} else {
    my $stopapachecmd = '/etc/init.d/httpd stop';
    my $apacheprocess = '/usr/sbin/httpd';
    my $stopapachecmd = '/etc/init.d/httpd stop';
    my $proc_owner = 'root';
    if ($distro =~ /^(suse|sles)/) {
        if ($distro =~ /^(suse|sles)9/) {
            $stopapachecmd = '/etc/init.d/apache stop';
        } else {
            $apacheprocess = '/usr/sbin/httpd2';
            $stopapachecmd = '/etc/init.d/apache2 stop';
        }
    } elsif ($distro =~ /^(?:debian|ubuntu)(\d+)/) {
        $apacheprocess = '/usr/sbin/apache2';
        $stopapachecmd = '/etc/init.d/apache2 stop';
    } elsif ($distro =~ /^(?:fedora)(\d+)/) {
        my $version = $1;
        if ($version >= 16) {
            $stopapachecmd = '/bin/systemctl stop httpd.service';
        }
    }
    if (open(PIPE,"ps -ef |grep '$apacheprocess' |grep -v grep 2>&1 |")) {
        my $status = <PIPE>;
        close(PIPE);
        chomp($status);
        if ($status =~ /^\Q$proc_owner\E\s+\d+\s+/) {
            print "\n".
                  &mt('You need to stop the Apache daemon before moving Construction Spaces.')."\n".
                  &mt('To do so use the following command: [_1]',"\n\n$stopapachecmd")."\n\n".
                  &mt('Now stopping the move_construction_spaces.pl script.')."\n";
            exit;
        }
    } else {
        print &mt('Could not determine if Apache daemon is running.')."\n";
    }
}

my $stoploncontrol = '/etc/init.d/loncontrol stop';
if (open(PIPE,"ps -ef |grep lond |grep -v grep 2>&1 |")) {
    my $status = <PIPE>;
    close(PIPE);
    chomp($status);
    if ($status =~ /^www\s+\d+\s+/) {
        print "\n".
              &mt('You need to stop the LON-CAPA daemons before moving Construction Spaces.')."\n".
              &mt('To do so use the following command: [_1]',"\n\n$stoploncontrol")."\n\n".
              &mt('Now stopping the move_construction_spaces.pl script.')."\n";
        exit;        
    }
}

# Abort if more than one argument.

my $parameter=$ARGV[0];
$parameter =~ s/^\s+//;
$parameter =~ s/\s+$//;

if ((@ARGV > 1) || (($parameter ne '') && ($parameter !~ /^(move|undo)$/))) {
    print &mt('usage: [_1]','move_construction_spaces.pl [move|undo]')."\n\n".
          &mt('You should enter either no arguments, or just one argument -- either move or undo.')."\n".
          &mt("move - to move authors' Construction Spaces from: [_1] to [_2].",
              "'/home'","'$londocroot/priv/'")."\n".
          &mt('undo - to reverse those changes and move Construction Spaces back from: [_1] to [_2].',
              "'$londocroot/priv/'","'/home'")."\n".
          &mt('no argument to do a dry run of the move option, without actually moving anything.')."\n";
    exit;
}

print "\n".&mt("Moving authors' Construction Spaces.")."\n".
      "-----------------------------\n\n".
      &mt('If run without an argument, the script will report what it would do when moving Construction Spaces from [_1] to [_2].',
          "'/home'","'$londocroot/priv/'")."\n\n".
      &mt('If there are ambiguities (i.e., the same username belongs to two domains), this will be flagged, and you will be able to decide how to proceed.')."\n";

my (undef,undef,$uid,$gid) = getpwnam('www');
my ($action) = ($parameter=~/^(move|undo)$/);
if ($action eq '') {
    $action = 'dryrun';
}

if ($action eq 'dryrun') {
    print "\n\n".
          &mt('Running in exploratory mode ...')."\n\n".
          &mt('Run with argument [_1] to actually move Construction Spaces to [_2], i.e., [_3]',
              "'move'","'$londocroot/priv'","\n\nperl move_construction_spaces.pl move")."\n\n\n".
          &mt('Run with argument [_1] to move Construction spaces back to [_2], i.e., [_3]',
              "'undo'","'/home'","\n\nperl move_construction_spaces.pl undo")."\n\n\n".
          &mt('Continue? ~[y/N~] ');
    if (!&get_user_selection()) {
        exit;
    } else {
        print "\n";
    }
} else {
    print "\n *** ".&mt('Running in a mode where changes will be made.')." ***\n";
    if ($action eq 'move') {
        print "\n".
              &mt('Mode is [_1] -- directories will be moved to [_2].',
                  "'$action'","'$londocroot/priv'")."\n";
    } else {
        print "\n".
              &mt('Mode is [_1] -- directories will be moved back to [_2].',
                  "'$action'","'/home'")."\n";
    }
    print &mt('Continue? ~[y/N~] ');
    if (!&get_user_selection()) {
        exit;
    } else {
        print "\n";
    }
}

my $logfh;
if ($action ne 'dryrun') {
    if (!open($logfh,">>$londaemons/logs/move_construction_spaces.log")) {
        print &mt('Could not open log file: [_1] for writing.',
                  "'$londaemons/logs/move_construction_spaces.log'")."\n".
              &mt('Stopping.')."\n";
    } else {
        &start_logging($logfh,$action);
    }
}

# Authors hosted on this server
my %allauthors;
my %pubusers;

if ($action eq 'move') {
    my $output;
    if (-d "$londocroot/priv") {
        $output = &mt('New Construction Spaces directory: [_1] already exists.',
                      "'$londocroot/priv'")."\n";
        print $output;
        print $logfh $output;
    } else {
        $output = &mt('Creating new directory: [_1] for Construction Spaces.',
                      "'$londocroot/priv'")."\n";
        if (mkdir("$londocroot/priv",0750)) {
            if (chown($uid,$gid,"$londocroot/priv")) {
                $output .= &mt('Creation Successful')."\n";
                print $output;
                print $logfh $output;
            } else {
                $output .= &mt('Failed to change ownership to [_1].',"'$uid:$gid'")."\n";
                print $output;
                &stop_logging($logfh,$output);
                print &mt('Stopping')."\n";
                exit;
            }
        } else {
            $output .=  &mt('Failed to create directory [_1].',"'$londocroot/priv'")."\n";
            print $output;
            &stop_logging($logfh,$output);
            print &mt('Stopping')."\n";
            exit;
        }
    }
}

my @machinedoms;
if ($lonusersdir) {
    my ($dir,$output);
    if (opendir($dir,$lonusersdir)) {
        my @contents = (grep(!/^\.{1,2}$/,readdir($dir)));
        closedir($dir);
        foreach my $item (@contents) {
            if (-d "$lonusersdir/$item") {
                if ($item =~ /^$match_domain$/) {
                    my $domain = $item;
                    unless (grep(/^\Q$domain\E$/,@machinedoms)) {
                        push(@machinedoms,$domain);  
                    }
                    my $dom_target="$londocroot/priv/$domain";
                    if ($action eq 'move') {
                        if (!-e $dom_target) {
                            if (mkdir($dom_target,0755)) {
                                chown($uid,$gid,$dom_target);
                                $output = &mt('Made [_1].',"'$dom_target'")."\n";
                                print $output;
                                print $logfh $output;
                            } else {
                                $output = &mt('Failed to make [_1].',"'$dom_target'")."\n";
                                print $output;
                                print $logfh $output;
                                &stop_logging($logfh,$output);
                                print &mt('Stopping')."\n";
                                exit;
                            }
                        } elsif ($action eq 'dryrun') {
                            print &mt('Would make [_1].',"'$dom_target'")."\n";
                        }
                    }
                    my %authors=();
                    my $fname = "$lonusersdir/$domain/nohist_domainroles.db";
                    my $dbref;
                    if (-e $fname) {
                        $dbref=&LONCAPA::locking_hash_tie($fname,&GDBM_READER());
                    }
                    if (!$dbref) {
                        print &mt('Unable to tie to [_1].',"'$fname'")."\n";
                    } elsif (ref($dbref) eq 'HASH') {
                        foreach my $key (keys(%{$dbref})) {
                            $key = &unescape($key);
                            if ($key =~ /^au\:($match_username)\Q:$domain\E/) {
                                push(@{$allauthors{$1}},$domain);
                            }
                        }
                        &LONCAPA::locking_hash_untie($dbref);
                    }
                }
            }
        }
    } else {
        $output = &mt('Could not open [_1].',"'$lonusersdir'")."\n";
        print $output;
        &stop_logging($logfh,$output);
        print &mt('Stopping')."\n";
        exit;
    }
}

if ($londocroot ne '') {
    if (-d "$londocroot/res") {
        my ($dir,$domdir);
        if (opendir($dir,"$londocroot/res")) {
            my @contents = (grep(!/^\.{1,2}$/,readdir($dir)));
            closedir($dir);
            foreach my $dom (@contents) {
                if ((grep(/^\Q$dom\E/,@machinedoms)) && (-d "$londocroot/res/$dom")) {
                    if (opendir($domdir,"$londocroot/res/$dom")) {
                        my @unames = (grep(!/^\.{1,2}$/,readdir($domdir)));
                        closedir($domdir);
                        foreach my $uname (@unames) {
                            if ($uname =~ /^$match_username$/) {
                                push(@{$pubusers{$uname}},$dom);
                            }
                        }
                    }
                }
            }
        }
    }
}

if ($action eq 'undo') {
    my %privspaces;
    if ($londocroot ne '') {
        if (-d "$londocroot/priv") {
            my ($dir,$domdir);
            if (opendir($dir,"$londocroot/priv")) {
                my @contents = (grep(!/^\.{1,2}/,readdir($dir)));
                closedir($dir);
                foreach my $dom (@contents) {
                    next if (!-d "$londocroot/priv/$dom");
                    if (opendir($domdir,"$londocroot/priv/$dom")) {
                        my @unames = (grep(!/^\.{1,2}$/,readdir($domdir)));
                        closedir($domdir);
                        foreach my $uname (@unames) {
                            if ($uname =~ /^$match_username$/) {
                                push(@{$privspaces{$uname}},$dom);
                            }
                        }
                    }
                }
            }
        }
    }
    foreach my $uname (keys(%privspaces)) {
        if (ref($privspaces{$uname}) eq 'ARRAY') {
            my $output;
            if (@{$privspaces{$uname}} > 1) {
                my $displaydoms = join(', ',@{$privspaces{$uname}});
                print &mt('Same username used for authors in multiple domains.')."\n".
                      &mt('This configuration is not supported where Construction Spaces are located in [_1].','/home').".\n".
                      &mt('You will be able to move files for just one of the domains, choose which one.')."\n".
                      &mt('The domains to choose from are: [_1].',"'$displaydoms'")."\n".
                      &mt('Enter choice: ');
                my $choice=<STDIN>;
                chomp($choice);
                if (grep(/^\Q$choice\E$/,@{$privspaces{$uname}})) {
                    $output = &move_priv_to_home($londocroot,$uid,$gid,$uname,$choice);
                } else {
                    print &mt('Invalid choice of domain:')." $choice\n";
                    $output = &mt('Skipping this user: [_1].',"'$uname'")."\n";
                    print $output;
                    print $logfh $output;
                    next;
                }
            } elsif (@{$privspaces{$uname}} == 1) {
                $output = &move_priv_to_home($londocroot,$uid,$gid,$uname,$privspaces{$uname}[0]);
            } else {
                print &mt('Username [_1] found in [_2] was not within a domain',
                          "'$uname'","'$londocroot/priv'")."\n";
                $output = &mt('Skipping this user: [_1].',"'$uname'")."\n";
            }
            print $output;
            print $logfh $output;
        }
    }
    if (-d "$londocroot/priv") {
        my $output;
        if (opendir(my $dir,"$londocroot/priv")) {
            my @doms = grep(!/^\.{1,2}/,readdir($dir));
            closedir($dir);
            foreach my $dom (@doms) {
                if (opendir(my $domdir,"$londocroot/priv/$dom")) {
                    my @contents =  grep(!/^\.{1,2}/,readdir($domdir));
                    closedir($domdir);
                    if (@contents == 0) {
                        if (rmdir("$londocroot/priv/$dom")) {
                            $output = &mt('Removed empty directory: [_1]',
                                          "'$londocroot/priv/$dom'")."\n";
                        } else {
                            $output = &mt('Failed to remove directory: [_1]',
                                          "'$londocroot/priv/$dom'")."\n";
                        }
                    }
                }
            }
        }
        my $warning = &mt('WARNING: Access to Construction Spaces in their old locations (i.e., in [_1]) via LON-CAPA with URLs of the form [_2] will not work until the directory at [_3] is moved or deleted.',"'/home/<user>/'","'/priv/<user>/'","'$londocroot/priv/'")."\n";
        if (opendir(my $dir,"$londocroot/priv")) {
            my @contents = (grep(!/^\.{1,2}/,readdir($dir)));
            closedir($dir);
            if (@contents == 0) {
                if (rmdir("$londocroot/priv")) {
                    $output .= &mt('Removed empty directory: [_1]',
                                   "'$londocroot/priv'")."\n";
                } else {
                    $output .= &mt('Failed to remove directory: [_1]',
                                   "'$londocroot/priv'")."\n".
                               $warning."\n";
                }
            } else {
                $output .= $warning."\n".
                           &mt('The attempt to remove the directory failed, because it is not empty.')."\n";
            }
        } else {
            $output .= $warning."\n".
                       &mt('The attempt to open the directory to see its contents failed, hence no attempt was made to remove it.')."\n";
        }
        print $output;
        print $logfh $output;
    }
    &stop_logging($logfh);
    print "\n".&mt('Done')."\n";
    exit;
}

my @allskipped;
my %allmoved;
my ($dbh,$dbflag);

# Iterate over directories in /home
if (opendir(my $dir,"/home")) {
    my @possibles = grep(!/^\.{1,2}$/,readdir($dir));
    closedir($dir);
    foreach my $item (sort(@possibles)) {
        next if ($item eq 'www');
        if ((-d "/home/$item") && ($item ne '')) {
# Is there a public_html-directory?
            if (-d "/home/$item/public_html") {
                my $author = $item;
                my ($domain,$skipped,$output,$stopnow);
                if (ref($allauthors{$author}) eq 'ARRAY') {
                    ($domain,$skipped,$stopnow) = 
                        &choose_domain($action,$author,$allauthors{$author});
                    if ($stopnow) {
                        if ($action ne 'dryrun') {
                            my $output = &mt('Stopped by user at author: [_1].',
                                             "'$author'")/"\n";
                            &stop_logging($logfh,$output);
                        }
                        if ($dbflag == 1) {
                            &disconnect_mysql($dbh);
                        }
                        print &mt('Stopped.')."\n";
                        exit;
                    }
                }
                if (($domain eq '') && (!$skipped)) {
                    if (ref($pubusers{$author}) eq 'ARRAY') {
                        ($domain,$skipped,$stopnow) = 
                            &choose_domain($action,$author,$pubusers{$author});
                    }
                    if ($stopnow) {
                        if ($action ne 'dryrun') {
                            my $output = &mt('Stopped by user at author: [_1].',
                                             "'$author'")/"\n";
                            &stop_logging($logfh,$output);
                        }
                        if ($dbflag == 1) {
                            &disconnect_mysql($dbh);
                        }
                        print &mt('Stopped.')."\n";
                        exit;
                    }
                }
                if (($domain eq '') && (!$skipped)) {
                    my @foundauthor = ();
                    foreach my $dom (@machinedoms) {
                        my $posspath = &LONCAPA::propath($dom,$author);
                        if (-e $posspath) {
                            my $rolesdbref;
                            my $fname = "$posspath/roles.db";
                            if (-e "$fname") {
                                $rolesdbref=&LONCAPA::locking_hash_tie($fname,&GDBM_READER());
                                if (!$rolesdbref) {
                                    print &mt('Unable to tie to [_1].',"'$fname'")."\n";
                                } elsif (ref($rolesdbref) eq 'HASH') {
                                    foreach my $key (keys(%{$rolesdbref})) {
                                        if ($key eq "/$dom/_au") {
                                            unless(grep(/^\Q$dom\E$/,@foundauthor)) { 
                                                push(@foundauthor,$dom);
                                            }
                                        }
                                    }
                                    &LONCAPA::locking_hash_untie($rolesdbref);
                                }
                            }
                        }
                    }
                    if (@foundauthor > 0) {
                        ($domain,$skipped,$stopnow) = 
                             &choose_domain($action,$author,\@foundauthor);
                        if ($stopnow) {
                            if ($action ne 'dryrun') {
                                my $output = &mt('Stopped by user at author: [_1].',
                                                 "'$author'")/"\n";
                                &stop_logging($logfh,$output);
                            }
                            if ($dbflag == 1) {
                                &disconnect_mysql($dbh);
                            }
                            print &mt('Stopped.')."\n";
                            exit;
                        }
                    }
                }
                if (($domain eq '') && (!$skipped)) {
                    if (!$dbflag) {
                        ($dbh,$dbflag) = &connect_mysql($lonsqlaccess);
                    }
                    if (defined($dbh)) {
                        my $foundusers = &search_allusers($dbh,$author);
                        if (ref($foundusers) eq 'HASH') {
                            ($domain,$skipped,$stopnow) = 
                                &choose_domain($action,$author,$foundusers);
                        }
                        if ($stopnow) {
                            if ($action ne 'dryrun') {
                                my $output = &mt('Stopped by user at author: [_1].',
                                                 "'$author'")/"\n";
                                &stop_logging($logfh,$output);
                            }
                            if ($dbflag == 1) {
                                &disconnect_mysql($dbh);
                            }
                            print &mt('Stopped.')."\n";
                            exit;
                        }
                    }
                }
                my $source_path="/home/$author/public_html";
                if ($domain) {
                    my $target_path="$londocroot/priv/$domain/$author";
                    if ($action eq 'move') {
                        if (move($source_path,$target_path)) {
                            my (undef,undef,$userid,$groupid) = getpwnam($author);
                            if ($userid eq '' && $groupid eq '' && $author ne '') {
                                chown($uid,$gid,$target_path);
                            }
                            $output = &mt('Moved [_1] to [_2].',
                                          "'$source_path'","'$target_path'")."\n";
                            push(@{$allmoved{$domain}},$author); 
                            my (undef,undef,$userid,$groupid) = getpwnam($author);
                            if ($userid eq '' && $groupid eq '' && $author ne '') {
                                &check_for_restore_files($londaemons,$author,$domain);
                                if (opendir(my $homedir,"/home/$author")) {
                                    my @contents = 
                                        grep(!/^\.{1,2}$/,readdir($homedir));
                                    closedir($homedir);
                                    if (@contents == 0) {
                                        if (rmdir("/home/$author/")) {
                                            $output .= &mt('Removed empty directory: [_1]',
                                                           "'/home/$author/'")."\n";
                                        } else {
                                            $output .= &mt('Failed to remove directory: [_1]',
                                                           "'/home/$author/'")."\n";
                                        }
                                    } else {
                                        $output .= &mt('Not removing directory [_1] as it still contains: [_2]',
                                                   "'/home/$author/'",
                                                   "\n".join("\n",@contents)."\n");
                                    }
                                }
                            } else {
                                $output .= &mt('Not removing directory [_1] for UNIX user account',
                                               "'/home/$author/'")."\n";  
                            }
                        } else {
                            $output = &mt('Failed to move [_1] to [_2].',
                                          "'$source_path'","'$target_path'")."\n";
                        }
                        print $output;
                        print $logfh $output;
                    } elsif ($action eq 'dryrun') {
                        push(@{$allmoved{$domain}},$author);
                        print &mt('Would move [_1] to [_2].',"'$source_path'","'$target_path'")."\n";
                    }
                } elsif ($skipped) {
                    push(@allskipped,$author); 
                    if ($action ne 'dryrun') {
                        my $output = &mt('Skipping this user: [_1].',"'$author'")."\n";
                        print $logfh $output;
                    }
                } else {
                    print '*** '.&mt('WARNING: [_1] has no domain.',"'$author'")."\n".
                          &mt('Enter [_1]: skip this user.','1')."\n".
                          &mt('Enter [_1]: stop.','2')."\n".
                          &mt('or enter domain for user to be placed into')."\n".
                          &mt('Your input: ');
                    my $choice=<STDIN>;
                    chomp($choice);
                    $choice =~ s/^\s+//;
                    $choice =~ s/\s+$//;
                    if ($choice == 1) {
                        my $output = &mt('Skipping -- no domain for user: [_1].',"'$author'")."\n";
                        print $output;
                        if ($action ne 'dryrun') {
                            print $logfh $output;
                        }
                        push(@allskipped,$author);
                        next;
                    }
                    if ($choice == 2) {
                        print &mt('Stopped.')."\n";
                        if ($action ne 'dryrun') {
                            my $output = &mt('Stopped by user because of author without domain: [_1].',
                                             "'$author'")/"\n";
                            &stop_logging($logfh,$output); 
                        }
                        if ($dbflag == 1) {
                            &disconnect_mysql($dbh);
                        } 
                        exit;
                    } elsif ($choice =~ /^$match_domain$/) {
                        print &mt('You entered:')." $choice\n".
                              &mt('Is this ok? ~[Y/n~] ');
                        if (!&get_user_selection(1)) {
                            print &mt('Try again ...')."\n".
                                  &mt('Enter [_1]: skip this user.','1')."\n".
                                  &mt('Enter [_1]: stop.','2')."\n".
                                  &mt('or enter domain for user to be placed into')."\n".
                                  &mt('Your input: ');
                            $choice=<STDIN>;
                            chomp($choice);
                            $choice =~ s/^\s+//;
                            $choice =~ s/\s+$//;
                            if ($choice == 1) {
                                my $output = &mt('Skipping -- no domain for user: [_1].',"'$author'")."\n";
                                print $output;
                                if ($action ne 'dryrun') {
                                    print $logfh $output;
                                }
                                push(@allskipped,$author);
                                next;
                            }
                            if ($choice == 2) {
                                print &mt('Stopped.')."\n";
                                if ($action ne 'dryrun') {
                                    my $output = &mt('Stopped by user because of author without domain: [_1].',
                                                     "'$author'")/"\n";
                                    &stop_logging($logfh,$output);
                                }
                                if ($dbflag == 1) {
                                    &disconnect_mysql($dbh);
                                }
                                exit;
                            } elsif ($choice !~ /^$match_domain$/) {
                                print &mt('Invalid domain entered:')." $choice\n";
                                my $output = &mt('Skipping -- no domain for user: [_1].',"'$author'")."\n";
                                print $output;
                                if ($action ne 'dryrun') {
                                    print $logfh $output;
                                }
                                push(@allskipped,$author);
                                next;
                            }
                        }
                        my $dompath="$londocroot/priv/$choice";
                        my $newpath="$londocroot/priv/$choice/$author";
                        unless (-e $dompath) {
                            if ($action eq 'move') {
                                print '*** '.&mt('WARNING: [_1] does not yet exist.',"'$dompath'")."\n";
                            }
                        }
                        if ($action eq 'move') {
                            unless (-e $dompath) {
                                $output .= &mt('Making [_1].',"'$dompath'")."\n";
                                if (mkdir($dompath,0755)) {
                                    chown($uid,$gid,$dompath);
                                }
                            }
                            if (-e $dompath) {
                                if (move($source_path,$newpath)) {
                                    chown($uid,$gid,$newpath);
                                    chmod(0750,$newpath);
                                    $output = &mt('Moved [_1] to [_2].',
                                                  "'$source_path'","'$newpath'")."\n";
                                } else {
                                    $output = &mt('Failed to move [_1] to [_2].',
                                                  "'$source_path'","'$newpath'")."\n"; 
                                }
                                print $output;
                                print $logfh $output;
                            } else {
                                $output = &mt('Failed to move [_1] to [_2] -- missing [_3].',
                                              "'$source_path'","'$newpath'","'$dompath'")."\n";
                            }
                        } elsif ($action eq 'dryrun') {
                           print &mt('Would make author [_1] in domain [_2].',"'$author'","'$choice'")."\n";
                           unless (-e $dompath) {
                               print &mt('Would make [_1].',"'$dompath'")."\n";
                           }
                           print &mt('Would make [_1].',"'$newpath'")."\n";
                        }
                    } else {
                        print &mt('Invalid domain:')." $choice\n";
                        if ($action eq 'move') {
                            print $logfh &mt('Skipping -- no domain for user: [_1].',"'$author'")."\n";
                        }
                        push(@allskipped,$author);
                        next;
                    }
                }
            }
        }
    }
}

my ($moveinfo,$skipcount);
if (keys(%allmoved) == 0) {
    $moveinfo = &mt('None')."\n";
} else {
    foreach my $dom (sort(keys(%allmoved))) {
        if (ref($allmoved{$dom}) eq 'ARRAY') {
            $moveinfo .= "\n      ".&mt('Domain: [_1], number of authors: [_2]',
                                        "'$dom'",scalar(@{$allmoved{$dom}}));
        }
    }
}

$skipcount = scalar(@allskipped);

print "\n";
if ($action ne 'dryrun') {
    my $output = &mt('You skipped: [_1].',$skipcount)."\n".
                 join("\n",sort(@allskipped))."\n\n".
                 &mt('Moved ... [_1]',$moveinfo);
    print $output;
    print $logfh $output;
    &stop_logging($logfh);
} else {
    print &mt('You would have skipped: [_1].',$skipcount)."\n".
          join("\n",sort(@allskipped))."\n\n".
          &mt('You would have moved ... [_1]',$moveinfo);
}
print "\n\n".&mt('Done.')."\n";

sub choose_domain {
    my ($action,$author,$domref) = @_;
    my ($domain,$skipped,$stopnow,@domains);
    if (ref($domref) eq 'ARRAY') {
        @domains = @{$domref};
    } elsif (ref($domref) eq 'HASH') {
        @domains = sort(keys(%{$domref}));
    }
    if (@domains > 1) {
        print '*** '.&mt('ERROR: [_1] found in multiple domains.',"'$author'")."\n".
               &mt('Enter a number to choose what action to take.')."\n";
        my $num = 1;
        print &mt('Enter [_1]: skip this user.',$num)."\n";
        for (my $i=0; $i<@domains; $i++) {
            my $shown = $domains[$i];
            if (ref($domref) eq 'HASH') {
                if ($domref->{$shown} ne '') {
                    $shown .= ' ('.$domref->{$shown}.') ';
                }
            }
            $num ++; 
            print &mt('Enter [_1]: use domain - [_2].',$num,$shown)."\n";
        }
        $num ++;
        print &mt('Enter [_1]: stop.',$num)."\n";
        print &mt('Your choice:').' ';
        my $choice=<STDIN>;
        chomp($choice);
        if ($choice =~ /^\d+$/) {
            if ($choice == 1) {
                $skipped = 1;       
            } elsif (($choice < $num) && ($choice > 1)) {
                $domain = $domains[$choice-2];
            } elsif ($choice == $num) {
                $stopnow = 1;
            } else {
                print &mt('Invalid choice:')." $choice\n".
                      &mt('Skipping this user.')."\n";
                $skipped = 1;
            }
        } else {
            print &mt('Invalid choice:')." $choice\n".
                  &mt('Skipping this user.')."\n";
            $skipped = 1;
        }
    } elsif (@domains == 1) {
        $domain = $domains[0];
    }
    return ($domain,$skipped,$stopnow);
}

sub move_priv_to_home {
    my ($londocroot,$uid,$gid,$uname,$domain) = @_;
    my $output;
    if ($uname =~ /^$match_username$/ && $domain =~ /^$match_domain$/) {
        my $source_path="$londocroot/priv/$domain/$uname";
        my $target_path="/home/$uname/public_html";
        my $userpath = &LONCAPA::propath($domain,$uname);
        my ($authtype,$userid,$groupid);
        if (-e "$userpath/passwd") {
            if (open(my $fh, "<$userpath/passwd")) {
                ($authtype) = split(/:/,<$fh>);
                close($fh);
            }
        }
        if (!-e "/home/$uname") {
            (undef,undef,$userid,$groupid) = getpwnam($uname);
            if (mkdir("/home/$uname",0711)) {
                if ($authtype eq 'unix' && $userid ne '' && $groupid ne '') {
                    chown($userid,$groupid,"/home/$uname");
                    if (&www_in_group($uname)) {
                        chmod(0710,"/home/$uname");
                    }
                }
            } else {
                $output = &mt('Failed to create directory [_1] -- not moving [_2].',
                          "'/home/$uname'","'$source_path'")."\n";
                return $output;
            }
        }
        if (-e "/home/$uname") {
            if (!-e $target_path) {
                move($source_path,$target_path);
                chown($uid,$gid,$target_path);
                if ($authtype eq 'unix' && $userid ne '' && $groupid ne '') {
                    if (&www_in_group($uname)) {
                        chown($userid,$groupid,$target_path);
                    }
                }
                chmod(02770,$target_path);
                if (-e $target_path && !-e $source_path) {
                    $output = &mt('Moved [_1] to [_2].',"'$source_path'","'$target_path'")."\n";
                } else {
                    $output = &mt('Failed to move [_1] to [_2].',"'$source_path'","'$target_path'")."\n";
                }
            } else {
                $output = &mt('Directory [_1] already exists -- not moving [_2].',
                              "'$target_path'","'$source_path'")."\n";
            }
        }
    }
    return $output;
}

sub www_in_group {
    my ($uname) = @_;
    my $hasuser;
    if (open(PIPE,"/usr/bin/groups www|")) {
        my $memberlist = <PIPE>;
        close(PIPE);
        chomp($memberlist);
        my ($safegroups)=($memberlist=~/:\s*([\s\w]+)$/);
        if ($safegroups =~ /\S/) {
            my @members = split(/\s+/,$safegroups);
            if (grep(/^\Q$uname\E$/,@members)) {
                $hasuser = 1;
            } else {
                my @ugrouplist=grep(!/www|$uname/,@members);
                my $gl=join(',',(@ugrouplist,$uname));
                if (system('/usr/sbin/usermod','-G',$gl,'www') == 0) {
                    $hasuser = 1;
                }
            }
        }
    }
    return $hasuser;
}

sub get_user_selection {
    my ($defaultrun) = @_;
    my $do_action = 0;
    my $choice = <STDIN>;
    chomp($choice);
    $choice =~ s/(^\s+|\s+$)//g;
    my $yes = &mt('y');
    if ($defaultrun) {
        if (($choice eq '') || ($choice =~ /^\Q$yes\E/i)) {
            $do_action = 1;
        }
    } else {
        if ($choice =~ /^\Q$yes\E/i) {
            $do_action = 1;
        }
    }
    return $do_action;
}

sub start_logging {
    my ($fh,$action) = @_;
    my $start = localtime(time);
    print $fh "*****************************************************\n".
              &mt('[_1] - mode is [_2].',
                  'move_construction_spaces.pl',"'$action'")."\n".
              &mt('Started -- time: [_1]',$start)."\n".
              "*****************************************************\n\n";
    return;
}

sub stop_logging {
    my ($fh) = @_;
    my $end = localtime(time);
    print $fh "*****************************************************\n".
               &mt('Ended -- time: [_1]',$end)."\n".
              "*****************************************************\n\n\n";
    close($fh);
    return;
}

sub check_for_restore_files {
    my ($londaemons,$author,$domain) = @_;
    if (opendir(my $homedir,"/home/$author")) {
        my @contents = grep(!/^\.{1,2}$/,readdir($homedir));
        closedir($homedir);
        if (@contents > 0) {
            if (grep(/^restore_\d+\.sh$/,@contents)) {
                if (!-e "$londaemons/logs/moved_construction_spaces") { 
                    mkdir("$londaemons/logs/moved_construction_spaces",0755);
                }
                if (!-e "$londaemons/logs/moved_construction_spaces/$domain") {
                    mkdir("$londaemons/logs/moved_construction_spaces/$domain",0755);
                }
                if (-e "$londaemons/logs/moved_construction_spaces/$domain") {
                    if (open(my $restorefh,">>$londaemons/logs/moved_construction_spaces/$domain/$author")) {
                        foreach my $item (@contents) {
                            if ($item =~ /^restore_\d+\.sh$/) {
                                my @stats = stat("/home/$author/$item");
                                my $lastmod = $stats[9];
                                if (open(my $fh,"</home/$author/$item")) {
                                    print $restorefh
                                          "*******************************\n".
                                          "$item -- ".localtime(time)."\n".
                                          "*******************************\n";
                                    while (<$fh>) {
                                        print $restorefh $_;
                                    }
                                    print $restorefh
                                          "*******************************\n\n";
                                    close($fh);
                                    unlink("/home/$author/$item");
                                }
                            }
                        }
                        close($restorefh); 
                    }
                }
            }
        }
    }
    return;
}

sub connect_mysql {
    my ($lonsqlaccess) = @_;
    my ($dbh,$dbflag);
    eval { $dbh = DBI->connect("DBI:mysql:loncapa","www",
                               $lonsqlaccess,
                               {RaiseError =>0,PrintError=>0}); 
    };
    if ($@) {
        $dbflag = -1;
    } else {
        if (defined($dbh)) {
            $dbflag = 1;
        }
    }
    return ($dbh,$dbflag);
}

sub disconnect_mysql {
    my ($dbh) = @_;
    if (ref($dbh)) {
        $dbh->disconnect;
    }
    return;
}

sub search_allusers {
    my ($dbh,$author) = @_;
    my %fullnames;
    if ((ref($dbh)) && ($author ne '')) {
        eval {
            my $statement = "SELECT domain, lastname, firstname FROM allusers WHERE username='$author'";
            my $sth = $dbh->prepare($statement);
            $sth->execute();
            while ( my ($dom,$last,$first) = $sth->fetchrow_array()) {
                if ($dom ne '') {
                    $fullnames{$dom} = "$first $last";
                }
            }
            $sth->finish;
        };
    }
    return \%fullnames;
}
