# The LearningOnline Network with CAPA
# Interface routines to R CAS
#
# $Id: lonr.pm,v 1.11 2014/11/19 21:14:47 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
 
package Apache::lonr;
 
use strict;
use IO::Socket;
use Apache::lonnet;
use Apache::response();
use LONCAPA;
use Tie::IxHash::Easy; # autoties all subhashes to keep index order
use Data::Dumper;  # used to output hash contents

my $errormsg='';

#
# Rcroak: for use with R-error messages
#
sub Rcroak {
   $errormsg=$_[0];
}

#
#
# Rpeel takes a string containing serialized values from R, 
# peels off the first syntactically complete unit (number, string or array),
# and returns a list (first unit, remainder).
#
sub Rpeel {
        my $x = $_[0];  # the string containing the serialized R object(s)
        if ($x =~ /^N\;(.*)$/) {
            return ('',$1);
        } elsif ($x =~ /^((?:i|d):(.+?);)(.*)$/) {
                return ($1, $+);  # x starts with a number
        } elsif ($x =~ /^s:(\d+):/) {
                my $n = $1;  # x starts with a string of length n
                if ($x =~ /^(s:\d+:\"(.{$n})\";)(.*)$/) {
                        return ($1, $+);  # x starts with a valid string
                } else {
                        &Rcroak('invalid string detected');
                }
        } elsif ($x =~ /^a:/) {
                # x starts with an array -- need to find the closing brace
                my $i = index $x, '{', 0;  # position of first opening brace
                if ($i < 0) {
                        &Rcroak('array with no opening brace');
                }
                my $open = 1;  # counts open braces
                my $j = index $x, '}', $i; # position of first closing brace
                $i = index $x, '{', $i + 1; # position of next opening brace (if any)
                my $pos = -1;  # position of final closing brace
                do {
                        if (($i < $j) && ($i > 0)) {
                                # encounter another opening brace before next closing brace
                                $open++;
                                $i = index $x, '{', $i + 1;  # find the next opening brace
                        } elsif ($j > 0) {
                                # next brace encountered is a closing brace
                                $open--;
                                $pos = $j;
                                $j = index $x, '}', $j + 1;
                        } else {
                                &Rcroak('unmatched left brace');
                        }
                } until ($open eq 0);
                # array runs from start to $pos
                my $a = substr $x, 0, $pos + 1;  # array
                my $b = substr $x, $pos + 1;     # remainder
                return ($a, $b);
        } else {
                &Rcroak('unrecognized R value');
        }
}
# --- end Rpeel ---

#
# Rreturn accepts a string containing a serialized R object
# and returns either the object's value (if it is scalar) or a reference
# to a hash containing the contents of the object.  Any null keys in the hash
# are replaced by 'resultNNN' where NNN is the index of the entry in the original
# R array.
#
sub Rreturn {
    my $x = $_[0];  # the string containing the serialized R object(s)
    $x=~s/^\"//;
    $x=~s/\"$//;
    $x=~s/\\\"/\"/g;
    $errormsg='';
    if ($x =~ /^(?:i|d):(.+?);$/) {
        return $1;  # return the value of the number
    } elsif ($x =~ /^s:(\d+):\"(.*)\";$/) {
        # string -- verify the length
        if (length($2) eq $1) {
           return $2;  # return the string
        } else {
           return 'mismatch in string length';
        }
    } elsif ($x =~ /^a:(\d+):\{(.*)\}$/) {
        # array
        my $dim = $1;  # array size
        $x = $2;  # array contents
        tie(my %h,'Tie::IxHash::Easy'); # start a hash
        keys(%h) = $dim; # allocate space for the hash
        my $key;
        my $y;
        for (my $i = 0; $i < $dim; $i++) {
           ($y, $x) = &Rpeel($x);  # strip off the entry for the key
           if ($y eq '') {
              &Rcroak('ran out of keys');
           }
           $key = &Rreturn($y);
           if ($key eq '') {
              $key = "result$i";  # correct null key
           }
           ($y, $x) = &Rpeel($x);  # strip off the value
           if ($y eq '') {
               &Rcroak('ran out of values');
           }
           if ($y =~ /^a:/) {
               $h{$key} = \&Rreturn($y);  # array value: store as reference
           } else {
               $h{$key} = &Rreturn($y);  # scalar value: store the entry in the hash
           }
        }
        if ($errormsg) { return $errormsg; }
        return \%h;  # return a reference to the hash
    } elsif ($x eq '') {
        return '';
    } else {
        return 'Unrecognized output';
    }
}
# --- end Rreturn ---

sub Rentry {
    my $hash = shift;  # pointer to tied hash
    my $i;
    if (ref($hash) ne 'HASH') {
       return 'Argument to cas_hashref_entry is not a hash!';
    }
    while ($i = shift) {
       if (exists($hash->{$i})) {
          $hash = $hash->{$i};
       } else {
          return undef;
       }
       if (ref($hash) eq 'REF') {
          $hash = $$hash;  # dereference one layer
       } elsif (ref($hash) ne 'HASH') {
          return $hash;  # drilled down to a scalar
       }
    }
}

sub Rarray {
    my $hash = shift;  # pointer to tied hash
    my $i;
    if (ref($hash) ne 'HASH') {
       return 'Argument to cas_hashref_array is not a hash!';
    }
    while ($i = shift) {
       if (exists($hash->{$i})) {
          $hash = $hash->{$i};
       } else {
          return undef;
       }
       if (ref($hash) eq 'REF') {
          $hash = $$hash;  # dereference one layer
       }
    }
    my @returnarray=();
    foreach my $key (keys(%{$hash})) {
        $returnarray[$key-1]=$$hash{$key};
    }
    return @returnarray;
}

sub connect {
   return IO::Socket::UNIX->new(Peer    => $Apache::lonnet::perlvar{'lonSockDir'}.'/rsock',
				Type    => SOCK_STREAM,
				Timeout => 10);
}

sub disconnect {
    my ($socket)=@_;
    if ($socket) { close($socket); }
}

sub rreply {
    my ($socket,$cmd)=@_;
    if ($socket) {
	print $socket &escape($cmd)."\n";
        my $reply=<$socket>;
        chomp($reply);
        if ($reply=~/^Incorrect/) { $reply='Error: '.$reply; }
        return &unescape($reply);
    } else {
        return 'Error: no connection.';
    }
}

sub blacklisted {
    my ($cmd)=@_;
    foreach my $forbidden (
        'read','write','scan','save','socket','connections',
        'open','close',
        'plot','X11','windows','quartz',
        'postscript','pdf','png','jpeg',
        'dev\.list','dev\.next','dev\.prev','dev\.set',
        'dev\.off','dev\.copy','dev\.print','graphics',
        'library','package','source','sink','objects',
        'Sys\.','unlink','file\.','on\.exit','error',
        'q\(\)'
     ) {
	if ($cmd=~/$forbidden/s) { return 1; }
    } 
    return 0;
}

sub r_allowed_libraries {
   return ('alr3','boot','car','class','cluster','datasets','Hmisc','KernSmooth','leaps','lmtest',
           'MASS','methods','mgcv','nlme','nnet','qAnalyst','quadprog','rpart','SuppDists','spatial',
           'splines','stats','stats4','survival','tseries','zoo');
}

sub r_is_allowed_library {
    my ($library)=@_;
    foreach my $allowed_library (&r_allowed_libraries()) {
       if ($library eq $allowed_library) { return 1; }
    }
    return 0;
}

sub runscript {
    my ($socket,$fullscript,$libraries)=@_;
    if (&blacklisted($fullscript)) { return 'Error: blacklisted'; }
    my $reply='';
    $fullscript=~s/[\n\r\l]//gs;
    if ($libraries) {
       foreach my $library (split(/\s*\,\s*/,$libraries)) {
          unless ($library=~/\w/) { next; }
          if (&r_is_allowed_library($library)) {
              $reply=&rreply($socket,'library('.$library.');');
              if ($reply=~/^Error\:/) { return $reply; }
          } else { 
             return 'Error: blacklisted'; 
          }
       }
    }
    $fullscript=~s/\;+\s*$//s;
    if ($fullscript=~/\w/) { $reply=&rreply($socket,$fullscript.';'); }
    if ($reply=~/^Error\:/) { return $reply; }
    $reply=~s/^\s*//gs;
    $reply=~s/\s*$//gs;
    &Apache::lonxml::debug("r $fullscript \n reply $reply");
    return $reply;
}

sub runserializedscript {
    my ($socket,$fullscript,$libraries)=@_;
    if (&blacklisted($fullscript)) { return 'Error: blacklisted'; }
    my $reply;
    $fullscript=~s/[\n\r\l]//gs;
    if ($libraries) {
       foreach my $library (split(/\s*\,\s*/,$libraries)) {
          unless ($library=~/\w/) { next; }
          if (&r_is_allowed_library($library)) {
              $reply=&rreply($socket,'library('.$library.');');
              if ($reply=~/^Error\:/) { return($reply,$reply); }
          } else {
             return 'Error: blacklisted';
          }
       }
    }
    $fullscript=~s/\;+\s*$//s;
    my $lastline='';
    my $firstpart='';
    if ($fullscript=~/\;/) {
       ($firstpart,$lastline)=($fullscript=~/^(.*\;)([^\;]+)$/);
    } else {
       $lastline=$fullscript;
    }
    if ($firstpart) {
        $firstpart=~s/\;+\s*$//s;
        $reply=&rreply($socket,$firstpart.';');
        if ($reply=~/^Error\:/) { return($reply,$reply); }
    }
# The last line needs to be serialized
    $reply=&Rreturn(&rreply($socket,"phpSerialize($lastline);"));
    return($reply,&Dumper($reply));
}

sub r_cas_formula_fix {
   my ($expression)=@_;
   return &Apache::response::implicit_multiplication($expression);
}

sub r_run {
    my ($script,$submission,$argument,$libraries) = @_;
    my $socket=&connect();
    my @submissionarray=split(/\s*\,\s*/,$submission);
    for (my $i=0;$i<=$#submissionarray;$i++) {
        my $n=$i+1;
        my $fixedsubmission=&r_cas_formula_fix($submissionarray[$i]);
        $script=~s/RESPONSE\[$n\]/$fixedsubmission/gs;
    }
    my @argumentarray=@{$argument};
    for (my $i=0;$i<=$#argumentarray;$i++) {
        my $n=$i+1;
        my $fixedargument=&r_cas_formula_fix($argumentarray[$i]);
        $script=~s/LONCAPALIST\[$n\]/$fixedargument/gs;
    }
    my $reply=&runscript($socket,$script,$libraries);
    &disconnect($socket);
    if ($reply=~/^\s*true\s*$/i) { return 'EXACT_ANS'; }
    if ($reply=~/^\s*false\s*$/i) { return 'INCORRECT'; } 
    return 'BAD_FORMULA';
}

sub r_eval {
    my ($script,$libraries,$hashflag) = @_;
    my $socket=&connect();
    my $reply;
    my $dump='';
    if ($hashflag) {
       ($reply,$dump)=&runserializedscript($socket,$script,$libraries);
    } else {
       $reply=&runscript($socket,$script,$libraries);
    }
    &disconnect($socket);
    return ($reply,$dump);
}


sub compareterms {
    my ($socket,$terma,$termb)=@_;
    my $difference=$terma.'-('.$termb.')';
    if (&blacklisted($difference)) { return 'Error: blacklisted'; }
    my $reply=&rreply($socket,$difference.';');
    if ($reply=~/^\s*0\s*$/) { return 'true'; }
    if ($reply=~/^Error\:/) { return $reply; }
    return 'false';
}

sub r_check {
    my ($response,$answer,$reterror) = @_;
    my $socket=&connect();
    my $reply=&compareterms($socket,$response,$answer);
    &disconnect($socket);
    # integer to string mappings come from capaParser.h
    # 1 maps to 'EXACT_ANS'
    if ($reply eq 'true') { return 1; }
    # 11 maps to 'BAD_FORMULA'
    if ($reply=~/^Error\:/) { return 11; }
    # 7 maps to 'INCORRECT'
    return 7;
}
 
1;
__END__;
