# The LearningOnline Network
# Allow access to password changing via a token sent to user's e-mail. 
#
# $Id: resetpw.pm,v 1.36 2013/08/17 00:34:29 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
#

=pod

=head1 NAME

Apache::resetpw: reset user password.

=head1 SYNOPSIS

Handles resetting of forgotten passwords.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.
 
=head1 OVERVIEW

A user with an e-mail address associated with his/her LON-CAPA username
can reset a forgotten password, using a link sent to the e-mail address
if the authentication type for the account is "internal".

=cut

package Apache::resetpw;

use strict;
use Apache::Constants qw(:common);
use Apache::lonacc;
use Apache::lonnet;
use Apache::loncommon;
use Apache::lonlocal;
use LONCAPA;

sub handler {
    my $r = shift;
    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;
    if ($r->header_only) {
        return OK;
    }
    my $contact_name = &mt('LON-CAPA helpdesk');
    my $origmail =  $r->dir_config('lonSupportEMail');
    my $server = $r->dir_config('lonHostID');
    my $defdom = &Apache::lonnet::default_login_domain();
    my $contacts =
        &Apache::loncommon::build_recipient_list(undef,'helpdeskmail',
                                                 $defdom,$origmail);
    my ($contact_email) = split(',',$contacts);
    my $handle = &Apache::lonnet::check_for_valid_session($r);
    my $lonidsdir=$r->dir_config('lonIDsDir');
    if ($handle ne '') {
        if ($handle=~/^publicuser\_/) {
            unlink($r->dir_config('lonIDsDir')."/$handle.id");
        } else {
            &Apache::lonnet::transfer_profile_to_env($lonidsdir,$handle);
        }
    }
    &Apache::lonacc::get_posted_cgi($r);
    &Apache::lonlocal::get_language_handle($r);
    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},['token']);
    
    my @emailtypes = ('permanentemail','critnotification','notification');
    my $uname = &unescape($env{'form.uname'});
    my $udom = $env{'form.udom'};
    my $token = $env{'form.token'};
    my $brcrum = [];
    if ($token) {
        push (@{$brcrum},
            {href => '/adm/resetpw',
             text => 'Update Password'});
    } else {
        push (@{$brcrum},
            {href => '/adm/resetpw',
             text => 'Account Information'});
        if ($uname && $udom) {
            push (@{$brcrum},
                {href => '/adm/resetpw',
                 text => 'Result'});
        }
    }
    my $args = {bread_crumbs => $brcrum};
    $r->print(&Apache::loncommon::start_page('Reset password','',$args));
    $r->print('<h2>'.&mt('Reset forgotten LON-CAPA password').'</h2>');
    my $output;
    if ($token) {
        $output = &reset_passwd($r,$token,$contact_name,$contact_email);
    } elsif ($uname && $udom) {
        my $domdesc = &Apache::lonnet::domain($udom,'description');
        my $authtype = &Apache::lonnet::queryauthenticate($uname,$udom);
        if ($authtype =~ /^internal/) {
            my $useremail = $env{'form.useremail'};
            if ($useremail !~ /^[^\@]+\@[^\@]+\.[^\@\.]+$/) {
                $output = &invalid_state('baduseremail',$domdesc,
                                         $contact_name,$contact_email);
            } else {
                my %userinfo = 
		    &Apache::lonnet::get('environment',\@emailtypes,
					 $udom,$uname);
                my @allemails;
                foreach my $type (@emailtypes) {
                    my $email = $userinfo{$type};
                    my @items;
                    if ($email =~ /,/) {
                        @items = split(',',$userinfo{$type});
                    } else {
                        @items = ($email);
                    }
                    foreach my $item (@items) {
                        if ($item =~ /^[^\@]+\@[^\@]+\.[^\@\.]+$/) {
                            unless(grep(/^\Q$item\E$/,@allemails)) { 
                                push(@allemails,$item);
                            }
                        }
                    }
                }
                if (@allemails > 0) {
                    if (grep(/^\Q$useremail\E$/,@allemails)) {
                        $output = &send_token($uname,$udom,$useremail,$server,
                                              $domdesc,$contact_name,
                                              $contact_email);
                    } else {
                        $output = &invalid_state('mismatch',$domdesc,
                                                 $contact_name,
                                                 $contact_email);
                    }
                } else {
                    $output = &invalid_state('missing',$domdesc,
                                             $contact_name,$contact_email);
                }
            }
        } elsif ($authtype =~ /^(krb|unix|local)/) { 
            $output = &invalid_state('authentication',$domdesc,
                                     $contact_name,$contact_email);
        } else {
            $output = &invalid_state('invalid',$domdesc,
                                     $contact_name,$contact_email);
        }
    } else {
        $output = &get_uname($defdom);
    }
    $r->print($output);
    $r->print(&Apache::loncommon::end_page());
    return OK;
}

sub get_uname {
    my ($defdom) = @_;
    my %lt = &Apache::lonlocal::texthash(
                                         unam => 'LON-CAPA username',
                                         udom => 'LON-CAPA domain',
                                         uemail => 'E-mail address in LON-CAPA',
                                         proc => 'Proceed');

    my $msg = &mt('If you use the same account for other campus services besides LON-CAPA, (e.g., e-mail, course registration, etc.), a separate centrally managed mechanism likely exists to reset a password. However, if your account is used for just LON-CAPA access you will probably be able to reset a password from this page.');
    $msg .= '<br /><br />'.&mt('Three conditions must be met:')
           .'<ul><li>'.&mt('An e-mail address must have previously been associated with your LON-CAPA username.').'</li>'
           .'<li>'.&mt('You must be able to access e-mail sent to that address.').'</li>'
           .'<li>'.&mt('Your LON-CAPA account must be of a type for which LON-CAPA can reset a password.')
           .'</ul>';
    $msg .= '<form name="forgotpw" method="post" action="/adm/resetpw">'.
            &Apache::lonhtmlcommon::start_pick_box(). 
            &Apache::lonhtmlcommon::row_title($lt{'unam'}).
            '<input type="text" name="uname" size="20" />'.
            &Apache::lonhtmlcommon::row_closure(1).
            &Apache::lonhtmlcommon::row_title($lt{'udom'}).
            &Apache::loncommon::select_dom_form($defdom,'udom').
            &Apache::lonhtmlcommon::row_closure(1).
            &Apache::lonhtmlcommon::row_title($lt{'uemail'}).
            '<input type="text" name="useremail" size="30" />'.
            &Apache::lonhtmlcommon::end_pick_box().
            '<br /><br /><input type="submit" name="resetter" value="'.$lt{'proc'}.'" /></form>';
    return $msg;
}

sub send_token {
    my ($uname,$udom,$email,$server,$domdesc,$contact_name,
        $contact_email) = @_;
    my $msg =
        '<p class="LC_info">'
       .&mt('Thank you for your request to reset the password for your LON-CAPA account.')
       .'</p>';

    my $now = time;
    my $temppasswd = &create_passwd();
    my %info = ('ip'         => $ENV{'REMOTE_ADDR'},
		'time'       => $now,
		'domain'     => $udom,
		'username'   => $uname,
		'email'      => $email,
		'temppasswd' => $temppasswd);

    my $token = &Apache::lonnet::tmpput(\%info,$server,'resetpw');
    if ($token !~ /^error/ && $token ne 'no_such_host') {
        my $esc_token = &escape($token);
        my $showtime = &Apache::lonlocal::locallocaltime(time);
        my $reseturl = &Apache::lonnet::absolute_url().'/adm/resetpw?token='.$esc_token;
        my $mailmsg = &mt('A request was submitted on [_1] for reset of the password for your LON-CAPA account.',$showtime)." \n".&mt('To complete this process please open a web browser and enter the following URL in the address/location box: [_1]',"\n\n".$reseturl);
        my $result = &send_mail($domdesc,$email,$mailmsg,$contact_name,
                                $contact_email);
        if ($result eq 'ok') {
            $msg .=
                &mt('An e-mail sent to the e-mail address associated with your LON-CAPA account includes the web address for the link you should use to complete the reset process.')
               .'<br /><br />'
               .&mt('The link included in the message will be valid for the next [_1]two[_2] hours.','<b>','</b>');
        } else {
            $msg .=
                '<p class="LC_error">'
               .&mt('An error occurred when sending a message to the e-mail address'
                   .' associated with your LON-CAPA account.')
               .'</p>'
               .&display_actions($contact_email,$domdesc);
        }
    } else {
        $msg .=
            '<p class="LC_error">'
           .&mt('An error occurred creating a token required for the'
               .' password reset process.')
           .'</p>'
           .&display_actions($contact_email,$domdesc);
    }
    return $msg;
}

sub send_mail {
    my ($domdesc,$email,$mailmsg,$contact_name,$contact_email) = @_;
    my $outcome;
    my $requestmail = "To: $email\n".
                      "From: $contact_name <$contact_email>\n".
                      "Subject: ".&mt('Your LON-CAPA account')."\n".
                      "Content-type: text/plain\;charset=UTF-8\n".
                      "\n\n".$mailmsg."\n\n".
                      &mt('[_1] LON-CAPA support team',$domdesc)."\n".
                      "$contact_email\n";
    if (open(MAIL, "|/usr/lib/sendmail -oi -t -odb")) {
        print MAIL $requestmail;
        close(MAIL);
        $outcome = 'ok';
    } else {
        $outcome = 'fail';
    }
    return $outcome;
}

sub invalid_state {
    my ($error,$domdesc,$contact_name,$contact_email) = @_;
    my $msg;
    if ($error eq 'invalid') {
        $msg =
            '<p class="LC_warning">'
            .&mt('The username you provided was not verified as a valid username'
                .' in the LON-CAPA system for the [_1] domain.','<i>'.$domdesc.'</i>')
              .'</p>';
        $msg .= &display_actions($contact_email,$domdesc);
    } else {
        if ($error eq 'baduseremail') {
            $msg = &mt('The e-mail address you provided does not appear to be a valid address.');
        } elsif ($error eq 'mismatch') {
            $msg = &mt('The e-mail address you provided does not match the address recorded in the LON-CAPA system for the username and domain you provided.');  
        } elsif ($error eq 'missing') {
            $msg = &mt('A valid e-mail address was not located in the LON-CAPA system for the username and domain you provided.');
        } elsif ($error eq 'authentication') {
            $msg = &mt('The username you provided uses an authentication type which can not be reset directly via LON-CAPA.');
        }
        $msg = '<p class="LC_warning">'.$msg.'</p>'
              .&display_actions($contact_email,$domdesc);
    }
    return $msg;
}

sub reset_passwd {
    my ($r,$token,$contact_name,$contact_email) = @_;
    my $msg;
    my %data = &Apache::lonnet::tmpget($token);
    my $now = time;
    if (keys(%data) == 0) {
        $msg = &mt('Sorry, the URL you provided to complete the reset of your password was invalid. Either the token included in the URL has been deleted or the URL you provided was invalid. Please submit a [_1]new request[_2] for a password reset, and follow the link to the new URL included in the e-mail that will be sent to you, to allow you to enter a new password.'
                  ,'<a href="/adm/resetpw">','</a>');
        return $msg;
    }
    if (($data{'time'} =~ /^\d+$/) && 
        ($data{'username'} ne '') && 
        ($data{'domain'} ne '') && 
        ($data{'email'}  =~ /^[^\@]+\@[^\@]+\.[^\@\.]+$/) && 
        ($data{'temppasswd'} =~/^\w+$/)) {
        my $reqtime = &Apache::lonlocal::locallocaltime($data{'time'});
        if ($now - $data{'time'} < 7200) {
            if ($env{'form.action'} eq 'verify_and_change_pass') {
                unless (($env{'form.uname'} eq $data{'username'}) && ($env{'form.udom'} eq $data{'domain'}) && ($env{'form.email'} eq $data{'email'})) {
                    $msg = &generic_failure_msg($contact_name,$contact_email);
                    return $msg;
                }
                my $change_failed = 
		    &Apache::lonpreferences::verify_and_change_password($r,'reset_by_email',$token);
                if (!$change_failed) {
                    my $delete = &Apache::lonnet::tmpdel($token);
                    my $now = &Apache::lonlocal::locallocaltime(time);
                    my $domdesc = 
			&Apache::lonnet::domain($data{'domain'},'description');
                    my $mailmsg = &mt('The password for your LON-CAPA account in the [_1] domain was changed [_2] from IP address: [_3].  If you did not perform this change or authorize it, please contact the [_4] ([_5]).',$domdesc,$now,$ENV{'REMOTE_ADDR'},$contact_name,$contact_email)."\n";
                    my $result = &send_mail($domdesc,$data{'email'},$mailmsg,
                                            $contact_name,$contact_email);
                    my $confirm_msg;
                    if ($result eq 'ok') {
                        $confirm_msg =
                            &Apache::lonhtmlcommon::confirm_success(
                                &mt('An e-mail confirming setting of the password'
                                   .' for your LON-CAPA account has been sent to [_1].'
                                    ,'<span class="LC_filename">'.$data{'email'}.'</span>'));
                    } else {
                        $confirm_msg =
                            &Apache::lonhtmlcommon::confirm_success(
                                &mt('An error occurred when sending e-mail to [_1]'
                                   .' confirming setting of your new password.'
                                    ,'<span class="LC_filename">'.$data{'email'}.'</span>'),1);
                    }
                    $msg .=
                        &Apache::loncommon::confirmwrapper($confirm_msg)
                       .&Apache::lonhtmlcommon::actionbox([
                            '<a href="/adm/login">'.&mt('Go to the login page').'</a>']);
                } elsif ($change_failed eq 'invalid_client') {
                    my $homeserver = &Apache::lonnet::homeserver($data{'username'},$data{'domain'});
                    if ($homeserver eq 'no_host') {
                        $msg .= &generic_failure_msg($contact_name,$contact_email);
                    } else {
                        my $protocol = $Apache::lonnet::protocol{$homeserver};
                        $protocol = 'http' if ($protocol ne 'https');
                        my $url = $protocol.'://'.&Apache::lonnet::hostname($homeserver).
                                  '/adm/resetpw';
                        my ($opentag,$closetag);
                        if ($url) {
                           $opentag = '<a href="'.$url.'">';
                           $closetag = '</a>';
                        }
                        $msg .=
                            '<p class="LC_warning">'
                           .&mt('A problem occurred when attempting to reset'
                               .' the password for your account.'
                               .' Please try again from your [_1]home server[_2].'
                                ,$opentag,$closetag)
                           .'</p>';
                    }
                } else {
                    $msg .= &generic_failure_msg($contact_name,$contact_email);
                }
            } else {
                $r->print(&mt('The token included in an e-mail sent to you [_1] has been verified, so you may now proceed to reset the password for your LON-CAPA account.',$reqtime).'<br /><br />');
                $r->print(&mt('Please enter the username and domain of the LON-CAPA account, and the associated e-mail address, for which you are setting a password. The new password must contain at least 7 characters.').' '.&mt('Your new password will be sent to the LON-CAPA server in an encrypted form.').'<br />');
                &Apache::lonpreferences::passwordchanger($r,'','reset_by_email',$token);
            }
        } else {
            $msg =
                '<p class="LC_warning">'
               .&mt('Sorry, the token generated when you requested a password reset has expired. Please submit a [_1]new request[_2], and follow the link to the web page included in the new e-mail that will be sent to you, to allow you to enter a new password.'
                    ,'<a href="/adm/resetpw">','</a>')
               .'</p>';
        }
    } else {
        $msg .=
            '<p class="LC_warning">'
           .&mt('Sorry, the URL generated when you requested reset of your password contained incomplete information. Please submit a [_1]new request[_2] for a password reset, and use the new URL that will be sent to your e-mail account to complete the process.'
                ,'<a href="/adm/resetpw">','</a>')
           .'</p>';
    }
    return $msg;
}

sub generic_failure_msg {
    my ($contact_name,$contact_email) = @_;
    return
        '<p class="LC_error">'
       .&mt('A problem occurred when attempting to reset the password for your account.')
       .'<br />'
       .&mt('Please contact the [_1] ([_2]) for assistance.',
              $contact_name,'<a href="mailto:'.$contact_email.'">'.$contact_email.'</a>')
       .'</p>';
}

sub create_passwd {
    my $passwd = '';
    my @letts = ("a".."z");
    for (my $i=0; $i<8; $i++) {
        my $lettnum = int(rand(2));
        my $item = '';
        if ($lettnum) {
            $item = $letts[int(rand(26))];
            my $uppercase = int(rand(2));
            if ($uppercase) {
                $item =~ tr/a-z/A-Z/;
            }
        } else {
            $item = int(rand(10));
        }
        $passwd .= $item;
    }
    return ($passwd);
}

sub display_actions {
    my ($contact_email, $domdesc) = @_;
    my @msg = (&mt('[_1]Go back[_2] and try again',
                   '<a href="javascript:history.go(-1)">','</a>'));
    my $msg2 = '';
    if ($contact_email ne '') {
        my $escuri = &HTML::Entities::encode('/adm/resetpw','&<>"');
        push(@msg, &mt('Contact the [_1]LON-CAPA helpdesk[_2] for the institution: [_3]',
                       '<a href="/adm/helpdesk?origurl='.$escuri.'">',
                       '</a>','<i>'.$domdesc.'</i>'));
    } else {
        $msg2 =
            '<p>'
           .&mt('You may wish to send an e-mail to the'
           .' server administrator: [_1] for the [_2] domain.',
                '<i>'.$Apache::lonnet::perlvar{'AdmEMail'}.'</i>',
                '<i>'.$domdesc.'</i>')
           .'</p>';
    }

    return &Apache::lonhtmlcommon::actionbox(\@msg).$msg2;

}

1;
