# The LearningOnline Network
# Access Handler for File Transfers
#
# $Id: lonracc.pm,v 1.23 2008/11/12 20:01:09 jms Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#

=pod

=head1 NAME

Apache::lonracc - Access Handler for File Transfers

=head1 SYNOPSIS

Invoked by /etc/httpd/conf/loncapa.conf:

 <LocationMatch "^/raw.*">
 PerlAccessHandler Apache::lonracc
 </LocationMatch>

=head1 INTRODUCTION

This module enables authentication for file transfers and works
against the /res tree.

Only lond invokes the /raw namespace through its subscribe function.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 HANDLER SUBROUTINE

This routine is called by Apache and mod_perl.

=over 4

=item *

Determine requesting host

=item *

See whether or not the requesting host is subscribed.

=item *

Respond with status of request and make log entry in case of unallowed
access.

=back

=cut

package Apache::lonracc;

use strict;
use Apache::Constants qw(:common :remotehost);
use Apache::lonnet;
use Apache::File();
use IO::Socket;

sub subscribed {
    my ($filename,$id) = @_;

    return 0 if (!-e "$filename.subscription");

    my $hostname=&Apache::lonnet::hostname($id);
    my (undef,undef,undef,undef,$ip) = gethostbyname($hostname);
    
    return 0 if (length($ip) != 4);

    $ip=inet_ntoa($ip);

    my $expr='^'.quotemeta($id).':'.quotemeta($ip).':';

    my $found=0;
    if (my $sh=Apache::File->new("$filename.subscription")) {
	while (my $subline=<$sh>) { if ($subline =~ /$expr/) { $found=1; } }
	$sh->close();
    }
    return $found;
}

sub handler {
    my $r = shift;

    my $filename=$r->filename;
    if (!-e $filename) {
	return NOT_FOUND;
    }

    my $reqhost = $r->get_remote_host(REMOTE_NOLOOKUP);
    my @hostids= &Apache::lonnet::get_hosts_from_ip($reqhost);
    if (!@hostids && $reqhost ne '127.0.0.1' ) {
	$r->log_reason("Unable to find a host for ".
		       $r->get_remote_host(REMOTE_NOLOOKUP));
	return FORBIDDEN;
    }
    if ($reqhost eq '127.0.0.1') {
	return OK;
    }
    my $return;
    my @ids;

    foreach my $id (@hostids) {
	my $uri =$r->uri;
	if (($filename=~/\.meta$/) ||
	    ($uri=~m|^/raw/uploaded|) ||
	    (-e "$filename.$id") ||
	    &subscribed($filename,$id) ) {
	    return OK;
	} else {
	    $return=FORBIDDEN;
	    push(@ids,$id);
	}
    }
    if ($return == FORBIDDEN) {
	$r->log_reason(join(':',@ids)." not subscribed", $r->filename);
	return FORBIDDEN;
    }
    $r->log_reason("Invalid request for file transfer from $reqhost", 
                   $r->filename); 
    return FORBIDDEN;
}

1;
__END__









