# The LearningOnline Network
# Starts a user off based of an existing token.
#
# $Id: migrateuser.pm,v 1.25 2016/12/05 00:51:43 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#

package Apache::migrateuser;

use strict;
use LONCAPA qw(:DEFAULT :match);
use Apache::Constants qw(:common :http :methods);
use Apache::lonauth;
use Apache::lonnet;
use Apache::lonlocal;
use Apache::lonlogin();

sub goto_login {
    my ($r,$domain) = @_;
    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;
    my $url = '/adm/login';
    if ($domain) {
        $url .= '?domain='.$domain;
    }
    $r->print(&Apache::loncommon::start_page('Going to login',undef,
					     {'redirect' => [0,$url],}).
	      '<h1>'.&mt('One moment please...').'</h1>'.
	      '<p>'.&mt('Transferring to login page.').'</p>'.
	      &Apache::loncommon::end_page());
    return OK;
}


sub sso_check {
    my ($data) = @_;
    my %extra_env;
    if (ref($data) eq 'HASH') {
        if ($data->{'sso.login'}) {
            $extra_env{'request.sso.login'} = $data->{'sso.login'};
        }
        if ($data->{'sso.reloginserver'}) {
            $extra_env{'request.sso.reloginserver'} = 
                $data->{'sso.reloginserver'};
        }
    }
    return \%extra_env;
}

sub ip_changed {
    my ($r,$udom,$camefrom,$dataref) = @_;
    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;
    if (ref($dataref) eq 'HASH') {
        my $title = 'LON-CAPA Session redirected';
        my $message = &mt('Your internet address has changed since you logged in.');  
        my $rule_in_effect;
        if ($dataref->{'balancer'}) {
            my $baldom = &Apache::lonnet::host_domain($camefrom);
            my $balprimaryid = &Apache::lonnet::domain($baldom,'primary');
            my $balintdom = &Apache::lonnet::internet_dom($balprimaryid);
            my $uprimaryid = &Apache::lonnet::domain($udom,'primary'); 
            my $uintdom = &Apache::lonnet::internet_dom($uprimaryid);
            my $dom_in_use;
            if (($uintdom ne '') && ($uintdom eq $balintdom)) {
                $dom_in_use = $udom;
            } else {
                $dom_in_use = $baldom;
            }
            my ($result,$cached)=&Apache::lonnet::is_cached_new('loadbalancing',$dom_in_use);
            unless (defined($cached)) {
                my $cachetime = 60*60*24; 
                my %domconfig =
                    &Apache::lonnet::get_dom('configuration',['loadbalancing'],$dom_in_use);
                if (ref($domconfig{'loadbalancing'}) eq 'HASH') {
                    $result = &Apache::lonnet::do_cache_new('loadbalancing',$dom_in_use,
                                                            $domconfig{'loadbalancing'},$cachetime);
                }
            }
            if (ref($result) eq 'HASH') {
                (undef,my $currtargets,my $currrules) =
                    &Apache::lonnet::check_balancer_result($result,$dataref->{'server'});
                if (ref($currrules) eq 'HASH') {
                    if ($dataref->{'sso.login'}) {
                        if ($currrules->{'_LC_ipchangesso'} ne '') {
                            $rule_in_effect = $currrules->{'_LC_ipchangesso'};
                        }
                    } else {
                        if ($currrules->{'_LC_ipchange'} ne '') {
                            $rule_in_effect = $currrules->{'_LC_ipchange'};
                        }
                    }
                }
            }
        }
        my $url;
        my $lonhost= $r->dir_config('lonHostID');
        my $switchto = $lonhost;
        if ($rule_in_effect ne 'offloadedto') {
            my $hosthere;
            my @ids=&Apache::lonnet::current_machine_ids();
            unless ($rule_in_effect eq 'balancer') {
                if (grep(/^\Q$rule_in_effect\E$/,@ids)) {
                    $hosthere = 1;
                }
            }
            unless ($hosthere) {
                if ($dataref->{'role'}) {
                    my ($adom,$aname);
                    if ($dataref->{'role'} =~ m{^au\./($match_domain)/$}) {
                        $adom = $1;
                        $aname = $dataref->{'username'};
                    } elsif ($dataref->{'role'} =~ m{^(?:ca|aa)\./($match_domain)/($match_username)$}) {
                        $adom = $1;
                        $aname = $2;
                    }
                    if ($adom ne '' && $aname ne '') {
                        my $ahome = &Apache::lonnet::homeserver($aname,$adom);
                        unless ($ahome eq 'no_host') {
                            if ($ahome && grep(/^\Q$ahome\E$/,@ids)) {
                                $hosthere = 1;
                            }
                        }
                    }
                }
            }
            unless ($hosthere) {
                my $hostname;
                if ($rule_in_effect eq 'balancer') {
                    $hostname = &Apache::lonnet::hostname($dataref->{'server'});
                    if ($hostname) {
                        $switchto = $dataref->{'server'};
                    }
                } else {
                    $hostname = &Apache::lonnet::hostname($rule_in_effect);
                    if ($hostname) {
                        $switchto = $rule_in_effect;
                    }
                }
                if ($hostname) {
                    my $protocol = $Apache::lonnet::protocol{$switchto};
                    $protocol = 'http' if ($protocol ne 'https');
                    $url = $protocol.'://'.$hostname;
                    if ($rule_in_effect eq 'balancer') {
                        $message .= '<br />'.
                                    &mt('As a result, your LON-CAPA session is being redirected to the server where you originally logged in.');
                    } else {
                        $message .= '<br />'.
                                    &mt('As a result, your LON-CAPA session is being redirected.');
                    }
                }
            }
        }
        if ($dataref->{'sso.login'}) {
            $url .= '/adm/roles';
        } else {
            $url .= '/adm/login';
            if ($udom) {
                $url .= '?domain='.$udom;
            }
            $message .= '<br />'.&mt('You will need to provide your password one more time.');
        }
        my %info= (
                    'domain'          => $udom,
                    'username'        => $dataref->{'username'},
                    'role'            => $dataref->{'role'},
                    'sessionserver'   => $lonhost,
                  );
        if ($dataref->{'origurl'}) {
            $info{'origurl'} = $dataref->{'origurl'};
        }
        if ($dataref->{'symb'}) {
            $info{'symb'} = $dataref->{'symb'};
        }
        my $iptoken = &Apache::lonnet::tmpput(\%info,$switchto);
        unless ($iptoken eq 'conlost') {
            $url .= ($url =~ /\?/) ? '&' : '?';
            $url .= 'iptoken='.$iptoken;
        }
        $r->print(&Apache::loncommon::start_page($title,undef,
                                                 {'redirect' =>
                                                  [2,$url],}).
                  '<h1>'.&mt('One moment please...').'</h1>'.
                  '<p class="LC_warning">'.$message.'</p>'.
                  &Apache::loncommon::end_page());
    } else {
        return &goto_login($r);
    }
    return OK;
}

sub handler {
    my ($r) = @_;
    
    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},['token']);
    my %data =   &Apache::lonnet::tmpget($env{'form.token'});
    if (keys(%data) == 0) {
        return &goto_login($r);
    }
    my $delete = &Apache::lonnet::tmpdel($env{'form.token'});

    &Apache::lonlocal::get_language_handle($r);

    if ($delete ne 'ok') {
	return &goto_login($r);
    }

    if (!defined($data{'username'}) || !defined($data{'domain'})) {
        return &goto_login($r);
    }
    if ($data{'ip'} ne $ENV{'REMOTE_ADDR'}) {
        &Apache::lonnet::logthis('IP change when session migration requested -- was: '.
                 $data{'ip'}.'; now: '.$ENV{'REMOTE_ADDR'}.' for '.$data{'username'}.':'.$data{'domain'});
	return &ip_changed($r,$data{'domain'},$data{'server'},\%data);
    }

    &Apache::lonnet::logthis("Allowing access for $data{'username'}:$data{'domain'} to $data{'role'}");
    my $home=&Apache::lonnet::homeserver($data{'username'},$data{'domain'});
    my $udom;
    if (&Apache::lonnet::domain($data{'domain'})) {
        $udom=$data{'domain'};
    }
    if ($home =~ /(con_lost|no_such_host)/) { return &goto_login($r,$udom); }

    my $extra_env = &sso_check(\%data);

    my %form;
    if ($data{'symb'} ne '') {
        $form{'symb'} = $data{'symb'};
    }
    if ($data{'iptoken'} ne '') {
        $form{'iptoken'} = $data{'iptoken'};
    }
    if ($data{'noloadbalance'} ne '') {
        $form{'noloadbalance'} = $data{'noloadbalance'};
    }

    if (!$data{'role'}) {
	my $handle = &Apache::lonnet::check_for_valid_session($r);
	if ($handle) {
	    &Apache::lonnet::transfer_profile_to_env($r->dir_config('lonIDsDir'),
						     $handle);
            if ($data{'origurl'} ne '') {
                $r->internal_redirect($data{'origurl'});
            } elsif ($env{'request.course.id'}) {
                $r->internal_redirect('/adm/navmaps');
	    } else {
		$r->internal_redirect('/adm/roles');
	    }
	} else {
            my $desturl = '/adm/roles';
            if ($data{'origurl'} ne '') {
                $desturl = $data{'origurl'};
            }
	    &Apache::lonauth::success($r,$data{'username'},$data{'domain'},
				      $home,$desturl,$extra_env,\%form);

	}
	return OK;

    }

    my $next_url='/adm/roles?selectrole=1&amp;'.&escape($data{'role'}).'=1';
    if ($data{'origurl'} ne '') {
        $next_url .= '&amp;orgurl='.&escape($data{'origurl'});
    }
    &Apache::lonauth::success($r,$data{'username'},$data{'domain'},$home,
			      $next_url,$extra_env,\%form);
    return OK;
}

1;
__END__
