# The LearningOnline Network
# Documents
#
# $Id: lonextresedit.pm,v 1.8.2.2 2017/03/14 21:09:20 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#

package Apache::lonextresedit;

use strict;
use Apache::Constants qw(:common :http);
use HTML::Entities;
use Apache::lonlocal;
use Apache::lonnet;
use Apache::loncommon;
use Apache::lonhtmlcommon;
use Apache::lonuserstate;
use LONCAPA::map();
use LONCAPA qw(:DEFAULT :match);

sub handler {
    my $r=shift;
    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;

    return OK if $r->header_only;

    # Check for access
    if (! &Apache::lonnet::allowed('mdc',$env{'request.course.id'})) {
        $env{'user.error.msg'}=
            $r->uri.":mdc:0:0:Cannot modify course content.";
            return HTTP_NOT_ACCEPTABLE;
    }

    my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
    my $chome = $env{'course.'.$env{'request.course.id'}.'.home'};
    my ($supplementalflag,$updated,$output,$errormsg,$residx,$url,$title,$symb);
    if (($env{'form.folderpath'} =~ /^supplemental/) && ($env{'form.suppurl'})) {
        $supplementalflag = 1;
    }
    if (($supplementalflag) || ($env{'form.symb'} =~ /^uploaded/)) {
        ($updated,$output,$errormsg,$residx,$url,$title,$symb) =
            &process_changes($supplementalflag,$cdom,$cnum,$chome);
        if ($supplementalflag) {
            if ($url ne $env{'form.suppurl'}) {
                 $env{'form.suppurl'} = $url;
            }
            if ($title ne $env{'form.title'}) {
                $env{'form.title'} = $title;
            }
            $env{'form.idx'} = $residx;
        } else {
            if ($symb ne $env{'form.symb'}) {
                $env{'form.symb'} = $symb;
            }
        }
    } else {
        $errormsg = &mt('Information about external resource to edit is missing.');
    }
    if ($updated) {
        $output = &Apache::lonhtmlcommon::confirm_success(&mt('External Resource updated'));
    }
    if ($errormsg) {
        $errormsg = '<p class="LC_error">'.$errormsg.'</p>';
    }
    my $js = &Apache::lonhtmlcommon::scripttag(&extedit_javascript());
    my $pathitem = '<input type="hidden" name="folderpath" value="'.
                   &HTML::Entities::encode($env{'form.folderpath'},'<>&"').'" />';
    $r->print(&Apache::loncommon::start_page('External Resource Editor',$js).
              '<div class="LC_left_float">'.
              $output.
              $errormsg.
              &extedit_form($supplementalflag,$residx,$url,$title,$pathitem,undef,
                           'direct',$env{'form.symb'}).
              '</div>'.&Apache::loncommon::end_page());
    return OK;
}

sub process_changes {
    my ($supplementalflag,$cdom,$cnum,$chome) = @_;
    my ($folder,$container,$output,$errormsg,$updated,$symb,$oldidx,$oldurl,
        $oldtitle,$newidx,$newurl,$newtitle,$residx,$url,$title);
    if ($env{'form.symb'}) {
        $symb = $env{'form.symb'};
        (my $map,$oldidx,$oldurl)=&Apache::lonnet::decode_symb($symb);
        if ($map =~ m{^uploaded/$cdom/$cnum/(default(_\d+|))\.(sequence|page)$}) {
            $folder = $1;
            $container = $3;
        }
        $oldtitle = &Apache::lonnet::gettitle($env{'form.symb'});
    } elsif ($env{'form.folderpath'}) {
        $folder = &unescape( (split('&',$env{'form.folderpath'}))[-2] );
        $oldurl = &unescape($env{'form.suppurl'});
        $oldtitle = &unescape($env{'form.title'});
        $container = 'sequence';
        $supplementalflag = 1;
    }
    if ($oldurl =~ m{^ext/(.+)$}) {
        my $external = $1; 
        if ($external =~ m{^https://}) {
            $oldurl = $external;
        } else {
            $oldurl = 'http://'.$oldurl;
        }
    }
    $url = $oldurl;
    $title = $oldtitle;
    if ($env{'form.importdetail'}) {
        ($newtitle,$newurl,$newidx) =
            map {&unescape($_)} split(/\=/,$env{'form.importdetail'});
    }
    if ($supplementalflag) {
        $residx = $newidx;
    } else {
        $residx = $oldidx;
    }
    if ($folder && $container) {
        if ($env{'form.importdetail'}) {
            my ($errtext,$fatal,$mismatchedid,@imports);
            if (!$supplementalflag) {
                if (($oldidx) && ($oldidx != $newidx)) {
                    $mismatchedid = 1;
                }
            }
            if ($mismatchedid) {
                $errormsg = 'Wrong item identifier';
            } elsif (($newtitle eq $oldtitle) && ($newurl eq $oldurl)) {
                $output = &mt('No change');
            } else {
                my $map = "/uploaded/$cdom/$cnum/$folder.$container";
                my ($errtext,$fatal) = &LONCAPA::map::mapread($map);
                if ($fatal) {
                    $errormsg = &mt('Update failed: [_1].',$errtext);
                } else {
                    my $saveurl = &LONCAPA::map::qtunescape($newurl);
                    my $savetitle = &LONCAPA::map::qtunescape($newtitle);
                    $LONCAPA::map::resources[$residx] =
                        join(':', ($savetitle,$saveurl,'true','normal','res'));
                    my ($outtext,$errtext) = &LONCAPA::map::storemap($map,1);
                    if ($errtext) {
                        $errormsg = &mt('Update failed: [_1].',$errtext);
                    } else {
                        $updated = 1;
                        $title = $newtitle;
                        if ($newurl ne $oldurl) {
                            $url = $newurl;
                            $newurl =~ s{^http://}{};
                            $newurl = "ext/$newurl";
                        }
                        if (!$supplementalflag) {
                            if ($newurl ne $oldurl) {
                                $symb = &Apache::lonnet::encode_symb($map,$residx,$newurl);
                            } else {
                                $symb = $env{'form.symb'};
                                if ($symb) {
                                    &Apache::lonnet::devalidate_title_cache($symb);
                                }
                            }
                        }
                        my ($furl,$ferr) = 
                            &Apache::lonuserstate::readmap("$cdom/$cnum");
                        if ($ferr) {
                            $errormsg = &mt('Reload failed: [_1].',$ferr);
                        } else {
                            unless ($supplementalflag) {
                                &Apache::loncommon::update_content_constraints($cdom,$cnum,$chome,$cdom.'_'.$cnum);
                            }
                        }
                    }
                }
            }
        } else {
            $output = &mt('No change');
        }
    } else {
        $errormsg = &mt('Information about current external resource is incomplete.');
    }
    return ($updated,$output,$errormsg,$residx,$url,$title,$symb);
}

sub extedit_form {
    my ($supplementalflag,$residx,$orig_url,$orig_title,$pathitem,$helpitem,$caller,$symb,$disabled) = @_;
    my %lt = &Apache::lonlocal::texthash(
        ex => 'External Resource',
        ed => 'Edit',
        ee => 'External Resource Editor',
        pr => 'Preview',
        sv => 'Save',
        ul => 'URL',
        ti => 'Title',
        al => 'Add Link',
    );
    my $formname = 'newext';
    my $tabid = 'aa';
    my $toggle = 'ext';
    my $fieldsetid = 'uploadextform';
    my $urlid = 'exturl';
    my $size = 60;
    if ($supplementalflag) {
        $formname = 'newsuppext';
        $tabid = 'ee';
        $toggle = 'suppext';
        $fieldsetid = 'uploadsuppextform';
        $urlid = 'suppexturl';
    }
    my ($link,$legend,$active,$srcclass,$extsrc,$preview,$title,$save,
        $fieldsetstyle,$action,$hiddenelem,$form);
    $fieldsetstyle = 'display: none;';
    $action = '/adm/coursedocs';
    my $protocol = ($ENV{'SERVER_PORT'} == 443?'https':'http');
    if ($residx) {
        if ($caller eq 'direct') {
            $fieldsetstyle = 'display: block;';
            $action = '/adm/extresedit';
            $legend = "<legend>$lt{'ee'}</legend>";
            if ($symb) {
                $hiddenelem = '<input type="hidden" name="symb" value="'.$symb.'" />';
            } elsif ($supplementalflag) {
                $hiddenelem = '<input type="hidden" name="suppurl" value="'.
                              &HTML::Entities::encode(&escape($orig_url),'<>&"').'" />'."\n".
                              '<input type="hidden" name="title" value="'.
                              &HTML::Entities::encode(&escape($orig_title),'<>&"').'" />';
            }
        } else {        
            $link = '<a class="LC_docs_ext_edit" href="javascript:editext('."'$residx'".');">'.$lt{'ed'}.'</a>&nbsp;'."\n";
            $size = 40;
            $active = '<input type="hidden" name="active" value="'.$tabid.'" />';
        }
        $formname = "editext_$residx";
        $fieldsetid = "uploadext$residx";
        $urlid = "exturl_$residx";
        $srcclass = ' class="LC_nobreak"';
        $extsrc = '<span class="LC_docs_ext_edit">'.$lt{'ul'}.'&nbsp;</span>';
        $preview = '&nbsp;<a class="LC_docs_ext_edit" href="javascript:extUrlPreview('."'$urlid','$protocol'".');">'.$lt{'pr'}.'</a>';
        $title = '<span class="LC_docs_ext_edit">'.$lt{'ti'}.'&nbsp;</span>';
        $save = $lt{'sv'};
    } else {
        $link = '<a class="LC_menubuttons_link" href="javascript:toggleUpload('."'$toggle'".');">'.$lt{'ex'}.'</a>'.$helpitem;
        $legend = "<legend>$lt{'ex'}</legend>";
        $extsrc = $lt{'ul'}.':<br />';
        $title = $lt{'ti'}.':<br />';
        $residx = 0;
        $orig_url = 'http://';
        $orig_title = $lt{'ex'};
        $preview = '<input type="button" name="view" value="'.$lt{'pr'}.'" onclick="javascript:extUrlPreview('."'$urlid','$protocol'".');"'.$disabled.' />';
        $save = $lt{'al'};
        $pathitem .= '<br />';
    }
    $form = <<ENDFORM;
<form action="$action" method="post" name="$formname">
<fieldset id="$fieldsetid" style="$fieldsetstyle">
$legend
$active
<span$srcclass>
$extsrc
<input type="text" size="$size" name="exturl" id="$urlid" value="$orig_url" $disabled />
$preview
</span> 
<br />
<span$srcclass>
$title
<input type="text" size="$size" name="exttitle" value="$orig_title" $disabled />
<input type="hidden" name="importdetail" value="" />
$pathitem
$hiddenelem
<input type="button" value="$save" onclick="javascript:setExternal(this.form,'$residx');" $disabled />
</span>
</fieldset>
</form>
ENDFORM
    if (wantarray) {
        return ($link,$form);
    } else {
        return $link.$form;
    }
}

sub display_editor {
    my ($url,$folderpath,$symb,$idx) = @_;
    my ($residx,$supplementalflag,$title,$pathitem,$output,$js,$navmap);
    if ($folderpath =~ /^supplemental/) {
        $supplementalflag = 1;
        $residx = $idx;
        $title = &unescape($env{'form.title'});
        $pathitem = '<input type="hidden" name="folderpath" value="'.&HTML::Entities::encode($folderpath,'<>&"').'" />';
    } elsif ($symb =~ /^uploaded/) {
        (my $map,$residx,my $res) =
            &Apache::lonnet::decode_symb($symb);
        $title = &Apache::lonnet::gettitle($symb);
        my $path = &Apache::loncommon::symb_to_docspath($symb,\$navmap);
        $pathitem = '<input type="hidden" name="folderpath" value="'.&HTML::Entities::encode($path,'<>&"').'" />';
    }
    $js = &Apache::lonhtmlcommon::scripttag(&extedit_javascript());
    my $args = { 'force_register' => $env{'form.register'} };
    return &Apache::loncommon::start_page('External Resource Editor',$js,$args).
           '<div class="LC_left_float">'.
           &extedit_form($supplementalflag,$residx,$url,$title,$pathitem,undef,'direct',$symb).
           '</div>'.
           &Apache::loncommon::end_page();
}

sub extedit_javascript {
    my %js_lt = &Apache::lonlocal::texthash(
        invurl  => 'Invalid URL',
        titbl   => 'Title is blank',
    );
    &js_escape(\%js_lt);

    my $urlregexp = <<'ENDREGEXP';
/^([a-z]([a-z]|\d|\+|-|\.)*):(\/\/(((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:)*@)?((\[(|(v[\da-f]{1,}\.(([a-z]|\d|-|\.|_|~)|[!\$&'\(\)\*\+,;=]|:)+))\])|((\d|[1-9]\d|1\d\d|2[0-4]\d|25[0-5])\.(\d|[1-9]\d|1\d\d|2[0-4]\d|25[0-5])\.(\d|[1-9]\d|1\d\d|2[0-4]\d|25[0-5])\.(\d|[1-9]\d|1\d\d|2[0-4]\d|25[0-5]))|(([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=])*)(:\d*)?)(\/(([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)*)*|(\/((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)+(\/(([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)*)*)?)|((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)+(\/(([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)*)*)|((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)){0})(\?((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)|[\uE000-\uF8FF]|\/|\?)*)?(\#((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)|\/|\?)*)?$/i
ENDREGEXP

    return <<ENDJS;

var regexp = $urlregexp;

function setExternal(extform,residx) {
    var title=extform.exttitle.value;
    if (!String.trim) {
        String.prototype.trim = function() {return this.replace(\/^\\s+|\\s+$\/g, "");};    }
    var url=extform.exturl.value;
    if (title == null || title.trim()=="") {
        alert("$js_lt{'titbl'}");
        extform.exttitle.focus();
        return;
    }
    if (regexp.test(url)) {
        url = escape(url);
        title = escape(title);
        if (residx > 0) {
            eval("extform.importdetail.value=title+'='+url+'='+residx;extform.submit();");
        } else {
            eval("extform.importdetail.value=title+'='+url;extform.submit();");
        }
    } else {
        alert("$js_lt{'invurl'}");
        extform.exturl.focus();
        return;
    }
}

function editext(residx) {
    if (document.getElementById('uploadext'+residx)) {
        var curr = document.getElementById('uploadext'+residx).style.display;
        if (curr == 'none') {
            disp = 'block';
        } else {
            disp = 'none';
        }
        document.getElementById('uploadext'+residx).style.display=disp;
    }
    resize_scrollbox('contentscroll','1','1');
    return;
}

function extUrlPreview(caller,protocol) {
    if (document.getElementById(caller)) {
        var url = document.getElementById(caller).value;
        if (regexp.test(url)) {
            var http_regex = /^http\:\/\//gi;
            if ((protocol == 'https') && (http_regex.test(url))) {
                window.open(url,"externalpreview","height=400,width=500,scrollbars=1,resizable=1,menubar=0,location=1");
            } else {
                openMyModal(url,500,400,'yes');
            }
        } else {
            alert("$js_lt{'invurl'}");
        }
    }
}

ENDJS

}

1;
