# The LearningOnline Network
# Login Screen
#
# $Id: lonlogin.pm,v 1.158.2.12 2021/01/04 03:49:10 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#

package Apache::lonlogin;

use strict;
use Apache::Constants qw(:common);
use Apache::File ();
use Apache::lonnet;
use Apache::loncommon();
use Apache::lonauth();
use Apache::lonlocal;
use Apache::migrateuser();
use lib '/home/httpd/lib/perl/';
use LONCAPA qw(:DEFAULT :match);
use CGI::Cookie();
 
sub handler {
    my $r = shift;

    &Apache::loncommon::get_unprocessed_cgi
	(join('&',$ENV{'QUERY_STRING'},$env{'request.querystring'},
	      $ENV{'REDIRECT_QUERY_STRING'}),
	 ['interface','username','domain','firsturl','localpath','localres',
	  'token','role','symb','iptoken','btoken']);
    if (!defined($env{'form.firsturl'})) {
        &Apache::lonacc::get_posted_cgi($r,['firsturl']);
    }

# -- check if they are a migrating user
    if (defined($env{'form.token'})) {
	return &Apache::migrateuser::handler($r);
    }

# For "public user" - remove any exising "public" cookie, as user really wants to log-in
    my ($handle,$lonidsdir,$expirepub,$userdom);
    $lonidsdir=$r->dir_config('lonIDsDir');
    unless ($r->header_only) {
        $handle = &Apache::lonnet::check_for_valid_session($r,'lonID',undef,\$userdom);
        if ($handle ne '') {
            if ($handle=~/^publicuser\_/) {
                unlink($r->dir_config('lonIDsDir')."/$handle.id");
                undef($handle);
                undef($userdom);
                $expirepub = 1;
            }
        }
    }

    &Apache::loncommon::no_cache($r);
    &Apache::lonlocal::get_language_handle($r);
    &Apache::loncommon::content_type($r,'text/html');
    if ($expirepub) {
        my $c = new CGI::Cookie(-name    => 'lonPubID',
                                -value   => '',
                                -expires => '-10y',);
        $r->header_out('Set-cookie' => $c);
    } elsif (($handle eq '') && ($userdom ne '')) {
        my %cookies=CGI::Cookie->parse($r->header_in('Cookie'));
        foreach my $name (keys(%cookies)) {
            next unless ($name =~ /^lon(|S|Link|Pub)ID$/);
            my $c = new CGI::Cookie(-name    => $name,
                                    -value   => '',
                                    -expires => '-10y',);
            $r->headers_out->add('Set-cookie' => $c);
        }
    }
    $r->send_http_header;
    return OK if $r->header_only;


# Are we re-routing?
    my $londocroot = $r->dir_config('lonDocRoot'); 
    if (-e "$londocroot/lon-status/reroute.txt") {
	&Apache::lonauth::reroute($r);
	return OK;
    }

    my $lonhost = $r->dir_config('lonHostID');
    $env{'form.firsturl'} =~ s/(`)/'/g;

# Check if browser sent a LON-CAPA load balancer cookie (and this is a balancer)

    my ($found_server,$balancer_cookie) = &Apache::lonnet::check_for_balancer_cookie($r,1);
    if ($found_server) {
        my $hostname = &Apache::lonnet::hostname($found_server);
        if ($hostname ne '') {
            my $protocol = $Apache::lonnet::protocol{$found_server};
            $protocol = 'http' if ($protocol ne 'https');
            my $dest = '/adm/roles';
            if ($env{'form.firsturl'} ne '') {
                $dest = $env{'form.firsturl'};
            }
            my %info = (
                         balcookie => $lonhost.':'.$balancer_cookie,
                       );
            my $balancer_token = &Apache::lonnet::tmpput(\%info,$found_server);
            if ($balancer_token) {
                $dest .=  (($dest=~/\?/)?'&;':'?') . 'btoken='.$balancer_token;
            }
            my $url = $protocol.'://'.$hostname.$dest;
            my $start_page =
                &Apache::loncommon::start_page('Switching Server ...',undef,
                                               {'redirect'       => [0,$url],});
            my $end_page   = &Apache::loncommon::end_page();
            $r->print($start_page.$end_page);
            return OK;
        }
    }

#
# Check if a LON-CAPA load balancer sent user here because user's browser sent
# it a balancer cookie for an active session on this server.
#

    my $balcookie;
    if ($env{'form.btoken'}) {
        my %info = &Apache::lonnet::tmpget($env{'form.btoken'});
        $balcookie = $info{'balcookie'};
        &Apache::lonnet::tmpdel($env{'form.btoken'});
        delete($env{'form.btoken'});
    }

#
# If browser sent an old cookie for which the session file had been removed
# check if configuration for user's domain has a portal URL set.  If so
# switch user's log-in to the portal.
#

    if (($handle eq '') && ($userdom ne '')) {
        my %domdefaults = &Apache::lonnet::get_domain_defaults($userdom);
        if ($domdefaults{'portal_def'} =~ /^https?\:/) {
            my $start_page = &Apache::loncommon::start_page('Switching Server ...',undef,
                                          {'redirect' => [0,$domdefaults{'portal_def'}],});
            my $end_page   = &Apache::loncommon::end_page();
            $r->print($start_page.$end_page);
            return OK;
        }
    }

    $env{'form.firsturl'} =~ s/(`)/'/g;

# -------------------------------- Prevent users from attempting to login twice
    if ($handle ne '') {
        &Apache::lonnet::transfer_profile_to_env($lonidsdir,$handle);
	my $start_page = 
	    &Apache::loncommon::start_page('Already logged in');
	my $end_page = 
	    &Apache::loncommon::end_page();
        my $dest = '/adm/roles';
        if ($env{'form.firsturl'} ne '') {
            $dest = $env{'form.firsturl'}; 
        }
	$r->print(
              $start_page
             .'<p class="LC_warning">'.&mt('You are already logged in!').'</p>'
             .'<p>'.&mt('Please either [_1]continue the current session[_2] or [_3]log out[_4].',
              '<a href="'.$dest.'">','</a>','<a href="/adm/logout">','</a>').'</p>'
             .$end_page
             );
        return OK;
    }

# ---------------------------------------------------- No valid token, continue

# ---------------------------- Not possible to really login to domain "public"
    if ($env{'form.domain'} eq 'public') {
	$env{'form.domain'}='';
	$env{'form.username'}='';
    }

# ------ Is this page requested because /adm/migrateuser detected an IP change?
    my %sessiondata;
    if ($env{'form.iptoken'}) {
        %sessiondata = &Apache::lonnet::tmpget($env{'form.iptoken'});
        unless ($sessiondata{'sessionserver'}) {
            my $delete = &Apache::lonnet::tmpdel($env{'form.iptoken'});
            delete($env{'form.iptoken'});
        }
    }
# ----------------------------------------------------------- Process Interface
    $env{'form.interface'}=~s/\W//g;

    (undef,undef,undef,undef,undef,undef,my $clientmobile) =
        &Apache::loncommon::decode_user_agent();

    my $iconpath= 
	&Apache::loncommon::lonhttpdurl($r->dir_config('lonIconsURL'));

    my $domain = &Apache::lonnet::default_login_domain();
    my $defdom = $domain;
    if ($lonhost ne '') {
        unless ($sessiondata{'sessionserver'}) {
            my $redirect = &check_loginvia($domain,$lonhost,$lonidsdir,$balcookie);
            if ($redirect) {
                $r->print($redirect);
                return OK;
            }
        }
    }

    if (($sessiondata{'domain'}) &&
        (&Apache::lonnet::domain($sessiondata{'domain'},'description'))) {
        $domain=$sessiondata{'domain'};
    } elsif (($env{'form.domain'}) && 
	(&Apache::lonnet::domain($env{'form.domain'},'description'))) {
	$domain=$env{'form.domain'};
    }

    my $role    = $r->dir_config('lonRole');
    my $loadlim = $r->dir_config('lonLoadLim');
    my $uloadlim= $r->dir_config('lonUserLoadLim');
    my $servadm = $r->dir_config('lonAdmEMail');
    my $tabdir  = $r->dir_config('lonTabDir');
    my $include = $r->dir_config('lonIncludes');
    my $expire  = $r->dir_config('lonExpire');
    my $version = $r->dir_config('lonVersion');
    my $host_name = &Apache::lonnet::hostname($lonhost);

# --------------------------------------------- Default values for login fields
    
    my ($authusername,$authdomain);
    if ($sessiondata{'username'}) {
        $authusername=$sessiondata{'username'};
    } else {
        $env{'form.username'} = &Apache::loncommon::cleanup_html($env{'form.username'});
        $authusername=($env{'form.username'}?$env{'form.username'}:'');
    }
    if ($sessiondata{'domain'}) {
        $authdomain=$sessiondata{'domain'};
    } else {
        $env{'form.domain'} = &Apache::loncommon::cleanup_html($env{'form.domain'});
        $authdomain=($env{'form.domain'}?$env{'form.domain'}:$domain);
    }

# ---------------------------------------------------------- Determine own load
    my $loadavg;
    {
	my $loadfile=Apache::File->new('/proc/loadavg');
	$loadavg=<$loadfile>;
    }
    $loadavg =~ s/\s.*//g;

    my ($loadpercent,$userloadpercent);
    if ($loadlim) {
        $loadpercent=sprintf("%.1f",100*$loadavg/$loadlim);
    }
    if ($uloadlim) {
        $userloadpercent=&Apache::lonnet::userload();
    }

    my $firsturl=
    ($env{'request.firsturl'}?$env{'request.firsturl'}:$env{'form.firsturl'});

# ----------------------------------------------------------- Get announcements
    my $announcements=&Apache::lonnet::getannounce();
# -------------------------------------------------------- Set login parameters

    my @hexstr=('0','1','2','3','4','5','6','7',
                '8','9','a','b','c','d','e','f');
    my $lkey='';
    for (0..7) {
        $lkey.=$hexstr[rand(15)];
    }

    my $ukey='';
    for (0..7) {
        $ukey.=$hexstr[rand(15)];
    }

    my $lextkey=hex($lkey);
    if ($lextkey>2147483647) { $lextkey-=4294967296; }

    my $uextkey=hex($ukey);
    if ($uextkey>2147483647) { $uextkey-=4294967296; }

# -------------------------------------------------------- Store away log token
    my $tokenextras;
    if ($env{'form.role'}) {
        $tokenextras = '&role='.&escape($env{'form.role'});
    }
    if ($env{'form.symb'}) {
        if (!$tokenextras) {
            $tokenextras = '&';
        }
        $tokenextras .= '&symb='.&escape($env{'form.symb'});
    }
    if ($env{'form.iptoken'}) {
        if (!$tokenextras) {
            $tokenextras = '&&';
        }
        $tokenextras .= '&iptoken='.&escape($env{'form.iptoken'});
    }
    my $logtoken=Apache::lonnet::reply(
       'tmpput:'.$ukey.$lkey.'&'.$firsturl.$tokenextras,
       $lonhost);

# -- If we cannot talk to ourselves, or hostID does not map to a hostname
#    we are in serious trouble

    if (($logtoken eq 'con_lost') || ($logtoken eq 'no_such_host')) {
        if ($logtoken eq 'no_such_host') {
            &Apache::lonnet::logthis('No valid logtoken for log-in page -- unable to determine hostname for hostID: '.$lonhost.'. Check entry in hosts.tab');
        }
        my $spares='';
        my (@sparehosts,%spareservers);
        my $sparesref = &Apache::lonnet::this_host_spares($defdom);
        if (ref($sparesref) eq 'HASH') {
            foreach my $key (keys(%{$sparesref})) {
                if (ref($sparesref->{$key}) eq 'ARRAY') {
                    my @sorted = sort { &Apache::lonnet::hostname($a) cmp
                                        &Apache::lonnet::hostname($b);
                                      } @{$sparesref->{$key}};
                    if (@sorted) {
                        if ($key eq 'primary') {
                            unshift(@sparehosts,@sorted);
                        } elsif ($key eq 'default') {
                            push(@sparehosts,@sorted);
                        }
                    }
                }
            }
        }
        foreach my $hostid (@sparehosts) {
            next if ($hostid eq $lonhost);
	    my $hostname = &Apache::lonnet::hostname($hostid);
	    next if (($hostname eq '') || ($spareservers{$hostname}));
            $spareservers{$hostname} = 1;
            my $protocol = $Apache::lonnet::protocol{$hostid};
            $protocol = 'http' if ($protocol ne 'https');
            $spares.='<br /><span style="font-size: larger;"><a href="'.$protocol.'://'.
                $hostname.
                '/adm/login?domain='.$authdomain.'">'.
                $hostname.'</a>'.
                ' '.&mt('(preferred)').'</span>'.$/;
        }
        if ($spares) {
            $spares.= '<br />';
        }
        my %all_hostnames = &Apache::lonnet::all_hostnames();
        foreach my $hostid (sort
		    {
			&Apache::lonnet::hostname($a) cmp
			    &Apache::lonnet::hostname($b);
		    }
		    keys(%all_hostnames)) {
            next if ($hostid eq $lonhost);
            my $hostname = &Apache::lonnet::hostname($hostid);
            next if (($hostname eq '') || ($spareservers{$hostname}));
            $spareservers{$hostname} = 1;
            my $protocol = $Apache::lonnet::protocol{$hostid};
            $protocol = 'http' if ($protocol ne 'https');
            $spares.='<br /><a href="'.$protocol.'://'.
	             $hostname.
	             '/adm/login?domain='.$authdomain.'">'.
	             $hostname.'</a>';
         }
         $r->print(
   '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">'
  .'<html xmlns="http://www.w3.org/1999/xhtml" xml:lang="en" lang="en">'
  .'<head><meta http-equiv="Content-Type" content="text/html; charset=utf-8" /><title>'
  .&mt('The LearningOnline Network with CAPA')
  .'</title></head>'
  .'<body bgcolor="#FFFFFF">'
  .'<h1>'.&mt('The LearningOnline Network with CAPA').'</h1>'
  .'<img src="/adm/lonKaputt/lonlogo_broken.gif" alt="broken icon" align="right" />'
  .'<h3>'.&mt('This LON-CAPA server is temporarily not available for login.').'</h3>');
        if ($spares) {
            $r->print('<p>'.&mt('Please attempt to login to one of the following servers:')
                     .'</p>'
                     .$spares);
        }
        $r->print('</body>'
                 .'</html>'
        );
        return OK;
    }

# ----------------------------------------------- Apparently we are in business
    $servadm=~s/\,/\<br \/\>/g;

# ----------------------------------------------------------- Front page design
    my $pgbg=&Apache::loncommon::designparm('login.pgbg',$domain);
    my $font=&Apache::loncommon::designparm('login.font',$domain);
    my $link=&Apache::loncommon::designparm('login.link',$domain);
    my $vlink=&Apache::loncommon::designparm('login.vlink',$domain);
    my $alink=&Apache::loncommon::designparm('login.alink',$domain);
    my $mainbg=&Apache::loncommon::designparm('login.mainbg',$domain);
    my $loginbox_bg=&Apache::loncommon::designparm('login.sidebg',$domain);
    my $loginbox_header_bgcol=&Apache::loncommon::designparm('login.bgcol',$domain);
    my $loginbox_header_textcol=&Apache::loncommon::designparm('login.textcol',$domain);
    my $logo=&Apache::loncommon::designparm('login.logo',$domain);
    my $img=&Apache::loncommon::designparm('login.img',$domain);
    my $domainlogo=&Apache::loncommon::domainlogo($domain);
    my $showbanner = 1;
    my $showmainlogo = 1;
    if (defined(&Apache::loncommon::designparm('login.showlogo_img',$domain))) {
        $showbanner = &Apache::loncommon::designparm('login.showlogo_img',$domain);
    }
    if (defined(&Apache::loncommon::designparm('login.showlogo_logo',$domain))) {
        $showmainlogo = &Apache::loncommon::designparm('login.showlogo_logo',$domain);
    }
    my $showadminmail;
    my @possdoms = &Apache::lonnet::current_machine_domains();
    if (grep(/^\Q$domain\E$/,@possdoms)) {
        $showadminmail=&Apache::loncommon::designparm('login.adminmail',$domain);
    }
    my $showcoursecat =
        &Apache::loncommon::designparm('login.coursecatalog',$domain);
    my $shownewuserlink = 
        &Apache::loncommon::designparm('login.newuser',$domain);
    my $showhelpdesk =
        &Apache::loncommon::designparm('login.helpdesk',$domain);
    my $now=time;
    my $js = (<<ENDSCRIPT);

<script type="text/javascript" language="JavaScript">
// <![CDATA[
function send()
{
this.document.server.elements.uname.value
=this.document.client.elements.uname.value;

this.document.server.elements.udom.value
=this.document.client.elements.udom.value;

uextkey=this.document.client.elements.uextkey.value;
lextkey=this.document.client.elements.lextkey.value;
initkeys();

this.document.server.elements.upass0.value
    =getCrypted(this.document.client.elements.upass$now.value);

this.document.client.elements.uname.value='';
this.document.client.elements.upass$now.value='';

this.document.server.submit();
return false;
}

function enableInput() {
    this.document.client.elements.upass$now.removeAttribute("readOnly");
    this.document.client.elements.uname.removeAttribute("readOnly");
    this.document.client.elements.udom.removeAttribute("readOnly");
    return;
}

// ]]>
</script>

ENDSCRIPT

# --------------------------------------------------- Print login screen header

    my %add_entries = (
	       bgcolor      => "$mainbg",
	       text         => "$font",
	       link         => "$link",
	       vlink        => "$vlink",
	       alink        => "$alink",
               onload       => 'javascript:enableInput();',);

    my ($lonhost_in_use,$headextra,$headextra_exempt,@hosts,%defaultdomconf);
    @hosts = &Apache::lonnet::current_machine_ids();
    $lonhost_in_use = $lonhost;
    if (@hosts > 1) {
        foreach my $hostid (@hosts) {
            if (&Apache::lonnet::host_domain($hostid) eq $defdom) {
                $lonhost_in_use = $hostid;
                last;
            }
        }
    }
    %defaultdomconf = &Apache::loncommon::get_domainconf($defdom);
    $headextra = $defaultdomconf{$defdom.'.login.headtag_'.$lonhost_in_use};
    $headextra_exempt = $defaultdomconf{$domain.'.login.headtag_exempt_'.$lonhost_in_use};
    if ($headextra) {
        my $omitextra;
        if ($headextra_exempt ne '') {
            my @exempt = split(',',$headextra_exempt);
            my $ip = &Apache::lonnet::get_requestor_ip();
            if (grep(/^\Q$ip\E$/,@exempt)) {
                $omitextra = 1;
            }
        }
        unless ($omitextra) {
            my $confname = $defdom.'-domainconfig';
            if ($headextra =~ m{^\Q/res/$defdom/$confname/login/headtag/$lonhost_in_use/\E}) {
                my $extra = &Apache::lonnet::getfile(&Apache::lonnet::filelocation("",$headextra));
                unless ($extra eq '-1') {
                    $js .= "\n".$extra."\n";
                }
            }
        }
    }

    $r->print(&Apache::loncommon::start_page('The LearningOnline Network with CAPA Login',$js,
			       { 'redirect'       => [$expire,'/adm/roles'], 
				 'add_entries' => \%add_entries,
				 'only_body'   => 1,}));

# ----------------------------------------------------------------------- Texts

    my %lt=&Apache::lonlocal::texthash(
          'un'       => 'Username',
          'pw'       => 'Password',
          'dom'      => 'Domain',
          'perc'     => 'percent',
          'load'     => 'Server Load',
          'userload' => 'User Load',
          'catalog'  => 'Course/Community Catalog',
          'log'      => 'Log in',
          'help'     => 'Log-in Help',
          'serv'     => 'Server',
          'servadm'  => 'Server Administration',
          'helpdesk' => 'Contact Helpdesk',
          'forgotpw' => 'Forgot password?',
          'newuser'  => 'New User?',
       );
# -------------------------------------------------- Change password field name

    my $forgotpw = &forgotpwdisplay(%lt);
    $forgotpw .= '<br />' if $forgotpw;
    my $loginhelp = &Apache::lonauth::loginhelpdisplay($authdomain);
    if ($loginhelp) {
        $loginhelp = '<a href="'.$loginhelp.'">'.$lt{'help'}.'</a><br />';
    }

# ---------------------------------------------------- Serve out DES JavaScript
    {
    my $jsh=Apache::File->new($include."/londes.js");
    $r->print(<$jsh>);
    }
# ---------------------------------------------------------- Serve rest of page

    $r->print(
    '<div class="LC_Box"'
   .' style="margin:0 auto; padding:10px; width:90%; height: auto; background-color:#FFFFFF;">'
);

    $r->print(<<ENDSERVERFORM);
<form name="server" action="/adm/authenticate" method="post" target="_top">
   <input type="hidden" name="logtoken" value="$logtoken" />
   <input type="hidden" name="serverid" value="$lonhost" />
   <input type="hidden" name="uname" value="" />
   <input type="hidden" name="upass0" value="" />
   <input type="hidden" name="udom" value="" />
   <input type="hidden" name="localpath" value="$env{'form.localpath'}" />
   <input type="hidden" name="localres" value="$env{'form.localres'}" />
  </form>
ENDSERVERFORM
    my $coursecatalog;
    if (($showcoursecat eq '') || ($showcoursecat)) {
        $coursecatalog = &coursecatalog_link($lt{'catalog'}).'<br />';
    }
    my $newuserlink;
    if ($shownewuserlink) {
        $newuserlink = &newuser_link($lt{'newuser'}).'<br />';
    }
    my $logintitle =
        '<h2 class="LC_hcell"'
       .' style="background:'.$loginbox_header_bgcol.';'
       .' color:'.$loginbox_header_textcol.'">'
       .$lt{'log'}
       .'</h2>';

    my $noscript_warning='<noscript><span class="LC_warning"><b>'
                        .&mt('Use of LON-CAPA requires Javascript to be enabled in your web browser.')
                        .'</b></span></noscript>';
    my $helpdeskscript;
    my $contactblock = &contactdisplay(\%lt,$servadm,$showadminmail,
                                       $authdomain,\$helpdeskscript,
                                       $showhelpdesk,\@possdoms);

    my $mobileargs;
    if ($clientmobile) {
        $mobileargs = 'autocapitalize="off" autocorrect="off"'; 
    }
    my $loginform=(<<LFORM);
<form name="client" action="" onsubmit="return(send())">
  <input type="hidden" name="lextkey" value="$lextkey" />
  <input type="hidden" name="uextkey" value="$uextkey" />
  <b><label for="uname">$lt{'un'}</label>:</b><br />
  <input type="text" name="uname" id="uname" size="15" value="$authusername" readonly="readonly" $mobileargs /><br />
  <b><label for="upass$now">$lt{'pw'}</label>:</b><br />
  <input type="password" name="upass$now" id="upass$now" size="15" readonly="readonly" /><br />
  <b><label for="udom">$lt{'dom'}</label>:</b><br />
  <input type="text" name="udom" id="udom" size="15" value="$authdomain" readonly="readonly" $mobileargs /><br />
  <input type="submit" value="$lt{'log'}" />
</form>
LFORM

    if ($showbanner) {
        $r->print(<<HEADER);
<!-- The LON-CAPA Header -->
<div style="background:$pgbg;margin:0;width:100%;">
  <img src="$img" border="0" alt="The Learning Online Network with CAPA" class="LC_maxwidth" />
</div>
HEADER
    }
    $r->print(<<ENDTOP);
<div style="float:left;margin-top:0;">
<div class="LC_Box" style="background:$loginbox_bg;">
  $logintitle
  $loginform
  $noscript_warning
</div>
  
<div class="LC_Box" style="padding-top: 10px;">
  $loginhelp
  $forgotpw
  $contactblock
  $newuserlink
  $coursecatalog
</div>
</div>

<div>
ENDTOP
    if ($showmainlogo) {
        $r->print(' <img src="'.$logo.'" alt="" class="LC_maxwidth" />'."\n");
    }
$r->print(<<ENDTOP);
$announcements
</div>
<hr style="clear:both;" />
ENDTOP
    my ($domainrow,$serverrow,$loadrow,$userloadrow,$versionrow);
    $domainrow = <<"END";
      <tr>
       <td  align="left" valign="top">
        <small><b>$lt{'dom'}:&nbsp;</b></small>
       </td>
       <td  align="left" valign="top">
        <small><tt>&nbsp;$domain</tt></small>
       </td>
      </tr>
END
    $serverrow = <<"END";
      <tr>
       <td  align="left" valign="top">
        <small><b>$lt{'serv'}:&nbsp;</b></small>
       </td>
       <td align="left" valign="top">
        <small><tt>&nbsp;$lonhost ($role)</tt></small>
       </td>
      </tr>
END
    if ($loadlim) {
        $loadrow = <<"END";
      <tr>
       <td align="left" valign="top">
        <small><b>$lt{'load'}:&nbsp;</b></small>
       </td>
       <td align="left" valign="top">
        <small><tt>&nbsp;$loadpercent $lt{'perc'}</tt></small>
       </td>
      </tr>
END
    }
    if ($uloadlim) {
        $userloadrow = <<"END";
      <tr>
       <td align="left" valign="top">
        <small><b>$lt{'userload'}:&nbsp;</b></small>
       </td>
       <td align="left" valign="top">
        <small><tt>&nbsp;$userloadpercent $lt{'perc'}</tt></small>
       </td>
      </tr>
END
    }
    if (($version ne '') && ($version ne '<!-- VERSION -->')) {
        $versionrow = <<"END";
      <tr>
       <td colspan="2" align="left">
        <small>$version</small>
       </td>
      </tr>
END
    }

    $r->print(<<ENDDOCUMENT);
    <div style="float: left;">
     <table border="0" cellspacing="0" cellpadding="0">
$domainrow
$serverrow
$loadrow    
$userloadrow
$versionrow
     </table>
    </div>
    <div style="float: right;">
    $domainlogo
    </div>
    <br style="clear:both;" />
 </div>

<script type="text/javascript">
// <![CDATA[
// the if prevents the script error if the browser can not handle this
if ( document.client.uname ) { document.client.uname.focus(); }
// ]]>
</script>
$helpdeskscript

ENDDOCUMENT
    my %endargs = ( 'noredirectlink' => 1, );
    $r->print(&Apache::loncommon::end_page(\%endargs));
    return OK;
}

sub check_loginvia {
    my ($domain,$lonhost,$lonidsdir,$balcookie) = @_;
    if ($domain eq '' || $lonhost eq '' || $lonidsdir eq '') {
        return;
    }
    my %domconfhash = &Apache::loncommon::get_domainconf($domain);
    my $loginvia = $domconfhash{$domain.'.login.loginvia_'.$lonhost};
    my $loginvia_exempt = $domconfhash{$domain.'.login.loginvia_exempt_'.$lonhost};
    my $output;
    if ($loginvia ne '') {
        my $noredirect;
        my $ip = &Apache::lonnet::get_requestor_ip();
        if ($ip eq '127.0.0.1') {
            $noredirect = 1;
        } else {
            if ($loginvia_exempt ne '') {
                my @exempt = split(',',$loginvia_exempt);
                if (grep(/^\Q$ip\E$/,@exempt)) {
                    $noredirect = 1;
                }
            }
        }
        unless ($noredirect) {
            my ($newhost,$path);
            if ($loginvia =~ /:/) {
                ($newhost,$path) = split(':',$loginvia);
            } else {
                $newhost = $loginvia;
            }
            if ($newhost ne $lonhost) {
                if (&Apache::lonnet::hostname($newhost) ne '') {
                    if ($balcookie) {
                        my ($balancer,$cookie) = split(/:/,$balcookie);
                        if ($cookie =~ /^($match_domain)_($match_username)_([a-f0-9]+)$/) {
                            my ($udom,$uname,$cookieid) = ($1,$2,$3);
                            unless (&Apache::lonnet::delbalcookie($cookie,$balancer) eq 'ok') {
                                if ((-d $lonidsdir) && (opendir(my $dh,$lonidsdir))) {
                                    while (my $filename=readdir($dh)) {
                                        if ($filename=~/^(\Q$uname\E_\d+_\Q$udom\E_$match_lonid)\.id$/) {
                                            my $handle = $1;
                                            my %hash =
                                                &Apache::lonnet::get_sessionfile_vars($handle,$lonidsdir,
                                                                                     ['request.balancercookie',
                                                                                      'user.linkedenv']);
                                            if ($hash{'request.balancercookie'} eq "$balancer:$cookieid") {
                                                if (unlink("$lonidsdir/$filename")) {
                                                    if (($hash{'user.linkedenv'} =~ /^[a-f0-9]+_linked$/) &&
                                                        (-l "$lonidsdir/$hash{'user.linkedenv'}.id") &&
                                                        (readlink("$lonidsdir/$hash{'user.linkedenv'}.id") eq "$lonidsdir/$filename")) {
                                                        unlink("$lonidsdir/$hash{'user.linkedenv'}.id");
                                                    }
                                                }
                                            }
                                            last;
                                        }
                                    }
                                    closedir($dh);
                                }
                            }
                        }
                    }
                    $output = &redirect_page($newhost,$path);
                }
            }
        }
    }
    return $output;
}

sub redirect_page {
    my ($desthost,$path) = @_;
    my $hostname = &Apache::lonnet::hostname($desthost);
    my $protocol = $Apache::lonnet::protocol{$desthost};
    $protocol = 'http' if ($protocol ne 'https');
    unless ($path =~ m{^/}) {
        $path = '/'.$path;
    }
    my $url = $protocol.'://'.$hostname.$path;
    if ($env{'form.firsturl'} ne '') {
        $url .='?firsturl='.$env{'form.firsturl'};
    }
    my $start_page = &Apache::loncommon::start_page('Switching Server ...',undef,
                                                    {'redirect' => [0,$url],});
    my $end_page   = &Apache::loncommon::end_page();
    return $start_page.$end_page;
}

sub contactdisplay {
    my ($lt,$servadm,$showadminmail,$authdomain,$helpdeskscript,$showhelpdesk,
        $possdoms) = @_;
    my $contactblock;
    my $origmail;
    if (ref($possdoms) eq 'ARRAY') {
        if (grep(/^\Q$authdomain\E$/,@{$possdoms})) { 
            $origmail = $Apache::lonnet::perlvar{'lonSupportEMail'};
        }
    }
    my $requestmail = 
        &Apache::loncommon::build_recipient_list(undef,'helpdeskmail',
                                                 $authdomain,$origmail);
    unless ($showhelpdesk eq '0') {
        if ($requestmail =~ m/[^\@]+\@[^\@]+/) {
            $showhelpdesk = 1;
        } else {
            $showhelpdesk = 0;
        }
    }
    if ($servadm && $showadminmail) {
        $contactblock .= $$lt{'servadm'}.':<br />'.
                         '<tt>'.$servadm.'</tt><br />';
    }
    if ($showhelpdesk) {
        $contactblock .= '<a href="javascript:helpdesk()">'.$lt->{'helpdesk'}.'</a><br />';
        my $thisurl = &escape('/adm/login');
        $$helpdeskscript = <<"ENDSCRIPT";
<script type="text/javascript">
// <![CDATA[
function helpdesk() {
    var possdom = document.client.udom.value;
    var codedom = possdom.replace( new RegExp("[^A-Za-z0-9.\\-]","g"),'');
    if (codedom == '') {
        codedom = "$authdomain";
    }
    var querystr = "origurl=$thisurl&codedom="+codedom;
    document.location.href = "/adm/helpdesk?"+querystr;
    return;
}
// ]]>
</script>
ENDSCRIPT
    }
    return $contactblock;
}

sub forgotpwdisplay {
    my (%lt) = @_;
    my $prompt_for_resetpw = 1; 
    if ($prompt_for_resetpw) {
        return '<a href="/adm/resetpw">'.$lt{'forgotpw'}.'</a>';
    }
    return;
}

sub coursecatalog_link {
    my ($linkname) = @_;
    return <<"END";
      <a href="/adm/coursecatalog">$linkname</a>
END
}

sub newuser_link {
    my ($linkname) = @_;
    return '<a href="/adm/createaccount">'.$linkname.'</a>';
}

1;
__END__
