#
# LON-CAPA helpers for cgi-bin scripts
#
# $Id: loncgi.pm,v 1.16 2018/07/04 16:58:26 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
#############################################
#############################################

=pod

=head1 NAME

loncgi

=head1 SYNOPSIS

Provides subroutines for checking a LON-CAPA cookie, loading the user's
environment, and retrieving arguments passed in via a CGI's Query String.

=head1 Subroutines

=over 4 

=cut

#############################################
#############################################
package LONCAPA::loncgi;

use strict;
use warnings FATAL=>'all';
no warnings 'uninitialized';

use lib '/home/httpd/lib/perl/';
use CGI qw(:standard);
use CGI::Cookie();
use MIME::Types();
use Fcntl qw(:flock);
use LONCAPA;
use LONCAPA::Configuration();
use GDBM_File;
use Apache::lonlocal;

my $lonidsdir;

BEGIN {
    my $perlvar=LONCAPA::Configuration::read_conf('loncapa.conf');
    delete $perlvar->{'lonReceipt'};
    $lonidsdir = $perlvar->{'lonIDsDir'};
}


#############################################
#############################################

=pod

=item check_cookie_and_load_env()

Inputs: 1 ( optional). When called from a handler in mod_perl,
        pass in the request object.

Returns: 1 if the user has a LON-CAPA cookie, 0 if not.
Side effect: Loads the user's environment into the %env hash
             if the cookie is correct.

=cut

#############################################
#############################################
sub check_cookie_and_load_env {
    my ($r) = @_;
    my ($hascookie,$handle) = &check_cookie($r);
    if (($hascookie) && ($handle)) {
        &transfer_profile_to_env($handle);
    }
    return $hascookie;
}

#############################################
#############################################

=pod

=item check_cookie()

Inputs: none

Array context:
Returns: (1,$handle) if the user has a LON-CAPA cookie;
(0) if user does not have a LON-CAPA cookie.

Scalar context:
Returns: 1 if the user has a LON-CAPA cookie and 0 if not.

=cut

#############################################
#############################################
sub check_cookie {
    my ($r) = @_;
    my %cookies;
    if (ref($r)) {
        %cookies = CGI::Cookie->fetch($r);
    } else {
        %cookies = CGI::Cookie->fetch();
    }
    if (keys(%cookies)) {
        my $name = 'lonID';
        my $secure = 'lonSID';
        my $linkname = 'lonLinkID';
        my $pubname = 'lonPubID';
        my $lonid;
        if (exists($cookies{$secure})) {
            $lonid=$cookies{$secure};
        } elsif (exists($cookies{$name})) {
            $lonid=$cookies{$name};
        } elsif (exists($cookies{$linkname})) {
            $lonid=$cookies{$linkname};
        } elsif (exists($cookies{$pubname})) {
            $lonid=$cookies{$pubname};
        }
        if ($lonid) {
            my $handle=&LONCAPA::clean_handle($lonid->value);
            if ($handle) {
                if (-l "$lonidsdir/$handle.id") {
                    my $link = readlink("$lonidsdir/$handle.id");
                    if ((-e $link) && ($link =~ m{^\Q$lonidsdir\E/(.+)\.id$})) {
                        $handle = $1;
                    }
                }
                if (-e "$lonidsdir/".$handle.".id") {
                    # valid cookie found
                    if (wantarray) {
                        return (1,$handle);
                    } else {
                        return 1;
                    }
                }
            }
        }
    }
    # No valid cookie found
    if (wantarray) {
        return (0);
    } else {
        return 0;
    }
}

#############################################
#############################################

=pod

=item transfer_profile_to_env()

Load the users environment into the %env hash.

Inputs: $handle, the name of the users LON-CAPA cookie.

Returns: undef

=cut

#############################################
#############################################
sub transfer_profile_to_env {
    my ($handle)=@_;
    if (tie(my %disk_env,'GDBM_File',"$lonidsdir/$handle.id",&GDBM_READER(),
	    0640)) {
	%Apache::lonnet::env = %disk_env;
	untie(%disk_env);
    }
    $Apache::lonnet::env{'user.environment'} = "$lonidsdir/$handle.id";
    return undef;
}

#############################################
#############################################

=pod

=item missing_cookie_msg()

Inputs: none
Returns: HTML for a page indicating cookie information absent. 

=cut

#############################################
#############################################
sub missing_cookie_msg {
    my %lt = &Apache::lonlocal::texthash (
                        cook => 'Bad Cookie',
                        your => 'Your cookie information is incorrect.',
             );
    return <<END;
<html>
<head><title>$lt{'cook'}</title></head>
<body>
$lt{'your'}
</body>
</html>
END

}

#############################################
#############################################

=pod

=cgi_getitems()

Inputs: $query - the CGI query string (required)
        $getitems - reference to a hash (required)
        $possname - permitted names of keys (optional)

Returns: nothing

Side Effects: populates $getitems hash ref with key => value
              where each key is the name of the form item in the query string
              and value is an array of corresponding values. 

=cut

#############################################
#############################################
sub cgi_getitems {
    my ($query,$getitems,$possnames)= @_;
    foreach (split(/&/,$query)) {
        my ($name, $value) = split(/=/,$_);
        $name = &unescape($name);
        if (ref($possnames) eq 'ARRAY') {
            next unless (grep(/^\Q$name\E$/,@{$possnames}));
        }
        $value =~ tr/+/ /;
        $value =~ s/%([a-fA-F0-9][a-fA-F0-9])/pack("C",hex($1))/eg;
        push(@{$$getitems{$name}},$value);
    }
    return;
}

#############################################
#############################################

=pod

=cgi_header()

Inputs: $contenttype - Content Type (e.g., text/html or text/plain)
        $nocache     - Boolean 1 = nocache
Returns: HTTP Response headers constructed using CGI.pm

=cut

#############################################
#############################################
sub cgi_header {
    my ($contenttype,$nocache) = @_;
    my $mimetypes = MIME::Types->new;
    my %headers;
    if ($contenttype ne '') {
        if ($mimetypes->type($contenttype) ne '') {
            $headers{'-type'} = $contenttype;
            if ($contenttype =~ m{^text/}) {
                $headers{'-charset'} = 'utf-8';
            }
        }
    }
    if ($nocache) {
       $headers{'-expires'} = 'now';
    }
    if (%headers) {
        return CGI::header(%headers);
    }
    return;
}

=pod

=back

=cut

1;

__END__
