# The LearningOnline Network with CAPA
# essay (ungraded) style responses
#
# $Id: essayresponse.pm,v 1.118.2.1 2019/02/19 14:24:38 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#

package Apache::essayresponse;
use strict;
use Apache::lonxml();
use Apache::lonhtmlcommon;
use Apache::loncommon;
use Apache::lonnet;
use Apache::lonnavmaps;
use Apache::lonlocal;
use LONCAPA qw(:DEFAULT :match);
 

BEGIN {
    &Apache::lonxml::register('Apache::essayresponse',('essayresponse'));
}

sub start_essayresponse {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    my $id = &Apache::response::start_response($parstack,$safeeval);
    if ($target eq 'meta') {
	$result=&Apache::response::meta_package_write('essayresponse');
    } elsif ($target eq 'web' &&
	     $Apache::inputtags::status[-1] eq 'CAN_ANSWER') {
	my $part= $Apache::inputtags::part;
	my $ncol= &Apache::lonnet::EXT("resource.$part".'_'."$id.maxcollaborators");
	my $coll= &HTML::Entities::encode($Apache::lonhomework::history{"resource.$part.$id.collaborators"},'<>&"');
	my $uploadedfiletypes= &Apache::lonnet::EXT("resource.$part".'_'."$id.uploadedfiletypes");
        $uploadedfiletypes=~s/[^\w\,]//g;
        my $maxfilesize=&Apache::lonnet::EXT("resource.$part".'_'."$id.maxfilesize");
        if (!defined($maxfilesize)) {
            $maxfilesize = 10.0; #FIXME This should become a domain configuration 
        }
        my $hiddendraft;
	if (($Apache::lonhomework::type eq 'survey') ||
            ($Apache::lonhomework::type eq 'surveycred') ||
            ($Apache::lonhomework::type eq 'anonsurvey') ||
            ($Apache::lonhomework::type eq 'anonsurveycred')) {
            $hiddendraft = '<input type="hidden" name="HWDRAFT'.$part.'_'.$id.'" value="yes" />';
        } else {
            my $status_text = &mt('Submission type');
            if ($Apache::lonhomework::history{"resource.$part.award"} eq 'DRAFT') {
                $status_text .= '<br />'.&mt('(Currently -- draft)');
            }
            $result = &Apache::lonhtmlcommon::row_title($status_text);
            my $closure;
            unless ($ncol || $uploadedfiletypes) {
                $closure = 1;
            }
	    $result.= 
		'<label>'.
		'<input type="radio" name="HWDRAFT'.$part.'_'.$id.'" value="yes" checked="checked" />&nbsp;'.
		&mt('Submit entries below as answer to receive credit').
		'</label> <br />'.
		'<label>'.
		'<input type="radio" name="HWDRAFT'.$part.'_'.$id.'" value="no" />&nbsp;'.
		&mt('Save entries below (not submitted for credit yet)').
		'</label>'.
                &Apache::lonhtmlcommon::row_closure($closure);
	}

	if ($ncol > 0) {
	    $result.= &Apache::lonhtmlcommon::row_title(&mt('Collaborators')).
                      '<label>'.
		      &mt('Collaborators:').' <input type="text" size="70" max="80" name="HWCOL'.
		      $part.'_'.$id.'" value="'.$coll.'" /><br />'.
		      &mt('Enter a maximum of [quant,_1,collaborator] using username or username:domain, e.g. smithje or smithje:[_2].',$ncol,$env{'user.domain'});
            if ($ncol > 1) {
                $result .= '<br />'.&mt('If entering more than one, use spaces to separate the collaborators.');
            }
            $result .= '</label><br />';
	    $result .= &check_collaborators($ncol,$coll) if ($coll =~ /\w+/);
	    $result .= &Apache::lonhtmlcommon::row_closure();
	}
        my $filesfrom = 'both';
        my $stuname = &Apache::lonnet::EXT('user.name');
        my $studom = &Apache::lonnet::EXT('user.domain');
        if (!&Apache::lonnet::usertools_access($stuname,$studom,'portfolio')) {
            $filesfrom = 'uploadonly';
        }
	$result.=&Apache::inputtags::file_selector($part,$id,$uploadedfiletypes,
						   $filesfrom,undef,$maxfilesize);
        if ($result) {
            $result =
                  '<div>'.$hiddendraft.
                  &Apache::lonhtmlcommon::start_pick_box().
                  $result.
                  &Apache::lonhtmlcommon::end_pick_box().'</div>';
        } else {
            $result = $hiddendraft;
        }
    } elsif ($target eq 'web' &&
	     $Apache::inputtags::status[-1] ne 'CAN_ANSWER') {
	my $part= $Apache::inputtags::part;
	my @msgs;
	if ($Apache::lonhomework::history{"resource.$part.$id.collaborators"} =~ /\S/) {
	    my $coll= &HTML::Entities::encode($Apache::lonhomework::history{"resource.$part.$id.collaborators"},'<>&"');
	    $result .= '<td><i>'.&mt('Collaborated with [_1]',$coll).'</i></td>';
	}

        my $current_files_display = &Apache::inputtags::current_file_submissions($part,$id);
        if ($current_files_display) {
            $result .= '<td><b>'.&mt('Submitted files:').'</b><br />'.
                       $current_files_display.'</td>';
        }

	if ($result ne '') {
	    $result = 
		'<table class="LC_pastsubmission"><tr>'.$result.
		'</tr></table>';
	}
    }
    return $result;
}

sub end_essayresponse {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $part          = $Apache::inputtags::part;
    my $id            = $Apache::inputtags::response[-1];
    my $increment     = &Apache::response::repetition();
    my $result;
    if ( $target eq 'grade' ) {
# Deal with collaborators
	my $collaborators = $env{'form.HWCOL'.$part.'_'.$id};
	my $previous_list= &HTML::Entities::encode($Apache::lonhomework::history{"resource.$part.$id.collaborators"},'<>&"');
	if ($collaborators ne $previous_list) { 
#           &Apache::lonnet::logthis("New collaborators [$collaborators] [$previous_list]");
           $Apache::lonhomework::results{"resource.$part.$id.collaborators"}=$collaborators; 
        }
# Scantron
	if (  &Apache::response::submitted('scantron') ) {
	    $increment=&Apache::response::scored_response($part,$id);
	} elsif ( &Apache::response::submitted() ) {
	    my $response      = $env{'form.HWVAL_'.$id};
            my $jspart=$part;
            $jspart=~s/\./_/g;
            my $filename = $env{'form.HWFILE'.$jspart.'_'.$id.'.filename'} || 
                           $env{'form.HWFILETOOBIG'.$part.'_'.$id};
            my $portfiles = $env{'form.HWPORT'.$jspart.'_'.$id};
            my @deletions = &Apache::loncommon::get_env_multiple('form.HWFILE'.$jspart.'_'.$id.'_delete');
            my ($is_submit,$was_draft);
            if ($env{'form.HWDRAFT'.$part.'_'.$id} eq 'yes') {
                 $is_submit = 1;
            }
            if ($Apache::lonhomework::history{"resource.$part.award"} eq 'DRAFT') {
                $was_draft = 1;
            }
	    if (($response =~ /[^\s]/) || ($filename =~ /[^\s]/) || ($portfiles =~ /[^\s]/) || 
                (@deletions > 0) || ($was_draft && $is_submit)) {
 		my $award='DRAFT';
        	if ($env{'form.HWDRAFT'.$part.'_'.$id} eq 'yes') {
                    if ($Apache::lonhomework::type eq 'anonsurvey') {
                        $award='ANONYMOUS';
                    } elsif ($Apache::lonhomework::type eq 'anonsurveycred') {
                        $award='ANONYMOUS_CREDIT';
                    } elsif ($Apache::lonhomework::type eq 'surveycred') {
                        $award='SUBMITTED_CREDIT';
                    } else {
		        $award='SUBMITTED';
                    }
		}
                my $uploadedflag=0;
                my $totalsize=0;
                &file_submission($part,$id,\$award,\$uploadedflag,\$totalsize,\@deletions);
		$Apache::lonhomework::results{"resource.$part.$id.submission"}=$response;
		$Apache::lonhomework::results{"resource.$part.$id.awarddetail"}=$award;
		my %previous=&Apache::response::check_for_previous($response,$part,$id);
		if ($uploadedflag) {
                    if ($award eq 'FILENAME_INUSE') {
                        delete($Apache::lonhomework::results{"resource.$id.tries"});
                    }
                } else {
                    &Apache::response::handle_previous(\%previous,$award);
                }
#
# Store with resource author for similarity testing
#
                if ($award eq 'SUBMITTED') {
		    my ($symb,$crsid,$domain,$name)=
			&Apache::lonnet::whichuser();
		    if ($crsid) {
			my $akey=join('.',&escape($name),&escape($domain),
				      &escape($crsid));
			my $essayurl=
			    &Apache::lonnet::declutter($ENV{'REQUEST_URI'});
                        if ($essayurl eq 'lib/templates/simpleproblem.problem') {
                            my %crsinfo = &Apache::lonnet::coursedescription($crsid);
                            my $cdom = $crsinfo{'domain'};
                            my $cnum = $crsinfo{'num'};
                            my ($map,$id,$res) = &Apache::lonnet::decode_symb($symb);
                            if ($map =~ m{^\Quploaded/$cdom/$cnum/\E(default(?:|_\d+)\.(?:sequence|page))$}) {
                                my $apath = $1.'_'.$id;
                                $apath=~s/\W/\_/gs;
                                my $akey = join('.',&escape($name),&escape($domain));
                                &Apache::lonnet::put('nohist_essay_'.$apath,
                                                     { $akey => $response },$cdom,$cnum);
                            }
                        } else {
			    my ($adom,$aname,$apath)=
			        ($essayurl=~/^($LONCAPA::domain_re)\/($LONCAPA::username_re)\/(.*)$/);
                            $apath=&escape($apath);
			    $apath=~s/\W/\_/gs;
			    &Apache::lonnet::put('nohist_essay_'.$apath,
                                                 { $akey => $response },$adom,$aname);
                        }
                    }
                }
            }
        }
    } elsif ($target eq 'edit') {
	$result.=&Apache::edit::end_table();

    } elsif ($target eq 'tex'
	     && $Apache::lonhomework::type eq 'exam') {
	$result .= &Apache::inputtags::exam_score_line($target);

    } elsif ($target eq 'answer') {
	$result.=&Apache::response::answer_header($$tagstack[-1]);
	my $answer = &mt('Essay will be hand graded.');
	$result.=&Apache::response::answer_part($$tagstack[-1],$answer,
						{'no_verbatim' => 1});
	$result.=&Apache::response::answer_footer($$tagstack[-1]);
    }
    if ($target eq 'web') {
	&Apache::response::setup_prior_tries_hash(\&format_prior_response,
						  ['portfiles',
						   'uploadedurl']);
    }

    if ($target eq 'grade' || $target eq 'web' || $target eq 'answer' ||
	$target eq 'tex' || $target eq 'analyze') {
	&Apache::lonxml::increment_counter($increment, "$part.$id");

	if ($target eq 'analyze') {
            $Apache::lonhomework::analyze{"$part.$id.type"} = 'essayresponse';
            push (@{ $Apache::lonhomework::analyze{"parts"} },"$part.$id");
	    &Apache::lonhomework::set_bubble_lines();
	}
    }
    &Apache::response::end_response;

    return $result;
}

sub format_prior_response {
    my ($mode,$answer,$other_data) = @_;
    my $output;

    my (undef,undef,$udom,$uname) = &Apache::lonnet::whichuser();
    my $port_url = '/uploaded/'.$udom.'/'.$uname.'/portfolio/';

    my $file_list;

    foreach my $file (split(/\s*,\s*/,
			    $other_data->[0].','.$other_data->[1])) {
	next if ($file!~/\S/);
	if ($file !~ m{^/uploaded/}) { $file=$port_url.$file; }
	$file=~s|/+|/|g;
	&Apache::lonnet::allowuploaded('/adm/essayresponse',$file);
	$file_list.='<li><span class="LC_nobreak"><a href="'.$file.'?rawmode=1" target="lonGRDs"><img src="'.
	    &Apache::loncommon::icon($file).'" alt="file icon" border="0" /> '.$file.
	    '</a></span></li>'."\n";
    }
    if ($file_list) {
	$output.= &mt('Submitted Files').'<ul>'.$file_list.'</ul>';
    }
    if ($answer =~ /\S/) {
	$output.='<p>'.&mt('Submitted text').
	    '<blockquote>'.&HTML::Entities::encode($answer, '"<>&').'</blockquote></p>';
    }

    return '<div class="LC_prior_essay">'.$output.'</div>';
}

sub file_submission {
    my ($part,$id,$award,$uploadedflag,$totalsize,$deletions)=@_;
    my $files;
    my $jspart=$part;
    $jspart=~s/\./_/g;
    my ($symb,$crsid,$udom,$uname) = &Apache::lonnet::whichuser();
    my %crsinfo = &Apache::lonnet::coursedescription($crsid);
    my $cdom = $crsinfo{'domain'};
    my $cnum = $crsinfo{'num'};
    my (@portfiles,$uploadedurl,@submitted_portfiles,$submitted_upload,
        @acceptable_portfiles,$acceptable_upload,@accepted_portfiles,
        $accepted_upload,@savedportfiles,$stored_upload,@tolock,
        %port_delete,$uploaded_delete);
    if ($Apache::lonhomework::history{"resource.$part.$id.portfiles"} ||
        $Apache::lonhomework::history{"resource.$part.$id.uploadedurl"}) {
        if ($Apache::lonhomework::history{"resource.$part.$id.portfiles"}) {
            @portfiles = split(/,/,$Apache::lonhomework::history{"resource.$part.$id.portfiles"});
        }
        $uploadedurl = $Apache::lonhomework::history{"resource.$part.$id.uploadedurl"};
        if (ref($deletions) eq 'ARRAY') {
            if (@{$deletions} > 0) {
                foreach my $file (@{$deletions}) {
                    $file = &HTML::Entities::decode($file);
                    if (grep(/^\Q$file\E$/,@portfiles)) {
                        $port_delete{$file} = 1;
                    } elsif ($file =~ m{^/uploaded/\Q$udom\E/\Q$uname\E/essayresponse/\Q$cdom\E/\Q$cnum\E/}) {
                        $uploaded_delete = $file; 
                    } elsif ($file =~ m{^/uploaded/\Q$udom\E/\Q$uname\E/essayresponse/[^/]+$}) {
                        $uploaded_delete = $file;
                    }
                }
            }
        }
        foreach my $current (@portfiles) {
            unless ($port_delete{$current}) {
                push(@savedportfiles,$current);
            }
        }
        if ($uploaded_delete) {
            if ($uploaded_delete eq $uploadedurl) {
                $Apache::lonhomework::results{"resource.$part.$id.uploadedfile"} = "";
                $Apache::lonhomework::results{"resource.$part.$id.uploadedurl"} = "";
            } else {
                undef($uploaded_delete);
            }
        }
    }
    if ($env{'form.HWPORT'.$jspart.'_'.$id} ne '') {
        my $newfiles= $env{'form.HWPORT'.$jspart.'_'.$id};
        $newfiles =~s/,$//;
        if ($newfiles =~ /[^\s]/) {
            foreach my $file (split(/\s*,\s*/,$newfiles)) {
                if ($file =~ /[^\s]/) {
                    push(@submitted_portfiles,$file);
                }
            }
        }
    }
    if ($env{'form.HWFILETOOBIG'.$part.'_'.$id} ne '') {
        $$award = 'EXCESS_FILESIZE';
    } elsif ($env{'form.HWFILE'.$jspart.'_'.$id.'.filename'} ne '') {
        my $newfile = $env{'form.HWFILE'.$jspart.'_'.$id.'.filename'};
        if ($newfile =~ /[^\s]/) {
            $submitted_upload = $newfile;
        }
    }
    if (@savedportfiles) {
        foreach my $file (reverse(@savedportfiles)) {
            unless(grep(/^\Q$file\E$/,@submitted_portfiles)) {
                unshift(@submitted_portfiles,$file);
            }
        }
    }
    if (@submitted_portfiles || $submitted_upload) {
        my $uploadedfiletypes=
            &Apache::lonnet::EXT("resource.$part".'_'."$id.uploadedfiletypes");
        if ($uploadedfiletypes ne '') {
            $uploadedfiletypes=~s/[^\w\,]//g;
            $uploadedfiletypes=','.$uploadedfiletypes.',';
            if (@submitted_portfiles) {
                foreach my $file (@submitted_portfiles) {
                    my ($extension)=($file=~/\.(\w+)$/);
                    if ($uploadedfiletypes=~/\,\s*\Q$extension\E\s*\,/i) {
                        push(@acceptable_portfiles,$file);
                    }
                }
            }
            if ($submitted_upload) {
                my ($upload_ext)=($submitted_upload=~/\.(\w+)$/);
                if ($uploadedfiletypes=~/\,\s*\Q$upload_ext\E\s*\,/i) {
                    $acceptable_upload = $submitted_upload;
                } else {
                    $$award='INVALID_FILETYPE';
                    &delete_form_items($jspart,$id);
                }
            }
        } else {
            @acceptable_portfiles = @submitted_portfiles;
            $acceptable_upload = $submitted_upload;
        }
    }
    if ((@acceptable_portfiles) || ($acceptable_upload ne '')) {
        my $maxfilesize=&Apache::lonnet::EXT("resource.$part".'_'."$id.maxfilesize");
        if (!$maxfilesize) {
            $maxfilesize = 10.0; #FIXME This should become a domain configuration
        }
        my %dirlist;
        if (@acceptable_portfiles) {
            foreach my $file (@acceptable_portfiles) {
                my ($path,$filename) = ($file =~ m{^(.*/)([^/]+)$});
                my $fullpath = '/userfiles/portfolio'.$path;
                if (!exists($dirlist{$fullpath})) {
                    my ($listref,$listerror) =
                        &Apache::lonnet::dirlist($fullpath,$udom,$uname,1);
                    if (ref($listref) eq 'ARRAY') {
                        $dirlist{$fullpath} = $listref;
                    }
                }
                if (ref($dirlist{$fullpath}) eq 'ARRAY') {
                    foreach my $dir_line (@{$dirlist{$fullpath}}) {
                        my ($fname,$dom,undef,$testdir,undef,undef,undef,undef,
                            $size,undef,$mtime,undef,undef,undef,$obs,undef) =
                        split(/\&/,$dir_line,16);
                        if ($filename eq $fname) {
                            my $mbsize = $size/(1024.0*1024.0);
                            if (ref($totalsize)) {
                                $$totalsize += $mbsize;
                            }
                            last;
                        }
                    }
                }
                if (ref($totalsize)) {
                    if ($$totalsize > $maxfilesize) {
                        $$award='EXCESS_FILESIZE';
                        &delete_form_items($jspart,$id);
                    } else {
                        push(@accepted_portfiles,$file);
                    }
                } else {
                    push(@accepted_portfiles,$file);
                }
            }
        }
        if ($acceptable_upload ne '') {
            if (ref($totalsize)) {
                $$totalsize += $env{'form.HWFILESIZE'.$jspart.'_'.$id};
                if ($$totalsize > $maxfilesize) {
                    $$award='EXCESS_FILESIZE';
                    delete($env{'form.HWFILE'.$jspart.'_'.$id});
                } else {
                    $accepted_upload = $acceptable_upload;
                }
            } else {
                $accepted_upload = $acceptable_upload;
            }
        }
    }
    if ($accepted_upload ne '') {
        my ($path,$multiresp) = 
            &Apache::loncommon::get_turnedin_filepath($symb,$uname,$udom,
                                                      'submission');
        if ($path eq '') {
            $$award = 'INTERNAL_ERROR';
        } else {
            if ($multiresp) {
                $path .= '/'.$jspart.'_'.$id;
            }
            my $prefix = 'portfolio';
            my $formelement = 'HWFILE'.$jspart.'_'.$id;
            my $fname = &Apache::lonnet::clean_filename($env{'form.'.$formelement.'.filename'});
            my $url = '/uploaded/'.$udom.'/'.$uname.'/'.$prefix.$path.'/'.$fname;
            my @stat = &Apache::lonnet::stat_file($url);
            my $conflicts = 0;
            if (@stat && $stat[0] ne 'no_such_dir') {
                my $current_permissions = 
                    &Apache::lonnet::get_portfile_permissions($udom,$uname);
                if (ref($current_permissions) eq 'HASH') {
                    if (ref($current_permissions->{$path.'/'.$fname}) eq 'ARRAY') {
                        foreach my $record (@{$current_permissions->{$path.'/'.$fname}}) {
                            if (ref($record) eq 'ARRAY') {
                                next if (($record->[0] eq $symb) && 
                                         ($record->[1] eq $crsid));
                                $conflicts ++;
                            } 
                        }
                    }
                }
                if ($conflicts) {
                    $$award = 'FILENAME_INUSE';
                }
            }
            unless ($conflicts) { 
                my ($mode,%allfiles,%codebase);
                my $result = &Apache::lonnet::userfileupload($formelement,'',
                             $prefix.$path,$mode,\%allfiles,\%codebase);
                if ($result =~ m{^/uploaded/}) {
                    $stored_upload = $path.'/'.$fname;
                    unless (grep(/^\Q$stored_upload\E$/,@accepted_portfiles)) {
                        $Apache::lonhomework::results{"resource.$part.$id.portfiles"} = $stored_upload;
                        push(@tolock,$stored_upload);
                    }
                } else {
                    $$award = 'INTERNAL_ERROR';
                }
            }
        }
        delete($env{'form.HWFILE'.$jspart.'_'.$id});
    }
    if (@accepted_portfiles) {
        if ($Apache::lonhomework::results{"resource.$part.$id.portfiles"}) {
            $Apache::lonhomework::results{"resource.$part.$id.portfiles"} .= ',';
        }
        $Apache::lonhomework::results{"resource.$part.$id.portfiles"}.=join(',',@accepted_portfiles);
        push(@tolock,@accepted_portfiles);
    }
    if (!defined($Apache::lonhomework::results{"resource.$part.$id.portfiles"})) {
        if (keys(%port_delete) > 0) {
            $Apache::lonhomework::results{"resource.$part.$id.portfiles"} = "";
        }
    }
    if (($Apache::lonhomework::history{"resource.$part.$id.portfiles"} ne  
         $Apache::lonhomework::results{"resource.$part.$id.portfiles"}) ||
        ($uploaded_delete)) {
        if (ref($uploadedflag)) {
            $$uploadedflag=1;
        }
    }
    &Apache::lonnet::unmark_as_readonly($udom,$uname,[$symb,$crsid]);
    &Apache::lonnet::mark_as_readonly($udom,$uname,[@tolock],[$symb,$crsid]);
    &Apache::lonnet::clear_selected_files($uname);
    return;
}

sub delete_form_items {
    my ($jspart,$id) = @_;
    delete($env{'form.HWFILE'.$jspart.'_'.$id.'.filename'});
    delete($env{'form.HWFILE'.$jspart.'_'.$id.'.mimetype'});
    delete($env{'form.HWFILE'.$jspart.'_'.$id});
}

sub check_collaborators {
    my ($ncol,$coll) = @_;
    my %classlist=&Apache::lonnet::dump('classlist',
					$env{'course.'.$env{'request.course.id'}.'.domain'},
					$env{'course.'.$env{'request.course.id'}.'.num'});
    my (@badcollaborators,$result);
  
    my (@collaborators) = split(/[,;\s]+/,$coll);
    foreach my $entry (@collaborators) {
        my $collaborator;
	if ($entry =~ /:/) {
	    $collaborator = $entry;
	} else {
	    $collaborator = $entry.':'.$env{'user.domain'};
	}
        if ($collaborator !~ /^$match_username:$match_domain$/) {
            if (!grep(/^\Q$entry\E$/,@badcollaborators)) {
	        push(@badcollaborators,$entry);
            }
        } elsif (!grep(/^\Q$collaborator\E$/i,keys(%classlist))) {
            if (!grep(/^\Q$entry\E$/,@badcollaborators)) {
                push(@badcollaborators,$entry);
            }
        }
    }
    
    my $numbad = scalar(@badcollaborators);
    if ($numbad) {
	$result = '<table border="0"><tr bgcolor="#ffbbbb"><td>';
        if ($numbad == 1) {
            $result .= &mt('The following user is invalid:');
        } else {
            $result .= &mt('The following [_1] users are invalid:',$numbad);
        }
	$result .= ' '.join(', ',@badcollaborators).'. '.&mt('Please correct.').
                   '</td></tr></table>';
    }
    my $toomany = scalar(@collaborators) - $ncol;
    if ($toomany > 0) {
	$result .= '<table border="0"><tr bgcolor="#ffbbbb"><td>'.
	           &mt('You have too many collaborators.').' '.
                   &mt('Please remove [quant,_1,collaborator].',$toomany).
	           '</td></tr></table>';
    }
    return $result;
}

1;
__END__


=pod

=head1 NAME

Apache::essayresponse

=head1 SYNOPSIS

Handler to evaluate essay (ungraded) style responses.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 SUBROUTINES

=over 

=item start_essayresponse()

=item end_essayresponse()

=item format_prior_response()

=item file_submission()

=item delete_form_items()

=item check_collaborators()

=back

=cut
