# The LearningOnline Network with CAPA 
# implements the tags that control the hints
#
# $Id: hint.pm,v 1.78 2011/06/08 20:38:47 www Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA me&aree software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#



package Apache::hinttags; 

use strict;
use Apache::lonnet();
use capa;
use Apache::caparesponse();
use Apache::lonmaxima();
use Apache::lonr();
use Apache::chemresponse();
use Apache::response();
use Apache::lonlocal;
use Storable qw(dclone);

BEGIN {
    &Apache::lonxml::register('Apache::hinttags',('hintgroup','hintpart','numericalhint','stringhint','formulahint','optionhint','radiobuttonhint','mathhint','customhint','reactionhint','organichint'));
}


@Apache::hint::which=();
sub start_hintgroup {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $skiptoend='0';
    my $result;

    if ($target eq 'web') {
	my $id=$Apache::inputtags::part;
	my $numtries=$Apache::lonhomework::history{"resource.$id.tries"};
	if ( $numtries eq '') { $numtries = 0; }
	my $hinttries=&Apache::response::get_response_param($id,"hinttries",1);
	&Apache::lonxml::debug("found :$id:$numtries:$hinttries:");
	my $gradestatus=
	    $Apache::lonhomework::history{"resource.$id.solved"};
	my $showoncorrect=lc(&Apache::lonxml::get_param('showoncorrect',$parstack,$safeeval));	
	&Apache::lonxml::debug("onc orrect $showoncorrect, $gradestatus");
	if ( ($showoncorrect ne 'yes' && &Apache::response::show_answer()) ||
	     ( $numtries < $hinttries) ) {
	    &Apache::lonxml::debug("Grabbin all");
	    &Apache::lonxml::get_all_text("/hintgroup",$parser,$style);
	}
	&Apache::lonxml::startredirection;
    } elsif ($target eq 'tex') {
	$result .= '\keephidden{';
    } elsif ($target eq 'edit') {
	$result.=&Apache::edit::tag_start($target,$token);
	$result.=&Apache::edit::select_arg('Show hint even if problem Correct:','showoncorrect',['no','yes'],$token);
	$result.=&Apache::edit::end_row().&Apache::edit::start_spanning_row();
    } elsif ($target eq 'modified') {
	my $constructtag=&Apache::edit::get_new_args($token,$parstack,$safeeval,'showoncorrect');
       if ($constructtag) {
           $result =&Apache::edit::rebuild_tag($token);
       }
    }
    @Apache::hint::which=();
    if (($#Apache::functionplotresponse::failedrules>=0) && ($target eq 'web')) {
        @Apache::hint::which=@Apache::functionplotresponse::failedrules;
        undef @Apache::functionplotresponse::failedrules;
    }
    return $result;
}

sub end_hintgroup {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;

    if ($target eq 'web') {
	my $id=$Apache::inputtags::part;
	my $numtries=$Apache::lonhomework::history{"resource.$id.tries"};
	if ( $numtries eq '') { $numtries = 0; }
	my $hinttries=&Apache::response::get_response_param($id,"hinttries",1);
	&Apache::lonxml::debug("found :$id:$numtries:$hinttries:");
	my $hinttext=&Apache::lonxml::endredirection;
	if ($Apache::lonhomework::type ne 'exam' &&
	    $numtries >= $hinttries && $hinttext =~/\S/) {
	    $result='<table bgcolor="#dddddd"><tr><td>'.
		$hinttext.'</td></tr></table>';
	}
    } elsif ($target eq 'edit') {
	$result.=&Apache::edit::end_row().&Apache::edit::end_table();
    } elsif ($target eq 'tex') {
	$result .= '}';
    }
    @Apache::hint::which=();
    return $result;
}

sub start_numericalhint {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    #do everything in end, so intervening <responseparams> work
    &Apache::response::start_hintresponse($parstack,$safeeval);
    &Apache::caparesponse::push_answer();
    my $result;
    if ($target eq 'edit') {
	$result.=&Apache::edit::tag_start($target,$token);
	$result.=&Apache::edit::text_arg('Name:','name',$token);
	$result.=&Apache::edit::text_arg('Answer:','answer',$token);
	if ($token->[1] eq 'numericalhint') {
	    $result.=&Apache::edit::text_arg('Unit:','unit',$token,5).
		&Apache::loncommon::help_open_topic('Physical_Units');
	    $result.=&Apache::edit::text_arg('Format:','format',$token,4).
		&Apache::loncommon::help_open_topic('Numerical_Response_Format');
	} elsif ($token->[1] eq 'formulahint') {
	    $result.=&Apache::edit::text_arg('Sample Points:','samples',
					     $token,40).
	        &Apache::loncommon::help_open_topic('Formula_Response_Sampling');
	}
        $result.=&Apache::edit::text_arg('Pre-Processor Subroutine:','preprocess',
                                             $token,10);
	$result.=&Apache::edit::end_row();
	$result.=&Apache::edit::start_spanning_row();
    } elsif ($target eq 'modified') {
	my $constructtag;
	if ($token->[1] eq 'numericalhint') {
	    $constructtag=&Apache::edit::get_new_args($token,$parstack,
						     $safeeval,'name',
						     'answer','unit','format','preprocess');
	} elsif ($token->[1] eq 'formulahint') {
	    $constructtag=&Apache::edit::get_new_args($token,$parstack,
						     $safeeval,'name','answer',
						     'samples','preprocess');
	}
	if ($constructtag) {
	    $result  = &Apache::edit::rebuild_tag($token);
	}
    } elsif ($target eq 'web') {
	&Apache::response::reset_params();
    }
    return $result;
}

sub end_numericalhint {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    if ($target eq 'web') {
	if (!$Apache::lonxml::default_homework_loaded) {
	    &Apache::lonxml::default_homework_load($safeeval);
	}
	$safeeval->share_from('capa',['&caparesponse_capa_check_answer']);
	my $hint_name= &Apache::lonxml::get_param('name',$parstack,$safeeval);
	&Apache::response::setup_params('numericalhint',$safeeval);

	my $partid=$Apache::inputtags::part;
	my $id=$Apache::inputtags::hint[-1];
	#id submissions occurred under
	my $submitid=$Apache::inputtags::response[-1];

	my $response = $Apache::lonhomework::history{
			    "resource.$partid.$submitid.submission"};
	&Apache::lonxml::debug("hintgroup is using $response<br />\n");

	my @args = ('type','tol','sig','ans_fmt','unit','calc','samples');
	my $args_ref =
	    &Apache::caparesponse::setup_capa_args($safeeval,$parstack,
						   \@args,$response);

	my $hideunit=&Apache::response::get_response_param($partid.'_'.$submitid,'turnoffunit');
	if (lc($hideunit) eq "yes") { delete($$args_ref{'unit'}); }

	if ($$tagstack[-1] eq 'formulahint') {
            if ($$args_ref{'samples'}) {
                $$args_ref{'type'}='fml';
            } else {
                $$args_ref{'type'}='math';
            }
	} elsif ($$tagstack[-1] eq 'numericalhint') {
	    $$args_ref{'type'}='float';
	}
	&Apache::caparesponse::add_in_tag_answer($parstack,$safeeval);
	my %answer = &Apache::caparesponse::get_answer();
	my (@final_awards,@final_msgs,@ans_names);
	foreach my $ans_name (keys(%answer)) {
	    &Apache::lonxml::debug(" doing $ans_name with ".join(':',@{ $answer{$ans_name}{'answers'} }));
	
	    ${$safeeval->varglob('LONCAPA::CAPAresponse_answer')}=
		dclone($answer{$ans_name});
	    &Apache::caparesponse::setup_capa_response($args_ref,$response);
	    my ($result,@msgs) = 
		&Apache::run::run("&caparesponse_check_list()",$safeeval);
	    &Apache::lonxml::debug("checking $ans_name $result with $response");
	    &Apache::lonxml::debug('msgs are '.join(':',@msgs));
	    my ($awards)=split(/:/,$result);
	    my @awards= split(/,/,$awards);
	    my ($ad, $msg) = 
		&Apache::inputtags::finalizeawards(\@awards,\@msgs);
	    push(@final_awards,$ad);
	    push(@final_msgs,$msg);
	    push(@ans_names,$ans_name);
	}
	my ($ad, $msg, $ans_name) =
	    &Apache::inputtags::finalizeawards(\@final_awards,
					       \@final_msgs,
					       \@ans_names,1);
	if ($ad eq 'EXACT_ANS' || $ad eq 'APPROX_ANS') {
	    push(@Apache::hint::which,$hint_name);	
	}
	$result='';
    } elsif ($target eq 'meta') {
	$result=&Apache::response::meta_package_write($token->[1]);
    } elsif ($target eq 'edit') {
	$result.=&Apache::edit::end_row().&Apache::edit::end_table();
    }
    &Apache::caparesponse::pop_answer();
    &Apache::response::end_hintresponse();
    return $result;
}

sub start_formulahint {
    return &start_numericalhint(@_);
}

sub end_formulahint {
    return &end_numericalhint(@_);
}

sub start_mathhint {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    #do everything in end, so intervening <responseparams> and <answer> work
    &Apache::response::start_hintresponse($parstack,$safeeval);
    &Apache::lonxml::register('Apache::response',('answer'));
    my $result;
    if ($target eq 'edit') {
        $result.=&Apache::edit::tag_start($target,$token);
        $result.=&Apache::edit::text_arg('Name:','name',$token);
        $result.=&Apache::edit::select_arg('Algebra System:',
                                           'cas',
                                           ['maxima','R'],
                                           $token);
        $result.=&Apache::edit::text_arg('Argument Array:',
                                         'args',$token);
        $result.=&Apache::edit::end_row();
        $result.=&Apache::edit::start_spanning_row();
    } elsif ($target eq 'modified') {
        my $constructtag;
        $constructtag=&Apache::edit::get_new_args($token,$parstack,
                                                  $safeeval,'name','cas',
                                                  'args');
        $result  = &Apache::edit::rebuild_tag($token);
    } elsif ($target eq 'web') {
        &Apache::response::reset_params();
    }
    return $result;
}

sub end_mathhint {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    if ($target eq 'web') {
        if (!$Apache::lonxml::default_homework_loaded) {
            &Apache::lonxml::default_homework_load($safeeval);
        }
	my $name= &Apache::lonxml::get_param('name',$parstack,$safeeval);
        &Apache::response::setup_params('mathhint',$safeeval);
        my $partid=$Apache::inputtags::part;
        my $submitid=$Apache::inputtags::response[-1];
        my $response = $Apache::lonhomework::history{
                            "resource.$partid.$submitid.submission"};
	
	my $cas = &Apache::lonxml::get_param('cas',$parstack,$safeeval);
	my $award;
	if ($cas eq 'maxima') {
	    my $args = [&Apache::lonxml::get_param_var('args',$parstack,$safeeval)];
	    $award=&Apache::lonmaxima::maxima_run($Apache::response::custom_answer[-1],$response,$args);
	}
        if ($cas eq 'R') {
            my $args = [&Apache::lonxml::get_param_var('args',$parstack,$safeeval)];
            $award=&Apache::lonr::r_run($Apache::response::custom_answer[-1],$response,$args);
        }
        if ($award eq 'EXACT_ANS' || $award eq 'APPROX_ANS') {
            push (@Apache::hint::which,$name);
        }
        $result='';
    } elsif ($target eq 'meta') {
        $result=&Apache::response::meta_package_write($token->[1]);
    } elsif ($target eq 'edit') {
        $result.=&Apache::edit::end_row().&Apache::edit::end_table();
    }
    pop(@Apache::response::custom_answer);
    pop(@Apache::response::custom_answer_type);
    &Apache::response::end_hintresponse();
    return $result;
}

sub start_customhint {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    #do everything in end, so intervening <responseparams> and <answer> work
    &Apache::response::start_hintresponse($parstack,$safeeval);
    &Apache::lonxml::register('Apache::response',('answer'));
    my $result;
    if ($target eq 'edit') {
        $result.=&Apache::edit::tag_start($target,$token);
        $result.=&Apache::edit::text_arg('Name:','name',$token);
        $result.=&Apache::edit::end_row();
        $result.=&Apache::edit::start_spanning_row();
    } elsif ($target eq 'modified') {
        my $constructtag;
        $constructtag=&Apache::edit::get_new_args($token,$parstack,
                                                  $safeeval,'name');
        $result  = &Apache::edit::rebuild_tag($token);
    } elsif ($target eq 'web') {
        &Apache::response::reset_params();
    }
    return $result;
}

sub end_customhint {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    if ($target eq 'web') {
        if (!$Apache::lonxml::default_homework_loaded) {
            &Apache::lonxml::default_homework_load($safeeval);
        }
	my $name= &Apache::lonxml::get_param('name',$parstack,$safeeval);
        &Apache::response::setup_params('customhint',$safeeval);
        my $partid=$Apache::inputtags::part;
        my $submitid=$Apache::inputtags::response[-1];
        my $response = $Apache::lonhomework::history{
                            "resource.$partid.$submitid.submission"};
        my $award;	
	if ( $response =~ /[^\s]/ && 
	     $Apache::response::custom_answer_type[-1] eq 'loncapa/perl') {
	    if (!$Apache::lonxml::default_homework_loaded) {
		&Apache::lonxml::default_homework_load($safeeval);
	    }
	    ${$safeeval->varglob('LONCAPA::customresponse_submission')}=
		$response;
	    
	    $award = &Apache::run::run('{ my $submission=$LONCAPA::customresponse_submission;'.$Apache::response::custom_answer[-1].'}',$safeeval);
	}
        if ($award eq 'EXACT_ANS' || $award eq 'APPROX_ANS') {
            push (@Apache::hint::which,$name);
        }
        $result='';
    } elsif ($target eq 'meta') {
        $result=&Apache::response::meta_package_write($token->[1]);
    } elsif ($target eq 'edit') {
        $result.=&Apache::edit::end_row().&Apache::edit::end_table();
    }
    pop(@Apache::response::custom_answer);
    pop(@Apache::response::custom_answer_type);
    &Apache::response::end_hintresponse();
    return $result;
}

sub start_stringhint {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    #do everything in end, so intervening <responseparams> work
    &Apache::response::start_hintresponse($parstack,$safeeval);
    my $result;
    if ($target eq 'edit') {
	$result.=&Apache::edit::tag_start($target,$token);
	$result.=&Apache::edit::text_arg('Name:','name',$token);
	$result.=&Apache::edit::text_arg('Answer:','answer',$token);
	$result.=&Apache::edit::select_arg('Type:','type',
			 [['cs','Case Sensitive'],['ci','Case Insensitive'],
			  ['mc','Case Insensitive, Any Order'],
			  ['re','Regular Expression']],$token);
        $result.=&Apache::edit::text_arg('Pre-Processor Subroutine:','preprocess',
                                             $token,10);
	$result.=&Apache::edit::end_row();
	$result.=&Apache::edit::start_spanning_row();
    } elsif ($target eq 'modified') {
	my $constructtag;
	$constructtag=&Apache::edit::get_new_args($token,$parstack,
						  $safeeval,'name','answer',
						  'type','preprocess');
	$result  = &Apache::edit::rebuild_tag($token);
    } elsif ($target eq 'web') {
	&Apache::response::reset_params();
    }
    return $result;
}

sub end_stringhint {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    if ($target eq 'web') {
	if (!$Apache::lonxml::default_homework_loaded) {
	    &Apache::lonxml::default_homework_load($safeeval);
	}
	$safeeval->share_from('capa',['&caparesponse_capa_check_answer']);
	my $hint_name= &Apache::lonxml::get_param('name',$parstack,$safeeval);
	&Apache::response::setup_params('stringhint',$safeeval);
	my $partid=$Apache::inputtags::part;
	my $id=$Apache::inputtags::hint[-1];
	#id submissions occurred under
	my $submitid=$Apache::inputtags::response[-1];
	my $response = $Apache::lonhomework::history{
			    "resource.$partid.$submitid.submission"};
	&Apache::lonxml::debug("hintgroup is using $response<br />\n");
	my $type = &Apache::lonxml::get_param('type',$parstack,$safeeval);
	my ($ad,$msg);
	if ($type eq 're' ) {
	    my $answer=&Apache::lonxml::get_param('answer',$parstack,
						  $safeeval);
	    ${$safeeval->varglob('LONCAPA::response')}=$response;
	    my $compare='=';
	    if ($answer=~/^\s*NOT\s*/) {
		$answer=~s/^\s*NOT\s*//;
		$compare='!';
	    }
	    my $test='$LONCAPA::response'.$compare.'~m'.$answer;
	    &Apache::lonxml::debug("test $test");
	    $result = &Apache::run::run("return $test",$safeeval);
	    &Apache::lonxml::debug("current $response");
	    &Apache::lonxml::debug("current $answer");
	    $ad = ($result) ? 'APPROX_ANS' : 'INCORRECT';
	} else { 
	    my @args = ('type');
	    my $args_ref = 
		&Apache::caparesponse::setup_capa_args($safeeval,$parstack,
						       \@args,$response);
            if ($$args_ref{'type'} eq '') {
                $$args_ref{'type'} = 'ci';
            }
	    &Apache::caparesponse::add_in_tag_answer($parstack,$safeeval);
	    my (@final_awards,@final_msgs,@ans_names);
	    my %answer = &Apache::caparesponse::get_answer();
	    foreach my $ans_name (keys(%answer)) {
		&Apache::lonxml::debug(" doing $ans_name with ".join(':',@{ $answer{$ans_name}{'answers'} }));
		${$safeeval->varglob('LONCAPA::CAPAresponse_answer')}=dclone($answer{$ans_name});
		my ($result, @msgs)=&Apache::run::run("&caparesponse_check_list()",$safeeval);
		&Apache::lonxml::debug('msgs are'.join(':',@msgs));
		my ($awards) = split(/:/,$result);
		my (@awards) = split(/,/,$awards);
		($ad,$msg) = 
		    &Apache::inputtags::finalizeawards(\@awards,\@msgs);
		push(@final_awards,$ad);
		push(@final_msgs,$msg);
		push(@ans_names,$ans_name);
		&Apache::lonxml::debug("\n<br>result:$result:$Apache::lonxml::curdepth<br>\n");
	    }
	    my ($ad, $msg, $ans_name) = 
		&Apache::inputtags::finalizeawards(\@final_awards,
						   \@final_msgs,
						   \@ans_names,1);
	}
	if ($ad eq 'EXACT_ANS' || $ad eq 'APPROX_ANS') { 
	    push(@Apache::hint::which,$hint_name);
	}
	$result='';
    } elsif ($target eq 'meta') {
	$result=&Apache::response::meta_package_write($token->[1]);
    } elsif ($target eq 'edit') {
	$result.=&Apache::edit::end_row().&Apache::edit::end_table();
    }
    &Apache::response::end_hintresponse();
    return $result;
}

# reactionhint

sub start_reactionhint {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    #do everything in end, so intervening <responseparams> work
    &Apache::response::start_hintresponse($parstack,$safeeval);
    my $result;
    if ($target eq 'edit') {
        $result.=&Apache::edit::tag_start($target,$token);
        $result.=&Apache::edit::text_arg('Name:','name',$token);
        $result.=&Apache::edit::text_arg('Answer:','answer',$token);
        $result.=&Apache::edit::end_row();
        $result.=&Apache::edit::start_spanning_row();
    } elsif ($target eq 'modified') {
        my $constructtag;
        $constructtag=&Apache::edit::get_new_args($token,$parstack,
                                                  $safeeval,'name','answer',
                                                  );
        $result  = &Apache::edit::rebuild_tag($token);
    } elsif ($target eq 'web') {
        &Apache::response::reset_params();
    }
    return $result;
}

sub end_reactionhint {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    if ($target eq 'web') {
        if (!$Apache::lonxml::default_homework_loaded) {
            &Apache::lonxml::default_homework_load($safeeval);
        }
        my $hint_name= &Apache::lonxml::get_param('name',$parstack,$safeeval);
        &Apache::response::setup_params('reactionhint',$safeeval);
        my $partid=$Apache::inputtags::part;
        my $id=$Apache::inputtags::hint[-1];
        #id submissions occurred under
        my $submitid=$Apache::inputtags::response[-1];
        my $response = $Apache::lonhomework::history{
                            "resource.$partid.$submitid.submission"};
        &Apache::lonxml::debug("hintgroup is using $response<br />\n");
        my $answer=&Apache::lonxml::get_param('answer',$parstack,
                                                  $safeeval);
        &Apache::lonxml::debug("current $response");
        &Apache::lonxml::debug("current $answer");
        if (&Apache::chemresponse::chem_standard_order($response) eq 
            &Apache::chemresponse::chem_standard_order($answer)) {
            push(@Apache::hint::which,$hint_name);
        }
        $result='';
    } elsif ($target eq 'meta') {
        $result=&Apache::response::meta_package_write($token->[1]);
    } elsif ($target eq 'edit') {
        $result.=&Apache::edit::end_row().&Apache::edit::end_table();
    }
    &Apache::response::end_hintresponse();
    return $result;
}

# organichint

sub start_organichint {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    #do everything in end, so intervening <responseparams> work
    &Apache::response::start_hintresponse($parstack,$safeeval);
    my $result;
    if ($target eq 'edit') {
        $result.=&Apache::edit::tag_start($target,$token);
        $result.=&Apache::edit::text_arg('Name:','name',$token);
        $result.=&Apache::edit::text_arg('Answer:','answer',$token);
        $result.=&Apache::edit::end_row();
        $result.=&Apache::edit::start_spanning_row();
    } elsif ($target eq 'modified') {
        my $constructtag;
        $constructtag=&Apache::edit::get_new_args($token,$parstack,
                                                  $safeeval,'name','answer',
                                                  );
        $result  = &Apache::edit::rebuild_tag($token);
    } elsif ($target eq 'web') {
        &Apache::response::reset_params();
    }
    return $result;
}

sub end_organichint {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    if ($target eq 'web') {
        if (!$Apache::lonxml::default_homework_loaded) {
            &Apache::lonxml::default_homework_load($safeeval);
        }
        my $hint_name= &Apache::lonxml::get_param('name',$parstack,$safeeval);
        &Apache::response::setup_params('organichint',$safeeval);
        my $partid=$Apache::inputtags::part;
        my $id=$Apache::inputtags::hint[-1];
        #id submissions occurred under
        my $submitid=$Apache::inputtags::response[-1];
        my $response = $Apache::lonhomework::history{
                            "resource.$partid.$submitid.submission"};
        &Apache::lonxml::debug("hintgroup is using $response<br />\n");
        my (@answers)=&Apache::lonxml::get_param_var('answer',$parstack,$safeeval);
        my %previous = &Apache::response::check_for_previous($response,$partid,$id);
        foreach my $answer (@answers) {
            &Apache::lonxml::debug("submitted a $response for $answer<br \>\n");
            if ($response eq $answer) {
               push(@Apache::hint::which,$hint_name);
               last;
            }
        }
        $result='';
    } elsif ($target eq 'meta') {
        $result=&Apache::response::meta_package_write($token->[1]);
    } elsif ($target eq 'edit') {
        $result.=&Apache::edit::end_row().&Apache::edit::end_table();
    }
    &Apache::response::end_hintresponse();
    return $result;
}


# a part shows if it is on, if no specific parts are on, then default shows
sub start_hintpart {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;

    my $show ='0';
    my $result = '';
    if ($target eq 'web') {
	my $on= &Apache::lonxml::get_param('on',$parstack,$safeeval);
	&Apache::lonxml::debug("hintpart sees $on and ,$#Apache::hint::which");
	if ( $on eq 'default' && $#Apache::hint::which == '-1') {
	    $show=1;
	} else {
	    my $which;
	    foreach $which (@Apache::hint::which) { if ($which eq $on) { $show = 1; last } }
	}
	if (!$show) {
	    &Apache::lonxml::get_all_text("/hintpart",$parser,$style);
	}
    } elsif ($target eq 'grade') {
	&Apache::lonxml::get_all_text("/hintpart",$parser,$style);
    } elsif ($target eq 'edit') {
	$result.= &Apache::edit::tag_start($target,$token);
	$result.= &Apache::edit::text_arg('On:','on',$token);
	$result.= &Apache::edit::end_row();
	$result.= &Apache::edit::start_spanning_row();
    } elsif ($target eq 'modified') {
	my $constructtag=&Apache::edit::get_new_args($token,$parstack,
						     $safeeval,'on');
	if ($constructtag) {
	    $result = &Apache::edit::rebuild_tag($token);
	}
    }
    return $result;
}

sub end_hintpart {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my $result;
    if ($target eq 'edit') {
	$result.=&Apache::edit::end_row().&Apache::edit::end_table();
    }
    return $result;
}

sub start_optionhint {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my $result;
    &Apache::response::start_hintresponse($parstack,$safeeval);
    if ($target eq 'edit') {
	$result.=&Apache::edit::tag_start($target,$token);
	$result.=&Apache::edit::text_arg('Name:','name',$token);
	$result.=&Apache::edit::text_arg('Answer:','answer',$token,40);
	$result.=&Apache::edit::text_arg('Concept:','concept',$token,50);
    } elsif ($target eq 'modified') {
	my $constructtag=&Apache::edit::get_new_args($token,$parstack,
						     $safeeval,'name',
						     'answer','concept');
	if ($constructtag) {
	    $result  = &Apache::edit::rebuild_tag($token);
	}
    } elsif ($target eq 'meta') {
	$result=&Apache::response::meta_package_write('numericalhint');
    }
    return $result;
}

sub end_optionhint {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my $result;
    if ($target eq 'web') {
	my ($foilmatch,$conceptmatch)=(-1,-1);
	my $name= &Apache::lonxml::get_param('name',$parstack,$safeeval);
	my $partid=$Apache::inputtags::part;
	#id submissions occurred under
	my $submitid=$Apache::inputtags::response[-1];
	my $part_id="$partid.$submitid";
	my %answer;
	my $answer=&Apache::lonxml::get_param('answer',$parstack,$safeeval);
	if ($answer) {
	    eval('%answer ='.$answer);
	    &Apache::lonxml::debug("answwer hash");
	    &Apache::lonhomework::showhash(%answer);
	    my $response = $Apache::lonhomework::history{
				            "resource.$part_id.submission"};
	    my %response=&Apache::lonnet::str2hash($response);
	    &Apache::lonhomework::showhash(%response);
	    foreach my $foil (keys(%answer)) {
		$foilmatch=1;
		if ($answer{$foil} ne $response{$foil}) {$foilmatch=0;last;}
	    }
	}
	my %concept;
	my $constr=&Apache::lonxml::get_param('concept',$parstack,$safeeval);
	if ( $constr ) { eval('%concept ='.$constr);	}
	my $response = $Apache::lonhomework::history{
					"resource.$part_id.submissiongrading"};
	my %response=&Apache::lonnet::str2hash($response);
	foreach my $concept (keys(%concept)) {
	    my $compare;
	    if ($concept{$concept} eq 'correct') {$compare=1}else{$compare=0}
	    $conceptmatch=1;
	    if (ref($Apache::hint::option{"$part_id.concepts"})) {
		foreach my $foil (@{ $Apache::hint::option{"$part_id.concept.$concept"} }) {
		    &Apache::lonxml::debug("compare -$foil- -$response{$foil}-$compare-");
		    if ( exists($response{$foil}) && 
			 $response{$foil} ne $compare) {$conceptmatch=0;last;}
		}
	    } else {
		$conceptmatch=0;
	    }
	    if ($conceptmatch eq '0') { last; }
	}
	if ( ($conceptmatch eq '-1' || $conceptmatch eq '1') &&
	     ($foilmatch    eq '-1' || $foilmatch    eq '1') ) {
	    push(@Apache::hint::which,$name);
	}
    } elsif ($target eq 'edit') {
	$result.=&Apache::edit::end_row().&Apache::edit::end_table();
    }
    &Apache::response::end_hintresponse();
    return $result;
}

sub start_radiobuttonhint {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my $result;
    &Apache::response::start_hintresponse($parstack,$safeeval);
    if ($target eq 'edit') {
	$result.=&Apache::edit::tag_start($target,$token);
	$result.=&Apache::edit::text_arg('Name:','name',$token);
	$result.=&Apache::edit::text_arg('Answer:','answer',$token);
    } elsif ($target eq 'modified') {
	my $constructtag=&Apache::edit::get_new_args($token,$parstack,
						     $safeeval,'name',
						     'answer');
	if ($constructtag) {
	    $result  = &Apache::edit::rebuild_tag($token);
	}
    } elsif ($target eq 'meta') {
	$result=&Apache::response::meta_package_write('numericalhint');
    }
    return $result;
}

sub end_radiobuttonhint {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval)=@_;
    my $result;
    if ($target eq 'web') {
	my $name= &Apache::lonxml::get_param('name',$parstack,$safeeval);
	my @answer;
	my $answer=&Apache::lonxml::get_param('answer',$parstack,$safeeval);
	eval('@answer ='.$answer);
	my $partid=$Apache::inputtags::part;
	#id submissions occurred under
	my $submitid=$Apache::inputtags::response[-1];
	my $part_id="$partid.$submitid";
	my $response = $Apache::lonhomework::history{
			    "resource.$part_id.submission"};
	($response)=&Apache::lonnet::str2hash($response);
	&Apache::lonxml::debug("response is $response");
	
	if ($answer[0] eq 'foil') {
	    shift(@answer);
	    foreach my $answer (@answer) {
		if ($response eq $answer) {
		    push (@Apache::hint::which,$name);
		    last;
		}
	    }
	} elsif ($answer[0] eq 'concept') {
	    shift(@answer);
	    foreach my $answer (@answer) {
		if (ref($Apache::hint::radiobutton{"$part_id.concept.".$answer})) {
		    my @names=@{ $Apache::hint::radiobutton{"$part_id.concept.".$answer} };
		    if (grep(/^\Q$response\E$/,@names)) {
			push(@Apache::hint::which,$name);
			last;
		    }
		}
	    }
	}
    } elsif ($target eq 'edit') {
	$result.=&Apache::edit::end_row().&Apache::edit::end_table();
    }
    &Apache::response::end_hintresponse();
    return $result;
}
1;
__END__


=head1 NAME

Apache::hinttags

=head1 SYNOPSIS

This handler coordinates the delivery of hints to students working on LON-CAPA problems and assignments.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 SUBROUTINES

=over

=item start_hintgroup()

=item end_hintgroup()

=item start_numericalhint()

=item end_numericalhint()

=item start_formulahint()

=item end_formulahint()

=item start_mathhint()

=item end_mathhint()

=item start_customhint()

=item end_customhint()

=item start_stringhint()

=item end_stringhint()

=item start_hintpart()

=item end_hintpart()

=item start_optionhint()

=item end_optionhint()

=item start_radiobuttonhint()

=item end_radiobuttonhint()

=back

=cut
