# The LearningOnline Network with CAPA
# Handler for displaying the course catalog interface
#
# $Id: coursecatalog.pm,v 1.88.2.7 2019/08/02 00:24:02 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#

package Apache::coursecatalog;

use strict;
use lib qw(/home/httpd/lib/perl);
use Apache::Constants qw(:common);
use Apache::loncommon;
use Apache::lonhtmlcommon;
use Apache::lonnet;
use Apache::lonlocal;
use Apache::courseclassifier;
use Apache::lonacc;
use LONCAPA;
use LONCAPA::lonauthcgi;

sub handler {
    my ($r) = @_;
    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;
    if ($r->header_only) {
        return OK;
    }
    my $handle = &Apache::lonnet::check_for_valid_session($r);
    my $lonidsdir=$r->dir_config('lonIDsDir');
    if ($handle ne '') {
        &Apache::lonnet::transfer_profile_to_env($lonidsdir,$handle);
    }
    &Apache::lonacc::get_posted_cgi($r);
    &Apache::lonlocal::get_language_handle($r);
    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},
                                            ['sortby','showdom']);

    my $codedom = &Apache::lonnet::default_login_domain();

    if (($env{'user.domain'} ne '') && ($env{'user.domain'} ne 'public')) { 
        $codedom = $env{'user.domain'};
        if ($env{'request.role.domain'} ne '') {
            $codedom = $env{'request.role.domain'};
        }
    }
    my $formname = 'coursecatalog';
    if ($env{'form.showdom'} ne '') {
        $env{'form.showdom'} = &LONCAPA::clean_domain($env{'form.showdom'});
        if (&Apache::lonnet::domain($env{'form.showdom'}) ne '') {
            $codedom = $env{'form.showdom'};
        } else {
            $env{'form.showdom'} = '';
        }
    }
    my $domdesc = &Apache::lonnet::domain($codedom,'description');
    &Apache::lonhtmlcommon::clear_breadcrumbs();

    my $crscats = &Apache::lonnet::get_dom_cats($codedom);
    my %domdefaults = &Apache::lonnet::get_domain_defaults($codedom);
    my $knownuser = &user_is_known();
    my $canviewall = &canview_all($knownuser,$codedom);
    
    my ($cathash,$cattype);
    if (ref($crscats) eq 'HASH') {
        %{$cathash} = %{$crscats};
    } else {
        $cathash = {};
    }
    if ($knownuser || $canviewall) {
        $cattype = $domdefaults{'catauth'};
    } else {
        $cattype = $domdefaults{'catunauth'};
    }
    if ($cattype eq '') {
        $cattype = 'std';
    }
    if ($cattype eq 'none') {
        $r->print(&Apache::loncommon::start_page('Course/Community Catalog'));
        &Apache::lonhtmlcommon::add_breadcrumb
        ({href=>"/adm/coursecatalog",
          text=>"Course/Community Catalog"});
        $r->print(&Apache::lonhtmlcommon::breadcrumbs('Course/Community Catalog'));
        if ($knownuser || $canviewall) {
            $r->print('<div>'.&mt('No catalog of LON-CAPA courses/communities is provided for: [_1]',$domdesc).'</div>');
        } else {
            if ($domdefaults{'catauth'} eq 'none') {
                $r->print('<div>'.&mt('No catalog of LON-CAPA courses/communities is provided for: [_1]',$domdesc).'</div>');
            } else {
                $r->print('<div>'.&mt('The catalog of LON-CAPA courses/communities provided for: "[_1]" is only available to users who are logged in.',$domdesc).'</div>');
            }
        }      
        $r->print(&Apache::loncommon::end_page());
        return OK;
    }

    my $cnum;
    if ($cattype eq 'codesrch') {
        my ($uniquecode,$codemsg,$brtext);
        if ($env{'form.uniquecode'}) {
            $uniquecode = $env{'form.uniquecode'};
            $uniquecode =~ s/^\s+|\s+$//g;
        }
        my $js = '<script type="text/javascript">'."\n".
                 '// <![CDATA['."\n".
                 &courselink_javascript($r)."\n".
                 '// ]]>'."\n".
                 '</script>'."\n";
        $r->print(&Apache::loncommon::start_page('Search for a Course/Community',$js));
        if ($uniquecode =~ /^\w{6}$/) {
            &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>"/adm/coursecatalog",
              text=>"Course/Community Catalog"});
            $brtext = 'Search Result';
        } else {
            $brtext = 'Course/Community Catalog';
        }
        &Apache::lonhtmlcommon::add_breadcrumb
        ({href=>"/adm/coursecatalog",
          text=>"$brtext"});
        $r->print(&Apache::lonhtmlcommon::breadcrumbs('Course/Community Catalog'));
        $r->print(&coursesearch($codedom,$domdesc,$uniquecode));
        if ($uniquecode =~ /^\w{6}$/) {
            $r->print('<hr />');
            my $confname = $codedom.'-domainconfig';
            my %codes = &Apache::lonnet::get('uniquecodes',[$uniquecode],$codedom,$confname);
            if ($codes{$uniquecode}) {
                $cnum = $codes{$uniquecode};
                my %courses = &Apache::lonnet::courseiddump($codedom,'.',1,'.','.',
                                                            $cnum,undef,undef,'.',1);
                if (keys(%courses)) {
                    $env{'form.coursenum'} = $cnum;
                    my %courseinfo = &build_courseinfo_hash(\%courses,$knownuser,$codedom,$canviewall);
                    undef($env{'form.coursenum'});
                    if (ref($courseinfo{$codedom.'_'.$cnum}) eq 'HASH') {
                        $r->print(&Apache::lonhtmlcommon::start_pick_box());
                        my @cols = ('title','ownerlastnames','seclist','access');
                        my %lt = &Apache::lonlocal::texthash(
                                     title          => 'Title',
                                     ownerlastnames => 'Owner & Co-owner(s)',
                                     seclist        => 'Sections',
                                     access         => 'Default Access Dates for Students',
                                 );
                        my @shown;
                        foreach my $item (@cols) {
                            if ($courseinfo{$codedom.'_'.$cnum}{$item}) {
                                push(@shown,$item);
                            }
                        }
                        my $num = 0;
                        foreach my $item (@shown) {
                            $num ++;
                            $r->print(&Apache::lonhtmlcommon::row_title($lt{$item}).
                                      $courseinfo{$codedom.'_'.$cnum}{$item});
                            if ($item eq 'title') {
                                if ($courseinfo{$codedom.'_'.$cnum}{'showsyllabus'}) {
                                    my $usehttp = 0;
                                    if (($ENV{'SERVER_PORT'} == 443) && ($courseinfo{$codedom.'_'.$cnum}{'extsyllplain'})) {
                                        unless (&Apache::lonnet::uses_sts()) {
                                            $usehttp = 1;
                                        }
                                    }
                                    $r->print('&nbsp;<font size="-2">'.
                                              '<a href="javascript:ToSyllabus('."'$codedom','$cnum','$usehttp'".')">'.
                                              &mt('Syllabus').'</a></font>');
                                }
                            }
                            my $arg = ( $num == scalar(@shown) ? 1 : '' );
                            $r->print(&Apache::lonhtmlcommon::row_closure($arg));
                        }
                        $r->print(&Apache::lonhtmlcommon::end_pick_box());
                        my $selfenroll = &selfenroll_status($courseinfo{$codedom.'_'.$cnum},$codedom.'_'.$cnum);
                        if ($selfenroll) {
                            $r->print('<br />'.$selfenroll);
                        }
                        $r->print('<form name="linklaunch" method="post" action="">'."\n".
                                  '<input type="hidden" name="backto" value="coursecatalog" />'."\n".
                                  '<input type="hidden" name="courseid" value="" />'."\n".
                                  '</form>'."\n");

                    } else {
                        $codemsg = &mt('Code matched, but course ID to which this mapped is invalid.');
                    }
                } else {
                    $codemsg = &mt('Code matched, but course ID to which this mapped is invalid.');              
                } 
            } else {
                $codemsg = &mt('No match');
            }
        }
        $r->print('<br />'.&Apache::loncommon::end_page());
        return OK;
    } else {
        if ($env{'form.coursenum'} ne '') {
            if ($env{'form.coursenum'} =~ /^$LONCAPA::match_courseid$/) {
                $cnum = $env{'form.coursenum'};
            } else {
                delete($env{'form.coursenum'});
            }
        }
    }

    my (@cats,@trails,%allitems,%idx,@jsarray,%subcathash,$subcats,%maxd,
        $toplevelstr,$maxdepthstr);
    if ($env{'form.withsubcats'}) {
        $subcats = \%subcathash;
    }
    &Apache::loncommon::extract_categories($cathash,\@cats,\@trails,\%allitems,
                                           \%idx,\@jsarray,$subcats,\%maxd);
    if (ref($cats[0]) eq 'ARRAY') {
        foreach my $item (@{$cats[0]}) {
            $toplevelstr .= "'".&js_escape($item)."::0',";
            $maxdepthstr .= "'$maxd{$item}',";
        }
        $toplevelstr =~ s/,$//;
        $maxdepthstr =~ s/,$//;
    }
    &validate_input($codedom,\@cats,\%maxd,$cathash);
    my ($numtitles,@codetitles);
    if (($env{'form.coursenum'} ne '') && ($knownuser)) {
        &course_details($r,$codedom,$formname,$domdesc,\@trails,\%allitems,\@codetitles);
    } else {
        my ($catlinks,$has_subcats,$selitem) = &category_breadcrumbs($codedom,@cats);
        my $wasacctext = &get_wasactive_text();
        my $catjs = <<"ENDSCRIPT";

function setCatDepth(depth) {
    var depth = parseInt(depth);
    if (depth !== NaN) {
        if (depth > 0) {
            var possmaxd = 0;
            var toplevel = new Array($toplevelstr);
            var maxdepths = new Array($maxdepthstr);
            if (toplevel.length) {
                for (var i=0; i<toplevel.length; i++) {
                    var item = unescape(toplevel[i]);
                    if (item == document.coursecats.currcat_0.value) {
                        possmaxd = maxdepths[i];
                        break;
                    }
                }
            }
            if (depth > possmaxd) {
                depth = possmaxd;
            }
        }
        document.coursecats.catalog_maxdepth.value = depth;
    } else {
        document.coursecats.currcat_0.value = '';
        document.coursecats.catalog_maxdepth.value = '';
    }
    document.coursecats.submit();
    return;
}

function changeSort(caller) {
    document.$formname.sortby.value = caller;
    document.$formname.submit();
}

function setCourseId(caller) {
    document.$formname.coursenum.value = caller;
    document.$formname.submit();
}

ENDSCRIPT
        $catjs .= &courselink_javascript($r);
        if (&user_is_dc($codedom) || $canviewall) {
            $catjs .= <<ENDTOGGJS

function toggleStatuses() {
    if (document.$formname.showdetails.checked) {
        document.getElementById('statuschoice').style.display='block';
        document.getElementById('statuscell').style.borderLeft='1px solid'; 
    } else {
        document.getElementById('statuschoice').style.display='none';
        document.getElementById('statuscell').style.borderLeft='0px';
    }
    return;
}

function toggleWasActive() {
    if (document.getElementById('counts_Previous')) {
        if (document.getElementById('counts_Previous').checked) {
            document.getElementById('choosewasactive').style.display='block';
            document.getElementById('choosewasacctext').innerHTML = '$wasacctext';
        } else {
            document.getElementById('choosewasactive').style.display='none';
            document.getElementById('choosewasacctext').innerHTML = '';
        }
    }
    return;
}

ENDTOGGJS
        }
        if ($env{'form.currcat_0'} eq 'instcode::0') {
            $numtitles = &instcode_course_selector($r,$codedom,$formname,$domdesc,
                                                   $catlinks,$catjs,\@codetitles,$cattype,$canviewall);
            if ($env{'form.state'} eq 'listing') {
                $r->print(&print_course_listing($codedom,$numtitles,undef,undef,undef,
                                                \@codetitles,$canviewall));
            }
        } else {
            my (%add_entries);
            my ($currdepth,$deeper) = &get_depth_values();
            if ($selitem) {
                my $alert = &mt('Choose a subcategory to display');
                if (!$deeper) {
                    $alert = &mt('Choose a category to display');
                }
                &js_escape(\$alert);
                $catjs .= <<ENDJS;
function check_selected() {
    if (document.coursecats.$selitem.options[document.coursecats.$selitem.selectedIndex].value == "") {
        alert('$alert');
        return false;
    }
}
ENDJS
            }
            $catjs = '<script type="text/javascript">'."\n".$catjs."\n".'</script>';
            &cat_header($r,$codedom,$catjs,\%add_entries,$catlinks,undef,$cattype);
            if ($env{'form.currcat_0'} ne '') {
                $r->print('<form name="'.$formname.
                          '" method="post" action="/adm/coursecatalog">'.
                          &additional_filters($codedom,$has_subcats,$canviewall)."\n");
                $r->print('<input type="hidden" name="catalog_maxdepth" value="'.
                          $deeper.'" />'."\n");
                for (my $i=0; $i<$deeper; $i++) {
                    $r->print('<input type="hidden" name="currcat_'.$i.'" value="'.$env{'form.currcat_'.$i}.'" />'."\n");
                }
                my $display_button;
                if ($env{'form.currcat_0'} eq 'communities::0') {
                    $display_button = &mt('Display communities');
                } else {
                    $display_button = &mt('Display courses');
                }
                $r->print('<input type="hidden" name="coursenum" value="" />'."\n".
                          '<input type="hidden" name="sortby" value="" />'."\n".
                          '<input type="hidden" name="state" value="listing" />'."\n".
                          '<input type="hidden" name="showdom" value="'.
                          $env{'form.showdom'}.'" />'.
                          '<input type="submit" name="catalogfilter" value="'.
                          $display_button.'" /></form><br /><br />');
            }
            if ($env{'form.state'} eq 'listing') {
                $r->print(&print_course_listing($codedom,undef,\@trails,\%allitems,$subcats,\@codetitles,
                                                $canviewall));
            }
        }
    }
    $r->print('<br />'.&Apache::loncommon::end_page());
    return OK;
}

sub validate_input {
    my ($codedom,$cats,$maxd,$crscatsref) = @_;
    my $currcat = '';
    my $depth = 0;
    if ($env{'form.catalog_maxdepth'} ne '') {
        $env{'form.catalog_maxdepth'} =~ s{\D}{}g;
    }
    if ((ref($cats) eq 'ARRAY') && (ref($maxd) eq 'HASH')) {
        if (ref($cats->[0]) eq 'ARRAY') {
            if (@{$cats->[0]} == 1) {
                if ($cats->[0][0] eq 'instcode') {
                    $currcat = 'instcode::0';
                } elsif ($cats->[0][0] eq 'communities') {
                    $currcat = 'communities::0';
                } elsif ($cats->[0][0] eq 'placement') {
                    $currcat = 'placement::0';
                } else {
                    my $name = $cats->[0][0];
                    $currcat = &escape($name).'::0';
                }
                if (exists($maxd->{$cats->[0][0]})) {
                    if ($env{'form.catalog_maxdepth'} <= $maxd->{$cats->[0][0]}) {
                        $depth = $env{'form.catalog_maxdepth'};
                    } else {
                        $depth = $maxd->{$cats->[0][0]};
                    }
                }
            } elsif ((@{$cats->[0]} > 1) && ($env{'form.currcat_0'} ne '')) {
                my ($escname) = ($env{'form.currcat_0'} =~ /^([^:]+)\:\:0$/);
                if ($escname =~ /^instcode|communities|placement$/) {
                    $currcat = $env{'form.currcat_0'};
                    if (exists($maxd->{$escname})) {
                        if ($env{'form.catalog_maxdepth'} <= $maxd->{$escname}) {
                            $depth = $env{'form.catalog_maxdepth'};
                        } else {
                            $depth = $maxd->{$escname};
                        }
                    } else {
                        $depth = 1;
                    }
                } elsif ($escname ne '') {
                    my $name = &unescape($escname);
                    if (grep(/^\Q$name\E$/,@{$cats->[0]})) {
                        $currcat = $env{'form.currcat_0'};
                        if (exists($maxd->{$name})) {
                            if ($env{'form.catalog_maxdepth'} <= $maxd->{$name}) {
                                $depth = $env{'form.catalog_maxdepth'};
                            } else {
                                $depth = $maxd->{$name};
                            }
                        }
                    }
                }
            }
        }
    }
    $env{'form.currcat_0'} = $currcat;
    $env{'form.catalog_maxdepth'} = $depth;

    my %possibles = (
        backto => 1,
        catalogfilter => 1,
        catalog_maxdepth => 1,
        courseid => 1,
        coursenum => 1,
        currcat_0 => 1,
        numtitles => 1,
        Year => 1,
        Semester => 1,
        Department => 1,
        Number => 1,
        showdom => 1,
        sortby => 1,
        showcounts => 1,
        showdetails => 1,
        showhidden => 1,
        showselfenroll => 1,
        state => 1,
        uniquecode => 1,
        wasactive => 1,
        wasactiveon_day => 1,
        wasactiveon_month => 1,
        wasactiveon_year => 1,
        withsubcats => 1,
    );
    my %currcats;
    if ($env{'form.catalog_maxdepth'} > 0) {
        for (my $i=1; $i<=$env{'form.catalog_maxdepth'}; $i++) {
            $currcats{'currcat_'.$i} = 1;
            $possibles{'currcat_'.$i} = 1;
        }
    }

    foreach my $key (sort(keys(%env))) {
        if ($key =~ /^form\.(.+)$/) {
            unless ($possibles{$1}) {
                delete($env{$key});
            }
        }
    }
    if (exists($env{'form.backto'})) {
        unless ($env{'form.backto'} eq 'coursecatalog') {
            delete($env{'form.backto'});
        }
    }
    if (exists($env{'form.catalogfilter'})) {
        unless (($env{'form.catalogfilter'} eq &mt('Display courses')) ||
                ($env{'form.catalogfilter'} eq &mt('Display communities')) ||
                ($env{'form.catalogfilter'} eq &mt('Display placement tests'))) {
            delete($env{'form.catalogfilter'});
        }
    }
    if (exists($env{'form.courseid'})) {
        if ($env{'form.courseid'} ne '') {
            unless ($env{'form.courseid'} =~ /^\Q$codedom\E_$LONCAPA::match_courseid$/) {
                $env{'form.courseid'} = '';
            }
        }
    }
    if (exists($env{'form.coursenum'})) {
        unless ($env{'form.coursenum'} =~ /^$LONCAPA::match_courseid$/) {
            $env{'form.coursenum'} = '';
        }
    }
    if (exists($env{'form.currcat_0'})) {
        unless ($env{'form.currcat_0'} =~ /^(instcode|communities|placement)\Q::0\E$/) {
            if ($env{'form.currcat_0'} =~ /^.+\Q::0\E$/) {
                if (ref($crscatsref) eq 'HASH') {
                    unless (exists($crscatsref->{$env{'form.currcat_0'}})) {
                        delete($env{'form.currcat_0'});
                    }
                } else {
                    delete($env{'form.currcat_0'});
                }
            } else {
                delete($env{'form.currcat_0'});
            }
        }
    }
    if (exists($env{'form.numtitles'})) {
        unless ($env{'form.numtitles'} =~ /^\d+$/) {
            delete($env{'form.numtitles'});
        }
    }
    my ($gotcodes,%possvals);
    foreach my $item ('Year','Semester','Department','Number') {
        if (exists($env{'form.'.$item})) {
            unless ($env{'form.'.$item} eq '0') {
                unless ($gotcodes) {
                    $gotcodes = &get_instcode_allowable($codedom,\%possvals);
                }
                if (ref($possvals{$item}) eq 'HASH') {
                    unless (exists($possvals{$item}{$env{'form.'.$item}})) {
                        delete($env{'form.'.$item});
                    }
                } else {
                    delete($env{'form.'.$item});
                }
            }
        }
    }
    if (exists($env{'form.showdom'})) {
        unless ($env{'form.showdom'} =~ /^$LONCAPA::match_domain$/) {
            delete($env{'form.showdom'});
        }
    }
    if (exists($env{'form.sortby'})) {
        unless ($env{'form.sortby'} =~ /^(title|code|owner|cats)$/) {
            delete($env{'form.sortby'});
        }
    }
    if (exists($env{'form.showcounts'})) {
        if (ref($env{'form.showcounts'}) eq 'ARRAY') {
            foreach my $item (@{$env{'form.showcounts'}}) {
                unless ($item =~ /^(Active|Future|Previous)$/) {
                    delete($env{'form.showcounts'});
                    last;
                }
            }
        } else {
            unless ($env{'form.showcounts'} =~ /^(Active|Future|Previous)$/) {
                delete($env{'form.showcounts'});
            }
        }
    }
    foreach my $item ('showhidden','showdetails','showselfenroll','withsubcats') {
        if (exists($env{'form.'.$item})) {
            unless ($env{'form.'.$item} eq '1') {
                delete($env{'form.'.$item});
            }
        }
    }
    if (exists($env{'form.state'})) {
        unless ($env{'form.state'} eq 'listing') {
            delete($env{'form.state'});
        }
    }
    if (exists($env{'form.uniquecode'})) {
        unless ($env{'form.uniquecode'} =~ /^\w{6}$/) {
            delete($env{'form.uniquecode'});
        }
    }
    if (exists($env{'form.wasactive'})) {
        unless (($env{'form.wasactive'} eq 'accessend') || ($env{'form.wasactive'} eq 'enrollend')) {
            delete($env{'form.wasactive'});
        }
    }
    if (exists($env{'form.wasactiveon_day'})) {
        my $tmpday = $env{'form.wasactiveon_day'};
        unless (($tmpday =~ /^\d+$/) && ($tmpday > 0) && ($tmpday < 32)) {
            delete($env{'form.wasactiveon_day'});
        }
    }
    if (exists($env{'form.wasactiveon_month'})) {
        my $tmpmonth = $env{'form.wasactiveon_month'};
        unless (($tmpmonth =~ /^\d+$/) && ($tmpmonth > 0) && ($tmpmonth < 13)) {
            delete($env{'form.wasactiveon_month'});
        }
    }
    if (exists($env{'form.wasactiveon_year'})) {
        my $tmpyear = $env{'form.wasactiveon_year'};
        unless (($tmpyear =~ /^\d+$/) && ($tmpyear >= 1970)) {
            delete($env{'form.wasactiveon_year'});
        }
    }
    if (keys(%currcats)) {
        foreach my $key (keys(%currcats)) {
            if ($env{'form.'.$key} eq '') {
                delete($env{'form.'.$key});
            } elsif (ref($crscatsref) eq 'HASH') {
                unless (exists($crscatsref->{$env{'form.'.$key}})) {
                    delete($env{'form.'.$key});
                }
            } else {
                delete($env{'form.'.$key});
            }
        }
    }
    return;
}

sub get_instcode_allowable {
    my ($codedom,$possvalref) = @_;
    return 1 unless (ref($possvalref) eq 'HASH');
    my $instcats = &Apache::lonnet::get_dom_instcats($codedom);
    if (ref($instcats) eq 'HASH') {
        if ((ref($instcats->{'codetitles'}) eq 'ARRAY') && (ref($instcats->{'codes'}) eq 'HASH')) {
            my %codes = %{$instcats->{'codes'}};
            my @codetitles = @{$instcats->{'codetitles'}};
            my @standardnames = &Apache::loncommon::get_standard_codeitems();
            my %local_to_standard;
            for (my $i=0; $i<@codetitles; $i++) {
                $local_to_standard{$codetitles[$i]} = $standardnames[$i];
                $possvalref->{$standardnames[$i]} = {};
            }
            foreach my $cid (sort(keys(%codes))) {
                if (ref($codes{$cid}) eq 'HASH') {
                    foreach my $item (keys(%{$codes{$cid}})) {
                        $possvalref->{$local_to_standard{$item}}{$codes{$cid}{$item}} = 1;
                    }
                }
            }
        }
    }
    return 1;
}

sub course_details {
    my ($r,$codedom,$formname,$domdesc,$trails,$allitems,$codetitles) = @_;
    my $output;
    my %add_entries = (topmargin    => "0",
                       marginheight => "0",);
    my $js = '<script type="text/javascript">'."\n".
             &courselink_javascript($r).'</script>'."\n";
    my $start_page =
        &Apache::loncommon::start_page('Course/Community Catalog',$js,
                                       {'add_entries' => \%add_entries, });
    $r->print($start_page);
    if ($env{'form.numtitles'} > 0) {
        &Apache::lonhtmlcommon::add_breadcrumb
                ({href=>"/adm/coursecatalog",
                  text=>"Course/Community Catalog"});
    }
    my $brtextone = 'Course listing';
    my $brtexttwo = 'Course details';
    if ($env{'form.currcat_0'} eq 'communities::0') {
        $brtextone = 'Community listing';
        $brtexttwo = 'Community details';
    }
    &Apache::lonhtmlcommon::add_breadcrumb
             ({href=>"javascript:document.$formname.submit()",
              text=>$brtextone},
             {text=>$brtexttwo});
    $r->print(
        &Apache::lonhtmlcommon::breadcrumbs('Course/Community Catalog').
        '<h2>'.
        (($env{'form.currcat_0'} eq 'communities::0') ?
            &mt('Detailed community information:') :
            &mt('Detailed course information:')).
        '</h2>'.
        &print_course_listing($codedom,undef,$trails,$allitems,undef,$codetitles).
        '<br />'.
        '<form name="'.$formname.'" method="post" action="/adm/coursecatalog">'.
        &Apache::lonhtmlcommon::actionbox([
            '<a href = "javascript:document.coursecatalog.submit()">'.
            (($env{'form.currcat_0'} eq 'communities::0') ?
                &mt('Back to community listing') : &mt('Back to course listing')).
            '</a>'
        ]).
        &Apache::lonhtmlcommon::echo_form_input(['coursenum','catalogfilter',
                                                 'showdetails','courseid']).
        '</form>');
    return;
}

sub coursesearch {
    my ($codedom,$domdesc,$uniquecode) = @_;
    my %lt = &Apache::lonlocal::texthash (
                                           crlk => 'Course look-up',  
                                           code => 'Code',
                                           ifyo => 'Enter the course code (six letters and numbers)',
                                           srch => 'Find course',
                                         );
    return <<"END";
<h3>$lt{'crlk'} ($domdesc)</h3>
$lt{'ifyo'}
<form name="searchbycode" method="post" action="">
<span class="LC_nobreak">
<input type="text" value="$uniquecode" size="6" name="uniquecode" />&nbsp;
<br />
<input type="submit" value="$lt{'srch'}" name="srch" /></span>
</form>
END
}

sub courselink_javascript {
    my ($r) = @_;
    my $hostname = $r->hostname();    
    return <<"END";

function ToSyllabus(cdom,cnum,usehttp) {
    if (cdom == '' || cdom == null) {
        return;
    }
    if (cnum == '' || cnum == null) {
        return;
    }
    document.linklaunch.action = "/public/"+cdom+"/"+cnum+"/syllabus";
    if (usehttp == 1) {
        var hostname = '$hostname';
        if (hostname != '') {
            document.linklaunch.action = 'http://'+hostname+document.linklaunch.action;
        }
        document.linklaunch.action += '?usehttp=1';
    }
    document.linklaunch.submit();
}

function ToSelfenroll(courseid) {
    if (courseid == '') {
        return;
    }
    document.linklaunch.action = "/adm/selfenroll";
    document.linklaunch.courseid.value = courseid;
    document.linklaunch.submit();
}

END
}

sub instcode_course_selector {
    my ($r,$codedom,$formname,$domdesc,$catlinks,$catjs,$codetitles,$cattype,$canviewall) = @_;
    my %cat_titles = ();
    my %cat_order = ();
    my %cat_items = ();
    my %add_entries = (topmargin    => "0",
                       marginheight => "0",);
    my ($jscript,$totcodes,$numtitles,$lasttitle) = 
        &Apache::courseclassifier::instcode_selectors_data($codedom,$formname,
                           \%cat_items,$codetitles,\%cat_titles,\%cat_order);
    my $js = '<script type="text/javascript">'."\n".
             '// <![CDATA['."\n".
             "$jscript\n$catjs\n".
             '// ]]>'."\n".
             '</script>';
    if ($totcodes) {
        if (($env{'form.state'} eq 'listing') && ($numtitles > 0)) {
            $add_entries{'onload'} = 'setElements();';
        }
        if (&user_is_dc($codedom) || $canviewall) {
            $add_entries{'onload'} .= ' toggleStatuses();toggleWasActive();'
        }
        &cat_header($r,$codedom,$js,\%add_entries,$catlinks,$numtitles,$cattype);
        my $cat_maxdepth = $env{'form.catalog_maxdepth'};
        $r->print('<form name="'.$formname.'" method="post" action="/adm/coursecatalog">'.
                  '<input type="hidden" name="catalog_maxdepth" value="'.$cat_maxdepth.'" />'."\n".
                  '<input type="hidden" name="showdom" value="'.$env{'form.showdom'}.'" />'."\n".
                  '<input type="hidden" name="currcat_0" value="instcode::0" />'.
                  &additional_filters($codedom,undef,$canviewall));
        if ($numtitles > 0) {
            $r->print('<b>'.&mt('Choose which course(s) to list.').'</b><br />'.
                      &Apache::courseclassifier::build_instcode_selectors($numtitles,
                       $lasttitle,\%cat_items,$codetitles,\%cat_titles,\%cat_order));
        }
        $r->print('<input type="hidden" name="coursenum" value="" />'."\n".
                  '<input type="hidden" name="sortby" value="" />'."\n".
                  '<input type="hidden" name="state" value="listing" />'."\n".
                  '<input type="submit" name="catalogfilter" value="'.
                  &mt('Display courses').'" />'.
                  '<input type="hidden" name="numtitles" value="'.$numtitles.
                  '" /></form><br /><br />');
    } else {
        $js = '<script type="text/javascript">'."\n".
              '// <![CDATA['."\n".
              "$catjs\n".
              '// ]]>'."\n".
              '</script>';
        &cat_header($r,$codedom,$js,\%add_entries,$catlinks,$numtitles,$cattype);
        my $cat_maxdepth = $env{'form.catalog_maxdepth'};
        $r->print('<form name="'.$formname.'" method="post" action="/adm/coursecatalog">'.
                  '<input type="hidden" name="catalog_maxdepth" value="'.$cat_maxdepth.'" />'.
                  '<input type="hidden" name="showdom" value="'.$env{'form.showdom'}.'" />'.
                  '<input type="hidden" name="currcat_0" value="instcode::0" />');
        $r->print('<br />'.&mt('No official courses to display for [_1].',$domdesc).'</form>');
    }
    return $numtitles;
}

sub cat_header {
    my ($r,$codedom,$js,$add_entries,$catlinks,$numtitles,$cattype) = @_;
    my $start_page =
        &Apache::loncommon::start_page('Course/Community Catalog',$js,
                                       { 'add_entries' => $add_entries, });
    $r->print($start_page);
    my $brtext = 'Course listing';
    if ($env{'form.currcat_0'} eq 'communities::0') {
        $brtext = 'Community listing';
    }
    if ($env{'form.state'} eq 'listing') {
        if ($numtitles > 0) {
            &Apache::lonhtmlcommon::add_breadcrumb
                ({href=>"/adm/coursecatalog",
                  text=>"Course/Community Catalog"},
                 {text=>$brtext});
        } else {
            &Apache::lonhtmlcommon::add_breadcrumb
            ({text=>$brtext});
        }
    } else {
        &Apache::lonhtmlcommon::add_breadcrumb
        ({href=>"/adm/coursecatalog",
          text=>"Course/Community Catalog"});
    }
    $r->print(&Apache::lonhtmlcommon::breadcrumbs('Course/Community Catalog'));
    if ($cattype eq 'std') {
        my $onchange = 'this.form.submit()';
        $r->print('<form name="coursecatdom" method="post" action="/adm/coursecatalog">'.
                  '<table border="0"><tr><td><b>'.&mt('Domain:').'</b></td><td>'.
                  &Apache::loncommon::select_dom_form($codedom,'showdom','',1,$onchange));
        if (!$onchange) {
	   $r->print('&nbsp;<input type="submit" name="godom" value="'.&mt('Change').'" />');
        }
        $r->print('</td></tr></table></form>');
    }
    $r->print('<form name="coursecats" method="post" action="/adm/coursecatalog"'.
              ' onsubmit="return check_selected();">'.
              '<table border="0"><tr>'.$catlinks.'</tr></table></form>');
    return;
}

sub category_breadcrumbs {
    my ($dom,@cats) = @_;
    my $crumbsymbol = ' &#x25b6; ';
    my ($currdepth,$deeper) = &get_depth_values();
    my $currcat_str = 
        '<input type="hidden" name="catalog_maxdepth" value="'.$deeper.'" />'.
        '<input type="hidden" name="showdom" value="'.$dom.'" />';
    my $catlinks = '<td valign="top"><b>'.&mt('Catalog:').'</b></td><td><table><tr><td>';
    my $has_subcats;
    my $selitem;
    if (ref($cats[0]) eq 'ARRAY') {
        if (@{$cats[0]} == 0) {
            $catlinks .= &mt('No categories defined in this domain'); 
        } elsif (@{$cats[0]} == 1) {
            if ($cats[0][0] eq 'instcode') {
                $catlinks .= &mt('Official courses (with institutional codes)');
                $env{'form.currcat_0'} = 'instcode::0';
            } elsif ($cats[0][0] eq 'communities') {
                $catlinks .= &mt('Communities');
                $env{'form.currcat_0'} = 'communities::0';
            } else {
                my $name = $cats[0][0];
                my $item = &escape($name).'::0';
                $catlinks .= $name;
                $env{'form.currcat_0'} = $item;
            }
            $currcat_str .= '<input type="hidden" name="currcat_0" value="'.$env{'form.currcat_0'}.'" />';
        } else {
            $catlinks .= &main_category_selector(@cats);
            if (($env{'form.currcat_0'} ne '') && 
                ($env{'form.currcat_0'} ne 'instcode::0')) {
                $catlinks .= $crumbsymbol;
            }
        }
    } else {
        $catlinks .= &mt('Official courses (with institutional codes)');
                     $env{'form.currcat_0'} = 'instcode::0';
         $currcat_str .= '<input type="hidden" name="currcat_0" value="'.$env{'form.currcat_0'}.'" />';
    }
    if ($deeper) {
        for (my $i=1; $i<=$deeper; $i++) {
            my $shallower = $i-1;
            next if ($shallower == 0);
            my ($cat,$container,$depth) = map { &unescape($_); } split(/:/,$env{'form.currcat_'.$shallower});
            if ($cat ne '') {
                $catlinks .= '<td valign="top">'.
                             '<select name="currcat_'.$shallower.'" onchange="'.
                             'setCatDepth('."'$shallower'".');this.form.submit();">';
                if (ref($cats[$shallower]{$container}) eq 'ARRAY') {
                    $catlinks .= '<option value="">'.&mt('De-select').'</option>';
                    for (my $j=0; $j<@{$cats[$shallower]{$container}}; $j++) {
                        my $name = $cats[$shallower]{$container}[$j];
                        my $item = &escape($name).':'.&escape($container).':'.$shallower;
                        my $selected = '';
                        if ($item eq $env{'form.currcat_'.$shallower}) {
                            $selected = ' selected="selected"';
                        }
                        $catlinks .= 
                           '<option value="'.$item.'"'.$selected.'>'.$name.'</option>';
                    }
                }
                $catlinks .= '</select>';
            }
            unless ($i == $deeper) {
                $catlinks .= $crumbsymbol;
            } 
        }
        my ($cat,$container,$depth);
        if ($env{'form.currcat_'.$currdepth} eq '') {
            my $shallower = $currdepth - 1;
            ($cat,$container,$depth) = map { &unescape($_); } split(/:/,$env{'form.currcat_'.$shallower});
        } else {
            ($cat,$container,$depth) = map { &unescape($_); } split(/:/,$env{'form.currcat_'.$currdepth});
        }
        my $deeperlevel = $depth +1;
        if (ref($cats[$deeperlevel]{$cat}) eq 'ARRAY') {
            $has_subcats = 1;
            my $buttontext = &mt('Show subcategories');
            my $selitem = 'currcat_'.$deeperlevel;
            $catlinks .= '&nbsp;<select name="'.$selitem.'" onchange="this.form.submit()">';
            if (@{$cats[$deeperlevel]{$cat}}) {
                $catlinks .= '<option value="" selected="selected">'.
                             &mt('Subcategory ...').'</option>';
            }
            for (my $k=0; $k<@{$cats[$deeperlevel]{$cat}}; $k++) {
                my $name = $cats[$deeperlevel]{$cat}[$k];
                my $item = &escape($name).':'.&escape($cat).':'.$deeperlevel;
                $catlinks .= '<option value="'.$item.'">'.$name.'</option>'."\n";
            }
            $catlinks .= '</select>'."\n";
        } elsif ($cat ne 'instcode') {
            $catlinks .= '&nbsp;'.&mt('(No subcategories)');
        }
    } else {
        $selitem = 'currcat_0';
    }
    $catlinks .= $currcat_str.'</td></tr></table></td>';
    return ($catlinks,$has_subcats,$selitem);
}

sub main_category_selector {
    my (@cats) = @_;
    my $maincatlinks = '<select name="currcat_0" onchange="setCatDepth('."'0'".');this.form.submit();">'."\n";
    if (ref($cats[0]) eq 'ARRAY') {
        if (@{$cats[0]} > 1) {
            my $selected = '';
            if ($env{'form.currcat_0'} eq '') {
                $selected = ' selected="selected"';    
            }
            $maincatlinks .= 
                '<option value=""'.$selected.'>'.&mt('Select').'</option>'."\n";
        }
        for (my $i=0; $i<@{$cats[0]}; $i++) {
            my $name = $cats[0][$i];
            my $item = &escape($name).'::0';
            my $selected;
            if ($env{'form.currcat_0'} eq $item) {
                $selected = ' selected="selected"';
            }
            $maincatlinks .= '<option value="'.$item.'"'.$selected.'>';
            if ($name eq 'instcode') {
                $maincatlinks .= &mt('Official courses (with institutional codes)');
            } elsif ($name eq 'communities') {
                $maincatlinks .= &mt('Communities');
            } else {
                $maincatlinks .= $name;
            }
            $maincatlinks .= '</option>'."\n";
        }
        $maincatlinks .= '</select>'."\n";
    }
    return $maincatlinks;
}

sub get_depth_values {
    my $currdepth = 0;
    my $deeper = 0;
    if ($env{'form.catalog_maxdepth'} ne '') {
        $currdepth = $env{'form.catalog_maxdepth'};
        if ($env{'form.currcat_'.$currdepth} eq '') {
            $deeper = $currdepth;
        } else {
            $deeper = $currdepth + 1;
        }
    }
    return ($currdepth,$deeper);
}

sub additional_filters {
    my ($codedom,$has_subcats,$canviewall) = @_;
    my $is_dc = &user_is_dc($codedom);
    my $output = '<div class="LC_left_float">';
    if ($is_dc || $canviewall) {
        $output .= '<fieldset><legend>'.&mt('Options').'</legend>';
    }
    $output .= '<table><tr><td valign="top">';
    if (($env{'form.currcat_0'} ne 'instcode::0') &&
        ($env{'form.currcat_0'} ne '') && ($has_subcats)) {
        my $include_subcat_status;
        if ($env{'form.withsubcats'}) {
            $include_subcat_status = 'checked="checked" ';
        }
        my $counter = $env{'form.catalog_maxdepth'};
        if ($counter > 0) {
            if ($env{'form.state'} eq 'listing') {
                $counter --;
            } elsif ($env{'form.currcat_'.$counter} eq '') {
                $counter --;
            }
        }
        my ($catname) = split(/:/,$env{'form.currcat_'.$counter});
        if ($catname ne '') {
            $output .= '<span class="LC_nobreak"><label>'.
                       '<input type="checkbox" name="withsubcats" value="1" '.
                       $include_subcat_status.'/>'.
                       &mt('Include subcategories within "[_1]"',
                           &unescape($catname)).'</label></span><br />';
        }
    }
    my $show_selfenroll_status;
    if ($env{'form.showselfenroll'}) {
        $show_selfenroll_status = 'checked="checked" ';
    }
    my $selfenroll_text;
    if ($env{'form.currcat_0'} eq 'communities::0') {
        $selfenroll_text = &mt('Only show communities which currently allow self-enrollment (or will allow it in the future)');
    } else {
        $selfenroll_text = &mt('Only show courses which currently allow self-enrollment (or will allow it in the future)');
    }
    $output .= '<span class="LC_nobreak">'.
               '<label><input type="checkbox" name="showselfenroll" value="1" '.
               $show_selfenroll_status.'/>'.$selfenroll_text.
               '</label></span><br />';
    if ($is_dc || $canviewall) {
        my ($titlesref,$orderref) = &get_statustitles('filters');
        my $showdetails_status;
        if ($env{'form.showdetails'}) {
            $showdetails_status = 'checked="checked" ';
        }
        my $showhidden_status;
        if ($env{'form.showhidden'}) {
            $showhidden_status = 'checked="checked" ';
        }
        my @currstatuses = &Apache::loncommon::get_env_multiple('form.showcounts');
        my ($details_text,$hidden_text,$statusdisplay,$cellborder);
        my $wasactivedisplay = 'none';
        if ($env{'form.showdetails'}) {
            $statusdisplay = 'block';
            $cellborder = 'border-left: 1px solid;';
            if (grep(/^Previous$/,@currstatuses)) {
                $wasactivedisplay = 'block';
            }
        } else {
            $statusdisplay = 'none';
            $cellborder = 'border-left: 0px';
        }
        if ($env{'form.currcat_0'} eq 'communities::0') {
            $details_text = &mt('Show full details for each community (domain staff only)');
            $hidden_text = &mt('Include communities set to be hidden from catalog (domain staff only)');
        } else {
            $details_text = &mt('Show full details for each course (domain staff only)');
            $hidden_text = &mt('Include courses set to be hidden from catalog (domain staff only)');
        }
        if ($is_dc) {
            $output .= '<span class="LC_nobreak">'.
                       '<label><input type="checkbox" name="showhidden" value="1" '.
                       $showhidden_status.'/>'.$hidden_text.
                       '</label></span><br />'."\n";
        }
        $output .= '<span class="LC_nobreak">'.
                   '<label><input type="checkbox" name="showdetails" value="1" '.
                   $showdetails_status.'onclick="toggleStatuses();" />'.
                   $details_text.'</label></span></td>'."\n".
                   '<td id="statuscell" valign="top" style="'.$cellborder.'">'.
                   '<div id="statuschoice" style="display:'.$statusdisplay.'">';
        if (ref($orderref) eq 'ARRAY') {
            if (@{$orderref} > 0) {
                foreach my $type (@{$orderref}) {
                    my $checked;
                    if (grep(/^\Q$type\E$/,@currstatuses)) {
                        $checked = ' checked="checked"';
                    }
                    my $title;
                    if (ref($titlesref) eq 'HASH') {
                        $title = $titlesref->{$type};
                    }
                    unless ($title) {
                        $title = &mt($type);
                    }
                    my $onclick;
                    if ($type eq 'Previous') {
                        $onclick = ' onclick="toggleWasActive();"'; 
                    }
                    $output .= '<span class="LC_nobreak">'.
                               '<label><input type="checkbox" id="counts_'.$type.'"'.
                               ' name="showcounts" value="'.$type.'"'.$checked.$onclick.
                               ' />'.$title.'</label></span>';
                    if ($type eq 'Previous') {
                        my %milestonetext = &Apache::lonlocal::texthash (
                            accessend => 'immediately prior to default end access date',
                            enrollend => 'immediately prior to end date for auto-enrollment',
                            date      => 'immediately prior to specific date:',
                        );
                        my @statuses = &Apache::loncommon::get_env_multiple('form.showcounts');
                        $output .= '<span id="choosewasacctext" class="LC_nobreak">';
                        if ($checked) {
                            $output .= &get_wasactive_text();
                        }
                        $output .= '</span>'.
                                   '<div id="choosewasactive" style="display:'.$wasactivedisplay.'">'.
                                   '<table>';
                        my @milestones = ('accessend');
                        if (&Apache::lonnet::auto_run(undef,$codedom)) {
                            push(@milestones,'enrollend');
                        }
                        push(@milestones,'date');
                        foreach my $item (@milestones) {
                            my $checked;
                            if ($env{'form.state'} eq 'listing') {
                                if ($env{'form.wasactive'} eq $item) {
                                    $checked = ' checked="checked"';
                                }
                            } elsif ($item eq 'accessend') {
                                $checked = ' checked="checked"';
                            }
                            $output .=
                                '<tr><td width="10">&nbsp;</td><td>'.
                                '<span class="LC_nobreak"><label>'.
                                '<input type="radio" value="'.$item.'" name="wasactive"'.$checked.' />'.
                                $milestonetext{$item}.'</label></span>';
                            if ($item eq 'date') {
                                my $wasactiveon;
                                if (grep(/^Previous$/,@currstatuses)) {
                                    $wasactiveon =
                                        &Apache::lonhtmlcommon::get_date_from_form('wasactiveon');
                                } else {
                                    $wasactiveon = 'now';
                                }
                                $output .= ' '.
                                    &Apache::lonhtmlcommon::date_setter('coursecatalog',
                                                                        'wasactiveon',
                                                                        $wasactiveon,
                                                                        '','','',1,'',
                                                                        '','',1);
                            }
                            $output .= '</td></tr>';
                        }
                        $output .= '</table></div>';
                    }
                    $output .= '<br />';
                }
            }
        }
        $output .= '</div></td>';
    } else {
        $output .= '</td>';  
    }
    $output .= '</tr></table></fieldset></div>'.
               '<div style="clear:both;margin:0;"></div>';  
    return $output;
}

sub user_is_dc {
    my ($codedom) = @_;
    if (exists($env{'user.role.dc./'.$codedom.'/'})) {
        my $livedc = 1;
        my $now = time;
        my ($start,$end)=split(/\./,$env{'user.role.dc./'.$codedom.'/'});
        if ($start && $start>$now) { $livedc = 0; }
        if ($end   && $end  <$now) { $livedc = 0; }
        return $livedc;
    }
    return;
}

sub canview_all {
    my ($knownuser,$codedom) = @_;
    my $canviewall = 0;
    my $page = 'coursecatalog';
    if (&LONCAPA::lonauthcgi::can_view($page)) {
        $canviewall = 1;
    } elsif (&LONCAPA::lonauthcgi::check_ipbased_access($page)) {
        $canviewall= 1;
    } elsif (($knownuser) && ($codedom ne '')) {
        if (&Apache::lonnet::allowed('dcd',$codedom)) {
            $canviewall = 1;
        }
    }
    return $canviewall;
}

sub get_statustitles {
    my ($caller) = @_;
    my @status_order = ('Active','Future','Previous');
    my %status_title;
    if ($caller eq 'filters') {
        %status_title = &Apache::lonlocal::texthash(
                           Previous => 'Show count for past access',
                           Active => 'Show count for current student access',
                           Future => 'Show count for future student access',
                        );
        if ($env{'form.currcat_0'} eq 'communities::0') {
            $status_title{'Active'} = 'Show count for current member access';
            $status_title{'Future'} = 'Show count for future member access'; 
        }
    } else {    
        %status_title = &Apache::lonlocal::texthash(
                           Previous => 'Previous access',
                           Active => 'Current access',
                           Future => 'Future access',
                        );
    }
    return (\%status_title,\@status_order);
}

sub get_wasactive_text {
    my $wasacctext = ' -- ';
    if ($env{'form.currcat_0'} eq 'communities::0') {
        $wasacctext .= &mt('where member access status was current ...');
    } else {
        $wasacctext .= &mt('where student access status was current ...');
    }
    return $wasacctext;
}

sub search_official_courselist {
    my ($domain,$numtitles,$codetitles) = @_;
    my $instcode = &Apache::courseclassifier::instcode_search_str($domain,$numtitles,$codetitles);
    my $showhidden;
    if (&user_is_dc($domain)) {
        $showhidden = $env{'form.showhidden'};
    }
    my %courses = 
        &Apache::lonnet::courseiddump($domain,'.',1,$instcode,'.','.',undef,undef,
                                      'Course',1,$env{'form.showselfenroll'},undef,
                                      $showhidden,'coursecatalog');
    return %courses;
}

sub search_courselist {
    my ($domain,$subcats) = @_;
    my $cat_maxdepth = $env{'form.catalog_maxdepth'};
    my $filter = $env{'form.currcat_'.$cat_maxdepth};
    if (($filter eq '') && ($cat_maxdepth > 0)) {
        my $shallower = $cat_maxdepth - 1;
        $filter = $env{'form.currcat_'.$shallower};
    }
    my %courses;
    my $filterstr;
    if ($filter ne '') {
        if ($env{'form.withsubcats'}) {
            if (ref($subcats) eq 'HASH') {
                if (ref($subcats->{$filter}) eq 'ARRAY') {
                    $filterstr = join('&',@{$subcats->{$filter}});
                    if ($filterstr ne '') {
                        $filterstr = $filter.'&'.$filterstr;
                    }
                } else {
                    $filterstr = $filter;
                }
            } else {
                $filterstr = $filter;
            }  
        } else {
            $filterstr = $filter; 
        }
        my ($showhidden,$typefilter);
        if (&user_is_dc($domain)) {
            $showhidden = $env{'form.showhidden'};
        }
        if ($env{'form.currcat_0'} eq 'communities::0') {
            $typefilter = 'Community';
        } else {
            $typefilter = '.';
        }
        %courses = 
            &Apache::lonnet::courseiddump($domain,'.',1,'.','.','.',undef,undef,
                                          $typefilter,1,$env{'form.showselfenroll'},
                                          $filterstr,$showhidden,'coursecatalog');
    }
    return %courses;
}

sub print_course_listing {
    my ($domain,$numtitles,$trails,$allitems,$subcats,$codetitles,$canviewall) = @_;
    my $output;
    my %courses;
    my $knownuser = &user_is_known();
    my $canviewall = &canview_all();
    my $details = $env{'form.coursenum'};
    if (&user_is_dc($domain) || $canviewall) {
        if ($env{'form.showdetails'}) {
            $details = 1;
        }
    }
    if ($env{'form.coursenum'} ne '') {
        %courses = &Apache::lonnet::courseiddump($domain,'.',1,'.','.',
                                                 $env{'form.coursenum'},
                                                 undef,undef,'.',1);
        if (keys(%courses) == 0) {
            $output = '<p class="LC_error">';
            if ($env{'form.currcat_0'} eq 'communities::0') {
                $output .= &mt('The courseID provided does not match a community in this domain.');
            } else { 
                $output .= &mt('The courseID provided does not match a course in this domain.');
            }
            $output .= '</p>';
            return $output;
        }
    } else {
        if ($env{'form.currcat_0'} eq 'instcode::0') {
            %courses = &search_official_courselist($domain,$numtitles,$codetitles);
        } else {
            %courses = &search_courselist($domain,$subcats);
        }
        if (keys(%courses) == 0) {
            $output = '<p class="LC_info">';
            if ($env{'form.currcat_0'} eq 'communities::0') {
                $output .= &mt('No communities match the criteria you selected.');
            } else {
                $output .= &mt('No courses match the criteria you selected.');
            }
            $output .= '</p>';
            return $output;
        }
        if (($knownuser) && (!$env{'form.showdetails'}) && (!&user_is_dc($domain)) && (!$canviewall)) {
            $output = '<b>'.&mt('Note for students:').'</b> '
                     .&mt('If you are officially enrolled in a course but the course is not listed in your LON-CAPA courses, click the "Show more details" link for the specific course and check the default access dates and/or automated enrollment settings.')
                     .'<br /><br />';
        }
    }
    my $now = time;
    $output .= &construct_data_table($knownuser,$domain,\%courses,$details,undef,
                                     $now,$trails,$allitems,$canviewall);
    $output .= "\n".'<form name="linklaunch" method="post" action="">'.
               '<input type="hidden" name="backto" value="coursecatalog" />'.
               '<input type="hidden" name="courseid" value="" />'.
               &Apache::lonhtmlcommon::echo_form_input(['catalogfilter','courseid']).'</form>';
    return $output;
}

sub construct_data_table {
    my ($knownuser,$domain,$courses,$details,$usersections,$now,$trails,$allitems,$canviewall) = @_;
    my %sortname;
    if (($details eq '') || ($env{'form.showdetails'})) {
        $sortname{'Code'} = 'code';
        $sortname{'Categories'} = 'cats';
        $sortname{'Title'} = 'title';
        $sortname{'Owner & Co-owner(s)'} = 'owner';
    }
    my $output = &Apache::loncommon::start_data_table().
                 &Apache::loncommon::start_data_table_header_row();
    my @coltitles = ('Count');
    if ($env{'form.currcat_0'} eq 'instcode::0') {
        push(@coltitles,'Code');
    } else {
        push(@coltitles,'Categories');
    }
    push(@coltitles,('Sections','Crosslisted','Title','Owner & Co-owner(s)'));
    if (ref($usersections) eq 'HASH') {
       $coltitles[1] = 'Your Section';
    }
    foreach my $item (@coltitles) {
        $output .= '<th>';
        if (defined($sortname{$item})) {
            $output .= '<a href="javascript:changeSort('."'$sortname{$item}'".')">'.&mt($item).'</a>';
        } elsif ($item eq 'Count') {
            $output .= '&nbsp;&nbsp;';
        } else {
            $output .= &mt($item);
        }
        $output .= '</th>';
    }
    my (@fields,%fieldtitles,$wasactiveon);
    if ($knownuser || ($canviewall && $details)) {
        if ($details) {
            if ($env{'form.currcat_0'} eq 'communities::0') {
                $output .= '<th>'.&mt('Default Access Dates for Members').'</th>'.
                           '<th>'.&mt('Member Counts').'</th>';
            } else {
                $output .=
                    '<th>'.&mt('Default Access Dates for Students').'</th>'.
                    '<th>'.&mt('Student Counts').'</th>'.
                    '<th>'.&mt('Auto-enrollment of[_1]registered students','<br />').'</th>';
            }
            my ($titlesref,$orderref) = &get_statustitles();
            my @statuses;
            if (&user_is_dc($domain) || $canviewall) {
                @statuses = &Apache::loncommon::get_env_multiple('form.showcounts');
                if (grep(/^Previous$/,@statuses)) {
                    if ($env{'form.wasactive'} eq 'date') { 
                        $wasactiveon = 
                            &Apache::lonhtmlcommon::get_date_from_form('wasactiveon');
                    } else {
                        $wasactiveon = $env{'form.wasactive'};
                    }
                }
                if (ref($orderref) eq 'ARRAY') {
                    foreach my $status (@{$orderref}) {
                        if (grep(/^\Q$status\E$/,@statuses)) {
                            push(@fields,$status); 
                        }
                    }
                }
            } else {
                @fields = ('Active','Future');
            }
            foreach my $status (@fields) {
                my $title;
                if (ref($titlesref) eq 'HASH') {
                    $title = $titlesref->{$status};
                }
                unless ($title) {
                    $title = &mt($status);
                }
                $fieldtitles{$status} = $title;
            }
        } else {
            $output .= '<th>'.&mt('Details').'</th>';
        }
    }
    $output .= '<th>'.&mt('Self-enroll (if permitted)').'</th>';
    &Apache::loncommon::end_data_table_header_row();
    my (%numbers,%creditsum);
    my ($showcredits,$defofficial,$defunofficial,$deftextbook);
    my %domdefaults = &Apache::lonnet::get_domain_defaults($domain);
    unless ($env{'form.currcat_0'} eq 'communities::0') {
        if ($domdefaults{'officialcredits'} || $domdefaults{'unofficialcredits'} || $domdefaults{'textbookcredits'}) {
            $showcredits = 1;
            $defofficial = $domdefaults{'officialcredits'};
            $defunofficial = $domdefaults{'unofficialcredits'};
            $deftextbook = $domdefaults{'textbookcredits'};
        }
    }
    my %courseinfo = &build_courseinfo_hash($courses,$knownuser,$domain,$canviewall,$details,
                                            $usersections,\@fields,\%fieldtitles,
                                            $wasactiveon,\%numbers,\%creditsum,
                                            $showcredits,$defofficial,$defunofficial,$deftextbook);
    my %Sortby;
    foreach my $course (sort(keys(%{$courses}))) {
        if ($env{'form.sortby'} eq 'code') {
            push(@{$Sortby{$courseinfo{$course}{'code'}}},$course);
        } elsif ($env{'form.sortby'} eq 'cats') {
            push(@{$Sortby{$courseinfo{$course}{'categories'}}},$course);
        } elsif ($env{'form.sortby'} eq 'owner') {
            push(@{$Sortby{$courseinfo{$course}{'ownerlastnames'}}},$course);
        } else {
            my $clean_title = $courseinfo{$course}{'title'};
            $clean_title =~ s/\W+//g;
            if ($clean_title eq '') {
                $clean_title = $courseinfo{$course}{'title'};
            }
            push(@{$Sortby{$clean_title}},$course);
        }
    }
    my @sorted_courses;
    if (($env{'form.sortby'} eq 'code') || ($env{'form.sortby'} eq 'owner') ||
        ($env{'form.sortby'} eq 'cats')) {
        @sorted_courses = sort(keys(%Sortby));
    } else {
        @sorted_courses = sort { lc($a) cmp lc($b) } (keys(%Sortby));
    }
    my $count = 1;
    my $totalsec = 0;
    foreach my $item (@sorted_courses) {
        foreach my $course (@{$Sortby{$item}}) {
            $output.=&Apache::loncommon::start_data_table_row(); 
            $output.=&courseinfo_row($courseinfo{$course},$knownuser,$details,
                                     \$count,$now,$course,$trails,$allitems,\%numbers,$canviewall);
            $output.=&Apache::loncommon::end_data_table_row();
        }
    }
    if (($knownuser || $canviewall) && ($count > 1) && $env{'form.showdetails'}) {
        if (&user_is_dc($domain) || $canviewall) {
            my %lt = &Apache::lonlocal::texthash (
                                                     'Active'   => 'Total current students',
                                                     'Future'   => 'Total future students',
                                                     'Previous' => 'Total previous students',
                                                     'courses'  => 'Total unique codes and courses without codes',
                                                     'sections' => 'Total sections',
                                                     'xlists'   => 'Total cross-listings',
                                                 );
            if ($showcredits) {
                $lt{'cr_Active'} = &mt('Total current student credit hours');
                $lt{'cr_Future'} = &mt('Total future student credit hours');
                $lt{'cr_Previous'} = &mt('Total previous student credit hours');
            }
            if ($env{'form.currcat_0'} eq 'communities::0') {
                $lt{'courses'} = &mt('Total communities');
                $lt{'Active'} = &mt('Total current members'); 
                $lt{'Future'} = &mt('Total future members');
                $lt{'Previous'} = &mt('Total previous members');
            }
            my $colspan = 8;
            if ($showcredits) {
                $colspan = 4;
            }
            $output .= '<tr class="LC_footer_row">'.
                       '<td colspan="2">&nbsp;</td>'.
                       '<td colspan="'.$colspan.'">'.
                       '<table border="0">';
            foreach my $item ('courses','sections','xlists') {
                $output .= '<tr>'.
                           '<td>'.$lt{$item}.'</td><td>&nbsp;</td>'.
                           '<td align="right">'.$numbers{$item}.'</td>'.
                           '</tr>'."\n";
            }
            if (@fields > 0) { 
                foreach my $status (@fields) {
                    $output .= '<tr>'.
                               '<td>'.$lt{$status}.'</td><td>&nbsp;</td>'.
                               '<td align="right">'.$numbers{$status}.'</td>'.
                               '</tr>'."\n";
                }
            }
            $output .= '</table></td>';
            if ($showcredits) {
                $output .= '<td colspan="'.$colspan.'" valign="bottom"><table>';
                foreach my $status (@fields) {
                    $output .= '<tr>'.
                               '<td>'.$lt{'cr_'.$status}.'</td><td>&nbsp;</td>'.
                               '<td align="right">'.$creditsum{$status}.'</td></tr>';
                }
                $output .= '</table></td></tr>';
            }
        }
    }
    $output .= &Apache::loncommon::end_data_table();
    return $output;
}

sub build_courseinfo_hash {
    my ($courses,$knownuser,$domain,$canviewall,$details,$usersections,$fields,$fieldtitles,
        $wasactiveon,$numbers,$creditsum,$showcredits,$defofficial,$defunofficial) = @_;
    my %courseinfo;
    my $now = time;
    my $gettotals;
    if ((keys(%{$courses}) > 0) && (&user_is_dc($domain) || $canviewall) && ($details)) {
        $gettotals = 1;
    }
    my (%uniquecodes,$nocodes,$defcreds);
    foreach my $course (keys(%{$courses})) {
        my $descr;
        if (ref($courses->{$course}) eq 'HASH') {
            $descr = $courses->{$course}{'description'}; 
        }
        my $cleandesc=&HTML::Entities::encode($descr,'<>&"');
        $cleandesc=~s/'/\\'/g;
        $cleandesc =~ s/^\s+//;
        my ($cdom,$cnum)=split(/\_/,$course);
        my ($instcode,$singleowner,$ttype,$selfenroll_types,$extsyllplain,
            $selfenroll_start,$selfenroll_end,@owners,%ownernames,$categories);
        if (ref($courses->{$course}) eq 'HASH') {
            $descr = $courses->{$course}{'description'};
            $instcode =  $courses->{$course}{'inst_code'};
            $singleowner = $courses->{$course}{'owner'};
            $ttype =  $courses->{$course}{'type'};
            $selfenroll_types = $courses->{$course}{'selfenroll_types'};
            $selfenroll_start = $courses->{$course}{'selfenroll_start_date'};
            $selfenroll_end = $courses->{$course}{'selfenroll_end_date'};
            $categories = $courses->{$course}{'categories'};
            $extsyllplain = $courses->{$course}{'extsyllplain'};
            push(@owners,$singleowner);
            if ($courses->{$course}{'co-owners'} ne '') {
                foreach my $item (split(/,/,$courses->{$course}{'co-owners'})) {
                    push(@owners,$item);
                }
            }
        }
        if ($instcode ne '') {
            $uniquecodes{$instcode} = 1;
        } else {
            $nocodes ++;
        } 
        foreach my $owner (@owners) {
            my ($ownername,$ownerdom); 
            if ($owner =~ /:/) {
                ($ownername,$ownerdom) = split(/:/,$owner);
            } else {
                $ownername = $owner;
                if ($owner ne '') {
                    $ownerdom = $cdom;
                }
            }
            if ($ownername ne '' && $ownerdom ne '') {
                my %namehash=&Apache::loncommon::getnames($ownername,$ownerdom);
                $ownernames{$ownername.':'.$ownerdom} = \%namehash; 
            }
        }
        $courseinfo{$course}{'cdom'} = $cdom;
        $courseinfo{$course}{'cnum'} = $cnum;
        $courseinfo{$course}{'code'} = $instcode;
        my @lastnames;
        foreach my $owner (keys(%ownernames)) {
            if (ref($ownernames{$owner}) eq 'HASH') {
                push(@lastnames,$ownernames{$owner}{'lastname'});
            }
        }
        $courseinfo{$course}{'ownerlastnames'} = join(', ',sort(@lastnames));
        $courseinfo{$course}{'title'} = $cleandesc;
        $courseinfo{$course}{'owner'} = $singleowner;
        $courseinfo{$course}{'selfenroll_types'} = $selfenroll_types;
        $courseinfo{$course}{'selfenroll_start'} = $selfenroll_start;
        $courseinfo{$course}{'selfenroll_end'} = $selfenroll_end;
        $courseinfo{$course}{'categories'} = $categories;
        $courseinfo{$course}{'extsyllplain'} = $extsyllplain;

        my %coursehash = &Apache::lonnet::dump('environment',$cdom,$cnum);
        my @classids;
        my @crosslistings;
        my ($seclist,$numsec) = 
            &identify_sections($coursehash{'internal.sectionnums'});
        $courseinfo{$course}{'seclist'} = $seclist;
        my ($xlist_items,$numxlist) = 
            &identify_sections($coursehash{'internal.crosslistings'});
        if (ref($numbers) eq 'HASH') {
            $numbers->{'sections'} += $numsec; 
            $numbers->{'xlists'} += $numxlist;  
        }
        my $showsyllabus = 1; # default is to include a syllabus link
        if (defined($coursehash{'showsyllabus'})) {
            $showsyllabus = $coursehash{'showsyllabus'};
        }
        $courseinfo{$course}{'showsyllabus'} = $showsyllabus;
        if ($showcredits) {
            if ($coursehash{'internal.defaultcredits'}) {
                $courseinfo{$course}{'defaultcredits'} = $coursehash{'internal.defaultcredits'};
            } elsif ($instcode ne '') {
                $courseinfo{$course}{'defaultcredits'} = $defofficial;
            } else {
                $courseinfo{$course}{'defaultcredits'} = $defunofficial;
            }
            $defcreds = $courseinfo{$course}{'defaultcredits'};
        }
        if (((defined($env{'form.coursenum'}) && ($cnum eq $env{'form.coursenum'}))) ||
            (($knownuser || $canviewall) && ($details == 1))) {
            my $milestone;
            if ($wasactiveon eq 'accessend') {
                if ($coursehash{'default_enrollment_end_date'}) {
                    $milestone = $coursehash{'default_enrollment_end_date'};
                } else {
                    $milestone = time;
                }
            } elsif ($wasactiveon eq 'enrollend') {
                if ($coursehash{'internal.autoend'}) {
                    $milestone = $coursehash{'internal.autoend'};
                } else {
                    $milestone = time;
                }
            } else {
                $milestone = $wasactiveon;
            }
            $courseinfo{$course}{'counts'} =  
                &count_students($cdom,$cnum,$numsec,$fields,$fieldtitles,$gettotals,
                                $numbers,$creditsum,$showcredits,$defcreds,$milestone);
            if ($instcode ne '') {
                $courseinfo{$course}{'autoenrollment'} =
                    &autoenroll_info(\%coursehash,$now,$seclist,$xlist_items,
                                     $instcode,\@owners,$cdom,$cnum);
            }
            my $startaccess = '';
            my $endaccess = '';
            my $accessdates;
            if ( defined($coursehash{'default_enrollment_start_date'}) ) {
                $startaccess = &Apache::lonlocal::locallocaltime($coursehash{'default_enrollment_start_date'});
            }
            if ( defined($coursehash{'default_enrollment_end_date'}) ) {
                $endaccess = &Apache::lonlocal::locallocaltime($coursehash{'default_enrollment_end_date'});
                if ($coursehash{'default_enrollment_end_date'} == 0) {
                    $endaccess = &mt('No ending date');
                }
            }
            if ($startaccess) {
                $accessdates .= '<i>'.&mt('From:[_1]','</i> '.$startaccess).'<br />';
            }
            if ($endaccess) {
                $accessdates .= '<i>'.&mt('To:[_1]','</i> '.$endaccess).'<br />';
            }
            if (($selfenroll_types ne '') && 
                ($selfenroll_end > 0 && $selfenroll_end > $now)) {
                my ($selfenroll_start_access,$selfenroll_end_access);
                if (($coursehash{'default_enrollment_start_date'} ne 
                     $coursehash{'internal.selfenroll_start_access'}) ||
                   ($coursehash{'default_enrollment_end_date'} ne 
                    $coursehash{'internal.selfenroll_end_access'})) {
                    if ( defined($coursehash{'internal.selfenroll_start_access'}) ) {
                        $selfenroll_start_access = &Apache::lonlocal::locallocaltime($coursehash{'internal.selfenroll_start_access'});
                    }
                    if ( defined($coursehash{'default_enrollment_end_date'}) ) {
                        $selfenroll_end_access = &Apache::lonlocal::locallocaltime($coursehash{'internal.selfenroll_end_access'});
                        if ($coursehash{'internal.selfenroll_end_access'} == 0) {
                            $selfenroll_end_access = &mt('No ending date');
                        }
                    }
                    if ($selfenroll_start_access || $selfenroll_end_access) {
                        $accessdates .= '<br/><br /><i>'.&mt('Self-enrollers:').'</i><br />';
                        if ($selfenroll_start_access) {
                            $accessdates .= '<i>'.&mt('From:[_1]','</i> '.$selfenroll_start_access).'<br />';
                        }
                        if ($selfenroll_end_access) {
                            $accessdates .= '<i>'.&mt('To:[_1]','</i> '.$selfenroll_end_access).'<br />';
                        }
                    }
                }
            }
            $courseinfo{$course}{'access'} = $accessdates;
        }
        if ($xlist_items eq '') {
            $xlist_items = &mt('No');
        }
        $courseinfo{$course}{'xlist'} = $xlist_items;
    }
    if (ref($numbers) eq 'HASH') {
        $numbers->{'courses'} = $nocodes + scalar(keys(%uniquecodes));
    }
    return %courseinfo;
}

sub count_students {
    my ($cdom,$cnum,$numsec,$fieldsref,$titlesref,$getcounts,$numbers,$creditsum,
        $showcredits,$defcreds,$wasactiveon) = @_;
    my $countslist = '<span class="LC_nobreak">'.
                     &mt('[quant,_1,section,sections,No sections]',$numsec).'</span>';
    my (@fields,%titles,$showexpired);
    if ((ref($fieldsref) eq 'ARRAY') && (ref($titlesref) eq 'HASH') &&
        (ref($numbers) eq 'HASH')) {
        @fields = @{$fieldsref};
        %titles = %{$titlesref};
        if (grep(/^Previous$/,@fields)) {
            $showexpired = 1;
        }
    } else {
        return;
    }
    my $classlist = &Apache::loncoursedata::get_classlist($cdom,$cnum);
    my (%student_count,%credit_count);
    %student_count = (
                           Active   => 0,
                           Future   => 0,
                           Previous => 0,
                     );
    if ($showcredits) {
        %credit_count = (
                          Active   => 0,
                          Future   => 0,
                          Previous => 0,
                        );
    }
    my %idx;
    $idx{'status'} = &Apache::loncoursedata::CL_STATUS();
    $idx{'end'}    = &Apache::loncoursedata::CL_END();
    $idx{'credits'} = &Apache::loncoursedata::CL_CREDITS();
    while (my ($student,$data) = each(%$classlist)) {
        my $status = $data->[$idx{'status'}];
        my $credits = $data->[$idx{'credits'}];
        if ($credits eq '') {
            $credits = $defcreds;  
        }
        if ($status eq 'Expired') {
            if (($showexpired) &&
                ($data->[$idx{'end'}] >= $wasactiveon)) {
                $student_count{'Previous'} ++;
                if ($showcredits) {
                    $credit_count{'Previous'} += $credits; 
                }
            }
        } else {
            $student_count{$status} ++;
            if ($showcredits) {
                $credit_count{$status} += $credits;
            }
        }
    }
    if (@fields) {
        $countslist .= ':<br />';
        foreach my $status (@fields) {
            $countslist .= '<span class="LC_nobreak">'.$titles{$status}.': '.
                           $student_count{$status}.'</span><br />';
            $numbers->{$status} += $student_count{$status};
            if ($showcredits) {
                $creditsum->{$status} += $credit_count{$status};
            }
        }
    }
    return $countslist;
}

sub courseinfo_row {
    my ($info,$knownuser,$details,$countref,$now,$course,$trails,$allitems,$numbers,$canviewall) = @_;
    my ($cdom,$cnum,$title,$ownerlast,$code,$owner,$seclist,$xlist_items,
        $accessdates,$showsyllabus,$counts,$autoenrollment,$output,$categories,
        $extsyllplain);
    if (ref($info) eq 'HASH') {
        $cdom = $info->{'cdom'};
        $cnum = $info->{'cnum'};
        $title = $info->{'title'};
        $ownerlast = $info->{'ownerlastnames'};
        $code = $info->{'code'};
        $owner = $info->{'owner'};
        $seclist = $info->{'seclist'};
        $xlist_items = $info->{'xlist'};
        $accessdates = $info->{'access'};
        $counts = $info->{'counts'};
        $autoenrollment = $info->{'autoenrollment'};
        $showsyllabus = $info->{'showsyllabus'};
        $extsyllplain = $info->{'extsyllplain'};
        $categories = $info->{'categories'};
    } else {
        $output = '<td colspan="8">'.&mt('No information available for [_1].',
                                         $code).'</td>';
        return $output;
    }
    $output .= '<td>'.$$countref.'</td>';
    if ($env{'form.currcat_0'} eq 'instcode::0') {
        $output .= '<td>'.$code.'</td>';
    } else {
        my ($categorylist,@cats);
        if ($categories ne '') {
            @cats = split('&',$categories);
        }
        if ((ref($trails) eq 'ARRAY') && (ref($allitems) eq 'HASH')) {
            my @categories = map { $trails->[$allitems->{$_}]; } @cats;
            $categorylist = join('<br />',@categories);
        }
        if ($categorylist eq '') {
            $categorylist = '&nbsp;';
        }
        $output .= '<td>'.$categorylist.'</td>';
    }
    $output .= '<td>'.$seclist.'</td>'.
               '<td>'.$xlist_items.'</td>'.
               '<td>'.$title.'&nbsp;<font size="-2">';
    if ($showsyllabus) {
        my $usehttp = 0;
        if (($ENV{'SERVER_PORT'} == 443) && ($extsyllplain)) {
            unless (&Apache::lonnet::uses_sts()) {
                $usehttp = 1;
            }
        }
        $output .= '<a href="javascript:ToSyllabus('."'$cdom','$cnum','$usehttp'".')">'.&mt('Syllabus').'</a>';
    } else {
        $output .= '&nbsp;';
    }
    $output .= '</font></td>'.
               '<td>'.$ownerlast.'</td>';
    if (($knownuser) || ($canviewall && $details)) {
        if ($details) {
            $output .=
                '<td>'.$accessdates.'</td>'.
                '<td>'.$counts.'</td>';
            unless ($env{'form.currcat_0'} eq 'communities::0') {
                $output .= '<td>'.$autoenrollment.'</td>';
            }
        } else {
            $output .= "<td><a href=\"javascript:setCourseId('$cnum')\">".&mt('Show more details').'</a></td>';
        }
    }
    my $selfenroll = &selfenroll_status($info,$course);
    if ($selfenroll) {
        $output .= '<td>'.$selfenroll.'</td>';
    } else {
        $output .= '<td>&nbsp;</td>';
    }
    $$countref ++;
    return $output;
}

sub selfenroll_status {
    my ($info,$course) = @_;
    my $now = time;
    my $output;
    if (ref($info) eq 'HASH') {
        if ($info->{'selfenroll_types'}) {
            my $showstart = &Apache::lonlocal::locallocaltime($info->{'selfenroll_start'});
            my $showend = &Apache::lonlocal::locallocaltime($info->{'selfenroll_end'});
            if (($info->{'selfenroll_end'} > 0) && ($info->{'selfenroll_end'} > $now)) {
                if (($info->{'selfenroll_start'} > 0) && ($info->{'selfenroll_start'} > $now)) {
                    $output = &mt('Starts: [_1]','<span class="LC_cusr_emph">'.$showstart.'</span>').'<br />'.&mt('Ends: [_1]','<span class="LC_cusr_emph">'.$showend.'</span>');
                } else {
                    $output = '<a href="javascript:ToSelfenroll('."'$course'".')">'.
                              &mt('Enroll in course').'</a><br />';
                    if ($info->{'selfenroll_end'} == 0) {
                        $output .= &mt('Available permanently');
                    } elsif ($info->{'selfenroll_end'} > $now) {
                        $output .= &mt('Self-enrollment ends: [_1]','<span class="LC_cusr_emph">'.$showend.'</span>');
                    }
                }
            }
        }
    }
    return $output;
}

sub identify_sections {
    my ($seclist) = @_;
    my @secnums;
    if ($seclist =~ /,/) {
        my @sections = split(/,/,$seclist);
        foreach my $sec (@sections) {
            $sec =~ s/:[^:]*$//;
            push(@secnums,$sec);
        }
    } else {
        if ($seclist =~ m/^([^:]+):/) {
            my $sec = $1;
            if (!grep(/^\Q$sec\E$/,@secnums)) {
                push(@secnums,$sec);
            }
        }
    }
    @secnums = sort {$a <=> $b} @secnums;
    $seclist = join(', ',@secnums);
    my $numsec = @secnums;
    return ($seclist,$numsec);
}

sub get_valid_classes {
    my ($seclist,$xlist_items,$crscode,$owners,$cdom,$cnum) = @_;
    my $response;
    my (@sections,@xlists,%possclasses,%okclasses,%validations);
    @{$validations{'sections'}} = ();
    @{$validations{'xlists'}} = ();
    my $totalitems = 0;
    if ($seclist) {
        @sections = split(/,\s+/,$seclist);
        map { $possclasses{$crscode.$_} = 1; } @sections;
    }
    if ($xlist_items) {
        @xlists = split(/,\s+/,$xlist_items);
        map { $possclasses{$_} = 1; } @xlists;
    }
    my %okclasses = &Apache::lonnet::auto_validate_instclasses($cdom,$cnum,$owners,
                                                               \%possclasses);
    if (keys(%okclasses)) {
        foreach my $sec (@sections) {
            if ($okclasses{$crscode.$sec}) {
                if (!grep(/^\Q$sec$\E/,@{$validations{'sections'}})) {
                    push(@{$validations{'sections'}},$sec);
                    $totalitems ++;
                }
            }
        }
        foreach my $item (@xlists) {
            if ($okclasses{$item}) {
                if (!grep(/^\Q$item\E$/,@{$validations{'xlists'}})) {
                    push(@{$validations{'xlists'}},$item);
                    $totalitems ++;
                }
            }
        }
    }
    if ($totalitems > 0) {
        if (@{$validations{'sections'}}) {
            $response = &mt('Sections:').' '.
                        join(', ',@{$validations{'sections'}}).'<br />';
        }
        if (@{$validations{'xlists'}}) {
            $response .= &mt('Courses:').' '.
                        join(', ',@{$validations{'xlists'}});
        }
    }
    return $response;
}

sub autoenroll_info {
    my ($coursehash,$now,$seclist,$xlist_items,$code,$owners,$cdom,$cnum) = @_;
    my $autoenrolldates = &mt('Not enabled');
    if (defined($coursehash->{'internal.autoadds'}) && $coursehash->{'internal.autoadds'} == 1) {
        my ($autostart,$autoend);
        if ( defined($coursehash->{'internal.autostart'}) ) {
            $autostart = &Apache::lonlocal::locallocaltime($coursehash->{'internal.autostart'});
        }
        if ( defined($coursehash->{'internal.autoend'}) ) {
            $autoend = &Apache::lonlocal::locallocaltime($coursehash->{'internal.autoend'});
        }
        if ($coursehash->{'internal.autostart'} > $now) {
            if ($coursehash->{'internal.autoend'} && $coursehash->{'internal.autoend'} < $now) {
                $autoenrolldates = &mt('Not enabled');
            } else {
                my $valid_classes = 
                   &get_valid_classes($seclist,$xlist_items,$code,
                                      $owners,$cdom,$cnum);
                if ($valid_classes ne '') {
                    $autoenrolldates = &mt('Not enabled').'<br />'
                                      .&mt('Starts: [_1]',$autostart)
                                      .'<br />'.$valid_classes;
                }
            }
        } else {
            if ($coursehash->{'internal.autoend'} && $coursehash->{'internal.autoend'} < $now) {
                $autoenrolldates = &mt('Not enabled').'<br />'
                                  .&mt('Ended: [_1]',$autoend);
            } else {
                my $valid_classes = &get_valid_classes($seclist,$xlist_items,
                                                       $code,$owners,$cdom,$cnum);
                if ($valid_classes ne '') {
                    $autoenrolldates = &mt('Currently enabled').'<br />'.
                                       $valid_classes;
                }
            }
        }
    }
    return $autoenrolldates;
}

sub user_is_known {
    my $known = 0;
    if ($env{'user.name'} ne '' && $env{'user.name'} ne 'public' 
        && $env{'user.domain'} ne '' && $env{'user.domain'} ne 'public') {
        $known = 1;
    }
    return $known;
}

1;
