# The LearningOnline Network with CAPA
# displays the blocking status table
#
# $Id: lonblockingstatus.pm,v 1.14.2.4 2020/09/29 19:55:00 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
#
package Apache::lonblockingstatus;

use strict;
use Apache::Constants qw(:common);
use Apache::loncommon();
use Apache::lonnet;
use Apache::lonlocal;
use LONCAPA qw(:DEFAULT :match);

sub handler {
    my $r = shift;
    &Apache::loncommon::no_cache($r);
    &Apache::loncommon::content_type($r,'text/html');

    $r->send_http_header;
    return OK if $r->header_only;

    my (%activities,$activity,$origurl,$origsymb);
    map { $activities{$_} = 1; } ('boards','chat','com','blogs','groups','port','printout','docs','passwd');

    # determine what kind of blocking we want details for
    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},['activity','url','symb']);
    $activity = $env{'form.activity'};

    my $title = 'Communication Blocking Status Information';

    if (($activity eq 'docs') || ($activity eq 'printout') ||
        ($activity eq 'passwd')) {
        $title = 'Blocking Status Information';
        if ($activity eq 'docs') {
            $origurl = $env{'form.url'};
            $origsymb = $env{'form.symb'};
        }
    }
    $r->print(&Apache::loncommon::start_page($title,undef,
                                            {'only_body' => 1}));

    if (($activity eq '') || (!$activities{$activity})) {
        $r->print('<p class="LC_error">'.&mt('Error: unknown activity type blocked').'</p>');
    } elsif (($activity eq 'docs') && ($origurl eq '') && ($origsymb eq '')) {
        $r->print('<p class="LC_error">'.&mt('Error: could not determine what content was blocked from access').'</p>');
    } else {
        $r->print(&blockpage($activity,$origurl,$origsymb));
    }

    $r->print(&Apache::loncommon::end_page());

    return OK;
}

sub blockpage {
    my ($activity,$origurl,$origsymb) = @_;

    # in case of a portfolio block we need to determine the owner of the files
    # we're trying to look at. This information is passed via query string.
    my ($uname, $udom);

    if (($activity eq 'port') ||
        (($activity eq 'passwd') && ($env{'user.name'} eq 'public') && ($env{'user.domain'} eq 'public'))) {
        &Apache::loncommon::get_unprocessed_cgi(
            $ENV{'QUERY_STRING'}, ['udom', 'uname'] );

        ($uname, $udom) = ($env{'form.uname'}, $env{'form.udom'});
        if (($uname !~ /^$match_username$/) || ($udom !~ /^$match_domain$/)) {
            if ($activity eq 'port') {
                return '<span class="LC_error">'.
                       &mt('Information about the owner of the portfolio files you were trying to view was missing or invalid.').
                       '</span><br />'.
                       &mt('Without valid owner information, the reason why access is blocked can not be determined');
            } else {
                return '<span class="LC_error">'.
                       &mt('Information about the username and/or domain for which you were trying to reset a password was missing or invalid.').
                       '</span><br />'.
                       &mt('Without valid information, the reason why access is blocked can not be determined');
            }
        }
    }

    # retrieve start/end of possible active blocking
    my (%setters,$startblock,$endblock,$triggerblock);

    if ($activity eq 'docs') {
        my ($cdom,$cnum);
        if ($env{'request.course.id'}) {
            $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
            $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
        }
        my $cancheck;
        if (($cdom ne '') && ($cnum ne '')) {
            if ($origsymb =~ m{^uploaded/($match_domain)/($match_courseid)/}) {
                my ($crsdom,$crsnum) = ($1,$2);
                if (($cdom eq $crsdom) && ($cnum eq $crsnum)) {
                    $cancheck = 1;
                }
            } else {
                $cancheck = 1;
            }
        }
        if ($cancheck) {
            ($startblock,$endblock,$triggerblock) =
                &Apache::loncommon::blockcheck(\%setters,$activity,$cnum,$cdom,$origurl,1,$origsymb,'blockingstatus');
        } else {
            return '<p class="LC_info">'.&mt('Could not determine why access is blocked.').'</p>';
        }
    } else {
        ($startblock,$endblock,$triggerblock) =
            &Apache::loncommon::blockcheck(\%setters,$activity,$uname,$udom,$origurl,undef,$origsymb,'blockingstatus');
    }

    # nothing to do if there's no active blocking
    unless ($startblock && $endblock) {
        if ($activity eq 'docs') {
            return '<p class="LC_info">'.&mt('Content no longer blocked from access').'</p>';
        }
        return '<p class="LC_info">'.&mt('Access no longer blocked for this activity').'</p>';
    }

    # lookup $activity -> description
                   #possible activity          #corresponding description
    my %descs = (
                   boards     => 'Discussion posts in this course',
                   chat       => 'Chat Room',
                   com        => 'This message',
                   blogs      => 'Blogs',
                   groups     => 'Groups in this course',
                   printout   => 'Printout generation',
                   docs       => 'Course Content',
                   passwd     => 'Changing of passwords',
                );

    if ($activity eq 'groups' || $activity eq 'boards') {
        if (&Apache::loncommon::course_type() eq 'Community') {
            $descs{'boards'} = 'Discussion posts in this community';
            $descs{'groups'} = 'Groups in this community';
            $descs{'docs'} = 'Community Content';
        }
    }

    my $description = $descs{$activity};
    if ($activity eq 'port') {
        $description = &get_portfolio_category($uname,$udom,\%setters);
    }
    if ($description eq '') {
        $description = 'Communication';
    }

    my $showstart = Apache::lonlocal::locallocaltime($startblock);
    my $showend   = Apache::lonlocal::locallocaltime($endblock);

    my $output;

    if ( ref($description) ne 'ARRAY' ) {
        #default: $description is one of the above descriptions
        if ($activity eq 'docs') {
            $output=&mt( 'Access to the content page you are attempting to'
                         . ' view will be unavailable between [_1] and [_2] because'
                         . ' access to selected '.$description.' is being blocked.'
                         ,$showstart, $showend);
        } elsif (($activity eq 'printout') || ($activity eq 'passwd')) {
            $output = mt( $description
                          . ' will be unavailable between [_1] and [_2] because'
                          . ' this functionality is being blocked.'
                          ,$showstart, $showend);
        } else {
            $output = mt( $description
                          . ' will be inaccessible between [_1] and [_2] because'
                          . ' communication is being blocked.'
                          ,$showstart, $showend);
        }
    } else {
        # @$description is is the array returned from get_portfolio_category()
        # and contains the description (e.g. "Portfolio files belonging to [_1]"
        # and the value for [_1]
        $output = mt( $$description[0]
                      . ' will be inaccessible between [_2] and [_3] because'
                      . ' communication is being blocked.'
                     ,$$description[1], $showstart, $showend);
    }

    $output = "<p class=\"LC_info\">$output</p>";

    # show a table containing details, except if user is trying to look
    # at a different user's portfolio files
    if (   $activity ne 'port'                        # no portfolio
        || (   $uname eq $env{'user.name'}            # or own portfolio
            && $udom  eq $env{'user.domain'} ) 
        || Apache::lonnet::is_course($udom, $uname) ) # or portfolio of a course
    {
        $output .= &build_block_table(\%setters);
    }

    return $output;
}

sub build_block_table {
    my ($setters) = @_;
    my %lt = &Apache::lonlocal::texthash(
        'cacb' => 'Currently active communication/content blocks',
        'cour' => 'Course',
        'dura' => 'Duration',
        'blse' => 'Block set by'
    );
    my $output;
    $output  = Apache::loncommon::start_data_table()
             . Apache::loncommon::data_table_caption($lt{'cacb'})
             . Apache::loncommon::start_data_table_header_row()
             . "<th>$lt{'cour'}</th> <th>$lt{'dura'}</th> <th>$lt{'blse'}</th>"
             . Apache::loncommon::end_data_table_header_row();

    foreach my $course (keys(%{$setters})) {
        my %courseinfo=&Apache::lonnet::coursedescription($course);
        for (my $i=0; $i<@{$$setters{$course}{staff}}; $i++) {
            my ($uname,$udom) = @{$$setters{$course}{staff}[$i]};
            my $fullname = Apache::loncommon::plainname($uname,$udom);
            if (defined($env{'user.name'}) && defined($env{'user.domain'})
                && $env{'user.name'} ne 'public'
                && $env{'user.domain'} ne 'public') 
            {
                $fullname = Apache::loncommon::aboutmewrapper($fullname,$uname,$udom);
            }
            my $triggertype = $$setters{$course}{triggers}[$i];
            if ($triggertype) {
                $fullname .= &mt(' (triggered by you when starting timer)');
            }
            my ($openblock,$closeblock) = @{$$setters{$course}{times}[$i]};
            $openblock = &Apache::lonlocal::locallocaltime($openblock);
            $closeblock= &Apache::lonlocal::locallocaltime($closeblock);
            my $duration = mt('[_1] to [_2]', $openblock, $closeblock);
            $output .= Apache::loncommon::start_data_table_row()
                     . "<td>$courseinfo{'description'}</td>"
                     . "<td>$duration</td>"
                     . "<td>$fullname</td>"
                     . Apache::loncommon::end_data_table_row();
        }
    }
    $output .= Apache::loncommon::end_data_table();
}

sub get_portfolio_category {
    my ($uname, $udom, $setters) = @_;

    if ($uname eq $env{'user.name'} && $udom eq $env{'user.domain'}) {
        # user's portolfio files

        return 'Your portfolio files';

    } elsif (Apache::lonnet::is_course($udom, $uname)) {
        # group portfolio files

        my $coursedesc;

        foreach my $course (keys(%{$setters})) {
            my %courseinfo = Apache::lonnet::coursedescription($course);
            $coursedesc    = $courseinfo{'description'};
        }

        return ["Group portfolio in the course '[_1]'", $coursedesc];
        
    } else {
        # different user's portfolio files
        
        my $plainname = Apache::loncommon::plainname($uname, $udom);

        unless (   $env{'user.name'}   eq 'public' 
                && $env{'user.domain'} eq 'public' ) 
        {
            $plainname = Apache::loncommon::aboutmewrapper(
                            $plainname, $uname, $udom);
        }

        return ['Portfolio files belonging to [_1]', $plainname];
    }
}

1;
__END__
