#!/usr/bin/perl
# The LearningOnline Network
#
# $Id: refresh_courseids_db.pl,v 1.22 2020/04/22 16:43:56 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
#################################################

=pod

=head1 NAME

refresh_courseids_db.pl

=head1 SYNOPSIS

refresh_courseids_db.pl is run on a library server and gathers 
course information for each course for which the current server is
the home server.  Entries (excluding last access time) for each course 
in nohist_courseids.db are updated.   

=head1 DESCRIPTION

refresh_courseids_db.pl will update course information, apart 
from last access time, in nohist_courseids.db, using course data   
from each course's environment.db file.

=cut

#################################################

use strict;
use lib '/home/httpd/lib/perl/';
use Apache::lonnet;
use Apache::loncommon;
use Apache::lonuserstate;
use Apache::loncoursedata;
use Apache::lonnavmaps;
use Apache::lonrelrequtils;
use LONCAPA qw(:DEFAULT :match);

exit if ($Apache::lonnet::perlvar{'lonRole'} ne 'library');

#  Make sure this process is running from user=www
my $wwwid=getpwnam('www');
if ($wwwid!=$<) {
    my $emailto="$Apache::lonnet::perlvar{'lonAdmEMail'},$Apache::lonnet::perlvar{'lonSysEMail'}";
    my $subj="LON: $Apache::lonnet::perlvar{'lonHostID'} User ID mismatch";
    system("echo 'User ID mismatch. refresh_courseids_db.pl must be run as user www.' |\
 mail -s '$subj' $emailto > /dev/null");
    exit 1;
}
#
# Let people know we are running
open(my $fh,'>>'.$Apache::lonnet::perlvar{'lonDaemons'}.'/logs/refreshcourseids_db.log');
print $fh "==== refresh_courseids_db.pl Run ".localtime()."====\n";

my @domains = sort(&Apache::lonnet::current_machine_domains());
my @ids=&Apache::lonnet::current_machine_ids();

&Apache::lonrelrequtils::init_global_hashes();
my $globals_set = 1;

$env{'allowed.bre'} = 'F';

foreach my $dom (@domains) {
    $env{'user.domain'} = $dom;
    $env{'user.name'} = &Apache::lonnet::get_domainconfiguser($dom);
    my %courseshash;
    my %currhash = &Apache::lonnet::courseiddump($dom,'.',1,'.','.','.',1,\@ids,'.');
    my %lastaccess = &Apache::lonnet::courselastaccess($dom,undef,\@ids);
    my $dir = $Apache::lonnet::perlvar{lonUsersDir}.'/'.$dom;
    my %domdesign = &Apache::loncommon::get_domainconf($dom);
    my $autoassign = $domdesign{$dom.'.autoassign.co-owners'};
    &recurse_courses($dom,$dir,0,\%courseshash,\%currhash,\%lastaccess,$autoassign,$fh);
    foreach my $lonhost (keys(%courseshash)) {
        if (ref($courseshash{$lonhost}) eq 'HASH') {
            if (&Apache::lonnet::courseidput($dom,$courseshash{$lonhost},$lonhost,'notime') eq 'ok') {
                print $fh "nohist_courseids.db updated successfully for domain $dom on lonHostID $lonhost\n";
            } else {
                print $fh "Error occurred when updating nohist_courseids.db for domain $dom on lonHostID $lonhost\n";
            }
        }
    }
    delete($env{'user.name'});
    delete($env{'user.domain'});
}

delete($env{'allowed.bre'});

## Finished!
print $fh "==== refresh_courseids.db completed ".localtime()." ====\n";
close($fh);

sub recurse_courses {
    my ($cdom,$dir,$depth,$courseshash,$currhash,$lastaccess,$autoassign,$fh) = @_;
    next unless (ref($currhash) eq 'HASH');
    if (-d $dir) {
        opendir(DIR,$dir);
        my @contents = grep(!/^\./,readdir(DIR));
        closedir(DIR);
        $depth ++;
        foreach my $item (@contents) {
            if (($depth < 4) && (length($item) == 1)) {
                &recurse_courses($cdom,$dir.'/'.$item,$depth,$courseshash,
                                 $currhash,$lastaccess,$autoassign,$fh);
            } elsif ($item =~ /^$match_courseid$/) {
                my $cnum = $item;
                my $cid = $cdom.'_'.$cnum;
                unless (ref($currhash->{$cid}) eq 'HASH') {
                    my $is_course = 0;
                    if (-e "$dir/$cnum/passwd") {
                        if (open(my $pwfh,"<$dir/$cnum/passwd")) {
                            while (<$pwfh>) {
                                if (/^none:/) {
                                    $is_course = 1;
                                    last;
                                }
                            } 
                        }
                    }
                    next unless ($is_course);
                    my @stats = stat("$dir/$cnum/passwd");
                    print $fh "Course missing from nohist_courseids.db: $cid, created:".localtime($stats[9])."\n";
                }
                my %courseinfo=&Apache::lonnet::coursedescription($cid,{'one_time' => '1'});
                my %changes = ();
                my $crstype = $courseinfo{'type'};
                if ($crstype eq '') {
                    if ($cnum =~ /^$match_community$/) {
                        $crstype = 'Community';
                    } else {
                        $crstype = 'Course';
                    }
                    $changes{'type'} = $crstype;
                }
                my $chome = &Apache::lonnet::homeserver($cnum,$cdom);
                my $owner = $courseinfo{'internal.courseowner'};
                my $twodaysago = time - 172800;
                my (%roleshash,$gotcc,$reqdmajor,$reqdminor);
                if ($owner eq '') {
                    %roleshash = &Apache::lonnet::get_my_roles($cnum,$cdom,undef,undef,['cc'],undef,undef,1);
                    $gotcc = 1;
                    if (keys(%roleshash) == 1) {
                        foreach my $key (keys(%roleshash)) {
                            if ($key =~ /^($match_username\:$match_domain)\:cc$/) {
                                $owner = $1;
                                $changes{'internal.courseowner'} = $owner;
                            }
                        }
                    }
                } elsif ($owner !~ /:/) {
                    if ($owner =~ /^$match_username$/) {
                        my $ownerhome=&Apache::lonnet::homeserver($owner,$cdom);
                        unless (($ownerhome eq '') || ($ownerhome eq 'no_host')) {
                            $owner .= ':'.$cdom;
                            $changes{'internal.courseowner'} = $owner;
                        }
                    }
                }
                my $created = $courseinfo{'internal.created'};
                my $creator = $courseinfo{'internal.creator'};
                my $creationcontext = $courseinfo{'internal.creationcontext'};
                my $inst_code = $courseinfo{'internal.coursecode'};
                my $releaserequired = $courseinfo{'internal.releaserequired'};
                my $uniquecode = $courseinfo{'internal.uniquecode'};
                $inst_code = '' if (!defined($inst_code));
                $owner = '' if (!defined($owner));
                $uniquecode = '' if (!defined($uniquecode));
                if ($created eq '') {
                    if (ref($currhash->{$cid}) eq 'HASH') {
                        $created = $currhash->{$cid}{'created'};
                        $creator = $currhash->{$cid}{'creator'};
                        $creationcontext = $currhash->{$cid}{'context'};
                        unless ($created eq '') {
                            $changes{'internal.created'} = $created;
                        }
                        if ($creator =~ /^($LONCAPA::match_username):($LONCAPA::match_domain)$/) {
                             $changes{'internal.creator'} = $creator;
                        }
                        unless ($creationcontext eq '') {
                            $changes{'internal.creationcontext'} = $creationcontext;
                        }
                    }
                    if ($created eq '') {
                        if (-e "$dir/$cnum/passwd") {
                            my @stats = stat("$dir/$cnum/passwd");
                            $created = $stats[9];
                        }
                        if ($lastaccess->{$cid}) {
                            if ($created eq '') {
                                $created = $lastaccess->{$cid};
                            } elsif ($lastaccess->{$cid} < $created) {
                                $created = $lastaccess->{$cid};
                            }
                        }
                        unless ($created eq '') {
                            $changes{'internal.created'} = $created;
                        }
                    }
                }
                 
                if (($chome ne '')  && ($lastaccess->{$cid} > $twodaysago)) {
                    $env{'request.course.id'} = $cdom.'_'.$cnum;
                    $env{'request.role'} = 'cc./'.$cdom.'/'.$cnum;
                    $env{'request.role.adv'} = 1;

                    my $readmap = 1;
                    ($reqdmajor,$reqdminor) = &Apache::lonrelrequtils::get_release_req($cnum,$cdom,
                                                                                       $crstype,$readmap,
                                                                                       $globals_set);
                    delete($env{'request.role.adv'});
                    delete($env{'request.course.id'});
                    delete($env{'request.role'});
                } elsif ($releaserequired) {
                    ($reqdmajor,$reqdminor) = split(/\./,$releaserequired);
                }

                unless ($chome eq 'no_host') {
                    if (($lastaccess->{$cid} eq '') ||
                        ($lastaccess->{$cid} > $twodaysago)) {
                        my $contentchange;
                        if ($courseinfo{'internal.created'} eq '') {
                            $contentchange = &last_map_update($cnum,$cdom);
                        } else {
                            unless ($courseinfo{'internal.created'} > $lastaccess->{$cid}) {
                                $contentchange = &last_map_update($cnum,$cdom);
                            }
                        }
                        if (($contentchange) && ($contentchange > $courseinfo{'internal.contentchange'})) {
                            $changes{'internal.contentchange'} = $contentchange;
                        }
                    }
                    $courseshash->{$chome}{$cid} = {
                        description => $courseinfo{'description'},
                        inst_code   => $inst_code,
                        owner       => $owner,
                        type        => $crstype,
                    };
                    if ($creator ne '') {
                        $courseshash->{$chome}{$cid}{'creator'} = $creator;
                    }
                    if ($created ne '') {
                        $courseshash->{$chome}{$cid}{'created'} = $created;
                    }
                    if ($creationcontext ne '') {
                        $courseshash->{$chome}{$cid}{'context'} = $creationcontext;
                    }
                    if (($inst_code ne '') && ($autoassign)) {
                        unless ($gotcc) {
                            %roleshash = &Apache::lonnet::get_my_roles($cnum,$cdom,undef,undef,['cc'],undef,undef,1);
                        }
                        my @currcoowners;
                        my @newcoowners;
                        if ($courseinfo{'internal.co-owners'} ne '') {
                            @currcoowners = split(',',$courseinfo{'internal.co-owners'});
                        }
                        foreach my $key (keys(%roleshash)) {
                            if ($key =~ /^($match_username\:$match_domain)\:cc$/) {
                                my $cc = $1;
                                unless ($cc eq $owner) {
                                    my ($result,$desc) = &Apache::lonnet::auto_validate_instcode($cnum,$cdom,$inst_code,$cc);
                                    if ($result eq 'valid') {
                                        if (@newcoowners > 0) {
                                            unless (grep(/^\Q$cc\E$/,@newcoowners)) { 
                                                push(@newcoowners,$cc);
                                            }
                                        } else {
                                            push(@newcoowners,$cc);
                                        }
                                    }
                                }
                            }
                        }
                        my @diffs = &Apache::loncommon::compare_arrays(\@currcoowners,\@newcoowners);
                        if (@diffs > 0) {
                            if (@newcoowners > 0) {
                                $changes{'internal.co-owners'} = join(',',@newcoowners);
                                $courseshash->{$chome}{$cid}{'co-owners'} = $changes{'internal.co-owners'};
                            } else {
                                if ($courseinfo{'internal.co-owners'} ne '') {
                                    if (&Apache::lonnet::del('environment',['internal.co-owners'],$cdom,$cnum) eq 'ok') {
                                        print $fh "Former co-owner(s): $courseinfo{'internal.co-owners'} for official course: $inst_code (".$cdom."_".$cnum.") no longer active CCs, co-ownership status deleted.\n";
                                    }
                                } else {
                                    print $fh "Error occurred when updating co-ownership in course's environment.db for ".$cdom."_".$cnum."\n";
                                }
                            }
                        } elsif (@currcoowners > 0) {
                            $courseshash->{$chome}{$cid}{'co-owners'} = $courseinfo{'internal.co-owners'};
                        }
                    } elsif ($courseinfo{'internal.co-owners'} ne '') {
                        $courseshash->{$chome}{$cid}{'co-owners'} = $courseinfo{'internal.co-owners'};
                    }
                    foreach my $item ('categories','cloners','hidefromcat') {
                        if ($courseinfo{$item} ne '') {
                            $courseshash->{$chome}{$cid}{$item} = $courseinfo{$item}; 
                        }
                    }
                    foreach my $item ('selfenroll_types','selfenroll_start_date','selfenroll_end_date','uniquecode') {
                        if ($courseinfo{'internal.'.$item} ne '') {
                            $courseshash->{$chome}{$cid}{$item} =
                                $courseinfo{'internal.'.$item};
                        }
                    }
                    if ($reqdmajor eq '' && $reqdminor eq '') {
                        if ($courseinfo{'internal.releaserequired'} ne '') {
                            $changes{'internal.releaserequired'} = '';
                        }
                    } else {
                        my $releasereq =  $reqdmajor.'.'.$reqdminor;
                        $courseshash->{$chome}{$cid}{'releaserequired'} = $releasereq;
                        if ($courseinfo{'internal.releaserequired'} eq '') {
                            $changes{'internal.releaserequired'} = $releasereq;
                        } else {
                            if ($courseinfo{'internal.releaserequired'} ne $releasereq) {
                                $changes{'internal.releaserequired'} = $releasereq;
                            }
                        }
                    }
                    if (keys(%changes)) {
                        if (&Apache::lonnet::put('environment',\%changes,$cdom,$cnum) eq 'ok') {
                            print $fh "Course's environment.db for ".$cdom."_".$cnum." successfully updated with following entries: ";
                            foreach my $key (sort(keys(%changes))) {
                                print $fh "$key => $changes{$key} ";
                            }
                            print $fh "\n";
                        } else {
                            print $fh "Error occurred when updating course's environment.db for ".$cdom."_".$cnum."\n";
                        }
                    }
                }
            }
        }
    }
    return;
}

sub last_map_update {
    my ($cnum,$cdom) = @_;
    my $lastupdate = 0;
    my $path = &LONCAPA::propath($cdom,$cnum);
    if (-d "$path/userfiles") {
        if (opendir(my $dirh, "$path/userfiles")) {
            my @maps = grep(/^default_?\d*\.(?:sequence|page)$/,readdir($dirh));
            foreach my $map (@maps) {
                my $mtime = (stat("$path/userfiles/$map"))[9];
                if ($mtime > $lastupdate) {
                    $lastupdate = $mtime;
                }
            }
        }
    }
    return $lastupdate;
}

