# The LearningOnline Network with CAPA
# Authoring Space Directory Lister
#
# $Id: lonpubdir.pm,v 1.160.2.5 2019/08/13 19:27:43 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
###

package Apache::lonpubdir;

use strict;
use Apache::File;
use File::Copy;
use Apache::Constants qw(:common :http :methods);
use Apache::loncommon();
use Apache::lonhtmlcommon();
use Apache::londiff();
use Apache::lonlocal;
use Apache::lonmsg;
use Apache::lonmenu;
use Apache::lonnet;
use LONCAPA qw(:DEFAULT :match);

sub handler {

    my $r=shift;

    # Validate access to the construction space and get username:domain.

    my ($uname,$udom)=&Apache::lonnet::constructaccess($r->uri); 
    unless (($uname) && ($udom)) {
        return HTTP_NOT_ACCEPTABLE;
    }

# ----------------------------------------------------------- Start page output

    my $fn=$r->filename;
    $fn=~s/\/$//;
    my $thisdisfn=$fn;

    my $docroot=$r->dir_config('lonDocRoot');     # Apache  londocument root.
    if ($thisdisfn eq "$docroot/priv/$udom") {
        if ((-d "/home/$uname/public_html/") && (!-e "$docroot/priv/$udom/$uname")) {
            my ($version) = ($r->dir_config('lonVersion') =~ /^\'?(\d+\.\d+)\./);
            &Apache::loncommon::content_type($r,'text/html');
            $r->send_http_header;

            &Apache::lonhtmlcommon::clear_breadcrumbs();
            $r->print(&Apache::loncommon::start_page('Authoring Space').
                      '<div class="LC_error">'.
                      '<br /><p>'.
                      &mt('Your Authoring Space is currently in the location used by LON-CAPA version 2.10 and older, but your domain is using a newer LON-CAPA version ([_1]).',$version).'</p>'.
                      '<p>'.
                      &mt('Please ask your Domain Coordinator to move your Authoring Space to the new location.').
                      '</p>'.
                      '</div>'.
                      &Apache::loncommon::end_page());
            return OK;
        }
    }
    $thisdisfn=~s/^\Q$docroot\E\/priv//;

    my $resdir=$docroot.'/res'.$thisdisfn; # Resource directory
    my $targetdir='/res'.$thisdisfn; # Publication target directory.
    my $linkdir='/priv'.$thisdisfn;      # Full URL name of constr space.

    my %bombs=&Apache::lonmsg::all_url_author_res_msg($uname,$udom);

    my $londocroot = $Apache::lonnet::perlvar{'lonDocRoot'};
    my $current_disk_usage = &Apache::lonnet::diskusage($udom,$uname,
                                "$londocroot/priv/$udom/$uname"); # expressed in kB
    my $disk_quota = &Apache::loncommon::get_user_quota($uname,$udom,
                                                 'author'); # expressed in MB

    # Put out the start of page.
    &startpage($r, $uname, $udom, $thisdisfn, $current_disk_usage, $disk_quota);

    if (!-d $fn) {
        if (-e $fn) {
            $r->print('<p class="LC_info">'.&mt('Requested item is a file not a directory.').'</p>');
        } else {
            $r->print('<p class="LC_info">'.&mt('The requested subdirectory does not exist.').'</p>');
        }
        $r->print(&Apache::loncommon::end_page());
        return OK;
    }
    my @files;
    if (opendir(DIR,$fn)) {
        @files = grep(!/^\.+$/,readdir(DIR));
        closedir(DIR);
    } else {
        $r->print('<p class="LC_error">'.&mt('Could not open directory.').'</p>');
        $r->print(&Apache::loncommon::end_page());
        return OK;
    }

    # Put out actions for directory, browse/upload + new file page.
    &dircontrols($r,$uname,$udom,$thisdisfn, $current_disk_usage, $disk_quota);
    &resourceactions($r,$uname,$udom,$thisdisfn); # Put out form used for printing/deletion etc.

    my $numdir = 0;
    my $numres = 0;
  
    if ((@files == 0) && ($thisdisfn =~ m{^/$match_domain/$match_username})) {
        if ($thisdisfn =~ m{^/$match_domain/$match_username$}) {
            $r->print('<p class="LC_info">'.&mt('This Authoring Space is currently empty.').'</p>');
        } else {
            $r->print('<p class="LC_info">'.&mt('This subdirectory is currently empty.').'</p>');
        }
        $r->print(&Apache::loncommon::end_page());
        return OK;
    }

    # Retrieving value for "sortby" and "sortorder" from QUERY_STRING
    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},
        ['sortby','sortorder']);

    # Sort by name as default, not reversed
    if (! exists($env{'form.sortby'})) { $env{'form.sortby'} = 'filename' }
    if (! exists($env{'form.sortorder'})) { $env{'form.sortorder'} = '' }
    my $sortby = $env{'form.sortby'};
    my $sortorder = $env{'form.sortorder'};

    # Order in which columns are displayed from left to right
    my @order = ('filetype','actions','filename','title',
                    'pubstatus','cmtime','size');

    # Up and down arrows to indicate sort order
    my @arrows = ('&nbsp;&#9650;','&nbsp;&#9660;','');

    # Default sort order and column title
    my %columns = (
        filetype =>     {
                            order => 'ascending',
                            text  => &mt('Type'),
                        },
        actions =>      {
                            # Not sortable
                            text  => &mt('Actions'),
                        },
        filename =>     {
                            order => 'ascending',
                            text  => &mt('Name'),
                        },
        title =>        {
                            order => 'ascending',
                            text  => &mt('Title'),
                        },
        pubstatus =>    {
                            order => 'ascending',
                            text  => &mt('Status'),
                            colspan => '2',
                        },
        cmtime =>       {
                            order => 'descending',
                            text  => &mt('Last Modified'),
                        },
        size =>         {
                            order => 'ascending',
                            text  => &mt('Size').' (kB)',
                        },
    ); 

    # Print column headers
    my $output = '';
    foreach my $key (@order) {
        my $idx;
        # Append an up or down arrow to sorted column
        if ($sortby eq $key) {
            $idx = ($columns{$key}{order} eq 'ascending') ? 0:1;
            if ($sortorder eq 'rev') { $idx ++; }
            $idx = $idx%2;
        } else { $idx = 2; } # No arrow if column is not sorted
        $output .= (($columns{$key}{order}) ?
            '<th'.($columns{$key}{colspan} ? ' colspan="'.$columns{$key}{colspan}.'"' : '')
            .'><a href="'.$linkdir.'/?sortby='.$key.'&amp;sortorder='
            .((($sortby eq $key) && ($sortorder ne 'rev')) ? 'rev' : '').'">'
            .$columns{$key}{text}.$arrows[$idx].'</a></th>' :
            '<th>'.$columns{$key}{text}.'</th>');
    }
    $r->print(&Apache::loncommon::start_data_table()
        .&Apache::loncommon::start_data_table_header_row() . $output
        .&Apache::loncommon::end_data_table_header_row()
    );

    my $dirptr=16384;		# Mask indicating a directory in stat.cmode.
    my $filehash = {};
    foreach my $filename (@files) {
        # Skip .DS_Store, .DAV and hidden files
        my ($extension) = ($filename=~/\.(\w+)$/);
        next if (($filename eq '.DS_Store')
                || ($filename eq '.DAV')
                || (&Apache::loncommon::fileembstyle($extension) eq 'hdn')
                || ($filename =~ /^\._/));

        my ($cmode,$csize,$cmtime)=(stat($fn.'/'.$filename))[2,7,9];
        my $linkfilename = &HTML::Entities::encode('/priv'.$thisdisfn.'/'.$filename,'<>&"');
        # Identify type of file according to icon used
        my ($filetype) = (&Apache::loncommon::icon($filename) =~ m{/(\w+).gif$}); 
        my $cstr_dir = $r->dir_config('lonDocRoot').'/priv'.$thisdisfn;
        my $meta_same = &isMetaSame($cstr_dir, $resdir, $filename);
        
        # Store size, title, and status for files but not directories
        my $size = (!($cmode&$dirptr)) ? $csize/1024. : 0;
        my ($status, $pubstatus, $title, $fulltitle);
        if (!($cmode&$dirptr)) {
            ($status, $pubstatus) = &getStatus($resdir, $targetdir, $cstr_dir, 
                $filename, $linkfilename, $cmtime, $meta_same);
            ($fulltitle, $title) = &getTitle($resdir, $targetdir, $filename, 
                                        $linkfilename, $meta_same, \%bombs);
        } else {
            ($status, $pubstatus) = ('','');
            ($fulltitle, $title) = ('','');
        }

        # This hash will allow sorting
        $filehash->{ $filename } = {
            "cmtime"            => $cmtime,
            "size"              => $size,
            "cmode"             => $cmode,
            "filetype"          => $filetype,
            "title"             => $title,
            "fulltitle"         => $fulltitle,
            "status"            => $status,
            "pubstatus"         => $pubstatus,
            "linkfilename"      => $linkfilename,
        }
    }
   
    my @sorted_files;
    # Sorting by something other than "Name".  Name is the secondary key.
    if ($sortby =~ m{cmtime|size}) {    # Numeric fields
        # First check if order should be reversed
        if ($sortorder eq "rev") {
            @sorted_files = sort {
                $filehash->{$a}->{$sortby} <=> $filehash->{$b}->{$sortby}
                    or
                uc($a) cmp uc($b)
            } (keys(%{$filehash}));
        } else {
            @sorted_files = sort {
                $filehash->{$b}->{$sortby} <=> $filehash->{$a}->{$sortby}
                    or
                uc($a) cmp uc($b)
            } (keys(%{$filehash}));
        }
    } elsif ($sortby =~ m{filetype|title|status}) {     # String fields
        if ($sortorder eq "rev") {
            @sorted_files = sort {
                $filehash->{$b}->{$sortby} cmp $filehash->{$a}->{$sortby}
                    or
                uc($a) cmp uc($b)
            } (keys(%{$filehash}));
        } else {
            @sorted_files = sort {
                $filehash->{$a}->{$sortby} cmp $filehash->{$b}->{$sortby}
                    or
                uc($a) cmp uc($b)
            } (keys(%{$filehash}));
        }

    # Sort by "Name" is the default
    } else { 
        if ($sortorder eq "rev") {
            @sorted_files = sort {uc($b) cmp uc($a)} (keys(%{$filehash}));
        } else {
            @sorted_files = sort {uc($a) cmp uc($b)} (keys(%{$filehash}));
        }
    }

    # Print the sorted resources
    foreach my $filename (@sorted_files) {
        if ($filehash->{$filename}->{"cmode"}&$dirptr) {        # Directories
            &putdirectory($r, $thisdisfn, $linkdir, $filename, 
                $filehash->{$filename}->{"cmtime"}, 
                $targetdir, \%bombs, \$numdir);
        } else {                                                # Files
            &putresource($r, $udom, $uname, $filename, $thisdisfn, $resdir,
                $targetdir, $linkdir, $filehash->{$filename}->{"cmtime"}, 
                $filehash->{$filename}->{"size"}, \$numres, 
                $filehash->{$filename}->{"linkfilename"},
                $filehash->{$filename}->{"fulltitle"},
                $filehash->{$filename}->{"status"},
                $filehash->{$filename}->{"pubstatus"});
        }
    }

    $r->print( &Apache::loncommon::end_data_table()
        .&Apache::loncommon::end_page() );

    return OK;
}



#   Output the header of the page.  This includes:
#   - The HTML header 
#   - The H1/H3  stuff which includes the directory.
#
#     startpage($r, $uame, $udom, $thisdisfn, $current_disk_usage, $disk_quota);
#      $r     - The apache request object.
#      $uname - User name.
#      $udom  - Domain name the user is logged in under.
#      $thisdisfn - Displayable version of the filename.
#      $current_disk_usage - User's current disk usage (in kB).
#      $disk_quota - Disk quota for user's authoring space (in MB).
#      $crstype - Course type, if this is for "course author"

sub startpage {
    my ($r, $uname, $udom, $thisdisfn, $current_disk_usage, $disk_quota) = @_;
    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;

    my $formaction='/priv'.$thisdisfn.'/';
    $formaction=~s|/+|/|g;
    &Apache::lonhtmlcommon::store_recent('construct',$formaction,$formaction);

    &Apache::lonhtmlcommon::clear_breadcrumbs();
    &Apache::lonhtmlcommon::add_breadcrumb({
        'text'  => 'Authoring Space',
        'href'  => &Apache::loncommon::authorspace($formaction),
    });
    # breadcrumbs (and tools) will be created 
    # in start_page->bodytag->innerregister

    $env{'request.noversionuri'}=$formaction;
    $r->print(&Apache::loncommon::start_page('Authoring Space'));

    my $londocroot = $Apache::lonnet::perlvar{'lonDocRoot'};
    my $current_disk_usage = &Apache::lonnet::diskusage($udom,$uname,"$londocroot/priv/$udom/$uname");
    my $disk_quota = &Apache::loncommon::get_user_quota($uname,$udom,'author'); #expressed in MB
    $disk_quota = 1024 * $disk_quota; # convert from MB to kB

    $r->print(&Apache::loncommon::head_subbox(
                     '<div style="float:right;padding-top:0;margin-top;0">'
                    .&Apache::lonhtmlcommon::display_usage($current_disk_usage,
                                                           $disk_quota,'authoring')
                    .'</div>'
                    .&Apache::loncommon::CSTR_pageheader()));

    my $esc_thisdisfn = &Apache::loncommon::escape_single($thisdisfn);
    my $doctitle = 'LON-CAPA '.&mt('Authoring Space');
    my $newname = &mt('New Name');
    my $pubdirscript=(<<ENDPUBDIRSCRIPT);
<script type="text/javascript">
top.document.title = '$esc_thisdisfn/ - $doctitle';
// Store directory location for menu bar to find

parent.lastknownpriv='/priv$esc_thisdisfn/';

// Confirmation dialogues

    function currdiract(theform) {
        if (theform.dirtask.options[theform.dirtask.selectedIndex].value == 'publish') {
            document.publishdir.filename.value = theform.filename.value;
	    document.publishdir.submit();
        }
        if (theform.dirtask.options[theform.dirtask.selectedIndex].value == 'editmeta') {
            top.location=theform.filename.value+'default.meta'
        }
        if (theform.dirtask.options[theform.dirtask.selectedIndex].value == 'printdir' ) {
            document.printdir.postdata.value=theform.filename.value
            document.printdir.submit();
        }
        if (theform.dirtask.options[theform.dirtask.selectedIndex].value == "delete") {
              var delform = document.delresource
              delform.filename.value = theform.filename.value
              delform.submit()
        }
    }
  
    function checkUpload(theform) {
        if (theform.file == '') {
            alert("Please use 'Browse..' to choose a file first, before uploading")
            return 
        }
        theform.submit()  
    }

    function SetPubDir(theform,printForm) {
        if (theform.diraction.options[theform.diraction.selectedIndex].value == "open") {
            top.location = theform.openname.value
            return
        }
        if (theform.diraction.options[theform.diraction.selectedIndex].value == "publish") {
            theform.submit();
        }
        if (theform.diraction.options[theform.diraction.selectedIndex].value == "editmeta") {
            top.location=theform.filename.value+'default.meta'
        }
        if (theform.diraction.options[theform.diraction.selectedIndex].value == "printdir") {
            theform.action = '/adm/printout'
            theform.postdata.value = theform.filename.value
            theform.submit()
        }
        if (theform.diraction.options[theform.diraction.selectedIndex].value == "delete") {
              var delform = document.delresource
              delform.filename.value = theform.filename.value
              delform.submit()
        }
        return
    }
    function SetResChoice(theform) {
      var activity = theform.reschoice.options[theform.reschoice.selectedIndex].value
      if ((activity == 'rename') || (activity == 'copy') || (activity == 'move')) {
          changename(theform,activity)
      }
      if (activity == 'publish') {
          var pubform = document.pubresource
          pubform.filename.value = theform.filename.value
          pubform.submit()
      }
      if (activity == 'delete') {
          var delform = document.delresource
          delform.filename.value = theform.filename.value
          delform.submit()
      }
      if (activity == 'obsolete') {
          var pubform = document.pubresource
          pubform.filename.value = theform.filename.value
          pubform.makeobsolete.value=1;
          pubform.submit()
      }
      if (activity == 'print') {
          document.printresource.postdata.value = theform.filename.value
          document.printresource.submit()
      }
      if (activity == 'retrieve') {
          document.retrieveres.filename.value = theform.filename.value
          document.retrieveres.submit()
      }
      if (activity == 'cleanup') {
          document.cleanup.filename.value = theform.filename.value
          document.cleanup.submit()
      }
      return
    }
    function changename(theform,activity) {
        var oldname=theform.dispfilename.value;
        var newname=prompt('$newname',oldname);
        if (newname == "" || !newname || newname == oldname)  {
            return
        }
        document.moveresource.newfilename.value = newname
        document.moveresource.filename.value = theform.filename.value
        document.moveresource.action.value = activity
        document.moveresource.submit();
    }
</script>
ENDPUBDIRSCRIPT
    $r->print($pubdirscript);
}

sub dircontrols {
    my ($r,$uname,$udom,$thisdisfn, $current_disk_usage, $disk_quota) = @_;
    my %lt=&Apache::lonlocal::texthash(
                                       cnpd => 'Cannot publish directory',
                                       cnrd => 'Cannot retrieve directory',
                                       mcdi => 'Must create new subdirectory inside a directory',
                                       pubr => 'Publish this Resource',
                                       pubd => 'Publish this Directory',
                                       dedr => 'Delete Directory',
                                       rtrv => 'Retrieve Old Version',
                                       list => 'List Directory',
                                       uplo => 'Upload file',  
                                       dele => 'Delete',
                                       edit => 'Edit Metadata', 
                                       sela => 'Select Action',
                                       nfil => 'New file',
                                       nhtm => 'New HTML file',
                                       nprb => 'New problem',
                                       npag => 'New assembled page',
                                       nseq => 'New assembled sequence',
                                       ncrf => 'New custom rights file',
                                       nsty => 'New style file',
                                       nlib => 'New library file',
                                       nbt  => 'New bridgetask file',
                                       nsub => 'New subdirectory',
                                       renm => 'Rename current file to',
                                       move => 'Move current file to',
                                       copy => 'Copy current file to',
                                       type => 'Type Name Here',
                                       go   => 'Go',
                                       prnt => 'Print contents of directory',
                                       crea => 'Create a new directory or LON-CAPA document',
				       acti => 'Actions for current directory',
				       updc => 'Upload a new document',
				       pick => 'Please select an action to perform using the new filename',
                                      );
    my $mytype = $lt{'type'}; # avoid conflict with " and ' in javascript
    # Calculate free space in bytes.
    # $disk_quota is in MB and $current_disk_usage is in kB
    my $free_space = 1024 * ((1024 * $disk_quota) - $current_disk_usage);
    $r->print(<<END);
<div class="LC_columnSection">
  <div>
    <form name="curractions" method="post" action="">
      <fieldset>
        <legend>$lt{'acti'}</legend>
        <select name="dirtask" onchange="currdiract(this.form)">
            <option>$lt{'sela'}</option>
            <option value="publish">$lt{'pubd'}</option>
            <option value="editmeta">$lt{'edit'}</option>
            <option value="printdir">$lt{'prnt'}</option>
            <option value="delete">$lt{'dedr'}</option>
        </select>
        <input type="hidden" name="filename" value="/priv$thisdisfn/" />
      </fieldset>
    </form>
    <form name="publishdir" method="post" action="/adm/publish" target="_parent">
      <input type="hidden" name="pubrec" value="" />
      <input type="hidden" name="filename" value="" />
    </form>
    <form name="printdir" method="post" action="/adm/printout" target="_parent">
      <input type="hidden" name="postdata" value="" />
    </form>
  </div>

  <div style="padding-bottom: 2px">
    <form name="upublisher" enctype="multipart/form-data" method="post" action="/adm/upload" target="_parent">
      <fieldset>
        <legend>$lt{'updc'}</legend>
        <input type="hidden" name="filename" value="/priv$thisdisfn/" />
        <input type="file" name="upfile" class="LC_flUpload" size="20" />
        <input type="hidden" id="LC_free_space" value="$free_space" />
        <input type="button" value="$lt{'uplo'}"  onclick="checkUpload(this.form)" />
      </fieldset>
    </form>
  </div>

  <div>
    <form name="fileaction" method="post" action="/adm/cfile" target="_parent">
      <fieldset>
              <legend>$lt{'crea'}</legend>
	      <span class="LC_nobreak">
		<input type="hidden" name="filename" value="/priv$thisdisfn/" />
                  <script type="text/javascript">
                    function validate_go() {
                        var selected = document.fileaction.action.selectedIndex;
                        if (selected == 0) {
                            alert('$lt{'pick'}');
                        } else {
                            document.fileaction.submit();
                        }
                    }
                  </script>
		  <select name="action">
		    <option value="none">$lt{'sela'}</option>
		    <option value="newfile">$lt{'nfil'}:</option>
		    <option value="newhtmlfile">$lt{'nhtm'}:</option>
		    <option value="newproblemfile">$lt{'nprb'}:</option>
                    <option value="newpagefile">$lt{'npag'}:</option>
                    <option value="newsequencefile">$lt{'nseq'}:</option>
                    <option value="newrightsfile">$lt{'ncrf'}:</option>
                    <option value="newstyfile">$lt{'nsty'}:</option>
                    <option value="newtaskfile">$lt{'nbt'}:</option>
                    <option value="newlibraryfile">$lt{'nlib'}:</option>
	            <option value="newdir">$lt{'nsub'}:</option>
		  </select>&nbsp;<input type="text" name="newfilename" placeholder="$lt{'type'}" value="" onfocus="if (this.value == is.empty()) this.value=''" />&nbsp;<input type="button" value="Go" onclick="validate_go();" />
                <br />
                <span>Quickactions:
                 <input type="hidden" name="mode"/>
                 <a href="javascript:void(0)" onclick="javascript:validate_action('blank')">
                    <img src="/adm/lonIcons/unknown.gif" title="Create blank problem file"></a>
                 <a href="javascript:void(0)" onclick="javascript:validate_action('problemtempl')">
                    <img src="/adm/lonIcons/problem.gif" title="Create new problem from template"></a>
                 <a href="javascript:void(0)" onclick="javascript:validate_action('blankhtml')">
                    <img src="/adm/lonIcons/html.gif" title="Create new blank HTML file"></a>
                 <a href="javascript:void(0)" onclick="javascript:validate_action('folder')">
                    <img src="/adm/lonIcons/navmap.folder.closed.gif" title="Create new subdirectory"></a>
                </span>
                 <script type="text/javascript">
                     function validate_action(action){

                         if (document.getElementsByName(\'newfilename\')[0].value != \'\'){
                             if (action == "blank") {
                                                                 document.fileaction.action.value=\'newproblemfile\';
                                                                 document.fileaction.mode.value=\'blank\';
                                                         } else if (action == "problemtempl") {
                                                                 document.fileaction.action.value=\'newproblemfile\';
                                 validate_go();
                             } else if (action == "blankhtml") {
                                 document.fileaction.action.value=\'newhtmlfile\';
                                 validate_go();
                             } else if (action == "folder") {
                                 document.fileaction.action.value=\'newdir\';
                                 document.fileaction.mode.value=\'folder\';
                             }
                             fileaction.submit();
                         } else {
                             alert(\'Please specify file name.\');
                             // TODO: ask for filename? if so, do some refactoring

                         }
                     }
                 </script>
		 </span>
      </fieldset>
    </form>
  </div>
</div>
END
}

sub resourceactions {
    my ($r,$uname,$udom,$thisdisfn) = @_;
    $r->print(<<END);
       <form name="moveresource" action="/adm/cfile" target="_parent" method="post">
         <input type="hidden" name="filename" value="" />
         <input type="hidden" name="newfilename" value="" />
         <input type="hidden" name="action" value="" />
       </form>
       <form name="delresource" action="/adm/cfile" target="_parent" method="post">
         <input type="hidden" name="filename" value="" />
         <input type="hidden" name="action" value="delete" />
       </form>
       <form name="pubresource" action="/adm/publish" target="_parent" method="post">
         <input type="hidden" name="filename" value="" />
         <input type="hidden" name="makeobsolete" value="0" />
       </form>
       <form name="printresource" action="/adm/printout" target="_parent" method="post">
           <input type="hidden" name="postdata" value="" />
       </form>
       <form name="retrieveres" action="/adm/retrieve" target="_parent" method="post">
           <input type="hidden" name="filename" value="" />
       </form>
       <form name="cleanup" action="/adm/cleanup" target="_parent" method="post">
           <input type="hidden" name="filename" value="" />
       </form>
END
}

#
#   Get the title string or "[untitled]" if the file has no title metadata:
#   Without the latter substitution, it's impossible to examine metadata for
#   untitled resources.  Resources may be legitimately untitled, to prevent
#   searches from locating them.
#
#   $str = getTitleString($fullname);
#       $fullname - Fully qualified filename to check.
#
sub getTitleString {
    my $fullname = shift;
    my $title    = &Apache::lonnet::metadata($fullname, 'title');

    unless ($title) {
	$title = "[".&mt('untitled')."]";
    }
    return $title;
}

sub getCopyRightString {
    my $fullname = shift;
    return &Apache::lonnet::metadata($fullname, 'copyright');
}

sub getSourceRightString {
    my $fullname = shift;
    return &Apache::lonnet::metadata($fullname, 'sourceavail');
}
#
#  Put out a directory table row:
#    putdirectory(r, base, here, dirname, modtime, targetdir, bombs, numdir)
#      r         - Apache request object.
#      reqfile   - File in request.
#      here      - Where we are in directory tree.
#      dirname   - Name of directory special file.
#      modtime   - Encoded modification time.
#      targetdir - Publication target directory.
#      bombs     - Reference to hash of URLs with runtime error messages.
#      numdir    - Reference to scalar used to track number of sub-directories
#                  in directory (used in form name for each "actions" dropdown).
#
sub putdirectory {
    my ($r, $reqfile, $here, $dirname, $modtime, $targetdir, $bombs, $numdir) = @_;

# construct the display filename: the directory name unless ..:
   
    my $actionitem;
 
    my $disfilename = $dirname;
# Don't display directory itself, and there is no way up from root directory
    unless ((($dirname eq '..') && ($reqfile=~/^\/[^\/]+\/[^\/]+$/)) || ($dirname eq '.')) {
        my $kaputt=0;
        if (ref($bombs) eq 'HASH') {
            foreach my $key (keys(%{$bombs})) {
                my $currentdir = &Apache::lonnet::declutter("$targetdir/$disfilename");
                if (($key) =~ m{^\Q$currentdir\E/}) { $kaputt=1; last; }
            }
        }
#
# Get the metadata from that directory's default.meta to display titles
#
	%Apache::lonpublisher::metadatafields=();
	%Apache::lonpublisher::metadatakeys=();
	&Apache::lonpublisher::metaeval(
                 &Apache::lonnet::getfile($r->dir_config('lonDocRoot').$here.'/'.$dirname.'/default.meta')
                                       );
        if ($dirname eq '..') {
            $actionitem = &mt('Go to ...');
            $disfilename = '<i>'.&mt('Parent Directory').'</i>';
        } else {
            $actionitem = 
                    '<form name="dirselect_'.$$numdir.
                    '" action="/adm/publish" target="_parent">'.
                    '<select name="diraction" onchange="SetPubDir(this.form,document)">'.
                      '<option selected="selected">'.&mt('Select action').'</option>'.
                      '<option value="open">'.&mt('Open').'</option>'.
                      '<option value="publish">'.&mt('Publish').'</option>'.
                      '<option value="editmeta">'.&mt('Edit Metadata').'</option>'.
                      '<option value="printdir">'.&mt('Print directory').'</option>'.
                      '<option value="delete">'.&mt('Delete directory').'</option>'.
                    '</select>'.
                     '<input type="hidden" name="filename" value="'.&HTML::Entities::encode($here.'/'.$dirname,'<>&"').'/" />'.
                     '<input type="hidden" name="openname" value="'.$here.'/'.$dirname.'/" />'.
                     '<input type="hidden" name="postdata" value="" />'.
                   '</form>';
            $$numdir ++;
        }
	$r->print('<tr class="LC_browser_folder">'.
		  '<td><img src="'.
		  $Apache::lonnet::perlvar{'lonIconsURL'}.'/navmap.folder.closed.gif" alt="folder" /></td>'.
		  '<td>'.$actionitem.'</td>'.
		  '<td><span class="LC_filename"><a href="'.&HTML::Entities::encode($here.'/'.$dirname,'<>&"').'/" target="_parent">'.
		  $disfilename.'</a></span></td>'.
		        '<td colspan="3">'.($kaputt?&Apache::lonhtmlcommon::authorbombs($targetdir.'/'.$disfilename.'/'):'').$Apache::lonpublisher::metadatafields{'title'});
	if ($Apache::lonpublisher::metadatafields{'subject'} ne '') {
	    $r->print(' <i>'.
		      $Apache::lonpublisher::metadatafields{'subject'}.
		      '</i> ');
	}
	$r->print($Apache::lonpublisher::metadatafields{'keywords'}.'</td>'.
		  '<td>'.&Apache::lonlocal::locallocaltime($modtime).'</td>'.
	          '<td></td>'.
		  "</tr>\n");
    }
    return;
}

sub getTitle {
    my ($resdir, $targetdir, $filename, $linkfilename, $meta_same, $bombs) = @_;
    my $title='';
    my $titleString = &getTitleString($targetdir.'/'.$filename);
    if (-e $resdir.'/'.$filename) {
	$title = '<a href="'.$targetdir.'/'.$filename.
	    '.meta" target="cat">'.$titleString.'</a>';
        if (!$meta_same) {
	    $title = &mt('Metadata Modified').'<br />'.$title.
		'<br />'.
                &Apache::loncommon::modal_link(
                    '/adm/diff?filename='.$linkfilename.'.meta'.'&amp;versiontwo=priv',
                    &mt('Metadata Diffs'),600,500);
	    $title.="\n".'<br />'.
                &Apache::loncommon::modal_link(
                    '/adm/retrieve?filename='.$linkfilename.'.meta&amp;inhibitmenu=yes&amp;add_modal=yes',
                    &mt('Retrieve Metadata'),600,500);
        } 
    }
    # Allow editing metadata of published and unpublished resources
    $title .= "\n".'<br />' if ($title);
    $title .= '<a href="'.$linkfilename.'.meta">'.
              ($$bombs{&Apache::lonnet::declutter($targetdir.'/'.$filename)}?
                  '<img src="/adm/lonMisc/bomb.gif" border="0" alt="'.&mt('bomb').'" />':
                  &mt('Edit Metadata')).
              '</a>';

    return ($title, $titleString);
}


sub isMetaSame {
    my ($cstr_dir, $resdir, $filename) = @_;
    my $meta_cmtime = (stat($cstr_dir.'/'.$filename.'.meta'))[9];
    my $meta_rmtime = (stat($resdir.'/'.$filename.'.meta'))[9];
    return (&Apache::londiff::are_different_files($resdir.'/'.$filename.'.meta',
            $cstr_dir.'/'.$filename.'.meta') && $meta_rmtime < $meta_cmtime) 
        ? 0 : 1;
}
    

sub getStatus {    
    my ($resdir, $targetdir, $cstr_dir, $filename,  
            $linkfilename, $cmtime, $meta_same) = @_;
    my $pubstatus = 'unpublished';
    my $status = &mt('Unpublished');

    if (-e $resdir.'/'.$filename) {
        my $same = 0;
        if ((stat($resdir.'/'.$filename))[9] >= $cmtime) {
            $same = 1;
        } else {
           if (&Apache::londiff::are_different_files($resdir.'/'.$filename,
						     $cstr_dir.'/'.$filename)) {
              $same = 0;
           } else {
              $same = 1;
           }
        }

        my $rights_status =
            &mt(&getCopyRightString($targetdir.'/'.$filename)).', ';

        my %lt_SourceRight = &Apache::lonlocal::texthash(
               'open'   => 'Source: open',
               'closed' => 'Source: closed',
        );
        $rights_status .=
            $lt_SourceRight{&getSourceRightString($targetdir.'/'.$filename)};

	if ($same) {
	    if (&Apache::lonnet::metadata($targetdir.'/'.$filename,'obsolete')) {
                $pubstatus = 'obsolete';
		$status=&mt('Obsolete');
            } else {
		if (!$meta_same) {
		    $pubstatus = 'metamodified';
		} else {
		    $pubstatus = 'published';
		}
		$status=&mt('Published').
		    '<br />'. $rights_status;
	    }
	} else {
            $pubstatus = 'modified';
	    $status=&mt('Modified').
		'<br />'. $rights_status;
	    if (&Apache::loncommon::fileembstyle(($filename=~/\.(\w+)$/)) eq 'ssi') {
		$status.='<br />'.
                         &Apache::loncommon::modal_link(
                             '/adm/diff?filename='.$linkfilename.'&amp;versiontwo=priv',
                             &mt('Diffs'),600,500);
	    }
	} 

	$status.="\n".'<br />'.
             &Apache::loncommon::modal_link(
                 '/adm/retrieve?filename='.$linkfilename.'&amp;inhibitmenu=yes&amp;add_modal=yes',&mt('Retrieve'),600,500);
    }

    return ($status, $pubstatus);
}


#
#   Put a table row for a file resource.
#
sub putresource {
    my ($r, $udom, $uname, $filename, $thisdisfn, $resdir, $targetdir, 
            $linkdir, $cmtime, $size, $numres, $linkfilename, $title, 
            $status, $pubstatus) = @_;
    &Apache::lonnet::devalidate_cache_new('meta',$targetdir.'/'.$filename);

    my $editlink='';
    my $editlink2='';
    if ($filename=~/\.(xml|html|htm|xhtml|xhtm|sty)$/) {
	$editlink=' <br />(<a href="'.$linkdir.'/'.$filename.'?editmode=Edit&amp;problemmode=edit">'.&mt('Edit').'</a>)';
    }
    if ($filename=~/$LONCAPA::assess_re/) {
	$editlink=' (<a href="'.$linkdir.'/'.$filename.'?editmode=Edit&amp;problemmode=editxml">'.&mt('EditXML').'</a>)';
	$editlink2=' <br />(<a href="'.$linkdir.'/'.$filename.'?editmode=Edit&amp;problemmode=edit">'.&mt('Edit').'</a>)';
    }
    if ($filename=~/\.(problem|exam|quiz|assess|survey|form|library|xml|html|htm|xhtml|xhtm|sty)$/) {
	$editlink.=' (<a href="/adm/cleanup?filename='.$linkfilename.'" target="_parent">'.&mt('Clean Up').')</a>';
    }
    if ($filename=~/\.(zip|tar|bz2|gz|tar.gz|tar.bz2|tgz)$/) {
	$editlink=' (<a target="_parent" href="/adm/cfile?decompress='.$linkfilename.'">'.&mt('Decompress').'</a>)';
    }
    my $publish_button = (-e $resdir.'/'.$filename) ? &mt('Re-publish') : &mt('Publish');
    my $pub_select = '';
    &create_pubselect($r,\$pub_select,$udom,$uname,$thisdisfn,$filename,$resdir,$pubstatus,$publish_button,$numres);
    $r->print(&Apache::loncommon::start_data_table_row().
	      '<td>'.($filename=~/[\#\~]$/?'&nbsp;':
		      '<img src="'.&Apache::loncommon::icon($filename).'" alt="" />').'</td>'.
              '<td>'.$pub_select.'</td>'.
	      '<td><span class="LC_filename">'.
	      '<a href="'.$linkdir.'/'.$filename.'" target="_parent">'.
               $filename.'</a></span>'.$editlink2.$editlink.
	      '</td>'.
	      '<td>'.$title.'</td>'.
              '<td class="LC_browser_file_'.$pubstatus.'">&nbsp;&nbsp;</td>'. # Display publication status
              '<td>'.$status.'</td>'.
	      '<td>'.&Apache::lonlocal::locallocaltime($cmtime).'</td>'.
	      '<td>'.sprintf("%.1f",$size).'</td>'.
	      &Apache::loncommon::end_data_table_row()
    );
    return;
}

sub create_pubselect {
    my ($r,$pub_select,$udom,$uname,$thisdisfn,$filename,$resdir,$pubstatus,$publish_button,$numres) = @_;
    $$pub_select = '
<form name="resselect_'.$$numres.'" action="">
<select name="reschoice"  onchange="SetResChoice(this.form)">
<option>'.&mt('Select action').'</option>'.
'<option value="copy">'.&mt('Copy').'</option>';
    if ($pubstatus eq 'obsolete' || $pubstatus eq 'unpublished') {
        $$pub_select .= 
'<option value="rename">'.&mt('Rename').'</option>'.
'<option value="move">'.&mt('Move').'</option>'.
'<option value="delete">'.&mt('Delete').'</option>';
    } else {
        $$pub_select .= '
<option value="obsolete">'.&mt('Mark obsolete').'</option>';
    }
# check for versions
    my $versions = &check_for_versions($r,'/'.$filename,$udom,$uname);
    if ($versions > 0) {
        $$pub_select .='
<option value="retrieve">'.&mt('Retrieve old version').'</option>';
    }
    $$pub_select .= '
<option value="publish">'.$publish_button.'</option>'.
'<option value="cleanup">'.&mt('Clean up').'</option>'.
'<option value="print">'.&mt('Print').'</option>'.
'</select>
<input type="hidden" name="filename" value="/priv'.
 &HTML::Entities::encode($thisdisfn.'/'.$filename,'<>&"').'" />
 <input type="hidden" name="dispfilename" value="'.
 &HTML::Entities::encode($filename).'" /></form>';
    $$numres ++;
}

sub check_for_versions {
    my ($r,$fn,$udom,$uname) = @_;
    my $versions = 0;
    my $docroot=$r->dir_config('lonDocRoot');
    my $resfn=$docroot.'/res/'.$udom.'/'.$uname.$fn;
    my $resdir=$resfn;
    $resdir=~s/\/[^\/]+$/\//;
    $fn=~/\/([^\/]+)\.(\w+)$/;
    my $main=$1;
    my $suffix=$2;
    opendir(DIR,$resdir);
    while (my $filename=readdir(DIR)) {
        if ($filename=~/^\Q$main\E\.(\d+)\.\Q$suffix\E$/) {
            $versions ++;        
        }
    }
    closedir(DIR);
    return $versions;
}

1;
__END__


=head1 NAME

Apache::lonpubdir - Authoring space directory lister

=head1 SYNOPSIS

Invoked (for various locations) by /etc/httpd/conf/srm.conf:

 <LocationMatch "^/+priv.*/$">
 PerlAccessHandler       Apache::loncacc
 SetHandler perl-script
 PerlHandler Apache::lonpubdir
 ErrorDocument     403 /adm/login
 ErrorDocument     404 /adm/notfound.html
 ErrorDocument     406 /adm/unauthorized.html
 ErrorDocument	  500 /adm/errorhandler
 </LocationMatch>

 <Location /adm/pubdir>
 PerlAccessHandler       Apache::lonacc
 SetHandler perl-script
 PerlHandler Apache::lonpubdir
 ErrorDocument     403 /adm/login
 ErrorDocument     404 /adm/notfound.html
 ErrorDocument     406 /adm/unauthorized.html
 ErrorDocument	  500 /adm/errorhandler
 </Location>

=head1 INTRODUCTION

This module publishes a directory of files.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 HANDLER SUBROUTINE

This routine is called by Apache and mod_perl.

=over 4

=item *

read in information

=item *

start page output

=item *

run through list of files and attempt to publish unhidden files

=back

=head1 SUBROUTINES:

=over

=item startpage($r, $uame, $udom, $thisdisfn)

Output the header of the page.  This includes:
 - The HTML header 
 - The H1/H3  stuff which includes the directory.
 
    startpage($r, $uame, $udom, $thisdisfn);
        $r     - The apache request object.
        $uname - User name.
        $udom  - Domain name the user is logged in under.
        $thisdisfn - Displayable version of the filename.

=item getTitleString($fullname)

    Get the title string or "[untitled]" if the file has no title metadata:
    Without the latter substitution, it's impossible to examine metadata for
    untitled resources.  Resources may be legitimately untitled, to prevent
    searches from locating them.
    
    $str = getTitleString($fullname);
        $fullname - Fully qualified filename to check.

=item putdirectory($r, $base, $here, $dirname, $modtime, $targetdir, $bombs,
                   $numdir)

    Put out a directory table row:
    
        $r        - Apache request object.
        $reqfile  - File in request.
        $here     - Where we are in directory tree.
        $dirname  - Name of directory special file.
        $modtime  - Encoded modification time.
        targetdir - Publication target directory.
        bombs     - Reference to hash of URLs with runtime error messages.
        numdir    - Reference to scalar used to track number of sub-directories
                    in directory (used in form name for each "actions" dropdown).

=back

=cut
