#!/usr/bin/perl
#
# Automated Userinfo update script
# $Id: Autoupdate.pl,v 1.25 2022/01/14 16:27:20 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
    use strict;
    use lib '/home/httpd/lib/perl';
    use localenroll;
    use GDBM_File;
    use Apache::lonnet;
    use Apache::loncommon;
    use Apache::lonlocal;
    use Apache::lonuserutils();
    use LONCAPA::Configuration;
    use LONCAPA;

    my $perlvarref = LONCAPA::Configuration::read_conf('loncapa.conf');
    exit if (ref($perlvarref) ne 'HASH');
    my $logfile = $perlvarref->{'lonDaemons'}.'/logs/autoupdate.log';
    exit if ($perlvarref->{'lonRole'} ne 'library');
    my $hostid = $perlvarref->{'lonHostID'};

    my @info = ('inststatus','lockedname','internalname','lastname',
                'firstname','middlename','generation','id','permanentemail');
    # Initialize language handler
    &Apache::lonlocal::get_language_handle();
    # find out which users we need to examine
    my @domains = sort(&Apache::lonnet::current_machine_domains());
    foreach my $dom (@domains) {
        my %domconfig = &Apache::lonnet::get_dom('configuration',['autoupdate'],
                                                 $dom);
        #only run if configured to
        my $run_update = 0;
        my $settings;
        my $allowed_inactivity;
        my $check_unexpired;
        if (ref($domconfig{'autoupdate'}) eq 'HASH') {
            $settings = $domconfig{'autoupdate'};
            if ($settings->{'run'} eq '1') {
                $run_update = 1;
            }
            if ($settings->{'lastactive'} =~/^\d+$/) {
                $allowed_inactivity = 86400 * $settings->{'lastactive'};
            }
            if ($settings->{'unexpired'}) {
                $check_unexpired = 1;
            }
        }
        next if (!$run_update);
        open(my $fh,">>$logfile");
        print $fh ("********************\n".&Apache::lonlocal::locallocaltime(time).' '.&mt('Autoupdate messages start for domain: [_1]',$dom).' --'."\n");
        # get courseIDs for domain
        my %courses=&Apache::lonnet::courseiddump($dom,'.',1,'.','.','.',1,[$hostid],'.');
        # get user information
        my (%users,%instusers,%instids,%counts);
        my $dir = $Apache::lonnet::perlvar{lonUsersDir}.'/'.$dom;
        &descend_tree($dom,$dir,0,\%users,\%courses,$allowed_inactivity,$check_unexpired);
        my $resp = &localenroll::allusers_info($dom,\%instusers,\%instids,\%users,\%counts);
        if ($resp ne 'ok') {
            print $fh &mt('Problem retrieving institutional data for users in domain: [_1].',$dom)."\n".
                      &mt('Error: [_1].',$resp)."\n".
                      "-- \n".&Apache::lonlocal::locallocaltime(time).' '.
                      &mt('Autoupdate messages end')."\n*******************\n\n";
            close($fh);
            next;
        } else {
            if (keys(%counts)) {
                foreach my $key (sort(keys(%counts))) {
                    if ($counts{$key} ne '') {
                        print $fh "Count for $key = $counts{$key}\n";
                    }
                }
            }
        }
        my (%unamechg,%possnames);
        my @types = ('active','future');
        my @roles = ('st');
        my @cdoms = ($dom);
        foreach my $uname (sort(keys(%users))) {
            my %userhash = &Apache::lonnet::userenvironment($dom,$uname,@info);
            my (@inststatuses);
            if (!$userhash{'internalname'}) {
                if (defined($instusers{$uname})) {
                    (my $insttypechg,@inststatuses) = 
                       &affiliations_check(\%userhash,$instusers{$uname});
                    if ($insttypechg) {
                        my $inststatusstr = join(':',map { &escape($_) } (@inststatuses));
                        my %statushash = ( inststatus => $inststatusstr );
                        my $statusres = &Apache::lonnet::put('environment',\%statushash,$dom,$uname);
                    }
                }
            }
            if (!$userhash{'lockedname'} && !$userhash{'internalname'}) {
                if (defined($instusers{$uname})) {
                    my (@fields,%changes,$changed);
                    if (@inststatuses > 0) {
                        foreach my $type (@inststatuses) {
                            if (ref($settings->{fields}{$type}) eq 'ARRAY') {
                                foreach my $field (@{$settings->{fields}{$type}}) {
                                    if (!grep(/^\Q$field\E$/,@fields)) {
                                        push(@fields,$field);
                                    }
                                }
                            } 
                        }
                    } else {
                        if (ref($settings->{fields}{'default'}) eq 'ARRAY') {
                            @fields = @{$settings->{fields}{'default'}};
                        }
                    }
                    foreach my $field (@fields) { 
                        if ($userhash{$field} ne $instusers{$uname}{$field}) {
                            $changed = 1;
                            if ($settings->{'classlists'} == 1) { 
                                if ($field eq 'id') {
                                    $changes{'id'} = 1;
                                } elsif ($field eq 'lastname' || $field eq 'firstname' || $field eq 'middlename' || $field eq 'generation') {
                                    $changes{'fullname'} = 1;
                                }
                            }
                        }
                    }
                    # Make the change
                    if ($changed) {
                        my %userupdate;
                        foreach my $item ('lastname','firstname','middlename','generation','id',
                                          'permanentemail') {
                            $userupdate{$item} = $userhash{$item};
                        }
                        foreach my $field (@fields) {
                            $userupdate{$field} = $instusers{$uname}{$field};
                        }
                        my $modresult = &Apache::lonnet::modifyuser($dom,$uname,$userupdate{'id'},undef,undef,$userupdate{'firstname'},$userupdate{'middlename'},$userupdate{'lastname'},$userupdate{'generation'},1,$hostid,$userupdate{'permanentemail'},undef,\@fields);
                        if ($modresult eq 'ok') {
                            print $fh "User change: $uname:$dom; New values: 1.Name: $userupdate{'firstname'} $userupdate{'middlename'} $userupdate{'lastname'} $userupdate{'generation'}; 2. StuID: $userupdate{'id'}; 3. Email: $userupdate{'permanentemail'}.\n";
                            if ($settings->{'classlists'} == 1) {
                                if ($changes{'id'} || $changes{'fullname'}) {
                                    my %roleshash = 
                                        &Apache::lonnet::get_my_roles($uname,
                                                  $dom,'userroles',\@types,\@roles,\@cdoms);
                                    foreach my $item (keys(%roleshash)) {
                                        my ($cnum,$cdom,$role) = split(/:/,$item);
                                        my ($start,$end) = split(/:/,$roleshash{$item});
                                        if (&Apache::lonnet::is_course($cdom,$cnum)) {
                                            my $result = 
                                                &Apache::lonuserutils::update_classlist($cdom,$cnum,$dom,$uname,\%userupdate);
                                            if ($result eq 'ok') {
                                                print $fh "Classlist change: $uname:$dom - class -> $cnum:$cdom\n";
                                            } else {
                                                print $fh "Error - $result -during classlist update for $uname:$dom in $cnum:$cdom\n";
                                            }
                                        }
                                    }
                                }
                            }
                        } else {
                            print $fh "User change detected for $uname:$dom. Error during update: $modresult\n";
                        }
                    }
                } else {
                    # check if the username has changed
                    if (defined($instids{$userhash{'id'}})) {
                        if (ref($instids{$userhash{'id'}}) eq 'ARRAY') {
                            foreach my $name (@{$instids{$userhash{'id'}}}) {
                                if (!exists($users{$name})) {
                                    push(@{$possnames{$uname}},$name);
                                }
                            }
                        } else {
                            if (!exists($users{$instids{$userhash{'id'}}})) {
                                $unamechg{$uname} = $instids{$userhash{'id'}};
                                print $fh (&mt('Username change to [_1] detected for [_2] in domain [_3].',$unamechg{$uname},$uname,$dom)."\n");
                            }
                        }
                    }
                }
            }
        }
        if (keys(%possnames) > 0) {
            foreach my $uname (keys(%possnames)) {
                my $altnames = join(' or ',@{$possnames{$uname}});
                print $fh (&mt('Possible username change to [_1] detected for [_2] in domain [_3].',$altnames,$uname,$dom)."\n");
            }
        }
        print $fh ("-- \n".&Apache::lonlocal::locallocaltime(time).' '.&mt('Autoupdate messages end')."\n*******************\n\n");
        close($fh);
    }

sub descend_tree {
    my ($dom,$dir,$depth,$alldomusers,$coursesref,$allowed_inactivity,$check_unexpired) = @_;
    if (-d $dir) {
        opendir(DIR,$dir);
        my @contents = grep(!/^\./,readdir(DIR));
        closedir(DIR);
        $depth ++;
        foreach my $item (@contents) {
            if (($depth < 4) && (length($item) == 1)) {
                &descend_tree($dom,$dir.'/'.$item,$depth,$alldomusers,$coursesref,
                              $allowed_inactivity,$check_unexpired);
            } elsif (-e $dir.'/'.$item.'/passwd') {
                if (ref($coursesref) eq 'HASH') {
                    next if (exists($coursesref->{$dom.'_'.$item}));
                }
                if ($allowed_inactivity) {
                    my $now = time;
                    my $aclog = $dir.'/'.$item.'/activity.log';
                    my $roledb = $dir.'/'.$item.'/roles.db';
                    if (-e $aclog) {
                        my $lastac=(stat($aclog))[9];
                        if (($now - $lastac) > $allowed_inactivity) {
                            if (-e $roledb) {
                                my $lastrolechg=(stat($roledb))[9];
                                next if (($now - $lastrolechg) > $allowed_inactivity);
                            } else {
                                next;
                            }
                        }
                    } elsif (-e $roledb) {
                        my $lastrolechg=(stat($roledb))[9];
                        next if (($now - $lastrolechg) > $allowed_inactivity);
                    } else {
                        next;
                    }
                }
                if ($check_unexpired) {
                    my $roledb = $dir.'/'.$item.'/roles.db';
                    my $unexpired = 0;
                    my $now = time;
                    if (-e $roledb) {
                        my $roleshash = &LONCAPA::tie_user_hash($dom,$item,'roles',&GDBM_READER()) or next;
                        if (ref($roleshash)) {
                            while (my ($key,$value) = each(%$roleshash)) {
                                next if ($key =~ /^rolesdef/);
                                my ($role,$roleend,$rolestart) = split(/\_/,$value);
                                next if ($role =~ /^gr\//);
                                if (!$roleend || $roleend > $now) {
                                    $unexpired = 1;
                                    last;
                                }
                            }
                            &LONCAPA::untie_user_hash($roleshash);
                            next unless ($unexpired);
                        } else {
                            next;
                        }
                    } else {
                        next;
                    }
                }
                $$alldomusers{$item} = '';
            }
        }
    }
}

sub affiliations_check {
    my ($userhash,$insthashref) = @_;
    my (@inststatuses,$insttypechg);;
    if (ref($insthashref) eq 'HASH') {
        if (ref($insthashref->{'inststatus'}) eq 'ARRAY') {
            @inststatuses = @{$insthashref->{'inststatus'}};
        }
    }
    my @currstatuses = map { &unescape($_); } (split(/:/,$userhash->{'inststatus'}));
    foreach my $status (@inststatuses) {
        if (!grep/^\Q$status\E/,@currstatuses) {
            $insttypechg = 1;
        }
    }
    if (!$insttypechg) {
        foreach my $status (@currstatuses) {
            if (!grep/^\Q$status\E/,@inststatuses) {
                $insttypechg = 1;
            }
        }
    }
    return ($insttypechg,@inststatuses);
}

