# The LearningOnline Network with CAPA
# chemical equation style response
#
# $Id: chemresponse.pm,v 1.101 2021/12/24 11:07:42 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
#
package Apache::chemresponse;
use strict;
use Apache::lonxml;
use Apache::lonnet;
use Apache::lonlocal;
use lib '/home/httpd/lib/perl/';
use LONCAPA;
 

BEGIN {
    &Apache::lonxml::register('Apache::chemresponse',('organicresponse','organicstructure','reactionresponse','chem'));
}

sub chem_standard_order {
    my ($reaction) = @_;
    my ($re,$pr) = split(/->|<=>/,$reaction);
    my @reactants = split(/\s\+/,$re);
    my @products =  split(/\s\+/,$pr);
    foreach my $substance (@reactants,@products) {
	$substance =~ s/(\^\d*)\s+/$1_/g;         # protect superscript space
	$substance =~ s/\s*//g;                   # strip whitespace
	$substance =~ s/_/ /g;                    # restore superscript space
    }
    @reactants = sort @reactants;
    @products = sort @products;
    my $standard = '';
    foreach my $substance (@reactants) {
	$standard .= $substance;
	$standard .= ' + ';
    }
    $standard =~ s/ \+ $//;              # get rid of trailing plus sign
    $standard .= ' -> ';
    foreach my $substance (@products) {
	$standard .= $substance;
	$standard .= ' + ';
    }
    $standard =~ s/ \+ $//;              # get rid of trailing plus sign
    return $standard;
}

sub separate_jme_window {
    my ($smile_input,$jme_input,$molecule,$options,$shown_text)=@_;
    my $usejsme = 1;
    if (($env{'request.course.id'}) && ($env{'request.state'} ne 'construct')) {
        if (exists($env{'course.'.$env{'request.course.id'}.'.usejsme'})) {
            if ($env{'course.'.$env{'request.course.id'}.'.usejsme'} eq '0') {
                $usejsme = 0;
            }
        } else {
            my %domdefs = &Apache::lonnet::get_domain_defaults($env{'course.'.$env{'request.course.id'}.'.domain'});
            if ($domdefs{'usejsme'} eq '0') {
                $usejsme = 0;
            }
        }
    } else {
        my %domdefs = &Apache::lonnet::get_domain_defaults($env{'course.'.$env{'request.course.id'}.'.domain'});
        if ($domdefs{'usejsme'} eq '0') {
            $usejsme = 0;
        }
    }
    if ($usejsme) {
        if ($env{'browser.type'} eq 'safari') {
            unless ($env{'browser.mobile'}) {
                if ($env{'browser.version'} < 534) {
                    $usejsme = 0;
                }
            }
        } elsif ($env{'browser.type'} eq 'mozilla') {
            if ($env{'browser.version'} < 5) {
                $usejsme = 0;
            } elsif ($env{'browser.info'} =~ /^firefox\-([\d\.]+)/) {
                my $firefox = $1;
                if ($firefox < 12) {
                    $usejsme = 0;
                }
            }
        } elsif ($env{'browser.type'} eq 'explorer') {
            if ($env{'browser.version'} < 7) {
                $usejsme = 0;
            }
        } elsif ($env{'browser.type'} eq 'opera') {
            if ($env{'browser.version'} < 15) {
                $usejsme = 0;
            }
        }
    } else {
        if ($env{'browser.mobile'}) {
            $usejsme = 1;
        }
    }
    my $linkstyle = 'display:none';
    my $creditstyle = 'display:inline';
    if ($env{'browser.type'} eq 'explorer') {
        if (($env{'browser.os'} eq 'win') && ($env{'browser.version'} < 9)) {
            $linkstyle = 'display:inline';
            $creditstyle = 'display:none';
        }
    }
    my $smilesection;
    if (defined($smile_input)) {
        my $smiles;
        if ($usejsme) {
            $smiles = 'document.JME.smiles()'; 
        } else {
            $smiles = 'document.applets.JME.smiles()';
        }
	$smilesection=<<SMILESECTION;
	opener.document.lonhomework.$smile_input.value = $smiles;
SMILESECTION
    }
    my $jmesection;
    my $jmefile;
    if (defined($jme_input)) {
        if ($usejsme) {
            $jmefile = 'document.JME.jmeFile()';
        } else {
            $jmefile = 'document.applets.JME.jmeFile()';
        }
	$jmesection=<<JMESECTION;
	opener.document.lonhomework.$jme_input.value = $jmefile;
JMESECTION
    }

    if ($molecule) {
        if ($usejsme) {
            $molecule = 'document.JME.readMolecule("'.$molecule.'")';
        } else {
            $molecule='<param name="jme" value="'.$molecule.'" />';
        }
    }
    my $insert_answer;
    if ($shown_text eq '') { 
	$insert_answer=
	    '<input type="button" name="submit" value="'.&mt('Insert Answer').'" onclick="javascript:submitSmiles();" /><br />';
    }

   

    my ($jsme_js,$js,$buttonstyle,$viewportjs,$resizejs);
    if ($usejsme) {
        $buttonstyle = 'display:none';
        $resizejs =<<RESIZEJS;
<script type="text/javascript">
function callResize() {
    var timer;
    clearTimeout(timer);
    timer=setTimeout('resize_jsme()',100);
}

window.onresize = callResize;

function resize_jsme() {
    init_geometry();
    var vph = Geometry.getViewportHeight();
    var vpw = Geometry.getViewportWidth();
    var lowerdivheight = document.getElementById('JMEbuttons').offsetHeight;
    var formheight = document.getElementById('JMEform').offsetHeight;
    var freevspace = vph-(lowerdivheight+50);
    var freehspace = vpw-20;
    if (typeof jsmeApplet !== 'undefined') {
        jsmeApplet.setSize(freehspace,freevspace);
    }
}
</script>
RESIZEJS
        $resizejs = &Apache::loncommon::js_ready($resizejs);
        $jsme_js = '
<script type="text/javascript" language="javascript" src="/adm/jsme/jsme.nocache.js"></script>'."\n";
        $js =<<CHEMJS;
<script type="text/javascript">
function jsmeOnLoad() {
    document.getElementById('JMErefresh').style.display="none";
    document.getElementById('JMEcredits').style.display="inline";
    jsmeApplet = new JSApplet.JSME("jme", "420px", "330px", {"options" : "$options"});
    document.JME = jsmeApplet;
    $molecule;
    document.getElementById('JMEbuttons').style.display="block";
    callResize();
}

function submitSmiles() {
    jmeFile = document.JME.jmeFile();
    if (jmeFile == "") {
        alert("Nothing to submit");
    } else {
        $jmesection
        $smilesection
        window.close();
    }
}
function openHelpWindow() {
    window.open("http://peter-ertl.com/jsme/2013_03/help.html","","scrollbars=yes,resizable=yes,width=500,height=600");
}

</script>

CHEMJS

        $viewportjs = &Apache::loncommon::viewport_geometry_js();
        $viewportjs = '<script type="text/javascript">'."\n".
                      $viewportjs."\n".
                      '</script>';

    } else {
        $buttonstyle = 'display:block';
        $js =<<CHEMJS; 
<script type="text/javascript">
function submitSmiles() {
    jmeFile = document.applets.JME.jmeFile();
    if (jmeFile == "") {
	alert("Nothing to submit");
    } else {
        $jmesection
        $smilesection
	window.close();
    }
}
function openHelpWindow() {
    window.open("/adm/jme/jme_help.html","","scrollbars=yes,resizable=yes,width=500,height=600");
}
function substituent(r) {document.applets.JME.setSubstituent(r);}
</script>
CHEMJS
    }

    my $start_page = 
        &Apache::loncommon::start_page('Molecule Editor',$viewportjs,
				       {'only_body' => 1,
					'js_ready'  => 1,
					'bgcolor'   => '#FFFFFF',});
    my $end_page =
 	&Apache::loncommon::end_page({'js_ready' => 1,});
    my $java_not_enabled=&Apache::lonhtmlcommon::java_not_enabled();
    my %lt = &Apache::lonlocal::texthash(
        'seltext' => 'Select substituent...',
        'close'   => 'Close',
        'help'    => 'Help',
       );
    my $body = "
$jsme_js
$js".'
<center>
<form action="" id="JMEform">
';
    if ($usejsme) {
        $body.= <<"ENDCHEM";
<div id="jme">
<div id="JMEcredits" style="$creditstyle">
<span style="font-size:small; font-family:arial,sans-serif;"><a href="http://peter-ertl.com/jsme/">JSME Molecular Editor</a> courtesy of Peter Ertl (Novartis) and Bruno Bienfait</span></div>
</div>
ENDCHEM
    } else {
        $body.=<<CHEMPAGE;
  <table width="440"><tr>
    <td></td>
    <td align="right">
      <select onchange="javascript:substituent(options[selectedIndex].text)">
        <option>$lt{'seltext'}</option>
        <option>-C(=O)OH</option>
        <option>-C(=O)OMe</option>
        <option>-OC(=O)Me</option>
        <option>-CMe3</option>
        <option>-CF3</option>
        <option>-CCl3</option>
        <option>-NO2</option>
        <option>-SO2-NH2</option>
        <option>-NH-SO2-Me</option>
        <option>-NMe2</option>
        <option>-C#N</option>
        <option>-C#C-Me</option>
        <option>-C#CH</option>
      </select>
    </td></tr>
  </table>
<applet code="JME.class" name="JME" archive="/adm/jme/JME.jar" width="440" height="390" mayscript>
$java_not_enabled
$molecule
<param name="options" value="$options" />
</applet>
<br />
<font face="arial,helvetica,sans-serif" size="-1"><a href="http://www.molinspiration.com/jme/index.html">JME Editor</a> courtesy of Peter Ertl, Novartis</font>
CHEMPAGE
    }
    $body .= <<CHEMPAGE;
<div id="JMEbuttons" style="$buttonstyle">
$insert_answer
<input type="button" value="$lt{'close'}" onclick="javascript:window.close()" />
&nbsp;&nbsp;
<input type="button" value="$lt{'help'}" onclick="javascript:openHelpWindow()" />
</div>
<div id="JMErefresh" style="$linkstyle">
<a href="javascript:location.reload();">Display Molecule Editor</a>
</div> 
</form>
</center>
CHEMPAGE

    $body=&Apache::loncommon::js_ready($body);
    my $nothing=&Apache::lonhtmlcommon::javascript_nothing();
    my $docopen=&Apache::lonhtmlcommon::javascript_docopen();
    my $display=&mt('Draw Molecule');
    if (defined($shown_text)) { $display=&mt($shown_text); }
    my $iconpath=$Apache::lonnet::perlvar{'lonIconsURL'};
    my $function =
	'LONCAPA_draw_molecule_'.&get_uniq_name();
    my $result=<<CHEMINPUT;
<script type="text/javascript">
    function $function() {
	editor=window.open($nothing,'jmeedit','width=500,height=500,menubar=no,scrollbars=no,resizable=yes');
        if (editor) {
	    editor.$docopen;
	    editor.document.write('$start_page $body $resizejs $end_page');
	    editor.document.close();
	    editor.focus();
        }
    }
</script>
CHEMINPUT
    my $jscall = "javascript:$function();void(0);";
    if ($shown_text eq '') {
        $result .=<<PENCIL;
<a href="$jscall"><img class="stift" src="$iconpath/stift.gif" alt="$display" title="$display" /></a>
PENCIL
    } else {
        $result .= '<input type="button" value="'.&mt($shown_text).'" onclick="$jscall" />';
    }
    return $result;
}
sub jme_img {
    my ($jme,$smile,$width,$options)=@_;
    my $id=&Apache::loncommon::get_cgi_id();
    my $result='<img alt="'.$smile.'" src="/cgi-bin/convertjme.pl?'.$id.'"';
    if ($options =~ /border/) { $result.= ' border="1"'; }
    $result.=' />';
    &Apache::lonnet::appenv({'cgi.'.$id.'.JME'   =>
			     &escape($jme),
			     'cgi.'.$id.'.PNG'   => 1,
			     'cgi.'.$id.'.WIDTH' => $width});
    return $result;
}

sub start_organicresponse {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    my $partid = $Apache::inputtags::part;
    my $id = &Apache::response::start_response($parstack,$safeeval);
    if ($target eq 'meta') {
	$result=&Apache::response::meta_package_write('organicresponse');
    } elsif ($target eq 'web') {
	my $jmeanswer=&Apache::lonxml::get_param('jmeanswer',$parstack,
						 $safeeval);
	if (&Apache::response::show_answer()) {
            my $jmeanswer=&Apache::lonxml::get_param('jmeanswer',$parstack,
                                                     $safeeval);
            if ($jmeanswer ne '') {
	        my $options=&Apache::lonxml::get_param('options',$parstack,
	    					       $safeeval);
	        my $width=&Apache::lonxml::get_param('width',$parstack,
						     $safeeval);
	        my (@answers)=&Apache::lonxml::get_param_var('answer',$parstack,
							     $safeeval);
	        $result.=&jme_img($jmeanswer,$answers[0],$width,$options);
            }
	} else {
	    $result.= '<input type="hidden" name="MOLECULE_'.$id.'" value="" />';
	}
    } elsif ($target eq 'edit') {
	$result .=&Apache::edit::tag_start($target,$token);
	my $options=&Apache::lonxml::get_param('options',$parstack,
					       $safeeval);
	if ($options !~ /multipart/) { $options.=',multipart'; }
	$result .='<span class="LC_nobreak">'.
	    &Apache::edit::text_arg('Starting Molecule:','molecule',
				    $token,40);
	my $molecule=&Apache::lonxml::get_param('molecule',$parstack,
						$safeeval);
	$result .=&separate_jme_window(undef,
		      &Apache::edit::html_element_name('molecule'),
		      $molecule,$options);
	$result .='</span><br /><span class="LC_nobreak">';
	$result .=&Apache::edit::text_arg('Correct Answer:','answer',
					  $token,40);
	$result .='</span><br /><span class="LC_nobreak">';
	$result .=&Apache::edit::text_arg('JME string of the answer - automatically updated by "Insert Answer" in the JME pop-up (click pencil):',
					  'jmeanswer',$token);
	my $jmeanswer=&Apache::lonxml::get_param('jmeanswer',$parstack,
						 $safeeval);
	$result .=&separate_jme_window(
                      &Apache::edit::html_element_name('answer'),
                      &Apache::edit::html_element_name('jmeanswer'),
		      $jmeanswer,$options);
	$result .='</span><br />';
	$result .=&Apache::edit::checked_arg('Options:','options',
				    [ ['autoez','Auto E,Z stereochemistry'],
				      ['multipart','Multipart Structures'],
				      ['nostereo','No stereochemistry'],
				      ['reaction','Is a reaction'],
				      ['number','Able to number atoms'] ],
					     ,$token);
	$result .=&Apache::edit::text_arg('Width of correct answer image:',
					  'width',$token,10);
	$result .=&Apache::edit::end_row().&Apache::edit::start_spanning_row();
    } elsif ($target eq 'modified') {
	my $constructtag=&Apache::edit::get_new_args($token,$parstack,
						     $safeeval,'molecule',
						     'answer','jmeanswer',
						     'options','width');
	if ($constructtag) { $result = &Apache::edit::rebuild_tag($token); }
    }

    return $result;
}

sub end_organicresponse {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;

    my $partid = $Apache::inputtags::part;
    my $id = $Apache::inputtags::response['-1'];

    if ($target eq 'grade' 
	&& &Apache::response::submitted()
	&& $Apache::lonhomework::type eq 'exam') {

	&Apache::response::scored_response($partid,$id);

    } elsif ($target eq 'grade' 
	&& &Apache::response::submitted()
	&& $Apache::lonhomework::type ne 'exam') {

	&Apache::response::setup_params($$tagstack[-1],$safeeval);
	my $response = &Apache::response::getresponse();
	if ( $response =~ /[^\s]/) {
	    my (@answers)=&Apache::lonxml::get_param_var('answer',$parstack,$safeeval);
	    my %previous = &Apache::response::check_for_previous($response,$partid,$id);
	    $Apache::lonhomework::results{"resource.$partid.$id.submission"}=$response;
	    my $ad;
	    foreach my $answer (@answers) {
		&Apache::lonxml::debug("submitted a $response for $answer<br \>\n");
		if ($response eq $answer) {
		    $ad='EXACT_ANS';
		    last;
		} else {
		    $ad='INCORRECT';
		}
	    }
            if ($ad) {
	        if ($Apache::lonhomework::type eq 'survey') {
		    $ad='SUBMITTED';
                } elsif ($Apache::lonhomework::type eq 'surveycred') {
                    $ad='SUBMITTED_CREDIT';
                } elsif ($Apache::lonhomework::type eq 'anonsurvey') {
                    $ad='ANONYMOUS';
                } elsif ($Apache::lonhomework::type eq 'anonsurveycred') {
                    $ad='ANONYMOUS_CREDIT';
                }
            }
	    &Apache::response::handle_previous(\%previous,$ad);
	    $Apache::lonhomework::results{"resource.$partid.$id.awarddetail"}=$ad;
	    $Apache::lonhomework::results{"resource.$partid.$id.molecule"}=$env{"form.MOLECULE_$id"};
	}
    } elsif ($target eq "edit") {
	$result.= &Apache::edit::tag_end($target,$token,'');
    } elsif ($target eq 'answer') {
	my (@answers)=&Apache::lonxml::get_param_var('answer',$parstack,
						     $safeeval);
	$result.=&Apache::response::answer_header('organicresponse');
	foreach my $answer (@answers) {
	    $result.=&Apache::response::answer_part('organicresponse',$answer);
	}
	$result.=&Apache::response::answer_footer('organicresponse');
    }
    if ($target eq 'web') {
	&Apache::response::setup_prior_tries_hash(\&format_prior_answer_organic,
						  ['molecule'])
    }

    if ($target eq 'grade' || $target eq 'web' || $target eq 'answer' || 
	$target eq 'tex' || $target eq 'analyze') {
        my $repetition = &Apache::response::repetition();
	&Apache::lonxml::increment_counter($repetition,"$partid.$id"); # part.response
	if ($target eq 'analyze') {
            $Apache::lonhomework::analyze{"$partid.$id.type"} = 'organicresponse';
            push (@{ $Apache::lonhomework::analyze{"parts"} },"$partid.$id");
	    &Apache::lonhomework::set_bubble_lines();
	}
    }
    if ($target eq 'web' ) {
        my ($showpencil,$shown_text);
        if ($Apache::inputtags::status['-1'] eq 'CAN_ANSWER') {
            $showpencil = 1;
        } elsif (&Apache::response::show_answer()) {
            my $jmeanswer=&Apache::lonxml::get_param('jmeanswer',$parstack,                                                         $safeeval);
            if ($jmeanswer eq '') {
                $showpencil = 1;
                $shown_text="Show Your Last Answer";
            }
        }
        if ($showpencil) {
            my $options=&Apache::lonxml::get_param('options',$parstack,
                                                   $safeeval);

	    my $molecule;
	    if (defined($Apache::lonhomework::history{"resource.$partid.$id.molecule"})) {
		$molecule=$Apache::lonhomework::history{"resource.$partid.$id.molecule"};
	    } else {
		$molecule=&Apache::lonxml::get_param('molecule',$parstack,
						     $safeeval);
	    }
	    $result.=&separate_jme_window("HWVAL_$id","MOLECULE_$id",$molecule,
                                          $options,$shown_text);
        }
    }
    &Apache::response::end_response();
    return $result;
}

sub format_prior_answer_organic {
    my ($mode,$answer,$other_data) = @_;
    my $result=&mt('Smile representation: [_1]','"<tt>'.$answer.'</tt>"');
    my $jme=$other_data->[0];
    $result.=&jme_img($jme,$answer,400);
    return $result;
}

sub start_organicstructure {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    if ($target eq 'web') {
	my $width=&Apache::lonxml::get_param('width',$parstack,$safeeval);
	my $molecule=&Apache::lonxml::get_param('molecule',$parstack,$safeeval);
	my $options=&Apache::lonxml::get_param('options',$parstack,$safeeval);
	my $id=&Apache::loncommon::get_cgi_id();
	$result="<img src='/cgi-bin/convertjme.pl?$id'";
	if ($options =~ /border/) { $result.= ' border="1"'; }
	$result.=' />';
	&Apache::lonnet::appenv(
            {'cgi.'.$id.'.JME'   => &escape($molecule),
	     'cgi.'.$id.'.PNG' => 1,
	     'cgi.'.$id.'.WIDTH' => $width});
    } elsif ($target eq 'tex') {
	my $texwidth=&Apache::lonxml::get_param('texwidth',$parstack,$safeeval,undef,1);
	my $webwidth=&Apache::lonxml::get_param('width', $parstack, $safeeval);
	my $webheight=&Apache::lonxml::get_param('height', $parstack, $safeeval);
	if (!$webheight) { $webheight = $webwidth; }
	if (!$texwidth) { $texwidth='90'; }
	$result = "%DYNAMICIMAGE:$webwidth:$webheight:$texwidth\n";
	my $molecule=&Apache::lonxml::get_param('molecule',$parstack,$safeeval);
	my $options=&Apache::lonxml::get_param('options',$parstack,$safeeval);
	my $filename = $env{'user.name'}.'_'.$env{'user.domain'}.
	    '_'.time.'_'.$$.int(rand(1000)).'_organicstructure';
	my $id=$filename;
	&Apache::lonnet::appenv(
		     {'cgi.'.$id.'.JME'   => &escape($molecule),
		      'cgi.'.$id.'.PS' => 1,
		      'cgi.'.$id.'.WIDTH' => $texwidth});
	$id=&escape($id);
	&Apache::lonxml::register_ssi("/cgi-bin/convertjme.pl?$id");
	if ($options =~ /border/) { $result.= '\fbox{'; }
	$result .= '\graphicspath{{'.LONCAPA::tempdir().
	    '}}\includegraphics[width='.$texwidth.' mm]{'.$filename.'.eps}';
	if ($options =~ /border/) { $result.= '} '; }
    } elsif ($target eq 'edit') {
	$result .=&Apache::edit::tag_start($target,$token);
	$result .=&Apache::edit::text_arg('Width (pixels):','width',$token,5);
	$result .=&Apache::edit::text_arg('TeXwidth (mm):','texwidth',$token,5);
	$result .='<span class="LC_nobreak">';
	$result .=&Apache::edit::text_arg('Molecule:','molecule',$token,40);
	my $molecule=&Apache::lonxml::get_param('molecule',$parstack,
						$safeeval);
	my $options=&Apache::lonxml::get_param('options',$parstack,
					       $safeeval);
	if ($options !~ /reaction/) {
	    $options.= ',multipart,number';
	}
						   
	$result .=&separate_jme_window(undef,
				 &Apache::edit::html_element_name('molecule'),
				       $molecule,$options);
	$result.="</span><br />";
	$result .=&Apache::edit::checked_arg('Options:','options',
					     [ ['reaction','Is a reaction'],
					       ['border','Draw a border'] ],
					     $token);
	$result .=&Apache::edit::end_row();
    } elsif ($target eq 'modified') {
	my $constructtag=&Apache::edit::get_new_args($token,$parstack,
						     $safeeval,'molecule',
						     'width','texwidth',
						     'options');
	if ($constructtag) { $result = &Apache::edit::rebuild_tag($token); }
    }
    return $result;
}

sub end_organicstructure {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    if ($target eq "edit") {
	$result.= &Apache::edit::tag_end($target,$token,'');
    }
    return $result;
}

sub edit_reaction_button {
    my ($id,$field,$reaction)=@_;
    my $id_es=&escape($id);
    my $field_es=&escape($field);
    my $reaction_es=&escape($reaction);
    my $docopen=&Apache::lonhtmlcommon::javascript_docopen();
    my $iconpath=$Apache::lonnet::perlvar{'lonIconsURL'};
    my $display=&mt('Edit Answer');
    my $start_page =
        &Apache::loncommon::start_page('LON-CAPA Reaction Editor',undef,
                                       {'frameset'    => 1,
                                        'js_ready'    => 1,
                                        'add_entries' => {
                                            'rows'   => "30%,*",
                                            'border' => "0",}},);
    my $end_page =
        &Apache::loncommon::end_page({'frameset' => 1,
                                      'js_ready' => 1});
    my $result=<<EDITREACTION;
<script type="text/javascript">
// <!--
    function create_reaction_window_${id}_${field} () {
        editor=window.open('','','width=500,height=270,scrollbars=no,resizable=yes');
        editor.$docopen;
        editor.document.writeln('$start_page <frame src="/adm/reactionresponse/reaction_viewer.html?inhibitmenu=yes" name="viewer" scrolling="no" />  <frame src="/adm/reactionresponse/reaction_editor.html?inhibitmenu=yes&reaction=$reaction_es&id=$id_es&field=$field_es" name="editor" scrolling="no" /> $end_page');
        editor.document.close();
    }
// -->
</script>
<a href="javascript:create_reaction_window_${id}_${field}();void(0);"><img class="stift" src='$iconpath/stift.gif' alt='$display' title='$display' /></a>
EDITREACTION
    return $result;
}

sub reaction_preview {
    my ($field, $reaction) = @_;
    
    # NOTE: $reaction should be encoded if the document was sent as XHTML
    $reaction =~ s/"//g;
    my $result=<<JS_PREVIEW;
<input type="button" value="Help" onclick = "window.open('/adm/reactionresponse/reaction_help.html','','scrollbars=yes,resizable=yes,width=550,height=600')" />
<script type="text/javascript">
    if (typeof reaction_preview_started === 'undefined') {
        var script = document.createElement('script');
        script.type = 'text/javascript';
        script.src = '/adm/reactionresponse/reaction_preview.js';
        document.body.appendChild(script);
        reaction_preview_started = true;
    }
    window.addEventListener('load', function(e) {
        var input = document.forms.lonhomework.$field;
        input.readonly = '';
        input.value = "$reaction";
        if (!input.id)
            input.id = "$field";
        var preview = new LC.HW.ReactionPreview(input.id);
        preview.start_preview();
    }, false);
</script>
JS_PREVIEW
    return $result;
}

sub start_reactionresponse {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;
    my $id = &Apache::response::start_response($parstack,$safeeval);
    if ($target eq 'meta') {
	$result=&Apache::response::meta_package_write('reactionresponse');
    } elsif ($target eq 'web') {
	my $partid = $Apache::inputtags::part;
	my $id = $Apache::inputtags::response['-1'];
	if (  &Apache::response::show_answer() ) {
	    my $ans=&Apache::lonxml::get_param('answer',$parstack,$safeeval);
	    if (!$Apache::lonxml::default_homework_loaded) {
		&Apache::lonxml::default_homework_load($safeeval);
	    }
	    @Apache::scripttag::parser_env = @_;
	    $Apache::inputtags::answertxt{$id}=[&Apache::run::run("return &chemparse(q\0$ans\0);",$safeeval)];
	}
    } elsif ($target eq "edit") {
	$result .=&Apache::edit::tag_start($target,$token);
	my $answer=&Apache::lonxml::get_param('answer',$parstack,
						$safeeval);
	$result .='<span class="LC_nobreak">'.
	    &Apache::edit::text_arg('Answer:','answer',$token,40);
        my $inline_chem = &use_inline_chem();
        if ($inline_chem) {
	    $result .= &reaction_preview(&Apache::edit::html_element_name('answer'), $answer).'</span>';
        } else {
            $result .=&edit_reaction_button($id,&Apache::edit::html_element_name('answer'),$answer).'</span>';
        }
	my $initial=&Apache::lonxml::get_param('initial',$parstack,$safeeval);
	$result.='<span class="LC_nobreak">'.
	    &Apache::edit::text_arg('Initial Reaction:','initial',$token,40);
        if ($inline_chem) { 
            $result .= &reaction_preview(&Apache::edit::html_element_name('initial'), $initial).'</span>';
        } else {
            $result .=&edit_reaction_button($id,&Apache::edit::html_element_name('initial'),$initial).'</span>';
        }
	$result .=&Apache::edit::end_row().&Apache::edit::start_spanning_row();
    }  elsif ($target eq 'modified') {
	my $constructtag=&Apache::edit::get_new_args($token,$parstack,
						     $safeeval,'answer',
						     'initial');
	if ($constructtag) { $result = &Apache::edit::rebuild_tag($token); }
    }
    return $result;
}

sub end_reactionresponse {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    my $result;

    my $partid = $Apache::inputtags::part;
    my $id = $Apache::inputtags::response['-1'];

    if ($target eq 'grade' 
	&& &Apache::response::submitted()
	&& $Apache::lonhomework::type eq 'exam') {

	&Apache::response::scored_response($partid,$id);

    } elsif ($target eq 'grade' 
	&& &Apache::response::submitted()
	&& $Apache::lonhomework::type ne 'exam') {

	&Apache::response::setup_params($$tagstack[-1],$safeeval);
	my $response = &Apache::response::getresponse();
	if ( $response =~ /[^\s]/) {
	    my (@answers)=&Apache::lonxml::get_param_var('answer',$parstack,$safeeval);
	    my %previous = &Apache::response::check_for_previous($response,$partid,$id);
	    $Apache::lonhomework::results{"resource.$partid.$id.submission"}=$response;
	    my $ad;
	    foreach my $answer (@answers) {
		&Apache::lonxml::debug("submitted a $response for $answer<br \>\n");
		if (&chem_standard_order($response) eq 
		    &chem_standard_order($answer)) {
		    $ad='EXACT_ANS';
		} else {
		    $ad='INCORRECT';
		}
	    }
            if ($ad) {
                if ($Apache::lonhomework::type eq 'survey') {
		    $ad='SUBMITTED';
	        } elsif ($ad && $Apache::lonhomework::type eq 'surveycred') {
                    $ad='SUBMITTED_CREDIT';
                } elsif ($ad && $Apache::lonhomework::type eq 'anonsurvey') {
                    $ad='ANONYMOUS';
                } elsif ($ad && $Apache::lonhomework::type eq 'anonsurveycred') {
                    $ad='ANONYMOUS_CREDIT';
                }
            }
	    &Apache::response::handle_previous(\%previous,$ad);
	    $Apache::lonhomework::results{"resource.$partid.$id.awarddetail"}=$ad;
	}
    }  elsif ($target eq "edit") {
	$result.= &Apache::edit::tag_end($target,$token,'');
    } elsif ($target eq 'answer') {
	my (@answers)=&Apache::lonxml::get_param_var('answer',$parstack,
						     $safeeval);
	$result.=&Apache::response::answer_header('reactionresponse');
	foreach my $answer (@answers) {
	    $result.=&Apache::response::answer_part('reactionresponse',
						    $answer);
	}
	$result.=&Apache::response::answer_footer('reactionresponse');
    }
    if ($target eq 'web') {
	&Apache::response::setup_prior_tries_hash(\&format_prior_response_reaction);
    }

    if ($target eq 'grade' || $target eq 'web' || $target eq 'answer' || 
	$target eq 'tex' || $target eq 'analyze') {
	my $repetition = &Apache::response::repetition();
        &Apache::lonxml::increment_counter($repetition,"$partid.$id");
        if ($target eq 'analyze') {
            $Apache::lonhomework::analyze{"$partid.$id.type"} = 'reactionresponse';
            push (@{ $Apache::lonhomework::analyze{"parts"} },"$partid.$id");
            &Apache::lonhomework::set_bubble_lines();
        }
    }
    my $status=$Apache::inputtags::status['-1'];
    if  (($target eq 'web') && ($Apache::lonhomework::type ne 'exam') && ($status eq 'CAN_ANSWER')) {
        my $reaction=$Apache::lonhomework::history{"resource.$partid.$id.submission"};
        if ($reaction eq '') {  $reaction=&Apache::lonxml::get_param('initial',$parstack,$safeeval); }
        if (&use_inline_chem()) {
            $result .= &reaction_preview("HWVAL_$id", $reaction);
        } else {
            $result.=&edit_reaction_button($id,"HWVAL_$id",$reaction);
        }
    }
    &Apache::response::end_response();
    return $result;
}

sub use_inline_chem {
    my $inline_chem = 1;
    if (($env{'request.course.id'}) && ($env{'request.state'} ne 'construct')) {
        if (exists($env{'course.'.$env{'request.course.id'}.'.inline_chem'})) {
            if ($env{'course.'.$env{'request.course.id'}.'.inline_chem'} eq '0') {
                $inline_chem = 0;
            }
        } else {
            my %domdefs = &Apache::lonnet::get_domain_defaults($env{'course.'.$env{'request.course.id'}.'.domain'});
            if ($domdefs{'inline_chem'} eq '0') {
                $inline_chem = 0;
            }
        }
    } else {
        my %domdefs = &Apache::lonnet::get_domain_defaults($env{'course.'.$env{'request.course.id'}.'.domain'});
        if ($domdefs{'inline_chem'} eq '0') {
            $inline_chem = 0;
        }
    }
    return $inline_chem;
}

sub format_prior_response_reaction {
    my ($mode,$answer) =@_;
    return '<span class="LC_prior_reaction">'.
	    &HTML::Entities::encode($answer,'"<>&').'</span>';
}

sub start_chem {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style) = @_;
    my $result = '';
    my $inside = &Apache::lonxml::get_all_text_unbalanced("/chem",$parser);
    if ($target eq 'tex' || $target eq 'web') {
	$inside=&Apache::run::evaluate($inside,$safeeval,$$parstack[-1]);
	if (!$Apache::lonxml::default_homework_loaded) {
	    &Apache::lonxml::default_homework_load($safeeval);
	}
	@Apache::scripttag::parser_env = @_;
	$result=&Apache::run::run("return &chemparse(q\0$inside\0);",$safeeval);
    }    
    return $result;
}

sub end_chem {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style) = @_;
    my $result = '';
    return $result;
}

my $uniq=0;
sub get_uniq_name {
    $uniq++;
    return 'uniquename'.$uniq;
}

1;
__END__
