# The LearningOnline Network
# Personal Information Page
#
# $Id: lonaboutme.pm,v 1.161 2021/11/30 15:55:37 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#

=pod

=head1 NAME

pache::lonaboutme

=head1 SYNOPSIS

(empty)

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 OVERVIEW

(empty)


=head1 SUBROUTINES

=over

=item handler()

=item aboutme_info()

=item print_portfiles_link()

=item build_query_string()

=item display_portfolio_header()

=item display_portfolio_files()

=item portfolio_files()

=item build_hierarchy()

=item parse_directory()

=back

=cut


package Apache::lonaboutme;

use strict;
use Apache::Constants qw(:common);
use Apache::loncommon;
use Apache::lonnet;
use Apache::lontexconvert;
use Apache::lonhtmlgateway;
use Apache::lonrss();
use Apache::lonlocal;
use Apache::lonmsgdisplay();
use Apache::lontemplate;
use Apache::longroup;
use HTML::Entities();
use Image::Magick;

sub handler {
    my $r = shift;
    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;
    return OK if $r->header_only;
    my $target=$env{'form.grade_target'};
# ------------------------------------------------------------ Print the screen
    if ($target eq 'tex') {
        $r->print(&Apache::lonprintout::print_latex_header($env{'form.latex_type'}));
    }
    my (undef,undef,$cdom,$cnum,undef,$action)=split(/\//,$r->uri);
    my $is_course;
# Is this even a user?
    if (&Apache::lonnet::homeserver($cnum,$cdom) eq 'no_host') {
        &Apache::loncommon::simple_error_page($r,'No info',
            'No user information available');
        return OK;
    } else {
        $is_course = &Apache::lonnet::is_course($cdom,$cnum);
    }

    my $clientip = &Apache::lonnet::get_requestor_ip($r);
    my $candisplay = 1;
    if (!$is_course) {
        if ($action ne 'portfolio') {
            $candisplay = &Apache::lonnet::usertools_access($cnum,$cdom,'aboutme');
            if ((!$candisplay) && ($env{'request.course.id'})) {
                $candisplay = &aboutme_access($cnum,$cdom);
            }
            if (!$candisplay) {
                if ($target eq 'tex') {
                    $r->print('\noindent{\large\textbf{'.&mt('No user personal information page available').'}}\\\\\\\\');
                } else {
                    $r->print(&Apache::loncommon::start_page("Personal Information Page"));
                    $r->print('<p class="_LC_info">'.&mt('No user personal information page available') .'</p>'.
                        &mt('This is a result of one of the following:').'<ul>'.
                        '<li>'.&mt('The administrator of this domain has disabled personal information page functionality for this specific user.').'</li>'.
                        '<li>'.&mt('The domain has been configured to disable, by default, personal information page functionality for all users in the domain.').'</li>'.
                        '</ul>');
                    $r->print(&Apache::loncommon::end_page());
                }
                return OK;
            }
        }
    }

# --------------------------------------------------------- The syllabus fields
    my %syllabusfields=&Apache::lonlocal::texthash(
       'aaa_contactinfo'   => 'Contact Information',
       'bbb_aboutme'       => 'Personal Information',
       'ccc_webreferences' => 'Web References');

# ------------------------------------------------------------ Get Query String
    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},
                        ['forceedit','forcestudent','todocs',
                         'register','popup','folderpath','title']);
# ----------------------------------------------- Available Portfolio file display
    if (($target ne 'tex') && ($action eq 'portfolio')) {
        &display_portfolio_header($r,$is_course);
        if ((!$is_course) && (!&Apache::lonnet::usertools_access($cnum,$cdom,'portfolio'))) {
            $r->print('<h2>'.&mt('No user portfolio available') .'</h2>'.
                &mt('This is a result of one of the following:').'<ul>'.
                '<li>'.&mt('The administrator of this domain has disabled portfolio functionality for this specific user.').'</li>'.
                '<li>'.&mt('The domain has been configured to disable, by default, portfolio functionality for all users in the domain.').'</li>'.
                '</ul>');
        } else {
            my ($blocked,$blocktext) =
                &Apache::loncommon::blocking_status('port',$clientip,$cnum,$cdom);
            if (!$blocked) {
                &display_portfolio_files($r,$is_course);
            } else {
                $r->print($blocktext);
            }
        }
        $r->print(&Apache::loncommon::end_page());
        return OK;
    }

    if ($is_course) {
        if ($target ne 'tex') {
            my $args = {'function'       => undef,
                        'domain'         => $cdom};
            if ($env{'form.register'}) {
                $args->{'force_register'} = $env{'form.register'};
            } else {
                my %coursedescription = 
                    &Apache::lonnet::coursedescription($cdom.'_'.$cnum);
                my $cdescr = $coursedescription{'description'};
                my $brcrum = [{href=>"/adm/$cdom/$cnum/aboutme",
                               text=>&mt('Course Information - [_1]',$cdescr),
                               no_mt=>1}
                             ];
                $args->{'bread_crumbs'} = $brcrum;
            }
            my $start_page = &Apache::loncommon::start_page(
                                 "Course Information",undef,$args);
            $r->print($start_page);
            $r->print('<h2>'.&mt('Group Portfolio').'</h2>');
            &print_portfiles_link($r,$is_course);
            $r->print(&Apache::loncommon::end_page());
        }
        return OK;
    }

#------------Get rights
    my %courseenv=&Apache::lonnet::dump('environment',$cdom,$cnum);
    my %syllabus=&Apache::lonnet::dump('aboutme',$cdom,$cnum);
    my ($allowed,$coursedomain,$coursenum);
    if ($env{'request.course.id'}) {
        $coursedomain = $env{'course.'.$env{'request.course.id'}.'.domain'};
        $coursenum = $env{'course.'.$env{'request.course.id'}.'.num'};
    }
    my ($cfile) = 
        &Apache::lonnet::can_edit_resource($env{'request.noversionuri'},
                                           $coursenum,$coursedomain,
                                           $env{'request.noversionuri'},
                                           $env{'request.symb'});
    if ($cfile ne '') {
        $allowed = 1;
    }

    if (!$env{'form.forceedit'} or $target eq 'tex') { $allowed=0; }

# --------------------------------------- There is such a user, get environment

    if ($target ne 'tex') {
        my $rss_link = &Apache::lonrss::rss_link($cnum,$cdom);
        my $args = {'function' => undef,
                    'domain'   => $cdom,
                    'force_register' => $env{'form.register'},
                   };
        if ($env{'form.popup'}) { # Don't show breadcrumbs in popup window 
            $args->{'no_nav_bar'} = 1;
        } elsif (!$env{'form.register'}) { #Don't show breadcrumbs twice, when this page is part of course content and you call it
            if (($env{'request.course.id'}) &&
                 ($env{'form.folderpath'} =~ /^supplemental/)) {
                my $crstype = &Apache::loncommon::course_type();
                my $title = $env{'form.title'};
                if ($title eq '') {
                    $title = &mt('Personal Information Page');
                }
                my $brcrum =
                    &Apache::lonhtmlcommon::docs_breadcrumbs(undef,$crstype,undef,$title,1);
                if (ref($brcrum) eq 'ARRAY') {
                    $args->{'bread_crumbs'} = $brcrum;
                }
            } else {
                $args->{'bread_crumbs'} = [{href=>"/adm/$cdom/$cnum/aboutme",
                                            text=>"Personal Information Page"}];
            }
        }
        my $start_page = &Apache::loncommon::start_page('Personal Information Page',$rss_link,$args);
        $r->print($start_page);
   }
   my ($blocked,$blocktext) =
       &Apache::loncommon::blocking_status('about',$clientip,$cnum,$cdom);
   if ($blocked) {
       if ($target eq 'tex') {
           $r->print('\noindent{\large\textbf{'.&mt('No user personal information page available').'}}\\\\\\\\');
       } else {
           $r->print($blocktext);
       }
       $r->print(&Apache::loncommon::end_page());
       return OK;
   }

#----------------Print Privacy note (edit mode) or last modified date. 

    if ($target ne 'tex') {
        #Print Privacy Note
        if ($allowed) {
            $r->print('<div class="LC_info">'
                .'<b>'.&mt('Privacy Note:').'</b> '
                .&mt('The information you submit can be viewed by anybody who is logged into LON-CAPA. Do not provide information that you are not ready to share publicly.')
                .'</div>'
            );
        } elsif ($syllabus{'uploaded.lastmodified'}) {
        #Print last modified
            my $lastmod=$syllabus{'uploaded.lastmodified'};
            $lastmod=($lastmod?&Apache::lonlocal::locallocaltime($lastmod):&mt('never'));
            $r->print('<div class="LC_info">');
            $r->print(&mt('Last updated').': '.$lastmod . '');
            $r->print('</div>');
        }
    }

#------Print Headtitle
     if ($target ne 'tex') {
         $r->print('<div class="LC_Box">'.
                   '<h2 class="LC_hcell">'.&Apache::loncommon::plainname($cnum,$cdom).'</h2>');
         if ($allowed) {
             $r->print('<div style="margin: 0; float:left;">');
             if ($courseenv{'nickname'}) {
                 $r->print('<h2>&quot;'.$courseenv{'nickname'}.'&quot;</h2>');
             }
             $r->print('<h3>'.&Apache::lonnet::domain($cdom,'description').'</h3></div>');
             #Print Help Text
             $r->print('<div style="margin: 0; float:right;">'.
                       &Apache::loncommon::help_open_topic('Uploaded_Templates_TextBoxes',&mt('Help with filling in text boxes')).
                       '</div><br clear="all" />');
         } else {
             if ($courseenv{'nickname'}) {
                 $r->print('<h2>&quot;'.$courseenv{'nickname'}.'&quot;</h2>');
             }
             $r->print('<h3>'.&Apache::lonnet::domain($cdom,'description').'</h3>');
         }
     } else {
        $r->print('\noindent{\large\textbf{'.&Apache::loncommon::plainname($cnum,$cdom).'}}\\\\\\\\');
        $r->print('\textbf{'.&Apache::lonnet::domain($cdom,'description').'}\\\\');
    }
# does this user have privileges to post, etc?


    my $query_string;

    if (($env{'form.uploaddoc.filename'}) &&
          ($env{'form.storeupl'}) && ($allowed)) {
        if ($env{'form.uploaddoc.filename'}=~/\.(gif|jpg|png|jpeg)$/i) {
            if ($syllabus{'uploaded.photourl'}) {
                &Apache::lonnet::removeuploadedurl($syllabus{'uploaded.photourl'});
            }
            $syllabus{'uploaded.photourl'}=
                &Apache::lonnet::userfileupload('uploaddoc',undef,'aboutme',
                    undef,undef,undef,undef,undef,undef,undef,'400','500');
         }
        $syllabus{'uploaded.lastmodified'}=time;
        &Apache::lonnet::put('aboutme',\%syllabus,$cdom,$cnum);
    }
    if ($allowed && $env{'form.delupl'}) {
        if ($syllabus{'uploaded.photourl'}) {
            &Apache::lonnet::removeuploadedurl($syllabus{'uploaded.photourl'});
            delete($syllabus{'uploaded.photourl'});
            &Apache::lonnet::del('aboutme',['uploaded.photourl'],$cdom,$cnum);
        }
    }
    if (($allowed) && ($env{'form.storesyl'})) {
        foreach my $syl_field (keys(%syllabusfields)) {
            my $field=$env{'form.'.$syl_field};
            chomp($field);
            my $gateway = Apache::lonhtmlgateway->new();
            $field = $gateway->process_incoming_html($field,1);
            $syllabus{$syl_field}=$field;
        }
        $syllabus{'uploaded.lastmodified'}=time;
        &Apache::lonnet::put('aboutme',\%syllabus,$cdom,$cnum);
    }

    my $image;
# ---------------------------------------------------------------- Get syllabus
    if (($syllabus{'uploaded.lastmodified'}) || ($allowed)) {
        if ($syllabus{'uploaded.photourl'}) {
            &Apache::lonnet::allowuploaded('/adm/aboutme',$syllabus{'uploaded.photourl'});

            $image=qq|<img name="userPhoto" src="$syllabus{'uploaded.photourl'} " class="LC_AboutMe_Image" alt="Photo of the user" />|;

            if ($target eq 'tex') {
                $image=&Apache::lonxml::xmlparse($r,'tex',$image);
            }
        }

        if ($allowed) {
            $r->print(
                '<form name="UploadPhoto" method="post" enctype="multipart/form-data" action="">'.
                '<h3>'.&mt('Upload a Photo').'</h3>'.
                '<p class="LC_info">'.
                &mt('LON-CAPA will automatically scale your uploaded file so the image will not exceed a width of 400px and a height of 500px.').'</p>'.
                '<input type="file" name="uploaddoc" size="50" />'.
                '<input type="submit" name="storeupl" value="'.&mt('Upload').'" />'.
                '<input type="hidden" name="popup" value="'.$env{'form.popup'}.'" />'.
                '</form>');
            if ($syllabus{'uploaded.photourl'}) {
                $r->print('<form name="delPhoto" method="post" action="" ><input type="submit" name="delupl" value="'.&mt('Delete Photo').'" /> </form>')
            }
            $r->print('<p></p>');
        }

        if($allowed) {
            $r->print('<form name="lonaboutmeFields" method="post" action="" >');
        }

        if ($target ne 'tex') { #print Image
            $r->print($image.'<div class="LC_clear_float_footer"></div>');

        } #End Print Image

       #Print Content eg. Contactinfo aboutme,...
        &Apache::lontemplate::print_aboutme_content_template($r,$allowed,$target,\%syllabusfields,\%syllabus);
       #End Print Content

        if ($target ne 'tex') { #Begin Print RSS and portfiles
            &print_portfiles_link($r,$is_course);
            if (&Apache::lonrss::advertisefeeds($cnum,$cdom) ne '') {
                &Apache::lontemplate::print_start_template($r,'RSS Feeds and Blogs','LC_Box');
                $r->print(&Apache::lonrss::advertisefeeds($cnum,$cdom));
                &Apache::lontemplate::print_end_template($r);
            }

        } #End  Print RSS and portfiles


        if ($allowed) {
            if ($env{'form.popup'}) {
                $r->print('<input type="hidden" name="popup" value="'.
                    $env{'form.popup'}.'" />');
            }
            $r->print('</form>');
        }
        if ($target ne 'tex') {$r->print('<br />');} else {$r->print('\\\\');}
    } else {
        $r->print('<p class="LC_info">'.&mt('No personal information provided').'.</p>');
    }

    if ($env{'request.course.id'}
        && &Apache::lonnet::allowed('srm',$env{'request.course.id'})
        && &Apache::lonnet::in_course($cdom,$cnum,$coursedomain,$coursenum,undef,1)) {
        if ($target ne 'tex') {
            $r->print('<a name="coursecomment" />');
            &Apache::lontemplate::print_start_template($r,&mt('User Notes, Records of Face-To-Face Discussions, and Critical Messages in Course'),'LC_Box');
            $r->print('<span class="LC_info">');
            $r->print(&mt('Shared by course faculty and staff').&Apache::loncommon::help_open_topic("Course_Face_To_Face_Records,Course_Critical_Message"));
            $r->print('</span>');
            &Apache::lonmsgdisplay::disfacetoface($r,$cnum,$cdom);
            &Apache::lontemplate::print_end_template($r);

        } else {
            $r->print('\\\\\textbf{'.&mt('User Notes, Records of Face-To-Face Discussions, and Critical Messages in Course').'}\\\\'.&mt('Shared by course faculty and staff').'\\\\\\\\');
            &Apache::lonmsgdisplay::disfacetoface($r,$cnum,$cdom,$target);
        }
    }
    if ($target ne 'tex') {
        $r->print('</div>');
        if ($env{'form.popup'}) {
            $r->print('<p><a href="javascript:window.close()">'.&mt('Close window').'</a></p>');
        }
        $r->print(&Apache::loncommon::end_page());
    } else {
        $r->print('\end{document}');
    }



    return OK;
}

sub aboutme_info {
    my ($r,$is_course) = @_;
    my (undef,undef,$cdom,$cnum)=split(/\//,$r->uri);
    my $name;
    if (!$is_course) {
        $name = &Apache::loncommon::plainname($cnum,$cdom);
    }
    return ($cdom,$cnum,$name);
}

sub print_portfiles_link {
    my ($r,$is_course) = @_;
    my ($cdom,$cnum,$name) = &aboutme_info($r,$is_course);
    my $filecounts = &portfolio_files($r,'showlink',undef,$is_course,
                                      $cdom,$cnum,$name);
    my $query_string = &build_query_string();
    my $output;
    my %lt = &Apache::lonlocal::texthash(
        'vpfi' => 'Viewable portfolio files',
        'vgpf' => 'Viewable group portfolio files',
        'difl' => 'Display file listing',
    );
    if ($filecounts->{'both'} > 0) {
        $output = '<div class="LC_Box"><h4 class="LC_hcell">';
        $output .= ($is_course?$lt{'vgpf'}:$lt{'vpfi'}).'</h4>';

       #$output = '<h4>'.($is_course?$lt{'vgpf'}:$lt{'vpfi'}).'</h4>';
        $output .= '<a href="/adm/'.$cdom.'/'.$cnum.'/aboutme/portfolio'.
            $query_string.'">'.$lt{'difl'}.
            '</a><br /><br />';
        if ($filecounts->{'both'} == 1) {
            if ($is_course) {
                $output .= &mt('One group portfolio file is available.').'<ul>';
            } else {
                $output .= &mt('One portfolio file owned by [_1] is available.',$name).'<ul>';
            }
        } else {
            if ($is_course) {
                $output .= &mt('A total of [_1] group portfolio files are available.',$filecounts->{'both'}).'<ul>';
            } else {
                $output .= &mt('A total of [_1] portfolio files owned by [_2] are available.',$filecounts->{'both'},$name).'<ul>';
            }
        }
        if ($filecounts->{'withoutpass'}) {
            $output .= '<li>'.&mt('[quant,_1,file is,files are] publicly accessible.',$filecounts->{'withoutpass'}).'</li>';
        }
        if ($filecounts->{'withpass'}) {
            $output .= '<li>'.&mt('[quant,_1,file requires,files require] a passphrase for access.',$filecounts->{'withpass'}).'</li>';
        }
        $output .= '</ul>';
        $output .= '</div>';
    } elsif ($is_course) {
        $output .= '<div class="LC_info">'.&mt('There are currently no publicly accessible or password protected group portfolio files.').'</div>'; 
    }
    $r->print($output);
    return;
}

sub build_query_string {
    my ($new_items) = @_;
    my $query_string;
    my @formelements = ('register');
    my $new = 0;
    if (ref($new_items) eq 'HASH') {
        $new = 1;
        if (!defined($new_items->{'forceedit'}) &&
            !defined($new_items->{'forcestudent'})) {
            push(@formelements,('forceedit','forcestudent'));
        }
    } else {
        push(@formelements,('forceedit','forcestudent'));
    }
    foreach my $element (@formelements) {
        if (exists($env{'form.'.$element})) {
            if ((!$new) || (!defined($new_items->{$element}))) {
                $query_string .= '&amp;'.$element.'='.$env{'form.'.$element};
            }
        }
    }
    if ($new) {
        foreach my $key (keys(%{$new_items})) {
            $query_string .= '&amp;'.$key.'='.$new_items->{$key};
        }
    }
    $query_string =~ s/^\&amp;/\?/;
    return $query_string;
}

sub display_portfolio_header {
    my ($r,$is_course) = @_;
    my ($cdom,$cnum,$name) = &aboutme_info($r,$is_course);
    my $query_string = &build_query_string();
    my $args = {'domain' => $cdom};
    if ($env{'form.forcestudent'}) {
        $args->{'function'} = 'student';
    }
    my $output;
    if ($is_course) {
        if (($env{'request.course.id'} eq $cdom.'_'.$cnum) && 
            ($env{'form.register'})) {
            $args->{force_register} = $env{'form.register'};
        } else {
            my %coursedescription = &Apache::lonnet::coursedescription($cdom.'_'.$cnum);
            my $cdescr = $coursedescription{'description'}; 
            my $brcrum = [{href=>"/adm/$cdom/$cnum/aboutme".$query_string,
                           text=>&mt('Course Information - [_1]',$cdescr),
                           no_mt=>1},
                          {href=>"/adm/$cdom/$cnum/aboutme/portfolio".$query_string,
                           text=>'Viewable group portfolio files'}
                         ];
            $args->{bread_crumbs} = $brcrum;
        }
        $output = &Apache::loncommon::start_page('Viewable group portfolio files',undef,$args).
                  '<h3>'.&mt('Group Portfolio files').'</h3>';
    } else {
        if ($env{'request.course.id'} && $env{'form.register'}) {
            $args->{force_register} = $env{'form.register'};
        } else {
            my $brcrum = [{href  => "/adm/$cdom/$cnum/aboutme".$query_string,
                          text  => &mt('Personal Information Page - [_1]',$name),
                          title => &mt('Go to personal information page for [_1]',$name),
                          no_mt => 1},
                         {href  => "/adm/$cdom/$cnum/aboutme/portfolio".$query_string,
                          text  => &mt('Viewable files'),
                          title => &mt('Viewable portfolio files for [_1]',$name),
                          no_mt => 1}
                         ];
            $args->{bread_crumbs} = $brcrum;
        } 
        $output  = 
            &Apache::loncommon::start_page('Viewable portfolio files',
                                           undef,$args).
            '<h3>'.&mt('Portfolio files for [_1]',$name).'</h3>';
    }
    $r->print($output);
    return;
}

sub display_portfolio_files {
    my ($r,$is_course) = @_;
    my ($cdom,$cnum,$name) = &aboutme_info($r,$is_course);
    my %lt = &Apache::lonlocal::texthash(
        'withoutpass' => 'passphrase not required',
        'withpass'    => 'passphrase protected',
        'both'        => 'all access types ',
    );

    my $portaccess = 'withoutpass';
    if (exists($env{'form.portaccess'})) {
        $portaccess = $env{'form.portaccess'};
    }

    my $output = '<form action="'.&HTML::Entities::encode($r->uri,'<>&"')
        .'" name="displaystatus" method="post">'.
        &mt('File access type: ').'<select name="portaccess">';
    foreach my $type ('withoutpass','withpass','both') {
        $output .= '<option value="'.$type.'" ';
        if ($portaccess eq $type) {
            $output .= 'selected="selected"';
        }
        $output .= '>'.$lt{$type}.'</option>';
    }
    $output .= '</select>'."\n";
    if ($env{'form.register'}) {
        $output .= '<input type="hidden" name="register" value="'.$env{'form.register'}.'" />'."\n";
    }
    $output .= '<input type="submit" name="portaccessbutton" value="'.
               &mt('Update display').'" />'.
               '</form><br /><br />';
    $r->print($output);
    my $filecounts = &portfolio_files($r,'listfiles',\%lt,$is_course,
                                      $cdom,$cnum,$name);
    if (!($env{'user.name'} eq 'public' && $env{'user.domain'} eq 'public')) {
        if ($env{'request.course.id'} && $env{'form.register'}) {
            my $query_string = &build_query_string();
            $r->print('<br /><a href="/adm/'.$cdom.'/'.$cnum.
                      '/aboutme'.$query_string.'">');
            if ($is_course) {
                $r->print(&mt('Course Information page'));
            } else {
                $r->print(&mt('Information about [_1]',$name));
            }
            $r->print('</a>');
        }
    }
    return;
}

sub portfolio_files {
    my ($r,$mode,$lt,$is_course,$cdom,$cnum,$name) = @_;
    my $filecounts = {
                       withpass    => 0,
                       withoutpass => 0,
                       both        => 0,
                     };
    my $current_permissions =
    &Apache::lonnet::get_portfile_permissions($cdom,$cnum);
    my %access_controls =
    &Apache::lonnet::get_access_controls($current_permissions);
    my $portaccess;
    if ($mode eq 'showlink') {
        $portaccess = 'both';
    } else {
        $portaccess = 'withoutpass';
        if (exists($env{'form.portaccess'})) {
            $portaccess = $env{'form.portaccess'};
        }
    }

    my $diroutput;
    if ($is_course) {
        my %files_by_group;
        my %curr_groups = &Apache::longroup::coursegroups($cdom,$cnum);
        foreach my $filename (sort(keys(%access_controls))) {
            my ($group,$path) = split('/',$filename,2);
            if (exists($curr_groups{$group})) {
                $files_by_group{$group}{$path} = $access_controls{$filename};
            }
        }
        foreach my $group (sort(keys(%files_by_group))) {
            my %fileshash;
            my $grpout .= &build_hierarchy($r,$cdom,$cnum,$portaccess,
                                           $is_course,$filecounts,$mode,
                                           $files_by_group{$group},
                                           \%fileshash,$group);
            if ($grpout) {
                $diroutput .= '<h3>'.$group.'</h3>'.$grpout.'<br />';
            }
        }
    } else {
        my %allfileshash;
        $diroutput = &build_hierarchy($r,$cdom,$cnum,$portaccess,$is_course,
                                      $filecounts,$mode,\%access_controls,
                                      \%allfileshash);
    }
    if ($mode eq 'listfiles') {
        if ($filecounts->{'both'}) {
             $r->print($diroutput);
        } else {
            my $access_text;
            if (ref($lt) eq 'HASH') {
                $access_text = $lt->{$portaccess};
            }
            $r->print(&mt('There are no available files of the specified access type: [_1]',$access_text));
        }
    }
    return $filecounts;
}

sub build_hierarchy {
    my ($r,$cdom,$cnum,$portaccess,$is_course,$filecounts,$mode,$access_info,
        $allfileshash,$group) = @_;
    my $clientip = &Apache::lonnet::get_requestor_ip($r);
    foreach my $filename (sort(keys(%{$access_info}))) {
        my $access_status =
           &Apache::lonnet::get_portfolio_access($cdom,$cnum,$filename,$group,$clientip,
                                                 $access_info->{$filename});
        if ($portaccess eq 'both') {
            if (($access_status ne 'ok') &&
                ($access_status !~  /^[^:]+:guest_/)) {
                next;
            }
        } elsif ($portaccess eq 'withoutpass') {
            if ($access_status ne 'ok') {
                next;
            }
        } elsif ($portaccess eq 'withpass') {
            if ($access_status !~  /^[^:]+:guest_/) {
                next;
            }
        }
        if ($mode eq 'listfiles') {
            $filename =~ s/^\///;
            my @pathitems = split('/',$filename);
            my $lasthash = $allfileshash;
            while (@pathitems > 1) {
                my $newlevel = shift(@pathitems);
                if (!exists($lasthash->{$newlevel})) {
                    $lasthash->{$newlevel} = {};
                }
                $lasthash = $lasthash->{$newlevel};
            }
            $lasthash->{$pathitems[0]} = $filename;
        }
        if ($access_status eq 'ok') {
            $filecounts->{'withoutpass'} ++;
        } elsif ($access_status =~  /^[^:]+:guest_/) {
            $filecounts->{'withpass'} ++;
        }
    }
    $filecounts->{'both'} =  $filecounts->{'withoutpass'} +
                              $filecounts->{'withpass'};
    my $output;
    if ($mode eq 'listfiles') {
        if ($filecounts->{'both'} > 0) {
            $output = &Apache::loncommon::start_data_table();
            $output .= &parse_directory($r,0,$allfileshash,'',$is_course,
                                        $group);
            $output .= &Apache::loncommon::end_data_table();
        }
    }
    return $output;
}

sub parse_directory {
    my ($r,$depth,$currhash,$path,$is_course,$group) = @_;
    my ($cdom,$cnum,$name) = &aboutme_info($r,$is_course);
    $depth++;
    my $output;

    my $portfolio_root = &Apache::portfolio::get_portfolio_root($cdom,$cnum,
                                                                $group);
    my $getpropath = 1;
    my ($listref,$listerror) =
        &Apache::lonnet::dirlist($portfolio_root.$path,$cdom,$cnum,$getpropath);
    my %dirlist;
    if (ref($listref) eq 'ARRAY') {
        %dirlist = map { ((split('&',$_,2))[0],1) } @{$listref};
    }
    foreach my $item (sort(keys(%{$currhash}))) {
        $output .= &Apache::loncommon::start_data_table_row();
        $output .= '<td style="padding-left: '.($depth*25).'px">';
        if (ref($currhash->{$item}) eq 'HASH') {
            my $title=&HTML::Entities::encode($item,'<>&"');
            $output .= '<img src="'.&Apache::loncommon::lonhttpdurl("/adm/lonIcons/navmap.folder.open.gif").'" alt="'.&mt('Folder').' '.$title.'" class="LC_icon" />&nbsp;'.$title;
            $output .= '</td><td>&nbsp;</td>'
                      .&Apache::loncommon::end_data_table_row();
            $output .= &parse_directory($r,$depth,$currhash->{$item},
                    $path.'/'.$item,$is_course,$group);
        } else {
            my $file_name;
            if ($currhash->{$item} =~ m|/([^/]+)$|) {
                $file_name = $1;
            } else {
                $file_name = $currhash->{$item};
            }
            my $have_meta = exists($dirlist{$file_name.'.meta'});
            my $url;
            if ($is_course) {
                $url = '/uploaded/'.$cdom.'/'.$cnum.'/groups/'.$group.
                    '/portfolio/'.$currhash->{$item};
            } else {
                $url = '/uploaded/'.$cdom.'/'.$cnum.'/portfolio/'.$currhash->{$item};
            }
            my $showname;
            if ($have_meta) {
                $showname = &Apache::lonnet::metadata($url,'title');
            }
            if ($showname eq '') {
                $showname = $file_name;
            } else {
                $showname = $file_name.' ('.$showname.')';
            }

            $showname=&HTML::Entities::encode($showname,'<>&"');
            $output .= '<a href="'.$url.'">'.
                '<img alt="" src="'.&Apache::loncommon::icon($currhash->{$item}).'" class="LC_icon" />'.
                '&nbsp;'.$showname.'</a>';
            $output.='</td><td>';
            if ($have_meta) {
                $output.= '<a href="'.$url.'.meta"><img alt="'.&mt('Metadata').'" src="'.
                    &Apache::loncommon::lonhttpdurl('/res/adm/pages/catalog.png').
                    '" class="LC_icon" /></a>';
            }
            $output .= '</td>'
                .&Apache::loncommon::end_data_table_row();
        }
    } 
    return $output;
}

sub aboutme_access {
    my ($uname,$udom) = @_;
    my $privcheck = $env{'request.course.id'};
    my $sec;
    if ($env{'request.course.sec'} ne '') {
        $sec = $env{'request.course.sec'};
        $privcheck .= '/'.$sec;
    }
    my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
    if (($cdom eq '') || ($cnum eq '')) {
        my %coursehash = &Apache::lonnet::coursedescription($env{'request.course.id'});
        $cdom = $coursehash{'domain'};
        $cnum = $coursehash{'cnum'};
    }
    if ((&Apache::lonnet::allowed('srm',$privcheck)) ||
        (&Apache::lonnet::allowed('dff',$privcheck))) {
        if (&Apache::lonnet::in_course($uname,$udom,$cnum,$cdom,undef,1)) {
            return 1;
        }
    }
    return;
}

1;
__END__
