# The LearningOnline Network with CAPA
# Routines to control the menu
#
# $Id: lonmenu.pm,v 1.369.2.83 2021/12/13 23:26:32 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
#

=head1 NAME

Apache::lonmenu

=head1 SYNOPSIS

Loads contents of /home/httpd/lonTabs/mydesk.tab, 
used to generate inline menu, and Main Menu page. 

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 GLOBAL VARIABLES

=over

=item @desklines

Each element of this array contains a line of mydesk.tab that doesn't start with
cat, prim or scnd. 
It gets filled in the BEGIN block of this module.

=item %category_names

The keys of this hash are the abbreviations used in mydesk.tab in those lines that 
start with cat, the values are strings representing titles. 
It gets filled in the BEGIN block of this module.

=item %category_members

TODO 

=item %category_positions

The keys of this hash are the abbreviations used in mydesk.tab in those lines that
start with cat, its values are position vectors (column, row). 
It gets filled in the BEGIN block of this module.

=item $readdesk

Indicates that mydesk.tab has been read. 
It is set to 'done' in the BEGIN block of this module.

=item @primary_menu

The elements of this array reference arrays that are made up of the components
of those lines of mydesk.tab that start with prim:.
It is used by primary_menu() to generate the corresponding menu.
It gets filled in the BEGIN block of this module.

=item %primary_sub_menu

The keys of this hash reference are the names of items in the primary_menu array 
which have sub-menus.  For each key, the corresponding value is a reference to
an array containing components extracted from lines in mydesk.tab which begin
with primsub:.
This hash, which is used by primary_menu to generate sub-menus, is populated in
the BEGIN block.

=item @secondary_menu

The elements of this array reference arrays that are made up of the components
of those lines of mydesk.tab that start with scnd.
It is used by secondary_menu() to generate the corresponding menu.
It gets filled in the BEGIN block of this module.

=back

=head1 SUBROUTINES

=over

=item prep_menuitems(\@menuitem)

This routine wraps a menuitem in proper HTML. It is used by primary_menu() and 
secondary_menu().

=item primary_menu()

This routine evaluates @primary_menu and returns a two item array, 
with the array elements containing XHTML for the left and right sides of 
the menu that contains the following links: About, Message, Roles, Help, Logout 
@primary_menu is filled within the BEGIN block of this module with 
entries from mydesk.tab

=item secondary_menu()

Same as primary_menu() but operates on @secondary_menu.

=item create_submenu()

Creates XHTML for unordered list of sub-menu items which belong to a
particular top-level menu item. Uses hover pseudo class in css to display
dropdown list when mouse hovers over top-level item. Support for IE6
(no hover psuedo class) via LC_hoverable class for <li> tag for top-
level item, which employs jQuery to handle behavior on mouseover.

Inputs: 6 - (a) link and (b) target for anchor href in top level item,
            (c) title for text wrapped by anchor tag in top level item.
            (d) reference to array of arrays of sub-menu items.
            (e) boolean to indicate whether to call &mt() to translate
                name of menu item,
            (f) optional class for <li> element in primary menu, for which
                sub menu is being generated.

The underlying datastructure used in (d) contains data from mydesk.tab.
It consists of an array which has an array for each item appearing in
the menu (e.g. [["link", "title", "condition"]] for a single-item menu).
create_submenu() supports also the creation of XHTML for nested dropdown
menus represented by unordered lists. This is done by replacing the
scalar used for the link with an arrayreference containing the menuitems
for the nested menu. This can be done recursively so that the next menu
may also contain nested submenus.

 Example:
 [                                                                                      # begin of datastructure
        ["/home/", "Home", "condition1"],               # 1st item of the 1st layer menu
        [                                                                               # 2nd item of the 1st layer menu
                [                                                                       # anon. array for nested menu
                        ["/path1", "Path1", undef],     # 1st item of the 2nd layer menu
                        ["/path2", "Path2", undef],     # 2nd item of the 2nd layer menu
                        [                                                               # 3rd item of the 2nd layer menu
                                [[...], [...], ..., [...]],     # containing another menu layer
                                "Sub-Sub-Menu",                         # title for this container
                                undef
                        ]
                ],                                                                      # end of array/nested menu
                "Sub-Menu",                                                     # title for the container item
                undef
        ]                                                                               # end of 2nd item of the 1st layer menu
]


=item innerregister()

This gets called in order to register a URL in the body of the document

=item loadevents()

=item unloadevents()

=item startupremote()

=item setflags()

=item maincall()

=item load_remote_msg()

=item get_menu_name()

=item reopenmenu()

=item open()

Open the menu

=item clear()

=item switch()

Switch a button or create a link
Switch acts on the javascript that is executed when a button is clicked.  
The javascript is usually similar to "go('/adm/roles')" or "cstrgo(..)".

=item secondlevel()

=item openmenu()

=item inlinemenu()

=item rawconfig()

=item utilityfunctions()

Output from this routine is a number of javascript functions called by
items in the inline menu, and in some cases items in the Main Menu page. 

=item serverform()

=item constspaceform()

=item get_nav_status()

=item hidden_button_check()

=item roles_selector()

=item jump_to_role()

=back

=cut

package Apache::lonmenu;

use strict;
use Apache::lonnet;
use Apache::lonhtmlcommon();
use Apache::loncommon();
use Apache::lonenc();
use Apache::lonlocal;
use Apache::lonmsg();
use LONCAPA qw(:DEFAULT :match);
use HTML::Entities();
use Apache::lonwishlist();

use vars qw(@desklines %category_names %category_members %category_positions 
            $readdesk @primary_menu %primary_submenu @secondary_menu %secondary_submenu);

my @inlineremote;

sub prep_menuitem {
    my ($menuitem,$listclass,$linkattr) = @_;
    return '' unless(ref($menuitem) eq 'ARRAY');
    my $link;
    if ($$menuitem[1]) { # graphical Link
        $link = "<img class=\"LC_noBorder\""
              . " src=\"" . &Apache::loncommon::lonhttpdurl($$menuitem[1]) . "\"" 
              . " alt=\"" . &mt($$menuitem[2]) . "\" />";
    } else {             # textual Link
        $link = &mt($$menuitem[3]);
    }
    return ($listclass?'<li class="'.$listclass.'">':'<li>').'<a'
           # highlighting for new messages
           . ( $$menuitem[4] eq 'newmsg' ? ' class="LC_new_message"' : '') 
           . qq| href="$$menuitem[0]" target="_top" $linkattr>$link</a></li>|;
}

# primary_menu() evaluates @primary_menu and returns a two item array,
# with the array elements containing XHTML for the left and right sides of 
# the menu that contains the following links:
# Personal, About, Message, Roles, Help, Logout
# @primary_menu is filled within the BEGIN block of this module with 
# entries from mydesk.tab
sub primary_menu {
    my ($links_disabled) = @_;
    my %menu;
    # each element of @primary contains following array:
    # (link url, icon path, alt text, link text, condition, position)
    my $public;
    if ((($env{'user.name'} eq 'public') && ($env{'user.domain'} eq 'public'))
        || (($env{'user.name'} eq '') && ($env{'user.domain'} eq ''))) {
        $public = 1;
    }
    foreach my $menuitem (@primary_menu) {
        # evaluate conditions 
        next if    ref($menuitem)       ne 'ARRAY';    #
        next if    $$menuitem[4]        eq 'nonewmsg'  # show links depending on
                && &Apache::lonmsg::mynewmail();       # whether a new msg 
        next if    $$menuitem[4]        eq 'newmsg'    # arrived or not
                && !&Apache::lonmsg::mynewmail();      # 
        next if    $$menuitem[4]        !~ /public/    ##we've a public user,
                && $public;                            ##who should not see all
                                                       ##links
        next if    $$menuitem[4]        eq 'onlypublic'# hide links which are 
                && !$public;                           # only visible to public
                                                       # users
        next if    $$menuitem[4]        eq 'roles'     ##show links depending on
                && &Apache::loncommon::show_course();  ##term 'Courses' or 
        next if    $$menuitem[4]        eq 'courses'   ##'Roles' wanted
                && !&Apache::loncommon::show_course(); ##
        
        my $title = $menuitem->[3];
        my $position = $menuitem->[5];
        if ($position eq '') {
            $position = 'right';
        }
        my ($listclass,$linkattr);
        if ($links_disabled) {
            $listclass = 'LCisDisabled';
            $linkattr = 'aria-disabled="true"';
        }
        if (defined($primary_submenu{$title})) {
            my ($link,$target);
            if ($menuitem->[0] ne '') {
                $link = $menuitem->[0];
                $target = '_top';
            } else {
                $link = '#';
            }
            my @primsub;
            if (ref($primary_submenu{$title}) eq 'ARRAY') {
                foreach my $item (@{$primary_submenu{$title}}) {
                    next if (($item->[2] eq 'wishlist') && (!$env{'user.adv'})); 
                    next if ((($item->[2] eq 'portfolio') || 
                             ($item->[2] eq 'blog')) && 
                             (!&Apache::lonnet::usertools_access('','',$item->[2],
                                                           undef,'tools')));
                    push(@primsub,$item);
                }
                if (@primsub > 0) {
                    if ($title eq 'Personal' && $env{'user.name'} && $env{'user.domain'} ) {
                        $title = &Apache::loncommon::plainname($env{'user.name'},$env{'user.domain'});
                    } else {
                        $title = &mt($title);
                    }
                    $menu{$position} .= &create_submenu($link,$target,$title,\@primsub,1,undef,$listclass,$linkattr);
                } elsif ($link) {
                    $menu{$position} .= ($listclass?'<li class="'.$listclass.'">':'<li>').
                                        '<a href="'.$link.'" target="'.$target.'" '.$linkattr.'>'.&mt($title).'</a></li>';
                }
            }
        } elsif ($$menuitem[3] eq 'Help') { # special treatment for helplink
            if ($public) {
                my $origmail = $Apache::lonnet::perlvar{'lonSupportEMail'};
                my $defdom = &Apache::lonnet::default_login_domain();
                my $to = &Apache::loncommon::build_recipient_list(undef,
                                                                  'helpdeskmail',
                                                                  $defdom,$origmail);
                if ($to ne '') {
                    $menu{$position} .= &prep_menuitem($menuitem,$listclass,$linkattr); 
                }
            } else {
                $menu{$position} .= ($listclass?'<li class="'.$listclass.'">':'<li>').
                                    &Apache::loncommon::top_nav_help('Help',$linkattr).
                                    '</li>';
            }
        } elsif ($$menuitem[3] eq 'Log In') {
            if ($public) {
                if (&Apache::lonnet::get_saml_landing()) {
                    $$menuitem[0] = '/adm/login';
                }
            }
            $menu{$position} .= prep_menuitem($menuitem,$listclass,$linkattr);
        } else {
            $menu{$position} .= prep_menuitem($menuitem,$listclass,$linkattr);
        }
    }
    my @output = ('','');
    if ($menu{'left'} ne '') {
        $output[0] = "<ol class=\"LC_primary_menu LC_floatleft\">$menu{'left'}</ol>";
    }
    if ($menu{'right'} ne '') {
        $output[1] = "<ol class=\"LC_primary_menu LC_floatright LC_right\">$menu{'right'}</ol>";
    }
    return @output;
}

#returns hashref {user=>'',dom=>''} containing:
#   own name, domain if user is au
#   name, domain of parent author if user is ca or aa
#empty return if user is not an author or not on homeserver
#
#TODO this should probably be moved somewhere more central
#since it can be used by different parts of the system
sub getauthor{
    return unless $env{'request.role'}=~/^(ca|aa|au)/; #nothing to do if user isn't some kind of author

                        #co- or assistent author?
    my ($dom, $user) = ($env{'request.role'} =~ /^(?:ca|aa)\.\/($match_domain)\/($match_username)$/)
                       ? ($1, $2) #domain, username of the parent author
                       : @env{ ('request.role.domain', 'user.name') }; #own domain, username

    # current server == home server?
    my $home =  &Apache::lonnet::homeserver($user,$dom);
    foreach (&Apache::lonnet::current_machine_ids()){
        return {user => $user, dom => $dom} if $_ eq $home;
    }

    # if wrong server
    return;
}

sub secondary_menu {
    my ($httphost,$links_disabled) = @_;
    my $menu;

    my $crstype = &Apache::loncommon::course_type();
    my $crs_sec = $env{'request.course.id'} . ($env{'request.course.sec'} 
                                               ? "/$env{'request.course.sec'}"
                                               : '');
    my $canedit       = &Apache::lonnet::allowed('mdc', $env{'request.course.id'});
    my $canvieweditor = &Apache::lonnet::allowed('cev', $env{'request.course.id'}); 
    my $canviewroster = $env{'course.'.$env{'request.course.id'}.'.student_classlist_view'};
    if ($canviewroster eq 'disabled') {
        undef($canviewroster);
    }
    my $canviewgrps   = &Apache::lonnet::allowed('vcg', $crs_sec);
    my $canmodifyuser = &Apache::lonnet::allowed('cst', $crs_sec);
    my $canviewusers  = &Apache::lonnet::allowed('vcl', $crs_sec);
    my $canviewwnew   = &Apache::lonnet::allowed('whn', $crs_sec);
    my $canviewpara   = &Apache::lonnet::allowed('vpa', $crs_sec);
    my $canmodpara    = &Apache::lonnet::allowed('opa', $crs_sec);
    my $canvgr        = &Apache::lonnet::allowed('vgr', $crs_sec);
    my $canmgr        = &Apache::lonnet::allowed('mgr', $crs_sec);
    my $author        = &getauthor();

    my ($cdom,$cnum,$showsyllabus,$showfeeds,$showresv,$grouptools);
    $grouptools = 0; 
    if ($env{'request.course.id'}) {
        $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
        $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
        unless ($canedit || $canvieweditor) {
            unless (&Apache::lonnet::is_on_map("public/$cdom/$cnum/syllabus")) {
                if (($env{'course.'.$env{'request.course.id'}.'.externalsyllabus'}) ||
                    ($env{'course.'.$env{'request.course.id'}.'.uploadedsyllabus'}) ||
                    ($env{'course.'.$env{'request.course.id'}.'.updatedsyllabus'}) ||
                    ($env{'request.course.syllabustime'})) {
                    $showsyllabus = 1;
                }
            }
            if ($env{'request.course.feeds'}) {
                $showfeeds = 1;
            }
        }
        unless ($canmgr || $canvgr) {
            my %slots = &Apache::lonnet::get_course_slots($cnum,$cdom);
            if (keys(%slots) > 0) {
                $showresv = 1;
            }
        }
        if ($env{'request.course.groups'} ne '') {
            foreach my $group (split(/:/,$env{'request.course.groups'})) {
                next unless ($group =~ /^\w+$/);
                my @privs = split(/:/,$env{"user.priv.$env{'request.role'}./$cdom/$cnum/$group"});
                shift(@privs);
                if (@privs) {
                    $grouptools ++;
                }
            }
        }
    }

    my ($listclass,$linkattr);
    if ($links_disabled) {
        $listclass = 'LCisDisabled';
        $linkattr = 'aria-disabled="true"';
    }

    my ($canmodifycoauthor);
    if ($env{'request.role'} eq "au./$env{'user.domain'}/") {
        my $extent = "$env{'user.domain'}/$env{'user.name'}";
        if ((&Apache::lonnet::allowed('cca',$extent)) ||
            (&Apache::lonnet::allowed('caa',$extent))) {
            $canmodifycoauthor = 1;
        }
    }

    my ($roleswitcher_js,$roleswitcher_form);

    foreach my $menuitem (@secondary_menu) {
        # evaluate conditions 
        next if    ref($menuitem)  ne 'ARRAY';
        next if    $$menuitem[4]   ne 'always'
                && ($$menuitem[4]  ne 'author' && $$menuitem[4] ne 'cca')
                && !$env{'request.course.id'};
        next if    $$menuitem[4]   =~ /^crsedit/
                && (!$canedit && !$canvieweditor);
        next if    $$menuitem[4]  eq 'crseditCourse'
                && ($crstype eq 'Community');
        next if    $$menuitem[4]  eq 'crseditCommunity'
                && ($crstype eq 'Course');
        next if    $$menuitem[4]  eq 'nvgr'
                && $canvgr;
        next if    $$menuitem[4]  eq 'vgr'
                && !$canvgr;
        next if    $$menuitem[4]   eq 'viewusers'
                && !$canmodifyuser && !$canviewusers;
        next if    $$menuitem[4]   eq 'noviewusers'
                && ($canmodifyuser || $canviewusers || !$canviewroster);
        next if    $$menuitem[4]   eq 'mgr'
                && !$canmgr;
        next if    $$menuitem[4]   eq 'showresv'
                && !$showresv;
        next if    $$menuitem[4]   eq 'whn'
                && !$canviewwnew;
        next if    $$menuitem[4]   eq 'params'
                && (!$canmodpara && !$canviewpara);
        next if    $$menuitem[4]   eq 'nvcg'
                && ($canviewgrps || !$grouptools);
        next if    $$menuitem[4]   eq 'showsyllabus'
                && !$showsyllabus;
        next if    $$menuitem[4]   eq 'showfeeds'
                && !$showfeeds;
        next if    $$menuitem[4]    eq 'author'
                && !$author;
        next if    $$menuitem[4]    eq 'cca'
                && !$canmodifycoauthor;

        my $title = $menuitem->[3];
        if (defined($secondary_submenu{$title})) {
            my ($link,$target);
            if ($menuitem->[0] ne '') {
                $link = $menuitem->[0];
                $target = '_top';
            } else {
                $link = '#';
            }
            my @scndsub;
            if (ref($secondary_submenu{$title}) eq 'ARRAY') {
                foreach my $item (@{$secondary_submenu{$title}}) {
                    if (ref($item) eq 'ARRAY') {
                        next if ($item->[2] eq 'vgr' && !$canvgr);
                        next if ($item->[2] eq 'opa' && !$canmodpara);
                        next if ($item->[2] eq 'vpa' && !$canviewpara);
                        next if ($item->[2] eq 'viewusers' && !($canmodifyuser || $canviewusers));
                        next if ($item->[2] eq 'mgr' && !$canmgr);
                        next if ($item->[2] eq 'vcg' && !$canviewgrps);
                        next if ($item->[2] eq 'crsedit' && !$canedit && !$canvieweditor);
                        next if ($item->[2] eq 'params' && !$canmodpara && !$canviewpara);
                        next if ($item->[2] eq 'author' && !$author);
                        next if ($item->[2] eq 'cca' && !$canmodifycoauthor);
                        push(@scndsub,$item); 
                    }
                }
                if (@scndsub > 0) {
                    $menu .= &create_submenu($link,$target,&mt($title),\@scndsub,1,undef,
                                             $listclass,$linkattr);
                } elsif ($link ne '#') {
                    $menu .= ($listclass?'<li class="'.$listclass.'">':'<li>').
                             '<a href="'.$link.'" target="'.$target.'" '.$linkattr.'>'.
                             &mt($title).'</a></li>';
                }
            }
        } elsif ($$menuitem[3] eq 'Roles' && $env{'request.course.id'}) {
            # special treatment for role selector
            ($roleswitcher_js,$roleswitcher_form,my $switcher) =
                &roles_selector(
                        $env{'course.' . $env{'request.course.id'} . '.domain'},
                        $env{'course.' . $env{'request.course.id'} . '.num'},
                        $httphost
                );
            $menu .= $switcher;
        } else {
            if ($$menuitem[3] eq 'Syllabus' && $env{'request.course.id'}) {
                my $url = $$menuitem[0];
                $url =~ s{\[cdom\]/\[cnum\]}{$cdom/$cnum};
                if (&Apache::lonnet::is_on_map($url)) {
                    unless ($$menuitem[0] =~ /(\?|\&)register=1/) {
                        $$menuitem[0] .= (($$menuitem[0]=~/\?/)? '&' : '?').'register=1';
                    }
                } else {
                    $$menuitem[0] =~ s{\&?register=1}{};
                }
                if ($env{'course.'.$env{'request.course.id'}.'.externalsyllabus'} =~ m{^http://}) {
                    if (($ENV{'SERVER_PORT'} == 443) || ($env{'request.use_absolute'} =~ m{^https://})) {
                        unless ((&Apache::lonnet::uses_sts()) || (&Apache::lonnet::waf_allssl())) {
                            unless ($$menuitem[0] =~ m{^https?://}) {
                                $$menuitem[0] = 'http://'.$ENV{'SERVER_NAME'}.$$menuitem[0];
                            }
                            unless ($$menuitem[0] =~ /(\&|\?)usehttp=1/) {
                                $$menuitem[0] .= (($$menuitem[0]=~/\?/) ? '&' : '?').'usehttp=1';
                            }
                        }
                    }
                }
                $$menuitem[0] = &HTML::Entities::encode($$menuitem[0],'&<>"');
            }
            $menu .= &prep_menuitem(\@$menuitem,$listclass,$linkattr);
        }
    }
    if ($menu =~ /\[url\].*\[symb\]/) {
        my $escurl  = &escape( &Apache::lonenc::check_encrypt(
                             $env{'request.noversionuri'}));

        my $escsymb = &escape( &Apache::lonenc::check_encrypt(
                             $env{'request.symb'})); 

        if (    $env{'request.state'} eq 'construct'
            and (   $env{'request.noversionuri'} eq '' 
                 || !defined($env{'request.noversionuri'}))) 
        {
            my $londocroot = $Apache::lonnet::perlvar{'lonDocRoot'};
            ($escurl = $env{'request.filename'}) =~ s{^\Q$londocroot\E}{};
            $escurl  = &escape($escurl);
        }
        $menu =~ s/\[url\]/$escurl/g;
        $menu =~ s/\[symb\]/$escsymb/g;
    }
    $menu =~ s/\[uname\]/$$author{user}/g;
    $menu =~ s/\[udom\]/$$author{dom}/g;
    if ($env{'request.course.id'}) {  
        $menu =~ s/\[cnum\]/$cnum/g;
        $menu =~ s/\[cdom\]/$cdom/g;
    }
    if ($menu) {
        $menu = "<ul id=\"LC_secondary_menu\">$menu</ul>";
    }
    if ($roleswitcher_form) {
        $menu .= "\n$roleswitcher_js\n$roleswitcher_form";
    }
    return $menu;
}

sub create_submenu {
    my ($link,$target,$title,$submenu,$translate,$addclass,$listclass,$linkattr) = @_;
    return unless (ref($submenu) eq 'ARRAY');
    my $disptarget;
    if ($target ne '') {
        $disptarget = ' target="'.$target.'"';
    }
    my $menu = '<li class="LC_hoverable '.$addclass.'">'.
               '<a href="'.$link.'"'.$disptarget.'>'.
               '<span class="LC_nobreak">'.$title.
               '<span class="LC_fontsize_small" style="font-weight:normal;">'.
               ' &#9660;</span></span></a>'.
               '<ul>';

    # $link and $title are only used in the initial string written in $menu
    # as seen above, not needed for nested submenus
    $menu .= &build_submenu($target, $submenu, $translate, '1', $listclass, $linkattr);
    $menu .= '</ul></li>';

    return $menu;
}

# helper routine for create_submenu
# build the dropdown (and nested submenus) recursively
# see perldoc create_submenu documentation for further information
sub build_submenu {
    my ($target, $submenu, $translate, $first_level, $listclass, $linkattr) = @_;
    unless (@{$submenu}) {
        return '';
    }

    my $menu = '';
    my $count = 0;
    my $numsub = scalar(@{$submenu});
    foreach my $item (@{$submenu}) {
        $count ++;
        if (ref($item) eq 'ARRAY') {
            my $href = $item->[0];
            my $bordertop;
            my $borderbot;
            my $title;

            if ($translate) {
                 $title = &mt($item->[1]);
            } else {
                $title = $item->[1];
            }

            if ($count == 1 && !$first_level) {
                $bordertop = 'border-top: 1px solid black;';
            }
            if ($count == $numsub) {
                $borderbot = 'border-bottom: 1px solid black;';
            }

            # href is a reference to another submenu
            if (ref($href) eq 'ARRAY') {
                $menu .= '<li style="margin:0;padding:0;'.$bordertop . $borderbot . '">';
                $menu .= '<p><span class="LC_primary_menu_innertitle">'
                                        . $title . '</span><span class="LC_primary_menu_innerarrow">&#9654;</span></p>';
                $menu .= '<ul>';
                $menu .= &build_submenu($target, $href, $translate);
                $menu .= '</ul>';
                $menu .= '</li>';
            } else {    # href is the actual hyperlink and does not represent another submenu
                        # for the current menu title
                if ($href =~ /(aboutme|rss\.html)$/) {
                    next unless (($env{'user.name'} ne '') && ($env{'user.domain'} ne ''));
                    $href =~ s/\[domain\]/$env{'user.domain'}/g;
                    $href =~ s/\[user\]/$env{'user.name'}/g;
                } elsif (($href =~ m{^/adm/preferences\?}) && ($href =~ /\[returnurl\]/)) {
                    my $returnurl = $ENV{'REQUEST_URI'};
                    if ($ENV{'REQUEST_URI'} =~ m{/adm/preferences\?action=(?:changedomcoord|authorsettings)\&returnurl=([^\&]+)$}) {
                        $returnurl = $1;
                    }
                    if (($returnurl =~ m{^/adm/createuser($|\?action=)}) ||
                        ($returnurl =~ m{^/priv/$match_domain/$match_username}) ||
                        ($returnurl =~ m{^/res(/?$|/$match_domain/$match_username)})) {
                        $returnurl =~ s{\?.*$}{};
                        $returnurl = '&amp;returnurl='.&HTML::Entities::encode($returnurl,'"<>&\'');
                    } else {
                        undef($returnurl);
                    }
                    $href =~ s/\[returnurl\]/$returnurl/;
                }
                unless (($href eq '') || ($href =~ /^\#/)) {
                    $target = ' target="_top"';
                }

                $menu .= '<li ';
                $menu .= ($listclass?'class="'.$listclass.'" ':'');
                $menu .= 'style="margin:0;padding:0;'. $bordertop . $borderbot .'">';
                $menu .= '<a href="'.$href.'"'.$target.' '.$linkattr.'>' .  $title . '</a>';
                $menu .= '</li>';
            }
        }
    }
    return $menu;
}

sub registerurl {
    my ($forcereg) = @_;
    my $result = '';
    if ($env{'request.noversionuri'} =~ m{^/res/adm/pages/}) { return ''; }
    my $force_title='';
    if ($env{'request.state'} eq 'construct') {
        $force_title=&Apache::lonxml::display_title();
    }
    if (($env{'environment.remote'} ne 'on') ||
        ((($env{'request.publicaccess'}) ||
         (!&Apache::lonnet::is_on_map(
           &unescape($env{'request.noversionuri'})))) &&
        (!$forcereg))) {
        return
        $result
       .'<script type="text/javascript">'."\n"
       .'// <![CDATA['."\n"
       .'function LONCAPAreg(){;} function LONCAPAstale(){}'."\n"
       .'// ]]>'."\n"
       .'</script>'
       .$force_title;
    }
# Graphical display after login only
    if ($env{'request.registered'} && !$forcereg) { return ''; }
    $result.=&innerregister($forcereg);
    return $result.$force_title;
}

sub innerregister {
    my ($forcereg,$bread_crumbs,$group,$pagebuttonshide,$hostname) = @_;
    my $const_space = ($env{'request.state'} eq 'construct');
    my $is_const_dir = 0;

    if ($env{'request.noversionuri'} =~ m{^/res/adm/pages/}) { return ''; }

    $env{'request.registered'} = 1;

    my $noremote = ($env{'environment.remote'} ne 'on');

    undef(@inlineremote);

    my $reopen=&Apache::lonmenu::reopenmenu();

    my $newmail='';

    if (&Apache::lonmsg::newmail() && !$noremote) {
        # We have new mail and remote is up
        $newmail= 'swmenu.setstatus("you have","messages");';
    }

    my ($mapurl,$resurl,$crstype,$navmap);

    if ($env{'request.course.id'}) {
#
#course_type:  Course or Community
#
        $crstype = &Apache::loncommon::course_type();
        if ($env{'request.symb'}) {
            my $ignorenull;
            unless ($env{'request.noversionuri'} eq '/adm/navmaps') {
                $ignorenull = 1;
            }
            my $symb = &Apache::lonnet::symbread('','',$ignorenull);
            ($mapurl, my $rid, $resurl) = &Apache::lonnet::decode_symb($symb);
            my $coursetitle = $env{'course.'.$env{'request.course.id'}.'.description'};

            my $maptitle = &Apache::lonnet::gettitle($mapurl);
            my $restitle = &Apache::lonnet::gettitle($symb);

            my (@crumbs,@mapcrumbs);
            if (($env{'request.noversionuri'} ne '/adm/navmaps') && ($mapurl ne '') &&
                ($mapurl ne $env{'course.'.$env{'request.course.id'}.'.url'})) {
                $navmap = Apache::lonnavmaps::navmap->new();
                if (ref($navmap)) {
                    @mapcrumbs = $navmap->recursed_crumbs($mapurl,$restitle);
                }
            }
            unless (($forcereg) && 
                    ($env{'request.noversionuri'} eq '/adm/navmaps') &&
                    ($mapurl eq $env{'course.'.$env{'request.course.id'}.'.url'})) {
                @crumbs = ({text  => $crstype.' Contents', 
                            href  => "Javascript:gopost('/adm/navmaps','')"});
            }
            if ($mapurl ne $env{'course.'.$env{'request.course.id'}.'.url'}) { 
                if (@mapcrumbs) {
                    push(@crumbs,@mapcrumbs);
                } else {
                    push(@crumbs, {text  => '...',
                                   no_mt => 1});
                }
            }

            unless ((@mapcrumbs) || (!$maptitle) || ($maptitle eq 'default.sequence') ||
                    ($mapurl eq $env{'course.'.$env{'request.course.id'}.'.url'})) {
                push @crumbs, {text => $maptitle, no_mt => 1,
                               href => &Apache::lonnet::clutter($mapurl).'?navmap=1'};
            }
            if ($restitle && !@mapcrumbs) {
                push(@crumbs,{text => $restitle, no_mt => 1});
            }
            my @tools;
            if ($env{'request.filename'} =~ /\.page$/) {
                my %breadcrumb_tools = &Apache::lonhtmlcommon::current_breadcrumb_tools();
                if (ref($breadcrumb_tools{'tools'}) eq 'ARRAY') {
                    @tools = @{$breadcrumb_tools{'tools'}};
                }
            }
            &Apache::lonhtmlcommon::clear_breadcrumbs();
            &Apache::lonhtmlcommon::add_breadcrumb(@crumbs);
            if (@tools) {
                &Apache::lonhtmlcommon::add_breadcrumb_tool('tools',@tools);
            }
        } else {
            $resurl = $env{'request.noversionuri'};
            my $courseurl = &Apache::lonnet::courseid_to_courseurl($env{'request.course.id'});
            my $title = &mt('View Resource');
            if ($resurl =~ m{^\Q/uploaded$courseurl/supplemental/\E(default|\d+)/}) {
                &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},['folderpath','title']);
                &Apache::lonhtmlcommon::clear_breadcrumbs();
                if ($env{'form.title'}) {
                    $title = $env{'form.title'};
                }
                my $trail;
                if ($env{'form.folderpath'}) {
                    &prepare_functions($resurl,$forcereg,$group,undef,undef,1,$hostname);
                    ($trail) =
                        &Apache::lonhtmlcommon::docs_breadcrumbs(undef,$crstype,undef,$title,1);
                } else {
                    &Apache::lonhtmlcommon::add_breadcrumb(
                    {text  => "Supplemental $crstype Content",
                     href  => "javascript:gopost('/adm/supplemental','')"});
                    $title = &mt('View Resource');
                    ($trail) =
                        &Apache::lonhtmlcommon::docs_breadcrumbs(undef,$crstype,undef,$title,1);
                }
                return $trail;
            } elsif ($resurl =~ m{^\Q/uploaded$courseurl/portfolio/syllabus/}) {
                &Apache::lonhtmlcommon::clear_breadcrumbs();
                &prepare_functions('/public'.$courseurl."/syllabus",
                                   $forcereg,$group,undef,undef,1,$hostname);
                $title = &mt('Syllabus File');
                my ($trail) =
                    &Apache::lonhtmlcommon::docs_breadcrumbs(undef,$crstype,undef,$title,1,$hostname);
                return $trail;
            }
            unless ($env{'request.state'} eq 'construct') {
                &Apache::lonhtmlcommon::clear_breadcrumbs();
                &Apache::lonhtmlcommon::add_breadcrumb({text => 'View Resource'});
            }
        }
    } elsif (! $const_space){
        #a situation when we're looking at a resource outside of context of a 
        #course or construction space (e.g. with cumulative rights)
        &Apache::lonhtmlcommon::clear_breadcrumbs();
        unless ($env{'request.noversionuri'} =~ m{^/adm/$match_domain/$match_username/aboutme$}) {
            &Apache::lonhtmlcommon::add_breadcrumb({text => 'View Resource'});
        }
    }
    my $timesync   = ( $noremote ? '' : 'swmenu.syncclock(1000*'.time.');' );
# =============================================================================
# ============================ This is for URLs that actually can be registered
    if ( ($env{'request.noversionuri'}!~m{^/(res/)*adm/})
                       || ($forcereg)) {

        my %swtext;
        if ($noremote) {
            %swtext = &get_inline_text();
        } else {
            %swtext = &get_rc_text();
        }
        my $hwkadd='';

        my ($cdom,$cnum,%perms,$cfile,$switchserver,$home,$forceedit,
            $forceview,$editbutton);
        if (($resurl =~ m{^/adm/($match_domain)/($match_username)/aboutme$}) ||
            ($env{'request.role'} !~/^(aa|ca|au)/)) {
            if (($env{'environment.remote'} eq 'on') && ($env{'request.symb'})) {
                &Apache::lonhtmlcommon::clear_breadcrumbs();
            }
            $editbutton = &prepare_functions($resurl,$forcereg,$group,'','','',$hostname);
        }
        if ($editbutton eq '') {
            $editbutton = &clear(6,1);
        }

#
# This applies in course context
#
        if ($env{'request.course.id'}) {
            $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
            $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
            $perms{'mdc'} = &Apache::lonnet::allowed('mdc',$env{'request.course.id'});
            $perms{'cev'} = &Apache::lonnet::allowed('cev',$env{'request.course.id'});
            my @privs;
            if ($env{'request.symb'} ne '') {
                if ($env{'request.filename'}=~/$LONCAPA::assess_re/) {
                    push(@privs,('mgr','vgr'));
                }
                push(@privs,('opa','vpa'));
            }
            foreach my $priv (@privs) {
                $perms{$priv} = &Apache::lonnet::allowed($priv,$env{'request.course.id'});
                if (!$perms{$priv} && $env{'request.course.sec'} ne '') {
                    $perms{$priv} =
                        &Apache::lonnet::allowed($priv,"$env{'request.course.id'}/$env{'request.course.sec'}");
                }
            }
#
# Determine whether or not to show Grades and Submissions buttons
#
            if ($env{'request.symb'} ne '' &&
                $env{'request.filename'}=~/$LONCAPA::assess_re/) {
                if ($perms{'mgr'}) {
                    $hwkadd.= &switch('','',7,2,'pgrd.png','Content Grades',
                                      'grades[_4]',
                                      "gocmd('/adm/grades','gradingmenu')",
                                      'Content Grades');
                } elsif ($perms{'vgr'}) {
                    $hwkadd .= &switch('','',7,2,'subm.png','Content Submissions',
                                       'missions[_1]',
                                       "gocmd('/adm/grades','submission')",
                                       'Content Submissions');
                }
            }
            if (($env{'request.symb'} ne '') && (($perms{'opa'}) || ($perms{'vpa'}))) {
                $hwkadd .= &switch('','',7,3,'pparm.png','Content Settings',
                                   'parms[_2]',"gocmd('/adm/parmset','set')",
                                   'Content Settings');
            }
# End grades/submissions check

#
# This applies to items inside a folder/page modifiable in the course.
#
            if (($env{'request.symb'}=~/^uploaded/) && (($perms{'mdc'}) || ($perms{'cev'}))) {
                my $text = 'Edit Folder';
                if (($mapurl =~ /\.page$/) ||
                    ($env{'request.symb'}=~
                         m{uploaded/$cdom/$cnum/default_\d+\.page$}))  {
                    $text = 'Edit Page';
                }
                $hwkadd .= &switch('','',7,4,'docs-22x22.png',$text,'parms[_2]',
                                   "gocmd('/adm/coursedocs','direct')",
                                   'Folder/Page Content');
            }
# End modifiable folder/page container check
        }
# End course context

# Prepare the rest of the buttons
        my ($menuitems,$got_prt,$got_wishlist,$cstritems);
        if ($const_space) {
#
# We are in construction space
#

            my $londocroot = $Apache::lonnet::perlvar{'lonDocRoot'};
	    my ($udom,$uname,$thisdisfn) =
		($env{'request.filename'}=~m{^\Q$londocroot/priv/\E([^/]+)/([^/]+)/(.*)$});
            my $currdir = '/priv/'.$udom.'/'.$uname.'/'.$thisdisfn;
            if ($currdir =~ m-/$-) {
                $is_const_dir = 1;
                if ($thisdisfn eq '') {
                    $is_const_dir = 2;
                }
            } else {
                $currdir =~ s|[^/]+$||;
		my $cleandisfn = &Apache::loncommon::escape_single($thisdisfn);
		my $esc_currdir = &Apache::loncommon::escape_single($currdir);
#
# Probably should be in mydesk.tab
#
                $menuitems=(<<ENDMENUITEMS);
s&6&1&list.png&Directory&dir[_1]&golist('$esc_currdir')&List current directory
s&6&2&rtrv.png&Retrieve&version[_1]&gocstr('/adm/retrieve','/priv/$udom/$uname/$cleandisfn')&Retrieve old version
s&6&3&pub.png&Publish&resource[_3]&gocstr('/adm/publish','/priv/$udom/$uname/$cleandisfn')&Publish this resource
s&7&1&del.png&Delete&resource[_2]&gocstr('/adm/cfile?action=delete','/priv/$udom/$uname/$cleandisfn')&Delete this resource
s&7&2&prt.png&Print&printout[_1]&gocstr('/adm/printout','/priv/$udom/$uname/$cleandisfn')&Prepare a printable document
ENDMENUITEMS
                unless ($noremote) {
                    $cstritems = $menuitems;
                    undef($menuitems);
                }
            }
            if (ref($bread_crumbs) eq 'ARRAY') {
                &Apache::lonhtmlcommon::clear_breadcrumbs();
                foreach my $crumb (@{$bread_crumbs}){
                     &Apache::lonhtmlcommon::add_breadcrumb($crumb);
                }
            }
        } elsif ( defined($env{'request.course.id'}) && 
		 $env{'request.symb'} ne '' ) {
#
# We are in a course and looking at a registered URL
# Should probably be in mydesk.tab
#

	    $menuitems=(<<ENDMENUITEMS);
c&3&1
s&2&1&back.png&$swtext{'back'}&&gopost('/adm/flip','back:'+currentURL)&Previous content resource&&1
s&2&3&forw.png&$swtext{'forw'}&&gopost('/adm/flip','forward:'+currentURL)&Next content resource&&3
c&6&3
c&8&1
c&8&2
s&8&3&prt.png&$swtext{'prt'}&printout[_1]&gopost('/adm/printout',currentURL)&Prepare a printable document
ENDMENUITEMS
            $got_prt = 1;
            if (($env{'user.adv'}) && ($env{'request.uri'} =~ /^\/res/)
                && (!$env{'request.enc'})) {
                # wishlist is only available for users with access to resource-pool
                # and links can only be set for resources within the resource-pool
                $menuitems .= (<<ENDMENUITEMS);
s&9&1&alnk.png&$swtext{'alnk'}&linkstor[_1]&set_wishlistlink('',currentURL)&Save a link for this resource in my personal Stored Links repository&&1
ENDMENUITEMS
                $got_wishlist = 1;
            }

my $currentURL = &Apache::loncommon::get_symb();
my ($symb_old,$symb_old_enc) = &Apache::loncommon::clean_symb($currentURL);
my $annotation = &Apache::loncommon::get_annotation($symb_old,$symb_old_enc);
$menuitems.="s&9&3&";
if(length($annotation) > 0){
	$menuitems.="anot2.png";
}else{
	$menuitems.="anot.png";
}
$menuitems.="&$swtext{'anot'}&tations[_1]&annotate()&";
$menuitems.="Make notes and annotations about this resource&&1\n";
my $is_mobile;
if ($env{'browser.mobile'}) {
    $is_mobile = 1;
}

            unless ($env{'request.noversionuri'}=~/\/(bulletinboard|smppg|navmaps|syllabus|aboutme|viewclasslist|portfolio)(\?|$)/) {
		if ((!$env{'request.enc'}) && ($env{'request.noversionuri'} !~ m{^/adm/wrapper/ext/}) &&
                     ($env{'request.noversionuri'} !~ m{^/uploaded/$match_domain/$match_courseid/(docs/|default_\d+\.page$)})) {
		    $menuitems.=(<<ENDREALRES);
s&6&3&catalog.png&$swtext{'catalog'}&info[_1]&catalog_info('$is_mobile')&Show Metadata
ENDREALRES
                }
                unless (($env{'request.noversionuri'} =~ m{^/uploaded/$match_domain/$match_courseid/(docs/|default_\d+\.page$)}) ||
                        ($env{'request.noversionuri'} =~ m{^\Q/adm/wrapper/\E(ext|uploaded)/})) { 
	            $menuitems.=(<<ENDREALRES);
s&8&1&eval.png&$swtext{'eval'}&this[_1]&gopost('/adm/evaluate',currentURL,1)&Provide my evaluation of this resource
ENDREALRES
                }
                unless ($env{'request.noversionuri'} =~ m{^\Q/adm/wrapper/\E(ext|uploaded)/}) {
                    $menuitems.=(<<ENDREALRES);
s&8&2&fdbk.png&$swtext{'fdbk'}&discuss[_1]&gopost('/adm/feedback',currentURL,1)&Provide feedback messages or contribute to the course discussion about this resource
ENDREALRES
                }
	    }
        }
	if ($env{'request.uri'} =~ /^\/res/) {
            unless ($got_prt) {
	        $menuitems .= (<<ENDMENUITEMS);
s&8&3&prt.png&$swtext{'prt'}&printout[_1]&gopost('/adm/printout',currentURL)&Prepare a printable document
ENDMENUITEMS
                $got_prt = 1;
            }
            unless ($got_wishlist) {
                if (($env{'user.adv'}) && (!$env{'request.enc'})) {
                    # wishlist is only available for users with access to resource-pool
                    $menuitems .= (<<ENDMENUITEMS);
s&9&1&alnk.png&$swtext{'alnk'}&linkstor[_1]&set_wishlistlink('',currentURL)&Save a link for this resource in your personal Stored Links repository&&1
ENDMENUITEMS
                    $got_wishlist = 1;
                }
            }
	}
        unless ($got_prt) {
            $menuitems .= (<<ENDMENUITEMS);
c&8&3
ENDMENUITEMS
        }
        unless ($got_wishlist) {
            $menuitems .= (<<ENDMENUITEMS);
c&9&1
ENDMENUITEMS
        }
        my $buttons='';
        foreach (split(/\n/,$menuitems)) {
	    my ($command,@rest)=split(/\&/,$_);
            my $idx=10*$rest[0]+$rest[1];
            if (&hidden_button_check() eq 'yes') {
                if ($idx == 21 ||$idx == 23) {
                    $buttons.=&switch('','',@rest);
                } else {
                    $buttons.=&clear(@rest);
                }
            } else {  
                if ($command eq 's') {
	            $buttons.=&switch('','',@rest);
                } else {
                    $buttons.=&clear(@rest);
                }
            }
        }
        if ($noremote) {
	    my $addremote=0;
	    foreach (@inlineremote) { if ($_ ne '') { $addremote=1; last;} }
            if ($addremote) {
                my ($countdown,$buttonshide);
                if ($env{'request.filename'} =~ /\.page$/) {
                    my %breadcrumb_tools = &Apache::lonhtmlcommon::current_breadcrumb_tools();
                    if (ref($breadcrumb_tools{'tools'}) eq 'ARRAY') {
                        $countdown = $breadcrumb_tools{'tools'}->[0];
                    }
                    $buttonshide = $pagebuttonshide;
                } else {
                    $countdown = &countdown_timer();
                    $buttonshide = &hidden_button_check();
                }

                &Apache::lonhtmlcommon::clear_breadcrumb_tools();

                &Apache::lonhtmlcommon::add_breadcrumb_tool(
                    'navigation', @inlineremote[21,23]);

                if ($buttonshide eq 'yes') {
                    if ($countdown) {
                        &Apache::lonhtmlcommon::add_breadcrumb_tool('tools',$countdown);
                    }
                } else {
                    my @tools = @inlineremote[93,91,81,82,83];
                    if ($countdown) {
                        unshift(@tools,$countdown);
                    }
                    &Apache::lonhtmlcommon::add_breadcrumb_tool(
                        'tools',@tools);

                    #publish button in construction space
                    if ($env{'request.state'} eq 'construct'){
                        &Apache::lonhtmlcommon::add_breadcrumb_tool(
                            'advtools', $inlineremote[63]);
                    } else {
                        &Apache::lonhtmlcommon::add_breadcrumb_tool(
                            'tools', $inlineremote[63]);
                    }
                    &advtools_crumbs(@inlineremote);
                }
            }
            my ($topic_help,$topic_help_text);
            if ($is_const_dir == 2) {
                if ((($ENV{'SERVER_PORT'} == 443) ||
                     ($Apache::lonnet::protocol{$Apache::lonnet::perlvar{'lonHostID'}} eq 'https')) &&
                     (&Apache::lonnet::usertools_access($env{'user.name'},$env{'user.domain'},'webdav'))) {
                    $topic_help = 'Authoring_WebDAV,Authoring_WebDAV_Mac_10v6,Authoring_WebDAV_Mac_10v10,'.
                                 'Authoring_WebDAV_Windows_v7,Authoring_WebDAV_Linux_Centos';
                    $topic_help_text = 'About WebDAV access';
                }
            }
            return   &Apache::lonhtmlcommon::scripttag('', 'start')
                   . &Apache::lonhtmlcommon::breadcrumbs(undef,undef,0,'','','','',$topic_help,$topic_help_text)
                   . &Apache::lonhtmlcommon::scripttag('', 'end');

        } else {
            my $cstrcrumbs;
            if ($const_space) {
                foreach (split(/\n/,$cstritems)) {
                    my ($command,@rest)=split(/\&/,$_);
                    my $idx=10*$rest[0]+$rest[1];
                    &switch('','',@rest);
                }
                &Apache::lonhtmlcommon::add_breadcrumb_tool('advtools',
                                                            @inlineremote[63,61,71,72]);

                $cstrcrumbs = &Apache::lonhtmlcommon::scripttag('', 'start')
                             .&Apache::lonhtmlcommon::breadcrumbs(undef,undef,0)
                             .&Apache::lonhtmlcommon::scripttag('', 'end');
            }
            my $requri=&Apache::lonnet::clutter(&Apache::lonnet::fixversion((split(/\?/,$env{'request.noversionuri'}))[0]));
            $requri=&Apache::lonenc::check_encrypt(&unescape($requri));
            my $cursymb=&Apache::lonenc::check_encrypt($env{'request.symb'});
            my $navstatus=&get_nav_status();
            my $clearcstr;

            if ($env{'user.adv'}) { $clearcstr='clearbut(6,1)'; }
            return <<ENDREGTHIS;

<script type="text/javascript">
// <![CDATA[
// BEGIN LON-CAPA Internal
var swmenu=null;

    function LONCAPAreg() {
          swmenu=$reopen;
          swmenu.clearTimeout(swmenu.menucltim);
          $timesync
          $newmail
          $buttons
          swmenu.currentURL="$requri";
          swmenu.reloadURL=swmenu.currentURL+window.location.search;
          swmenu.currentSymb="$cursymb";
          swmenu.reloadSymb="$cursymb";
          swmenu.currentStale=0;
          $navstatus
          $hwkadd
          $editbutton
    }

    function LONCAPAstale() {
          swmenu=$reopen
          swmenu.currentStale=1;
          if (swmenu.reloadURL!='' && swmenu.reloadURL!= null) {
             swmenu.switchbutton
             (3,1,'reload.gif','return','location','go(reloadURL)','Return to the last known location in the course sequence');
          }
          swmenu.clearbut(7,2);
          swmenu.clearbut(7,3);
          swmenu.menucltim=swmenu.setTimeout(
 'clearbut(2,1);clearbut(2,3);clearbut(8,1);clearbut(8,2);clearbut(8,3);'+
 'clearbut(9,1);clearbut(9,3);clearbut(6,3);$clearcstr',
                          2000);
      }

// END LON-CAPA Internal
// ]]>
</script>

$cstrcrumbs
ENDREGTHIS
        }
    } else {
       unless ($noremote) {
# Not registered, graphical
           return (<<ENDDONOTREGTHIS);

<script type="text/javascript">
// <![CDATA[
// BEGIN LON-CAPA Internal
var swmenu=null;

    function LONCAPAreg() {
          swmenu=$reopen
          $timesync
          swmenu.currentStale=1;
          swmenu.clearbut(2,1);
          swmenu.clearbut(2,3);
          swmenu.clearbut(8,1);
          swmenu.clearbut(8,2);
          swmenu.clearbut(8,3);
          swmenu.clearbut(9,1);
          if (swmenu.currentURL) {
             swmenu.switchbutton
              (3,1,'reload.gif','return','location','go(currentURL)');
          } else {
              swmenu.clearbut(3,1);
          }
    }

    function LONCAPAstale() {
    }

// END LON-CAPA Internal
// ]]>
</script>
ENDDONOTREGTHIS

        }
        return '';
    }
}

sub get_inline_text {
    my %text = (
                 pgrd     => 'Content Grades',
                 subm     => 'Content Submissions',
                 pparm    => 'Content Settings',
                 docs     => 'Folder/Page Content',
                 pcstr    => 'Edit',
                 prt      => 'Print',
                 alnk     => 'Stored Links',
                 anot     => 'Notes',
                 catalog  => 'Info',
                 eval     => 'Evaluate',
                 fdbk     => 'Feedback',
    );
    return %text;
}

sub get_rc_text {
    my %text = (
                   pgrd    => 'problem[_1]',
                   subm    => 'view sub-[_1]',
                   pparm   => 'problem[_2]',
                   pcstr   => 'edit[_1]',
                   prt     => 'prepare[_1]',
                   back    => 'backward[_1]',
                   forw    => 'forward[_1]',
                   alnk    => 'add to[_1]',
                   anot    => 'anno-[_1]',
                   catalog => 'catalog[_2]',
                   eval    => 'evaluate[_1]',
                   fdbk    => 'feedback[_1]',
    );
    return %text;
}

sub loadevents() {
    if ($env{'request.state'} eq 'construct' ||
        $env{'request.noversionuri'} =~ m{^/res/adm/pages/}) { return ''; }
    return 'LONCAPAreg();';
}

sub unloadevents() {
    if ($env{'request.state'} eq 'construct' ||
        $env{'request.noversionuri'} =~ m{^/res/adm/pages/}) { return ''; }
    return 'LONCAPAstale();';
}

sub startupremote {
    my ($lowerurl)=@_;
    unless ($env{'environment.remote'} eq 'on') {
        return ('<meta HTTP-EQUIV="Refresh" CONTENT="0.5; url='.$lowerurl.'" />');
    }
#
# The Remote actually gets launched!
#
    my $configmenu=&rawconfig();
    my $esclowerurl=&escape($lowerurl);
    my $message=&mt('"Waiting for Remote Control window to load: "+[_1]','waited');
    return(<<ENDREMOTESTARTUP);
<script type="text/javascript">
// <![CDATA[
var timestart;
function wheelswitch() {
    if (typeof(document.wheel) != 'undefined') {
        if (typeof(document.wheel.spin) != 'undefined') {
            var date=new Date();
            var waited=Math.round(30-((date.getTime()-timestart)/1000));
            document.wheel.spin.value=$message;
        }
    }
   if (window.status=='|') {
      window.status='/';
   } else {
      if (window.status=='/') {
         window.status='-';
      } else {
         if (window.status=='-') {
            window.status='\\\\';
         } else {
            if (window.status=='\\\\') { window.status='|'; }
         }
      }
   }
}

// ---------------------------------------------------------- The wait function
var canceltim;
function wait() {
   if ((menuloaded==1) || (tim==1)) {
      window.status='Done.';
      if (tim==0) {
         clearTimeout(canceltim);
         $configmenu
         window.location='$lowerurl';
      } else {
          window.location='/adm/remote?action=collapse&url=$esclowerurl';
      }
   } else {
      wheelswitch();
      setTimeout('wait();',200);
   }
}

function main() {
   canceltim=setTimeout('tim=1;',30000);
   window.status='-';
   var date=new Date();
   timestart=date.getTime();
   wait();
}

// ]]>
</script>
ENDREMOTESTARTUP
}

sub setflags() {
    return(<<ENDSETFLAGS);
<script type="text/javascript">
// <![CDATA[
    menuloaded=0;
    tim=0;
// ]]>
</script>
ENDSETFLAGS
}

sub maincall() {
    unless ($env{'environment.remote'} eq 'on') { return ''; }
    return(<<ENDMAINCALL);
<script type="text/javascript">
// <![CDATA[
    main();
// ]]>
</script>
ENDMAINCALL
}

sub load_remote_msg {
    my ($lowerurl)=@_;

    unless ($env{'environment.remote'} eq 'on') { return ''; }

    my $esclowerurl=&escape($lowerurl);
    my $link=&mt('[_1]Continue[_2] on in Inline Menu mode'
                ,'<a href="/adm/remote?action=collapse&amp;url='.$esclowerurl.'">'
                ,'</a>');
    return(<<ENDREMOTEFORM);
<p>
<form name="wheel">
<input name="spin" type="text" size="60" />
</form>
</p>
<p>$link</p>
ENDREMOTEFORM
}

sub get_menu_name {
    my $hostid = $Apache::lonnet::perlvar{'lonHostID'};
    $hostid =~ s/\W//g;
    return 'LCmenu'.$hostid;
}


sub reopenmenu {
   unless ($env{'environment.remote'} eq 'on') { return ''; }
   my $menuname = &get_menu_name();
   my $nothing = &Apache::lonhtmlcommon::javascript_nothing();
   return('window.open('.$nothing.',"'.$menuname.'","",false);');
}


sub open {
    my $returnval='';
    unless ($env{'environment.remote'} eq 'on') {
        return
        '<script type="text/javascript">'."\n"
       .'// <![CDATA['."\n"
       .'self.name="loncapaclient";'."\n"
       .'// ]]>'."\n"
       .'</script>';
    }
    my $menuname = &get_menu_name();

#    unless (shift eq 'unix') {
# resizing does not work on linux because of virtual desktop sizes
#       $returnval.=(<<ENDRESIZE);
#if (window.screen) {
#    self.resizeTo(screen.availWidth-215,screen.availHeight-55);
#    self.moveTo(190,15);
#}
#ENDRESIZE
#    }
    $returnval=(<<ENDOPEN);
// <![CDATA[
window.status='Opening LON-CAPA Remote Control';
var menu=window.open("/res/adm/pages/menu.html?inhibitmenu=yes","$menuname",
"height=375,width=150,scrollbars=no,menubar=no,top=5,left=5,screenX=5,screenY=5");
self.name='loncapaclient';
// ]]>
ENDOPEN
    return '<script type="text/javascript">'.$returnval.'</script>';
}

sub get_editbutton {
    my ($cfile,$home,$switchserver,$forceedit,$forceview,$forcereg,$hostname) = @_;
    my $jscall;
    if (($forceview) && ($env{'form.todocs'})) {
        my ($folderpath,$command,$navmap);
        if ($env{'request.symb'}) {
            $folderpath = &Apache::loncommon::symb_to_docspath($env{'request.symb'},\$navmap);
        } elsif ($env{'form.folderpath'} =~ /^supplemental/) {
            $folderpath = $env{'form.folderpath'};
            $command = '&forcesupplement=1';
        }
        $folderpath = &escape(&HTML::Entities::encode(&escape($folderpath),'<>&"'));
        $jscall = "go('/adm/coursedocs?folderpath=$folderpath$command')";
    } else {
        my $suppanchor;
        if ($env{'form.folderpath'}) {
            $suppanchor = $env{'form.anchor'};
        }
        $jscall = &Apache::lonhtmlcommon::jump_to_editres($cfile,$home,$switchserver,
                                                $forceedit,$forcereg,$env{'request.symb'},
                                                &escape($env{'form.folderpath'}),
                                                &escape($env{'form.title'}),$hostname,
                                                $env{'form.idx'},&escape($env{'form.suppurl'}),
                                                $env{'form.todocs'},$suppanchor);
    }
    if ($jscall) {
        my $icon = 'pcstr.png';
        my $label = 'Edit';
        if ($forceview) {
            $icon = 'tolastloc.png';
            $label = 'Exit Editing';
        }
        my $infunc = 1;
        my $clearbutton;
        if ($env{'environment.remote'} eq 'on') {
            if ($cfile =~ m{^/priv/}) {
                undef($infunc);
                $label = 'edit';
            } else {
                $clearbutton = 1;
            }
        }
        my $editor = &switch('','',6,1,$icon,$label,'resource[_2]',
                             $jscall,"Edit this resource",'','',$infunc);
        if ($infunc) {
            return 1;
        } elsif ($clearbutton) {
            return &clear(6,1);
        } else {
            return $editor;
        }
    }
    return;
}

sub prepare_functions {
    my ($resurl,$forcereg,$group,$bread_crumbs,$advtools,$docscrumbs,$hostname,$forbodytag) = @_;
    unless ($env{'request.registered'}) {
        undef(@inlineremote);
    }
    my ($cdom,$cnum,%perms,$cfile,$switchserver,$home,$forceedit,
        $forceview);

    if ($env{'request.course.id'}) {
        $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
        $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
        $perms{'mdc'} = &Apache::lonnet::allowed('mdc',$env{'request.course.id'});
    }

    my $editbutton = '';
    my $viewsrcbutton = '';
    my $clientip = &Apache::lonnet::get_requestor_ip();
#
# Determine whether or not to display 'Edit' or 'View Source' icon/button
#
    if ($resurl =~ m{^/?adm/($match_domain)/($match_username)/aboutme$}) {
        my $blocked = &Apache::loncommon::blocking_status('about',$clientip,$2,$1);
        my $file=&Apache::lonnet::declutter($env{'request.filename'});
        ($cfile,$home,$switchserver,$forceedit,$forceview) =
            &Apache::lonnet::can_edit_resource($file,$cnum,$cdom,
                &Apache::lonnet::clutter($resurl),$env{'request.symb'},$group);
        if (($cfile) && ($home ne '') && ($home ne 'no_host') && (!$blocked)) {
            $editbutton = &get_editbutton($cfile,$home,$switchserver,
                                          $forceedit,$forceview,$forcereg);
        }
    } elsif ((!$env{'request.course.id'}) &&
             ($env{'user.author'}) && ($env{'request.filename'}) &&
             ($env{'request.role'} !~/^(aa|ca|au)/)) {
#
# Currently do not have the role of author or co-author.
# Do we have authoring privileges for the resource?
#
        my $file=&Apache::lonnet::declutter($env{'request.filename'});
        ($cfile,$home,$switchserver,$forceedit,$forceview) =
            &Apache::lonnet::can_edit_resource($file,$cnum,$cdom,
                &Apache::lonnet::clutter($resurl),$env{'request.symb'},$group);
        if (($cfile) && ($home ne '') && ($home ne 'no_host')) {
            $editbutton = &get_editbutton($cfile,$home,$switchserver,
                                          $forceedit,$forceview,$forcereg);
        }
    } elsif ($env{'request.course.id'}) {
#
# This applies in course context
#
        if (($perms{'mdc'}) &&
            (($resurl =~ m{^/?public/$cdom/$cnum/syllabus}) ||
            ($resurl =~ m{^/?uploaded/$cdom/$cnum/portfolio/syllabus/}) || 
            (($resurl =~ m{^/?uploaded/$cdom/$cnum/default_\d+\.sequence$}) && ($env{'form.navmap'})))) {
            if ($resurl =~ m{^/}) {
                $cfile = $resurl;
            } else {
                $cfile = "/$resurl";
            }
            $home = &Apache::lonnet::homeserver($cnum,$cdom);
            if ($env{'form.forceedit'}) {
                $forceview = 1;
            } else {
                $forceedit = 1;
            }
            if ($cfile =~ m{^/uploaded/$cdom/$cnum/default_\d+\.sequence$}) {
                my $text = 'Edit Folder';
                &switch('','',7,4,'docs-22x22.png','Edit Folder','parms[_2]',
                        "gocmd('/adm/coursedocs','direct')",
                        'Folder/Page Content');
                $editbutton = 1;
            } else {
                $editbutton = &get_editbutton($cfile,$home,$switchserver,
                                              $forceedit,$forceview,$forcereg,
                                              $hostname);
            }
        } elsif (($resurl eq '/adm/extresedit') &&
                 (($env{'form.symb'}) || ($env{'form.folderpath'}))) {
            ($cfile,$home,$switchserver,$forceedit,$forceview) =
            &Apache::lonnet::can_edit_resource($resurl,$cnum,$cdom,$resurl,
                                               $env{'form.symb'});
            if ($cfile ne '') {
                $editbutton = &get_editbutton($cfile,$home,$switchserver,
                                              $forceedit,$forceview,$forcereg,
                                              $env{'form.title'},$env{'form.suppurl'});
            }
        } elsif (($resurl =~ m{^/?adm/viewclasslist$}) &&
                 (&Apache::lonnet::allowed('opa',$env{'request.course.id'}))) {
            ($cfile,$home,$switchserver,$forceedit,$forceview) =
            &Apache::lonnet::can_edit_resource($resurl,$cnum,$cdom,$resurl,
                                               $env{'form.symb'});
            $editbutton = &get_editbutton($cfile,$home,$switchserver,
                                          $forceedit,$forceview,$forcereg);
        } elsif (($resurl !~ m{^/?adm/($match_domain)/($match_username)/aboutme$}) &&
                 ($resurl ne '/cgi-bin/printout.pl')) {
            if ($env{'request.filename'}) {
                my $file=&Apache::lonnet::declutter($env{'request.filename'});
                ($cfile,$home,$switchserver,$forceedit,$forceview) =
                    &Apache::lonnet::can_edit_resource($file,$cnum,$cdom,
                        &Apache::lonnet::clutter($resurl),$env{'request.symb'},$group);
                if ($cfile ne '') {
                    $editbutton = &get_editbutton($cfile,$home,$switchserver,
                                                  $forceedit,$forceview,$forcereg);
                }
                if ((($cfile eq '') || (!$editbutton)) &&
                    ($resurl =~ /$LONCAPA::assess_re/)) {
                    my $showurl = &Apache::lonnet::clutter($resurl);
                    my $crs_sec = $env{'request.course.id'} . (($env{'request.course.sec'} ne '')
                                                              ? "/$env{'request.course.sec'}"
                                                              : '');
                    if ((&Apache::lonnet::allowed('cre','/')) &&
                        (&Apache::lonnet::metadata($resurl,'sourceavail') eq 'open')) {
                        $viewsrcbutton = 1;
                    } elsif (&Apache::lonnet::allowed('vxc',$crs_sec)) {
                        if ($showurl =~ m{^\Q/res/$cdom/\E($match_username)/}) {
                            my $auname = $1;
                            if (($env{'request.course.adhocsrcaccess'} ne '') &&
                                (grep(/^\Q$auname\E$/,split(/,/,$env{'request.course.adhocsrcaccess'})))) {
                                $viewsrcbutton = 1;
                            } elsif ((&Apache::lonnet::metadata($resurl,'sourceavail') eq 'open') &&
                                     (&Apache::lonnet::allowed('bre',$crs_sec))) {
                                $viewsrcbutton = 1;
                            }
                        }
                    }
                    if ($viewsrcbutton) {
                        &switch('','',6,1,'pcstr.png','View Source','resource[_2]','open_source()',
                                'View source code');
                    }
                }
            }
        }
    }
# End determination of 'Edit' icon/button display

    if ($env{'request.course.id'}) {
# This applies to about me page for users in a course
        if ($resurl =~ m{^/?adm/($match_domain)/($match_username)/aboutme$}) {
            my ($sdom,$sname) = ($1,$2);
            unless (&Apache::lonnet::is_course($sdom,$sname)) {
                my $blocked = &Apache::loncommon::blocking_status('about',$clientip,$sname,$sdom);
                unless ($blocked) {
                    &switch('','',6,4,'mail-message-new-22x22.png','Message to user',
                            '',
                            "go('/adm/email?compose=individual&amp;recname=$sname&amp;recdom=$sdom')",
                                'Send message to specific user','','',1);
                }
            }
            my $hideprivileged = 1;
            if (&Apache::lonnet::in_course($sdom,$sname,$cdom,$cnum,undef,
                                           $hideprivileged)) {
                foreach my $priv ('vsa','vgr','srm') {
                    $perms{$priv} = &Apache::lonnet::allowed($priv,$env{'request.course.id'});
                    if (!$perms{$priv} && $env{'request.course.sec'} ne '') {
                        $perms{$priv} =
                            &Apache::lonnet::allowed($priv,"$env{'request.course.id'}/$env{'request.course.sec'}");
                    }
                }
                if ($perms{'vsa'}) {
                    &switch('','',6,5,'trck-22x22.png','Activity',
                            '',
                            "go('/adm/trackstudent?selected_student=$sname:$sdom')",
                            'View recent activity by this person','','',1);
                }
                if ($perms{'vgr'}) {
                    &switch('','',6,6,'rsrv-22x22.png','Reservations',
                            '',
                            "go('/adm/slotrequest?command=showresv&amp;origin=aboutme&amp;uname=$sname&amp;udom=$sdom')",
                            'Slot reservation history','','',1);
                }
                if ($perms{'srm'}) {
                    &switch('','',6,7,'contact-new-22x22.png','Records',
                            '',
                            "go('/adm/email?recordftf=retrieve&amp;recname=$sname&amp;recdom=$sdom')",
                            'Add records','','',1);
                }
            }
        }
        if (($env{'form.folderpath'} =~ /^supplemental/) &&
            (&Apache::lonnet::allowed('mdc',$env{'request.course.id'})) &&
            (($resurl =~ m{^/adm/wrapper/ext/}) ||
             ($resurl =~ m{^/uploaded/$cdom/$cnum/supplemental/}) ||
             ($resurl eq '/adm/supplemental') ||
             ($resurl =~ m{^/public/$cdom/$cnum/syllabus$}) ||
             ($resurl =~ m{^/adm/$match_domain/$match_username/aboutme$}))) {
            my @folders=split('&',$env{'form.folderpath'});
            if ((@folders > 2) || ($resurl ne '/adm/supplemental')) {
                my $suppanchor;
                if ($resurl =~ m{^/adm/wrapper/ext/}) {
                    $suppanchor = $env{'form.anchor'};
                }
                my $esc_path=&escape(&HTML::Entities::encode(&escape($env{'form.folderpath'}),'<>&"'));
                my $link = '/adm/coursedocs?command=direct&amp;forcesupplement=1&amp;supppath='.
                           "$esc_path&amp;anchor=$suppanchor";
                if ($env{'request.use_absolute'} ne '') {
                    $link = $env{'request.use_absolute'}.$link;
                }
                &switch('','',7,4,'docs-22x22.png','Edit Folder','parms[_2]',
                        "location.href='$link'",'Folder/Page Content');
            }
        }
    }

# End checking for items for about me page for users in a course
    if ($docscrumbs) {
        &Apache::lonhtmlcommon::clear_breadcrumb_tools();
        &advtools_crumbs(@inlineremote);
        return $editbutton;
    } elsif (($env{'request.registered'}) && (!ref($forbodytag))) {
        return $editbutton || $viewsrcbutton;
    } else {
        if (ref($bread_crumbs) eq 'ARRAY') {
            if (@inlineremote > 0) {
                if (ref($advtools) eq 'ARRAY') {
                    @{$advtools} = @inlineremote;
                }
            }
            return;
        } elsif (@inlineremote > 0) {
            &Apache::lonhtmlcommon::clear_breadcrumb_tools();
            &advtools_crumbs(@inlineremote);
            if (ref($forbodytag)) {
                $$forbodytag =
                    &Apache::lonhtmlcommon::scripttag('', 'start')
                   .&Apache::lonhtmlcommon::breadcrumbs(undef,undef,0)
                   .&Apache::lonhtmlcommon::scripttag('', 'end');
            }
            return;
        }
    }
}

sub advtools_crumbs {
    my @funcs = @_;
    if ($env{'request.noversionuri'} =~ m{^/adm/$match_domain/$match_username/aboutme$}) {
        &Apache::lonhtmlcommon::add_breadcrumb_tool(
            'advtools', @funcs[61,64,65,66,67,74]);
    } elsif ($env{'request.noversionuri'} !~ m{^/adm/(navmaps|viewclasslist)(\?|$)}) {
        &Apache::lonhtmlcommon::add_breadcrumb_tool(
            'advtools', @funcs[61,71,72,73,74,92]);
    } elsif ($env{'request.noversionuri'} eq '/adm/viewclasslist') {
        &Apache::lonhtmlcommon::add_breadcrumb_tool(
            'advtools', $funcs[61]);
    }
}

# ================================================================== Raw Config

sub clear {
    my ($row,$col)=@_;
    if ($env{'environment.remote'} eq 'on') {
       if (($row<1) || ($row>13)) { return ''; }
       return "\n".qq(window.status+='.';swmenu.clearbut($row,$col););
    } else {
        $inlineremote[10*$row+$col]='';
        return '';
    }
}

# ============================================ Switch a button or create a link
# Switch acts on the javascript that is executed when a button is clicked.  
# The javascript is usually similar to "go('/adm/roles')" or "cstrgo(..)".

sub switch {
    my ($uname,$udom,$row,$col,$img,$top,$bot,$act,$desc,$cat,$nobreak,$infunc)=@_;
    $act=~s/\$uname/$uname/g;
    $act=~s/\$udom/$udom/g;
    $top=&mt($top);
    $bot=&mt($bot);
    $desc=&mt($desc);
    my $idx=10*$row+$col;
    $category_members{$cat}.=':'.$idx;

    if (($env{'environment.remote'} eq 'on') && (!$infunc)) {
        if (($row<1) || ($row>13)) { return ''; }
        if ($env{'request.state'} eq 'construct') {
            my $text = $top.' '.$bot;
            $text=~s/\s*\-\s*//gs;
            my $pic = '<img alt="'.$text.'" src="'.
                      &Apache::loncommon::lonhttpdurl('/res/adm/pages/'.$img).
                      '" align="'.($nobreak==3?'right':'left').'" class="LC_icon" />';
           $inlineremote[$idx] =
               '<a title="'.$desc.'" class="LC_menubuttons_link" href="javascript:'.$act.';">'.
               $pic.'<span class="LC_menubuttons_inline_text">'.$top.'&nbsp;</span></a>';
        }
# Remote
        $img=~s/\.png$/\.gif/;
        return "\n".
 qq(window.status+='.';swmenu.switchbutton($row,$col,"$img","$top","$bot","$act","$desc"););
    }

# Inline Menu
    if ($nobreak==2) { return ''; }
    my $text=$top.' '.$bot;
    $text=~s/\s*\-\s*//gs;

    my $pic=
	   '<img alt="'.$text.'" src="'.
	   &Apache::loncommon::lonhttpdurl('/res/adm/pages/'.$img).
	   '" align="'.($nobreak==3?'right':'left').'" class="LC_icon" />';
    if ($env{'browser.interface'} eq 'faketextual') {
# Main Menu
	   if ($nobreak==3) {
	       $inlineremote[$idx]="\n".
		   '<td class="LC_menubuttons_text" align="right">'.$text.
		   '</td><td align="left">'.
		   '<a href="javascript:'.$act.';">'.$pic.'</a></td></tr>';
	   } elsif ($nobreak) {
	       $inlineremote[$idx]="\n<tr>".
		   '<td align="left">'.
		   '<a href="javascript:'.$act.';">'.$pic.'</a></td>
                    <td class="LC_menubuttons_text" align="left"><a class="LC_menubuttons_link" href="javascript:'.$act.';"><span class="LC_menubuttons_inline_text">'.$text.'</span></a></td>';
	   } else {
	       $inlineremote[$idx]="\n<tr>".
		   '<td align="left">'.
		   '<a href="javascript:'.$act.';">'.$pic.
		   '</a></td><td class="LC_menubuttons_text" colspan="3">'.
		   '<a class="LC_menubuttons_link" href="javascript:'.$act.';"><span class="LC_menubuttons_inline_text">'.$desc.'</span></a></td></tr>';
	   }
    } else {
# Inline Menu
        my @tools = (93,91,81,82,83);
        unless ($env{'request.state'} eq 'construct') {
            push(@tools,63);
        }
        if (($env{'environment.icons'} eq 'iconsonly') &&
            (grep(/^$idx$/,@tools))) {
            $inlineremote[$idx] =
        '<a title="'.$desc.'" class="LC_menubuttons_link" href="javascript:'.$act.';">'.$pic.'</a>';
        } else {
            $inlineremote[$idx] =
       '<a title="'.$desc.'" class="LC_menubuttons_link" href="javascript:'.$act.';">'.$pic.
       '<span class="LC_menubuttons_inline_text">'.$top.'&nbsp;</span></a>';
        }
    }
    return '';
}

sub secondlevel {
    my $output='';
    my 
    ($uname,$udom,$rol,$crs,$pub,$con,$row,$col,$prt,$img,$top,$bot,$act,$desc,$cat)=@_;
    if ($prt eq 'any') {
	   $output.=&switch($uname,$udom,$row,$col,$img,$top,$bot,$act,$desc,$cat);
    } elsif ($prt=~/^r(\w+)/) {
        if ($rol eq $1) {
           $output.=&switch($uname,$udom,$row,$col,$img,$top,$bot,$act,$desc,$cat);
        }
    }
    return $output;
}

sub openmenu {
    my $menuname = &get_menu_name();
    unless ($env{'environment.remote'} eq 'on') { return ''; }
    my $nothing = &Apache::lonhtmlcommon::javascript_nothing();
    return "window.open(".$nothing.",'".$menuname."');";
}

sub inlinemenu {
    undef(@inlineremote);
    undef(%category_members);
# calling rawconfig with "1" will evaluate mydesk.tab, even if there is no active remote control
    &rawconfig(1);
    my $output='<table><tr>';
    for (my $col=1; $col<=2; $col++) {
        $output.='<td class="LC_mainmenu_col_fieldset">';
        for (my $row=1; $row<=8; $row++) {
            foreach my $cat (keys(%category_members)) {
               if ($category_positions{$cat} ne "$col,$row") { next; }
               #$output.='<table><tr><td colspan="4" class="LC_menubuttons_category">'.&mt($category_names{$cat}).'</td></tr>';
               $output.='<div class="LC_Box LC_400Box">';
	       $output.='<h3 class="LC_hcell">'.&mt($category_names{$cat}).'</h3>';
               $output.='<table>';
               my %active=();
               foreach my $menu_item (split(/\:/,$category_members{$cat})) {
                  if ($inlineremote[$menu_item]) {
                     $active{$menu_item}=1;
                  }
               }  
               foreach my $item (sort(keys(%active))) {
                  $output.=$inlineremote[$item];
               }
               $output.='</table>';
               $output.='</div>';
            }
         }
         $output.="</td>";
    }
    $output.="</tr></table>";
    return $output;
}

sub rawconfig {
#
# This evaluates mydesk.tab
# Need to add more positions and more privileges to deal with all
# menu items.
#
    my $textualoverride=shift;
    my $output='';
    if ($env{'environment.remote'} eq 'on') {
       $output.=
 "window.status='Opening Remote Control';var swmenu=".&openmenu().
"\nwindow.status='Configuring Remote Control ';";
    } else {
        unless ($textualoverride) { return ''; }
    }
    my $uname=$env{'user.name'};
    my $udom=$env{'user.domain'};
    my $adv=$env{'user.adv'};
    my $show_course=&Apache::loncommon::show_course();
    my $author=$env{'user.author'};
    my $crs='';
    my $crstype='';
    if ($env{'request.course.id'}) {
       $crs='/'.$env{'request.course.id'};
       if ($env{'request.course.sec'}) {
	   $crs.='_'.$env{'request.course.sec'};
       }
       $crs=~s/\_/\//g;
       $crstype = &Apache::loncommon::course_type();
    }
    my $pub=($env{'request.state'} eq 'published');
    my $con=($env{'request.state'} eq 'construct');
    my $rol=$env{'request.role'};
    my $requested_domain;
    if ($rol) {
       $requested_domain = $env{'request.role.domain'};
    }
    foreach my $line (@desklines) {
        my ($row,$col,$pro,$prt,$img,$top,$bot,$act,$desc,$cat)=split(/\:/,$line);
        $prt=~s/\$uname/$uname/g;
        $prt=~s/\$udom/$udom/g;
        if ($prt =~ /\$crs/) {
            next unless ($env{'request.course.id'});
            next if ($crstype eq 'Community');
            $prt=~s/\$crs/$crs/g;
        } elsif ($prt =~ /\$cmty/) {
            next unless ($env{'request.course.id'});
            next if ($crstype ne 'Community');
            $prt=~s/\$cmty/$crs/g;
        }
        if ($prt =~ m/\$requested_domain/) {
            if ((!$requested_domain) && ($pro eq 'pbre') && ($env{'user.adv'})) {
                $prt=~s/\$requested_domain/$env{'user.domain'}/g;
            } else {
                $prt=~s/\$requested_domain/$requested_domain/g;
            }
        }
        if ($category_names{$cat}!~/\w/) { $cat='oth'; }
        if ($pro eq 'clear') {
	    $output.=&clear($row,$col);
        } elsif ($pro eq 'any') {
               $output.=&secondlevel(
	  $uname,$udom,$rol,$crs,$pub,$con,$row,$col,$prt,$img,$top,$bot,$act,$desc,$cat);
	} elsif ($pro eq 'smp') {
            unless ($adv) {
               $output.=&secondlevel(
          $uname,$udom,$rol,$crs,$pub,$con,$row,$col,$prt,$img,$top,$bot,$act,$desc,$cat);
            }
        } elsif ($pro eq 'adv') {
            if ($adv) {
               $output.=&secondlevel(
	  $uname,$udom,$rol,$crs,$pub,$con,$row,$col,$prt,$img,$top,$bot,$act,$desc,$cat);
            }
	} elsif ($pro eq 'shc') {
            if ($show_course) {
               $output.=&secondlevel(
          $uname,$udom,$rol,$crs,$pub,$con,$row,$col,$prt,$img,$top,$bot,$act,$desc,$cat);
            }
        } elsif ($pro eq 'nsc') {
            if (!$show_course) {
               $output.=&secondlevel(
	  $uname,$udom,$rol,$crs,$pub,$con,$row,$col,$prt,$img,$top,$bot,$act,$desc,$cat);
            }
        } elsif (($pro=~/^p(\w+)/) && ($prt)) {
            my $priv = $1;
            if ($priv =~ /^mdc(Course|Community)/) {
                if ($crstype eq $1) {
                    $priv = 'mdc';
                } else {
                    next;
                }
            }
            if ((($priv eq 'bre') && (&Apache::lonnet::allowed($priv,$prt) eq 'F')) ||
                (($priv ne 'bre') && (&Apache::lonnet::allowed($priv,$prt)))) {
                $output.=&switch($uname,$udom,$row,$col,$img,$top,$bot,$act,$desc,$cat);
            }
        } elsif ($pro eq 'course')  {
            if (($env{'request.course.fn'}) && ($crstype ne 'Community')) {
               $output.=&switch($uname,$udom,$row,$col,$img,$top,$bot,$act,$desc,$cat);
	    }
        } elsif ($pro eq 'community')  {
            if (($env{'request.course.fn'}) && ($crstype eq 'Community')) {
               $output.=&switch($uname,$udom,$row,$col,$img,$top,$bot,$act,$desc,$cat);
            }
        } elsif ($pro =~ /^courseenv_(.*)$/) {
            my $key = $1;
            if ($crstype ne 'Community') {
                my $coursepref = $env{'course.'.$env{'request.course.id'}.'.'.$key};
                if ($key eq 'canuse_pdfforms') {
                    if ($env{'request.course.id'} && $coursepref eq '') {
                        my %domdefs = &Apache::lonnet::get_domain_defaults($env{'course.'.$env{'request.course.id'}.'.domain'});
                        $coursepref = $domdefs{'canuse_pdfforms'};
                    }
                }
                if ($coursepref) { 
                    $output.=&switch($uname,$udom,$row,$col,$img,$top,$bot,$act,$desc,$cat);
                }
            }
        } elsif ($pro =~ /^communityenv_(.*)$/) {
            my $key = $1;
            if ($crstype eq 'Community') {
                my $coursepref = $env{'course.'.$env{'request.course.id'}.'.'.$key};
                if ($key eq 'canuse_pdfforms') {
                    if ($env{'request.course.id'} && $coursepref eq '') {
                        my %domdefs = &Apache::lonnet::get_domain_defaults($env{'course.'.$env{'request.course.id'}.'.domain'});
                        $coursepref = $domdefs{'canuse_pdfforms'};
                    }
                }
                if ($coursepref) { 
                    $output.=&switch($uname,$udom,$row,$col,$img,$top,$bot,$act,$desc,$cat);
                }
            }
        } elsif ($pro =~ /^course_(.*)$/) {
            # Check for permissions inside of a course
            if (($env{'request.course.id'}) && ($crstype ne 'Community') && 
                (&Apache::lonnet::allowed($1,$env{'request.course.id'}.
            ($env{'request.course.sec'}?'/'.$env{'request.course.sec'}:''))
                 )) {
                $output.=&switch($uname,$udom,$row,$col,$img,$top,$bot,$act,$desc,$cat);
	    }
        } elsif ($pro =~ /^community_(.*)$/) {
            # Check for permissions inside of a community
            if (($env{'request.course.id'}) && ($crstype eq 'Community') &&   
                (&Apache::lonnet::allowed($1,$env{'request.course.id'}.
            ($env{'request.course.sec'}?'/'.$env{'request.course.sec'}:''))
                 )) {
                $output.=&switch($uname,$udom,$row,$col,$img,$top,$bot,$act,$desc,$cat);
            }
        } elsif ($pro eq 'author') {
            if ($author) {
                if ((($prt eq 'rca') && ($env{'request.role'}=~/^ca/)) ||
                    (($prt eq 'raa') && ($env{'request.role'}=~/^aa/)) || 
                    (($prt eq 'rau') && ($env{'request.role'}=~/^au/))) {
                    # Check that we are on the correct machine
                    my $cadom=$requested_domain;
                    my $caname=$env{'user.name'};
                    if (($prt eq 'rca') || ($prt eq 'raa')) {
		       ($cadom,$caname)=
                               ($env{'request.role'}=~/($match_domain)\/($match_username)$/);
                    }                       
                    $act =~ s/\$caname/$caname/g;
                    $act =~ s/\$cadom/$cadom/g;
                    my $home = &Apache::lonnet::homeserver($caname,$cadom);
		    my $allowed=0;
		    my @ids=&Apache::lonnet::current_machine_ids();
		    foreach my $id (@ids) { if ($id eq $home) { $allowed=1; } }
		    if ($allowed) {
                        $output.=&switch($caname,$cadom,
                                        $row,$col,$img,$top,$bot,$act,$desc,$cat);
                    }
                }
            }
        } elsif ($pro eq 'tools') {
            my @tools = ('aboutme','blog','portfolio');
            if (grep(/^\Q$prt\E$/,@tools)) {
                if (!&Apache::lonnet::usertools_access($env{'user.name'},
                                                       $env{'user.domain'},
                                                       $prt,undef,'tools')) {
                    $output.=&clear($row,$col);
                    next;
                }
            } elsif (($prt eq 'reqcrsnsc') || ($prt eq 'reqcrsshc')) {
                if (($prt eq 'reqcrsnsc') && ($show_course))   {
                    next;
                }
                if (($prt eq 'reqcrsshc') && (!$show_course)) {
                    next;
                }
                my $showreqcrs = &check_for_rcrs();
                if (!$showreqcrs) {
                    $output.=&clear($row,$col);
                    next;
                }
            }
            $prt='any';
            $output.=&secondlevel(
          $uname,$udom,$rol,$crs,$pub,$con,$row,$col,$prt,$img,$top,$bot,$act,$desc,$cat);
        }
    }
    if ($env{'environment.remote'} eq 'on') {
        $output.="\nwindow.status='Synchronizing Time';swmenu.syncclock(1000*".time.");\nwindow.status='Remote Control Configured.';";
        if (&Apache::lonmsg::newmail()) {
            $output.='swmenu.setstatus("you have","messages");';
        }
    }
    return $output;
}

sub check_for_rcrs {
    my $showreqcrs = 0;
    my @reqtypes = ('official','unofficial','community','textbook');
    foreach my $type (@reqtypes) {
        if (&Apache::lonnet::usertools_access($env{'user.name'},
                                              $env{'user.domain'},
                                              $type,undef,'requestcourses')) {
            $showreqcrs = 1;
            last;
        }
    }
    if (!$showreqcrs) {
        foreach my $type (@reqtypes) {
            if ($env{'environment.reqcrsotherdom.'.$type} ne '') {
                $showreqcrs = 1;
                last;
            }
        }
    }
    return $showreqcrs;
}

# ======================================================================= Close

sub close {
    unless ($env{'environment.remote'} eq 'on') { return ''; }
    my $menuname = &get_menu_name();
    return(<<ENDCLOSE);
<script type="text/javascript">
// <![CDATA[
window.status='Accessing Remote Control';
menu=window.open("/adm/rat/empty.html","$menuname",
                 "height=350,width=150,scrollbars=no,menubar=no");
window.status='Disabling Remote Control';
menu.active=0;
menu.autologout=0;
window.status='Closing Remote Control';
menu.close();
window.status='Done.';
// ]]>
</script>
ENDCLOSE
}

sub dc_popup_js {
    my %lt = &Apache::lonlocal::texthash(
                                          more => '(More ...)',
                                          less => '(Less ...)',
                                        );
    return <<"END";

function showCourseID() {
    document.getElementById('dccid').style.display='block';
    document.getElementById('dccid').style.textAlign='left';
    document.getElementById('dccid').style.textFace='normal';
    document.getElementById('dccidtext').innerHTML ='<a href="javascript:hideCourseID();" class="LC_menubuttons_link">$lt{'less'}</a>';
    return;
}

function hideCourseID() {
    document.getElementById('dccid').style.display='none';
    document.getElementById('dccidtext').innerHTML ='<a href="javascript:showCourseID()" class="LC_menubuttons_link">$lt{'more'}</a>';
    return;
}

END

}

sub countdown_toggle_js {
    return <<"END";

function toggleCountdown() {
    var countdownid = document.getElementById('duedatecountdown');
    var currstyle = countdownid.style.display;
    if (currstyle == 'inline') {
        countdownid.style.display = 'none';
        document.getElementById('ddcountcollapse').innerHTML='';
        document.getElementById('ddcountexpand').innerHTML='&#9668;&nbsp;';
    } else {
        countdownid.style.display = 'inline';
        document.getElementById('ddcountcollapse').innerHTML='&#9658;&nbsp;';
        document.getElementById('ddcountexpand').innerHTML='';
    }
    return;
}

END
}

sub utilityfunctions {
    my ($httphost) = @_;
    my $currenturl=&Apache::lonnet::clutter(&Apache::lonnet::fixversion((split(/\?/,$env{'request.noversionuri'}))[0]));
    my $currentsymb=&Apache::lonenc::check_encrypt($env{'request.symb'});
    if ($currenturl =~ m{^/adm/wrapper/ext/}) {
        if ($env{'request.external.querystring'}) {
            $currenturl .= ($currenturl=~/\?/)?'&':'?'.$env{'request.external.querystring'};
        }
        my ($anchor) = ($env{'request.symb'} =~ /(\#[^\#]+)$/);
        if (($anchor) && ($currenturl !~ /\Q$anchor\E$/)) {
            $currenturl .= $1;
        }
    }
    $currenturl=&Apache::lonenc::check_encrypt(&unescape($currenturl));
    
    my $dc_popup_cid;
    if ($env{'user.adv'} && exists($env{'user.role.dc./'.
                        $env{'course.'.$env{'request.course.id'}.
                                 '.domain'}.'/'})) {
        $dc_popup_cid = &dc_popup_js();
    }

    my $start_page_annotate = 
        &Apache::loncommon::start_page('Annotator',undef,
				       {'only_body' => 1,
					'js_ready'  => 1,
					'bgcolor'   => '#BBBBBB',
					'add_entries' => {
					    'onload' => 'javascript:document.goannotate.submit();'}});

    my $end_page_annotate = 
        &Apache::loncommon::end_page({'js_ready' => 1});

    my $jumptores = &Apache::lonhtmlcommon::javascript_jumpto_resource();

    my $esc_url=&escape($currenturl);
    my $esc_symb=&escape($currentsymb);

    my $countdown = &countdown_toggle_js();

    my $annotateurl = '/adm/annotation';
    if ($httphost) {
        $annotateurl = '/adm/annotations';
    }
    my $hostvar = '
function setLCHost() {
    var lcHostname="";
';
    if ($httphost =~ m{^https?\://}) {
        $hostvar .= '    var lcServer="'.$httphost.'";'."\n".
                    '    var hostReg = /^https?:\/\/([^\/]+)$/i;'."\n".
                    '    var match = hostReg.exec(lcServer);'."\n".
                    '    if (match.length) {'."\n".
                    '        if (match[1] == location.hostname) {'."\n".
                    '            lcHostname=lcServer;'."\n".
                    '        }'."\n".
                    '    }'."\n";
    }

    $hostvar .= '    return lcHostname;'."\n".
'}'."\n";

return (<<ENDUTILITY)
    $hostvar
    var currentURL=unescape("$esc_url");
    var reloadURL=unescape("$esc_url");
    var currentSymb=unescape("$esc_symb");

$dc_popup_cid

$jumptores

function gopost(url,postdata) {
   if (url!='') {
      var lcHostname = setLCHost();
      this.document.server.action=lcHostname+url;
      this.document.server.postdata.value=postdata;
      this.document.server.command.value='';
      this.document.server.url.value='';
      this.document.server.symb.value='';
      this.document.server.submit();
   }
}

function gocmd(url,cmd) {
   if (url!='') {
      var lcHostname = setLCHost();
      this.document.server.action=lcHostname+url;
      this.document.server.postdata.value='';
      this.document.server.command.value=cmd;
      this.document.server.url.value=currentURL;
      this.document.server.symb.value=currentSymb;
      this.document.server.submit();
   }
}

function gocstr(url,filename) {
    if (url == '/adm/cfile?action=delete') {
        this.document.cstrdelete.filename.value = filename
        this.document.cstrdelete.submit();
        return;
    }
    if (url == '/adm/printout') {
        this.document.cstrprint.postdata.value = filename
        this.document.cstrprint.curseed.value = 0;
        this.document.cstrprint.problemtype.value = 0;
        if (this.document.lonhomework) {
            if ((this.document.lonhomework.rndseed) && (this.document.lonhomework.rndseed.value != null) && (this.document.lonhomework.rndseed.value != '')) {
                this.document.cstrprint.curseed.value = this.document.lonhomework.rndseed.value
            }
            if (this.document.lonhomework.problemtype) {
		if (this.document.lonhomework.problemtype.value) {
		    this.document.cstrprint.problemtype.value = 
			this.document.lonhomework.problemtype.value;
		} else if (this.document.lonhomework.problemtype.options) {
		    for (var i=0; i<this.document.lonhomework.problemtype.options.length; i++) {
			if (this.document.lonhomework.problemtype.options[i].selected) {
			    if (this.document.lonhomework.problemtype.options[i].value != null && this.document.lonhomework.problemtype.options[i].value != '') { 
				this.document.cstrprint.problemtype.value = this.document.lonhomework.problemtype.options[i].value
				}
			}
		    }
		}
	    }
	}
        this.document.cstrprint.submit();
        return;
    }
    if (url !='') {
        this.document.constspace.filename.value = filename;
        this.document.constspace.action = url;
        this.document.constspace.submit();
    }
}

function golist(url) {
   if (url!='' && url!= null) {
       currentURL = null;
       currentSymb= null;
       var lcHostname = setLCHost();
       top.location.href=lcHostname+url;
   }
}



function catalog_info(isMobile) {
    if (isMobile == 1) {
        openMyModal(window.location.pathname+'.meta?modal=1',500,400,'yes');
    } else {
        loncatinfo=window.open(window.location.pathname+'.meta',"LONcatInfo",'height=500,width=400,resizable=yes,scrollbars=yes,location=no,menubar=no,toolbar=no');
    }
}

function chat_win() {
   var lcHostname = setLCHost();
   lonchat=window.open(lcHostname+'/res/adm/pages/chatroom.html',"LONchat",'height=320,width=480,resizable=yes,location=no,menubar=no,toolbar=no');
}

function group_chat(group) {
   var lcHostname = setLCHost();
   var url = lcHostname+'/adm/groupchat?group='+group;
   var winName = 'LONchat_'+group;
   grpchat=window.open(url,winName,'height=320,width=280,resizable=yes,location=no,menubar=no,toolbar=no');
}

function annotate() {
   w_Annotator_flag=1;
   annotator=window.open('','Annotator','width=365,height=265,scrollbars=0');
   annotator.document.write(
   '$start_page_annotate'
  +"<form name='goannotate' target='Annotator' method='post' "
  +"action='$annotateurl'>"
  +"<input type='hidden' name='symbnew' value='"+currentSymb+"' />"
  +"<\\/form>"
  +'$end_page_annotate');
   annotator.document.close();
}

function open_StoredLinks_Import(rat) {
   var newWin;
   var lcHostname = setLCHost();
   if (rat) {
       newWin = window.open(lcHostname+'/adm/wishlist?inhibitmenu=yes&mode=import&rat='+rat,
                            'wishlistImport','scrollbars=1,resizable=1,menubar=0');
   }
   else {
       newWin = window.open(lcHostname+'/adm/wishlist?inhibitmenu=yes&mode=import',
                            'wishlistImport','scrollbars=1,resizable=1,menubar=0');
   }
   newWin.focus();
}

function open_source() {
   sourcewin=window.open('/adm/source?inhibitmenu=yes&viewonly=1&filename='+currentURL,'LONsource',
                         'height=500,width=600,resizable=yes,location=no,menubar=no,toolbar=no,scrollbars=yes');
}

(function (\$) {
  \$(document).ready(function () {
    \$.single=function(a){return function(b){a[0]=b;return a}}(\$([1]));
    /*\@cc_on
      if (!window.XMLHttpRequest) {
        \$('.LC_hoverable').each(function () {
          this.attachEvent('onmouseenter', function (evt) { \$.single(evt.srcElement).addClass('hover'); });
          this.attachEvent('onmouseleave', function (evt) { \$.single(evt.srcElement).removeClass('hover'); });
        });
      }
    \@*/
  });
}(jQuery));

$countdown

ENDUTILITY
}

sub serverform {
    return(<<ENDSERVERFORM);
<form name="server" action="/adm/logout" method="post" target="_top">
<input type="hidden" name="postdata" value="none" />
<input type="hidden" name="command" value="none" />
<input type="hidden" name="url" value="none" />
<input type="hidden" name="symb" value="none" />
</form>
ENDSERVERFORM
}

sub constspaceform {
    return(<<ENDCONSTSPACEFORM);
<form name="constspace" action="/adm/logout" method="post" target="_top">
<input type="hidden" name="filename" value="" />
</form>
<form name="cstrdelete" action="/adm/cfile" method="post" target="_top">
<input type="hidden" name="action" value="delete" /> 
<input type="hidden" name="filename" value="" />
</form>
<form name="cstrprint" action="/adm/printout" target="_parent" method="post">
<input type="hidden" name="postdata" value="" />
<input type="hidden" name="curseed" value="" />
<input type="hidden" name="problemtype" value="" />
</form>

ENDCONSTSPACEFORM
}

sub get_nav_status {
    my $navstatus="swmenu.w_loncapanav_flag=";
    if ($env{'environment.remotenavmap'} eq 'on') {
        $navstatus.="1";
    } else {
        $navstatus.="-1";
    }
    return $navstatus;
}

sub hidden_button_check {
    if ( $env{'request.course.id'} eq ''
         || $env{'request.role.adv'} ) {

        return;
    }
    my $buttonshide = &Apache::lonnet::EXT('resource.0.buttonshide');
    return $buttonshide; 
}

sub roles_selector {
    my ($cdom,$cnum,$httphost) = @_;
    my $crstype = &Apache::loncommon::course_type();
    my $now = time;
    my (%courseroles,%seccount,%courseprivs,%roledesc);
    my $is_cc;
    my ($js,$form,$switcher);
    my $ccrole;
    if ($crstype eq 'Community') {
        $ccrole = 'co';
    } else {
        $ccrole = 'cc';
    }
    my ($privref,$gotsymb,$destsymb);
    my $destinationurl = $ENV{'REQUEST_URI'};
    if ($destinationurl =~ /(\?|\&)symb=/) {
        $gotsymb = 1;
    } elsif ($destinationurl =~ m{^/enc/}) {
        my $plainurl = &Apache::lonenc::unencrypted($destinationurl);
        if ($plainurl =~ /(\?|\&)symb=/) {
            $gotsymb = 1;
        }
    }
    unless ($gotsymb) {
        $destsymb = &Apache::lonnet::symbread();
        if ($destsymb ne '') {
            $destsymb = &Apache::lonenc::check_encrypt($destsymb);
        }
    }
    my $reqprivs = &required_privs();
    if (ref($reqprivs) eq 'HASH') {
        my $destination = $destinationurl;
        $destination =~ s/(\?.*)$//;
        if (exists($reqprivs->{$destination})) {
            if ($reqprivs->{$destination} =~ /,/) {
                @{$privref} = split(/,/,$reqprivs->{$destination});
            } else {
                $privref = [$reqprivs->{$destination}];
            }
        }
    }
    if ($env{'user.role.'.$ccrole.'./'.$cdom.'/'.$cnum}) {
        my ($start,$end) = split(/\./,$env{'user.role.'.$ccrole.'./'.$cdom.'/'.$cnum});
        if ((($start) && ($start<0)) || 
            (($end) && ($end<$now))  ||
            (($start) && ($now<$start))) {
            $is_cc = 0;
        } else {
            $is_cc = 1;
        }
    }
    if ($is_cc) {
        &get_all_courseroles($cdom,$cnum,\%courseroles,\%seccount,\%courseprivs);
    } elsif ($env{'request.role'} =~ m{^\Qcr/$cdom/$cdom-domainconfig/\E(\w+)\.\Q/$cdom/$cnum\E}) {
        &get_customadhoc_roles($cdom,$cnum,\%courseroles,\%seccount,\%courseprivs,\%roledesc,$privref);
    } else {
        my %gotnosection;
        foreach my $item (keys(%env)) {
            if ($item =~ m-^user\.role\.([^.]+)\./\Q$cdom\E/\Q$cnum\E/?(\w*)$-) {
                my $role = $1;
                my $sec = $2;
                next if ($role eq 'gr');
                my ($start,$end) = split(/\./,$env{$item});
                next if (($start && $start > $now) || ($end && $end < $now));
                if ($sec eq '') {
                    if (!$gotnosection{$role}) {
                        $seccount{$role} ++;
                        $gotnosection{$role} = 1;
                    }
                }
                if ((ref($privref) eq 'ARRAY') && (@{$privref} > 0)) {
                    my $cnumsec = $cnum;
                    if ($sec ne '') {
                        $cnumsec .= "/$sec";
                    }
                    $courseprivs{"$role./$cdom/$cnumsec./"} =
                        $env{"user.priv.$role./$cdom/$cnumsec./"};
                    $courseprivs{"$role./$cdom/$cnumsec./$cdom/"} =
                        $env{"user.priv.$role./$cdom/$cnumsec./$cdom/"};
                    $courseprivs{"$role./$cdom/$cnumsec./$cdom/$cnumsec"} =
                        $env{"user.priv.$role./$cdom/$cnumsec./$cdom/$cnumsec"};
                }
                if (ref($courseroles{$role}) eq 'ARRAY') {
                    if ($sec ne '') {
                        if (!grep(/^\Q$sec\E$/,@{$courseroles{$role}})) {
                            push(@{$courseroles{$role}},$sec);
                            $seccount{$role} ++;
                        }
                    }
                } else {
                    @{$courseroles{$role}} = ();
                    if ($sec ne '') {
                        $seccount{$role} ++;
                        push(@{$courseroles{$role}},$sec);
                    }
                }
            }
        }
    }
    my @roles_order = ($ccrole,'in','ta','ep','ad','st');
    my $numdiffsec;
    if (keys(%seccount) == 1) {
        foreach my $key (keys(%seccount)) {
            $numdiffsec = $seccount{$key};
        }
    }
    if ((keys(%seccount) > 1) || ($numdiffsec > 1)) {
        my @submenu;
        $js = &jump_to_role($cdom,$cnum,\%seccount,\%courseroles,\%courseprivs,\%roledesc,$privref);
        $form = 
            '<form name="rolechooser" method="post" action="'.$httphost.'/adm/roles">'."\n".
            '  <input type="hidden" name="destinationurl" value="'.
            &HTML::Entities::encode($destinationurl).'" />'."\n".
            '  <input type="hidden" name="gotorole" value="1" />'."\n".
            '  <input type="hidden" name="selectrole" value="" />'."\n".
            '  <input type="hidden" name="switchrole" value="" />'."\n";
        if ($destsymb ne '') {
            $form .= '   <input type="hidden" name="destsymb" value="'.
                         &HTML::Entities::encode($destsymb).'" />'."\n";
        }
        $form .= '</form>'."\n";
        foreach my $role (@roles_order) {
            my $include;
            if (defined($courseroles{$role})) {
                if ($env{'request.role'} =~ m{^\Q$role\E}) {
                    if ($seccount{$role} > 1) {
                        $include = 1;
                    }
                } else {
                    $include = 1;
                }
            }
            if ($include) {
                push(@submenu,['javascript:adhocRole('."'$role'".')',
                               &Apache::lonnet::plaintext($role,$crstype)]);
            }
        }
        foreach my $role (sort(keys(%courseroles))) {
            if ($role =~ /^cr/) {
                my $include;
                if ($env{'request.role'} =~ m{^\Q$role\E}) {
                    if ($seccount{$role} > 1) {
                        $include = 1;
                    }
                } else {
                    $include = 1;
                }
                if ($include) {
                    my $rolename;
                    if ($role =~ m{^cr/$cdom/$cdom\-domainconfig/(\w+)(?:/\w+|$)}) {
                        $rolename = $roledesc{$role};
                        if ($rolename eq '') {
                            $rolename = &mt('Helpdesk [_1]',$1);
                        }
                    } else {
                        $rolename = &Apache::lonnet::plaintext($role);
                    }
                    push(@submenu,['javascript:adhocRole('."'$role'".')',
                                   $rolename]);
                }
            }
        }
        if (@submenu > 0) {
            $switcher = &create_submenu('','',&mt('Switch role'),\@submenu);
        }
    }
    return ($js,$form,$switcher);
}

sub get_all_courseroles {
    my ($cdom,$cnum,$courseroles,$seccount,$courseprivs) = @_;
    unless ((ref($courseroles) eq 'HASH') && (ref($seccount) eq 'HASH') &&
            (ref($courseprivs) eq 'HASH')) {
        return;
    }
    my ($result,$cached) = 
        &Apache::lonnet::is_cached_new('getcourseroles',$cdom.'_'.$cnum);
    if (defined($cached)) {
        if (ref($result) eq 'HASH') {
            if ((ref($result->{'roles'}) eq 'HASH') && 
                (ref($result->{'seccount'}) eq 'HASH') && 
                (ref($result->{'privs'}) eq 'HASH')) {
                %{$courseroles} = %{$result->{'roles'}};
                %{$seccount} = %{$result->{'seccount'}};
                %{$courseprivs} = %{$result->{'privs'}};
                return;
            }
        }
    }
    my %gotnosection;
    my %adv_roles =
         &Apache::lonnet::get_course_adv_roles($env{'request.course.id'},1);
    foreach my $role (keys(%adv_roles)) {
        my ($urole,$usec) = split(/:/,$role);
        if (!$gotnosection{$urole}) {
            $seccount->{$urole} ++;
            $gotnosection{$urole} = 1;
        }
        if (ref($courseroles->{$urole}) eq 'ARRAY') {
            if ($usec ne '') {
                if (!grep(/^Q$usec\E$/,@{$courseroles->{$urole}})) {
                    push(@{$courseroles->{$urole}},$usec);
                    $seccount->{$urole} ++;
                }
            }
        } else {
            @{$courseroles->{$urole}} = ();
            if ($usec ne '') {
                $seccount->{$urole} ++;
                push(@{$courseroles->{$urole}},$usec);
            }
        }
        my $area = '/'.$cdom.'/'.$cnum;
        if ($usec ne '') {
            $area .= '/'.$usec;
        }
        if ($role =~ /^cr\//) {
            &Apache::lonnet::custom_roleprivs($courseprivs,$urole,$cdom,$cnum,$urole.'.'.$area,$area);
        } else {
            &Apache::lonnet::standard_roleprivs($courseprivs,$urole,$cdom,$urole.'.'.$area,$cnum,$area);
        }
    }
    my %sections_count = &Apache::loncommon::get_sections($cdom,$cnum,['st']);
    @{$courseroles->{'st'}} = ();
    &Apache::lonnet::standard_roleprivs($courseprivs,'st',$cdom,"st./$cdom/$cnum",$cnum,"/$cdom/$cnum");
    if (keys(%sections_count) > 0) {
        push(@{$courseroles->{'st'}},keys(%sections_count));
        $seccount->{'st'} = scalar(keys(%sections_count));
    }
    $seccount->{'st'} ++; # Increment for a section-less student role.
    my $rolehash = {
                     'roles'    => $courseroles,
                     'seccount' => $seccount,
                     'privs'    => $courseprivs,
                   };
    &Apache::lonnet::do_cache_new('getcourseroles',$cdom.'_'.$cnum,$rolehash);
    return;
}

sub get_customadhoc_roles {
    my ($cdom,$cnum,$courseroles,$seccount,$courseprivs,$roledesc,$privref) = @_;
    unless ((ref($courseroles) eq 'HASH') && (ref($seccount) eq 'HASH') &&
            (ref($courseprivs) eq 'HASH') && (ref($roledesc) eq 'HASH')) {
        return;
    }
    my $is_helpdesk = 0;
    my $now = time;
    foreach my $role ('dh','da') {
        if ($env{"user.role.$role./$cdom/"}) {
            my ($start,$end)=split(/\./,$env{"user.role.$role./$cdom/"});
            if (!($start && ($now<$start)) && !($end && ($now>$end))) {
                $is_helpdesk = 1;
                last;
            }
        }
    }
    if ($is_helpdesk) {
        my ($possroles,$description) = &Apache::lonnet::get_my_adhocroles($cdom.'_'.$cnum);
        my %available;
        if (ref($possroles) eq 'ARRAY') {
            map { $available{$_} = 1; } @{$possroles};
        }
        my %domdefaults = &Apache::lonnet::get_domain_defaults($cdom);
        if (ref($domdefaults{'adhocroles'}) eq 'HASH') {
            if (keys(%{$domdefaults{'adhocroles'}})) {
                my $numsec = 1;
                my @sections;
                my ($allseclist,$cached) =
                    &Apache::lonnet::is_cached_new('courseseclist',$cdom.'_'.$cnum);
                if (defined($cached)) {
                    if ($allseclist ne '') {
                        @sections = split(/,/,$allseclist);
                        $numsec += scalar(@sections);
                    }
                } else {
                    my %sections_count = &Apache::loncommon::get_sections($cdom,$cnum);
                    @sections = sort(keys(%sections_count));
                    $numsec += scalar(@sections);
                    $allseclist = join(',',@sections);
                    &Apache::lonnet::do_cache_new('courseseclist',$cdom.'_'.$cnum,$allseclist);
                }
                my (%adhoc,$gotprivs);
                my $prefix = "cr/$cdom/$cdom".'-domainconfig';
                foreach my $role (keys(%{$domdefaults{'adhocroles'}})) {
                    next if (($role eq '') || ($role =~ /\W/));
                    $seccount->{"$prefix/$role"} = $numsec;
                    $roledesc->{"$prefix/$role"} = $description->{$role};  
                    if ((ref($privref) eq 'ARRAY') && (@{$privref} > 0)) {
                        if (exists($env{"user.priv.$prefix/$role./$cdom/$cnum./"})) {
                            $courseprivs->{"$prefix/$role./$cdom/$cnum./"} =
                                $env{"user.priv.$prefix/$role./$cdom/$cnum./"};
                            $courseprivs->{"$prefix/$role./$cdom/$cnum./$cdom/"} =
                                $env{"user.priv.$prefix/$role./$cdom/$cnum./$cdom/"};
                            $courseprivs->{"$prefix/$role./$cdom/$cnum./$cdom/$cnum"} =
                                $env{"user.priv.$prefix/$role./$cdom/$cnum./$cdom/$cnum"};
                        } else {
                            unless ($gotprivs) {
                                my ($adhocroles,$privscached) =
                                    &Apache::lonnet::is_cached_new('adhocroles',$cdom);
                                if ((defined($privscached)) && (ref($adhocroles) eq 'HASH')) {
                                    %adhoc = %{$adhocroles};
                                } else {
                                    my $confname = &Apache::lonnet::get_domainconfiguser($cdom);
                                    my %roledefs = &Apache::lonnet::dump('roles',$cdom,$confname,'rolesdef_');
                                    foreach my $key (keys(%roledefs)) {
                                        (undef,my $rolename) = split(/_/,$key);
                                        if ($rolename ne '') {
                                            my ($systempriv,$domainpriv,$coursepriv) = split(/\_/,$roledefs{$key});
                                            $coursepriv = &Apache::lonnet::course_adhocrole_privs($rolename,$cdom,$cnum,$coursepriv);
                                            $adhoc{$rolename} = join('_',($systempriv,$domainpriv,$coursepriv));
                                        }
                                    }
                                    &Apache::lonnet::do_cache_new('adhocroles',$cdom,\%adhoc);
                                }
                                $gotprivs = 1;
                            }
                            ($courseprivs->{"$prefix/$role./$cdom/$cnum./"},
                             $courseprivs->{"$prefix/$role./$cdom/$cnum./$cdom/"},
                             $courseprivs->{"$prefix/$role./$cdom/$cnum./$cdom/$cnum"}) =
                                 split(/\_/,$adhoc{$role});
                        }
                    }
                    if ($available{$role}) {
                        $courseroles->{"$prefix/$role"} = \@sections;
                    }
                }
            }
        }
    }
    return;
}

sub jump_to_role {
    my ($cdom,$cnum,$seccount,$courseroles,$courseprivs,$roledesc,$privref) = @_;
    my %lt = &Apache::lonlocal::texthash(
                this => 'This role has section(s) associated with it.',
                ente => 'Enter a specific section.',
                orlb => 'Enter a specific section, or leave blank for no section.',
                avai => 'Available sections are:',
                youe => 'You entered an invalid section choice:',
                plst => 'Please try again.',
                role => 'The role you selected is not permitted to view the current page.',
                swit => 'Switch role, but display Main Menu page instead?',
    );
    my $js;
    if (ref($courseroles) eq 'HASH') {
        $js = '    var secpick = new Array("'.$lt{'ente'}.'","'.$lt{'orlb'}.'");'."\n". 
              '    var numsec = new Array();'."\n".
              '    var rolesections = new Array();'."\n".
              '    var rolenames = new Array();'."\n".
              '    var roleseclist = new Array();'."\n";
        my @items = keys(%{$courseroles});
        for (my $i=0; $i<@items; $i++) {
            $js .= '    rolenames['.$i.'] = "'.$items[$i].'";'."\n";
            my ($secs,$secstr);
            if (ref($courseroles->{$items[$i]}) eq 'ARRAY') {
                my @sections = sort { $a <=> $b } @{$courseroles->{$items[$i]}};
                $secs = join('","',@sections);
                $secstr = join(', ',@sections);
            }
            $js .= '    rolesections['.$i.'] = new Array("'.$secs.'");'."\n".
                   '    roleseclist['.$i.'] = "'.$secstr.'";'."\n".
                   '    numsec['.$i.'] = "'.$seccount->{$items[$i]}.'";'."\n";
        }
    }
    my $checkroles = 0;
    if ((ref($privref) eq 'ARRAY') && (@{$privref} > 0) && (ref($courseprivs) eq 'HASH')) {
        my %disallowed;
        foreach my $role (sort(keys(%{$courseprivs}))) {
            my $trole;
            if ($role =~ m{^(.+?)\Q./$cdom/$cnum\E}) {
                $trole = $1;
            }
            if (($trole ne '') && ($trole ne 'cm')) {
                $disallowed{$trole} = 1;
                foreach my $priv (@{$privref}) { 
                    if ($courseprivs->{$role} =~ /\Q:$priv\E($|:|\&\w+)/) {
                        delete($disallowed{$trole});
                        last;
                    }
                }
            }
        }
        if (keys(%disallowed) > 0) {
            $checkroles = 1;
            $js .= "    var disallow = new Array('".join("','",keys(%disallowed))."');\n".
                   "    var rolecheck = 1;\n";
        }
    }
    if (!$checkroles) {
        $js .=  "    var disallow = new Array();\n".
                "    rolecheck = 0;\n";
    }
    return <<"END";
<script type="text/javascript">
//<![CDATA[
function adhocRole(newrole) {
    $js
    if (newrole == '') {
        return;
    } 
    var fullrole = newrole+'./$cdom/$cnum';
    var selidx = '';
    for (var i=0; i<rolenames.length; i++) {
        if (rolenames[i] == newrole) {
            selidx = i;
        }
    }
    if (rolecheck > 0) {
        for (var i=0; i<disallow.length; i++) {
            if (disallow[i] == newrole) {
                if (confirm("$lt{'role'}\\n$lt{'swit'}")) {
                    document.rolechooser.destinationurl.value = '/adm/menu';
                } else {
                    return;
                }
            }
        }
    }
    var secok = 1;
    var secchoice = '';
    if (selidx >= 0) {
        if (numsec[selidx] > 1) {
            secok = 0;
            var numrolesec = rolesections[selidx].length;
            var msgidx = numsec[selidx] - numrolesec;
            secchoice = prompt("$lt{'this'} "+secpick[msgidx]+"\\n$lt{'avai'} "+roleseclist[selidx],"");
            if (secchoice == '') {
                if (msgidx > 0) {
                    secok = 1;
                }
            } else {
                for (var j=0; j<rolesections[selidx].length; j++) {
                    if (rolesections[selidx][j] == secchoice) {
                        secok = 1;
                    }
                }
            }
        } else {
            if (rolesections[selidx].length == 1) {
                secchoice = rolesections[selidx][0];
            }
        }
    }
    if (secok == 1) {
        if (secchoice != '') {
            fullrole += '/'+secchoice;
        }
    } else {
        document.rolechooser.elements[roleitem].selectedIndex = 0;
        if (secchoice != null) {
            alert("$lt{'youe'} \\""+secchoice+"\\".\\n $lt{'plst'}");
        }
        return;
    }
    if (fullrole == "$env{'request.role'}") {
        return;
    }
    itemid = retrieveIndex('gotorole');
    if (itemid != -1) {
        document.rolechooser.elements[itemid].name = fullrole;
    }
    document.rolechooser.switchrole.value = fullrole;
    document.rolechooser.selectrole.value = '1';
    document.rolechooser.submit();
    return;
}

function retrieveIndex(item) {
    for (var i=0;i<document.rolechooser.elements.length;i++) {
        if (document.rolechooser.elements[i].name == item) {
            return i;
        }
    }
    return -1;
}
// ]]>
</script>
END
}

sub required_privs {
    my $privs =  {
             '/adm/parmset'      => 'opa,vpa',
             '/adm/courseprefs'  => 'opa,vpa',
             '/adm/whatsnew'     => 'whn',
             '/adm/populate'     => 'cst,vpa,vcl',
             '/adm/trackstudent' => 'vsa',
             '/adm/statistics'   => 'mgr,vgr',
             '/adm/setblock'     => 'dcm,vcb',
             '/adm/coursedocs'   => 'mdc',
           };
    unless ($env{'course.'.$env{'request.course.id'}.'.grading'} eq 'spreadsheet') {
        $privs->{'/adm/classcalc'}   = 'vgr',
        $privs->{'/adm/assesscalc'}  = 'vgr',
        $privs->{'/adm/studentcalc'} = 'vgr';
    }
    return $privs;
}

sub countdown_timer {
    if (($env{'request.course.id'}) && ($env{'request.symb'} ne '') &&
        ($env{'request.filename'}=~/$LONCAPA::assess_re/)) {
        my ($type,$hastimeleft,$slothastime);
        my $now = time;
        if ($env{'request.filename'} =~ /\.task$/) {
            $type = 'Task';
        } else {
            $type = 'problem';
        }
        my ($status,$accessmsg,$slot_name,$slot) =
            &Apache::lonhomework::check_slot_access('0',$type);
        if ($slot_name ne '') {
            if (ref($slot) eq 'HASH') {
                if (($slot->{'starttime'} < $now) &&
                    ($slot->{'endtime'} > $now)) {
                    $slothastime = 1;
                }
            }
        }
        if ($status ne 'CAN_ANSWER') {
            return;
        }
        my $duedate = &Apache::lonnet::EXT("resource.0.duedate");
        my @interval=&Apache::lonnet::EXT("resource.0.interval");
        if (@interval > 1) {
            my $first_access=&Apache::lonnet::get_first_access($interval[1]);
            if ($first_access > 0) {
                if ($first_access+$interval[0] > time) {
                    $hastimeleft = 1;
                }
            }
        }
        if (($duedate && $duedate > time) ||
            (!$duedate && $hastimeleft) ||
            ($slot_name ne '' && $slothastime)) {
            my ($collapse,$expand,$alttxt,$title,$currdisp);
            if ((@interval > 1 && $hastimeleft) ||
                ($type eq 'Task' && $slothastime)) {
                $currdisp = 'inline';
                $collapse = '&#9658;&nbsp;';
            } else {
                $currdisp = 'none';
                $expand = '&#9668;&nbsp;';
            }
            unless ($env{'environment.icons'} eq 'iconsonly') {
                $alttxt = &mt('Timer');
                $title = $alttxt.'&nbsp;';
            }
            my $desc = &mt('Countdown to due date/time');
            return <<END;

<a href="javascript:toggleCountdown();" class="LC_menubuttons_link">
<span id="ddcountcollapse" class="LC_menubuttons_inline_text">
$collapse
</span></a>
<span id="duedatecountdown" class="LC_menubuttons_inline_text" style="display: $currdisp;"></span>
<a href="javascript:toggleCountdown();" class="LC_menubuttons_link">
<span id="ddcountexpand" class="LC_menubuttons_inline_text" >$expand</span>
<img src="/res/adm/pages/timer.png" title="$desc" class="LC_icon" alt="$alttxt" /><span class="LC_menubuttons_inline_text">$title</span></a>
END
        }
    }
    return;
}

# ================================================================ Main Program

BEGIN {
    if (! defined($readdesk)) {
        {
            my $tabfile = $Apache::lonnet::perlvar{'lonTabDir'}.'/mydesk.tab';
            if ( CORE::open( my $config,"<$tabfile") ) {
                while (my $configline=<$config>) {
                    $configline=(split(/\#/,$configline))[0];
                    $configline=~s/^\s+//;
                    chomp($configline);
                    if ($configline=~/^cat\:/) {
                        my @entries=split(/\:/,$configline);
                        $category_positions{$entries[2]}=$entries[1];
                        $category_names{$entries[2]}=$entries[3];
                    } elsif ($configline=~/^prim\:/) {
                        my @entries = (split(/\:/, $configline))[1..6];
                        push(@primary_menu,\@entries);
                    } elsif ($configline=~/^primsub\:/) {
                        my ($parent,@entries) = (split(/\:/, $configline))[1..4];
                        push(@{$primary_submenu{$parent}},\@entries);
                    } elsif ($configline=~/^scnd\:/) {
                        my @entries = (split(/\:/, $configline))[1..5];
                        push(@secondary_menu,\@entries);
                    } elsif ($configline=~/^scndsub\:/) {
                        my ($parent,@entries) = (split(/\:/, $configline))[1..4];
                        push(@{$secondary_submenu{$parent}},\@entries);
                    } elsif ($configline) {
                        push(@desklines,$configline);
                    }
                }
                CORE::close($config);
            }
        }
        $readdesk='done';
    }
}

1;
__END__

