# The LearningOnline Network
# Allow access to password changing via a token sent to user's e-mail. 
#
# $Id: resetpw.pm,v 1.51 2021/11/30 16:08:44 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
#

=pod

=head1 NAME

Apache::resetpw: reset user password.

=head1 SYNOPSIS

Handles resetting of forgotten passwords.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.
 
=head1 OVERVIEW

A user with an e-mail address associated with his/her LON-CAPA username
can reset a forgotten password, using a link sent to the e-mail address
if the authentication type for the account is "internal".

=cut

package Apache::resetpw;

use strict;
use Apache::Constants qw(:common);
use Apache::lonacc;
use Apache::lonnet;
use Apache::loncommon;
use Apache::lonpreferences;
use Apache::lonlocal;
use LONCAPA;
use HTML::Entities;

sub handler {
    my $r = shift;
    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;
    if ($r->header_only) {
        return OK;
    }
    my $contact_name = &mt('LON-CAPA helpdesk');
    my $origmail =  $r->dir_config('lonSupportEMail');
    my $server = $r->dir_config('lonHostID');
    my $defdom = &Apache::lonnet::default_login_domain();
    my $contacts =
        &Apache::loncommon::build_recipient_list(undef,'helpdeskmail',
                                                 $defdom,$origmail);
    my ($contact_email) = split(',',$contacts);
    my $handle = &Apache::lonnet::check_for_valid_session($r);
    my $lonidsdir=$r->dir_config('lonIDsDir');
    if ($handle ne '') {
        if ($handle=~/^publicuser\_/) {
            unlink($r->dir_config('lonIDsDir')."/$handle.id");
        } else {
            &Apache::lonnet::transfer_profile_to_env($lonidsdir,$handle);
        }
    }
    &Apache::lonacc::get_posted_cgi($r);
    &Apache::lonlocal::get_language_handle($r);
    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},['token']);
    
    my @emailtypes = ('permanentemail','critnotification','notification');
    my $uname = $env{'form.uname'};
    $uname =~ s/^\s+|\s+$//g;
    $uname = &LONCAPA::clean_username($uname);
    my $udom = &LONCAPA::clean_domain($env{'form.udom'});
    my ($domdesc,$otherinst,$lookup);
    if ($udom) {
        $domdesc = &Apache::lonnet::domain($udom,'description');
        if ($domdesc) {
            $otherinst = 1;
            my @ids=&Apache::lonnet::current_machine_ids();
            my %servers = &Apache::lonnet::internet_dom_servers($udom);
            foreach my $hostid (keys(%servers)) {
                if (grep(/^\Q$hostid\E$/,@ids)) {
                    $otherinst = 0;
                    last;
                }
            }
        }
    }
    my $dom_in_effect = $defdom;
    if (($udom ne '') && ($domdesc ne '')) {
        unless ($otherinst) {
            $dom_in_effect = $udom;
        }
    }
    my %passwdconf = &Apache::lonnet::get_passwdconf($dom_in_effect);
    my $clientip = &Apache::lonnet::get_requestor_ip($r);
    my $token = $env{'form.token'};
    my $useremail = $env{'form.useremail'};
    if (($udom ne '') && (!$otherinst) && (!$token)) {
        if ($uname ne '') {
            my $uhome = &Apache::lonnet::homeserver($uname,$udom);
            if ($uhome eq 'no_host') {
                my %srch = (srchby     => 'uname_ci',
                            srchdomain => $udom,
                            srchterm   => $uname,
                            srchtype   => 'exact');
                my %srch_results = &Apache::lonnet::usersearch(\%srch);
                if (keys(%srch_results) > 1) {
                    $lookup = 'nonunique';
                    if ($useremail =~ /^[^\@]+\@[^\@]+\.[^\@\.]+$/) {
                        foreach my $key (keys(%srch_results)) {
                            if (ref($srch_results{$key}) eq 'HASH') {
                                if ($srch_results{$key}{permanentemail} =~ /^\Q$useremail\E$/i) {
                                    ($uname) = split(/:/,$key);
                                    undef($lookup);
                                    last;
                                }
                            }
                        }
                    }
                } elsif (keys(%srch_results) == 1) {
                    my $match = (keys(%srch_results))[0];
                    ($uname) = split(/:/,$match);
                } else {
                    $lookup = 'nomatch';
                }
            }
        }
        if (($lookup eq 'nomatch') || ($uname eq '')) {
            if (($useremail =~ /^[^\@]+\@[^\@]+\.[^\@\.]+$/) &&
                ($passwdconf{'resetprelink'} eq 'either')) {
                my %srch = (srchby     => 'email',
                            srchdomain => $udom,
                            srchterm   => $useremail,
                            srchtype   => 'exact');
                my %srch_results = &Apache::lonnet::usersearch(\%srch);
                if (keys(%srch_results) > 1) {
                    $lookup = 'nonunique';
                } elsif (keys(%srch_results) == 1) {
                    my $match = (keys(%srch_results))[0];
                    ($uname) = split(/:/,$match);
                    undef($lookup);
                } else {
                    $lookup = 'nomatch';
                }
            }
        }
    }
    my $brcrum = [];
    if ($token) {
        push (@{$brcrum},
            {href => '/adm/resetpw',
             text => 'Update Password'});
    } else {
        push (@{$brcrum},
            {href => '/adm/resetpw',
             text => 'Account Information'});
        if ($uname && $udom) {
            push (@{$brcrum},
                {href => '/adm/resetpw',
                 text => 'Result'});
        }
    }
    my $args = {bread_crumbs => $brcrum};
    my $js;
    unless ($token || $otherinst || ($uname && $udom)) {
        my (@intdoms,@instdoms);
        my $internet_names = &Apache::lonnet::get_internet_names($server);
        if (ref($internet_names) eq 'ARRAY') {
            @intdoms = @{$internet_names};
        }
        if (@intdoms) {
            my %iphost = &Apache::lonnet::get_iphost();
            foreach my $ip (keys(%iphost)) {
                if (ref($iphost{$ip}) eq 'ARRAY') {
                    foreach my $id (@{$iphost{$ip}}) {
                        my $location = &Apache::lonnet::internet_dom($id);
                        if ($location) {
                            if (grep(/^\Q$location\E$/,@intdoms)) {
                                my $dom = &Apache::lonnet::host_domain($id);
                                unless (grep(/^\Q$dom\E/,@instdoms)) {
                                    push(@instdoms,$dom);
                                }
                            }
                        }
                    }
                }
            }
        }
        my $instdomstr;
        if (@instdoms > 0) {
            $instdomstr = "'".join("','",@instdoms)."'";
        }
        my %js_lt = &Apache::lonlocal::texthash(
            thdo  => 'The domain you have selected is for another institution.',
            yowi  => 'You will be switched to the Forgot Password utility at that institution.',
            unam  => 'You must enter a username.',
            mail  => 'You must enter an e-mail address.',
            eith  => 'Enter a username and/or an e-mail address.',
        );
        &js_escape(\%js_lt);
        $js = <<"END";
function verifyDomain(caller,form) {
    var redirect = 1; 
    var instdoms = new Array($instdomstr);
    if (instdoms.length > 0) {
        for (var i=0; i<instdoms.length; i++) {
            if (caller.options[caller.selectedIndex].value == instdoms[i]) {
                redirect = 0;
                break;
            }
        }
    }
    if (redirect == 0) {
        if (caller.options[caller.selectedIndex].value != '$dom_in_effect') {
            document.forgotpw.uname.value = '';
            document.forgotpw.useremail.value = '';
            form.submit();
        }
    }
    if (redirect == 1) {
        if (confirm('$js_lt{thdo}\\n$js_lt{yowi}')) {
            form.submit();
        } else {
            for (var i=0; i<caller.options.length; i++) { 
                if (caller.option[i].value == '$dom_in_effect') {
                    caller.selectedIndex = i;
                    break;
                }
            }
        }
    }
    return;
}

END
        if ($passwdconf{resetprelink} eq 'either') {
            $js .= <<"END";
function validInfo() {
    if ((document.forgotpw.uname.value == '') &&
        (document.forgotpw.useremail.value == '')) {
        alert("$js_lt{'eith'}");
        return false;
    }
    return true;
}
END
        } else {
            $js .= <<"END";

function validInfo() {
    if (document.forgotpw.uname.value == '') {
        alert("$js_lt{'unam'}");
        return false;
    }
    if (document.forgotpw.useremail.value == '') {
        alert("$js_lt{'mail'}");
        return false;
    }
    return true;
}
END
        }
    }
    $js = &Apache::lonhtmlcommon::scripttag($js);
    if (($passwdconf{'captcha'} eq 'recaptcha') && ($passwdconf{'recaptchaversion'} >=2)) {
        $js.= "\n".'<script src="https://www.google.com/recaptcha/api.js"></script>'."\n";
    }
    my $header = &Apache::loncommon::start_page('Reset password',$js,$args).
                 '<h2>'.&mt('Reset forgotten LON-CAPA password').'</h2>';
    my $output;
    if ($token) {
        $r->print($header);
        &reset_passwd($r,$token,$contact_name,$contact_email,$clientip,\%passwdconf);
        $r->print(&Apache::loncommon::end_page());
        return OK;
    } elsif ($udom) {
        if (!$domdesc) {
            $output = &invalid_state('baddomain',$domdesc,
                                     $contact_name,$contact_email); 
        } elsif ($otherinst) {
            ($header,$output) = &homeserver_redirect($r,$uname,$udom,$domdesc,$brcrum);
        } elsif (($uname) || ($useremail)) {
            my $earlyout;
            unless ($passwdconf{'captcha'} eq 'unused') {
                my ($captcha_chk,$captcha_error) =
                    &Apache::loncommon::captcha_response('passwords',$server,$dom_in_effect);
                if ($captcha_chk != 1) {
                    my $error = 'captcha'; 
                    if ($passwdconf{'captcha'} eq 'recaptcha') {
                        $error = 'recaptcha';
                    }
                    $output = &invalid_state($error,$domdesc,
                                             $contact_name,$contact_email);
                    $earlyout = 1;
                }
            }
            unless ($earlyout) {
                if ($lookup) {
                    $output = &invalid_state($lookup,$domdesc,
                                             $contact_name,$contact_email);
                    $earlyout = 1;
                }
            }
            unless ($earlyout) {
                my $authtype = &Apache::lonnet::queryauthenticate($uname,$udom);
                if ($authtype =~ /^internal/) {
                    my ($blocked,$blocktext) =
                        &Apache::loncommon::blocking_status('passwd',$clientip,$uname,$udom);
                    if ($blocked) {
                        $output = '<p class="LC_warning">'.$blocktext.'</p>'
                                  .&display_actions($contact_email,$domdesc);
                    } elsif (($passwdconf{'resetprelink'} ne 'either') && 
                             ($useremail !~ /^[^\@]+\@[^\@]+\.[^\@\.]+$/)) {
                        $output = &invalid_state('baduseremail',$domdesc,
                                                 $contact_name,$contact_email);
                    } else {
                        my %userinfo = 
	                    &Apache::lonnet::get('environment',\@emailtypes,
					         $udom,$uname);
                        my @allemails;
                        foreach my $type (@emailtypes) {
                            if (ref($passwdconf{resetemail}) eq 'ARRAY') {
                                if ($type eq 'permanentemail') {
                                    next unless (grep(/^permanent$/,@{$passwdconf{resetemail}}));
                                } elsif ($type eq 'critnotification') {
                                    next unless (grep(/^critical$/,@{$passwdconf{resetemail}}));
                                } elsif ($type eq 'notification') {
                                    next unless (grep(/^notify$/,@{$passwdconf{resetemail}}));
                                }
                            }
                            my $email = $userinfo{$type};
                            my @items;
                            if ($email =~ /,/) {
                                @items = split(',',$userinfo{$type});
                            } else {
                                @items = ($email);
                            }
                            foreach my $item (@items) {
                                if ($item =~ /^[^\@]+\@[^\@]+\.[^\@\.]+$/) {
                                    unless (grep(/^\Q$item\E$/i,@allemails)) {
                                        push(@allemails,$item);
                                    }
                                }
                            }
                        }
                        if (@allemails > 0) {
                            my ($sendto,$warning,$timelimit);
                            my $timelimit = 2;
                            if ($passwdconf{'resetlink'} =~ /^\d+(|\.\d*)$/) {
                                $timelimit = $passwdconf{'resetlink'};
                            }
                            if ($passwdconf{'resetprelink'} eq 'either') {
                                if ($useremail ne '') {
                                    if (grep(/^\Q$useremail\E$/i,@allemails)) {
                                        $sendto = $useremail;
                                    } else {
                                        $warning = &mt('The e-mail address you entered did not match the expected e-mail address.');
                                    }
                                } elsif (@allemails > 1) {
                                    $warning = &mt('More than one e-mail address is associated with your username, and one has been selected to receive the message sent by LON-CAPA.');
                                }
                                unless ($sendto) {
                                    $sendto = $allemails[0];
                                }
                            } else {
                                if (grep(/^\Q$useremail\E$/i,@allemails)) {
                                    $sendto = $useremail;
                                } else {
                                    $output = &invalid_state('mismatch',$domdesc,
                                                             $contact_name,
                                                             $contact_email);
                                }
                            }
                            if ($sendto) {
                                $output = &send_token($uname,$udom,$sendto,$server,
                                                      $domdesc,$contact_name,
                                                      $contact_email,$timelimit,$warning);
                            }
                        } else {
                            $output = &invalid_state('missing',$domdesc,
                                                     $contact_name,$contact_email);
                        }
                    }
                } elsif ($authtype =~ /^(krb|unix|local)/) {
                    $output = &invalid_state('authentication',$domdesc,
                                         $contact_name,$contact_email);
                } else {
                    $output = &invalid_state('invalid',$domdesc,
                                             $contact_name,$contact_email);
                }
            }
        } else {
            $output = &get_uname($server,$dom_in_effect,\%passwdconf);
        }
    } else {
        $output = &get_uname($server,$defdom,\%passwdconf);
    }
    $r->print($header.$output);
    $r->print(&Apache::loncommon::end_page());
    return OK;
}

sub get_uname {
    my ($server,$defdom,$passwdconf) = @_;
    return unless (ref($passwdconf) eq 'HASH');
    my %lt = &Apache::lonlocal::texthash(
            unam => 'LON-CAPA username',
            udom => 'LON-CAPA domain',
            uemail => 'E-mail address in LON-CAPA',
            vali   => 'Validation',
            proc => 'Proceed');
    my $msg;
    unless ($passwdconf->{'resetremove'}) {
        $msg = '<p>'.&mt('If you use the same account for other campus services besides LON-CAPA, (e.g., e-mail, course registration, etc.), a separate centrally managed mechanism likely exists to reset a password. However, if your account is used for just LON-CAPA access you will probably be able to reset a password from this page.').'</p>';
    }
    if ($passwdconf->{'resetcustom'} eq "/res/$defdom/$defdom-domainconfig/customtext/resetpw/resetpw.html") {
        my $contents = &Apache::lonnet::getfile(&Apache::lonnet::filelocation('',$passwdconf->{'resetcustom'}));
        unless ($contents eq '-1') {
            $msg .= $contents;
        }
    }
    $msg .= '<p>'.&mt('Three conditions must be met:')
           .'<ul><li>'.&mt('An e-mail address must have previously been associated with your LON-CAPA username.').'</li>'
           .'<li>'.&mt('You must be able to access e-mail sent to that address.').'</li>'
           .'<li>'.&mt('Your LON-CAPA account must be of a type for which LON-CAPA can reset a password.')
           .'</ul></p>';
    my $onchange = 'javascript:verifyDomain(this,this.form);';
    $msg .= '<form name="forgotpw" method="post" action="/adm/resetpw" onsubmit="return validInfo();">'.
            &Apache::lonhtmlcommon::start_pick_box().
            &Apache::lonhtmlcommon::row_title($lt{'udom'}).
            &Apache::loncommon::select_dom_form($defdom,'udom',undef,undef,$onchange).
            &Apache::lonhtmlcommon::row_closure(1).
            &Apache::lonhtmlcommon::row_title($lt{'unam'}).
            '<input type="text" name="uname" size="20" autocapitalize="off" autocorrect="off" />'.
            &Apache::lonhtmlcommon::row_closure(1).
            &Apache::lonhtmlcommon::row_title($lt{'uemail'}).
            '<input type="text" name="useremail" size="30" autocapitalize="off" autocorrect="off" />'.
            &Apache::lonhtmlcommon::row_closure(1);
    unless ($passwdconf->{'captcha'} eq 'notused') {
        my ($captcha_form,$captcha_error,$captcha,$recaptcha_version) =
            &Apache::loncommon::captcha_display('passwords',$server,$defdom);
        if ($captcha_form) {
            $msg .= &Apache::lonhtmlcommon::row_title($lt{'vali'}).
                    $captcha_form."\n".
                    &Apache::lonhtmlcommon::row_closure(1);
        }
    }
    $msg .= &Apache::lonhtmlcommon::end_pick_box().
            '<br /><br /><input type="submit" name="resetter" value="'.$lt{'proc'}.'" /></form>';
    return $msg;
}

sub send_token {
    my ($uname,$udom,$email,$server,$domdesc,$contact_name,
        $contact_email,$timelimit,$warning) = @_;
    my $msg =
        '<p class="LC_info">'
       .&mt('Thank you for your request to reset the password for your LON-CAPA account.')
       .'</p>';

    my $now = time;
    my $temppasswd = &create_passwd();
    my $ip = &Apache::lonnet::get_requestor_ip(); 
    my %info = ('ip'         => $ip,
		'time'       => $now,
		'domain'     => $udom,
		'username'   => $uname,
		'email'      => $email,
		'temppasswd' => $temppasswd);

    my $token = &Apache::lonnet::tmpput(\%info,$server,'resetpw');
    if ($token !~ /^error/ && $token ne 'no_such_host') {
        my $esc_token = &escape($token);
        my $showtime = &Apache::lonlocal::locallocaltime(time);
        my $reseturl = &Apache::lonnet::absolute_url().'/adm/resetpw?token='.$esc_token;
        my $mailmsg = &mt('A request was submitted on [_1] for reset of the password for your LON-CAPA account.',$showtime)." \n".&mt('To complete this process please open a web browser and enter the following URL in the address/location box: [_1]',"\n\n".$reseturl);
        my $result = &send_mail($domdesc,$email,$mailmsg,$contact_name,
                                $contact_email);
        if ($result eq 'ok') {
            $msg .=
                &mt('An e-mail sent to the e-mail address associated with your LON-CAPA account includes the web address for the link you should use to complete the reset process.')
               .'<br /><br />'
               .&mt('The link included in the message will be valid for the next [_1][quant,_2,hour][_3].','<b>',$timelimit,'</b>');
        } else {
            $msg .=
                '<p class="LC_error">'
               .&mt('An error occurred when sending a message to the e-mail address'
                   .' associated with your LON-CAPA account.')
               .'</p>'
               .&display_actions($contact_email,$domdesc);
        }
    } else {
        $msg .=
            '<p class="LC_error">'
           .&mt('An error occurred creating a token required for the'
               .' password reset process.')
           .'</p>'
           .&display_actions($contact_email,$domdesc);
    }
    return $msg;
}

sub send_mail {
    my ($domdesc,$email,$mailmsg,$contact_name,$contact_email) = @_;
    my $outcome;
    my $requestmail = "To: $email\n".
                      "From: $contact_name <$contact_email>\n".
                      "Subject: ".&mt('Your LON-CAPA account')."\n".
                      "Content-type: text/plain\;charset=UTF-8\n".
                      "\n\n".$mailmsg."\n\n".
                      &mt('[_1] LON-CAPA support team',$domdesc)."\n".
                      "$contact_email\n";
    if (open(MAIL, "|/usr/lib/sendmail -oi -t -odb")) {
        print MAIL $requestmail;
        close(MAIL);
        $outcome = 'ok';
    } else {
        $outcome = 'fail';
    }
    return $outcome;
}

sub invalid_state {
    my ($error,$domdesc,$contact_name,$contact_email) = @_;
    my $msg;
    if ($error eq 'invalid') {
        $msg =
            '<p class="LC_warning">'
            .&mt('The username you provided was not verified as a valid username'
                .' in the LON-CAPA system for the [_1] domain.','<i>'.$domdesc.'</i>')
              .'</p>';
        $msg .= &display_actions($contact_email,$domdesc);
    } else {
        if ($error eq 'captcha') {
            $msg = &mt('Validation of the code you entered failed');
        } elsif ($error eq 'recaptcha') {
            $msg = &mt('Validation of human, not robot, failed'); 
        } elsif ($error eq 'nonunique') {
            $msg = &mt('More than one username was identified from the information you provided; try providing both a username and e-mail address');
        } elsif ($error eq 'nomatch') {
            $msg = &mt('A valid user could not be identified from the username and/or e-mail address you provided');
        } elsif ($error eq 'baduseremail') {
            $msg = &mt('The e-mail address you provided does not appear to be a valid address.');
        } elsif ($error eq 'mismatch') {
            $msg = &mt('The e-mail address you provided does not match the address recorded in the LON-CAPA system for the username and domain you provided.');  
        } elsif ($error eq 'missing') {
            $msg = &mt('A valid e-mail address was not located in the LON-CAPA system for the username and domain you provided.');
        } elsif ($error eq 'authentication') {
            $msg = &mt('The username you provided uses an authentication type which can not be reset directly via LON-CAPA.');
        } elsif ($error eq 'baddomain') {
            $msg = &mt('The domain you provided was not verified as a valid domain in the LON-CAPA system.');
        }
        $msg = '<p class="LC_warning">'.$msg.'</p>'
              .&display_actions($contact_email,$domdesc);
    }
    return $msg;
}

sub homeserver_redirect {
    my ($r,$uname,$udom,$domdesc,$brcrum) = @_;
    my $uhome;
    if (($uname ne '') && ($udom ne '')) {
        $uhome = &Apache::lonnet::homeserver($uname,$udom);
    }
    if (($uhome eq 'no_host') || ($uhome eq '')) {
        $uhome = &Apache::lonnet::domain($udom,'primary');
    }
    my $hostname = &Apache::lonnet::hostname($uhome);
    my $protocol = $Apache::lonnet::protocol{$uhome};
    $protocol = 'http' if ($protocol ne 'https');
    my $alias = &Apache::lonnet::use_proxy_alias($r,$uhome);
    $hostname = $alias if ($alias ne '');
    my $url = $protocol.'://'.$hostname.'/adm/resetpw';
    # Breadcrumbs
    my $start_page = &Apache::loncommon::start_page('Switching Server',undef,
                                                    {'redirect' => [0,$url],
                                                     'bread_crumbs' => $brcrum,});
    my $output = '<p>'.&mt('This LON-CAPA server belongs to a different domain.').' '.
                       &mt('You are being switched to your domain ([_1]), to use the "Forgot Password" tool.',$domdesc).
                 '</p>';
    return ($start_page,$output);
}

sub reset_passwd {
    my ($r,$token,$contact_name,$contact_email,$clientip,$passwdconf) = @_;
    return unless (ref($passwdconf) eq 'HASH');
    my %data = &Apache::lonnet::tmpget($token);
    my $now = time;
    if (keys(%data) == 0) {
        $r->print(&mt('Sorry, the URL you provided to complete the reset of your password was invalid. Either the token included in the URL has been deleted or the URL you provided was invalid. Please submit a [_1]new request[_2] for a password reset, and follow the link to the new URL included in the e-mail that will be sent to you, to allow you to enter a new password.'
                  ,'<a href="/adm/resetpw">','</a>')
        );
        return;
    }
    if (($data{'time'} =~ /^\d+$/) && 
        ($data{'username'} ne '') && 
        ($data{'domain'} ne '') && 
        ($data{'email'}  =~ /^[^\@]+\@[^\@]+\.[^\@\.]+$/) && 
        ($data{'temppasswd'} =~/^\w+$/)) {
        my $timelimit = 7200;
        if ($passwdconf->{resetlink} =~ /^\d+(|\.\d*)$/) {
            $timelimit = 3600 * $passwdconf->{resetlink};
        }
        my $reqtime = &Apache::lonlocal::locallocaltime($data{'time'});
        my ($blocked,$blocktext) =
            &Apache::loncommon::blocking_status('passwd',$clientip,$data{'username'},$data{'domain'});
        if ($blocked) {
            $r->print('<p class="LC_warning">'.$blocktext.'</p>');
            return;
        } elsif ($now - $data{'time'} < $timelimit) {
            my ($needscase,%formfields) = &reset_requires($data{'username'},$data{'domain'},
                                                          $passwdconf);
            if ($env{'form.action'} eq 'verify_and_change_pass') {
                my $invalidinfo;
                if ($formfields{'username'}) {
                    $env{'form.uname'} =~ s/^\s+|\s+$//g;
                    $env{'form.udom'} =~ s/^\s+|\s+$//g;
                    if ($needscase) {
                        unless (($env{'form.uname'} eq $data{'username'}) && ($env{'form.udom'} eq $data{'domain'})) {
                            $invalidinfo = "||$env{'form.uname'}|| ||$env{'form.udom'}|| ";
                        }
                    } else {
                        if ((lc($env{'form.uname'}) eq lc($data{'username'})) && (lc($env{'form.udom'}) eq lc($data{'domain'}))) {
                            $env{'form.uname'} = $data{'username'};
                        } else {
                            $invalidinfo = "||$env{'form.uname'}|| ||$env{'form.udom'}|| ";
                        }
                    }
                } else {
                    $env{'form.uname'} = $data{'username'};
                    $env{'form.udom'} = $data{'domain'};
                }
                if ($formfields{'email'}) {
                    $env{'form.email'} =~ s/^\s+|\s+$//g;
                    if ($needscase) {
                        unless ($env{'form.email'} eq $data{'email'}) {
                            $invalidinfo .= "||$env{'form.email'}||";
                        }
                    } else {
                        unless (lc($env{'form.email'}) eq lc($data{'email'})) {
                            $invalidinfo = "||$env{'form.email'}||";
                        }
                    }
                }
                if ($invalidinfo) {
                    &Apache::lonnet::logthis("Forgot Password -- token data: ||$data{'username'}|| ||$data{'domain'}|| ||$data{'email'}|| differs from form: $invalidinfo");
                    my $retry;
                    $r->print(
                              '<p class="LC_warning">'
                             .&mt('A problem occurred when attempting to reset'
                             .' the password for your account.').'</p>');
                    if (($formfields{'username'}) && ($formfields{'email'})) {
                        if ($needscase) {
                            $r->print('<p>'
                                     .&mt('Please verify you entered the correct username and e-mail address, '
                                     .'including the correct lower and/or upper case letters')
                                     .'</p>');
                        } else {
                            $r->print('<p>'
                                     .&mt('Please verify you entered the correct username and e-mail address.')
                                     .'</p>');
                        }
                        $retry = 1;
                    } elsif ($formfields{'username'}) {
                        if ($needscase) {
                            $r->print('<p>'
                                     .&mt('Please verify you entered the correct username, '
                                     .'including the correct lower and/or upper case letters')
                                     .'</p>');
                        } else {
                            $r->print('<p>'
                                     .&mt('Please verify you entered the correct username.')
                                     .'</p>');
                        }
                        $retry = 1;
                    } elsif ($formfields{'email'}) {
                        if ($needscase) {
                            $r->print('<p>'
                                     .&mt('Please verify you entered the correct e-mail address, '
                                     .'including the correct lower and/or upper case letters')
                                     .'</p>');
                        } else {
                            $r->print('<p>'
                                     .&mt('Please verify you entered the correct e-mail address.')
                                     .'</p>');
                        }
                        $retry = 1;
                    }
                    if ($retry) {
                         &Apache::lonpreferences::passwordchanger($r,'','reset_by_email',$token,$timelimit,\%formfields);
                    } else {
                        $r->print(&generic_failure_msg($contact_name,$contact_email));
                    }
                    unless ($formfields{'username'}) {
                        delete($env{'form.uname'});
                        delete($env{'form.udom'});
                    }
                    return;
                }
                my $change_failed =
		    &Apache::lonpreferences::verify_and_change_password($r,'reset_by_email',$token,$timelimit,\%formfields);
                if (!$change_failed) {
                    my $delete = &Apache::lonnet::tmpdel($token);
                    my $now = &Apache::lonlocal::locallocaltime(time);
                    my $domdesc = 
			&Apache::lonnet::domain($data{'domain'},'description');
                    my $ip = &Apache::lonnet::get_requestor_ip();
                    my $mailmsg = &mt('The password for your LON-CAPA account in the [_1] domain was changed [_2] from IP address: [_3].  If you did not perform this change or authorize it, please contact the [_4] ([_5]).',$domdesc,$now,$ip,$contact_name,$contact_email)."\n";
                    my $result = &send_mail($domdesc,$data{'email'},$mailmsg,
                                            $contact_name,$contact_email);
                    my $confirm_msg;
                    if ($result eq 'ok') {
                        $confirm_msg =
                            &Apache::lonhtmlcommon::confirm_success(
                                &mt('An e-mail confirming setting of the password'
                                   .' for your LON-CAPA account has been sent to [_1].'
                                    ,'<span class="LC_filename">'.$data{'email'}.'</span>'));
                    } else {
                        $confirm_msg =
                            &Apache::lonhtmlcommon::confirm_success(
                                &mt('An error occurred when sending e-mail to [_1]'
                                   .' confirming setting of your new password.'
                                    ,'<span class="LC_filename">'.$data{'email'}.'</span>'),1);
                    }
                    $r->print(
                        &Apache::loncommon::confirmwrapper($confirm_msg)
                       .&Apache::lonhtmlcommon::actionbox([
                            '<a href="/adm/login">'.&mt('Go to the login page').'</a>'])
                    );
                } elsif ($change_failed eq 'invalid_client') {
                    my $homeserver = &Apache::lonnet::homeserver($data{'username'},$data{'domain'});
                    if ($homeserver eq 'no_host') {
                        $r->print(&generic_failure_msg($contact_name,$contact_email));
                    } else {
                        my $hostname = &Apache::lonnet::hostname($homeserver);
                        my $protocol = $Apache::lonnet::protocol{$homeserver};
                        $protocol = 'http' if ($protocol ne 'https');
                        my $alias = &Apache::lonnet::use_proxy_alias($r,$homeserver);
                        $hostname = $alias if ($alias ne '');
                        my $url = $protocol.'://'.$hostname.'/adm/resetpw';
                        my ($opentag,$closetag);
                        if ($url) {
                           $opentag = '<a href="'.$url.'">';
                           $closetag = '</a>';
                        }
                        $r->print(
                            '<p class="LC_warning">'
                           .&mt('A problem occurred when attempting to reset'
                               .' the password for your account.'
                               .' Please try again from your [_1]home server[_2].'
                                ,$opentag,$closetag)
                           .'</p>'
                        );
                    }
                } elsif (($change_failed eq 'prioruse') && ($passwdconf->{'numsaved'})) {
                    my $domdesc =
                        &Apache::lonnet::domain($data{'domain'},'description');
                    $r->print(
                          '<p class="LC_warning">'
                         .&mt('Please enter a password that you have not used recently.')
                         .'</p>'
                         .&display_actions($contact_email,$domdesc,$token)
                    );
                } elsif (($change_failed eq 'internalerror') || ($change_failed eq 'missingtemp') ||
                         ($change_failed eq 'error')) {
                    $r->print(&generic_failure_msg($contact_name,$contact_email));
                }
                unless ($formfields{'username'}) {
                    delete($env{'form.uname'});
                    delete($env{'form.udom'});
                }
            } else {
                $r->print(&mt('The token included in an e-mail sent to you [_1] has been verified, so you may now proceed to reset the password for your LON-CAPA account.',$reqtime).'<br /><br />');
                if (keys(%formfields)) {
                    if (($formfields{'username'}) && ($formfields{'email'})) {
                        $r->print(&mt('Please enter the username and domain of the LON-CAPA account, and the associated e-mail address, for which you are setting a password.'));
                    } elsif ($formfields{'username'}) {
                        $r->print(&mt('Please enter the username and domain of the LON-CAPA account for which you are setting a password.'));
                    } elsif ($formfields{'email'}) {
                        $r->print(&mt('Please enter the e-mail address associated with the LON-CAPA account for which you are setting a password.'));
                    }
                    if ($needscase) {
                        $r->print(' '.&mt('User data entered must match LON-CAPA account information (including case).'));
                    }
                    $r->print('<br />');
                }
                my ($min,$max,$minrule,$maxrule);
                if ($passwdconf->{min}) {
                    $min = $passwdconf->{min};
                } else {
                    $min = $Apache::lonnet::passwdmin;
                }
                if ($min) {
                    $minrule = &mt('Minimum password length: [_1]',$min);
                }
                if ($passwdconf->{max}) {
                    $max = $passwdconf->{max};
                    $maxrule = &mt('Maximum password length: [_1]',$max);
                }
                if (ref($passwdconf->{chars}) eq 'ARRAY') {
                    my %rules;
                    map { $rules{$_} = 1; } @{$passwdconf->{chars}};
                    my %rulenames = &Apache::lonlocal::texthash(
                                                     uc => 'At least one upper case letter',
                                                     lc => 'At least one lower case letter',
                                                     num => 'At least one number',
                                                     spec => 'At least one non-alphanumeric',
                                                   );
                    $r->print(&mt('The new password must satisfy the following:').'<ul>');
                    foreach my $poss ('uc','lc','num','spec') {
                        if ($rules{$poss}) {
                            $r->print('<li>'.$rulenames{$poss}.'</li>');
                        }
                    }
                    if ($min) {
                        $r->print('<li>'.$minrule.'</li>');
                    }
                    if ($max) {
                        $r->print('<li>'.$maxrule.'</li>');
                    }
                    $r->print('</ul>');
                } else {
                    if ($min && $max) {
                        $r->print(&mt('The new password must satisfy the following:').'<ul>'."\n".
                                  '<li>'.$minrule.'</li>'."\n".
                                  '<li>'.$maxrule.'</li>'."\n".
                                  '</ul>'."\n");
                    } elsif ($min) {
                        $r->print($minrule.'<br />');
                    } elsif ($max) {
                        $r->print($maxrule.'<br />');
                    }
                }
                $r->print(&mt('Your new password will be sent to the LON-CAPA server in an encrypted form.').'<br />');
                &Apache::lonpreferences::passwordchanger($r,'','reset_by_email',$token,$timelimit,\%formfields);
            }
        } else {
            $r->print(
                '<p class="LC_warning">'
               .&mt('Sorry, the token generated when you requested a password reset has expired.').'<br />'
               .&mt('Please submit a [_1]new request[_2], and follow the link to the web page included in the new e-mail that will be sent to you, to allow you to enter a new password.'
                    ,'<a href="/adm/resetpw">','</a>')
               .'</p>'
            );
        }
    } else {
        $r->print(
            '<p class="LC_warning">'
           .&mt('Sorry, the URL generated when you requested reset of your password contained incomplete information. Please submit a [_1]new request[_2] for a password reset, and use the new URL that will be sent to your e-mail account to complete the process.'
                ,'<a href="/adm/resetpw">','</a>')
           .'</p>'
        );
    }
    return;
}

sub generic_failure_msg {
    my ($contact_name,$contact_email) = @_;
    return
        '<p class="LC_error">'
       .&mt('A problem occurred when attempting to reset the password for your account.')
       .'<br />'
       .&mt('Please contact the [_1] ([_2]) for assistance.',
              $contact_name,'<a href="mailto:'.$contact_email.'">'.$contact_email.'</a>')
       .'</p>';
}

sub create_passwd {
    my $passwd = '';
    my @letts = ("a".."z");
    for (my $i=0; $i<8; $i++) {
        my $lettnum = int(rand(2));
        my $item = '';
        if ($lettnum) {
            $item = $letts[int(rand(26))];
            my $uppercase = int(rand(2));
            if ($uppercase) {
                $item =~ tr/a-z/A-Z/;
            }
        } else {
            $item = int(rand(10));
        }
        $passwd .= $item;
    }
    return ($passwd);
}

sub display_actions {
    my ($contact_email,$domdesc,$token) = @_;
    my $url = '/adm/resetpw';
    if ($token) {
        $url .= '?token='.&escape($token);
    }
    my @msg = (&mt('[_1]Go back[_2] and try again',
                   '<a href="'.$url.'">','</a>'));
    my $msg2 = '';
    if ($contact_email ne '') {
        my $escuri = &HTML::Entities::encode('/adm/resetpw','&<>"');
        push(@msg, &mt('Contact the [_1]LON-CAPA helpdesk[_2] for the institution: [_3]',
                       '<a href="/adm/helpdesk?origurl='.$escuri.'">',
                       '</a>','<i>'.$domdesc.'</i>'));
    } else {
        $msg2 =
            '<p>'
           .&mt('You may wish to send an e-mail to the'
           .' server administrator: [_1] for the [_2] domain.',
                '<i>'.$Apache::lonnet::perlvar{'AdmEMail'}.'</i>',
                '<i>'.$domdesc.'</i>')
           .'</p>';
    }
    return &Apache::lonhtmlcommon::actionbox(\@msg).$msg2;
}

sub reset_requires {
    my ($username,$domain,$passwdconf) = @_;
    my (%fields,$needscase);
    return ($needscase,%fields) unless (ref($passwdconf) eq 'HASH');
    my (%postlink,%resetcase);
    if (ref($passwdconf->{resetpostlink}) eq 'HASH') {
        %postlink = %{$passwdconf->{resetpostlink}};
    }
    if (ref($passwdconf->{resetcase}) eq 'ARRAY') {
        map { $resetcase{$_} = 1; } (@{$passwdconf->{resetcase}});
    } else {
        $needscase = 1;
    }
    my %userenv =
        &Apache::lonnet::get('environment',['inststatus'],$domain,$username);
    my @inststatuses;
    if ($userenv{'inststatus'} ne '') {
        @inststatuses = split(/:/,$userenv{'inststatus'});
    } else {
        @inststatuses = ('default');
    }
    foreach my $status (@inststatuses) {
        if (ref($postlink{$status}) eq 'ARRAY') {
            map { $fields{$_} = 1; } (@{$postlink{$status}});
        } else {
            map { $fields{$_} = 1; } ('username','email');
        }
        if ($resetcase{$status}) {
            $needscase = 1;
        }
    }
    return ($needscase,%fields);
}

1;
