# The LearningOnline Network
#
#  Read maps into a 'big hash'.
#
# $Id: lonmap.pm,v 1.15 2014/12/11 01:50:27 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
###

package Apache::lonmap;
use strict;

#------------- Required external modules.

use Error qw(:try);

use HTML::TokeParser;


use LONCAPA;
use Apache::lonnet;
use Apache::lonlocal;

use Data::Dumper;


#------------- File scoped variables:

my $map_number = 0;		# keep track of maps within the course.
my $course_id;     		# Will be the id of the course being read in.

#
# The variables below are auxiliary hashes.  They will be tacked onto the
# big hash though currently not used by clients.. you never know about later.
#

my %randompick;
my %randompickseed;
my %randomorder;
my %encurl;
my %hiddenurl;
my %parmhash;
my @cond;			# Array of conditions.
my $retfrid;
#
#  Other stuff we make global (sigh) so that it does not need
#  to be passed around all the time:
#

my $username;			# User for whom the map is being read.
my $userdomain;  		# Domain the user lives in.
my $short_name;			# Course shortname.
my %mapalias_cache;		# Keeps track of map aliases -> resources detects duplicates.
my %cenv;			# Course environment.

#------------- Executable code: 


#----------------------------------------------------------------
#
#  General utilities:


#
#  I _think_ this does common sub-expression simplification and 
#  optimization for condition strings...based on common pattern matching.
# Parameters:
#    expression - the condition expression string.
# Returns:
#    The optimized expression if an optimization could be found.
#
# NOTE:
#   Added repetetive optimization..it's possible that an
#   optimization results in an expression that can be recognized further in
#   a subsequent optimization pass:
#

sub simplify {
    my $expression=shift;

# (0&1) = 1
	$expression=~s/\(0\&([_\.\d]+)\)/$1/g;
# (8)=8
	$expression=~s/\(([_\.\d]+)\)/$1/g;
# 8&8=8
	$expression=~s/([^_\.\d])([_\.\d]+)\&\2([^_\.\d])/$1$2$3/g;
# 8|8=8
	$expression=~s/([^_\.\d])([_\.\d]+)(?:\|\2)+([^_\.\d])/$1$2$3/g;
# (5&3)&4=5&3&4
	$expression=~s/\(([_\.\d]+)((?:\&[_\.\d]+)+)\)\&([_\.\d]+[^_\.\d])/$1$2\&$3/g;
# (((5&3)|(4&6)))=((5&3)|(4&6))
	$expression=~
	    s/\((\(\([_\.\d]+(?:\&[_\.\d]+)*\)(?:\|\([_\.\d]+(?:\&[_\.\d]+)*\))+\))\)/$1/g;
# ((5&3)|(4&6))|(1&2)=(5&3)|(4&6)|(1&2)
	$expression=~
	    s/\((\([_\.\d]+(?:\&[_\.\d]+)*\))((?:\|\([_\.\d]+(?:\&[_\.\d]+)*\))+)\)\|(\([_\.\d]+(?:\&[_\.\d]+)*\))/\($1$2\|$3\)/g;


    return $expression;
}
    

#
#  Merge the conditions into the big hash
#  these will result in hash entries of the form:
#   'condition.n'  where 'n' is the array index of the condition in the
#   @cond array above.
#
#  Parameters:
#    $hash - big hashthat's being built up.
#
sub merge_conditions {
    my $hash = shift;

    for (my $i = 0; $i < scalar(@cond); $i++) {
	$hash->{'condition' . '.' . $i} = $cond[$i];
    }
}

# Merge the contents of a 'child hash' into a parent hash hanging it off another key.
# This is _not_ done by hanging a reference to the child hash as the parent may be
# bound to a GDBM file e.g. and shared by more than one process ..and references are
# pretty clearly not going to work across process boundaries.
#
# Parameters:
#   $parent  - The hash to which the child will be merged (reference)
#   $key     - The key in the parent hash on which the child elements will be hung.
#              given a key named $childkey the final parent hash entry will be
#              $parent . '.' $childkey
#  $child    - The hash whose contents we merge into the parent (reference)
#
sub merge_hash {
    my ($parent, $key, $child) = @_;

    if ($key ne '') {
	$key .= '.';		# If we are prefixing, prefix then .
    }

    foreach my $childkey (keys(%$child)) {
	$parent->{$key . $childkey} = $child->{$childkey};
    }
}

#----------------------------------------------------------------------------------
#
#   Code to keep track of map aliases and to determine if there are doubly 
#   defined aliases.
#

#
#  Maintains the mapalias hash.  This is a hash of arrays.  Each array
#  is indexed by the alias and contains the set of resource ids pointed to by that
#  alias.  In an ideal world, there will only be one element in each array.
#  The point of this, however is to determine which aliases might be doubley defined
#  due to map nesting e.g.
#
#  Parameters:
#    $value   - Alias name.
#    $resid   - Resource id pointed to by the alias.
#
#    
sub count_mapalias {
    my ($value,$resid) = @_;
    push(@{ $mapalias_cache{$value} }, $resid);
}
#
#  Looks at each key in the mapalias hash and, for each case where an
#  alias points to more than one value adds an error text to the
#  result string.'
#
#  Parameters:
#     hash - Reference to the hash we are trying t build up.
#  Implicit inputs
#     %mapalias - a hash that is indexed by map aliases and contains for each key
#                 an array of the resource id's the alias 'points to'.
# Returns:
#    A hopefully empty string of messages that descsribe the aliases that have more
#    than one value.  This string is formatted like an html list.
#
#
sub get_mapalias_errors {
    my $hash = shift;
    my $error_text;
    foreach my $mapalias (sort(keys(%mapalias_cache))) {
	next if (scalar(@{ $mapalias_cache{$mapalias} } ) == 1);
	my $count;
	my $which =
	    join('</li><li>', 
		 map {
		     my $id = $_;
		     if (exists($hash->{'src_'.$id})) {
			 $count++;
		     }
		     my ($mapid) = split(/\./,$id);
		     &mt('Resource [_1][_2]in Map [_3]',
			 '"'.$hash->{'title_'.$id}.'"',
                         '<br />',
			 '"'.$hash->{'title_'.$hash->{'ids_'.$hash->{'map_id_'.$mapid}}}.'"');
		 } (@{ $mapalias_cache{$mapalias} }));
	next if ($count < 2);
	$error_text .= '<div class="LC_error">'.
	    &mt('Error: Found the mapalias "[_1]" defined multiple times.',
		$mapalias).
		'</div><ul><li>'.$which.'</li></ul>';
    }
    &clear_mapalias_count();
    return $error_text;
}
#
#   Clears the map aliase hash.
#
sub clear_mapalias_count {
    undef(%mapalias_cache);
}

#----------------------------------------------------------------
#
#  Code dealing with resource versions.
#

#
#  Put a version into a src element of a hash or url:
#
#  Parameters:
#     uri - URI into which the version must be added.
#    hash - Reference to the hash being built up.
#    short- Short coursename.
#

sub putinversion {
    my ($uri, $hash, $short) = @_;
    my $key=$short.'_'.&Apache::lonnet::clutter($uri);
    if ($hash->{'version_'.$uri}) {
	my $version=$hash->{'version_'.$uri};
	if ($version eq 'mostrecent') { return $uri; }
	if ($version eq &Apache::lonnet::getversion(
			&Apache::lonnet::filelocation('',$uri))) 
	             { return $uri; }
	$uri=~s/\.(\w+)$/\.$version\.$1/;
    }
    &Apache::lonnet::do_cache_new('courseresversion',$key,&Apache::lonnet::declutter($uri),600);
    return $uri;
}


#
#  Create hash entries for each version of the course.
# Parameters:
#   $cenv    - Reference to a course environment from lonnet::coursedescription.
#   $hash    - Reference to a hash that will be populated.

#
sub process_versions {
    my ($cenv, $hash) = @_;

    
    my %versions = &Apache::lonnet::dump('resourceversions',
					 $cenv->{'domain'},
					 $cenv->{'num'});

    foreach my $ver (keys(%versions)) {
	if ($ver =~/^error\:/) { # lonc/lond transaction failed.
	    throw Error::Simple('lonc/lond returned error: ' . $ver);
	}
	$hash->{'version_'.$ver} = $versions{$ver};
    }
}

#
#  Generate text for a version discrepancy error:
# Parameters:
#  $uri   - URI of the resource.
#  $used  - Version used.
#  $unused - Veresion of duplicate.
#
sub versionerror {
    my ($uri, $used, $unused) = @_;
    return '<br />'.
	&mt('Version discrepancy: resource [_1] included in both version [_2] and version [_3]. Using version [_2].',
	    $uri,$used,$unused).'<br />';

}

#  Removes the version number from a URI and returns the resulting
#  URI (e.g. mumbly.version.stuff => mumbly.stuff).
#
#   If the URI has not been seen with a version before the
#   hash{'version_'.resultingURI} is set to the  version number.
#   If the hash has already been seen, but differs then
#   an error is raised.
#
# Parameters:
#   $uri  -  potentially with a version.
#   $hash -  reference to a hash to fill in. 
# Returns:
#   URI with the version cut out.
#
sub versiontrack {
    my ($uri, $hash) = @_;


    if ($uri=~/\.(\d+)\.\w+$/) { # URI like *.n.text it's version 'n'
	my $version=$1;
	$uri=~s/\.\d+\.(\w+)$/\.$1/; # elide the version.
        unless ($hash->{'version_'.$uri}) {
	    $hash->{'version_'.$uri}=$version;
	} elsif ($version!=$hash->{'version_'.$uri}) {
	    throw Error::Simple(&versionerror($uri, $hash->{'version_'.$uri}, $version));
        }
    }
    return $uri;
}
#
#  Appends the version of a resource to its uri and also caches the 
#  URI (contents?) on the local server
#
#  Parameters:
#     $uri   - URI of the course (without version informatino.
#     $hash  - What we have of  the big hash.
#
# Side-Effects:
#   The URI is cached by memcached.
#
# Returns:
#    The version appended URI.
#
sub append_version {
    my ($uri, $hash) = @_;

    # Create the key for the cache entry.

    my $key = $course_id . '_' . &Apache::lonnet::clutter($uri);

    # If there is a version it will already be  in the hash:

    if ($hash->{'version_' . $uri}) {
	my $version = $hash->{'version_' . $uri};
	if ($version eq 'mostrecent') {
	    return $uri;     # Most recent version does not require decoration (or caching?).
	}
	if ($version eq 
	    &Apache::lonnet::getversion(&Apache::lonnet::filelocation('', $uri))) {
	    return $uri;	# version matches the most recent file version?
	}
	$uri =~ s/\.(\w+)$/\.$version\.$1/; # insert the versino prior to the last .word.
    }
 
   # cache the version:

   &Apache::lonnet::do_cache_new('courseresversion', $key, 
				 &Apache::lonnet::declutter($uri), 600);

    return $uri;

}
#------------------------------------------------------------------------------
#
#  Misc. utilities that don't fit into the other classifications.

# Determine if the specified user has an 'advanced' role in a course.
# Parameters:
#   cenv       - reference to a course environment.
#   username   - Name of the user we care about.
#   domain     - Domain in which the user is defined.
# Returns:
#    0  - User does not have an advanced role in the course.
#    1  - User does have an advanced role in the course.
#
sub has_advanced_role {
    my ($username, $domain) = @_;

    my %adv_roles = &Apache::lonnet::get_course_adv_roles($short_name);
    my $merged_username = $username . ':' . $domain;
    foreach my $user (values %adv_roles) {
	if ($merged_username eq $user) {
	    return 1;
	}
    }
    return 0;
}

#--------------------------------------------------------------------------------
# Post processing subs:
sub hiddenurls {
    my $hash = shift;

    my $uname    = $hash->{'context.username'};
    my $udom     = $hash->{'context.userdom'};
    my $courseid = $hash->{'context.courseid'};

    my $randomoutentry='';
    foreach my $rid (keys(%randompick)) {
        my $rndpick=$randompick{$rid};
        my $mpc=$hash->{'map_pc_'.$hash->{'src_'.$rid}};
# ------------------------------------------- put existing resources into array
        my @currentrids=();
        foreach my $key (sort(keys(%$hash))) {
	    if ($key=~/^src_($mpc\.\d+)/) {
		if ($hash->{'src_'.$1}) { push @currentrids, $1; }
            }
        }
	# rids are number.number and we want to numercially sort on 
        # the second number
	@currentrids=sort {
	    my (undef,$aid)=split(/\./,$a);
	    my (undef,$bid)=split(/\./,$b);
	    $aid <=> $bid;
	} @currentrids;
        next if ($#currentrids<$rndpick);
# -------------------------------- randomly eliminate the ones that should stay
	my (undef,$id)=split(/\./,$rid);
        if ($randompickseed{$rid}) { $id=$randompickseed{$rid}; }
	my $rndseed=&Apache::lonnet::rndseed($id, $courseid, $udom, $uname, \%cenv); # use id instead of symb
	&Apache::lonnet::setup_random_from_rndseed($rndseed);
	my @whichids=&Math::Random::random_permuted_index($#currentrids+1);
        for (my $i=1;$i<=$rndpick;$i++) { $currentrids[$whichids[$i]]=''; }

# -------------------------------------------------------- delete the leftovers
        for (my $k=0; $k<=$#currentrids; $k++) {
            if ($currentrids[$k]) {
		$hash->{'randomout_'.$currentrids[$k]}='1';
                my ($mapid,$resid)=split(/\./,$currentrids[$k]);
                $randomoutentry.='&'.
		    &Apache::lonnet::encode_symb($hash->{'map_id_'.$mapid},
						 $resid,
						 $hash->{'src_'.$currentrids[$k]}
						 ).'&';
            }
        }
    }
# ------------------------------ take care of explicitly hidden urls or folders
    foreach my $rid (keys(%hiddenurl)) {
	$hash->{'randomout_'.$rid}='1';
	my ($mapid,$resid)=split(/\./,$rid);
	$randomoutentry.='&'.
	    &Apache::lonnet::encode_symb($hash->{'map_id_'.$mapid},$resid,
					 $hash->{'src_'.$rid}).'&';
    }
# --------------------------------------- add randomout to the hash.
    if ($randomoutentry) {
	$hash->{'acc.randomout'} = $randomoutentry;

    }
}

#
# It's not so clear to me what this sub does.
#
#  Parameters
#     uri   - URI from the course description hash.
#     short - Course short name.
#     fn    - Course filename.
#     hash  - Reference to the big hash as filled in so far
#       

sub accinit {
    my ($uri, $short, $hash)=@_;
    my %acchash=();
    my %captured=();
    my $condcounter=0;
    $acchash{'acc.cond.'.$short.'.0'}=0;

    # This loop is only interested in conditions and 
    # parameters in the big hash:

    foreach my $key (keys(%$hash)) {

	# conditions:

	if ($key=~/^conditions/) {
	    my $expr=$hash->{$key};

	    # try to find and factor out common sub-expressions
	    # Any subexpression that is found is simplified, removed from
	    # the original condition expression and the simplified sub-expression
	    # substituted back in to the epxression..I'm not actually convinced this
	    # factors anything out...but instead maybe simplifies common factors(?)

	    foreach my $sub ($expr=~m/(\(\([_\.\d]+(?:\&[_\.\d]+)+\)(?:\|\([_\.\d]+(?:\&[_\.\d]+)+\))+\))/g) {
		my $orig=$sub;

		my ($factor) = ($sub=~/\(\(([_\.\d]+\&(:?[_\.\d]+\&)*)(?:[_\.\d]+\&*)+\)(?:\|\(\1(?:[_\.\d]+\&*)+\))+\)/);
		next if (!defined($factor));

		$sub=~s/\Q$factor\E//g;
		$sub=~s/^\(/\($factor\(/;
		$sub.=')';
		$sub=simplify($sub);
		$expr=~s/\Q$orig\E/$sub/;
	    }
	    $hash->{$key}=$expr;

           # If not yet seen, record in acchash and that we've seen it.

	    unless (defined($captured{$expr})) {
		$condcounter++;
		$captured{$expr}=$condcounter;
		$acchash{'acc.cond.'.$short.'.'.$condcounter}=$expr;
	    } 
        # Parameters:

	} elsif ($key=~/^param_(\d+)\.(\d+)/) {
	    my $prefix=&Apache::lonnet::encode_symb($hash->{'map_id_'.$1},$2,
						    $hash->{'src_'.$1.'.'.$2});
	    foreach my $param (split(/\&/,$hash->{$key})) {
		my ($typename,$value)=split(/\=/,$param);
		my ($type,$name)=split(/\:/,$typename);
		$parmhash{$prefix.'.'.&unescape($name)}=
		    &unescape($value);
		$parmhash{$prefix.'.'.&unescape($name).'.type'}=
		    &unescape($type);
	    }
	}
    }
    # This loop only processes id entries in the big hash.

    foreach my $key (keys(%$hash)) {
	if ($key=~/^ids/) {
	    foreach my $resid (split(/\,/,$hash->{$key})) {
		my $uri=$hash->{'src_'.$resid};
		my ($uripath,$urifile) =
		    &Apache::lonnet::split_uri_for_cond($uri);
		if ($uripath) {
		    my $uricond='0';
		    if (defined($hash->{'conditions_'.$resid})) {
			$uricond=$captured{$hash->{'conditions_'.$resid}};
		    }
		    if (defined($acchash{'acc.res.'.$short.'.'.$uripath})) {
			if ($acchash{'acc.res.'.$short.'.'.$uripath}=~
			    /(\&\Q$urifile\E\:[^\&]*)/) {
			    my $replace=$1;
			    my $regexp=$replace;
			    #$regexp=~s/\|/\\\|/g;
			    $acchash{'acc.res.'.$short.'.'.$uripath} =~
				s/\Q$regexp\E/$replace\|$uricond/;
			} else {
			    $acchash{'acc.res.'.$short.'.'.$uripath}.=
				$urifile.':'.$uricond.'&';
			}
		    } else {
			$acchash{'acc.res.'.$short.'.'.$uripath}=
			    '&'.$urifile.':'.$uricond.'&';
		    }
		} 
	    }
        }
    }
    $acchash{'acc.res.'.$short.'.'}='&:0&';
    my $courseuri=$uri;
    $courseuri=~s/^\/res\///;
    my $regexp = 1;
 
    &merge_hash($hash, '', \%acchash); # there's already an acc prefix in the hash keys.


}


#
#  Traces a route recursively through the map after it has been loaded
#  (I believe this really visits each resource that is reachable fromt he
#  start top node.
#
#  - Marks hidden resources as hidden.
#  - Marks which resource URL's must be encrypted.
#  - Figures out (if necessary) the first resource in the map.
#  - Further builds the chunks of the big hash that define how 
#    conditions work
#
#  Note that the tracing strategy won't visit resources that are not linked to
#  anything or islands in the map (groups of resources that form a path but are not
#  linked in to the path that can be traced from the start resource...but that's ok
#  because by definition, those resources are not reachable by users of the course.
#
# Parameters:
#   sofar    - _URI of the prior entry or 0 if this is the top.
#   rid      - URI of the resource to visit.
#   beenhere - list of resources (each resource enclosed by &'s) that have
#              already been visited.
#   encflag  - If true the resource that resulted in a recursive call to us
#              has an encoded URL (which means contained resources should too). 
#   hdnflag  - If true,the resource that resulted in a recursive call to us
#              was hidden (which means contained resources should be hidden too).
#   hash     - Reference to the hash we are traversing.
# Returns
#    new value indicating how far the map has been traversed (the sofar).
#
sub traceroute {
    my ($sofar, $rid, $beenhere, $encflag, $hdnflag, $hash)=@_;
    my $newsofar=$sofar=simplify($sofar);

    unless ($beenhere=~/\&\Q$rid\E\&/) {
	$beenhere.=$rid.'&';  
	my ($mapid,$resid)=split(/\./,$rid);
	my $symb=&Apache::lonnet::encode_symb($hash->{'map_id_'.$mapid},$resid,
					      $hash->{'src_'.$rid});
	my $hidden=&Apache::lonnet::EXT('resource.0.hiddenresource',$symb);
        my $ignorehidden;

	if ($hdnflag || lc($hidden) eq 'yes') {
            if (defined($hash->{'is_map_'.$rid})) {
                if (($hash->{'context.nohideurl'}) && ($hash->{'context.nohideurl'} eq $hash->{'src_'.$rid})) {
                    $ignorehidden = 1; # Hidden parameter explicitly deleted 
                                       # if printing/grading bubblesheet exam
                }
            }
            unless ($ignorehidden) {
	        $hiddenurl{$rid}=1;
            }
	}
	if (!$hdnflag && lc($hidden) eq 'no') {
	    delete($hiddenurl{$rid});
	}

	my $encrypt=&Apache::lonnet::EXT('resource.0.encrypturl',$symb);
	if ($encflag || lc($encrypt) eq 'yes') { $encurl{$rid}=1; }

	if (($retfrid eq '') && ($hash->{'src_'.$rid})
	    && ($hash->{'src_'.$rid}!~/\.sequence$/)) {
	    $retfrid=$rid;
	}

	if (defined($hash->{'conditions_'.$rid})) {
	    $hash->{'conditions_'.$rid}=simplify(
           '('.$hash->{'conditions_'.$rid}.')|('.$sofar.')');
	} else {
	    $hash->{'conditions_'.$rid}=$sofar;
	}

	# if the expression is just the 0th condition keep it
	# otherwise leave a pointer to this condition expression

	$newsofar = ($sofar eq '0') ? $sofar : '_'.$rid;

	# Recurse if the resource is a map:

	if (defined($hash->{'is_map_'.$rid})) {
	    if (defined($hash->{'map_start_'.$hash->{'src_'.$rid}})) {
                my $maphidden;
                unless ($ignorehidden) {
                    $maphidden = $hdnflag || $hiddenurl{$rid};
                }
		$sofar=$newsofar=
		    &traceroute($sofar,
				$hash->{'map_start_'.$hash->{'src_'.$rid}},
				$beenhere,
				$encflag || $encurl{$rid},
				$maphidden, $hash);
	    }
	}

	# Processes  links to this resource:
	#  - verify the existence of any conditionals on the link to here.
	#  - Recurse to any resources linked to us.
	#
	if (defined($hash->{'to_'.$rid})) {
	    foreach my $id (split(/\,/,$hash->{'to_'.$rid})) {
		my $further=$sofar;
		#
		# If there's a condition associated with this link be sure
		# it's been defined else that's an error:
		#
                if ($hash->{'undercond_'.$id}) {
		    if (defined($hash->{'condid_'.$hash->{'undercond_'.$id}})) {
			$further=simplify('('.'_'.$rid.')&('.
					  $hash->{'condid_'.$hash->{'undercond_'.$id}}.')');
		    } else {
			my $errtext.='<br />'.&mt('Undefined condition ID: [_1]',$hash->{'undercond_'.$id});
			throw Error::Simple($errtext);
		    }
                }
		#  Recurse to resoruces that have to's to us.
                $newsofar=&traceroute($further,$hash->{'goesto_'.$id},$beenhere,
				      $encflag,$hdnflag, $hash);
	    }
	}
    }
    return $newsofar;
}


#---------------------------------------------------------------------------------
#
#  Map parsing code:
#

# 
#  Parse the <param> tag.  for most parameters, the only action is to define/extend
#  a has entry for {'param_{refid}'} where refid is the resource the parameter is
#  attached to and the value built up is an & separated list of parameters of the form:
#  type:part.name=value
#
#   In addition there is special case code for:
#   - randompick
#   - randompickseed
#   - randomorder
#
#   - encrypturl
#   - hiddenresource
#
# Parameters:
#    token - The token array from HTML::TokeParse  we mostly care about element [2]
#            which is a hash of attribute => values supplied in the tag
#            (remember this sub is only processing start tag tokens).
#    mno   - Map number.  This is used to qualify resource ids within a map
#            to make them unique course wide (a process known as uniquifaction).
#    hash  - Reference to the hash we are building.
#
sub parse_param {
    my ($token, $mno, $hash)  = @_;

    # Qualify the reference and name by the map number and part number.
    # if no explicit part number is supplied, 0 is the implicit part num.

    my $referid=$mno.'.'.$token->[2]->{'to'}; # Resource param applies to.
    my $name=$token->[2]->{'name'};	      # Name of parameter
    my $part;


    if ($name=~/^parameter_(.*)_/) { 
	$part=$1;
    } else {
	$part=0;
    }

    # Peel the parameter_ off the parameter name.

    $name=~s/^.*_([^_]*)$/$1/;

    # The value is:
    #   type.part.name.value

    my $newparam=
	&escape($token->[2]->{'type'}).':'.
	&escape($part.'.'.$name).'='.
	&escape($token->[2]->{'value'});

    # The hash key is param_resourceid.
    # Multiple parameters for a single resource are & separated in the hash.


    if (defined($hash->{'param_'.$referid})) {
	$hash->{'param_'.$referid}.='&'.$newparam;
    } else {
	$hash->{'param_'.$referid}=''.$newparam;
    }
    #
    #  These parameters have to do with randomly selecting
    # resources, therefore a separate hash is also created to 
    # make it easy to locate them when actually computing the resource set later on
    # See the code conditionalized by ($randomize) in read_map().

    if ($token->[2]->{'name'}=~/^parameter_(0_)*randompick$/) { # Random selection turned on
	$randompick{$referid}=$token->[2]->{'value'};
    }
    if ($token->[2]->{'name'}=~/^parameter_(0_)*randompickseed$/) { # Randomseed provided.
	$randompickseed{$referid}=$token->[2]->{'value'};
    }
    if ($token->[2]->{'name'}=~/^parameter_(0_)*randomorder$/) { # Random order turned on.
	$randomorder{$referid}=$token->[2]->{'value'};
    }

    # These parameters have to do with how the URLs of resources are presented to
    # course members(?).  encrypturl presents encypted url's while
    # hiddenresource hides the URL.
    #

    if ($token->[2]->{'name'}=~/^parameter_(0_)*encrypturl$/) {
	if ($token->[2]->{'value'}=~/^yes$/i) {
	    $encurl{$referid}=1;
	}
    }
    if ($token->[2]->{'name'}=~/^parameter_(0_)*hiddenresource$/) {
	if ($token->[2]->{'value'}=~/^yes$/i) {
	    $hiddenurl{$referid}=1;
	}
    }
}


#
#  Parses a resource tag to produce the value to push into the
#  map_ids array.
# 
#
#  Information about the actual type of resource is provided by the file extension
#  of the uri (e.g. .problem, .sequence etc. etc.).
#
#  Parameters:
#    $token   - A token from HTML::TokeParser
#               This is an array that describes the most recently parsed HTML item.
#    $lpc     - Map nesting level (?)
#    $ispage  - True if this resource is encapsulated in a .page (assembled resourcde).
#    $uri     - URI of the enclosing resource.
#    $code    - CODE for which resource is being parsed (CODEd assignments).
#    $hash    - Reference to the hash we are building.
#
# Returns:
#   Value of the id attribute of the tag.
#
# Note:
#   The token is an array that contains the following elements:
#   [0]   => 'S' indicating this is a start token
#   [1]   => 'resource'  indicating this tag is a <resource> tag.
#   [2]   => Hash of attribute =>value pairs.
#   [3]   => @(keys [2]).
#   [4]   => unused.
#
#   The attributes of the resourcde tag include:
#
#   id     - The resource id.
#   src    - The URI of the resource.
#   type   - The resource type (e.g. start and finish).
#   title  - The resource title.
#

sub parse_resource {
    my ($token,$lpc,$ispage,$uri,$code,$hash) = @_;
    
    # I refuse to countenance code like this that has 
    # such a dirty side effect (and forcing this sub to be called within a loop).
    #
    #  if ($token->[2]->{'type'} eq 'zombie') { next; }
    #
    #  The original code both returns _and_ skips to the next pass of the >caller's<
    #  loop, that's just dirty.
    #

    # Zombie resources don't produce anything useful.

    if ($token->[2]->{'type'} eq 'zombie') {
	return undef;
    }

    my $rid=$lpc.'.'.$token->[2]->{'id'}; # Resource id in hash is levelcounter.id-in-xml.

    # Save the hash element type and title:
	    
    $hash->{'kind_'.$rid}='res';
    $hash->{'title_'.$rid}=$token->[2]->{'title'};

    # Get the version free URI for the resource.
    # If a 'version' attribute was supplied, and this resource's version 
    # information has not yet been stored, store it.
    #


    my $turi=&versiontrack($token->[2]->{'src'});
    if ($token->[2]->{'version'}) {
	unless ($hash->{'version_'.$turi}) {

	    #Where does the value of $1 below come from?
	    #$1 for the regexps in versiontrack should have gone out of scope.
	    #
	    # I think this may be dead code since versiontrack ought to set
	    # this hash element(?).
	    #
	    $hash->{'version_'.$turi}=$1;
	}
    }
    # Pull out the title and do entity substitution on &colon
    # Q: Why no other entity substitutions?

    my $title=$token->[2]->{'title'};
    $title=~s/\&colon\;/\:/gs;



    # I think the point of all this code is to construct a final
    # URI that apache and its rewrite rules can use to
    # fetch the resource.   Thi s sonly necessary if the resource
    # is not a page.  If the resource is a page then it must be
    # assembled (at fetch time?).

    unless ($ispage) {
	$turi=~/\.(\w+)$/;
	my $embstyle=&Apache::loncommon::fileembstyle($1);
	if ($token->[2]->{'external'} eq 'true') { # external
	    $turi=~s/^https?\:\/\//\/adm\/wrapper\/ext\//;
	} elsif ($turi=~/^\/*uploaded\//) { # uploaded
	    if (($embstyle eq 'img') 
		|| ($embstyle eq 'emb')
		|| ($embstyle eq 'wrp')) {
		$turi='/adm/wrapper'.$turi;
	    } elsif ($embstyle eq 'ssi') {
		#do nothing with these
	    } elsif ($turi!~/\.(sequence|page)$/) {
		$turi='/adm/coursedocs/showdoc'.$turi;
	    }
	} elsif ($turi=~/\S/) { # normal non-empty internal resource
	    my $mapdir=$uri;
	    $mapdir=~s/[^\/]+$//;
	    $turi=&Apache::lonnet::hreflocation($mapdir,$turi);
	    if (($embstyle eq 'img') 
		|| ($embstyle eq 'emb')
		|| ($embstyle eq 'wrp')) {
		$turi='/adm/wrapper'.$turi;
	    }
	}
    }
    # Store reverse lookup, remove query string resource 'ids'_uri => resource id.
    # If the URI appears more than one time in the sequence, it's resourcde
    # id's are constructed as a comma spearated list.

    my $idsuri=$turi;
    $idsuri=~s/\?.+$//;
    if (defined($hash->{'ids_'.$idsuri})) {
	$hash->{'ids_'.$idsuri}.=','.$rid;
    } else {
	$hash->{'ids_'.$idsuri}=''.$rid;
    }
    


    if ($turi=~/\/(syllabus|aboutme|navmaps|smppg|bulletinboard|viewclasslist)$/) {
	$turi.='?register=1';
    }
    

    # resource id lookup:  'src'_resourc-di  => URI decorated with a query
    # parameter as above if necessary due to the resource type.
    
    $hash->{'src_'.$rid}=$turi;

    # Mark the external-ness of the resource:
    
    if ($token->[2]->{'external'} eq 'true') {
	$hash->{'ext_'.$rid}='true:';
    } else {
	$hash->{'ext_'.$rid}='false:';
    }

    # If the resource is a start/finish resource set those
    # entries in the has so that navigation knows where everything starts.
    #   If there is a malformed sequence that has no start or no finish
    # resource, should this be detected and errors thrown?  How would such a 
    # resource come into being other than being manually constructed by a person
    # and then uploaded?  Could that happen if an author decided a sequence was almost
    # right edited it by hand and then reuploaded it to 'fix it' but accidently cut the
    #  start or finish resources?
    #
    #  All resourcess also get a type_id => (start | finish | normal)    hash entr.
    #
    if ($token->[2]->{'type'}) {
	$hash->{'type_'.$rid}=$token->[2]->{'type'};
	if ($token->[2]->{'type'} eq 'start') {
	    $hash->{'map_start_'.$uri}="$rid";
	}
	if ($token->[2]->{'type'} eq 'finish') {
	    $hash->{'map_finish_'.$uri}="$rid";
	}
    }  else {
	$hash->{'type_'.$rid}='normal';
    }

    # Sequences end pages are constructed entities.  They require that the 
    # map that defines _them_ be loaded as well into the hash...with this resourcde
    # as the base of the nesting.
    # Resources like that are also marked with is_map_id => 1 entries.
    #
    
    if (($turi=~/\.sequence$/) ||
	($turi=~/\.page$/)) {
	$hash->{'is_map_'.$rid}='1'; # String in lonuserstate.
        if ($hiddenurl{$rid}) {
            if (($hash->{'context.nohideurl'}) &&
                ($hash->{'context.nohideurl'} eq $hash->{'src_'.$rid})) {
                delete($hiddenurl{$rid}); # Hidden parameter explicitly deleted
                                          # if printing/grading bubblesheet exam  
            }
        }

	&read_map($turi,$rid,$code,$hash);
    } 
    return $token->[2]->{'id'};
}

#  Links define how you are allowed to move from one resource to another.
#  They are the transition edges in the directed graph that a map is.
#  This sub takes informatino from a <link> tag and constructs the
#  navigation bits and pieces of a map.  There is no requirement that the
#  resources that are linke are already defined, however clearly the map is 
#  badly broken if they are not _eventually_ defined.
#
#  Note that links can be unconditional or conditional.
#
#  Parameters:
#     linkpc   - The link counter for this level of map nesting (this is 
#                reset to zero by read_map prior to starting to process
#                links for map).
#     lpc      - The map level ocounter (how deeply nested this map is in
#                the hierarchy of maps that are recursively read in.
#     to       - resource id (within the XML) of the target of the edge.
#     from     - resource id (within the XML) of the source of the edge.
#     condition- id of condition associated with the edge (also within the XML).
#     hash     - reference to the hash we are building.

#

sub make_link {
    my ($linkpc,$lpc,$to,$from,$condition, $hash) = @_;
    
    #  Compute fully qualified ids for the link, the 
    # and from/to by prepending lpc.
    #

    my $linkid=$lpc.'.'.$linkpc;
    my $goesto=$lpc.'.'.$to;
    my $comesfrom=$lpc.'.'.$from;
    my $undercond='0';


    # If there is a condition, qualify it with the level counter.

    if ($condition) {
	$undercond=$lpc.'.'.$condition;
    }

    # Links are represnted by:
    #  goesto_.fuullyqualifedlinkid => fully qualified to
    #  comesfrom.fullyqualifiedlinkid => fully qualified from
    #  undercond_.fullyqualifiedlinkid => fully qualified condition id.

    $hash->{'goesto_'.$linkid}=$goesto;
    $hash->{'comesfrom_'.$linkid}=$comesfrom;
    $hash->{'undercond_'.$linkid}=$undercond;

    # In addition:
    #   to_.fully qualified from => comma separated list of 
    #   link ids with that from.
    # Similarly:
    #   from_.fully qualified to => comma separated list of link ids`
    #                               with that to.
    #  That allows us given a resource id to know all edges that go to it
    #  and leave from it.
    #

    if (defined($hash->{'to_'.$comesfrom})) {
	$hash->{'to_'.$comesfrom}.=','.$linkid;
    } else {
	$hash->{'to_'.$comesfrom}=''.$linkid;
    }
    if (defined($hash->{'from_'.$goesto})) {
	$hash->{'from_'.$goesto}.=','.$linkid;
    } else {
	$hash->{'from_'.$goesto}=''.$linkid;
    }
}

# ------------------------------------------------------------------- Condition
#
#  Processes <condition> tags, storing sufficient information about them
#  in the hash so that they can be evaluated and used to conditionalize
#  what is presented to the student.
#
#  these can have the following attributes 
#
#    id    = A unique identifier of the condition within the map.
#
#    value = Is a perl script-let that, when evaluated in safe space
#            determines whether or not the condition is true.
#            Normally this takes the form of a test on an  Apache::lonnet::EXT call
#            to find the value of variable associated with a resource in the
#            map identified by a mapalias.
#            Here's a fragment of XML code that illustrates this:
#
#           <param to="5" value="mainproblem" name="parameter_0_mapalias" type="string" />
#           <resource src="" id="1" type="start" title="Start" />
#           <resource src="/res/msu/albertel/b_and_c/p1.problem" id="5"  title="p1.problem" />
#           <condition value="&EXT('user.resource.resource.0.tries','mainproblem')
#           <2 " id="61" type="stop" />
#           <link to="5" index="1" from="1" condition="61" />    
#
#           In this fragment:
#             - The param tag establishes an alias to resource id 5 of 'mainproblem'.
#             - The resource that is the start of the map is identified.
#             - The resource tag identifies the resource associated with this tag
#               and gives it the id 5.
#             - The condition is true if the tries variable associated with mainproblem
#               is less than 2 (that is the user has had more than 2 tries).
#               The condition type is a stop condition which inhibits(?) the associated
#               link if the condition  is false. 
#             - The link to resource 5 from resource 1 is affected by this condition.    
#            
#    type  = Type of the condition. The type determines how the condition affects the
#            link associated with it and is one of
#            -  'force'
#            -  'stop'
#              anything else including not supplied..which treated as:
#            - 'normal'.
#            Presumably maps get created by the resource assembly tool and therefore
#            illegal type values won't squirm their way into the XML.
#   hash   - Reference to the hash we are trying to build up.
#
# Side effects:
#   -  The kind_level-qualified-condition-id hash element is set to 'cond'.
#   -  The condition text is pushed into the cond array and its element number is
#      set in the condid_level-qualified-condition-id element of the hash.
#   - The condition type is colon appneded to the cond array element for this condition.
sub parse_condition {
    my ($token, $lpc, $hash) = @_;
    my $rid=$lpc.'.'.$token->[2]->{'id'};
    
    $hash->{'kind_'.$rid}='cond';

    my $condition = $token->[2]->{'value'};
    $condition =~ s/[\n\r]+/ /gs;
    push(@cond, $condition);
    $hash->{'condid_'.$rid}=$#cond;
    if ($token->[2]->{'type'}) {
	$cond[$#cond].=':'.$token->[2]->{'type'};
    }  else {
	$cond[$#cond].=':normal';
    }
}

#
#  Parse mapalias parameters.
#  these are tags of the form:
#  <param to="nn" 
#         value="some-alias-for-resourceid-nn" 
#         name="parameter_0_mapalias" 
#         type="string" />
#  A map alias is a textual name for a resource:
#    - The to  attribute identifies the resource (this gets level qualified below)
#    - The value attributes provides the alias string.
#    - name must be of the regexp form: /^parameter_(0_)*mapalias$/
#    - e.g. the string 'parameter_' followed by 0 or more "0_" strings
#      terminating with the string 'mapalias'.
#      Examples:
#         'parameter_mapalias', 'parameter_0_mapalias', parameter_0_0_mapalias'
#  Invalid to ids are silently ignored.
#
#  Parameters:
#     token - The token array fromthe HMTML::TokeParser
#     lpc   - The current map level counter.
#     hash  - Reference to the hash that we are building.
#
sub parse_mapalias_param {
    my ($token, $lpc, $hash) = @_;

    # Fully qualify the to value and ignore the alias if there is no
    # corresponding resource.

    my $referid=$lpc.'.'.$token->[2]->{'to'};
    return if (!exists($hash->{'src_'.$referid}));

    # If this is a valid mapalias parameter, 
    # Append the target id to the count_mapalias element for that
    # alias so that we can detect doubly defined aliases
    # e.g.:
    #  <param to="1" value="george" name="parameter_0_mapalias" type="string" />
    #  <param to="2" value="george" name="parameter_0_mapalias" type="string" />
    #
    #  The example above is trivial but the case that's important has to do with
    #  constructing a map that includes a nested map where the nested map may have
    #  aliases that conflict with aliases established in the enclosing map.
    #
    # ...and create/update the hash mapalias entry to actually store the alias.
    #

    if ($token->[2]->{'name'}=~/^parameter_(0_)*mapalias$/) {
	&count_mapalias($token->[2]->{'value'},$referid);
	$hash->{'mapalias_'.$token->[2]->{'value'}}=$referid;
    }
}


#---------------------------------------------------------------------------------
#
#  Code to process the map file.

#  read a map file and add it to the hash.  Since a course map can contain resources
#  that are themselves maps, read_map might be recursively called.
#
# Parameters:
#   $uri         - URI of the course itself (not the map file).
#   $parent_rid  - map number qualified id of the parent of the map being read.
#                  For the top level course map this is 0.0.  For the first nested
#                  map 1.n  where n is the id of the resource within the
#                  top level map and so on.
#   $code        - CODE for which map is being read (CODEd assignments).
#   $hash        - Reference to a hash that will become the big hash for the course
#                  This hash is modified as per the map description.
# Side-effects:
#   $map_number - Will be  incremented.   This keeps track of the number of the map
#                 we are currently working on (see parent_rid above, the number to the
#                 left of the . in $parent_rid is the map number).
#
#  
sub read_map {
    my ($uri, $parent_rid, $code, $hash) = @_;


    # Check for duplication: A map may only be included once.

    if($hash->{'map_pc_' . $uri}) {
	throw Error::Simple('Duplicate map: ', $uri);
    }
    # count the map number and save it locally so that we don't lose it
    # when we recurse.

    $map_number++;
    my $lmap_no = $map_number;

    # save the map_pc and map_id elements of the hash for this map:
    #  map_pc_uri is the map number of the map with that URI.
    #  map_id_$lmap_no is the URI for this map level.
    #
    $hash->{'map_pc_' . $uri}     = "$lmap_no"; # string form in lonuserstate.
    $hash->{'map_id_' . $lmap_no} = "$uri";

    # Create the path up to the top of the course.
    # this is in 'map_hierarchy_mapno'  that's a comma separated path down to us
    # in the hierarchy:

    if ($parent_rid =~/^(\d+).\d+$/) { 
	my $parent_no = $1;	       # Parent's map number.
	if (defined($hash->{'map_hierarchy_' . $parent_no})) {
	    $hash->{'map_hierarchy_' . $lmap_no} =
		$hash->{'map_hierarchy_' . $parent_no} . ',' . $parent_no;
	} else {
	    # Only 1 level deep ..nothing to append to:

	    $hash->{'map_hierarchy_' . $lmap_no} = $parent_no;
	}
    }

    # figure out the name of the map file we need to read.
    # ensure that it is a .page or a .sequence as those are the only 
    # sorts of files that make sense for this sub 

    my $filename = &Apache::lonnet::filelocation('', &append_version($uri, $hash));


    my $ispage = ($filename =~/\.page$/);
    unless ($ispage || ($filename =~ /\.sequence$/)) {
	&Apache::lonnet::logthis("invalid: $filename : $uri");
	throw Error::Simple('<br />'.&mt('Invalid map: [_1]','<span class="LC_filename">'.$filename.'</span>'));
    }

    $filename =~ /\.(\w+)$/;

    $hash->{'map_type_'.$lmap_no}=$1;

    # Repcopy the file and get its contents...report errors if we can't
   
    my $contents = &Apache::lonnet::getfile($filename);
    if($contents eq -1) {
        throw Error::Simple('<br />'.&mt('Map not loaded: The file [_1] does not exist.',
				'<span class="LC_filename">'.$filename.'</span>'));
    }
    # Now that we succesfully retrieved the file we can make our parsing passes over it:
    # parsing is done in passes:
    # 1. Parameters are parsed.
    # 2. Resource, links and conditions are parsed.
    #
    # post processing takes care of the case where the sequence is random ordered
    # or randomselected.

    # Parse the parameters,  This pass only cares about start tags for <param>
    # tags.. this is because there is no body to a <param> tag.
    #

    my $parser  = HTML::TokeParser->new(\$contents);
    $parser->attr_encoded(1);	# Don't interpret entities in attributes (leave &xyz; alone).

    while (my $token = $parser->get_token()) {
	if (($token->[0] eq 'S') && ($token->[1] eq 'param')) { 
	    &parse_param($token, $map_number, $hash);
	}
    }

    # ready for pass 2: Resource links and conditions.
    # Note that if the map is random-ordered link tags are computed by randomizing
    # resource order.  Furthermore, since conditions are set on links rather than
    # resources, they are also not processed if random order is turned on.
    #

    $parser = HTML::TokeParser->new(\$contents); # no way to reset the existing parser
    $parser->attr_encoded(1);

    my $linkpc=0;
    my $randomize = ($randomorder{$parent_rid} =~ /^yes$/i);

    my @map_ids;
    while (my $token = $parser->get_token) {
	next if ($token->[0] ne 'S');

	# Resource

	if ($token->[1] eq 'resource') {
	    my $resource_id = &parse_resource($token,$lmap_no,$ispage,$uri,$code,$hash);
	    if (defined $resource_id) {
		push(@map_ids, $resource_id); 
	    }

       # Link

	} elsif ($token->[1] eq 'link' && !$randomize) {
	    &make_link(++$linkpc,$lmap_no,$token->[2]->{'to'},
		       $token->[2]->{'from'},
		       $token->[2]->{'condition'}, $hash); # note ..condition may be undefined.

	# condition

	} elsif ($token->[1] eq 'condition' && !$randomize) {
	    &parse_condition($token,$lmap_no, $hash);
	}
    }

    #  This section handles random ordering by permuting the 
    # IDs of the map according to the user's random seed.
    # 

    if ($randomize) {
	if (!&has_advanced_role($username, $userdomain) || $code) {
	    my $seed;

	    # In the advanced role, the map's random seed
	    # parameter is used as the basis for computing the
	    # seed ... if it has been specified:

	    if (defined($randompickseed{$parent_rid})) {
		$seed = $randompickseed{$parent_rid};
	    } else {

		# Otherwise the parent's fully encoded symb is used.

		my ($mapid,$resid)=split(/\./,$parent_rid);
		my $symb=
		    &Apache::lonnet::encode_symb($hash->{'map_id_'.$mapid},
						 $resid,$hash->{'src_'.$parent_rid});
		
		$seed = $symb;
	    }


	    my $rndseed=&Apache::lonnet::rndseed($seed, '', 
						 $userdomain, $username,
						 \%cenv);
						 

	    &Apache::lonnet::setup_random_from_rndseed($rndseed);

	    # Take the set of map ids we have decoded and permute them to a
	    # random order based on the seed set above. All of this is
	    # processing the randomorder parameter if it is set, not
	    # randompick.

	    @map_ids=&Math::Random::random_permutation(@map_ids); 
	}
	my $from = shift(@map_ids);
	my $from_rid = $lmap_no.'.'.$from;
	$hash->{'map_start_'.$uri} = $from_rid;
	$hash->{'type_'.$from_rid}='start';

	# Create links to reflect the random re-ordering done above.
	# In the code to process the map XML, we did not process links or conditions
	# if randomorder was set.  This means that for an instructor to choose

	while (my $to = shift(@map_ids)) {
	    &make_link(++$linkpc,$lmap_no,$to,$from, 0, $hash);
	    my $to_rid =  $lmap_no.'.'.$to;
	    $hash->{'type_'.$to_rid}='normal';
	    $from = $to;
	    $from_rid = $to_rid;
	}

	$hash->{'map_finish_'.$uri}= $from_rid;
	$hash->{'type_'.$from_rid}='finish';
    }


    #  The last parsing pass parses the <mapalias> tags that associate a name
    #  with resource ids.

    $parser = HTML::TokeParser->new(\$contents);
    $parser->attr_encoded(1);

    while (my $token = $parser->get_token) {
	next if ($token->[0] ne 'S');
	if ($token->[1] eq 'param') {
	    &parse_mapalias_param($token,$lmap_no, $hash);  
	} 
    }

}


#
#  Load a map from file into a target hash.  This is done by first parsing the 
#  map file into local hashes and then unrolling those hashes into the big hash.
# 
# Parameters:
#
#    $cnum       - number of course being read.
#    $cdom       - Domain in which the course is evaluated.
#    $uname      - Name of the user for whom the course is being read
#    $udom       - Name of the domain of the user for whom the course is being read.
#    $code       - CODE for which course is being read (CODEd assignments)
#    $nohideurl  - URL for an exam folder for which hidden state is to be ignored.
#    $target_hash- Reference to the target hash into which all of this is read.
#                  Note tht some of the hash entries we need to build require knowledge of the
#                  course URI.. these are expected to be filled in by the caller.
#
# Errors are logged to lonnet and are managed via the Perl structured exception package.
#
#  
sub loadmap {
    my ($cnum, $cdom, $uname, $udom, $code, $nohideurl, $target_hash) = @_;



    # Clear the auxiliary hashes and the cond array.


    %randompick     = ();
    %randompickseed = ();
    %encurl         = ();
    %hiddenurl      = ();
    %parmhash       = ();
    @cond           = ('true:normal'); # Initial value for cond 0.
    $retfrid        = '';
    $username       = '';
    $userdomain     = '';
    %mapalias_cache = ();
    %cenv           = ();
    $map_number     =  0;
    
    # 

    $username   = $uname;
    $userdomain = $udom;

    $short_name = $cdom .'/' . $cnum;
    my $retfurl;

    try {

	
	# Get the information we need about the course.
 	# Return without filling in anything if we can't get any info:
 	
 	%cenv = &Apache::lonnet::coursedescription($short_name,
 						     {'freshen_cache' => 1,
 						      'user'          => $uname}); 
 
 	unless ($cenv{'url'}) {
 	    &Apache::lonnet::logthis("lonmap::loadmap failed: $cnum/$cdom - did not get url");
 	    return; 
 	}
 
 	$course_id = $cdom . '_' . $cnum; # Long course id.
 
 	# Load the version information into the hash
 
 	
	&process_versions(\%cenv, $target_hash);
	
	
	# Figure out the map filename's URI, and set up some starting points for the map.
	
	my $course_uri = $cenv{'url'};
	my $map_uri    = &Apache::lonnet::clutter($course_uri);
	
	$target_hash->{'src_0.0'}            = &versiontrack($map_uri, $target_hash); 
	$target_hash->{'title_0.0'}          = &Apache::lonnet::metadata($course_uri, 'title');
	if(!defined $target_hash->{'title_0.0'}) {
	    $target_hash->{'title_0.0'} = '';
	}
	$target_hash->{'ids_'.$map_uri} = '0.0';
	$target_hash->{'is_map_0.0'}         = '1';

	# In some places we need a username a domain and the courseid...store that
	# in the target hash in the context.xxxx keys:

	$target_hash->{'context.username'} = $username;
	$target_hash->{'context.userdom'}  = $userdomain;
	$target_hash->{'context.courseid'} = $course_id;
 
        # When grading or printing a bubblesheet exam ignore
        # "hidden" parameter set in the map containing the exam folder.
        $target_hash->{'context.nohideurl'} = $nohideurl;

        &read_map($course_uri, '0.0', $code, $target_hash);

	if (defined($target_hash->{'map_start_'.$map_uri})) {

	    &traceroute('0',$target_hash->{'map_start_'.$course_uri},'&', 0, 0, $target_hash);
	    &accinit($course_uri, $short_name,  $target_hash);
	    &hiddenurls($target_hash);
	}
	my $errors = &get_mapalias_errors($target_hash);
	if ($errors ne "") {
	    throw Error::Simple("Map alias errors: ", $errors);
	}

	# Put the versions in to src:

	foreach my $key (keys(%$target_hash)) {
	    if ($key =~ /^src_/) {
		$target_hash->{$key} = 
		    &putinversion($target_hash->{$key}, $target_hash, $short_name);
	    } elsif ($key =~ /^(map_(?:start|finish|pc)_)(.*)/) {
		my ($type, $url) = ($1,$2);
		my $value = $target_hash->{$key};
		$target_hash->{$type.&putinversion($url, $target_hash, $short_name)}=$value;
	    }
	}
	#  Mark necrypted URLS.

	foreach my $id (keys(%encurl)) {
	    $target_hash->{'encrypted_'.$id}=1;
	}

	# Store first keys.

	$target_hash->{'first_rid'}=$retfrid;
	my ($mapid,$resid)=split(/\./,$retfrid);
	$target_hash->{'first_mapurl'}=$target_hash->{'map_id_'.$mapid};
	my $symb=&Apache::lonnet::encode_symb($target_hash->{'map_id_'.$mapid},
					      $resid,
					      $target_hash->{'src_'.$retfrid});
	$retfurl=&add_get_param($target_hash->{'src_'.$retfrid},{ 'symb' => $symb });
	if ($target_hash->{'encrypted_'.$retfrid}) {
	    $retfurl=&Apache::lonenc::encrypted($retfurl,
						(&Apache::lonnet::allowed('adv') ne 'F'));
	}
	$target_hash->{'first_url'}=$retfurl;	

	# Merge in the child hashes in case the caller wants that information as well.


	&merge_hash($target_hash, 'randompick', \%randompick);
	&merge_hash($target_hash, 'randompickseed', \%randompickseed);
	&merge_hash($target_hash, 'randomorder', \%randomorder);
	&merge_hash($target_hash, 'encurl', \%encurl);
	&merge_hash($target_hash, 'hiddenurl', \%hiddenurl);
	&merge_hash($target_hash, 'param', \%parmhash);
	&merge_conditions($target_hash);
    }
    otherwise {
	my $e = shift;
	&Apache::lonnet::logthis("lonmap::loadmap failed: " . $e->stringify());
    }

}


1;

#
#  Module initialization code:
#  TODO:  Fix the pod docs below.

1;
__END__

=head1 NAME

Apache::lonmap - Construct a hash that represents a course (Big Hash).

=head1 SYNOPSIS

&Apache::lonmap::loadmap($cnum, $cdom, $uname, $udom, $code, $nohideurl, \%target_hash);

=head1 INTRODUCTION

This module reads a course filename into a hash reference.  It's up to the caller
to do things like decide that the hash should be tied to some external file and handle the
the locking if this file should be shared amongst several Apache children.

=head1 SUBROUTINES

=over

=item loadmap($cnum, $cdom, $uname, $udom, $code, $nohideurl, $targethash)


Reads the top-level map file into a target hash. This is done by first parsing the
map file into local hashes and then unrolling those hashes into the big hash.

=over

=item $cnum - number of course being read.

=item $cdom - domain in which the course is evaluated.

=item $uname - name of the user for whom the course is being read.

=item $udom  - name of the domain of the user for whom the course is being read.

=item $code  - CODE for which course is being read (CODEd assignments).

=item $nohideurl - URL for an exam folder for which hidden state is to be ignored.

=item $targethash - A reference to hash into which the course is read

=back

=item process_versions($cenv, $hash)

Makes hash entries for each version of a course described by a course environment
returned from Apache::lonnet::coursedescription.

=over

=item $cenv - Reference to the environment hash returned by Apache::lonnet::coursedescription

=item $hash - Hash to be filled in with 'version_xxx' entries as per the big hash.

=back

=back
 

=cut
