# The LearningOnline Network with CAPA
# Construct and maintain state and binary representation of course for user
#
# $Id: lonuserstate.pm,v 1.149.2.5 2021/12/14 22:52:33 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
###

package Apache::lonuserstate;

# ------------------------------------------------- modules used by this module
use strict;
use HTML::TokeParser;
use Apache::lonnet;
use Apache::lonlocal;
use Apache::loncommon();
use GDBM_File;
use Apache::lonmsg;
use Safe;
use Safe::Hole;
use Opcode;
use Apache::lonenc;
use Fcntl qw(:flock);
use LONCAPA;
use File::Basename;

 

# ---------------------------------------------------- Globals for this package

my $pc;      # Package counter is this what 'Guts' calls the map counter?
my %hash;    # The big tied hash
my %parmhash;# The hash with the parameters
my @cond;    # Array with all of the conditions
my $errtext; # variable with all errors
my $retfrid; # variable with the very first RID in the course
my $retfurl; # first URL
my %randompick; # randomly picked resources
my %randompickseed; # optional seed for randomly picking resources
my %randomorder; # maps to order contents randomly
my %randomizationcode; # code used to grade folder for bubblesheet exam 
my %encurl; # URLs in this folder are supposed to be encrypted
my %hiddenurl; # this URL (or complete folder) is supposed to be hidden
my %rescount; # count of unhidden items in each map
my %mapcount; # count of unhidden maps in each map

# ----------------------------------- Remove version from URL and store in hash

sub versionerror {
    my ($uri,$usedversion,$unusedversion)=@_;
    return '<br />'.&mt('Version discrepancy: resource [_1] included in both version [_2] and version [_3]. Using version [_2].',
                    $uri,$usedversion,$unusedversion).'<br />';
}

#  Removes the version number from a URI and returns the resulting
#  URI (e.g. mumbly.version.stuff => mumbly.stuff).
#
#   If the URI has not been seen with a versio before the
#   hash{'version_'.resultingURI} is set to the  version number.
#   If the URI has been seen and the version does not match and error
#   is added to the error string.
#
# Parameters:
#   URI potentially with a version.
# Returns:
#   URI with the version cut out.
# See above for side effects.
#

sub versiontrack {
    my $uri=shift;
    if ($uri=~/\.(\d+)\.\w+$/) {
	my $version=$1;
	$uri=~s/\.\d+\.(\w+)$/\.$1/;
        unless ($hash{'version_'.$uri}) {
	    $hash{'version_'.$uri}=$version;
	} elsif ($version!=$hash{'version_'.$uri}) {
            $errtext.=&versionerror($uri,$hash{'version_'.$uri},$version);
        }
    }
    return $uri;
}

# -------------------------------------------------------------- Put in version

sub putinversion {
    my $uri=shift;
    my $key=$env{'request.course.id'}.'_'.&Apache::lonnet::clutter($uri);
    if ($hash{'version_'.$uri}) {
	my $version=$hash{'version_'.$uri};
	if ($version eq 'mostrecent') { return $uri; }
	if ($version eq &Apache::lonnet::getversion(
			&Apache::lonnet::filelocation('',$uri))) 
	             { return $uri; }
	$uri=~s/\.(\w+)$/\.$version\.$1/;
    }
    &Apache::lonnet::do_cache_new('courseresversion',$key,&Apache::lonnet::declutter($uri),600);
    return $uri;
}

# ----------------------------------------- Processing versions file for course

sub processversionfile {
    my %cenv=@_;
    my %versions=&Apache::lonnet::dump('resourceversions',
				       $cenv{'domain'},
				       $cenv{'num'});
    foreach my $ver (keys(%versions)) {
	if ($ver=~/^error\:/) { return; }
	$hash{'version_'.$ver}=$versions{$ver};
    }
}

# --------------------------------------------------------- Loads from disk


#
#  Loads a map file.
#  Note that this may implicitly recurse via parse_resource if one of the resources
#  is itself composed.
#
# Parameters:
#    uri         - URI of the map file.
#    parent_rid  - Resource id in the map of the parent resource (0.0 for the top level map)
#    courseid    - Course id for the course for which the map is being loaded
#
sub loadmap { 
    my ($uri,$parent_rid,$courseid)=@_;

    # Is the map already included?

    if ($hash{'map_pc_'.$uri}) { 
	$errtext.='<p class="LC_error">'.
	    &mt('Multiple use of sequence/page [_1]! The course will not function properly.','<tt>'.$uri.'</tt>').
	    '</p>';
	return; 
    }
    # Register the resource in it's map_pc_ [for the URL]
    # map_id.nnn is the nesting level -> to the URI.

    $pc++;
    my $lpc=$pc;
    $hash{'map_pc_'.$uri}=$lpc;
    $hash{'map_id_'.$lpc}=$uri;

    # If the parent is of the form n.m hang this map underneath it in the
    # map hierarchy.

    if ($parent_rid =~ /^(\d+)\.\d+$/) {
        my $parent_pc = $1;
        if (defined($hash{'map_hierarchy_'.$parent_pc})) {
            $hash{'map_hierarchy_'.$lpc}=$hash{'map_hierarchy_'.$parent_pc}.','.
                                         $parent_pc;
        } else {
            $hash{'map_hierarchy_'.$lpc}=$parent_pc;
        }
    }

# Determine and check filename of the sequence we need to read:

    my $fn=&Apache::lonnet::filelocation('',&putinversion($uri));

    my $ispage=($fn=~/\.page$/);

    # We can only nest sequences or pages.  Anything else is an illegal nest.

    unless (($fn=~/\.sequence$/) || $ispage) { 
	$errtext.='<br />'.&mt('Invalid map: [_1]',"<tt>$fn</tt>");
	return; 
    }

    # Read the XML that constitutes the file.

    my $instr=&Apache::lonnet::getfile($fn);

    if ($instr eq -1) {
        $errtext.= '<br />'
                  .&mt('Map not loaded: The file [_1] does not exist.',
                       "<tt>$fn</tt>");
	return;
    }

    # Successfully got file, parse it

    # parse for parameter processing.
    # Note that these are <param... / > tags
    # so we only care about 'S' (tag start) nodes.


    my $parser = HTML::TokeParser->new(\$instr);
    $parser->attr_encoded(1);

    # first get all parameters


    while (my $token = $parser->get_token) {
	next if ($token->[0] ne 'S');
	if ($token->[1] eq 'param') {
	    &parse_param($token,$lpc);
	} 
    }

    # Get set to take another pass through the XML:
    # for resources and links.

    $parser = HTML::TokeParser->new(\$instr);
    $parser->attr_encoded(1);

    my $linkpc=0;

    $fn=~/\.(\w+)$/;

    $hash{'map_type_'.$lpc}=$1;

    my $randomize = ($randomorder{$parent_rid} =~ /^yes$/i);

    # Parse the resources, link and condition tags.
    # Note that if randomorder or random select is chosen the links and
    # conditions are meaningless but are determined by the randomization.
    # This is handled in the next chunk of code.

    my @map_ids;
    my $codechecked;
    $rescount{$lpc} = 0;
    $mapcount{$lpc} = 0;
    while (my $token = $parser->get_token) {
	next if ($token->[0] ne 'S');

	# Resource

	if ($token->[1] eq 'resource') {
	    my $resource_id = &parse_resource($token,$lpc,$ispage,$uri,$courseid);
	    if (defined $resource_id) {
		push(@map_ids, $resource_id);
                if ($hash{'src_'.$lpc.'.'.$resource_id}) {
                    $rescount{$lpc} ++;
                    if (($hash{'src_'.$lpc.'.'.$resource_id}=~/\.sequence$/) ||
                        ($hash{'src_'.$lpc.'.'.$resource_id}=~/\.page$/)) {
                        $mapcount{$lpc} ++;
                    }
                }
                unless ($codechecked) {
                    my $startsymb =
                       &Apache::lonnet::encode_symb($hash{'map_id_'.$lpc},$resource_id,
                                                    $hash{'src_'."$lpc.$resource_id"});
                    my $code = 
                        &Apache::lonnet::EXT('resource.0.examcode',$startsymb,undef,undef,
                                             undef,undef,$courseid);
                    if ($code) {
                        $randomizationcode{$parent_rid} = $code;
                    }
                    $codechecked = 1; 
                }
	    }

       # Link

	} elsif ($token->[1] eq 'link' && !$randomize) {
	    &make_link(++$linkpc,$lpc,$token->[2]->{'to'},
		       $token->[2]->{'from'},
		       $token->[2]->{'condition'}); # note ..condition may be undefined.

	# condition

	} elsif ($token->[1] eq 'condition' && !$randomize) {
	    &parse_condition($token,$lpc);
	}
    }
    undef($codechecked);

    # Handle randomization and random selection

    if ($randomize) {
        unless (&is_advanced($courseid)) {
            # Order of resources is not randomized if user has and advanced role in the course.
	    my $seed;

            # If the map's random seed parameter has been specified
            # it is used as the basis for computing the seed ...

	    if (defined($randompickseed{$parent_rid})) {
		$seed = $randompickseed{$parent_rid};
	    } else {

		# Otherwise the parent's fully encoded symb is used.

		my ($mapid,$resid)=split(/\./,$parent_rid);
		my $symb=
		    &Apache::lonnet::encode_symb($hash{'map_id_'.$mapid},
						 $resid,$hash{'src_'.$parent_rid});
		
		$seed = $symb;
	    }

	    # TODO: Here for sure we need to pass along the username/domain
	    # so that we can impersonate users in lonprintout e.g.

            my $setcode;
            if (defined($randomizationcode{$parent_rid})) {
                if ($env{'form.CODE'} eq '') {
                    $env{'form.CODE'} = $randomizationcode{$parent_rid};
                    $setcode = 1;
                }
            }

	    my $rndseed=&Apache::lonnet::rndseed($seed);
	    &Apache::lonnet::setup_random_from_rndseed($rndseed);

            if ($setcode) {
                undef($env{'form.CODE'});
                undef($setcode);
            }

	    # Take the set of map ids we have decoded and permute them to a
	    # random order based on the seed set above. All of this is
	    # processing the randomorder parameter if it is set, not
	    # randompick.

	    @map_ids=&Math::Random::random_permutation(@map_ids);
	}

	my $from = shift(@map_ids);
	my $from_rid = $lpc.'.'.$from;
	$hash{'map_start_'.$uri} = $from_rid;
	$hash{'type_'.$from_rid}='start';

	# Create links to reflect the random re-ordering done above.
	# In the code to process the map XML, we did not process links or conditions
	# if randomorder was set.  This means that for an instructor to choose

	while (my $to = shift(@map_ids)) {
	    &make_link(++$linkpc,$lpc,$to,$from);
	    my $to_rid =  $lpc.'.'.$to;
	    $hash{'type_'.$to_rid}='normal';
	    $from = $to;
	    $from_rid = $to_rid;
	}

	$hash{'map_finish_'.$uri}= $from_rid;
	$hash{'type_'.$from_rid}='finish';
    }

    $parser = HTML::TokeParser->new(\$instr);
    $parser->attr_encoded(1);

    # last parse out the mapalias params.  These provide mnemonic
    # tags to resources that can be used in conditions

    while (my $token = $parser->get_token) {
	next if ($token->[0] ne 'S');
	if ($token->[1] eq 'param') {
	    &parse_mapalias_param($token,$lpc);
	} 
    }
}

sub is_advanced {
    my ($courseid) = @_;
    my $advanced;
    if ($env{'request.course.id'}) {
        $advanced = (&Apache::lonnet::allowed('adv') eq 'F');
    } else {
        $env{'request.course.id'} = $courseid;
        $advanced = (&Apache::lonnet::allowed('adv') eq 'F');
        $env{'request.course.id'} = '';
    }
    return $advanced;
}

# -------------------------------------------------------------------- Resource
#
#  Parses a resource tag to produce the value to push into the
#  map_ids array.
# 
#
#  Information about the actual type of resource is provided by the file extension
#  of the uri (e.g. .problem, .sequence etc. etc.).
#
#  Parameters:
#    $token   - A token from HTML::TokeParser
#               This is an array that describes the most recently parsed HTML item.
#    $lpc     - Map nesting level (?)
#    $ispage  - True if this resource is encapsulated in a .page (assembled resourcde).
#    $uri     - URI of the enclosing resource.
#    $courseid - Course id of the course containing the resource being parsed. 
# Returns:
#   Value of the id attribute of the tag.
#
# Note:
#   The token is an array that contains the following elements:
#   [0]   => 'S' indicating this is a start token
#   [1]   => 'resource'  indicating this tag is a <resource> tag.
#   [2]   => Hash of attribute =>value pairs.
#   [3]   => @(keys [2]).
#   [4]   => unused.
#
#   The attributes of the resourcde tag include:
#
#   id     - The resource id.
#   src    - The URI of the resource.
#   type   - The resource type (e.g. start and finish).
#   title  - The resource title.


sub parse_resource {
    my ($token,$lpc,$ispage,$uri,$courseid) = @_;
    
    # I refuse to countenance code like this that has 
    # such a dirty side effect (and forcing this sub to be called within a loop).
    #
    #  if ($token->[2]->{'type'} eq 'zombie') { next; }
    #
    #  The original code both returns _and_ skips to the next pass of the >caller's<
    #  loop, that's just dirty.
    #

    # Zombie resources don't produce anything useful.

    if ($token->[2]->{'type'} eq 'zombie') {
	return undef;
    }

    my $rid=$lpc.'.'.$token->[2]->{'id'}; # Resource id in hash is levelcounter.id-in-xml.

    # Save the hash element type and title:
	    
    $hash{'kind_'.$rid}='res';
    $hash{'title_'.$rid}=$token->[2]->{'title'};

    # Get the version free URI for the resource.
    # If a 'version' attribute was supplied, and this resource's version 
    # information has not yet been stored, store it.
    #

    my $turi=&versiontrack($token->[2]->{'src'});
    if ($token->[2]->{'version'}) {
	unless ($hash{'version_'.$turi}) {
	    $hash{'version_'.$turi}=$1;
	}
    }
    # Pull out the title and do entity substitution on &colon
    # Q: Why no other entity substitutions?

    my $title=$token->[2]->{'title'};
    $title=~s/\&colon\;/\:/gs;



    # I think the point of all this code is to construct a final
    # URI that apache and its rewrite rules can use to
    # fetch the resource.   Thi s sonly necessary if the resource
    # is not a page.  If the resource is a page then it must be
    # assembled (at fetch time?).

    if ($ispage) {
        if ($token->[2]->{'external'} eq 'true') { # external
            $turi=~s{^http\://}{/ext/};
        }
    } else {
	$turi=~/\.(\w+)$/;
	my $embstyle=&Apache::loncommon::fileembstyle($1);
	if ($token->[2]->{'external'} eq 'true') { # external
	    $turi=~s/^https?\:\/\//\/adm\/wrapper\/ext\//;
	} elsif ($turi=~/^\/*uploaded\//) { # uploaded
	    if (($embstyle eq 'img') 
		|| ($embstyle eq 'emb')
		|| ($embstyle eq 'wrp')) {
		$turi='/adm/wrapper'.$turi;
	    } elsif ($embstyle eq 'ssi') {
		#do nothing with these
	    } elsif ($turi!~/\.(sequence|page)$/) {
		$turi='/adm/coursedocs/showdoc'.$turi;
	    }
	} elsif ($turi=~/\S/) { # normal non-empty internal resource
	    my $mapdir=$uri;
	    $mapdir=~s/[^\/]+$//;
	    $turi=&Apache::lonnet::hreflocation($mapdir,$turi);
	    if (($embstyle eq 'img') 
		|| ($embstyle eq 'emb')
		|| ($embstyle eq 'wrp')) {
		$turi='/adm/wrapper'.$turi;
	    }
	}
    }
    # Store reverse lookup, remove query string resource 'ids'_uri => resource id.
    # If the URI appears more than one time in the sequence, it's resourcde
    # id's are constructed as a comma spearated list.

    my $idsuri=$turi;
    $idsuri=~s/\?.+$//;
    if (defined($hash{'ids_'.$idsuri})) {
	$hash{'ids_'.$idsuri}.=','.$rid;
    } else {
	$hash{'ids_'.$idsuri}=''.$rid;
    }
    


    if ($turi=~/\/(syllabus|aboutme|navmaps|smppg|bulletinboard|viewclasslist)$/) {
	$turi.='?register=1';
    }
    

    # resource id lookup:  'src'_resourc-di  => URI decorated with a query
    # parameter as above if necessary due to the resource type.
    
    $hash{'src_'.$rid}=$turi;

    # Mark the external-ness of the resource:
    
    if ($token->[2]->{'external'} eq 'true') {
	$hash{'ext_'.$rid}='true:';
    } else {
	$hash{'ext_'.$rid}='false:';
    }

    # If the resource is a start/finish resource set those
    # entries in the has so that navigation knows where everything starts.
    # TODO?  If there is a malformed sequence that has no start or no finish
    # resource, should this be detected and errors thrown?  How would such a 
    # resource come into being other than being manually constructed by a person
    # and then uploaded?  Could that happen if an author decided a sequence was almost
    # right edited it by hand and then reuploaded it to 'fix it' but accidently cut the
    #  start or finish resources?
    #
    #  All resourcess also get a type_id => (start | finish | normal)    hash entr.
    #
    if ($token->[2]->{'type'}) {
	$hash{'type_'.$rid}=$token->[2]->{'type'};
	if ($token->[2]->{'type'} eq 'start') {
	    $hash{'map_start_'.$uri}="$rid";
	}
	if ($token->[2]->{'type'} eq 'finish') {
	    $hash{'map_finish_'.$uri}="$rid";
	}
    }  else {
	$hash{'type_'.$rid}='normal';
    }

    # Sequences end pages are constructed entities.  They require that the 
    # map that defines _them_ be loaded as well into the hash...with this resourcde
    # as the base of the nesting.
    # Resources like that are also marked with is_map_id => 1 entries.
    #
    
    if (($turi=~/\.sequence$/) ||
	($turi=~/\.page$/)) {
	$hash{'is_map_'.$rid}=1;
	if ((!$hiddenurl{$rid}) || (&is_advanced($courseid))) {
	    &loadmap($turi,$rid,$courseid);
	}
    } 
    return $token->[2]->{'id'};
}

#-------------------------------------------------------------------- link
#  Links define how you are allowed to move from one resource to another.
#  They are the transition edges in the directed graph that a map is.
#  This sub takes informatino from a <link> tag and constructs the
#  navigation bits and pieces of a map.  There is no requirement that the
#  resources that are linke are already defined, however clearly the map is 
#  badly broken if they are not _eventually_ defined.
#
#  Note that links can be unconditional or conditional.
#
#  Parameters:
#     linkpc   - The link counter for this level of map nesting (this is 
#                reset to zero by loadmap prior to starting to process
#                links for map).
#     lpc      - The map level ocounter (how deeply nested this map is in
#                the hierarchy of maps that are recursively read in.
#     to       - resource id (within the XML) of the target of the edge.
#     from     - resource id (within the XML) of the source of the edge.
#     condition- id of condition associated with the edge (also within the XML).
#

sub make_link {
    my ($linkpc,$lpc,$to,$from,$condition) = @_;
    
    #  Compute fully qualified ids for the link, the 
    # and from/to by prepending lpc.
    #

    my $linkid=$lpc.'.'.$linkpc;
    my $goesto=$lpc.'.'.$to;
    my $comesfrom=$lpc.'.'.$from;
    my $undercond=0;


    # If there is a condition, qualify it with the level counter.

    if ($condition) {
	$undercond=$lpc.'.'.$condition;
    }

    # Links are represnted by:
    #  goesto_.fuullyqualifedlinkid => fully qualified to
    #  comesfrom.fullyqualifiedlinkid => fully qualified from
    #  undercond_.fullyqualifiedlinkid => fully qualified condition id.

    $hash{'goesto_'.$linkid}=$goesto;
    $hash{'comesfrom_'.$linkid}=$comesfrom;
    $hash{'undercond_'.$linkid}=$undercond;

    # In addition:
    #   to_.fully qualified from => comma separated list of 
    #   link ids with that from.
    # Similarly:
    #   from_.fully qualified to => comma separated list of link ids`
    #                               with that to.
    #  That allows us given a resource id to know all edges that go to it
    #  and leave from it.
    #

    if (defined($hash{'to_'.$comesfrom})) {
	$hash{'to_'.$comesfrom}.=','.$linkid;
    } else {
	$hash{'to_'.$comesfrom}=''.$linkid;
    }
    if (defined($hash{'from_'.$goesto})) {
	$hash{'from_'.$goesto}.=','.$linkid;
    } else {
	$hash{'from_'.$goesto}=''.$linkid;
    }
}

# ------------------------------------------------------------------- Condition
#
#  Processes <condition> tags, storing sufficient information about them
#  in the hash so that they can be evaluated and used to conditionalize
#  what is presented to the student.
#
#  these can have the following attributes 
#
#    id    = A unique identifier of the condition within the map.
#
#    value = Is a perl script-let that, when evaluated in safe space
#            determines whether or not the condition is true.
#            Normally this takes the form of a test on an  Apache::lonnet::EXT call
#            to find the value of variable associated with a resource in the
#            map identified by a mapalias.
#            Here's a fragment of XML code that illustrates this:
#
#           <param to="5" value="mainproblem" name="parameter_0_mapalias" type="string" />
#           <resource src="" id="1" type="start" title="Start" />
#           <resource src="/res/msu/albertel/b_and_c/p1.problem" id="5"  title="p1.problem" />
#           <condition value="&EXT('user.resource.resource.0.tries','mainproblem')
#           <2 " id="61" type="stop" />
#           <link to="5" index="1" from="1" condition="61" />    
#
#           In this fragment:
#             - The param tag establishes an alias to resource id 5 of 'mainproblem'.
#             - The resource that is the start of the map is identified.
#             - The resource tag identifies the resource associated with this tag
#               and gives it the id 5.
#             - The condition is true if the tries variable associated with mainproblem
#               is less than 2 (that is the user has had more than 2 tries).
#               The condition type is a stop condition which inhibits(?) the associated
#               link if the condition  is false. 
#             - The link to resource 5 from resource 1 is affected by this condition.    
#            
#    type  = Type of the condition. The type determines how the condition affects the
#            link associated with it and is one of
#            -  'force'
#            -  'stop'
#              anything else including not supplied..which treated as:
#            - 'normal'.
#            Presumably maps get created by the resource assembly tool and therefore
#            illegal type values won't squirm their way into the XML.
#
# Side effects:
#   -  The kind_level-qualified-condition-id hash element is set to 'cond'.
#   -  The condition text is pushed into the cond array and its element number is
#      set in the condid_level-qualified-condition-id element of the hash.
#   - The condition type is colon appneded to the cond array element for this condition.
sub parse_condition {
    my ($token,$lpc) = @_;
    my $rid=$lpc.'.'.$token->[2]->{'id'};
    
    $hash{'kind_'.$rid}='cond';

    my $condition = $token->[2]->{'value'};
    $condition =~ s/[\n\r]+/ /gs;
    push(@cond, $condition);
    $hash{'condid_'.$rid}=$#cond;
    if ($token->[2]->{'type'}) {
	$cond[$#cond].=':'.$token->[2]->{'type'};
    }  else {
	$cond[$#cond].=':normal';
    }
}

# ------------------------------------------------------------------- Parameter
# Parse a <parameter> tag in the map.
# Parmameters:
#    $token Token array for a start tag from HTML::TokeParser
#           [0] = 'S'
#           [1] = tagname ("param")
#           [2] = Hash of {attribute} = values.
#           [3] = Array of the keys in [2].
#           [4] = unused.
#    $lpc   Current map nesting level.a
#
#  Typical attributes:
#     to=n      - Number of the resource the parameter applies to.
#     type=xx   - Type of parameter value (e.g. string_yesno or int_pos).
#     name=xxx  - Name of parameter (e.g. parameter_randompick or parameter_randomorder).
#     value=xxx - value of the parameter.

sub parse_param {
    my ($token,$lpc) = @_;
    my $referid=$lpc.'.'.$token->[2]->{'to'}; # Resource param applies to.
    my $name=$token->[2]->{'name'};	      # Name of parameter
    my $part;


    if ($name=~/^parameter_(.*)_/) { 
	$part=$1;
    } else {
	$part=0;
    }

    # Peel the parameter_ off the parameter name.

    $name=~s/^.*_([^_]*)$/$1/;

    # The value is:
    #   type.part.name.value

    my $newparam=
	&escape($token->[2]->{'type'}).':'.
	&escape($part.'.'.$name).'='.
	&escape($token->[2]->{'value'});

    # The hash key is param_resourceid.
    # Multiple parameters for a single resource are & separated in the hash.


    if (defined($hash{'param_'.$referid})) {
	$hash{'param_'.$referid}.='&'.$newparam;
    } else {
	$hash{'param_'.$referid}=''.$newparam;
    }
    #
    #  These parameters have to do with randomly selecting
    # resources, therefore a separate hash is also created to 
    # make it easy to locate them when actually computing the resource set later on
    # See the code conditionalized by ($randomize) in loadmap().

    if ($token->[2]->{'name'}=~/^parameter_(0_)*randompick$/) { # Random selection turned on
	$randompick{$referid}=$token->[2]->{'value'};
    }
    if ($token->[2]->{'name'}=~/^parameter_(0_)*randompickseed$/) { # Randomseed provided.
	$randompickseed{$referid}=$token->[2]->{'value'};
    }
    if ($token->[2]->{'name'}=~/^parameter_(0_)*randomorder$/) { # Random order turned on.
	$randomorder{$referid}=$token->[2]->{'value'};
    }

    # These parameters have to do with how the URLs of resources are presented to
    # course members(?).  encrypturl presents encypted url's while
    # hiddenresource hides the URL.
    #

    if ($token->[2]->{'name'}=~/^parameter_(0_)*encrypturl$/) {
	if ($token->[2]->{'value'}=~/^yes$/i) {
	    $encurl{$referid}=1;
	}
    }
    if ($token->[2]->{'name'}=~/^parameter_(0_)*hiddenresource$/) {
	if ($token->[2]->{'value'}=~/^yes$/i) {
	    $hiddenurl{$referid}=1;
	}
    }
}
#
#  Parse mapalias parameters.
#  these are tags of the form:
#  <param to="nn" 
#         value="some-alias-for-resourceid-nn" 
#         name="parameter_0_mapalias" 
#         type="string" />
#  A map alias is a textual name for a resource:
#    - The to  attribute identifies the resource (this gets level qualified below)
#    - The value attributes provides the alias string.
#    - name must be of the regexp form: /^parameter_(0_)*mapalias$/
#    - e.g. the string 'parameter_' followed by 0 or more "0_" strings
#      terminating with the string 'mapalias'.
#      Examples:
#         'parameter_mapalias', 'parameter_0_mapalias', parameter_0_0_mapalias'
#  Invalid to ids are silently ignored.
#
#  Parameters:
#     token - The token array fromthe HMTML::TokeParser
#     lpc   - The current map level counter.
#
sub parse_mapalias_param {
    my ($token,$lpc) = @_;

    # Fully qualify the to value and ignore the alias if there is no
    # corresponding resource.

    my $referid=$lpc.'.'.$token->[2]->{'to'};
    return if (!exists($hash{'src_'.$referid}));

    # If this is a valid mapalias parameter, 
    # Append the target id to the count_mapalias element for that
    # alias so that we can detect doubly defined aliases
    # e.g.:
    #  <param to="1" value="george" name="parameter_0_mapalias" type="string" />
    #  <param to="2" value="george" name="parameter_0_mapalias" type="string" />
    #
    #  The example above is trivial but the case that's important has to do with
    #  constructing a map that includes a nested map where the nested map may have
    #  aliases that conflict with aliases established in the enclosing map.
    #
    # ...and create/update the hash mapalias entry to actually store the alias.
    #

    if ($token->[2]->{'name'}=~/^parameter_(0_)*mapalias$/) {
	&count_mapalias($token->[2]->{'value'},$referid);
	$hash{'mapalias_'.$token->[2]->{'value'}}=$referid;
    }
}

# --------------------------------------------------------- Simplify expression


#
#  Someone should really comment this to describe what it does to what and why.
#
sub simplify {
    my $expression=shift;
# (0&1) = 1
    $expression=~s/\(0\&([_\.\d]+)\)/$1/g;
# (8)=8
    $expression=~s/\(([_\.\d]+)\)/$1/g;
# 8&8=8
    $expression=~s/([^_\.\d])([_\.\d]+)\&\2([^_\.\d])/$1$2$3/g;
# 8|8=8
    $expression=~s/([^_\.\d])([_\.\d]+)(?:\|\2)+([^_\.\d])/$1$2$3/g;
# (5&3)&4=5&3&4
    $expression=~s/\(([_\.\d]+)((?:\&[_\.\d]+)+)\)\&([_\.\d]+[^_\.\d])/$1$2\&$3/g;
# (((5&3)|(4&6)))=((5&3)|(4&6))
    $expression=~
	s/\((\(\([_\.\d]+(?:\&[_\.\d]+)*\)(?:\|\([_\.\d]+(?:\&[_\.\d]+)*\))+\))\)/$1/g;
# ((5&3)|(4&6))|(1&2)=(5&3)|(4&6)|(1&2)
    $expression=~
	s/\((\([_\.\d]+(?:\&[_\.\d]+)*\))((?:\|\([_\.\d]+(?:\&[_\.\d]+)*\))+)\)\|(\([_\.\d]+(?:\&[_\.\d]+)*\))/\($1$2\|$3\)/g;
    return $expression;
}

# -------------------------------------------------------- Build condition hash

#
#  Traces a route recursively through the map after it has been loaded
#  (I believe this really visits each resourcde that is reachable fromt he
#  start top node.
#
#  - Marks hidden resources as hidden.
#  - Marks which resource URL's must be encrypted.
#  - Figures out (if necessary) the first resource in the map.
#  - Further builds the chunks of the big hash that define how 
#    conditions work
#
#  Note that the tracing strategy won't visit resources that are not linked to
#  anything or islands in the map (groups of resources that form a path but are not
#  linked in to the path that can be traced from the start resource...but that's ok
#  because by definition, those resources are not reachable by users of the course.
#
# Parameters:
#   sofar    - _URI of the prior entry or 0 if this is the top.
#   rid      - URI of the resource to visit.
#   beenhere - list of resources (each resource enclosed by &'s) that have
#              already been visited.
#   encflag  - If true the resource that resulted in a recursive call to us
#              has an encoded URL (which means contained resources should too). 
#   hdnflag  - If true,the resource that resulted in a recursive call to us
#              was hidden (which means contained resources should be hidden too).
# Returns
#    new value indicating how far the map has been traversed (the sofar).
#
sub traceroute {
    my ($sofar,$rid,$beenhere,$encflag,$hdnflag)=@_;
    my $newsofar=$sofar=simplify($sofar);

    unless ($beenhere=~/\&\Q$rid\E\&/) {
	$beenhere.=$rid.'&';  
	my ($mapid,$resid)=split(/\./,$rid);
	my $symb=&Apache::lonnet::encode_symb($hash{'map_id_'.$mapid},$resid,$hash{'src_'.$rid});
	my $hidden=&Apache::lonnet::EXT('resource.0.hiddenresource',$symb);

	if ($hdnflag || lc($hidden) eq 'yes') {
	    $hiddenurl{$rid}=1;
	}
	if (!$hdnflag && lc($hidden) eq 'no') {
	    delete($hiddenurl{$rid});
	}

	my $encrypt=&Apache::lonnet::EXT('resource.0.encrypturl',$symb);
	if ($encflag || lc($encrypt) eq 'yes') { $encurl{$rid}=1; }

	if (($retfrid eq '') && ($hash{'src_'.$rid})
	    && ($hash{'src_'.$rid}!~/\.sequence$/)) {
	    $retfrid=$rid;
	}

	if (defined($hash{'conditions_'.$rid})) {
	    $hash{'conditions_'.$rid}=simplify(
           '('.$hash{'conditions_'.$rid}.')|('.$sofar.')');
	} else {
	    $hash{'conditions_'.$rid}=$sofar;
	}

	# if the expression is just the 0th condition keep it
	# otherwise leave a pointer to this condition expression

	$newsofar = ($sofar eq '0') ? $sofar : '_'.$rid;

	# Recurse if the resource is a map:

	if (defined($hash{'is_map_'.$rid})) {
	    if (defined($hash{'map_start_'.$hash{'src_'.$rid}})) {
		$sofar=$newsofar=
		    &traceroute($sofar,
				$hash{'map_start_'.$hash{'src_'.$rid}},
				$beenhere,
				$encflag || $encurl{$rid},
				$hdnflag || $hiddenurl{$rid});
	    }
	}

	# Processes  links to this resource:
	#  - verify the existence of any conditionals on the link to here.
	#  - Recurse to any resources linked to us.
	#
	if (defined($hash{'to_'.$rid})) {
	    foreach my $id (split(/\,/,$hash{'to_'.$rid})) {
		my $further=$sofar;
		#
		# If there's a condition associated with this link be sure
		# it's been defined else that's an error:
		#
                if ($hash{'undercond_'.$id}) {
		    if (defined($hash{'condid_'.$hash{'undercond_'.$id}})) {
			$further=simplify('('.'_'.$rid.')&('.
					  $hash{'condid_'.$hash{'undercond_'.$id}}.')');
		    } else {
			$errtext.= '<br />'.
                                   &mt('Undefined condition ID: [_1]',
                                       $hash{'undercond_'.$id});
		    }
                }
		#  Recurse to resoruces that have to's to us.
                $newsofar=&traceroute($further,$hash{'goesto_'.$id},$beenhere,
				      $encflag,$hdnflag);
	    }
	}
    }
    return $newsofar;
}

# ------------------------------ Cascading conditions, quick access, parameters

#
#  Seems a rather strangely named sub given what the comment above says it does.


sub accinit {
    my ($uri,$short,$fn)=@_;
    my %acchash=();
    my %captured=();
    my $condcounter=0;
    $acchash{'acc.cond.'.$short.'.0'}=0;

    # This loop is only interested in conditions and 
    # parameters in the big hash:

    foreach my $key (keys(%hash)) {

	# conditions:

	if ($key=~/^conditions/) {
	    my $expr=$hash{$key};

	    # try to find and factor out common sub-expressions
	    # Any subexpression that is found is simplified, removed from
	    # the original condition expression and the simplified sub-expression
	    # substituted back in to the epxression..I'm not actually convinced this
	    # factors anything out...but instead maybe simplifies common factors(?)

	    foreach my $sub ($expr=~m/(\(\([_\.\d]+(?:\&[_\.\d]+)+\)(?:\|\([_\.\d]+(?:\&[_\.\d]+)+\))+\))/g) {
		my $orig=$sub;

		my ($factor) = ($sub=~/\(\(([_\.\d]+\&(:?[_\.\d]+\&)*)(?:[_\.\d]+\&*)+\)(?:\|\(\1(?:[_\.\d]+\&*)+\))+\)/);
		next if (!defined($factor));

		$sub=~s/\Q$factor\E//g;
		$sub=~s/^\(/\($factor\(/;
		$sub.=')';
		$sub=simplify($sub);
		$expr=~s/\Q$orig\E/$sub/;
	    }
	    $hash{$key}=$expr;

           # If not yet seen, record in acchash and that we've seen it.

	    unless (defined($captured{$expr})) {
		$condcounter++;
		$captured{$expr}=$condcounter;
		$acchash{'acc.cond.'.$short.'.'.$condcounter}=$expr;
	    } 
        # Parameters:

	} elsif ($key=~/^param_(\d+)\.(\d+)/) {
	    my $prefix=&Apache::lonnet::encode_symb($hash{'map_id_'.$1},$2,
						    $hash{'src_'.$1.'.'.$2});
	    foreach my $param (split(/\&/,$hash{$key})) {
		my ($typename,$value)=split(/\=/,$param);
		my ($type,$name)=split(/\:/,$typename);
		$parmhash{$prefix.'.'.&unescape($name)}=
		    &unescape($value);
		$parmhash{$prefix.'.'.&unescape($name).'.type'}=
		    &unescape($type);
	    }
	}
    }
    # This loop only processes id entries in the big hash.

    foreach my $key (keys(%hash)) {
	if ($key=~/^ids/) {
	    foreach my $resid (split(/\,/,$hash{$key})) {
		my $uri=$hash{'src_'.$resid};
		my ($uripath,$urifile) =
		    &Apache::lonnet::split_uri_for_cond($uri);
		if ($uripath) {
		    my $uricond='0';
		    if (defined($hash{'conditions_'.$resid})) {
			$uricond=$captured{$hash{'conditions_'.$resid}};
		    }
		    if (defined($acchash{'acc.res.'.$short.'.'.$uripath})) {
			if ($acchash{'acc.res.'.$short.'.'.$uripath}=~
			    /(\&\Q$urifile\E\:[^\&]*)/) {
			    my $replace=$1;
			    my $regexp=$replace;
			    #$regexp=~s/\|/\\\|/g;
			    $acchash{'acc.res.'.$short.'.'.$uripath} =~
				s/\Q$regexp\E/$replace\|$uricond/;
			} else {
			    $acchash{'acc.res.'.$short.'.'.$uripath}.=
				$urifile.':'.$uricond.'&';
			}
		    } else {
			$acchash{'acc.res.'.$short.'.'.$uripath}=
			    '&'.$urifile.':'.$uricond.'&';
		    }
		} 
	    }
	}
    }
    $acchash{'acc.res.'.$short.'.'}='&:0&';
    my $courseuri=$uri;
    $courseuri=~s/^\/res\///;
    my $regexp = 1;
    &Apache::lonnet::delenv('(acc\.|httpref\.)',$regexp);
    &Apache::lonnet::appenv(\%acchash);
}

# ---------------- Selectively delete from randompick maps and hidden url parms

sub hiddenurls {
    my $randomoutentry='';
    foreach my $rid (keys(%randompick)) {
        my $rndpick=$randompick{$rid};
        my $mpc=$hash{'map_pc_'.$hash{'src_'.$rid}};
# ------------------------------------------- put existing resources into array
        my @currentrids=();
        foreach my $key (sort(keys(%hash))) {
	    if ($key=~/^src_($mpc\.\d+)/) {
		if ($hash{'src_'.$1}) { push @currentrids, $1; }
            }
        }
	# rids are number.number and we want to numercially sort on 
        # the second number
	@currentrids=sort {
	    my (undef,$aid)=split(/\./,$a);
	    my (undef,$bid)=split(/\./,$b);
	    $aid <=> $bid;
	} @currentrids;
        next if ($#currentrids<$rndpick);
# -------------------------------- randomly eliminate the ones that should stay
	my (undef,$id)=split(/\./,$rid);
        if ($randompickseed{$rid}) { $id=$randompickseed{$rid}; }
        my $setcode;
        if (defined($randomizationcode{$rid})) {
            if ($env{'form.CODE'} eq '') {
                $env{'form.CODE'} = $randomizationcode{$rid};
                $setcode = 1;
            }
        }
	my $rndseed=&Apache::lonnet::rndseed($id); # use id instead of symb
        if ($setcode) {
            undef($env{'form.CODE'});
            undef($setcode);
        }
	&Apache::lonnet::setup_random_from_rndseed($rndseed);
	my @whichids=&Math::Random::random_permuted_index($#currentrids+1);
        for (my $i=1;$i<=$rndpick;$i++) { $currentrids[$whichids[$i]]=''; }
	#&Apache::lonnet::logthis("$id,$rndseed,".join(':',@whichids));
# -------------------------------------------------------- delete the leftovers
        for (my $k=0; $k<=$#currentrids; $k++) {
            if ($currentrids[$k]) {
		$hash{'randomout_'.$currentrids[$k]}=1;
                my ($mapid,$resid)=split(/\./,$currentrids[$k]);
                if ($rescount{$mapid}) {
                    $rescount{$mapid} --;
                }
                if ($hash{'is_map_'.$currentrids[$k]}) {
                    if ($mapcount{$mapid}) {
                        $mapcount{$mapid} --;
                    }
                }
                $randomoutentry.='&'.
		    &Apache::lonnet::encode_symb($hash{'map_id_'.$mapid},
						 $resid,
						 $hash{'src_'.$currentrids[$k]}
						 ).'&';
            }
        }
    }
# ------------------------------ take care of explicitly hidden urls or folders
    foreach my $rid (keys(%hiddenurl)) {
	$hash{'randomout_'.$rid}=1;
	my ($mapid,$resid)=split(/\./,$rid);
        if ($rescount{$mapid}) {
            $rescount{$mapid} --;
        }
        if ($hash{'is_map_'.$rid}) {
            if ($mapcount{$mapid}) {
                $mapcount{$mapid} --;
            }
        }
	$randomoutentry.='&'.
	    &Apache::lonnet::encode_symb($hash{'map_id_'.$mapid},$resid,
					 $hash{'src_'.$rid}).'&';
    }
# --------------------------------------- append randomout entry to environment
    if ($randomoutentry) {
	&Apache::lonnet::appenv({'acc.randomout' => $randomoutentry});
    }
}

# -------------------------------------- populate big hash with map breadcrumbs

# Create map_breadcrumbs_$pc from map_hierarchy_$pc by omitting intermediate
# maps not shown in Course Contents table.

sub mapcrumbs {
    foreach my $key (keys(%rescount)) {
        if ($hash{'map_hierarchy_'.$key}) {
            my $skipnext = 0;
            foreach my $id (split(/,/,$hash{'map_hierarchy_'.$key}),$key) {
                unless ($skipnext) {
                    $hash{'map_breadcrumbs_'.$key} .= "$id,";
                }
                unless (($id == 0) || ($id == 1)) {
                    if ((!$rescount{$id}) || ($rescount{$id} == 1 && $mapcount{$id} == 1)) {
                        $skipnext = 1;
                    } else {
                        $skipnext = 0;
                    }
                }
            }
            $hash{'map_breadcrumbs_'.$key} =~ s/,$//;
        }
    }
}

# ---------------------------------------------------- Read map and all submaps

sub readmap {
    my ($short,$critmsg_check) = @_;
    $short=~s/^\///;

    # TODO:  Hidden dependency on current user:

    my %cenv=&Apache::lonnet::coursedescription($short,{'freshen_cache'=>1}); 

    my $fn=$cenv{'fn'};
    my $uri;
    $short=~s/\//\_/g;
    unless ($uri=$cenv{'url'}) { 
	&Apache::lonnet::logthis('<font color="blue">WARNING: '.
				 "Could not load course $short.</font>"); 
	return ('',&mt('No course data available.'));;
    }
    @cond=('true:normal');

    unless (open(LOCKFILE,">","$fn.db.lock")) {
	# 
	# Most likely a permissions problem on the lockfile or its directory.
	#
        $retfurl = '';
        return ($retfurl,'<br />'.&mt('Map not loaded - Lock file could not be opened when reading map:').' <tt>'.$fn.'</tt>.');
    }
    my $lock=0;
    my $gotstate=0;
    
    # If we can get the lock without delay any files there are idle
    # and from some prior request.  We'll kill them off and regenerate them:

    if (flock(LOCKFILE,LOCK_EX|LOCK_NB)) {	
	$lock=1;		# Remember that we hold the lock.
        &unlink_tmpfiles($fn);
    }
    undef %randompick;
    undef %randompickseed;
    undef %randomorder;
    undef %randomizationcode;
    undef %hiddenurl;
    undef %encurl;
    undef %rescount;
    undef %mapcount;
    $retfrid='';
    $errtext='';
    my ($untiedhash,$untiedparmhash,$tiedhash,$tiedparmhash); # More state flags.

    # if we got the lock, regenerate course regnerate empty files and tie them.

    if ($lock) {
        if (tie(%hash,'GDBM_File',"$fn.db",&GDBM_WRCREAT(),0640)) {
            $tiedhash = 1;
            if (tie(%parmhash,'GDBM_File',$fn.'_parms.db',&GDBM_WRCREAT(),0640)) {
                $tiedparmhash = 1;
                $gotstate = &build_tmp_hashes($uri,
					      $fn,
					      $short,
					      \%cenv); # TODO: Need to provide requested user@dom
                unless ($gotstate) {
                    &Apache::lonnet::logthis('Failed to write statemap at first attempt '.$fn.' for '.$uri.'.</font>');
                }
                $untiedparmhash = untie(%parmhash);
                unless ($untiedparmhash) {
                    &Apache::lonnet::logthis('<font color="blue">WARNING: '.
                        'Could not untie coursemap parmhash '.$fn.' for '.$uri.'.</font>');
                }
            }
            $untiedhash = untie(%hash);
            unless ($untiedhash) {
                &Apache::lonnet::logthis('<font color="blue">WARNING: '.
                    'Could not untie coursemap hash '.$fn.' for '.$uri.'.</font>');
            }
        }
	flock(LOCKFILE,LOCK_UN); # RF: this is what I don't get unless there are other
	                         # unlocked places the remainder happens..seems like if we
                                 # just kept the lock here the rest of the code would have
                                 # been much easier? 
    }
    unless ($lock && $tiedhash && $tiedparmhash) { 
	# if we are here it is likely because we are already trying to 
	# initialize the course in another child, busy wait trying to 
	# tie the hashes for the next 90 seconds, if we succeed forward 
	# them on to navmaps, if we fail, throw up the Could not init 
	# course screen
	#
	# RF: I'm not seeing the case where the ties/unties can fail in a way
	#     that can be remedied by this.  Since we owned the lock seems
	#     Tie/untie failures are a result of something like a permissions problem instead?
	#

	#  In any vent, undo what we did manage to do above first:
	if ($lock) {
	    # Got the lock but not the DB files
	    flock(LOCKFILE,LOCK_UN);
            $lock = 0;
	}
        if ($tiedhash) {
            unless($untiedhash) {
	        untie(%hash);
            }
        }
        if ($tiedparmhash) {
            unless($untiedparmhash) {
                untie(%parmhash);
            }
        }
	# Log our failure:

	&Apache::lonnet::logthis('<font color="blue">WARNING: '.
				 "Could not tie coursemap $fn for $uri.</font>");
        $tiedhash = '';
        $tiedparmhash = '';
	my $i=0;

	# Keep on retrying the lock for 90 sec until we succeed.

	while($i<90) {
	    $i++;
	    sleep(1);
	    if (flock(LOCKFILE,LOCK_EX|LOCK_NB)) {

		# Got the lock, tie the hashes...the assumption in this code is
		# that some other worker thread has created the db files quite recently
		# so no load is needed:

                $lock = 1;
		if (tie(%hash,'GDBM_File',"$fn.db",&GDBM_READER(),0640)) {
                    $tiedhash = 1;
		    if (tie(%parmhash,'GDBM_File',$fn.'_parms.db',&GDBM_READER(),0640)) {
                        $tiedparmhash = 1;
                        if (-e "$fn.state") {
		            $retfurl='/adm/navmaps';

			    # BUG BUG: Side effect!
			    # Should conditionalize on something so that we can use this
			    # to load maps for courses that are not current?
			    #
		            &Apache::lonnet::appenv({"request.course.id"  => $short,
		   			             "request.course.fn"  => $fn,
					             "request.course.uri" => $uri,
                                                     "request.course.tied" => time});
                            
		            $untiedhash = untie(%hash);
		            $untiedparmhash = untie(%parmhash);
                            $gotstate = 1;
		            last;
		        }
                        $untiedparmhash = untie(%parmhash);
	            }
	            $untiedhash = untie(%hash);
                }
            }
	}
        if ($lock) {
            flock(LOCKFILE,LOCK_UN);
            $lock = 0;
            if ($tiedparmhash) {
                unless ($untiedparmhash) {
                    &Apache::lonnet::logthis('<font color="blue">WARNING: '.
                        'Could not untie coursemap parmhash '.$fn.' for '.$uri.'.</font>');
                }
            }
            if ($tiedparmhash) {
                unless ($untiedhash) {
                    &Apache::lonnet::logthis('<font color="blue">WARNING: '.
                        'Could not untie coursemap hash '.$fn.' for '.$uri.'.</font>');
                }
            }
        }
    }
    # I think this branch of code is all about what happens if we just did the stuff above, 
    # but found that the  state file did not exist...again if we'd just held the lock
    # would that have made this logic simpler..as generating all the files would be
    # an atomic operation with respect to the lock.
    #
    unless ($gotstate) {
        $lock = 0;
        &Apache::lonnet::logthis('<font color="blue">WARNING: '.
                     'Could not read statemap '.$fn.' for '.$uri.'.</font>');
        &unlink_tmpfiles($fn);
        if (flock(LOCKFILE,LOCK_EX|LOCK_NB)) {
            $lock=1;
        }
        undef %randompick;
        undef %randompickseed;
        undef %randomorder;
        undef %randomizationcode;
        undef %hiddenurl;
        undef %encurl;
        undef %rescount;
        undef %mapcount;
        $errtext='';
        $retfrid='';
	#
	# Once more through the routine of tying and loading and so on.
	#
        if ($lock) {
            if (tie(%hash,'GDBM_File',"$fn.db",&GDBM_WRCREAT(),0640)) {
                if (tie(%parmhash,'GDBM_File',$fn.'_parms.db',&GDBM_WRCREAT(),0640)) {
                    $gotstate = &build_tmp_hashes($uri,$fn,$short,\%cenv); # TODO: User dependent?
                    unless ($gotstate) {
                        &Apache::lonnet::logthis('<font color="blue">WARNING: '.
                            'Failed to write statemap at second attempt '.$fn.' for '.$uri.'.</font>');
                    }
                    unless (untie(%parmhash)) {
                        &Apache::lonnet::logthis('<font color="blue">WARNING: '.
                            'Could not untie coursemap parmhash '.$fn.'.db for '.$uri.'.</font>');
                    }
                } else {
                    &Apache::lonnet::logthis('<font color="blue">WARNING: '.
                        'Could not tie coursemap '.$fn.'__parms.db for '.$uri.'.</font>');
                }
                unless (untie(%hash)) {
                    &Apache::lonnet::logthis('<font color="blue">WARNING: '.
                        'Could not untie coursemap hash '.$fn.'.db for '.$uri.'.</font>');
                }
            } else {
               &Apache::lonnet::logthis('<font color="blue">WARNING: '.
                   'Could not tie coursemap '.$fn.'.db for '.$uri.'.</font>');
            }
            flock(LOCKFILE,LOCK_UN);
            $lock = 0;
        } else {
	    # Failed to get the immediate lock.

            &Apache::lonnet::logthis('<font color="blue">WARNING: '.
            'Could not obtain lock to tie coursemap hash '.$fn.'.db for '.$uri.'.</font>');
        }
    }
    close(LOCKFILE);
    unless (($errtext eq '') || ($env{'request.course.uri'} =~ m{^/uploaded/})) {
        &Apache::lonmsg::author_res_msg($env{'request.course.uri'},
                                        $errtext); # TODO: User dependent?
    }
# ------------------------------------------------- Check for critical messages

#  Depends on user must parameterize this as well..or separate as this is:
#  more part of determining what someone sees on entering a course?
#  When lonuserstate::readmap() is called from lonroles.pm, i.e.,
#  after selecting a role in a course, critical_redirect will be called,
#  unless the course has a blocking event in effect, which suppresses
#  critical message checking (users without evb priv).
#

    if ($critmsg_check) {
        my ($redirect,$url) = &Apache::loncommon::critical_redirect();
        if ($redirect) {
            $retfurl = $url;
        }
    } 
    return ($retfurl,$errtext);
}

#
#  This sub is called when the course hash and the param hash have been tied and
#  their lock file is held.
#  Parameters:
#     $uri      -  URI that identifies the course.
#     $fn       -  The base path/filename of the files that make up the context
#                  being built.
#     $short    -  Short course name.
#     $cenvref  -  Reference to the course environment hash returned by 
#                  Apache::lonnet::coursedescription
#
#  Assumptions:
#    The globals
#    %hash, %paramhash are tied to their gdbm files and we hold the lock on them.
#
sub build_tmp_hashes {
    my ($uri,$fn,$short,$cenvref) = @_;
    
    unless(ref($cenvref) eq 'HASH') {
        return;
    }
    my %cenv = %{$cenvref};
    my $gotstate = 0;
    %hash=();			# empty the global course and  parameter hashes.
    %parmhash=();
    $errtext='';		# No error messages yet.
    $pc=0;
    &clear_mapalias_count();
    &processversionfile(%cenv);

    # URI Of the map file.

    my $furi=&Apache::lonnet::clutter($uri);
    #
    #  the map staring points.
    #
    $hash{'src_0.0'}=&versiontrack($furi);
    $hash{'title_0.0'}=&Apache::lonnet::metadata($uri,'title');
    $hash{'ids_'.$furi}='0.0';
    $hash{'is_map_0.0'}=1;

    # Load the map.. note that loadmap may implicitly recurse if the map contains 
    # sub-maps.


    &loadmap($uri,'0.0',$short);

    #  The code below only executes if there is a starting point for the map>
    #  Q/BUG??? If there is no start resource for the map should that be an error?
    #

    if (defined($hash{'map_start_'.$uri})) {
        &Apache::lonnet::appenv({"request.course.id"  => $short,
                                 "request.course.fn"  => $fn,
                                 "request.course.uri" => $uri,
                                 "request.course.tied" => time});
        $env{'request.course.id'}=$short;
        &traceroute('0',$hash{'map_start_'.$uri},'&');
        &accinit($uri,$short,$fn);
        &hiddenurls();
        &mapcrumbs();
    }
    $errtext .= &get_mapalias_errors();
# ------------------------------------------------------- Put versions into src
    foreach my $key (keys(%hash)) {
        if ($key=~/^src_/) {
            $hash{$key}=&putinversion($hash{$key});
        } elsif ($key =~ /^(map_(?:start|finish|pc)_)(.*)/) {
            my ($type, $url) = ($1,$2);
            my $value = $hash{$key};
            $hash{$type.&putinversion($url)}=$value;
        }
    }
# ---------------------------------------------------------------- Encrypt URLs
    foreach my $id (keys(%encurl)) {
#           $hash{'src_'.$id}=&Apache::lonenc::encrypted($hash{'src_'.$id});
        $hash{'encrypted_'.$id}=1;
    }
# ----------------------------------------------- Close hashes to finally store
# --------------------------------- Routine must pass this point, no early outs
    $hash{'first_rid'}=$retfrid;
    my ($mapid,$resid)=split(/\./,$retfrid);
    $hash{'first_mapurl'}=$hash{'map_id_'.$mapid};
    my $symb=&Apache::lonnet::encode_symb($hash{'map_id_'.$mapid},$resid,$hash{'src_'.$retfrid});
    $retfurl=&add_get_param($hash{'src_'.$retfrid},{ 'symb' => $symb });
    if ($hash{'encrypted_'.$retfrid}) {
        $retfurl=&Apache::lonenc::encrypted($retfurl,(&Apache::lonnet::allowed('adv') ne 'F'));
    }
    $hash{'first_url'}=$retfurl;
# ---------------------------------------------------- Store away initial state
    {
        my $cfh;
        if (open($cfh,">","$fn.state")) {
            print $cfh join("\n",@cond);
            $gotstate = 1;
        } else {
            &Apache::lonnet::logthis("<font color=blue>WARNING: ".
                                     "Could not write statemap $fn for $uri.</font>");
        }
    }
    return $gotstate;
}

sub unlink_tmpfiles {
    my ($fn) = @_;
    my $file_dir = dirname($fn);

    if ("$file_dir/" eq LONCAPA::tempdir()) {
        my @files = qw (.db _symb.db .state _parms.db);
        foreach my $file (@files) {
            if (-e $fn.$file) {
                unless (unlink($fn.$file)) {
                    &Apache::lonnet::logthis("<font color=blue>WARNING: ".
                                 "Could not unlink ".$fn.$file."</font>");
                }
            }
        }
    }
    return;
}

# ------------------------------------------------------- Evaluate state string

sub evalstate {
    my $fn=$env{'request.course.fn'}.'.state';
    my $state='';
    if (-e $fn) {
	my @conditions=();
	{
	    open(my $fh,"<",$fn);
	    @conditions=<$fh>;
            close($fh);
	}  
	my $safeeval = new Safe;
	my $safehole = new Safe::Hole;
	$safeeval->permit("entereval");
	$safeeval->permit(":base_math");
	$safeeval->deny(":base_io");
	$safehole->wrap(\&Apache::lonnet::EXT,$safeeval,'&EXT');
	foreach my $line (@conditions) {
	    chomp($line);
	    my ($condition,$weight)=split(/\:/,$line);
	    if ($safeeval->reval($condition)) {
		if ($weight eq 'force') {
		    $state.='3';
		} else {
		    $state.='2';
		}
	    } else {
		if ($weight eq 'stop') {
		    $state.='0';
		} else {
		    $state.='1';
		}
	    }
	}
    }
    &Apache::lonnet::appenv({'user.state.'.$env{'request.course.id'} => $state});
    return $state;
}

#  This block seems to have code to manage/detect doubly defined
#  aliases in maps.

{
    my %mapalias_cache;
    sub count_mapalias {
	my ($value,$resid) = @_;
 	push(@{ $mapalias_cache{$value} }, $resid);
    }

    sub get_mapalias_errors {
	my $error_text;
	foreach my $mapalias (sort(keys(%mapalias_cache))) {
	    next if (scalar(@{ $mapalias_cache{$mapalias} } ) == 1);
	    my $count;
	    my $which =
		join('</li><li>', 
		     map {
			 my $id = $_;
			 if (exists($hash{'src_'.$id})) {
			     $count++;
			 }
			 my ($mapid) = split(/\./,$id);
                         &mt('Resource [_1][_2]in Map [_3]',
			     $hash{'title_'.$id},'<br />',
			     $hash{'title_'.$hash{'ids_'.$hash{'map_id_'.$mapid}}});
		     } (@{ $mapalias_cache{$mapalias} }));
	    next if ($count < 2);
	    $error_text .= '<div class="LC_error">'.
		&mt('Error: Found the mapalias "[_1]" defined multiple times.',
		    $mapalias).
		'</div><ul><li>'.$which.'</li></ul>';
	}
	&clear_mapalias_count();
	return $error_text;
    }
    sub clear_mapalias_count {
	undef(%mapalias_cache);
    }
}
1;
__END__

=head1 NAME

Apache::lonuserstate - Construct and maintain state and binary representation
of course for user

=head1 SYNOPSIS

Invoked by lonroles.pm.

&Apache::lonuserstate::readmap($cdom.'/'.$cnum);

=head1 INTRODUCTION

This module constructs and maintains state and binary representation
of course for user.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 SUBROUTINES

=over

=item loadmap()

Loads map from disk

=item simplify()

Simplify expression

=item traceroute()

Build condition hash

=item accinit()

Cascading conditions, quick access, parameters

=item readmap()

Read map and all submaps

=item evalstate()

Evaluate state string

=back

=cut
