# The LearningOnline Network with CAPA
# Wrapper for external and binary files as standalone resources
#
# $Id: lonwrapper.pm,v 1.49.2.11 2021/12/15 00:41:48 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#


package Apache::lonwrapper;

use strict;
use Apache::Constants qw(:common);
use Apache::lonenc();
use Apache::lonnet;
use Apache::lonlocal;
use Apache::loncommon();
use Apache::lonhtmlcommon();
use Apache::lonextresedit();

# ================================================================ Main Handler
sub wrapper {
    my ($r,$url,$brcrum,$absolute,$is_ext,$is_pdf,$title) = @_;

    my $forcereg;
    unless ($env{'form.folderpath'}) {
        $forcereg = 1;
    }

    my %lt = &Apache::lonlocal::texthash(
                                          'noif' => 'No iframe support.',
                                          'show' => 'Show content in pop-up window',
                                        );

    (undef,undef,undef,undef,undef,undef,my $clientmobile) =
        &Apache::loncommon::decode_user_agent($r);

    my ($anchor,$uselink,$linktext);
    if ($is_ext) {
        if ($env{'form.symb'}) {
            (undef,undef,my $res) = &Apache::lonnet::decode_symb($env{'form.symb'});
            if ($res =~ /(#[^#]+)$/) {
                $anchor = $1;
            }
        } elsif ($env{'form.anchor'} ne '') {
            $anchor = '#'.$env{'form.anchor'};
        }
        unless (($is_pdf) && ($clientmobile)) {
            my $hostname = $r->hostname();
            my $lonhost = $r->dir_config('lonHostID');
            my $ip = &Apache::lonnet::get_host_ip($lonhost);
            $uselink = &Apache::loncommon::is_nonframeable($url,$absolute,$hostname,$ip);
        }
    }

    my $noiframe = &Apache::loncommon::modal_link($url.$anchor,$lt{'show'},500,400);
    my $args = {'bgcolor' => '#FFFFFF'};
    if ($forcereg) {
        $args->{'force_register'} = $forcereg;
    }
    if (ref($brcrum) eq 'ARRAY') {
        $args->{'bread_crumbs'} = $brcrum;
    }
    if ($absolute) {
        $args->{'use_absolute'} = $absolute;
    }
    if ($env{'form.only_body'}) {
        $args->{'only_body'} = $env{'form.only_body'};
    }

    my $headjs;

#
# Where iframe is in use, if window.onload() executes before the custom resize function
# has been defined (jQuery), two global javascript vars (LCnotready and LCresizedef)
# are used to ensure document.ready() triggers a call to resize, so the iframe contents
# do not obscure the Functions menu.
#

    unless ($clientmobile || $uselink) {
        $headjs = '
<script type="text/javascript">
// <![CDATA[
var LCnotready = 0;
var LCresizedef = 0;
// ]]>
</script>'."\n";
    }

    my $startpage = &Apache::loncommon::start_page('Menu',$headjs,$args);
    my $endpage = &Apache::loncommon::end_page();

    if (($uselink) && ($title eq '')) {
        if ($env{'form.symb'}) {
            $title=&Apache::lonnet::gettitle($env{'form.symb'});
        } else {
            my $symb=&Apache::lonnet::symbread($r->uri);
            if ($symb) {
                $title=&Apache::lonnet::gettitle($symb);
            }
        }
    }
    if ($clientmobile) {
        my $output = $startpage;
        if ($is_pdf) {
            $linktext = &mt('Link to PDF (for mobile devices)');
            $output .= &create_link($url,$anchor,$title,$linktext);
        } else {
            if ($uselink) {
                $linktext = &mt('Link to resource');
                $output .= &create_link($url,$anchor,$title,$linktext);
            } else {
                my $dest = &HTML::Entities::encode($url.$anchor,'&<>"');
                $output .= '<div style="overflow:scroll; -webkit-overflow-scrolling:touch;">'."\n".
                           '<iframe src="'.$dest.'" height="100%" width="100%" frameborder="0">'."\n".
                           "$lt{'noif'} $noiframe\n".
                           "</iframe>\n".
                           "</div>\n";
            }
        }
        $output .= $endpage;
        return $output;
    } elsif ($uselink) {
        $linktext = &mt('Link to resource');
        return $startpage.&create_link($url,$anchor,$title,$linktext).$endpage;
    } else {
        my $offset = 5;
        &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},['inhibitmenu']);
        if ($env{'form.inhibitmenu'} eq 'yes') {
            $offset = 0;
        }
        my $script = &Apache::lonhtmlcommon::scripttag(<<SCRIPT);
        \$(document).ready( function() {
            \$(window).unbind('resize').resize(function(){
                var header = null;
                var offset = $offset;
                var height = 0;
                var hdrtop = 0;
                if (\$('div.LC_head_subbox:first').length) {
                    header = \$('div.LC_head_subbox:first');
                    offset = 9;
                } else {
                    if (\$('#LC_breadcrumbs').length) {
                        header = \$('#LC_breadcrumbs');
                    }
                }
                if (header != null && header.length) {
                    height = header.height();
                    hdrtop = header.position().top;
                }
                var pos = height + hdrtop + offset;
                \$('.LC_iframecontainer').css('top', pos);
            });
            LCresizedef = 1;
            if (LCnotready == 1) {
                LCnotready = 0;
                \$(window).trigger('resize');
            }
        });
        window.onload = function(){
            if (LCresizedef) {
                LCnotready = 0;
                \$(window).trigger('resize');
            } else {
                LCnotready = 1;
            }
        };
SCRIPT
        # javascript will position the iframe if window was resized (or zoomed)
        return <<ENDFRAME;
        $startpage
        $script
        <div class="LC_iframecontainer">
            <iframe src="$url$anchor">$lt{'noif'} $noiframe</iframe>
        </div>
        $endpage
ENDFRAME
    }
}

sub create_link {
    my ($url,$anchor,$title,$linktext) = @_;
    my $shownlink;
    if ($title eq '') {
        $title = $env{'form.title'};
        if ($title eq '') {
            unless ($env{'request.enc'}) {
                ($title) = ($url =~ m{/([^/]+)$});
                $title =~ s/(\?[^\?]+)$//;
            }
        }
    }
    unless ($title eq '') {
        $shownlink = '<span style="font-weight:bold;">'.$title.'</span><br />';
    }
    my $dest = &HTML::Entities::encode($url.$anchor,'&<>"');
    $shownlink .= '<a href="'.$dest.'">'.$linktext.'</a>';
    return $shownlink;
}

sub handler {
    my $r=shift;
    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;

    return OK if $r->header_only;

    my $url = $r->uri;
    my ($is_ext,$brcrum,$absolute,$is_pdf,$cdom,$cnum,$hostname);

    for ($url){
        s|^/adm/wrapper||;
        $is_ext = $_ =~ s|^/ext/|http://|;
        s|http://https://|https://| if ($is_ext);
        s|&colon;|:|g;
    }

    if ($url =~ /\.pdf$/i) {
        $is_pdf = 1;
    }
 
    if ($is_ext) {
        &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},
            ['forceedit','register','folderpath','symb','idx','title','anchor']);
        if (($env{'form.forceedit'}) &&
            (&Apache::lonnet::allowed('mdc',$env{'request.course.id'})) &&
            (($env{'form.folderpath'} =~ /^supplemental/) ||
             ($env{'form.symb'} =~ /^uploaded/))) {
            if ($env{'form.symb'}) {
                (undef,undef,my $res) = &Apache::lonnet::decode_symb($env{'form.symb'});
                if ($res =~ /(#[^#]+)$/) {
                    $url .= $1;
                }
            } elsif ($env{'form.folderpath'} =~ /^supplemental/) {
                if ($env{'form.anchor'} ne '') {
                    $url .= '#'.$env{'form.anchor'};
                }
            }
            if (($url =~ /^http:/) && ($ENV{'SERVER_PORT'} == 443)) {
                $hostname = $r->hostname();
            }
            my $type = 'ext';
            $r->print(
                &Apache::lonextresedit::display_editor($url,$env{'form.folderpath'},
                                                       $env{'form.symb'},
                                                       $env{'form.idx'},$type,$cdom,
                                                       $cnum,$hostname));
            return OK;
        } elsif ($env{'form.folderpath'} =~ /^supplemental/) {
            my $crstype = &Apache::loncommon::course_type();
            my $title = $env{'form.title'};
            if ($title eq '') {
                $title = &mt('External Resource');
            }
            $brcrum =
                &Apache::lonhtmlcommon::docs_breadcrumbs(undef,$crstype,undef,$title,1);
        }
    }

#
# Actual URL
#
    if ($url=~/$LONCAPA::assess_re/) {
#
# This is uploaded homework
#
        $env{'request.state'}='uploaded';
        &Apache::lonhomework::renderpage($r,$url);
    } else {
#
# This is not homework
#
        if ($is_ext) {
            $absolute = $env{'request.use_absolute'};
            $ENV{'QUERY_STRING'} =~ s/(^|\&)symb=[^\&]*/$1/;
            $ENV{'QUERY_STRING'} =~ s/\&$//;
        }

        unless ($ENV{'QUERY_STRING'} eq '') {
            $url.=(($url=~/\?/)?'&':'?').$ENV{'QUERY_STRING'};
        }

        # encrypt url if not external
        &Apache::lonenc::check_encrypt(\$url) if $url !~ /^https?\:/ ;

        $r->print( wrapper($r,$url,$brcrum,$absolute,$is_ext,$is_pdf) );

    } # not just the menu
    
    return OK;
} # handler

1;
__END__

=pod

=head1 NAME

Apache::lonwrapper - External and binary file management.

=head1 SYNOPSIS

Wrapper for external and binary files as standalone resources. Edit handler for rat maps; TeX content handler.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 Subroutines

=over

=item wrapper($r,$url,$brcrum,$absolute,$is_ext,$is_pdf,$title))

=over

=item $r

request object

=item $url

url to display by including in an iframe within a
LON-CAPA page which has a standard LON-CAPA inline menu.

=item $brcrum

breadcrumbs for unregistered urls
(i.e., external resources in Supplemental Content).

=item $absolute

contains protocol (http or https) followed by
the hostname, if menu items in the standard LON-CAPA
interface created by the call to loncommon::start_page()
within &wrapper() need to use absolute URLs rather than
relative URLs.

That will be the case where an external resource has been 
served from port 80, when the server customarily serves
requests using Apache/SSL (i.e., port 443). mod_rewrite 
is used to switch requests for external resources and
the syllabus: /public/<domain>/<courseid>/syllabus
(which might also point at an external resource)
from https:// to http:// where the the URL of the remote site 
specified in the resource itself is http://.

This is done to avoid default mixed content blocking
in Firefox 23 and later, when serving from Apache/SSL.

=item $is_ext

true if URL is for an external resource.

=item $is_pdf

true if URL is for a PDF (based on file extension).

=item $title

optional. If wrapped item is a PDF, and $clientmobile is true,
a link to a PDF is shown. The "title" will be displayed
above the link, but if not provided as an arg, $env{'form.title'}
will be used, otherwise, the filename will be displayed (unless
hidden URL set for the resource).

=back

Returns markup for the entire page.

=item handler()

=back

=cut

