#!/usr/bin/perl -w
# The LearningOnline Network 
# Red Hat 7.3 installation script
#
# $Id: install.pl,v 1.25 2003/09/17 18:51:47 albertel Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# http://www.lon-capa.org/
#

##
## Obvious flaws of this program: 
##   Dieing on every error may be a little extreme.  On the other hand, 
##       how the heck am I supposed to know what absurd things the user 
##       has done with their system before inflicting LON-CAPA on it?
##   The links to /etc/init.d for httpd and mysqld do not seem to work :(
##   The user is never informed of the log file (/tmp/loncapa_install.log).
##   It does not test the system at the end.  Again, there are limits to 
##       what nonsense we can put up with.  Of course, we will have to 
##       explain that to people at some point...
##   There is probably an overuse of elipses (...) in the comments.
##   It might be nice to check that all the files we need are here.
##   Appletalk is installed but does not work and gives errors on
##       boot up.  I have not been able to find a clean way to get the
##       appletalk support working but the powers that be insist on it.
##

#
# Needed files:
#
#    The following files are assumed to be present in the current
#    directory:
#      RPMS:
#        ImageMagick-5.4.3.11-1.i386.rpm
#        ImageMagick-devel-5.4.3.11-1.i386.rpm
#        ImageMagick-perl-5.4.3.11-1.i386.rpm
#        gnuplot-3.7.1-5.i386.rpm
#        libgd-1.3-4.i386.rpm
#        libungif-progs-4.1.0-9.i386.rpm
#        ncurses4-5.0-5.i386.rpm
#        readline-2.2.1-6.i386.rpm
#        readline-4.2a-4.i386.rpm
#        perl-DBD-MySQL-1.2216-4.i386.rpm
#        perl-DBI-1.21-1.i386.rpm
#        mod_perl-1.26-5.i386.rpm
#        perl-suidperl-5.6.1-34.99.6.i386.rpm
#        LON-CAPA-systemperl-3.5-rh7.i386.rpm
#        mysql-3.23.49-3.i386.rpm
#        mysqlclient9-3.23.22-6.i386.rpm
#        mysql-server-3.23.49-3.i386.rpm
#        hwcrypto-1.0-3.i386.rpm
#        m2crypto-0.05_snap4-2.i386.rpm
#        netpbm-9.24-3.i386.rpm
#        netpbm-progs-9.24-3.i386.rpm
#        krb5-libs-1.2.4-3.i386.rpm
#      Other files:
#        httpd.conf
#        mod_auth_external-2.1.13.tar.gz
#
#    The contingency plan for a 7.2 install tells the user to install these
#    from the current directory.
#        perl-5.6.1-34.99.6.i386.rpm
#        perl-CGI-2.752-34.99.6.i386.rpm
#

use strict;
use File::Copy;

my $result; 
my $test;

# note: The filehandle LOG is global.
open LOG,">loncapa_install.log" || die "Unable to open log file.\n";

print LOG '$Id: install.pl,v 1.25 2003/09/17 18:51:47 albertel Exp $'."\n";

# Some friendly subroutines
sub die_if_nonempty {
    my ($string,$error)=@_;
    return if (! defined($error));
    chomp($string);chomp($error);
    if ($string ne '') {
        print_and_log("$error\nHalting.\n");
        die;
    }
}

sub make_link_or_die {
    my ($source,$dest)=@_;
    &die_if_nonempty
        (`ln -fs $source $dest`,"Unable to link $source to $dest.");
    print LOG "Link from $source to $dest made successfully\n";
}

sub writelog {
    while ($_ = shift) {
        chomp;
        print LOG "$_\n";
    }
}

sub print_and_log {
    while ($_=shift) {
        chomp;
        print "$_\n";
        print LOG "$_\n";
    }
}

##
## First, make sure it's a red hat system.
##
if (! -e "/etc/redhat-release") {
    print_and_log(<<"END");
*********************************************************************

This does not a appear to be a Red-Hat system.  More than likely the 
installation will not be successful!  Press control-c to abort now, 
otherwise press enter to forge ahead and damn the torpedos.

*********************************************************************
END
    undef = <STDIN>;
}



#
# The installation work begins now...
#

print <<"END";
********************************************************************

                    Welcome to LON-CAPA 

This script will install the base software that LON-CAPA needs to
run properly. 

********************************************************************
END

##
## Install needed RPMS
##
my $instdir = `pwd`;
chomp($instdir);
# 
# This list of rpms needs to be pared down to some extent.
#

my @apache_rpms = (
             "$instdir/apache-1.3.23-14.i386.rpm",
                   );

my @openssh_rpms = (
             "$instdir/openssh-3.1p1-6.i386.rpm",
             "$instdir/openssh-askpass-3.1p1-6.i386.rpm",
             "$instdir/openssh-clients-3.1p1-6.i386.rpm",
             "$instdir/openssh-server-3.1p1-6.i386.rpm"
                );

# Check for gnome-askpass installation.
if (-e "/etc/profile.d/gnome-ssh-askpass.sh") {
    push @openssh_rpms,"$instdir/openssh-askpass-gnome-3.1p1-6.i386.rpm";
}

my @ImageMagick_rpms = ( 
             "$instdir/ImageMagick-5.4.3.11-1.i386.rpm",
             "$instdir/ImageMagick-devel-5.4.3.11-1.i386.rpm",
             "$instdir/ImageMagick-perl-5.4.3.11-1.i386.rpm",
                       );

my @mysql_rpms = (
             "$instdir/mysql-3.23.49-3.i386.rpm",
             "$instdir/mysqlclient9-3.23.22-6.i386.rpm",
             "$instdir/mysql-server-3.23.49-3.i386.rpm",
                  );

my @perl_rpms = ( 
             "$instdir/perl-DBD-MySQL-1.2216-4.i386.rpm",
             "$instdir/perl-DBI-1.21-1.i386.rpm",
             "$instdir/perl-suidperl-5.6.1-34.99.6.i386.rpm",
                 );

my @old_readline_rpms = (
             "$instdir/readline-2.2.1-6.i386.rpm", # requires -i --oldpackage, 
                                                   # not -Uvh
                         );
my @gnuplot_rpms = ( # must be done after readline-2.2.1-6
             "$instdir/libgd-1.3-4.i386.rpm", 
             "$instdir/libungif-progs-4.1.0-9.i386.rpm",
             "$instdir/ncurses4-5.0-5.i386.rpm",
             "$instdir/gnuplot-3.7.1-5.i386.rpm",
                    );

my @loncapa_perl_rpms = (
             "$instdir/netpbm-9.24-3.i386.rpm",
             "$instdir/netpbm-progs-9.24-3.i386.rpm",
             "$instdir/krb5-libs-1.2.4-3.i386.rpm",
             "$instdir/krb5-devel-1.2.4-3.i386.rpm",
             "$instdir/LON-CAPA-krb4-3.1-1.i386.rpm",
                    );
my @misc_rpms = (
             "$instdir/m2crypto-0.05_snap4-2.i386.rpm",
             "$instdir/tetex-dvips-1.0.7-47.i386.rpm",
             "$instdir/ntp-4.1.1-1.i386.rpm",
             "$instdir/libcap-1.10-8.i386.rpm",
             );

my $systemperl = "$instdir/LON-CAPA-systemperl-3.7-rh7.i386.rpm";

##
## Some of these rpm commands require being obnoxious (--force --nodeps)
## this is not a nice thing to do and we should be careful about it.
##



&print_and_log("Installing Apache packages.\n");
&writelog (`rpm -Uvh --replacepkgs @apache_rpms`);
&print_and_log("Installing openssh packages.\n");
&writelog (`rpm -Uvh --replacepkgs @openssh_rpms`);
&writelog(`/etc/init.d/sshd start`);
&print_and_log("Installing ImageMagick packages.\n");
&writelog (`rpm -Uvh --replacepkgs @ImageMagick_rpms`);
&print_and_log("Installing mysql packages.\n");
&writelog (`rpm -Uvh --replacepkgs @mysql_rpms`);
&print_and_log("Installing Perl packages.\n");
&writelog (`rpm -Uvh --replacepkgs @perl_rpms`);
&print_and_log("Installing legacy readline package (required for gnuplot).");
&writelog(`rpm -i --oldpackage @old_readline_rpms`);
&print_and_log("Installing gnuplot packages.\n");
&writelog (`rpm -ivh --force --nodeps @gnuplot_rpms`);
&print_and_log("Installing LON-CAPA Perl packages.\n");
&writelog (`rpm -Uvh --replacepkgs @loncapa_perl_rpms`);
&print_and_log("Installing misc packages.\n");
&writelog (`rpm -Uvh --replacepkgs @misc_rpms`);
&print_and_log("Installing LON-CAPA systemperl rpm");
&writelog(`rpm -ivh --force --nodeps $systemperl`);
&print_and_log("\n");


##
## Remove conflicting packages
##
my @php_rpms = ("php-imap-4.1.2-7",
                "asp2php-0.76.2-1",
                "php-ldap-4.1.2-7",
                "php-devel-4.1.2-7",
                "php-4.1.2-7",
	        "php-pgsql-4.1.2-7");

&print_and_log("Removing php packages");
foreach my $php_rpm (@php_rpms) {
    my $remove_error = system("rpm -e --nodeps ".$php_rpm);
    if ($remove_error) {
        &print_and_log("Unable to remove ".$php_rpm.".  ".
                       "Assuming it is not present.\n");
    } else {
        &writelog("Successfully removed ".$php_rpm);
    }
}

&print_and_log("Removing mod_throttle");
system("rpm -e `rpm -q -a | grep mod_throttle`");
&print_and_log("Removing mod_bandwidth");
system("rpm -e `rpm -q -a | grep mod_bandwidth`");

##
## Fix that stupid little sendmail bug
##
print_and_log("changing permissions on root directory.\n");
$result = `chmod g-w,u+w /`;
if ($result eq '') {
    $result = "successful\n";
} else {
    die "Unable to change permissions on root directory.  Halting.\n";
}
writelog ($result);
print_and_log("\n");

##
## Set up www and authentication
##
print_and_log("Creating user 'www'\n");
$result = `/usr/sbin/useradd www`;
if (! (($result eq '') || ($result =~ /user www exists/))) {
    die "Unable to add user www.  Halting.\n";
}
writelog ($result);
my $num = `grep ^www /etc/passwd | cut -d':' -f3`;
chomp $num;
if (int($num) == $num) {
    writelog ("uid of www = $num\n");
} else {
    die "Unable to determine UID of user www\n  Halting.\n";
}
print_and_log("\n");

##
## Patch mod_auth_external
##
print_and_log("Setting up authentication for 'www'\n");
my $patch = <<"ENDPATCH";
148c148
< #define SERVER_UIDS 99		/* user "nobody" */
---
> #define SERVER_UIDS $num		/* user "www" */
ENDPATCH

if (! -e "/usr/bin/patch") {
    print_and_log("You must install the software development tools package ".
                  "when installing RedHat.\n");
    die;
}
&die_if_nonempty(`cd /tmp; tar zxf $instdir/mod_auth_external-2.1.13.tar.gz`,
                 "Unable to extract mod_auth_external\n");
my $dir = "/tmp/mod_auth_external-2.1.13/pwauth";
open PATCH, "| patch $dir/config.h" || 
    die "Unable to start patch for mod_auth_external.  Halting\n";
print PATCH $patch;
close PATCH;
print_and_log("\n");

##
## Compile patched pwauth
##
print_and_log("Compiling pwauth\n");
$result = `cd $dir/; make`;
my $expected = <<"END";
gcc -g    -c -o pwauth.o pwauth.c
gcc -o pwauth -g  pwauth.o -lcrypt
END

if ($result ne $expected) {
    die "Unable to compile patched pwauth.  Halting.\n";
}    
print_and_log( $result );

##
## Install patched pwauth
##
print_and_log("Copying pwauth to /usr/local/sbin\n");
if (! copy "$dir/pwauth","/usr/local/sbin/pwauth") {
    die "Unable to copy $dir/pwauth to /usr/local/sbin/pwauth.\n$!\nHalting\n";
}
if (! chmod (06755, "/usr/local/sbin/pwauth")) {
    die "Unable to set permissions on /usr/local/sbin/pwauth.\n";
}
print_and_log("\n");

##
## Set up mysql
##
print_and_log("Setting mysqld to start on boot up.\n");
system("/sbin/chkconfig --add mysqld");
system("/sbin/chkconfig mysqld on");
&writelog(`/sbin/chkconfig --list mysqld`);

writelog("mysql links created successfully\n");
writelog(`/etc/rc.d/init.d/mysqld start`);
print_and_log("Waiting for mysql daemon to start.\n");
sleep 5;
my $status = system("/etc/rc.d/init.d/mysqld status");
if ($status != 0) {
    die "Unable to start mysql daemon\nHalting\n";
} else {
    print_and_log("Mysql daemon is running.\n");
}
print_and_log("\n");

##
## Get root password for mysql client
##
print <<END;
Please enter a root password for the mysql database.
It does not have to match your root account password, but you will need
to remember it.
END
my $rootpass = <>;
chomp $rootpass;
print_and_log("\n");

##
## Run the damn thing (mysql, not LON-CAPA)
##
print_and_log("Starting mysql client.\n");
open MYSQL, "|mysql -u root mysql" || die "Unable to start mysql\n";
print MYSQL <<"ENDMYSQL";
CREATE DATABASE loncapa;
INSERT INTO user (Host, User, Password)
VALUES ('localhost','www',password('localhostkey'));
INSERT INTO db VALUES ('localhost','loncapa','www',
'Y','Y','Y','Y','Y','Y','N','Y','Y','Y');
SET PASSWORD FOR root\@localhost=PASSWORD('$rootpass');
DELETE FROM user WHERE host<>'localhost';
FLUSH PRIVILEGES;
USE loncapa;
CREATE TABLE IF NOT EXISTS metadata (title TEXT, author TEXT, subject TEXT, url TEXT, keywords TEXT, version TEXT, notes TEXT, abstract TEXT, mime TEXT, language TEXT, creationdate DATETIME, lastrevisiondate DATETIME, owner TEXT, copyright TEXT, FULLTEXT idx_title (title), FULLTEXT idx_author (author), FULLTEXT idx_subject (subject), FULLTEXT idx_url (url), FULLTEXT idx_keywords (keywords), FULLTEXT idx_version (version), FULLTEXT idx_notes (notes), FULLTEXT idx_abstract (abstract), FULLTEXT idx_mime (mime), FULLTEXT idx_language (language), FULLTEXT idx_owner (owner), FULLTEXT idx_copyright (copyright)) TYPE=MYISAM;
EXIT
ENDMYSQL

close MYSQL;
print_and_log("\n");

##
## Remove the firewall.
##
system("/sbin/chkconfig ipchains off");
system("/etc/init.d/ipchains stop");
system("/sbin/chkconfig iptables off");
system("/etc/init.d/iptables stop");

# Someday we will add these to /etc/sysconfig/ipchains.
#  "-A input -s 0/0 -d 0/0 8080 -p tcp -y -j ACCEPT",
#  "-A input -s 0/0 -d 0/0 5663 -p tcp -y -j ACCEPT"
# Someday we will deal with iptables, too.  Soon.

##
## Set up httpd 
##
print_and_log("Setting httpd to start on boot up.\n");
system("/sbin/chkconfig httpd on");

##
## Copy our (probably lousy) httpd.conf to its rightful place
##
print_and_log("Copying our httpd.conf to /etc/httpd/conf/httpd.conf\n");
copy "$instdir/httpd.conf","/etc/httpd/conf/httpd.conf";
chmod 0444,"/etc/httpd/conf/httpd.conf";
print_and_log("\n");

##
## Retrieve loncapa.tar.gz
##
my $lctarball = 'loncapa-current.tar.gz';
if (! -e "$instdir/$lctarball") {
    print_and_log("Retrieving LON-CAPA source files from install.loncapa.org\n")
;
    system("wget http://install.loncapa.org/versions/$lctarball 2>/dev/null 1>/dev/null");
    if (! -e "./$lctarball") {
        die("Unable to retrieve LON-CAPA source files from\n".
            "http://install.loncapa.org/versions/$lctarball\n");
    }
    print_and_log("\n");
} else {
    print_and_log(<<"END");
------------------------------------------------------------------------

You seem to have a version of loncapa-current.tar.gz in $instdir.  
This copy will be used and a new version will NOT be downloaded.  
If you wish, you may download a new version by executing:

wget http://install.loncapa.org/versions/loncapa-current.tar.gz

------------------------------------------------------------------------
END
}

##
## untar loncapa.tar.gz
##
print_and_log("Extracting LON-CAPA source files\n");
writelog(`cd ~root; tar zxf $instdir/loncapa-current.tar.gz`);
print_and_log("\n");

my $version = `cat /etc/redhat-release`;
if ($version =~ /7\.2/) {
    print_and_log(<<"END");
This appears to be a Red Hat 7.2 system.  You need to execute the following
commands now:
rpm -Uvh perl-5.6.1-34.99.6.i386.rpm
rpm -Uvh perl-CGI-2.752-34.99.6.i386.rpm

cd /root/loncapa-N.N     (N.N should correspond to a version number like '0.4')
./UPDATE

END
} else {
    ##
    ## Assure them that everything worked okay....
    ##
    print <<"ENDMSG";
All of the extra files seem to have been installed correctly.  It remains for 
you to execute the following commands:

cd /root/loncapa-N.N     (N.N should correspond to a version number like '0.4')
./UPDATE

If you have any trouble, please see http://install.loncapa.org/ and 
http://help.loncapa.org/.  
ENDMSG
}

close LOG;

