# LearningOnline Network with CAPA
# option list style responses
#
# $Id: optionresponse.pm,v 1.195.2.3 2018/03/01 13:11:47 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#





package Apache::optionresponse;
use strict;
use Apache::response();
use Apache::lonlocal;
use Apache::lonnet;

BEGIN {
  &Apache::lonxml::register('Apache::optionresponse',('optionresponse'));
}

sub start_optionresponse {
  my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
  my $result='';
  #when in a option response use these
  &Apache::lonxml::register('Apache::optionresponse',('foilgroup','foil','conceptgroup','drawoptionlist'));
  push (@Apache::lonxml::namespace,'optionresponse');
  my $id = &Apache::response::start_response($parstack,$safeeval);
  %Apache::hint::option=();
  undef(%Apache::response::foilnames);
  if ($target eq 'edit') {
    $result.=&Apache::edit::start_table($token)
       .'<tr><td><span class="LC_nobreak">'.&Apache::loncommon::insert_folding_button().' '
       .&mt('Multiple Option Response Question').'</span> '
       .&Apache::loncommon::help_open_topic('Option_Response_Problems')."</td>"
       .'<td><span class="LC_nobreak">'.&mt('Delete?').' '
       .&Apache::edit::deletelist($target,$token)
       .'</span></td>'
       ."<td>&nbsp;"
       .&Apache::edit::end_row()
       .&Apache::edit::start_spanning_row()
       ."\n";
    $result.=&Apache::edit::text_arg('Max Number Of Shown Foils:','max',
				     $token,'4').
        &Apache::edit::select_arg('Randomize Foil Order:','randomize',
				  ['yes','no'],$token).
        &Apache::edit::select_arg('Display of options when printed','TeXlayout',
				  [['horizontal','Normal list'],
				   ['vertical','Listed in a vertical column']],$token).
        &Apache::edit::end_row().&Apache::edit::start_spanning_row();
  } elsif ($target eq 'modified') {
    my $constructtag=&Apache::edit::get_new_args($token,$parstack,
						 $safeeval,'max','randomize',
						 'TeXlayout');
    if ($constructtag) { $result = &Apache::edit::rebuild_tag($token); }
  } elsif ($target eq 'meta') {
    $result=&Apache::response::meta_package_write('optionresponse');
  } elsif ($target eq 'analyze') {
    my $part_id="$Apache::inputtags::part.$id";
    $Apache::lonhomework::analyze{"$part_id.type"} = 'optionresponse';
    push (@{ $Apache::lonhomework::analyze{"parts"} },$part_id);
    push (@{ $Apache::lonhomework::analyze{"$part_id.bubble_lines"} }, 1);

  }
  return $result;
}

sub end_optionresponse {
  my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
  &Apache::response::end_response;
  pop @Apache::lonxml::namespace;
  &Apache::lonxml::deregister('Apache::optionresponse',('foilgroup','foil','conceptgroup','drawoptionlist'));
  my $result;
  if ($target eq 'edit') { $result=&Apache::edit::end_table(); }
  undef(%Apache::response::foilnames);
  return $result;
}

%Apache::response::foilgroup=();
sub start_foilgroup {
  my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;

  my $result='';
  %Apache::response::foilgroup=();
  $Apache::optionresponse::conceptgroup=0;
  &Apache::response::pushrandomnumber(undef,$target);
  if ($target eq 'edit') {
    my $optionlist="<option></option>\n";
    my $option;
    my @opt;
    my @raw_options = &Apache::lonxml::get_param('options', $parstack, $safeeval, 0,  0, 1);


    eval '@opt ='. &Apache::lonxml::get_param('options',$parstack,$safeeval, 0, 0, 1);

    my $count=1;
    foreach $option (@opt) {
      $optionlist.="<option value=\"$count\">$option</option>\n";
      $count++;
    }
    my $insertlist=&Apache::edit::insertlist($target,$token);
    $result.=&Apache::edit::start_table($token);
    my %lt = &Apache::lonlocal::texthash(
        sel => 'Select Options',
        add => 'Add new Option:',
        del => 'Delete an Option:',
        maxcheck => 'Enter maximum number students may check (e.g., 3)',
        mincheck => 'Enter minimum number students need to check (e.g., 1)',
    );
    $result.= sprintf(<<ENDTABLE, &Apache::loncommon::insert_folding_button());
      <tr><td>%s $lt{'sel'}</td>
        <td>
	  $lt{'add'} <input type="text" name="$Apache::lonxml::curdepth.options" />
        </td>
        <td>$lt{'del'}
	  <select name="$Apache::lonxml::curdepth.deleteopt">$optionlist</select>
        </td>
ENDTABLE
    $result.= '<td>'.&Apache::edit::checked_arg('Print options:','texoptions',
					 [ ['nochoice',"Don't show option list"] ]
					 ,$token).'</td>';
    $result.= '<td><label>'.&mt('Two-option checkboxes for:').
              '<select name="checkboxvalue_'.$Apache::lonxml::curdepth.'">';
    foreach $option (('',@opt)) {
       $result.='<option value="'.$option.'"';
       if ($option eq &Apache::lonxml::get_param('checkboxvalue',$parstack,$safeeval)) {
          $result.=' selected="selected"';
       }
       $result.='>'.$option.'</option>';
    }
    $result.='</select></label> ';
    my %checknumtxt = (
                        maxcheck => 'Max. boxes permitted to check',
                        mincheck => 'Min. boxes required to check',
    );
    my (%onclicknum,%checknum);
    foreach my $item ('maxcheck','mincheck') {
        $onclicknum{$item} = 'onclick="'."updateNumber('$item','$Apache::lonxml::curdepth','check','$lt{$item}');".'"';
        $checknum{$item} = &Apache::lonxml::get_param($item,$parstack,$safeeval);
    }
    $result.=&Apache::edit::checked_arg('Checkbox options:','checkboxoptions',
                                         [ ['nochoice',"Don't show option list"] ]
                                         ,$token).' '.
             &Apache::edit::checked_arg('','noprompt',
                                         [ ['nochoice','Omit "Select all that are ... "' ] ]
                                         ,$token);
    foreach my $item ('maxcheck','mincheck') {
        my $currnum = $checknum{$item};
        unless ($currnum =~ /^\d+$/) {
            $currnum = 'nochoice';
        } 
        $result.= ' <span class="LC_nobreak">'.
                  &Apache::edit::checked_arg('',$item,[ [$currnum,$checknumtxt{$item}] ],
                                        $token,$onclicknum{$item},1).' '.
                  '<span id="'.$item.'_numtext_'.$Apache::lonxml::curdepth.'">';
        if ($checknum{$item} ne '') {
            $result .= ':&nbsp;<a href="javascript:updateNumber('."'$item','$Apache::lonxml::curdepth','link','$lt{$item}'".')">'.
                       $checknum{$item}.'</a>';
        }
        $result .= '</span></span>';
    } 
    $result.= '</td>';
    $result.= &Apache::edit::end_row();
    $result.= &Apache::edit::start_spanning_row();
    $result.= $insertlist.'<br />';
  }
  if ($target eq 'modified') {
    my @options;
    my $optchanged=0;


    eval '@options ='.&Apache::lonxml::get_param('options',$parstack,$safeeval, 0, 0, 1);

    if ($env{"form.$Apache::lonxml::curdepth.deleteopt"}) {
      my $delopt=$env{"form.$Apache::lonxml::curdepth.deleteopt"};
      &Apache::lonxml::debug("Deleting :$delopt:");
      splice(@options,$delopt-1,1);
      $optchanged=1;
    }
    if ($env{"form.$Apache::lonxml::curdepth.options"} ne '') {
      my $newopt = $env{"form.$Apache::lonxml::curdepth.options"};
      push(@options,$newopt);
      $optchanged=1;
    }
    my $rebuildtag = &Apache::edit::get_new_args($token,$parstack,$safeeval,
						 'texoptions','checkboxvalue','checkboxoptions',
                                                 'noprompt','maxcheck','mincheck');
    if ($rebuildtag) {
        my $maxcheck = $token->[2]->{'maxcheck'};
        my $mincheck = $token->[2]->{'mincheck'};
        if ($mincheck =~ /D/) {
            $token->[2]->{'mincheck'}='';
        }
        if ($maxcheck =~ /D/) {
            $token->[2]->{'maxcheck'}='';
        }
        if (($mincheck ne '') && ($maxcheck ne '')) {
            if ($mincheck > $maxcheck) {
                $token->[2]->{'mincheck'}=$maxcheck; 
            }
        }
    }
    if ($optchanged || $rebuildtag ) {
      $result = "<foilgroup options=\"(";
      foreach my $option (@options) {
	$option=~s/\'/\\\'/g;
	&Apache::lonxml::debug("adding option :$option:");
	$result .="'".$option."',";
      }
      chop $result;
      $result.=')" ';
      $result .= 'texoptions="'.$token->[2]{'texoptions'}.'" ';
      $result .= 'checkboxoptions="'.$token->[2]{'checkboxoptions'}.'" ';
      $result .= 'checkboxvalue="'.$token->[2]{'checkboxvalue'}.'" ';
      $result .= 'noprompt="'.$token->[2]{'noprompt'}.'" ';
      $result .= 'maxcheck="'.$token->[2]{'maxcheck'}.'" ';
      $result .= 'mincheck="'.$token->[2]{'mincheck'}.'"';
      $result .= '>';
    } # else nothing changed so just use the default mechanism
  }
  if ($target eq 'tex' and $Apache::lonhomework::type ne 'exam') {
    if($env{'form.pdfFormFields'} ne 'yes') {
      $result .= ' \begin{itemize} ';
    } else {
      $result .= "\\\\";
    }
  }
  return $result;
}

sub end_foilgroup {
  my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
  
  my $result;
  if ($target eq 'grade' || $target eq 'web' || $target eq 'answer' ||
      $target eq 'tex' || $target eq 'analyze') {
    my $tex_option_switch=&Apache::lonxml::get_param('texoptions',$parstack,$safeeval);
    my $name;
    my @opt;
    eval '@opt ='.&Apache::lonxml::get_param('options',$parstack,$safeeval);
    &Apache::lonxml::debug("options:".join(':',@opt));

    my $TeXlayout=&Apache::lonxml::get_param('TeXlayout',$parstack,$safeeval,
					     -2,0);
    my $checkboxvalue=&Apache::lonxml::get_param('checkboxvalue',$parstack,$safeeval);
    my $checkboxchoices=(&Apache::lonxml::get_param('checkboxoptions',$parstack,$safeeval) ne 'nochoice');
    my $maxcheck=&Apache::lonxml::get_param('maxcheck', $parstack, $safeeval);
    $maxcheck =~ s/\D+//g;
    my $mincheck=&Apache::lonxml::get_param('mincheck', $parstack, $safeeval);
    $mincheck =~ s/\D+//g;
    my $noprompt=&Apache::lonxml::get_param('noprompt', $parstack, $safeeval);

    if ($target eq 'tex' && $tex_option_switch eq 'nochoice') {@opt=();}
    &Apache::lonxml::debug("Options are $#opt");
    my $max = &Apache::lonxml::get_param('max',$parstack,$safeeval,'-2');
    my $randomize = &Apache::lonxml::get_param('randomize',$parstack,
					       $safeeval,'-2');
    if ($target eq 'web' || $target eq 'tex') {
      $result.=&displayfoils($target,$max,$randomize,$TeXlayout,$checkboxvalue,$checkboxchoices,$tex_option_switch, $noprompt,$maxcheck,$mincheck, @opt);
      $Apache::lonxml::post_evaluate=0;
    } elsif ( $target eq 'answer') {
      $result.=&displayanswers($max,$randomize,@opt);
      
    } elsif ( $target eq 'analyze') {
	my @shown = &whichfoils($max,$randomize);
	&Apache::response::analyze_store_foilgroup(\@shown,
						  ['text','value','location']);
	my $part_id="$Apache::inputtags::part.$Apache::inputtags::response[-1]";
	push (@{ $Apache::lonhomework::analyze{"$part_id.options"} },@opt);
    } elsif ( $target eq 'grade') {

      if ( &Apache::response::submitted()) {

	my @whichopt = &whichfoils($max,$randomize);
	my $temp=1;my $name;
	my %responsehash;
	my %grade;
	my $right=0;
	my $wrong=0;
	my $ignored=0;
        my $checked=0;
        my ($numrows,$bubbles_per_row);
        if ($Apache::lonhomework::scantronmode) {
            my $numitems = scalar(@opt);
            ($numrows,$bubbles_per_row) =
                &Apache::optionresponse::getnumrows($numitems);
        }
        if ($numrows < 1) {
            $numrows = 1;
        }
        my $checkboxopt;
        if ($checkboxvalue) {
            $checkboxopt = &check_box_opt($target,$checkboxvalue,@opt);
            if (($checkboxopt) && ($Apache::lonhomework::type !~ /survey/)) {
                my $corrnumcheck = 0;
                if (($maxcheck ne '') || ($mincheck ne '')) {
                    foreach $name (@whichopt) {
                        if ($Apache::response::foilgroup{$name.'.value'} eq $checkboxopt) {
                            $corrnumcheck ++; 
                        }
                    }
                    if (($maxcheck ne '') && ($corrnumcheck > $maxcheck)) {
                        $maxcheck = $corrnumcheck;
                    }
                    if (($mincheck ne '') && ($corrnumcheck < $mincheck)) {
                        $mincheck = $corrnumcheck;
                    }
                }
            }
        }
	foreach $name (@whichopt) {
	  my $response;
          if ($env{'form.submitted'} eq 'scantron') {
              if ($numrows > 1) {
                  my $num = $temp;
                  my $totalnum;
                  for (my $i=0; $i<$numrows; $i++) {
                      my $item = &Apache::response::getresponse($num);
                      if ($item =~ /^\d+$/) {
                          $totalnum = $i*$bubbles_per_row + $item;
                      }
                      $num ++;
                  }
                  if ($totalnum =~ /^\d+$/) {
                      $response = $opt[$totalnum];
                  }
                  $temp += $numrows;
              } else {
                  $response=&Apache::response::getresponse($temp);
                  if ($response=~/\S/) {
                      $response = $opt[$response];
                  }
                  $temp ++;
              }
	  } else {
              $response=&Apache::response::getresponse($temp);
              $temp ++;
          }
	  if ( $response =~ /[^\s]/) {
	    $responsehash{$name}=$response;
	    my $value=$Apache::response::foilgroup{$name.'.value'};
	    &Apache::lonxml::debug("submitted a $response for $value<br />\n");
	    if ($value eq $response) {
		$grade{$name}='1'; $right++;
	    } else {
		$grade{$name}='0'; $wrong++;
	    }
            if ($checkboxopt) {
                if ($response eq $checkboxopt) {
                    $checked++;
                }
            } 
	  } else {
	    $ignored++;
	  }
	}
	my $part=$Apache::inputtags::part;
        my $id = $Apache::inputtags::response['-1'];
        my $nonlenient=&grading_is_nonlenient($part,$id);
	my $responsestr=&Apache::lonnet::hash2str(%responsehash);
	my $gradestr   =&Apache::lonnet::hash2str(%grade);
	my %previous=&Apache::response::check_for_previous($responsestr,
							   $part,$id);
	&Apache::lonxml::debug("Got $right right and $wrong wrong, and $ignored were ignored");
	$Apache::lonhomework::results{"resource.$part.$id.submission"}=
	    $responsestr;
	$Apache::lonhomework::results{"resource.$part.$id.submissiongrading"}=$gradestr;
        if ($Apache::lonhomework::type eq 'randomizetry') {
            $Apache::lonhomework::results{"resource.$part.$id.foilorder"} = &Apache::lonnet::array2str(@whichopt);
        }
        my $skipaward;
        if ($checkboxvalue) {
            if ((($maxcheck ne '') && ($checked > $maxcheck)) ||
                (($mincheck ne '') && ($checked < $mincheck))) {
                my $ad=$Apache::lonhomework::results{"resource.$part.$id.awarddetail"}='WRONG_NUMBOXESCHECKED';
                &Apache::response::handle_previous(\%previous,$ad);
                $skipaward = 1;
            }
        }
        unless ($skipaward) {
	    if (($Apache::lonhomework::type eq 'survey') ||
                ($Apache::lonhomework::type eq 'surveycred') ||
                ($Apache::lonhomework::type eq 'anonsurvey') ||
                ($Apache::lonhomework::type eq 'anonsurveycred')) {
	        if ($ignored == 0) {
                    my $ad;
                    if ($Apache::lonhomework::type eq 'anonsurveycred') {
                        $ad=$Apache::lonhomework::results{"resource.$part.$id.awarddetail"}='ANONYMOUS_CREDIT';
                    } elsif ($Apache::lonhomework::type eq 'anonsurvey') {
                        $ad=$Apache::lonhomework::results{"resource.$part.$id.awarddetail"}='ANONYMOUS';
                    } elsif ($Apache::lonhomework::type eq 'surveycred') {
                        $ad=$Apache::lonhomework::results{"resource.$part.$id.awarddetail"}='SUBMITTED_CREDIT';
                    } else {
		        $ad=$Apache::lonhomework::results{"resource.$part.$id.awarddetail"}='SUBMITTED';
                    }
		    &Apache::response::handle_previous(\%previous,$ad);
	        } elsif ($wrong==0 && $right==0) {
	        } else {
		    my $ad=$Apache::lonhomework::results{"resource.$part.$id.awarddetail"}='MISSING_ANSWER';
		    &Apache::response::handle_previous(\%previous,$ad);
	        }
	    } elsif ($nonlenient) {
#
# Non-lenient mode. All right or all wrong
#
	        my $ad;
	        if ($wrong==0 && $ignored==0) {
		    $ad='EXACT_ANS';
	        } elsif ($wrong==0 && $right==0) {
		    #nothing submitted
	        } else {
		    if ($ignored==0) {
		        $ad='INCORRECT';
		    } else {
		        $ad='MISSING_ANSWER';
		    }
	        }
	        $Apache::lonhomework::results{"resource.$part.$id.awarddetail"}=$ad;
	        &Apache::response::handle_previous(\%previous,$ad);
	     } else {
#
# This is lenient mode
#
	        my $ad;
	        if ($wrong==0 && $right==0) {
		    #nothing submitted only assign a score if we 
		    #need to override a previous grade
		    if (defined($Apache::lonhomework::history{"resource.$part.$id.awarddetail"})) {
		        $ad='ASSIGNED_SCORE';
		    }
	        } else {
		    $ad='ASSIGNED_SCORE';
	        }
	        $Apache::lonhomework::results{"resource.$part.$id.awarddetail"}=$ad;
	        $Apache::lonhomework::results{"resource.$part.$id.awarded"}=
		    $right/(scalar(@whichopt));
	        $Apache::lonhomework::results{"resource.$part.$id.numfoils"}=
		    scalar(@whichopt);
	     }
       }
      }
    }
    my $part_id     = $Apache::inputtags::part;
    my $response_id = $Apache::inputtags::response[-1];
    my ($numrows,$bubbles_per_row);
    if (($target eq 'tex') && ($Apache::lonhomework::type eq 'exam')) {
        ($numrows,$bubbles_per_row) =
            &Apache::optionresponse::getnumrows(scalar(@opt));
    }
    if ($numrows < 1) {
        $numrows = 1;
    }
    my $increment = &getfoilcounts($max) * $numrows;
    &Apache::lonxml::increment_counter($increment,"$part_id.$response_id");
    if ($target eq 'analyze') {
	&Apache::lonhomework::set_bubble_lines();
    }

  } elsif ($target eq 'edit') {
    $result.=&Apache::edit::end_table();
  }
  if ($target eq 'tex' and $Apache::lonhomework::type ne 'exam') {
    if($env{'form.pdfFormFields'} ne 'yes') {
      $result .= '\end{itemize}'; 
    } else {
      $result .= "\\\\";
    }
  }
  &Apache::response::poprandomnumber();
  return $result;
}

sub grading_is_nonlenient {
    my ($part,$id) = @_;
# Web mode: we are non-lenient unless told otherwise
    my $defaultparm = 'off';
    my $nonlenient = 1;
# Grading a bubblesheet exam: we are grading lenient unless told otherwise
    if ($Apache::lonhomework::scantronmode) {
        $defaultparm = 'on';
        $nonlenient = 0;
    }
    my $lenientparm =
        &Apache::response::get_response_param($part,'lenient',$defaultparm);
    if ($lenientparm eq 'default') {
        $lenientparm = $defaultparm;
    }
    $lenientparm =
        &Apache::response::get_response_param($part.'_'.$id,'lenient',$lenientparm);
    if ($lenientparm eq 'default') {
        $lenientparm = $defaultparm;
    }
    if ($lenientparm=~/^0|off|no$/i) {
        $nonlenient = 1;
    } elsif ($lenientparm=~/^1|on|yes$/i) {
        $nonlenient = 0;
    }
    if (!$nonlenient) {
        $Apache::inputtags::leniency=1;
    }
    return $nonlenient;
}

sub getfoilcounts {
  my ($max)=@_;
  # +1 since instructors will count from 1
  my $count = $#{ $Apache::response::foilgroup{'names'} }+1;
  if (&Apache::response::showallfoils()) { $max=$count; }
  if ($count>$max) { $count=$max } 
  &Apache::lonxml::debug("Count is $count from $max");
  return $count;
}

sub whichfoils {
    my ($max,$randomize)=@_;
    return &Apache::response::whichorder($max,$randomize,
					 &Apache::response::showallfoils(),
					 \%Apache::response::foilgroup);
}

sub displayanswers {
    my ($max,$randomize,@opt)=@_;
    my @names;
    if (ref($Apache::response::foilgroup{'names'}) eq 'ARRAY') {
        @names = @{ $Apache::response::foilgroup{'names'} };
    }
    return if (!@names);
    my @whichopt = &whichfoils($max,$randomize);
    my $result;
    if ($Apache::lonhomework::type eq 'exam') {
	my $i = 0;
	my %opt = map { ($_,$i++) } @opt;
	
	$i = 0;
	my @alphabet = ('A'..'Z');
	foreach my $name (@whichopt) {
	    $result.=&Apache::response::answer_header('optionresponse',$i++);
	    $result.=&Apache::response::answer_part('optionresponse',
						    $alphabet[$opt{$Apache::response::foilgroup{$name.'.value'}}]);
	    $result.=&Apache::response::answer_part('optionresponse',
						    $Apache::response::foilgroup{$name.'.value'});
	    $result.=&Apache::response::answer_footer('optionresponse');
	}
    } else {
	$result=&Apache::response::answer_header('optionresponse');
	foreach my $name (@whichopt) {
	    $result.=&Apache::response::answer_part('optionresponse',
						    $Apache::response::foilgroup{$name.'.value'});
	}
	$result.=&Apache::response::answer_footer('optionresponse');
    }
    return $result;
}

sub check_box_opt {
    my ($target,$checkboxvalue,@opt)=@_;
# Check if we are in checkbox mode: checkboxvalue specified, on web, only two options.
# If so, return "checked" value
    if ($#opt!=1) { return ''; }
    unless (($target eq 'web') || ($target eq 'grade')) { return ''; }
    return $checkboxvalue;
}

sub check_for_invalid {
    my ($names,$options) = @_;
    my %bad_names;
    foreach my $name (@{ $names }) {
	my $value=$Apache::response::foilgroup{$name.'.value'};
	my $found=0;
	foreach my $option (@{ $options }) {
	    if ($value eq $option) { $found=1; }
	}
	if (!$found) { $bad_names{$name}=$value; }
    }
    if (%bad_names) {
	my $error=&mt('The question can not be gotten correct, '.
	    'the following foils in the &lt;optionresponse&gt; '.
	    'have invalid correct options').' <br /><tt>'.
	    join('<br />',(map { $_=&mt("[_1] with value [_2]",$_,$bad_names{$_}) } (keys(%bad_names)))).
	    "</tt>";
	&Apache::lonxml::error($error);
    }
}

sub displayfoils {
  my ($target,$max,$randomize,$TeXlayout,$checkboxvalue,$checkboxchoices,
      $tex_option_switch, $no_tfprompt, $maxcheck, $mincheck, @opt)=@_;
  my @names; 
  if (ref($Apache::response::foilgroup{'names'}) eq 'ARRAY') {
      @names = @{ $Apache::response::foilgroup{'names'} };
  }
  unless (@names > 0) { return;}
  my @truelist;
  my @falselist;
  my $result;  
  my $name;
  my $displayoptionintex=1;
  my @alphabet = ('A'..'Z');
  my @whichopt = &whichfoils($max,$randomize);
  unless (($target eq 'tex') && ($tex_option_switch eq 'nochoice')) {
      &check_for_invalid(\@whichopt,\@opt);
  }
  my $part=$Apache::inputtags::part;
  my $id=$Apache::inputtags::response[-1];
  my $break;
  if ( ($target ne 'tex') &&
       &Apache::response::show_answer() ) {
    my $temp=1;
    foreach $name (@whichopt) {
	my $text=$Apache::response::foilgroup{$name.'.text'};
        my $lastresp;
        if ((($env{'form.grade_username'} eq '') && ($env{'form.grade_domain'} eq '')) ||
            (($env{'form.grade_username'} eq $env{'user.name'}) &&
             ($env{'form.grade_domain'} eq $env{'user.domain'}))) {
            $lastresp = $Apache::lonhomework::history{"resource.$part.$id.submission"};
        } else {
            unless (($Apache::lonhomework::history{"resource.$part.type"} eq 'anonsurvey') ||
                    ($Apache::lonhomework::history{"resource.$part.type"} eq 'anonsurveycred')) {
                $lastresp = $Apache::lonhomework::history{"resource.$part.$id.submission"};
            }
        }
	my %lastresponse=&Apache::lonnet::str2hash($lastresp);
	my $lastopt=$lastresponse{$name};
	if ($text!~/^\s*$/) { $break='<br />'; }
	$result.=$break;
	if ($target eq 'web') {
	    my $value=$Apache::response::foilgroup{$name.'.value'};
	    if (!($text=~s|<drawoptionlist\s*/>|$value|)) {
		if ($text=~/^\s*$/) {
		    $text=$value.$text;
		} else {
		    $text='<b>'.$value.':</b> '.$text;
		}
	    } else {
                if (@whichopt > 1) {
                    $text='&#149;'.$text;
                }
	    }
	    $result.=$text."\n";
	}
      if ($Apache::lonhomework::type eq 'exam') {
	  $result.=&webbubbles(\@opt,\@alphabet,$temp,$lastopt);
      }
      $temp++;
    }
  } else {
    my $temp=1;
    my %lastresponse;
    my $newvariation;
    if ((($Apache::lonhomework::history{"resource.$part.type"} eq 'randomizetry') ||
        ($Apache::lonhomework::type eq 'randomizetry')) &&
        ($Apache::inputtags::status[-1] eq 'CAN_ANSWER')) {
        if ($env{'form.'.$part.'.rndseed'} ne
            $Apache::lonhomework::history{"resource.$part.rndseed"}) {
            $newvariation = 1;
        }
    }
    unless ($newvariation) {
        if ((($env{'form.grade_username'} eq '') && ($env{'form.grade_domain'} eq '')) ||
            (($env{'form.grade_username'} eq $env{'user.name'}) &&
             ($env{'form.grade_domain'} eq $env{'user.domain'}))) {
                 %lastresponse =
                     &Apache::lonnet::str2hash($Apache::lonhomework::history{"resource.$part.$id.submission"});
        } else {
            unless (($Apache::lonhomework::history{"resource.$part.type"} eq 'anonsurvey') ||
                    ($Apache::lonhomework::history{"resource.$part.type"} eq 'anonsurveycred')) {
                %lastresponse =
                    &Apache::lonnet::str2hash($Apache::lonhomework::history{"resource.$part.$id.submission"});
            }
        }
    }
    my $internal_counter=$Apache::lonxml::counter;
    my $checkboxopt;
    if ($target eq 'web') {
        $checkboxopt=&check_box_opt($target,$checkboxvalue,@opt);
    }
    if ($checkboxopt && (!$no_tfprompt)) {
       $result.='<br />'.
                ($checkboxchoices?&mt('Choices: ').'<b>'.$opt[0].','.$opt[1].'</b>. ':'').
                 &mt('Select all that are [_1].','<b>'.$checkboxopt.'</b>');
    }
    foreach $name (@whichopt) {
      my $text=$Apache::response::foilgroup{$name.'.text'};
      if ($text!~/^\s*$/) {
	  if ($target eq 'tex') {
	      $break='\vskip 0 mm ';
	  } elsif ($target eq 'web') {
	      $break='<br />';
	  }
      }
      my $lastopt=$lastresponse{$name};
      my $optionlist="<option></option>\n";

      if($target eq 'tex' and $env{'form.pdfFormFields'} eq 'yes'
         && $Apache::inputtags::status[-1] eq 'CAN_ANSWER') {
          my $fieldname = $env{'request.symb'}.'&part_'.$Apache::inputtags::part.'&optionresponse'.'&HWVAL_'.$Apache::inputtags::response['-1'].':'.$temp;
          $optionlist =  &Apache::lonxml::print_pdf_start_combobox($fieldname);
      }

      foreach my $option (@opt) {
	  my $escopt=&HTML::Entities::encode($option,'\'"&<>');
          if ($option eq $lastopt) {
              if ($target eq 'tex' && $env{'form.pdfFormFields'} eq 'yes'
                  && $Apache::inputtags::status[-1] eq 'CAN_ANSWER'
                  && $Apache::lonhomework::type ne 'exam') {
                  $optionlist .= &Apache::lonxml::print_pdf_add_combobox_option($option);
              } else {
                  $optionlist.="<option value='".$escopt."' selected=\"selected\">$option</option>\n";
              }
          } else {
              if ($target eq 'tex' && $env{'form.pdfFormFields'} eq 'yes'
                  && $Apache::inputtags::status[-1] eq 'CAN_ANSWER'
                  && $Apache::lonhomework::type ne 'exam') {
                  $optionlist .= &Apache::lonxml::print_pdf_add_combobox_option($option);
              } else {
                  $optionlist.="<option value='".$escopt."'>$option</option>\n";
              }
          }
      }
      if ($target ne 'tex') {
	  if ($Apache::lonhomework::type ne 'exam') {
# we are on the web, this is not an exam, and the problem can be answered
              if ($checkboxopt) {
# generate checkboxes
                  my $fieldname=$Apache::inputtags::response['-1'].':'.$temp;
                  my $altopt=$opt[0];
                  if ($opt[0] eq $checkboxopt) {
                     $altopt=$opt[1];
                  }
                  my $defopt=$lastopt;
                  unless ($defopt) { $defopt=$altopt; }
                  my $escdefopt=&HTML::Entities::encode($defopt,'\'"&<>');
                  my $esccheckboxopt=&HTML::Entities::encode($checkboxopt,'\'"&<>');
                  my $escaltopt=&HTML::Entities::encode($altopt,'\'"&<>');
# checkboxopt is how the box is labelled
# altopt is the alternative option
# lastopt is what the user submitted before
# defopt is what the field is going to start out with: either previous choice or altopt
# fieldname is this input field's name after HWVAL_
                  $optionlist='<input type="hidden" name="HWVAL_'.$fieldname.'" value="'.$escdefopt.'" />'.
                  '<input type="checkbox" name="HWCHK_'.$fieldname.'" onclick="javascript:if (this.form.elements[\'HWCHK_'.
                  $fieldname.'\'].checked) { this.form.elements[\'HWVAL_'.$fieldname.'\'].value=\''.$esccheckboxopt.'\'; } else { this.form.elements[\'HWVAL_'.$fieldname.'\'].value=\''.$escaltopt.'\'; };javascript:setSubmittedPart(\''.$part.'\');"'.($defopt eq $checkboxopt?' checked="checked"':'')." />\n";
              } else {
# classic selection list
	          $optionlist='<select onchange="javascript:setSubmittedPart(\''.
		  $part.'\');" name="HWVAL_'.
		  $Apache::inputtags::response['-1'].':'.$temp.'">'.
		  $optionlist."</select>\n";
              }
	  } else {
	      $optionlist='<u>'.('&nbsp;'x10).'</u>';
	  }
	  if ($text=~s|<drawoptionlist\s*/>|$optionlist|) {
	      if ($Apache::lonhomework::type ne 'exam') {
                  if (@whichopt > 1) {  
                      $text='&#149;'.$text;
                  }
	      }
	  } else {
	      if ($Apache::lonhomework::type ne 'exam') {
		  $text=$optionlist.$text;
	      }
	  }
	  $result.=$break.$text."\n";
	  if ($Apache::lonhomework::type eq 'exam') {
	      $result.=&webbubbles(\@opt,\@alphabet,$temp,$lastopt);
	  }
	  $temp++;
      } else {
	  my $texoptionlist='';
	  if ($displayoptionintex &&
	      $Apache::lonhomework::type ne 'exam') {
	      $texoptionlist = &optionlist_correction($TeXlayout,@opt);
	  }
	  if ($text=~/<drawoptionlist\s*\/>/) {
	      $text=~s|<drawoptionlist\s*\/>| \\makebox\[0\.3in\]\[b\]\{\\hrulefill\} |g;
	  }

	  if ($text=~m/\\item /) {
	      if ($Apache::lonhomework::type eq 'exam') {
	          $text=~s/\\item/\\vskip 2 mm/;
	      } elsif ($env{'form.pdfFormFields'} ne 'yes') {
                  $result.= $texoptionlist.$text;
              }
          } else {
	      if ($Apache::lonhomework::type eq 'exam') {
		  $result.= $texoptionlist.'  '.$text;
	      } elsif ($env{'form.pdfFormFields'} ne 'yes') {
		  if ($text=~/\S/) {
                      $result.= $texoptionlist.'\vspace*{-2 mm}\item '.$text;
                  } else {
                      $result.= $texoptionlist;
                  }
	      }
	  }
	  if ($Apache::lonhomework::type eq 'exam') {
	      $result.='\vskip -1 mm\noindent';
              my ($numrows,$bubbles_per_row) = &getnumrows(scalar(@opt)); 
              if ($numrows == 1) {  
                  $result .= '\textbf{'.$internal_counter.'}. \vskip -3mm';
              } else {
                  my $linetext;
                  for (my $i=0; $i<$numrows; $i++) {
                      $linetext .= $internal_counter+$i.', ';
                  }
                  $linetext =~ s/,\s$//;
                  $result .= '\small {\textbf{'.$linetext.'}} '.
                             '\hskip 2 mm {\footnotesize '.
                             &mt('(Bubble once in [_1] lines)',$numrows).
                             '} \vskip 1 mm';
              }
              $result.= &bubbles(\@alphabet,\@opt,undef,undef,$numrows,
                                 $bubbles_per_row,$internal_counter).
                        ' \strut ';
	      $internal_counter += $numrows;
	  }
          if ($target eq 'tex' && $env{'form.pdfFormFields'} eq 'yes'
              && $Apache::inputtags::status[-1] eq 'CAN_ANSWER'
              && $Apache::lonhomework::type ne 'exam') {
              $text =~ s/\\item//m;
              $result .= " $optionlist ". &Apache::lonxml::print_pdf_end_combobox($text).'\strut \\\\';
              $temp++;
          }
	  $displayoptionintex=0;
      }
    }
  }

  if ($target eq 'web') {
      my $data = [\@whichopt,'submissiongrading'];
      my $questiontype;
      if ($Apache::lonhomework::type eq 'randomizetry') {
          $questiontype = $Apache::lonhomework::type,
      }
      &Apache::response::setup_prior_tries_hash(\&Apache::rankresponse::format_prior_answer,$data,$questiontype);
  }

  if ($target ne 'tex') {
      return $result.$break;
  } else {
      return $result;
  }
}


sub optionlist_correction {
    my ($TeXlayout,@options) = @_;
    my $texoptionlist='\\item [] '.&mt('Choices:').' ';
    if ($TeXlayout eq 'vertical') {$texoptionlist='\\item []';}
    if (scalar(@options) > 0) {
	foreach my $option (@options) {
	    $texoptionlist.='{\bf '.
		&Apache::lonxml::latex_special_symbols($option).
		'}';
	    if ($TeXlayout eq 'vertical') {
		$texoptionlist.=' \vskip 0 mm ',
	    } else {
		$texoptionlist.=', ';
	    }
	}
	$texoptionlist=~s/, $//;
	if ($TeXlayout ne 'vertical') {$texoptionlist.='.';}
    } else {
	if ($TeXlayout ne 'vertical') {$texoptionlist='\\item [] \\vskip -5 mm';}
    }
    return $texoptionlist;
}


sub webbubbles {
    my ($ropt,$ralphabet,$temp,$lastopt)=@_;
    my @opt=@$ropt; 
    my @alphabet=@$ralphabet;
    my $result='';
    my $number_of_bubbles = $#opt + 1;
    $result.= '<table border="1"><tr>';
    for (my $ind=0;$ind<$number_of_bubbles;$ind++) {
	my $checked='';
	if ($lastopt eq $opt[$ind]) {
	    $checked=' checked="on" ';
	}
	$result.='<td><input type="radio" name="HWVAL_'.
	    $Apache::inputtags::response['-1'].':'.$temp.
	    '" value="'.$opt[$ind].'" '.$checked.' />';
	if ($alphabet[$ind]) {
	    $result.=$alphabet[$ind].': ';
	}
	$result.=$opt[$ind].'</td>';
    }
    $result.='</tr></table>';
    return $result;
}


sub bubbles {
    my ($ralphabet,$ropt,$response,$max_width,$numrows,$bubbles_per_row,
        $internal_counter) = @_;
    my @alphabet = @$ralphabet;
    my @opt = @$ropt;
    my ($result,$head,$line) =('','','');
    my $number_of_bubbles = $#opt + 1;
    my $current_length = 0;
    my $textwidth;
    if (defined($max_width)) {
	$textwidth=$max_width;
	&Apache::lonxml::debug("Max width passed in: $max_width");
    } elsif ($env{'form.textwidth'} ne '') {
	$env{'form.textwidth'}=~/(\d+)/;
	$textwidth=$1;
	&Apache::lonxml::debug("Max width from form: $textwidth");
    } else {
	$env{'form.textwidth'}=~/(\d*)\.?(\d*)/;
	$textwidth=$1.'.'.$2;
	&Apache::lonxml::debug("Max width defaults? $textwidth");
    }
    &Apache::lonxml::debug("Final maxwidth: $textwidth");
    for (my $ind=0;$ind<=$number_of_bubbles;$ind++) {
        my $item;
        if ($numrows > 1) {
            my $num = $internal_counter+int($ind/$bubbles_per_row);
            my $idx = int($ind % $bubbles_per_row);
            $item = $num.$alphabet[$idx];
        } else {
            $item = $alphabet[$ind];
        }
	my $leftmargin;
	$opt[$ind]=&Apache::lonxml::latex_special_symbols($opt[$ind]);
	if ($response eq 'rankresponse') {$opt[$ind]='Rank '.$opt[$ind];}
	if ($ind==0) {$leftmargin=6;} else {$leftmargin=10;}

	$current_length += (length($opt[$ind])+length($item)+5)*2;
        
	if ($current_length<($textwidth-$leftmargin) and $ind!=$number_of_bubbles) {
            
	    $line.='\hskip 4 mm {\small \textbf{'.$item.'}}$\bigcirc$\hskip -1 mm & \hskip -3 mm {\small '.$opt[$ind].'} & ';
	    $head.='lr';
	} else {
	    $line=~s/\&\s*$//;
	    $result.='\vskip -1 mm\noindent\setlength{\tabcolsep}{2 mm}\renewcommand{\arraystretch}{1.25}\begin{tabular}{'.$head.'}'.$line.'\\\\\end{tabular}\vskip 0 mm';
	    $line = '\hskip 4 mm {\small \textbf{'.$item.'}}$\bigcirc$\hskip -1 mm & \hskip -3 mm {\small '.$opt[$ind].'} & ';
	    $head ='lr';
	    $current_length = (length($opt[$ind])+length($item))*2;
	}

    }
    return $result;
}

sub start_conceptgroup {
  my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
  $Apache::optionresponse::conceptgroup=1;
  %Apache::response::conceptgroup=();
  my $result;
  if ($target eq 'edit') {
    $result.=&Apache::edit::tag_start($target,$token,"Concept Grouped Foils");
    $result.=&Apache::edit::text_arg('Concept:','concept',$token,'50').
        &Apache::edit::end_row().&Apache::edit::start_spanning_row();
  }
  if ($target eq 'modified') {
    my $constructtag=&Apache::edit::get_new_args($token,$parstack,$safeeval,'concept');
    if ($constructtag) { $result = &Apache::edit::rebuild_tag($token); }
  }
  return $result;
}

sub end_conceptgroup {
  my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
  $Apache::optionresponse::conceptgroup=0;
  my $result='';
  if ($target eq 'web' || $target eq 'grade' || $target eq 'answer' ||
      $target eq 'tex' || $target eq 'analyze') {
    #if not there aren't any foils to display and thus no question
      &Apache::response::pick_foil_for_concept($target,
					       ['value','text','location'],
					       \%Apache::hint::option,
					       $parstack,$safeeval);
  } elsif ($target eq 'edit') {
    $result=&Apache::edit::end_table();
  }
  return $result;
}

sub insert_conceptgroup {
  my $result="\n\t\t<conceptgroup concept=\"\">".&insert_foil()."\n\t\t</conceptgroup>\n";
  return $result;
}

sub start_foil {
  my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
  my $result='';
  if ($target eq 'web' || $target eq 'tex' || $target eq 'analyze' ) {
      &Apache::lonxml::startredirection;
      if ($target eq 'analyze') {
	  &Apache::response::check_if_computed($token,$parstack,$safeeval,'value');
      }
  } elsif ($target eq 'edit') {
    $result=&Apache::edit::tag_start($target,$token,"Foil");
    my $level='-2';
    if ($$tagstack['-2'] eq 'conceptgroup') { $level = '-3'; }
    my @opt;
    eval '@opt ='.&Apache::lonxml::get_param('options',$parstack,$safeeval,$level);
    $result.=&Apache::edit::text_arg('Name:','name',$token);
    $result.= &Apache::edit::select_or_text_arg('Correct Option:','value',
					       ['unused',(@opt)],$token,'15');
    my $randomize=&Apache::lonxml::get_param('randomize',$parstack,
					     $safeeval,'-3');
    if ($randomize ne 'no') {
	$result.=&Apache::edit::select_arg('Location:','location',
					   ['random','top','bottom'],$token);
    }
    $result .=&Apache::edit::end_row().&Apache::edit::start_spanning_row();
  } elsif ($target eq 'modified') {
    my $constructtag=&Apache::edit::get_new_args($token,$parstack,$safeeval,
						 'value','name','location');
    if ($constructtag) { $result = &Apache::edit::rebuild_tag($token); }
  } 
  return $result;
}

sub end_foil {
  my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
  my $text ='';
  my $result = '';
  if ($target eq 'web' || $target eq 'tex' || $target eq 'analyze') { 
      $text=&Apache::lonxml::endredirection;
      if ($target eq 'tex') {$text=~s/\\strut\s*\\\\\s*\\strut/\\vskip 0 mm/;}
      if (($target eq 'tex') and ($Apache::lonhomework::type ne 'exam') and ($text=~/\S/)) {$text = '\vspace*{-2 mm}\item '.$text;} 
  }
  if ($target eq 'web' || $target eq 'grade' || $target eq 'answer' 
      || $target eq 'tex' || $target eq 'analyze') {
    my $value = &Apache::lonxml::get_param('value',$parstack,$safeeval);
    if ($target eq 'tex' && $Apache::lonhomework::type eq 'exam') {
	$text='\vskip 5mm $\triangleright$ '.$text;
    }
    if ($value ne 'unused') {
      my $name = &Apache::lonxml::get_param('name',$parstack,$safeeval);
      &Apache::lonxml::debug("Got a name of :$name:");
      if ($name eq "") {
	  &Apache::lonxml::warning(&mt('Foils without names exist. This can cause problems to malfunction.'));
	  $name=$Apache::lonxml::curdepth;
      }
      &Apache::lonxml::debug("Using a name of :$name:");
      if (defined($Apache::response::foilnames{$name})) {
	  &Apache::lonxml::error(&mt("Foil name [_1] appears more than once. Foil names need to be unique.",'<b><tt>'.$name.'</tt></b>'));
      }
      $Apache::response::foilnames{$name}++;
      my $location =&Apache::lonxml::get_param('location',$parstack,$safeeval);
      if ( $Apache::optionresponse::conceptgroup
	   && !&Apache::response::showallfoils() ) {
	push @{ $Apache::response::conceptgroup{'names'} }, $name;
	$Apache::response::conceptgroup{"$name.value"} = $value;
	$Apache::response::conceptgroup{"$name.text"} = $text;
	$Apache::response::conceptgroup{"$name.location"} = $location;
      } else {
	push @{ $Apache::response::foilgroup{'names'} }, $name;
	$Apache::response::foilgroup{"$name.value"} = $value;
	$Apache::response::foilgroup{"$name.text"} = $text;
	$Apache::response::foilgroup{"$name.location"} = $location;
      }
    }
  }
  if ($target eq 'edit') {
    $result.= &Apache::edit::tag_end($target,$token,'');
  }
  return $result;
}

sub start_drawoptionlist {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    if ($target !~ /^(?:meta|answer|modified|edit)$/) {
	return $token->[4];
    }
}

sub end_drawoptionlist {
    my ($target,$token,$tagstack,$parstack,$parser,$safeeval,$style)=@_;
    if ($target !~ /^(?:meta|answer|modified|edit)$/) {
	return $token->[2];
    }
}

sub insert_foil {
    return '
<foil name="" value="unused">
<startouttext />
<endouttext />
</foil>';
}

sub insert_drawoptionlist {
    return '<drawoptionlist />';
}

sub getnumrows {
    my ($numitems) = @_;
    my $bubbles_per_row;
    my $default_numbubbles = 10;
    if (($env{'form.bubbles_per_row'} =~ /^\d+$/) &&
        ($env{'form.bubbles_per_row'} > 0)) {
        $bubbles_per_row = $env{'form.bubbles_per_row'};
    } else {
        $bubbles_per_row = $default_numbubbles;
    }
    my $numrows = int ($numitems/$bubbles_per_row);
    if (($numitems % $bubbles_per_row) != 0) {
        $numrows ++;
    }
    return ($numrows,$bubbles_per_row);
}

1;
__END__
 
=head1 NAME

Apache::optionresponse.pm;

=head1 SYNOPSIS

Handles tags associated with showing a list of
options.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 HANDLER SUBROUTINE

start_optionresponse()

=head1 OTHER SUBROUTINES

=over

=item end_optionresponse()

=item start_foilgroup()

=item end_foilgroup()

=item getfoilcounts()

=item displayanswers()

=item check_for_invalid()

=item displayfoils()

=item optionlist_correction()

=item webbubbles()

=item bubbles()

=item start_conceptgroup()

=item end_conceptgroup()

=item insert_conceptgroup()

=item start_foil()

=item end_foil()

=item start_drawoptionlist()

=item end_drawoptionlist()

=item insert_foil()

=item insert_drawoptionlist()

=back

=cut
