# The LearningOnline Network with CAPA
# This will take annotations and then plug them into a page.
#
# $Id: admannotations.pm,v 1.37 2021/12/24 00:48:31 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
##################


package Apache::admannotations;

use strict;
use Apache::Constants qw(:common);
use Apache::loncommon();
use Apache::lonnet;
use Apache::lonlocal;

# --------------------------------------------------------------Put annotation

sub write_annotation {
    my ($symb, $enc, $annotation) = @_;

    $annotation=~s/^\s*$//;
    my $key = $symb;
    if (!$enc) {
	$key = 
	    &Apache::lonnet::clutter((&Apache::lonnet::decode_symb($symb))[2]);
    }

    &Apache::lonnet::put('nohist_annotations',{$key => $annotation});
    return;
}


# ------------------------------------------------------------Construct editor
sub js_get_symb {
    return <<ENDJS;
function get_symb() {
    var symb;
    if (typeof(window.opener.clientwindow) != 'undefined') {
	symb=window.opener.clientwindow.location.href;
    } else {
	symb=window.opener.location.href;
    }
    symb=window.opener.currentSymb;
    return symb;
}
ENDJS
}

sub construct_editor {
    my ($annotation,$symbnew) = @_;
    if ($annotation=~/^error:/) { $annotation=''; }

    my $get_symb = &js_get_symb();

    my %lt = &Apache::lonlocal::texthash(
        'close no save' => 'Close',
        'save'          => 'Save',
        'cancel'        => 'Cancel',
    );

    my $js = <<ENDJS;
<script type="text/javascript">
var timeout;

$get_symb

function changed() {
    var symbnew=get_symb();
    if (symbnew!=document.annotInfo.symbold.value) {
	document.annotInfo.symbnew.value=symbnew;
        document.annotInfo.submit();
    }
    timeout=setTimeout('changed();','1000');
}

function renameButton(){
    document.getElementById("close").value="$lt{'cancel'}";
}

</script>
ENDJS

    my %load_items = ('onload'   => "timeout=setTimeout('changed()','1000')",
		      'onunload' => 'clearTimeout(timeout);');
    my $start_page =
	&Apache::loncommon::start_page('Annotations',$js,
				       {'only_body'   => 1,
					'add_entries' => \%load_items,});

    my $end_page =
        &Apache::loncommon::end_page();
    
	 
    return(<<END_HTML)
$start_page
 <center>
  <form name="annotInfo" method="post" action="">
    <textarea name="annotation" wrap="on" rows="12" cols="36" onfocus="javascript:renameButton();">$annotation</textarea><br />
    <input type="hidden" name="symbold" value="$symbnew" />
    <input type="hidden" name="symbnew" value="" />
    <input type="button" name="send" value="$lt{'save'}"
           onclick="javascript:this.form.symbnew.value=get_symb();this.form.submit();" />
    <input type="button" id="close" name="close" value="$lt{'close no save'}" 
           onclick="javascript:window.close();" />
  </form>
 </center>
$end_page
END_HTML
}

# ----------------------------------------------------Constructs error window

sub construct_error {
    my ($annot_error,$button_name) = @_;

    my $get_symb = &js_get_symb();
    my $js = <<ENDJS;
<script type="text/javascript">
$get_symb
</script>
ENDJS

    my $start_page = &Apache::loncommon::start_page('Annotations',$js,
						    {'only_body'   => 1,});
    my $end_page =  &Apache::loncommon::end_page();

    my %lt = &Apache::lonlocal::texthash(
        'close'  => 'Close',
        'button' => $button_name,
    );

    return(<<END_HTML2)
$start_page
 <center>
  <form name="annotInfo" method="post" action="">
<table bgcolor="#FFFFFF" width="100%" height="90%" align="center">
<td>
<font size="+1"><i>
$annot_error
</i></font>
</td>
</table>
    <input type="hidden" name="symbold" value="" />
    <input type="hidden" name="symbnew" value="" />
    <input type="button" name="send" value="$lt{'button'}"
           onclick="javascript:this.form.symbnew.value=get_symb();this.form.submit();" />
    <input type="button" name="close" value="$lt{'close'}" 
           onclick="javascript:window.close();" />
  </form>
 </center>
$end_page
END_HTML2
}

sub block_check {
    my ($clientip) = @_;
    my ($blocked,$blocktext) =
        &Apache::loncommon::blocking_status('annotate',$clientip);
    if ($blocked) {
        my %lt = &Apache::lonlocal::texthash(
                                              'close' => 'Close',
                                            );
        my $start_page =
            &Apache::loncommon::start_page('Annotations',undef,
                                           {'only_body'   => 1,});
        my $end_page = '<form name="annotInfo" method="post" action="">'."\n".
                       '<input type="button" id="close" name="close" value="'.$lt{'close'}.'"'.
                       ' onclick="javascript:window.close();" />'."\n".
                       '</form>'."\n".
                       &Apache::loncommon::end_page();
        return ($blocked,$start_page.$blocktext.$end_page);
    } else {
        return ();
    }
}

# ---------------------------------------------------------------Main Handler

sub handler {
    my ($r) = @_;

    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;
    return OK if $r->header_only;

    my $clientip = &Apache::lonnet::get_requestor_ip($r);
    if ($clientip ne '') {
        my ($blocked,$output) = &block_check($clientip);
        if ($blocked) {
            $r->print($output);
            return OK;
        }
    }

    my $page;

    my ($symb_old,$symb_old_enc) = &Apache::loncommon::clean_symb($env{'form.symbold'},1);
    my ($symb_new,$symb_new_enc) = &Apache::loncommon::clean_symb($env{'form.symbnew'},1);

    my $annotation=$env{'form.annotation'};

    if ($symb_old) {
	&write_annotation($symb_old,$symb_old_enc,$annotation);
    }
    if (exists($env{'form.symbnew'})) {
	unless ($symb_new) {
	    $page=&construct_error(&mt("Cannot annotate current window. Please point your browser to a LON-CAPA page and then 'continue'."),"continue");
	} else {
	    if ($symb_old ne $symb_new) {
		$annotation=&Apache::loncommon::get_annotation($symb_new,$symb_new_enc);
	    }
	    # pass same symb through.
	    $page=&construct_editor($annotation,$env{'form.symbnew'});
	}
    }
    $r->print($page);
    return OK;
}

1;
__END__

=pod

=head1 NAME

Apache::admannotations

=head1 SYNOPSIS

This will take annotations and then plug them into a page.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 OVERVIEW

(empty)

=head1 SUBROUTINES

=over write_annotation()

Put annotation

=item js_get_symb()

Construct editor

=item construct_editor()

=item construct_error()

=item handler()

=back

=cut


