# The LearningOnline Network
# Generate a menu page containing links to server status pages accessible
# to user. 
#
# $Id: domainstatus.pm,v 1.7 2014/03/17 02:36:56 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
#
package Apache::domainstatus;

use strict;
use Apache::Constants qw(:common);
use Apache::lonnet;
use Apache::loncommon;
use Apache::lonhtmlcommon;
use Apache::lonlocal;
use LONCAPA::lonauthcgi;

sub handler {
    my $r = shift;
    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;
    if ($r->header_only) {
        return OK;
    }

    &Apache::lonlocal::get_language_handle();
    my $dom = &Apache::lonnet::default_login_domain();

    if (($env{'user.domain'} ne '') && ($env{'user.domain'} ne 'public')) {
        $dom = $env{'user.domain'};
        if ($env{'request.role.domain'} ne '') {
            $dom = $env{'request.role.domain'};
        }
    }
    my $domdesc = &Apache::lonnet::domain($dom,'description');

    my %candisplay;
    my $titles = &LONCAPA::lonauthcgi::serverstatus_titles();
    if (ref($titles) eq 'HASH') {
        foreach my $page (keys(%{$titles})) {
            if (&LONCAPA::lonauthcgi::can_view($page)) {
                $candisplay{$page} = 'F';       
            } elsif (&LONCAPA::lonauthcgi::check_ipbased_access($page)) {
                $candisplay{$page} = 'F';
            }
        }
    }

    &Apache::lonhtmlcommon::clear_breadcrumbs();
    &Apache::lonhtmlcommon::add_breadcrumb
    ({href=>"/adm/domainstatus",
      text=>"Server Status Information"});

    $r->print(&Apache::loncommon::start_page('Server Status').
              &Apache::lonhtmlcommon::breadcrumbs('Server Status Information'));

    if (keys(%candisplay) > 0) {
        $r->print('<h2>'.&mt('Server Utilities for Domain: [_1]','<i>'.$domdesc.'</i>').
                  '</h2>'.&print_status_menu(\%candisplay,$dom));
    } else {
        $r->print(
            '<h2>'.&mt('No information available').'</h2>'
           .'<p class="LC_warning">'
           .&mt('Your current role and/or IP address does not have permission to display information about server status for servers in the domain: [_1].',
                '<b>'.$domdesc.'</b> ('.$dom.')')
           .'</p>'
        ); 
    }
    $r->print(&Apache::loncommon::end_page());
    return OK;
}

sub print_status_menu {
    my ($candisplay,$dom) = @_;
    return '' if (ref($candisplay) ne 'HASH');
    return &Apache::lonhtmlcommon::generate_menu(
               &servermenu_items($candisplay,$dom));
}

sub servermenu_items {
    my ($candisplay,$dom) = @_;
    my $titles = &LONCAPA::lonauthcgi::serverstatus_titles();
    my $linknames = &serverstatus_links();
    my @menu;
    if ((ref($candisplay) eq 'HASH') && (ref($titles) eq 'HASH') && 
        (ref($linknames) eq 'HASH')) { 
        @menu = ( {categorytitle => 'Status information',
                  items =>
                 [{
                 linktext => $linknames->{'userstatus'},
                 icon => '',
                 alttext => '',
                 #help => 'Domain_Coordination_Userstatus',
                 url => '/cgi-bin/userstatus.pl',
                 permission => $candisplay->{'userstatus'},
                 linktitle =>  $titles->{'userstatus'},
             },
             {
                 linktext => $linknames->{'lonstatus'},
                 icon => '',
                 alttext => '',
                 #help => 'Domain_Coordination_Connection_Status',
                 url => '/lon-status/',
                 permission => $candisplay->{'lonstatus'},
                 linktitle => $titles->{'lonstatus'},
             },
             {
                 linktext => $linknames->{'server-status'},
                 icon => '',
                 alttext => '',
                 #help => 'Domain_Coordination_Apache_Status',
                 url => '/server-status',
                 permission => $candisplay->{'server-status'},
                 linktitle => $titles->{'server-status'},
             },
             {
                 linktext => $linknames->{'clusterstatus'},
                 icon => '',
                 alttext => '',
                 #help => 'Domain_Coordination_Apache_Status',
                 url => '/cgi-bin/clusterstatus.pl',
                 permission => $candisplay->{'clusterstatus'},
                 linktitle => $titles->{'clusterstatus'},
             },
             {
                 linktext => $linknames->{'codeversions'},
                 icon => '',
                 alttext => '',
                 #help => 'Domain_Coordination_Code_Versions',
                 url => '/cgi-bin/lonversions.pl',
                 permission => $candisplay->{'codeversions'},
                 linktitle => $titles->{'codeversions'},
             },
             {
                 linktext => $linknames->{'checksums'},
                 icon => '',
                 alttext => '',
                 #help => 'Domain_Coordination_Code_Checksums',
                 url => '/cgi-bin/lonmodulecheck.pl',
                 permission => $candisplay->{'checksums'},
                 linktitle => $titles->{'checksums'},
             },
             {
                 linktext => $linknames->{'diskusage'},
                 icon => '',
                 alttext => '',
                 #help => 'Domain_Coordination_Disk_Usage',
                 url => '/cgi-bin/quotacheck.pl?domain='.$dom,
                 permission => $candisplay->{'diskusage'},
                 linktitle => $titles->{'diskusage'},
             },
             {
                 linktext => $linknames->{'uniquecodes'},
                 icon => '',
                 alttext => '',
                 #help => 'Domain_Coordination_Uniquecodes',
                 url => '/cgi-bin/listcodes.pl?domain='.$dom.'&amp;format=html',
                 permission => $candisplay->{'uniquecodes'},
                 linktitle => $titles->{'uniquecodes'},
             },
             {
                 linktext => $linknames->{'showenv'},
                 icon => '',
                 alttext => '',
                 #help => 'Domain_Coordination_User_Environment',
                 url => '/adm/test',
                 permission => $candisplay->{'showenv'},
                 linktitle => $titles->{'showenv'},
             }]},
             {categorytitle => 'Server Actions',
             items =>
             [{
                 linktext => $linknames->{'loncron'},
                 icon => '',
                 alttext => '',
                 #help => 'Domain_Coordination_Connection_Reload',
                 url => '/cgi-bin/loncron.pl',
                 permission => $candisplay->{'loncron'},
                 linktitle => $titles->{'loncron'},
             },
             {
                 linktext => $linknames->{'takeoffline'},
                 icon => '',
                 alttext => '',
                 #help => 'Domain_Coordination_Server_Offline',
                 url => '/cgi-bin/takeoffline.pl',
                 permission => $candisplay->{'takeoffline'},
                 linktitle => $titles->{'takeoffline'},
             },
             {
                 linktext => $linknames->{'takeonline'},
                 icon => '',
                 alttext => '',
                 #help => 'Domain_Coordination_Server_Online',
                 url => '/cgi-bin/takeonline.pl',
                 permission => $candisplay->{'takeonline'},
                 linktitle => $titles->{'takeonline'},
             }]},
             {categorytitle => 'Metadata Information',
             items =>
             [{
                 linktext => $linknames->{'metadata_keywords'},
                 icon => '',
                 alttext => '',
                 #help => 'Domain_Coordination_Metadata_Keywords',
                 url => '/cgi-bin/metadata_keywords.pl',
                 permission => $candisplay->{'metadata_keywords'},
                 linktitle => $titles->{'metadata_keywords'},
             },
             {
                 linktext => $linknames->{'metadata_harvest'},
                 icon => '',
                 alttext => '',
                 #help => 'Domain_Coordination_Metadata_Harvest',
                 url => '/cgi-bin/metadata_harvest.pl',
                 permission => $candisplay->{'metadata_harvest'},
                 linktitle => $titles->{'metadata_harvest'},
             }]},
           );
        }
    return @menu;
}

sub serverstatus_links {
    my $linkname = { 
                    'userstatus' => 'User Sessions',
                    'lonstatus' => 'Connection Status',
                    'server-status' => 'Apache Server Status',
                    'clusterstatus' => 'Domain Status',
                    'codeversions' => 'LON-CAPA Modules',
                    'checksums'    => 'Check for LON-CAPA Module changes',   
                    'diskusage'    => 'Display quotas and usage for Course/Community Content',
                    'showenv' => 'User Environment for current log-in',
                    'loncron' => 'Update Connections and Refresh Status Information',
                    'uniquecodes' => "List Domain's Courses with Assigned Six Character codes",
                    'takeoffline' => 'Replace log-in page with offline notice',
                    'takeonline' => 'Replace offline notice with log-in page',
                    'metadata_keywords' => 'Display Metadata Keywords',
                    'metadata_harvest' => 'Harvest Metadata Keywords',
    };
    return $linkname;
}

1;
