# The LearningOnline Network with CAPA
# displays the blocking status table
#
# $Id: lonblockingstatus.pm,v 1.14.2.6 2021/12/24 21:32:03 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
#
package Apache::lonblockingstatus;

use strict;
use Apache::Constants qw(:common);
use Apache::loncommon();
use Apache::lonnet;
use Apache::lonlocal;
use LONCAPA qw(:DEFAULT :match);

sub handler {
    my $r = shift;
    &Apache::loncommon::no_cache($r);
    &Apache::loncommon::content_type($r,'text/html');

    $r->send_http_header;
    return OK if $r->header_only;

    my (%activities,$activity,$origurl,$origsymb);
    map { $activities{$_} = 1; } ('boards','chat','com','blogs','about','groups','port','printout','docs','grades','passwd','search','wishlist','annotate');

    # determine what kind of blocking we want details for
    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},['activity','url','symb']);
    $activity = $env{'form.activity'};

    my $title = 'Communication Blocking Status Information';

    if (($activity eq 'docs') || ($activity eq 'printout') ||
        ($activity eq 'grades') || ($activity eq 'passwd') ||
        ($activity eq 'search') || ($activity eq 'wishlist') ||
        ($activity eq 'annotate')) {
        $title = 'Blocking Status Information';
        if ($activity eq 'docs') {
            $origurl = $env{'form.url'};
            $origsymb = $env{'form.symb'};
        }
    }
    $r->print(&Apache::loncommon::start_page($title,undef,
                                            {'only_body' => 1}));

    if (($activity eq '') || (!$activities{$activity})) {
        $r->print('<p class="LC_error">'.&mt('Error: unknown activity type blocked').'</p>');
    } elsif (($activity eq 'docs') && ($origurl eq '') && ($origsymb eq '')) {
        $r->print('<p class="LC_error">'.&mt('Error: could not determine what content was blocked from access').'</p>');
    } else {
        my $clientip = &Apache::lonnet::get_requestor_ip($r);
        $r->print(&blockpage($activity,$origurl,$origsymb,$clientip));
    }

    $r->print(&Apache::loncommon::end_page());

    return OK;
}

sub blockpage {
    my ($activity,$origurl,$origsymb,$clientip) = @_;

    # in case of a portfolio block we need to determine the owner of the files
    # we're trying to look at. This information is passed via query string.
    my ($uname, $udom);

    if (($activity eq 'port') || ($activity eq 'about') ||
        (($activity eq 'passwd') && ($env{'user.name'} eq 'public') && ($env{'user.domain'} eq 'public'))) {
        &Apache::loncommon::get_unprocessed_cgi(
            $ENV{'QUERY_STRING'}, ['udom', 'uname'] );

        ($uname, $udom) = ($env{'form.uname'}, $env{'form.udom'});
        if (($uname !~ /^$match_username$/) || ($udom !~ /^$match_domain$/)) {
            if ($activity eq 'port') {
                return '<span class="LC_error">'.
                       &mt('Information about the owner of the portfolio files you were trying to view was missing or invalid.').
                       '</span><br />'.
                       &mt('Without valid owner information, the reason why access is blocked can not be determined');
            } elsif ($activity eq 'about') {
                return '<span class="LC_error">'.
                       &mt('The username and/or domain for the User Information page you were trying to view was missing or invalid.').
                       '</span><br />'.
                       &mt('Without valid information, the reason why access is blocked can not be determined');
            } else {
                return '<span class="LC_error">'.
                       &mt('Information about the username and/or domain for which you were trying to reset a password was missing or invalid.').
                       '</span><br />'.
                       &mt('Without valid information, the reason why access is blocked can not be determined');
            }
        }
    }

    # retrieve start/end of possible active blocking
    my (%setters,$startblock,$endblock,$triggerblock,$by_ip,$blockdom);

    if ($activity eq 'docs') {
        my ($cdom,$cnum);
        if ($env{'request.course.id'}) {
            $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
            $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
        }
        my $cancheck;
        if (($cdom ne '') && ($cnum ne '')) {
            if ($origsymb =~ m{^uploaded/($match_domain)/($match_courseid)/}) {
                my ($crsdom,$crsnum) = ($1,$2);
                if (($cdom eq $crsdom) && ($cnum eq $crsnum)) {
                    $cancheck = 1;
                }
            } else {
                $cancheck = 1;
            }
        }
        if ($cancheck) {
            ($startblock,$endblock,$triggerblock) =
                &Apache::loncommon::blockcheck(\%setters,$activity,$clientip,$cnum,$cdom,$origurl,1,$origsymb,'blockingstatus');
        } else {
            return '<p class="LC_info">'.&mt('Could not determine why access is blocked.').'</p>';
        }
    } else {
        ($startblock,$endblock,$triggerblock,$by_ip,$blockdom) =
            &Apache::loncommon::blockcheck(\%setters,$activity,$clientip,$uname,$udom,$origurl,undef,$origsymb,'blockingstatus');
    }

    # nothing to do if there's no active blocking
    unless (($startblock && $endblock) || ($by_ip)) {
        if ($activity eq 'docs') {
            return '<p class="LC_info">'.&mt('Content no longer blocked from access').'</p>';
        }
        return '<p class="LC_info">'.&mt('Access no longer blocked for this activity').'</p>';
    }

    # lookup $activity -> description
                   #possible activity          #corresponding description
    my %descs = (
                   boards     => 'Discussion posts in this course',
                   chat       => 'Chat Room',
                   com        => 'This message',
                   blogs      => 'Blogs',
                   about      => 'User information pages',
                   groups     => 'Groups in this course',
                   printout   => 'Printout generation',
                   docs       => 'Course Content',
                   passwd     => 'Changing of passwords',
                   grades     => 'Course Gradebook',
                   search     => 'Content Search',
                   wishlist   => 'Stored Links',
                   annotate   => 'Annotations',
                );

    if ($activity eq 'groups' || $activity eq 'boards') {
        if (&Apache::loncommon::course_type() eq 'Community') {
            $descs{'boards'} = 'Discussion posts in this community';
            $descs{'groups'} = 'Groups in this community';
            $descs{'docs'} = 'Community Content';
            $descs{'grades'} = 'Community Gradebook';
        }
    }

    my $description = $descs{$activity};
    if ($activity eq 'port') {
        $description = &get_portfolio_category($uname,$udom,$by_ip,\%setters);
    }
    if ($description eq '') {
        $description = 'Communication';
    }

    my ($showstart,$showend,$output);
    unless ($by_ip) {
        $showstart = Apache::lonlocal::locallocaltime($startblock);
        $showend   = Apache::lonlocal::locallocaltime($endblock);
    }

    if ( ref($description) ne 'ARRAY' ) {
        #default: $description is one of the above descriptions
        if ($activity eq 'docs') {
            $output=&mt( 'Access to the content page you are attempting to'
                         . ' view will be unavailable between [_1] and [_2] because'
                         . ' access to selected '.$description.' is being blocked.'
                         ,$showstart, $showend);
        } elsif (($activity eq 'printout') || ($activity eq 'passwd') ||
                 ($activity eq 'grades') || ($activity eq 'search') ||
                 ($activity eq 'about') || ($activity eq 'wishlist') ||
                 ($activity eq 'annotate')) {
            if ($by_ip) {
                $output = mt( $description
                              . ' unavailable from your current IP address: [_1] '
                              . 'because this functionality is being blocked for certain IP address(es).'
                              ,$clientip);
            } else {
                $output = mt( $description
                              . ' will be unavailable between [_1] and [_2] because'
                              . ' this functionality is being blocked.'
                              ,$showstart, $showend);
            }
        } else {
            if ($by_ip) {
                $output = mt( $description
                              . ' unavailable from your current IP address: [_1] '
                              . 'because communication is being blocked for certain IP address(es).'
                              ,$clientip);
            } else {
                $output = mt( $description
                              . ' will be inaccessible between [_1] and [_2] because'
                              . ' communication is being blocked.'
                              ,$showstart, $showend);
            }
        }
    } else {
        # @$description is is the array returned from get_portfolio_category()
        # and contains the description (e.g. "Portfolio files belonging to [_1]"
        # and the value for [_1]
        if ($by_ip) {
            $output = mt( $$description[0]
                          . ' are inaccessible from your current IP address: [_2] '
                          . 'because communication is being blocked for certain IP address(es).'
                          , $$description[1], $clientip);
        } else {
            $output = mt( $$description[0]
                          . ' will be inaccessible between [_2] and [_3] because'
                          . ' communication is being blocked.'
                          ,$$description[1], $showstart, $showend);
        }
    }

    $output = "<p class=\"LC_info\">$output</p>";

    # show a table containing details, except if user is trying to look
    # at a different user's portfolio files
    if (   $activity ne 'port'                        # no portfolio
        || (   $uname eq $env{'user.name'}            # or own portfolio
            && $udom  eq $env{'user.domain'} ) 
        || Apache::lonnet::is_course($udom, $uname) ) # or portfolio of a course
    {
        if ($by_ip) {
            my $showdom = &Apache::lonnet::domain($blockdom);
            if ($showdom eq '') {
                $showdom = $blockdom;
            }
            $output .= '<br />'.
                       &mt('This restriction was set by an administrator in the [_1] LON-CAPA domain.'
                          ,$showdom);
        } else {
            $output .= &build_block_table(\%setters);
        }
    }

    return $output;
}

sub build_block_table {
    my ($setters) = @_;
    my %lt = &Apache::lonlocal::texthash(
        'cacb' => 'Currently active communication/content blocks',
        'cour' => 'Course',
        'dura' => 'Duration',
        'blse' => 'Block set by'
    );
    my $output;
    $output  = Apache::loncommon::start_data_table()
             . Apache::loncommon::data_table_caption($lt{'cacb'})
             . Apache::loncommon::start_data_table_header_row()
             . "<th>$lt{'cour'}</th> <th>$lt{'dura'}</th> <th>$lt{'blse'}</th>"
             . Apache::loncommon::end_data_table_header_row();

    foreach my $course (keys(%{$setters})) {
        my %courseinfo=&Apache::lonnet::coursedescription($course);
        for (my $i=0; $i<@{$$setters{$course}{staff}}; $i++) {
            my ($uname,$udom) = @{$$setters{$course}{staff}[$i]};
            my $fullname = Apache::loncommon::plainname($uname,$udom);
            if (defined($env{'user.name'}) && defined($env{'user.domain'})
                && $env{'user.name'} ne 'public'
                && $env{'user.domain'} ne 'public') 
            {
                $fullname = Apache::loncommon::aboutmewrapper($fullname,$uname,$udom);
            }
            my $triggertype = $$setters{$course}{triggers}[$i];
            if ($triggertype) {
                $fullname .= &mt(' (triggered by you when starting timer)');
            }
            my ($openblock,$closeblock) = @{$$setters{$course}{times}[$i]};
            $openblock = &Apache::lonlocal::locallocaltime($openblock);
            $closeblock= &Apache::lonlocal::locallocaltime($closeblock);
            my $duration = mt('[_1] to [_2]', $openblock, $closeblock);
            $output .= Apache::loncommon::start_data_table_row()
                     . "<td>$courseinfo{'description'}</td>"
                     . "<td>$duration</td>"
                     . "<td>$fullname</td>"
                     . Apache::loncommon::end_data_table_row();
        }
    }
    $output .= Apache::loncommon::end_data_table();
}

sub get_portfolio_category {
    my ($uname, $udom, $by_ip, $setters) = @_;

    if ($uname eq $env{'user.name'} && $udom eq $env{'user.domain'}) {
        # user's portolfio files

        return 'Your portfolio files';

    } elsif (Apache::lonnet::is_course($udom, $uname)) {
        # group portfolio files

        my $coursedesc;

        if ($by_ip) {
            my %courseinfo = Apache::lonnet::coursedescription($udom.'_'.$uname);
            $coursedesc    = $courseinfo{'description'};
        } else {
            foreach my $course (keys(%{$setters})) {
                my %courseinfo = Apache::lonnet::coursedescription($course);
                $coursedesc    = $courseinfo{'description'};
            }
        }

        return ["Group portfolio files in the course '[_1]'", $coursedesc];
        
    } else {
        # different user's portfolio files
        
        my $plainname = Apache::loncommon::plainname($uname, $udom);

        unless (   $env{'user.name'}   eq 'public' 
                && $env{'user.domain'} eq 'public' ) 
        {
            $plainname = Apache::loncommon::aboutmewrapper(
                            $plainname, $uname, $udom);
        }

        return ['Portfolio files belonging to [_1]', $plainname];
    }
}

1;
__END__
