# The LearningOnline Network with CAPA
# Handler to check if external resource can be shown in iframe
#
# $Id: lonexturlcheck.pm,v 1.4 2020/02/15 03:54:43 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
#
###############################################################
###############################################################

=pod

=head1 NAME

Apache::lonexturlcheck - External Resource URL checker

=head1 SYNOPSIS

Called in course context by course personnel either with the course editing
privilege or with view-only access to course editing tools.

Query string contains one item: name=exturl, value=URL of external resource
(format: http://hostname/path or https://hostname/path).
 
The resource URL is sent to &loncommon::is_nonframeable() to check whether
it can be displayed in an iframe in a page served by the current host.

=head1 OVERVIEW

Input: external resource URL (from query string passed to /adm/exturlcheck).

Hostname, lonHostID, and IP address for this node are retrieved from Apache.

Dependencies: calls &loncommon::is_nonframeable() to check if server where
external resource is hosted is configured with a Content-Security-Policy or
with X-Frame-options settings which prohibit display of the resource within
an iframe in a LON-CAPA page served from this node.

Output to print buffer: (content-type: text/plain):  1, 0, -1 or empty string.
'' -- display in iframe is allowed
1  -- display in iframe not allowed
0  -- invalid URL
-1 -- could not verify course editing privilege or view-only access to
      course editing tools

HTTP Return codes:
406 -- if user is not in a course
200 -- otherwise

=cut

package Apache::lonexturlcheck;

use strict;
use Apache::Constants qw(:common :http);
use Apache::lonnet;
use Apache::loncommon;
use HTTP::Request;

sub handler {
    my $r=shift;
    if ($r->header_only) {
        &Apache::loncommon::content_type($r,'text/html');
        $r->send_http_header;
        return OK;
    }
    if (!$env{'request.course.fn'}) {
        # Not in a course.
        $env{'user.error.msg'}="/adm/lonexturlcheck:bre:0:0:Not in a course";
        return HTTP_NOT_ACCEPTABLE;
    }
    &Apache::loncommon::content_type($r,'text/plain');
    $r->send_http_header;
    my $uselink;
    if (($env{'request.course.id'}) &&
        ((&Apache::lonnet::allowed('mdc',$env{'request.course.id'})) ||
         (&Apache::lonnet::allowed('cev',$env{'request.course.id'})))) {
        &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},['exturl']);
        $env{'form.exturl'} =~ s/^\s+|\s+$//g;
        if ($env{'form.exturl'} =~ m{^https?\://([a-z0-9]+(-[a-z0-9]+)*\.)+[a-z]{2,}}i) {
            my $hostname = $r->hostname();
            my $lonhost = $r->dir_config('lonHostID');
            my $ip = &Apache::lonnet::get_host_ip($lonhost);
            $r->print(&Apache::loncommon::is_nonframeable($env{'form.exturl'},'',$hostname,$ip,1));
        } else {
            $r->print(0);
        }
    } else {
        $r->print(-1);
    }
    return OK;
}

1;
