#!/usr/bin/perl
# CGI-script to run LaTeX, dvips, ps2ps, ps2pdf etc.
#
# $Id: printout.pl,v 1.172 2024/07/11 15:24:01 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#

use lib '/home/httpd/lib/perl';
use LONCAPA::loncgi;
use File::Path;
use File::Basename;
use File::Copy;
use IO::File;
use Image::Magick;
use Apache::lonhtmlcommon();
use Apache::lonnet;
use Apache::loncommon();
use Apache::lonlocal;
use Apache::lonmsg();
use LONCAPA::Enrollment;
use LONCAPA::Configuration;
use LONCAPA;
use Archive::Zip qw( :ERROR_CODES );

use strict;

my $busy_wait_timeout = 30; 
my $pdfs_converted    = 0;	# non zero if PDF includes (need to fixps).

my $debugging = 0;

sub debug {
    if ($debugging) {
	my ($text) = @_;
	print "$text <br />\n";
    }
}

#   Determine if a user is operating as a student for this course/domain.
#Parameters:
#    none
#Implicit:
#    $env{request.role} contains the role under which this user operated this
#                       this request.
sub is_student {
    return ($env{'request.role'}=~/^st\./);
}

#
#   Debugging:  Dump the environment for debugging.
#
sub dumpenv  {
    print "<br />-------------------<br />";
    foreach my $key (sort (keys %env)) {
	print "<br />$key -> $env{$key}";
    }
    print "<br />-------------------<br />";
}

#
#   This sub sends a message to the appropriate person if there was an error
#   rendering the latex  At present, there's only one case to consider:
#   a student printing inside a course results in messages to the course coordinator.
#Parmaeters:
#    identifier -  The unique identifier of this cgi request.
#    badresource-  Filepath to most likely failing 
#    logfile    -  The contents of the log file (included in the message).
#    texfile    -  reference to an array containing the LaTeX input file
#                  (included in the message).
#Implicit inputs:
#   From the environment:
#       cgi.$identifier.user     - User doing the printing.
#       cgi.$identifier.domain   - Domain the user is logged in on with printing.
#       cgi.$identifier.courseid - Id of the course (if this is a course).
#       cgi.$identifier.coursedom- Domain in which course was constituted.
#       cgi.$identifier.resources - List of resource URL's for which the print
#                                  was attempted.
# 
sub send_error_mail {
    my ($identifier, $badresource, $logfile, $texfile) = @_;
    my $user     = $env{"cgi.$identifier.user"};
    my $domain   = $env{"cgi.$identifier.domain"};
    my $courseid = $env{"cgi.$identifier.courseid"};
    my $coursedom= $env{"cgi.$identifier.coursedom"};
    my $resources= $env{"cgi.$identifier.resources"};

    #  resource file->URL
    #
    my $badurl = &Apache::lonnet::declutter($badresource);

    # &dumpenv();



    #  Only continue if there is a user, domain, courseid, course domain
    #  and resources:

    if(defined($user)       && defined($domain) && defined($courseid) &&
       defined($coursedom)  && defined($resources) ){
	   
	#  Only mail if the conditions are ripe for it:
	#  The user is a student in the course:
	#
	
	if (&is_student()) {
	    # build the subject and message body:
	    &debug("sending message to course coordinators.");

	    # Todo: Convert badurl into a url from file path:

	    my $subject  = "Error [$badurl] Print failed for $user".':'.$domain;
	    my $message .= "Print failed to render LaTeX for $user".':'."$domain\n";
	    $message    .= "  User was attempting to print: \n";
	    foreach my $resource (split(/:/,$resources)) {
		$message    .= "       $resource\n";
	    }
	    $message    .= "--------------------LaTeX logfile:------------ \n";
	    $message    .= $logfile;
	    $message    .= "-----------------LaTeX source file: ------------\n";
	    
	    foreach my $line (@$texfile) {
		$message .= "$line\n";
	    }
	    my (undef, %receivers) = &Apache::lonmsg::decide_receiver(undef, 0,
								      1,1,1);
	    &debug("sending...section:  $env{'request.course.sec'}");
	    foreach my $dest (keys %receivers) {
		&debug("dest is $dest");
		my @destinfo = split(/:/,$dest);
		my $user = $destinfo[0];
		my $dom  = $destinfo[1];

		&Apache::lonmsg::user_normal_msg($user, $dom,
						 $subject, $message);
		
		# No point in looking at the return status as there's no good
		# error action I can think of right now (log maybe?).
	    }
	}
    }
}

$|=1;
if (! &LONCAPA::loncgi::check_cookie_and_load_env()) {
    print <<END;
Content-type: text/html

<html>
<head><title>Bad Cookie</title></head>
<body>
Your cookie information is incorrect.
</body>
</html>
END
    exit;
}

my %perlvar=%{&LONCAPA::Configuration::read_conf('loncapa.conf')};
&Apache::lonlocal::get_language_handle();
&Apache::loncommon::content_type(undef,'text/html');
$env{'request.noversionuri'} = '/cgi-bin/printout.pl';
# Breadcrumbs
#FIXME: Choose better/different breadcrumbs?!? Links?
my $brcrum = [{'href' => '',
               'text' => 'Helper'}, #FIXME: Different origin possible than print out helper?
              {'href' => '',
               'text' => 'Preparing Printout'},
              {'href' => '',
               'text' => 'Creating PDF'}];
print(&Apache::loncommon::start_page('Creating PDF',
                                     undef,
                                     {'bread_crumbs' => $brcrum,}));

my $identifier = $ENV{'QUERY_STRING'};
my $texfile = $env{'cgi.'.$identifier.'.file'};
my $laystyle = $env{'cgi.'.$identifier.'.layout'};
my $numberofcolumns = $env{'cgi.'.$identifier.'.numcol'};
my $paper = $env{'cgi.'.$identifier.'.paper'};
my $selectionmade = $env{'cgi.'.$identifier.'.selection'};
my $tableofcontents = $env{'cgi.'.$identifier.'.tableofcontents'};
my $tableofindex = $env{'cgi.'.$identifier.'.tableofindex'};
my $advanced_role = $env{'cgi.'.$identifier.'.role'};
my $number_of_files = $env{'cgi.'.$identifier.'.numberoffiles'}+1;
my $student_names = $env{'cgi.'.$identifier.'.studentnames'};
my $backref = &unescape($env{'cgi.'.$identifier.'.backref'});


my @names_pack=();
if ($student_names=~/_END_/) {  
    @names_pack=split(/_ENDPERSON_/,$student_names);
}
if ($backref) {
    if ($backref =~ m{^(/uploaded/$LONCAPA::match_domain/$LONCAPA::match_courseid/default_\d+.page)}) {
        $backref = $1;
    }
    print('<p>'.&mt("[_1]Return[_2] to resource.",
		    "<a href=\"$backref\"><b>","</b></a>").'</p>');
    print('<p><a href="javascript:gopost(\'/adm/printout\',\''.$backref.'\');">'.
        &mt("Change Printing Options").'</a></p>'."\n");
}
my $figfile = $texfile;
$figfile =~ s/^(.*_printout)_\d+_\d+_\d+\.tex/$1\.dat/;
my $duefile = $texfile;
$duefile =~ s/^(.*_printout)_\d+_\d+_\d+\.tex/$1\.due/;


#-------------------------------------------------------------------------------------
#
#   Each print may have associated with it a file that contains a set of figures
#   that need to be converted to .eps from whatever form they were in when included
#   in the resource.  The name of the figure file is in $figfile.  If it exists,
#   it contains the names of the files that need to be converted, one per line.
#

&debug("Figure file is $figfile");

if (-e $figfile) {
    &debug( "Figure file exists");
    &debug("$figfile exists");
    my %done_conversion;
    my $temporary_file=IO::File->new($figfile) || die "Couldn't open fig file $figfile for reading: $!\n";
    my @content_of_file = <$temporary_file>;
    close $temporary_file;  
    my $noteps;
    my %prog_state;
    if ($advanced_role) { %prog_state=&Apache::lonhtmlcommon::Create_PrgWin('',$#content_of_file);  }
    print('<br />');
    foreach my $not_eps (@content_of_file) {
	chomp($not_eps);
	&debug( "Being asked to convert $not_eps");
	if ($not_eps ne '') {
	    $not_eps=~s|\/\.\/|\/|g;
	    if (!$done_conversion{$not_eps}) { #  Only convert multiple includes once.
		&convert_figure($not_eps);
		$done_conversion{$not_eps} = 1;
	    }
	}
    }
    if ($advanced_role) { 
	&Apache::lonhtmlcommon::Close_PrgWin('',\%prog_state); 
    }
    unlink($figfile);
}
#     End of figure conversion section:
#
#--------------------------------------------------------------------------------------------
#
#  Figure out which Tex files we need to process.  If this is a large class e.g.
#  the instructor may have asked that the printout be by section, one section per file
#  in that case, the output tex fiels will be the base filename with 3 digit serial numbers
#  just prior to the .tex file type.
#  By the time this loop exits, @texfile is an array of the files to process.
#

my @texfile=($texfile);
if ($number_of_files>1) {
    @texfile=();
    for (my $i=1;$i<=$number_of_files;$i++) {
	my $new_texfile=$texfile;
	$new_texfile=~s/\.tex//;
	$new_texfile = sprintf("%s_%03d.tex", $new_texfile,$i);
	push @texfile,$new_texfile;
    } 
}

#--------------------------------------------------------------------------------------------

my $ind=-1;

my %prog_state;
if ($advanced_role) { 
    %prog_state=&Apache::lonhtmlcommon::Create_PrgWin('',$number_of_files); 
}
print "<br />";
my $num_files = @texfile;	# How does this differ from $number_of_files , can that be 0?




foreach $texfile (@texfile) {
  my $status_statement='';
  my $link_text='download PDF';
  $ind++;

  #---------------------------------------------------------------------------------------
  #  This chunk of code 
  #  determines the status message for the printout, and the download link text.
  #  If the print is for one or more students, both will contain the range of students
  #  covered by this file.
  #  

  my @stud_info=split(/_END_/,$names_pack[$ind]);
  my @tempo_array=split(/:/,$stud_info[0]);       # username:domain:section:full name:...
  my $name;
  my $name_range='';

  # $name       -> Either user's full name or username:domain
  # $name_range -> Either user's last name or username.

  if ($tempo_array[3]) {
      $name=$tempo_array[3];
      $name =~ s{^\s+|\s+$}{}g;
      if ($name =~ /,/) { 
          ($name_range) = split(/,/,$name, 2);
      } elsif ($name =~ /\s/) {
          $name_range = $name;
          $name_range =~ s/\s+/_/;
      } else {
          $name_range = $name;  
      }
      $name_range =~ s/[^\w\:\-]+//g;
  } else {
      $name=$tempo_array[0].':'.$tempo_array[1];
      $name_range = $tempo_array[0];
  }

  # If there truly is a name add it to the status text so we know which
  # user is getting printed.
  #

  if (($name ne "") && ($name ne ':') ) { # Could be printing codes...
      $link_text='<b>'.$name.'</b>';
      $status_statement.=$name;
  }

  # Group of students being printed...
  # $name_range -> first student's name - last student's name
  #

  if ($#stud_info>0) {
      @tempo_array=split(/:/,$stud_info[-1]);
      if ($tempo_array[3]) {
	  $name=$tempo_array[3];
          $name =~ s{^\s+|\s+$}{}g;
          my $lastname;
          if ($name =~ /,/) {
	      ($lastname) = split(/,/, $name,2);
          } elsif ($name =~ /\s/) {
              $lastname = $name;
              $lastname =~ s/\s+/_/;
          } else {
              $lastname = $name;
          }
	  $name_range .= "-".$lastname;
          $name_range =~ s/[^\w\:\-]+//g;
      } else {
	  $name=$tempo_array[0].':'.$tempo_array[1];
	  $name_range .= '-'.$tempo_array[0];
      }
      if (($name ne "") && ($name ne ':')) {
	  $link_text.=' - <b>'.$name.'</b>';
	  $status_statement.=' -  '.$name;
  
      }
  }

  # $name_range is the range of names in this file.
  # $name is the first of the names in this file.
  #
  #-----------------------------------------------------------------------

  #  If the number of files is > 1, but we don't have multiple
  #  student names, we must be printing an exam from codes.
  #  The download link becomes the filename (basename.tex
  #  The status info has the basename appended to it.
  #
  if(($num_files > 1) && ($link_text eq 'download PDF')) {
      $link_text = '<b>'.basename($texfile,'.tex').'.pdf</b>';
      $status_statement .= basename($texfile);
  }


  #------------------------------------------------------------------------

  $name_range =~ s/'//g;	# O'Neil -> ONeil e.g.


  print "<br/>";
  if ($advanced_role) { 
      &Apache::lonhtmlcommon::Update_PrgWin('',\%prog_state,&mt('Creating PDF for: [_1]',$status_statement)); 
  }

  if (-e $texfile) {		# Ensure the tex file exists:

      #---------------------------------------------------------------------
      #
      #  Put username ranges into the original tex
      #  Tex filename from which they'll propagate into the other filenames as well.
      #

      if (($name_range ne '') && ($num_files > 1)) {
	  my $newtexfile = $texfile;
	  $newtexfile    =~ s/\.tex/$name_range\.tex/;
	  rename($texfile, $newtexfile);
	  $texfile       = $newtexfile;
      }

      #---------------------------------------------------------------------

      $texfile =~ m/^(.*)\/([^\/]+)$/; 
      my $name_file = $2;
      my $path_file = $1.'/';
      chdir $path_file;
      my $dvi_file= $name_file; $dvi_file =~ s/\.tex$/\.dvi/;
      &make_dvi_file($name_file,
		     $dvi_file,
		     $tableofcontents,
		     $tableofindex,
		     $status_statement,
		     \%prog_state,
		     $busy_wait_timeout);


      #Do we have a latex error in the log file?


      my $logfilename = $texfile; $logfilename =~ s/\.tex$/\.log/;


      if (&analyze_logfile($logfilename, $texfile, $advanced_role)) {
	  

	  #LaTeX successfully parsed tex file 
	  $name_file =~ s/\.tex/\.dvi/;
	  my $new_name_file = $name_file;
	  $new_name_file =~ s/\.dvi/\.ps/;
# Explicitly include a switch for papertype, otherwise dvips will default
# to whatever is listed first in config.ps (which in most cases is a4).
# Historically (since 2004) LON-CAPA printing expected to use the default,
# i.e., a papertype of a4, when the user selected letter [8 1/2 x 11 in] 
# in the Layout options, so I follow that convention if $papera is letter.  
	  my $papera=$paper;
	  if ($papera eq 'letter') {$papera='a4';}
	  if ($papera ne '') {$papera='-t'.$papera;}
	  my $extra_ps_header = $perlvar{'lonLib'} .'/includepsheader.ps';
	  my $comma = "dvips $papera -h $extra_ps_header -Ppdf -G0 -o  $new_name_file";
	  &busy_wait_command("$comma $name_file 1>/dev/null 2>/dev/null",
			     "for $status_statement now Converting to PS",
			     \%prog_state,$new_name_file);
	  #
	  #  One last little hinky kinky thing.
	  #  It's just possible that some fonts could not be maded
	  #  at the resolution of the pdf print driver.
	  #  In that case a file called missfont.log will have been
	  #  created that will contain the commands that were attempted
	  # to create the missing fonts.  If we basically
	  # take all the 8000 strings in that file, and
	  # replace them with 600 (the ljfour resolution)
	  # run the commands in that file and redvips,
	  # we'll be able to print the missing glyphs at 600dpi.
	  #
	  # Supposedly it is possible to tune TeX/Metafont to do this
	  # right but I failed to get that to work when following the
	  # docs at the tug site, hence this rather kludgey fix.
	  #

	  my $print_directory = dirname($name_file);
	  my $missfonts_file  = $print_directory."/missfont.log";
	  #print("<br /> Missing fonts file is: $missfonts_file");
	  if (-e $missfonts_file) {
	      #print("<br />Missing fonts file exists\n");
	      &create_missing_fonts($missfonts_file,\%prog_state);
	      &busy_wait_command("$comma $name_file 1>/dev/null 2>/dev/null",
				 "for $status_statement dvips generated missing fonts",
				 \%prog_state, $new_name_file);
	  }
	  if (-e $new_name_file) {
	      my $latex_file = $name_file;
	      $latex_file =~ s/\.dvi/\.tex/;
	      &repaginate($new_name_file, $latex_file,  $numberofcolumns);

	      &make_dvi_file($latex_file,
			     $name_file,
			     $tableofcontents,
			     $tableofindex,
			     $status_statement,
			     \%prog_state,
			     $busy_wait_timeout);


	      &busy_wait_command("$comma $name_file 1>/dev/null 2>/dev/null",
				 "for $status_statement dvips to repaginate",
				 \%prog_state, $new_name_file);

	      print "<br />";
	      $new_name_file =~ m/^(.*)\./;
	      my $ps_file = my $tempo_file = $1.'temporar.ps';
	      my $pdf_file = $1.'.pdf';
	      $papera=~s/t/p/;
#----
# The code below uses fixps to make pdf include in sequences work.
#
#              $comma = "fixps --force $new_name_file";
#              &debug("FIXPS command: $comma");
#              &busy_wait_command("$comma 1>$tempo_file  2>/dev/null",
#                                 "for $status_statement now validating PS",
#                                 \%prog_state,$tempo_file);

#--- 
#  The code below uses gs to make pdf includes in sequences work

	      # Use gs to fix the postscript -> level 1.5 
	      # .. if pdfs were included
	      #
	      # pswrite device was removed from ghostscript 9.09 and later,
	      # (ps2write device is used instead).
	      # check which device is available, and use as the value
              # passed via -sDEVICE= arg in gs call to fix the postscript.
	      #

	      if ($pdfs_converted > 0) {
		  my @possdevices = qw(ps2write pswrite);
		  my $device;
		  foreach my $poss (@possdevices) {
		      if (open(PIPE,"gs -h |grep ' $poss ' 2>&1 |")) {
		          my $output = <PIPE>;
		          close(PIPE);
		          chomp($output);
		          if ($output =~ /\Q $poss \E/) {
		              $device = $poss;
		          }
		      }
		      last if ($device ne '');
		  }
		  if ($device ne '') {
		      my ($major,$minor);
		      if (open(PIPE,"gs -v |grep 'GPL Ghostscript' 2>&1 |")) {
		          my $info = <PIPE>;
		          close(PIPE);
		          chomp($info);
		          if ($info =~ /Ghostscript\s+(\d+)\.(\d+)/) {
		              ($major,$minor) = ($1,$2);
		          }
		      }
		      $comma = "gs -sDEVICE=$device -dLanguageLevel=1.5 ";
		      if (($major > 9) || (($major == 9) && ($minor >= 50))) {
		          $comma .= '--permit-file-read=* ';
		      }
		      &busy_wait_command("$comma -o $tempo_file $new_name_file 2>/dev/null 1>/dev/null",
				         "for $status_statement now validating PS",
				         \%prog_state, $tempo_file);

#---
		      if (-e $tempo_file) {
		          &busy_wait_command("mv $tempo_file $new_name_file",
				             'File move', \%prog_state, $new_name_file);
		      }
		  }
	      }
	      if ($laystyle eq 'album' and $numberofcolumns eq '2') {
		  $comma = "psnup $papera -2 -s1.0 $new_name_file";
		  &debug("PSNUP command: $comma");
		  &busy_wait_command("$comma $tempo_file 1>/dev/null 2>/dev/null",
				     "for $status_statement now Modifying PS layout",
				     \%prog_state,$tempo_file);
	      } elsif ($laystyle eq 'book' and $numberofcolumns eq '2') {
		  $comma = 'pstops '.$papera.' "2:0+1(0.48w,0)" '.$new_name_file;
		  &debug("PSTOPS command: $comma ");
		  &busy_wait_command("$comma $tempo_file 1>/dev/null 2>/dev/null",
				     "for $status_statement now Modifying PS layout",
				     \%prog_state,$tempo_file); 
	      } else {
		  $ps_file=$new_name_file;
	      }
	      my $addtoPSfile={'legal'=>'<< /PageSize [612 1008] >> setpagedevice',
                               'tabloid'=>'<< /PageSize [792 1224] >> setpagedevice',
                               'executive'=>,'<< /PageSize [540 720] >> setpagedevice',
                               'a2'=>'<< /PageSize [1195.02 1690.09] >> setpagedevice',
                               'a3'=>'<< /PageSize [842 1195.02] >> setpagedevice',
                               'a4'=>'<< /PageSize [595.2 842] >> setpagedevice',
                               'a5'=>'<< /PageSize [421.1 595.2] >> setpagedevice',
                               'a6'=>'<< /PageSize [298.75 421.1] >> setpagedevice',
			   };
	      if ($paper ne 'letter') {
		  open(FFH,'<',$ps_file) || die "Couldn't open ps file $ps_file for reading: $!\n";
		  my $new_ps_file='new'.$ps_file;
		  open(FFHS,'>',$new_ps_file) || die "Couldn't open new ps file $new_ps_file for reading: $!\n";
		  print FFHS $addtoPSfile->{$paper}."\n";
		  while (<FFH>) {
		      print FFHS $_;
		  }
		  close(FFH);
		  close(FFHS);
		  $ps_file=$new_ps_file;	  
	      }
	      &busy_wait_command("ps2pdf13 $ps_file $pdf_file 1>/dev/null 2>/dev/null",
				 "for $status_statement now Converting PS to PDF",
				 \%prog_state,$pdf_file);
	    
	      my $texlog = $texfile;
	      my $texaux = $texfile;
	      my $texdvi = $texfile;
	      my $texps = $texfile;
	      $texlog =~ s/\.tex/\.log/;
	      $texaux =~ s/\.tex/\.aux/;
	      $texdvi =~ s/\.tex/\.dvi/;
	      $texps =~ s/\.tex/\.ps/;
	      my @garb = ($texlog,$texaux,$texdvi,$texps);
#	  unlink @garb;
	      unlink($duefile);
	      print
                  '<p>'
                 .&mt('[_1] - [_2]Your PDF file[_3] is ready for download.',
                      $link_text,'<a href="/prtspool/'.$pdf_file.'">','</a>')
                 .'</p>'."\n";
	  }
	  unlink($missfonts_file);

      }  
  } else {
      print
          '<p class="LC_error">'
         .&mt('The LaTeX file [_1] was not created successfully.',
              '<span class="LC_filename">'.$texfile.'</span>')
         .'</p>';
  }
}
if ($advanced_role) { &Apache::lonhtmlcommon::Close_PrgWin('',\%prog_state); }
print "<br />";
if ($number_of_files>1) {
    print('<p>'.&mt('Zip Output:')."\n");
    my %zip_prog_state;
    if ($advanced_role) { %zip_prog_state=&Apache::lonhtmlcommon::Create_PrgWin('',$number_of_files); }
    my $zipfile=$texfile[0];
    $zipfile=~s/\.tex/\.zip/;
    my $zip = Archive::Zip->new();
    my $counter = 0;
    foreach my $file (@texfile) {
        $file=~s/\.tex/.\pdf/;
        my $dest=$file;
        $dest=~s{^\Q$perlvar{'lonPrtDir'}\E}{prtspool};
        $zip->addFile($file,$dest);
        $dest=~s/^prtspool//;
        $counter ++;
        if ($advanced_role) {
            &Apache::lonhtmlcommon::Update_PrgWin('',\%zip_prog_state,
                                                  &mt('[_1] added to zip archive ([_2] of [_3]',
                                                  $dest,$counter,$number_of_files));
        }
    }
    if ($advanced_role) {
        &Apache::lonhtmlcommon::Update_PrgWin('',\%zip_prog_state,&mt('Writing zip file'));
    }
    if ($zip->writeToFileNamed($zipfile) == AZ_OK) {
        $zipfile=~s{^\Q$perlvar{'lonPrtDir'}\E}{/prtspool};
        print
            '<p>'
           .&mt('A [_1]ZIP file[_2] of all the PDF files is ready for download.',
                '<a href="'.$zipfile.'">','</a>')
           .'</p>';
    } else {
        print '<p class="LC_error">'.
              &mt('An error occurred creating a ZIP file of all the PDF files').
             '</p>';
    }
    if ($advanced_role) { &Apache::lonhtmlcommon::Close_PrgWin('',\%zip_prog_state); }
}
print(&Apache::loncommon::end_page());
my $done;

sub REAPER {
    $done=1;
}
#
#  Execute a command updating the status window as the command's
#  output file builds up (at intervals of a second).
#
#   If the timeout argument defined, then if that many seconds
#   elapses without an increase in the size of the output file,
#   the command will be killed (this deals with the case when
#   latex crawls into an infinite loop).
#
sub busy_wait_command {
    my ($command,$message,$progress_win,$output_file, $timeout)=@_;
    
    $SIG{CHLD} = \&REAPER;
    $done=0;
    my $pid=open(CMD,"$command |");
    if ($advanced_role) {
	&Apache::lonhtmlcommon::Update_PrgWin('',$progress_win,$message);
    }
    my $last_size      = 0;
    my $unchanged_time = 0;
    while(!$done) {
	sleep 1;
	my $extra_msg;
	if ($output_file) {
	    my $size=(stat($output_file))[7];
	    $extra_msg=", $size bytes generated";
	    if ($size == $last_size) {
		$unchanged_time++;
		if ($timeout && ($unchanged_time > $timeout)) {
		    print '<p class="LC_error">'.&mt('Operation timed out!')."</p>\n";
		    print "<p>Executing $command, the output file $output_file did not grow\n";
		    print "after $timeout seconds.  This <em>may</em> indicate $command\n";
		    print "is in an infinite loop.\n";
		    print "See if printing fewer copies helps.  Please contact LON-CAPA\n";
		    print "support about this in any event.";
		    print "</p>";
		    kill(9, $pid); # Reaper will do the rest...I hope there's errors in the log.
		}
	    } else {
		$last_size      = $size;
		$unchanged_time = 0;
	    }
	}
	if ($advanced_role) {
	    &Apache::lonhtmlcommon::Update_PrgWin('',$progress_win,$message.$extra_msg);
	}
    }
    $SIG{CHLD}='IGNORE';
    close(CMD);
}

# Make the dvi file (or rather try to), from the latex file and the
# various bits and pieces that control how the latex file is processed:
# LaTeX is run as many times a needed to make this all happen... this may
# result in several runs of LaTeX that just are errors if the LaTeX is
# bad, but the printing subsystem is _supposed_ to not do that.
#
# Parameters:
#   name_file        - Name of the LaTeX file to process.
#   dvi_file         - Name of resulting dvi file.
#   tableofcontents  - "yes" if we are supposed to make a table of contents.
#   tableofindex     - "yes" if we are suposed to make an index.
#   status_statement - Part of the status statement for ths status window.
#   prog_state       - Reference to the program state hash.
#   busy_wait_timeout- Seconds without any progress that imply a problem.
#
#
sub make_dvi_file {
    my ($name_file,
	$dvi_file,
	$tableofcontents,
	$tableofindex,
	$status_statement,
	$prog_state,
	$busy_wait_timeout) = @_;
    
    
    &busy_wait_command("latex $name_file 1>/dev/null 2>/dev/null",
		       "for $status_statement now LaTeXing file",
		       $prog_state,$dvi_file, $busy_wait_timeout);

    # If the tableof contents was requested, we need to run 
    # LaTex a couple more times to get all the references sorted out.

    if ($tableofcontents eq 'yes') {
	&busy_wait_command("latex $name_file 1>/dev/null 2>/dev/null",
			   "for $status_statement First LaTeX of file for table of contents",
			   $prog_state,$dvi_file, $busy_wait_timeout);
	&busy_wait_command("latex $name_file 1>/dev/null 2>/dev/null",
			   "for $status_statement Second LaTeX of file for table of contents",
			   $prog_state,$dvi_file,$busy_wait_timeout);
    } 

    # And makeindex and another run of LaTeX to incorporate it if the index
    # is enabled.


    if ($tableofindex eq 'yes') {
	my $idxname=$name_file;
	$idxname=~s/\.tex$/\.idx/;
	&busy_wait_command("makeindex $idxname",
			   "making index file",
			   $prog_state,$idxname);
	&busy_wait_command("latex $name_file 1>/dev/null 2>/dev/null",
			   "for $status_statement now LaTeXing file for index section",
			   $prog_state,$dvi_file, $busy_wait_timeout);
    } 
    
}    


#  Repagninate
#  What we need to do:
#   - Count the number of pages in each student.
#   - Rewrite the latex file replacing the \specials that
#     mark the end of student with an appropriate number of newlines.
#   parameters:
#     psfile     - Postscript filename
#     latexfile  - LaTeX filename
#     columns    - number of columns.
sub repaginate {

    # We will try to do this in 2 passes through the postscript since
    # the postscript is potentially large, to do 2 passes, the first pass
    # must be able to calculate the total number of document pages so that
    # at the beginning of the second pass we already know how to replace
    #  %%Pages:

    #  Figure out
    #    1. Number of pages in the document
    #    2. Maximum number of pages in a student
    #    3. Number of pages in each student.

    my ($postscript_filename, $latex_filename, $num_columns) = @_;
    open(PSFILE, "<$postscript_filename");
    my $line;
    my $total_pages;		# Total pages in document.
    my $seen_pages        = 0;	# There are several %%Pages only the first is useful
    my @pages_in_student;	# For each student his/her initial page count.
    my $max_pages = 0;		# Pages in 'longest' student.
    my $page_number = 0;
    &Apache::lonhtmlcommon::Update_PrgWin('',\%prog_state,&mt("Counting pages for student: [_1]",1));

    while ($line = <PSFILE>) {
	
	# Check for total pages (%%Pages:)

	if (($line =~ /^%%Pages:/) && (!$seen_pages)) {
	    my @pageinfo = split(/ /,$line);
	    $total_pages = $pageinfo[1];
	    $seen_pages  = 1;
	}
	#  Check for %%Page: n m  $page_number will be the
	#  biggest of these until we see an endofstudent.
	#  Note that minipages generate spurious %Page: 1 1's so
	#  we only are looking for the largest n (n is page number at the
	#  bottom of the page, m the page number within the document.
	#

	if ($line =~ /^%%Page:\s+\d+\s+\d+/) {
	    my @pageinfo = split(/\s+/, $line);
	    if ($page_number < $pageinfo[1]) {
		$page_number = $pageinfo[1];
	    } elsif ($pageinfo[2] ne 1) {
		#  current page count reset, and it's not because of a 
		#    minipage 
		# - save the page_number, reset and, if necessary
		#    update max_pages.
		push(@pages_in_student, $page_number);
		&Apache::lonhtmlcommon::Update_PrgWin('',\%prog_state,&mt("Counting pages for student: [_1]", scalar(@pages_in_student)));
		if ($page_number > $max_pages) {
		    $max_pages = $page_number;
		}
		$page_number = $pageinfo[1];
	    }
	}

	
    }
    # file ended so one more student
    push(@pages_in_student, $page_number);
    &Apache::lonhtmlcommon::Update_PrgWin('',\%prog_state,&mt("Counting pages for student: [_1]",scalar(@pages_in_student)));
    if ($page_number > $max_pages) {
	$max_pages = $page_number;
    }
    $page_number = 0;
    
    close(PSFILE);

    #  If 2 columns, max_pages must go to an even number of columns:

   
    if ($num_columns == 2) {
	if ($max_pages % 2) {
	    $max_pages++;
	}
    }
    
    #  Now rewrite the LaTex file, substituting our \special
    #  with an appropriate number of \newpage directives.

    my $outfilename = $latex_filename."temp";

    open(LATEXIN, "<$latex_filename");
    open(LATEXOUT, ">$outfilename");


    my $student_number    = 0;	# Index of student we're working on.
    &Apache::lonhtmlcommon::Update_PrgWin('',\%prog_state,&mt("Repaginating student: [_1]",$student_number+1));

    while (my $line = <LATEXIN>) {
	if ($line eq "\\special{ps:ENDOFSTUDENTSTAMP}\n") {
	    # only end of student stamp if next line is ENDOFSTUDENTSTAMP:


	    # End of student replace with 0 or more newpages.
	    
	    my $addlines = $max_pages - $pages_in_student[$student_number];
	    while($addlines)  {
		print LATEXOUT '\clearpage \strut \clearpage';

		$addlines--;
	    }
	    
	    $student_number++;
	    &Apache::lonhtmlcommon::Update_PrgWin('',\%prog_state,&mt("Repaginating student: [_1]",$student_number+1));
	    
	} else {
	    print LATEXOUT $line;
	}
    }

    close(LATEXIN);
    close(LATEXOUT);
    rename($outfilename, $latex_filename);

}

#
#   Create missing fonts given a latex missfonts.log file.
#   This file will have lines like:
#
#   mktexpk --mfmode ljfour --bdpi 8000 --mag 1+0/8000 --dpi 8000 tcrm0500
#
#  We want to execute those lines with the 8000's changed to 600's
#  in order to match the resolution of the ljfour printer.
#  Of course if some wiseguy has changed the default printer from ljfour
#  in the dvips's config.ps file that will break so we'll also
#  ensure that --mfmode is ljfour.
#
sub create_missing_fonts {
    my ($fontfile, $state) = @_;

    # Open and read in the font file..we'll read it into the array
    #  font_commands.
    #
    open(my $font_handle, $fontfile);
    my @font_commands = <$font_handle>;

    # make the list contain each command only once
    my %uniq;
    @font_commands = map { $uniq{$_}++ == 0 ? $_ : () } @font_commands;

    #  Now process each command replacing the appropriate 8000's with
    #  600's ensuring that font names with 8000's in them are not corrupted.
    #  and if the --mfmode is not ljfour we turn it into ljfour.
    #   Then we execute the command.
    #
    
    foreach my $command (@font_commands) {
	#print("<br />Raw command: $command");
	$command =~ s/ 8000/ 600/g;    # dpi directives.
	$command =~ s/\/8000/\/600/g;  # mag directives.
	#print("<br />After dpi replacements: $command");

	my @cmdarray = split(/ /,$command);
	for (my $i =0; $i < scalar(@cmdarray); $i++) {
	    if ($cmdarray[$i] eq '--mfmode') {
		$cmdarray[$i+1] = "ljfour";
	    }
	}
	#print("<br /> before reassembly : (@cmdarray)");
	$command = join(" ", (@cmdarray));

	#print("<br />Creating fonts via command: $command");
	&busy_wait_command("$command 1>/dev/null 2>/dev/null",
			   "Creating missing font",
			   $state);
			   
    }

}
#
#  Convert a figure file to encapsulated postscript:
#  At present, this is using a lot of file scoped globals to pass data around. 
# Parameters:
#    not_eps  - The name of the file to convert which, presumably, is not
#               already an eps file.
#
sub convert_figure {
    my ($not_eps) = @_;
    &debug("in convert_figure");

    my $status_statement='EPS picture for '.$not_eps;
    my $eps_f = $not_eps;

    if ($eps_f=~/\/home\/httpd\/html\/priv\/[^\/]+\/([^\/]+)\//) {
	$eps_f=~s/\/home\/httpd\/html\/priv\/[^\/]+\/([^\/]+)/$1/;
    } elsif ($eps_f=~/$perlvar{'lonDocRoot'}\/res\//) {
	$eps_f=~ s/$perlvar{'lonDocRoot'}\/res\/(.+)/$1/;
    } elsif ($eps_f=~/$perlvar{'lonUsersDir'}\//) {
	$eps_f=~ s/$perlvar{'lonUsersDir'}\/([^\/]+)\/\w\/\w\/\w\/(.+)/$1\/$2/;
    }

    $eps_f = $perlvar{'lonPrtDir'}.'/'.$eps_f;

    # Spaces are problematic for system commands and LaTeX, replace with _

    $eps_f  =~ s/ /\_/g;

    # 
    # If the file is already an .eps or .ps file (eps_f still has the original
    # file type),
    # We really just need to copy it from where it was to prtspool
    # but with the spaces substituted to _'s.
    #
    my ($nsname,$path, $sext) = &fileparse($eps_f, qr/\.(ps|eps)/i);
    if ($sext =~/ps$/i) {
	&File::Path::mkpath($path,0,0777);
	copy("$not_eps", "$eps_f"); 
    } else {
	
	$eps_f .= '.eps';	# Just append the eps ext.
	my $path= &dirname($eps_f);
	&File::Path::mkpath($path,0,0777);
	$not_eps =~ s/^\s+//;
	$not_eps =~ s/\s+$//;
    my $prettyname=$not_eps;
	if ($advanced_role) {
	    $prettyname=~s|$perlvar{'lonDocRoot'}/|/|;
	    &Apache::lonhtmlcommon::Update_PrgWin('',\%prog_state,&mt('Converting to EPS: [_1]',$prettyname));
	}
	#
	#  If the file is a PDF, need to use pdftops to convert it to a ps file.
	#  otherwise use imagemagik:
	#
	if($not_eps =~/\.(pdf|PDF)$/) {

	    #
	    # For whatever reason, pure postscript conversions have to be
	    # in the same dir as the base file:
	    #
	    $eps_f = &basename($eps_f);
	    $eps_f = $perlvar{'lonPrtDir'}.'/'.$eps_f;

	    &debug("Converting pdf $not_eps to postscript: $eps_f");
            my @args = ('pdftops',$not_eps,$eps_f);
            system({$args[0]} @args); # Indirect object forces list processing mode.
                                      # See perlfunc documentation for exec().
            if ($? and $advanced_role) {
                print '<p class="LC_warning">'
                     .&mt('An error occurred during the conversion of [_1] to postscript.',
                          '<span class="LC_filename">'.$prettyname.'</span>')
                     .'</p>';
            } else {
                $pdfs_converted++; # Need to fix ps in last pass.
            }
	} else {
            my @args = ('convert',$not_eps,$eps_f);
            system({$args[0]} @args); # Indirect object forces list processing mode.
                                      # See perlfunc documentation for exec().
            if ($? and $advanced_role) {
                print '<p class="LC_warning">'
                     .&mt('An error occurred during the conversion of [_1].',
                          '<span class="LC_filename">'.$prettyname.'</span>')
                     .'<br />'
                     .&mt('If possible try to save this image using different settings and republish it.')
                     .'</p>';
            }
	}

	if (not -e $eps_f) {
	    # converting an animated gif creates either:
	    # anim.gif.eps.0
	    # or
	    # anim.gif-0.eps
	    for (my $i=0;$i<10000;$i++) {
		if (-e $eps_f.'.'.$i) {
		    rename($eps_f.'.'.$i, $eps_f);
		    last;
		}
		my $anim_eps = $eps_f;
		$anim_eps =~ s/(\.[^.]*)\.eps$/$1-$i\.eps/i;
		if (-e $anim_eps) {
		    rename($anim_eps, $eps_f);
		    last;
		}
	    }
	}
	
	# imagemagick 6.2.0-6.2.7 fails to properly handle
	# convert anim.gif anim.gif.eps
	# it creates anim.eps instead. 
	if (not -e $eps_f) {
	    my $eps_f2 = $eps_f;
	    $eps_f2 =~ s/\.[^.]*\.eps$/\.eps/i;
	    if(-e $eps_f2) {
		rename($eps_f2,$eps_f);
	    }
	}
    }
    
}
#
#   Analyze a LaTeX logfile producing appropriate  output on error and 
#   returning a boolean to let the caller know if, in our opinion, it's
#   worth continuing on to produce the PDF file.
#
# Parameters:
#   logfilename   - Name of the logfile.
#   texfile       - Name of the LaTeX file that was being processed.
#   advanced_role - True if the user is privileged with respect to the printout
#                   (e.g. is the course coordinator or some such thing).
# Returns:
#    1            - Caller is advised to continue to create the PDF.
#    0            - Caller need not bother creating the PDF.
# Side Effects:
#   Messages are printed to describe any errors that have been encountered.
# NOTE:
#    The current policy is to assume that if LaTeX decided to insert some text
#    it has salvaged the resource and the resource can be printed.. in that case
#    a message is emitted from this sub.
#
sub analyze_logfile {
    my ($logfilename, $texfile, $advanced_role) = @_;

    my $temporary_file=IO::File->new($logfilename) || die "Couldn't open log file $logfilename for reading: $!\n";
    my @content_of_file = <$temporary_file>;
    close $temporary_file; 
    my $body_log_file = join(' ',@content_of_file);
    $logfilename =~ s/\.log$/\.html/;
    $temporary_file = IO::File->new('>'.$logfilename); 
    print $temporary_file '<html><head><title>LOGFILE</title></head><body><pre>'.$body_log_file.'</pre></body></html>'."\n";
    if ($body_log_file=~m/!\s+Emergency stop/) {
	my $whereitbegins = rindex $body_log_file,'STAMPOFPASSEDRESOURCESTART';
	my $whereitends = rindex $body_log_file,'STAMPOFPASSEDRESOURCEEND';
	my $badresource;
	my $badtext;
	if ($whereitbegins!=-1 and $whereitends!=-1) {
	    $badtext = substr($body_log_file,$whereitbegins+26, $whereitends-$whereitbegins-26);
	    $whereitbegins  = rindex $badtext,'located in';
	    if ($whereitbegins != -1) {
		
		$badresource = substr($badtext, $whereitbegins+27, 
				      length($badtext) - $whereitbegins - 48);
		# print "<br />failing resourcename: $badresource<br />";
	    }
        }

	# Guys with privileged roles get a more detailed error output:

	if ($advanced_role) {  
	    #LaTeX failed to parse tex file 
	    print "<h2>".&mt('LaTeX could not successfully parse your TeX file.')."</h2>";
	    print &mt('It probably has errors in it.')."<br />";
	    if ($badtext) {
                print &mt('With very high probability this error occurred in [_1].',$badtext)
                     ."<br /><br />";
            }
	    print &mt('Here are the error messages in the LaTeX log file:')
                 ."<br /><pre>";
	    
	    my $sygnal = 0;
	    for (my $i=0;$i<=$#content_of_file;$i++) {
		if ($content_of_file[$i]=~m/^Runaway argument?/ or $content_of_file[$i]=~m/^!/) {
		    $sygnal = 1;
		} 
		if ($content_of_file[$i]=~m/Here is how much of/) {
		    $sygnal = 0;
		} 
		if ($sygnal) {
		    print "$content_of_file[$i]";
		}  
	    }
	    print "</pre>\n";
	    # print "<br /> Advanced role <br />";
	    $logfilename=~s{^\Q$perlvar{'lonPrtDir'}\E}{/prtspool};
	    print "<b><big>"
                 .&mt('The link to [_1]Your log file[_2]','<a href="'.$logfilename.'">','</a>')
	         ."</big></b>\n";
	    #link to original LaTeX file
	    my $tex_temporary_file=IO::File->new($texfile) || die "Couldn't open tex file $texfile for reading: $!\n";
	    my @tex_content_of_file = <$tex_temporary_file>;
	    close $tex_temporary_file; 
	    my $body_tex_file = join(' ',@tex_content_of_file);
	    $texfile =~ s/\.tex$/aaaaa\.html/;
	    $tex_temporary_file = IO::File->new('>'.$texfile); 
	    print $tex_temporary_file '<html><head><title>LOGFILE</title></head><body><pre>'.$body_tex_file.'</pre></body></html>'."\n";
	    print "<br /><br />";
	    $texfile=~s{^\Q$perlvar{'lonPrtDir'}\E}{/prtspool};
	    print "<b><big>"
                 .&mt('The link to [_1]Your original LaTeX file[_2]','<a href="'.$texfile.'">','</a>')
	         ."</big></b><br /><br />\n";
	    my $help_text = &Apache::loncommon::help_open_topic("Print_Resource", &mt('Help on printing'));
	    print ("$help_text");

	    # Students on the other hand get a minimal error message, since they won't
	    # be able to correct the error message. A message is sent to the 
	    # instructor:

	} else {		# Student role...
	    #  at this point:
	    #    $body_log_file - contains the log file.
	    #    $name_file     - is the name of the LaTeX file.
	    #    $identifier    - is the unique LaTeX identifier.l
	    
            print '<p class="LC_error">';
	    if ($badtext) {
                print &mt('There are errors in [_1].',$badtext);
            } else {
                print &mt('There are errors.');
            }
            print '</p>'
                 .&mt('These errors prevent this resource from printing correctly.');

	    my $tex_handle = IO::File->new($texfile);
	    my @tex_contents = <$tex_handle>;
	    &send_error_mail($identifier, $badresource, $body_log_file, \@tex_contents);
	    print "<p>"
                 .&mt('A message has been sent to the instructor describing this failure.')
                 ."</p>";
	    my $help_text = &Apache::loncommon::help_open_topic("Print_Resource", &mt('Help on printing'));
	    print  ("$help_text");
	    
	  }

	# Either way, an emergency stop does not allow us to continue so:

	return 0;
	
	# The branch of code below is taken if it appears that 
	# there was no emergency stop but LaTeX had to correct the
	# input file to run.
	# In that case we need to provide error feedback, as the correction >may< not be
	# sufficient, we can let the game continue as there's a dvi file to process.

    } elsif ($body_log_file=~m/<inserted text>/) {
	my $whereitbegins = index $body_log_file,'<inserted text>';
	print &mt('You are running LaTeX in [_1]batch mode[_2].','<b>','</b>');
	while ($whereitbegins != -1) {
	    my $tempobegin=$whereitbegins;
	    $whereitbegins = rindex $body_log_file,'STAMPOFPASSEDRESOURCESTART',$whereitbegins;
	    my $whereitends = index $body_log_file,'STAMPOFPASSEDRESOURCEEND',$whereitbegins;
	    print "<br />"
                 .&mt('It has found an error in [_1][_2]and corrected it.',substr($body_log_file,$whereitbegins+26,$whereitends-$whereitbegins-26),"<br />")."\n";
	    print &mt('Usually this correction is valid but you probably need to check the indicated resource one more time and implement necessary corrections by yourself.')."\n";
	    $whereitbegins = index $body_log_file,'<inserted text>',$tempobegin+10;
	}

	if ($advanced_role) {  
	    print "<br /><br />";
	    $logfilename=~s{^\Q$perlvar{'lonPrtDir'}\E}{/prtspool};
	    print "<b><big>"
                 .&mt('The link to [_1]Your log file[_2]'
                     ,'<a href="'.$logfilename.'">','</a>')
	         ."</big></b>\n";
	    #link to original LaTeX file
	    my $tex_temporary_file=IO::File->new($texfile) || die "Couldn't open tex file $texfile for reading: $!\n";
	    my @tex_content_of_file = <$tex_temporary_file>;
	    close $tex_temporary_file; 
	    my $body_tex_file = join(' ',@tex_content_of_file);
	    $texfile =~ s/\.tex$/aaaaa\.html/;
	    $tex_temporary_file = IO::File->new('>'.$texfile); 
	    print $tex_temporary_file '<html><head><title>LOGFILE</title></head><body><pre>'.$body_tex_file.'</pre></body></html>'."\n";
	    print "<br /><br />";
	    $texfile=~s{^\Q$perlvar{'lonPrtDir'}\E}{/prtspool};
	    print "<b><big>"
                  .&mt('The link to [_1]Your original LaTeX file[_2]','<a href="'.$texfile.'">','</a>');
	    print "</big></b>\n";
	}
	return 1;
    }
    return 1;			# NO log file issues at all.
}
