# The LearningOnline Network with CAPA
#
# $Id: lonstudentassessment.pm,v 1.169.2.8 2024/07/02 14:19:01 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
# (Navigate problems for statistical reports
#
#######################################################
#######################################################

=pod

=head1 NAME

lonstudentassessment

=head1 SYNOPSIS

Presents assessment data about a student or a group of students.

=head1 Subroutines

=over 4 

=cut

#######################################################
#######################################################

package Apache::lonstudentassessment;

use strict;
use Apache::lonstatistics();
use Apache::lonhtmlcommon();
use Apache::loncommon();
use Apache::loncoursedata;
use Apache::lonnet; # for logging porpoises
use Apache::lonlocal;
use Apache::grades();
use Apache::lonmsgdisplay();
use Time::HiRes;
use Spreadsheet::WriteExcel;
use Spreadsheet::WriteExcel::Utility();
use lib '/home/httpd/lib/perl/';
use LONCAPA;
 

#######################################################
#######################################################
=pod

=item Package Variables

=over 4

=item $Statistics Hash ref to store student data.  Indexed by symb,
      contains hashes with keys 'score' and 'max'.

=cut

#######################################################
#######################################################

my $Statistics;

#######################################################
#######################################################

=pod

=item $show_links 'yes' or 'no' for linking to student performance data

=item $output_mode 'html', 'excel', or 'csv' for output mode

=item $show 'all', 'totals', or 'scores' determines how much data is output

=item $single_student_mode evaluates to true if we are showing only one
student.

=cut

#######################################################
#######################################################
my $show_links;
my $output_mode;
my $chosen_output;
my $single_student_mode;

#######################################################
#######################################################
# End of package variable declarations

=pod

=back

=cut

#######################################################
#######################################################

=pod

=item &BuildStudentAssessmentPage()

Inputs: 

=over 4

=item $r Apache Request

=item $c Apache Connection 

=back

=cut

#######################################################
#######################################################
sub BuildStudentAssessmentPage {
    my ($r,$c)=@_;
    #
    undef($Statistics);
    undef($show_links);
    undef($output_mode);
    undef($chosen_output);
    undef($single_student_mode);
    #
    my %Saveable_Parameters = ('Status' => 'scalar',
                               'chartoutputmode' => 'scalar',
                               'chartoutputdata' => 'scalar',
                               'Section' => 'array',
                               'Groups' => 'array',
                               'StudentData' => 'array',
                               'Maps' => 'array');
    &Apache::loncommon::store_course_settings('chart',\%Saveable_Parameters);
    &Apache::loncommon::restore_course_settings('chart',\%Saveable_Parameters);
    #
    &Apache::lonstatistics::PrepareClasslist();
    #
    $single_student_mode = 0;
    $single_student_mode = 1 if ($env{'form.SelectedStudent'});
    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},
                                            ['selectstudent']);
    if ($env{'form.selectstudent'}) {
        &Apache::lonstatistics::DisplayClasslist($r);
        return;
    }
    $r->print(&Apache::lonhtmlcommon::breadcrumbs('Chart','Chart_Description:Chart_Sections:Chart_Student_Data:Chart_Enrollment_Status:Chart_Sequences:Chart_Output_Formats:Chart_Output_Data'));

    #
    # Print out the HTML headers for the interface
    #    This also parses the output mode selector
    #    This step must *always* be done.
    $r->print(&CreateInterface());
    $r->print('<input type="hidden" name="notfirstrun" value="true" />');
    $r->print('<input type="hidden" name="sort" value="'.
              $env{'form.sort'}.'" />');
    $r->rflush();
    #
    if (! exists($env{'form.notfirstrun'}) && ! $single_student_mode) {
        return;
    }
    $r->print('<h4>'.
              &Apache::lonstatistics::section_and_enrollment_description().
              '</h4>');
    #
    my $initialize     = \&html_initialize;
    my $output_student = \&html_outputstudent;
    my $finish         = \&html_finish;
    #
    if ($output_mode eq 'excel') {
        $initialize     = \&excel_initialize;
        $output_student = \&excel_outputstudent;
        $finish         = \&excel_finish;
    } elsif ($output_mode eq 'csv') {
        $initialize     = \&csv_initialize;
        $output_student = \&csv_outputstudent;
        $finish         = \&csv_finish;
    }
    #
    if($c->aborted()) {  return ; }
    #
    # Determine which students we want to look at
    my @Students;
    if ($single_student_mode) {
        @Students = (&Apache::lonstatistics::current_student());
        $r->print(&next_and_previous_buttons());
        $r->rflush();
    } else {
        @Students = @Apache::lonstatistics::Students;
    }
    #
    # Perform generic initialization tasks
    #       Since we use lonnet::EXT to retrieve problem weights,
    #       to ensure current data we must clear the caches out.
    #       This makes sure that parameter changes at the student level
    #       are immediately reflected in the chart.
    &Apache::lonnet::clear_EXT_cache_status();
    #
    # Clean out loncoursedata's package data, just to be safe.
    &Apache::loncoursedata::clear_internal_caches();
    #
    # Call the initialize routine selected above
    $initialize->($r);
    foreach my $student (@Students) {
        if($c->aborted()) { 
            $finish->($r);
            return ; 
        }
        # Call the output_student routine selected above
        $output_student->($r,$student);
    }
    # Call the "finish" routine selected above
    $finish->($r);
    #
    return;
}

#######################################################
#######################################################
sub next_and_previous_buttons {
    my $Str = '';
    $Str .= '<input type="hidden" name="SelectedStudent" value="'.
        $env{'form.SelectedStudent'}.'" />';
    #
    # Build the previous student link
    my $previous = &Apache::lonstatistics::previous_student();
    my $previousbutton = '';
    if (defined($previous)) {
        my $sname = $previous->{'username'}.':'.$previous->{'domain'};
        $previousbutton .= '<input type="button" value="'.
            &mt('Previous Student ([_1])',
            $previous->{'username'}.':'.$previous->{'domain'}).
            '" onclick="document.Statistics.SelectedStudent.value='.
            "'".$sname."'".';'.
            'document.Statistics.submit();" />';
    } else {
        $previousbutton .= '<input type="button" value="'.
            &mt('Previous Student').'" disabled="disabled" />';
    }
    #
    # Build the next student link
    my $next = &Apache::lonstatistics::next_student();
    my $nextbutton = '';
    if (defined($next)) {
        my $sname = $next->{'username'}.':'.$next->{'domain'};
        $nextbutton .= '<input type="button" value="'.
            &mt('Next Student ([_1])',
            $next->{'username'}.':'.$next->{'domain'}).
            '" onclick="document.Statistics.SelectedStudent.value='.
            "'".$sname."'".';'.
            'document.Statistics.submit();" />';
    } else {
        $nextbutton .= '<input type="button" value="'.
            &mt('Next Student').'" disabled="disabled" />';
    }
    #
    # Build the 'all students' button
    my $all = '';
    $all .= '<input type="button" value="'.&mt('All Students').'" '.
            ' onclick="document.Statistics.SelectedStudent.value='.
            "''".';'.'document.Statistics.submit();" />';
    $Str .= $previousbutton.('&nbsp;'x5).$all.('&nbsp;'x5).$nextbutton;
    return $Str;
}

#######################################################
#######################################################

sub get_student_fields_to_show {
    my @to_show = @Apache::lonstatistics::SelectedStudentData;
    foreach (@to_show) {
        if ($_ eq 'all') {
            @to_show = @Apache::lonstatistics::StudentDataOrder;
            last;
        }
    }
    return @to_show;
}

#######################################################
#######################################################

=pod

=item &CreateInterface()

Called by &BuildStudentAssessmentPage to create the top part of the
page which displays the chart.

Inputs: None

Returns:  A string containing the HTML for the headers and top table for 
the chart page.

=cut

#######################################################
#######################################################
sub CreateInterface {
    my $Str = '';
    $Str .= '<table cellspacing="5">'."\n";
    $Str .= '<tr>';
    $Str .= '<td align="center"><b>'.&mt('Sections').'</b>'.
	&Apache::loncommon::help_open_topic("Chart_Sections").
	'</td>';
    $Str .= '<td align="center"><b>'.&mt('Groups').'</b>'.
	'</td>';
    $Str .= '<td align="center"><b>'.&mt('Student Data').'</b>'.
	&Apache::loncommon::help_open_topic("Chart_Student_Data").
	'</td>';
    $Str .= '<td align="center"><b>'.&mt('Access Status').'</b>'.
	&Apache::loncommon::help_open_topic("Chart_Enrollment_Status").
	'</td>';
    $Str .= '<td align="center"><b>'.&mt('Sequences and Folders').'</b>'.
	&Apache::loncommon::help_open_topic("Chart_Sequences").
	'</td>';
    $Str .= '<td align="center"><b>'.&mt('Output Format').'</b>'.
        &Apache::loncommon::help_open_topic("Chart_Output_Formats").
        '</td>';
    $Str .= '<td align="center"><b>'.&mt('Output Data').'</b>'.
        &Apache::loncommon::help_open_topic("Chart_Output_Data").
        '</td>';
    $Str .= '</tr>'."\n";
    #
    $Str .= '<tr><td align="center">'."\n";
    $Str .= &Apache::lonstatistics::SectionSelect('Section','multiple',5);
    $Str .= '</td><td align="center">';
    $Str .= &Apache::lonstatistics::GroupSelect('Group','multiple',5);
    $Str .= '</td><td align="center">';
    $Str .= &Apache::lonstatistics::StudentDataSelect('StudentData','multiple',
                                                      5,undef);
    $Str .= '</td><td>'."\n";
    $Str .= &Apache::lonhtmlcommon::StatusOptions(undef,undef,5);
    $Str .= '</td><td>'."\n";
    $Str .= &Apache::lonstatistics::map_select('Maps','multiple,all',5);
    $Str .= '</td><td>'."\n";
    $Str .= &CreateAndParseOutputSelector();
    $Str .= '</td><td>'."\n";
    $Str .= &CreateAndParseOutputDataSelector();
    $Str .= '</td></tr>'."\n";
    $Str .= '</table>'."\n";
    $Str .= '<input type="submit" name="selectstudent" value="'.
        &mt('Select One Student').'" />';
    $Str .= '&nbsp;'x5;
    $Str .= '<input type="submit" name="ClearCache" value="'.
        &mt('Clear Caches').'" />';
    $Str .= '<p>'
           .'<input type="submit" name="Generate Chart"'
           .' value="'.&mt('Generate Chart').'" />'
           .'</p>';
    return $Str;
}

#######################################################
#######################################################

=pod

=item &CreateAndParseOutputSelector()

=cut

#######################################################
#######################################################
my @OutputOptions = 
    ({ name  => 'HTML, with links',
       value => 'html, with links',
       description => 'Output HTML with each symbol linked to the problem '.
	   'which generated it.',
       mode => 'html',
       show_links => 'yes',
       },
     { name  => 'HTML, with all links',
       value => 'html, with all links',
       description => 'Output HTML with each symbol linked to the problem '.
	   'which generated it.  '.
           'This includes links for unattempted problems.',
       mode => 'html',
       show_links => 'all',
       },
     { name  => 'HTML, without links',
       value => 'html, without links',
       description => 'Output HTML.  By not including links, the size of the'.
	   ' web page is greatly reduced.  If your browser crashes on the '.
	   'full display, try this.',
       mode => 'html',
       show_links => 'no',
           },
     { name  => 'Excel',
       value => 'excel',
       description => 'Output an Excel file (compatable with Excel 95).',
       mode => 'excel',
       show_links => 'no',
   },
     { name  => 'CSV',
       value => 'csv',
       description => 'Output a comma separated values file suitable for '.
           'import into a spreadsheet program.  Using this method as opposed '.
           'to Excel output allows you to organize your data before importing'.
           ' it into a spreadsheet program.',
       mode => 'csv',
       show_links => 'no',
           },
     );

sub OutputDescriptions {
    my $Str = '';
    $Str .= '<h2>'.&mt('Output Formats')."</h2>\n";
    $Str .= "<dl>\n";
    foreach my $outputmode (@OutputOptions) {
	$Str .="    <dt>".$outputmode->{'name'}."</dt>\n";
	$Str .="        <dd>".$outputmode->{'description'}."</dd>\n";
    }
    $Str .= "</dl>\n";
    return $Str;
}

sub CreateAndParseOutputSelector {
    my $Str = '';
    my $elementname = 'chartoutputmode';
    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},
                                            [$elementname]);
    #
    # Format for output options is 'mode, restrictions';
    my $selected = (&Apache::loncommon::get_env_multiple('form.'.$elementname))[0];
    $selected = 'html, without links' if (!$selected);

    #
    # Set package variables describing output mode
    $show_links  = 'no';
    $output_mode = 'html';
    foreach my $option (@OutputOptions) {
        next if ($option->{'value'} ne $selected);
        $output_mode = $option->{'mode'};
        $show_links  = $option->{'show_links'};
    }

    #
    # Build the form element
    $Str = qq/<select size="5" name="$elementname">/;
    foreach my $option (@OutputOptions) {
        $Str .= "\n".'    <option value="'.$option->{'value'}.'"';
        $Str .= ' selected="selected"' if ($option->{'value'} eq $selected);
        $Str .= ">".&mt($option->{'name'})."<\/option>";
    }
    $Str .= "\n</select>";
    return $Str;
}

##
## Data selector stuff
##
my @OutputDataOptions =
    (
     { name  => 'Scores Summary',
       base  => 'scores',
       value => 'sum and total',
       scores => 1,
       tries  => 0,
       every_problem => 0,
       sequence_sum => 1,
       sequence_max => 1,
       grand_total => 1,
       grand_maximum => 1,
       summary_table => 1,
       maximum_row => 1,
       ignore_weight => 0,
       shortdesc => 'Total Score and Maximum Possible for each '.
           'Sequence or Folder',
       longdesc => 'The score of each student as well as the '.
           ' maximum possible on each Sequence or Folder.',
       },
     { name  => 'Scores Per Problem',
       base  => 'scores',
       value => 'scores',
       scores => 1,
       tries  => 0,
       correct => 0,
       every_problem => 1,
       sequence_sum => 1,
       sequence_max => 1,
       grand_total => 1,
       grand_maximum => 1,
       summary_table => 1,
       maximum_row => 1,
       ignore_weight => 0,
       shortdesc => 'Score on each Problem Part',
       longdesc =>'The students score on each problem part, computed as'.
           'the part weight * part awarded',
       },
     { name  =>'Tries',
       base  =>'tries',
       value => 'tries',
       scores => 0,
       tries  => 1,
       correct => 0,
       every_problem => 1,
       sequence_sum => 0,
       sequence_max => 0,
       grand_total => 0,
       grand_maximum => 0,
       summary_table => 0,
       maximum_row => 0,
       ignore_weight => 0,
       shortdesc => 'Number of Tries before success on each Problem Part',
       longdesc =>'The number of tries before success on each problem part.',
       non_html_notes => 'negative values indicate an incorrect problem',
       },
     { name  =>'Parts Correct',
       base  =>'tries',
       value => 'parts correct total',
       scores => 0,
       tries  => 0,
       correct => 1,
       every_problem => 1,
       sequence_sum => 1,
       sequence_max => 1,
       grand_total => 1,
       grand_maximum => 1,
       summary_table => 1,
       maximum_row => 0,
       ignore_weight => 1,
       shortdesc => 'Number of Problem Parts completed successfully',
       longdesc => 'The Number of Problem Parts completed successfully and '.
           'the maximum possible for each student',
       },
     );

sub HTMLifyOutputDataDescriptions {
    my $Str = '';
    $Str .= '<h2>'.&mt('Output Data').'</h2>'."\n";
    $Str .= "<dl>\n";
    foreach my $option (@OutputDataOptions) {
        $Str .= '    <dt>'.$option->{'name'}.'</dt>';
        $Str .= '<dd>'.$option->{'longdesc'}.'</dd>'."\n";
    }
    $Str .= "</dl>\n";
    return $Str;
}

sub CreateAndParseOutputDataSelector {
    my $Str = '';
    my $elementname = 'chartoutputdata';
    #
    my $selected = (&Apache::loncommon::get_env_multiple('form.'.$elementname))[0];
    $selected = 'scores' if (!$selected);

    #
    $chosen_output = $OutputDataOptions[0];
    foreach my $option (@OutputDataOptions) {
        if ($option->{'value'} eq $selected) {
            $chosen_output = $option;
        }
    }
    #
    # Build the form element
    $Str = qq/<select size="5" name="$elementname">/;
    foreach my $option (@OutputDataOptions) {
        $Str .= "\n".'    <option value="'.$option->{'value'}.'"';
        $Str .= ' selected="selected"' if ($option->{'value'} eq $chosen_output->{'value'});
        $Str .= ">".&mt($option->{'name'})."<\/option>";
    }
    $Str .= "\n</select>";
    return $Str;

}

#######################################################
#######################################################
sub count_parts {
    my ($navmap,$sequence) = @_;
    my @resources = &get_resources($navmap,$sequence);
    my $count = 0;
    foreach my $res (@resources) {
        $count += scalar(@{$res->parts});
    }
    return $count;
}

sub get_resources {
    my ($navmap,$sequence) = @_;
    my @resources = $navmap->retrieveResources($sequence,
                                               sub { shift->is_problem(); },
                                               0,0,0);
    return @resources;
}

#######################################################
#######################################################

=pod

=head2 HTML output routines

=item &html_initialize($r)

Create labels for the columns of student data to show.

=item &html_outputstudent($r,$student)

Return a line of the chart for a student.

=item &html_finish($r)

=cut

#######################################################
#######################################################
{
    my $padding;
    my $count;

    my $nodata_count; # The number of students for which there is no data
    my %prog_state;   # progress state used by loncommon PrgWin routines
    my $total_sum_width;

    my %width; # Holds sequence width information
    my @sequences;
    my $navmap; # Have to keep this around since weakref is a bit zealous

sub html_cleanup {
    undef(%prog_state);
    undef(%width);
    #
    undef($navmap);
    undef(@sequences);
}

sub html_initialize {
    my ($r) = @_;
    #
    $padding = ' 'x3;
    $count = 0;
    $nodata_count = 0;
    &html_cleanup();
    ($navmap,@sequences) = 
        &Apache::lonstatistics::selected_sequences_with_assessments();
    if (! ref($navmap)) {
        # Unable to get data, so bail out
        $r->print('<p class="LC_error">'
                 .&mt('Unable to retrieve course information.')
                 .'</p>');
    }

    # If we're showing links, show a checkbox to open in new
    # windows.
    if ($show_links ne 'no') {
        my $labeltext = &mt('Show links in new window');
        $r->print(<<NEW_WINDOW_CHECKBOX);
<script type="text/javascript">new_window = true;</script>
<p><label> 
<input type="checkbox" checked="checked" onclick="new_window=this.checked" />
$labeltext
</label></p>
NEW_WINDOW_CHECKBOX
    }

    #
    $r->print("<h3>".$env{'course.'.$env{'request.course.id'}.'.description'}.
              "&nbsp;&nbsp;".&Apache::lonlocal::locallocaltime(time)."</h3>");
    #
    if ($chosen_output->{'base'} !~ /^final table/) {
        $r->print("<h3>".&mt($chosen_output->{'shortdesc'})."</h3>");        
    }
    my $Str = "<pre>\n";
    # First, the @StudentData fields need to be listed
    my @to_show = &get_student_fields_to_show();
    foreach my $field (@to_show) {
        my $title=$Apache::lonstatistics::StudentData{$field}->{'title'};
        my $base =$Apache::lonstatistics::StudentData{$field}->{'base_width'};
        my $width=$Apache::lonstatistics::StudentData{$field}->{'width'};
        $Str .= $title.' 'x($width-$base).$padding;
    }
    #
    # Compute the column widths and output the sequence titles
    my $total_count;
    #
    # Compute sequence widths
    my $starttime = Time::HiRes::time;
    foreach my $seq (@sequences) {
        my $symb = $seq->symb;
        my $title = $seq->compTitle;
        $width{$symb}->{'width_sum'} = 0;
        # Compute width of sum
        if ($chosen_output->{'sequence_sum'}) {
            if ($chosen_output->{'every_problem'}) {
                # Use 1 digit for a space
                $width{$symb}->{'width_sum'} += 1;            
            }
	    $total_count += &count_parts($navmap,$seq);
            # Use 6 digits for the sum
            $width{$symb}->{'width_sum'} += 6;
        }
        # Compute width of maximum
        if ($chosen_output->{'sequence_max'}) {
            if ($width{$symb}->{'width_sum'}>0) {
                # One digit for the '/'
                $width{$symb}->{'width_sum'} +=1;
            }
            # Use 6 digits for the total
            $width{$symb}->{'width_sum'}+=6;
        }
	#
        if ($chosen_output->{'every_problem'}) {
            # one problem per digit
            $width{$symb}->{'width_parts'}= &count_parts($navmap,$seq);
            $width{$symb}->{'width_problem'} += $width{$symb}->{'width_parts'};
        } else {
            $width{$symb}->{'width_problem'} = 0;
        }
        $width{$symb}->{'width_total'} = $width{$symb}->{'width_problem'} + 
                                     $width{$symb}->{'width_sum'};
        if ($width{$symb}->{'width_total'} < length(&HTML::Entities::decode($title))) {
            $width{$symb}->{'width_total'} = length(&HTML::Entities::decode($title));
        }
        #
        # Output the sequence titles
        $Str .= $title.(' 'x($width{$symb}->{'width_total'}-
                            length($title)
                            )).$padding;
    }
    $total_sum_width = length($total_count);
    if ($total_sum_width < 6) {
        $total_sum_width = 6;
    }
    $Str .= "    total</pre>\n";
    $Str .= "<pre>";

    $r->print(<<JS);
<script type="text/javascript">
// get the left offset of a given widget as an absolute position
function getLeftOffset (element) {
    return collect(element, "offsetLeft");
}

// get the top offset of a given widget as an absolute position
function getTopOffset (element) {
    return collect(element, "offsetTop");
}

function collect(element, att) {
    var val = 0;
    while(element) {
        val += element[att];
        element = element.offsetParent;
    }
    return val;
}

var currentDiv;
var currentElement;
function popup_score(element, score) {
    popdown_score();
    var left = getLeftOffset(element);
    var top = getTopOffset(element);
    var div = document.createElement("div");
    div.className = "LC_chrt_popup";
    div.appendChild(document.createTextNode(score));
    div.style.position = "absolute";
    div.style.top = (top - 25) + "px";
    div.style.left = (left - 10) + "px";
    currentDiv = div;
    document.body.insertBefore(div, document.body.childNodes[0]);
    element.className = "LC_chrt_popup_up";
    currentElement = element;
}

function popdown_score() {
    if (currentDiv) {
        document.body.removeChild(currentDiv);
    }
    if (currentElement) {
        currentElement.className = 'LC_chrt_popup_exists';
    }
    currentDiv = undefined;
}
</script>
JS

    #
    # Let the user know what we are doing
    my $studentcount = scalar(@Apache::lonstatistics::Students); 
    if ($env{'form.SelectedStudent'}) {
        $studentcount = '1';
    }
    #
    # Initialize progress window
    #
    %prog_state=&Apache::lonhtmlcommon::Create_PrgWin($r,$studentcount);
    &Apache::lonhtmlcommon::Update_PrgWin($r,\%prog_state,
                                          'Processing first student');
    $r->print($Str);
    $r->rflush();

    return;
}

sub html_outputstudent {
    my ($r,$student) = @_;
    my $Str = '';
    return if (! defined($navmap));
    #
    if($count++ % 5 == 0 && $count > 0) {
#       $r->print("</pre><pre>");
        $r->print('</pre>');
        &Apache::lonhtmlcommon::Increment_PrgWin(
            $r,\%prog_state,'last five students',5);
        $r->rflush();
        $r->print('<pre>');
    }
    # First, the @StudentData fields need to be listed
    my @to_show = &get_student_fields_to_show();
    foreach my $field (@to_show) {
        my $title=$student->{$field};
        # Deal with 'comments' - how I love special cases
        if ($field eq 'comments') {
            $title = '<a href="/adm/'.$student->{'domain'}.'/'.$student->{'username'}.'/'.'aboutme#coursecomment">'.&mt('Comments').'</a>';
        }
        utf8::decode($title);
        my $base = length($title);
        my $width=$Apache::lonstatistics::StudentData{$field}->{'width'};
        $Str .= $title.' 'x($width-$base).$padding;
    }
    # Get ALL the students data
    my %StudentsData;
    my @tmp = &Apache::loncoursedata::get_current_state
        ($student->{'username'},$student->{'domain'},undef,
         $env{'request.course.id'});
    if ((scalar @tmp > 0) && ($tmp[0] !~ /^error:(.*)/)) {
        %StudentsData = @tmp;
    } else {
	my $error = $1;
	if (scalar(@tmp) < 1) {
	    $Str .= '<span class="LC_warning">'
                   .&mt('No Course Data')
                   .'</span>'."\n";
	} else {
            $Str .= '<span class="LC_error">'
                   .&mt('Error getting student data ([_1])',$error)
                   .'</span>'."\n";
	}
        $nodata_count++;
        $r->print($Str);
        $r->rflush();
        return;
    }
    #
    # By sequence build up the data
    my $studentstats;
    my $PerformanceStr = '';
    foreach my $seq (@sequences) {
        my $symb = $seq->symb;
        my $randompick = $seq->randompick();
        my ($performance,$performance_length,$score,$seq_max,$rawdata);
        if ($chosen_output->{'tries'}) {
            ($performance,$performance_length,$score,$seq_max,$rawdata) =
                &student_tries_on_sequence($student,\%StudentsData,
                                           $navmap,$seq,$show_links,$randompick);
        } else {
            ($performance,$performance_length,$score,$seq_max,$rawdata) =
                &student_performance_on_sequence($student,\%StudentsData,
                                                 $navmap,$seq,$show_links,
                                                 $chosen_output->{ignore_weight},
                                                 $randompick);
        }
        my $ratio='';
        if ($chosen_output->{'every_problem'} && 
            $chosen_output->{'sequence_sum'}) {
            $ratio .= ' ';
        }
        if ($chosen_output->{'sequence_sum'} && $score ne ' ') {
            my $score .= sprintf("%3.2f",$score);
            $ratio .= (' 'x(6-length($score))).$score;
        } elsif($chosen_output->{'sequence_sum'}) {
            $ratio .= ' 'x6;
        }
        if ($chosen_output->{'sequence_max'}) {
            if ($chosen_output->{'sequence_sum'}) {
                $ratio .= '/';
            }
            my $sequence_total=sprintf("%3.2f",$seq_max);
            $ratio .= $sequence_total.(' 'x(6-length($sequence_total)));
        }
        #
        if (! $chosen_output->{'every_problem'}) {
            $performance = '';
	    $performance_length=0;
        }
        $performance .= ' 'x($width{$symb}->{'width_total'} -
                             $performance_length -
                             $width{$symb}->{'width_sum'}).
            $ratio;
        #
        $Str .= $performance.$padding;
        #
        $studentstats->{$symb}->{'score'}= $score;
        $studentstats->{$symb}->{'max'}  = $seq_max;
    }
    #
    # Total it up and store the statistics info.
    my ($score,$max);
    while (my ($symb,$seq_stats) = each (%{$studentstats})) {
        $Statistics->{$symb}->{'score'} += $seq_stats->{'score'};
        if ($Statistics->{$symb}->{'max'} < $seq_stats->{'max'}) {
            $Statistics->{$symb}->{'max'} = $seq_stats->{'max'};
        }
        if ($seq_stats->{'score'} ne ' ') {
            $score += $seq_stats->{'score'};
            $Statistics->{$symb}->{'num_students'}++;
        }
        $max   += $seq_stats->{'max'};
    }
    if (! defined($score)) {
        $score = ' ' x $total_sum_width;
    } else {
        $score = sprintf("%.2f",$score);
        $score = (' 'x(6-length($score))).$score;
    }
    $max = sprintf("%.2f",$max);
    $Str .= ' '.' 'x($total_sum_width-length($score)).$score.' / '.$max;
    $Str .= " \n";
    #
    $r->print($Str);
    #
#   $r->rflush();
#   &Apache::lonhtmlcommon::Increment_PrgWin($r,\%prog_state,'last student');
    return;
}    

sub html_finish {
    my ($r) = @_;
    return if (! defined($navmap));
    #
    # Check for suppressed output and close the progress window if so
    $r->print("</pre>\n"); 
    if ($chosen_output->{'summary_table'}) {
        if ($single_student_mode) {
            $r->print(&SingleStudentTotal());
        } else {
            $r->print(&StudentAverageTotal());
        }
    }
    $r->rflush();
    &Apache::lonhtmlcommon::Close_PrgWin($r,\%prog_state);
    &html_cleanup();
    return;
}

sub StudentAverageTotal {
    my $Str = '<h3>'.&mt('Summary Tables').'</h3>'.$/;
    $Str .= &Apache::loncommon::start_data_table();
    $Str .= &Apache::loncommon::start_data_table_header_row().
        '<th>'.&mt('Title').'</th>'.
        '<th>'.&mt('Average').'</th>'.
        '<th>'.&mt('Maximum').'</th>'.
        &Apache::loncommon::end_data_table_header_row().$/;
    foreach my $seq (@sequences) {
        my $symb = $seq->symb;
        my $ave;
        my $num_students = $Statistics->{$symb}->{'num_students'};
        if ($num_students > 0) {
            $ave = int(100*
                       ($Statistics->{$symb}->{'score'}/$num_students)
                       )/100;
        } else {
            $ave = 0;
        }
        my $max = $Statistics->{$symb}->{'max'};
        $ave = sprintf("%.2f",$ave);
        $Str .= &Apache::loncommon::start_data_table_row().
            '<td>'.$seq->compTitle.'</td>'.
            '<td align="right">'.$ave.'&nbsp;</td>'.
            '<td align="right">'.$max.'&nbsp;'.'</td>'.
            &Apache::loncommon::end_data_table_row()."\n";
    }
    $Str .= &Apache::loncommon::end_data_table()."\n";
    return $Str;
}

sub SingleStudentTotal {
    return if (! defined($navmap));
    my $student = &Apache::lonstatistics::current_student();
    my $Str = '<h3>'.&mt('Summary table for [_1] ([_2])',
                         $student->{'fullname'},
                         $student->{'username'}.':'.$student->{'domain'}).'</h3>';
    $Str .= $/;
    $Str .= &Apache::loncommon::start_data_table()."\n";
    $Str .= 
        &Apache::loncommon::start_data_table_header_row().
        '<th>'.&mt('Sequence or Folder').'</th>';
    if ($chosen_output->{'base'} eq 'tries') {
        $Str .= '<th>'.&mt('Parts Correct').'</th>';
    } else {
        $Str .= '<th>'.&mt('Score').'</th>';
    }
    $Str .= '<th>'.&mt('Maximum').'</th>'.
            &Apache::loncommon::end_data_table_header_row()."\n";
    my $total = 0;
    my $total_max = 0;
    foreach my $seq (@sequences) {
        my $value = $Statistics->{$seq->symb}->{'score'};
        my $max = $Statistics->{$seq->symb}->{'max'};
        $Str .= &Apache::loncommon::start_data_table_row().
            '<td>'.&HTML::Entities::encode($seq->compTitle).'</td>'.
            '<td align="right">'.$value.'</td>'.
            '<td align="right">'.$max.'</td>'.
            &Apache::loncommon::end_data_table_row()."\n";
        $total += $value;
        $total_max +=$max;
    }
    $Str .= &Apache::loncommon::start_data_table_row().
        '<td><b>'.&mt('Total').'</b></td>'.
        '<td align="right">'.$total.'</td>'.
        '<td align="right">'.$total_max.'</td>'.
        &Apache::loncommon::end_data_table_row()."\n";
    $Str .= &Apache::loncommon::end_data_table()."\n";
    return $Str;
}

}

#######################################################
#######################################################

=pod

=head2 EXCEL subroutines

=item &excel_initialize($r)

=item &excel_outputstudent($r,$student)

=item &excel_finish($r)

=cut

#######################################################
#######################################################
{

my $excel_sheet;
my $excel_workbook;
my $format;

my $filename;
my $rows_output;
my $cols_output;

my %prog_state; # progress window state
my $request_aborted;

my $total_formula;
my $maximum_formula;
my %formula_data;

my $navmap;
my @sequences;

sub excel_cleanup {
    #
    undef ($excel_sheet);
    undef ($excel_workbook);
    undef ($filename);
    undef ($rows_output);
    undef ($cols_output);
    undef (%prog_state);
    undef ($request_aborted);
    undef ($total_formula);
    undef ($maximum_formula);
    #
    undef(%formula_data);
    #
    undef($navmap);
    undef(@sequences);
}

sub excel_initialize {
    my ($r) = @_;

    &excel_cleanup();
    ($navmap,@sequences) = 
        &Apache::lonstatistics::selected_sequences_with_assessments();
    if (! ref($navmap)) {
        # Unable to get data, so bail out
        $r->print('<p class="LC_error">'.
                  &mt('Unable to retrieve course information.').
                  '</p>');
    }
    #
    my $total_columns = scalar(&get_student_fields_to_show());
    my $num_students = scalar(@Apache::lonstatistics::Students);
    #
    foreach my $seq (@sequences) {
        if ($chosen_output->{'every_problem'}) {
            $total_columns+=&count_parts($navmap,$seq);
        }
        # Add 2 because we need a 'sequence_sum' and 'total' column for each
        $total_columns += 2;
    }
    my $too_many_cols_error_message = 
        '<h2>'.&mt('Unable to Complete Request').'</h2>'.$/.
        '<p class="LC_warning">'.&mt('LON-CAPA is unable to produce your Excel spreadsheet because your selections will result in more than 255 columns.  Excel allows only 255 columns in a spreadsheet.').'</p>'.$/.
        '<p>'.&mt('You may consider reducing the number of [_1]Sequences or Folders[_2] you have selected.','<b>','</b>').'</p>'.$/.
        '<p>'.&mt('LON-CAPA can produce [_1]CSV[_2] files of this data or Excel files of the [_1]Scores Summary[_2] data.','<b>','</b>').'</p>'.$/;
    if ($chosen_output->{'base'} eq 'tries' && $total_columns > 255) {
        $r->print($too_many_cols_error_message);
        $request_aborted = 1;
    }
    if ($chosen_output->{'base'} eq 'scores' && $total_columns > 255) {
        $r->print($too_many_cols_error_message);
        $request_aborted = 1;
    }
    return if ($request_aborted);
    #
    #
    $excel_workbook = undef;
    $excel_sheet = undef;
    #
    $rows_output = 0;
    $cols_output = 0;
    #
    # Determine rows 
    my $header_row = $rows_output++;
    my $description_row = $rows_output++;
    my $notes_row = $rows_output++;
    $rows_output++;        # blank row
    my $summary_header_row;
    if ($chosen_output->{'summary_table'}) {
        $summary_header_row = $rows_output++;
        $rows_output+= scalar(@sequences);
        $rows_output++;
    }
    my $sequence_name_row = $rows_output++;
    my $resource_name_row = $rows_output++;
    my $maximum_data_row = $rows_output++;
    if (! $chosen_output->{'maximum_row'}) {
        $rows_output--;
    }
    my $first_data_row = $rows_output++;
    #
    # Create sheet
    ($excel_workbook,$filename,$format)=
        &Apache::loncommon::create_workbook($r);
    return if (! defined($excel_workbook));
    #
    # Add a worksheet
    my $sheetname = $env{'course.'.$env{'request.course.id'}.'.description'};
    $sheetname = &Apache::loncommon::clean_excel_name($sheetname);
    $excel_sheet = $excel_workbook->addworksheet($sheetname);
    #
    # Put the course description in the header
    $excel_sheet->write($header_row,$cols_output++,
                   $env{'course.'.$env{'request.course.id'}.'.description'},
                        $format->{'h1'});
    $cols_output += 3;
    #
    # Put a description of the sections listed
    my $sectionstring = '';
    my @Sections = &Apache::lonstatistics::get_selected_sections();
    $excel_sheet->write($header_row,$cols_output++,
                        &Apache::lonstatistics::section_and_enrollment_description('localized'),
                        $format->{'h3'});
    #
    # Put the date in there too
    $excel_sheet->write($header_row,$cols_output++,
                        &mt('Compiled on [_1]',&Apache::lonlocal::locallocaltime(time)),$format->{'h3'});
    #
    $cols_output = 0;
    $excel_sheet->write($description_row,$cols_output++,
                        &mt($chosen_output->{'shortdesc'}),
                        $format->{'b'});
    #
    $cols_output = 0;
    $excel_sheet->write($notes_row,$cols_output++,
                        $chosen_output->{'non_html_notes'},
                        $format->{'i'});
    
    ##############################################
    # Output headings for the raw data
    ##############################################
    #
    # Add the student headers
    $cols_output = 0;
    foreach my $field (&get_student_fields_to_show()) {
        $excel_sheet->write($resource_name_row,$cols_output++,&mt($field),
                            $format->{'bold'});
    }
    #
    # Add the remaining column headers
    my $total_formula_string = '=0';
    my $maximum_formula_string = '=0';
    foreach my $seq (@sequences) {
        my $symb = $seq->symb;
        $excel_sheet->write($sequence_name_row,,
                            $cols_output,$seq->compTitle,$format->{'bold'});
        # Determine starting cell
        $formula_data{$symb}->{'Excel:startcell'}=
            &Spreadsheet::WriteExcel::Utility::xl_rowcol_to_cell
            ($first_data_row,$cols_output);
        $formula_data{$symb}->{'Excel:startcol'}=$cols_output;
        my $count = 0;
        if ($chosen_output->{'every_problem'}) {
            # Put the names of the problems and parts into the sheet
            foreach my $res (&get_resources($navmap,$seq)) {
                if (scalar(@{$res->parts}) > 1) {
                    foreach my $part (@{$res->parts}) {
                        $excel_sheet->write($resource_name_row,
                                            $cols_output++,
                                            $res->compTitle.' part '.$res->part_display($part),
                                            $format->{'bold'});
                        $count++;
                    }
                } else {
                    $excel_sheet->write($resource_name_row,
                                        $cols_output++,
                                        $res->compTitle,$format->{'bold'});
                    $count++;
                }
            }
        }
        # Determine ending cell
        if ($count <= 1) {
            $formula_data{$symb}->{'Excel:endcell'} = $formula_data{$symb}->{'Excel:startcell'};
            $formula_data{$symb}->{'Excel:endcol'}  = $formula_data{$symb}->{'Excel:startcol'};
        } else {
            $formula_data{$symb}->{'Excel:endcell'} = 
                &Spreadsheet::WriteExcel::Utility::xl_rowcol_to_cell
                ($first_data_row,$cols_output-1);
            $formula_data{$symb}->{'Excel:endcol'} = $cols_output-1;
        }
        # Create the formula for summing up this sequence
        if (! exists($formula_data{$symb}->{'Excel:endcell'}) ||
            ! defined($formula_data{$symb}->{'Excel:endcell'})) {
            $formula_data{$symb}->{'Excel:endcell'} = $formula_data{$symb}->{'Excel:startcell'};
        }

        my $start = $formula_data{$symb}->{'Excel:startcell'};
        my $end = $formula_data{$symb}->{'Excel:endcell'};
        $formula_data{$symb}->{'Excel:sum'}= $excel_sheet->store_formula
            ("=IF(COUNT($start\:$end),SUM($start\:$end),\"\")");
        # Determine cell the score is held in
        $formula_data{$symb}->{'Excel:scorecell'} = 
            &Spreadsheet::WriteExcel::Utility::xl_rowcol_to_cell
            ($first_data_row,$cols_output);
        $formula_data{$symb}->{'Excel:scorecol'}=$cols_output;
        if ($chosen_output->{'base'} eq 'parts correct total') {
            $excel_sheet->write($resource_name_row,$cols_output++,
                                &mt('parts correct'),
                                $format->{'bold'});
        } elsif ($chosen_output->{'sequence_sum'}) {
            if ($chosen_output->{'correct'}) {
                # Only reporting the number correct, so do not call it score
                $excel_sheet->write($resource_name_row,$cols_output++,
                                    &mt('sum'),
                                    $format->{'bold'});
            } else {
                $excel_sheet->write($resource_name_row,$cols_output++,
                                    &mt('score'),
                                    $format->{'bold'});
            }
        }
        #
        $total_formula_string.='+'.
            &Spreadsheet::WriteExcel::Utility::xl_rowcol_to_cell
            ($first_data_row,$cols_output-1);
        if ($chosen_output->{'sequence_max'}) {
            $excel_sheet->write($resource_name_row,$cols_output,
                                &mt('maximum'),
                                $format->{'bold'});
            $formula_data{$symb}->{'Excel:maxcell'} = 
                &Spreadsheet::WriteExcel::Utility::xl_rowcol_to_cell
                ($first_data_row,$cols_output);
            $formula_data{$symb}->{'Excel:maxcol'}=$cols_output;
            $maximum_formula_string.='+'.
                &Spreadsheet::WriteExcel::Utility::xl_rowcol_to_cell
                ($first_data_row,$cols_output);
            $cols_output++;

        }
    }
    if ($chosen_output->{'grand_total'}) {
        $excel_sheet->write($resource_name_row,$cols_output++,&mt('Total'),
                            $format->{'bold'});
    }
    if ($chosen_output->{'grand_maximum'}) {
        $excel_sheet->write($resource_name_row,$cols_output++,&mt('Max. Total'),
                            $format->{'bold'});
    }
    $total_formula = $excel_sheet->store_formula($total_formula_string);
    $maximum_formula = $excel_sheet->store_formula($maximum_formula_string);
    ##############################################
    # Output a row for MAX, if appropriate
    ##############################################
    if ($chosen_output->{'maximum_row'}) {
        $cols_output = 0;
        foreach my $field (&get_student_fields_to_show()) {
            if ($field eq 'username' || $field eq 'fullname' || 
                $field eq 'id') {
                $excel_sheet->write($maximum_data_row,$cols_output++,'Maximum',
                                    $format->{'bold'});
            } else {
                $excel_sheet->write($maximum_data_row,$cols_output++,'');
            }
        }
        #
        # Add the maximums for each sequence or assessment
        my %total_cell_translation;
        my %maximum_cell_translation;
        foreach my $seq (@sequences) {
            my $symb = $seq->symb;
            $cols_output=$formula_data{$symb}->{'Excel:startcol'};
            $total_cell_translation{$formula_data{$symb}->{'Excel:scorecell'}}=
                &Spreadsheet::WriteExcel::Utility::xl_rowcol_to_cell
                ($maximum_data_row,$formula_data{$symb}->{'Excel:scorecol'});
            $maximum_cell_translation{$formula_data{$symb}->{'Excel:maxcell'}}=
                &Spreadsheet::WriteExcel::Utility::xl_rowcol_to_cell
                ($maximum_data_row,$formula_data{$symb}->{'Excel:maxcol'});
            my $weight;
            my $max = 0;
            foreach my $resource (&get_resources($navmap,$seq)) {
                foreach my $part (@{$resource->parts}){
                    $weight = 1;
                    if ($chosen_output->{'scores'}) {
                        $weight = &Apache::lonnet::EXT
                            ('resource.'.$part.'.weight',$resource->symb,
                             undef,undef,undef);
                        if (!defined($weight) || ($weight eq '')) { 
                            $weight=1;
                        }
                    }
                    if ($chosen_output->{'scores'} &&
                        $chosen_output->{'every_problem'}) {
                        $excel_sheet->write($maximum_data_row,$cols_output++,
                                            $weight);
                    }
                    $max += $weight;
                }
            } 
            #
            if ($chosen_output->{'sequence_sum'} && 
                $chosen_output->{'every_problem'}) {
		my %replaceCells=
		    ('^'.$formula_data{$symb}->{'Excel:startcell'}.':'.
		         $formula_data{$symb}->{'Excel:endcell'}.'$' =>
		     &Spreadsheet::WriteExcel::Utility::xl_rowcol_to_cell($maximum_data_row,$formula_data{$symb}->{'Excel:startcol'}).':'.
		     &Spreadsheet::WriteExcel::Utility::xl_rowcol_to_cell($maximum_data_row,$formula_data{$symb}->{'Excel:endcol'}));
                $excel_sheet->repeat_formula($maximum_data_row,$cols_output++,
                                             $formula_data{$symb}->{'Excel:sum'},undef,
					     %replaceCells, %replaceCells);
			
            } elsif ($chosen_output->{'sequence_sum'}) {
                $excel_sheet->write($maximum_data_row,$cols_output++,$max);
            }
            if ($chosen_output->{'sequence_max'}) {
                $excel_sheet->write($maximum_data_row,$cols_output++,$max);
            }
            #
        }
        if ($chosen_output->{'grand_total'}) {
            $excel_sheet->repeat_formula($maximum_data_row,$cols_output++,
                                         $total_formula,undef,
                                         %total_cell_translation);
        }
        if ($chosen_output->{'grand_maximum'}) {
            $excel_sheet->repeat_formula($maximum_data_row,$cols_output++,
                                         $maximum_formula,undef,
                                         %maximum_cell_translation);
        }
    } # End of MAXIMUM row output  if ($chosen_output->{'maximum_row'}) {
    $rows_output = $first_data_row;
    ##############################################
    # Output summary table, which actually is above the sequence name row.
    ##############################################
    if ($chosen_output->{'summary_table'}) {
        $cols_output = 0;
        $excel_sheet->write($summary_header_row,$cols_output++,
                            &mt('Summary Table'),$format->{'bold'});
        if ($chosen_output->{'maximum_row'}) {
            $excel_sheet->write($summary_header_row,$cols_output++,
                                &mt('Maximum'),$format->{'bold'});
        }
        $excel_sheet->write($summary_header_row,$cols_output++,
                            &mt('Average'),$format->{'bold'});
        $excel_sheet->write($summary_header_row,$cols_output++,
                            &mt('Median'),$format->{'bold'});
        $excel_sheet->write($summary_header_row,$cols_output++,
                            &mt('Std Dev'),$format->{'bold'});
        my $row = $summary_header_row+1;
        foreach my $seq (@sequences) {
            my $symb = $seq->symb;
            $cols_output = 0;
            $excel_sheet->write($row,$cols_output++,
                                $seq->compTitle,
                                $format->{'bold'});
            if ($chosen_output->{'maximum_row'}) {
                $excel_sheet->write
                    ($row,$cols_output++,
                     '='.
                     &Spreadsheet::WriteExcel::Utility::xl_rowcol_to_cell
                     ($maximum_data_row,$formula_data{$symb}->{'Excel:scorecol'})
                     );
            }
            my $range = 
                &Spreadsheet::WriteExcel::Utility::xl_rowcol_to_cell
                ($first_data_row,$formula_data{$symb}->{'Excel:scorecol'}).
                ':'.
                &Spreadsheet::WriteExcel::Utility::xl_rowcol_to_cell
                ($first_data_row+$num_students-1,$formula_data{$symb}->{'Excel:scorecol'});
            $excel_sheet->write($row,$cols_output++,
                                '=AVERAGE('.$range.')');
            $excel_sheet->write($row,$cols_output++,
                                '=MEDIAN('.$range.')');
            $excel_sheet->write($row,$cols_output++,
                                '=STDEV('.$range.')');
            $row++;
        }
    }
    ##############################################
    #   Take care of non-excel initialization
    ##############################################
    #
    # Let the user know what we are doing
    my $studentcount = scalar(@Apache::lonstatistics::Students); 
    if ($env{'form.SelectedStudent'}) {
        $studentcount = '1';
    }
    $r->print('<p>'
             .&mt('Compiling Excel spreadsheet for [quant,_1,student]...',$studentcount)
            ."</p>\n"
    );
    $r->rflush();
    #
    # Initialize progress window
    %prog_state=&Apache::lonhtmlcommon::Create_PrgWin($r,$studentcount);
    #
    &Apache::lonhtmlcommon::Update_PrgWin($r,\%prog_state,
                                          'Processing first student');
    return;
}

sub excel_outputstudent {
    my ($r,$student) = @_;
    if ($request_aborted || ! defined($navmap) || ! defined($excel_sheet)) {
        return;
    }
    $cols_output=0;
    #
    # Write out student data
    my @to_show = &get_student_fields_to_show();
    foreach my $field (@to_show) {
        my $value = $student->{$field};
        if ($field eq 'comments') {
            $value = &Apache::lonmsgdisplay::retrieve_instructor_comments
                ($student->{'username'},$student->{'domain'});
        }
        $excel_sheet->write($rows_output,$cols_output++,$value);
    }
    #
    # Get student assessment data
    my %StudentsData;
    my @tmp = &Apache::loncoursedata::get_current_state($student->{'username'},
                                                        $student->{'domain'},
                                                        undef,
                                                   $env{'request.course.id'});
    if ((scalar @tmp > 0) && ($tmp[0] !~ /^error:/)) {
        %StudentsData = @tmp;
    }
    #
    # Write out sequence scores and totals data
    my %total_cell_translation;
    my %maximum_cell_translation;
    foreach my $seq (@sequences) {
        my $symb = $seq->symb;
        my $randompick = $seq->randompick();
        $cols_output = $formula_data{$symb}->{'Excel:startcol'};
        # Keep track of cells to translate in total cell
        $total_cell_translation{$formula_data{$symb}->{'Excel:scorecell'}} = 
            &Spreadsheet::WriteExcel::Utility::xl_rowcol_to_cell
                        ($rows_output,$formula_data{$symb}->{'Excel:scorecol'});
        # and maximum cell
        $maximum_cell_translation{$formula_data{$symb}->{'Excel:maxcell'}} = 
            &Spreadsheet::WriteExcel::Utility::xl_rowcol_to_cell
            ($rows_output,$formula_data{$symb}->{'Excel:maxcol'});
        #
        my ($performance,$performance_length,$score,$seq_max,$rawdata);
        if ($chosen_output->{'tries'} || $chosen_output->{'correct'}){
            ($performance,$performance_length,$score,$seq_max,$rawdata) =
                &student_tries_on_sequence($student,\%StudentsData,
                                           $navmap,$seq,'no',$randompick);
        } else {
            ($performance,$performance_length,$score,$seq_max,$rawdata) =
                &student_performance_on_sequence($student,\%StudentsData,
                                                 $navmap,$seq,'no',
                                                 $chosen_output->{ignore_weight},
                                                 $randompick);
        } 
        if ($chosen_output->{'every_problem'}) {
            if ($chosen_output->{'correct'}) {
                # only indiciate if each item is correct or not
                foreach my $value (@$rawdata) {
                    # positive means correct, 0 or negative means
                    # incorrect
                    $value = $value > 0 ? 1 : 0;
                    $excel_sheet->write($rows_output,$cols_output++,$value);
                }
            } else {
                foreach my $value (@$rawdata) {
                    if ($score eq ' ' || !defined($value)) {
                        $cols_output++;
                    } else {                        
                        $excel_sheet->write($rows_output,$cols_output++,
                                            $value);
                    }
                }
            }
        }
        if ($chosen_output->{'sequence_sum'} && 
            $chosen_output->{'every_problem'}) {
            # Write a formula for the sum of this sequence
            my %replaceCells=
		('^'.$formula_data{$symb}->{'Excel:startcell'}.':'.$formula_data{$symb}->{'Excel:endcell'}.'$'
		 => 
		 &Spreadsheet::WriteExcel::Utility::xl_rowcol_to_cell($rows_output,$formula_data{$symb}->{'Excel:startcol'}).':'.
		 &Spreadsheet::WriteExcel::Utility::xl_rowcol_to_cell($rows_output,$formula_data{$symb}->{'Excel:endcol'})
		 );
            # The undef is for the format	    
	    $excel_sheet->repeat_formula($rows_output,$cols_output++,
					 $formula_data{$symb}->{'Excel:sum'},undef,
					 %replaceCells, %replaceCells);
        } elsif ($chosen_output->{'sequence_sum'}) {
            if ($score eq ' ') {
                $cols_output++;
            } else {
                $excel_sheet->write($rows_output,$cols_output++,$score);
            }
        }
        if ($chosen_output->{'sequence_max'}) {
            $excel_sheet->write($rows_output,$cols_output++,$seq_max);
        }
    }
    #
    if ($chosen_output->{'grand_total'}) {
        $excel_sheet->repeat_formula($rows_output,$cols_output++,
                                     $total_formula,undef,
                                     %total_cell_translation);
    }
    if ($chosen_output->{'grand_maximum'}) {
        $excel_sheet->repeat_formula($rows_output,$cols_output++,
                                     $maximum_formula,undef,
                                     %maximum_cell_translation);
    }
    #
    # Bookkeeping
    $rows_output++; 
    $cols_output=0;
    #
    # Update the progress window
    &Apache::lonhtmlcommon::Increment_PrgWin($r,\%prog_state,'last student');
    return;
}

sub excel_finish {
    my ($r) = @_;
    if ($request_aborted || ! defined($navmap) || ! defined($excel_sheet)) {
	&excel_cleanup();
        return;
    }
    #
    # Write the excel file
    $excel_workbook->close();
    #
    # Close the progress window
    &Apache::lonhtmlcommon::Close_PrgWin($r,\%prog_state);
    #
    # Tell the user where to get their excel file
    $r->print('<br />'.
              '<a href="'.$filename.'">'.&mt('Your Excel spreadsheet').'</a>'."\n");
    $r->rflush();
    &excel_cleanup();
    return;
}

}
#######################################################
#######################################################

=pod

=head2 CSV output routines

=item &csv_initialize($r)

=item &csv_outputstudent($r,$student)

=item &csv_finish($r)

=cut

#######################################################
#######################################################
{

my $outputfile;
my $filename;
my $request_aborted;
my %prog_state; # progress window state
my $navmap;
my @sequences;

sub csv_cleanup {
    undef($outputfile);
    undef($filename);
    undef($request_aborted);
    undef(%prog_state);
    #
    undef($navmap);
    undef(@sequences);
}

sub csv_initialize{
    my ($r) = @_;

    &csv_cleanup();
    ($navmap,@sequences) = 
        &Apache::lonstatistics::selected_sequences_with_assessments();
    if (! ref($navmap)) {
        # Unable to get data, so bail out
        $r->print('<p class="LC_error">'.
                  &mt('Unable to retrieve course information.').
                  '</p>');
    }
    #
    # Deal with unimplemented requests
    $request_aborted = undef;
    if ($chosen_output->{'base'} =~ /final table/) {
        $r->print(
            '<h2>'.&mt('Unable to Complete Request').'</h2>'
           .'<p class="LC_warning">'
           .&mt('The [_1]Summary Table (Scores)[_2] option'
               .' is not available for non-HTML output.','<b>','</b>')
           .'</p>'
        );
        $request_aborted = 1;
    }
    return if ($request_aborted);
    #
    # Initialize progress window
    my $studentcount = scalar(@Apache::lonstatistics::Students);
    %prog_state=&Apache::lonhtmlcommon::Create_PrgWin($r,$studentcount);
    #
    # Open a file
    ($outputfile,$filename) = &Apache::loncommon::create_text_file($r,'csv');
    if (! defined($outputfile)) { return ''; }
    #
    # Datestamp
    my $description = $env{'course.'.$env{'request.course.id'}.'.description'};
    print $outputfile '"'.&Apache::loncommon::csv_translate($description).'",'.
        '"'.&Apache::loncommon::csv_translate(scalar(&Apache::lonlocal::locallocaltime(time))).'"'.
            "\n";
    print $outputfile '"'.
        &Apache::loncommon::csv_translate
        (&Apache::lonstatistics::section_and_enrollment_description()).
        '"'."\n";
    foreach my $item ('shortdesc','non_html_notes') {
        next if (! exists($chosen_output->{$item}));
        print $outputfile 
            '"'.&Apache::loncommon::csv_translate($chosen_output->{$item}).'"'.
            "\n";
    }
    #
    # Print out the headings
    my $sequence_row = '';
    my $resource_row = undef;
    foreach my $field (&get_student_fields_to_show()) {
        $sequence_row .='"",';
        $resource_row .= '"'.&Apache::loncommon::csv_translate($field).'",';
    }
    foreach my $seq (@sequences) {
        $sequence_row .= '"'.
            &Apache::loncommon::csv_translate($seq->compTitle).'",';
        my $count = 0;
        if ($chosen_output->{'every_problem'}) {
            foreach my $res (&get_resources($navmap,$seq)) {
                if (scalar(@{$res->parts}) < 1) {
                    next;
                }
                foreach my $part (@{$res->parts}) {
                    $resource_row .= '"'.
                        &Apache::loncommon::csv_translate
                        ($res->compTitle.', Part '.$res->part_display($part)).'",';
                    $count++;
                }
            }
        }
        $sequence_row.='"",'x$count;
        if ($chosen_output->{'sequence_sum'}) {
            if($chosen_output->{'correct'}) {
                $resource_row .= '"'.&mt('sum').'",';
            } else {
                $resource_row .= '"'.&mt('score').'",';
            }
        }
        if ($chosen_output->{'sequence_max'}) {
            $sequence_row.= '"",';
            $resource_row .= '"'.&mt('maximum possible').'",';
        }
    }
    if ($chosen_output->{'grand_total'}) {
        $sequence_row.= '"",';
        $resource_row.= '"'.&mt('Total').'",';
    } 
    if ($chosen_output->{'grand_maximum'}) {
        $sequence_row.= '"",';
        $resource_row.= '"'.&mt('Maximum').'",';
    } 
    chomp($sequence_row);
    chomp($resource_row);
    print $outputfile $sequence_row."\n";
    print $outputfile $resource_row."\n";
    return;
}

sub csv_outputstudent {
    my ($r,$student) = @_;
    if ($request_aborted || ! defined($navmap) || ! defined($outputfile)) {
        return;
    }
    my $Str = '';
    #
    # Output student fields
    my @to_show = &get_student_fields_to_show();
    foreach my $field (@to_show) {
        my $value = $student->{$field};
        if ($field eq 'comments') {
            $value = &Apache::lonmsgdisplay::retrieve_instructor_comments
                ($student->{'username'},$student->{'domain'});
        }        
        $Str .= '"'.&Apache::loncommon::csv_translate($value).'",';
    }
    #
    # Get student assessment data
    my %StudentsData;
    my @tmp = &Apache::loncoursedata::get_current_state($student->{'username'},
                                                        $student->{'domain'},
                                                        undef,
                                                   $env{'request.course.id'});
    if ((scalar @tmp > 0) && ($tmp[0] !~ /^error:/)) {
        %StudentsData = @tmp;
    }
    #
    # Output performance data
    my $total = 0;
    my $maximum = 0;
    foreach my $seq (@sequences) {
        my $randompick = $seq->randompick();
        my ($performance,$performance_length,$score,$seq_max,$rawdata);
        if ($chosen_output->{'tries'}){
            ($performance,$performance_length,$score,$seq_max,$rawdata) =
                &student_tries_on_sequence($student,\%StudentsData,
                                           $navmap,$seq,'no',$randompick);
        } else {
            ($performance,$performance_length,$score,$seq_max,$rawdata) =
                &student_performance_on_sequence($student,\%StudentsData,
                                                 $navmap,$seq,'no',
                                                 $chosen_output->{ignore_weight},
                                                 $randompick);
        }
        if ($chosen_output->{'every_problem'}) {
            if ($chosen_output->{'correct'}) {
                $score = 0;
                # Deal with number of parts correct data
                $Str .= '"'.join('","',( map { if ($_>0) { 
                                                   $score += 1;
                                                   1; 
                                               } else { 
                                                   0; 
                                               }
                                             } @$rawdata)).'",';
            } else {
                $Str .= '"'.join('","',(@$rawdata)).'",';
            }
        }
        if ($chosen_output->{'sequence_sum'}) {
            $Str .= '"'.$score.'",';
        } 
        if ($chosen_output->{'sequence_max'}) {
            $Str .= '"'.$seq_max.'",';
        }
        $total+=$score;
        $maximum += $seq_max;
    }
    if ($chosen_output->{'grand_total'}) {
        $Str .= '"'.$total.'",';
    }
    if ($chosen_output->{'grand_maximum'}) {
        $Str .= '"'.$maximum.'",';
    }
    chop($Str);
    $Str .= "\n";
    print $outputfile $Str;
    #
    # Update the progress window
    &Apache::lonhtmlcommon::Increment_PrgWin($r,\%prog_state,'last student');
    return;
}

sub csv_finish {
    my ($r) = @_;
    if ($request_aborted || ! defined($navmap) || ! defined($outputfile)) {
	&csv_cleanup();
        return;
    }
    close($outputfile);
    #
    # Close the progress window
    &Apache::lonhtmlcommon::Close_PrgWin($r,\%prog_state);
    #
    # Tell the user where to get their csv file
    $r->print('<br />'.
              '<a href="'.$filename.'">'.&mt('Your CSV file.').'</a>'."\n");
    $r->rflush();
    &csv_cleanup();
    return;
    
}

}

# This function will return an HTML string including a star, with
# a mouseover popup showing the "real" value. An optional second
# argument lets you show something other than a star.
sub show_star {
    my $popup = shift;
    my $symbol = shift || '*';
    # Escape the popup for JS.
    $popup =~ s/([^-a-zA-Z0-9:;,._ ()|!\/?=&*])/'\\' . sprintf("%lo", ord($1))/ge;
    
    return "<span class=\"LC_chrt_popup_exists\" onmouseover='popup_score(this, \"$popup\");return false;' onmouseout='popdown_score();return false;'>$symbol</span>";
}

#######################################################
#######################################################

=pod

=item &StudentTriesOnSequence()

Inputs:

=over 4

=item $student

=item $studentdata Hash ref to all student data

=item $seq Hash ref, the sequence we are working on

=item $links if defined we will output links to each resource.

=back

=cut

#######################################################
#######################################################
sub student_tries_on_sequence {
    my ($student,$studentdata,$navmap,$seq,$links,$randompick) = @_;
    $links = 'no' if (! defined($links));
    my $Str = '';
    my ($sum,$max) = (0,0);
    my $performance_length = 0;
    my @TriesData = ();
    my $tries;
    my $hasdata = 0; # flag - true if the student has any data on the sequence
    foreach my $resource (&get_resources($navmap,$seq)) {
        my $resource_data = $studentdata->{$resource->symb};
        my $value = '';
        foreach my $partnum (@{$resource->parts()}) {
            $tries = undef;
            $max++;
            $performance_length++;
            my $symbol = ' '; # default to space
            #
            my $awarded = 0;
            if (exists($resource_data->{'resource.'.$partnum.'.awarded'})) {
                $awarded = $resource_data->{'resource.'.$partnum.'.awarded'};
                $awarded = 0 if (! $awarded);
            }
            #
            my $status = '';
            if (exists($resource_data->{'resource.'.$partnum.'.solved'})) {
                $status = $resource_data->{'resource.'.$partnum.'.solved'};
            }
            #
            my $tries = 0;
            if(exists($resource_data->{'resource.'.$partnum.'.tries'})) {
                $tries = $resource_data->{'resource.'.$partnum.'.tries'};
                $hasdata =1;
            }
            #
            if ($awarded > 0) {
                # The student has gotten the problem correct to some degree
                if ($status eq 'excused') {
                    $symbol = 'x';
                    $max--;
                } elsif ($status eq 'correct_by_override' && !$resource->is_task()) {
                    $symbol = '+';
                    $sum++;
                } elsif ($tries > 0) {
                    if ($tries > 9) {
                        $symbol = show_star($tries);
                    } else {
                        $symbol = $tries;
                    }
                    $sum++;
                } else {
                    $symbol = '+';
                    $sum++;
                }
            } else {
                # The student has the problem incorrect or it is ungraded
                if ($status eq 'excused') {
                    $symbol = 'x';
                    $max--;
                } elsif ($status eq 'incorrect_by_override') {
                    $symbol = '-';
                } elsif ($status eq 'ungraded_attempted') {
                    $symbol = 'u';
                } elsif ($status eq 'incorrect_attempted' ||
                         $tries > 0)  {
                    $symbol = '.';
                } else {
                    # Problem is wrong and has not been attempted.
                    $symbol=' ';
                }
            }
            #
            if (! defined($tries)) {
                $tries = 0;
            }
            if ($status =~ /^(incorrect|ungraded)/) {
                # Bug 3390: show '-' for tries on incorrect problems 
                # (csv & excel only)
                push(@TriesData,-$tries);
            } else {
                push (@TriesData,$tries);
            }
            #
            if ( ($links eq 'yes' && $symbol ne ' ') ||
                 ($links eq 'all')) {
                my $link = '/adm/grades'.
                    '?symb='.&escape($resource->shown_symb).
                        '&amp;student='.$student->{'username'}.
                            '&amp;userdom='.$student->{'domain'}.
                                '&amp;command=submission';
                $symbol = &link($symbol, $link);
            }
            $value .= $symbol;
        }
        $Str .= $value;
    }
    if ($randompick) {
        $max = $randompick;
    }
    if (! $hasdata && $sum == 0) {
        $sum = ' ';
    }
    return ($Str,$performance_length,$sum,$max,\@TriesData);
}

=pod

=item &link

Inputs:

=over 4

=item $text

=item $target

=back

Takes the text and creates a link to the $text that honors
the value of 'new window' if clicked on, but uses a real 
'href' so middle and right clicks still work.

$target and $text are assumed to be already correctly escaped; i.e., it
can be dumped out directly into the output stream as-is.

=cut

sub link {
    my ($text,$target) = @_;
    return 
        "<a href='$target' onclick=\"t=this.href;if(new_window)"
        ."{window.open(t)}else{return void(window."
        ."location=t)};return false;\">$text</a>";
}

#######################################################
#######################################################

=pod

=item &student_performance_on_sequence

Inputs:

=over 4

=item $student

=item $studentdata Hash ref to all student data

=item $seq Hash ref, the sequence we are working on

=item $links if defined we will output links to each resource.

=back

=cut

#######################################################
#######################################################
sub student_performance_on_sequence {
    my ($student,$studentdata,$navmap,$seq,$links,$awarded_only,$randompick) = @_;
    $links = 'no' if (! defined($links));
    my $Str = ''; # final result string
    my ($score,$max) = (0,0);
    my $performance_length = 0;
    my $symbol;
    my @ScoreData = ();
    my $partscore;
    my $hasdata = 0; # flag, 0 if there were no submissions on the sequence
    my %ptsfreq;
    foreach my $resource (&get_resources($navmap,$seq)) {
        my $symb = $resource->symb;
        my $resource_data = $studentdata->{$symb};
        my $resmax = 0;
        foreach my $part (@{$resource->parts()}) {
            $partscore = undef;
            my $weight;
            if (!$awarded_only){
                $weight = &Apache::lonnet::EXT('resource.'.$part.'.weight',
                                               $symb,
                                               $student->{'domain'},
                                               $student->{'username'},
                                               $student->{'section'});
            }
            if (!defined($weight) || ($weight eq '')) { 
                $weight=1;
            }
            #
            $max += $weight; # see the 'excused' branch below...
            $resmax += $weight;
            $performance_length++; # one character per part
            $symbol = ' '; # default to space
            #
            my $awarded;
            if (exists($resource_data->{'resource.'.$part.'.awarded'})) {
                $awarded = $resource_data->{'resource.'.$part.'.awarded'};
                $awarded = 0 if (! $awarded);
                $hasdata = 1;
            }
            #
            $partscore = &Apache::grades::compute_points($weight,$awarded);
            if (! defined($awarded)) {
                $partscore = undef;
            }
            $score += $partscore;
            $symbol = $partscore; 
            if (abs($symbol - sprintf("%.0f",$symbol)) < 0.001) {
                $symbol = sprintf("%.0f",$symbol);
            }
            if (length($symbol) > 1) {
                $symbol = show_star($symbol);
            }
            if (exists($resource_data->{'resource.'.$part.'.solved'}) &&
                $resource_data->{'resource.'.$part.'.solved'} ne '') {
                my $status = $resource_data->{'resource.'.$part.'.solved'};
                if ($status eq 'excused') {
                    $symbol = 'x';
                    $max -= $weight; # Do not count 'excused' problems.
                } elsif ($status eq 'ungraded_attempted') {
                    $symbol = 'u';
                }
                $hasdata = 1;
            } elsif ($resource_data->{'resource.'.$part.'.award'} eq 'DRAFT') {
                $symbol = 'd';
                $hasdata = 1;
            } elsif (!exists($resource_data->{'resource.'.$part.'.awarded'})){
                # Unsolved.  Did they try?
                if (exists($resource_data->{'resource.'.$part.'.tries'})){
                    $symbol = '.';
                    $hasdata = 1;
                } else {
                    $symbol = ' ';
                }
            }
            #
            if (! defined($partscore)) {
                $partscore = $symbol;
            }
            push (@ScoreData,$partscore);
            #
            if ( ($links eq 'yes' && $symbol ne ' ') || ($links eq 'all')) {
                my $link = '/adm/grades' .
                    '?symb='.&escape($resource->shown_symb).
                    '&amp;student='.$student->{'username'}.
                    '&amp;userdom='.$student->{'domain'}.
                    '&amp;command=submission';
                $symbol = &link($symbol, $link);
            }
            $Str .= $symbol;
        }
        if ($ptsfreq{$resmax}) {
            $ptsfreq{$resmax} ++;
        } else {
            $ptsfreq{$resmax} = 1;
        }
    }
    if ($randompick) {
        my @uniquetotals = keys(%ptsfreq);
        if ((@uniquetotals == 1) && ($ptsfreq{$uniquetotals[0]} > 0)) {
            $max = $max * $randompick/$ptsfreq{$uniquetotals[0]};
        }
    }
    if (! $hasdata && $score == 0) {
        $score = ' ';
    }
    return ($Str,$performance_length,$score,$max,\@ScoreData);
}

#######################################################
#######################################################

=pod

=item &CreateLegend()

This function returns a formatted string containing the legend for the
chart.  The legend describes the symbols used to represent grades for
problems.

=cut

#######################################################
#######################################################
sub CreateLegend {
    my $Str = "<p><pre>".
              " digit score or number of tries to get correct ".
              "   *  correct by student in more than 9 tries\n".
	      "   +  correct by hand grading or override\n".
              "   -  incorrect by override\n".
	      "   .  incorrect attempted\n".
	      "   u  ungraded attempted\n".
              "   d  draft answer saved but not submitted\n".
              "      not attempted (blank field)\n".
	      "   x  excused".
              "</pre><p>";
    return $Str;
}

#######################################################
#######################################################

=pod 

=back

=cut

#######################################################
#######################################################

1;

__END__
