# The LearningOnline Network
# User Authentication Module
#
# $Id: lonauth.pm,v 1.121.2.27 2024/10/09 17:56:24 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#

package Apache::lonauth;

use strict;
use LONCAPA qw(:DEFAULT :match);
use Apache::Constants qw(:common);
use CGI qw(:standard);
use Apache::loncommon();
use Apache::lonnet;
use Apache::lonmenu();
use Apache::createaccount;
use Fcntl qw(:flock);
use Apache::lonlocal;
use Apache::File();
use HTML::Entities;
use Digest::MD5;
use CGI::Cookie();
 
# ------------------------------------------------------------ Successful login
sub success {
    my ($r, $username, $domain, $authhost, $lowerurl, $extra_env,
	$form,$cid,$expirepub) = @_;

# ------------------------------------------------------------ Get cookie ready
    my $cookie =
	&Apache::loncommon::init_user_environment($r, $username, $domain,
						  $authhost, $form,
						  {'extra_env' => $extra_env,});

    my $public=($username eq 'public' && $domain eq 'public');

    if ($public or $lowerurl eq 'noredirect') { return $cookie; }

# -------------------------------------------------------------------- Log this

    my $ip = &Apache::lonnet::get_requestor_ip();
    &Apache::lonnet::log($domain,$username,$authhost,
                         "Login $ip");

# ------------------------------------------------- Check for critical messages

    my @what=&Apache::lonnet::dump('critical',$domain,$username);
    if ($what[0]) {
	if (($what[0] ne 'con_lost') && ($what[0]!~/^error\:/)) {
	    $lowerurl='/adm/email?critical=display';
        }
    }

# ------------------------------------------------------------ Get cookies ready
    my ($securecookie,$defaultcookie);
    my $ssl = $r->subprocess_env('https');
    if ($ssl) {
        $securecookie="lonSID=$cookie; path=/; HttpOnly; secure";
        my $lonidsdir=$r->dir_config('lonIDsDir');
        if (($lonidsdir) && (-e "$lonidsdir/$cookie.id")) {
            my $linkname=substr(Digest::MD5::md5_hex(Digest::MD5::md5_hex(time(). {}. rand(). $$)), 0, 32).'_linked';
            if (-e "$lonidsdir/$linkname.id") {
                unlink("$lonidsdir/$linkname.id");
            }
            my $made_symlink = eval { symlink("$lonidsdir/$cookie.id",
                                              "$lonidsdir/$linkname.id"); 1 };
            if ($made_symlink) {
                $defaultcookie = "lonLinkID=$linkname; path=/; HttpOnly;";
                &Apache::lonnet::appenv({'user.linkedenv' => $linkname});
            }
        }
    } else {
        $defaultcookie = "lonID=$cookie; path=/; HttpOnly;";
    }
# -------------------------------------------------------- Menu script and info
    my $destination = $lowerurl;

    if (defined($form->{role})) {
        my $envkey = 'user.role.'.$form->{role};
        my $now=time;
        my $then=$env{'user.login.time'};
        my $refresh=$env{'user.refresh.time'};
        my $update=$env{'user.update.time'};
        if (!$update) {
            $update = $then;
        }
        if (exists($env{$envkey})) {
            my ($role,$where,$trolecode,$tstart,$tend,$tremark,$tstatus);
            &Apache::lonnet::role_status($envkey,$update,$refresh,$now,\$role,\$where,
                                         \$trolecode,\$tstatus,\$tstart,\$tend);
            if ($tstatus eq 'is') {
                $destination  .= ($destination =~ /\?/) ? '&' : '?';
                my $newrole = &HTML::Entities::encode($form->{role},'"<>&');
                $destination .= 'selectrole=1&'.$newrole.'=1';
            }
        }
    }
    if (defined($form->{symb})) {
        my $destsymb = $form->{symb};
        my $encrypted;
        if ($destsymb =~ m{^/enc/}) {
            $encrypted = 1;
            if ($cid) {
                $destsymb = &Apache::lonenc::unencrypted($destsymb,$cid);
            }
        }
        $destination  .= ($destination =~ /\?/) ? '&' : '?';
        if ($destsymb =~ /___/) {
            my ($map,$resid,$desturl)=split(/___/,$destsymb);
            $desturl = &Apache::lonnet::clutter($desturl);
            if ($encrypted) {
                $desturl = &Apache::lonenc::encrypted($desturl,1,$cid);
                $destsymb = $form->{symb};
            }
            $desturl = &HTML::Entities::encode($desturl,'"<>&');
            $destsymb = &HTML::Entities::encode($destsymb,'"<>&');
            $destination .= 'destinationurl='.$desturl.
                            '&destsymb='.$destsymb;
        } elsif (!$encrypted) {
            $destsymb = &HTML::Entities::encode($destsymb,'"<>&');
            $destination .= 'destinationurl='.$destsymb;
        }
    }
    if ($destination =~ m{^/adm/roles}) {
        $destination  .= ($destination =~ /\?/) ? '&' : '?';
        $destination .= 'source=login';
    }

    my $windowinfo=&Apache::lonmenu::open($env{'browser.os'});
    if ($env{'environment.remote'} eq 'on') {
        my $checkexempt;
        if ($env{'user.loadbalexempt'} eq $r->dir_config('lonHostID')) {
            if ($env{'user.loadbalcheck.time'} + 600 > time) {
                $checkexempt = 1;
            }
        }
        if ($env{'user.noloadbalance'} eq $r->dir_config('lonHostID')) {
            $checkexempt = 1;
        }
        unless (($checkexempt) || 
                (($destination =~ m{^/adm/switchserver}) && (!$r->is_initial_req()))) {
            my ($is_balancer,$otherserver) =
                &Apache::lonnet::check_loadbalancing($env{'user.name'},
                                                     $env{'user.domain'});
            if (($is_balancer) && ($otherserver ne '') && 
                ($otherserver ne $r->dir_config('lonHostID'))) {
                $env{'environment.remote'} = 'off';
            }
        }
    }
    my $startupremote=&Apache::lonmenu::startupremote($destination);
    my $remoteinfo=&Apache::lonmenu::load_remote_msg($lowerurl);
    my $setflags=&Apache::lonmenu::setflags();
    my $maincall=&Apache::lonmenu::maincall();
    my $brcrum = [{'href' => '',
                   'text' => 'Successful Login'},];
    my $start_page=&Apache::loncommon::start_page('Successful Login',
                                                  $startupremote,
                                                  {'no_inline_link' => 1,
                                                   'bread_crumbs' => $brcrum,});
    my $end_page  =&Apache::loncommon::end_page();

    my $continuelink;
    if ($env{'environment.remote'} eq 'off') {
	$continuelink='<a href="'.$destination.'">'.&mt('Continue').'</a>';
    }
# ------------------------------------------------- Output for successful login

    &Apache::loncommon::content_type($r,'text/html');
    if ($securecookie) {
        $r->headers_out->add('Set-cookie' => $securecookie);
    }
    if ($defaultcookie) {
        $r->headers_out->add('Set-cookie' => $defaultcookie);
    }
    if ($expirepub) {
        my $c = new CGI::Cookie(-name    => 'lonPubID',
                                -value   => '',
                                -expires => '-10y',);
        $r->headers_out->add('Set-cookie' => $c);
    }
    $r->send_http_header;

    my %lt=&Apache::lonlocal::texthash(
				       'wel' => 'Welcome',
				       'pro' => 'Login problems?',
				       );
    my $loginhelp = &loginhelpdisplay($domain);
    if ($loginhelp) {
        $loginhelp = '<p><a href="'.$loginhelp.'">'.$lt{'pro'}.'</a></p>';
    }

    my $welcome = &mt('Welcome to the Learning[_1]Online[_2] Network with CAPA. Please wait while your session is being set up.','<i>','</i>'); 
    $r->print(<<ENDSUCCESS);
$start_page
$setflags
$windowinfo
<h1>$lt{'wel'}</h1>
$welcome
$loginhelp
$remoteinfo
$maincall
$continuelink
$end_page
ENDSUCCESS
    return;
}

# --------------------------------------------------------------- Failed login!

sub failed {
    my ($r,$message,$form,$authhost) = @_;
    (undef,undef,undef,my $clientmathml,my $clientunicode) =
        &Apache::loncommon::decode_user_agent();
    my $args = {};
    if ($clientunicode && !$clientmathml) {
        $args = {'browser.unicode' => 1};
    }

    my $start_page = &Apache::loncommon::start_page('Unsuccessful Login',undef,$args);
    my $uname = &Apache::loncommon::cleanup_html($form->{'uname'});
    my $udom = &Apache::loncommon::cleanup_html($form->{'udom'});
    if (&Apache::lonnet::domain($udom,'description') eq '') {
        undef($udom);
    }
    my $retry = '/adm/login';
    if ($uname eq $form->{'uname'}) {
        $retry .= '?username='.$uname;
    }
    if ($udom) {
        $retry .= (($retry=~/\?/)?'&amp;':'?').'domain='.$udom;
    }
    my $lonhost = $r->dir_config('lonHostID');
    my $querystr;
    my $result = &set_retry_token($form,$lonhost,\$querystr);
    if ($result eq 'fail') {
        if (exists($form->{role})) {
            my $role = &Apache::loncommon::cleanup_html($form->{role});
            if ($role ne '') {
                $retry .= (($retry=~/\?/)?'&amp;':'?').'role='.$role;
            }
        }
        if (exists($form->{symb})) {
            my $symb = &Apache::loncommon::cleanup_html($form->{symb});
            if ($symb ne '') {
                $retry .= (($retry=~/\?/)?'&amp;':'?').'symb='.$symb;
            }
        }
        if (exists($form->{firsturl})) {
            my $firsturl = &Apache::loncommon::cleanup_html($form->{firsturl});
            if ($firsturl ne '') {
                $retry .= (($retry=~/\?/)?'&amp;':'?').'firsturl='.$firsturl;
            }
        }
    } elsif ($querystr ne '') {
        $retry .= (($retry=~/\?/)?'&amp;':'?').$querystr;
    }
    my $end_page = &Apache::loncommon::end_page();
    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;
    my @actions =
         (&mt('Please [_1]log in again[_2].','<a href="'.$retry.'">','</a>'));
    my $loginhelp = &loginhelpdisplay($udom);
    if ($loginhelp) {
        push(@actions, '<a href="'.$loginhelp.'">'.&mt('Login problems?').'</a>');
    }
    #FIXME: link to helpdesk might be added here

    $r->print(
       $start_page
      .'<h2>'.&mt('Sorry ...').'</h2>'
      .&Apache::lonhtmlcommon::confirm_success(&mt($message),1).'<br /><br />'
      .&Apache::lonhtmlcommon::actionbox(\@actions)
      .$end_page
    );
 }

# ------------------------------------------------------------------ Rerouting!

sub reroute {
    my ($r) = @_;
    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;
    my $msg='<b>'.&mt('Sorry ...').'</b><br />'
           .&mt('Please [_1]log in again[_2].');
    &Apache::loncommon::simple_error_page($r,'Rerouting',$msg,{'no_auto_mt_msg' => 1});
}

# ---------------------------------------------------------------- Main handler

sub handler {
    my $r = shift;
    my $londocroot = $r->dir_config('lonDocRoot');
# Are we re-routing?
    if (-e "$londocroot/lon-status/reroute.txt") {
	&reroute($r);
	return OK;
    }

    &Apache::lonlocal::get_language_handle($r);

# -------------------------------- Prevent users from attempting to login twice
    my $handle = &Apache::lonnet::check_for_valid_session($r);
    if ($handle ne '') {
        my $lonidsdir=$r->dir_config('lonIDsDir');
        if ($handle=~/^publicuser\_/) {
# For "public user" - remove it, we apparently really want to login
            unlink($r->dir_config('lonIDsDir')."/$handle.id");
        } else {
# Indeed, a valid token is found
            &Apache::lonnet::transfer_profile_to_env($lonidsdir,$handle);
	    &Apache::loncommon::content_type($r,'text/html');
	    $r->send_http_header;
	    my $start_page = 
	        &Apache::loncommon::start_page('Already logged in');
	    my $end_page = 
	        &Apache::loncommon::end_page();
            my $dest = '/adm/roles';
            my %form = &get_form_items($r);
            if ($form{'logtoken'}) {
                my $tmpinfo = &Apache::lonnet::reply('tmpget:'.$form{'logtoken'},
                                                     $form{'serverid'});
                unless (($tmpinfo=~/^error/) || ($tmpinfo eq 'con_lost') ||
                        ($tmpinfo eq 'no_such_host')) {
                    my ($des_key,$firsturl,@rest)=split(/&/,$tmpinfo);
                    $firsturl = &unescape($firsturl);
                    my %info;
                    foreach my $item (@rest) {
                        my ($key,$value) = split(/=/,$item);
                        $info{$key} = &unescape($value);
                    }
                    if ($firsturl ne '') {
                        $info{'firsturl'} = $firsturl;
                        $dest = $firsturl;
                    }
                }
            }
            $r->print(
               $start_page
              .'<p class="LC_warning">'.&mt('You are already logged in!').'</p>'
              .'<p>'.&mt('Please either [_1]continue the current session[_2] or [_3]log out[_4].'
                    ,'<a href="'.$dest.'">','</a>','<a href="/adm/logout">','</a>')
              .'</p>'
              .$end_page
            );
            return OK;
        }
    }

# ---------------------------------------------------- No valid token, continue

    my %form = &get_form_items($r);
    if ((!$form{'uname'}) || (!$form{'upass0'}) || (!$form{'udom'})) {
	&failed($r,'Username, password and domain need to be specified.',
		\%form);
        return OK;
    }

# split user logging in and "su"-user

    ($form{'uname'},$form{'suname'},$form{'sudom'})=split(/\:/,$form{'uname'});
    $form{'uname'} = &LONCAPA::clean_username($form{'uname'});
    $form{'suname'}= &LONCAPA::clean_username($form{'suname'});
    $form{'udom'}  = &LONCAPA::clean_domain($form{'udom'});
    $form{'sudom'} = &LONCAPA::clean_domain($form{'sudom'});

    my $role   = $r->dir_config('lonRole');
    my $domain = $r->dir_config('lonDefDomain');
    my $prodir = $r->dir_config('lonUsersDir');
    my $contact_name = &mt('LON-CAPA helpdesk');

# ---------------------------------------- Get the information from login token

    my $tmpinfo=Apache::lonnet::reply('tmpget:'.$form{'logtoken'},
                                      $form{'serverid'});

    if (($tmpinfo=~/^error/) || ($tmpinfo eq 'con_lost') || 
        ($tmpinfo eq 'no_such_host')) {
	&failed($r,'Information needed to verify your login information is missing, inaccessible or expired.',\%form);
        return OK;
    } else {
	my $reply = &Apache::lonnet::reply('tmpdel:'.$form{'logtoken'},
					   $form{'serverid'});
        if ( $reply ne 'ok' ) {
            &failed($r,'Session could not be opened.',\%form);
	    &Apache::lonnet::logthis("ERROR got a reply of $reply when trying to contact ". $form{'serverid'}." to get login token");
	    return OK;
	}
    }

    if (!&Apache::lonnet::domain($form{'udom'})) {
        &failed($r,'The domain you provided is not a valid LON-CAPA domain.',\%form);
        return OK;
    }

    my ($des_key,$firsturl,@rest)=split(/&/,$tmpinfo);
    $firsturl = &unescape($firsturl);
    foreach my $item (@rest) {
        my ($key,$value) = split(/=/,$item);
        $form{$key} = &unescape($value);
    }
    my $upass = &Apache::loncommon::des_decrypt($des_key,$form{'upass0'});

# ---------------------------------------------------------------- Authenticate

    my %domconfig = &Apache::lonnet::get_dom('configuration',['usercreation'],$form{'udom'});
    my ($cancreate,$statustocreate) =
        &Apache::createaccount::get_creation_controls($form{'udom'},$domconfig{'usercreation'});
    my $defaultauth;
    if (ref($cancreate) eq 'ARRAY') {
        if (grep(/^login$/,@{$cancreate})) {
            $defaultauth = 1;
        }
    }
    my $clientcancheckhost = 1;
    my $authhost=Apache::lonnet::authenticate($form{'uname'},$upass,
                                              $form{'udom'},$defaultauth,
                                              $clientcancheckhost);
    
# --------------------------------------------------------------------- Failed?

    if ($authhost eq 'no_host') {
        my $pwdverify;
        if (&Apache::lonnet::homeserver($form{'uname'},$form{'udom'}) eq 'no_host') {
            my %possunames = &alternate_unames_check($form{'uname'},$form{'udom'});
            if (keys(%possunames) > 0) {
                foreach my $rulematch (keys(%possunames)) {
                    my $possuname = $possunames{$rulematch};
                    if (($possuname ne '') && ($possuname =~ /^$match_username$/)) {
                        $authhost=Apache::lonnet::authenticate($possuname,$upass,
                                                               $form{'udom'},undef,
                                                               $clientcancheckhost);
                        if (($authhost eq 'no_host') || ($authhost eq 'no_account_on_host')) {
                            next;
                        } elsif (($authhost ne '') && (&Apache::lonnet::hostname($authhost) ne '')) {
                            $pwdverify = 1;
                            &Apache::lonnet::logthis("Authenticated user: $possuname was submitted as: $form{'uname'}");
                            $form{'uname'} = $possuname;
                            last;
                        }
                    }
                }
            }
        }
        unless ($pwdverify) {
            &failed($r,'Username and/or password could not be authenticated.',
                    \%form);
            return OK;
        }
    } elsif ($authhost eq 'no_account_on_host') {
        if ($defaultauth) {
            my $domdesc = &Apache::lonnet::domain($form{'udom'},'description');
            unless (&check_can_host($r,\%form,'no_account_on_host',$domdesc)) {
                return OK;
            }
            my $start_page = 
                &Apache::loncommon::start_page('Create a user account in LON-CAPA',
                                               '',{'no_inline_link'   => 1,});
            my $lonhost = $r->dir_config('lonHostID');
            my $origmail = $Apache::lonnet::perlvar{'lonSupportEMail'};
            my $contacts = 
                &Apache::loncommon::build_recipient_list(undef,'helpdeskmail',
                                                        $form{'udom'},$origmail);
            my ($contact_email) = split(',',$contacts); 
            my $output = 
                &Apache::createaccount::username_check($form{'uname'},$form{'udom'},
                                                       $domdesc,'',$lonhost,
                                                       $contact_email,$contact_name,
                                                       undef,$statustocreate);
            &Apache::loncommon::content_type($r,'text/html');
            $r->send_http_header;
            &Apache::createaccount::print_header($r,$start_page);
            $r->print('<h3>'.&mt('Account creation').'</h3>'.
                      &mt('Although your username and password were authenticated, you do not currently have a LON-CAPA account at this institution.').'<br />'.
                      $output.&Apache::loncommon::end_page());
            return OK;
        } else {
            &failed($r,'Although your username and password were authenticated, you do not currently have a LON-CAPA account in this domain, and you are not permitted to create one.',\%form);
            return OK;
        }
    }

    if (($firsturl eq '') || 
	($firsturl=~/^\/adm\/(logout|remote)/)) {
	$firsturl='/adm/roles';
    }

    my ($hosthere,%sessiondata);
    if ($form{'iptoken'}) {
        %sessiondata = &Apache::lonnet::tmpget($form{'iptoken'});
        my $delete = &Apache::lonnet::tmpdel($form{'iptoken'});
        if (($sessiondata{'domain'} eq $form{'udom'}) &&
            ($sessiondata{'username'} eq $form{'uname'})) {
            $hosthere = 1;
        }
    }

# --------------------------------- Are we attempting to login as somebody else?
    if ($form{'suname'}) {
        my ($suname,$sudom,$sudomref);
        $suname = $form{'suname'};
        $sudom = $form{'udom'};
        if ($form{'sudom'}) {
            unless ($sudom eq $form{'sudom'}) {
                if (&Apache::lonnet::domain($form{'sudom'})) {
                    $sudomref = [$form{'sudom'}];
                    $sudom = $form{'sudom'};
                }
            }
        }
# ------------ see if the original user has enough privileges to pull this stunt
	if (&Apache::lonnet::privileged($form{'uname'},$form{'udom'},$sudomref)) {
# ---------------------------------------------------- see if the su-user exists
	    unless (&Apache::lonnet::homeserver($suname,$sudom) eq 'no_host') {
# ------------------------------ see if the su-user is not too highly privileged
		if (&Apache::lonnet::privileged($suname,$sudom)) {
                    &Apache::lonnet::logthis('Attempted switch user to privileged user');
                } else {
                    my $noprivswitch;
#
# su-user's home server and user's home server must have one of:
# (a) same domain
# (b) same primary library server for the two domains
# (c) same "internet domain" for primary library server(s) for home servers' domains
#
                    my $suprim = &Apache::lonnet::domain($sudom,'primary');
                    my $suintdom = &Apache::lonnet::internet_dom($suprim);
                    unless ($sudom eq $form{'udom'}) {
                        my $uprim = &Apache::lonnet::domain($form{'udom'},'primary');
                        my $uintdom = &Apache::lonnet::internet_dom($uprim);
                        unless ($suprim eq $uprim) {
                            unless ($suintdom eq $uintdom) {
                                &Apache::lonnet::logthis('Attempted switch user '
                                   .'to user with different "internet domain".');
                                $noprivswitch = 1;
                            }
                        }
                    }

                    unless ($noprivswitch) {
#
# server where log-in occurs must have same "internet domain" as su-user's home
# server
#
                        my $lonhost = $r->dir_config('lonHostID');
                        my $hostintdom = &Apache::lonnet::internet_dom($lonhost);
                        if ($hostintdom ne $suintdom) {
                            &Apache::lonnet::logthis('Attempted switch user on a '
                                .'server with a different "internet domain".');
                        } else {

# -------------------------------------------------------- actually switch users

			    &Apache::lonnet::logperm('User '.$form{'uname'}.' at '.
				$form{'udom'}.' logging in as '.$suname.':'.$sudom);
			    $form{'uname'}=$suname;
                            if ($form{'udom'} ne $sudom) {
                                $form{'udom'}=$sudom;
                            }
                        }
                    }
		}
	    }
	} else {
	    &Apache::lonnet::logthis('Non-privileged user attempting switch user');
	}
    }

    my ($is_balancer,$otherserver);

    unless ($hosthere) {
        ($is_balancer,$otherserver) =
            &Apache::lonnet::check_loadbalancing($form{'uname'},$form{'udom'},'login');
        if ($is_balancer) {
            # Check if browser sent a LON-CAPA load balancer cookie (and this is a balancer)
            my ($found_server,$balancer_cookie) = &Apache::lonnet::check_for_balancer_cookie($r);
            if (($found_server) && ($balancer_cookie =~ /^\Q$env{'user.domain'}\E_\Q$env{'user.name'}\E_/)) {
                $otherserver = $found_server;
            }
            if ($otherserver eq '') {
                my $lowest_load;
                ($otherserver,undef,undef,undef,$lowest_load) = &Apache::lonnet::choose_server($form{'udom'});
                if ($lowest_load > 100) {
                    $otherserver = &Apache::lonnet::spareserver($r,$lowest_load,$lowest_load,1,$form{'udom'});
                }
            }
            if ($otherserver ne '') {
                my @hosts = &Apache::lonnet::current_machine_ids();
                if (grep(/^\Q$otherserver\E$/,@hosts)) {
                    $hosthere = $otherserver;
                }
            }
        }
    }

    if (($is_balancer) && (!$hosthere)) {
        if ($otherserver) {
            &success($r,$form{'uname'},$form{'udom'},$authhost,'noredirect',undef,
                     \%form);
            my $switchto = '/adm/switchserver?otherserver='.$otherserver;
            if (($firsturl) && ($firsturl ne '/adm/switchserver') && ($firsturl ne '/adm/roles')) {
                $switchto .= '&origurl='.$firsturl;
            }
            if ($form{'role'}) {
                $switchto .= '&role='.$form{'role'};
            }
            if ($form{'symb'}) {
                $switchto .= '&symb='.$form{'symb'};
            }
            $r->internal_redirect($switchto);
        } else {
            &Apache::loncommon::content_type($r,'text/html');
            $r->send_http_header;
            $r->print(&noswitch());
        }
        return OK;
    } else {
        if (!&check_can_host($r,\%form,$authhost)) {
            my ($otherserver) = &Apache::lonnet::choose_server($form{'udom'});
            if ($otherserver) {
                &success($r,$form{'uname'},$form{'udom'},$authhost,'noredirect',undef,
                         \%form);
                my $switchto = '/adm/switchserver?otherserver='.$otherserver;
                if (($firsturl) && ($firsturl ne '/adm/switchserver') && ($firsturl ne '/adm/roles')) {
                    $switchto .= '&origurl='.$firsturl;
                }
                if ($form{'role'}) {
                    $switchto .= '&role='.$form{'role'};
                }
                if ($form{'symb'}) {
                    $switchto .= '&symb='.$form{'symb'};
                }
                $r->internal_redirect($switchto);
            } else {
                &Apache::loncommon::content_type($r,'text/html');
                $r->send_http_header;
                $r->print(&noswitch());
            }
            return OK;
        }

# ------------------------------------------------------- Do the load balancing

# ---------------------------------------------------------- Determine own load
        my $loadlim = $r->dir_config('lonLoadLim');
        my $loadavg;
        {
            my $loadfile=Apache::File->new('/proc/loadavg');
            $loadavg=<$loadfile>;
        }
        $loadavg =~ s/\s.*//g;
        my $loadpercent=sprintf("%.1f",100*$loadavg/$loadlim);
        my $userloadpercent=&Apache::lonnet::userload();

# ---------------------------------------------------------- Are we overloaded?
        if ((($userloadpercent>100.0)||($loadpercent>100.0))) {
            my $unloaded=Apache::lonnet::spareserver($r,$loadpercent,$userloadpercent,1,$form{'udom'});
            if (!$unloaded) {
                ($unloaded) = &Apache::lonnet::choose_server($form{'udom'});
            }
            if ($unloaded) {
                &success($r,$form{'uname'},$form{'udom'},$authhost,'noredirect',
                         undef,\%form);
                $r->internal_redirect('/adm/switchserver?otherserver='.$unloaded.'&origurl='.$firsturl);
                return OK;
            }
        }
        if (($is_balancer) && ($hosthere)) {
            $form{'noloadbalance'} = $hosthere;
        }
        my $extra_env;
        if (($hosthere) && ($sessiondata{'sessionserver'} ne '')) {
            if ($sessiondata{'origurl'} ne '') {
                $firsturl = $sessiondata{'origurl'};
                $form{'firsturl'} = $sessiondata{'origurl'};
                my @names = ('role','symb','linkprot','linkkey');
                foreach my $item (@names) {
                    if ($sessiondata{$item} ne '') {
                        $form{$item} = $sessiondata{$item};
                    }
                }
            }
        }
        &success($r,$form{'uname'},$form{'udom'},$authhost,$firsturl,undef,
                 \%form);
        return OK;
    }
}

sub get_form_items {
    my ($r) = @_;
    my $buffer;
    if ($r->header_in('Content-length') > 0) {
        $r->read($buffer,$r->header_in('Content-length'),0);
    }
    my %form;
    foreach my $pair (split(/&/,$buffer)) {
       my ($name,$value) = split(/=/,$pair);
       $value =~ tr/+/ /;
       $value =~ s/%([a-fA-F0-9][a-fA-F0-9])/pack("C",hex($1))/eg;
       $form{$name}=$value;
    }
    return %form;
}

sub set_retry_token {
    my ($form,$lonhost,$querystr) = @_;
    if (ref($form) eq 'HASH') {
        my ($firsturl,$token,$extras,@names);
        @names = ('role','symb','iptoken');
        foreach my $name (@names) {
            if ($form->{$name} ne '') {
                $extras .= '&'.$name.'='.&escape($form->{$name});
                last if ($name eq 'linkprot');
            }
        }
        my $firsturl = $form->{'firsturl'};
        if (($firsturl ne '') || ($extras ne '')) {
            $extras .= ':retry';
            $token = &Apache::lonnet::reply('tmpput:'.&escape($firsturl).
                                            $extras,$lonhost);
            if (($token eq 'con_lost') || ($token eq 'no_such_host')) {
                return 'fail';
            } else {
                if (ref($querystr)) {
                    $$querystr = 'retry='.$token;
                }
                return 'ok';
            }
        }
    }
    return;
}

sub check_can_host {
    my ($r,$form,$authhost,$domdesc) = @_;
    return unless (ref($form) eq 'HASH');
    my $canhost = 1;
    my $lonhost = $r->dir_config('lonHostID');
    my $udom = $form->{'udom'};
    my @intdoms;
    my $internet_names = &Apache::lonnet::get_internet_names($lonhost);
    if (ref($internet_names) eq 'ARRAY') {
        @intdoms = @{$internet_names};
    }
    my $uprimary_id = &Apache::lonnet::domain($udom,'primary');
    my $uint_dom = &Apache::lonnet::internet_dom($uprimary_id);
    unless ($uint_dom ne '' && grep(/^\Q$uint_dom\E$/,@intdoms)) {
        my $machine_dom = &Apache::lonnet::host_domain($lonhost);
        my $hostname = &Apache::lonnet::hostname($lonhost);
        my $serverhomeID = &Apache::lonnet::get_server_homeID($hostname);
        my $serverhomedom = &Apache::lonnet::host_domain($serverhomeID);
        my %defdomdefaults = &Apache::lonnet::get_domain_defaults($serverhomedom);
        my %udomdefaults = &Apache::lonnet::get_domain_defaults($udom);
        my $loncaparev;
        if ($authhost eq 'no_account_on_host') {
            $loncaparev = &Apache::lonnet::get_server_loncaparev($machine_dom);
        } else {
            $loncaparev = &Apache::lonnet::get_server_loncaparev($machine_dom,$lonhost);
        }
        $canhost = &Apache::lonnet::can_host_session($udom,$lonhost,$loncaparev,
                                                     $udomdefaults{'remotesessions'},
                                                     $defdomdefaults{'hostedsessions'});
    }
    unless ($canhost) {
        if ($authhost eq 'no_account_on_host') {
            my $checkloginvia = 1;
            my ($login_host,$hostname) = 
                &Apache::lonnet::choose_server($udom,$checkloginvia);
            &Apache::loncommon::content_type($r,'text/html');
            $r->send_http_header;
            if ($login_host ne '') {
                my $protocol = $Apache::lonnet::protocol{$login_host};
                $protocol = 'http' if ($protocol ne 'https');
                my $alias = &Apache::lonnet::use_proxy_alias($r,$login_host);
                $hostname = $alias if ($alias ne '');
                my $newurl = $protocol.'://'.$hostname.'/adm/createaccount';
                $r->print(&Apache::loncommon::start_page('Create a user account in LON-CAPA').
                          '<h3>'.&mt('Account creation').'</h3>'.
                          &mt('You do not currently have a LON-CAPA account at this institution.').'<br />'.
                          '<p>'.&mt('You will be able to create one by logging into a LON-CAPA server within the [_1] domain.',$domdesc).'</p>'.
                          '<p>'.&mt('[_1]Log in[_2]','<a href="'.$newurl.'">','</a>').
                          &Apache::loncommon::end_page());
            } else {
                $r->print(&Apache::loncommon::start_page('Access to LON-CAPA unavailable').
                          '<h3>'.&mt('Account creation unavailable').'</h3>'.
                          &mt('You do not currently have a LON-CAPA account at this institution.').'<br />'.
                          '<p>'.&mt('Currently a LON-CAPA server is not available within the [_1] domain for you to log-in to, to create an account.',$domdesc).'</p>'.
                          &Apache::loncommon::end_page());
            }
        } else {
            &success($r,$form->{'uname'},$udom,$authhost,'noredirect',undef,
                     $form);
            my ($otherserver) = &Apache::lonnet::choose_server($udom);
            $r->internal_redirect('/adm/switchserver?otherserver='.$otherserver);
        }
    }
    return $canhost;
}

sub noswitch {
    my $result = &Apache::loncommon::start_page('Access to LON-CAPA unavailable').
                 '<h3>'.&mt('Session unavailable').'</h3>'.
                 &mt('This LON-CAPA server is unable to host your session.').'<br />'.
                 '<p>'.&mt('Currently no other LON-CAPA server is available to host your session either.').'</p>'.
                 &Apache::loncommon::end_page();
    return $result;
}

sub loginhelpdisplay {
    my ($authdomain) = @_;
    my $login_help = 1;
    my $lang = &Apache::lonlocal::current_language();
    if ($login_help) {
        my $dom = $authdomain;
        if ($dom eq '') {
            $dom = &Apache::lonnet::default_login_domain();
        }
        my %domconfhash = &Apache::loncommon::get_domainconf($dom);
        my $loginhelp_url;
        if ($lang) {
            $loginhelp_url = $domconfhash{$dom.'.login.helpurl_'.$lang};
            if ($loginhelp_url ne '') {
                return $loginhelp_url;
            }
        }
        $loginhelp_url = $domconfhash{$dom.'.login.helpurl_nolang'};
        if ($loginhelp_url ne '') {
            return $loginhelp_url;
        } else {
            return '/adm/loginproblems.html';
        }
    }
    return;
}

sub alternate_unames_check {
    my ($uname,$udom) = @_;
    my %possunames;
    my %domdefs = &Apache::lonnet::get_domain_defaults($udom);
    if (ref($domdefs{'unamemap_rule'}) eq 'ARRAY') {
        if (@{$domdefs{'unamemap_rule'}} > 0) {
            %possunames =
                &Apache::lonnet::inst_rulecheck($udom,$uname,undef,
                                                'unamemap',$domdefs{'unamemap_rule'});
        }
    }
    return %possunames;
}

1;
__END__


