# The LearningOnline Network with CAPA
# The LON-CAPA Grading handler
#
# $Id: grades.pm,v 1.596.2.12.2.63 2024/12/09 01:24:24 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#



package Apache::grades;
use strict;
use Apache::style;
use Apache::lonxml;
use Apache::lonnet;
use Apache::loncommon;
use Apache::lonhtmlcommon;
use Apache::lonnavmaps;
use Apache::lonhomework;
use Apache::lonpickcode;
use Apache::loncoursedata;
use Apache::lonmsg();
use Apache::Constants qw(:common :http);
use Apache::lonlocal;
use Apache::lonenc;
use Apache::lonstathelpers;
use Apache::bridgetask();
use Apache::lontexconvert();
use HTML::Parser();
use File::MMagic;
use String::Similarity;
use LONCAPA;

use POSIX qw(floor);



my %perm=();
my %old_essays=();

#  These variables are used to recover from ssi errors

my $ssi_retries = 5;
my $ssi_error;
my $ssi_error_resource;
my $ssi_error_message;


sub ssi_with_retries {
    my ($resource, $retries, %form) = @_;
    my ($content, $response) = &Apache::loncommon::ssi_with_retries($resource, $retries, %form);
    if ($response->is_error) {
	$ssi_error          = 1;
	$ssi_error_resource = $resource;
	$ssi_error_message  = $response->code . " " . $response->message;
    }

    return $content;

}
#
#  Prodcuces an ssi retry failure error message to the user:
#

sub ssi_print_error {
    my ($r) = @_;
    my $helpurl = &Apache::loncommon::top_nav_help('Helpdesk');
    $r->print('
<br />
<h2>'.&mt('An unrecoverable network error occurred:').'</h2>
<p>
'.&mt('Unable to retrieve a resource from a server:').'<br />
'.&mt('Resource:').' '.$ssi_error_resource.'<br />
'.&mt('Error:').' '.$ssi_error_message.'
</p>
<p>'.
&mt('It is recommended that you try again later, as this error may mean the server was just temporarily unavailable, or is down for maintenance.').'<br />'.
&mt('If the error persists, please contact the [_1] for assistance.',$helpurl).
'</p>');
    return;
}

#
# --- Retrieve the parts from the metadata file.---
# Returns an array of everything that the resources stores away
#

sub getpartlist {
    my ($symb,$errorref) = @_;

    my $navmap   = Apache::lonnavmaps::navmap->new();
    unless (ref($navmap)) {
        if (ref($errorref)) { 
            $$errorref = 'navmap';
            return;
        }
    }
    my $res      = $navmap->getBySymb($symb);
    my $partlist = $res->parts();
    my $url      = $res->src();
    my @metakeys = split(/,/,&Apache::lonnet::metadata($url,'keys'));

    my @stores;
    foreach my $part (@{ $partlist }) {
	foreach my $key (@metakeys) {
	    if ($key =~ m/^stores_\Q$part\E_/) { push(@stores,$key); }
	}
    }
    return @stores;
}

#--- Format fullname, username:domain if different for display
#--- Use anywhere where the student names are listed
sub nameUserString {
    my ($type,$fullname,$uname,$udom) = @_;
    if ($type eq 'header') {
	return '<b>&nbsp;'.&mt('Fullname').'&nbsp;</b><span class="LC_internal_info">('.&mt('Username').')</span>';
    } else {
	return '&nbsp;'.$fullname.'<span class="LC_internal_info">&nbsp;('.$uname.
	    ($env{'user.domain'} eq $udom ? '' : ' ('.$udom.')').')</span>';
    }
}

#--- Get the partlist and the response type for a given problem. ---
#--- Indicate if a response type is coded handgraded or not. ---
#--- Count responseIDs, essayresponse items, and dropbox items ---
#--- Sets response_error pointer to "1" if navmaps object broken ---
sub response_type {
    my ($symb,$response_error) = @_;

    my $navmap = Apache::lonnavmaps::navmap->new();
    unless (ref($navmap)) {
        if (ref($response_error)) {
            $$response_error = 1;
        }
        return;
    }
    my $res = $navmap->getBySymb($symb);
    unless (ref($res)) {
        $$response_error = 1;
        return;
    }
    my $partlist = $res->parts();
    my ($numresp,$numessay,$numdropbox) = (0,0,0);
    my %vPart = 
	map { $_ => 1 } (&Apache::loncommon::get_env_multiple('form.vPart'));
    my (%response_types,%handgrade);
    foreach my $part (@{ $partlist }) {
	next if (%vPart && !exists($vPart{$part}));

	my @types = $res->responseType($part);
	my @ids = $res->responseIds($part);
	for (my $i=0; $i < scalar(@ids); $i++) {
            $numresp ++;
	    $response_types{$part}{$ids[$i]} = $types[$i];
            if ($types[$i] eq 'essay') {
                $numessay ++;
                if (&Apache::lonnet::EXT("resource.$part".'_'.$ids[$i].".uploadedfiletypes",$symb)) {
                    $numdropbox ++;
                }
            }
	    $handgrade{$part.'_'.$ids[$i]} = 
		&Apache::lonnet::EXT('resource.'.$part.'_'.$ids[$i].
				     '.handgrade',$symb);
	}
    }
    return ($partlist,\%handgrade,\%response_types,$numresp,$numessay,$numdropbox);
}

sub flatten_responseType {
    my ($responseType) = @_;
    my @part_response_id =
	map { 
	    my $part = $_;
	    map {
		[$part,$_]
		} sort(keys(%{ $responseType->{$part} }));
	} sort(keys(%$responseType));
    return @part_response_id;
}

sub get_display_part {
    my ($partID,$symb)=@_;
    my $display=&Apache::lonnet::EXT('resource.'.$partID.'.display',$symb);
    if (defined($display) and $display ne '') {
        $display.= ' (<span class="LC_internal_info">'
                  .&mt('Part ID: [_1]',$partID).'</span>)';
    } else {
	$display=$partID;
    }
    return $display;
}

#--- Show parts and response type
sub showResourceInfo {
    my ($symb,$partlist,$responseType,$formname,$checkboxes,$uploads) = @_;
    unless ((ref($partlist) eq 'ARRAY') && (ref($responseType) eq 'HASH')) {
        return '<br clear="all">';
    }
    my $coltitle = &mt('Problem Part Shown');
    if ($checkboxes) {
        $coltitle = &mt('Problem Part');
    } else {
        my $checkedparts = 0;
        foreach my $partid (&Apache::loncommon::get_env_multiple('form.vPart')) {
            if (grep(/^\Q$partid\E$/,@{$partlist})) {
                $checkedparts ++;
            }
        }
        if ($checkedparts == scalar(@{$partlist})) {
            return '<br clear="all">';
        }
        if ($uploads) {
            $coltitle = &mt('Problem Part Selected');
        }
    }
    my $result = '<div class="LC_left_float" style="display:inline-block;">';
    if ($checkboxes) {
        my $legend = &mt('Parts to display');
        if ($uploads) {
            $legend = &mt('Part(s) with dropbox');
        }
        $result .= '<fieldset style="display:inline-block;"><legend>'.$legend.'</legend>'.
                   '<span class="LC_nobreak">'.
                   '<label><input type="radio" name="chooseparts" value="0" onclick="toggleParts('."'$formname'".');" checked="checked" />'.
                   &mt('All parts').'</label>'.('&nbsp;'x2).
                   '<label><input type="radio" name="chooseparts" value="1" onclick="toggleParts('."'$formname'".');" />'.
                   &mt('Selected parts').'</label></span>'.
                   '<div id="LC_partselector" style="display:none">';
    }
    $result .= &Apache::loncommon::start_data_table()
              .&Apache::loncommon::start_data_table_header_row();
    if ($checkboxes) {
        $result .= '<th>'.&mt('Display?').'</th>';
    }
    $result .= '<th>'.$coltitle.'</th>'
              .'<th>'.&mt('Res. ID').'</th>'
              .'<th>'.&mt('Type').'</th>'
              .&Apache::loncommon::end_data_table_header_row();
    my %partsseen;
    foreach my $partID (sort(keys(%$responseType))) {
        foreach my $resID (sort(keys(%{ $responseType->{$partID} }))) {
            my $responsetype = $responseType->{$partID}->{$resID};
            if ($uploads) {
                next unless ($responsetype eq 'essay');
                next unless (&Apache::lonnet::EXT("resource.$partID".'_'."$resID.uploadedfiletypes",$symb));
            }
            my $display_part=&get_display_part($partID,$symb);
            if (exists($partsseen{$partID})) {
                $result.=&Apache::loncommon::continue_data_table_row();
            } else {
                $partsseen{$partID}=scalar(keys(%{$responseType->{$partID}}));
                $result.=&Apache::loncommon::start_data_table_row().
                         '<td rowspan="'.$partsseen{$partID}.'" style="vertical-align:middle">';
                if ($checkboxes) {
                    $result.='<input type="checkbox" name="vPart" checked="checked" value="'.$partID.'" /></td>'.
                             '<td rowspan="'.$partsseen{$partID}.'" style="vertical-align:middle">'.$display_part.'</td>';
                } else {
                    $result.=$display_part.'</td>';
                }
            }
            $result.='<td>'.'<span class="LC_internal_info">'.$resID.'</span></td>'
                    .'<td>'.&mt($responsetype).'</td>'
                    .&Apache::loncommon::end_data_table_row();
        }
    }
    $result.=&Apache::loncommon::end_data_table();
    if ($checkboxes) {
        $result .= '</div></fieldset>';
    }
    $result .= '</div><div style="padding:0;clear:both;margin:0;border:0"></div>';
    if (!keys(%partsseen)) {
        $result = '';
        if ($uploads) {
            return '<div style="padding:0;clear:both;margin:0;border:0"></div>'.
                   '<p class="LC_info">'.
                    &mt('No dropbox items or essayresponse items with uploadedfiletypes set.').
                   '</p>';
        } else {
            return '<br clear="all" />';
        }
    }  
    return $result;
}

sub part_selector_js {
    my $js = <<"END";
function toggleParts(formname) {
    if (document.getElementById('LC_partselector')) {
        var index = '';
        if (document.forms.length) {
            for (var i=0; i<document.forms.length; i++) {
                if (document.forms[i].name == formname) {
                    index = i;
                    break;
                }
            }
        }
        if ((index != '') && (document.forms[index].elements['chooseparts'].length > 1)) {
            for (var i=0; i<document.forms[index].elements['chooseparts'].length; i++) {
                if (document.forms[index].elements['chooseparts'][i].checked) {
                   var val = document.forms[index].elements['chooseparts'][i].value;
                    if (document.forms[index].elements['chooseparts'][i].value == 1) {
                        document.getElementById('LC_partselector').style.display = 'block';
                    } else {
                        document.getElementById('LC_partselector').style.display = 'none';
                    }
                }
            }
        }
    }
}
END
    return &Apache::lonhtmlcommon::scripttag($js);
}

sub reset_caches {
    &reset_analyze_cache();
    &reset_perm();
    &reset_old_essays();
}

{
    my %analyze_cache;
    my %analyze_cache_formkeys;

    sub reset_analyze_cache {
	undef(%analyze_cache);
        undef(%analyze_cache_formkeys);
    }

    sub get_analyze {
	my ($symb,$uname,$udom,$no_increment,$add_to_hash,$type,$trial,$rndseed,$bubbles_per_row)=@_;
	my $key = "$symb\0$uname\0$udom";
        if ($type eq 'randomizetry') {
            if ($trial ne '') {
                $key .= "\0".$trial;
            }
        }
	if (exists($analyze_cache{$key})) {
            my $getupdate = 0;
            if (ref($add_to_hash) eq 'HASH') {
                foreach my $item (keys(%{$add_to_hash})) {
                    if (ref($analyze_cache_formkeys{$key}) eq 'HASH') {
                        if (!exists($analyze_cache_formkeys{$key}{$item})) {
                            $getupdate = 1;
                            last;
                        }
                    } else {
                        $getupdate = 1;
                    }
                }
            }
            if (!$getupdate) {
                return $analyze_cache{$key};
            }
        }

	my (undef,undef,$url)=&Apache::lonnet::decode_symb($symb);
	$url=&Apache::lonnet::clutter($url);
        my %form = ('grade_target'      => 'analyze',
                    'grade_domain'      => $udom,
                    'grade_symb'        => $symb,
                    'grade_courseid'    =>  $env{'request.course.id'},
                    'grade_username'    => $uname,
                    'grade_noincrement' => $no_increment);
        if ($bubbles_per_row ne '') {
            $form{'bubbles_per_row'} = $bubbles_per_row;
        }
        if ($type eq 'randomizetry') {
            $form{'grade_questiontype'} = $type;
            if ($rndseed ne '') {
                $form{'grade_rndseed'} = $rndseed;
            }
        }
        if (ref($add_to_hash)) {
            %form = (%form,%{$add_to_hash});
        }
	my $subresult=&ssi_with_retries($url, $ssi_retries,%form);
	(undef,$subresult)=split(/_HASH_REF__/,$subresult,2);
	my %analyze=&Apache::lonnet::str2hash($subresult);
        if (ref($add_to_hash) eq 'HASH') {
            $analyze_cache_formkeys{$key} = $add_to_hash;
        } else {
            $analyze_cache_formkeys{$key} = {};
        }
	return $analyze_cache{$key} = \%analyze;
    }

    sub get_order {
	my ($partid,$respid,$symb,$uname,$udom,$no_increment,$type,$trial,$rndseed)=@_;
	my $analyze = &get_analyze($symb,$uname,$udom,$no_increment,undef,$type,$trial,$rndseed);
	return $analyze->{"$partid.$respid.shown"};
    }

    sub get_radiobutton_correct_foil {
	my ($partid,$respid,$symb,$uname,$udom,$type,$trial,$rndseed)=@_;
	my $analyze = &get_analyze($symb,$uname,$udom,undef,undef,$type,$trial,$rndseed);
        my $foils = &get_order($partid,$respid,$symb,$uname,$udom,undef,$type,$trial,$rndseed);
        if (ref($foils) eq 'ARRAY') {
	    foreach my $foil (@{$foils}) {
	        if ($analyze->{"$partid.$respid.foil.value.$foil"} eq 'true') {
		    return $foil;
	        }
	    }
	}
    }

    sub scantron_partids_tograde {
        my ($resource,$cid,$uname,$udom,$check_for_randomlist,$bubbles_per_row,$scancode) = @_;
        my (%analysis,@parts);
        if (ref($resource)) {
            my $symb = $resource->symb();
            my $add_to_form;
            if ($check_for_randomlist) {
                $add_to_form = { 'check_parts_withrandomlist' => 1,};
            }
            if ($scancode) {
                if (ref($add_to_form) eq 'HASH') {
                    $add_to_form->{'code_for_randomlist'} = $scancode;
                } else {
                    $add_to_form = { 'code_for_randomlist' => $scancode,};
                }
            }
            my $analyze =
                &get_analyze($symb,$uname,$udom,undef,$add_to_form,
                             undef,undef,undef,$bubbles_per_row);
            if (ref($analyze) eq 'HASH') {
                %analysis = %{$analyze};
            }
            if (ref($analysis{'parts'}) eq 'ARRAY') {
                foreach my $part (@{$analysis{'parts'}}) {
                    my ($id,$respid) = split(/\./,$part);
                    if (!&Apache::loncommon::check_if_partid_hidden($id,$symb,$udom,$uname)) {
                        push(@parts,$part);
                    }
                }
            }
        }
        return (\%analysis,\@parts);
    }

}

#--- Clean response type for display
#--- Currently filters option/rank/radiobutton/match/essay/Task
#        response types only.
sub cleanRecord {
    my ($answer,$response,$symb,$partid,$respid,$record,$order,$version,
	$uname,$udom,$type,$trial,$rndseed) = @_;
    my $grayFont = '<span class="LC_internal_info">';
    if ($response =~ /^(option|rank)$/) {
	my %answer=&Apache::lonnet::str2hash($answer);
        my @answer = %answer;
        %answer = map {&HTML::Entities::encode($_, '"<>&')} @answer;
	my %grading=&Apache::lonnet::str2hash($record->{$version."resource.$partid.$respid.submissiongrading"});
	my ($toprow,$bottomrow);
	foreach my $foil (@$order) {
	    if ($grading{$foil} == 1) {
		$toprow.='<td><b>'.$answer{$foil}.'&nbsp;</b></td>';
	    } else {
		$toprow.='<td><i>'.$answer{$foil}.'&nbsp;</i></td>';
	    }
	    $bottomrow.='<td>'.$grayFont.$foil.'</span>&nbsp;</td>';
	}
	return '<blockquote><table border="1">'.
	    '<tr valign="top"><td>'.&mt('Answer').'</td>'.$toprow.'</tr>'.
	    '<tr valign="top"><td>'.$grayFont.&mt('Option ID').'</span></td>'.
	    $bottomrow.'</tr></table></blockquote>';
    } elsif ($response eq 'match') {
	my %answer=&Apache::lonnet::str2hash($answer);
        my @answer = %answer;
        %answer = map {&HTML::Entities::encode($_, '"<>&')} @answer;
	my %grading=&Apache::lonnet::str2hash($record->{$version."resource.$partid.$respid.submissiongrading"});
	my @items=&Apache::lonnet::str2array($record->{$version."resource.$partid.$respid.submissionitems"});
	my ($toprow,$middlerow,$bottomrow);
	foreach my $foil (@$order) {
	    my $item=shift(@items);
	    if ($grading{$foil} == 1) {
		$toprow.='<td><b>'.$item.'&nbsp;</b></td>';
		$middlerow.='<td><b>'.$grayFont.$answer{$foil}.'&nbsp;</span></b></td>';
	    } else {
		$toprow.='<td><i>'.$item.'&nbsp;</i></td>';
		$middlerow.='<td><i>'.$grayFont.$answer{$foil}.'&nbsp;</span></i></td>';
	    }
	    $bottomrow.='<td>'.$grayFont.$foil.'</span>&nbsp;</td>';
	}
	return '<blockquote><table border="1">'.
	    '<tr valign="top"><td>'.&mt('Answer').'</td>'.$toprow.'</tr>'.
	    '<tr valign="top"><td>'.$grayFont.&mt('Item ID').'</span></td>'.
	    $middlerow.'</tr>'.
	    '<tr valign="top"><td>'.$grayFont.&mt('Option ID').'</span></td>'.
	    $bottomrow.'</tr></table></blockquote>';
    } elsif ($response eq 'radiobutton') {
	my %answer=&Apache::lonnet::str2hash($answer);
        my @answer = %answer;
        %answer = map {&HTML::Entities::encode($_, '"<>&')}  @answer;
	my ($toprow,$bottomrow);
	my $correct = 
	    &get_radiobutton_correct_foil($partid,$respid,$symb,$uname,$udom,$type,$trial,$rndseed);
	foreach my $foil (@$order) {
	    if (exists($answer{$foil})) {
		if ($foil eq $correct) {
		    $toprow.='<td><b>'.&mt('true').'</b></td>';
		} else {
		    $toprow.='<td><i>'.&mt('true').'</i></td>';
		}
	    } else {
		$toprow.='<td>'.&mt('false').'</td>';
	    }
	    $bottomrow.='<td>'.$grayFont.$foil.'</span>&nbsp;</td>';
	}
	return '<blockquote><table border="1">'.
	    '<tr valign="top"><td>'.&mt('Answer').'</td>'.$toprow.'</tr>'.
	    '<tr valign="top"><td>'.$grayFont.&mt('Option ID').'</span></td>'.
	    $bottomrow.'</tr></table></blockquote>';
    } elsif ($response eq 'essay') {
	if (! exists ($env{'form.'.$symb})) {
	    my (%keyhash) = &Apache::lonnet::dump('nohist_handgrade',
						  $env{'course.'.$env{'request.course.id'}.'.domain'},
						  $env{'course.'.$env{'request.course.id'}.'.num'});

	    my $loginuser = $env{'user.name'}.':'.$env{'user.domain'};
	    $env{'form.keywords'} = $keyhash{$symb.'_keywords'} ne '' ? $keyhash{$symb.'_keywords'} : '';
	    $env{'form.kwclr'}    = $keyhash{$loginuser.'_kwclr'} ne '' ? $keyhash{$loginuser.'_kwclr'} : 'red';
	    $env{'form.kwsize'}   = $keyhash{$loginuser.'_kwsize'} ne '' ? $keyhash{$loginuser.'_kwsize'} : '0';
	    $env{'form.kwstyle'}  = $keyhash{$loginuser.'_kwstyle'} ne '' ? $keyhash{$loginuser.'_kwstyle'} : '';
	    $env{'form.'.$symb} = 1; # so that we don't have to read it from disk for multiple sub of the same prob.
	}
        $answer = &Apache::lontexconvert::msgtexconverted($answer);
	return '<br /><br /><blockquote><tt>'.&keywords_highlight($answer).'</tt></blockquote>';
    } elsif ( $response eq 'organic') {
        my $result=&mt('Smile representation: [_1]',
                           '"<tt>'.&HTML::Entities::encode($answer, '"<>&').'</tt>"');
	my $jme=$record->{$version."resource.$partid.$respid.molecule"};
	$result.=&Apache::chemresponse::jme_img($jme,$answer,400);
	return $result;
    } elsif ( $response eq 'Task') {
	if ( $answer eq 'SUBMITTED') {
	    my $files = $record->{$version."resource.$respid.$partid.bridgetask.portfiles"};
	    my $result = &Apache::bridgetask::file_list($files,$uname,$udom);
	    return $result;
	} elsif ( grep(/^\Q$version\E.*?\.instance$/, keys(%{$record})) ) {
	    my @matches = grep(/^\Q$version\E.*?\.instance$/,
			       keys(%{$record}));
	    return join('<br />',($version,@matches));
			       
			       
	} else {
	    my $result =
		'<p>'
		.&mt('Overall result: [_1]',
		     $record->{$version."resource.$respid.$partid.status"})
		.'</p>';
	    
	    $result .= '<ul>';
	    my @grade = grep(/^\Q${version}resource.$respid.$partid.\E[^.]*[.]status$/,
			     keys(%{$record}));
	    foreach my $grade (sort(@grade)) {
		my ($dim) = ($grade =~/[.]([^.]+)[.]status$/);
		$result.= '<li>'.&mt("Dimension: [_1], status [_2] ",
				     $dim, $record->{$grade}).
			  '</li>';
	    }
	    $result.='</ul>';
	    return $result;
	}
    } elsif ( $response =~ m/(?:numerical|formula|custom)/) {
        # Respect multiple input fields, see Bug #5409 
	$answer = 
	    &Apache::loncommon::format_previous_attempt_value('submission',
							      $answer);
	return $answer;
    }
    return &HTML::Entities::encode($answer, '"<>&');
}

#-- A couple of common js functions
sub commonJSfunctions {
    my $request = shift;
    $request->print(&Apache::lonhtmlcommon::scripttag(<<COMMONJSFUNCTIONS));
    function radioSelection(radioButton) {
	var selection=null;
	if (radioButton.length > 1) {
	    for (var i=0; i<radioButton.length; i++) {
		if (radioButton[i].checked) {
		    return radioButton[i].value;
		}
	    }
	} else {
	    if (radioButton.checked) return radioButton.value;
	}
	return selection;
    }

    function pullDownSelection(selectOne) {
	var selection="";
	if (selectOne.length > 1) {
	    for (var i=0; i<selectOne.length; i++) {
		if (selectOne[i].selected) {
		    return selectOne[i].value;
		}
	    }
	} else {
            // only one value it must be the selected one
	    return selectOne.value;
	}
    }
COMMONJSFUNCTIONS
}

#--- Dumps the class list with usernames,list of sections,
#--- section, ids and fullnames for each user.
sub getclasslist {
    my ($getsec,$filterbyaccstatus,$getgroup,$symb,$submitonly,$filterbysubmstatus) = @_;
    my @getsec;
    my @getgroup;
    my $stu_status = join(':',&Apache::loncommon::get_env_multiple('form.Status'));
    if (!ref($getsec)) {
	if ($getsec ne '' && $getsec ne 'all') {
	    @getsec=($getsec);
	}
    } else {
	@getsec=@{$getsec};
    }
    if (grep(/^all$/,@getsec)) { undef(@getsec); }
    if (!ref($getgroup)) {
	if ($getgroup ne '' && $getgroup ne 'all') {
	    @getgroup=($getgroup);
	}
    } else {
	@getgroup=@{$getgroup};
    }
    if (grep(/^all$/,@getgroup)) { undef(@getgroup); }

    my ($classlist,$keylist)=&Apache::loncoursedata::get_classlist();
    # Bail out if we were unable to get the classlist
    return if (! defined($classlist));
    &Apache::loncoursedata::get_group_memberships($classlist,$keylist);
    #
    my %sections;
    my %fullnames;
    my ($cdom,$cnum,$partlist);
    if (($filterbysubmstatus) && ($submitonly ne 'all') && ($symb ne '')) {
        $cdom = $env{"course.$env{'request.course.id'}.domain"};
        $cnum = $env{"course.$env{'request.course.id'}.num"};
        my $res_error;
        ($partlist) = &response_type($symb,\$res_error);
    }
    foreach my $student (keys(%$classlist)) {
        my $end      = 
            $classlist->{$student}->[&Apache::loncoursedata::CL_END()];
        my $start    = 
            $classlist->{$student}->[&Apache::loncoursedata::CL_START()];
        my $id       = 
            $classlist->{$student}->[&Apache::loncoursedata::CL_ID()];
        my $section  = 
            $classlist->{$student}->[&Apache::loncoursedata::CL_SECTION()];
        my $fullname = 
            $classlist->{$student}->[&Apache::loncoursedata::CL_FULLNAME()];
        my $status   = 
            $classlist->{$student}->[&Apache::loncoursedata::CL_STATUS()];
        my $group   = 
            $classlist->{$student}->[&Apache::loncoursedata::CL_GROUP()];
	# filter students according to status selected
	if ($filterbyaccstatus && (!($stu_status =~ /Any/))) {
	    if (!($stu_status =~ $status)) {
		delete($classlist->{$student});
		next;
	    }
	}
	# filter students according to groups selected
	my @stu_groups = split(/,/,$group);
	if (@getgroup) {
	    my $exclude = 1;
	    foreach my $grp (@getgroup) {
	        foreach my $stu_group (@stu_groups) {
	            if ($stu_group eq $grp) {
	                $exclude = 0;
    	            } 
	        }
    	        if (($grp eq 'none') && !$group) {
        	    $exclude = 0;
        	}
	    }
	    if ($exclude) {
	        delete($classlist->{$student});
		next;
	    }
	}
        if (($filterbysubmstatus) && ($submitonly ne 'all') && ($symb ne '')) {
            my $udom =
                $classlist->{$student}->[&Apache::loncoursedata::CL_SDOM()];
            my $uname =
                $classlist->{$student}->[&Apache::loncoursedata::CL_SNAME()];
            if (($symb ne '') && ($udom ne '') && ($uname ne '')) {
                if ($submitonly eq 'queued') {
                    my %queue_status =
                        &Apache::bridgetask::get_student_status($symb,$cdom,$cnum,
                                                                $udom,$uname);
                    if (!defined($queue_status{'gradingqueue'})) {
                        delete($classlist->{$student});
                        next;
                    }
                } else {
                    my (%status) =&student_gradeStatus($symb,$udom,$uname,$partlist);
                    my $submitted = 0;
                    my $graded = 0;
                    my $incorrect = 0;
                    foreach (keys(%status)) {
                        $submitted = 1 if ($status{$_} ne 'nothing');
                        $graded = 1 if ($status{$_} =~ /^ungraded/);
                        $incorrect = 1 if ($status{$_} =~ /^incorrect/);

                        my ($foo,$partid,$foo1) = split(/\./,$_);
                        if ($status{'resource.'.$partid.'.submitted_by'} ne '') {
                            $submitted = 0;
                        }
                    }
                    if (!$submitted && ($submitonly eq 'yes' ||
                                        $submitonly eq 'incorrect' ||
                                        $submitonly eq 'graded')) {
                        delete($classlist->{$student});
                        next;
                    } elsif (!$graded && ($submitonly eq 'graded')) {
                        delete($classlist->{$student});
                        next;
                    } elsif (!$incorrect && $submitonly eq 'incorrect') {
                        delete($classlist->{$student});
                        next;
                    }
                }
            }
        }
	$section = ($section ne '' ? $section : 'none');
	if (&canview($section)) {
	    if (!@getsec || grep(/^\Q$section\E$/,@getsec)) {
		$sections{$section}++;
		if ($classlist->{$student}) {
		    $fullnames{$student}=$fullname;
		}
	    } else {
		delete($classlist->{$student});
	    }
	} else {
	    delete($classlist->{$student});
	}
    }
    my @sections = sort(keys(%sections));
    return ($classlist,\@sections,\%fullnames);
}

sub canmodify {
    my ($sec)=@_;
    if ($perm{'mgr'}) {
	if (!defined($perm{'mgr_section'})) {
	    # can modify whole class
	    return 1;
	} else {
	    if ($sec eq $perm{'mgr_section'}) {
		#can modify the requested section
		return 1;
	    } else {
		# can't modify the requested section
		return 0;
	    }
	}
    }
    #can't modify
    return 0;
}

sub canview {
    my ($sec)=@_;
    if ($perm{'vgr'}) {
	if (!defined($perm{'vgr_section'})) {
	    # can view whole class
	    return 1;
	} else {
	    if ($sec eq $perm{'vgr_section'}) {
		#can view the requested section
		return 1;
	    } else {
		# can't view the requested section
		return 0;
	    }
	}
    }
    #can't view
    return 0;
}

#--- Retrieve the grade status of a student for all the parts
sub student_gradeStatus {
    my ($symb,$udom,$uname,$partlist) = @_;
    my %record     = &Apache::lonnet::restore($symb,$env{'request.course.id'},$udom,$uname);
    my %partstatus = ();
    foreach (@$partlist) {
	my ($status,undef)   = split(/_/,$record{"resource.$_.solved"},2);
	$status              = 'nothing' if ($status eq '');
	$partstatus{$_}      = $status;
	my $subkey           = "resource.$_.submitted_by";
	$partstatus{$subkey} = $record{$subkey} if ($record{$subkey} ne '');
    }
    return %partstatus;
}

# hidden form and javascript that calls the form
# Use by verifyscript and viewgrades
# Shows a student's view of problem and submission
sub jscriptNform {
    my ($symb) = @_;
    my $stu_status = join(':',&Apache::loncommon::get_env_multiple('form.Status'));
    my $jscript= &Apache::lonhtmlcommon::scripttag(
	'    function viewOneStudent(user,domain) {'."\n".
	'	document.onestudent.student.value = user;'."\n".
	'	document.onestudent.userdom.value = domain;'."\n".
	'	document.onestudent.submit();'."\n".
	'    }'."\n".
	"\n");
    $jscript.= '<form action="/adm/grades" method="post" name="onestudent">'."\n".
	'<input type="hidden" name="symb"    value="'.&Apache::lonenc::check_encrypt($symb).'" />'."\n".
	'<input type="hidden" name="Status"  value="'.$stu_status.'" />'."\n".
	'<input type="hidden" name="command" value="submission" />'."\n".
	'<input type="hidden" name="student" value="" />'."\n".
	'<input type="hidden" name="userdom" value="" />'."\n".
	'</form>'."\n";
    return $jscript;
}



# Given the score (as a number [0-1] and the weight) what is the final
# point value? This function will round to the nearest tenth, third,
# or quarter if one of those is within the tolerance of .00001.
sub compute_points {
    my ($score, $weight) = @_;
    
    my $tolerance = .00001;
    my $points = $score * $weight;

    # Check for nearness to 1/x.
    my $check_for_nearness = sub {
        my ($factor) = @_;
        my $num = ($points * $factor) + $tolerance;
        my $floored_num = floor($num);
        if ($num - $floored_num < 2 * $tolerance * $factor) {
            return $floored_num / $factor;
        }
        return $points;
    };

    $points = $check_for_nearness->(10);
    $points = $check_for_nearness->(3);
    $points = $check_for_nearness->(4);
    
    return $points;
}

#------------------ End of general use routines --------------------

#
# Find most similar essay
#

sub most_similar {
    my ($uname,$udom,$symb,$uessay)=@_;

    unless ($symb) { return ''; }

    unless (ref($old_essays{$symb}) eq 'HASH') { return ''; }

# ignore spaces and punctuation

    $uessay=~s/\W+/ /gs;

# ignore empty submissions (occuring when only files are sent)

    unless ($uessay=~/\w+/s) { return ''; }

# these will be returned. Do not care if not at least 50 percent similar
    my $limit=0.6;
    my $sname='';
    my $sdom='';
    my $scrsid='';
    my $sessay='';
# go through all essays ...
    foreach my $tkey (keys(%{$old_essays{$symb}})) {
	my ($tname,$tdom,$tcrsid)=map {&unescape($_)} (split(/\./,$tkey));
# ... except the same student
        next if (($tname eq $uname) && ($tdom eq $udom));
	my $tessay=$old_essays{$symb}{$tkey};
	$tessay=~s/\W+/ /gs;
# String similarity gives up if not even limit
	my $tsimilar=&String::Similarity::similarity($uessay,$tessay,$limit);
# Found one
	if ($tsimilar>$limit) {
	    $limit=$tsimilar;
	    $sname=$tname;
	    $sdom=$tdom;
	    $scrsid=$tcrsid;
	    $sessay=$old_essays{$symb}{$tkey};
	}
    }
    if ($limit>0.6) {
       return ($sname,$sdom,$scrsid,$sessay,$limit);
    } else {
       return ('','','','',0);
    }
}

#-------------------------------------------------------------------

#------------------------------------ Receipt Verification Routines
#

sub initialverifyreceipt {
   my ($request,$symb) = @_;
   &commonJSfunctions($request);
   return '<form name="gradingMenu" action=""><input type="submit" value="'.&mt('Verify Receipt Number.').'" />'.
        &Apache::lonnet::recprefix($env{'request.course.id'}).
        '-<input type="text" name="receipt" size="4" />'.
        '<input type="hidden" name="symb" value="'.&Apache::lonenc::check_encrypt($symb).'" />'."\n".
        '<input type="hidden" name="command" value="verify" />'.
        "</form>\n";
}

#--- Check whether a receipt number is valid.---
sub verifyreceipt {
    my ($request,$symb) = @_;

    my $courseid = $env{'request.course.id'};
    my $receipt  = &Apache::lonnet::recprefix($courseid).'-'.
	$env{'form.receipt'};
    $receipt     =~ s/[^\-\d]//g;

    my $title =
	'<h3><span class="LC_info">'.
	&mt('Verifying Receipt Number [_1]',$receipt).
	'</span></h3>'."\n";

    my ($string,$contents,$matches) = ('','',0);
    my (undef,undef,$fullname) = &getclasslist('all','0');
    
    my $receiptparts=0;
    if ($env{"course.$courseid.receiptalg"} eq 'receipt2' ||
	$env{"course.$courseid.receiptalg"} eq 'receipt3') { $receiptparts=1; }
    my $parts=['0'];
    if ($receiptparts) {
        my $res_error; 
        ($parts)=&response_type($symb,\$res_error);
        if ($res_error) {
            return &navmap_errormsg();
        } 
    }
    
    my $header = 
	&Apache::loncommon::start_data_table().
	&Apache::loncommon::start_data_table_header_row().
	'<th>&nbsp;'.&mt('Fullname').'&nbsp;</th>'."\n".
	'<th>&nbsp;'.&mt('Username').'&nbsp;</th>'."\n".
	'<th>&nbsp;'.&mt('Domain').'&nbsp;</th>';
    if ($receiptparts) {
	$header.='<th>&nbsp;'.&mt('Problem Part').'&nbsp;</th>';
    }
    $header.=
	&Apache::loncommon::end_data_table_header_row();

    foreach (sort 
	     {
		 if (lc($$fullname{$a}) ne lc($$fullname{$b})) {
		     return (lc($$fullname{$a}) cmp lc($$fullname{$b}));
		 }
		 return $a cmp $b;
	     } (keys(%$fullname))) {
	my ($uname,$udom)=split(/\:/);
	foreach my $part (@$parts) {
	    if ($receipt eq &Apache::lonnet::ireceipt($uname,$udom,$courseid,$symb,$part)) {
		$contents.=
		    &Apache::loncommon::start_data_table_row().
		    '<td>&nbsp;'."\n".
		    '<a href="javascript:viewOneStudent(\''.$uname.'\',\''.$udom.
		    '\');" target="_self">'.$$fullname{$_}.'</a>&nbsp;</td>'."\n".
		    '<td>&nbsp;'.$uname.'&nbsp;</td>'.
		    '<td>&nbsp;'.$udom.'&nbsp;</td>';
		if ($receiptparts) {
		    $contents.='<td>&nbsp;'.$part.'&nbsp;</td>';
		}
		$contents.= 
		    &Apache::loncommon::end_data_table_row()."\n";
		
		$matches++;
	    }
	}
    }
    if ($matches == 0) {
        $string = $title
                 .'<p class="LC_warning">'
                 .&mt('No match found for the above receipt number.')
                 .'</p>';
    } else {
	$string = &jscriptNform($symb).$title.
	    '<p>'.
	    &mt('The above receipt number matches the following [quant,_1,student].',$matches).
	    '</p>'.
	    $header.
	    $contents.
	    &Apache::loncommon::end_data_table()."\n";
    }
    return $string;
}

#--- This is called by a number of programs.
#--- Called from the Grading Menu - View/Grade an individual student
#--- Also called directly when one clicks on the subm button 
#    on the problem page.
sub listStudents {
    my ($request,$symb,$submitonly,$divforres) = @_;

    my $cdom      = $env{"course.$env{'request.course.id'}.domain"};
    my $cnum      = $env{"course.$env{'request.course.id'}.num"};
    my $getsec    = $env{'form.section'} eq '' ? 'all' : $env{'form.section'};
    my $getgroup  = $env{'form.group'} eq '' ? 'all' : $env{'form.group'};
    unless ($submitonly) {
        $submitonly = $env{'form.submitonly'} eq '' ? 'all' : $env{'form.submitonly'};
    }

    my $result='';
    my $res_error;
    my ($partlist,$handgrade,$responseType,$numresp,$numessay) = &response_type($symb,\$res_error);

    my $table;
    if (ref($partlist) eq 'ARRAY') {
        if (scalar(@$partlist) > 1 ) {
            $table = &showResourceInfo($symb,$partlist,$responseType,'gradesub',1);
        } elsif ($divforres) {
            $table = '<div style="padding:0;clear:both;margin:0;border:0"></div>';
        } else {
            $table = '<br clear="all" />';
        }
    }

    my %js_lt = &Apache::lonlocal::texthash (
		'multiple' => 'Please select a student or group of students before clicking on the Next button.',
		'single'   => 'Please select the student before clicking on the Next button.',
	     );
    &js_escape(\%js_lt);
    $request->print(&Apache::lonhtmlcommon::scripttag(<<LISTJAVASCRIPT));
    function checkSelect(checkBox) {
	var ctr=0;
	var sense="";
	if (checkBox.length > 1) {
	    for (var i=0; i<checkBox.length; i++) {
		if (checkBox[i].checked) {
		    ctr++;
		}
	    }
	    sense = '$js_lt{'multiple'}';
	} else {
	    if (checkBox.checked) {
		ctr = 1;
	    }
	    sense = '$js_lt{'single'}';
	}
	if (ctr == 0) {
	    alert(sense);
	    return false;
	}
	document.gradesub.submit();
    }

    function reLoadList(formname) {
	if (formname.saveStatusOld.value == pullDownSelection(formname.Status)) {return;}
	formname.command.value = 'submission';
	formname.submit();
    }
LISTJAVASCRIPT

    &commonJSfunctions($request);
    $request->print($result);

    my $gradeTable='<form action="/adm/grades" method="post" name="gradesub">'.
	"\n".$table;

    $gradeTable .= &Apache::lonhtmlcommon::start_pick_box();
    $gradeTable .= &Apache::lonhtmlcommon::row_title(&mt('View Problem Text'))
                  .'<label><input type="radio" name="vProb" value="no" checked="checked" /> '.&mt('no').' </label>'."\n"
                  .'<label><input type="radio" name="vProb" value="yes" /> '.&mt('one student').' </label>'."\n"
                  .'<label><input type="radio" name="vProb" value="all" /> '.&mt('all students').' </label><br />'."\n"
                  .&Apache::lonhtmlcommon::row_closure();
    $gradeTable .= &Apache::lonhtmlcommon::row_title(&mt('View Answer'))
                  .'<label><input type="radio" name="vAns" value="no"  /> '.&mt('no').' </label>'."\n"
                  .'<label><input type="radio" name="vAns" value="yes" /> '.&mt('one student').' </label>'."\n"
                  .'<label><input type="radio" name="vAns" value="all" checked="checked" /> '.&mt('all students').' </label><br />'."\n"
                  .&Apache::lonhtmlcommon::row_closure();

    my $stu_status = join(':',&Apache::loncommon::get_env_multiple('form.Status'));
    my $saveStatus = $stu_status eq '' ? 'Active' : $stu_status;
    $env{'form.Status'} = $saveStatus;
    my %optiontext = &Apache::lonlocal::texthash (
                          lastonly => 'last submission',
                          last     => 'last submission with details',
                          datesub  => 'all submissions',
                          all      => 'all submissions with details',
                      );
    my $submission_options =
        '<span class="LC_nobreak">'.
        '<label><input type="radio" name="lastSub" value="lastonly" /> '.
        $optiontext{'lastonly'}.' </label></span>'."\n".
        '<span class="LC_nobreak">'.
        '<label><input type="radio" name="lastSub" value="last" /> '.
        $optiontext{'last'}.' </label></span>'."\n".
        '<span class="LC_nobreak">'.
        '<label><input type="radio" name="lastSub" value="datesub" checked="checked" /> '.
        $optiontext{'datesub'}.'</label></span>'."\n".
        '<span class="LC_nobreak">'.
        '<label><input type="radio" name="lastSub" value="all" /> '.
        $optiontext{'all'}.'</label></span>';
    my ($compmsg,$nocompmsg);
    $nocompmsg = ' checked="checked"';
    if ($numessay) {
        $compmsg = $nocompmsg;
        $nocompmsg = '';
    }
    $gradeTable .= &Apache::lonhtmlcommon::row_title(&mt('Submissions'))
                  .$submission_options;
# Check if any gradable
    my $showmore;
    if ($perm{'mgr'}) {
        my @sections;
        if ($env{'request.course.sec'} ne '') {
            @sections = ($env{'request.course.sec'});
        } elsif ($env{'form.section'} eq '') {
            @sections = ('all');
        } else {
            @sections = &Apache::loncommon::get_env_multiple('form.section');
        }
        if (grep(/^all$/,@sections)) {
            $showmore = 1;
        } else {
            foreach my $sec (@sections) {
                if (&canmodify($sec)) {
                    $showmore = 1;
                    last;
                }
            }
        }
    }

    if ($showmore) {
        $gradeTable .=
                   &Apache::lonhtmlcommon::row_closure()
                  .&Apache::lonhtmlcommon::row_title(&mt('Send Messages'))
                  .'<span class="LC_nobreak">'
                  .'<label><input type="radio" name="compmsg" value="0"'.$nocompmsg.' />'
                  .&mt('No').('&nbsp;'x2).'</label>'
                  .'<label><input type="radio" name="compmsg" value="1"'.$compmsg.' />'
                  .&mt('Yes').('&nbsp;'x2).'</label>'
                  .&Apache::lonhtmlcommon::row_closure();

        $gradeTable .= 
                   &Apache::lonhtmlcommon::row_title(&mt('Grading Increments'))
                  .'<select name="increment">'
                  .'<option value="1">'.&mt('Whole Points').'</option>'
                  .'<option value=".5">'.&mt('Half Points').'</option>'
                  .'<option value=".25">'.&mt('Quarter Points').'</option>'
                  .'<option value=".1">'.&mt('Tenths of a Point').'</option>'
                  .'</select>';
    }
    $gradeTable .= 
        &build_section_inputs().
	'<input type="hidden" name="submitonly"  value="'.$submitonly.'" />'."\n".
	'<input type="hidden" name="symb" value="'.&Apache::lonenc::check_encrypt($symb).'" />'."\n".
	'<input type="hidden" name="saveStatusOld" value="'.$saveStatus.'" />'."\n";
    if (exists($env{'form.Status'})) {
	$gradeTable .= '<input type="hidden" name="Status" value="'.$env{'form.Status'}.'" />'."\n";
    } else {
        $gradeTable .= &Apache::lonhtmlcommon::row_closure()
                      .&Apache::lonhtmlcommon::row_title(&mt('Student Status'))
                      .&Apache::lonhtmlcommon::StatusOptions(
                           $saveStatus,undef,1,'javascript:reLoadList(this.form);');
    }
    if ($numessay) {
        $gradeTable .= &Apache::lonhtmlcommon::row_closure()
                      .&Apache::lonhtmlcommon::row_title(&mt('Check For Plagiarism'))
                      .'<input type="checkbox" name="checkPlag" checked="checked" />';
    }
    $gradeTable .= &Apache::lonhtmlcommon::row_closure(1)
                  .&Apache::lonhtmlcommon::end_pick_box();

    $gradeTable .= '<p>'
                  .&mt("To view/grade/regrade a submission or a group of submissions, click on the check box(es) next to the student's name(s). Then click on the Next button.")."\n"
                  .'<input type="hidden" name="command" value="processGroup" />'
                  .'</p>';

# checkall buttons
    $gradeTable.=&check_script('gradesub', 'stuinfo');
    $gradeTable.='<input type="button" '."\n".
        'onclick="javascript:checkSelect(this.form.stuinfo);" '."\n".
        'value="'.&mt('Next').' &rarr;" /> <br />'."\n";
    $gradeTable.=&check_buttons();
    my ($classlist, undef, $fullname) = &getclasslist($getsec,'1',$getgroup);
    $gradeTable.= &Apache::loncommon::start_data_table().
	&Apache::loncommon::start_data_table_header_row();
    my $loop = 0;
    while ($loop < 2) {
	$gradeTable.='<th>'.&mt('No.').'</th><th>'.&mt('Select').'</th>'.
	    '<th>'.&nameUserString('header').'&nbsp;'.&mt('Section/Group').'</th>';
	if (($submitonly ne 'queued') && ($submitonly ne 'all')) {
	    foreach my $part (sort(@$partlist)) {
		my $display_part=
		    &get_display_part((split(/_/,$part))[0],$symb);
		$gradeTable.=
		    '<th>'.&mt('Part: [_1] Status',$display_part).'</th>';
	    }
	} elsif ($submitonly eq 'queued') {
	    $gradeTable.='<th>'.&mt('Queue Status').'&nbsp;</th>';
	}
	$loop++;
#	$gradeTable.='<td></td>' if ($loop%2 ==1);
    }
    $gradeTable.=&Apache::loncommon::end_data_table_header_row()."\n";

    my $ctr = 0;
    foreach my $student (sort 
			 {
			     if (lc($$fullname{$a}) ne lc($$fullname{$b})) {
				 return (lc($$fullname{$a}) cmp lc($$fullname{$b}));
			     }
			     return $a cmp $b;
			 }
			 (keys(%$fullname))) {
	my ($uname,$udom) = split(/:/,$student);

	my %status = ();

	if ($submitonly eq 'queued') {
	    my %queue_status = 
		&Apache::bridgetask::get_student_status($symb,$cdom,$cnum,
							$udom,$uname);
	    next if (!defined($queue_status{'gradingqueue'}));
	    $status{'gradingqueue'} = $queue_status{'gradingqueue'};
	}

	if (($submitonly ne 'queued') && ($submitonly ne 'all')) {
	    (%status) =&student_gradeStatus($symb,$udom,$uname,$partlist);
	    my $submitted = 0;
	    my $graded = 0;
	    my $incorrect = 0;
	    foreach (keys(%status)) {
		$submitted = 1 if ($status{$_} ne 'nothing');
		$graded = 1 if ($status{$_} =~ /^ungraded/);
		$incorrect = 1 if ($status{$_} =~ /^incorrect/);
		
		my ($foo,$partid,$foo1) = split(/\./,$_);
		if ($status{'resource.'.$partid.'.submitted_by'} ne '') {
		    $submitted = 0;
		    my ($part)=split(/\./,$partid);
		    $gradeTable.='<input type="hidden" name="'.
			$student.':'.$part.':submitted_by" value="'.
			$status{'resource.'.$partid.'.submitted_by'}.'" />';
		}
	    }
	    
	    next if (!$submitted && ($submitonly eq 'yes' ||
				     $submitonly eq 'incorrect' ||
				     $submitonly eq 'graded'));
	    next if (!$graded && ($submitonly eq 'graded'));
	    next if (!$incorrect && $submitonly eq 'incorrect');
	}

	$ctr++;
	my $section = $classlist->{$student}->[&Apache::loncoursedata::CL_SECTION()];
        my $group = $classlist->{$student}->[&Apache::loncoursedata::CL_GROUP()];
	if ( $perm{'vgr'} eq 'F' ) {
	    if ($ctr%2 ==1) {
		$gradeTable.= &Apache::loncommon::start_data_table_row();
	    }
	    $gradeTable.='<td align="right">'.$ctr.'&nbsp;</td>'.
               '<td align="center"><label><input type="checkbox" name="stuinfo" value="'.
               $student.':'.$$fullname{$student}.':::SECTION'.$section.
	       ')&nbsp;" />&nbsp;&nbsp;</label></td>'."\n".'<td>'.
	       &nameUserString(undef,$$fullname{$student},$uname,$udom).
	       '&nbsp;'.$section.($group ne '' ?'/'.$group:'').'</td>'."\n";

	    if ($submitonly ne 'all') {
		foreach (sort(keys(%status))) {
		    next if ($_ =~ /^resource.*?submitted_by$/);
		    $gradeTable.='<td align="center">&nbsp;'.&mt($status{$_}).'&nbsp;</td>'."\n";
		}
	    }
#	    $gradeTable.='<td></td>' if ($ctr%2 ==1);
	    if ($ctr%2 ==0) {
		$gradeTable.=&Apache::loncommon::end_data_table_row()."\n";
	    }
	}
    }
    if ($ctr%2 ==1) {
	$gradeTable.='<td>&nbsp;</td><td>&nbsp;</td><td>&nbsp;</td>';
	    if (($submitonly ne 'queued') && ($submitonly ne 'all')) {
		foreach (@$partlist) {
		    $gradeTable.='<td>&nbsp;</td>';
		}
	    } elsif ($submitonly eq 'queued') {
		$gradeTable.='<td>&nbsp;</td>';
	    }
	$gradeTable.=&Apache::loncommon::end_data_table_row();
    }

    $gradeTable.=&Apache::loncommon::end_data_table()."\n".
        '<input type="button" '.
        'onclick="javascript:checkSelect(this.form.stuinfo);" '.
        'value="'.&mt('Next').' &rarr;" /></form>'."\n";
    if ($ctr == 0) {
	my $num_students=(scalar(keys(%$fullname)));
	if ($num_students eq 0) {
	    $gradeTable='<br />&nbsp;<span class="LC_warning">'.&mt('There are no students currently enrolled.').'</span>';
	} else {
	    my $submissions='submissions';
	    if ($submitonly eq 'incorrect') { $submissions = 'incorrect submissions'; }
	    if ($submitonly eq 'graded'   ) { $submissions = 'ungraded submissions'; }
	    if ($submitonly eq 'queued'   ) { $submissions = 'queued submissions'; }
	    $gradeTable='<br />&nbsp;<span class="LC_warning">'.
		&mt('No '.$submissions.' found for this resource for any students. ([quant,_1,student] checked for '.$submissions.')',
		    $num_students).
		'</span><br />';
	}
    } elsif ($ctr == 1) {
	$gradeTable =~ s/type="checkbox"/type="checkbox" checked="checked"/;
    }
    $request->print($gradeTable);
    return '';
}

#---- Called from the listStudents routine

sub check_script {
    my ($form,$type) = @_;
    my $chkallscript = &Apache::lonhtmlcommon::scripttag('
    function checkall() {
        for (i=0; i<document.forms.'.$form.'.elements.length; i++) {
            ele = document.forms.'.$form.'.elements[i];
            if (ele.name == "'.$type.'") {
            document.forms.'.$form.'.elements[i].checked=true;
                                       }
        }
    }

    function checksec() {
        for (i=0; i<document.forms.'.$form.'.elements.length; i++) {
            ele = document.forms.'.$form.'.elements[i];
           string = document.forms.'.$form.'.chksec.value;
           if
          (ele.value.indexOf(":::SECTION"+string)>0) {
              document.forms.'.$form.'.elements[i].checked=true;
            }
        }
    }


    function uncheckall() {
        for (i=0; i<document.forms.'.$form.'.elements.length; i++) {
            ele = document.forms.'.$form.'.elements[i];
            if (ele.name == "'.$type.'") {
            document.forms.'.$form.'.elements[i].checked=false;
                                       }
        }
    }

'."\n");
    return $chkallscript;
}

sub check_buttons {
    my $buttons.='<input type="button" onclick="checkall()" value="'.&mt('Check All').'" />';
    $buttons.='<input type="button" onclick="uncheckall()" value="'.&mt('Uncheck All').'" />&nbsp;';
    $buttons.='<input type="button" onclick="checksec()" value="'.&mt('Check Section/Group').'" />';
    $buttons.='<input type="text" size="5" name="chksec" />&nbsp;';
    return $buttons;
}

#     Displays the submissions for one student or a group of students
sub processGroup {
    my ($request,$symb) = @_;
    my $ctr        = 0;
    my @stuchecked = &Apache::loncommon::get_env_multiple('form.stuinfo');
    my $total      = scalar(@stuchecked)-1;

    foreach my $student (@stuchecked) {
	my ($uname,$udom,$fullname) = split(/:/,$student);
	$env{'form.student'}        = $uname;
	$env{'form.userdom'}        = $udom;
	$env{'form.fullname'}       = $fullname;
	&submission($request,$ctr,$total,$symb);
	$ctr++;
    }
    return '';
}

#------------------------------------------------------------------------------------
#
#-------------------------- Next few routines handles grading by student, essentially
#                           handles essay response type problem/part
#
#--- Javascript to handle the submission page functionality ---
sub sub_page_js {
    my $request = shift;
    my $alertmsg = &mt('A number equal or greater than 0 is expected. Entered value = ');
    &js_escape(\$alertmsg);
    $request->print(&Apache::lonhtmlcommon::scripttag(<<SUBJAVASCRIPT));
    function updateRadio(formname,id,weight) {
	var gradeBox = formname["GD_BOX"+id];
	var radioButton = formname["RADVAL"+id];
	var oldpts = formname["oldpts"+id].value;
	var pts = checkSolved(formname,id) == 'update' ? gradeBox.value : oldpts;
	gradeBox.value = pts;
	var resetbox = false;
	if (isNaN(pts) || pts < 0) {
	    alert("$alertmsg"+pts);
	    for (var i=0; i<radioButton.length; i++) {
		if (radioButton[i].checked) {
		    gradeBox.value = i;
		    resetbox = true;
		}
	    }
	    if (!resetbox) {
		formtextbox.value = "";
	    }
	    return;
	}

	if (pts > weight) {
	    var resp = confirm("You entered a value ("+pts+
			       ") greater than the weight for the part. Accept?");
	    if (resp == false) {
		gradeBox.value = oldpts;
		return;
	    }
	}

	for (var i=0; i<radioButton.length; i++) {
	    radioButton[i].checked=false;
	    if (pts == i && pts != "") {
		radioButton[i].checked=true;
	    }
	}
	updateSelect(formname,id);
	formname["stores"+id].value = "0";
    }

    function writeBox(formname,id,pts) {
	var gradeBox = formname["GD_BOX"+id];
	if (checkSolved(formname,id) == 'update') {
	    gradeBox.value = pts;
	} else {
	    var oldpts = formname["oldpts"+id].value;
	    gradeBox.value = oldpts;
	    var radioButton = formname["RADVAL"+id];
	    for (var i=0; i<radioButton.length; i++) {
		radioButton[i].checked=false;
		if (i == oldpts) {
		    radioButton[i].checked=true;
		}
	    }
	}
	formname["stores"+id].value = "0";
	updateSelect(formname,id);
	return;
    }

    function clearRadBox(formname,id) {
	if (checkSolved(formname,id) == 'noupdate') {
	    updateSelect(formname,id);
	    return;
	}
	gradeSelect = formname["GD_SEL"+id];
	for (var i=0; i<gradeSelect.length; i++) {
	    if (gradeSelect[i].selected) {
		var selectx=i;
	    }
	}
	var stores = formname["stores"+id];
	if (selectx == stores.value) { return };
	var gradeBox = formname["GD_BOX"+id];
	gradeBox.value = "";
	var radioButton = formname["RADVAL"+id];
	for (var i=0; i<radioButton.length; i++) {
	    radioButton[i].checked=false;
	}
	stores.value = selectx;
    }

    function checkSolved(formname,id) {
	if (formname["solved"+id].value == "correct_by_student" && formname.overRideScore.value == 'no') {
	    var reply = confirm("This problem has been graded correct by the computer. Do you want to change the score?");
	    if (!reply) {return "noupdate";}
	    formname.overRideScore.value = 'yes';
	}
	return "update";
    }

    function updateSelect(formname,id) {
	formname["GD_SEL"+id][0].selected = true;
	return;
    }

//=========== Check that a point is assigned for all the parts  ============
    function checksubmit(formname,val,total,parttot) {
	formname.gradeOpt.value = val;
	if (val == "Save & Next") {
	    for (i=0;i<=total;i++) {
		for (j=0;j<parttot;j++) {
		    var partid = formname["partid"+i+"_"+j].value;
		    if (formname["GD_SEL"+i+"_"+partid][0].selected) {
			var points = formname["GD_BOX"+i+"_"+partid].value;
			if (points == "") {
			    var name = formname["name"+i].value;
			    var studentID = (name != '' ? name : formname["unamedom"+i].value);
			    var resp = confirm("You did not assign a score for "+studentID+
					       ", part "+partid+". Continue?");
			    if (resp == false) {
				formname["GD_BOX"+i+"_"+partid].focus();
				return false;
			    }
			}
		    }
		}
	    }
	}
	formname.submit();
    }

//======= Check that a score is assigned for all the problems (page/sequence grading only) =========
    function checkSubmitPage(formname,total) {
	noscore = new Array(100);
	var ptr = 0;
	for (i=1;i<total;i++) {
	    var partid = formname["q_"+i].value;
	    if (formname["GD_SEL"+i+"_"+partid][0].selected) {
		var points = formname["GD_BOX"+i+"_"+partid].value;
		var status = formname["solved"+i+"_"+partid].value;
		if (points == "" && status != "correct_by_student") {
		    noscore[ptr] = i;
		    ptr++;
		}
	    }
	}
	if (ptr != 0) {
	    var sense = ptr == 1 ? ": " : "s: ";
	    var prolist = "";
	    if (ptr == 1) {
		prolist = noscore[0];
	    } else {
		var i = 0;
		while (i < ptr-1) {
		    prolist += noscore[i]+", ";
		    i++;
		}
		prolist += "and "+noscore[i];
	    }
	    var resp = confirm("You did not assign any score for the following problem"+sense+prolist+". Continue?");
	    if (resp == false) {
		return false;
	    }
	}

	formname.submit();
    }
SUBJAVASCRIPT
}

#--- javascript for grading message center
sub sub_grademessage_js {
    my $request = shift;
    my $iconpath = $request->dir_config('lonIconsURL');
    &commonJSfunctions($request);

    my $inner_js_msg_central= (<<INNERJS);
<script type="text/javascript">
    function checkInput() {
      opener.document.SCORE.msgsub.value = opener.checkEntities(document.msgcenter.msgsub.value);
      var nmsg   = opener.document.SCORE.savemsgN.value;
      var usrctr = document.msgcenter.usrctr.value;
      var newval = opener.document.SCORE["newmsg"+usrctr];
      newval.value = opener.checkEntities(document.msgcenter.newmsg.value);

      var msgchk = "";
      if (document.msgcenter.subchk.checked) {
         msgchk = "msgsub,";
      }
      var includemsg = 0;
      for (var i=1; i<=nmsg; i++) {
          var opnmsg = opener.document.SCORE["savemsg"+i];
          var frmmsg = document.msgcenter["msg"+i];
          opnmsg.value = opener.checkEntities(frmmsg.value);
          var showflg = opener.document.SCORE["shownOnce"+i];
          showflg.value = "1";
          var chkbox = document.msgcenter["msgn"+i];
          if (chkbox.checked) {
             msgchk += "savemsg"+i+",";
             includemsg = 1;
          }
      }
      if (document.msgcenter.newmsgchk.checked) {
         msgchk += "newmsg"+usrctr;
         includemsg = 1;
      }
      imgformname = opener.document.SCORE["mailicon"+usrctr];
      imgformname.src = "$iconpath/"+((includemsg) ? "mailto.gif" : "mailbkgrd.gif");
      var includemsg = opener.document.SCORE["includemsg"+usrctr];
      includemsg.value = msgchk;

      self.close()

    }
</script>
INNERJS

    my $start_page_msg_central =
        &Apache::loncommon::start_page('Message Central',$inner_js_msg_central,
				       {'js_ready'  => 1,
					'only_body' => 1,
					'bgcolor'   =>'#FFFFFF',});
    my $end_page_msg_central =
	&Apache::loncommon::end_page({'js_ready' => 1});


    my $docopen=&Apache::lonhtmlcommon::javascript_docopen();
    $docopen=~s/^document\.//;

    my %html_js_lt = &Apache::lonlocal::texthash(
                comp => 'Compose Message for: ',
                incl => 'Include',
                type => 'Type',
                subj => 'Subject',
                mesa => 'Message',
                new  => 'New',
                save => 'Save',
                canc => 'Cancel',
             );
    &html_escape(\%html_js_lt);
    &js_escape(\%html_js_lt);
    $request->print(&Apache::lonhtmlcommon::scripttag(<<SUBJAVASCRIPT));

//===================== Script to view submitted by ==================
  function viewSubmitter(submitter) {
    document.SCORE.refresh.value = "on";
    document.SCORE.NCT.value = "1";
    document.SCORE.unamedom0.value = submitter;
    document.SCORE.submit();
    return;
  }

//====================== Script for composing message ==============
   // preload images
   img1 = new Image();
   img1.src = "$iconpath/mailbkgrd.gif";
   img2 = new Image();
   img2.src = "$iconpath/mailto.gif";

  function msgCenter(msgform,usrctr,fullname) {
    var Nmsg  = msgform.savemsgN.value;
    savedMsgHeader(Nmsg,usrctr,fullname);
    var subject = msgform.msgsub.value;
    var msgchk = document.SCORE["includemsg"+usrctr].value;
    re = /msgsub/;
    var shwsel = "";
    if (re.test(msgchk)) { shwsel = "checked" }
    subject = (document.SCORE.shownSub.value == 0 ? checkEntities(subject) : subject);
    displaySubject(checkEntities(subject),shwsel);
    for (var i=1; i<=Nmsg; i++) {
	var testmsg = "savemsg"+i+",";
	re = new RegExp(testmsg,"g");
	shwsel = "";
	if (re.test(msgchk)) { shwsel = "checked" }
	var message = document.SCORE["savemsg"+i].value;
	message = (document.SCORE["shownOnce"+i].value == 0 ? checkEntities(message) : message);
	displaySavedMsg(i,message,shwsel); //I do not get it. w/o checkEntities on saved messages,
	                                   //any &lt; is already converted to <, etc. However, only once!!
    }
    newmsg = document.SCORE["newmsg"+usrctr].value;
    shwsel = "";
    re = /newmsg/;
    if (re.test(msgchk)) { shwsel = "checked" }
    newMsg(newmsg,shwsel);
    msgTail(); 
    return;
  }

  function checkEntities(strx) {
    if (strx.length == 0) return strx;
    var orgStr = ["&", "<", ">", '"']; 
    var newStr = ["&amp;", "&lt;", "&gt;", "&quot;"];
    var counter = 0;
    while (counter < 4) {
	strx = strReplace(strx,orgStr[counter],newStr[counter]);
	counter++;
    }
    return strx;
  }

  function strReplace(strx, orgStr, newStr) {
    return strx.split(orgStr).join(newStr);
  }

  function savedMsgHeader(Nmsg,usrctr,fullname) {
    var height = 70*Nmsg+250;
    if (height > 600) {
	height = 600;
    }
    var xpos = (screen.width-600)/2;
    xpos = (xpos < 0) ? '0' : xpos;
    var ypos = (screen.height-height)/2-30;
    ypos = (ypos < 0) ? '0' : ypos;

    pWin = window.open('', 'MessageCenter', 'resizable=yes,toolbar=no,location=no,scrollbars=yes,screenx='+xpos+',screeny='+ypos+',width=700,height='+height);
    pWin.focus();
    pDoc = pWin.document;
    pDoc.$docopen;
    pDoc.write('$start_page_msg_central');

    pDoc.write("<form action=\\"inactive\\" name=\\"msgcenter\\">");
    pDoc.write("<input value=\\""+usrctr+"\\" name=\\"usrctr\\" type=\\"hidden\\">");
    pDoc.write("<h1>&nbsp;$html_js_lt{'comp'}\"+fullname+\"<\\/h1>");

    pDoc.write('<table style="border:1px solid black;"><tr>');
    pDoc.write("<td><b>$html_js_lt{'incl'}<\\/b><\\/td><td><b>$html_js_lt{'type'}<\\/b><\\/td><td><b>$html_js_lt{'mesa'}<\\/td><\\/tr>");
}
    function displaySubject(msg,shwsel) {
    pDoc = pWin.document;
    pDoc.write("<tr>");
    pDoc.write("<td align=\\"center\\"><input name=\\"subchk\\" type=\\"checkbox\\"" +shwsel+"><\\/td>");
    pDoc.write("<td>$html_js_lt{'subj'}<\\/td>");
    pDoc.write("<td><input name=\\"msgsub\\" type=\\"text\\" value=\\""+msg+"\\"size=\\"40\\" maxlength=\\"80\\"><\\/td><\\/tr>");
}

  function displaySavedMsg(ctr,msg,shwsel) {
    pDoc = pWin.document;
    pDoc.write("<tr>");
    pDoc.write("<td align=\\"center\\"><input name=\\"msgn"+ctr+"\\" type=\\"checkbox\\"" +shwsel+"><\\/td>");
    pDoc.write("<td align=\\"center\\">"+ctr+"<\\/td>");
    pDoc.write("<td><textarea name=\\"msg"+ctr+"\\" cols=\\"60\\" rows=\\"3\\">"+msg+"<\\/textarea><\\/td><\\/tr>");
}

  function newMsg(newmsg,shwsel) {
    pDoc = pWin.document;
    pDoc.write("<tr>");
    pDoc.write("<td align=\\"center\\"><input name=\\"newmsgchk\\" type=\\"checkbox\\"" +shwsel+"><\\/td>");
    pDoc.write("<td align=\\"center\\">$html_js_lt{'new'}<\\/td>");
    pDoc.write("<td><textarea name=\\"newmsg\\" cols=\\"60\\" rows=\\"3\\" onchange=\\"javascript:this.form.newmsgchk.checked=true\\" >"+newmsg+"<\\/textarea><\\/td><\\/tr>");
}

  function msgTail() {
    pDoc = pWin.document;
    pDoc.write("<\\/td><\\/tr><\\/table>&nbsp;");
    pDoc.write("<input type=\\"button\\" value=\\"$html_js_lt{'save'}\\" onclick=\\"javascript:checkInput()\\">&nbsp;&nbsp;");
    pDoc.write("<input type=\\"button\\" value=\\"$html_js_lt{'canc'}\\" onclick=\\"self.close()\\"><br /><br />");
    pDoc.write("<\\/form>");
    pDoc.write('$end_page_msg_central');
    pDoc.close();
}

SUBJAVASCRIPT
}

#--- javascript for essay type problem --
sub sub_page_kw_js {
    my $request = shift;

    unless ($env{'form.compmsg'}) {
        &commonJSfunctions($request);
    }

    my $inner_js_highlight_central= (<<INNERJS);
<script type="text/javascript">
    function updateChoice(flag) {
      opener.document.SCORE.kwclr.value = opener.radioSelection(document.hlCenter.kwdclr);
      opener.document.SCORE.kwsize.value = opener.radioSelection(document.hlCenter.kwdsize);
      opener.document.SCORE.kwstyle.value = opener.radioSelection(document.hlCenter.kwdstyle);
      opener.document.SCORE.refresh.value = "on";
      if (opener.document.SCORE.keywords.value!=""){
         opener.document.SCORE.submit();
      }
      self.close()
    }
</script>
INNERJS

    my $start_page_highlight_central =
        &Apache::loncommon::start_page('Highlight Central',
                                       $inner_js_highlight_central,
                                       {'js_ready'  => 1,
                                        'only_body' => 1,
                                        'bgcolor'   =>'#FFFFFF',});
    my $end_page_highlight_central =
        &Apache::loncommon::end_page({'js_ready' => 1});

    my $docopen=&Apache::lonhtmlcommon::javascript_docopen();
    $docopen=~s/^document\.//;

    my %js_lt = &Apache::lonlocal::texthash(
                keyw => 'Keywords list, separated by a space. Add/delete to list if desired.',
                plse => 'Please select a word or group of words from document and then click this link.',
                adds => 'Add selection to keyword list? Edit if desired.',
                col1 => 'red',
                col2 => 'green',
                col3 => 'blue',
                siz1 => 'normal',
                siz2 => '+1',
                siz3 => '+2',
                sty1 => 'normal',
                sty2 => 'italic',
                sty3 => 'bold',
             );
    my %html_js_lt = &Apache::lonlocal::texthash(
                save => 'Save',
                canc => 'Cancel',
                kehi => 'Keyword Highlight Options',
                txtc => 'Text Color',
                font => 'Font Size',
                fnst => 'Font Style',
             );
    &js_escape(\%js_lt);
    &html_escape(\%html_js_lt);
    &js_escape(\%html_js_lt);
    $request->print(&Apache::lonhtmlcommon::scripttag(<<SUBJAVASCRIPT));

//===================== Show list of keywords ====================
  function keywords(formname) {
    var nret = prompt("$js_lt{'keyw'}",formname.keywords.value);
    if (nret==null) return;
    formname.keywords.value = nret;

    if (formname.keywords.value != "") {
        formname.refresh.value = "on";
        formname.submit();
    }
    return;
  }

//===================== Script to add keyword(s) ==================
  function getSel() {
    if (document.getSelection) txt = document.getSelection();
    else if (document.selection) txt = document.selection.createRange().text;
    else return;
    if (typeof(txt) != 'string') {
        txt = String(txt);
    }
    var cleantxt = txt.replace(new RegExp('([\\f\\n\\r\\t\\v ])+', 'g')," ");
    if (cleantxt=="") {
        alert("$js_lt{'plse'}");
        return;
    }
    var nret = prompt("$js_lt{'adds'}",cleantxt);
    if (nret==null) return;
    document.SCORE.keywords.value = document.SCORE.keywords.value+" "+nret;
    if (document.SCORE.keywords.value != "") {
        document.SCORE.refresh.value = "on";
        document.SCORE.submit();
    }
    return;
  }

//====================== Script for keyword highlight options ==============
  function kwhighlight() {
    var kwclr    = document.SCORE.kwclr.value;
    var kwsize   = document.SCORE.kwsize.value;
    var kwstyle  = document.SCORE.kwstyle.value;
    var redsel = "";
    var grnsel = "";
    var blusel = "";
    var txtcol1 = "$js_lt{'col1'}";
    var txtcol2 = "$js_lt{'col2'}";
    var txtcol3 = "$js_lt{'col3'}";
    var txtsiz1 = "$js_lt{'siz1'}";
    var txtsiz2 = "$js_lt{'siz2'}";
    var txtsiz3 = "$js_lt{'siz3'}";
    var txtsty1 = "$js_lt{'sty1'}";
    var txtsty2 = "$js_lt{'sty2'}";
    var txtsty3 = "$js_lt{'sty3'}";
    if (kwclr=="red")   {var redsel="checked='checked'"};
    if (kwclr=="green") {var grnsel="checked='checked'"};
    if (kwclr=="blue")  {var blusel="checked='checked'"};
    var sznsel = "";
    var sz1sel = "";
    var sz2sel = "";
    if (kwsize=="0")  {var sznsel="checked='checked'"};
    if (kwsize=="+1") {var sz1sel="checked='checked'"};
    if (kwsize=="+2") {var sz2sel="checked='checked'"};
    var synsel = "";
    var syisel = "";
    var sybsel = "";
    if (kwstyle=="")    {var synsel="checked='checked'"};
    if (kwstyle=="<i>") {var syisel="checked='checked'"};
    if (kwstyle=="<b>") {var sybsel="checked='checked'"};
    highlightCentral();
    highlightbody('red',txtcol1,redsel,'0',txtsiz1,sznsel,'',txtsty1,synsel);
    highlightbody('green',txtcol2,grnsel,'+1',txtsiz2,sz1sel,'<i>',txtsty2,syisel);
    highlightbody('blue',txtcol3,blusel,'+2',txtsiz3,sz2sel,'<b>',txtsty3,sybsel);
    highlightend();
    return;
  }

  function highlightCentral() {
//    if (window.hwdWin) window.hwdWin.close();
    var xpos = (screen.width-400)/2;
    xpos = (xpos < 0) ? '0' : xpos;
    var ypos = (screen.height-330)/2-30;
    ypos = (ypos < 0) ? '0' : ypos;

    hwdWin = window.open('', 'KeywordHighlightCentral', 'resizeable=yes,toolbar=no,location=no,scrollbars=no,width=400,height=300,screenx='+xpos+',screeny='+ypos);
    hwdWin.focus();
    var hDoc = hwdWin.document;
    hDoc.$docopen;
    hDoc.write('$start_page_highlight_central');
    hDoc.write("<form action=\\"inactive\\" name=\\"hlCenter\\">");
    hDoc.write("<h1>$html_js_lt{'kehi'}<\\/h1>");

    hDoc.write('<table border="0" width="100%"><tr style="background-color:#A1D676">');
    hDoc.write("<th>$html_js_lt{'txtc'}<\\/th><th>$html_js_lt{'font'}<\\/th><th>$html_js_lt{'fnst'}<\\/th><\\/tr>");
  }

  function highlightbody(clrval,clrtxt,clrsel,szval,sztxt,szsel,syval,sytxt,sysel) { 
    var hDoc = hwdWin.document;
    hDoc.write("<tr>");
    hDoc.write("<td align=\\"left\\">");
    hDoc.write("<input name=\\"kwdclr\\" type=\\"radio\\" value=\\""+clrval+"\\" "+clrsel+" \\/>&nbsp;"+clrtxt+"<\\/td>");
    hDoc.write("<td align=\\"left\\">");
    hDoc.write("<input name=\\"kwdsize\\" type=\\"radio\\" value=\\""+szval+"\\" "+szsel+" \\/>&nbsp;"+sztxt+"<\\/td>");
    hDoc.write("<td align=\\"left\\">");
    hDoc.write("<input name=\\"kwdstyle\\" type=\\"radio\\" value=\\""+syval+"\\" "+sysel+" \\/>&nbsp;"+sytxt+"<\\/td>");
    hDoc.write("<\\/tr>");
  }

  function highlightend() { 
    var hDoc = hwdWin.document;
    hDoc.write("<\\/table><br \\/>");
    hDoc.write("<input type=\\"button\\" value=\\"$html_js_lt{'save'}\\" onclick=\\"javascript:updateChoice(1)\\" \\/>&nbsp;&nbsp;");
    hDoc.write("<input type=\\"button\\" value=\\"$html_js_lt{'canc'}\\" onclick=\\"self.close()\\" \\/><br /><br />");
    hDoc.write("<\\/form>");
    hDoc.write('$end_page_highlight_central');
    hDoc.close();
  }

SUBJAVASCRIPT
}

sub get_increment {
    my $increment = $env{'form.increment'};
    if ($increment != 1 && $increment != .5 && $increment != .25 &&
        $increment != .1) {
        $increment = 1;
    }
    return $increment;
}

sub gradeBox_start {
    return (
        &Apache::loncommon::start_data_table()
       .&Apache::loncommon::start_data_table_header_row()
       .'<th>'.&mt('Part').'</th>'
       .'<th>'.&mt('Points').'</th>'
       .'<th>&nbsp;</th>'
       .'<th>'.&mt('Assign Grade').'</th>'
       .'<th>'.&mt('Weight').'</th>'
       .'<th>'.&mt('Grade Status').'</th>'
       .&Apache::loncommon::end_data_table_header_row()
    );
}

sub gradeBox_end {
    return (
        &Apache::loncommon::end_data_table()
    );
}
#--- displays the grading box, used in essay type problem and grading by page/sequence
sub gradeBox {
    my ($request,$symb,$uname,$udom,$counter,$partid,$record) = @_;
    my $checkIcon = '<img alt="'.&mt('Check Mark').
	'" src="'.&Apache::loncommon::lonhttpdurl($request->dir_config('lonIconsURL').'/check.gif').'" height="16" border="0" />';
    my $wgt    = &Apache::lonnet::EXT('resource.'.$partid.'.weight',$symb,$udom,$uname);
    my $wgtmsg = ($wgt > 0) ? &mt('(problem weight)') 
                           : '<span class="LC_info">'.&mt('problem weight assigned by computer').'</span>';
    $wgt       = ($wgt > 0 ? $wgt : '1');
    my $score  = ($$record{'resource.'.$partid.'.awarded'} eq '' ?
		  '' : &compute_points($$record{'resource.'.$partid.'.awarded'},$wgt));
    my $data_WGT='<input type="hidden" name="WGT'.$counter.'_'.$partid.'" value="'.$wgt.'" />'."\n";
    my $display_part= &get_display_part($partid,$symb);
    my %last_resets = &get_last_resets($symb,$env{'request.course.id'},
				       [$partid]);
    my $aggtries = $$record{'resource.'.$partid.'.tries'};
    if ($last_resets{$partid}) {
        $aggtries = &get_num_tries($record,$last_resets{$partid},$partid);
    }
    my $result=&Apache::loncommon::start_data_table_row();
    my $ctr = 0;
    my $thisweight = 0;
    my $increment = &get_increment();

    my $radio.='<table border="0"><tr>'."\n";  # display radio buttons in a nice table 10 across
    while ($thisweight<=$wgt) {
	$radio.= '<td><span class="LC_nobreak"><label><input type="radio" name="RADVAL'.$counter.'_'.$partid.'" '.
        'onclick="javascript:writeBox(this.form,\''.$counter.'_'.$partid.'\','.
	    $thisweight.')" value="'.$thisweight.'" '.
	    ($score eq $thisweight ? 'checked="checked"':'').' /> '.$thisweight."</label></span></td>\n";
	$radio.=(($ctr+1)%10 == 0 ? '</tr><tr>' : '');
        $thisweight += $increment;
	$ctr++;
    }
    $radio.='</tr></table>';

    my $line.='<input type="text" name="GD_BOX'.$counter.'_'.$partid.'"'.
	($score ne ''? ' value = "'.$score.'"':'').' size="4" '.
	'onchange="javascript:updateRadio(this.form,\''.$counter.'_'.$partid.'\','.
	$wgt.')" /></td>'."\n";
    $line.='<td>/'.$wgt.' '.$wgtmsg.
	($$record{'resource.'.$partid.'.solved'} eq 'correct_by_student' ? '&nbsp;'.$checkIcon : '').
	' </td>'."\n";
    $line.='<td><select name="GD_SEL'.$counter.'_'.$partid.'" '.
	'onchange="javascript:clearRadBox(this.form,\''.$counter.'_'.$partid.'\')" >'."\n";
    if ($$record{'resource.'.$partid.'.solved'} eq 'excused') {
	$line.='<option></option>'.
	    '<option value="excused" selected="selected">'.&mt('excused').'</option>';
    } else {
	$line.='<option selected="selected"></option>'.
	    '<option value="excused" >'.&mt('excused').'</option>';
    }
    $line.='<option value="reset status">'.&mt('reset status').'</option></select>'."\n";


    $result .= 
	    '<td>'.$data_WGT.$display_part.'</td><td>'.$radio.'</td><td>'.&mt('or').'</td><td>'.$line.'</td>';
    $result.=&Apache::loncommon::end_data_table_row();
    $result.=&Apache::loncommon::start_data_table_row().'<td colspan="6">';
    $result.='<input type="hidden" name="stores'.$counter.'_'.$partid.'" value="" />'."\n".
	'<input type="hidden" name="oldpts'.$counter.'_'.$partid.'" value="'.$score.'" />'."\n".
	'<input type="hidden" name="solved'.$counter.'_'.$partid.'" value="'.
	$$record{'resource.'.$partid.'.solved'}.'" />'."\n".
        '<input type="hidden" name="totaltries'.$counter.'_'.$partid.'" value="'.
        $$record{'resource.'.$partid.'.tries'}.'" />'."\n".
        '<input type="hidden" name="aggtries'.$counter.'_'.$partid.'" value="'.
        $aggtries.'" />'."\n";
    my $res_error;
    $result.=&handback_box($symb,$uname,$udom,$counter,$partid,$record,\$res_error);
    $result.='</td>'.&Apache::loncommon::end_data_table_row();
    if ($res_error) {
        return &navmap_errormsg();
    }
    return $result;
}

sub handback_box {
    my ($symb,$uname,$udom,$counter,$partid,$record,$res_error_pointer) = @_;
    my ($partlist,$handgrade,$responseType,$numresp,$numessay) = &response_type($symb,$res_error_pointer);
    return unless ($numessay);
    my (@respids);
    my @part_response_id = &flatten_responseType($responseType);
    foreach my $part_response_id (@part_response_id) {
    	my ($part,$resp) = @{ $part_response_id };
        if ($part eq $partid) {
            push(@respids,$resp);
        }
    }
    my $result;
    foreach my $respid (@respids) {
	my $prefix = $counter.'_'.$partid.'_'.$respid.'_';
	my $files=&get_submitted_files($udom,$uname,$partid,$respid,$record);
	next if (!@$files);
	my $file_counter = 0;
	foreach my $file (@$files) {
	    if ($file =~ /\/portfolio\//) {
                $file_counter++;
    	        my ($file_path, $file_disp) = ($file =~ m|(.+/)(.+)$|);
    	        my ($name,$version,$ext) = &file_name_version_ext($file_disp);
    	        $file_disp = "$name.$ext";
    	        $file = $file_path.$file_disp;
    	        $result.=&mt('Return commented version of [_1] to student.',
    			 '<span class="LC_filename">'.$file_disp.'</span>');
    	        $result.='<input type="file"   name="'.$prefix.'returndoc'.$file_counter.'" />'."\n";
    	        $result.='<input type="hidden" name="'.$prefix.'origdoc'.$file_counter.'" value="'.$file.'" /><br />'."\n";
	    }
	}
        if ($file_counter) {
            $result .= '<input type="hidden" name="'.$prefix.'countreturndoc" value="'.$file_counter.'" />'."\n".
                       '<span class="LC_info">'.
                       '('.&mt('File(s) will be uploaded when you click on Save &amp; Next below.',$file_counter).')</span><br /><br />';
        }
    }
    return $result;    
}

sub show_problem {
    my ($request,$symb,$uname,$udom,$removeform,$viewon,$mode,$form) = @_;
    my $rendered;
    my %form = ((ref($form) eq 'HASH')? %{$form} : ());
    &Apache::lonxml::remember_problem_counter();
    if ($mode eq 'both' or $mode eq 'text') {
	$rendered=&Apache::loncommon::get_student_view($symb,$uname,$udom,
						       $env{'request.course.id'},
						       undef,\%form);
    }
    if ($removeform) {
	$rendered=~s|<form(.*?)>||g;
	$rendered=~s|</form>||g;
	$rendered=~s|(<input[^>]*name\s*=\s*"?)(\w+)("?)|$1would_have_been_$2$3|g;
    }
    my $companswer;
    if ($mode eq 'both' or $mode eq 'answer') {
	&Apache::lonxml::restore_problem_counter();
	$companswer=
	    &Apache::loncommon::get_student_answers($symb,$uname,$udom,
						    $env{'request.course.id'},
						    %form);
    }
    if ($removeform) {
	$companswer=~s|<form(.*?)>||g;
	$companswer=~s|</form>||g;
	$companswer=~s|name="submit"|name="would_have_been_submit"|g;
    }
    my $renderheading = &mt('View of the problem');
    my $answerheading = &mt('Correct answer');
    if (($uname ne $env{'user.name'}) || ($udom ne $env{'user.domain'})) {
        my $stu_fullname = $env{'form.fullname'};
        if ($stu_fullname eq '') {
            $stu_fullname = &Apache::loncommon::plainname($uname,$udom,'lastname');
        }
        my $forwhom = &nameUserString(undef,$stu_fullname,$uname,$udom);
        if ($forwhom ne '') {
            $renderheading = &mt('View of the problem for[_1]',$forwhom);
            $answerheading = &mt('Correct answer for[_1]',$forwhom);
        }
    }
    $rendered=
        '<div class="LC_Box">'
       .'<h3 class="LC_hcell">'.$renderheading.'</h3>'
       .$rendered
       .'</div>';
    $companswer=
        '<div class="LC_Box">'
       .'<h3 class="LC_hcell">'.$answerheading.'</h3>'
       .$companswer
       .'</div>';
    my $result;
    if ($mode eq 'both') {
        $result=$rendered.$companswer;
    } elsif ($mode eq 'text') {
        $result=$rendered;
    } elsif ($mode eq 'answer') {
        $result=$companswer;
    }
    return $result;
}

sub files_exist {
    my ($r, $symb) = @_;
    my @students = &Apache::loncommon::get_env_multiple('form.stuinfo');
    foreach my $student (@students) {
        my ($uname,$udom,$fullname) = split(/:/,$student);
        my %record = &Apache::lonnet::restore($symb,$env{'request.course.id'},
					      $udom,$uname);
        my ($string)= &get_last_submission(\%record);
        foreach my $submission (@$string) {
            my ($partid,$respid) =
		($submission =~ /^resource\.([^\.]*)\.([^\.]*)\.submission/);
            my $files=&get_submitted_files($udom,$uname,$partid,$respid,
					   \%record);
            return 1 if (@$files);
        }
    }
    return 0;
}

sub download_all_link {
    my ($r,$symb) = @_;
    unless (&files_exist($r, $symb)) {
        $r->print(&mt('There are currently no submitted documents.'));
        return;
    }
    my $all_students = 
	join("\n", &Apache::loncommon::get_env_multiple('form.stuinfo'));

    my $parts =
	join("\n",&Apache::loncommon::get_env_multiple('form.vPart'));

    my $identifier = &Apache::loncommon::get_cgi_id();
    &Apache::lonnet::appenv({'cgi.'.$identifier.'.students' => $all_students,
                             'cgi.'.$identifier.'.symb' => $symb,
                             'cgi.'.$identifier.'.parts' => $parts,});
    $r->print('<a href="/cgi-bin/multidownload.pl?'.$identifier.'">'.
	      &mt('Download All Submitted Documents').'</a>');
    return;
}

sub submit_download_link {
    my ($request,$symb) = @_;
    if (!$symb) { return ''; }
    my $res_error;
    my ($partlist,$handgrade,$responseType,$numresp,$numessay,$numdropbox) =
        &response_type($symb,\$res_error);
    if ($res_error) {
        $request->print(&mt('An error occurred retrieving response types'));
        return;
    }
    unless ($numessay) {
        $request->print(&mt('No essayresponse items found'));
        return;
    }
    my @chosenparts = &Apache::loncommon::get_env_multiple('form.vPart');
    if (@chosenparts) {
        $request->print(&showResourceInfo($symb,$partlist,$responseType,
                                          undef,undef,1));
    }
    if ($numessay) {
        my $submitonly= $env{'form.submitonly'} eq '' ? 'all' : $env{'form.submitonly'};
        my $getsec    = $env{'form.section'} eq '' ? 'all' : $env{'form.section'};
        my $getgroup  = $env{'form.group'} eq '' ? 'all' : $env{'form.group'};
        (undef,undef,my $fullname) = &getclasslist($getsec,1,$getgroup,$symb,$submitonly,1);
        if (ref($fullname) eq 'HASH') {
            my @students = map { $_.':'.$fullname->{$_} } (keys(%{$fullname}));
            if (@students) {
                @{$env{'form.stuinfo'}} = @students;
                if ($numdropbox) {
                    &download_all_link($request,$symb);
                } else {
                    $request->print(&mt('No essayrespose items with dropbox found'));
                }
# FIXME Need a mechanism to download essays, i.e., if $numessay > $numdropbox
# Needs to omit user's identity if resource instance is for an anonymous survey.
            } else {
                $request->print(&mt('No students match the criteria you selected'));
            }
        } else {
            $request->print(&mt('Could not retrieve student information'));
        }
    } else {
        $request->print(&mt('No essayresponse items found'));
    }
    return;
}

sub build_section_inputs {
    my $section_inputs;
    if ($env{'form.section'} eq '') {
        $section_inputs .= '<input type="hidden" name="section" value="all" />'."\n";
    } else {
        my @sections = &Apache::loncommon::get_env_multiple('form.section');
        foreach my $section (@sections) {
            $section_inputs .= '<input type="hidden" name="section" value="'.$section.'" />'."\n";
        }
    }
    return $section_inputs;
}

# --------------------------- show submissions of a student, option to grade 
sub submission {
    my ($request,$counter,$total,$symb,$divforres,$calledby) = @_;
    my ($uname,$udom)     = ($env{'form.student'},$env{'form.userdom'});
    $udom = ($udom eq '' ? $env{'user.domain'} : $udom); #has form.userdom changed for a student?
    my $usec = &Apache::lonnet::getsection($udom,$uname,$env{'request.course.id'});
    $env{'form.fullname'} = &Apache::loncommon::plainname($uname,$udom,'lastname') if $env{'form.fullname'} eq '';

    if ($symb eq '') { $request->print("Unable to handle ambiguous references:."); return ''; }
    my $probtitle=&Apache::lonnet::gettitle($symb);
    my ($essayurl,%coursedesc_by_cid);

    if (!&canview($usec)) {
        $request->print(
            '<span class="LC_warning">'.
            &mt('Unable to view requested student.').
            ' '.&mt('([_1] in section [_2] in course id [_3])',
                        $uname.':'.$udom,$usec,$env{'request.course.id'}).
            '</span>');
	return;
    }

    my $res_error;
    my ($partlist,$handgrade,$responseType,$numresp,$numessay) =
        &response_type($symb,\$res_error);
    if ($res_error) {
        $request->print(&navmap_errormsg());
        return;
    }

    if (!$env{'form.lastSub'}) { $env{'form.lastSub'} = 'datesub'; }
    if (!$env{'form.vProb'}) { $env{'form.vProb'} = 'yes'; }
    if (!$env{'form.vAns'}) { $env{'form.vAns'} = 'yes'; }
    if (($numessay) && ($calledby eq 'submission') && (!exists($env{'form.compmsg'}))) {
        $env{'form.compmsg'} = 1;
    }
    my $last = ($env{'form.lastSub'} eq 'last' ? 'last' : '');
    my $checkIcon = '<img alt="'.&mt('Check Mark').
	'" src="'.$request->dir_config('lonIconsURL').
	'/check.gif" height="16" border="0" />';

    # header info
    if ($counter == 0) {
        my @chosenparts = &Apache::loncommon::get_env_multiple('form.vPart');
        if (@chosenparts) {
            $request->print(&showResourceInfo($symb,$partlist,$responseType,'gradesub'));
        } elsif ($divforres) {
            $request->print('<div style="padding:0;clear:both;margin:0;border:0"></div>');
        } else {
            $request->print('<br clear="all" />');
        }
	&sub_page_js($request);
        &sub_grademessage_js($request) if ($env{'form.compmsg'});
	&sub_page_kw_js($request) if ($numessay);

	# option to display problem, only once else it cause problems 
        # with the form later since the problem has a form.
	if ($env{'form.vProb'} eq 'yes' or $env{'form.vAns'} eq 'yes') {
	    my $mode;
	    if ($env{'form.vProb'} eq 'yes' && $env{'form.vAns'} eq 'yes') {
		$mode='both';
	    } elsif ($env{'form.vProb'} eq 'yes') {
		$mode='text';
	    } elsif ($env{'form.vAns'} eq 'yes') {
		$mode='answer';
	    }
	    &Apache::lonxml::clear_problem_counter();
	    $request->print(&show_problem($request,$symb,$uname,$udom,0,1,$mode));
	}

	my %keyhash = ();
	if (($env{'form.kwclr'} eq '' && $numessay) || ($env{'form.compmsg'})) {
	    %keyhash = &Apache::lonnet::dump('nohist_handgrade',
					     $env{'course.'.$env{'request.course.id'}.'.domain'},
					     $env{'course.'.$env{'request.course.id'}.'.num'});
	}
	# kwclr is the only variable that is guaranteed not to be blank
	# if this subroutine has been called once.
	if ($env{'form.kwclr'} eq '' && $numessay) {
	    my $loginuser = $env{'user.name'}.':'.$env{'user.domain'};
	    $env{'form.keywords'} = $keyhash{$symb.'_keywords'} ne '' ? $keyhash{$symb.'_keywords'} : '';
	    $env{'form.kwclr'}    = $keyhash{$loginuser.'_kwclr'} ne '' ? $keyhash{$loginuser.'_kwclr'} : 'red';
	    $env{'form.kwsize'}   = $keyhash{$loginuser.'_kwsize'} ne '' ? $keyhash{$loginuser.'_kwsize'} : '0';
	    $env{'form.kwstyle'}  = $keyhash{$loginuser.'_kwstyle'} ne '' ? $keyhash{$loginuser.'_kwstyle'} : '';
	}
	if ($env{'form.compmsg'}) {
	    $env{'form.msgsub'}   = $keyhash{$symb.'_subject'} ne '' ?
		$keyhash{$symb.'_subject'} : $probtitle;
	    $env{'form.savemsgN'} = $keyhash{$symb.'_savemsgN'} ne '' ? $keyhash{$symb.'_savemsgN'} : '0';
	}

	my $overRideScore = $env{'form.overRideScore'} eq '' ? 'no' : $env{'form.overRideScore'};
	my $stu_status = join(':',&Apache::loncommon::get_env_multiple('form.Status'));
	$request->print('<form action="/adm/grades" method="post" name="SCORE" enctype="multipart/form-data">'."\n".
			'<input type="hidden" name="command"    value="handgrade" />'."\n".
			'<input type="hidden" name="Status"     value="'.$stu_status.'" />'."\n".
			'<input type="hidden" name="overRideScore" value="'.$overRideScore.'" />'."\n".
			'<input type="hidden" name="refresh"    value="off" />'."\n".
			'<input type="hidden" name="studentNo"  value="" />'."\n".
			'<input type="hidden" name="gradeOpt"   value="" />'."\n".
			'<input type="hidden" name="symb"       value="'.&Apache::lonenc::check_encrypt($symb).'" />'."\n".
			'<input type="hidden" name="vProb"      value="'.$env{'form.vProb'}.'" />'."\n".
			'<input type="hidden" name="vAns"       value="'.$env{'form.vAns'}.'" />'."\n".
			'<input type="hidden" name="lastSub"    value="'.$env{'form.lastSub'}.'" />'."\n".
			'<input type="hidden" name="compmsg"    value="'.$env{'form.compmsg'}.'" />'."\n".
			&build_section_inputs().
			'<input type="hidden" name="submitonly" value="'.$env{'form.submitonly'}.'" />'."\n".
			'<input type="hidden" name="NCT"'.
			' value="'.($env{'form.NTSTU'} ne '' ? $env{'form.NTSTU'} : $total+1).'" />'."\n");
	if ($env{'form.compmsg'}) {
	    $request->print('<input type="hidden" name="msgsub"   value="'.$env{'form.msgsub'}.'" />'."\n".
			    '<input type="hidden" name="shownSub" value="0" />'."\n".
			    '<input type="hidden" name="savemsgN" value="'.$env{'form.savemsgN'}.'" />'."\n");
	}
	if ($numessay) {
	    $request->print('<input type="hidden" name="keywords" value="'.$env{'form.keywords'}.'" />'."\n".
			    '<input type="hidden" name="kwclr"    value="'.$env{'form.kwclr'}.'" />'."\n".
			    '<input type="hidden" name="kwsize"   value="'.$env{'form.kwsize'}.'" />'."\n".
			    '<input type="hidden" name="kwstyle"  value="'.$env{'form.kwstyle'}.'" />'."\n");
	}

	my ($cts,$prnmsg) = (1,'');
	while ($cts <= $env{'form.savemsgN'}) {
	    $prnmsg.='<input type="hidden" name="savemsg'.$cts.'" value="'.
		(!exists($keyhash{$symb.'_savemsg'.$cts}) ? 
		 &Apache::lonfeedback::clear_out_html($env{'form.savemsg'.$cts}) :
		 &Apache::lonfeedback::clear_out_html($keyhash{$symb.'_savemsg'.$cts})).
		'" />'."\n".
		'<input type="hidden" name="shownOnce'.$cts.'" value="0" />'."\n";
	    $cts++;
	}
	$request->print($prnmsg);

	if ($numessay) {

            my %lt = &Apache::lonlocal::texthash(
                          keyh => 'Keyword Highlighting for Essays',
                          keyw => 'Keyword Options',
                          list => 'List',
                          past => 'Paste Selection to List',
                          high => 'Highlight Attribute',
                     );
#
# Print out the keyword options line
#
	    $request->print(
                '<div class="LC_columnSection">'
               .'<fieldset><legend>'.$lt{'keyh'}.'</legend>'
               .&Apache::lonhtmlcommon::funclist_from_array(
                    ['<a href="javascript:keywords(document.SCORE);" target="_self">'.$lt{'list'}.'</a>',
                     '<a href="#" onmousedown="javascript:getSel(); return false"
 class="page">'.$lt{'past'}.'</a>',
                     '<a href="javascript:kwhighlight();" target="_self">'.$lt{'high'}.'</a>'],
                    {legend => $lt{'keyw'}})
               .'</fieldset></div>'
            );

#
# Load the other essays for similarity check
#
            (undef,undef,$essayurl) = &Apache::lonnet::decode_symb($symb);
            if ($essayurl eq 'lib/templates/simpleproblem.problem') {
                my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
                my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
                if ($cdom ne '' && $cnum ne '') {
                    my ($map,$id,$res) = &Apache::lonnet::decode_symb($symb);
                    if ($map =~ m{^\Quploaded/$cdom/$cnum/\E(default(?:|_\d+)\.(?:sequence|page))$}) {
                        my $apath = $1.'_'.$id;
                        $apath=~s/\W/\_/gs;
                        &init_old_essays($symb,$apath,$cdom,$cnum);
                    }
                }
            } else {
	        my ($adom,$aname,$apath)=($essayurl=~/^($LONCAPA::domain_re)\/($LONCAPA::username_re)\/(.*)$/);
	        $apath=&escape($apath);
	        $apath=~s/\W/\_/gs;
                &init_old_essays($symb,$apath,$adom,$aname);
            }
        }
    }

# This is where output for one specific student would start
    my $add_class = ($counter%2) ? ' LC_grade_show_user_odd_row' : '';
    $request->print(
        "\n\n"
       .'<div class="LC_grade_show_user'.$add_class.'">'
       .'<h2>'.&nameUserString(undef,$env{'form.fullname'},$uname,$udom).'</h2>'
       ."\n"
    );

    # Show additional functions if allowed
    if ($perm{'vgr'}) {
        $request->print(
            &Apache::loncommon::track_student_link(
                'View recent activity',
                $uname,$udom,'check')
           .' '
        );
    }
    if ($perm{'opa'}) {
        $request->print(
            &Apache::loncommon::pprmlink(
                &mt('Set/Change parameters'),
                $uname,$udom,$symb,'check'));
    }

    # Show Problem
    if ($env{'form.vProb'} eq 'all' or $env{'form.vAns'} eq 'all') {
	my $mode;
	if ($env{'form.vProb'} eq 'all' && $env{'form.vAns'} eq 'all') {
	    $mode='both';
	} elsif ($env{'form.vProb'} eq 'all' ) {
	    $mode='text';
	} elsif ($env{'form.vAns'} eq 'all') {
	    $mode='answer';
	}
	&Apache::lonxml::clear_problem_counter();
	$request->print(&show_problem($request,$symb,$uname,$udom,1,1,$mode,{'request.prefix' => 'ctr'.$counter}));
    }

    my %record = &Apache::lonnet::restore($symb,$env{'request.course.id'},$udom,$uname);

    # Display student info
    $request->print(($counter == 0 ? '' : '<br />'));

    my $result='<div class="LC_Box">'
              .'<h3 class="LC_hcell">'.&mt('Submissions').'</h3>';
    $result.='<input type="hidden" name="name'.$counter.
             '" value="'.$env{'form.fullname'}.'" />'."\n";
    if ($numresp > $numessay) {
        $result.='<p class="LC_info">'
                .&mt('Part(s) graded correct by the computer is marked with a [_1] symbol.',$checkIcon)
                ."</p>\n";
    }

    # If any part of the problem is an essayresponse, then check for collaborators
    my $fullname;
    my $col_fullnames = [];
    if ($numessay) {
	(my $sub_result,$fullname,$col_fullnames)=
	    &check_collaborators($symb,$uname,$udom,\%record,$handgrade,
				 $counter);
	$result.=$sub_result;
    }
    $request->print($result."\n");

    # print student answer/submission
    # Options are (1) Last submission only
    #             (2) Last submission (with detailed information for that submission)
    #             (3) All transactions (by date)
    #             (4) The whole record (with detailed information for all transactions)

    my ($string,$timestamp,$lastgradetime,$lastsubmittime) =
        &get_last_submission(\%record);

    my $lastsubonly;

    if ($timestamp eq '') {
        $lastsubonly.='<div class="LC_grade_submissions_body">'.$string->[0].'</div>'; 
    } else {
        my ($shownsubmdate,$showngradedate);
        if ($lastsubmittime && $lastgradetime) {
            $shownsubmdate = &Apache::lonlocal::locallocaltime($lastsubmittime);
            if ($lastgradetime > $lastsubmittime) {
                 $showngradedate = &Apache::lonlocal::locallocaltime($lastgradetime);
             }
        } else {
            $shownsubmdate = $timestamp;
        }
        $lastsubonly =
            '<div class="LC_grade_submissions_body">'
           .'<b>'.&mt('Date Submitted:').'</b> '.$shownsubmdate."\n";
        if ($showngradedate) {
            $lastsubonly .= '<br /><b>'.&mt('Date Graded:').'</b> '.$showngradedate."\n";
        }

	my %seenparts;
	my @part_response_id = &flatten_responseType($responseType);
	foreach my $part (@part_response_id) {
	    my ($partid,$respid) = @{ $part };
	    my $display_part=&get_display_part($partid,$symb);
	    if ($env{"form.$uname:$udom:$partid:submitted_by"}) {
		if (exists($seenparts{$partid})) { next; }
		$seenparts{$partid}=1;
                $request->print(
                    '<b>'.&mt('Part: [_1]',$display_part).'</b>'.
                    ' <b>'.&mt('Collaborative submission by: [_1]',
                               '<a href="javascript:viewSubmitter(\''.
                               $env{"form.$uname:$udom:$partid:submitted_by"}.
                               '\');" target="_self">'.
                               $$fullname{$env{"form.$uname:$udom:$partid:submitted_by"}}.'</a>').
                    '<br />');
		next;
	    }
	    my $responsetype = $responseType->{$partid}->{$respid};
	    if (!exists($record{"resource.$partid.$respid.submission"})) {
                $lastsubonly.="\n".'<div class="LC_grade_submission_part">'.
                    '<b>'.&mt('Part: [_1]',$display_part).'</b>'.
                    ' <span class="LC_internal_info">'.
                    '('.&mt('Response ID: [_1]',$respid).')'.
                    '</span>&nbsp; &nbsp;'.
	            '<span class="LC_warning">'.&mt('Nothing submitted - no attempts.').'</span><br /><br /></div>';
		next;
	    }
	    foreach my $submission (@$string) {
		my ($partid,$respid) = ($submission =~ /^resource\.([^\.]*)\.([^\.]*)\.submission/);
		if (join('_',@{$part}) ne ($partid.'_'.$respid)) { next; }
		my ($ressub,$hide,$draft,$subval) = split(/:/,$submission,4);
		# Similarity check
                my $similar='';
                my ($type,$trial,$rndseed);
                if ($hide eq 'rand') {
                    $type = 'randomizetry';
                    $trial = $record{"resource.$partid.tries"};
                    $rndseed = $record{"resource.$partid.rndseed"};
                }
		if ($env{'form.checkPlag'}) {
		    my ($oname,$odom,$ocrsid,$oessay,$osim)=
		        &most_similar($uname,$udom,$symb,$subval);
		    if ($osim) {
		        $osim=int($osim*100.0);
                        if ($hide eq 'anon') {
                            $similar='<hr /><span class="LC_warning">'.&mt("Essay was found to be similar to another essay submitted for this assignment.").'<br />'.
                                     &mt('As the current submission is for an anonymous survey, no other details are available.').'</span><hr />';
                        } else {
			    $similar='<hr />';
                            if ($essayurl eq 'lib/templates/simpleproblem.problem') {
                                $similar .= '<h3><span class="LC_warning">'.
                                            &mt('Essay is [_1]% similar to an essay by [_2]',
                                                $osim,
                                                &Apache::loncommon::plainname($oname,$odom).' ('.$oname.':'.$odom.')').
                                            '</span></h3>';
                            } elsif ($ocrsid ne '') {
                                my %old_course_desc;
                                if (ref($coursedesc_by_cid{$ocrsid}) eq 'HASH') {
                                    %old_course_desc = %{$coursedesc_by_cid{$ocrsid}};
                                } else {
                                    my $args;
                                    if ($ocrsid ne $env{'request.course.id'}) {
                                        $args = {'one_time' => 1};
                                    }
                                    %old_course_desc =
                                        &Apache::lonnet::coursedescription($ocrsid,$args);
                                    $coursedesc_by_cid{$ocrsid} = \%old_course_desc;
                                }
                                $similar .=
                                    '<h3><span class="LC_warning">'.
				    &mt('Essay is [_1]% similar to an essay by [_2] in course [_3] (course id [_4]:[_5])',
				        $osim,
				        &Apache::loncommon::plainname($oname,$odom).' ('.$oname.':'.$odom.')',
				        $old_course_desc{'description'},
				        $old_course_desc{'num'},
				        $old_course_desc{'domain'}).
				    '</span></h3>';
                            } else {
                                $similar .=
                                    '<h3><span class="LC_warning">'.
                                    &mt('Essay is [_1]% similar to an essay by [_2] in an unknown course',
                                        $osim,
                                        &Apache::loncommon::plainname($oname,$odom).' ('.$oname.':'.$odom.')').
                                    '</span></h3>';
                            }
                            $similar .= '<blockquote><i>'.
                                        &keywords_highlight($oessay).
                                        '</i></blockquote><hr />';
		        }
                    }
                }
		my $order=&get_order($partid,$respid,$symb,$uname,$udom,
                                     undef,$type,$trial,$rndseed);
                if (($env{'form.lastSub'} eq 'lastonly') ||
                    ($env{'form.lastSub'} eq 'datesub')  ||
                    ($env{'form.lastSub'} =~ /^(last|all)$/)) {
		    my $display_part=&get_display_part($partid,$symb);
                    $lastsubonly.='<div class="LC_grade_submission_part">'.
                        '<b>'.&mt('Part: [_1]',$display_part).'</b>'.
                        ' <span class="LC_internal_info">'.
                        '('.&mt('Response ID: [_1]',$respid).')'.
                        '</span>&nbsp; &nbsp;';
		    my $files=&get_submitted_files($udom,$uname,$partid,$respid,\%record);
		    if (@$files) {
                        if ($hide eq 'anon') {
                            $lastsubonly.='<br />'.&mt('[quant,_1,file] uploaded to this anonymous survey',scalar(@{$files}));
                        } else {
                            $lastsubonly.='<br /><br />'.'<b>'.&mt('Submitted Files:').'</b>'
                                         .'<br /><span class="LC_warning">';
                            if(@$files == 1) {
                                $lastsubonly .= &mt('Like all files provided by users, this file may contain viruses!');
                            } else {
                                $lastsubonly .= &mt('Like all files provided by users, these files may contain viruses!');
                            }
                            $lastsubonly .= '</span>';
                            foreach my $file (@$files) {
                                &Apache::lonnet::allowuploaded('/adm/grades',$file);
                                $lastsubonly.='<br /><a href="'.$file.'?rawmode=1" target="lonGRDs"><img src="'.&Apache::loncommon::icon($file).'" border="0" alt="" /> '.$file.'</a>';
                            }
                        }
			$lastsubonly.='<br />';
		    }
                    if ($hide eq 'anon') {
                        $lastsubonly.='<br /><b>'.&mt('Anonymous Survey').'</b>'; 
                    } else {
                        $lastsubonly.='<br /><b>'.&mt('Submitted Answer:').' </b>';
                        if ($draft) {
                            $lastsubonly.= ' <span class="LC_warning">'.&mt('Draft Copy').'</span>';
                        }
                        $subval =
			    &cleanRecord($subval,$responsetype,$symb,$partid,
					 $respid,\%record,$order,undef,$uname,$udom,$type,$trial,$rndseed);
                        if ($responsetype eq 'essay') {
                            $subval =~ s{\n}{<br />}g;
                        }
                        $lastsubonly.=$subval."\n";
                    }
                    if ($similar) {$lastsubonly.="<br /><br />$similar\n";}
		    $lastsubonly.='</div>';
		}
	    }
	}
	$lastsubonly.='</div>'."\n"; # End: LC_grade_submissions_body
    }
    $request->print($lastsubonly);
    if ($env{'form.lastSub'} eq 'datesub') {
        my ($parts,$handgrade,$responseType) = &response_type($symb,\$res_error);
	$request->print(&displaySubByDates($symb,\%record,$parts,$responseType,$checkIcon,$uname,$udom));
    }
    if ($env{'form.lastSub'} =~ /^(last|all)$/) {
        my $identifier = (&canmodify($usec)? $counter : '');
	$request->print(&Apache::loncommon::get_previous_attempt($symb,$uname,$udom,
								 $env{'request.course.id'},
								 $last,'.submission',
								 'Apache::grades::keywords_highlight',
                                                                 $usec,$identifier));
    }
    $request->print('<input type="hidden" name="unamedom'.$counter.'" value="'.$uname.':'
	.$udom.'" />'."\n");
    # return if view submission with no grading option
    if (!&canmodify($usec)) {
        $request->print('<p><span class="LC_warning">'.&mt('No grading privileges').'</span></p></div>');
        return;
    } else {
	$request->print('</div>'."\n");
    }

    # grading message center

    if ($env{'form.compmsg'}) {
        my $result='<div class="LC_Box">'.
                   '<h3 class="LC_hcell">'.&mt('Send Message').'</h3>'.
                   '<div class="LC_grade_message_center_body">';
        my ($lastname,$givenn) = split(/,/,$env{'form.fullname'});
        my $msgfor = $givenn.' '.$lastname;
        if (scalar(@$col_fullnames) > 0) {
            my $lastone = pop(@$col_fullnames);
            $msgfor .= ', '.(join ', ',@$col_fullnames).' and '.$lastone.'.';
        }
        $msgfor =~ s/\'/\\'/g; #' stupid emacs - no! javascript
        $result.='<input type="hidden" name="includemsg'.$counter.'" value="" />'."\n".
                 '<input type="hidden" name="newmsg'.$counter.'" value="" />'."\n".
	         '&nbsp;<a href="javascript:msgCenter(document.SCORE,'.$counter.
                 ',\''.$msgfor.'\');" target="_self">'.
                 &mt('Compose message to student'.(scalar(@$col_fullnames) >= 1 ? 's' : '')).'</a><label> ('.
                 &mt('incl. grades').' <input type="checkbox" name="withgrades'.$counter.'" /></label>)'.
                 ' <img src="'.$request->dir_config('lonIconsURL').
                 '/mailbkgrd.gif" width="14" height="10" alt="" name="mailicon'.$counter.'" />'."\n".
                 '<br />&nbsp;('.
                 &mt('Message will be sent when you click on Save &amp; Next below.').")\n".
	         '</div></div>';
        $request->print($result);
    }

    my %seen = ();
    my @partlist;
    my @gradePartRespid;
    my @part_response_id = &flatten_responseType($responseType);
    $request->print(
        '<div class="LC_Box">'
       .'<h3 class="LC_hcell">'.&mt('Assign Grades').'</h3>'
    );
    $request->print(&gradeBox_start());
    foreach my $part_response_id (@part_response_id) {
    	my ($partid,$respid) = @{ $part_response_id };
	my $part_resp = join('_',@{ $part_response_id });
	next if ($seen{$partid} > 0);
	$seen{$partid}++;
	push(@partlist,$partid);
	push(@gradePartRespid,$partid.'.'.$respid);
	$request->print(&gradeBox($request,$symb,$uname,$udom,$counter,$partid,\%record));
    }
    $request->print(&gradeBox_end()); # </div>
    $request->print('</div>');

    $request->print('<div class="LC_grade_info_links">');
    $request->print('</div>');

    $result='<input type="hidden" name="partlist'.$counter.
	'" value="'.(join ":",@partlist).'" />'."\n";
    $result.='<input type="hidden" name="gradePartRespid'.
	'" value="'.(join ":",@gradePartRespid).'" />'."\n" if ($counter == 0);
    my $ctr = 0;
    while ($ctr < scalar(@partlist)) {
	$result.='<input type="hidden" name="partid'.$counter.'_'.$ctr.'" value="'.
	    $partlist[$ctr].'" />'."\n";
	$ctr++;
    }
    $request->print($result.''."\n");

# Done with printing info for one student

    $request->print('</div>');#LC_grade_show_user


    # print end of form
    if ($counter == $total) {
        my $endform='<br /><hr /><table border="0"><tr><td>'."\n";
	$endform.='<input type="button" value="'.&mt('Save &amp; Next').'" '.
	    'onclick="javascript:checksubmit(this.form,\'Save & Next\','.
	    $total.','.scalar(@partlist).');" target="_self" /> &nbsp;'."\n";
	my $ntstu ='<select name="NTSTU">'.
	    '<option>1</option><option>2</option>'.
	    '<option>3</option><option>5</option>'.
	    '<option>7</option><option>10</option></select>'."\n";
	my $nsel = ($env{'form.NTSTU'} ne '' ? $env{'form.NTSTU'} : '1');
	$ntstu =~ s/<option>$nsel</<option selected="selected">$nsel</;
        $endform.=&mt('[_1]student(s)',$ntstu);
	$endform.='&nbsp;&nbsp;<input type="button" value="'.&mt('Previous').'" '.
	    'onclick="javascript:checksubmit(this.form,\'Previous\');" target="_self" /> &nbsp;'."\n".
	    '<input type="button" value="'.&mt('Next').'" '.
	    'onclick="javascript:checksubmit(this.form,\'Next\');" target="_self" /> &nbsp;';
        $endform.='<span class="LC_warning">'.
                  &mt('(Next and Previous (student) do not save the scores.)').
                  '</span>'."\n" ;
        $endform.="<input type='hidden' value='".&get_increment().
            "' name='increment' />";
	$endform.='</td></tr></table></form>';
	$request->print($endform);
    }
    return '';
}

sub check_collaborators {
    my ($symb,$uname,$udom,$record,$handgrade,$counter) = @_;
    my ($result,@col_fullnames);
    my ($classlist,undef,$fullname) = &getclasslist('all','0');
    foreach my $part (keys(%$handgrade)) {
	my $ncol = &Apache::lonnet::EXT('resource.'.$part.
					'.maxcollaborators',
					$symb,$udom,$uname);
	next if ($ncol <= 0);
	$part =~ s/\_/\./g;
	next if ($record->{'resource.'.$part.'.collaborators'} eq '');
	my (@good_collaborators, @bad_collaborators);
	foreach my $possible_collaborator
	    (split(/[,;\s]+/,$record->{'resource.'.$part.'.collaborators'})) { 
	    $possible_collaborator =~ s/[\$\^\(\)]//g;
	    next if ($possible_collaborator eq '');
	    my ($co_name,$co_dom) = split(/:/,$possible_collaborator);
	    $co_dom = $udom if (! defined($co_dom) || $co_dom =~ /^domain$/i);
	    next if ($co_name eq $uname && $co_dom eq $udom);
	    # Doing this grep allows 'fuzzy' specification
	    my @matches = grep(/^\Q$co_name\E:\Q$co_dom\E$/i, 
			       keys(%$classlist));
	    if (! scalar(@matches)) {
		push(@bad_collaborators, $possible_collaborator);
	    } else {
		push(@good_collaborators, @matches);
	    }
	}
	if (scalar(@good_collaborators) != 0) {
	    $result.='<br />'.&mt('Collaborators:').'<ol>';
	    foreach my $name (@good_collaborators) {
		my ($lastname,$givenn) = split(/,/,$$fullname{$name});
		push(@col_fullnames, $givenn.' '.$lastname);
		$result.='<li>'.$fullname->{$name}.'</li>';
	    }
	    $result.='</ol><br />'."\n";
	    my ($part)=split(/\./,$part);
	    $result.='<input type="hidden" name="collaborator'.$counter.
		'" value="'.$part.':'.(join ':',@good_collaborators).'" />'.
		"\n";
	}
	if (scalar(@bad_collaborators) > 0) {
	    $result.='<div class="LC_warning">';
	    $result.=&mt('This student has submitted [quant,_1,invalid collaborator]: [_2]',scalar(@bad_collaborators),join(', ',@bad_collaborators));
	    $result .= '</div>';
	}         
	if (scalar(@bad_collaborators > $ncol)) {
	    $result .= '<div class="LC_warning">';
	    $result .= &mt('This student has submitted too many '.
		'collaborators.  Maximum is [_1].',$ncol);
	    $result .= '</div>';
	}
    }
    return ($result,$fullname,\@col_fullnames);
}

#--- Retrieve the last submission for all the parts
sub get_last_submission {
    my ($returnhash)=@_;
    my (@string,$timestamp,$lastgradetime,$lastsubmittime);
    if ($$returnhash{'version'}) {
	my %lasthash=();
        my %prevsolved=();
        my %solved=();
	my $version;
	for ($version=1;$version<=$$returnhash{'version'};$version++) {
            my %handgraded = ();
	    foreach my $key (sort(split(/\:/,
					$$returnhash{$version.':keys'}))) {
		$lasthash{$key}=$$returnhash{$version.':'.$key};
                if ($key =~ /\.([^.]+)\.regrader$/) {
                    $handgraded{$1} = 1;
                } elsif ($key =~ /\.portfiles$/) {
                    if (($$returnhash{$version.':'.$key} ne '') &&
                        ($$returnhash{$version.':'.$key} !~ /\.\d+\.\w+$/)) {
                        $lastsubmittime = $$returnhash{$version.':timestamp'};
                    }
                } elsif ($key =~ /\.submission$/) {
                    if ($$returnhash{$version.':'.$key} ne '') {
                        $lastsubmittime = $$returnhash{$version.':timestamp'};
                    }
                } elsif ($key =~ /\.([^.]+)\.solved$/) {
                    $prevsolved{$1} = $solved{$1};
                    $solved{$1} = $lasthash{$key};
                }
	    }
            foreach my $partid (keys(%handgraded)) {
                if (($prevsolved{$partid} eq 'ungraded_attempted') &&
                    (($solved{$partid} eq 'incorrect_by_override') ||
                     ($solved{$partid} eq 'correct_by_override'))) {
                    $lastgradetime = $$returnhash{$version.':timestamp'};
                }
                if ($solved{$partid} ne '') {
                    $prevsolved{$partid} = $solved{$partid};
                }
            }
	}
#
# Timestamp is for last transaction for this resource, which does not
# necessarily correspond to the time of last submission for problem (or part).
#
        if ($lasthash{'timestamp'} ne '') {
            $timestamp = &Apache::lonlocal::locallocaltime($lasthash{'timestamp'});
        }
        my (%typeparts,%randombytry);
        my $showsurv = 
            &Apache::lonnet::allowed('vas',$env{'request.course.id'});
        foreach my $key (sort(keys(%lasthash))) {
            if ($key =~ /\.type$/) {
                if (($lasthash{$key} eq 'anonsurvey') || 
                    ($lasthash{$key} eq 'anonsurveycred') ||
                    ($lasthash{$key} eq 'randomizetry')) {
                    my ($ign,@parts) = split(/\./,$key);
                    pop(@parts);
                    my $id = join('.',@parts);
                    if ($lasthash{$key} eq 'randomizetry') {
                        $randombytry{$ign.'.'.$id} = $lasthash{$key};
                    } else {
                        unless ($showsurv) {
                            $typeparts{$ign.'.'.$id} = $lasthash{$key};
                        }
                    }
                    delete($lasthash{$key});
                }
            }
        }
        my @hidden = keys(%typeparts);
        my @randomize = keys(%randombytry);
	foreach my $key (keys(%lasthash)) {
	    next if ($key !~ /\.submission$/);
            my $hide;
            if (@hidden) {
                foreach my $id (@hidden) {
                    if ($key =~ /^\Q$id\E/) {
                        $hide = 'anon';
                        last;
                    }
                }
            }
            unless ($hide) {
                if (@randomize) {
                    foreach my $id (@randomize) {
                        if ($key =~ /^\Q$id\E/) {
                            $hide = 'rand';
                            last;
                        }
                    }
                }
            }
	    my ($partid,$foo) = split(/submission$/,$key);
	    my $draft  = $lasthash{$partid.'awarddetail'} eq 'DRAFT' ? 1 : 0;
            push(@string, join(':', $key, $hide, $draft, (
                ref($lasthash{$key}) eq 'ARRAY' ?
                    join(',', @{$lasthash{$key}}) : $lasthash{$key}) ));
	}
    }
    if (!@string) {
	$string[0] =
	    '<span class="LC_warning">'.&mt('Nothing submitted - no attempts.').'</span>';
    }
    return (\@string,$timestamp,$lastgradetime,$lastsubmittime);
}

#--- High light keywords, with style choosen by user.
sub keywords_highlight {
    my $string    = shift;
    my $size      = $env{'form.kwsize'} eq '0' ? '' : 'size='.$env{'form.kwsize'};
    my $styleon   = $env{'form.kwstyle'} eq ''  ? '' : $env{'form.kwstyle'};
    (my $styleoff = $styleon) =~ s/\</\<\//;
    my @keylist   = split(/[,\s+]/,$env{'form.keywords'});
    foreach my $keyword (@keylist) {
	$string =~ s/\b\Q$keyword\E(\b|\.)/<font color\=$env{'form.kwclr'} $size\>$styleon$keyword$styleoff<\/font>/gi;
    }
    return $string;
}

# For Tasks provide a mechanism to display previous version for one specific student

sub show_previous_task_version {
    my ($request,$symb) = @_;
    if ($symb eq '') {
        $request->print(
            '<span class="LC_error">'.
            &mt('Unable to handle ambiguous references.').
            '</span>');
        return '';
    }
    my ($uname,$udom) = ($env{'form.student'},$env{'form.userdom'});
    my $usec = &Apache::lonnet::getsection($udom,$uname,$env{'request.course.id'});
    if (!&canview($usec)) {
        $request->print('<span class="LC_warning">'.
                        &mt('Unable to view previous version for requested student.').
                        ' '.&mt('([_1] in section [_2] in course id [_3])',
                                $uname.':'.$udom,$usec,$env{'request.course.id'}).
                        '</span>');
        return;
    }
    my $mode = 'both';
    my $isTask = ($symb =~/\.task$/);
    if ($isTask) {
        if ($env{'form.previousversion'} =~ /^\d+$/) {
            if ($env{'form.fullname'} eq '') {
                $env{'form.fullname'} =
                    &Apache::loncommon::plainname($uname,$udom,'lastname');
            }
            my $probtitle=&Apache::lonnet::gettitle($symb);
            $request->print("\n\n".
                            '<div class="LC_grade_show_user">'.
                            '<h2>'.&nameUserString(undef,$env{'form.fullname'},$uname,$udom).
                            '</h2>'."\n");
            &Apache::lonxml::clear_problem_counter();
            $request->print(&show_problem($request,$symb,$uname,$udom,1,1,$mode,
                            {'previousversion' => $env{'form.previousversion'} }));
            $request->print("\n</div>");
        }
    }
    return;
}

sub choose_task_version_form {
    my ($symb,$uname,$udom,$nomenu) = @_;
    my $isTask = ($symb =~/\.task$/);
    my ($current,$version,$result,$js,$displayed,$rowtitle);
    if ($isTask) {
        my %record = &Apache::lonnet::restore($symb,$env{'request.course.id'},
                                              $udom,$uname);
        if (($record{'resource.0.version'} eq '') ||
            ($record{'resource.0.version'} < 2)) {
            return ($record{'resource.0.version'},
                    $record{'resource.0.version'},$result,$js);
        } else {
            $current = $record{'resource.0.version'};
        }
        if ($env{'form.previousversion'}) {
            $displayed = $env{'form.previousversion'};
            $rowtitle = &mt('Choose another version:')
        } else {
            $displayed = $current;
            $rowtitle = &mt('Show earlier version:');
        }
        $result = '<div class="LC_left_float">';
        my $list;
        my $numversions = 0;
        for (my $i=1; $i<=$record{'resource.0.version'}; $i++) {
            if ($i == $current) {
                if (!$env{'form.previousversion'} || $nomenu) {
                    next;
                } else {
                    $list .= '<option value="'.$i.'">'.&mt('Current').'</option>'."\n";
                    $numversions ++;
                }
            } elsif (defined($record{'resource.'.$i.'.0.status'})) {
                unless ($i == $env{'form.previousversion'}) {
                    $numversions ++;
                }
                $list .= '<option value="'.$i.'">'.$i.'</option>'."\n";
            }
        }
        if ($numversions) {
            $symb = &HTML::Entities::encode($symb,'<>"&');
            $result .=
                '<form name="getprev" method="post" action=""'.
                ' onsubmit="return previousVersion('."'$uname','$udom','$symb','$displayed'".');">'.
                &Apache::loncommon::start_data_table().
                &Apache::loncommon::start_data_table_row().
                '<th align="left">'.$rowtitle.'</th>'.
                '<td><select name="version">'.
                '<option>'.&mt('Select').'</option>'.
                $list.
                '</select></td>'.
                &Apache::loncommon::end_data_table_row();
            unless ($nomenu) {
                $result .= &Apache::loncommon::start_data_table_row().
                '<th align="left">'.&mt('Open in new window').'</th>'.
                '<td><span class="LC_nobreak">'.
                '<label><input type="radio" name="prevwin" value="1" />'.
                &mt('Yes').'</label>'.
                '<label><input type="radio" name="prevwin" value="0" checked="checked" />'.&mt('No').'</label>'.
                '</span></td>'.
                &Apache::loncommon::end_data_table_row();
            }
            $result .=
                &Apache::loncommon::start_data_table_row().
                '<th align="left">&nbsp;</th>'.
                '<td>'.
                '<input type="submit" name="prevsub" value="'.&mt('Display').'" />'.
                '</td>'.
                &Apache::loncommon::end_data_table_row().
                &Apache::loncommon::end_data_table().
                '</form>';
            $js = &previous_display_javascript($nomenu,$current);
        } elsif ($displayed && $nomenu) {
            $result .= '<a href="javascript:window.close()">'.&mt('Close window').'</a>';
        } else {
            $result .= &mt('No previous versions to show for this student');
        }
        $result .= '</div>';
    }
    return ($current,$displayed,$result,$js);
}

sub previous_display_javascript {
    my ($nomenu,$current) = @_;
    my $js = <<"JSONE";
<script type="text/javascript">
// <![CDATA[
function previousVersion(uname,udom,symb) {
    var current = '$current';
    var version = document.getprev.version.options[document.getprev.version.selectedIndex].value;
    var prevstr = new RegExp("^\\\\d+\$");
    if (!prevstr.test(version)) {
        return false;
    }
    var url = '';
    if (version == current) {
        url = '/adm/grades?student='+uname+'&userdom='+udom+'&symb='+symb+'&command=submission';
    } else {
        url = '/adm/grades?student='+uname+'&userdom='+udom+'&symb='+symb+'&command=versionsub&previousversion='+version;
    }
JSONE
    if ($nomenu) {
        $js .= <<"JSTWO";
    document.location.href = url;
JSTWO
    } else {
        $js .= <<"JSTHREE";
    var newwin = 0;
    for (var i=0; i<document.getprev.prevwin.length; i++) {
        if (document.getprev.prevwin[i].checked == true) {
            newwin = document.getprev.prevwin[i].value;
        }
    }
    if (newwin == 1) {
        var options = 'height=600,width=800,resizable=yes,scrollbars=yes,location=no,menubar=no,toolbar=no';
        url = url+'&inhibitmenu=yes';
        if (typeof(previousWin) == 'undefined' || previousWin.closed) {
            previousWin = window.open(url,'',options,1);
        } else {
            previousWin.location.href = url;
        }
        previousWin.focus();
        return false;
    } else {
        document.location.href = url;
        return false;
    }
JSTHREE
    }
    $js .= <<"ENDJS";
    return false;
}
// ]]>
</script>
ENDJS

}

#--- Called from submission routine
sub processHandGrade {
    my ($request,$symb) = @_;
    my (undef,undef,$url) = &Apache::lonnet::decode_symb($symb);
    my $button = $env{'form.gradeOpt'};
    my $ngrade = $env{'form.NCT'};
    my $ntstu  = $env{'form.NTSTU'};
    my $cdom   = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $cnum   = $env{'course.'.$env{'request.course.id'}.'.num'};
    my ($res_error,%queueable);
    my ($partlist,$handgrade,$responseType,$numresp,$numessay) = &response_type($symb,\$res_error);
    if ($res_error) {
        $request->print(&navmap_errormsg());
        return;
    } else {
        foreach my $part (@{$partlist}) {
            if (ref($responseType->{$part}) eq 'HASH') {
                foreach my $id (keys(%{$responseType->{$part}})) {
                    if (($responseType->{$part}->{$id} eq 'essay') ||
                        (lc($handgrade->{$part.'_'.$id}) eq 'yes')) {
                        $queueable{$part} = 1;
                        last;
                    }
                }
            }
        }
    }

    if ($button eq 'Save & Next') {
	my $ctr = 0;
	while ($ctr < $ngrade) {
	    my ($uname,$udom) = split(/:/,$env{'form.unamedom'.$ctr});
	    my ($errorflag,$pts,$wgt,$numhidden) = 
                &saveHandGrade($request,$symb,$uname,$udom,$ctr,undef,undef,\%queueable);
	    if ($errorflag eq 'no_score') {
		$ctr++;
		next;
	    }
	    if ($errorflag eq 'not_allowed') {
                $request->print(
                    '<span class="LC_error">'
                   .&mt('Not allowed to modify grades for [_1]',"$uname:$udom")
                   .'</span>');
		$ctr++;
		next;
	    }
            if ($numhidden) {
                $request->print(
                    '<span class="LC_info">'
                   .&mt('For [_1]: [quant,_2,transaction] hidden',"$uname:$udom",$numhidden)
                   .'</span><br />');
            }
	    my $includemsg = $env{'form.includemsg'.$ctr};
	    my ($subject,$message,$msgstatus) = ('','','');
	    my $restitle = &Apache::lonnet::gettitle($symb);
            my ($feedurl,$showsymb) =
		&get_feedurl_and_symb($symb,$uname,$udom);
	    my $messagetail;
	    if ($includemsg =~ /savemsg|newmsg\Q$ctr\E/) {
		$subject = $env{'form.msgsub'} if ($includemsg =~ /msgsub/);
		unless ($subject=~/\w/) { $subject=&mt('Grading Feedback'); }
		$subject.=' ['.$restitle.']';
		my (@msgnum) = split(/,/,$includemsg);
		foreach (@msgnum) {
		    $message.=$env{'form.'.$_} if ($_ =~ /savemsg|newmsg/ && $_ ne '');
		}
		$message =&Apache::lonfeedback::clear_out_html($message);
		if ($env{'form.withgrades'.$ctr}) {
		    $message.="\n\nPoint".($pts > 1 ? 's':'').' awarded = '.$pts.' out of '.$wgt;
		    $messagetail = " for <a href=\"".
		                   $feedurl."?symb=$showsymb\">$restitle</a>";
		}
		$msgstatus = 
                    &Apache::lonmsg::user_normal_msg($uname,$udom,$subject,
						     $message.$messagetail,
                                                     undef,$feedurl,undef,
                                                     undef,undef,$showsymb,
                                                     $restitle);
		$request->print('<br />'.&mt('Sending message to [_1]',$uname.':'.$udom).': '.
				$msgstatus.'<br />');
	    }
	    if ($env{'form.collaborator'.$ctr}) {
		my @collabstrs=&Apache::loncommon::get_env_multiple("form.collaborator$ctr");
		foreach my $collabstr (@collabstrs) {
		    my ($part,@collaborators) = split(/:/,$collabstr);
		    foreach my $collaborator (@collaborators) {
			my ($errorflag,$pts,$wgt) = 
			    &saveHandGrade($request,$symb,$collaborator,$udom,$ctr,
					   $env{'form.unamedom'.$ctr},$part,\%queueable);
			if ($errorflag eq 'not_allowed') {
			    $request->print("<span class=\"LC_error\">".&mt('Not allowed to modify grades for [_1]',"$collaborator:$udom")."</span>");
			    next;
			} elsif ($message ne '') {
			    my ($baseurl,$showsymb) = 
				&get_feedurl_and_symb($symb,$collaborator,
						      $udom);
			    if ($env{'form.withgrades'.$ctr}) {
				$messagetail = " for <a href=\"".
                                    $baseurl."?symb=$showsymb\">$restitle</a>";
			    }
			    $msgstatus = 
				&Apache::lonmsg::user_normal_msg($collaborator,$udom,$subject,$message.$messagetail,undef,$baseurl,undef,undef,undef,$showsymb,$restitle);
			}
		    }
		}
	    }
	    $ctr++;
	}
    }

    my %keyhash = ();
    if ($numessay) {
	# Keywords sorted in alphabatical order
	my $loginuser = $env{'user.name'}.':'.$env{'user.domain'};
	$env{'form.keywords'}           =~ s/,\s{0,}|\s+/ /g;
	$env{'form.keywords'}           =~ s/^\s+|\s+$//g;
	my (@keywords) = sort(split(/\s+/,$env{'form.keywords'}));
	$env{'form.keywords'} = join(' ',@keywords);
	$keyhash{$symb.'_keywords'}     = $env{'form.keywords'};
	$keyhash{$symb.'_subject'}      = $env{'form.msgsub'};
	$keyhash{$loginuser.'_kwclr'}   = $env{'form.kwclr'};
	$keyhash{$loginuser.'_kwsize'}  = $env{'form.kwsize'};
	$keyhash{$loginuser.'_kwstyle'} = $env{'form.kwstyle'};
    }

    if ($env{'form.compmsg'}) {
	# message center - Order of message gets changed. Blank line is eliminated.
	# New messages are saved in env for the next student.
	# All messages are saved in nohist_handgrade.db
	my ($ctr,$idx) = (1,1);
	while ($ctr <= $env{'form.savemsgN'}) {
	    if ($env{'form.savemsg'.$ctr} ne '') {
		$keyhash{$symb.'_savemsg'.$idx} = $env{'form.savemsg'.$ctr};
		$idx++;
	    }
	    $ctr++;
	}
	$ctr = 0;
	while ($ctr < $ngrade) {
	    if ($env{'form.newmsg'.$ctr} ne '') {
		$keyhash{$symb.'_savemsg'.$idx} = $env{'form.newmsg'.$ctr};
		$env{'form.savemsg'.$idx} = $env{'form.newmsg'.$ctr};
		$idx++;
	    }
	    $ctr++;
	}
	$env{'form.savemsgN'} = --$idx;
	$keyhash{$symb.'_savemsgN'} = $env{'form.savemsgN'};
    }
    if (($numessay) || ($env{'form.compmsg'})) {
	my $putresult = &Apache::lonnet::put
	    ('nohist_handgrade',\%keyhash,$cdom,$cnum);
    }

    # Called by Save & Refresh from Highlight Attribute Window
    my (undef,undef,$fullname) = &getclasslist($env{'form.section'},'1');
    if ($env{'form.refresh'} eq 'on') {
	my ($ctr,$total) = (0,0);
	while ($ctr < $ngrade) {
	    $total++ if  $env{'form.unamedom'.$ctr} ne '';
	    $ctr++;
	}
	$env{'form.NTSTU'}=$ngrade;
	$ctr = 0;
	while ($ctr < $total) {
	    my $processUser = $env{'form.unamedom'.$ctr};
	    ($env{'form.student'},$env{'form.userdom'}) = split(/:/,$processUser);
	    $env{'form.fullname'} = $$fullname{$processUser};
	    &submission($request,$ctr,$total-1,$symb);
	    $ctr++;
	}
	return '';
    }

    # Get the next/previous one or group of students
    my $firststu = $env{'form.unamedom0'};
    my $laststu = $env{'form.unamedom'.($ngrade-1)};
    my $ctr = 2;
    while ($laststu eq '') {
	$laststu  = $env{'form.unamedom'.($ngrade-$ctr)};
	$ctr++;
	$laststu = $firststu if ($ctr > $ngrade);
    }

    my (@parsedlist,@nextlist);
    my ($nextflg) = 0;
    foreach my $item (sort 
	     {
		 if (lc($$fullname{$a}) ne lc($$fullname{$b})) {
		     return (lc($$fullname{$a}) cmp lc($$fullname{$b}));
		 }
		 return $a cmp $b;
	     } (keys(%$fullname))) {
	if ($nextflg == 1 && $button =~ /Next$/) {
	    push(@parsedlist,$item);
	}
	$nextflg = 1 if ($item eq $laststu);
	if ($button eq 'Previous') {
	    last if ($item eq $firststu);
	    push(@parsedlist,$item);
	}
    }
    $ctr = 0;
    @parsedlist = reverse @parsedlist if ($button eq 'Previous');
    foreach my $student (@parsedlist) {
	my $submitonly=$env{'form.submitonly'};
	my ($uname,$udom) = split(/:/,$student);
	
	if ($submitonly eq 'queued') {
	    my %queue_status = 
		&Apache::bridgetask::get_student_status($symb,$cdom,$cnum,
							$udom,$uname);
	    next if (!defined($queue_status{'gradingqueue'}));
	}

	if ($submitonly =~ /^(yes|graded|incorrect)$/) {
#	    my %record = &Apache::lonnet::restore($symb,$env{'request.course.id'},$udom,$uname);
	    my %status=&student_gradeStatus($symb,$udom,$uname,$partlist);
	    my $submitted = 0;
	    my $ungraded = 0;
	    my $incorrect = 0;
	    foreach my $item (keys(%status)) {
		$submitted = 1 if ($status{$item} ne 'nothing');
		$ungraded = 1 if ($status{$item} =~ /^ungraded/);
		$incorrect = 1 if ($status{$item} =~ /^incorrect/);
		my ($foo,$partid,$foo1) = split(/\./,$item);
		if ($status{'resource.'.$partid.'.submitted_by'} ne '') {
		    $submitted = 0;
		}
	    }
	    next if (!$submitted && ($submitonly eq 'yes' ||
				     $submitonly eq 'incorrect' ||
				     $submitonly eq 'graded'));
	    next if (!$ungraded && ($submitonly eq 'graded'));
	    next if (!$incorrect && $submitonly eq 'incorrect');
	}
	push(@nextlist,$student) if ($ctr < $ntstu);
	last if ($ctr == $ntstu);
	$ctr++;
    }

    $ctr = 0;
    my $total = scalar(@nextlist)-1;

    foreach (sort(@nextlist)) {
	my ($uname,$udom,$submitter) = split(/:/);
	$env{'form.student'}  = $uname;
	$env{'form.userdom'}  = $udom;
	$env{'form.fullname'} = $$fullname{$_};
	&submission($request,$ctr,$total,$symb);
	$ctr++;
    }
    if ($total < 0) {
        my $the_end.='<p>'.&mt('[_1]Message:[_2] No more students for this section or class.','<b>','</b>').'</p>'."\n";
	$request->print($the_end);
    }
    return '';
}

#---- Save the score and award for each student, if changed
sub saveHandGrade {
    my ($request,$symb,$stuname,$domain,$newflg,$submitter,$part,$queueable) = @_;
    my @version_parts;
    my $usec = &Apache::lonnet::getsection($domain,$stuname,
					   $env{'request.course.id'});
    if (!&canmodify($usec)) { return('not_allowed'); }
    my %record = &Apache::lonnet::restore($symb,$env{'request.course.id'},$domain,$stuname);
    my @parts_graded;
    my %newrecord  = ();
    my ($pts,$wgt,$totchg) = ('','',0);
    my %aggregate = ();
    my $aggregateflag = 0;
    if ($env{'form.HIDE'.$newflg}) {
        my ($version,$parts) = split(/:/,$env{'form.HIDE'.$newflg},2);
        my $numchgs = &makehidden($version,$parts,\%record,$symb,$domain,$stuname,1);
        $totchg += $numchgs;
    }
    my @parts = split(/:/,$env{'form.partlist'.$newflg});
    foreach my $new_part (@parts) {
	#collaborator ($submi may vary for different parts
	if ($submitter && $new_part ne $part) { next; }
	my $dropMenu = $env{'form.GD_SEL'.$newflg.'_'.$new_part};
	if ($dropMenu eq 'excused') {
	    if ($record{'resource.'.$new_part.'.solved'} ne 'excused') {
		$newrecord{'resource.'.$new_part.'.solved'} = 'excused';
		if (exists($record{'resource.'.$new_part.'.awarded'})) {
		    $newrecord{'resource.'.$new_part.'.awarded'} = '';
		}
	        $newrecord{'resource.'.$new_part.'.regrader'}="$env{'user.name'}:$env{'user.domain'}";
	    }
	} elsif ($dropMenu eq 'reset status'
		 && exists($record{'resource.'.$new_part.'.solved'})) { #don't bother if no old records -> no attempts
	    foreach my $key (keys(%record)) {
		if ($key=~/^resource\.\Q$new_part\E\./) { $newrecord{$key} = ''; }
	    }
	    $newrecord{'resource.'.$new_part.'.regrader'}=
		"$env{'user.name'}:$env{'user.domain'}";
            my $totaltries = $record{'resource.'.$part.'.tries'};

            my %last_resets = &get_last_resets($symb,$env{'request.course.id'},
					       [$new_part]);
            my $aggtries =$totaltries;
            if ($last_resets{$new_part}) {
                $aggtries = &get_num_tries(\%record,$last_resets{$new_part},
					   $new_part);
            }

            my $solvedstatus = $record{'resource.'.$new_part.'.solved'};
            if ($aggtries > 0) {
                &decrement_aggs($symb,$new_part,\%aggregate,$aggtries,$totaltries,$solvedstatus);
                $aggregateflag = 1;
            }
	} elsif ($dropMenu eq '') {
	    $pts = ($env{'form.GD_BOX'.$newflg.'_'.$new_part} ne '' ? 
		    $env{'form.GD_BOX'.$newflg.'_'.$new_part} : 
		    $env{'form.RADVAL'.$newflg.'_'.$new_part});
	    if ($pts eq '' && $env{'form.GD_SEL'.$newflg.'_'.$new_part} eq '') {
		next;
	    }
	    $wgt = $env{'form.WGT'.$newflg.'_'.$new_part} eq '' ? 1 : 
		$env{'form.WGT'.$newflg.'_'.$new_part};
	    my $partial= $pts/$wgt;
	    if ($partial eq $record{'resource.'.$new_part.'.awarded'}) {
		#do not update score for part if not changed.
                &handback_files($request,$symb,$stuname,$domain,$newflg,$new_part,\%newrecord);
		next;
	    } else {
	        push(@parts_graded,$new_part);
	    }
	    if ($record{'resource.'.$new_part.'.awarded'} ne $partial) {
		$newrecord{'resource.'.$new_part.'.awarded'}  = $partial;
	    }
	    my $reckey = 'resource.'.$new_part.'.solved';
	    if ($partial == 0) {
		if ($record{$reckey} ne 'incorrect_by_override') {
		    $newrecord{$reckey} = 'incorrect_by_override';
		}
	    } else {
		if ($record{$reckey} ne 'correct_by_override') {
		    $newrecord{$reckey} = 'correct_by_override';
		}
	    }	    
	    if ($submitter && 
		($record{'resource.'.$new_part.'.submitted_by'} ne $submitter)) {
		$newrecord{'resource.'.$new_part.'.submitted_by'} = $submitter;
	    }
	    $newrecord{'resource.'.$new_part.'.regrader'}=
		"$env{'user.name'}:$env{'user.domain'}";
	}
	# unless problem has been graded, set flag to version the submitted files
	unless ($record{'resource.'.$new_part.'.solved'} =~ /^correct_/  || 
	        $record{'resource.'.$new_part.'.solved'} eq 'incorrect_by_override' ||
	        $dropMenu eq 'reset status')
	   {
	    push(@version_parts,$new_part);
	}
    }
    my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};

    if (%newrecord) {
        if (@version_parts) {
            my @changed_keys = &version_portfiles(\%record, \@parts_graded, 
                                $env{'request.course.id'}, $symb, $domain, $stuname, \@version_parts);
	    @newrecord{@changed_keys} = @record{@changed_keys};
	    foreach my $new_part (@version_parts) {
		&handback_files($request,$symb,$stuname,$domain,$newflg,
				$new_part,\%newrecord);
	    }
        }
	&Apache::lonnet::cstore(\%newrecord,$symb,
				$env{'request.course.id'},$domain,$stuname);
	&check_and_remove_from_queue(\@parts,\%record,\%newrecord,$symb,
				     $cdom,$cnum,$domain,$stuname,$queueable);
    }
    if ($aggregateflag) {
        &Apache::lonnet::cinc('nohist_resourcetracker',\%aggregate,
			      $cdom,$cnum);
    }
    return ('',$pts,$wgt,$totchg);
}

sub makehidden {
    my ($version,$parts,$record,$symb,$domain,$stuname,$tolog) = @_;
    return unless (ref($record) eq 'HASH');
    my %modified;
    my $numchanged = 0;
    if (exists($record->{$version.':keys'})) {
        my $partsregexp = $parts;
        $partsregexp =~ s/,/|/g;
        foreach my $key (split(/\:/,$record->{$version.':keys'})) {
            if ($key =~ /^resource\.(?:$partsregexp)\.([^\.]+)$/) {
                 my $item = $1;
                 unless (($item eq 'solved') || ($item =~ /^award(|msg|ed)$/)) {
                     $modified{$key} = $record->{$version.':'.$key};
                 }
            } elsif ($key =~ m{^(resource\.(?:$partsregexp)\.[^\.]+\.)(.+)$}) {
                $modified{$1.'hidden'.$2} = $record->{$version.':'.$key};
            } elsif ($key =~ /^(ip|timestamp|host)$/) {
                $modified{$key} = $record->{$version.':'.$key};
            }
        }
        if (keys(%modified)) {
            if (&Apache::lonnet::putstore($env{'request.course.id'},$symb,$version,\%modified,
                                          $domain,$stuname,$tolog) eq 'ok') {
                $numchanged ++;
            }
        }
    }
    return $numchanged;
}

sub check_and_remove_from_queue {
    my ($parts,$record,$newrecord,$symb,$cdom,$cnum,$domain,$stuname,$queueable) = @_;
    my @ungraded_parts;
    foreach my $part (@{$parts}) {
	if (    $record->{   'resource.'.$part.'.awarded'} eq ''
	     && $record->{   'resource.'.$part.'.solved' } ne 'excused'
	     && $newrecord->{'resource.'.$part.'.awarded'} eq ''
	     && $newrecord->{'resource.'.$part.'.solved' } ne 'excused'
		) {
            if ($queueable->{$part}) {
	        push(@ungraded_parts, $part);
            }
	}
    }
    if ( !@ungraded_parts ) {
	&Apache::bridgetask::remove_from_queue('gradingqueue',$symb,$cdom,
					       $cnum,$domain,$stuname);
    }
}

sub handback_files {
    my ($request,$symb,$stuname,$domain,$newflg,$new_part,$newrecord) = @_;
    my $portfolio_root = '/userfiles/portfolio';
    my $res_error;
    my ($partlist,$handgrade,$responseType) = &response_type($symb,\$res_error);
    if ($res_error) {
        $request->print('<br />'.&navmap_errormsg().'<br />');
        return;
    }
    my @handedback;
    my $file_msg;
    my @part_response_id = &flatten_responseType($responseType);
    foreach my $part_response_id (@part_response_id) {
    	my ($part_id,$resp_id) = @{ $part_response_id };
	my $part_resp = join('_',@{ $part_response_id });
        if (($env{'form.'.$newflg.'_'.$part_resp.'_countreturndoc'} =~ /^\d+$/) & ($new_part eq $part_id)) {
            for (my $counter=1; $counter<=$env{'form.'.$newflg.'_'.$part_resp.'_countreturndoc'}; $counter++) {
                # if multiple files are uploaded names will be 'returndoc2','returndoc3'
		if ($env{'form.'.$newflg.'_'.$part_resp.'_returndoc'.$counter}) {
                    my $fname=$env{'form.'.$newflg.'_'.$part_resp.'_returndoc'.$counter.'.filename'};
                    my ($directory,$answer_file) = 
                        ($env{'form.'.$newflg.'_'.$part_resp.'_origdoc'.$counter} =~ /^(.*?)([^\/]*)$/);
                    my ($answer_name,$answer_ver,$answer_ext) =
		        &file_name_version_ext($answer_file);
		    my ($portfolio_path) = ($directory =~ /^.+$stuname\/portfolio(.*)/);
                    my $getpropath = 1;
                    my ($dir_list,$listerror) =
                        &Apache::lonnet::dirlist($portfolio_root.$portfolio_path,
                                                 $domain,$stuname,$getpropath);
		    my $version = &get_next_version($answer_name,$answer_ext,$dir_list);
                    # fix filename
                    my ($save_file_name) = (($directory.$answer_name.".$version.".$answer_ext) =~ /^.+\/${stuname}\/(.*)/);
                    my $result=&Apache::lonnet::finishuserfileupload($stuname,$domain,
            	                                $newflg.'_'.$part_resp.'_returndoc'.$counter,
            	                                $save_file_name);
                    if ($result !~ m|^/uploaded/|) {
                        $request->print('<br /><span class="LC_error">'.
                            &mt('An error occurred ([_1]) while trying to upload [_2].',
                                $result,$newflg.'_'.$part_resp.'_returndoc'.$counter).
                                        '</span>');
                    } else {
                        # mark the file as read only
                        push(@handedback,$save_file_name);
			if (exists($$newrecord{"resource.$new_part.$resp_id.handback"})) {
			    $$newrecord{"resource.$new_part.$resp_id.handback"}.=',';
			}
                        $$newrecord{"resource.$new_part.$resp_id.handback"} .= $save_file_name;
			$file_msg.='<span class="LC_filename"><a href="/uploaded/'."$domain/$stuname/".$save_file_name.'">'.$save_file_name."</a></span> <br />";

                    }
                    $request->print('<br />'.&mt('[_1] will be the uploaded filename [_2]','<span class="LC_info">'.$fname.'</span>','<span class="LC_filename">'.$env{'form.'.$newflg.'_'.$part_resp.'_origdoc'.$counter}.'</span>'));
                }
            }
        }
    }
    if (@handedback > 0) {
        $request->print('<br />');
        my @what = ($symb,$env{'request.course.id'},'handback');
        &Apache::lonnet::mark_as_readonly($domain,$stuname,\@handedback,\@what);
        my $user_lh = &Apache::loncommon::user_lang($stuname,$domain,$env{'request.course.id'});
        my ($subject,$message);
        if (scalar(@handedback) == 1) {
            $subject = &mt_user($user_lh,'File Handed Back by Instructor');
            $message = &mt_user($user_lh,'A file has been returned that was originally submitted in response to: ');
        } else {
            $subject = &mt_user($user_lh,'Files Handed Back by Instructor');
            $message = &mt_user($user_lh,'Files have been returned that were originally submitted in response to: ');
        }
        $message .= "<p><strong>".&Apache::lonnet::gettitle($symb)." </strong></p>";
        $message .= &mt_user($user_lh,'The returned file(s) are named: [_1]',"<br />$file_msg <br />").
                    &mt_user($user_lh,'The file(s) can be found in your [_1]portfolio[_2].','<a href="/adm/portfolio">','</a>');
        my ($feedurl,$showsymb) =
            &get_feedurl_and_symb($symb,$domain,$stuname);
        my $restitle = &Apache::lonnet::gettitle($symb);
        $subject .= ' '.&mt_user($user_lh,'(File Returned)').' ['.$restitle.']';
        my $msgstatus =
             &Apache::lonmsg::user_normal_msg($stuname,$domain,$subject,
                 $message,undef,$feedurl,undef,undef,undef,$showsymb,
                 $restitle);
        if ($msgstatus) {
            $request->print(&mt('Notification message status: [_1]','<span class="LC_info">'.$msgstatus.'</span>').'<br />');
        }
    }
    return;
}

sub get_feedurl_and_symb {
    my ($symb,$uname,$udom) = @_;
    my (undef,undef,$url) = &Apache::lonnet::decode_symb($symb);
    $url = &Apache::lonnet::clutter($url);
    my $encrypturl=&Apache::lonnet::EXT('resource.0.encrypturl',
					$symb,$udom,$uname);
    if ($encrypturl =~ /^yes$/i) {
	&Apache::lonenc::encrypted(\$url,1);
	&Apache::lonenc::encrypted(\$symb,1);
    }
    return ($url,$symb);
}

sub get_submitted_files {
    my ($udom,$uname,$partid,$respid,$record) = @_;
    my @files;
    if ($$record{"resource.$partid.$respid.portfiles"}) {
        my $file_url = '/uploaded/'.$udom.'/'.$uname.'/portfolio';
        foreach my $file (split(',',$$record{"resource.$partid.$respid.portfiles"})) {
    	    push(@files,$file_url.$file);
        }
    }
    if ($$record{"resource.$partid.$respid.uploadedurl"}) {
        push(@files,$$record{"resource.$partid.$respid.uploadedurl"});
    }
    return (\@files);
}

# ----------- Provides number of tries since last reset.
sub get_num_tries {
    my ($record,$last_reset,$part) = @_;
    my $timestamp = '';
    my $num_tries = 0;
    if ($$record{'version'}) {
        for (my $version=$$record{'version'};$version>=1;$version--) {
            if (exists($$record{$version.':resource.'.$part.'.solved'})) {
                $timestamp = $$record{$version.':timestamp'};
                if ($timestamp > $last_reset) {
                    $num_tries ++;
                } else {
                    last;
                }
            }
        }
    }
    return $num_tries;
}

# ----------- Determine decrements required in aggregate totals 
sub decrement_aggs {
    my ($symb,$part,$aggregate,$aggtries,$totaltries,$solvedstatus) = @_;
    my %decrement = (
                        attempts => 0,
                        users => 0,
                        correct => 0
                    );
    $decrement{'attempts'} = $aggtries;
    if ($solvedstatus =~ /^correct/) {
        $decrement{'correct'} = 1;
    }
    if ($aggtries == $totaltries) {
        $decrement{'users'} = 1;
    }
    foreach my $type (keys(%decrement)) {
        $$aggregate{$symb."\0".$part."\0".$type} = -$decrement{$type};
    }
    return;
}

# ----------- Determine timestamps for last reset of aggregate totals for parts  
sub get_last_resets {
    my ($symb,$courseid,$partids) =@_;
    my %last_resets;
    my $cdom = $env{'course.'.$courseid.'.domain'};
    my $cname = $env{'course.'.$courseid.'.num'};
    my @keys;
    foreach my $part (@{$partids}) {
	push(@keys,"$symb\0$part\0resettime");
    }
    my %results=&Apache::lonnet::get('nohist_resourcetracker',\@keys,
				     $cdom,$cname);
    foreach my $part (@{$partids}) {
	$last_resets{$part}=$results{"$symb\0$part\0resettime"};
    }
    return %last_resets;
}

# ----------- Handles creating versions for portfolio files as answers
sub version_portfiles {
    my ($record, $parts_graded, $courseid, $symb, $domain, $stu_name, $v_flag) = @_;
    my $version_parts = join('|',@$v_flag);
    my @returned_keys;
    my $parts = join('|', @$parts_graded);
    my $portfolio_root = '/userfiles/portfolio';
    foreach my $key (keys(%$record)) {
        my $new_portfiles;
        if ($key =~ /^resource\.($version_parts)\./ && $key =~ /\.portfiles$/ ) {
            my @versioned_portfiles;
            my @portfiles = split(/\s*,\s*/,$$record{$key});
            foreach my $file (@portfiles) {
                &Apache::lonnet::unmark_as_readonly($domain,$stu_name,[$symb,$env{'request.course.id'}],$file);
                my ($directory,$answer_file) =($file =~ /^(.*?)([^\/]*)$/);
		my ($answer_name,$answer_ver,$answer_ext) =
		    &file_name_version_ext($answer_file);
                my $getpropath = 1;
                my ($dir_list,$listerror) =
                    &Apache::lonnet::dirlist($portfolio_root.$directory,$domain,
                                             $stu_name,$getpropath);
                my $version = &get_next_version($answer_name,$answer_ext,$dir_list);
                my $new_answer = &version_selected_portfile($domain, $stu_name, $directory, $answer_file, $version);
                if ($new_answer ne 'problem getting file') {
                    push(@versioned_portfiles, $directory.$new_answer);
                    &Apache::lonnet::mark_as_readonly($domain,$stu_name,
                        [$directory.$new_answer],
                        [$symb,$env{'request.course.id'},'graded']);
                }
            }
            $$record{$key} = join(',',@versioned_portfiles);
            push(@returned_keys,$key);
        }
    }
    return (@returned_keys);
}

sub get_next_version {
    my ($answer_name, $answer_ext, $dir_list) = @_;
    my $version;
    if (ref($dir_list) eq 'ARRAY') {
        foreach my $row (@{$dir_list}) {
            my ($file) = split(/\&/,$row,2);
            my ($file_name,$file_version,$file_ext) =
	        &file_name_version_ext($file);
            if (($file_name eq $answer_name) && 
	        ($file_ext eq $answer_ext)) {
                # gets here if filename and extension match, 
                # regardless of version
                if ($file_version ne '') {
                    # a versioned file is found  so save it for later
                    if ($file_version > $version) {
		        $version = $file_version;
                    }
	        }
            }
        }
    }
    $version ++;
    return($version);
}

sub version_selected_portfile {
    my ($domain,$stu_name,$directory,$file_name,$version) = @_;
    my ($answer_name,$answer_ver,$answer_ext) =
        &file_name_version_ext($file_name);
    my $new_answer;
    $env{'form.copy'} = &Apache::lonnet::getfile("/uploaded/$domain/$stu_name/portfolio$directory$file_name");
    if($env{'form.copy'} eq '-1') {
        $new_answer = 'problem getting file';
    } else {
        $new_answer = $answer_name.'.'.$version.'.'.$answer_ext;
        my $copy_result = &Apache::lonnet::finishuserfileupload(
                            $stu_name,$domain,'copy',
		        '/portfolio'.$directory.$new_answer);
    }    
    return ($new_answer);
}

sub file_name_version_ext {
    my ($file)=@_;
    my @file_parts = split(/\./, $file);
    my ($name,$version,$ext);
    if (@file_parts > 1) {
	$ext=pop(@file_parts);
	if (@file_parts > 1 && $file_parts[-1] =~ /^\d+$/) {
	    $version=pop(@file_parts);
	}
	$name=join('.',@file_parts);
    } else {
	$name=join('.',@file_parts);
    }
    return($name,$version,$ext);
}

#--------------------------------------------------------------------------------------
#
#-------------------------- Next few routines handles grading by section or whole class
#
#--- Javascript to handle grading by section or whole class
sub viewgrades_js {
    my ($request) = shift;

    my $alertmsg = &mt('A number equal or greater than 0 is expected. Entered value = ');
    &js_escape(\$alertmsg);
    $request->print(&Apache::lonhtmlcommon::scripttag(<<VIEWJAVASCRIPT));
   function writePoint(partid,weight,point) {
	var radioButton = document.classgrade["RADVAL_"+partid];
	var textbox = document.classgrade["TEXTVAL_"+partid];
	if (point == "textval") {
	    point = document.classgrade["TEXTVAL_"+partid].value;
	    if (isNaN(point) || parseFloat(point) < 0) {
		alert("$alertmsg"+parseFloat(point));
		var resetbox = false;
		for (var i=0; i<radioButton.length; i++) {
		    if (radioButton[i].checked) {
			textbox.value = i;
			resetbox = true;
		    }
		}
		if (!resetbox) {
		    textbox.value = "";
		}
		return;
	    }
	    if (parseFloat(point) > parseFloat(weight)) {
		var resp = confirm("You entered a value ("+parseFloat(point)+
				   ") greater than the weight for the part. Accept?");
		if (resp == false) {
		    textbox.value = "";
		    return;
		}
	    }
	    for (var i=0; i<radioButton.length; i++) {
		radioButton[i].checked=false;
		if (parseFloat(point) == i) {
		    radioButton[i].checked=true;
		}
	    }

	} else {
	    textbox.value = parseFloat(point);
	}
	for (i=0;i<document.classgrade.total.value;i++) {
	    var user = document.classgrade["ctr"+i].value;
	    user = user.replace(new RegExp(':', 'g'),"_");
	    var scorename = document.classgrade["GD_"+user+"_"+partid+"_awarded"];
	    var saveval   = document.classgrade["GD_"+user+"_"+partid+"_solved_s"].value;
	    var selname   = document.classgrade["GD_"+user+"_"+partid+"_solved"];
	    if (saveval != "correct") {
		scorename.value = point;
		if (selname[0].selected != true) {
		    selname[0].selected = true;
		}
	    }
	}
	document.classgrade["SELVAL_"+partid][0].selected = true;
    }

    function writeRadText(partid,weight) {
	var selval   = document.classgrade["SELVAL_"+partid];
	var radioButton = document.classgrade["RADVAL_"+partid];
        var override = document.classgrade["FORCE_"+partid].checked;
	var textbox = document.classgrade["TEXTVAL_"+partid];
	if (selval[1].selected || selval[2].selected) {
	    for (var i=0; i<radioButton.length; i++) {
		radioButton[i].checked=false;

	    }
	    textbox.value = "";

	    for (i=0;i<document.classgrade.total.value;i++) {
		var user = document.classgrade["ctr"+i].value;
		user = user.replace(new RegExp(':', 'g'),"_");
		var scorename = document.classgrade["GD_"+user+"_"+partid+"_awarded"];
		var saveval   = document.classgrade["GD_"+user+"_"+partid+"_solved_s"].value;
		var selname   = document.classgrade["GD_"+user+"_"+partid+"_solved"];
		if ((saveval != "correct") || override) {
		    scorename.value = "";
		    if (selval[1].selected) {
			selname[1].selected = true;
		    } else {
			selname[2].selected = true;
			if (Number(document.classgrade["GD_"+user+"_"+partid+"_tries"].value)) 
			{document.classgrade["GD_"+user+"_"+partid+"_tries"].value = '0';}
		    }
		}
	    }
	} else {
	    for (i=0;i<document.classgrade.total.value;i++) {
		var user = document.classgrade["ctr"+i].value;
		user = user.replace(new RegExp(':', 'g'),"_");
		var scorename = document.classgrade["GD_"+user+"_"+partid+"_awarded"];
		var saveval   = document.classgrade["GD_"+user+"_"+partid+"_solved_s"].value;
		var selname   = document.classgrade["GD_"+user+"_"+partid+"_solved"];
		if ((saveval != "correct") || override) {
		    scorename.value = document.classgrade["GD_"+user+"_"+partid+"_awarded_s"].value;
		    selname[0].selected = true;
		}
	    }
	}	    
    }

    function changeSelect(partid,user) {
	var selval = document.classgrade["GD_"+user+'_'+partid+"_solved"];
	var textbox = document.classgrade["GD_"+user+'_'+partid+"_awarded"];
	var point  = textbox.value;
	var weight = document.classgrade["weight_"+partid].value;

	if (isNaN(point) || parseFloat(point) < 0) {
	    alert("$alertmsg"+parseFloat(point));
	    textbox.value = "";
	    return;
	}
	if (parseFloat(point) > parseFloat(weight)) {
	    var resp = confirm("You entered a value ("+parseFloat(point)+
			       ") greater than the weight of the part. Accept?");
	    if (resp == false) {
		textbox.value = "";
		return;
	    }
	}
	selval[0].selected = true;
    }

    function changeOneScore(partid,user) {
	var selval = document.classgrade["GD_"+user+'_'+partid+"_solved"];
	if (selval[1].selected || selval[2].selected) {
	    document.classgrade["GD_"+user+'_'+partid+"_awarded"].value = "";
	    if (selval[2].selected) {
		document.classgrade["GD_"+user+'_'+partid+"_tries"].value = "0";
	    }
        }
    }

    function resetEntry(numpart) {
	for (ctpart=0;ctpart<numpart;ctpart++) {
	    var partid = document.classgrade["partid_"+ctpart].value;
	    var radioButton = document.classgrade["RADVAL_"+partid];
	    var textbox = document.classgrade["TEXTVAL_"+partid];
	    var selval  = document.classgrade["SELVAL_"+partid];
	    for (var i=0; i<radioButton.length; i++) {
		radioButton[i].checked=false;

	    }
	    textbox.value = "";
	    selval[0].selected = true;

	    for (i=0;i<document.classgrade.total.value;i++) {
		var user = document.classgrade["ctr"+i].value;
		user = user.replace(new RegExp(':', 'g'),"_");
		var resetscore = document.classgrade["GD_"+user+"_"+partid+"_awarded"];
		resetscore.value = document.classgrade["GD_"+user+"_"+partid+"_awarded_s"].value;
		var resettries = document.classgrade["GD_"+user+"_"+partid+"_tries"];
		resettries.value = document.classgrade["GD_"+user+"_"+partid+"_tries_s"].value;
		var saveselval   = document.classgrade["GD_"+user+"_"+partid+"_solved_s"].value;
		var selname   = document.classgrade["GD_"+user+"_"+partid+"_solved"];
		if (saveselval == "excused") {
		    if (selname[1].selected == false) { selname[1].selected = true;}
		} else {
		    if (selname[0].selected == false) {selname[0].selected = true};
		}
	    }
	}
    }

VIEWJAVASCRIPT
}

#--- show scores for a section or whole class w/ option to change/update a score
sub viewgrades {
    my ($request,$symb) = @_;
    &viewgrades_js($request);

    #need to make sure we have the correct data for later EXT calls, 
    #thus invalidate the cache
    &Apache::lonnet::devalidatecourseresdata(
                 $env{'course.'.$env{'request.course.id'}.'.num'},
                 $env{'course.'.$env{'request.course.id'}.'.domain'});
    &Apache::lonnet::clear_EXT_cache_status();

    my $result='<h3><span class="LC_info">'.&mt('Manual Grading').'</span></h3>';

    #view individual student submission form - called using Javascript viewOneStudent
    $result.=&jscriptNform($symb);

    #beginning of class grading form
    my $stu_status = join(':',&Apache::loncommon::get_env_multiple('form.Status'));
    $result.= '<form action="/adm/grades" method="post" name="classgrade">'."\n".
	'<input type="hidden" name="symb"    value="'.&Apache::lonenc::check_encrypt($symb).'" />'."\n".
	'<input type="hidden" name="command" value="editgrades" />'."\n".
	&build_section_inputs().
	'<input type="hidden" name="Status" value="'.$env{'stu_status'}.'" />'."\n".

    #retrieve selected groups
    my (@groups,$group_display);
    @groups = &Apache::loncommon::get_env_multiple('form.group');
    if (grep(/^all$/,@groups)) {
        @groups = ('all');
    } elsif (grep(/^none$/,@groups)) {
        @groups = ('none');
    } elsif (@groups > 0) {
        $group_display = join(', ',@groups);
    }

    my ($common_header,$specific_header,@sections,$section_display);
    if ($env{'request.course.sec'} ne '') {
        @sections = ($env{'request.course.sec'});
    } else {
        @sections = &Apache::loncommon::get_env_multiple('form.section');
    }

# Check if Save button should be usable
    my $disabled = ' disabled="disabled"';
    if ($perm{'mgr'}) {
        if (grep(/^all$/,@sections)) {
            undef($disabled);
        } else {
            foreach my $sec (@sections) {
                if (&canmodify($sec)) {
                    undef($disabled);
                    last;
                }
            }
        }
    }
    if (grep(/^all$/,@sections)) {
        @sections = ('all');
        if ($group_display) {
            $common_header = &mt('Assign Common Grade to Students in Group(s) [_1]',$group_display);
            $specific_header = &mt('Assign Grade to Specific Students in Group(s) [_1]',$group_display);
        } elsif (grep(/^none$/,@groups)) {
            $common_header = &mt('Assign Common Grade to Students not assigned to any groups');
            $specific_header = &mt('Assign Grade to Specific Students not assigned to any groups');
        } else {
            $common_header = &mt('Assign Common Grade to Class');
            $specific_header = &mt('Assign Grade to Specific Students in Class');
        }
    } elsif (grep(/^none$/,@sections)) {
        @sections = ('none');
        if ($group_display) {
            $common_header = &mt('Assign Common Grade to Students in no Section and in Group(s) [_1]',$group_display);
            $specific_header = &mt('Assign Grade to Specific Students in no Section and in Group(s)',$group_display);
        } elsif (grep(/^none$/,@groups)) {
            $common_header = &mt('Assign Common Grade to Students in no Section and in no Group');
            $specific_header = &mt('Assign Grade to Specific Students in no Section and in no Group');
        } else {
            $common_header = &mt('Assign Common Grade to Students in no Section');
            $specific_header = &mt('Assign Grade to Specific Students in no Section');
        }
    } else {
        $section_display = join (", ",@sections);
        if ($group_display) {
            $common_header = &mt('Assign Common Grade to Students in Section(s) [_1], and in Group(s) [_2]',
                                 $section_display,$group_display);
            $specific_header = &mt('Assign Grade to Specific Students in Section(s) [_1], and in Group(s) [_2]',
                                   $section_display,$group_display);
        } elsif (grep(/^none$/,@groups)) {
            $common_header = &mt('Assign Common Grade to Students in Section(s) [_1] and no Group',$section_display);
            $specific_header = &mt('Assign Grade to Specific Students in Section(s) [_1] and no Group',$section_display);
        } else {
            $common_header = &mt('Assign Common Grade to Students in Section(s) [_1]',$section_display);
            $specific_header = &mt('Assign Grade to Specific Students in Section(s) [_1]',$section_display);
        }
    }
    my %submit_types = &substatus_options();
    my $submission_status = $submit_types{$env{'form.submitonly'}};

    if ($env{'form.submitonly'} eq 'all') {
        $result.= '<h3>'.$common_header.'</h3>';
    } else {
        $result.= '<h3>'.$common_header.'&nbsp;'.&mt('(submission status: "[_1]")',$submission_status).'</h3>'; 
    }
    $result .= &Apache::loncommon::start_data_table();
    #radio buttons/text box for assigning points for a section or class.
    #handles different parts of a problem
    my $res_error;
    my ($partlist,$handgrade,$responseType) = &response_type($symb,\$res_error);
    if ($res_error) {
        return &navmap_errormsg();
    }
    my %weight = ();
    my $ctsparts = 0;
    my %seen = ();
    my @part_response_id = &flatten_responseType($responseType);
    foreach my $part_response_id (@part_response_id) {
    	my ($partid,$respid) = @{ $part_response_id };
	my $part_resp = join('_',@{ $part_response_id });
	next if $seen{$partid};
	$seen{$partid}++;
	my $wgt = &Apache::lonnet::EXT('resource.'.$partid.'.weight',$symb);
	$weight{$partid} = $wgt eq '' ? '1' : $wgt;

	my $display_part=&get_display_part($partid,$symb);
	my $radio.='<table border="0"><tr>';  
	my $ctr = 0;
	while ($ctr<=$weight{$partid}) { # display radio buttons in a nice table 10 across
	    $radio.= '<td><label><input type="radio" name="RADVAL_'.$partid.'" '.
		'onclick="javascript:writePoint(\''.$partid.'\','.$weight{$partid}.
		','.$ctr.')" />'.$ctr."</label></td>\n";
	    $result.=(($ctr+1)%10 == 0 ? '</tr><tr>' : '');
	    $ctr++;
	}
	$radio.='</tr></table>';
	my $line = '<input type="text" name="TEXTVAL_'.
	    $partid.'" size="4" '.'onchange="javascript:writePoint(\''.
		$partid.'\','.$weight{$partid}.',\'textval\')" /> /'.
	    $weight{$partid}.' '.&mt('(problem weight)').'</td>'."\n";
	$line.= '<td><b>'.&mt('Grade Status').':</b>'.
                '<select name="SELVAL_'.$partid.'" '.
	        'onchange="javascript:writeRadText(\''.$partid.'\','.
		$weight{$partid}.')"> '.
	    '<option selected="selected"> </option>'.
	    '<option value="excused">'.&mt('excused').'</option>'.
	    '<option value="reset status">'.&mt('reset status').'</option>'.
	    '</select></td>'.
            '<td><label><input type="checkbox" name="FORCE_'.$partid.'" />'.&mt('Override "Correct"').'</label>';
	$line.='<input type="hidden" name="partid_'.
	    $ctsparts.'" value="'.$partid.'" />'."\n";
	$line.='<input type="hidden" name="weight_'.
	    $partid.'" value="'.$weight{$partid}.'" />'."\n";

	$result.=
	    &Apache::loncommon::start_data_table_row()."\n".
	    '<td><b>'.&mt('Part:').'</b></td><td>'.$display_part.'</td><td><b>'.&mt('Points:').'</b></td><td>'.$radio.'</td><td>'.&mt('or').'</td><td>'.$line.'</td>'.
	    &Apache::loncommon::end_data_table_row()."\n";
	$ctsparts++;
    }
    $result.=&Apache::loncommon::end_data_table()."\n".
	'<input type="hidden" name="totalparts" value="'.$ctsparts.'" />';
    $result.='<input type="button" value="'.&mt('Revert to Default').'" '.
	'onclick="javascript:resetEntry('.$ctsparts.');" />';

    #table listing all the students in a section/class
    #header of table
    if ($env{'form.submitonly'} eq 'all') { 
        $result.= '<h3>'.$specific_header.'</h3>';
    } else {
        $result.= '<h3>'.$specific_header.'&nbsp;'.&mt('(submission status: "[_1]")',$submission_status).'</h3>';
    }
    $result.= &Apache::loncommon::start_data_table().
	      &Apache::loncommon::start_data_table_header_row().
	      '<th>'.&mt('No.').'</th>'.
	      '<th>'.&nameUserString('header')."</th>\n";
    my $partserror;
    my (@parts) = sort(&getpartlist($symb,\$partserror));
    if ($partserror) {
        return &navmap_errormsg();
    }
    my (undef,undef,$url)=&Apache::lonnet::decode_symb($symb);
    my @partids = ();
    foreach my $part (@parts) {
	my $display=&Apache::lonnet::metadata($url,$part.'.display');
        my $narrowtext = &mt('Tries');
	$display =~ s|^Number of Attempts|$narrowtext <br />|; # makes the column narrower
	if  (!$display) { $display = &Apache::lonnet::metadata($url,$part.'.name'); }
	my ($partid) = &split_part_type($part);
        push(@partids,$partid);
#
# FIXME: Looks like $display looks at English text
#
	my $display_part=&get_display_part($partid,$symb);
	if ($display =~ /^Partial Credit Factor/) {
	    $result.='<th>'.
                &mt('Score Part: [_1][_2](weight = [_3])',
                    $display_part,'<br />',$weight{$partid}).'</th>'."\n";
	    next;
	    
	} else {
	    if ($display =~ /Problem Status/) {
		my $grade_status_mt = &mt('Grade Status');
		$display =~ s{Problem Status}{$grade_status_mt<br />};
	    }
	    my $part_mt = &mt('Part:');
	    $display =~s{\[Part: \Q$partid\E\]}{$part_mt $display_part};
	}

	$result.='<th>'.$display.'</th>'."\n";
    }
    $result.=&Apache::loncommon::end_data_table_header_row();

    my %last_resets = 
	&get_last_resets($symb,$env{'request.course.id'},\@partids);

    #get info for each student
    #list all the students - with points and grade status
    my (undef,undef,$fullname) = &getclasslist(\@sections,'1',\@groups);
    my $ctr = 0;
    foreach (sort 
	     {
		 if (lc($$fullname{$a}) ne lc($$fullname{$b})) {
		     return (lc($$fullname{$a}) cmp lc($$fullname{$b}));
		 }
		 return $a cmp $b;
	     } (keys(%$fullname))) {
	$result.=&viewstudentgrade($symb,$env{'request.course.id'},
				   $_,$$fullname{$_},\@parts,\%weight,\$ctr,\%last_resets);
    }
    $result.=&Apache::loncommon::end_data_table();
    $result.='<input type="hidden" name="total" value="'.$ctr.'" />'."\n";
    $result.='<input type="button" value="'.&mt('Save').'"'.$disabled.' '.
	'onclick="javascript:submit();" target="_self" /></form>'."\n";
    if ($ctr == 0) {
        my $stu_status = join(' or ',&Apache::loncommon::get_env_multiple('form.Status'));
        $result='<h3><span class="LC_info">'.&mt('Manual Grading').'</span></h3>'.
                '<span class="LC_warning">';
        if ($env{'form.submitonly'} eq 'all') {
            if (grep(/^all$/,@sections)) {
                if (grep(/^all$/,@groups)) {
                    $result .= &mt('There are no students with enrollment status [_1] to modify or grade.',
                                   $stu_status);
                } elsif (grep(/^none$/,@groups)) {
                    $result .= &mt('There are no students with no group assigned and with enrollment status [_1] to modify or grade.',
                                   $stu_status);
                } else {
                    $result .= &mt('There are no students in group(s) [_1] with enrollment status [_2] to modify or grade.',
                                   $group_display,$stu_status);
                }
            } elsif (grep(/^none$/,@sections)) {
                if (grep(/^all$/,@groups)) {
                    $result .= &mt('There are no students in no section with enrollment status [_1] to modify or grade.',
                                   $stu_status);
                } elsif (grep(/^none$/,@groups)) {
                    $result .= &mt('There are no students in no section and no group with enrollment status [_1] to modify or grade.',
                                   $stu_status);
                } else {
                    $result .= &mt('There are no students in no section in group(s) [_1] with enrollment status [_2] to modify or grade.',
                                   $group_display,$stu_status);
                }
            } else {
                if (grep(/^all$/,@groups)) {
                    $result .= &mt('There are no students in section(s) [_1] with enrollment status [_2] to modify or grade.',
                                   $section_display,$stu_status);
                } elsif (grep(/^none$/,@groups)) {
                    $result .= &mt('There are no students in section(s) [_1] and no group with enrollment status [_2] to modify or grade.',
                                   $section_display,$stu_status);
                } else {
                    $result .= &mt('There are no students in section(s) [_1] and group(s) [_2] with enrollment status [_3] to modify or grade.',
                                   $section_display,$group_display,$stu_status);
                }
            }
        } else {
            if (grep(/^all$/,@sections)) {
                if (grep(/^all$/,@groups)) {
                    $result .= &mt('There are no students with enrollment status [_1] and submission status "[_2]" to modify or grade.',
                                   $stu_status,$submission_status);
                } elsif (grep(/^none$/,@groups)) {
                    $result .= &mt('There are no students with no group assigned with enrollment status [_1] and submission status "[_2]" to modify or grade.',
                                   $stu_status,$submission_status);
                } else {
                    $result .= &mt('There are no students in group(s) [_1] with enrollment status [_2] and submission status "[_3]" to modify or grade.',
                                   $group_display,$stu_status,$submission_status);
                }
            } elsif (grep(/^none$/,@sections)) {
                if (grep(/^all$/,@groups)) {
                    $result .= &mt('There are no students in no section with enrollment status [_1] and submission status "[_2]" to modify or grade.',
                                   $stu_status,$submission_status);
                } elsif (grep(/^none$/,@groups)) {
                    $result .= &mt('There are no students in no section and no group with enrollment status [_1] and submission status "[_2]" to modify or grade.',
                                   $stu_status,$submission_status);
                } else {
                    $result .= &mt('There are no students in no section in group(s) [_1] with enrollment status [_2] and submission status "[_3]" to modify or grade.',
                                   $group_display,$stu_status,$submission_status);
                }
            } else {
                if (grep(/^all$/,@groups)) {
                    $result .= &mt('There are no students in section(s) [_1] with enrollment status [_2] and submission status "[_3]" to modify or grade.',
                                   $section_display,$stu_status,$submission_status);
                } elsif (grep(/^none$/,@groups)) {
                    $result .= &mt('There are no students in section(s) [_1] and no group with enrollment status [_2] and submission status "[_3]" to modify or grade.',
                                   $section_display,$stu_status,$submission_status);
                } else {
                    $result .= &mt('There are no students in section(s) [_1] and group(s) [_2] with enrollment status [_3] and submission status "[_4]" to modify or grade.',
                                   $section_display,$group_display,$stu_status,$submission_status);
                }
            }
	}
	$result .= '</span><br />';
    }
    return $result;
}

#--- call by previous routine to display each student who satisfies submission filter.
sub viewstudentgrade {
    my ($symb,$courseid,$student,$fullname,$parts,$weight,$ctr,$last_resets) = @_;
    my ($uname,$udom) = split(/:/,$student);
    my %record=&Apache::lonnet::restore($symb,$courseid,$udom,$uname);
    my $submitonly = $env{'form.submitonly'};
    unless (($submitonly eq 'all') || ($submitonly eq 'queued')) {
        my %partstatus = ();
        if (ref($parts) eq 'ARRAY') {
            foreach my $apart (@{$parts}) {
                my ($part,$type) = &split_part_type($apart);
                my ($status,undef) = split(/_/,$record{"resource.$part.solved"},2);
                $status = 'nothing' if ($status eq '');
                $partstatus{$part}      = $status;
                my $subkey = "resource.$part.submitted_by";
                $partstatus{$subkey} = $record{$subkey} if ($record{$subkey} ne '');
            }
            my $submitted = 0;
            my $graded = 0;
            my $incorrect = 0;
            foreach my $key (keys(%partstatus)) {
                $submitted = 1 if ($partstatus{$key} ne 'nothing');
                $graded = 1 if ($partstatus{$key} =~ /^ungraded/);
                $incorrect = 1 if ($partstatus{$key} =~ /^incorrect/);

                my $partid = (split(/\./,$key))[1];
                if ($partstatus{'resource.'.$partid.'.'.$key.'.submitted_by'} ne '') {
                    $submitted = 0;
                }
            }
            return if (!$submitted && ($submitonly eq 'yes' ||
                                       $submitonly eq 'incorrect' ||
                                       $submitonly eq 'graded'));
            return if (!$graded && ($submitonly eq 'graded'));
            return if (!$incorrect && $submitonly eq 'incorrect');
        }
    }
    if ($submitonly eq 'queued') {
        my ($cdom,$cnum) = split(/_/,$courseid);
        my %queue_status =
            &Apache::bridgetask::get_student_status($symb,$cdom,$cnum,
                                                    $udom,$uname);
        return if (!defined($queue_status{'gradingqueue'}));
    }
    $$ctr++;
    my %aggregates = ();
    my $result=&Apache::loncommon::start_data_table_row().'<td align="right">'.
	'<input type="hidden" name="ctr'.($$ctr-1).'" value="'.$student.'" />'.
	"\n".$$ctr.'&nbsp;</td><td>&nbsp;'.
	'<a href="javascript:viewOneStudent(\''.$uname.'\',\''.$udom.
	'\');" target="_self">'.$fullname.'</a> '.
	'<span class="LC_internal_info">('.$uname.($env{'user.domain'} eq $udom ? '' : ':'.$udom).')</span></td>'."\n";
    $student=~s/:/_/; # colon doen't work in javascript for names
    foreach my $apart (@$parts) {
	my ($part,$type) = &split_part_type($apart);
	my $score=$record{"resource.$part.$type"};
        $result.='<td align="center">';
        my ($aggtries,$totaltries);
        unless (exists($aggregates{$part})) {
	    $totaltries = $record{'resource.'.$part.'.tries'};

	    $aggtries = $totaltries;
            if ($$last_resets{$part}) {  
                $aggtries = &get_num_tries(\%record,$$last_resets{$part},
					   $part);
            }
            $result.='<input type="hidden" name="'.
                'GD_'.$student.'_'.$part.'_aggtries" value="'.$aggtries.'" />'."\n";
            $result.='<input type="hidden" name="'.
                'GD_'.$student.'_'.$part.'_totaltries" value="'.$totaltries.'" />'."\n";
            $aggregates{$part} = 1;
        }
	if ($type eq 'awarded') {
	    my $pts = $score eq '' ? '' : &compute_points($score,$$weight{$part});
	    $result.='<input type="hidden" name="'.
		'GD_'.$student.'_'.$part.'_awarded_s" value="'.$pts.'" />'."\n";
	    $result.='<input type="text" name="'.
		'GD_'.$student.'_'.$part.'_awarded" '.
                'onchange="javascript:changeSelect(\''.$part.'\',\''.$student.
		'\')" value="'.$pts.'" size="4" /></td>'."\n";
	} elsif ($type eq 'solved') {
	    my ($status,$foo)=split(/_/,$score,2);
	    $status = 'nothing' if ($status eq '');
	    $result.='<input type="hidden" name="'.'GD_'.$student.'_'.
		$part.'_solved_s" value="'.$status.'" />'."\n";
	    $result.='&nbsp;<select name="'.
		'GD_'.$student.'_'.$part.'_solved" '.
                'onchange="javascript:changeOneScore(\''.$part.'\',\''.$student.'\')" >'."\n";
	    $result.= (($status eq 'excused') ? '<option> </option><option selected="selected" value="excused">'.&mt('excused').'</option>' 
		: '<option selected="selected"> </option><option value="excused">'.&mt('excused').'</option>')."\n";
	    $result.='<option value="reset status">'.&mt('reset status').'</option>';
	    $result.="</select>&nbsp;</td>\n";
	} else {
	    $result.='<input type="hidden" name="'.
		'GD_'.$student.'_'.$part.'_'.$type.'_s" value="'.$score.'" />'.
		    "\n";
	    $result.='<input type="text" name="'.
		'GD_'.$student.'_'.$part.'_'.$type.'" '.
		'value="'.$score.'" size="4" /></td>'."\n";
	}
    }
    $result.=&Apache::loncommon::end_data_table_row();
    return $result;
}

#--- change scores for all the students in a section/class
#    record does not get update if unchanged
sub editgrades {
    my ($request,$symb) = @_;

    my $section_display = join (", ",&Apache::loncommon::get_env_multiple('form.section'));
    my $title='<h2>'.&mt('Current Grade Status').'</h2>';
    $title.='<h4><b>'.&mt('Section:').'</b> '.$section_display.'</h4>'."\n";

    my $result= &Apache::loncommon::start_data_table().
	&Apache::loncommon::start_data_table_header_row().
	'<th rowspan="2" valign="middle">'.&mt('No.').'</th>'.
	'<th rowspan="2" valign="middle">'.&nameUserString('header')."</th>\n";
    my %scoreptr = (
		    'correct'  =>'correct_by_override',
		    'incorrect'=>'incorrect_by_override',
		    'excused'  =>'excused',
		    'ungraded' =>'ungraded_attempted',
                    'credited' =>'credit_attempted',
		    'nothing'  => '',
		    );
    my ($classlist,undef,$fullname) = &getclasslist($env{'form.section'},'0');

    my (@partid);
    my %weight = ();
    my %columns = ();
    my ($i,$ctr,$count,$rec_update) = (0,0,0,0);

    my $partserror;
    my (@parts) = sort(&getpartlist($symb,\$partserror));
    if ($partserror) {
        return &navmap_errormsg();
    }
    my $header;
    while ($ctr < $env{'form.totalparts'}) {
	my $partid = $env{'form.partid_'.$ctr};
	push(@partid,$partid);
	$weight{$partid} = $env{'form.weight_'.$partid};
	$ctr++;
    }
    my (undef,undef,$url) = &Apache::lonnet::decode_symb($symb);
    my $totcolspan = 0;
    foreach my $partid (@partid) {
	$header .= '<th align="center">'.&mt('Old Score').'</th>'.
	    '<th align="center">'.&mt('New Score').'</th>';
	$columns{$partid}=2;
	foreach my $stores (@parts) {
	    my ($part,$type) = &split_part_type($stores);
	    if ($part !~ m/^\Q$partid\E/) { next;}
	    if ($type eq 'awarded' || $type eq 'solved') { next; }
	    my $display=&Apache::lonnet::metadata($url,$stores.'.display');
	    $display =~ s/\[Part: \Q$part\E\]//;
            my $narrowtext = &mt('Tries');
	    $display =~ s/Number of Attempts/$narrowtext/;
	    $header .= '<th align="center">'.&mt('Old').' '.$display.'</th>'.
		'<th align="center">'.&mt('New').' '.$display.'</th>';
	    $columns{$partid}+=2;
	}
        $totcolspan += $columns{$partid};
    }
    foreach my $partid (@partid) {
	my $display_part=&get_display_part($partid,$symb);
	$result .= '<th colspan="'.$columns{$partid}.'" align="center">'.
	    &mt('Part: [_1] (Weight = [_2])',$display_part,$weight{$partid}).
	    '</th>';

    }
    $result .= &Apache::loncommon::end_data_table_header_row().
	&Apache::loncommon::start_data_table_header_row().
	$header.
	&Apache::loncommon::end_data_table_header_row();
    my @noupdate;
    my ($updateCtr,$noupdateCtr) = (1,1);
    my ($got_types,%queueable);
    for ($i=0; $i<$env{'form.total'}; $i++) {
	my $user = $env{'form.ctr'.$i};
	my ($uname,$udom)=split(/:/,$user);
	my %newrecord;
	my $updateflag = 0;
        my $usec=$classlist->{"$uname:$udom"}[5];
        my $canmodify = &canmodify($usec);
        my $line = '<td'.($canmodify?'':' colspan="2"').'>'.
                   &nameUserString(undef,$$fullname{$user},$uname,$udom).'</td>';
        if (!$canmodify) {
            push(@noupdate,
                 $line."<td colspan=\"$totcolspan\"><span class=\"LC_warning\">".
                 &mt('Not allowed to modify student')."</span></td>");
            next;
        }
        my %aggregate = ();
        my $aggregateflag = 0;
	$user=~s/:/_/; # colon doen't work in javascript for names
	foreach (@partid) {
	    my $old_aw    = $env{'form.GD_'.$user.'_'.$_.'_awarded_s'};
	    my $old_part_pcr = $old_aw/($weight{$_} ne '0' ? $weight{$_}:1);
	    my $old_part  = $old_aw eq '' ? '' : $old_part_pcr;
	    my $old_score = $scoreptr{$env{'form.GD_'.$user.'_'.$_.'_solved_s'}};
	    my $awarded   = $env{'form.GD_'.$user.'_'.$_.'_awarded'};
	    my $pcr       = $awarded/($weight{$_} ne '0' ? $weight{$_} : 1);
	    my $partial   = $awarded eq '' ? '' : $pcr;
	    my $score;
	    if ($partial eq '') {
		$score = $scoreptr{$env{'form.GD_'.$user.'_'.$_.'_solved_s'}};
	    } elsif ($partial > 0) {
		$score = 'correct_by_override';
	    } elsif ($partial == 0) {
		$score = 'incorrect_by_override';
	    }
	    my $dropMenu = $env{'form.GD_'.$user.'_'.$_.'_solved'};
	    $score = 'excused' if (($dropMenu eq 'excused') && ($score ne 'excused'));

	    $newrecord{'resource.'.$_.'.regrader'}=
		"$env{'user.name'}:$env{'user.domain'}";
	    if ($dropMenu eq 'reset status' &&
		$old_score ne '') { # ignore if no previous attempts => nothing to reset
		$newrecord{'resource.'.$_.'.tries'} = '';
		$newrecord{'resource.'.$_.'.solved'} = '';
		$newrecord{'resource.'.$_.'.award'} = '';
		$newrecord{'resource.'.$_.'.awarded'} = '';
		$updateflag = 1;
                if ($env{'form.GD_'.$user.'_'.$_.'_aggtries'} > 0) {
                    my $aggtries = $env{'form.GD_'.$user.'_'.$_.'_aggtries'};
                    my $totaltries = $env{'form.GD_'.$user.'_'.$_.'_totaltries'};
                    my $solvedstatus = $env{'form.GD_'.$user.'_'.$_.'_solved_s'};
                    &decrement_aggs($symb,$_,\%aggregate,$aggtries,$totaltries,$solvedstatus);
                    $aggregateflag = 1;
                }
	    } elsif (!($old_part eq $partial && $old_score eq $score)) {
		$updateflag = 1;
		$newrecord{'resource.'.$_.'.awarded'}  = $partial if $partial ne '';
		$newrecord{'resource.'.$_.'.solved'}   = $score;
		$rec_update++;
	    }

	    $line .= '<td align="center">'.$old_aw.'&nbsp;</td>'.
		'<td align="center">'.$awarded.
		($score eq 'excused' ? $score : '').'&nbsp;</td>';


	    my $partid=$_;
	    foreach my $stores (@parts) {
		my ($part,$type) = &split_part_type($stores);
		if ($part !~ m/^\Q$partid\E/) { next;}
		if ($type eq 'awarded' || $type eq 'solved') { next; }
		my $old_aw    = $env{'form.GD_'.$user.'_'.$part.'_'.$type.'_s'};
		my $awarded   = $env{'form.GD_'.$user.'_'.$part.'_'.$type};
		if ($awarded ne '' && $awarded ne $old_aw) {
		    $newrecord{'resource.'.$part.'.'.$type}= $awarded;
		    $newrecord{'resource.'.$part.'.regrader'}="$env{'user.name'}:$env{'user.domain'}";
		    $updateflag=1;
		}
		$line .= '<td align="center">'.$old_aw.'&nbsp;</td>'.
		    '<td align="center">'.$awarded.'&nbsp;</td>';
	    }
	}
	$line.="\n";

	my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
	my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};

	if ($updateflag) {
	    $count++;
	    &Apache::lonnet::cstore(\%newrecord,$symb,$env{'request.course.id'},
				    $udom,$uname);

	    if (&Apache::bridgetask::in_queue('gradingqueue',$symb,$cdom,
					      $cnum,$udom,$uname)) {
		# need to figure out if should be in queue.
		my %record =  
		    &Apache::lonnet::restore($symb,$env{'request.course.id'},
					     $udom,$uname);
		my $all_graded = 1;
		my $none_graded = 1;
                unless ($got_types) {
                    my $error;
                    my ($plist,$handgrd,$resptype) = &response_type($symb,\$error);
                    unless ($error) {
                        foreach my $part (@parts) {
                            if (ref($resptype->{$part}) eq 'HASH') {
                                foreach my $id (keys(%{$resptype->{$part}})) {
                                    if (($resptype->{$part}->{$id} eq 'essay') ||
                                        (lc($handgrd->{$part.'_'.$id}) eq 'yes')) {
                                        $queueable{$part} = 1;
                                        last;
                                    }
                                }
                            }
                        }
                    }
                    $got_types = 1;
                }
		foreach my $part (@parts) {
                    if ($queueable{$part}) {
		        if ( $record{'resource.'.$part.'.awarded'} eq '' ) {
			    $all_graded = 0;
		        } else {
			    $none_graded = 0;
		        }
                    }
		}

		if ($all_graded || $none_graded) {
		    &Apache::bridgetask::remove_from_queue('gradingqueue',
							   $symb,$cdom,$cnum,
							   $udom,$uname);
		}
	    }

	    $result.=&Apache::loncommon::start_data_table_row().
		'<td align="right">&nbsp;'.$updateCtr.'&nbsp;</td>'.$line.
		&Apache::loncommon::end_data_table_row();
	    $updateCtr++;
	} else {
	    push(@noupdate,
		 '<td align="right">&nbsp;'.$noupdateCtr.'&nbsp;</td>'.$line);
	    $noupdateCtr++;
	}
        if ($aggregateflag) {
            &Apache::lonnet::cinc('nohist_resourcetracker',\%aggregate,
				  $cdom,$cnum);
        }
    }
    if (@noupdate) {
        my $numcols=$totcolspan+2;
	$result .= &Apache::loncommon::start_data_table_row('LC_empty_row').
	    '<td align="center" colspan="'.$numcols.'">'.
	    &mt('No Changes Occurred For the Students Below').
	    '</td>'.
	    &Apache::loncommon::end_data_table_row();
	foreach my $line (@noupdate) {
	    $result.=
		&Apache::loncommon::start_data_table_row().
		$line.
		&Apache::loncommon::end_data_table_row();
	}
    }
    $result .= &Apache::loncommon::end_data_table();
    my $msg = '<p><b>'.
	&mt('Number of records updated = [_1] for [quant,_2,student].',
	    $rec_update,$count).'</b><br />'.
	'<b>'.&mt('Total number of students = [_1]',$env{'form.total'}).
	'</b></p>';
    return $title.$msg.$result;
}

sub split_part_type {
    my ($partstr) = @_;
    my ($temp,@allparts)=split(/_/,$partstr);
    my $type=pop(@allparts);
    my $part=join('_',@allparts);
    return ($part,$type);
}

#------------- end of section for handling grading by section/class ---------
#
#----------------------------------------------------------------------------


#----------------------------------------------------------------------------
#
#-------------------------- Next few routines handles grading by csv upload
#
#--- Javascript to handle csv upload
sub csvupload_javascript_reverse_associate {
    my $error1=&mt('You need to specify the username or the student/employee ID');
    my $error2=&mt('You need to specify at least one grading field');
  &js_escape(\$error1);
  &js_escape(\$error2);
  return(<<ENDPICK);
  function verify(vf) {
    var foundsomething=0;
    var founduname=0;
    var foundID=0;
    for (i=0;i<=vf.nfields.value;i++) {
      tw=eval('vf.f'+i+'.selectedIndex');
      if (i==0 && tw!=0) { foundID=1; }
      if (i==1 && tw!=0) { founduname=1; }
      if (i!=0 && i!=1 && i!=2 && tw!=0) { foundsomething=1; }
    }
    if (founduname==0 && foundID==0) {
	alert('$error1');
	return;
    }
    if (foundsomething==0) {
	alert('$error2');
	return;
    }
    vf.submit();
  }
  function flip(vf,tf) {
    var nw=eval('vf.f'+tf+'.selectedIndex');
    var i;
    for (i=0;i<=vf.nfields.value;i++) {
      //can not pick the same destination field for both name and domain
      if (((i ==0)||(i ==1)) && 
          ((tf==0)||(tf==1)) && 
          (i!=tf) &&
          (eval('vf.f'+i+'.selectedIndex')==nw)) {
        eval('vf.f'+i+'.selectedIndex=0;')
      }
    }
  }
ENDPICK
}

sub csvupload_javascript_forward_associate {
    my $error1=&mt('You need to specify the username or the student/employee ID');
    my $error2=&mt('You need to specify at least one grading field');
  &js_escape(\$error1);
  &js_escape(\$error2);
  return(<<ENDPICK);
  function verify(vf) {
    var foundsomething=0;
    var founduname=0;
    var foundID=0;
    for (i=0;i<=vf.nfields.value;i++) {
      tw=eval('vf.f'+i+'.selectedIndex');
      if (tw==1) { foundID=1; }
      if (tw==2) { founduname=1; }
      if (tw>3) { foundsomething=1; }
    }
    if (founduname==0 && foundID==0) {
	alert('$error1');
	return;
    }
    if (foundsomething==0) {
	alert('$error2');
	return;
    }
    vf.submit();
  }
  function flip(vf,tf) {
    var nw=eval('vf.f'+tf+'.selectedIndex');
    var i;
    //can not pick the same destination field twice
    for (i=0;i<=vf.nfields.value;i++) {
      if ((i!=tf) && (eval('vf.f'+i+'.selectedIndex')==nw)) {
        eval('vf.f'+i+'.selectedIndex=0;')
      }
    }
  }
ENDPICK
}

sub csvuploadmap_header {
    my ($request,$symb,$datatoken,$distotal)= @_;
    my $javascript;
    if ($env{'form.upfile_associate'} eq 'reverse') {
	$javascript=&csvupload_javascript_reverse_associate();
    } else {
	$javascript=&csvupload_javascript_forward_associate();
    }

    my $checked=(($env{'form.noFirstLine'})?' checked="checked"':'');
    my $ignore=&mt('Ignore First Line');
    $symb = &Apache::lonenc::check_encrypt($symb);
    $request->print('<form method="post" enctype="multipart/form-data" action="/adm/grades" name="gradesupload">'.
                    &mt('Total number of records found in file: [_1]',$distotal).'<hr />'.
                    &mt('Associate entries from the uploaded file with as many fields as you can.'));
    my $reverse=&mt("Reverse Association");
    $request->print(<<ENDPICK);
<br />
<input type="button" value="$reverse" onclick="javascript:this.form.associate.value='Reverse Association';submit(this.form);" />
<label><input type="checkbox" name="noFirstLine" $checked />$ignore</label>
<input type="hidden" name="associate"  value="" />
<input type="hidden" name="phase"      value="three" />
<input type="hidden" name="datatoken"  value="$datatoken" />
<input type="hidden" name="fileupload" value="$env{'form.fileupload'}" />
<input type="hidden" name="upfiletype" value="$env{'form.upfiletype'}" />
<input type="hidden" name="upfile_associate" 
                                       value="$env{'form.upfile_associate'}" />
<input type="hidden" name="symb"       value="$symb" />
<input type="hidden" name="command"    value="csvuploadoptions" />
<hr />
ENDPICK
    $request->print(&Apache::lonhtmlcommon::scripttag($javascript));
    return '';

}

sub csvupload_fields {
    my ($symb,$errorref) = @_;
    my (@parts) = &getpartlist($symb,$errorref);
    if (ref($errorref)) {
        if ($$errorref) {
            return;
        }
    }

    my @fields=(['ID','Student/Employee ID'],
		['username','Student Username'],
		['domain','Student Domain']);
    my (undef,undef,$url) = &Apache::lonnet::decode_symb($symb);
    foreach my $part (sort(@parts)) {
	my @datum;
	my $display=&Apache::lonnet::metadata($url,$part.'.display');
	my $name=$part;
	if  (!$display) { $display = $name; }
	@datum=($name,$display);
	if ($name=~/^stores_(.*)_awarded/) {
	    push(@fields,['stores_'.$1.'_points',"Points [Part: $1]"]);
	}
	push(@fields,\@datum);
    }
    return (@fields);
}

sub csvuploadmap_footer {
    my ($request,$i,$keyfields) =@_;
    my $buttontext = &mt('Assign Grades');
    $request->print(<<ENDPICK);
</table>
<input type="hidden" name="nfields" value="$i" />
<input type="hidden" name="keyfields" value="$keyfields" />
<input type="button" onclick="javascript:verify(this.form)" value="$buttontext" /><br />
</form>
ENDPICK
}

sub checkforfile_js {
    my $alertmsg = &mt('Please use the browse button to select a file from your local directory.');
    &js_escape(\$alertmsg);
    my $result = &Apache::lonhtmlcommon::scripttag(<<CSVFORMJS);
    function checkUpload(formname) {
	if (formname.upfile.value == "") {
	    alert("$alertmsg");
	    return false;
	}
	formname.submit();
    }
CSVFORMJS
    return $result;
}

sub upcsvScores_form {
    my ($request,$symb) = @_;
    if (!$symb) {return '';}
    my $result=&checkforfile_js();
    $result.=&Apache::loncommon::start_data_table().
             &Apache::loncommon::start_data_table_header_row().
             '<th>'.&mt('Specify a file containing the class scores for current resource.').'</th>'.
             &Apache::loncommon::end_data_table_header_row().
             &Apache::loncommon::start_data_table_row().'<td>';
    my $upload=&mt("Upload Scores");
    my $upfile_select=&Apache::loncommon::upfile_select_html();
    my $ignore=&mt('Ignore First Line');
    $symb = &Apache::lonenc::check_encrypt($symb);
    $result.=<<ENDUPFORM;
<form method="post" enctype="multipart/form-data" action="/adm/grades" name="gradesupload">
<input type="hidden" name="symb" value="$symb" />
<input type="hidden" name="command" value="csvuploadmap" />
$upfile_select
<br /><input type="button" onclick="javascript:checkUpload(this.form);" value="$upload" />
<label><input type="checkbox" name="noFirstLine" />$ignore</label>
</form>
ENDUPFORM
    $result.=&Apache::loncommon::help_open_topic("Course_Convert_To_CSV",
                           &mt("How do I create a CSV file from a spreadsheet")).
            '</td>'.
            &Apache::loncommon::end_data_table_row().
            &Apache::loncommon::end_data_table();
    return $result;
}


sub csvuploadmap {
    my ($request,$symb) = @_;
    if (!$symb) {return '';}

    my $datatoken;
    if (!$env{'form.datatoken'}) {
	$datatoken=&Apache::loncommon::upfile_store($request);
    } else {
        $datatoken=&Apache::loncommon::valid_datatoken($env{'form.datatoken'});
        if ($datatoken ne '') { 
	    &Apache::loncommon::load_tmp_file($request,$datatoken);
        }
    }
    my @records=&Apache::loncommon::upfile_record_sep();
    if ($env{'form.noFirstLine'}) { shift(@records); }
    &csvuploadmap_header($request,$symb,$datatoken,$#records+1);
    my ($i,$keyfields);
    if (@records) {
        my $fieldserror;
	my @fields=&csvupload_fields($symb,\$fieldserror);
        if ($fieldserror) {
            $request->print(&navmap_errormsg());
            return;
        }
	if ($env{'form.upfile_associate'} eq 'reverse') {	
	    &Apache::loncommon::csv_print_samples($request,\@records);
	    $i=&Apache::loncommon::csv_print_select_table($request,\@records,
							  \@fields);
	    foreach (@fields) { $keyfields.=$_->[0].','; }
	    chop($keyfields);
	} else {
	    unshift(@fields,['none','']);
	    $i=&Apache::loncommon::csv_samples_select_table($request,\@records,
							    \@fields);
            foreach my $rec (@records) {
                my %temp = &Apache::loncommon::record_sep($rec);
                if (%temp) {
                    $keyfields=join(',',sort(keys(%temp)));
                    last;
                }
            }
	}
    }
    &csvuploadmap_footer($request,$i,$keyfields);

    return '';
}

sub csvuploadoptions {
    my ($request,$symb)= @_;
    my $overwrite=&mt('Overwrite any existing score');
    my $checked=(($env{'form.noFirstLine'})?'1':'0');
    my $ignore=&mt('Ignore First Line');
    $request->print(<<ENDPICK);
<form method="post" enctype="multipart/form-data" action="/adm/grades" name="gradesupload">
<input type="hidden" name="command"    value="csvuploadassign" />
<p>
<label>
   <input type="checkbox" name="overwite_scores" checked="checked" />
   $overwrite
</label>
</p>
ENDPICK
    my %fields=&get_fields();
    if (!defined($fields{'domain'})) {
	my $domform = &Apache::loncommon::select_dom_form($env{'request.role.domain'},'default_domain');
        $request->print("\n<p>".&mt('Users are in domain: [_1]',$domform)."</p>\n");
    }
    foreach my $key (sort(keys(%env))) {
	if ($key !~ /^form\.(.*)$/) { next; }
	my $cleankey=$1;
	if ($cleankey eq 'command') { next; }
	$request->print('<input type="hidden" name="'.$cleankey.
			'"  value="'.$env{$key}.'" />'."\n");
    }
    # FIXME do a check for any duplicated user ids...
    # FIXME do a check for any invalid user ids?...
    $request->print('<input type="submit" value="'.&mt('Assign Grades').'" /><br />
<hr /></form>'."\n");
    return '';
}

sub get_fields {
    my %fields;
    my @keyfields = split(/\,/,$env{'form.keyfields'});
    for (my $i=0; $i<=$env{'form.nfields'}; $i++) {
	if ($env{'form.upfile_associate'} eq 'reverse') {
	    if ($env{'form.f'.$i} ne 'none') {
		$fields{$keyfields[$i]}=$env{'form.f'.$i};
	    }
	} else {
	    if ($env{'form.f'.$i} ne 'none') {
		$fields{$env{'form.f'.$i}}=$keyfields[$i];
	    }
	}
    }
    return %fields;
}

sub csvuploadassign {
    my ($request,$symb) = @_;
    if (!$symb) {return '';}
    my $error_msg = '';
    my $datatoken = &Apache::loncommon::valid_datatoken($env{'form.datatoken'});
    if ($datatoken ne '') {
        &Apache::loncommon::load_tmp_file($request,$datatoken);
    }
    my @gradedata = &Apache::loncommon::upfile_record_sep();
    if ($env{'form.noFirstLine'}) { shift(@gradedata); }
    my %fields=&get_fields();
    my $courseid=$env{'request.course.id'};
    my ($classlist) = &getclasslist('all',0);
    my @notallowed;
    my @skipped;
    my @warnings;
    my $countdone=0;
    foreach my $grade (@gradedata) {
	my %entries=&Apache::loncommon::record_sep($grade);
	my $domain;
	if ($entries{$fields{'domain'}}) {
	    $domain=$entries{$fields{'domain'}};
	} else {
	    $domain=$env{'form.default_domain'};
	}
	$domain=~s/\s//g;
	my $username=$entries{$fields{'username'}};
	$username=~s/\s//g;
	if (!$username) {
	    my $id=$entries{$fields{'ID'}};
	    $id=~s/\s//g;
	    my %ids=&Apache::lonnet::idget($domain,$id);
	    $username=$ids{$id};
	}
	if (!exists($$classlist{"$username:$domain"})) {
	    my $id=$entries{$fields{'ID'}};
	    $id=~s/\s//g;
	    if ($id) {
		push(@skipped,"$id:$domain");
	    } else {
		push(@skipped,"$username:$domain");
	    }
	    next;
	}
	my $usec=$classlist->{"$username:$domain"}[5];
	if (!&canmodify($usec)) {
	    push(@notallowed,"$username:$domain");
	    next;
	}
	my %points;
	my %grades;
	foreach my $dest (keys(%fields)) {
	    if ($dest eq 'ID' || $dest eq 'username' ||
		$dest eq 'domain') { next; }
	    if ($entries{$fields{$dest}} =~ /^\s*$/) { next; }
	    if ($dest=~/stores_(.*)_points/) {
		my $part=$1;
		my $wgt =&Apache::lonnet::EXT('resource.'.$part.'.weight',
					      $symb,$domain,$username);
                if ($wgt) {
                    $entries{$fields{$dest}}=~s/\s//g;
                    my $pcr=$entries{$fields{$dest}} / $wgt;
                    my $award=($pcr == 0) ? 'incorrect_by_override'
                                          : 'correct_by_override';
                    if ($pcr>1) {
                        push(@warnings,&mt("[_1]: point value larger than weight","$username:$domain"));
                    }
                    $grades{"resource.$part.awarded"}=$pcr;
                    $grades{"resource.$part.solved"}=$award;
                    $points{$part}=1;
                } else {
                    $error_msg = "<br />" .
                        &mt("Some point values were assigned"
                            ." for problems with a weight "
                            ."of zero. These values were "
                            ."ignored.");
                }
	    } else {
		if ($dest=~/stores_(.*)_awarded/) { if ($points{$1}) {next;} }
		if ($dest=~/stores_(.*)_solved/)  { if ($points{$1}) {next;} }
		my $store_key=$dest;
		$store_key=~s/^stores/resource/;
		$store_key=~s/_/\./g;
		$grades{$store_key}=$entries{$fields{$dest}};
	    }
	}
	if (! %grades) {
           push(@skipped,&mt("[_1]: no data to save","$username:$domain")); 
        } else {
	   $grades{"resource.regrader"}="$env{'user.name'}:$env{'user.domain'}";
	   my $result=&Apache::lonnet::cstore(\%grades,$symb,
					   $env{'request.course.id'},
					   $domain,$username);
	   if ($result eq 'ok') {
# Successfully stored
	      $request->print('.');
# Remove from grading queue
              &Apache::bridgetask::remove_from_queue('gradingqueue',$symb,
                                             $env{'course.'.$env{'request.course.id'}.'.domain'},
                                             $env{'course.'.$env{'request.course.id'}.'.num'},
                                             $domain,$username);
              $countdone++;
	   } else {
	      $request->print("<p><span class=\"LC_error\">".
                              &mt("Failed to save data for student [_1]. Message when trying to save was: [_2]",
                                  "$username:$domain",$result)."</span></p>");
	   }
	   $request->rflush();
        }
    }
    $request->print('<br />'.&Apache::lonhtmlcommon::confirm_success(&mt("Saved scores for [quant,_1,student]",$countdone),$countdone==0));
    if (@warnings) {
        $request->print('<br />'.&Apache::lonhtmlcommon::confirm_success(&mt('Warnings generated for the following saved scores:'),1).'<br />');
        $request->print(join(', ',@warnings));
    }
    if (@skipped) {
	$request->print('<br />'.&Apache::lonhtmlcommon::confirm_success(&mt('No scores stored for the following username(s):'),1).'<br />');
        $request->print(join(', ',@skipped));
    }
    if (@notallowed) {
	$request->print('<br />'.&Apache::lonhtmlcommon::confirm_success(&mt('Modification of scores not allowed for the following username(s):'),1).'<br />');
	$request->print(join(', ',@notallowed));
    }
    $request->print("<br />\n");
    return $error_msg;
}
#------------- end of section for handling csv file upload ---------
#
#-------------------------------------------------------------------
#
#-------------- Next few routines handle grading by page/sequence
#
#--- Select a page/sequence and a student to grade
sub pickStudentPage {
    my ($request,$symb) = @_;

    my $alertmsg = &mt('Please select the student you wish to grade.');
    &js_escape(\$alertmsg);
    $request->print(&Apache::lonhtmlcommon::scripttag(<<LISTJAVASCRIPT));

function checkPickOne(formname) {
    if (radioSelection(formname.student) == null) {
	alert("$alertmsg");
	return;
    }
    ptr = pullDownSelection(formname.selectpage);
    formname.page.value = formname["page"+ptr].value;
    formname.title.value = formname["title"+ptr].value;
    formname.submit();
}

LISTJAVASCRIPT
    &commonJSfunctions($request);

    my $cdom      = $env{"course.$env{'request.course.id'}.domain"};
    my $cnum      = $env{"course.$env{'request.course.id'}.num"};
    my $getsec    = $env{'form.section'} eq '' ? 'all' : $env{'form.section'};
    my $getgroup  = $env{'form.group'} eq '' ? 'all' : $env{'form.group'};

    my $result='<h3><span class="LC_info">&nbsp;'.
	&mt('Manual Grading by Page or Sequence').'</span></h3>';

    $result.='<form action="/adm/grades" method="post" name="displayPage">'."\n";
    my $map_error;
    my ($titles,$symbx) = &getSymbMap($map_error);
    if ($map_error) {
        $request->print(&navmap_errormsg());
        return; 
    }
    my ($curpage) =&Apache::lonnet::decode_symb($symb); 
#    my ($curpage,$mapId) =&Apache::lonnet::decode_symb($symb); 
#    my $type=($curpage =~ /\.(page|sequence)/);

    # Collection of hidden fields
    my $ctr=0;
    foreach (@$titles) {
	my ($minder,$showtitle) = ($_ =~ /(\d+)\.(.*)/);
	$result.='<input type="hidden" name="page'.$ctr.'" value="'.$$symbx{$_}.'" />'."\n";
	$result.='<input type="hidden" name="title'.$ctr.'" value="'.$showtitle.'" />'."\n";
	$ctr++;
    }
    $result.='<input type="hidden" name="page" />'."\n".
	'<input type="hidden" name="title" />'."\n";

    $result.=&build_section_inputs();
    my $stu_status = join(':',&Apache::loncommon::get_env_multiple('form.Status'));
    $result.='<input type="hidden" name="Status"  value="'.$stu_status.'" />'."\n".
        '<input type="hidden" name="command" value="displayPage" />'."\n".
        '<input type="hidden" name="symb"    value="'.&Apache::lonenc::check_encrypt($symb).'" />'."\n";

    # Show grading options
    $result.=&Apache::lonhtmlcommon::start_pick_box();
    my $select = '<select name="selectpage">'."\n";
    $ctr=0;
    foreach (@$titles) {
        my ($minder,$showtitle) = ($_ =~ /(\d+)\.(.*)/);
        $select.='<option value="'.$ctr.'"'.
            ($$symbx{$_} =~ /$curpage$/ ? ' selected="selected"' : '').
            '>'.$showtitle.'</option>'."\n";
        $ctr++;
    }
    $select.= '</select>';

    $result.=
        &Apache::lonhtmlcommon::row_title(&mt('Problems from'))
       .$select
       .&Apache::lonhtmlcommon::row_closure();

    $result.=
        &Apache::lonhtmlcommon::row_title(&mt('View Problem Text'))
       .'<label><input type="radio" name="vProb" value="no"'
           .' checked="checked" /> '.&mt('no').' </label>'."\n"
       .'<label><input type="radio" name="vProb" value="yes" />'
           .&mt('yes').'</label>'."\n"
       .&Apache::lonhtmlcommon::row_closure();

    $result.=
        &Apache::lonhtmlcommon::row_title(&mt('View Submissions'))
       .'<label><input type="radio" name="lastSub" value="none" /> '
           .&mt('none').' </label>'."\n"
       .'<label><input type="radio" name="lastSub" value="datesub"'
           .' checked="checked" /> '.&mt('all submissions').'</label>'."\n"
       .'<label><input type="radio" name="lastSub" value="all" /> '
           .&mt('all submissions with details').' </label>'
       .&Apache::lonhtmlcommon::row_closure();

    $result.=
        &Apache::lonhtmlcommon::row_title(&mt('Use CODE'))
       .'<input type="text" name="CODE" value="" />'
       .&Apache::lonhtmlcommon::row_closure(1)
       .&Apache::lonhtmlcommon::end_pick_box();

    # Show list of students to select for grading
    $result.='<br /><input type="button" '.
             'onclick="javascript:checkPickOne(this.form);" value="'.&mt('Next').' &rarr;" /><br />'."\n";

    $request->print($result);

    my $studentTable.='&nbsp;<b>'.&mt('Select a student you wish to grade and then click on the Next button.').'</b><br />'.
	&Apache::loncommon::start_data_table().
	&Apache::loncommon::start_data_table_header_row().
	'<th align="right">&nbsp;'.&mt('No.').'</th>'.
	'<th>'.&nameUserString('header').'</th>'.
	'<th align="right">&nbsp;'.&mt('No.').'</th>'.
	'<th>'.&nameUserString('header').'</th>'.
	&Apache::loncommon::end_data_table_header_row();
 
    my (undef,undef,$fullname) = &getclasslist($getsec,'1',$getgroup);
    my $ptr = 1;
    foreach my $student (sort 
			 {
			     if (lc($$fullname{$a}) ne lc($$fullname{$b})) {
				 return (lc($$fullname{$a}) cmp lc($$fullname{$b}));
			     }
			     return $a cmp $b;
			 } (keys(%$fullname))) {
	my ($uname,$udom) = split(/:/,$student);
	$studentTable.=($ptr%2==1 ? &Apache::loncommon::start_data_table_row()
                                  : '</td>');
	$studentTable.='<td align="right">'.$ptr.'&nbsp;</td>';
	$studentTable.='<td>&nbsp;<label><input type="radio" name="student" value="'.$student.'" /> '
	    .&nameUserString(undef,$$fullname{$student},$uname,$udom)."</label>\n";
	$studentTable.=
	    ($ptr%2 == 0 ? '</td>'.&Apache::loncommon::end_data_table_row() 
                         : '');
	$ptr++;
    }
    if ($ptr%2 == 0) {
	$studentTable.='</td><td>&nbsp;</td><td>&nbsp;</td>'.
	    &Apache::loncommon::end_data_table_row();
    }
    $studentTable.=&Apache::loncommon::end_data_table()."\n";
    $studentTable.='<input type="button" '.
                   'onclick="javascript:checkPickOne(this.form);" value="'.&mt('Next').' &rarr;" /></form>'."\n";

    $request->print($studentTable);

    return '';
}

sub getSymbMap {
    my ($map_error) = @_;
    my $navmap = Apache::lonnavmaps::navmap->new();
    unless (ref($navmap)) {
        if (ref($map_error)) {
            $$map_error = 'navmap';
        }
        return;
    }
    my %symbx = ();
    my @titles = ();
    my $minder = 0;

    # Gather every sequence that has problems.
    my @sequences = $navmap->retrieveResources(undef, sub { shift->is_map(); },
					       1,0,1);
    for my $sequence ($navmap->getById('0.0'), @sequences) {
	if ($navmap->hasResource($sequence, sub { shift->is_problem(); }, 0) ) {
	    my $title = $minder.'.'.
		&HTML::Entities::encode($sequence->compTitle(),'"\'&');
	    push(@titles, $title); # minder in case two titles are identical
	    $symbx{$title} = &HTML::Entities::encode($sequence->symb(),'"\'&');
	    $minder++;
	}
    }
    return \@titles,\%symbx;
}

#
#--- Displays a page/sequence w/wo problems, w/wo submissions
sub displayPage {
    my ($request,$symb) = @_;
    my $cdom      = $env{"course.$env{'request.course.id'}.domain"};
    my $cnum      = $env{"course.$env{'request.course.id'}.num"};
    my $getsec    = $env{'form.section'} eq '' ? 'all' : $env{'form.section'};
    my $pageTitle = $env{'form.page'};
    my ($classlist,undef,$fullname) = &getclasslist($getsec,'1');
    my ($uname,$udom) = split(/:/,$env{'form.student'});
    my $usec=$classlist->{$env{'form.student'}}[5];

    #need to make sure we have the correct data for later EXT calls, 
    #thus invalidate the cache
    &Apache::lonnet::devalidatecourseresdata(
                 $env{'course.'.$env{'request.course.id'}.'.num'},
                 $env{'course.'.$env{'request.course.id'}.'.domain'});
    &Apache::lonnet::clear_EXT_cache_status();

    if (!&canview($usec)) {
	$request->print(
            '<span class="LC_warning">'.
            &mt('Unable to view requested student. ([_1])',
                $env{'form.student'}).
            '</span>');
        return;
    }
    my $result='<h3><span class="LC_info">&nbsp;'.$env{'form.title'}.'</span></h3>';
    $result.='<h3>&nbsp;'.&mt('Student: [_1]',&nameUserString(undef,$$fullname{$env{'form.student'}},$uname,$udom)).
	'</h3>'."\n";
    $env{'form.CODE'} = uc($env{'form.CODE'});
    if (&Apache::lonnet::validCODE(uc($env{'form.CODE'}))) {
	$result.='<h3>&nbsp;'.&mt('CODE: [_1]',$env{'form.CODE'}).'</h3>'."\n";
    } else {
	delete($env{'form.CODE'});
    }
    &sub_page_js($request);
    $request->print($result);

    my $navmap = Apache::lonnavmaps::navmap->new();
    unless (ref($navmap)) {
        $request->print(&navmap_errormsg());
        return;
    }
    my ($mapUrl, $id, $resUrl)=&Apache::lonnet::decode_symb($env{'form.page'});
    my $map = $navmap->getResourceByUrl($resUrl); # add to navmaps
    if (!$map) {
	$request->print('<span class="LC_warning">'.&mt('Unable to view requested sequence. ([_1])',$resUrl).'</span>');
	return; 
    }
    my $iterator = $navmap->getIterator($map->map_start(),
					$map->map_finish());

    my $studentTable='<form action="/adm/grades" method="post" name="gradePage">'."\n".
	'<input type="hidden" name="command" value="gradeByPage" />'."\n".
	'<input type="hidden" name="fullname" value="'.$$fullname{$env{'form.student'}}.'" />'."\n".
	'<input type="hidden" name="student" value="'.$env{'form.student'}.'" />'."\n".
	'<input type="hidden" name="page"    value="'.$pageTitle.'" />'."\n".
	'<input type="hidden" name="title"   value="'.$env{'form.title'}.'" />'."\n".
	'<input type="hidden" name="symb"    value="'.&Apache::lonenc::check_encrypt($symb).'" />'."\n".
	'<input type="hidden" name="overRideScore" value="no" />'."\n";

    if (defined($env{'form.CODE'})) {
	$studentTable.=
	    '<input type="hidden" name="CODE" value="'.$env{'form.CODE'}.'" />'."\n";
    }
    my $checkIcon = '<img alt="'.&mt('Check Mark').
	'" src="'.&Apache::loncommon::lonhttpdurl($request->dir_config('lonIconsURL').'/check.gif').'" height="16" border="0" />';

    $studentTable.='&nbsp;<span class="LC_info">'.
        &mt('Problems graded correct by the computer are marked with a [_1] symbol.',$checkIcon).
        '</span>'."\n".
	&Apache::loncommon::start_data_table().
	&Apache::loncommon::start_data_table_header_row().
	'<th>'.&mt('Prob.').'</th>'.
	'<th>&nbsp;'.($env{'form.vProb'} eq 'no' ? &mt('Title') : &mt('Problem Text')).'/'.&mt('Grade').'</th>'.
	&Apache::loncommon::end_data_table_header_row();

    &Apache::lonxml::clear_problem_counter();
    my ($depth,$question,$prob) = (1,1,1);
    $iterator->next(); # skip the first BEGIN_MAP
    my $curRes = $iterator->next(); # for "current resource"
    while ($depth > 0) {
        if($curRes == $iterator->BEGIN_MAP) { $depth++; }
        if($curRes == $iterator->END_MAP) { $depth--; }

        if (ref($curRes) && $curRes->is_problem()) {
	    my $parts = $curRes->parts();
            my $title = $curRes->compTitle();
	    my $symbx = $curRes->symb();
	    $studentTable.=
		&Apache::loncommon::start_data_table_row().
		'<td align="center" valign="top" >'.$prob.
		(scalar(@{$parts}) == 1 ? '' 
		                        : '<br />('.&mt('[_1]parts',
							scalar(@{$parts}).'&nbsp;').')'
		 ).
		 '</td>';
	    $studentTable.='<td valign="top">';
	    my %form = ('CODE' => $env{'form.CODE'},);
	    if ($env{'form.vProb'} eq 'yes' ) {
		$studentTable.=&show_problem($request,$symbx,$uname,$udom,1,
					     undef,'both',\%form);
	    } else {
		my $companswer = &Apache::loncommon::get_student_answers($symbx,$uname,$udom,$env{'request.course.id'},%form);
		$companswer =~ s|<form(.*?)>||g;
		$companswer =~ s|</form>||g;
#		while ($companswer =~ /(<a href\=\"javascript:newWindow.*?Script Vars<\/a>)/s) { #<a href="javascript:newWindow</a>
#		    $companswer =~ s/$1/ /ms;
#		    $request->print('match='.$1."<br />\n");
#		}
#		$companswer =~ s|<table border=\"1\">|<table border=\"0\">|g;
		$studentTable.='&nbsp;<b>'.$title.'</b>&nbsp;<br />&nbsp;<b>'.&mt('Correct answer').':</b><br />'.$companswer;
	    }

	    my %record = &Apache::lonnet::restore($symbx,$env{'request.course.id'},$udom,$uname);

	    if ($env{'form.lastSub'} eq 'datesub') {
		if ($record{'version'} eq '') {
		    $studentTable.='<br />&nbsp;<span class="LC_warning">'.&mt('No recorded submission for this problem.').'</span><br />';
		} else {
		    my %responseType = ();
		    foreach my $partid (@{$parts}) {
			my @responseIds =$curRes->responseIds($partid);
			my @responseType =$curRes->responseType($partid);
			my %responseIds;
			for (my $i=0;$i<=$#responseIds;$i++) {
			    $responseIds{$responseIds[$i]}=$responseType[$i];
			}
			$responseType{$partid} = \%responseIds;
		    }
		    $studentTable.= &displaySubByDates($symbx,\%record,$parts,\%responseType,$checkIcon,$uname,$udom);

		}
	    } elsif ($env{'form.lastSub'} eq 'all') {
		my $last = ($env{'form.lastSub'} eq 'last' ? 'last' : '');
                my $identifier = (&canmodify($usec)? $prob : '');
		$studentTable.=&Apache::loncommon::get_previous_attempt($symbx,$uname,$udom,
									$env{'request.course.id'},
									'','.submission',undef,
                                                                        $usec,$identifier);
 
	    }
	    if (&canmodify($usec)) {
            $studentTable.=&gradeBox_start();
		foreach my $partid (@{$parts}) {
		    $studentTable.=&gradeBox($request,$symbx,$uname,$udom,$question,$partid,\%record);
		    $studentTable.='<input type="hidden" name="q_'.$question.'" value="'.$partid.'" />'."\n";
		    $question++;
		}
            $studentTable.=&gradeBox_end();
		$prob++;
	    }
	    $studentTable.='</td></tr>';

	}
        $curRes = $iterator->next();
    }
    my $disabled;
    unless (&canmodify($usec)) {
        $disabled = ' disabled="disabled"';
    }

    $studentTable.=
        '</table>'."\n".
        '<input type="button" value="'.&mt('Save').'"'.$disabled.' '.
        'onclick="javascript:checkSubmitPage(this.form,'.$question.');" />'.
        '</form>'."\n";
    $request->print($studentTable);

    return '';
}

sub displaySubByDates {
    my ($symb,$record,$parts,$responseType,$checkIcon,$uname,$udom) = @_;
    my $isCODE=0;
    my $isTask = ($symb =~/\.task$/);
    if (exists($record->{'resource.CODE'})) { $isCODE=1; }
    my $studentTable=&Apache::loncommon::start_data_table().
	&Apache::loncommon::start_data_table_header_row().
	'<th>'.&mt('Date/Time').'</th>'.
	($isCODE?'<th>'.&mt('CODE').'</th>':'').
        ($isTask?'<th>'.&mt('Version').'</th>':'').
	'<th>'.&mt('Submission').'</th>'.
	'<th>'.&mt('Status').'</th>'.
	&Apache::loncommon::end_data_table_header_row();
    my ($version);
    my %mark;
    my %orders;
    $mark{'correct_by_student'} = $checkIcon;
    if (!exists($$record{'1:timestamp'})) {
	return '<br />&nbsp;<span class="LC_warning">'.&mt('Nothing submitted - no attempts.').'</span><br />';
    }

    my $interaction;
    my $no_increment = 1;
    my (%lastrndseed,%lasttype);
    for ($version=1;$version<=$$record{'version'};$version++) {
	my $timestamp = 
	    &Apache::lonlocal::locallocaltime($$record{$version.':timestamp'});
	if (exists($$record{$version.':resource.0.version'})) {
	    $interaction = $$record{$version.':resource.0.version'};
	}
        if ($isTask && $env{'form.previousversion'}) {
            next unless ($interaction == $env{'form.previousversion'});
        }
	my $where = ($isTask ? "$version:resource.$interaction"
		             : "$version:resource");
	$studentTable.=&Apache::loncommon::start_data_table_row().
	    '<td>'.$timestamp.'</td>';
	if ($isCODE) {
	    $studentTable.='<td>'.$record->{$version.':resource.CODE'}.'</td>';
	}
        if ($isTask) {
            $studentTable.='<td>'.$interaction.'</td>';
        }
	my @versionKeys = split(/\:/,$$record{$version.':keys'});
	my @displaySub = ();
	foreach my $partid (@{$parts}) {
            my ($hidden,$type);
            $type = $$record{$version.':resource.'.$partid.'.type'};
            if (($type eq 'anonsurvey') || ($type eq 'anonsurveycred')) {
                $hidden = 1;
            }
	    my @matchKey;
            if ($isTask) {
                @matchKey = sort(grep(/^resource\.\d+\.\Q$partid\E\.award$/,@versionKeys));
            } else {
		@matchKey = sort(grep(/^resource\.\Q$partid\E\..*?\.submission$/,@versionKeys));
            }
#	    next if ($$record{"$version:resource.$partid.solved"} eq '');
	    my $display_part=&get_display_part($partid,$symb);
	    foreach my $matchKey (@matchKey) {
		if (exists($$record{$version.':'.$matchKey}) &&
		    $$record{$version.':'.$matchKey} ne '') {
                    
		    my ($responseId)= ($isTask ? ($matchKey=~ /^resource\.(.*?)\.\Q$partid\E\.award$/)
				               : ($matchKey=~ /^resource\.\Q$partid\E\.(.*?)\.submission$/));
                    $displaySub[0].='<span class="LC_nobreak">';
                    $displaySub[0].='<b>'.&mt('Part: [_1]',$display_part).'</b>'
                                   .' <span class="LC_internal_info">'
                                   .'('.&mt('Response ID: [_1]',$responseId).')'
                                   .'</span>'
                                   .' <b>';
                    if ($hidden) {
                        $displaySub[0].= &mt('Anonymous Survey').'</b>';
                    } else {
                        my ($trial,$rndseed,$newvariation);
                        if ($type eq 'randomizetry') {
                            $trial = $$record{"$where.$partid.tries"};
                            $rndseed = $$record{"$where.$partid.rndseed"};
                        }
		        if ($$record{"$where.$partid.tries"} eq '') {
			    $displaySub[0].=&mt('Trial not counted');
		        } else {
			    $displaySub[0].=&mt('Trial: [_1]',
					    $$record{"$where.$partid.tries"});
                            if (($rndseed ne '')  && ($lastrndseed{$partid} ne '')) {
                                if (($rndseed ne $lastrndseed{$partid}) &&
                                    (($type eq 'randomizetry') || ($lasttype{$partid} eq 'randomizetry'))) {
                                    $newvariation = '&nbsp;('.&mt('New variation this try').')';
                                }
                            }
                            $lastrndseed{$partid} = $rndseed;
                            $lasttype{$partid} = $type;
		        }
		        my $responseType=($isTask ? 'Task'
                                              : $responseType->{$partid}->{$responseId});
		        if (!exists($orders{$partid})) { $orders{$partid}={}; }
		        if ((!exists($orders{$partid}->{$responseId})) || ($trial)) {
			    $orders{$partid}->{$responseId}=
			        &get_order($partid,$responseId,$symb,$uname,$udom,
                                           $no_increment,$type,$trial,$rndseed);
		        }
		        $displaySub[0].='</b>'.$newvariation.'</span>'; # /nobreak
		        $displaySub[0].='&nbsp; '.
			    &cleanRecord($$record{$version.':'.$matchKey},$responseType,$symb,$partid,$responseId,$record,$orders{$partid}->{$responseId},"$version:",$uname,$udom,$type,$trial,$rndseed).'<br />';
                    }
		}
	    }
	    if (exists($$record{"$where.$partid.checkedin"})) {
		$displaySub[1].=&mt('Checked in by [_1] into slot [_2]',
				    $$record{"$where.$partid.checkedin"},
				    $$record{"$where.$partid.checkedin.slot"}).
					'<br />';
	    }
	    if (exists $$record{"$where.$partid.award"}) {
		$displaySub[1].='<b>'.&mt('Part:').'</b>&nbsp;'.$display_part.' &nbsp;'.
		    lc($$record{"$where.$partid.award"}).' '.
		    $mark{$$record{"$where.$partid.solved"}}.
		    '<br />';
	    }
	    if (exists $$record{"$where.$partid.regrader"}) {
		$displaySub[2].=$$record{"$where.$partid.regrader"}.
		    ' (<b>'.&mt('Part').':</b> '.$display_part.')';
	    } elsif ($$record{"$version:resource.$partid.regrader"} =~ /\S/) {
		$displaySub[2].=
		    $$record{"$version:resource.$partid.regrader"}.
		    ' (<b>'.&mt('Part').':</b> '.$display_part.')';
	    }
	}
	# needed because old essay regrader has not parts info
	if (exists $$record{"$version:resource.regrader"}) {
	    $displaySub[2].=$$record{"$version:resource.regrader"};
	}
	$studentTable.='<td>'.$displaySub[0].'&nbsp;</td><td>'.$displaySub[1];
	if ($displaySub[2]) {
	    $studentTable.=&mt('Manually graded by [_1]',$displaySub[2]);
	}
	$studentTable.='&nbsp;</td>'.
	    &Apache::loncommon::end_data_table_row();
    }
    $studentTable.=&Apache::loncommon::end_data_table();
    return $studentTable;
}

sub updateGradeByPage {
    my ($request,$symb) = @_;

    my $cdom      = $env{"course.$env{'request.course.id'}.domain"};
    my $cnum      = $env{"course.$env{'request.course.id'}.num"};
    my $getsec    = $env{'form.section'} eq '' ? 'all' : $env{'form.section'};
    my $pageTitle = $env{'form.page'};
    my ($classlist,undef,$fullname) = &getclasslist($getsec,'1');
    my ($uname,$udom) = split(/:/,$env{'form.student'});
    my $usec=$classlist->{$env{'form.student'}}[5];
    if (!&canmodify($usec)) {
	$request->print('<span class="LC_warning">'.&mt('Unable to modify requested student ([_1])',$env{'form.student'}).'</span>');
	return;
    }
    my $result='<h3><span class="LC_info">&nbsp;'.$env{'form.title'}.'</span></h3>';
    $result.='<h3>&nbsp;'.&mt('Student: ').&nameUserString(undef,$env{'form.fullname'},$uname,$udom).
	'</h3>'."\n";

    $request->print($result);


    my $navmap = Apache::lonnavmaps::navmap->new();
    unless (ref($navmap)) {
        $request->print(&navmap_errormsg());
        return;
    }
    my ($mapUrl, $id, $resUrl) = &Apache::lonnet::decode_symb( $env{'form.page'});
    my $map = $navmap->getResourceByUrl($resUrl); # add to navmaps
    if (!$map) {
	$request->print('<span class="LC_warning">'.&mt('Unable to grade requested sequence ([_1]).',$resUrl).'</span>');
	return; 
    }
    my $iterator = $navmap->getIterator($map->map_start(),
					$map->map_finish());

    my $studentTable=
	&Apache::loncommon::start_data_table().
	&Apache::loncommon::start_data_table_header_row().
	'<th align="center">&nbsp;'.&mt('Prob.').'&nbsp;</th>'.
	'<th>&nbsp;'.&mt('Title').'&nbsp;</th>'.
	'<th>&nbsp;'.&mt('Previous Score').'&nbsp;</th>'.
	'<th>&nbsp;'.&mt('New Score').'&nbsp;</th>'.
	&Apache::loncommon::end_data_table_header_row();

    $iterator->next(); # skip the first BEGIN_MAP
    my $curRes = $iterator->next(); # for "current resource"
    my ($depth,$question,$prob,$changeflag,$hideflag)= (1,1,1,0,0);
    while ($depth > 0) {
        if($curRes == $iterator->BEGIN_MAP) { $depth++; }
        if($curRes == $iterator->END_MAP) { $depth--; }

        if (ref($curRes) && $curRes->is_problem()) {
	    my $parts = $curRes->parts();
            my $title = $curRes->compTitle();
	    my $symbx = $curRes->symb();
	    $studentTable.=
		&Apache::loncommon::start_data_table_row().
		'<td align="center" valign="top" >'.$prob.
		(scalar(@{$parts}) == 1 ? '' 
                                        : '<br />('.&mt('[quant,_1,part]',scalar(@{$parts}))
		.')').'</td>';
	    $studentTable.='<td valign="top">&nbsp;<b>'.$title.'</b>&nbsp;</td>';

	    my %newrecord=();
	    my @displayPts=();
            my %aggregate = ();
            my $aggregateflag = 0;
            my %queueable;
            if ($env{'form.HIDE'.$prob}) {
                my %record = &Apache::lonnet::restore($symbx,$env{'request.course.id'},$udom,$uname);
                my ($version,$parts) = split(/:/,$env{'form.HIDE'.$prob},2);
                my $numchgs = &makehidden($version,$parts,\%record,$symbx,$udom,$uname,1);
                $hideflag += $numchgs;
            }
	    foreach my $partid (@{$parts}) {
		my $newpts = $env{'form.GD_BOX'.$question.'_'.$partid};
		my $oldpts = $env{'form.oldpts'.$question.'_'.$partid};
                my @types = $curRes->responseType($partid);
                if (grep(/^essay$/,@types)) {
                    $queueable{$partid} = 1;
                } else {
                    my @ids = $curRes->responseIds($partid);
                    for (my $i=0; $i < scalar(@ids); $i++) {
                        my $hndgrd = &Apache::lonnet::EXT('resource.'.$partid.'_'.$ids[$i].
                                                          '.handgrade',$symb);
                        if (lc($hndgrd) eq 'yes') {
                            $queueable{$partid} = 1;
                            last;
                        }
                    }
                }
		my $wgt = $env{'form.WGT'.$question.'_'.$partid} != 0 ? 
		    $env{'form.WGT'.$question.'_'.$partid} : 1;
		my $partial = $newpts/$wgt;
		my $score;
		if ($partial > 0) {
		    $score = 'correct_by_override';
		} elsif ($newpts ne '') { #empty is taken as 0
		    $score = 'incorrect_by_override';
		}
		my $dropMenu = $env{'form.GD_SEL'.$question.'_'.$partid};
		if ($dropMenu eq 'excused') {
		    $partial = '';
		    $score = 'excused';
		} elsif ($dropMenu eq 'reset status'
			 && $env{'form.solved'.$question.'_'.$partid} ne '') { #update only if previous record exists
		    $newrecord{'resource.'.$partid.'.tries'} = 0;
		    $newrecord{'resource.'.$partid.'.solved'} = '';
		    $newrecord{'resource.'.$partid.'.award'} = '';
		    $newrecord{'resource.'.$partid.'.awarded'} = 0;
		    $newrecord{'resource.'.$partid.'.regrader'} = "$env{'user.name'}:$env{'user.domain'}";
		    $changeflag++;
		    $newpts = '';
                    
                    my $aggtries =  $env{'form.aggtries'.$question.'_'.$partid};
                    my $totaltries = $env{'form.totaltries'.$question.'_'.$partid};
                    my $solvedstatus = $env{'form.solved'.$question.'_'.$partid};
                    if ($aggtries > 0) {
                        &decrement_aggs($symbx,$partid,\%aggregate,$aggtries,$totaltries,$solvedstatus);
                        $aggregateflag = 1;
                    }
		}
		my $display_part=&get_display_part($partid,$curRes->symb());
		my $oldstatus = $env{'form.solved'.$question.'_'.$partid};
		$displayPts[0].='&nbsp;<b>'.&mt('Part').':</b> '.$display_part.' = '.
		    (($oldstatus eq 'excused') ? 'excused' : $oldpts).
		    '&nbsp;<br />';
		$displayPts[1].='&nbsp;<b>'.&mt('Part').':</b> '.$display_part.' = '.
		     (($score eq 'excused') ? 'excused' : $newpts).
		    '&nbsp;<br />';
		$question++;
		next if ($dropMenu eq 'reset status' || ($newpts eq $oldpts && $score ne 'excused'));

		$newrecord{'resource.'.$partid.'.awarded'}  = $partial if $partial ne '';
		$newrecord{'resource.'.$partid.'.solved'}   = $score if $score ne '';
		$newrecord{'resource.'.$partid.'.regrader'} = "$env{'user.name'}:$env{'user.domain'}"
		    if (scalar(keys(%newrecord)) > 0);

		$changeflag++;
	    }
	    if (scalar(keys(%newrecord)) > 0) {
		my %record = 
		    &Apache::lonnet::restore($symbx,$env{'request.course.id'},
					     $udom,$uname);

		if (&Apache::lonnet::validCODE($env{'form.CODE'})) {
		    $newrecord{'resource.CODE'} = $env{'form.CODE'};
		} elsif (&Apache::lonnet::validCODE($record{'resource.CODE'})) {
		    $newrecord{'resource.CODE'} = '';
		}
		&Apache::lonnet::cstore(\%newrecord,$symbx,$env{'request.course.id'},
					$udom,$uname);
		%record = &Apache::lonnet::restore($symbx,
						   $env{'request.course.id'},
						   $udom,$uname);
		&check_and_remove_from_queue($parts,\%record,undef,$symbx,
					     $cdom,$cnum,$udom,$uname,\%queueable);
	    }
	    
            if ($aggregateflag) {
                &Apache::lonnet::cinc('nohist_resourcetracker',\%aggregate,
                      $env{'course.'.$env{'request.course.id'}.'.domain'},
                      $env{'course.'.$env{'request.course.id'}.'.num'});
            }

	    $studentTable.='<td valign="top">'.$displayPts[0].'</td>'.
		'<td valign="top">'.$displayPts[1].'</td>'.
		&Apache::loncommon::end_data_table_row();

	    $prob++;
	}
        $curRes = $iterator->next();
    }

    $studentTable.=&Apache::loncommon::end_data_table();
    my $grademsg=($changeflag == 0 ? &mt('No score was changed or updated.') :
		  &mt('The scores were changed for [quant,_1,problem].',
		  $changeflag).'<br />');
    my $hidemsg=($hideflag == 0 ? '' :
                 &mt('Submissions were marked "hidden" for [quant,_1,transaction].',
                     $hideflag).'<br />');
    $request->print($hidemsg.$grademsg.$studentTable);

    return '';
}

#-------- end of section for handling grading by page/sequence ---------
#
#-------------------------------------------------------------------

#-------------------- Bubblesheet (Scantron) Grading -------------------
#
#------ start of section for handling grading by page/sequence ---------

=pod

=head1 Bubble sheet grading routines

  For this documentation:

   'scanline' refers to the full line of characters
   from the file that we are parsing that represents one entire sheet

   'bubble line' refers to the data
   representing the line of bubbles that are on the physical bubblesheet


The overall process is that a scanned in bubblesheet data is uploaded
into a course. When a user wants to grade, they select a
sequence/folder of resources, a file of bubblesheet info, and pick
one of the predefined configurations for what each scanline looks
like.

Next each scanline is checked for any errors of either 'missing
bubbles' (it's an error because it may have been mis-scanned
because too light bubbling), 'double bubble' (each bubble line should
have no more than one letter picked), invalid or duplicated CODE,
invalid student/employee ID

If the CODE option is used that determines the randomization of the
homework problems, either way the student/employee ID is looked up into a
username:domain.

During the validation phase the instructor can choose to skip scanlines. 

After the validation phase, there are now 3 bubblesheet files

  scantron_original_filename (unmodified original file)
  scantron_corrected_filename (file where the corrected information has replaced the original information)
  scantron_skipped_filename (contains the exact text of scanlines that where skipped)

Also there is a separate hash nohist_scantrondata that contains extra
correction information that isn't representable in the bubblesheet
file (see &scantron_getfile() for more information)

After all scanlines are either valid, marked as valid or skipped, then
foreach line foreach problem in the picked sequence, an ssi request is
made that simulates a user submitting their selected letter(s) against
the homework problem.

=over 4



=item defaultFormData

  Returns html hidden inputs used to hold context/default values.

 Arguments:
  $symb - $symb of the current resource 

=cut

sub defaultFormData {
    my ($symb)=@_;
    return '<input type="hidden" name="symb" value="'.&Apache::lonenc::check_encrypt($symb).'" />';
}


=pod 

=item getSequenceDropDown

   Return html dropdown of possible sequences to grade
 
 Arguments:
   $symb - $symb of the current resource
   $map_error - ref to scalar which will container error if
                $navmap object is unavailable in &getSymbMap().

=cut

sub getSequenceDropDown {
    my ($symb,$map_error)=@_;
    my $result='<select name="selectpage">'."\n";
    my ($titles,$symbx) = &getSymbMap($map_error);
    if (ref($map_error)) {
        return if ($$map_error);
    }
    my ($curpage)=&Apache::lonnet::decode_symb($symb); 
    my $ctr=0;
    foreach (@$titles) {
	my ($minder,$showtitle) = ($_ =~ /(\d+)\.(.*)/);
	$result.='<option value="'.$$symbx{$_}.'" '.
	    ($$symbx{$_} =~ /$curpage$/ ? 'selected="selected"' : '').
	    '>'.$showtitle.'</option>'."\n";
	$ctr++;
    }
    $result.= '</select>';
    return $result;
}

my %bubble_lines_per_response;     # no. bubble lines for each response.
                                   # key is zero-based index - 0, 1, 2 ...

my %first_bubble_line;             # First bubble line no. for each bubble.

my %subdivided_bubble_lines;       # no. bubble lines for optionresponse, 
                                   # matchresponse or rankresponse, where 
                                   # an individual response can have multiple 
                                   # lines

my %responsetype_per_response;     # responsetype for each response

my %masterseq_id_responsenum;      # src_id (e.g., 12.3_0.11 etc.) for each
                                   # numbered response. Needed when randomorder
                                   # or randompick are in use. Key is ID, value 
                                   # is response number.

# Save and restore the bubble lines array to the form env.


sub save_bubble_lines {
    foreach my $line (keys(%bubble_lines_per_response)) {
	$env{"form.scantron.bubblelines.$line"}  = $bubble_lines_per_response{$line};
	$env{"form.scantron.first_bubble_line.$line"} =
	    $first_bubble_line{$line};
        $env{"form.scantron.sub_bubblelines.$line"} = 
            $subdivided_bubble_lines{$line};
        $env{"form.scantron.responsetype.$line"} =
            $responsetype_per_response{$line};
    }
    foreach my $resid (keys(%masterseq_id_responsenum)) {
        my $line = $masterseq_id_responsenum{$resid};
        $env{"form.scantron.residpart.$line"} = $resid;
    }
}


sub restore_bubble_lines {
    my $line = 0;
    %bubble_lines_per_response = ();
    %masterseq_id_responsenum = ();
    while ($env{"form.scantron.bubblelines.$line"}) {
	my $value = $env{"form.scantron.bubblelines.$line"};
	$bubble_lines_per_response{$line} = $value;
	$first_bubble_line{$line}  =
	    $env{"form.scantron.first_bubble_line.$line"};
        $subdivided_bubble_lines{$line} =
            $env{"form.scantron.sub_bubblelines.$line"};
        $responsetype_per_response{$line} =
            $env{"form.scantron.responsetype.$line"};
        my $id = $env{"form.scantron.residpart.$line"};
        $masterseq_id_responsenum{$id} = $line;
	$line++;
    }
}

=pod 

=item scantron_filenames

   Returns a list of the scantron files in the current course 

=cut

sub scantron_filenames {
    my $cdom=$env{'course.'.$env{'request.course.id'}.'.domain'};
    my $cname=$env{'course.'.$env{'request.course.id'}.'.num'};
    my $getpropath = 1;
    my ($dirlist,$listerror) = &Apache::lonnet::dirlist('userfiles',$cdom,
                                                        $cname,$getpropath);
    my @possiblenames;
    if (ref($dirlist) eq 'ARRAY') {
        foreach my $filename (sort(@{$dirlist})) {
	    ($filename)=split(/&/,$filename);
	    if ($filename!~/^scantron_orig_/) { next ; }
	    $filename=~s/^scantron_orig_//;
	    push(@possiblenames,$filename);
        }
    }
    return @possiblenames;
}

=pod 

=item scantron_uploads

   Returns  html drop-down list of scantron files in current course.

 Arguments:
   $file2grade - filename to set as selected in the dropdown

=cut

sub scantron_uploads {
    my ($file2grade) = @_;
    my $result=	'<select name="scantron_selectfile">';
    $result.="<option></option>";
    foreach my $filename (sort(&scantron_filenames())) {
	$result.="<option".($filename eq $file2grade ? ' selected="selected"':'').">$filename</option>\n";
    }
    $result.="</select>";
    return $result;
}

=pod 

=item scantron_scantab

  Returns html drop down of the scantron formats in the scantronformat.tab
  file.

=cut

sub scantron_scantab {
    my $result='<select name="scantron_format">'."\n";
    $result.='<option></option>'."\n";
    my @lines = &Apache::lonnet::get_scantronformat_file();
    if (@lines > 0) {
        foreach my $line (@lines) {
            next if (($line =~ /^\#/) || ($line eq ''));
	    my ($name,$descrip)=split(/:/,$line);
	    $result.='<option value="'.$name.'">'.$descrip.'</option>'."\n";
        }
    }
    $result.='</select>'."\n";
    return $result;
}

=pod 

=item scantron_CODElist

  Returns html drop down of the saved CODE lists from current course,
  generated from earlier printings.

=cut

sub scantron_CODElist {
    my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
    my @names=&Apache::lonnet::getkeys('CODEs',$cdom,$cnum);
    my $namechoice='<option></option>';
    foreach my $name (sort {uc($a) cmp uc($b)} @names) {
	if ($name =~ /^error: 2 /) { next; }
	if ($name =~ /^type\0/) { next; }
	$namechoice.='<option value="'.$name.'">'.$name.'</option>';
    }
    $namechoice='<select name="scantron_CODElist">'.$namechoice.'</select>';
    return $namechoice;
}

=pod 

=item scantron_CODEunique

  Returns the html for "Each CODE to be used once" radio.

=cut

sub scantron_CODEunique {
    my $result='<span class="LC_nobreak">
                 <label><input type="radio" name="scantron_CODEunique"
                        value="yes" checked="checked" />'.&mt('Yes').' </label>
                </span>
                <span class="LC_nobreak">
                 <label><input type="radio" name="scantron_CODEunique"
                        value="no" />'.&mt('No').' </label>
                </span>';
    return $result;
}

=pod 

=item scantron_selectphase

  Generates the initial screen to start the bubblesheet process.
  Allows for - starting a grading run.
             - downloading existing scan data (original, corrected
                                                or skipped info)

             - uploading new scan data

 Arguments:
  $r          - The Apache request object
  $file2grade - name of the file that contain the scanned data to score

=cut

sub scantron_selectphase {
    my ($r,$file2grade,$symb) = @_;
    if (!$symb) {return '';}
    my $map_error;
    my $sequence_selector=&getSequenceDropDown($symb,\$map_error);
    if ($map_error) {
        $r->print('<br />'.&navmap_errormsg().'<br />');
        return;
    }
    my $default_form_data=&defaultFormData($symb);
    my $file_selector=&scantron_uploads($file2grade);
    my $format_selector=&scantron_scantab();
    my $CODE_selector=&scantron_CODElist();
    my $CODE_unique=&scantron_CODEunique();
    my $result;

    $ssi_error = 0;

    if (&Apache::lonnet::allowed('usc',$env{'request.role.domain'}) ||
        &Apache::lonnet::allowed('usc',$env{'request.course.id'})) {

        # Chunk of form to prompt for a scantron file upload.

        $r->print('
    <br />');
        my $cdom= $env{'course.'.$env{'request.course.id'}.'.domain'};
        my $cnum= $env{'course.'.$env{'request.course.id'}.'.num'};
        my $alertmsg = &mt('Please use the browse button to select a file from your local directory.');
        &js_escape(\$alertmsg);
        my ($formatoptions,$formattitle,$formatjs) = &scantron_upload_dataformat($cdom);
        $r->print(&Apache::lonhtmlcommon::scripttag('
    function checkUpload(formname) {
        if (formname.upfile.value == "") {
            alert("'.$alertmsg.'");
            return false;
        }
        formname.submit();
    }'."\n".$formatjs));
        $r->print('
              <form enctype="multipart/form-data" action="/adm/grades" name="rules" method="post">
                '.$default_form_data.'
                <input name="courseid" type="hidden" value="'.$cnum.'" />
                <input name="domainid" type="hidden" value="'.$cdom.'" />
                <input name="command" value="scantronupload_save" type="hidden" />
              '.&Apache::loncommon::start_data_table('LC_scantron_action').'
              '.&Apache::loncommon::start_data_table_header_row().'
                <th>
                &nbsp;'.&mt('Specify a bubblesheet data file to upload.').'
                </th>
              '.&Apache::loncommon::end_data_table_header_row().'
              '.&Apache::loncommon::start_data_table_row().'
            <td>
                '.&mt('File to upload: [_1]','<input type="file" name="upfile" size="50" />').'<br />'."\n");
        if ($formatoptions) {
            $r->print('</td>
                 '.&Apache::loncommon::end_data_table_row().'
                 '.&Apache::loncommon::start_data_table_row().'
                 <td>'.$formattitle.('&nbsp;'x2).$formatoptions.'
                 </td>
                 '.&Apache::loncommon::end_data_table_row().'
                 '.&Apache::loncommon::start_data_table_row().'
                 <td>'
            );
        } else {
            $r->print(' <br />');
        }
        $r->print('<input type="button" onclick="javascript:checkUpload(this.form);" value="'.&mt('Upload Bubblesheet Data').'" />
              </td>
             '.&Apache::loncommon::end_data_table_row().'
             '.&Apache::loncommon::end_data_table().'
             </form>'
        );

    }

    # Chunk of form to prompt for a file to grade and how:

    $result.= '
    <br />
    <form method="post" enctype="multipart/form-data" action="/adm/grades" name="scantron_process">
    <input type="hidden" name="command" value="scantron_warning" />
    '.$default_form_data.'
    '.&Apache::loncommon::start_data_table('LC_scantron_action').'
       '.&Apache::loncommon::start_data_table_header_row().'
            <th colspan="2">
              &nbsp;'.&mt('Specify file and which Folder/Sequence to grade').'
            </th>
       '.&Apache::loncommon::end_data_table_header_row().'
       '.&Apache::loncommon::start_data_table_row().'
            <td> '.&mt('Sequence to grade:').' </td><td> '.$sequence_selector.' </td>
       '.&Apache::loncommon::end_data_table_row().'
       '.&Apache::loncommon::start_data_table_row().'
            <td> '.&mt('Filename of bubblesheet data file:').' </td><td> '.$file_selector.' </td>
       '.&Apache::loncommon::end_data_table_row().'
       '.&Apache::loncommon::start_data_table_row().'
            <td> '.&mt('Format of bubblesheet data file:').' </td><td> '.$format_selector.' </td>
       '.&Apache::loncommon::end_data_table_row().'
       '.&Apache::loncommon::start_data_table_row().'
            <td> '.&mt('Saved CODEs to validate against:').' </td><td> '.$CODE_selector.' </td>
       '.&Apache::loncommon::end_data_table_row().'
       '.&Apache::loncommon::start_data_table_row().'
            <td> '.&mt('Each CODE is only to be used once:').'</td><td> '.$CODE_unique.' </td>
       '.&Apache::loncommon::end_data_table_row().'
       '.&Apache::loncommon::start_data_table_row().'
	    <td> '.&mt('Options:').' </td>
            <td>
	       <label><input type="checkbox" name="scantron_options_redo" value="redo_skipped"/> '.&mt('Do only previously skipped records').'</label> <br />
               <label><input type="checkbox" name="scantron_options_ignore" value="ignore_corrections"/> '.&mt('Remove all existing corrections').'</label> <br />
               <label><input type="checkbox" name="scantron_options_hidden" value="ignore_hidden"/> '.&mt('Skip hidden resources when grading').'</label>
	    </td>
       '.&Apache::loncommon::end_data_table_row().'
       '.&Apache::loncommon::start_data_table_row().'
            <td colspan="2">
              <input type="submit" value="'.&mt('Grading: Validate Bubblesheet Records').'" />
            </td>
       '.&Apache::loncommon::end_data_table_row().'
    '.&Apache::loncommon::end_data_table().'
    </form>
';
   
    $r->print($result);

    # Chunk of the form that prompts to view a scoring office file,
    # corrected file, skipped records in a file.

    $r->print('
   <br />
   <form action="/adm/grades" name="scantron_download">
     '.$default_form_data.'
     <input type="hidden" name="command" value="scantron_download" />
     '.&Apache::loncommon::start_data_table('LC_scantron_action').'
       '.&Apache::loncommon::start_data_table_header_row().'
              <th>
                &nbsp;'.&mt('Download a scoring office file').'
              </th>
       '.&Apache::loncommon::end_data_table_header_row().'
       '.&Apache::loncommon::start_data_table_row().'
              <td> '.&mt('Filename of scoring office file: [_1]',$file_selector).' 
                <br />
                <input type="submit" value="'.&mt('Download: Show List of Associated Files').'" />
       '.&Apache::loncommon::end_data_table_row().'
     '.&Apache::loncommon::end_data_table().'
   </form>
   <br />
');

    &Apache::lonpickcode::code_list($r,2);

    $r->print('<br /><form method="post" name="checkscantron" action="">'.
             $default_form_data."\n".
             &Apache::loncommon::start_data_table('LC_scantron_action')."\n".
             &Apache::loncommon::start_data_table_header_row()."\n".
             '<th colspan="2">
              &nbsp;'.&mt('Review bubblesheet data and submissions for a previously graded folder/sequence')."\n".
             '</th>'."\n".
              &Apache::loncommon::end_data_table_header_row()."\n".
              &Apache::loncommon::start_data_table_row()."\n".
              '<td> '.&mt('Graded folder/sequence:').' </td>'."\n".
              '<td> '.$sequence_selector.' </td>'.
              &Apache::loncommon::end_data_table_row()."\n".
              &Apache::loncommon::start_data_table_row()."\n".
              '<td> '.&mt('Filename of scoring office file:').' </td>'."\n".
              '<td> '.$file_selector.' </td>'."\n".
              &Apache::loncommon::end_data_table_row()."\n".
              &Apache::loncommon::start_data_table_row()."\n".
              '<td> '.&mt('Format of data file:').' </td>'."\n".
              '<td> '.$format_selector.' </td>'."\n".
              &Apache::loncommon::end_data_table_row()."\n".
              &Apache::loncommon::start_data_table_row()."\n".
              '<td> '.&mt('Options').' </td>'."\n".
              '<td> <label><input type="checkbox" name="scantron_options_hidden" value="ignore_hidden"/> '.&mt('Skip hidden resources').'</label></td>'.
              &Apache::loncommon::end_data_table_row()."\n".
              &Apache::loncommon::start_data_table_row()."\n".
              '<td colspan="2">'."\n".
              '<input type="hidden" name="command" value="checksubmissions" />'."\n".
              '<input type="submit" value="'.&mt('Review Bubblesheet Data and Submission Records').'" />'."\n".
              '</td>'."\n".
              &Apache::loncommon::end_data_table_row()."\n".
              &Apache::loncommon::end_data_table()."\n".
              '</form><br />');
    return;
}

=pod 

=item username_to_idmap

    creates a hash keyed by student/employee ID with values of the corresponding
    student username:domain.

  Arguments:

    $classlist - reference to the class list hash. This is a hash
                 keyed by student name:domain  whose elements are references
                 to arrays containing various chunks of information
                 about the student. (See loncoursedata for more info).

  Returns
    %idmap - the constructed hash

=cut

sub username_to_idmap {
    my ($classlist)= @_;
    my %idmap;
    foreach my $student (keys(%$classlist)) {
        my $id = $classlist->{$student}->[&Apache::loncoursedata::CL_ID];
        unless ($id eq '') {
            if (!exists($idmap{$id})) {
                $idmap{$id} = $student;
            } else {
                my $status = $classlist->{$student}->[&Apache::loncoursedata::CL_STATUS];
                if ($status eq 'Active') {
                    $idmap{$id} = $student;
                }
            }
        }
    }
    return %idmap;
}

=pod

=item scantron_fixup_scanline

   Process a requested correction to a scanline.

  Arguments:
    $scantron_config   - hash from &Apache::lonnet::get_scantron_config()
    $scan_data         - hash of correction information 
                          (see &scantron_getfile())
    $line              - existing scanline
    $whichline         - line number of the passed in scanline
    $field             - type of change to process 
                         (either 
                          'ID'     -> correct the student/employee ID
                          'CODE'   -> correct the CODE
                          'answer' -> fixup the submitted answers)
    
   $args               - hash of additional info,
                          - 'ID' 
                               'newid' -> studentID to use in replacement
                                          of existing one
                          - 'CODE' 
                               'CODE_ignore_dup' - set to true if duplicates
                                                   should be ignored.
	                       'CODE' - is new code or 'use_unfound'
                                        if the existing unfound code should
                                        be used as is
                          - 'answer'
                               'response' - new answer or 'none' if blank
                               'question' - the bubble line to change
                               'questionnum' - the question identifier,
                                               may include subquestion. 

  Returns:
    $line - the modified scanline

  Side effects: 
    $scan_data - may be updated

=cut


sub scantron_fixup_scanline {
    my ($scantron_config,$scan_data,$line,$whichline,$field,$args)=@_;
    if ($field eq 'ID') {
	if (length($args->{'newid'}) > $$scantron_config{'IDlength'}) {
	    return ($line,1,'New value too large');
	}
	if (length($args->{'newid'}) < $$scantron_config{'IDlength'}) {
	    $args->{'newid'}=sprintf('%-'.$$scantron_config{'IDlength'}.'s',
				     $args->{'newid'});
	}
	substr($line,$$scantron_config{'IDstart'}-1,
	       $$scantron_config{'IDlength'})=$args->{'newid'};
	if ($args->{'newid'}=~/^\s*$/) {
	    &scan_data($scan_data,"$whichline.user",
		       $args->{'username'}.':'.$args->{'domain'});
	}
    } elsif ($field eq 'CODE') {
	if ($args->{'CODE_ignore_dup'}) {
	    &scan_data($scan_data,"$whichline.CODE_ignore_dup",'1');
	}
	&scan_data($scan_data,"$whichline.useCODE",'1');
	if ($args->{'CODE'} ne 'use_unfound') {
	    if (length($args->{'CODE'}) > $$scantron_config{'CODElength'}) {
		return ($line,1,'New CODE value too large');
	    }
	    if (length($args->{'CODE'}) < $$scantron_config{'CODElength'}) {
		$args->{'CODE'}=sprintf('%-'.$$scantron_config{'CODElength'}.'s',$args->{'CODE'});
	    }
	    substr($line,$$scantron_config{'CODEstart'}-1,
		   $$scantron_config{'CODElength'})=$args->{'CODE'};
	}
    } elsif ($field eq 'answer') {
	my $length=$scantron_config->{'Qlength'};
	my $off=$scantron_config->{'Qoff'};
	my $on=$scantron_config->{'Qon'};
	my $answer=${off}x$length;
	if ($args->{'response'} eq 'none') {
	    &scan_data($scan_data,
		       "$whichline.no_bubble.".$args->{'questionnum'},'1');
	} else {
	    if ($on eq 'letter') {
		my @alphabet=('A'..'Z');
		$answer=$alphabet[$args->{'response'}];
	    } elsif ($on eq 'number') {
		$answer=$args->{'response'}+1;
		if ($answer == 10) { $answer = '0'; }
	    } else {
		substr($answer,$args->{'response'},1)=$on;
	    }
	    &scan_data($scan_data,
		       "$whichline.no_bubble.".$args->{'questionnum'},undef,'1');
	}
	my $where=$length*($args->{'question'}-1)+$scantron_config->{'Qstart'};
	substr($line,$where-1,$length)=$answer;
    }
    return $line;
}

=pod

=item scan_data

    Edit or look up  an item in the scan_data hash.

  Arguments:
    $scan_data  - The hash (see scantron_getfile)
    $key        - shorthand of the key to edit (actual key is
                  scantronfilename_key).
    $data        - New value of the hash entry.
    $delete      - If true, the entry is removed from the hash.

  Returns:
    The new value of the hash table field (undefined if deleted).

=cut


sub scan_data {
    my ($scan_data,$key,$value,$delete)=@_;
    my $filename=$env{'form.scantron_selectfile'};
    if (defined($value)) {
	$scan_data->{$filename.'_'.$key} = $value;
    }
    if ($delete) { delete($scan_data->{$filename.'_'.$key}); }
    return $scan_data->{$filename.'_'.$key};
}

# ----- These first few routines are general use routines.----

# Return the number of occurences of a pattern in a string.

sub occurence_count {
    my ($string, $pattern) = @_;

    my @matches = ($string =~ /$pattern/g);

    return scalar(@matches);
}


# Take a string known to have digits and convert all the
# digits into letters in the range J,A..I.

sub digits_to_letters {
    my ($input) = @_;

    my @alphabet = ('J', 'A'..'I');

    my @input    = split(//, $input);
    my $output ='';
    for (my $i = 0; $i < scalar(@input); $i++) {
	if ($input[$i] =~ /\d/) {
	    $output .= $alphabet[$input[$i]];
	} else {
	    $output .= $input[$i];
	}
    }
    return $output;
}

=pod 

=item scantron_parse_scanline

  Decodes a scanline from the selected scantron file

 Arguments:
    line             - The text of the scantron file line to process
    whichline        - Line number
    scantron_config  - Hash describing the format of the scantron lines.
    scan_data        - Hash of extra information about the scanline
                       (see scantron_getfile for more information)
    just_header      - True if should not process question answers but only
                       the stuff to the left of the answers.
    randomorder      - True if randomorder in use
    randompick       - True if randompick in use
    sequence         - Exam folder URL
    master_seq       - Ref to array containing symbs in exam folder
    symb_to_resource - Ref to hash of symbs for resources in exam folder
                       (corresponding values are resource objects)
    partids_by_symb  - Ref to hash of symb -> array ref of partIDs
    orderedforcode   - Ref to hash of arrays. keys are CODEs and values
                       are refs to an array of resource objects, ordered
                       according to order used for CODE, when randomorder
                       and or randompick are in use.
    respnumlookup    - Ref to hash mapping question numbers in bubble lines
                       for current line to question number used for same question
                        in "Master Sequence" (as seen by Course Coordinator).
    startline        - Ref to hash where key is question number (0 is first)
                       and value is number of first bubble line for current 
                       student or code-based randompick and/or randomorder.
    totalref         - Ref of scalar used to score total number of bubble
                       lines needed for responses in a scan line (used when
                       randompick in use. 

 Returns:
   Hash containing the result of parsing the scanline

   Keys are all proceeded by the string 'scantron.'

       CODE    - the CODE in use for this scanline
       useCODE - 1 if the CODE is invalid but it usage has been forced
                 by the operator
       CODE_ignore_dup - 1 if the CODE is a duplicated use when unique
                            CODEs were selected, but the usage has been
                            forced by the operator
       ID  - student/employee ID
       PaperID - if used, the ID number printed on the sheet when the 
                 paper was scanned
       FirstName - first name from the sheet
       LastName  - last name from the sheet

     if just_header was not true these key may also exist

       missingerror - a list of bubble ranges that are considered to be answers
                      to a single question that don't have any bubbles filled in.
                      Of the form questionnumber:firstbubblenumber:count.
       doubleerror  - a list of bubble ranges that are considered to be answers
                      to a single question that have more than one bubble filled in.
                      Of the form questionnumber::firstbubblenumber:count
   
                In the above, count is the number of bubble responses in the
                input line needed to represent the possible answers to the question.
                e.g. a radioresponse with 15 choices in an answer sheet with 10 choices
                per line would have count = 2.

       maxquest     - the number of the last bubble line that was parsed

       (<number> starts at 1)
       <number>.answer - zero or more letters representing the selected
                         letters from the scanline for the bubble line 
                         <number>.
                         if blank there was either no bubble or there where
                         multiple bubbles, (consult the keys missingerror and
                         doubleerror if this is an error condition)

=cut

sub scantron_parse_scanline {
    my ($line,$whichline,$scantron_config,$scan_data,$just_header,$idmap,
        $randomorder,$randompick,$sequence,$master_seq,$symb_to_resource,
        $partids_by_symb,$orderedforcode,$respnumlookup,$startline,$totalref)=@_;

    my %record;
    my $data=substr($line,0,$$scantron_config{'Qstart'}-1); # stuff before answers
    if (!($$scantron_config{'CODElocation'} eq 0 ||
	  $$scantron_config{'CODElocation'} eq 'none')) {
	if ($$scantron_config{'CODElocation'} < 0 ||
	    $$scantron_config{'CODElocation'} eq 'letter' ||
	    $$scantron_config{'CODElocation'} eq 'number') {
	    $record{'scantron.CODE'}=substr($data,
					    $$scantron_config{'CODEstart'}-1,
					    $$scantron_config{'CODElength'});
	    if (&scan_data($scan_data,"$whichline.useCODE")) {
		$record{'scantron.useCODE'}=1;
	    }
	    if (&scan_data($scan_data,"$whichline.CODE_ignore_dup")) {
		$record{'scantron.CODE_ignore_dup'}=1;
	    }
	} else {
	    #FIXME interpret first N questions
	}
    }
    $record{'scantron.ID'}=substr($data,$$scantron_config{'IDstart'}-1,
				  $$scantron_config{'IDlength'});
    $record{'scantron.PaperID'}=
	substr($data,$$scantron_config{'PaperID'}-1,
	       $$scantron_config{'PaperIDlength'});
    $record{'scantron.FirstName'}=
	substr($data,$$scantron_config{'FirstName'}-1,
	       $$scantron_config{'FirstNamelength'});
    $record{'scantron.LastName'}=
	substr($data,$$scantron_config{'LastName'}-1,
	       $$scantron_config{'LastNamelength'});
    if ($just_header) { return \%record; }

    my @alphabet=('A'..'Z');
    my $questnum=0;
    my $ansnum  =1;		# Multiple 'answer lines'/question.

    my $lastpos = $env{'form.scantron_maxbubble'}*$$scantron_config{'Qlength'};
    if ($randompick || $randomorder) {
        my $total = &get_respnum_lookups($sequence,$scan_data,$idmap,$line,\%record,
                                         $master_seq,$symb_to_resource,
                                         $partids_by_symb,$orderedforcode,
                                         $respnumlookup,$startline);
        if ($total) {
            $lastpos = $total*$$scantron_config{'Qlength'};
        }
        if (ref($totalref)) {
            $$totalref = $total;
        }
    }
    my $questions=substr($line,$$scantron_config{'Qstart'}-1,$lastpos);  # Answers
    chomp($questions);		# Get rid of any trailing \n.
    $questions =~ s/\r$//;      # Get rid of trailing \r too (MAC or Win uploads).
    while (length($questions)) {
        my $answers_needed;
        if (($randompick || $randomorder) && (ref($respnumlookup) eq 'HASH')) {
            $answers_needed = $bubble_lines_per_response{$respnumlookup->{$questnum}};
        } else {
            $answers_needed = $bubble_lines_per_response{$questnum};
        }
        my $answer_length  = ($$scantron_config{'Qlength'} * $answers_needed)
                             || 1;
        $questnum++;
        my $quest_id = $questnum;
        my $currentquest = substr($questions,0,$answer_length);
        $questions       = substr($questions,$answer_length);
        if (length($currentquest) < $answer_length) { next; }

        my $subdivided;
        if (($randompick || $randomorder) && (ref($respnumlookup) eq 'HASH')) {
            $subdivided = $subdivided_bubble_lines{$respnumlookup->{$questnum-1}};
        } else {
            $subdivided = $subdivided_bubble_lines{$questnum-1};
        }
        if ($subdivided =~ /,/) {
            my $subquestnum = 1;
            my $subquestions = $currentquest;
            my @subanswers_needed = split(/,/,$subdivided);
            foreach my $subans (@subanswers_needed) {
                my $subans_length =
                    ($$scantron_config{'Qlength'} * $subans)  || 1;
                my $currsubquest = substr($subquestions,0,$subans_length);
                $subquestions   = substr($subquestions,$subans_length);
                $quest_id = "$questnum.$subquestnum";
                if (($$scantron_config{'Qon'} eq 'letter') ||
                    ($$scantron_config{'Qon'} eq 'number')) {
                    $ansnum = &scantron_validator_lettnum($ansnum, 
                        $questnum,$quest_id,$subans,$currsubquest,$whichline,
                        \@alphabet,\%record,$scantron_config,$scan_data,
                        $randomorder,$randompick,$respnumlookup);
                } else {
                    $ansnum = &scantron_validator_positional($ansnum,
                        $questnum,$quest_id,$subans,$currsubquest,$whichline,
                        \@alphabet,\%record,$scantron_config,$scan_data,
                        $randomorder,$randompick,$respnumlookup);
                }
                $subquestnum ++;
            }
        } else {
            if (($$scantron_config{'Qon'} eq 'letter') ||
                ($$scantron_config{'Qon'} eq 'number')) {
                $ansnum = &scantron_validator_lettnum($ansnum,$questnum,
                    $quest_id,$answers_needed,$currentquest,$whichline,
                    \@alphabet,\%record,$scantron_config,$scan_data,
                    $randomorder,$randompick,$respnumlookup);
            } else {
                $ansnum = &scantron_validator_positional($ansnum,$questnum,
                    $quest_id,$answers_needed,$currentquest,$whichline,
                    \@alphabet,\%record,$scantron_config,$scan_data,
                    $randomorder,$randompick,$respnumlookup);
            }
        }
    }
    $record{'scantron.maxquest'}=$questnum;
    return \%record;
}

sub get_master_seq {
    my ($resources,$master_seq,$symb_to_resource,$need_symb_in_map,$symb_for_examcode) = @_;
    return unless ((ref($resources) eq 'ARRAY') && (ref($master_seq) eq 'ARRAY') &&
                   (ref($symb_to_resource) eq 'HASH'));
    if ($need_symb_in_map) {
        return unless (ref($symb_for_examcode) eq 'HASH');
    }
    my $resource_error;
    foreach my $resource (@{$resources}) {
        my $ressymb;
        if (ref($resource)) {
            $ressymb = $resource->symb();
            push(@{$master_seq},$ressymb);
            $symb_to_resource->{$ressymb} = $resource;
            if ($need_symb_in_map) {
                unless ($resource->is_map()) {
                    my $map=(&Apache::lonnet::decode_symb($ressymb))[0];
                    unless (exists($symb_for_examcode->{$map})) {
                        $symb_for_examcode->{$map} = $ressymb;
                    }
                }
            }
        } else {
            $resource_error = 1;
            last;
        }
    }
    return $resource_error;
}

sub get_respnum_lookups {
    my ($sequence,$scan_data,$idmap,$line,$record,$master_seq,$symb_to_resource,
        $partids_by_symb,$orderedforcode,$respnumlookup,$startline) = @_;
    return unless ((ref($record) eq 'HASH') && (ref($master_seq) eq 'ARRAY') &&
                   (ref($symb_to_resource) eq 'HASH') && (ref($partids_by_symb) eq 'HASH') &&
                   (ref($orderedforcode) eq 'HASH') && (ref($respnumlookup) eq 'HASH') &&
                   (ref($startline) eq 'HASH'));
    my ($user,$scancode);
    if ((exists($record->{'scantron.CODE'})) &&
        (&Apache::lonnet::validCODE($record->{'scantron.CODE'}))) {
        $scancode = $record->{'scantron.CODE'};
    } else {
        $user = &scantron_find_student($record,$scan_data,$idmap,$line);
    }
    my @mapresources =
        &users_order($user,$scancode,$sequence,$master_seq,$symb_to_resource,
                     $orderedforcode);
    my $total = 0;
    my $count = 0;
    foreach my $resource (@mapresources) {
        my $id = $resource->id();
        my $symb = $resource->symb();
        if (ref($partids_by_symb->{$symb}) eq 'ARRAY') {
            foreach my $partid (@{$partids_by_symb->{$symb}}) {
                my $respnum = $masterseq_id_responsenum{$id.'_'.$partid};
                if ($respnum ne '') {
                    $respnumlookup->{$count} = $respnum;
                    $startline->{$count} = $total;
                    $total += $bubble_lines_per_response{$respnum};
                    $count ++;
                }
            }
        }
    }
    return $total;
}

sub scantron_validator_lettnum {
    my ($ansnum,$questnum,$quest_id,$answers_needed,$currquest,$whichline,
        $alphabet,$record,$scantron_config,$scan_data,$randomorder,
        $randompick,$respnumlookup) = @_;

    # Qon 'letter' implies for each slot in currquest we have:
    #    ? or * for doubles, a letter in A-Z for a bubble, and
    #    about anything else (esp. a value of Qoff) for missing
    #    bubbles.
    #
    # Qon 'number' implies each slot gives a digit that indexes the
    #    bubbles filled, or Qoff, or a non-number for unbubbled lines,
    #    and * or ? for double bubbles on a single line.
    #

    my $matchon;
    if ($$scantron_config{'Qon'} eq 'letter') {
        $matchon = '[A-Z]';
    } elsif ($$scantron_config{'Qon'} eq 'number') {
        $matchon = '\d';
    }
    my $occurrences = 0;
    my $responsenum = $questnum-1;
    if (($randompick || $randomorder) && (ref($respnumlookup) eq 'HASH')) {
       $responsenum = $respnumlookup->{$questnum-1}
    }
    if (($responsetype_per_response{$responsenum} eq 'essayresponse') ||
        ($responsetype_per_response{$responsenum} eq 'formularesponse') ||
        ($responsetype_per_response{$responsenum} eq 'stringresponse') ||
        ($responsetype_per_response{$responsenum} eq 'imageresponse') ||
        ($responsetype_per_response{$responsenum} eq 'reactionresponse') ||
        ($responsetype_per_response{$responsenum} eq 'organicresponse')) {
        my @singlelines = split('',$currquest);
        foreach my $entry (@singlelines) {
            $occurrences = &occurence_count($entry,$matchon);
            if ($occurrences > 1) {
                last;
            }
        }
    } else {
        $occurrences = &occurence_count($currquest,$matchon); 
    }
    if (($currquest =~ /\?/ || $currquest =~ /\*/) || ($occurrences > 1)) {
        push(@{$record->{'scantron.doubleerror'}},$quest_id);
        for (my $ans=0; $ans<$answers_needed; $ans++) {
            my $bubble = substr($currquest,$ans,1);
            if ($bubble =~ /$matchon/ ) {
                if ($$scantron_config{'Qon'} eq 'number') {
                    if ($bubble == 0) {
                        $bubble = 10; 
                    }
                    $record->{"scantron.$ansnum.answer"} = 
                        $alphabet->[$bubble-1];
                } else {
                    $record->{"scantron.$ansnum.answer"} = $bubble;
                }
            } else {
                $record->{"scantron.$ansnum.answer"}='';
            }
            $ansnum++;
        }
    } elsif (!defined($currquest)
            || (&occurence_count($currquest, $$scantron_config{'Qoff'}) == length($currquest))
            || (&occurence_count($currquest,$matchon) == 0)) {
        for (my $ans=0; $ans<$answers_needed; $ans++ ) {
            $record->{"scantron.$ansnum.answer"}='';
            $ansnum++;
        }
        if (!&scan_data($scan_data,"$whichline.no_bubble.$quest_id")) {
            push(@{$record->{'scantron.missingerror'}},$quest_id);
        }
    } else {
        if ($$scantron_config{'Qon'} eq 'number') {
            $currquest = &digits_to_letters($currquest);            
        }
        for (my $ans=0; $ans<$answers_needed; $ans++) {
            my $bubble = substr($currquest,$ans,1);
            $record->{"scantron.$ansnum.answer"} = $bubble;
            $ansnum++;
        }
    }
    return $ansnum;
}

sub scantron_validator_positional {
    my ($ansnum,$questnum,$quest_id,$answers_needed,$currquest,
        $whichline,$alphabet,$record,$scantron_config,$scan_data,
        $randomorder,$randompick,$respnumlookup) = @_;

    # Otherwise there's a positional notation;
    # each bubble line requires Qlength items, and there are filled in
    # bubbles for each case where there 'Qon' characters.
    #

    my @array=split($$scantron_config{'Qon'},$currquest,-1);

    # If the split only gives us one element.. the full length of the
    # answer string, no bubbles are filled in:

    if ($answers_needed eq '') {
        return;
    }

    if (length($array[0]) eq $$scantron_config{'Qlength'}*$answers_needed) {
        for (my $ans=0; $ans<$answers_needed; $ans++ ) {
            $record->{"scantron.$ansnum.answer"}='';
            $ansnum++;
        }
        if (!&scan_data($scan_data,"$whichline.no_bubble.$quest_id")) {
            push(@{$record->{"scantron.missingerror"}},$quest_id);
        }
    } elsif (scalar(@array) == 2) {
        my $location = length($array[0]);
        my $line_num = int($location / $$scantron_config{'Qlength'});
        my $bubble   = $alphabet->[$location % $$scantron_config{'Qlength'}];
        for (my $ans=0; $ans<$answers_needed; $ans++) {
            if ($ans eq $line_num) {
                $record->{"scantron.$ansnum.answer"} = $bubble;
            } else {
                $record->{"scantron.$ansnum.answer"} = ' ';
            }
            $ansnum++;
         }
    } else {
        #  If there's more than one instance of a bubble character
        #  That's a double bubble; with positional notation we can
        #  record all the bubbles filled in as well as the
        #  fact this response consists of multiple bubbles.
        #
        my $responsenum = $questnum-1;
        if (($randompick || $randomorder) && (ref($respnumlookup) eq 'HASH')) {
            $responsenum = $respnumlookup->{$questnum-1}
        }
        if (($responsetype_per_response{$responsenum} eq 'essayresponse') ||
            ($responsetype_per_response{$responsenum} eq 'formularesponse') ||
            ($responsetype_per_response{$responsenum} eq 'stringresponse') ||
            ($responsetype_per_response{$responsenum} eq 'imageresponse') ||
            ($responsetype_per_response{$responsenum} eq 'reactionresponse') ||
            ($responsetype_per_response{$responsenum} eq 'organicresponse')) {
            my $doubleerror = 0;
            while (($currquest >= $$scantron_config{'Qlength'}) && 
                   (!$doubleerror)) {
               my $currline = substr($currquest,0,$$scantron_config{'Qlength'});
               $currquest = substr($currquest,$$scantron_config{'Qlength'});
               my @currarray = split($$scantron_config{'Qon'},$currline,-1);
               if (length(@currarray) > 2) {
                   $doubleerror = 1;
               } 
            }
            if ($doubleerror) {
                push(@{$record->{'scantron.doubleerror'}},$quest_id);
            }
        } else {
            push(@{$record->{'scantron.doubleerror'}},$quest_id);
        }
        my $item = $ansnum;
        for (my $ans=0; $ans<$answers_needed; $ans++) {
            $record->{"scantron.$item.answer"} = '';
            $item ++;
        }

        my @ans=@array;
        my $i=0;
        my $increment = 0;
        while ($#ans) {
            $i+=length($ans[0]) + $increment;
            my $line   = int($i/$$scantron_config{'Qlength'} + $ansnum);
            my $bubble = $i%$$scantron_config{'Qlength'};
            $record->{"scantron.$line.answer"}.=$alphabet->[$bubble];
            shift(@ans);
            $increment = 1;
        }
        $ansnum += $answers_needed;
    }
    return $ansnum;
}

=pod

=item scantron_add_delay

   Adds an error message that occurred during the grading phase to a
   queue of messages to be shown after grading pass is complete

 Arguments:
   $delayqueue  - arrary ref of hash ref of error messages
   $scanline    - the scanline that caused the error
   $errormesage - the error message
   $errorcode   - a numeric code for the error

 Side Effects:
   updates the $delayqueue to have a new hash ref of the error

=cut

sub scantron_add_delay {
    my ($delayqueue,$scanline,$errormessage,$errorcode)=@_;
    push(@$delayqueue,
	 {'line' => $scanline, 'emsg' => $errormessage,
	  'ecode' => $errorcode }
	 );
}

=pod

=item scantron_find_student

   Finds the username for the current scanline

  Arguments:
   $scantron_record - hash result from scantron_parse_scanline
   $scan_data       - hash of correction information 
                      (see &scantron_getfile() form more information)
   $idmap           - hash from &username_to_idmap()
   $line            - number of current scanline
 
  Returns:
   Either 'username:domain' or undef if unknown

=cut

sub scantron_find_student {
    my ($scantron_record,$scan_data,$idmap,$line)=@_;
    my $scanID=$$scantron_record{'scantron.ID'};
    if ($scanID =~ /^\s*$/) {
 	return &scan_data($scan_data,"$line.user");
    }
    foreach my $id (keys(%$idmap)) {
 	if (lc($id) eq lc($scanID)) {
 	    return $$idmap{$id};
 	}
    }
    return undef;
}

=pod

=item scantron_filter

   Filter sub for lonnavmaps, filters out hidden resources if ignore
   hidden resources was selected

=cut

sub scantron_filter {
    my ($curres)=@_;

    if (ref($curres) && $curres->is_problem()) {
	# if the user has asked to not have either hidden
	# or 'randomout' controlled resources to be graded
	# don't include them
	if ($env{'form.scantron_options_hidden'} eq 'ignore_hidden'
	    && $curres->randomout) {
	    return 0;
	}
	return 1;
    }
    return 0;
}

=pod

=item scantron_process_corrections

   Gets correction information out of submitted form data and corrects
   the scanline

=cut

sub scantron_process_corrections {
    my ($r) = @_;
    my %scantron_config=&Apache::lonnet::get_scantron_config($env{'form.scantron_format'});
    my ($scanlines,$scan_data)=&scantron_getfile();
    my $classlist=&Apache::loncoursedata::get_classlist();
    my $which=$env{'form.scantron_line'};
    my $line=&scantron_get_line($scanlines,$scan_data,$which);
    my ($skip,$err,$errmsg);
    if ($env{'form.scantron_skip_record'}) {
	$skip=1;
    } elsif ($env{'form.scantron_corrections'} =~ /^(duplicate|incorrect)ID$/) {
	my $newstudent=$env{'form.scantron_username'}.':'.
	    $env{'form.scantron_domain'};
	my $newid=$classlist->{$newstudent}->[&Apache::loncoursedata::CL_ID];
	($line,$err,$errmsg)=
	    &scantron_fixup_scanline(\%scantron_config,$scan_data,$line,$which,
				     'ID',{'newid'=>$newid,
				    'username'=>$env{'form.scantron_username'},
				    'domain'=>$env{'form.scantron_domain'}});
    } elsif ($env{'form.scantron_corrections'} =~ /^(duplicate|incorrect)CODE$/) {
	my $resolution=$env{'form.scantron_CODE_resolution'};
	my $newCODE;
	my %args;
	if      ($resolution eq 'use_unfound') {
	    $newCODE='use_unfound';
	} elsif ($resolution eq 'use_found') {
	    $newCODE=$env{'form.scantron_CODE_selectedvalue'};
	} elsif ($resolution eq 'use_typed') {
	    $newCODE=$env{'form.scantron_CODE_newvalue'};
	} elsif ($resolution =~ /^use_closest_(\d+)/) {
	    $newCODE=$env{"form.scantron_CODE_closest_$1"};
	}
	if ($env{'form.scantron_corrections'} eq 'duplicateCODE') {
	    $args{'CODE_ignore_dup'}=1;
	}
	$args{'CODE'}=$newCODE;
	($line,$err,$errmsg)=
	    &scantron_fixup_scanline(\%scantron_config,$scan_data,$line,$which,
				     'CODE',\%args);
    } elsif ($env{'form.scantron_corrections'} =~ /^(missing|double)bubble$/) {
	foreach my $question (split(',',$env{'form.scantron_questions'})) {
	    ($line,$err,$errmsg)=
		&scantron_fixup_scanline(\%scantron_config,$scan_data,$line,
					 $which,'answer',
					 { 'question'=>$question,
		      		   'response'=>$env{"form.scantron_correct_Q_$question"},
                                   'questionnum'=>$env{"form.scantron_questionnum_Q_$question"}});
	    if ($err) { last; }
	}
    }
    if ($err) {
	$r->print(
            '<p class="LC_error">'
           .&mt('Unable to accept last correction, an error occurred: [_1]',
                $errmsg)
           .'</p>');
    } else {
	&scantron_put_line($scanlines,$scan_data,$which,$line,$skip);
	&scantron_putfile($scanlines,$scan_data);
    }
}

=pod

=item reset_skipping_status

   Forgets the current set of remember skipped scanlines (and thus
   reverts back to considering all lines in the
   scantron_skipped_<filename> file)

=cut

sub reset_skipping_status {
    my ($scanlines,$scan_data)=&scantron_getfile();
    &scan_data($scan_data,'remember_skipping',undef,1);
    &scantron_putfile(undef,$scan_data);
}

=pod

=item start_skipping

   Marks a scanline to be skipped. 

=cut

sub start_skipping {
    my ($scan_data,$i)=@_;
    my %remembered=split(':',&scan_data($scan_data,'remember_skipping'));
    if ($env{'form.scantron_options_redo'} =~ /^redo_/) {
	$remembered{$i}=2;
    } else {
	$remembered{$i}=1;
    }
    &scan_data($scan_data,'remember_skipping',join(':',%remembered));
}

=pod

=item should_be_skipped

   Checks whether a scanline should be skipped.

=cut

sub should_be_skipped {
    my ($scanlines,$scan_data,$i)=@_;
    if ($env{'form.scantron_options_redo'} !~ /^redo_/) {
	# not redoing old skips
	if ($scanlines->{'skipped'}[$i]) { return 1; }
	return 0;
    }
    my %remembered=split(':',&scan_data($scan_data,'remember_skipping'));

    if (exists($remembered{$i}) && $remembered{$i} != 2 ) {
	return 0;
    }
    return 1;
}

=pod

=item remember_current_skipped

   Discovers what scanlines are in the scantron_skipped_<filename>
   file and remembers them into scan_data for later use.

=cut

sub remember_current_skipped {
    my ($scanlines,$scan_data)=&scantron_getfile();
    my %to_remember;
    for (my $i=0;$i<=$scanlines->{'count'};$i++) {
	if ($scanlines->{'skipped'}[$i]) {
	    $to_remember{$i}=1;
	}
    }

    &scan_data($scan_data,'remember_skipping',join(':',%to_remember));
    &scantron_putfile(undef,$scan_data);
}

=pod

=item check_for_error

    Checks if there was an error when attempting to remove a specific
    scantron_.. bubblesheet data file. Prints out an error if
    something went wrong.

=cut

sub check_for_error {
    my ($r,$result)=@_;
    if ($result ne 'ok' && $result ne 'not_found' ) {
	$r->print(&mt("An error occurred ([_1]) when trying to remove the existing corrections.",$result));
    }
}

=pod

=item scantron_warning_screen

   Interstitial screen to make sure the operator has selected the
   correct options before we start the validation phase.

=cut

sub scantron_warning_screen {
    my ($button_text,$symb)=@_;
    my $title=&Apache::lonnet::gettitle($env{'form.selectpage'});
    my %scantron_config=&Apache::lonnet::get_scantron_config($env{'form.scantron_format'});
    my $CODElist;
    if ($scantron_config{'CODElocation'} &&
	$scantron_config{'CODEstart'} &&
	$scantron_config{'CODElength'}) {
	$CODElist=$env{'form.scantron_CODElist'};
	if ($env{'form.scantron_CODElist'} eq '') { $CODElist='<span class="LC_warning">'.&mt('None').'</span>'; }
	$CODElist=
	    '<tr><td><b>'.&mt('List of CODES to validate against:').'</b></td><td><tt>'.
	    $env{'form.scantron_CODElist'}.'</tt></td></tr>';
    }
    my $lastbubblepoints;
    if ($env{'form.scantron_lastbubblepoints'} ne '') {
        $lastbubblepoints =
            '<tr><td><b>'.&mt('Hand-graded items: points from last bubble in row').'</b></td><td><tt>'.
            $env{'form.scantron_lastbubblepoints'}.'</tt></td></tr>';
    }
    return ('
<p>
<span class="LC_warning">
'.&mt("Please double check the information below before clicking on '[_1]'",&mt($button_text)).'</span>
</p>
<table>
<tr><td><b>'.&mt('Sequence to be Graded:').'</b></td><td>'.$title.'</td></tr>
<tr><td><b>'.&mt('Data File that will be used:').'</b></td><td><tt>'.$env{'form.scantron_selectfile'}.'</tt></td></tr>
'.$CODElist.$lastbubblepoints.'
</table>
<p> '.&mt("If this information is correct, please click on '[_1]'.",&mt($button_text)).'<br />
'.&mt('If something is incorrect, please return to [_1]Grade/Manage/Review Bubblesheets[_2] to start over.','<a href="/adm/grades?symb='.$symb.'&command=scantron_selectphase" class="LC_info">','</a>').'</p>

<br />
');
}

=pod

=item scantron_do_warning

   Check if the operator has picked something for all required
   fields. Error out if something is missing.

=cut

sub scantron_do_warning {
    my ($r,$symb)=@_;
    if (!$symb) {return '';}
    my $default_form_data=&defaultFormData($symb);
    $r->print(&scantron_form_start().$default_form_data);
    if ( $env{'form.selectpage'} eq '' ||
	 $env{'form.scantron_selectfile'} eq '' ||
	 $env{'form.scantron_format'} eq '' ) {
	$r->print("<p>".&mt('You have forgotten to specify some information. Please go Back and try again.')."</p>");
	if ( $env{'form.selectpage'} eq '') {
	    $r->print('<p><span class="LC_error">'.&mt('You have not selected a Sequence to grade').'</span></p>');
	} 
	if ( $env{'form.scantron_selectfile'} eq '') {
	    $r->print('<p><span class="LC_error">'.&mt("You have not selected a file that contains the student's response data.").'</span></p>');
	} 
	if ( $env{'form.scantron_format'} eq '') {
	    $r->print('<p><span class="LC_error">'.&mt("You have not selected the format of the student's response data.").'</span></p>');
	} 
    } else {
	my $warning=&scantron_warning_screen('Grading: Validate Records',$symb);
        my $bubbledbyhand=&hand_bubble_option();
	$r->print('
'.$warning.$bubbledbyhand.'
<input type="submit" name="submit" value="'.&mt('Grading: Validate Records').'" />
<input type="hidden" name="command" value="scantron_validate" />
');
    }
    $r->print("</form><br />");
    return '';
}

=pod

=item scantron_form_start

    html hidden input for remembering all selected grading options

=cut

sub scantron_form_start {
    my ($max_bubble)=@_;
    my $result= <<SCANTRONFORM;
<form method="post" enctype="multipart/form-data" action="/adm/grades" name="scantronupload">
  <input type="hidden" name="selectpage" value="$env{'form.selectpage'}" />
  <input type="hidden" name="scantron_format" value="$env{'form.scantron_format'}" />
  <input type="hidden" name="scantron_selectfile" value="$env{'form.scantron_selectfile'}" />
  <input type="hidden" name="scantron_maxbubble" value="$max_bubble" />
  <input type="hidden" name="scantron_CODElist" value="$env{'form.scantron_CODElist'}" />
  <input type="hidden" name="scantron_CODEunique" value="$env{'form.scantron_CODEunique'}" />
  <input type="hidden" name="scantron_options_redo" value="$env{'form.scantron_options_redo'}" />
  <input type="hidden" name="scantron_options_ignore" value="$env{'form.scantron_options_ignore'}" />
  <input type="hidden" name="scantron_options_hidden" value="$env{'form.scantron_options_hidden'}" />
SCANTRONFORM

  my $line = 0;
    while (defined($env{"form.scantron.bubblelines.$line"})) {
       my $chunk =
	   '<input type="hidden" name="scantron.bubblelines.'.$line.'" value="'.$env{"form.scantron.bubblelines.$line"}.'" />'."\n";
       $chunk .=
	   '<input type="hidden" name="scantron.first_bubble_line.'.$line.'" value="'.$env{"form.scantron.first_bubble_line.$line"}.'" />'."\n";
       $chunk .= 
           '<input type="hidden" name="scantron.sub_bubblelines.'.$line.'" value="'.$env{"form.scantron.sub_bubblelines.$line"}.'" />'."\n";
       $chunk .=
           '<input type="hidden" name="scantron.responsetype.'.$line.'" value="'.$env{"form.scantron.responsetype.$line"}.'" />'."\n";
       $chunk .=
           '<input type="hidden" name="scantron.residpart.'.$line.'" value="'.$env{"form.scantron.residpart.$line"}.'" />'."\n";
       $result .= $chunk;
       $line++;
    }
    return $result;
}

=pod

=item scantron_validate_file

    Dispatch routine for doing validation of a bubblesheet data file.

    Also processes any necessary information resets that need to
    occur before validation begins (ignore previous corrections,
    restarting the skipped records processing)

=cut

sub scantron_validate_file {
    my ($r,$symb) = @_;
    if (!$symb) {return '';}
    my $default_form_data=&defaultFormData($symb);
    
    # do the detection of only doing skipped records first before we delete
    # them when doing the corrections reset
    if ($env{'form.scantron_options_redo'} ne 'redo_skipped_ready') {
	&reset_skipping_status();
    }
    if ($env{'form.scantron_options_redo'} eq 'redo_skipped') {
	&remember_current_skipped();
	$env{'form.scantron_options_redo'}='redo_skipped_ready';
    }

    if ($env{'form.scantron_options_ignore'} eq 'ignore_corrections') {
	&check_for_error($r,&scantron_remove_file('corrected'));
	&check_for_error($r,&scantron_remove_file('skipped'));
	&check_for_error($r,&scantron_remove_scan_data());
	$env{'form.scantron_options_ignore'}='done';
    }

    if ($env{'form.scantron_corrections'}) {
	&scantron_process_corrections($r);
    }
    $r->print('<p>'.&mt('Gathering necessary information.').'</p>');$r->rflush();
    #get the student pick code ready
    $r->print(&Apache::loncommon::studentbrowser_javascript());
    my $nav_error;
    my %scantron_config=&Apache::lonnet::get_scantron_config($env{'form.scantron_format'});
    my $max_bubble=&scantron_get_maxbubble(\$nav_error,\%scantron_config);
    if ($nav_error) {
        $r->print(&navmap_errormsg());
        return '';
    }
    my $result=&scantron_form_start($max_bubble).$default_form_data;
    if ($env{'form.scantron_lastbubblepoints'} ne '') {
        $result .= '<input type="hidden" name="scantron_lastbubblepoints" value="'.$env{'form.scantron_lastbubblepoints'}.'" />';
    }
    $r->print($result);
    
    my @validate_phases=( 'sequence',
			  'ID',
			  'CODE',
			  'doublebubble',
			  'missingbubbles');
    if (!$env{'form.validatepass'}) {
	$env{'form.validatepass'} = 0;
    }
    my $currentphase=$env{'form.validatepass'};


    my $stop=0;
    while (!$stop && $currentphase < scalar(@validate_phases)) {
	$r->print(&mt('Validating '.$validate_phases[$currentphase]).'<br />');
	$r->rflush();

	my $which="scantron_validate_".$validate_phases[$currentphase];
	{
	    no strict 'refs';
	    ($stop,$currentphase)=&$which($r,$currentphase);
	}
    }
    if (!$stop) {
	my $warning=&scantron_warning_screen('Start Grading',$symb);
	$r->print(&mt('Validation process complete.').'<br />'.
                  $warning.
                  &mt('Perform verification for each student after storage of submissions?').
                  '&nbsp;<span class="LC_nobreak"><label>'.
                  '<input type="radio" name="verifyrecord" value="1" />'.&mt('Yes').'</label>'.
                  ('&nbsp;'x3).'<label>'.
                  '<input type="radio" name="verifyrecord" value="0" checked="checked" />'.&mt('No').
                  '</label></span><br />'.
                  &mt('Grading will take longer if you use verification.').'<br />'.
                  &mt('Otherwise, Grade/Manage/Review Bubblesheets [_1] Review bubblesheet data can be used once grading is complete.','&raquo;').'<br /><br />'.
                  '<input type="submit" name="submit" value="'.&mt('Start Grading').'" />'.
                  '<input type="hidden" name="command" value="scantron_process" />'."\n");
    } else {
	$r->print('<input type="hidden" name="command" value="scantron_validate" />');
	$r->print("<input type='hidden' name='validatepass' value='".$currentphase."' />");
    }
    if ($stop) {
	if ($validate_phases[$currentphase] eq 'sequence') {
	    $r->print('<input type="submit" name="submit" value="'.&mt('Ignore').' &rarr; " />');
	    $r->print(' '.&mt('this error').' <br />');

            $r->print('<p>'.&mt('Or return to [_1]Grade/Manage/Review Bubblesheets[_2] to start over.','<a href="/adm/grades?symb='.$symb.'&command=scantron_selectphase" class="LC_info">','</a>').'</p>');
	} else {
            if ($validate_phases[$currentphase] eq 'doublebubble' || $validate_phases[$currentphase] eq 'missingbubbles') {
	        $r->print('<input type="button" name="submitbutton" value="'.&mt('Continue').' &rarr;" onclick="javascript:verify_bubble_radio(this.form)" />');
            } else {
                $r->print('<input type="submit" name="submit" value="'.&mt('Continue').' &rarr;" />');
            }
	    $r->print(' '.&mt('using corrected info').' <br />');
	    $r->print("<input type='submit' value='".&mt("Skip")."' name='scantron_skip_record' />");
	    $r->print(" ".&mt("this scanline saving it for later."));
	}
    }
    $r->print(" </form><br />");
    return '';
}


=pod

=item scantron_remove_file

   Removes the requested bubblesheet data file, makes sure that
   scantron_original_<filename> is never removed


=cut

sub scantron_remove_file {
    my ($which)=@_;
    my $cname=$env{'course.'.$env{'request.course.id'}.'.num'};
    my $cdom=$env{'course.'.$env{'request.course.id'}.'.domain'};
    my $file='scantron_';
    if ($which eq 'corrected' || $which eq 'skipped') {
	$file.=$which.'_';
    } else {
	return 'refused';
    }
    $file.=$env{'form.scantron_selectfile'};
    return &Apache::lonnet::removeuserfile($cname,$cdom,$file);
}


=pod

=item scantron_remove_scan_data

   Removes all scan_data correction for the requested bubblesheet
   data file.  (In the case that both the are doing skipped records we need
   to remember the old skipped lines for the time being so that element
   persists for a while.)

=cut

sub scantron_remove_scan_data {
    my $cname=$env{'course.'.$env{'request.course.id'}.'.num'};
    my $cdom=$env{'course.'.$env{'request.course.id'}.'.domain'};
    my @keys=&Apache::lonnet::getkeys('nohist_scantrondata',$cdom,$cname);
    my @todelete;
    my $filename=$env{'form.scantron_selectfile'};
    foreach my $key (@keys) {
	if ($key=~/^\Q$filename\E_/) {
	    if ($env{'form.scantron_options_redo'} eq 'redo_skipped_ready' &&
		$key=~/remember_skipping/) {
		next;
	    }
	    push(@todelete,$key);
	}
    }
    my $result;
    if (@todelete) {
	$result = &Apache::lonnet::del('nohist_scantrondata',
				       \@todelete,$cdom,$cname);
    } else {
	$result = 'ok';
    }
    return $result;
}


=pod

=item scantron_getfile

    Fetches the requested bubblesheet data file (all 3 versions), and
    the scan_data hash
  
  Arguments:
    None

  Returns:
    2 hash references

     - first one has 
         orig      -
         corrected -
         skipped   -  each of which points to an array ref of the specified
                      file broken up into individual lines
         count     - number of scanlines
 
     - second is the scan_data hash possible keys are
       ($number refers to scanline numbered $number and thus the key affects
        only that scanline
        $bubline refers to the specific bubble line element and the aspects
        refers to that specific bubble line element)

       $number.user - username:domain to use
       $number.CODE_ignore_dup 
                    - ignore the duplicate CODE error 
       $number.useCODE
                    - use the CODE in the scanline as is
       $number.no_bubble.$bubline
                    - it is valid that there is no bubbled in bubble
                      at $number $bubline
       remember_skipping
                    - a frozen hash containing keys of $number and values
                      of either 
                        1 - we are on a 'do skipped records pass' and plan
                            on processing this line
                        2 - we are on a 'do skipped records pass' and this
                            scanline has been marked to skip yet again

=cut

sub scantron_getfile {
    #FIXME really would prefer a scantron directory
    my $cname=$env{'course.'.$env{'request.course.id'}.'.num'};
    my $cdom=$env{'course.'.$env{'request.course.id'}.'.domain'};
    my $lines;
    $lines=&Apache::lonnet::getfile('/uploaded/'.$cdom.'/'.$cname.'/'.
		       'scantron_orig_'.$env{'form.scantron_selectfile'});
    my %scanlines;
    $scanlines{'orig'}=[(split("\n",$lines,-1))];
    my $temp=$scanlines{'orig'};
    $scanlines{'count'}=$#$temp;

    $lines=&Apache::lonnet::getfile('/uploaded/'.$cdom.'/'.$cname.'/'.
		       'scantron_corrected_'.$env{'form.scantron_selectfile'});
    if ($lines eq '-1') {
	$scanlines{'corrected'}=[];
    } else {
	$scanlines{'corrected'}=[(split("\n",$lines,-1))];
    }
    $lines=&Apache::lonnet::getfile('/uploaded/'.$cdom.'/'.$cname.'/'.
		       'scantron_skipped_'.$env{'form.scantron_selectfile'});
    if ($lines eq '-1') {
	$scanlines{'skipped'}=[];
    } else {
	$scanlines{'skipped'}=[(split("\n",$lines,-1))];
    }
    my @tmp=&Apache::lonnet::dump('nohist_scantrondata',$cdom,$cname);
    if ($tmp[0] =~ /^(error:|no_such_host)/) { @tmp=(); }
    my %scan_data = @tmp;
    return (\%scanlines,\%scan_data);
}

=pod

=item lonnet_putfile

   Wrapper routine to call &Apache::lonnet::finishuserfileupload

 Arguments:
   $contents - data to store
   $filename - filename to store $contents into

 Returns:
   result value from &Apache::lonnet::finishuserfileupload

=cut

sub lonnet_putfile {
    my ($contents,$filename)=@_;
    my $docuname=$env{'course.'.$env{'request.course.id'}.'.num'};
    my $docudom=$env{'course.'.$env{'request.course.id'}.'.domain'};
    $env{'form.sillywaytopassafilearound'}=$contents;
    &Apache::lonnet::finishuserfileupload($docuname,$docudom,'sillywaytopassafilearound',$filename);

}

=pod

=item scantron_putfile

    Stores the current version of the bubblesheet data files, and the
    scan_data hash. (Does not modify the original version only the
    corrected and skipped versions.

 Arguments:
    $scanlines - hash ref that looks like the first return value from
                 &scantron_getfile()
    $scan_data - hash ref that looks like the second return value from
                 &scantron_getfile()

=cut

sub scantron_putfile {
    my ($scanlines,$scan_data) = @_;
    #FIXME really would prefer a scantron directory
    my $cname=$env{'course.'.$env{'request.course.id'}.'.num'};
    my $cdom=$env{'course.'.$env{'request.course.id'}.'.domain'};
    if ($scanlines) {
	my $prefix='scantron_';
# no need to update orig, shouldn't change
#   &lonnet_putfile(join("\n",@{$scanlines->{'orig'}}),$prefix.'orig_'.
#		    $env{'form.scantron_selectfile'});
	&lonnet_putfile(join("\n",@{$scanlines->{'corrected'}}),
			$prefix.'corrected_'.
			$env{'form.scantron_selectfile'});
	&lonnet_putfile(join("\n",@{$scanlines->{'skipped'}}),
			$prefix.'skipped_'.
			$env{'form.scantron_selectfile'});
    }
    &Apache::lonnet::put('nohist_scantrondata',$scan_data,$cdom,$cname);
}

=pod

=item scantron_get_line

   Returns the correct version of the scanline

 Arguments:
    $scanlines - hash ref that looks like the first return value from
                 &scantron_getfile()
    $scan_data - hash ref that looks like the second return value from
                 &scantron_getfile()
    $i         - number of the requested line (starts at 0)

 Returns:
   A scanline, (either the original or the corrected one if it
   exists), or undef if the requested scanline should be
   skipped. (Either because it's an skipped scanline, or it's an
   unskipped scanline and we are not doing a 'do skipped scanlines'
   pass.

=cut

sub scantron_get_line {
    my ($scanlines,$scan_data,$i)=@_;
    if (&should_be_skipped($scanlines,$scan_data,$i)) { return undef; }
    #if ($scanlines->{'skipped'}[$i]) { return undef; }
    if ($scanlines->{'corrected'}[$i]) {return $scanlines->{'corrected'}[$i];}
    return $scanlines->{'orig'}[$i]; 
}

=pod

=item scantron_todo_count

    Counts the number of scanlines that need processing.

 Arguments:
    $scanlines - hash ref that looks like the first return value from
                 &scantron_getfile()
    $scan_data - hash ref that looks like the second return value from
                 &scantron_getfile()

 Returns:
    $count - number of scanlines to process

=cut

sub get_todo_count {
    my ($scanlines,$scan_data)=@_;
    my $count=0;
    for (my $i=0;$i<=$scanlines->{'count'};$i++) {
	my $line=&scantron_get_line($scanlines,$scan_data,$i);
	if ($line=~/^[\s\cz]*$/) { next; }
	$count++;
    }
    return $count;
}

=pod

=item scantron_put_line

    Updates the 'corrected' or 'skipped' versions of the bubblesheet
    data file.

 Arguments:
    $scanlines - hash ref that looks like the first return value from
                 &scantron_getfile()
    $scan_data - hash ref that looks like the second return value from
                 &scantron_getfile()
    $i         - line number to update
    $newline   - contents of the updated scanline
    $skip      - if true make the line for skipping and update the
                 'skipped' file

=cut

sub scantron_put_line {
    my ($scanlines,$scan_data,$i,$newline,$skip)=@_;
    if ($skip) {
	$scanlines->{'skipped'}[$i]=$newline;
	&start_skipping($scan_data,$i);
	return;
    }
    $scanlines->{'corrected'}[$i]=$newline;
}

=pod

=item scantron_clear_skip

   Remove a line from the 'skipped' file

 Arguments:
    $scanlines - hash ref that looks like the first return value from
                 &scantron_getfile()
    $scan_data - hash ref that looks like the second return value from
                 &scantron_getfile()
    $i         - line number to update

=cut

sub scantron_clear_skip {
    my ($scanlines,$scan_data,$i)=@_;
    if (exists($scanlines->{'skipped'}[$i])) {
	undef($scanlines->{'skipped'}[$i]);
	return 1;
    }
    return 0;
}

=pod

=item scantron_filter_not_exam

   Filter routine used by &Apache::lonnavmaps::retrieveResources(), to
   filter out resources that are not marked as 'exam' mode

=cut

sub scantron_filter_not_exam {
    my ($curres)=@_;
    
    if (ref($curres) && $curres->is_problem() && !$curres->is_exam()) {
	# if the user has asked to not have either hidden
	# or 'randomout' controlled resources to be graded
	# don't include them
	if ($env{'form.scantron_options_hidden'} eq 'ignore_hidden'
	    && $curres->randomout) {
	    return 0;
	}
	return 1;
    }
    return 0;
}

=pod

=item scantron_validate_sequence

    Validates the selected sequence, checking for resource that are
    not set to exam mode.

=cut

sub scantron_validate_sequence {
    my ($r,$currentphase) = @_;

    my $navmap=Apache::lonnavmaps::navmap->new();
    unless (ref($navmap)) {
        $r->print(&navmap_errormsg());
        return (1,$currentphase);
    }
    my (undef,undef,$sequence)=
	&Apache::lonnet::decode_symb($env{'form.selectpage'});

    my $map=$navmap->getResourceByUrl($sequence);

    $r->print('<input type="hidden" name="validate_sequence_exam"
                                    value="ignore" />');
    if ($env{'form.validate_sequence_exam'} ne 'ignore') {
	my @resources=
	    $navmap->retrieveResources($map,\&scantron_filter_not_exam,1,0);
	if (@resources) {
	    $r->print('<p class="LC_warning">'
               .&mt('Some resources in the sequence currently are not set to'
                   .' exam mode. Grading these resources currently may not'
                   .' work correctly.')
               .'</p>'
            );
	    return (1,$currentphase);
	}
    }

    return (0,$currentphase+1);
}



sub scantron_validate_ID {
    my ($r,$currentphase) = @_;
    
    #get student info
    my $classlist=&Apache::loncoursedata::get_classlist();
    my %idmap=&username_to_idmap($classlist);

    #get scantron line setup
    my %scantron_config=&Apache::lonnet::get_scantron_config($env{'form.scantron_format'});
    my ($scanlines,$scan_data)=&scantron_getfile();

    my $nav_error;
    &scantron_get_maxbubble(\$nav_error,\%scantron_config); # parse needs the bubble_lines.. array.
    if ($nav_error) {
        $r->print(&navmap_errormsg());
        return(1,$currentphase);
    }

    my %found=('ids'=>{},'usernames'=>{});
    for (my $i=0;$i<=$scanlines->{'count'};$i++) {
	my $line=&scantron_get_line($scanlines,$scan_data,$i);
	if ($line=~/^[\s\cz]*$/) { next; }
	my $scan_record=&scantron_parse_scanline($line,$i,\%scantron_config,
						 $scan_data);
	my $id=$$scan_record{'scantron.ID'};
	my $found;
	foreach my $checkid (keys(%idmap)) {
	    if (lc($checkid) eq lc($id)) { $found=$checkid;last; }
	}
	if ($found) {
	    my $username=$idmap{$found};
	    if ($found{'ids'}{$found}) {
		&scantron_get_correction($r,$i,$scan_record,\%scantron_config,
					 $line,'duplicateID',$found);
		return(1,$currentphase);
	    } elsif ($found{'usernames'}{$username}) {
		&scantron_get_correction($r,$i,$scan_record,\%scantron_config,
					 $line,'duplicateID',$username);
		return(1,$currentphase);
	    }
	    #FIXME store away line we previously saw the ID on to use above
	    $found{'ids'}{$found}++;
	    $found{'usernames'}{$username}++;
	} else {
	    if ($id =~ /^\s*$/) {
		my $username=&scan_data($scan_data,"$i.user");
		if (defined($username) && $found{'usernames'}{$username}) {
		    &scantron_get_correction($r,$i,$scan_record,
					     \%scantron_config,
					     $line,'duplicateID',$username);
		    return(1,$currentphase);
		} elsif (!defined($username)) {
		    &scantron_get_correction($r,$i,$scan_record,
					     \%scantron_config,
					     $line,'incorrectID');
		    return(1,$currentphase);
		}
		$found{'usernames'}{$username}++;
	    } else {
		&scantron_get_correction($r,$i,$scan_record,\%scantron_config,
					 $line,'incorrectID');
		return(1,$currentphase);
	    }
	}
    }

    return (0,$currentphase+1);
}


sub scantron_get_correction {
    my ($r,$i,$scan_record,$scan_config,$line,$error,$arg,
        $randomorder,$randompick,$respnumlookup,$startline)=@_;
#FIXME in the case of a duplicated ID the previous line, probably need
#to show both the current line and the previous one and allow skipping
#the previous one or the current one

    if ( $$scan_record{'scantron.PaperID'} =~ /\S/) {
        $r->print(
            '<p class="LC_warning">'
           .&mt('An error was detected ([_1]) for PaperID [_2]',
                "<b>$error</b>",
                '<tt>'.$$scan_record{'scantron.PaperID'}.'</tt>')
           ."</p> \n");
    } else {
        $r->print(
            '<p class="LC_warning">'
           .&mt('An error was detected ([_1]) in scanline [_2] [_3]',
                "<b>$error</b>", $i, "<pre>$line</pre>")
           ."</p> \n");
    }
    my $message =
        '<p>'
       .&mt('The ID on the form is [_1]',
            "<tt>$$scan_record{'scantron.ID'}</tt>")
       .'<br />'
       .&mt('The name on the paper is [_1], [_2]',
            $$scan_record{'scantron.LastName'},
            $$scan_record{'scantron.FirstName'})
       .'</p>';

    $r->print('<input type="hidden" name="scantron_corrections" value="'.$error.'" />'."\n");
    $r->print('<input type="hidden" name="scantron_line" value="'.$i.'" />'."\n");
                           # Array populated for doublebubble or
    my @lines_to_correct;  # missingbubble errors to build javascript
                           # to validate radio button checking   

    if ($error =~ /ID$/) {
	if ($error eq 'incorrectID') {
	    $r->print('<p class="LC_warning">'.&mt("The encoded ID is not in the classlist").
		      "</p>\n");
	} elsif ($error eq 'duplicateID') {
	    $r->print('<p class="LC_warning">'.&mt("The encoded ID has also been used by a previous paper [_1]",$arg)."</p>\n");
	}
	$r->print($message);
	$r->print("<p>".&mt("How should I handle this?")." <br /> \n");
	$r->print("\n<ul><li> ");
	#FIXME it would be nice if this sent back the user ID and
	#could do partial userID matches
	$r->print(&Apache::loncommon::selectstudent_link('scantronupload',
				       'scantron_username','scantron_domain'));
	$r->print(": <input type='text' name='scantron_username' value='' />");
	$r->print("\n:\n".
		 &Apache::loncommon::select_dom_form($env{'request.role.domain'},'scantron_domain'));

	$r->print('</li>');
    } elsif ($error =~ /CODE$/) {
	if ($error eq 'incorrectCODE') {
	    $r->print('<p class="LC_warning">'.&mt("The encoded CODE is not in the list of possible CODEs.")."</p>\n");
	} elsif ($error eq 'duplicateCODE') {
	    $r->print('<p class="LC_warning">'.&mt("The encoded CODE has also been used by a previous paper [_1], and CODEs are supposed to be unique.",join(', ',@{$arg}))."</p>\n");
	}
        $r->print("<p>".&mt('The CODE on the form is [_1]',
                            "<tt>'$$scan_record{'scantron.CODE'}'</tt>")
                 ."</p>\n");
	$r->print($message);
	$r->print("<p>".&mt("How should I handle this?")."</p>\n");
	$r->print("\n<br /> ");
	my $i=0;
	if ($error eq 'incorrectCODE' 
	    && $$scan_record{'scantron.CODE'}=~/\S/ ) {
	    my ($max,$closest)=&scantron_get_closely_matching_CODEs($arg,$$scan_record{'scantron.CODE'});
	    if ($closest > 0) {
		foreach my $testcode (@{$closest}) {
		    my $checked='';
		    if (!$i) { $checked=' checked="checked"'; }
		    $r->print("
   <label>
       <input type='radio' name='scantron_CODE_resolution' value='use_closest_$i'$checked />
       ".&mt("Use the similar CODE [_1] instead.",
	    "<b><tt>".$testcode."</tt></b>")."
    </label>
    <input type='hidden' name='scantron_CODE_closest_$i' value='$testcode' />");
		    $r->print("\n<br />");
		    $i++;
		}
	    }
	}
	if ($$scan_record{'scantron.CODE'}=~/\S/ ) {
	    my $checked; if (!$i) { $checked=' checked="checked"'; }
	    $r->print("
    <label>
        <input type='radio' name='scantron_CODE_resolution' value='use_unfound'$checked />
       ".&mt("Use the CODE [_1] that was on the paper, ignoring the error.",
	     "<b><tt>".$$scan_record{'scantron.CODE'}."</tt></b>")."
    </label>");
	    $r->print("\n<br />");
	}

	$r->print(&Apache::lonhtmlcommon::scripttag(<<ENDSCRIPT));
function change_radio(field) {
    var slct=document.scantronupload.scantron_CODE_resolution;
    var i;
    for (i=0;i<slct.length;i++) {
        if (slct[i].value==field) { slct[i].checked=true; }
    }
}
ENDSCRIPT
	my $href="/adm/pickcode?".
	   "form=".&escape("scantronupload").
	   "&scantron_format=".&escape($env{'form.scantron_format'}).
	   "&scantron_CODElist=".&escape($env{'form.scantron_CODElist'}).
	   "&curCODE=".&escape($$scan_record{'scantron.CODE'}).
	   "&scantron_selectfile=".&escape($env{'form.scantron_selectfile'});
	if ($env{'form.scantron_CODElist'} =~ /\S/) { 
	    $r->print("
    <label>
       <input type='radio' name='scantron_CODE_resolution' value='use_found' />
       ".&mt("[_1]Select[_2] a CODE from the list of all CODEs and use it.",
	     "<a target='_blank' href='$href'>","</a>")."
    </label> 
    ".&mt("Selected CODE is [_1]",'<input readonly="readonly" type="text" size="8" name="scantron_CODE_selectedvalue" onfocus="javascript:change_radio(\'use_found\')" onchange="javascript:change_radio(\'use_found\')" />'));
	    $r->print("\n<br />");
	}
	$r->print("
    <label>
       <input type='radio' name='scantron_CODE_resolution' value='use_typed' />
       ".&mt("Use [_1] as the CODE.",
	     "</label><input type='text' size='8' name='scantron_CODE_newvalue' onfocus=\"javascript:change_radio('use_typed')\" onkeypress=\"javascript:change_radio('use_typed')\" />"));
	$r->print("\n<br /><br />");
    } elsif ($error eq 'doublebubble') {
	$r->print('<p class="LC_warning">'.&mt("There have been multiple bubbles scanned for some question(s)")."</p>\n");

	# The form field scantron_questions is acutally a list of line numbers.
	# represented by this form so:

	my $line_list = &questions_to_line_list($arg,$randomorder,$randompick,
                                                $respnumlookup,$startline);

	$r->print('<input type="hidden" name="scantron_questions" value="'.
		  $line_list.'" />');
	$r->print($message);
	$r->print("<p>".&mt("Please indicate which bubble should be used for grading")."</p>");
	foreach my $question (@{$arg}) {
	    my @linenums = &prompt_for_corrections($r,$question,$scan_config,
                                                   $scan_record, $error,
                                                   $randomorder,$randompick,
                                                   $respnumlookup,$startline);
            push(@lines_to_correct,@linenums);
	}
        $r->print(&verify_bubbles_checked(@lines_to_correct));
    } elsif ($error eq 'missingbubble') {
	$r->print('<p class="LC_warning">'.&mt("There have been [_1]no[_2] bubbles scanned for some question(s)",'<b>','</b>')."</p>\n");
	$r->print($message);
	$r->print("<p>".&mt("Please indicate which bubble should be used for grading.")."</p>");
	$r->print(&mt("Some questions have no scanned bubbles.")."\n");

	# The form field scantron_questions is actually a list of line numbers not
	# a list of question numbers. Therefore:
	#
	
	my $line_list = &questions_to_line_list($arg,$randomorder,$randompick,
                                                $respnumlookup,$startline);

	$r->print('<input type="hidden" name="scantron_questions" value="'.
		  $line_list.'" />');
	foreach my $question (@{$arg}) {
	    my @linenums = &prompt_for_corrections($r,$question,$scan_config,
                                                   $scan_record, $error,
                                                   $randomorder,$randompick,
                                                   $respnumlookup,$startline);
            push(@lines_to_correct,@linenums);
	}
        $r->print(&verify_bubbles_checked(@lines_to_correct));
    } else {
	$r->print("\n<ul>");
    }
    $r->print("\n</li></ul>");
}

sub verify_bubbles_checked {
    my (@ansnums) = @_;
    my $ansnumstr = join('","',@ansnums);
    my $warning = &mt("A bubble or 'No bubble' selection has not been made for one or more lines.");
    &js_escape(\$warning);
    my $output = &Apache::lonhtmlcommon::scripttag(<<ENDSCRIPT);
function verify_bubble_radio(form) {
    var ansnumArray = new Array ("$ansnumstr");
    var need_bubble_count = 0;
    for (var i=0; i<ansnumArray.length; i++) {
        if (form.elements["scantron_correct_Q_"+ansnumArray[i]].length > 1) {
            var bubble_picked = 0; 
            for (var j=0; j<form.elements["scantron_correct_Q_"+ansnumArray[i]].length; j++) {
                if (form.elements["scantron_correct_Q_"+ansnumArray[i]][j].checked == true) {
                    bubble_picked = 1;
                }
            }
            if (bubble_picked == 0) {
                need_bubble_count ++;
            }
        }
    }
    if (need_bubble_count) {
        alert("$warning");
        return;
    }
    form.submit(); 
}
ENDSCRIPT
    return $output;
}

=pod

=item  questions_to_line_list

Converts a list of questions into a string of comma separated
line numbers in the answer sheet used by the questions.  This is
used to fill in the scantron_questions form field.

  Arguments:
     questions    - Reference to an array of questions.
     randomorder  - True if randomorder in use.
     randompick   - True if randompick in use.
     respnumlookup - Reference to HASH mapping question numbers in bubble lines
                     for current line to question number used for same question
                     in "Master Seqence" (as seen by Course Coordinator).
     startline    - Reference to hash where key is question number (0 is first)
                    and key is number of first bubble line for current student
                    or code-based randompick and/or randomorder.

=cut


sub questions_to_line_list {
    my ($questions,$randomorder,$randompick,$respnumlookup,$startline) = @_;
    my @lines;

    foreach my $item (@{$questions}) {
        my $question = $item;
        my ($first,$count,$last);
        if ($item =~ /^(\d+)\.(\d+)$/) {
            $question = $1;
            my $subquestion = $2;
            my $responsenum = $question-1;
            if (($randomorder || $randompick) && (ref($respnumlookup) eq 'HASH')) {
                $responsenum = $respnumlookup->{$question-1};
                if (ref($startline) eq 'HASH') {
                    $first = $startline->{$question-1} + 1;
                }
            } else {
                $first = $first_bubble_line{$responsenum} + 1;
            }
            my @subans = split(/,/,$subdivided_bubble_lines{$responsenum});
            my $subcount = 1;
            while ($subcount<$subquestion) {
                $first += $subans[$subcount-1];
                $subcount ++;
            }
            $count = $subans[$subquestion-1];
        } else {
            my $responsenum = $question-1;
            if (($randomorder || $randompick) && (ref($respnumlookup) eq 'HASH')) {
                $responsenum = $respnumlookup->{$question-1};
                if (ref($startline) eq 'HASH') {
                    $first = $startline->{$question-1} + 1;
                }
            } else {
                $first = $first_bubble_line{$responsenum} + 1;
            }
            $count   = $bubble_lines_per_response{$responsenum};
        }
        $last = $first+$count-1;
        push(@lines, ($first..$last));
    }
    return join(',', @lines);
}

=pod 

=item prompt_for_corrections

Prompts for a potentially multiline correction to the
user's bubbling (factors out common code from scantron_get_correction
for multi and missing bubble cases).

 Arguments:
   $r           - Apache request object.
   $question    - The question number to prompt for.
   $scan_config - The scantron file configuration hash.
   $scan_record - Reference to the hash that has the the parsed scanlines.
   $error       - Type of error
   $randomorder - True if randomorder in use.
   $randompick  - True if randompick in use.
   $respnumlookup - Reference to HASH mapping question numbers in bubble lines
                    for current line to question number used for same question
                    in "Master Seqence" (as seen by Course Coordinator).
   $startline   - Reference to hash where key is question number (0 is first)
                  and value is number of first bubble line for current student
                  or code-based randompick and/or randomorder.

 Implicit inputs:
   %bubble_lines_per_response   - Starting line numbers for each question.
                                  Numbered from 0 (but question numbers are from
                                  1.
   %first_bubble_line           - Starting bubble line for each question.
   %subdivided_bubble_lines     - optionresponse, matchresponse and rankresponse 
                                  type problems render as separate sub-questions, 
                                  in exam mode. This hash contains a 
                                  comma-separated list of the lines per 
                                  sub-question.
   %responsetype_per_response   - essayresponse, formularesponse,
                                  stringresponse, imageresponse, reactionresponse,
                                  and organicresponse type problem parts can have
                                  multiple lines per response if the weight
                                  assigned exceeds 10.  In this case, only
                                  one bubble per line is permitted, but more 
                                  than one line might contain bubbles, e.g.
                                  bubbling of: line 1 - J, line 2 - J, 
                                  line 3 - B would assign 22 points.  

=cut

sub prompt_for_corrections {
    my ($r, $question, $scan_config, $scan_record, $error, $randomorder,
        $randompick, $respnumlookup, $startline) = @_;
    my ($current_line,$lines);
    my @linenums;
    my $questionnum = $question;
    my ($first,$responsenum);
    if ($question =~ /^(\d+)\.(\d+)$/) {
        $question = $1;
        my $subquestion = $2;
        if (($randomorder || $randompick) && (ref($respnumlookup) eq 'HASH')) {
            $responsenum = $respnumlookup->{$question-1};
            if (ref($startline) eq 'HASH') {
                $first = $startline->{$question-1};
            }
        } else {
            $responsenum = $question-1;
            $first = $first_bubble_line{$responsenum};
        }
        $current_line = $first + 1 ;
        my @subans = split(/,/,$subdivided_bubble_lines{$responsenum});
        my $subcount = 1;
        while ($subcount<$subquestion) {
            $current_line += $subans[$subcount-1];
            $subcount ++;
        }
        $lines = $subans[$subquestion-1];
    } else {
        if (($randomorder || $randompick) && (ref($respnumlookup) eq 'HASH')) {
            $responsenum = $respnumlookup->{$question-1};
            if (ref($startline) eq 'HASH') {
                $first = $startline->{$question-1};
            }
        } else {
            $responsenum = $question-1;
            $first = $first_bubble_line{$responsenum};
        }
        $current_line = $first + 1;
        $lines        = $bubble_lines_per_response{$responsenum};
    }
    if ($lines > 1) {
        $r->print(&mt('The group of bubble lines below responds to a single question.').'<br />');
        if (($responsetype_per_response{$responsenum} eq 'essayresponse') ||
            ($responsetype_per_response{$responsenum} eq 'formularesponse') ||
            ($responsetype_per_response{$responsenum} eq 'stringresponse') ||
            ($responsetype_per_response{$responsenum} eq 'imageresponse') ||
            ($responsetype_per_response{$responsenum} eq 'reactionresponse') ||
            ($responsetype_per_response{$responsenum} eq 'organicresponse')) {
            $r->print(&mt("Although this particular question type requires handgrading, the instructions for this question in the bubblesheet exam directed students to leave [quant,_1,line] blank on their bubblesheets.",$lines).'<br /><br />'.&mt('A non-zero score can be assigned to the student during bubblesheet grading by selecting a bubble in at least one line.').'<br />'.&mt('The score for this question will be a sum of the numeric values for the selected bubbles from each line, where A=1 point, B=2 points etc.').'<br />'.&mt("To assign a score of zero for this question, mark all lines as 'No bubble'.").'<br /><br />');
        } else {
            $r->print(&mt("Select at most one bubble in a single line and select 'No Bubble' in all the other lines. ")."<br />");
        }
    }
    for (my $i =0; $i < $lines; $i++) {
        my $selected = $$scan_record{"scantron.$current_line.answer"};
	&scantron_bubble_selector($r,$scan_config,$current_line,
	        		  $questionnum,$error,split('', $selected));
        push(@linenums,$current_line);
	$current_line++;
    }
    if ($lines > 1) {
	$r->print("<hr /><br />");
    }
    return @linenums;
}

=pod

=item scantron_bubble_selector
  
   Generates the html radiobuttons to correct a single bubble line
   possibly showing the existing the selected bubbles if known

 Arguments:
    $r           - Apache request object
    $scan_config - hash from &Apache::lonnet::get_scantron_config()
    $line        - Number of the line being displayed.
    $questionnum - Question number (may include subquestion)
    $error       - Type of error.
    @selected    - Array of bubbles picked on this line.

=cut

sub scantron_bubble_selector {
    my ($r,$scan_config,$line,$questionnum,$error,@selected)=@_;
    my $max=$$scan_config{'Qlength'};

    my $scmode=$$scan_config{'Qon'};
    if ($scmode eq 'number' || $scmode eq 'letter') {
        if (($$scan_config{'BubblesPerRow'} =~ /^\d+$/) &&
            ($$scan_config{'BubblesPerRow'} > 0)) {
            $max=$$scan_config{'BubblesPerRow'};
            if (($scmode eq 'number') && ($max > 10)) {
                $max = 10;
            } elsif (($scmode eq 'letter') && $max > 26) {
                $max = 26;
            }
        } else {
            $max = 10;
        }
    }

    my @alphabet=('A'..'Z');
    $r->print(&Apache::loncommon::start_data_table().
              &Apache::loncommon::start_data_table_row());
    $r->print('<td rowspan="2" class="LC_leftcol_header">'.$line.'</td>');
    for (my $i=0;$i<$max+1;$i++) {
	$r->print("\n".'<td align="center">');
	if ($selected[0] eq $alphabet[$i]) { $r->print('X'); shift(@selected) }
	else { $r->print('&nbsp;'); }
	$r->print('</td>');
    }
    $r->print(&Apache::loncommon::end_data_table_row().
              &Apache::loncommon::start_data_table_row());
    for (my $i=0;$i<$max;$i++) {
	$r->print("\n".
		  '<td><label><input type="radio" name="scantron_correct_Q_'.
		  $line.'" value="'.$i.'" />'.$alphabet[$i]."</label></td>");
    }
    my $nobub_checked = ' ';
    if ($error eq 'missingbubble') {
        $nobub_checked = ' checked = "checked" ';
    }
    $r->print("\n".'<td><label><input type="radio" name="scantron_correct_Q_'.
	      $line.'" value="none"'.$nobub_checked.'/>'.&mt('No bubble').
              '</label>'."\n".'<input type="hidden" name="scantron_questionnum_Q_'.
              $line.'" value="'.$questionnum.'" /></td>');
    $r->print(&Apache::loncommon::end_data_table_row().
              &Apache::loncommon::end_data_table());
}

=pod

=item num_matches

   Counts the number of characters that are the same between the two arguments.

 Arguments:
   $orig - CODE from the scanline
   $code - CODE to match against

 Returns:
   $count - integer count of the number of same characters between the
            two arguments

=cut

sub num_matches {
    my ($orig,$code) = @_;
    my @code=split(//,$code);
    my @orig=split(//,$orig);
    my $same=0;
    for (my $i=0;$i<scalar(@code);$i++) {
	if ($code[$i] eq $orig[$i]) { $same++; }
    }
    return $same;
}

=pod

=item scantron_get_closely_matching_CODEs

   Cycles through all CODEs and finds the set that has the greatest
   number of same characters as the provided CODE

 Arguments:
   $allcodes - hash ref returned by &get_codes()
   $CODE     - CODE from the current scanline

 Returns:
   2 element list
    - first elements is number of how closely matching the best fit is 
      (5 means best set has 5 matching characters)
    - second element is an arrary ref containing the set of valid CODEs
      that best fit the passed in CODE

=cut

sub scantron_get_closely_matching_CODEs {
    my ($allcodes,$CODE)=@_;
    my @CODEs;
    foreach my $testcode (sort(keys(%{$allcodes}))) {
	push(@{$CODEs[&num_matches($CODE,$testcode)]},$testcode);
    }

    return ($#CODEs,$CODEs[-1]);
}

=pod

=item get_codes

   Builds a hash which has keys of all of the valid CODEs from the selected
   set of remembered CODEs.

 Arguments:
  $old_name - name of the set of remembered CODEs
  $cdom     - domain of the course
  $cnum     - internal course name

 Returns:
  %allcodes - keys are the valid CODEs, values are all 1

=cut

sub get_codes {
    my ($old_name, $cdom, $cnum) = @_;
    if (!$old_name) {
	$old_name=$env{'form.scantron_CODElist'};
    }
    if (!$cdom) {
	$cdom =$env{'course.'.$env{'request.course.id'}.'.domain'};
    }
    if (!$cnum) {
	$cnum =$env{'course.'.$env{'request.course.id'}.'.num'};
    }
    my %result=&Apache::lonnet::get('CODEs',[$old_name,"type\0$old_name"],
				    $cdom,$cnum);
    my %allcodes;
    if ($result{"type\0$old_name"} eq 'number') {
	%allcodes=map {($_,1)} split(',',$result{$old_name});
    } else {
	%allcodes=map {(&Apache::lonprintout::num_to_letters($_),1)} split(',',$result{$old_name});
    }
    return %allcodes;
}

=pod

=item scantron_validate_CODE

   Validates all scanlines in the selected file to not have any
   invalid or underspecified CODEs and that none of the codes are
   duplicated if this was requested.

=cut

sub scantron_validate_CODE {
    my ($r,$currentphase) = @_;
    my %scantron_config=&Apache::lonnet::get_scantron_config($env{'form.scantron_format'});
    if ($scantron_config{'CODElocation'} &&
	$scantron_config{'CODEstart'} &&
	$scantron_config{'CODElength'}) {
	if (!defined($env{'form.scantron_CODElist'})) {
	    &FIXME_blow_up()
	}
    } else {
	return (0,$currentphase+1);
    }
    
    my %usedCODEs;

    my %allcodes=&get_codes();

    my $nav_error;
    &scantron_get_maxbubble(\$nav_error,\%scantron_config); # parse needs the lines per response array.
    if ($nav_error) {
        $r->print(&navmap_errormsg());
        return(1,$currentphase);
    }

    my ($scanlines,$scan_data)=&scantron_getfile();
    for (my $i=0;$i<=$scanlines->{'count'};$i++) {
	my $line=&scantron_get_line($scanlines,$scan_data,$i);
	if ($line=~/^[\s\cz]*$/) { next; }
	my $scan_record=&scantron_parse_scanline($line,$i,\%scantron_config,
						 $scan_data);
	my $CODE=$$scan_record{'scantron.CODE'};
	my $error=0;
	if (!&Apache::lonnet::validCODE($CODE)) {
	    &scantron_get_correction($r,$i,$scan_record,
				     \%scantron_config,
				     $line,'incorrectCODE',\%allcodes);
	    return(1,$currentphase);
	}
	if (%allcodes && !exists($allcodes{$CODE}) 
	    && !$$scan_record{'scantron.useCODE'}) {
	    &scantron_get_correction($r,$i,$scan_record,
				     \%scantron_config,
				     $line,'incorrectCODE',\%allcodes);
	    return(1,$currentphase);
	}
	if (exists($usedCODEs{$CODE}) 
	    && $env{'form.scantron_CODEunique'} eq 'yes'
	    && !$$scan_record{'scantron.CODE_ignore_dup'}) {
	    &scantron_get_correction($r,$i,$scan_record,
				     \%scantron_config,
				     $line,'duplicateCODE',$usedCODEs{$CODE});
	    return(1,$currentphase);
	}
	push(@{$usedCODEs{$CODE}},$$scan_record{'scantron.PaperID'});
    }
    return (0,$currentphase+1);
}

=pod

=item scantron_validate_doublebubble

   Validates all scanlines in the selected file to not have any
   bubble lines with multiple bubbles marked.

=cut

sub scantron_validate_doublebubble {
    my ($r,$currentphase) = @_;
    #get student info
    my $classlist=&Apache::loncoursedata::get_classlist();
    my %idmap=&username_to_idmap($classlist);
    my (undef,undef,$sequence)=
        &Apache::lonnet::decode_symb($env{'form.selectpage'});

    #get scantron line setup
    my %scantron_config=&Apache::lonnet::get_scantron_config($env{'form.scantron_format'});
    my ($scanlines,$scan_data)=&scantron_getfile();

    my $navmap = Apache::lonnavmaps::navmap->new();
    unless (ref($navmap)) {
        $r->print(&navmap_errormsg());
        return(1,$currentphase);
    }
    my $map=$navmap->getResourceByUrl($sequence);
    my @resources=$navmap->retrieveResources($map,\&scantron_filter,1,0);
    my ($randomorder,$randompick,@master_seq,%symb_to_resource,%grader_partids_by_symb,
        %grader_randomlists_by_symb,%orderedforcode,%respnumlookup,%startline);
    my $bubbles_per_row = &bubblesheet_bubbles_per_row(\%scantron_config);

    my $nav_error;
    if (ref($map)) {
        $randomorder = $map->randomorder();
        $randompick = $map->randompick();
        unless ($randomorder || $randompick) {
            foreach my $res ($navmap->retrieveResources($map,sub { $_[0]->is_map() },1,0,1)) {
                if ($res->randomorder()) {
                    $randomorder = 1;
                }
                if ($res->randompick()) {
                    $randompick = 1;
                }
                last if ($randomorder || $randompick);
            }
        }
        if ($randomorder || $randompick) {
            $nav_error = &get_master_seq(\@resources,\@master_seq,\%symb_to_resource);
            if ($nav_error) {
                $r->print(&navmap_errormsg());
                return(1,$currentphase);
            }
            &graders_resources_pass(\@resources,\%grader_partids_by_symb,
                                    \%grader_randomlists_by_symb,$bubbles_per_row);
        }
    } else {
        $r->print(&navmap_errormsg());
        return(1,$currentphase);
    }

    &scantron_get_maxbubble(\$nav_error,\%scantron_config); # parse needs the bubble line array.
    if ($nav_error) {
        $r->print(&navmap_errormsg());
        return(1,$currentphase);
    }

    for (my $i=0;$i<=$scanlines->{'count'};$i++) {
	my $line=&scantron_get_line($scanlines,$scan_data,$i);
	if ($line=~/^[\s\cz]*$/) { next; }
	my $scan_record=&scantron_parse_scanline($line,$i,\%scantron_config,
						 $scan_data,undef,\%idmap,$randomorder,
                                                 $randompick,$sequence,\@master_seq,
                                                 \%symb_to_resource,\%grader_partids_by_symb,
                                                 \%orderedforcode,\%respnumlookup,\%startline);
	if (!defined($$scan_record{'scantron.doubleerror'})) { next; }
	&scantron_get_correction($r,$i,$scan_record,\%scantron_config,$line,
				 'doublebubble',
				 $$scan_record{'scantron.doubleerror'},
                                 $randomorder,$randompick,\%respnumlookup,\%startline);
    	return (1,$currentphase);
    }
    return (0,$currentphase+1);
}


sub scantron_get_maxbubble {
    my ($nav_error,$scantron_config) = @_;
    if (defined($env{'form.scantron_maxbubble'}) &&
	$env{'form.scantron_maxbubble'}) {
	&restore_bubble_lines();
	return $env{'form.scantron_maxbubble'};
    }

    my (undef, undef, $sequence) =
	&Apache::lonnet::decode_symb($env{'form.selectpage'});

    my $navmap=Apache::lonnavmaps::navmap->new();
    unless (ref($navmap)) {
        if (ref($nav_error)) {
            $$nav_error = 1;
        }
        return;
    }
    my $map=$navmap->getResourceByUrl($sequence);
    my @resources=$navmap->retrieveResources($map,\&scantron_filter,1,0);
    my $bubbles_per_row = &bubblesheet_bubbles_per_row($scantron_config);

    &Apache::lonxml::clear_problem_counter();

    my $uname       = $env{'user.name'};
    my $udom        = $env{'user.domain'};
    my $cid         = $env{'request.course.id'};
    my $total_lines = 0;
    %bubble_lines_per_response = ();
    %first_bubble_line         = ();
    %subdivided_bubble_lines   = ();
    %responsetype_per_response = ();
    %masterseq_id_responsenum  = ();

    my $response_number = 0;
    my $bubble_line     = 0;
    foreach my $resource (@resources) {
        my $resid = $resource->id();
        my ($analysis,$parts) = &scantron_partids_tograde($resource,$cid,$uname,
                                                          $udom,undef,$bubbles_per_row);
        if ((ref($analysis) eq 'HASH') && (ref($parts) eq 'ARRAY')) {
	    foreach my $part_id (@{$parts}) {
                my $lines;

	        # TODO - make this a persistent hash not an array.

                # optionresponse, matchresponse and rankresponse type items 
                # render as separate sub-questions in exam mode.
                if (($analysis->{$part_id.'.type'} eq 'optionresponse') ||
                    ($analysis->{$part_id.'.type'} eq 'matchresponse') ||
                    ($analysis->{$part_id.'.type'} eq 'rankresponse')) {
                    my ($numbub,$numshown);
                    if ($analysis->{$part_id.'.type'} eq 'optionresponse') {
                        if (ref($analysis->{$part_id.'.options'}) eq 'ARRAY') {
                            $numbub = scalar(@{$analysis->{$part_id.'.options'}});
                        }
                    } elsif ($analysis->{$part_id.'.type'} eq 'matchresponse') {
                        if (ref($analysis->{$part_id.'.items'}) eq 'ARRAY') {
                            $numbub = scalar(@{$analysis->{$part_id.'.items'}});
                        }
                    } elsif ($analysis->{$part_id.'.type'} eq 'rankresponse') {
                        if (ref($analysis->{$part_id.'.foils'}) eq 'ARRAY') {
                            $numbub = scalar(@{$analysis->{$part_id.'.foils'}});
                        }
                    }
                    if (ref($analysis->{$part_id.'.shown'}) eq 'ARRAY') {
                        $numshown = scalar(@{$analysis->{$part_id.'.shown'}});
                    }
                    my $bubbles_per_row =
                        &bubblesheet_bubbles_per_row($scantron_config);
                    my $inner_bubble_lines = int($numbub/$bubbles_per_row);
                    if (($numbub % $bubbles_per_row) != 0) {
                        $inner_bubble_lines++;
                    }
                    for (my $i=0; $i<$numshown; $i++) {
                        $subdivided_bubble_lines{$response_number} .= 
                            $inner_bubble_lines.',';
                    }
                    $subdivided_bubble_lines{$response_number} =~ s/,$//;
                    $lines = $numshown * $inner_bubble_lines;
                } else {
                    $lines = $analysis->{"$part_id.bubble_lines"};
                }

                $first_bubble_line{$response_number} = $bubble_line;
	        $bubble_lines_per_response{$response_number} = $lines;
                $responsetype_per_response{$response_number} = 
                    $analysis->{$part_id.'.type'};
                $masterseq_id_responsenum{$resid.'_'.$part_id} = $response_number;
	        $response_number++;

	        $bubble_line +=  $lines;
	        $total_lines +=  $lines;
	    }
        }
    }
    &Apache::lonnet::delenv('scantron.');

    &save_bubble_lines();
    $env{'form.scantron_maxbubble'} =
	$total_lines;
    return $env{'form.scantron_maxbubble'};
}

sub bubblesheet_bubbles_per_row {
    my ($scantron_config) = @_;
    my $bubbles_per_row;
    if (ref($scantron_config) eq 'HASH') {
        $bubbles_per_row = $scantron_config->{'BubblesPerRow'};
    }
    if ((!$bubbles_per_row) || ($bubbles_per_row < 1)) {
        $bubbles_per_row = 10;
    }
    return $bubbles_per_row;
}

sub scantron_validate_missingbubbles {
    my ($r,$currentphase) = @_;
    #get student info
    my $classlist=&Apache::loncoursedata::get_classlist();
    my %idmap=&username_to_idmap($classlist);
    my (undef,undef,$sequence)=
        &Apache::lonnet::decode_symb($env{'form.selectpage'});

    #get scantron line setup
    my %scantron_config=&Apache::lonnet::get_scantron_config($env{'form.scantron_format'});
    my ($scanlines,$scan_data)=&scantron_getfile();

    my $navmap = Apache::lonnavmaps::navmap->new();
    unless (ref($navmap)) {
        $r->print(&navmap_errormsg());
        return(1,$currentphase);
    }

    my $map=$navmap->getResourceByUrl($sequence);
    my @resources=$navmap->retrieveResources($map,\&scantron_filter,1,0);
    my ($randomorder,$randompick,@master_seq,%symb_to_resource,%grader_partids_by_symb,
        %grader_randomlists_by_symb,%orderedforcode,%respnumlookup,%startline);
    my $bubbles_per_row = &bubblesheet_bubbles_per_row(\%scantron_config);

    my $nav_error;
    if (ref($map)) {
        $randomorder = $map->randomorder();
        $randompick = $map->randompick();
        unless ($randomorder || $randompick) {
            foreach my $res ($navmap->retrieveResources($map,sub { $_[0]->is_map() },1,0,1)) {
                if ($res->randomorder()) {
                    $randomorder = 1;
                }
                if ($res->randompick()) {
                    $randompick = 1;
                }
                last if ($randomorder || $randompick);
            }
        }
        if ($randomorder || $randompick) {
            $nav_error = &get_master_seq(\@resources,\@master_seq,\%symb_to_resource);
            if ($nav_error) {
                $r->print(&navmap_errormsg());
                return(1,$currentphase);
            }
            &graders_resources_pass(\@resources,\%grader_partids_by_symb,
                                    \%grader_randomlists_by_symb,$bubbles_per_row);
        }
    } else {
        $r->print(&navmap_errormsg());
        return(1,$currentphase);
    }


    my $max_bubble=&scantron_get_maxbubble(\$nav_error,\%scantron_config);
    if ($nav_error) {
        $r->print(&navmap_errormsg());
        return(1,$currentphase);
    }

    if (!$max_bubble) { $max_bubble=2**31; }
    for (my $i=0;$i<=$scanlines->{'count'};$i++) {
	my $line=&scantron_get_line($scanlines,$scan_data,$i);
	if ($line=~/^[\s\cz]*$/) { next; }
        my $scan_record =
            &scantron_parse_scanline($line,$i,\%scantron_config,$scan_data,undef,\%idmap,
                                     $randomorder,$randompick,$sequence,\@master_seq,
                                     \%symb_to_resource,\%grader_partids_by_symb,
                                     \%orderedforcode,\%respnumlookup,\%startline);
	if (!defined($$scan_record{'scantron.missingerror'})) { next; }
	my @to_correct;
	
	# Probably here's where the error is...

	foreach my $missing (@{$$scan_record{'scantron.missingerror'}}) {
            my $lastbubble;
            if ($missing =~ /^(\d+)\.(\d+)$/) {
                my $question = $1;
                my $subquestion = $2;
                my ($first,$responsenum);
                if ($randomorder || $randompick) {
                    $responsenum = $respnumlookup{$question-1};
                    $first = $startline{$question-1};
                } else {
                    $responsenum = $question-1;
                    $first = $first_bubble_line{$responsenum};
                }
                if (!defined($first)) { next; }
                my @subans = split(/,/,$subdivided_bubble_lines{$responsenum});
                my $subcount = 1;
                while ($subcount<$subquestion) {
                    $first += $subans[$subcount-1];
                    $subcount ++;
                }
                my $count = $subans[$subquestion-1];
                $lastbubble = $first + $count;
            } else {
                my ($first,$responsenum);
                if ($randomorder || $randompick) {
                    $responsenum = $respnumlookup{$missing-1};
                    $first = $startline{$missing-1};
                } else {
                    $responsenum = $missing-1;
                    $first = $first_bubble_line{$responsenum};
                }
                if (!defined($first)) { next; }
                $lastbubble = $first + $bubble_lines_per_response{$responsenum};
            }
            if ($lastbubble > $max_bubble) { next; }
	    push(@to_correct,$missing);
	}
	if (@to_correct) {
	    &scantron_get_correction($r,$i,$scan_record,\%scantron_config,
				     $line,'missingbubble',\@to_correct,
                                     $randomorder,$randompick,\%respnumlookup,
                                     \%startline);
	    return (1,$currentphase);
	}

    }
    return (0,$currentphase+1);
}

sub hand_bubble_option {
    my (undef, undef, $sequence) =
        &Apache::lonnet::decode_symb($env{'form.selectpage'});
    return if ($sequence eq '');
    my $navmap = Apache::lonnavmaps::navmap->new();
    unless (ref($navmap)) {
        return;
    }
    my $needs_hand_bubbles;
    my $map=$navmap->getResourceByUrl($sequence);
    my @resources=$navmap->retrieveResources($map,\&scantron_filter,1,0);
    foreach my $res (@resources) {
        if (ref($res)) {
            if ($res->is_problem()) {
                my $partlist = $res->parts();
                foreach my $part (@{ $partlist }) {
                    my @types = $res->responseType($part);
                    if (grep(/^(chem|essay|image|formula|math|string|functionplot)$/,@types)) {
                        $needs_hand_bubbles = 1;
                        last;
                    }
                }
            }
        }
    }
    if ($needs_hand_bubbles) {
        my %scantron_config=&Apache::lonnet::get_scantron_config($env{'form.scantron_format'});
        my $bubbles_per_row = &bubblesheet_bubbles_per_row(\%scantron_config);
        return &mt('The sequence to be graded contains response types which are handgraded.').'<p>'.
               &mt('If you have already graded these by bubbling sheets to indicate points awarded, [_1]what point value is assigned to a filled last bubble in each row?','<br />').
               '<label><input type="radio" name="scantron_lastbubblepoints" value="'.$bubbles_per_row.'" checked="checked" />'.&mt('[quant,_1,point]',$bubbles_per_row).'</label>&nbsp;'.&mt('or').'&nbsp;'.
               '<label><input type="radio" name="scantron_lastbubblepoints" value="0" />'.&mt('0 points').'</label></p>';
    }
    return;
}

sub scantron_process_students {
    my ($r,$symb) = @_;

    my (undef,undef,$sequence)=&Apache::lonnet::decode_symb($env{'form.selectpage'});
    if (!$symb) {
	return '';
    }
    my $default_form_data=&defaultFormData($symb);

    my %scantron_config=&Apache::lonnet::get_scantron_config($env{'form.scantron_format'});
    my $bubbles_per_row = &bubblesheet_bubbles_per_row(\%scantron_config);
    my ($scanlines,$scan_data)=&scantron_getfile();
    my $classlist=&Apache::loncoursedata::get_classlist();
    my %idmap=&username_to_idmap($classlist);
    my $navmap=Apache::lonnavmaps::navmap->new();
    unless (ref($navmap)) {
        $r->print(&navmap_errormsg());
        return '';
    }
    my $map=$navmap->getResourceByUrl($sequence);
    my ($randomorder,$randompick,@master_seq,%symb_to_resource,%grader_partids_by_symb,
        %grader_randomlists_by_symb,%symb_for_examcode);
    if (ref($map)) {
        $randomorder = $map->randomorder();
        $randompick = $map->randompick();
        unless ($randomorder || $randompick) {
            foreach my $res ($navmap->retrieveResources($map,sub { $_[0]->is_map() },1,0,1)) {
                if ($res->randomorder()) {
                    $randomorder = 1;
                }
                if ($res->randompick()) {
                    $randompick = 1;
                }
                last if ($randomorder || $randompick);
            }
        }
    } else {
        $r->print(&navmap_errormsg());
        return '';
    }
    my $nav_error;
    my @resources=$navmap->retrieveResources($map,\&scantron_filter,1,0);
    if ($randomorder || $randompick) {
        $nav_error = &get_master_seq(\@resources,\@master_seq,\%symb_to_resource,1,\%symb_for_examcode);
        if ($nav_error) {
            $r->print(&navmap_errormsg());
            return '';
        }
    }
    &graders_resources_pass(\@resources,\%grader_partids_by_symb,
                            \%grader_randomlists_by_symb,$bubbles_per_row);

    my ($uname,$udom);
    my $result= <<SCANTRONFORM;
<form method="post" enctype="multipart/form-data" action="/adm/grades" name="scantronupload">
  <input type="hidden" name="command" value="scantron_configphase" />
  $default_form_data
SCANTRONFORM
    $r->print($result);

    my @delayqueue;
    my (%completedstudents,%scandata);
    
    my $lock=&Apache::lonnet::set_lock(&mt('Grading bubblesheet exam'));
    my $count=&get_todo_count($scanlines,$scan_data);
    my %prog_state=&Apache::lonhtmlcommon::Create_PrgWin($r,$count);
    &Apache::lonhtmlcommon::Update_PrgWin($r,\%prog_state,'Processing first student');
    $r->print('<br />');
    my $start=&Time::HiRes::time();
    my $i=-1;
    my $started;

    &scantron_get_maxbubble(\$nav_error,\%scantron_config); # Need the bubble lines array to parse.
    if ($nav_error) {
        $r->print(&navmap_errormsg());
        return '';
    }

    # If an ssi failed in scantron_get_maxbubble, put an error message out to
    # the user and return.

    if ($ssi_error) {
	$r->print("</form>");
	&ssi_print_error($r);
        &Apache::lonnet::remove_lock($lock);
	return '';		# Dunno why the other returns return '' rather than just returning.
    }

    my %lettdig = &Apache::lonnet::letter_to_digits();
    my $numletts = scalar(keys(%lettdig));
    my %orderedforcode;

    while ($i<$scanlines->{'count'}) {
 	($uname,$udom)=('','');
 	$i++;
 	my $line=&scantron_get_line($scanlines,$scan_data,$i);
 	if ($line=~/^[\s\cz]*$/) { next; }
	if ($started) {
	    &Apache::lonhtmlcommon::Increment_PrgWin($r,\%prog_state,'last student');
	}
	$started=1;
        my %respnumlookup = ();
        my %startline = ();
        my $total;
 	my $scan_record=&scantron_parse_scanline($line,$i,\%scantron_config,
 						 $scan_data,undef,\%idmap,$randomorder,
                                                 $randompick,$sequence,\@master_seq,
                                                 \%symb_to_resource,\%grader_partids_by_symb,
                                                 \%orderedforcode,\%respnumlookup,\%startline,
                                                 \$total);
 	unless ($uname=&scantron_find_student($scan_record,$scan_data,
 					      \%idmap,$i)) {
  	    &scantron_add_delay(\@delayqueue,$line,
 				'Unable to find a student that matches',1);
 	    next;
  	}
 	if (exists $completedstudents{$uname}) {
 	    &scantron_add_delay(\@delayqueue,$line,
 				'Student '.$uname.' has multiple sheets',2);
 	    next;
 	}
        my $usec = $classlist->{$uname}->[&Apache::loncoursedata::CL_SECTION];
        my $user = $uname.':'.$usec;
  	($uname,$udom)=split(/:/,$uname);

        my $scancode;
        if ((exists($scan_record->{'scantron.CODE'})) &&
            (&Apache::lonnet::validCODE($scan_record->{'scantron.CODE'}))) {
            $scancode = $scan_record->{'scantron.CODE'};
        } else {
            $scancode = '';
        }

        my @mapresources = @resources;
        if ($randomorder || $randompick) {
            @mapresources =
                &users_order($user,$scancode,$sequence,\@master_seq,\%symb_to_resource,
                             \%orderedforcode);
        }
        my (%partids_by_symb,$res_error);
        foreach my $resource (@mapresources) {
            my $ressymb;
            if (ref($resource)) {
                $ressymb = $resource->symb();
            } else {
                $res_error = 1;
                last;
            }
            if ((exists($grader_randomlists_by_symb{$ressymb})) ||
                (ref($grader_partids_by_symb{$ressymb}) ne 'ARRAY')) {
                my $currcode;
                if (exists($grader_randomlists_by_symb{$ressymb})) {
                    $currcode = $scancode;
                }
                my ($analysis,$parts) =
                    &scantron_partids_tograde($resource,$env{'request.course.id'},
                                              $uname,$udom,undef,$bubbles_per_row,
                                              $currcode);
                $partids_by_symb{$ressymb} = $parts;
            } else {
                $partids_by_symb{$ressymb} = $grader_partids_by_symb{$ressymb};
            }
        }

        if ($res_error) {
            &scantron_add_delay(\@delayqueue,$line,
                                'An error occurred while grading student '.$uname,2);
            next;
        }

	&Apache::lonxml::clear_problem_counter();
  	&Apache::lonnet::appenv($scan_record);

	if (&scantron_clear_skip($scanlines,$scan_data,$i)) {
	    &scantron_putfile($scanlines,$scan_data);
	}
	
        if (&grade_student_bubbles($r,$uname,$udom,$scan_record,$scancode,
                                   \@mapresources,\%partids_by_symb,
                                   $bubbles_per_row,$randomorder,$randompick,
                                   \%respnumlookup,\%startline) 
            eq 'ssi_error') {
            $ssi_error = 0; # So end of handler error message does not trigger.
            $r->print("</form>");
            &ssi_print_error($r);
            &Apache::lonnet::remove_lock($lock);
            return '';      # Why return ''?  Beats me.
        }

        if (($scancode) && ($randomorder || $randompick)) {
            foreach my $key (keys(%symb_for_examcode)) {
                my $symb_in_map = $symb_for_examcode{$key};
                if ($symb_in_map ne '') {
                    my $parmresult =
                        &Apache::lonparmset::storeparm_by_symb($symb_in_map,
                                                               '0_examcode',2,$scancode,
                                                               'string_examcode',$uname,
                                                               $udom);
                }
            }
        }
	$completedstudents{$uname}={'line'=>$line};
        if ($env{'form.verifyrecord'}) {
            my $lastpos = $env{'form.scantron_maxbubble'}*$scantron_config{'Qlength'};
            if ($randompick) {
                if ($total) {
                    $lastpos = $total*$scantron_config{'Qlength'};
                }
            }

            my $studentdata = substr($line,$scantron_config{'Qstart'}-1,$lastpos);
            chomp($studentdata);
            $studentdata =~ s/\r$//;
            my $studentrecord = '';
            my $counter = -1;
            foreach my $resource (@mapresources) {
                my $ressymb = $resource->symb();
                ($counter,my $recording) =
                    &verify_scantron_grading($resource,$udom,$uname,$env{'request.course.id'},
                                             $counter,$studentdata,$partids_by_symb{$ressymb},
                                             \%scantron_config,\%lettdig,$numletts,$randomorder,
                                             $randompick,\%respnumlookup,\%startline);
                $studentrecord .= $recording;
            }
            if ($studentrecord ne $studentdata) {
                &Apache::lonxml::clear_problem_counter();
                if (&grade_student_bubbles($r,$uname,$udom,$scan_record,$scancode,
                                           \@mapresources,\%partids_by_symb,
                                           $bubbles_per_row,$randomorder,$randompick,
                                           \%respnumlookup,\%startline)
                    eq 'ssi_error') {
                    $ssi_error = 0; # So end of handler error message does not trigger.
                    $r->print("</form>");
                    &ssi_print_error($r);
                    &Apache::lonnet::remove_lock($lock);
                    delete($completedstudents{$uname});
                    return '';
                }
                $counter = -1;
                $studentrecord = '';
                foreach my $resource (@mapresources) {
                    my $ressymb = $resource->symb();
                    ($counter,my $recording) =
                        &verify_scantron_grading($resource,$udom,$uname,$env{'request.course.id'},
                                                 $counter,$studentdata,$partids_by_symb{$ressymb},
                                                 \%scantron_config,\%lettdig,$numletts,
                                                 $randomorder,$randompick,\%respnumlookup,
                                                 \%startline);
                    $studentrecord .= $recording;
                }
                if ($studentrecord ne $studentdata) {
                    $r->print('<p><span class="LC_warning">');
                    if ($scancode eq '') {
                        $r->print(&mt('Mismatch grading bubblesheet for user: [_1] with ID: [_2].',
                                  $uname.':'.$udom,$scan_record->{'scantron.ID'}));
                    } else {
                        $r->print(&mt('Mismatch grading bubblesheet for user: [_1] with ID: [_2] and CODE: [_3].',
                                  $uname.':'.$udom,$scan_record->{'scantron.ID'},$scancode));
                    }
                    $r->print('</span><br />'.&Apache::loncommon::start_data_table()."\n".
                              &Apache::loncommon::start_data_table_header_row()."\n".
                              '<th>'.&mt('Source').'</th><th>'.&mt('Bubbled responses').'</th>'.
                              &Apache::loncommon::end_data_table_header_row()."\n".
                              &Apache::loncommon::start_data_table_row().
                              '<td>'.&mt('Bubblesheet').'</td>'.
                              '<td><span class="LC_nobreak" style="white-space: pre;"><tt>'.$studentdata.'</tt></span></td>'.
                              &Apache::loncommon::end_data_table_row().
                              &Apache::loncommon::start_data_table_row().
                              '<td>'.&mt('Stored submissions').'</td>'.
                              '<td><span class="LC_nobreak" style="white-space: pre;"><tt>'.$studentrecord.'</tt></span></td>'."\n".
                              &Apache::loncommon::end_data_table_row().
                              &Apache::loncommon::end_data_table().'</p>');
                } else {
                    $r->print('<br /><span class="LC_warning">'.
                             &mt('A second grading pass was needed for user: [_1] with ID: [_2], because a mismatch was seen on the first pass.',$uname.':'.$udom,$scan_record->{'scantron.ID'}).'<br />'.
                             &mt("As a consequence, this user's submission history records two tries.").
                                 '</span><br />');
                }
            }
        }
        if (&Apache::loncommon::connection_aborted($r)) { last; }
    } continue {
	&Apache::lonxml::clear_problem_counter();
	&Apache::lonnet::delenv('scantron.');
    }
    &Apache::lonhtmlcommon::Close_PrgWin($r,\%prog_state);
    &Apache::lonnet::remove_lock($lock);
#    my $lasttime = &Time::HiRes::time()-$start;
#    $r->print("<p>took $lasttime</p>");

    $r->print("</form>");
    return '';
}

sub graders_resources_pass {
    my ($resources,$grader_partids_by_symb,$grader_randomlists_by_symb,
        $bubbles_per_row) = @_;
    if ((ref($resources) eq 'ARRAY') && (ref($grader_partids_by_symb)) && 
        (ref($grader_randomlists_by_symb) eq 'HASH')) {
        foreach my $resource (@{$resources}) {
            my $ressymb = $resource->symb();
            my ($analysis,$parts) =
                &scantron_partids_tograde($resource,$env{'request.course.id'},
                                          $env{'user.name'},$env{'user.domain'},
                                          1,$bubbles_per_row);
            $grader_partids_by_symb->{$ressymb} = $parts;
            if (ref($analysis) eq 'HASH') {
                if (ref($analysis->{'parts_withrandomlist'}) eq 'ARRAY') {
                    $grader_randomlists_by_symb->{$ressymb} =
                        $analysis->{'parts_withrandomlist'};
                }
            }
        }
    }
    return;
}

=pod

=item users_order

  Returns array of resources in current map, ordered based on either CODE,
  if this is a CODEd exam, or based on student's identity if this is a
  "NAMEd" exam.

  Should be used when randomorder and/or randompick applied when the 
  corresponding exam was printed, prior to students completing bubblesheets 
  for the version of the exam the student received.

=cut

sub users_order  {
    my ($user,$scancode,$mapurl,$master_seq,$symb_to_resource,$orderedforcode) = @_;
    my @mapresources;
    unless ((ref($master_seq) eq 'ARRAY') && (ref($symb_to_resource) eq 'HASH')) {
        return @mapresources;
    }
    if ($scancode) {
        if ((ref($orderedforcode) eq 'HASH') && (ref($orderedforcode->{$scancode}) eq 'ARRAY')) {
            @mapresources = @{$orderedforcode->{$scancode}};
        } else {
            $env{'form.CODE'} = $scancode;
            my $actual_seq =
                &Apache::lonprintout::master_seq_to_person_seq($mapurl,
                                                               $master_seq,
                                                               $user,$scancode,1);
            if (ref($actual_seq) eq 'ARRAY') {
                @mapresources = map { $symb_to_resource->{$_}; } @{$actual_seq};
                if (ref($orderedforcode) eq 'HASH') {
                    if (@mapresources > 0) {
                        $orderedforcode->{$scancode} = \@mapresources;
                    }
                }
            }
            delete($env{'form.CODE'});
        }
    } else {
        my $actual_seq =
            &Apache::lonprintout::master_seq_to_person_seq($mapurl,
                                                           $master_seq,
                                                           $user,undef,1);
        if (ref($actual_seq) eq 'ARRAY') {
            @mapresources =
                map { $symb_to_resource->{$_}; } @{$actual_seq};
        }
    }
    return @mapresources;
}

sub grade_student_bubbles {
    my ($r,$uname,$udom,$scan_record,$scancode,$resources,$parts,$bubbles_per_row,
        $randomorder,$randompick,$respnumlookup,$startline) = @_;
    my $uselookup = 0;
    if (($randomorder || $randompick) && (ref($respnumlookup) eq 'HASH') &&
        (ref($startline) eq 'HASH')) {
        $uselookup = 1;
    }

    if (ref($resources) eq 'ARRAY') {
        my $count = 0;
        foreach my $resource (@{$resources}) {
            my $ressymb = $resource->symb();
            my %form = ('submitted'      => 'scantron',
                        'grade_target'   => 'grade',
                        'grade_username' => $uname,
                        'grade_domain'   => $udom,
                        'grade_courseid' => $env{'request.course.id'},
                        'grade_symb'     => $ressymb,
                        'CODE'           => $scancode
                       );
            if ($bubbles_per_row ne '') {
                $form{'bubbles_per_row'} = $bubbles_per_row;
            }
            if ($env{'form.scantron_lastbubblepoints'} ne '') {
                $form{'scantron_lastbubblepoints'} = $env{'form.scantron_lastbubblepoints'};
            }
            if (ref($parts) eq 'HASH') {
                if (ref($parts->{$ressymb}) eq 'ARRAY') {
                    foreach my $part (@{$parts->{$ressymb}}) {
                        if ($uselookup) {
                            $form{'scantron_questnum_start.'.$part} = $startline->{$count} + 1;
                        } else {
                            $form{'scantron_questnum_start.'.$part} =
                                1+$env{'form.scantron.first_bubble_line.'.$count};
                        }
                        $count++;
                    }
                }
            }
            my $result=&ssi_with_retries($resource->src(),$ssi_retries,%form);
            return 'ssi_error' if ($ssi_error);
            last if (&Apache::loncommon::connection_aborted($r));
        }
    }
    return;
}

sub scantron_upload_scantron_data {
    my ($r,$symb) = @_;
    my $dom = $env{'request.role.domain'};
    my ($formatoptions,$formattitle,$formatjs) = &scantron_upload_dataformat($dom);
    my $domdesc = &Apache::lonnet::domain($dom,'description');
    $r->print(&Apache::loncommon::coursebrowser_javascript($dom));
    my $select_link=&Apache::loncommon::selectcourse_link('rules','courseid',
							  'domainid',
							  'coursename',$dom);
    my $syllabuslink = '<a href="javascript:ToSyllabus();">'.&mt('Syllabus').'</a>'.
                       ('&nbsp'x2).&mt('(shows course personnel)');
    my $default_form_data=&defaultFormData($symb);
    my $nofile_alert = &mt('Please use the browse button to select a file from your local directory.');
    &js_escape(\$nofile_alert);
    my $nocourseid_alert = &mt("Please use the 'Select Course' link to open a separate window where you can search for a course to which a file can be uploaded.");
    &js_escape(\$nocourseid_alert);
    $r->print(&Apache::lonhtmlcommon::scripttag('
    function checkUpload(formname) {
	if (formname.upfile.value == "") {
	    alert("'.$nofile_alert.'");
	    return false;
	}
        if (formname.courseid.value == "") {
            alert("'.$nocourseid_alert.'");
            return false;
        }
	formname.submit();
    }

    function ToSyllabus() {
        var cdom = '."'$dom'".';
        var cnum = document.rules.courseid.value;
        if (cdom == "" || cdom == null) {
            return;
        }
        if (cnum == "" || cnum == null) {
           return;
        }
        syllwin=window.open("/public/"+cdom+"/"+cnum+"/syllabus","LONCAPASyllabus",
                            "height=350,width=350,scrollbars=yes,menubar=no");
        return;
    }

    '.$formatjs.'
'));
    $r->print('
<h3>'.&mt('Send bubblesheet data to a course').'</h3>

<form enctype="multipart/form-data" action="/adm/grades" name="rules" method="post">
'.$default_form_data.
  &Apache::lonhtmlcommon::start_pick_box().
  &Apache::lonhtmlcommon::row_title(&mt('Course ID')).
  '<input name="courseid" type="text" size="30" />'.$select_link.
  &Apache::lonhtmlcommon::row_closure().
  &Apache::lonhtmlcommon::row_title(&mt('Course Name')).
  '<input name="coursename" type="text" size="30" />'.$syllabuslink.
  &Apache::lonhtmlcommon::row_closure().
  &Apache::lonhtmlcommon::row_title(&mt('Domain')).
  '<input name="domainid" type="hidden" />'.$domdesc.
  &Apache::lonhtmlcommon::row_closure());
    if ($formatoptions) {
        $r->print(&Apache::lonhtmlcommon::row_title($formattitle).$formatoptions.
                  &Apache::lonhtmlcommon::row_closure());
    }
    $r->print(
  &Apache::lonhtmlcommon::row_title(&mt('File to upload')).
  '<input type="file" name="upfile" size="50" />'.
  &Apache::lonhtmlcommon::row_closure(1).
  &Apache::lonhtmlcommon::end_pick_box().'<br />

<input name="command" value="scantronupload_save" type="hidden" />
<input type="button" onclick="javascript:checkUpload(this.form);" value="'.&mt('Upload Bubblesheet Data').'" />
</form>
');
    return '';
}

sub scantron_upload_dataformat {
    my ($dom) = @_;
    my ($formatoptions,$formattitle,$formatjs);
    $formatjs = <<'END';
function toggleScantab(form) {
   return;
}
END
    my %domconfig = &Apache::lonnet::get_dom('configuration',['scantron'],$dom);
    if (ref($domconfig{'scantron'}) eq 'HASH') {
        if (ref($domconfig{'scantron'}{'config'}) eq 'HASH') {
            if (keys(%{$domconfig{'scantron'}{'config'}}) > 1) {
                if (($domconfig{'scantron'}{'config'}{'dat'}) &&
                    (ref($domconfig{'scantron'}{'config'}{'csv'}) eq 'HASH')) {
                    if (ref($domconfig{'scantron'}{'config'}{'csv'}{'fields'}) eq 'HASH') {
                        if (keys(%{$domconfig{'scantron'}{'config'}{'csv'}{'fields'}})) {
                            my ($onclick,$formatextra,$singleline);
                            my @lines = &Apache::lonnet::get_scantronformat_file();
                            my $count = 0;
                            foreach my $line (@lines) {
                                next if (($line =~ /^\#/) || ($line eq ''));
                                $singleline = $line;
                                $count ++;
                            }
                            if ($count > 1) {
                                $formatextra = '<div style="display:none" id="bubbletype">'.
                                               '<span class="LC_nobreak">'.
                                               &mt('Bubblesheet type').':&nbsp;'.
                                               &scantron_scantab().'</span></div>';
                                $onclick = ' onclick="toggleScantab(this.form);"';
                                $formatjs = <<"END";
function toggleScantab(form) {
    var divid = 'bubbletype';
    if (document.getElementById(divid)) {
        var radioname = 'fileformat';
        var num = form.elements[radioname].length;
        if (num) {
            for (var i=0; i<num; i++) {
                if (form.elements[radioname][i].checked) {
                    var chosen = form.elements[radioname][i].value;
                    if (chosen == 'dat') {
                        document.getElementById(divid).style.display = 'none';
                    } else if (chosen == 'csv') {
                        document.getElementById(divid).style.display = 'block';
                    }
                }
            }
        }
    }
    return;
}

END
                            } elsif ($count == 1) {
                                my $formatname = (split(/:/,$singleline,2))[0];
                                $formatextra = '<input type="hidden" name="scantron_format" value="'.$formatname.'" />';
                            }
                            $formattitle = &mt('File format');
                            $formatoptions = '<label><input name="fileformat" type="radio" value="dat" checked="checked"'.$onclick.' />'.
                                             &mt('Plain Text (no delimiters)').
                                             '</label>'.('&nbsp;'x2).
                                             '<label><input name="fileformat" type="radio" value="csv"'.$onclick.' />'.
                                             &mt('Comma separated values').'</label>'.$formatextra;
                        }
                    }
                }
            } elsif (keys(%{$domconfig{'scantron'}{'config'}}) == 1) {
                if (ref($domconfig{'scantron'}{'config'}{'csv'}{'fields'}) eq 'HASH') {
                    if (keys(%{$domconfig{'scantron'}{'config'}{'csv'}{'fields'}})) {
                        $formattitle = &mt('Bubblesheet type');
                        $formatoptions = &scantron_scantab();
                    }
                }
            }
        }
    }
    return ($formatoptions,$formattitle,$formatjs);
}

sub scantron_upload_scantron_data_save {
    my ($r,$symb) = @_;
    my $doanotherupload=
	'<br /><form action="/adm/grades" method="post">'."\n".
	'<input type="hidden" name="command" value="scantronupload" />'."\n".
	'<input type="submit" name="submit" value="'.&mt('Do Another Upload').'" />'."\n".
	'</form>'."\n";
    if (!&Apache::lonnet::allowed('usc',$env{'form.domainid'}) &&
	!&Apache::lonnet::allowed('usc',
			    $env{'form.domainid'}.'_'.$env{'form.courseid'})) {
	$r->print(&mt("You are not allowed to upload bubblesheet data to the requested course.")."<br />");
        unless ($symb) {
	    $r->print($doanotherupload);
	}
	return '';
    }
    my %coursedata=&Apache::lonnet::coursedescription($env{'form.domainid'}.'_'.$env{'form.courseid'});
    my $uploadedfile;
    $r->print('<p>'.&mt("Uploading file to [_1]",$coursedata{'description'}).'</p>');
    if (length($env{'form.upfile'}) < 2) {
        $r->print(
            &Apache::lonhtmlcommon::confirm_success(
                &mt('The file: [_1] you attempted to upload contained no information. Please check that you entered the correct filename.',
                        '<span class="LC_filename">'.&HTML::Entities::encode($env{'form.upfile.filename'},'<>&"').'</span>'),1));
    } else {
        my %domconfig = &Apache::lonnet::get_dom('configuration',['scantron'],$env{'form.domainid'});
        my $parser;
        if (ref($domconfig{'scantron'}) eq 'HASH') {
            if (ref($domconfig{'scantron'}{'config'}) eq 'HASH') {
                my $is_csv;
                my @possibles = keys(%{$domconfig{'scantron'}{'config'}});
                if (@possibles > 1) {
                    if ($env{'form.fileformat'} eq 'csv') {
                        if (ref($domconfig{'scantron'}{'config'}{'csv'}) eq 'HASH') {
                            if (ref($domconfig{'scantron'}{'config'}{'csv'}{'fields'}) eq 'HASH') {
                                if (keys(%{$domconfig{'scantron'}{'config'}{'csv'}{'fields'}}) > 1) {
                                    $is_csv = 1;
                                }
                            }
                        }
                    }
                } elsif (@possibles == 1) {
                    if (ref($domconfig{'scantron'}{'config'}{'csv'}) eq 'HASH') {
                        if (ref($domconfig{'scantron'}{'config'}{'csv'}{'fields'}) eq 'HASH') {
                            if (keys(%{$domconfig{'scantron'}{'config'}{'csv'}{'fields'}}) > 1) {
                                $is_csv = 1;
                            }
                        }
                    }
                }
                if ($is_csv) {
                   $parser = $domconfig{'scantron'}{'config'}{'csv'};
                }
            }
        }
        my $result =
            &Apache::lonnet::userfileupload('upfile','scantron','scantron',$parser,'','',
                                            $env{'form.courseid'},$env{'form.domainid'});
	if ($result =~ m{^/uploaded/}) {
            $r->print(
                &Apache::lonhtmlcommon::confirm_success(&mt('Upload successful')).'<br />'.
                &mt('Uploaded [_1] bytes of data into location: [_2]',
                        (length($env{'form.upfile'})-1),
                        '<span class="LC_filename">'.$result.'</span>'));
            ($uploadedfile) = ($result =~ m{/([^/]+)$});
            $r->print(&validate_uploaded_scantron_file($env{'form.domainid'},
                                                       $env{'form.courseid'},$uploadedfile));
	} else {
            $r->print(
                &Apache::lonhtmlcommon::confirm_success(&mt('Upload failed'),1).'<br />'.
                    &mt('An error ([_1]) occurred when attempting to upload the file: [_2]',
                          $result,
			  '<span class="LC_filename">'.&HTML::Entities::encode($env{'form.upfile.filename'},'<>&"').'</span>'));
	}
    }
    if ($symb) {
	$r->print(&scantron_selectphase($r,$uploadedfile,$symb));
    } else {
	$r->print($doanotherupload);
    }
    return '';
}

sub validate_uploaded_scantron_file {
    my ($cdom,$cname,$fname) = @_;
    my $scanlines=&Apache::lonnet::getfile('/uploaded/'.$cdom.'/'.$cname.'/'.$fname);
    my @lines;
    if ($scanlines ne '-1') {
        @lines=split("\n",$scanlines,-1);
    }
    my $output;
    if (@lines) {
        my (%counts,$max_match_format);
        my ($found_match_count,$max_match_count,$max_match_pct) = (0,0,0);
        my $classlist = &Apache::loncoursedata::get_classlist($cdom,$cname);
        my %idmap = &username_to_idmap($classlist);
        foreach my $key (keys(%idmap)) {
            my $lckey = lc($key);
            $idmap{$lckey} = $idmap{$key};
        }
        my %unique_formats;
        my @formatlines = &Apache::lonnet::get_scantronformat_file();
        foreach my $line (@formatlines) {
            next if (($line =~ /^\#/) || ($line eq ''));
            my @config = split(/:/,$line);
            my $idstart = $config[5];
            my $idlength = $config[6];
            if (($idstart ne '') && ($idlength > 0)) {
                if (ref($unique_formats{$idstart.':'.$idlength}) eq 'ARRAY') {
                    push(@{$unique_formats{$idstart.':'.$idlength}},$config[0].':'.$config[1]); 
                } else {
                    $unique_formats{$idstart.':'.$idlength} = [$config[0].':'.$config[1]];
                }
            }
        }
        foreach my $key (keys(%unique_formats)) {
            my ($idstart,$idlength) = split(':',$key);
            %{$counts{$key}} = (
                               'found'   => 0,
                               'total'   => 0,
                              );
            foreach my $line (@lines) {
                next if ($line =~ /^#/);
                next if ($line =~ /^[\s\cz]*$/);
                my $id = substr($line,$idstart-1,$idlength);
                $id = lc($id);
                if (exists($idmap{$id})) {
                    $counts{$key}{'found'} ++;
                }
                $counts{$key}{'total'} ++;
            }
            if ($counts{$key}{'total'}) {
                my $percent_match = (100*$counts{$key}{'found'})/($counts{$key}{'total'});
                if (($max_match_format eq '') || ($percent_match > $max_match_pct)) {
                    $max_match_pct = $percent_match;
                    $max_match_format = $key;
                    $found_match_count = $counts{$key}{'found'};
                    $max_match_count = $counts{$key}{'total'};
                }
            }
        }
        if (ref($unique_formats{$max_match_format}) eq 'ARRAY') {
            my $format_descs;
            my $numwithformat = @{$unique_formats{$max_match_format}};
            for (my $i=0; $i<$numwithformat; $i++) {
                my ($name,$desc) = split(':',$unique_formats{$max_match_format}[$i]);
                if ($i<$numwithformat-2) {
                    $format_descs .= '"<i>'.$desc.'</i>", ';
                } elsif ($i==$numwithformat-2) {
                    $format_descs .= '"<i>'.$desc.'</i>" '.&mt('and').' ';
                } elsif ($i==$numwithformat-1) {
                    $format_descs .= '"<i>'.$desc.'</i>"';
                }
            }
            my $showpct = sprintf("%.0f",$max_match_pct).'%';
            $output .= '<br />';
            if ($found_match_count == $max_match_count) {
                # 100% matching entries
                $output .= &Apache::lonhtmlcommon::confirm_success(
                     &mt('Comparison of student IDs: [_1] matching ([quant,_2,entry,entries])',
                            '<b>'.$showpct.'</b>',$found_match_count)).'<br />'.
                &mt('Comparison of student IDs in the uploaded file with'.
                    ' the course roster found matches for [_1] of the [_2] entries'.
                    ' in the file (for the format defined for [_3]).',
                        '<b>'.$showpct.'</b>','<b>'.$max_match_count.'</b>',$format_descs);
            } else {
                # Not all entries matching? -> Show warning and additional info
                $output .=
                    &Apache::lonhtmlcommon::confirm_success(
                        &mt('Comparison of student IDs: [_1] matching ([_2]/[quant,_3,entry,entries])',
                                '<b>'.$showpct.'</b>',$found_match_count,$max_match_count).'<br />'.
                        &mt('Not all entries could be matched!'),1).'<br />'.
                    &mt('Comparison of student IDs in the uploaded file with'.
                        ' the course roster found matches for [_1] of the [_2] entries'.
                        ' in the file (for the format defined for [_3]).',
                            '<b>'.$showpct.'</b>','<b>'.$max_match_count.'</b>',$format_descs).
                    '<p class="LC_info">'.
                    &mt('A low percentage of matches results from one of the following:').
                    '</p><ul>'.
                    '<li>'.&mt('The file was uploaded to the wrong course.').'</li>'.
                    '<li>'.&mt('The data is not in the format expected for the domain: [_1]',
                               '<i>'.$cdom.'</i>').'</li>'.
                    '<li>'.&mt('Students did not bubble their IDs, or mis-bubbled them').'</li>'.
                    '<li>'.&mt('The course roster is not up to date.').'</li>'.
                    '</ul>';
            }
        }
    } else {
        $output = '<p class="LC_warning">'.&mt('Uploaded file contained no data').'</p>';
    }
    return $output;
}

sub valid_file {
    my ($requested_file)=@_;
    foreach my $filename (sort(&scantron_filenames())) {
	if ($requested_file eq $filename) { return 1; }
    }
    return 0;
}

sub scantron_download_scantron_data {
    my ($r,$symb) = @_;
    my $default_form_data=&defaultFormData($symb);
    my $cname=$env{'course.'.$env{'request.course.id'}.'.num'};
    my $cdom=$env{'course.'.$env{'request.course.id'}.'.domain'};
    my $file=$env{'form.scantron_selectfile'};
    if (! &valid_file($file)) {
	$r->print('
	<p>
	    '.&mt('The requested filename was invalid.').'
        </p>
');
	return;
    }
    my $orig='/uploaded/'.$cdom.'/'.$cname.'/scantron_orig_'.$file;
    my $corrected='/uploaded/'.$cdom.'/'.$cname.'/scantron_corrected_'.$file;
    my $skipped='/uploaded/'.$cdom.'/'.$cname.'/scantron_skipped_'.$file;
    &Apache::lonnet::allowuploaded('/adm/grades',$orig);
    &Apache::lonnet::allowuploaded('/adm/grades',$corrected);
    &Apache::lonnet::allowuploaded('/adm/grades',$skipped);
    $r->print('
    <p>
	'.&mt('[_1]Original[_2] file as uploaded by the bubblesheet scanning office.',
	      '<a href="'.$orig.'">','</a>').'
    </p>
    <p>
	'.&mt('[_1]Corrections[_2], a file of corrected records that were used in grading.',
	      '<a href="'.$corrected.'">','</a>').'
    </p>
    <p>
	'.&mt('[_1]Skipped[_2], a file of records that were skipped.',
	      '<a href="'.$skipped.'">','</a>').'
    </p>
');
    return '';
}

sub checkscantron_results {
    my ($r,$symb) = @_;
    if (!$symb) {return '';}
    my $cid = $env{'request.course.id'};
    my %lettdig = &Apache::lonnet::letter_to_digits();
    my $numletts = scalar(keys(%lettdig));
    my $cnum = $env{'course.'.$cid.'.num'};
    my $cdom = $env{'course.'.$cid.'.domain'};
    my (undef, undef, $sequence) = &Apache::lonnet::decode_symb($env{'form.selectpage'});
    my %record;
    my %scantron_config =
        &Apache::lonnet::get_scantron_config($env{'form.scantron_format'});
    my $bubbles_per_row = &bubblesheet_bubbles_per_row(\%scantron_config);
    my ($scanlines,$scan_data)=&Apache::grades::scantron_getfile();
    my $classlist=&Apache::loncoursedata::get_classlist();
    my %idmap=&Apache::grades::username_to_idmap($classlist);
    my $navmap=Apache::lonnavmaps::navmap->new();
    unless (ref($navmap)) {
        $r->print(&navmap_errormsg());
        return '';
    }
    my $map=$navmap->getResourceByUrl($sequence);
    my ($randomorder,$randompick,@master_seq,%symb_to_resource,%grader_partids_by_symb,
        %grader_randomlists_by_symb,%orderedforcode);
    if (ref($map)) {
        $randomorder=$map->randomorder();
        $randompick=$map->randompick();
        unless ($randomorder || $randompick) {
            foreach my $res ($navmap->retrieveResources($map,sub { $_[0]->is_map() },1,0,1)) {
                if ($res->randomorder()) {
                    $randomorder = 1;
                }
                if ($res->randompick()) {
                    $randompick = 1;
                }
                last if ($randomorder || $randompick);
            }
        }
    }
    my @resources=$navmap->retrieveResources($map,\&scantron_filter,1,0);
    my $nav_error = &get_master_seq(\@resources,\@master_seq,\%symb_to_resource);
    if ($nav_error) {
        $r->print(&navmap_errormsg());
        return '';
    }
    &graders_resources_pass(\@resources,\%grader_partids_by_symb,
                            \%grader_randomlists_by_symb,$bubbles_per_row);
    my ($uname,$udom);
    my (%scandata,%lastname,%bylast);
    $r->print('
<form method="post" enctype="multipart/form-data" action="/adm/grades" name="checkscantron">'."\n");

    my @delayqueue;
    my %completedstudents;

    my $count=&get_todo_count($scanlines,$scan_data);
    my %prog_state=&Apache::lonhtmlcommon::Create_PrgWin($r,$count);
    my ($username,$domain,$started);
    &scantron_get_maxbubble(\$nav_error,\%scantron_config); # Need the bubble lines array to parse.
    if ($nav_error) {
        $r->print(&navmap_errormsg());
        return '';
    }

    &Apache::lonhtmlcommon::Update_PrgWin($r,\%prog_state,
                                          'Processing first student');
    my $start=&Time::HiRes::time();
    my $i=-1;

    while ($i<$scanlines->{'count'}) {
        ($username,$domain,$uname)=('','','');
        $i++;
        my $line=&Apache::grades::scantron_get_line($scanlines,$scan_data,$i);
        if ($line=~/^[\s\cz]*$/) { next; }
        if ($started) {
            &Apache::lonhtmlcommon::Increment_PrgWin($r,\%prog_state,
                                                     'last student');
        }
        $started=1;
        my $scan_record=
            &Apache::grades::scantron_parse_scanline($line,$i,\%scantron_config,
                                                     $scan_data);
        unless ($uname=&scantron_find_student($scan_record,$scan_data,
                                              \%idmap,$i)) {
            &Apache::grades::scantron_add_delay(\@delayqueue,$line,
                                'Unable to find a student that matches',1);
            next;
        }
        if (exists $completedstudents{$uname}) {
            &Apache::grades::scantron_add_delay(\@delayqueue,$line,
                                'Student '.$uname.' has multiple sheets',2);
            next;
        }
        my $pid = $scan_record->{'scantron.ID'};
        $lastname{$pid} = $scan_record->{'scantron.LastName'};
        push(@{$bylast{$lastname{$pid}}},$pid);
        my $usec = $classlist->{$uname}->[&Apache::loncoursedata::CL_SECTION];
        my $user = $uname.':'.$usec;
        ($username,$domain)=split(/:/,$uname);

        my $scancode;
        if ((exists($scan_record->{'scantron.CODE'})) &&
            (&Apache::lonnet::validCODE($scan_record->{'scantron.CODE'}))) {
            $scancode = $scan_record->{'scantron.CODE'};
        } else {
            $scancode = '';
        }

        my @mapresources = @resources;
        my $lastpos = $env{'form.scantron_maxbubble'}*$scantron_config{'Qlength'};
        my %respnumlookup=();
        my %startline=();
        if ($randomorder || $randompick) {
            @mapresources =
                &users_order($user,$scancode,$sequence,\@master_seq,\%symb_to_resource,
                             \%orderedforcode);
            my $total = &get_respnum_lookups($sequence,$scan_data,\%idmap,$line,
                                             $scan_record,\@master_seq,\%symb_to_resource,
                                             \%grader_partids_by_symb,\%orderedforcode,
                                             \%respnumlookup,\%startline);
            if ($randompick && $total) {
                $lastpos = $total*$scantron_config{'Qlength'};
            }
        }
        $scandata{$pid} = substr($line,$scantron_config{'Qstart'}-1,$lastpos);
        chomp($scandata{$pid});
        $scandata{$pid} =~ s/\r$//;

        my $counter = -1;
        foreach my $resource (@mapresources) {
            my $parts;
            my $ressymb = $resource->symb();
            if ((exists($grader_randomlists_by_symb{$ressymb})) ||
                (ref($grader_partids_by_symb{$ressymb}) ne 'ARRAY')) {
                my $currcode;
                if (exists($grader_randomlists_by_symb{$ressymb})) {
                    $currcode = $scancode;
                }
                (my $analysis,$parts) =
                    &scantron_partids_tograde($resource,$env{'request.course.id'},
                                              $username,$domain,undef,
                                              $bubbles_per_row,$currcode);
            } else {
                $parts = $grader_partids_by_symb{$ressymb};
            }
            ($counter,my $recording) =
                &verify_scantron_grading($resource,$domain,$username,$cid,$counter,
                                         $scandata{$pid},$parts,
                                         \%scantron_config,\%lettdig,$numletts,
                                         $randomorder,$randompick,
                                         \%respnumlookup,\%startline);
            $record{$pid} .= $recording;
        }
    }
    &Apache::lonhtmlcommon::Close_PrgWin($r,\%prog_state);
    $r->print('<br />');
    my ($okstudents,$badstudents,$numstudents,$passed,$failed);
    $passed = 0;
    $failed = 0;
    $numstudents = 0;
    foreach my $last (sort(keys(%bylast))) {
        if (ref($bylast{$last}) eq 'ARRAY') {
            foreach my $pid (sort(@{$bylast{$last}})) {
                my $showscandata = $scandata{$pid};
                my $showrecord = $record{$pid};
                $showscandata =~ s/\s/&nbsp;/g;
                $showrecord =~ s/\s/&nbsp;/g;
                if ($scandata{$pid} eq $record{$pid}) {
                    my $css_class = ($passed % 2)?'LC_odd_row':'LC_even_row';
                    $okstudents .= '<tr class="'.$css_class.'">'.
'<td>'.&mt('Bubblesheet').'</td><td>'.$showscandata.'</td><td rowspan="2">'.$last.'</td><td rowspan="2">'.$pid.'</td>'."\n".
'</tr>'."\n".
'<tr class="'.$css_class.'">'."\n".
'<td>'.&mt('Submissions').'</td><td>'.$showrecord.'</td></tr>'."\n";
                    $passed ++;
                } else {
                    my $css_class = ($failed % 2)?'LC_odd_row':'LC_even_row';
                    $badstudents .= '<tr class="'.$css_class.'"><td>'.&mt('Bubblesheet').'</td><td><span class="LC_nobreak">'.$scandata{$pid}.'</span></td><td rowspan="2">'.$last.'</td><td rowspan="2">'.$pid.'</td>'."\n".
'</tr>'."\n".
'<tr class="'.$css_class.'">'."\n".
'<td>'.&mt('Submissions').'</td><td><span class="LC_nobreak">'.$record{$pid}.'</span></td>'."\n".
'</tr>'."\n";
                    $failed ++;
                }
                $numstudents ++;
            }
        }
    }
    $r->print('<p>'.
              &mt('Comparison of bubblesheet data (including corrections) with corresponding submission records (most recent submission) for [_1][quant,_2,student][_3] ([quant,_4,bubblesheet line] per student).',
                  '<b>',
                  $numstudents,
                  '</b>',
                  $env{'form.scantron_maxbubble'}).
              '</p>'
    );
    $r->print('<p>'
             .&mt('Exact matches for [_1][quant,_2,student][_3].','<b>',$passed,'</b>')
             .'<br />'
             .&mt('Discrepancies detected for [_1][quant,_2,student][_3].','<b>',$failed,'</b>')
             .'</p>');
    if ($passed) {
        $r->print(&mt('Students with exact correspondence between bubblesheet data and submissions are as follows:').'<br /><br />');
        $r->print(&Apache::loncommon::start_data_table()."\n".
                 &Apache::loncommon::start_data_table_header_row()."\n".
                 '<th>'.&mt('Source').'</th><th>'.&mt('Bubble records').'</th><th>'.&mt('Name').'</th><th>'.&mt('ID').'</th>'.
                 &Apache::loncommon::end_data_table_header_row()."\n".
                 $okstudents."\n".
                 &Apache::loncommon::end_data_table().'<br />');
    }
    if ($failed) {
        $r->print(&mt('Students with differences between bubblesheet data and submissions are as follows:').'<br /><br />');
        $r->print(&Apache::loncommon::start_data_table()."\n".
                 &Apache::loncommon::start_data_table_header_row()."\n".
                 '<th>'.&mt('Source').'</th><th>'.&mt('Bubble records').'</th><th>'.&mt('Name').'</th><th>'.&mt('ID').'</th>'.
                 &Apache::loncommon::end_data_table_header_row()."\n".
                 $badstudents."\n".
                 &Apache::loncommon::end_data_table()).'<br />'.
                 &mt('Differences can occur if submissions were modified using manual grading after a bubblesheet grading pass.').'<br />'.&mt('If unexpected discrepancies were detected, it is recommended that you inspect the original bubblesheets.');  
    }
    $r->print('</form><br />');
    return;
}

sub verify_scantron_grading {
    my ($resource,$domain,$username,$cid,$counter,$scandata,$partids,
        $scantron_config,$lettdig,$numletts,$randomorder,$randompick,
        $respnumlookup,$startline) = @_;
    my ($record,%expected,%startpos);
    return ($counter,$record) if (!ref($resource));
    return ($counter,$record) if (!$resource->is_problem());
    my $symb = $resource->symb();
    return ($counter,$record) if (ref($partids) ne 'ARRAY');
    foreach my $part_id (@{$partids}) {
        $counter ++;
        $expected{$part_id} = 0;
        my $respnum = $counter;
        if ($randomorder || $randompick) {
            $respnum = $respnumlookup->{$counter};
            $startpos{$part_id} = $startline->{$counter} + 1;
        } else {
            $startpos{$part_id} = $env{"form.scantron.first_bubble_line.$counter"};
        }
        if ($env{"form.scantron.sub_bubblelines.$respnum"}) {
            my @sub_lines = split(/,/,$env{"form.scantron.sub_bubblelines.$respnum"});
            foreach my $item (@sub_lines) {
                $expected{$part_id} += $item;
            }
        } else {
            $expected{$part_id} = $env{"form.scantron.bubblelines.$respnum"};
        }
    }
    if ($symb) {
        my %recorded;
        my (%returnhash) = &Apache::lonnet::restore($symb,$cid,$domain,$username);
        if ($returnhash{'version'}) {
            my %lasthash=();
            my $version;
            for ($version=1;$version<=$returnhash{'version'};$version++) {
                foreach my $key (sort(split(/\:/,$returnhash{$version.':keys'}))) {
                    $lasthash{$key}=$returnhash{$version.':'.$key};
                }
            }
            foreach my $key (keys(%lasthash)) {
                if ($key =~ /\.scantron$/) {
                    my $value = &unescape($lasthash{$key});
                    my ($part_id) = ($key =~ /^resource\.(.+)\.scantron$/);
                    if ($value eq '') {
                        for (my $i=0; $i<$expected{$part_id}; $i++) {
                            for (my $j=0; $j<$scantron_config->{'length'}; $j++) {
                                $recorded{$part_id} .= $scantron_config->{'Qoff'};
                            }
                        }
                    } else {
                        my @tocheck;
                        my @items = split(//,$value);
                        if (($scantron_config->{'Qon'} eq 'letter') ||
                            ($scantron_config->{'Qon'} eq 'number')) {
                            if (@items < $expected{$part_id}) {
                                my $fragment = substr($scandata,$startpos{$part_id},$expected{$part_id});
                                my @singles = split(//,$fragment);
                                foreach my $pos (@singles) {
                                    if ($pos eq ' ') {
                                        push(@tocheck,$pos);
                                    } else {
                                        my $next = shift(@items);
                                        push(@tocheck,$next);
                                    }
                                }
                            } else {
                                @tocheck = @items;
                            }
                            foreach my $letter (@tocheck) {
                                if ($scantron_config->{'Qon'} eq 'letter') {
                                    if ($letter !~ /^[A-J]$/) {
                                        $letter = $scantron_config->{'Qoff'};
                                    }
                                    $recorded{$part_id} .= $letter;
                                } elsif ($scantron_config->{'Qon'} eq 'number') {
                                    my $digit;
                                    if ($letter !~ /^[A-J]$/) {
                                        $digit = $scantron_config->{'Qoff'};
                                    } else {
                                        $digit = $lettdig->{$letter};
                                    }
                                    $recorded{$part_id} .= $digit;
                                }
                            }
                        } else {
                            @tocheck = @items;
                            for (my $i=0; $i<$expected{$part_id}; $i++) {
                                my $curr_sub = shift(@tocheck);
                                my $digit;
                                if ($curr_sub =~ /^[A-J]$/) {
                                    $digit = $lettdig->{$curr_sub}-1;
                                }
                                if ($curr_sub eq 'J') {
                                    $digit += scalar($numletts);
                                }
                                for (my $j=0; $j<$scantron_config->{'Qlength'}; $j++) {
                                    if ($j == $digit) {
                                        $recorded{$part_id} .= $scantron_config->{'Qon'};
                                    } else {
                                        $recorded{$part_id} .= $scantron_config->{'Qoff'};
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        foreach my $part_id (@{$partids}) {
            if ($recorded{$part_id} eq '') {
                for (my $i=0; $i<$expected{$part_id}; $i++) {
                    for (my $j=0; $j<$scantron_config->{'Qlength'}; $j++) {
                        $recorded{$part_id} .= $scantron_config->{'Qoff'};
                    }
                }
            }
            $record .= $recorded{$part_id};
        }
    }
    return ($counter,$record);
}

#-------- end of section for handling grading scantron forms -------
#
#-------------------------------------------------------------------

#-------------------------- Menu interface -------------------------
#
#--- Href with symb and command ---

sub href_symb_cmd {
    my ($symb,$cmd)=@_;
    return '/adm/grades?symb='.&HTML::Entities::encode(&Apache::lonenc::check_encrypt($symb),'<>&"').'&amp;command='.$cmd;
}

sub grading_menu {
    my ($request,$symb) = @_;
    if (!$symb) {return '';}

    my %fields = ('symb'=>&Apache::lonenc::check_encrypt($symb),
                  'command'=>'individual');

    my $url1a = &Apache::lonhtmlcommon::build_url('grades/',\%fields);

    $fields{'command'}='ungraded';
    my $url1b=&Apache::lonhtmlcommon::build_url('grades/',\%fields);

    $fields{'command'}='table';
    my $url1c=&Apache::lonhtmlcommon::build_url('grades/',\%fields);

    $fields{'command'}='all_for_one';
    my $url1d=&Apache::lonhtmlcommon::build_url('grades/',\%fields);

    $fields{'command'}='downloadfilesselect';
    my $url1e=&Apache::lonhtmlcommon::build_url('grades/',\%fields);
    
    $fields{'command'} = 'csvform';
    my $url2 = &Apache::lonhtmlcommon::build_url('grades/',\%fields);
    
    $fields{'command'} = 'processclicker';
    my $url3 = &Apache::lonhtmlcommon::build_url('grades/',\%fields);
    
    $fields{'command'} = 'scantron_selectphase';
    my $url4 = &Apache::lonhtmlcommon::build_url('grades/',\%fields);

    $fields{'command'} = 'initialverifyreceipt';
    my $url5 = &Apache::lonhtmlcommon::build_url('grades/',\%fields);

    my %permissions;
    if ($perm{'mgr'}) {
        $permissions{'either'} = 'F';
        $permissions{'mgr'} = 'F';
    }
    if ($perm{'vgr'}) {
        $permissions{'either'} = 'F';
        $permissions{'vgr'} = 'F';
    }

    my @menu = ({	categorytitle=>'Hand Grading',
            items =>[
                        {       linktext => 'Select individual students to grade',
                                url => $url1a,
                                permission => $permissions{'either'},
                                icon => 'grade_students.png',
                                linktitle => 'Grade current resource for a selection of students.'
                        },
                        {       linktext => 'Grade ungraded submissions',
                                url => $url1b,
                                permission => $permissions{'either'},
                                icon => 'ungrade_sub.png',
                                linktitle => 'Grade all submissions that have not been graded yet.'
                        },

                        {       linktext => 'Grading table',
                                url => $url1c,
                                permission => $permissions{'either'},
                                icon => 'grading_table.png',
                                linktitle => 'Grade current resource for all students.'
                        },
                        {       linktext => 'Grade page/folder for one student',
                                url => $url1d,
                                permission => $permissions{'either'},
                                icon => 'grade_PageFolder.png',
                                linktitle => 'Grade all resources in current page/sequence/folder for one student.'
                        },
                        {       linktext => 'Download submitted files',
                                url => $url1e,
                                permission => $permissions{'either'},
                                icon => 'download_sub.png',
                                linktitle => 'Download all files submitted by students.'
                        }]},
                         { categorytitle=>'Automated Grading',
               items =>[

                	    {	linktext => 'Upload Scores',
                    		url => $url2,
                    		permission => $permissions{'mgr'},
                    		icon => 'uploadscores.png',
                    		linktitle => 'Specify a file containing the class scores for current resource.'
                	    },
                	    {	linktext => 'Process Clicker',
                    		url => $url3,
                    		permission => $permissions{'mgr'},
                    		icon => 'addClickerInfoFile.png',
                    		linktitle => 'Specify a file containing the clicker information for this resource.'
                	    },
                	    {	linktext => 'Grade/Manage/Review Bubblesheets',
                    		url => $url4,
                    		permission => $permissions{'mgr'},
                    		icon => 'bubblesheet.png',
                    		linktitle => 'Grade bubblesheet exams, upload/download bubblesheet data files, and review previously graded bubblesheet exams.'
                	    },
                            {   linktext => 'Verify Receipt Number',
                                url => $url5,
                                permission => $permissions{'either'},
                                icon => 'receipt_number.png',
                                linktitle => 'Verify a system-generated receipt number for correct problem solution.'
                            }

                    ]
            });

    # Create the menu
    my $Str;
    $Str .= '<form method="post" action="" name="gradingMenu">';
    $Str .= '<input type="hidden" name="command" value="" />'.
    	'<input type="hidden" name="symb"        value="'.&Apache::lonenc::check_encrypt($symb).'" />'."\n";

    $Str .= &Apache::lonhtmlcommon::generate_menu(@menu);
    return $Str;    
}

sub ungraded {
    my ($request)=@_;
    &submit_options($request);
}

sub submit_options_sequence {
    my ($request,$symb) = @_;
    if (!$symb) {return '';}
    &commonJSfunctions($request);
    my $result;

    $result.='<form action="/adm/grades" method="post" name="gradingMenu">'."\n".
        '<input type="hidden" name="symb"        value="'.&Apache::lonenc::check_encrypt($symb).'" />'."\n";
    $result.=&selectfield(0).
            '<input type="hidden" name="command" value="pickStudentPage" />
            <div>
              <input type="submit" value="'.&mt('Next').' &rarr;" />
            </div>
        </div>
  </form>';
    return $result;
}

sub submit_options_table {
    my ($request,$symb) = @_;
    if (!$symb) {return '';}
    &commonJSfunctions($request);
    my $result;

    $result.='<form action="/adm/grades" method="post" name="gradingMenu">'."\n".
        '<input type="hidden" name="symb"        value="'.&Apache::lonenc::check_encrypt($symb).'" />'."\n";

    $result.=&selectfield(1).
            '<input type="hidden" name="command" value="viewgrades" />
            <div>
              <input type="submit" value="'.&mt('Next').' &rarr;" />
            </div>
        </div>
  </form>';
    return $result;
}

sub submit_options_download {
    my ($request,$symb) = @_;
    if (!$symb) {return '';}

    my $res_error;
    my ($partlist,$handgrade,$responseType,$numresp,$numessay,$numdropbox) =
        &response_type($symb,\$res_error);
    if ($res_error) {
        $request->print(&mt('An error occurred retrieving response types'));
        return;
    }
    unless ($numessay) {
        $request->print(&mt('No essayresponse items found'));
        return;
    }
    my $table;
    if (ref($partlist) eq 'ARRAY') {
        if (scalar(@$partlist) > 1 ) {
            $table = &showResourceInfo($symb,$partlist,$responseType,'gradingMenu',1,1);
        }
    }

    &commonJSfunctions($request);

    my $result='<form action="/adm/grades" method="post" name="gradingMenu">'."\n".
        $table."\n".
        '<input type="hidden" name="symb" value="'.&Apache::lonenc::check_encrypt($symb).'" />'."\n";
    $result.='
<h2>
  '.&mt('Select Students for whom to Download Submitted Files').'
</h2>'.&selectfield(1).'
                <input type="hidden" name="command" value="downloadfileslink" />
              <input type="submit" value="'.&mt('Next').' &rarr;" />
            </div>
          </div>


  </form>';
    return $result;
}

#--- Displays the submissions first page -------
sub submit_options {
    my ($request,$symb) = @_;
    if (!$symb) {return '';}

    &commonJSfunctions($request);
    my $result;

    $result.='<form action="/adm/grades" method="post" name="gradingMenu">'."\n".
	'<input type="hidden" name="symb"        value="'.&Apache::lonenc::check_encrypt($symb).'" />'."\n";
    $result.=&selectfield(1).'
                <input type="hidden" name="command" value="submission" />
              <input type="submit" value="'.&mt('Next').' &rarr;" />
            </div>
          </div>
  </form>';
    return $result;
}

sub selectfield {
   my ($full)=@_;
   my %options =
       (&substatus_options,
        'select_form_order' => ['yes','queued','graded','incorrect','all']);

  #
  # PrepareClasslist() needs to be called to avoid getting a sections list
  # for a different course from the @Sections global in lonstatistics.pm,
  # populated by an earlier request.
  #
   &Apache::lonstatistics::PrepareClasslist();

   my $result='<div class="LC_columnSection">

    <fieldset>
      <legend>
       '.&mt('Sections').'
      </legend>
      '.&Apache::lonstatistics::SectionSelect('section','multiple',5).'
    </fieldset>

    <fieldset>
      <legend>
        '.&mt('Groups').'
      </legend>
      '.&Apache::lonstatistics::GroupSelect('group','multiple',5).'
    </fieldset>
 
    <fieldset>
      <legend>
        '.&mt('Access Status').'
      </legend>
      '.&Apache::lonhtmlcommon::StatusOptions(undef,undef,5,undef,'mult').'
    </fieldset>';
    if ($full) {
        $result.='
    <fieldset>
      <legend>
        '.&mt('Submission Status').'
      </legend>'.
       &Apache::loncommon::select_form('all','submitonly',\%options).
   '</fieldset>';
    }
    $result.='</div><br />';
    return $result;
}

sub substatus_options {
    return &Apache::lonlocal::texthash(
                                      'yes'       => 'with submissions',
                                      'queued'    => 'in grading queue',
                                      'graded'    => 'with ungraded submissions',
                                      'incorrect' => 'with incorrect submissions',
                                      'all'       => 'with any status',
                                      );
}

sub transtatus_options {
    return &Apache::lonlocal::texthash(
                                       'yes'       => 'with score transactions',
                                       'incorrect' => 'with less than full credit',
                                       'all'       => 'with any status',
                                      );
}

sub reset_perm {
    undef(%perm);
}

sub init_perm {
    &reset_perm();
    foreach my $test_perm ('vgr','mgr','opa') {

	my $scope = $env{'request.course.id'};
	if (!($perm{$test_perm}=&Apache::lonnet::allowed($test_perm,$scope))) {

	    $scope .= '/'.$env{'request.course.sec'};
	    if ( $perm{$test_perm}=
		 &Apache::lonnet::allowed($test_perm,$scope)) {
		$perm{$test_perm.'_section'}=$env{'request.course.sec'};
	    } else {
		delete($perm{$test_perm});
	    }
	}
    }
}

sub init_old_essays {
    my ($symb,$apath,$adom,$aname) = @_;
    if ($symb ne '') {
        my %essays = &Apache::lonnet::dump('nohist_essay_'.$apath,$adom,$aname);
        if (keys(%essays) > 0) {
            $old_essays{$symb} = \%essays;
        }
    }
    return;
}

sub reset_old_essays {
    undef(%old_essays);
}

sub gather_clicker_ids {
    my %clicker_ids;

    my $classlist = &Apache::loncoursedata::get_classlist();

    # Set up a couple variables.
    my $username_idx = &Apache::loncoursedata::CL_SNAME();
    my $domain_idx   = &Apache::loncoursedata::CL_SDOM();
    my $status_idx   = &Apache::loncoursedata::CL_STATUS();

    foreach my $student (keys(%$classlist)) {
        if ($classlist->{$student}->[$status_idx] ne 'Active') { next; }
        my $username = $classlist->{$student}->[$username_idx];
        my $domain   = $classlist->{$student}->[$domain_idx];
        my $clickers =
	    (&Apache::lonnet::userenvironment($domain,$username,'clickers'))[1];
        foreach my $id (split(/\,/,$clickers)) {
            $id=~s/^[\#0]+//;
            $id=~s/[\-\:]//g;
            if (exists($clicker_ids{$id})) {
		$clicker_ids{$id}.=','.$username.':'.$domain;
            } else {
		$clicker_ids{$id}=$username.':'.$domain;
            }
        }
    }
    return %clicker_ids;
}

sub gather_adv_clicker_ids {
    my %clicker_ids;
    my $cnum=$env{'course.'.$env{'request.course.id'}.'.num'};
    my $cdom=$env{'course.'.$env{'request.course.id'}.'.domain'};
    my %coursepersonnel=&Apache::lonnet::get_course_adv_roles($cdom.'/'.$cnum);
    foreach my $element (sort(keys(%coursepersonnel))) {
        foreach my $person (split(/\,/,$coursepersonnel{$element})) {
            my ($puname,$pudom)=split(/\:/,$person);
            my $clickers =
		(&Apache::lonnet::userenvironment($pudom,$puname,'clickers'))[1];
            foreach my $id (split(/\,/,$clickers)) {
		$id=~s/^[\#0]+//;
                $id=~s/[\-\:]//g;
		if (exists($clicker_ids{$id})) {
		    $clicker_ids{$id}.=','.$puname.':'.$pudom;
		} else {
		    $clicker_ids{$id}=$puname.':'.$pudom;
		}
            }
        }
    }
    return %clicker_ids;
}

sub clicker_grading_parameters {
    return ('gradingmechanism' => 'scalar',
            'upfiletype' => 'scalar',
            'specificid' => 'scalar',
            'pcorrect' => 'scalar',
            'pincorrect' => 'scalar');
}

sub process_clicker {
    my ($r,$symb)=@_;
    if (!$symb) {return '';}
    my $result=&checkforfile_js();
    $result.=&Apache::loncommon::start_data_table().
             &Apache::loncommon::start_data_table_header_row().
             '<th>'.&mt('Specify a file containing clicker information and set grading options.').'</th>'.
             &Apache::loncommon::end_data_table_header_row().
             &Apache::loncommon::start_data_table_row()."<td>\n";
# Attempt to restore parameters from last session, set defaults if not present
    my %Saveable_Parameters=&clicker_grading_parameters();
    &Apache::loncommon::restore_course_settings('grades_clicker',
                                                 \%Saveable_Parameters);
    if (!$env{'form.pcorrect'}) { $env{'form.pcorrect'}=100; }
    if (!$env{'form.pincorrect'}) { $env{'form.pincorrect'}=100; }
    if (!$env{'form.gradingmechanism'}) { $env{'form.gradingmechanism'}='attendance'; }
    if (!$env{'form.upfiletype'}) { $env{'form.upfiletype'}='iclicker'; }

    my %checked;
    foreach my $gradingmechanism ('attendance','personnel','specific','given') {
       if ($env{'form.gradingmechanism'} eq $gradingmechanism) {
          $checked{$gradingmechanism}=' checked="checked"';
       }
    }

    my $upload=&mt("Evaluate File");
    my $type=&mt("Type");
    my $attendance=&mt("Award points just for participation");
    my $personnel=&mt("Correctness determined from response by course personnel");
    my $specific=&mt("Correctness determined from response with clicker ID(s)"); 
    my $given=&mt("Correctness determined from given list of answers").' '.
              '<font size="-2"><tt>('.&mt("Provide comma-separated list. Use '*' for any answer correct, '-' for skip").')</tt></font>';
    my $pcorrect=&mt("Percentage points for correct solution");
    my $pincorrect=&mt("Percentage points for incorrect solution");
    my $selectform=&Apache::loncommon::select_form($env{'form.upfiletype'},'upfiletype',
                                                   {'iclicker' => 'i>clicker',
                                                    'interwrite' => 'interwrite PRS',
                                                    'turning' => 'Turning Technologies'});
    $symb = &Apache::lonenc::check_encrypt($symb);
    $result.= &Apache::lonhtmlcommon::scripttag(<<ENDUPFORM);
function sanitycheck() {
// Accept only integer percentages
   document.forms.gradesupload.pcorrect.value=Math.round(document.forms.gradesupload.pcorrect.value);
   document.forms.gradesupload.pincorrect.value=Math.round(document.forms.gradesupload.pincorrect.value);
// Find out grading choice
   for (i=0; i<document.forms.gradesupload.gradingmechanism.length; i++) {
      if (document.forms.gradesupload.gradingmechanism[i].checked) {
         gradingchoice=document.forms.gradesupload.gradingmechanism[i].value;
      }
   }
// By default, new choice equals user selection
   newgradingchoice=gradingchoice;
// Not good to give more points for false answers than correct ones
   if (Math.round(document.forms.gradesupload.pcorrect.value)<Math.round(document.forms.gradesupload.pincorrect.value)) {
      document.forms.gradesupload.pcorrect.value=document.forms.gradesupload.pincorrect.value;
   }
// If new choice is attendance only, and old choice was correctness-based, restore defaults
   if ((gradingchoice=='attendance') && (document.forms.gradesupload.waschecked.value!='attendance')) {
      document.forms.gradesupload.pcorrect.value=100;
      document.forms.gradesupload.pincorrect.value=100;
   }
// If the values are different, cannot be attendance only
   if ((Math.round(document.forms.gradesupload.pcorrect.value)!=Math.round(document.forms.gradesupload.pincorrect.value)) &&
       (gradingchoice=='attendance')) {
       newgradingchoice='personnel';
   }
// Change grading choice to new one
   for (i=0; i<document.forms.gradesupload.gradingmechanism.length; i++) {
      if (document.forms.gradesupload.gradingmechanism[i].value==newgradingchoice) {
         document.forms.gradesupload.gradingmechanism[i].checked=true;
      } else {
         document.forms.gradesupload.gradingmechanism[i].checked=false;
      }
   }
// Remember the old state
   document.forms.gradesupload.waschecked.value=newgradingchoice;
}
ENDUPFORM
    $result.= <<ENDUPFORM;
<form method="post" enctype="multipart/form-data" action="/adm/grades" name="gradesupload">
<input type="hidden" name="symb" value="$symb" />
<input type="hidden" name="command" value="processclickerfile" />
<input type="file" name="upfile" size="50" />
<br /><label>$type: $selectform</label>
ENDUPFORM
    $result.='</td>'.&Apache::loncommon::end_data_table_row().
                     &Apache::loncommon::start_data_table_row().'<td>'.(<<ENDGRADINGFORM);
      <label><input type="radio" name="gradingmechanism" value="attendance"$checked{'attendance'} onclick="sanitycheck()" />$attendance </label>
<br /><label><input type="radio" name="gradingmechanism" value="personnel"$checked{'personnel'} onclick="sanitycheck()" />$personnel</label>
<br /><label><input type="radio" name="gradingmechanism" value="specific"$checked{'specific'} onclick="sanitycheck()" />$specific </label>
<input type="text" name="specificid" value="$env{'form.specificid'}" size="20" />
<br /><label><input type="radio" name="gradingmechanism" value="given"$checked{'given'} onclick="sanitycheck()" />$given </label>
<br />&nbsp;&nbsp;&nbsp;
<input type="text" name="givenanswer" size="50" />
<input type="hidden" name="waschecked" value="$env{'form.gradingmechanism'}" />
ENDGRADINGFORM
    $result.='</td>'.&Apache::loncommon::end_data_table_row().
                     &Apache::loncommon::start_data_table_row().'<td>'.(<<ENDPERCFORM);
      <label>$pcorrect: <input type="text" name="pcorrect" size="4" value="$env{'form.pcorrect'}" onchange="sanitycheck()" /></label>
<br /><label>$pincorrect: <input type="text" name="pincorrect" size="4" value="$env{'form.pincorrect'}" onchange="sanitycheck()" /></label>
<br /><input type="button" onclick="javascript:checkUpload(this.form);" value="$upload" />
</form>
ENDPERCFORM
    $result.='</td>'.
             &Apache::loncommon::end_data_table_row().
             &Apache::loncommon::end_data_table();
    return $result;
}

sub process_clicker_file {
    my ($r,$symb) = @_;
    if (!$symb) {return '';}

    my %Saveable_Parameters=&clicker_grading_parameters();
    &Apache::loncommon::store_course_settings('grades_clicker',
                                              \%Saveable_Parameters);
    my $result='';
    if (($env{'form.gradingmechanism'} eq 'specific') && ($env{'form.specificid'}!~/\w/)) {
	$result.='<span class="LC_error">'.&mt('You need to specify a clicker ID for the correct answer').'</span>';
	return $result;
    }
    if (($env{'form.gradingmechanism'} eq 'given') && ($env{'form.givenanswer'}!~/\S/)) {
        $result.='<span class="LC_error">'.&mt('You need to specify the correct answer').'</span>';
        return $result;
    }
    my $foundgiven=0;
    if ($env{'form.gradingmechanism'} eq 'given') {
        $env{'form.givenanswer'}=~s/^\s*//gs;
        $env{'form.givenanswer'}=~s/\s*$//gs;
        $env{'form.givenanswer'}=~s/[^a-zA-Z0-9\.\*\-\+]+/\,/g;
        $env{'form.givenanswer'}=uc($env{'form.givenanswer'});
        my @answers=split(/\,/,$env{'form.givenanswer'});
        $foundgiven=$#answers+1;
    }
    my %clicker_ids=&gather_clicker_ids();
    my %correct_ids;
    if ($env{'form.gradingmechanism'} eq 'personnel') {
	%correct_ids=&gather_adv_clicker_ids();
    }
    if ($env{'form.gradingmechanism'} eq 'specific') {
	foreach my $correct_id (split(/[\s\,]/,$env{'form.specificid'})) {;
	   $correct_id=~tr/a-z/A-Z/;
	   $correct_id=~s/\s//gs;
	   $correct_id=~s/^[\#0]+//;
           $correct_id=~s/[\-\:]//g;
           if ($correct_id) {
	      $correct_ids{$correct_id}='specified';
           }
        }
    }
    if ($env{'form.gradingmechanism'} eq 'attendance') {
	$result.=&mt('Score based on attendance only');
    } elsif ($env{'form.gradingmechanism'} eq 'given') {
        $result.=&mt('Score based on [_1] ([_2] answers)','<tt>'.$env{'form.givenanswer'}.'</tt>',$foundgiven);
    } else {
	my $number=0;
	$result.='<p><b>'.&mt('Correctness determined by the following IDs').'</b>';
	foreach my $id (sort(keys(%correct_ids))) {
	    $result.='<br /><tt>'.$id.'</tt> - ';
	    if ($correct_ids{$id} eq 'specified') {
		$result.=&mt('specified');
	    } else {
		my ($uname,$udom)=split(/\:/,$correct_ids{$id});
		$result.=&Apache::loncommon::plainname($uname,$udom);
	    }
	    $number++;
	}
        $result.="</p>\n";
        if ($number==0) {
            $result .=
                 &Apache::lonhtmlcommon::confirm_success(
                     &mt('No IDs found to determine correct answer'),1);
            return $result;
        }
    }
    if (length($env{'form.upfile'}) < 2) {
        $result .=
            &Apache::lonhtmlcommon::confirm_success(
                &mt('The file: [_1] you attempted to upload contained no information. Please check that you entered the correct filename.',
                        '<span class="LC_filename">'.&HTML::Entities::encode($env{'form.upfile.filename'},'<>&"').'</span>'),1);
        return $result;
    }
    my $mimetype;
    if ($env{'form.upfiletype'} eq 'iclicker') {
        my $mm = new File::MMagic;
        $mimetype = $mm->checktype_contents($env{'form.upfile'});
        unless (($mimetype eq 'text/plain') || ($mimetype eq 'text/html')) {
            $result.= '<p>'.
                &Apache::lonhtmlcommon::confirm_success(
                    &mt('File format is neither csv (iclicker 6) nor xml (iclicker 7)'),1).'</p>';
            return $result;
        }
    } elsif (($env{'form.upfiletype'} ne 'interwrite') && ($env{'form.upfiletype'} ne 'turning')) {
        $result .= '<p>'.
            &Apache::lonhtmlcommon::confirm_success(
                &mt('Invalid clicker type: choose one of: i>clicker, Interwrite PRS, or Turning Technologies.'),1).'</p>';
        return $result;
    }

# Were able to get all the info needed, now analyze the file

    $result.=&Apache::loncommon::studentbrowser_javascript();
    $symb = &Apache::lonenc::check_encrypt($symb);
    $result.=&Apache::loncommon::start_data_table().
             &Apache::loncommon::start_data_table_header_row().
             '<th>'.&mt('Evaluate clicker file').'</th>'.
             &Apache::loncommon::end_data_table_header_row().
             &Apache::loncommon::start_data_table_row().(<<ENDHEADER);
<td>
<form method="post" action="/adm/grades" name="clickeranalysis">
<input type="hidden" name="symb" value="$symb" />
<input type="hidden" name="command" value="assignclickergrades" />
<input type="hidden" name="gradingmechanism" value="$env{'form.gradingmechanism'}" />
<input type="hidden" name="pcorrect" value="$env{'form.pcorrect'}" />
<input type="hidden" name="pincorrect" value="$env{'form.pincorrect'}" />
ENDHEADER
    if ($env{'form.gradingmechanism'} eq 'given') {
       $result.='<input type="hidden" name="correct:given" value="'.$env{'form.givenanswer'}.'" />';
    } 
    my %responses;
    my @questiontitles;
    my $errormsg='';
    my $number=0;
    if ($env{'form.upfiletype'} eq 'iclicker') {
        if ($mimetype eq 'text/plain') {
            ($errormsg,$number)=&iclicker_eval(\@questiontitles,\%responses);
        } elsif ($mimetype eq 'text/html') {
            ($errormsg,$number)=&iclickerxml_eval(\@questiontitles,\%responses);
        }
    } elsif ($env{'form.upfiletype'} eq 'interwrite') {
        ($errormsg,$number)=&interwrite_eval(\@questiontitles,\%responses);
    } elsif ($env{'form.upfiletype'} eq 'turning') {
        ($errormsg,$number)=&turning_eval(\@questiontitles,\%responses);
    }
    $result.='<br />'.&mt('Found [_1] question(s)',$number).'<br />'.
             '<input type="hidden" name="number" value="'.$number.'" />'.
             &mt('Awarding [_1] percent for correct and [_2] percent for incorrect responses',
                 $env{'form.pcorrect'},$env{'form.pincorrect'}).
             '<br />';
    if (($env{'form.gradingmechanism'} eq 'given') && ($number!=$foundgiven)) {
       $result.='<span class="LC_error">'.&mt('Number of given answers does not agree with number of questions in file.').'</span>';
       return $result;
    } 
# Remember Question Titles
# FIXME: Possibly need delimiter other than ":"
    for (my $i=0;$i<$number;$i++) {
        $result.='<input type="hidden" name="question:'.$i.'" value="'.
                 &HTML::Entities::encode($questiontitles[$i],'"&<>').'" />';
    }
    my $correct_count=0;
    my $student_count=0;
    my $unknown_count=0;
# Match answers with usernames
# FIXME: Possibly need delimiter other than ":"
    foreach my $id (keys(%responses)) {
       if ($correct_ids{$id}) {
          $result.="\n".'<input type="hidden" name="correct:'.$correct_count.':'.$correct_ids{$id}.'" value="'.$responses{$id}.'" />';
          $correct_count++;
       } elsif ($clicker_ids{$id}) {
          if ($clicker_ids{$id}=~/\,/) {
# More than one user with the same clicker!
             $result.="</td>".&Apache::loncommon::end_data_table_row().
                           &Apache::loncommon::start_data_table_row()."<td>".
                       &mt('Clicker registered more than once').": <tt>".$id."</tt><br />";
             $result.="\n".'<input type="hidden" name="unknown:'.$id.'" value="'.$responses{$id}.'" />'.
                           "<select name='multi".$id."'>";
             foreach my $reguser (sort(split(/\,/,$clicker_ids{$id}))) {
                 $result.="<option value='".$reguser."'>".&Apache::loncommon::plainname(split(/\:/,$reguser)).' ('.$reguser.')</option>';
             }
             $result.='</select>';
             $unknown_count++;
          } else {
# Good: found one and only one user with the right clicker
             $result.="\n".'<input type="hidden" name="student:'.$clicker_ids{$id}.'" value="'.$responses{$id}.'" />';
             $student_count++;
          }
       } else {
          $result.="</td>".&Apache::loncommon::end_data_table_row().
                           &Apache::loncommon::start_data_table_row()."<td>".
                    &mt('Unregistered Clicker')." <tt>".$id."</tt><br />";
          $result.="\n".'<input type="hidden" name="unknown:'.$id.'" value="'.$responses{$id}.'" />'.
                   "\n".&mt("Username").": <input type='text' name='uname".$id."' />&nbsp;".
                   "\n".&mt("Domain").": ".
                   &Apache::loncommon::select_dom_form($env{'course.'.$env{'request.course.id'}.'.domain'},'udom'.$id).'&nbsp;'.
                   &Apache::loncommon::selectstudent_link('clickeranalysis','uname'.$id,'udom'.$id,'',$id);
          $unknown_count++;
       }
    }
    $result.='<hr />'.
             &mt('Found [_1] registered and [_2] unregistered clickers.',$student_count,$unknown_count);
    if (($env{'form.gradingmechanism'} ne 'attendance') && ($env{'form.gradingmechanism'} ne 'given')) {
       if ($correct_count==0) {
          $errormsg.="Found no correct answers for grading!";
       } elsif ($correct_count>1) {
          $result.='<br /><span class="LC_warning">'.&mt("Found [_1] entries for grading!",$correct_count).'</span>';
       }
    }
    if ($number<1) {
       $errormsg.="Found no questions.";
    }
    if ($errormsg) {
       $result.='<br /><span class="LC_error">'.&mt($errormsg).'</span>';
    } else {
       $result.='<br /><input type="submit" name="finalize" value="'.&mt('Finalize Grading').'" />';
    }
    $result.='</form></td>'.
             &Apache::loncommon::end_data_table_row().
             &Apache::loncommon::end_data_table();
    return $result;
}

sub iclicker_eval {
    my ($questiontitles,$responses)=@_;
    my $number=0;
    my $errormsg='';
    foreach my $line (split(/[\n\r]/,$env{'form.upfile'})) {
        my %components=&Apache::loncommon::record_sep($line);
        my @entries=map {$components{$_}} (sort(keys(%components)));
	if ($entries[0] eq 'Question') {
	    for (my $i=3;$i<$#entries;$i+=6) {
		$$questiontitles[$number]=$entries[$i];
		$number++;
	    }
	}
	if ($entries[0]=~/^\#/) {
	    my $id=$entries[0];
	    my @idresponses;
	    $id=~s/^[\#0]+//;
	    for (my $i=0;$i<$number;$i++) {
		my $idx=3+$i*6;
                $entries[$idx]=~s/[^a-zA-Z0-9\.\*\-\+]+//g;
		push(@idresponses,$entries[$idx]);
	    }
	    $$responses{$id}=join(',',@idresponses);
	}
    }
    return ($errormsg,$number);
}

sub iclickerxml_eval {
    my ($questiontitles,$responses)=@_;
    my $number=0;
    my $errormsg='';
    my @state;
    my %respbyid;
    my $p = HTML::Parser->new
    (
        xml_mode => 1,
        start_h =>
            [sub {
                 my ($tagname,$attr) = @_;
                 push(@state,$tagname);
                 if ("@state" eq "ssn p") {
                     my $title = $attr->{qn};
                     $title =~ s/(^\s+|\s+$)//g;
                     $questiontitles->[$number]=$title;
                 } elsif ("@state" eq "ssn p v") {
                     my $id = $attr->{id};
                     my $entry = $attr->{ans};
                     $id=~s/^[\#0]+//;
                     $entry =~s/[^a-zA-Z0-9\.\*\-\+]+//g;
                     $respbyid{$id}[$number] = $entry;
                 }
            }, "tagname, attr"],
         end_h =>
               [sub {
                   my ($tagname) = @_;
                   if ("@state" eq "ssn p") {
                       $number++;
                   }
                   pop(@state);
                }, "tagname"],
    );

    $p->parse($env{'form.upfile'});
    $p->eof;
    foreach my $id (keys(%respbyid)) {
        $responses->{$id}=join(',',@{$respbyid{$id}});
    }
    return ($errormsg,$number);
}

sub interwrite_eval {
    my ($questiontitles,$responses)=@_;
    my $number=0;
    my $errormsg='';
    my $skipline=1;
    my $questionnumber=0;
    my %idresponses=();
    foreach my $line (split(/[\n\r]/,$env{'form.upfile'})) {
        my %components=&Apache::loncommon::record_sep($line);
        my @entries=map {$components{$_}} (sort(keys(%components)));
        if ($entries[1] eq 'Time') { $skipline=0; next; }
        if ($entries[1] eq 'Response') { $skipline=1; }
        next if $skipline;
        if ($entries[0]!=$questionnumber) {
           $questionnumber=$entries[0];
           $$questiontitles[$number]=&mt('Question [_1]',$questionnumber);
           $number++;
        }
        my $id=$entries[4];
        $id=~s/^[\#0]+//;
        $id=~s/^v\d*\://i;
        $id=~s/[\-\:]//g;
        $idresponses{$id}[$number]=$entries[6];
    }
    foreach my $id (keys(%idresponses)) {
       $$responses{$id}=join(',',@{$idresponses{$id}});
       $$responses{$id}=~s/^\s*\,//;
    }
    return ($errormsg,$number);
}

sub turning_eval {
    my ($questiontitles,$responses)=@_;
    my $number=0;
    my $errormsg='';
    foreach my $line (split(/[\n\r]/,$env{'form.upfile'})) {
        my %components=&Apache::loncommon::record_sep($line);
        my @entries=map {$components{$_}} (sort(keys(%components)));
        if ($#entries>$number) { $number=$#entries; }
        my $id=$entries[0];
        my @idresponses;
        $id=~s/^[\#0]+//;
        unless ($id) { next; }
        for (my $idx=1;$idx<=$#entries;$idx++) {
            $entries[$idx]=~s/\,/\;/g;
            $entries[$idx]=~s/[^a-zA-Z0-9\.\*\-\+\;]+//g;
            push(@idresponses,$entries[$idx]);
        }
        $$responses{$id}=join(',',@idresponses);
    }
    for (my $i=1; $i<=$number; $i++) {
        $$questiontitles[$i]=&mt('Question [_1]',$i);
    }
    return ($errormsg,$number);
}

sub assign_clicker_grades {
    my ($r,$symb) = @_;
    if (!$symb) {return '';}
# See which part we are saving to
    my $res_error;
    my ($partlist,$handgrade,$responseType) = &response_type($symb,\$res_error);
    if ($res_error) {
        return &navmap_errormsg();
    }
# FIXME: This should probably look for the first handgradeable part
    my $part=$$partlist[0];
# Start screen output
    my $result = &Apache::loncommon::start_data_table(). 
                 &Apache::loncommon::start_data_table_header_row().
                 '<th>'.&mt('Assigning grades based on clicker file').'</th>'.
                 &Apache::loncommon::end_data_table_header_row().
                 &Apache::loncommon::start_data_table_row().'<td>';
# Get correct result
# FIXME: Possibly need delimiter other than ":"
    my @correct=();
    my $gradingmechanism=$env{'form.gradingmechanism'};
    my $number=$env{'form.number'};
    if ($gradingmechanism ne 'attendance') {
       foreach my $key (keys(%env)) {
          if ($key=~/^form\.correct\:/) {
             my @input=split(/\,/,$env{$key});
             for (my $i=0;$i<=$#input;$i++) {
                 if (($correct[$i]) && ($input[$i]) &&
                     ($correct[$i] ne $input[$i])) {
                    $result.='<br /><span class="LC_warning">'.
                             &mt('More than one correct result given for question "[_1]": [_2] versus [_3].',
                                 $env{'form.question:'.$i},$correct[$i],$input[$i]).'</span>';
                 } elsif (($input[$i]) || ($input[$i] eq '0')) {
                    $correct[$i]=$input[$i];
                 }
             }
          }
       }
       for (my $i=0;$i<$number;$i++) {
          if ((!$correct[$i]) && ($correct[$i] ne '0')) {
             $result.='<br /><span class="LC_error">'.
                      &mt('No correct result given for question "[_1]"!',
                          $env{'form.question:'.$i}).'</span>';
          }
       }
       $result.='<br />'.&mt("Correct answer: [_1]",join(', ',map { ((($_) || ($_ eq '0'))?$_:'-') } @correct));
    }
# Start grading
    my $pcorrect=$env{'form.pcorrect'};
    my $pincorrect=$env{'form.pincorrect'};
    my $storecount=0;
    my %users=();
    foreach my $key (keys(%env)) {
       my $user='';
       if ($key=~/^form\.student\:(.*)$/) {
          $user=$1;
       }
       if ($key=~/^form\.unknown\:(.*)$/) {
          my $id=$1;
          if (($env{'form.uname'.$id}) && ($env{'form.udom'.$id})) {
             $user=$env{'form.uname'.$id}.':'.$env{'form.udom'.$id};
          } elsif ($env{'form.multi'.$id}) {
             $user=$env{'form.multi'.$id};
          }
       }
       if ($user) {
          if ($users{$user}) {
             $result.='<br /><span class="LC_warning">'.
                      &mt('More than one entry found for [_1]!','<tt>'.$user.'</tt>').
                      '</span><br />';
          }
          $users{$user}=1;
          my @answer=split(/\,/,$env{$key});
          my $sum=0;
          my $realnumber=$number;
          for (my $i=0;$i<$number;$i++) {
             if  ($correct[$i] eq '-') {
                $realnumber--;
             } elsif (($answer[$i]) || ($answer[$i]=~/^[0\.]+$/)) {
                if ($gradingmechanism eq 'attendance') {
                   $sum+=$pcorrect;
                } elsif ($correct[$i] eq '*') {
                   $sum+=$pcorrect;
                } else {
# We actually grade if correct or not
                   my $increment=$pincorrect;
# Special case: numerical answer "0"
                   if ($correct[$i] eq '0') {
                      if ($answer[$i]=~/^[0\.]+$/) {
                         $increment=$pcorrect;
                      }
# General numerical answer, both evaluate to something non-zero
                   } elsif ((1.0*$correct[$i]!=0) && (1.0*$answer[$i]!=0)) {
                      if (1.0*$correct[$i]==1.0*$answer[$i]) {
                         $increment=$pcorrect;
                      }
# Must be just alphanumeric
                   } elsif ($answer[$i] eq $correct[$i]) {
                      $increment=$pcorrect;
                   }
                   $sum+=$increment;
                }
             }
          }
          my $ave=$sum/(100*$realnumber);
# Store
          my ($username,$domain)=split(/\:/,$user);
          my %grades=();
          $grades{"resource.$part.solved"}='correct_by_override';
          $grades{"resource.$part.awarded"}=$ave;
          $grades{"resource.regrader"}="$env{'user.name'}:$env{'user.domain'}";
          my $returncode=&Apache::lonnet::cstore(\%grades,$symb,
                                                 $env{'request.course.id'},
                                                 $domain,$username);
          if ($returncode ne 'ok') {
             $result.="<br /><span class=\"LC_error\">Failed to save student $username:$domain. Message when trying to save was ($returncode)</span>";
          } else {
             $storecount++;
          }
       }
    }
# We are done
    $result.='<br />'.&mt('Successfully stored grades for [quant,_1,student].',$storecount).
             '</td>'.
             &Apache::loncommon::end_data_table_row().
             &Apache::loncommon::end_data_table();
    return $result;
}

sub navmap_errormsg {
    return '<div class="LC_error">'.
           &mt('An error occurred retrieving information about resources in the course.').'<br />'.
           &mt('It is recommended that you [_1]re-initialize the course[_2] and then return to this grading page.','<a href="/adm/roles?selectrole=1&newrole='.$env{'request.role'}.'">','</a>').
           '</div>';
}

sub startpage {
    my ($r,$symb,$crumbs,$onlyfolderflag,$nodisplayflag,$stuvcurrent,$stuvdisp,$nomenu,$head_extra,$onload,$divforres) = @_;
    my %args;
    if ($onload) {
         my %loaditems = (
                        'onload' => $onload,
                      );
         $args{'add_entries'} = \%loaditems;
    }
    if ($nomenu) {
        $args{'only_body'} = 1;
        $r->print(&Apache::loncommon::start_page("Student's Version",$head_extra,\%args));
    } else {
        if ($env{'request.course.id'}) {
            unshift(@$crumbs,{href=>&href_symb_cmd($symb,'gradingmenu'),text=>"Grading"});
        }
        $args{'bread_crumbs'} = $crumbs;
        $r->print(&Apache::loncommon::start_page('Grading',$head_extra,\%args));
    }
    unless ($nodisplayflag) {
       $r->print(&Apache::lonhtmlcommon::resource_info_box($symb,$onlyfolderflag,$stuvcurrent,$stuvdisp,$divforres));
    }
}

sub select_problem {
    my ($r)=@_;
    $r->print('<h3>'.&mt('Select the problem or one of the problems you want to grade').'</h3><form action="/adm/grades">');
    $r->print(&Apache::lonstathelpers::problem_selector('.',undef,1,undef,undef,1));
    $r->print('<input type="hidden" name="command" value="gradingmenu" />');
    $r->print('<input type="submit" value="'.&mt('Next').' &rarr;" /></form>');
}

sub handler {
    my $request=$_[0];
    &reset_caches();
    if ($request->header_only) {
        &Apache::loncommon::content_type($request,'text/html');
        $request->send_http_header;
        return OK;
    }
    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'});

# see what command we need to execute
 
    my @commands=&Apache::loncommon::get_env_multiple('form.command');
    my $command=$commands[0];

    &init_perm();
    if (!$env{'request.course.id'}) {
        unless ((&Apache::lonnet::allowed('usc',$env{'request.role.domain'})) &&
                ($command =~ /^scantronupload/)) {
            # Not in a course.
            $env{'user.error.msg'}="/adm/grades::vgr:0:0:Cannot display grades page outside course context";
            return HTTP_NOT_ACCEPTABLE;
        }
    } elsif (!%perm) {
        $request->internal_redirect('/adm/quickgrades');
        return OK;
    }
    &Apache::loncommon::content_type($request,'text/html');
    $request->send_http_header;

    if ($#commands > 0) {
	&Apache::lonnet::logthis("grades got multiple commands ".join(':',@commands));
    }

# see what the symb is

    my $symb=$env{'form.symb'};
    unless ($symb) {
       (my $url=$env{'form.url'}) =~ s-^https*://($ENV{'SERVER_NAME'}|$ENV{'HTTP_HOST'})--;
       $symb=&Apache::lonnet::symbread($url);
    }
    &Apache::lonenc::check_decrypt(\$symb);

    $ssi_error = 0;
    if (($symb eq '' || $command eq '') && ($env{'request.course.id'})) {
#
# Not called from a resource, but inside a course
#
        &startpage($request,undef,[],1,1);
        &select_problem($request);
    } else {
        if ($command eq 'submission' && $perm{'vgr'}) {
            my ($stuvcurrent,$stuvdisp,$versionform,$js,$onload);
            if (($env{'form.student'} ne '') && ($env{'form.userdom'} ne '')) {
                ($stuvcurrent,$stuvdisp,$versionform,$js) =
                    &choose_task_version_form($symb,$env{'form.student'},
                                              $env{'form.userdom'});
            }
            my $divforres;
            if ($env{'form.student'} eq '') {
                $js .= &part_selector_js();
                $onload = "toggleParts('gradesub');";
            } else {
                $divforres = 1;
            }
            my $head_extra = $js;
            unless ($env{'form.vProb'} eq 'no') {
                my $csslinks = &Apache::loncommon::css_links($symb);
                if ($csslinks) {
                    $head_extra .= "\n$csslinks";
                }
            }
            &startpage($request,$symb,[{href=>"", text=>"Student Submissions"}],undef,undef,
                       $stuvcurrent,$stuvdisp,undef,$head_extra,$onload,$divforres);
            if ($versionform) {
                if ($divforres) {
                    $request->print('<div style="padding:0;clear:both;margin:0;border:0"></div>');
                }
                $request->print($versionform);
            }
            ($env{'form.student'} eq '' ? &listStudents($request,$symb,'',$divforres) : &submission($request,0,0,$symb,$divforres,$command));
        } elsif ($command eq 'versionsub' && $perm{'vgr'}) {
            my ($stuvcurrent,$stuvdisp,$versionform,$js) =
                &choose_task_version_form($symb,$env{'form.student'},
                                          $env{'form.userdom'},
                                          $env{'form.inhibitmenu'});
            my $head_extra = $js;
            unless ($env{'form.vProb'} eq 'no') {
                my $csslinks = &Apache::loncommon::css_links($symb);
                if ($csslinks) {
                    $head_extra .= "\n$csslinks";
                }
            }
            &startpage($request,$symb,[{href=>"", text=>"Previous Student Version"}],undef,undef,
                       $stuvcurrent,$stuvdisp,$env{'form.inhibitmenu'},$head_extra);
            if ($versionform) {
                $request->print($versionform);
            }
            $request->print('<br clear="all" />');
            $request->print(&show_previous_task_version($request,$symb));
        } elsif ($command eq 'pickStudentPage' && $perm{'vgr'}) {
            &startpage($request,$symb,[{href=>&href_symb_cmd($symb,'all_for_one'),text=>'Grade page/folder for one student'},
                                       {href=>'',text=>'Select student'}],1,1);
            &pickStudentPage($request,$symb);
        } elsif ($command eq 'displayPage' && $perm{'vgr'}) {
            my $csslinks;
            unless ($env{'form.vProb'} eq 'no') {
                $csslinks = &Apache::loncommon::css_links($symb,'map');
            }
            &startpage($request,$symb,
                                      [{href=>&href_symb_cmd($symb,'all_for_one'),text=>'Grade page/folder for one student'},
                                       {href=>'',text=>'Select student'},
                                       {href=>'',text=>'Grade student'}],1,1,undef,undef,undef,$csslinks);
            &displayPage($request,$symb);
        } elsif ($command eq 'gradeByPage' && $perm{'mgr'}) {
            &startpage($request,$symb,[{href=>&href_symb_cmd($symb,'all_for_one'),text=>'Grade page/folder for one student'},
                                       {href=>'',text=>'Select student'},
                                       {href=>'',text=>'Grade student'},
                                       {href=>'',text=>'Store grades'}],1,1);
            &updateGradeByPage($request,$symb);
        } elsif ($command eq 'processGroup' && $perm{'vgr'}) {
            my $csslinks;
            unless ($env{'form.vProb'} eq 'no') {
                $csslinks = &Apache::loncommon::css_links($symb);
            }
            &startpage($request,$symb,[{href=>'',text=>'...'},
                                       {href=>'',text=>'Modify grades'}],undef,undef,undef,undef,undef,$csslinks,undef,1);
            &processGroup($request,$symb);
        } elsif ($command eq 'gradingmenu' && $perm{'vgr'}) {
            &startpage($request,$symb);
            $request->print(&grading_menu($request,$symb));
        } elsif ($command eq 'individual' && $perm{'vgr'}) {
            &startpage($request,$symb,[{href=>'',text=>'Select individual students to grade'}]);
            $request->print(&submit_options($request,$symb));
        } elsif ($command eq 'ungraded' && $perm{'vgr'}) {
            my $js = &part_selector_js();
            my $onload = "toggleParts('gradesub');";
            &startpage($request,$symb,[{href=>'',text=>'Grade ungraded submissions'}],
                       undef,undef,undef,undef,undef,$js,$onload);
            $request->print(&listStudents($request,$symb,'graded'));
        } elsif ($command eq 'table' && $perm{'vgr'}) {
            &startpage($request,$symb,[{href=>"", text=>"Grading table"}]);
            $request->print(&submit_options_table($request,$symb));
        } elsif ($command eq 'all_for_one' && $perm{'vgr'}) {
            &startpage($request,$symb,[{href=>'',text=>'Grade page/folder for one student'}],1,1);
            $request->print(&submit_options_sequence($request,$symb));
        } elsif ($command eq 'viewgrades' && $perm{'vgr'}) {
            &startpage($request,$symb,[{href=>&href_symb_cmd($symb,"table"), text=>"Grading table"},{href=>'', text=>"Modify grades"}]);
            $request->print(&viewgrades($request,$symb));
        } elsif ($command eq 'handgrade' && $perm{'mgr'}) {
            &startpage($request,$symb,[{href=>'',text=>'...'},
                                       {href=>'',text=>'Store grades'}]);
            $request->print(&processHandGrade($request,$symb));
        } elsif ($command eq 'editgrades' && $perm{'mgr'}) {
            &startpage($request,$symb,[{href=>&href_symb_cmd($symb,"table"), text=>"Grading table"},
                                       {href=>&href_symb_cmd($symb,'viewgrades').'&group=all&section=all&Status=Active',
                                                                             text=>"Modify grades"},
                                       {href=>'', text=>"Store grades"}]);
            $request->print(&editgrades($request,$symb));
        } elsif ($command eq 'initialverifyreceipt' && $perm{'vgr'}) {
            &startpage($request,$symb,[{href=>'',text=>'Verify Receipt Number'}]);
            $request->print(&initialverifyreceipt($request,$symb));
        } elsif ($command eq 'verify' && $perm{'vgr'}) {
            &startpage($request,$symb,[{href=>&href_symb_cmd($symb,"initialverifyreceipt"),text=>'Verify Receipt Number'},
                                       {href=>'',text=>'Verification Result'}]);
            $request->print(&verifyreceipt($request,$symb));
        } elsif ($command eq 'processclicker' && $perm{'mgr'}) {
            &startpage($request,$symb,[{href=>'', text=>'Process clicker'}]);
            $request->print(&process_clicker($request,$symb));
        } elsif ($command eq 'processclickerfile' && $perm{'mgr'}) {
            &startpage($request,$symb,[{href=>&href_symb_cmd($symb,'processclicker'), text=>'Process clicker'},
                                       {href=>'', text=>'Process clicker file'}]);
            $request->print(&process_clicker_file($request,$symb));
        } elsif ($command eq 'assignclickergrades' && $perm{'mgr'}) {
            &startpage($request,$symb,[{href=>&href_symb_cmd($symb,'processclicker'), text=>'Process clicker'},
                                       {href=>'', text=>'Process clicker file'},
                                       {href=>'', text=>'Store grades'}]);
            $request->print(&assign_clicker_grades($request,$symb));
        } elsif ($command eq 'csvform' && $perm{'mgr'}) {
            &startpage($request,$symb,[{href=>'', text=>'Upload Scores'}],1,1);
            $request->print(&upcsvScores_form($request,$symb));
        } elsif ($command eq 'csvupload' && $perm{'mgr'}) {
            &startpage($request,$symb,[{href=>'', text=>'Upload Scores'}],1,1);
            $request->print(&csvupload($request,$symb));
        } elsif ($command eq 'csvuploadmap' && $perm{'mgr'} ) {
            &startpage($request,$symb,[{href=>'', text=>'Upload Scores'}],1,1);
            $request->print(&csvuploadmap($request,$symb));
        } elsif ($command eq 'csvuploadoptions' && $perm{'mgr'}) {
            if ($env{'form.associate'} ne 'Reverse Association') {
                &startpage($request,$symb,[{href=>'', text=>'Upload Scores'}],1,1);
                $request->print(&csvuploadoptions($request,$symb));
            } else {
                if ( $env{'form.upfile_associate'} ne 'reverse' ) {
                    $env{'form.upfile_associate'} = 'reverse';
                } else {
                    $env{'form.upfile_associate'} = 'forward';
                }
                &startpage($request,$symb,[{href=>'', text=>'Upload Scores'}],1,1);
                $request->print(&csvuploadmap($request,$symb));
            }
        } elsif ($command eq 'csvuploadassign' && $perm{'mgr'} ) {
            &startpage($request,$symb,[{href=>'', text=>'Upload Scores'}],1,1);
            $request->print(&csvuploadassign($request,$symb));
        } elsif ($command eq 'scantron_selectphase' && $perm{'mgr'}) {
            &startpage($request,$symb,[{href=>'', text=>'Grade/Manage/Review Bubblesheets'}],1,1,
                       undef,undef,undef,undef,'toggleScantab(document.rules);');
            $request->print(&scantron_selectphase($request,undef,$symb));
        } elsif ($command eq 'scantron_warning' && $perm{'mgr'}) {
            &startpage($request,$symb,[{href=>'', text=>'Grade/Manage/Review Bubblesheets'}],1,1);
            $request->print(&scantron_do_warning($request,$symb));
        } elsif ($command eq 'scantron_validate' && $perm{'mgr'}) {
            &startpage($request,$symb,[{href=>'', text=>'Grade/Manage/Review Bubblesheets'}],1,1);
            $request->print(&scantron_validate_file($request,$symb));
        } elsif ($command eq 'scantron_process' && $perm{'mgr'}) {
            &startpage($request,$symb,[{href=>'', text=>'Grade/Manage/Review Bubblesheets'}],1,1);
            $request->print(&scantron_process_students($request,$symb));
        } elsif ($command eq 'scantronupload' &&
                 (&Apache::lonnet::allowed('usc',$env{'request.role.domain'})||
                  &Apache::lonnet::allowed('usc',$env{'request.course.id'}))) {
            &startpage($request,$symb,[{href=>'', text=>'Grade/Manage/Review Bubblesheets'}],1,1,
                       undef,undef,undef,undef,'toggleScantab(document.rules);');
            $request->print(&scantron_upload_scantron_data($request,$symb));
        } elsif ($command eq 'scantronupload_save' &&
                 (&Apache::lonnet::allowed('usc',$env{'request.role.domain'})||
                  &Apache::lonnet::allowed('usc',$env{'request.course.id'}))) {
            &startpage($request,$symb,[{href=>'', text=>'Grade/Manage/Review Bubblesheets'}],1,1);
            $request->print(&scantron_upload_scantron_data_save($request,$symb));
        } elsif ($command eq 'scantron_download' &&
                 &Apache::lonnet::allowed('usc',$env{'request.course.id'})) {
            &startpage($request,$symb,[{href=>'', text=>'Grade/Manage/Review Bubblesheets'}],1,1);
            $request->print(&scantron_download_scantron_data($request,$symb));
        } elsif ($command eq 'checksubmissions' && $perm{'vgr'}) {
            &startpage($request,$symb,[{href=>'', text=>'Grade/Manage/Review Bubblesheets'}],1,1);
            $request->print(&checkscantron_results($request,$symb));
        } elsif ($command eq 'downloadfilesselect' && $perm{'vgr'}) {
            my $js = &part_selector_js();
            my $onload = "toggleParts('gradingMenu');";
            &startpage($request,$symb,[{href=>'', text=>'Select which submissions to download'}],
                       undef,undef,undef,undef,undef,$js,$onload);
            $request->print(&submit_options_download($request,$symb));
         } elsif ($command eq 'downloadfileslink' && $perm{'vgr'}) {
            &startpage($request,$symb,
   [{href=>&href_symb_cmd($symb,'downloadfilesselect'), text=>'Select which submissions to download'},
    {href=>'', text=>'Download submitted files'}],
               undef,undef,undef,undef,undef,undef,undef,1);
            $request->print('<div style="padding:0;clear:both;margin:0;border:0"></div>');
            &submit_download_link($request,$symb);
        } elsif ($command) {
            &startpage($request,$symb,[{href=>'', text=>'Access denied'}]);
            $request->print('<p class="LC_error">'.&mt('Access Denied ([_1])',$command).'</p>');
        }
    }
    if ($ssi_error) {
	&ssi_print_error($request);
    }
    $request->print(&Apache::loncommon::end_page());
    &reset_caches();
    return OK;
}

1;

__END__;


=head1 NAME

Apache::grades

=head1 SYNOPSIS

Handles the viewing of grades.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 OVERVIEW

Do an ssi with retries:
While I'd love to factor out this with the vesrion in lonprintout,
that would either require a data coupling between modules, which I refuse to perpetuate (there's quite enough of that already), or would require the invention of another infrastructure
I'm not quite ready to invent (e.g. an ssi_with_retry object).

At least the logic that drives this has been pulled out into loncommon.



ssi_with_retries - Does the server side include of a resource.
                     if the ssi call returns an error we'll retry it up to
                     the number of times requested by the caller.
                     If we still have a problem, no text is appended to the
                     output and we set some global variables.
                     to indicate to the caller an SSI error occurred.  
                     All of this is supposed to deal with the issues described
                     in LON-CAPA BZ 5631 see:
                     http://bugs.lon-capa.org/show_bug.cgi?id=5631
                     by informing the user that this happened.

Parameters:
  resource   - The resource to include.  This is passed directly, without
               interpretation to lonnet::ssi.
  form       - The form hash parameters that guide the interpretation of the resource
               
  retries    - Number of retries allowed before giving up completely.
Returns:
  On success, returns the rendered resource identified by the resource parameter.
Side Effects:
  The following global variables can be set:
   ssi_error                - If an unrecoverable error occurred this becomes true.
                              It is up to the caller to initialize this to false
                              if desired.
   ssi_error_resource  - If an unrecoverable error occurred, this is the value
                              of the resource that could not be rendered by the ssi
                              call.
   ssi_error_message   - The error string fetched from the ssi response
                              in the event of an error.


=head1 HANDLER SUBROUTINE

ssi_with_retries()

=head1 SUBROUTINES

=over

=item scantron_get_correction() : 

   Builds the interface screen to interact with the operator to fix a
   specific error condition in a specific scanline

 Arguments:
    $r           - Apache request object
    $i           - number of the current scanline
    $scan_record - hash ref as returned from &scantron_parse_scanline()
    $scan_config - hash ref as returned from &Apache::lonnet::get_scantron_config()
    $line        - full contents of the current scanline
    $error       - error condition, valid values are
                   'incorrectCODE', 'duplicateCODE',
                   'doublebubble', 'missingbubble',
                   'duplicateID', 'incorrectID'
    $arg         - extra information needed
       For errors:
         - duplicateID   - paper number that this studentID was seen before on
         - duplicateCODE - array ref of the paper numbers this CODE was
                           seen on before
         - incorrectCODE - current incorrect CODE 
         - doublebubble  - array ref of the bubble lines that have double
                           bubble errors
         - missingbubble - array ref of the bubble lines that have missing
                           bubble errors

   $randomorder - True if exam folder (or a sub-folder) has randomorder set
   $randompick  - True if exam folder (or a sub-folder) has randompick set
   $respnumlookup - Reference to HASH mapping question numbers in bubble lines
                     for current line to question number used for same question
                     in "Master Seqence" (as seen by Course Coordinator).
   $startline   - Reference to hash where key is question number (0 is first)
                  and value is number of first bubble line for current student
                  or code-based randompick and/or randomorder.


=item  scantron_get_maxbubble() : 

   Arguments:
       $nav_error  - Reference to scalar which is a flag to indicate a
                      failure to retrieve a navmap object.
       if $nav_error is set to 1 by scantron_get_maxbubble(), the 
       calling routine should trap the error condition and display the warning
       found in &navmap_errormsg().

       $scantron_config - Reference to bubblesheet format configuration hash.

   Returns the maximum number of bubble lines that are expected to
   occur. Does this by walking the selected sequence rendering the
   resource and then checking &Apache::lonxml::get_problem_counter()
   for what the current value of the problem counter is.

   Caches the results to $env{'form.scantron_maxbubble'},
   $env{'form.scantron.bubble_lines.n'}, 
   $env{'form.scantron.first_bubble_line.n'} and
   $env{"form.scantron.sub_bubblelines.n"}
   which are the total number of bubble lines, the number of bubble
   lines for response n and number of the first bubble line for response n,
   and a comma separated list of numbers of bubble lines for sub-questions
   (for optionresponse, matchresponse, and rankresponse items), for response n.  


=item  scantron_validate_missingbubbles() : 

   Validates all scanlines in the selected file to not have any
    answers that don't have bubbles that have not been verified
    to be bubble free.

=item  scantron_process_students() : 

   Routine that does the actual grading of the bubblesheet information.

   The parsed scanline hash is added to %env 

   Then foreach unskipped scanline it does an &Apache::lonnet::ssi()
   foreach resource , with the form data of

	'submitted'     =>'scantron' 
	'grade_target'  =>'grade',
	'grade_username'=> username of student
	'grade_domain'  => domain of student
	'grade_courseid'=> of course
	'grade_symb'    => symb of resource to grade

    This triggers a grading pass. The problem grading code takes care
    of converting the bubbled letter information (now in %env) into a
    valid submission.

=item  scantron_upload_scantron_data() :

    Creates the screen for adding a new bubblesheet data file to a course.

=item  scantron_upload_scantron_data_save() : 

   Adds a provided bubble information data file to the course if user
   has the correct privileges to do so. 

=item  valid_file() :

   Validates that the requested bubble data file exists in the course.

=item  scantron_download_scantron_data() : 

   Shows a list of the three internal files (original, corrected,
   skipped) for a specific bubblesheet data file that exists in the
   course.

=item  scantron_validate_ID() : 

   Validates all scanlines in the selected file to not have any
   invalid or underspecified student/employee IDs

=item navmap_errormsg() :

   Returns HTML mark-up inside a <div></div> with a link to re-initialize the course.
   Should be called whenever the request to instantiate a navmap object fails.  

=back

=cut
