# The LearningOnline Network with CAPA
# a pile of common routines
#
# $Id: loncommon.pm,v 1.1075.2.176 2024/12/29 23:15:23 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#

# Makes a table out of the previous attempts
# Inputs result_from_symbread, user, domain, course_id
# Reads in non-network-related .tab files

# POD header:

=pod

=head1 NAME

Apache::loncommon - pile of common routines

=head1 SYNOPSIS

Common routines for manipulating connections, student answers,
    domains, common Javascript fragments, etc.

=head1 OVERVIEW

A collection of commonly used subroutines that don't have a natural
home anywhere else. This collection helps remove
redundancy from other modules and increase efficiency of memory usage.

=cut 

# End of POD header
package Apache::loncommon;

use strict;
use Apache::lonnet;
use GDBM_File;
use POSIX qw(strftime mktime);
use Apache::lonmenu();
use Apache::lonenc();
use Apache::lonlocal;
use HTML::Entities;
use Apache::lonhtmlcommon();
use Apache::loncoursedata();
use Apache::lontexconvert();
use Apache::lonclonecourse();
use Apache::lonuserutils();
use Apache::lonuserstate();
use Apache::courseclassifier();
use LONCAPA qw(:DEFAULT :match);
use HTTP::Request;
use DateTime::TimeZone;
use DateTime::Locale;
use Encode();
use Authen::Captcha;
use Captcha::reCAPTCHA;
use JSON::DWIW;
use LWP::UserAgent;
use Crypt::DES;
use DynaLoader; # for Crypt::DES version
use File::Copy();
use File::Path();

# ---------------------------------------------- Designs
use vars qw(%defaultdesign);

my $readit;


##
## Global Variables
##


# ----------------------------------------------- SSI with retries:
#

=pod

=head1 Server Side include with retries:

=over 4

=item * &ssi_with_retries(resource,retries form)

Performs an ssi with some number of retries.  Retries continue either
until the result is ok or until the retry count supplied by the
caller is exhausted.  

Inputs:

=over 4

resource   - Identifies the resource to insert.

retries    - Count of the number of retries allowed.

form       - Hash that identifies the rendering options.

=back

Returns:

=over 4

content    - The content of the response.  If retries were exhausted this is empty.

response   - The response from the last attempt (which may or may not have been successful.

=back

=back

=cut

sub ssi_with_retries {
    my ($resource, $retries, %form) = @_;


    my $ok = 0;			# True if we got a good response.
    my $content;
    my $response;

    # Try to get the ssi done. within the retries count:

    do {
	($content, $response) = &Apache::lonnet::ssi($resource, %form);
	$ok      = $response->is_success;
        if (!$ok) {
            &Apache::lonnet::logthis("Failed ssi_with_retries on $resource: ".$response->is_success.', '.$response->code.', '.$response->message);
        }
	$retries--;
    } while (!$ok && ($retries > 0));

    if (!$ok) {
	$content = '';		# On error return an empty content.
    }
    return ($content, $response);

}



# ----------------------------------------------- Filetypes/Languages/Copyright
my %language;
my %supported_language;
my %latex_language;		# For choosing hyphenation in <transl..>
my %latex_language_bykey;	# for choosing hyphenation from metadata
my %cprtag;
my %scprtag;
my %fe; my %fd; my %fm;
my %category_extensions;

# ---------------------------------------------- Thesaurus variables
#
# %Keywords:
#      A hash used by &keyword to determine if a word is considered a keyword.
# $thesaurus_db_file 
#      Scalar containing the full path to the thesaurus database.

my %Keywords;
my $thesaurus_db_file;

#
# Initialize values from language.tab, copyright.tab, filetypes.tab,
# thesaurus.tab, and filecategories.tab.
#
BEGIN {
    # Variable initialization
    $thesaurus_db_file = $Apache::lonnet::perlvar{'lonTabDir'}."/thesaurus.db";
    #
    unless ($readit) {
# ------------------------------------------------------------------- languages
    {
        my $langtabfile = $Apache::lonnet::perlvar{'lonTabDir'}.
                                   '/language.tab';
        if ( open(my $fh,'<',$langtabfile) ) {
            while (my $line = <$fh>) {
                next if ($line=~/^\#/);
                chomp($line);
                my ($key,$two,$country,$three,$enc,$val,$sup,$latex)=(split(/\t/,$line));
                $language{$key}=$val.' - '.$enc;
                if ($sup) {
                    $supported_language{$key}=$sup;
                }
		if ($latex) {
		    $latex_language_bykey{$key} = $latex;
		    $latex_language{$two} = $latex;
		}
            }
            close($fh);
        }
    }
# ------------------------------------------------------------------ copyrights
    {
        my $copyrightfile = $Apache::lonnet::perlvar{'lonIncludes'}.
                                  '/copyright.tab';
        if ( open (my $fh,'<',$copyrightfile) ) {
            while (my $line = <$fh>) {
                next if ($line=~/^\#/);
                chomp($line);
                my ($key,$val)=(split(/\s+/,$line,2));
                $cprtag{$key}=$val;
            }
            close($fh);
        }
    }
# ----------------------------------------------------------- source copyrights
    {
        my $sourcecopyrightfile = $Apache::lonnet::perlvar{'lonIncludes'}.
                                  '/source_copyright.tab';
        if ( open (my $fh,'<',$sourcecopyrightfile) ) {
            while (my $line = <$fh>) {
                next if ($line =~ /^\#/);
                chomp($line);
                my ($key,$val)=(split(/\s+/,$line,2));
                $scprtag{$key}=$val;
            }
            close($fh);
        }
    }

# -------------------------------------------------------------- default domain designs
    my $designdir=$Apache::lonnet::perlvar{'lonTabDir'}.'/lonDomColors';
    my $designfile = $designdir.'/default.tab';
    if ( open (my $fh,'<',$designfile) ) {
        while (my $line = <$fh>) {
            next if ($line =~ /^\#/);
            chomp($line);
            my ($key,$val)=(split(/\=/,$line));
            if ($val) { $defaultdesign{$key}=$val; }
        }
        close($fh);
    }

# ------------------------------------------------------------- file categories
    {
        my $categoryfile = $Apache::lonnet::perlvar{'lonTabDir'}.
                                  '/filecategories.tab';
        if ( open (my $fh,'<',$categoryfile) ) {
	    while (my $line = <$fh>) {
		next if ($line =~ /^\#/);
		chomp($line);
                my ($extension,$category)=(split(/\s+/,$line,2));
                push(@{$category_extensions{lc($category)}},$extension);
            }
            close($fh);
        }

    }
# ------------------------------------------------------------------ file types
    {
        my $typesfile = $Apache::lonnet::perlvar{'lonTabDir'}.
               '/filetypes.tab';
        if ( open (my $fh,'<',$typesfile) ) {
            while (my $line = <$fh>) {
		next if ($line =~ /^\#/);
		chomp($line);
                my ($ending,$emb,$mime,$descr)=split(/\s+/,$line,4);
                if ($descr ne '') {
                    $fe{$ending}=lc($emb);
                    $fd{$ending}=$descr;
                    if ($mime ne 'unk') { $fm{$ending}=$mime; }
                }
            }
            close($fh);
        }
    }
    &Apache::lonnet::logthis(
             "<span style='color:yellow;'>INFO: Read file types</span>");
    $readit=1;
    }  # end of unless($readit) 
    
}

###############################################################
##           HTML and Javascript Helper Functions            ##
###############################################################

=pod 

=head1 HTML and Javascript Functions

=over 4

=item * &browser_and_searcher_javascript()

X<browsing, javascript>X<searching, javascript>Returns a string
containing javascript with two functions, C<openbrowser> and
C<opensearcher>. Returned string does not contain E<lt>scriptE<gt>
tags.

=item * &openbrowser(formname,elementname,only,omit) [javascript]

inputs: formname, elementname, only, omit

formname and elementname indicate the name of the html form and name of
the element that the results of the browsing selection are to be placed in. 

Specifying 'only' will restrict the browser to displaying only files
with the given extension.  Can be a comma separated list.

Specifying 'omit' will restrict the browser to NOT displaying files
with the given extension.  Can be a comma separated list.

=item * &opensearcher(formname,elementname) [javascript]

Inputs: formname, elementname

formname and elementname specify the name of the html form and the name
of the element the selection from the search results will be placed in.

=cut

sub browser_and_searcher_javascript {
    my ($mode)=@_;
    if (!defined($mode)) { $mode='edit'; }
    my $resurl=&escape_single(&lastresurl());
    return <<END;
// <!-- BEGIN LON-CAPA Internal
    var editbrowser = null;
    function openbrowser(formname,elementname,only,omit,titleelement) {
        var url = '$resurl/?';
        if (editbrowser == null) {
            url += 'launch=1&';
        }
        url += 'catalogmode=interactive&';
        url += 'mode=$mode&';
        url += 'inhibitmenu=yes&';
        url += 'form=' + formname + '&';
        if (only != null) {
            url += 'only=' + only + '&';
        } else {
            url += 'only=&';
	}
        if (omit != null) {
            url += 'omit=' + omit + '&';
        } else {
            url += 'omit=&';
	}
        if (titleelement != null) {
            url += 'titleelement=' + titleelement + '&';
        } else {
	    url += 'titleelement=&';
	}
        url += 'element=' + elementname + '';
        var title = 'Browser';
        var options = 'scrollbars=1,resizable=1,menubar=0,toolbar=1,location=1';
        options += ',width=700,height=600';
        editbrowser = open(url,title,options,'1');
        editbrowser.focus();
    }
    var editsearcher;
    function opensearcher(formname,elementname,titleelement) {
        var url = '/adm/searchcat?';
        if (editsearcher == null) {
            url += 'launch=1&';
        }
        url += 'catalogmode=interactive&';
        url += 'mode=$mode&';
        url += 'form=' + formname + '&';
        if (titleelement != null) {
            url += 'titleelement=' + titleelement + '&';
        } else {
	    url += 'titleelement=&';
	}
        url += 'element=' + elementname + '';
        var title = 'Search';
        var options = 'scrollbars=1,resizable=1,menubar=0,toolbar=1,location=1';
        options += ',width=700,height=600';
        editsearcher = open(url,title,options,'1');
        editsearcher.focus();
    }
// END LON-CAPA Internal -->
END
}

sub lastresurl {
    if ($env{'environment.lastresurl'}) {
	return $env{'environment.lastresurl'}
    } else {
	return '/res';
    }
}

sub storeresurl {
    my $resurl=&Apache::lonnet::clutter(shift);
    unless ($resurl=~/^\/res/) { return 0; }
    $resurl=~s/\/$//;
    &Apache::lonnet::put('environment',{'lastresurl' => $resurl});
    &Apache::lonnet::appenv({'environment.lastresurl' => $resurl});
    return 1;
}

sub studentbrowser_javascript {
   unless (
            (($env{'request.course.id'}) && 
             (&Apache::lonnet::allowed('srm',$env{'request.course.id'})
	      || &Apache::lonnet::allowed('srm',$env{'request.course.id'}.
					  '/'.$env{'request.course.sec'})
	      ))
         || ($env{'request.role'}=~/^(au|dc|su)/)
          ) { return ''; }  
   return (<<'ENDSTDBRW');
<script type="text/javascript" language="Javascript">
// <![CDATA[
    var stdeditbrowser;
    function openstdbrowser(formname,uname,udom,clicker,roleflag,ignorefilter,courseadv) {
        var url = '/adm/pickstudent?';
        var filter;
	if (!ignorefilter) {
	    eval('filter=document.'+formname+'.'+uname+'.value;');
	}
        if (filter != null) {
           if (filter != '') {
               url += 'filter='+filter+'&';
	   }
        }
        url += 'form=' + formname + '&unameelement='+uname+
                                    '&udomelement='+udom+
                                    '&clicker='+clicker;
	if (roleflag) { url+="&roles=1"; }
        if (courseadv == 'condition') {
            if (document.getElementById('courseadv')) {
                courseadv = document.getElementById('courseadv').value;
            }
        }
        if ((courseadv == 'only') || (courseadv == 'none')) { url+="&courseadv="+courseadv; }
        var title = 'Student_Browser';
        var options = 'scrollbars=1,resizable=1,menubar=0';
        options += ',width=700,height=600';
        stdeditbrowser = open(url,title,options,'1');
        stdeditbrowser.focus();
    }
// ]]>
</script>
ENDSTDBRW
}

sub resourcebrowser_javascript {
   unless ($env{'request.course.id'}) { return ''; }
   return (<<'ENDRESBRW');
<script type="text/javascript" language="Javascript">
// <![CDATA[
    var reseditbrowser;
    function openresbrowser(formname,reslink) {
        var url = '/adm/pickresource?form='+formname+'&reslink='+reslink;
        var title = 'Resource_Browser';
        var options = 'scrollbars=1,resizable=1,menubar=0';
        options += ',width=700,height=500';
        reseditbrowser = open(url,title,options,'1');
        reseditbrowser.focus();
    }
// ]]>
</script>
ENDRESBRW
}

sub selectstudent_link {
   my ($form,$unameele,$udomele,$courseadv,$clickerid)=@_;
   my $callargs = "'".&Apache::lonhtmlcommon::entity_encode($form)."','".
                      &Apache::lonhtmlcommon::entity_encode($unameele)."','".
                      &Apache::lonhtmlcommon::entity_encode($udomele)."'";
   if ($env{'request.course.id'}) {  
       if (!&Apache::lonnet::allowed('srm',$env{'request.course.id'})
	   && !&Apache::lonnet::allowed('srm',$env{'request.course.id'}.
					'/'.$env{'request.course.sec'})) {
	   return '';
       }
       $callargs.=",'".&Apache::lonhtmlcommon::entity_encode($clickerid)."'";
       if ($courseadv eq 'only') {
           $callargs .= ",'',1,'$courseadv'";
       } elsif ($courseadv eq 'none') {
           $callargs .= ",'','','$courseadv'";
       } elsif ($courseadv eq 'condition') {
           $callargs .= ",'','','$courseadv'";
       }
       return '<span class="LC_nobreak">'.
              '<a href="javascript:openstdbrowser('.$callargs.');">'.
              &mt('Select User').'</a></span>';
   }
   if ($env{'request.role'}=~/^(au|dc|su)/) {
       $callargs .= ",'',1"; 
       return '<span class="LC_nobreak">'.
              '<a href="javascript:openstdbrowser('.$callargs.');">'.
              &mt('Select User').'</a></span>';
   }
   return '';
}

sub selectresource_link {
   my ($form,$reslink,$arg)=@_;
   
   my $callargs = "'".&Apache::lonhtmlcommon::entity_encode($form)."','".
                      &Apache::lonhtmlcommon::entity_encode($reslink)."'";
   unless ($env{'request.course.id'}) { return $arg; }
   return '<span class="LC_nobreak">'.
              '<a href="javascript:openresbrowser('.$callargs.');">'.
              $arg.'</a></span>';
}



sub authorbrowser_javascript {
    return <<"ENDAUTHORBRW";
<script type="text/javascript" language="JavaScript">
// <![CDATA[
var stdeditbrowser;

function openauthorbrowser(formname,udom) {
    var url = '/adm/pickauthor?';
    url += 'form='+formname+'&roledom='+udom;
    var title = 'Author_Browser';
    var options = 'scrollbars=1,resizable=1,menubar=0';
    options += ',width=700,height=600';
    stdeditbrowser = open(url,title,options,'1');
    stdeditbrowser.focus();
}

// ]]>
</script>
ENDAUTHORBRW
}

sub coursebrowser_javascript {
    my ($domainfilter,$sec_element,$formname,$role_element,$crstype,
        $credits_element,$instcode) = @_;
    my $wintitle = 'Course_Browser';
    if ($crstype eq 'Community') {
        $wintitle = 'Community_Browser';
    }
    my $id_functions = &javascript_index_functions();
    my $output = '
<script type="text/javascript" language="JavaScript">
// <![CDATA[
    var stdeditbrowser;'."\n";

    $output .= <<"ENDSTDBRW";
    function opencrsbrowser(formname,uname,udom,desc,extra_element,multflag,type,type_elem) {
        var url = '/adm/pickcourse?';
        var formid = getFormIdByName(formname);
        var domainfilter = getDomainFromSelectbox(formname,udom);
        if (domainfilter != null) {
           if (domainfilter != '') {
               url += 'domainfilter='+domainfilter+'&';
	   }
        }
        url += 'form=' + formname + '&cnumelement='+uname+
	                            '&cdomelement='+udom+
                                    '&cnameelement='+desc;
        if (extra_element !=null && extra_element != '') {
            if (formname == 'rolechoice' || formname == 'studentform') {
                url += '&roleelement='+extra_element;
                if (domainfilter == null || domainfilter == '') {
                    url += '&domainfilter='+extra_element;
                }
            }
            else {
                if (formname == 'portform') {
                    url += '&setroles='+extra_element;
                } else {
                    if (formname == 'rules') {
                        url += '&fixeddom='+extra_element; 
                    }
                }
            }     
        }
        if (type != null && type != '') {
            url += '&type='+type;
        }
        if (type_elem != null && type_elem != '') {
            url += '&typeelement='+type_elem;
        }
        if (formname == 'ccrs') {
            var ownername = document.forms[formid].ccuname.value;
            var ownerdom =  document.forms[formid].ccdomain.options[document.forms[formid].ccdomain.selectedIndex].value;
            url += '&cloner='+ownername+':'+ownerdom;
            if (type == 'Course') {
                url += '&crscode='+document.forms[formid].crscode.value;
            }
        }
        if (formname == 'requestcrs') {
            url += '&crsdom=$domainfilter&crscode=$instcode';
        }
        if (multflag !=null && multflag != '') {
            url += '&multiple='+multflag;
        }
        var title = '$wintitle';
        var options = 'scrollbars=1,resizable=1,menubar=0';
        options += ',width=700,height=600';
        stdeditbrowser = open(url,title,options,'1');
        stdeditbrowser.focus();
    }
$id_functions
ENDSTDBRW
    if (($sec_element ne '') || ($role_element ne '') || ($credits_element ne '')) {
        $output .= &setsec_javascript($sec_element,$formname,$role_element,
                                      $credits_element);
    }
    $output .= '
// ]]>
</script>';
    return $output;
}

sub javascript_index_functions {
    return <<"ENDJS";

function getFormIdByName(formname) {
    for (var i=0;i<document.forms.length;i++) {
        if (document.forms[i].name == formname) {
            return i;
        }
    }
    return -1;
}

function getIndexByName(formid,item) {
    for (var i=0;i<document.forms[formid].elements.length;i++) {
        if (document.forms[formid].elements[i].name == item) {
            return i;
        }
    }
    return -1;
}

function getDomainFromSelectbox(formname,udom) {
    var userdom;
    var formid = getFormIdByName(formname);
    if (formid > -1) {
        var domid = getIndexByName(formid,udom);
        if (domid > -1) {
            if (document.forms[formid].elements[domid].type == 'select-one') {
                userdom=document.forms[formid].elements[domid].options[document.forms[formid].elements[domid].selectedIndex].value;
            }
            if (document.forms[formid].elements[domid].type == 'hidden') {
                userdom=document.forms[formid].elements[domid].value;
            }
        }
    }
    return userdom;
}

ENDJS

}

sub javascript_array_indexof {
    return <<ENDJS;
<script type="text/javascript" language="JavaScript">
// <![CDATA[

if (!Array.prototype.indexOf) {
    Array.prototype.indexOf = function (searchElement /*, fromIndex */ ) {
        "use strict";
        if (this === void 0 || this === null) {
            throw new TypeError();
        }
        var t = Object(this);
        var len = t.length >>> 0;
        if (len === 0) {
            return -1;
        }
        var n = 0;
        if (arguments.length > 0) {
            n = Number(arguments[1]);
            if (n !== n) { // shortcut for verifying if it's NaN
                n = 0;
            } else if (n !== 0 && n !== (1 / 0) && n !== -(1 / 0)) {
                n = (n > 0 || -1) * Math.floor(Math.abs(n));
            }
        }
        if (n >= len) {
            return -1;
        }
        var k = n >= 0 ? n : Math.max(len - Math.abs(n), 0);
        for (; k < len; k++) {
            if (k in t && t[k] === searchElement) {
                return k;
            }
        }
        return -1;
    }
}

// ]]>
</script>

ENDJS

}

sub userbrowser_javascript {
    my $id_functions = &javascript_index_functions();
    return <<"ENDUSERBRW";

function openuserbrowser(formname,uname,udom,ulast,ufirst,uemail,hideudom,crsdom,caller) {
    var url = '/adm/pickuser?';
    var userdom = getDomainFromSelectbox(formname,udom);
    if (userdom != null) {
       if (userdom != '') {
           url += 'srchdom='+userdom+'&';
       }
    }
    url += 'form=' + formname + '&unameelement='+uname+
                                '&udomelement='+udom+
                                '&ulastelement='+ulast+
                                '&ufirstelement='+ufirst+
                                '&uemailelement='+uemail+
                                '&hideudomelement='+hideudom+
                                '&coursedom='+crsdom;
    if ((caller != null) && (caller != undefined)) {
        url += '&caller='+caller;
    }
    var title = 'User_Browser';
    var options = 'scrollbars=1,resizable=1,menubar=0';
    options += ',width=700,height=600';
    var stdeditbrowser = open(url,title,options,'1');
    stdeditbrowser.focus();
}

function fix_domain (formname,udom,origdom,uname) {
    var formid = getFormIdByName(formname);
    if (formid > -1) {
        var unameid = getIndexByName(formid,uname);
        var domid = getIndexByName(formid,udom);
        var hidedomid = getIndexByName(formid,origdom);
        if (hidedomid > -1) {
            var fixeddom = document.forms[formid].elements[hidedomid].value;
            var unameval = document.forms[formid].elements[unameid].value;
            if ((fixeddom != '') && (fixeddom != undefined) && (fixeddom != null) && (unameval != '') && (unameval != undefined) && (unameval != null)) {
                if (domid > -1) {
                    var slct = document.forms[formid].elements[domid];
                    if (slct.type == 'select-one') {
                        var i;
                        for (i=0;i<slct.length;i++) {
                            if (slct.options[i].value==fixeddom) { slct.selectedIndex=i; }
                        }
                    }
                    if (slct.type == 'hidden') {
                        slct.value = fixeddom;
                    }
                }
            }
        }
    }
    return;
}

$id_functions
ENDUSERBRW
}

sub setsec_javascript {
    my ($sec_element,$formname,$role_element,$credits_element) = @_;
    my (@courserolenames,@communityrolenames,$rolestr,$courserolestr,
        $communityrolestr);
    if ($role_element ne '') {
        my @allroles = ('st','ta','ep','in','ad');
        foreach my $crstype ('Course','Community') {
            if ($crstype eq 'Community') {
                foreach my $role (@allroles) {
                    push(@communityrolenames,&Apache::lonnet::plaintext($role,$crstype));
                }
                push(@communityrolenames,&Apache::lonnet::plaintext('co'));
            } else {
                foreach my $role (@allroles) {
                    push(@courserolenames,&Apache::lonnet::plaintext($role,$crstype));
                }
                push(@courserolenames,&Apache::lonnet::plaintext('cc'));
            }
        }
        $rolestr = '"'.join('","',@allroles).'"';
        $courserolestr = '"'.join('","',@courserolenames).'"';
        $communityrolestr = '"'.join('","',@communityrolenames).'"';
    }
    my $setsections = qq|
function setSect(sectionlist) {
    var sectionsArray = new Array();
    if ((sectionlist != '') && (typeof sectionlist != "undefined")) {
        sectionsArray = sectionlist.split(",");
    }
    var numSections = sectionsArray.length;
    document.$formname.$sec_element.length = 0;
    if (numSections == 0) {
        document.$formname.$sec_element.multiple=false;
        document.$formname.$sec_element.size=1;
        document.$formname.$sec_element.options[0] = new Option('No existing sections','',false,false)
    } else {
        if (numSections == 1) {
            document.$formname.$sec_element.multiple=false;
            document.$formname.$sec_element.size=1;
            document.$formname.$sec_element.options[0] = new Option('Select','',true,true);
            document.$formname.$sec_element.options[1] = new Option('No section','',false,false)
            document.$formname.$sec_element.options[2] = new Option(sectionsArray[0],sectionsArray[0],false,false);
        } else {
            for (var i=0; i<numSections; i++) {
                document.$formname.$sec_element.options[i] = new Option(sectionsArray[i],sectionsArray[i],false,false)
            }
            document.$formname.$sec_element.multiple=true
            if (numSections < 3) {
                document.$formname.$sec_element.size=numSections;
            } else {
                document.$formname.$sec_element.size=3;
            }
            document.$formname.$sec_element.options[0].selected = false
        }
    }
}

function setRole(crstype) {
|;
    if ($role_element eq '') {
        $setsections .= '    return;
}
';
    } else {
        $setsections .= qq|
    var elementLength = document.$formname.$role_element.length;
    var allroles = Array($rolestr);
    var courserolenames = Array($courserolestr);
    var communityrolenames = Array($communityrolestr);
    if (elementLength != undefined) {
        if (document.$formname.$role_element.options[5].value == 'cc') {
            if (crstype == 'Course') {
                return;
            } else {
                allroles[5] = 'co';
                for (var i=0; i<6; i++) {
                    document.$formname.$role_element.options[i].value = allroles[i];
                    document.$formname.$role_element.options[i].text = communityrolenames[i];
                }
            }
        } else {
            if (crstype == 'Community') {
                return;
            } else {
                allroles[5] = 'cc';
                for (var i=0; i<6; i++) {
                    document.$formname.$role_element.options[i].value = allroles[i];
                    document.$formname.$role_element.options[i].text = courserolenames[i];
                }
            }
        }
    }
    return;
}
|;
    }
    if ($credits_element) {
        $setsections .= qq|
function setCredits(defaultcredits) {
    document.$formname.$credits_element.value = defaultcredits;
    return;
}
|;
    }
    return $setsections;
}

sub selectcourse_link {
   my ($form,$unameele,$udomele,$desc,$extra_element,$multflag,$selecttype,
       $typeelement) = @_;
   my $type = $selecttype;
   my $linktext = &mt('Select Course');
   if ($selecttype eq 'Community') {
       $linktext = &mt('Select Community');
   } elsif ($selecttype eq 'Course/Community') {
       $linktext = &mt('Select Course/Community');
       $type = '';
   } elsif ($selecttype eq 'Select') {
       $linktext = &mt('Select');
       $type = '';
   }
   return '<span class="LC_nobreak">'
         ."<a href='"
         .'javascript:opencrsbrowser("'.$form.'","'.$unameele
         .'","'.$udomele.'","'.$desc.'","'.$extra_element
         .'","'.$multflag.'","'.$type.'","'.$typeelement.'");'
         ."'>".$linktext.'</a>'
         .'</span>';
}

sub selectauthor_link {
   my ($form,$udom)=@_;
   return '<a href="javascript:openauthorbrowser('."'$form','$udom'".');">'.
          &mt('Select Author').'</a>';
}

sub selectuser_link {
    my ($form,$unameelem,$domelem,$lastelem,$firstelem,$emailelem,$hdomelem,
        $coursedom,$linktext,$caller) = @_;
    return '<a href="javascript:openuserbrowser('."'$form','$unameelem','$domelem',".
           "'$lastelem','$firstelem','$emailelem','$hdomelem','$coursedom','$caller'".
           ');">'.$linktext.'</a>';
}

sub check_uncheck_jscript {
    my $jscript = <<"ENDSCRT";
function checkAll(field) {
    if (field.length > 0) {
        for (i = 0; i < field.length; i++) {
            if (!field[i].disabled) {
                field[i].checked = true;
            }
        }
    } else {
        if (!field.disabled) {
            field.checked = true;
        }
    }
}
 
function uncheckAll(field) {
    if (field.length > 0) {
        for (i = 0; i < field.length; i++) {
            field[i].checked = false ;
        }
    } else {
        field.checked = false ;
    }
}
ENDSCRT
    return $jscript;
}

sub select_timezone {
   my ($name,$selected,$onchange,$includeempty,$disabled)=@_;
   my $output='<select name="'.$name.'" '.$onchange.$disabled.'>'."\n";
   if ($includeempty) {
       $output .= '<option value=""';
       if (($selected eq '') || ($selected eq 'local')) {
           $output .= ' selected="selected" ';
       }
       $output .= '> </option>';
   }
   my @timezones = DateTime::TimeZone->all_names;
   foreach my $tzone (@timezones) {
       $output.= '<option value="'.$tzone.'"';
       if ($tzone eq $selected) {
           $output.=' selected="selected"';
       }
       $output.=">$tzone</option>\n";
   }
   $output.="</select>";
   return $output;
}

sub select_datelocale {
    my ($name,$selected,$onchange,$includeempty,$disabled)=@_;
    my $output='<select name="'.$name.'" '.$onchange.$disabled.'>'."\n";
    if ($includeempty) {
        $output .= '<option value=""';
        if ($selected eq '') {
            $output .= ' selected="selected" ';
        }
        $output .= '> </option>';
    }
    my @languages = &Apache::lonlocal::preferred_languages();
    my (@possibles,%locale_names);
    my @locales = DateTime::Locale->ids();
    foreach my $id (@locales) {
        if ($id ne '') {
            my ($en_terr,$native_terr);
            my $loc = DateTime::Locale->load($id);
            if (ref($loc)) {
                $en_terr = $loc->name();
                $native_terr = $loc->native_name();
                if (grep(/^en$/,@languages) || !@languages) {
                    if ($en_terr ne '') {
                        $locale_names{$id} = '('.$en_terr.')';
                    } elsif ($native_terr ne '') {
                        $locale_names{$id} = $native_terr;
                    }
                } else {
                    if ($native_terr ne '') {
                        $locale_names{$id} = $native_terr.' ';
                    } elsif ($en_terr ne '') {
                        $locale_names{$id} = '('.$en_terr.')';
                    }
                }
                $locale_names{$id} = Encode::encode('UTF-8',$locale_names{$id});
                push(@possibles,$id);
            }
        }
    }
    foreach my $item (sort(@possibles)) {
        $output.= '<option value="'.$item.'"';
        if ($item eq $selected) {
            $output.=' selected="selected"';
        }
        $output.=">$item";
        if ($locale_names{$item} ne '') {
            $output.='  '.$locale_names{$item};
        }
        $output.="</option>\n";
    }
    $output.="</select>";
    return $output;
}

sub select_language {
    my ($name,$selected,$includeempty,$noedit) = @_;
    my %langchoices;
    if ($includeempty) {
        %langchoices = ('' => 'No language preference');
    }
    foreach my $id (&languageids()) {
        my $code = &supportedlanguagecode($id);
        if ($code) {
            $langchoices{$code} = &plainlanguagedescription($id);
        }
    }
    %langchoices = &Apache::lonlocal::texthash(%langchoices);
    return &select_form($selected,$name,\%langchoices,undef,$noedit);
}

=pod

=item * &linked_select_forms(...)

linked_select_forms returns a string containing a <script></script> block
and html for two <select> menus.  The select menus will be linked in that
changing the value of the first menu will result in new values being placed
in the second menu.  The values in the select menu will appear in alphabetical
order unless a defined order is provided.

linked_select_forms takes the following ordered inputs:

=over 4

=item * $formname, the name of the <form> tag

=item * $middletext, the text which appears between the <select> tags

=item * $firstdefault, the default value for the first menu

=item * $firstselectname, the name of the first <select> tag

=item * $secondselectname, the name of the second <select> tag

=item * $hashref, a reference to a hash containing the data for the menus.

=item * $menuorder, the order of values in the first menu

=item * $onchangefirst, additional javascript call to execute for an onchange
        event for the first <select> tag

=item * $onchangesecond, additional javascript call to execute for an onchange
        event for the second <select> tag

=back 

Below is an example of such a hash.  Only the 'text', 'default', and 
'select2' keys must appear as stated.  keys(%menu) are the possible 
values for the first select menu.  The text that coincides with the 
first menu value is given in $menu{$choice1}->{'text'}.  The values 
and text for the second menu are given in the hash pointed to by 
$menu{$choice1}->{'select2'}.  

 my %menu = ( A1 => { text =>"Choice A1" ,
                       default => "B3",
                       select2 => { 
                           B1 => "Choice B1",
                           B2 => "Choice B2",
                           B3 => "Choice B3",
                           B4 => "Choice B4"
                           },
                       order => ['B4','B3','B1','B2'],
                   },
               A2 => { text =>"Choice A2" ,
                       default => "C2",
                       select2 => { 
                           C1 => "Choice C1",
                           C2 => "Choice C2",
                           C3 => "Choice C3"
                           },
                       order => ['C2','C1','C3'],
                   },
               A3 => { text =>"Choice A3" ,
                       default => "D6",
                       select2 => { 
                           D1 => "Choice D1",
                           D2 => "Choice D2",
                           D3 => "Choice D3",
                           D4 => "Choice D4",
                           D5 => "Choice D5",
                           D6 => "Choice D6",
                           D7 => "Choice D7"
                           },
                       order => ['D4','D3','D2','D1','D7','D6','D5'],
                   }
               );

=cut

sub linked_select_forms {
    my ($formname,
        $middletext,
        $firstdefault,
        $firstselectname,
        $secondselectname, 
        $hashref,
        $menuorder,
        $onchangefirst,
        $onchangesecond
        ) = @_;
    my $second = "document.$formname.$secondselectname";
    my $first = "document.$formname.$firstselectname";
    # output the javascript to do the changing
    my $result = '';
    $result.='<script type="text/javascript" language="JavaScript">'."\n";
    $result.="// <![CDATA[\n";
    $result.="var select2data = new Object();\n";
    $" = '","';
    my $debug = '';
    foreach my $s1 (sort(keys(%$hashref))) {
        $result.="select2data.d_$s1 = new Object();\n";        
        $result.="select2data.d_$s1.def = new String('".
            $hashref->{$s1}->{'default'}."');\n";
        $result.="select2data.d_$s1.values = new Array(";
        my @s2values = sort(keys( %{ $hashref->{$s1}->{'select2'} } ));
        if (ref($hashref->{$s1}->{'order'}) eq 'ARRAY') {
            @s2values = @{$hashref->{$s1}->{'order'}};
        }
        $result.="\"@s2values\");\n";
        $result.="select2data.d_$s1.texts = new Array(";        
        my @s2texts;
        foreach my $value (@s2values) {
            push(@s2texts, $hashref->{$s1}->{'select2'}->{$value});
        }
        $result.="\"@s2texts\");\n";
    }
    $"=' ';
    $result.= <<"END";

function select1_changed() {
    // Determine new choice
    var newvalue = "d_" + $first.value;
    // update select2
    var values     = select2data[newvalue].values;
    var texts      = select2data[newvalue].texts;
    var select2def = select2data[newvalue].def;
    var i;
    // out with the old
    for (i = 0; i < $second.options.length; i++) {
        $second.options[i] = null;
    }
    // in with the nuclear
    for (i=0;i<values.length; i++) {
        $second.options[i] = new Option(values[i]);
        $second.options[i].value = values[i];
        $second.options[i].text = texts[i];
        if (values[i] == select2def) {
            $second.options[i].selected = true;
        }
    }
}
// ]]>
</script>
END
    # output the initial values for the selection lists
    $result .= "<select size=\"1\" name=\"$firstselectname\" onchange=\"select1_changed();$onchangefirst\">\n";
    my @order = sort(keys(%{$hashref}));
    if (ref($menuorder) eq 'ARRAY') {
        @order = @{$menuorder};
    }
    foreach my $value (@order) {
        $result.="    <option value=\"$value\" ";
        $result.=" selected=\"selected\" " if ($value eq $firstdefault);
        $result.=">".&mt($hashref->{$value}->{'text'})."</option>\n";
    }
    $result .= "</select>\n";
    my %select2 = %{$hashref->{$firstdefault}->{'select2'}};
    $result .= $middletext;
    $result .= "<select size=\"1\" name=\"$secondselectname\"";
    if ($onchangesecond) {
        $result .= ' onchange="'.$onchangesecond.'"';
    }
    $result .= ">\n";
    my $seconddefault = $hashref->{$firstdefault}->{'default'};
    
    my @secondorder = sort(keys(%select2));
    if (ref($hashref->{$firstdefault}->{'order'}) eq 'ARRAY') {
        @secondorder = @{$hashref->{$firstdefault}->{'order'}};
    }
    foreach my $value (@secondorder) {
        $result.="    <option value=\"$value\" ";        
        $result.=" selected=\"selected\" " if ($value eq $seconddefault);
        $result.=">".&mt($select2{$value})."</option>\n";
    }
    $result .= "</select>\n";
    #    return $debug;
    return $result;
}   #  end of sub linked_select_forms {

=pod

=item * &help_open_topic($topic,$text,$stayOnPage,$width,$height,$imgid)

Returns a string corresponding to an HTML link to the given help
$topic, where $topic corresponds to the name of a .tex file in
/home/httpd/html/adm/help/tex, with underscores replaced by
spaces. 

$text will optionally be linked to the same topic, allowing you to
link text in addition to the graphic. If you do not want to link
text, but wish to specify one of the later parameters, pass an
empty string. 

$stayOnPage is a value that will be interpreted as a boolean. If true,
the link will not open a new window. If false, the link will open
a new window using Javascript. (Default is false.) 

$width and $height are optional numerical parameters that will
override the width and height of the popped up window, which may
be useful for certain help topics with big pictures included.

$imgid is the id of the img tag used for the help icon. This may be
used in a javascript call to switch the image src.  See 
lonhtmlcommon::htmlareaselectactive() for an example.

=cut

sub help_open_topic {
    my ($topic, $text, $stayOnPage, $width, $height, $imgid) = @_;
    $text = "" if (not defined $text);
    $stayOnPage = 0 if (not defined $stayOnPage);
    $width = 500 if (not defined $width);
    $height = 400 if (not defined $height);
    my $filename = $topic;
    $filename =~ s/ /_/g;

    my $template = "";
    my $link;
    
    $topic=~s/\W/\_/g;

    if (!$stayOnPage) {
        if ($env{'browser.mobile'}) {
	    $link = "javascript:openMyModal('/adm/help/${filename}.hlp',$width,$height,'yes');";
        } else {
            $link = "javascript:void(open('/adm/help/${filename}.hlp', 'Help_for_$topic', 'menubar=0,toolbar=1,scrollbars=1,width=$width,height=$height,resizable=yes'))";
        }
    } elsif ($stayOnPage eq 'popup') {
        $link = "javascript:void(open('/adm/help/${filename}.hlp', 'Help_for_$topic', 'menubar=0,toolbar=1,scrollbars=1,width=$width,height=$height,resizable=yes'))";
    } else {
	$link = "/adm/help/${filename}.hlp";
    }

    # Add the text
    if ($text ne "") {	
	$template.='<span class="LC_help_open_topic">'
                  .'<a target="_top" href="'.$link.'">'
                  .$text.'</a>';
    }

    # (Always) Add the graphic
    my $title = &mt('Online Help');
    my $helpicon=&lonhttpdurl("/adm/help/help.png");
    if ($imgid ne '') {
        $imgid = ' id="'.$imgid.'"';
    }
    $template.=' <a target="_top" href="'.$link.'" title="'.$title.'">'
              .'<img src="'.$helpicon.'" border="0"'
              .' alt="'.&mt('Help: [_1]',$topic).'"'
              .' title="'.$title.'" style="vertical-align:middle;"'.$imgid 
              .' /></a>';
    if ($text ne "") {	
        $template.='</span>';
    }
    return $template;

}

# This is a quicky function for Latex cheatsheet editing, since it 
# appears in at least four places
sub helpLatexCheatsheet {
    my ($topic,$text,$not_author,$stayOnPage) = @_;
    my $out;
    my $addOther = '';
    if ($topic) {
	$addOther = '<span>'.&help_open_topic($topic,&mt($text),$stayOnPage, undef, 600).'</span> ';
    }
    $out = '<span>' # Start cheatsheet
	  .$addOther
          .'<span>'
	  .&help_open_topic('Greek_Symbols',&mt('Greek Symbols'),$stayOnPage,undef,600)
	  .'</span> <span>'
	  .&help_open_topic('Other_Symbols',&mt('Other Symbols'),$stayOnPage,undef,600)
	  .'</span>';
    unless ($not_author) {
        $out .= ' <span>'
	       .&help_open_topic('Authoring_Output_Tags',&mt('Output Tags'),$stayOnPage,undef,600)
	       .'</span> <span>'
               .&help_open_topic('Authoring_Multilingual_Problems',&mt('Languages'),$stayOnPage,undef,600)
               .'</span>';
    }
    $out .= '</span>'; # End cheatsheet
    return $out;
}

sub general_help {
    my $helptopic='Student_Intro';
    if ($env{'request.role'}=~/^(ca|au)/) {
	$helptopic='Authoring_Intro';
    } elsif ($env{'request.role'}=~/^(cc|co)/) {
	$helptopic='Course_Coordination_Intro';
    } elsif ($env{'request.role'}=~/^dc/) {
        $helptopic='Domain_Coordination_Intro';
    }
    return $helptopic;
}

sub update_help_link {
    my ($topic,$component_help,$faq,$bug,$stayOnPage) = @_;
    my $origurl = $ENV{'REQUEST_URI'};
    $origurl=~s|^/~|/priv/|;
    my $timestamp = time;
    foreach my $datum (\$topic,\$component_help,\$faq,\$bug,\$origurl) {
        $$datum = &escape($$datum);
    }

    my $banner_link = "/adm/helpmenu?page=banner&amp;topic=$topic&amp;component_help=$component_help&amp;faq=$faq&amp;bug=$bug&amp;origurl=$origurl&amp;stamp=$timestamp&amp;stayonpage=$stayOnPage";
    my $output .= <<"ENDOUTPUT";
<script type="text/javascript">
// <![CDATA[
banner_link = '$banner_link';
// ]]>
</script>
ENDOUTPUT
    return $output;
}

# now just updates the help link and generates a blue icon
sub help_open_menu {
    my ($topic,$component_help,$faq,$bug,$stayOnPage,$width,$height,$text) 
	= @_;    
    $stayOnPage = 1;
    my $output;
    if ($component_help) {
	if (!$text) {
	    $output=&help_open_topic($component_help,undef,$stayOnPage,
				       $width,$height);
	} else {
	    my $help_text;
	    $help_text=&unescape($topic);
	    $output='<table><tr><td>'.
		&help_open_topic($component_help,$help_text,$stayOnPage,
				 $width,$height).'</td></tr></table>';
	}
    }
    my $banner_link = &update_help_link($topic,$component_help,$faq,$bug,$stayOnPage);
    return $output.$banner_link;
}

sub top_nav_help {
    my ($text,$linkattr) = @_;
    $text = &mt($text);
    my $stay_on_page;
    unless ($env{'environment.remote'} eq 'on') {
        $stay_on_page = 1;
    }
    my ($link,$banner_link);
    unless ($env{'request.noversionuri'} =~ m{^/adm/helpmenu}) {
        $link = ($stay_on_page) ? "javascript:helpMenu('display')"
	                         : "javascript:helpMenu('open')";
        $banner_link = &update_help_link(undef,undef,undef,undef,$stay_on_page);
    }
    my $title = &mt('Get help');
    if ($link) {
        return <<"END";
$banner_link
<a href="$link" title="$title" $linkattr>$text</a>
END
    } else {
        return '&nbsp;'.$text.'&nbsp;';
    }
}

sub help_menu_js {
    my ($httphost) = @_;
    my $stayOnPage = 1;
    my $width = 620;
    my $height = 600;
    my $helptopic=&general_help();
    my $details_link = $httphost.'/adm/help/'.$helptopic.'.hlp';
    my $nothing=&Apache::lonhtmlcommon::javascript_nothing();
    my $start_page =
        &Apache::loncommon::start_page('Help Menu', undef,
				       {'frameset'    => 1,
					'js_ready'    => 1,
                                        'use_absolute' => $httphost,
					'add_entries' => {
					    'border' => '0',
					    'rows'   => "110,*",},});
    my $end_page =
        &Apache::loncommon::end_page({'frameset' => 1,
				      'js_ready' => 1,});

    my $template .= <<"ENDTEMPLATE";
<script type="text/javascript">
// <![CDATA[
// <!-- BEGIN LON-CAPA Internal
var banner_link = '';
function helpMenu(target) {
    var caller = this;
    if (target == 'open') {
        var newWindow = null;
        try {
            newWindow =  window.open($nothing,"helpmenu","HEIGHT=$height,WIDTH=$width,resizable=yes,scrollbars=yes" )
        }
        catch(error) {
            writeHelp(caller);
            return;
        }
        if (newWindow) {
            caller = newWindow;
        }
    }
    writeHelp(caller);
    return;
}
function writeHelp(caller) {
    caller.document.writeln('$start_page\\n<frame name="bannerframe" src="'+banner_link+'" marginwidth="0" marginheight="0" frameborder="0">\\n');
    caller.document.writeln('<frame name="bodyframe" src="$details_link" marginwidth="0" marginheight="0" frameborder="0">\\n$end_page');
    caller.document.close();
    caller.focus();
}
// END LON-CAPA Internal -->
// ]]>
</script>
ENDTEMPLATE
    return $template;
}

sub help_open_bug {
    my ($topic, $text, $stayOnPage, $width, $height) = @_;
    unless ($env{'user.adv'}) { return ''; }
    unless ($Apache::lonnet::perlvar{'BugzillaHost'}) { return ''; }
    $text = "" if (not defined $text);
	$stayOnPage=1;
    $width = 600 if (not defined $width);
    $height = 600 if (not defined $height);

    $topic=~s/\W+/\+/g;
    my $link='';
    my $template='';
    my $url=$Apache::lonnet::perlvar{'BugzillaHost'}.'enter_bug.cgi?product=LON-CAPA&amp;bug_file_loc='.
	&escape($ENV{'REQUEST_URI'}).'&amp;component='.$topic;
    if (!$stayOnPage)
    {
	$link = "javascript:void(open('$url', 'Bugzilla', 'menubar=0,toolbar=1,scrollbars=1,width=$width,height=$height,resizable=yes'))";
    }
    else
    {
	$link = $url;
    }
    # Add the text
    if ($text ne "")
    {
	$template .= 
  "<table bgcolor='#AA3333' cellspacing='1' cellpadding='1' border='0'><tr>".
  "<td bgcolor='#FF5555'><a target=\"_top\" href=\"$link\"><span style=\"color:#FFFFFF;font-size:10pt;\">$text</span></a>";
    }

    # Add the graphic
    my $title = &mt('Report a Bug');
    my $bugicon=&lonhttpdurl("/adm/lonMisc/smallBug.gif");
    $template .= <<"ENDTEMPLATE";
 <a target="_top" href="$link" title="$title"><img src="$bugicon" border="0" alt="(Bug: $topic)" /></a>
ENDTEMPLATE
    if ($text ne '') { $template.='</td></tr></table>' };
    return $template;

}

sub help_open_faq {
    my ($topic, $text, $stayOnPage, $width, $height) = @_;
    unless ($env{'user.adv'}) { return ''; }
    unless ($Apache::lonnet::perlvar{'FAQHost'}) { return ''; }
    $text = "" if (not defined $text);
	$stayOnPage=1;
    $width = 350 if (not defined $width);
    $height = 400 if (not defined $height);

    $topic=~s/\W+/\+/g;
    my $link='';
    my $template='';
    my $url=$Apache::lonnet::perlvar{'FAQHost'}.'/fom/cache/'.$topic.'.html';
    if (!$stayOnPage)
    {
	$link = "javascript:void(open('$url', 'FAQ-O-Matic', 'menubar=0,toolbar=1,scrollbars=1,width=$width,height=$height,resizable=yes'))";
    }
    else
    {
	$link = $url;
    }

    # Add the text
    if ($text ne "")
    {
	$template .= 
  "<table bgcolor='#337733' cellspacing='1' cellpadding='1' border='0'><tr>".
  "<td bgcolor='#448844'><a target=\"_top\" href=\"$link\"><span style=\"color:#FFFFFF; font-size:10pt;\">$text</span></a>";
    }

    # Add the graphic
    my $title = &mt('View the FAQ');
    my $faqicon=&lonhttpdurl("/adm/lonMisc/smallFAQ.gif");
    $template .= <<"ENDTEMPLATE";
 <a target="_top" href="$link" title="$title"><img src="$faqicon" border="0" alt="(FAQ: $topic)" /></a>
ENDTEMPLATE
    if ($text ne '') { $template.='</td></tr></table>' };
    return $template;

}

###############################################################
###############################################################

=pod

=item * &change_content_javascript():

This and the next function allow you to create small sections of an
otherwise static HTML page that you can update on the fly with
Javascript, even in Netscape 4.

The Javascript fragment returned by this function (no E<lt>scriptE<gt> tag)
must be written to the HTML page once. It will prove the Javascript
function "change(name, content)". Calling the change function with the
name of the section 
you want to update, matching the name passed to C<changable_area>, and
the new content you want to put in there, will put the content into
that area.

B<Note>: Netscape 4 only reserves enough space for the changable area
to contain room for the original contents. You need to "make space"
for whatever changes you wish to make, and be B<sure> to check your
code in Netscape 4. This feature in Netscape 4 is B<not> powerful;
it's adequate for updating a one-line status display, but little more.
This script will set the space to 100% width, so you only need to
worry about height in Netscape 4.

Modern browsers are much less limiting, and if you can commit to the
user not using Netscape 4, this feature may be used freely with
pretty much any HTML.

=cut

sub change_content_javascript {
    # If we're on Netscape 4, we need to use Layer-based code
    if ($env{'browser.type'} eq 'netscape' &&
	$env{'browser.version'} =~ /^4\./) {
	return (<<NETSCAPE4);
	function change(name, content) {
	    doc = document.layers[name+"___escape"].layers[0].document;
	    doc.open();
	    doc.write(content);
	    doc.close();
	}
NETSCAPE4
    } else {
	# Otherwise, we need to use semi-standards-compliant code
	# (technically, "innerHTML" isn't standard but the equivalent
	# is really scary, and every useful browser supports it
	return (<<DOMBASED);
	function change(name, content) {
	    element = document.getElementById(name);
	    element.innerHTML = content;
	}
DOMBASED
    }
}

=pod

=item * &changable_area($name,$origContent):

This provides a "changable area" that can be modified on the fly via
the Javascript code provided in C<change_content_javascript>. $name is
the name you will use to reference the area later; do not repeat the
same name on a given HTML page more then once. $origContent is what
the area will originally contain, which can be left blank.

=cut

sub changable_area {
    my ($name, $origContent) = @_;

    if ($env{'browser.type'} eq 'netscape' &&
	$env{'browser.version'} =~ /^4\./) {
	# If this is netscape 4, we need to use the Layer tag
	return "<ilayer width='100%' id='${name}___escape' overflow='none'><layer width='100%' id='$name' overflow='none'>$origContent</layer></ilayer>";
    } else {
	return "<span id='$name'>$origContent</span>";
    }
}

=pod

=item * &viewport_geometry_js 

Provides javascript object (Geometry) which can provide information about the viewport geometry for the client browser.

=cut


sub viewport_geometry_js { 
    return <<"GEOMETRY";
var Geometry = {};
function init_geometry() {
    if (Geometry.init) { return };
    Geometry.init=1;
    if (window.innerHeight) {
        Geometry.getViewportHeight   = function() { return window.innerHeight; };
        Geometry.getViewportWidth   = function() { return window.innerWidth; };
        Geometry.getHorizontalScroll = function() { return window.pageXOffset; };
        Geometry.getVerticalScroll   = function() { return window.pageYOffset; };
    }
    else if (document.documentElement && document.documentElement.clientHeight) {
        Geometry.getViewportHeight =
            function() { return document.documentElement.clientHeight; };
        Geometry.getViewportWidth =
            function() { return document.documentElement.clientWidth; };

        Geometry.getHorizontalScroll =
            function() { return document.documentElement.scrollLeft; };
        Geometry.getVerticalScroll =
            function() { return document.documentElement.scrollTop; };
    }
    else if (document.body.clientHeight) {
        Geometry.getViewportHeight =
            function() { return document.body.clientHeight; };
        Geometry.getViewportWidth =
            function() { return document.body.clientWidth; };
        Geometry.getHorizontalScroll =
            function() { return document.body.scrollLeft; };
        Geometry.getVerticalScroll =
            function() { return document.body.scrollTop; };
    }
}

GEOMETRY
}

=pod

=item * &viewport_size_js()

Provides a javascript function to set values of two form elements - width and height (elements are passed in as arguments to the javascript function) to the dimensions of the user's browser window. 

=cut

sub viewport_size_js {
    my $geometry = &viewport_geometry_js();
    return <<"DIMS";

$geometry

function getViewportDims(width,height) {
    init_geometry();
    width.value = Geometry.getViewportWidth();
    height.value = Geometry.getViewportHeight();
    return;
}

DIMS
}

=pod

=item * &resize_textarea_js()

emits the needed javascript to resize a textarea to be as big as possible

creates a function resize_textrea that takes two IDs first should be
the id of the element to resize, second should be the id of a div that
surrounds everything that comes after the textarea, this routine needs
to be attached to the <body> for the onload and onresize events.

=back

=cut

sub resize_textarea_js {
    my $geometry = &viewport_geometry_js();
    return <<"RESIZE";
    <script type="text/javascript">
// <![CDATA[
$geometry

function getX(element) {
    var x = 0;
    while (element) {
	x += element.offsetLeft;
	element = element.offsetParent;
    }
    return x;
}
function getY(element) {
    var y = 0;
    while (element) {
	y += element.offsetTop;
	element = element.offsetParent;
    }
    return y;
}


function resize_textarea(textarea_id,bottom_id) {
    init_geometry();
    var textarea        = document.getElementById(textarea_id);
    //alert(textarea);

    var textarea_top    = getY(textarea);
    var textarea_height = textarea.offsetHeight;
    var bottom          = document.getElementById(bottom_id);
    var bottom_top      = getY(bottom);
    var bottom_height   = bottom.offsetHeight;
    var window_height   = Geometry.getViewportHeight();
    var fudge           = 23;
    var new_height      = window_height-fudge-textarea_top-bottom_height;
    if (new_height < 300) {
	new_height = 300;
    }
    textarea.style.height=new_height+'px';
}
// ]]>
</script>
RESIZE

}

sub colorfuleditor_js {
    return <<"COLORFULEDIT"
<script type="text/javascript">
// <![CDATA[>
    function fold_box(curDepth, lastresource){

    // we need a list because there can be several blocks you need to fold in one tag
        var block = document.getElementsByName('foldblock_'+curDepth);
    // but there is only one folding button per tag
        var foldbutton = document.getElementById('folding_btn_'+curDepth);

        if(block.item(0).style.display == 'none'){

            foldbutton.value = '@{[&mt("Hide")]}';
            for (i = 0; i < block.length; i++){
                block.item(i).style.display = '';
            }
        }else{

            foldbutton.value = '@{[&mt("Show")]}';
            for (i = 0; i < block.length; i++){
                // block.item(i).style.visibility = 'collapse';
                block.item(i).style.display = 'none';
            }
        };
        saveState(lastresource);
    }

    function saveState (lastresource) {

        var tag_list = getTagList();
        if(tag_list != null){
            var timestamp = new Date().getTime();
            var key = lastresource;

            // the value pattern is: 'time;key1,value1;key2,value2; ... '
            // starting with timestamp
            var value = timestamp+';';

            // building the list of key-value pairs
            for(var i = 0; i < tag_list.length; i++){
                value += tag_list[i]+',';
                value += document.getElementsByName(tag_list[i])[0].style.display+';';
            }

            // only iterate whole storage if nothing to override
            if(localStorage.getItem(key) == null){

                // prevent storage from growing large
                if(localStorage.length > 50){
                    var regex_getTimestamp = /^(?:\d)+;/;
                    var oldest_timestamp = regex_getTimestamp.exec(localStorage.key(0));
                    var oldest_key;

                    for(var i = 1; i < localStorage.length; i++){
                        if (regex_getTimestamp.exec(localStorage.key(i)) < oldest_timestamp) {
                            oldest_key = localStorage.key(i);
                            oldest_timestamp = regex_getTimestamp.exec(oldest_key);
                        }
                    }
                    localStorage.removeItem(oldest_key);
                }
            }
            localStorage.setItem(key,value);
        }
    }

    // restore folding status of blocks (on page load)
    function restoreState (lastresource) {
        if(localStorage.getItem(lastresource) != null){
            var key = lastresource;
            var value = localStorage.getItem(key);
            var regex_delTimestamp = /^\d+;/;

            value.replace(regex_delTimestamp, '');

            var valueArr = value.split(';');
            var pairs;
            var elements;
            for (var i = 0; i < valueArr.length; i++){
                pairs = valueArr[i].split(',');
                elements = document.getElementsByName(pairs[0]);

                for (var j = 0; j < elements.length; j++){
                    elements[j].style.display = pairs[1];
                    if (pairs[1] == "none"){
                        var regex_id = /([_\\d]+)\$/;
                        regex_id.exec(pairs[0]);
                        document.getElementById("folding_btn"+RegExp.\$1).value = "Show";
                    }
                }
            }
        }
    }

    function getTagList () {

        var stringToSearch = document.lonhomework.innerHTML;

        var ret = new Array();
        var regex_findBlock = /(foldblock_.*?)"/g;
        var tag_list = stringToSearch.match(regex_findBlock);

        if(tag_list != null){
            for(var i = 0; i < tag_list.length; i++){
                ret.push(tag_list[i].replace(/"/, ''));
            }
        }
        return ret;
    }

    function saveScrollPosition (resource) {
        var tag_list = getTagList();

        // we dont always want to jump to the first block
        // 170 is roughly above the "Problem Editing" header. we just want to save if the user scrolled down further than this
        if(\$(window).scrollTop() > 170){
            if(tag_list != null){
                var result;
                for(var i = 0; i < tag_list.length; i++){
                    if(isElementInViewport(tag_list[i])){
                        result += tag_list[i]+';';
                    }
                }
                sessionStorage.setItem('anchor_'+resource, result);
            }
        } else {
            // we dont need to save zero, just delete the item to leave everything tidy
            sessionStorage.removeItem('anchor_'+resource);
        }
    }

    function restoreScrollPosition(resource){

        var elem = sessionStorage.getItem('anchor_'+resource);
        if(elem != null){
            var tag_list = elem.split(';');
            var elem_list;

            for(var i = 0; i < tag_list.length; i++){
                elem_list = document.getElementsByName(tag_list[i]);

                if(elem_list.length > 0){
                    elem = elem_list[0];
                    break;
                }
            }
            elem.scrollIntoView();
        }
    }

    function isElementInViewport(el) {

        // change to last element instead of first
        var elem = document.getElementsByName(el);
        var rect = elem[0].getBoundingClientRect();

        return (
            rect.top >= 0 &&
            rect.left >= 0 &&
            rect.bottom <= (window.innerHeight || document.documentElement.clientHeight) && /*or $(window).height() */
            rect.right <= (window.innerWidth || document.documentElement.clientWidth) /*or $(window).width() */
        );
    }

    function autosize(depth){
        var cmInst = window['cm'+depth];
        var fitsizeButton = document.getElementById('fitsize'+depth);

        // is fixed size, switching to dynamic
        if (sessionStorage.getItem("autosized_"+depth) == null) {
            cmInst.setSize("","auto");
            fitsizeButton.value = "@{[&mt('Fixed size')]}";
            sessionStorage.setItem("autosized_"+depth, "yes");

        // is dynamic size, switching to fixed
        } else {
            cmInst.setSize("","300px");
            fitsizeButton.value = "@{[&mt('Dynamic size')]}";
            sessionStorage.removeItem("autosized_"+depth);
        }
    }



// ]]>
</script>
COLORFULEDIT
}

sub xmleditor_js {
    return <<XMLEDIT
<script type="text/javascript" src="/adm/jQuery/addons/jquery-scrolltofixed.js"></script>
<script type="text/javascript">
// <![CDATA[>

    function saveScrollPosition (resource) {

        var scrollPos = \$(window).scrollTop();
        sessionStorage.setItem(resource,scrollPos);
    }

    function restoreScrollPosition(resource){

        var scrollPos = sessionStorage.getItem(resource);
        \$(window).scrollTop(scrollPos);
    }

    // unless internet explorer
    if (!(window.navigator.appName == "Microsoft Internet Explorer" && (document.documentMode || document.compatMode))){

        \$(document).ready(function() {
             \$(".LC_edit_actionbar").scrollToFixed(\{zIndex: 100\});
        });
    }

    // inserts text at cursor position into codemirror (xml editor only)
    function insertText(text){
        cm.focus();
        var curPos = cm.getCursor();
        cm.replaceRange(text.replace(/ESCAPEDSCRIPT/g,'script'), {line: curPos.line,ch: curPos.ch});
    }
// ]]>
</script>
XMLEDIT
}

sub insert_folding_button {
    my $curDepth = $Apache::lonxml::curdepth;
    my $lastresource = $env{'request.ambiguous'};

    return "<input type=\"button\" id=\"folding_btn_$curDepth\"
            value=\"".&mt('Hide')."\" onclick=\"fold_box('$curDepth','$lastresource')\">";
}


=pod

=head1 Excel and CSV file utility routines

=cut

###############################################################
###############################################################

=pod

=over 4

=item * &csv_translate($text) 

Translate $text to allow it to be output as a 'comma separated values' 
format.

=cut

###############################################################
###############################################################
sub csv_translate {
    my $text = shift;
    $text =~ s/\"/\"\"/g;
    $text =~ s/\n/ /g;
    return $text;
}

###############################################################
###############################################################

=pod

=item * &define_excel_formats()

Define some commonly used Excel cell formats.

Currently supported formats:

=over 4

=item header

=item bold

=item h1

=item h2

=item h3

=item h4

=item i

=item date

=back

Inputs: $workbook

Returns: $format, a hash reference.


=cut

###############################################################
###############################################################
sub define_excel_formats {
    my ($workbook) = @_;
    my $format;
    $format->{'header'} = $workbook->add_format(bold      => 1, 
                                                bottom    => 1,
                                                align     => 'center');
    $format->{'bold'} = $workbook->add_format(bold=>1);
    $format->{'h1'}   = $workbook->add_format(bold=>1, size=>18);
    $format->{'h2'}   = $workbook->add_format(bold=>1, size=>16);
    $format->{'h3'}   = $workbook->add_format(bold=>1, size=>14);
    $format->{'h4'}   = $workbook->add_format(bold=>1, size=>12);
    $format->{'i'}    = $workbook->add_format(italic=>1);
    $format->{'date'} = $workbook->add_format(num_format=>
                                            'mm/dd/yyyy hh:mm:ss');
    return $format;
}

###############################################################
###############################################################

=pod

=item * &create_workbook()

Create an Excel worksheet.  If it fails, output message on the
request object and return undefs.

Inputs: Apache request object

Returns (undef) on failure, 
    Excel worksheet object, scalar with filename, and formats 
    from &Apache::loncommon::define_excel_formats on success

=cut

###############################################################
###############################################################
sub create_workbook {
    my ($r) = @_;
        #
    # Create the excel spreadsheet
    my $filename = '/prtspool/'.
        $env{'user.name'}.'_'.$env{'user.domain'}.'_'.
        time.'_'.rand(1000000000).'.xls';
    my $workbook  = Spreadsheet::WriteExcel->new('/home/httpd'.$filename);
    if (! defined($workbook)) {
        $r->log_error("Error creating excel spreadsheet $filename: $!");
        $r->print(
            '<p class="LC_error">'
           .&mt('Problems occurred in creating the new Excel file.')
           .' '.&mt('This error has been logged.')
           .' '.&mt('Please alert your LON-CAPA administrator.')
           .'</p>'
        );
        return (undef);
    }
    #
    $workbook->set_tempdir(LONCAPA::tempdir());
    #
    my $format = &Apache::loncommon::define_excel_formats($workbook);
    return ($workbook,$filename,$format);
}

###############################################################
###############################################################

=pod

=item * &create_text_file()

Create a file to write to and eventually make available to the user.
If file creation fails, outputs an error message on the request object and 
return undefs.

Inputs: Apache request object, and file suffix

Returns (undef) on failure, 
    Filehandle and filename on success.

=cut

###############################################################
###############################################################
sub create_text_file {
    my ($r,$suffix) = @_;
    if (! defined($suffix)) { $suffix = 'txt'; };
    my $fh;
    my $filename = '/prtspool/'.
        $env{'user.name'}.'_'.$env{'user.domain'}.'_'.
        time.'_'.rand(1000000000).'.'.$suffix;
    $fh = Apache::File->new('>/home/httpd'.$filename);
    if (! defined($fh)) {
        $r->log_error("Couldn't open $filename for output $!");
        $r->print(
            '<p class="LC_error">'
           .&mt('Problems occurred in creating the output file.')
           .' '.&mt('This error has been logged.')
           .' '.&mt('Please alert your LON-CAPA administrator.')
           .'</p>'
        );
    }
    return ($fh,$filename)
}


=pod 

=back

=cut

###############################################################
##        Home server <option> list generating code          ##
###############################################################

# ------------------------------------------

sub domain_select {
    my ($name,$value,$multiple)=@_;
    my %domains=map { 
	$_ => $_.' '. &Apache::lonnet::domain($_,'description') 
    } &Apache::lonnet::all_domains();
    if ($multiple) {
	$domains{''}=&mt('Any domain');
	$domains{'select_form_order'} = [sort {lc($a) cmp lc($b) } (keys(%domains))];
	return &multiple_select_form($name,$value,4,\%domains);
    } else {
	$domains{'select_form_order'} = [sort {lc($a) cmp lc($b) } (keys(%domains))];
	return &select_form($name,$value,\%domains);
    }
}

#-------------------------------------------

=pod

=head1 Routines for form select boxes

=over 4

=item * &multiple_select_form($name,$value,$size,$hash,$order)

Returns a string containing a <select> element int multiple mode


Args:
  $name - name of the <select> element
  $value - scalar or array ref of values that should already be selected
  $size - number of rows long the select element is
  $hash - the elements should be 'option' => 'shown text'
          (shown text should already have been &mt())
  $order - (optional) array ref of the order to show the elements in

=cut

#-------------------------------------------
sub multiple_select_form {
    my ($name,$value,$size,$hash,$order)=@_;
    my %selected = map { $_ => 1 } ref($value)?@{$value}:($value);
    my $output='';
    if (! defined($size)) {
        $size = 4;
        if (scalar(keys(%$hash))<4) {
            $size = scalar(keys(%$hash));
        }
    }
    $output.="\n".'<select name="'.$name.'" size="'.$size.'" multiple="multiple">';
    my @order;
    if (ref($order) eq 'ARRAY')  {
        @order = @{$order};
    } else {
        @order = sort(keys(%$hash));
    }
    if (exists($$hash{'select_form_order'})) {
        @order = @{$$hash{'select_form_order'}};
    }
        
    foreach my $key (@order) {
        $output.='<option value="'.&HTML::Entities::encode($key,'"<>&').'" ';
        $output.='selected="selected" ' if ($selected{$key});
        $output.='>'.$hash->{$key}."</option>\n";
    }
    $output.="</select>\n";
    return $output;
}

#-------------------------------------------

=pod

=item * &select_form($defdom,$name,$hashref,$onchange,$readonly)

Returns a string containing a <select name='$name' size='1'> form to 
allow a user to select options from a ref to a hash containing:
option_name => displayed text. An optional $onchange can include
a javascript onchange item, e.g., onchange="this.form.submit();".
An optional arg -- $readonly -- if true will cause the select form
to be disabled, e.g., for the case where an instructor has a section-
specific role, and is viewing/modifying parameters.  

See lonrights.pm for an example invocation and use.

=cut

#-------------------------------------------
sub select_form {
    my ($def,$name,$hashref,$onchange,$readonly) = @_;
    return unless (ref($hashref) eq 'HASH');
    if ($onchange) {
        $onchange = ' onchange="'.$onchange.'"';
    }
    my $disabled;
    if ($readonly) {
        $disabled = ' disabled="disabled"';
    }
    my $selectform = "<select name=\"$name\" size=\"1\"$onchange$disabled>\n";
    my @keys;
    if (exists($hashref->{'select_form_order'})) {
	@keys=@{$hashref->{'select_form_order'}};
    } else {
	@keys=sort(keys(%{$hashref}));
    }
    foreach my $key (@keys) {
        $selectform.=
	    '<option value="'.&HTML::Entities::encode($key,'"<>&').'" '.
            ($key eq $def ? 'selected="selected" ' : '').
                ">".$hashref->{$key}."</option>\n";
    }
    $selectform.="</select>";
    return $selectform;
}

# For display filters

sub display_filter {
    my ($context) = @_;
    if (!$env{'form.show'}) { $env{'form.show'}=10; }
    if (!$env{'form.displayfilter'}) { $env{'form.displayfilter'}='currentfolder'; }
    my $phraseinput = 'hidden';
    my $includeinput = 'hidden';
    my ($checked,$includetypestext);
    if ($env{'form.displayfilter'} eq 'containing') {
        $phraseinput = 'text'; 
        if ($context eq 'parmslog') {
            $includeinput = 'checkbox';
            if ($env{'form.includetypes'}) {
                $checked = ' checked="checked"';
            }
            $includetypestext = &mt('Include parameter types');
        }
    } else {
        $includetypestext = '&nbsp;';
    }
    my ($additional,$secondid,$thirdid);
    if ($context eq 'parmslog') {
        $additional = 
            '<label><input type="'.$includeinput.'" name="includetypes"'. 
            $checked.' name="includetypes" value="1" id="includetypes" />'.
            '&nbsp;<span id="includetypestext">'.$includetypestext.'</span>'.
            '</label>';
        $secondid = 'includetypes';
        $thirdid = 'includetypestext';
    }
    my $onchange = "javascript:toggleHistoryOptions(this,'containingphrase','$context',
                                                    '$secondid','$thirdid')";
    return '<span class="LC_nobreak"><label>'.&mt('Records: [_1]',
			       &Apache::lonmeta::selectbox('show',$env{'form.show'},undef,
							   (&mt('all'),10,20,50,100,1000,10000))).
	   '</label></span> <span class="LC_nobreak">'.
           &mt('Filter: [_1]',
	   &select_form($env{'form.displayfilter'},
			'displayfilter',
			{'currentfolder' => 'Current folder/page',
			 'containing' => 'Containing phrase',
			 'none' => 'None'},$onchange)).'&nbsp;'.
			 '<input type="'.$phraseinput.'" name="containingphrase" id="containingphrase" size="30" value="'.
                         &HTML::Entities::encode($env{'form.containingphrase'}).
                         '" />'.$additional;
}

sub display_filter_js {
    my $includetext = &mt('Include parameter types');
    return <<"ENDJS";
  
function toggleHistoryOptions(setter,firstid,context,secondid,thirdid) {
    var firstType = 'hidden';
    if (setter.options[setter.selectedIndex].value == 'containing') {
        firstType = 'text';
    }
    firstObject = document.getElementById(firstid);
    if (typeof(firstObject) == 'object') {
        if (firstObject.type != firstType) {
            changeInputType(firstObject,firstType);
        }
    }
    if (context == 'parmslog') {
        var secondType = 'hidden';
        if (firstType == 'text') {
            secondType = 'checkbox';
        }
        secondObject = document.getElementById(secondid);  
        if (typeof(secondObject) == 'object') {
            if (secondObject.type != secondType) {
                changeInputType(secondObject,secondType);
            }
        }
        var textItem = document.getElementById(thirdid);
        var currtext = textItem.innerHTML;
        var newtext;
        if (firstType == 'text') {
            newtext = '$includetext';
        } else {
            newtext = '&nbsp;';
        }
        if (currtext != newtext) {
            textItem.innerHTML = newtext;
        }
    }
    return;
}

function changeInputType(oldObject,newType) {
    var newObject = document.createElement('input');
    newObject.type = newType;
    if (oldObject.size) {
        newObject.size = oldObject.size;
    }
    if (oldObject.value) {
        newObject.value = oldObject.value;
    }
    if (oldObject.name) {
        newObject.name = oldObject.name;
    }
    if (oldObject.id) {
        newObject.id = oldObject.id;
    }
    oldObject.parentNode.replaceChild(newObject,oldObject);
    return;
}

ENDJS
}

sub gradeleveldescription {
    my $gradelevel=shift;
    my %gradelevels=(0 => 'Not specified',
		     1 => 'Grade 1',
		     2 => 'Grade 2',
		     3 => 'Grade 3',
		     4 => 'Grade 4',
		     5 => 'Grade 5',
		     6 => 'Grade 6',
		     7 => 'Grade 7',
		     8 => 'Grade 8',
		     9 => 'Grade 9',
		     10 => 'Grade 10',
		     11 => 'Grade 11',
		     12 => 'Grade 12',
		     13 => 'Grade 13',
		     14 => '100 Level',
		     15 => '200 Level',
		     16 => '300 Level',
		     17 => '400 Level',
		     18 => 'Graduate Level');
    return &mt($gradelevels{$gradelevel});
}

sub select_level_form {
    my ($deflevel,$name)=@_;
    unless ($deflevel) { $deflevel=0; }
    my $selectform = "<select name=\"$name\" size=\"1\">\n";
    for (my $i=0; $i<=18; $i++) {
        $selectform.="<option value=\"$i\" ".
            ($i==$deflevel ? 'selected="selected" ' : '').
                ">".&gradeleveldescription($i)."</option>\n";
    }
    $selectform.="</select>";
    return $selectform;
}

#-------------------------------------------

=pod

=item * &select_dom_form($defdom,$name,$includeempty,$showdomdesc,$onchange,$incdoms,$excdoms,$disabled)

Returns a string containing a <select name='$name' size='1'> form to 
allow a user to select the domain to preform an operation in.  
See loncreateuser.pm for an example invocation and use.

If the $includeempty flag is set, it also includes an empty choice ("no domain
selected");

If the $showdomdesc flag is set, the domain name is followed by the domain description.

The optional $onchange argument specifies what should occur if the domain selector is changed, e.g., 'this.form.submit()' if the form is to be automatically submitted.

The optional $incdoms is a reference to an array of domains which will be the only available options.

The optional $excdoms is a reference to an array of domains which will be excluded from the available options.

The optional $disabled argument, if true, adds the disabled attribute to the select tag. 

=cut

#-------------------------------------------
sub select_dom_form {
    my ($defdom,$name,$includeempty,$showdomdesc,$onchange,$incdoms,$excdoms,$disabled) = @_;
    if ($onchange) {
        $onchange = ' onchange="'.$onchange.'"';
    }
    if ($disabled) {
        $disabled = ' disabled="disabled"';
    }
    my (@domains,%exclude);
    if (ref($incdoms) eq 'ARRAY') {
        @domains = sort {lc($a) cmp lc($b)} (@{$incdoms});
    } else {
        @domains = sort {lc($a) cmp lc($b)} (&Apache::lonnet::all_domains());
    }
    if ($includeempty) { @domains=('',@domains); }
    if (ref($excdoms) eq 'ARRAY') {
        map { $exclude{$_} = 1; } @{$excdoms};
    }
    my $selectdomain = "<select name=\"$name\" size=\"1\"$onchange$disabled>\n";
    foreach my $dom (@domains) {
        next if ($exclude{$dom});
        $selectdomain.="<option value=\"$dom\" ".
            ($dom eq $defdom ? 'selected="selected" ' : '').'>'.$dom;
        if ($showdomdesc) {
            if ($dom ne '') {
                my $domdesc = &Apache::lonnet::domain($dom,'description');
                if ($domdesc ne '') {
                    $selectdomain .= ' ('.$domdesc.')';
                }
            } 
        }
        $selectdomain .= "</option>\n";
    }
    $selectdomain.="</select>";
    return $selectdomain;
}

#-------------------------------------------

=pod

=item * &home_server_form_item($domain,$name,$defaultflag)

input: 4 arguments (two required, two optional) - 
    $domain - domain of new user
    $name - name of form element
    $default - Value of 'default' causes a default item to be first 
                            option, and selected by default. 
    $hide - Value of 'hide' causes hiding of the name of the server, 
                            if 1 server found, or default, if 0 found.
output: returns 2 items: 
(a) form element which contains either:
   (i) <select name="$name">
        <option value="$hostid1">$hostid $servers{$hostid}</option>
        <option value="$hostid2">$hostid $servers{$hostid}</option>       
       </select>
       form item if there are multiple library servers in $domain, or
   (ii) an <input type="hidden" name="$name" value="$hostid" /> form item 
       if there is only one library server in $domain.

(b) number of library servers found.

See loncreateuser.pm for example of use.

=cut

#-------------------------------------------
sub home_server_form_item {
    my ($domain,$name,$default,$hide) = @_;
    my %servers = &Apache::lonnet::get_servers($domain,'library');
    my $result;
    my $numlib = keys(%servers);
    if ($numlib > 1) {
        $result .= '<select name="'.$name.'" />'."\n";
        if ($default) {
            $result .= '<option value="default" selected="selected">'.&mt('default').
                       '</option>'."\n";
        }
        foreach my $hostid (sort(keys(%servers))) {
            $result.= '<option value="'.$hostid.'">'.
	              $hostid.' '.$servers{$hostid}."</option>\n";
        }
        $result .= '</select>'."\n";
    } elsif ($numlib == 1) {
        my $hostid;
        foreach my $item (keys(%servers)) {
            $hostid = $item;
        }
        $result .= '<input type="hidden" name="'.$name.'" value="'.
                   $hostid.'" />';
                   if (!$hide) {
                       $result .= $hostid.' '.$servers{$hostid};
                   }
                   $result .= "\n";
    } elsif ($default) {
        $result .= '<input type="hidden" name="'.$name.
                   '" value="default" />';
                   if (!$hide) {
                       $result .= &mt('default');
                   }
                   $result .= "\n";
    }
    return ($result,$numlib);
}

=pod

=back 

=cut

###############################################################
##                  Decoding User Agent                      ##
###############################################################

=pod

=head1 Decoding the User Agent

=over 4

=item * &decode_user_agent()

Inputs: $r

Outputs:

=over 4

=item * $httpbrowser

=item * $clientbrowser

=item * $clientversion

=item * $clientmathml

=item * $clientunicode

=item * $clientos

=item * $clientmobile

=item * $clientinfo

=item * $clientosversion

=back

=back 

=cut

###############################################################
###############################################################
sub decode_user_agent {
    my ($r)=@_;
    my @browsertype=split(/\&/,$Apache::lonnet::perlvar{"lonBrowsDet"});
    my %mathcap=split(/\&/,$$Apache::lonnet::perlvar{"lonMathML"});
    my $httpbrowser=$ENV{"HTTP_USER_AGENT"};
    if (!$httpbrowser && $r) { $httpbrowser=$r->header_in('User-Agent'); }
    my $clientbrowser='unknown';
    my $clientversion='0';
    my $clientmathml='';
    my $clientunicode='0';
    my $clientmobile=0;
    my $clientosversion='';
    for (my $i=0;$i<=$#browsertype;$i++) {
        my ($bname,$match,$notmatch,$vreg,$minv,$univ)=split(/\%/,$browsertype[$i]);
	if (($httpbrowser=~/$match/i)  && ($httpbrowser!~/$notmatch/i)) {
	    $clientbrowser=$bname;
            $httpbrowser=~/$vreg/i;
	    $clientversion=$1;
            $clientmathml=($clientversion>=$minv);
            $clientunicode=($clientversion>=$univ);
	}
    }
    my $clientos='unknown';
    my $clientinfo;
    if (($httpbrowser=~/linux/i) ||
        ($httpbrowser=~/unix/i) ||
        ($httpbrowser=~/ux/i) ||
        ($httpbrowser=~/solaris/i)) { $clientos='unix'; }
    if (($httpbrowser=~/vax/i) ||
        ($httpbrowser=~/vms/i)) { $clientos='vms'; }
    if ($httpbrowser=~/next/i) { $clientos='next'; }
    if (($httpbrowser=~/mac/i) ||
        ($httpbrowser=~/powerpc/i)) { $clientos='mac'; }
    if ($httpbrowser=~/win/i) {
        $clientos='win';
        if ($httpbrowser =~/Windows\s+NT\s+(\d+\.\d+)/i) {
            $clientosversion = $1;
        }
    }
    if ($httpbrowser=~/embed/i) { $clientos='pda'; }
    if ($httpbrowser=~/(Android|iPod|iPad|iPhone|webOS|Blackberry|Windows Phone|Opera m(?:ob|in)|Fennec)/i) {
        $clientmobile=lc($1);
    }
    if ($httpbrowser=~ m{Firefox/(\d+\.\d+)}) {
        $clientinfo = 'firefox-'.$1;
    } elsif ($httpbrowser=~ m{chromeframe/(\d+\.\d+)\.}) {
        $clientinfo = 'chromeframe-'.$1;
    }
    return ($httpbrowser,$clientbrowser,$clientversion,$clientmathml,
            $clientunicode,$clientos,$clientmobile,$clientinfo,
            $clientosversion);
}

###############################################################
##    Authentication changing form generation subroutines    ##
###############################################################
##
## All of the authform_xxxxxxx subroutines take their inputs in a
## hash, and have reasonable default values.
##
##    formname = the name given in the <form> tag.
#-------------------------------------------

=pod

=head1 Authentication Routines

=over 4

=item * &authform_xxxxxx()

The authform_xxxxxx subroutines provide javascript and html forms which 
handle some of the conveniences required for authentication forms.  
This is not an optimal method, but it works.  

=over 4

=item * authform_header

=item * authform_authorwarning

=item * authform_nochange

=item * authform_kerberos

=item * authform_internal

=item * authform_filesystem

=back

See loncreateuser.pm for invocation and use examples.

=cut

#-------------------------------------------
sub authform_header{  
    my %in = (
        formname => 'cu',
        kerb_def_dom => '',
        @_,
    );
    $in{'formname'} = 'document.' . $in{'formname'};
    my $result='';

#---------------------------------------------- Code for upper case translation
    my $Javascript_toUpperCase;
    unless ($in{kerb_def_dom}) {
        $Javascript_toUpperCase =<<"END";
        switch (choice) {
           case 'krb': currentform.elements[choicearg].value =
               currentform.elements[choicearg].value.toUpperCase();
               break;
           default:
        }
END
    } else {
        $Javascript_toUpperCase = "";
    }

    my $radioval = "'nochange'";
    if (defined($in{'curr_authtype'})) {
        if ($in{'curr_authtype'} ne '') {
            $radioval = "'".$in{'curr_authtype'}."arg'";
        }
    }
    my $argfield = 'null';
    if (defined($in{'mode'})) {
        if ($in{'mode'} eq 'modifycourse')  {
            if (defined($in{'curr_autharg'})) {
                if ($in{'curr_autharg'} ne '') {
                    $argfield = "'$in{'curr_autharg'}'";
                }
            }
        }
    }

    $result.=<<"END";
var current = new Object();
current.radiovalue = $radioval;
current.argfield = $argfield;

function changed_radio(choice,currentform) {
    var choicearg = choice + 'arg';
    // If a radio button in changed, we need to change the argfield
    if (current.radiovalue != choice) {
        current.radiovalue = choice;
        if (current.argfield != null) {
            currentform.elements[current.argfield].value = '';
        }
        if (choice == 'nochange') {
            current.argfield = null;
        } else {
            current.argfield = choicearg;
            switch(choice) {
                case 'krb': 
                    currentform.elements[current.argfield].value = 
                        "$in{'kerb_def_dom'}";
                break;
              default:
                break;
            }
        }
    }
    return;
}

function changed_text(choice,currentform) {
    var choicearg = choice + 'arg';
    if (currentform.elements[choicearg].value !='') {
        $Javascript_toUpperCase
        // clear old field
        if ((current.argfield != choicearg) && (current.argfield != null)) {
            currentform.elements[current.argfield].value = '';
        }
        current.argfield = choicearg;
    }
    set_auth_radio_buttons(choice,currentform);
    return;
}

function set_auth_radio_buttons(newvalue,currentform) {
    var numauthchoices = currentform.login.length;
    if (typeof numauthchoices  == "undefined") {
        return;
    } 
    var i=0;
    while (i < numauthchoices) {
        if (currentform.login[i].value == newvalue) { break; }
        i++;
    }
    if (i == numauthchoices) {
        return;
    }
    current.radiovalue = newvalue;
    currentform.login[i].checked = true;
    return;
}
END
    return $result;
}

sub authform_authorwarning {
    my $result='';
    $result='<i>'.
        &mt('As a general rule, only authors or co-authors should be '.
            'filesystem authenticated '.
            '(which allows access to the server filesystem).')."</i>\n";
    return $result;
}

sub authform_nochange {
    my %in = (
              formname => 'document.cu',
              kerb_def_dom => 'MSU.EDU',
              @_,
          );
    my ($authnum,%can_assign) = &get_assignable_auth($in{'domain'}); 
    my $result;
    if (!$authnum) {
        $result = &mt('Under your current role you are not permitted to change login settings for this user');
    } else {
        $result = '<label>'.&mt('[_1] Do not change login data',
                  '<input type="radio" name="login" value="nochange" '.
                  'checked="checked" onclick="'.
            "javascript:changed_radio('nochange',$in{'formname'});".'" />').
	    '</label>';
    }
    return $result;
}

sub authform_kerberos {
    my %in = (
              formname => 'document.cu',
              kerb_def_dom => 'MSU.EDU',
              kerb_def_auth => 'krb4',
              @_,
              );
    my ($check4,$check5,$krbcheck,$krbarg,$krbver,$result,$authtype,
        $autharg,$jscall,$disabled);
    my ($authnum,%can_assign) = &get_assignable_auth($in{'domain'});
    if ($in{'kerb_def_auth'} eq 'krb5') {
       $check5 = ' checked="checked"';
    } else {
       $check4 = ' checked="checked"';
    }
    if ($in{'readonly'}) {
        $disabled = ' disabled="disabled"';
    }
    $krbarg = $in{'kerb_def_dom'};
    if (defined($in{'curr_authtype'})) {
        if ($in{'curr_authtype'} eq 'krb') {
            $krbcheck = ' checked="checked"';
            if (defined($in{'mode'})) {
                if ($in{'mode'} eq 'modifyuser') {
                    $krbcheck = '';
                }
            }
            if (defined($in{'curr_kerb_ver'})) {
                if ($in{'curr_krb_ver'} eq '5') {
                    $check5 = ' checked="checked"';
                    $check4 = '';
                } else {
                    $check4 = ' checked="checked"';
                    $check5 = '';
                }
            }
            if (defined($in{'curr_autharg'})) {
                $krbarg = $in{'curr_autharg'};
            }
            if (!$can_assign{'krb4'} && !$can_assign{'krb5'}) {
                if (defined($in{'curr_autharg'})) {
                    $result = 
    &mt('Currently Kerberos authenticated with domain [_1] Version [_2].',
        $in{'curr_autharg'},$krbver);
                } else {
                    $result =
    &mt('Currently Kerberos authenticated, Version [_1].',$krbver);
                }
                return $result; 
            }
        }
    } else {
        if ($authnum == 1) {
            $authtype = '<input type="hidden" name="login" value="krb" />';
        }
    }
    if (!$can_assign{'krb4'} && !$can_assign{'krb5'}) {
        return;
    } elsif ($authtype eq '') {
        if (defined($in{'mode'})) {
            if ($in{'mode'} eq 'modifycourse') {
                if ($authnum == 1) {
                    $authtype = '<input type="radio" name="login" value="krb"'.$disabled.' />';
                }
            }
        }
    }
    $jscall = "javascript:changed_radio('krb',$in{'formname'});";
    if ($authtype eq '') {
        $authtype = '<input type="radio" name="login" value="krb" '.
                    'onclick="'.$jscall.'" onchange="'.$jscall.'"'.
                    $krbcheck.$disabled.' />';
    }
    if (($can_assign{'krb4'} && $can_assign{'krb5'}) ||
        ($can_assign{'krb4'} && !$can_assign{'krb5'} &&
         $in{'curr_authtype'} eq 'krb5') ||
        (!$can_assign{'krb4'} && $can_assign{'krb5'} &&
         $in{'curr_authtype'} eq 'krb4')) {
        $result .= &mt
        ('[_1] Kerberos authenticated with domain [_2] '.
         '[_3] Version 4 [_4] Version 5 [_5]',
         '<label>'.$authtype,
         '</label><input type="text" size="10" name="krbarg" '.
             'value="'.$krbarg.'" '.
             'onchange="'.$jscall.'"'.$disabled.' />',
         '<label><input type="radio" name="krbver" value="4" '.$check4.$disabled.' />',
         '</label><label><input type="radio" name="krbver" value="5" '.$check5.$disabled.' />',
	 '</label>');
    } elsif ($can_assign{'krb4'}) {
        $result .= &mt
        ('[_1] Kerberos authenticated with domain [_2] '.
         '[_3] Version 4 [_4]',
         '<label>'.$authtype,
         '</label><input type="text" size="10" name="krbarg" '.
             'value="'.$krbarg.'" '.
             'onchange="'.$jscall.'"'.$disabled.' />',
         '<label><input type="hidden" name="krbver" value="4" />',
         '</label>');
    } elsif ($can_assign{'krb5'}) {
        $result .= &mt
        ('[_1] Kerberos authenticated with domain [_2] '.
         '[_3] Version 5 [_4]',
         '<label>'.$authtype,
         '</label><input type="text" size="10" name="krbarg" '.
             'value="'.$krbarg.'" '.
             'onchange="'.$jscall.'"'.$disabled.' />',
         '<label><input type="hidden" name="krbver" value="5" />',
         '</label>');
    }
    return $result;
}

sub authform_internal {
    my %in = (
                formname => 'document.cu',
                kerb_def_dom => 'MSU.EDU',
                @_,
                );
    my ($intcheck,$intarg,$result,$authtype,$autharg,$jscall,$disabled);
    my ($authnum,%can_assign) = &get_assignable_auth($in{'domain'});
    if ($in{'readonly'}) {
        $disabled = ' disabled="disabled"';
    }
    if (defined($in{'curr_authtype'})) {
        if ($in{'curr_authtype'} eq 'int') {
            if ($can_assign{'int'}) {
                $intcheck = 'checked="checked" ';
                if (defined($in{'mode'})) {
                    if ($in{'mode'} eq 'modifyuser') {
                        $intcheck = '';
                    }
                }
                if (defined($in{'curr_autharg'})) {
                    $intarg = $in{'curr_autharg'};
                }
            } else {
                $result = &mt('Currently internally authenticated.');
                return $result;
            }
        }
    } else {
        if ($authnum == 1) {
            $authtype = '<input type="hidden" name="login" value="int" />';
        }
    }
    if (!$can_assign{'int'}) {
        return;
    } elsif ($authtype eq '') {
        if (defined($in{'mode'})) {
            if ($in{'mode'} eq 'modifycourse') {
                if ($authnum == 1) {
                    $authtype = '<input type="radio" name="login" value="int"'.$disabled.' />';
                }
            }
        }
    }
    $jscall = "javascript:changed_radio('int',$in{'formname'});";
    if ($authtype eq '') {
        $authtype = '<input type="radio" name="login" value="int" '.$intcheck.
                    ' onchange="'.$jscall.'" onclick="'.$jscall.'"'.$disabled.' />';
    }
    $autharg = '<input type="password" size="10" name="intarg" value="'.
               $intarg.'" onchange="'.$jscall.'"'.$disabled.' />';
    $result = &mt
        ('[_1] Internally authenticated (with initial password [_2])',
         '<label>'.$authtype,'</label>'.$autharg);
    $result.='<label><input type="checkbox" name="visible" onclick="if (this.checked) { this.form.intarg.type='."'text'".' } else { this.form.intarg.type='."'password'".' }"'.$disabled.' />'.&mt('Visible input').'</label>';
    return $result;
}

sub authform_local {
    my %in = (
              formname => 'document.cu',
              kerb_def_dom => 'MSU.EDU',
              @_,
              );
    my ($loccheck,$locarg,$result,$authtype,$autharg,$jscall,$disabled);
    my ($authnum,%can_assign) = &get_assignable_auth($in{'domain'});
    if ($in{'readonly'}) {
        $disabled = ' disabled="disabled"';
    }
    if (defined($in{'curr_authtype'})) {
        if ($in{'curr_authtype'} eq 'loc') {
            if ($can_assign{'loc'}) {
                $loccheck = 'checked="checked" ';
                if (defined($in{'mode'})) {
                    if ($in{'mode'} eq 'modifyuser') {
                        $loccheck = '';
                    }
                }
                if (defined($in{'curr_autharg'})) {
                    $locarg = $in{'curr_autharg'};
                }
            } else {
                $result = &mt('Currently using local (institutional) authentication.');
                return $result;
            }
        }
    } else {
        if ($authnum == 1) {
            $authtype = '<input type="hidden" name="login" value="loc" />';
        }
    }
    if (!$can_assign{'loc'}) {
        return;
    } elsif ($authtype eq '') {
        if (defined($in{'mode'})) {
            if ($in{'mode'} eq 'modifycourse') {
                if ($authnum == 1) {
                    $authtype = '<input type="radio" name="login" value="loc"'.$disabled.' />';
                }
            }
        }
    }
    $jscall = "javascript:changed_radio('loc',$in{'formname'});";
    if ($authtype eq '') {
        $authtype = '<input type="radio" name="login" value="loc" '.
                    $loccheck.' onchange="'.$jscall.'" onclick="'.
                    $jscall.'"'.$disabled.' />';
    }
    $autharg = '<input type="text" size="10" name="locarg" value="'.
               $locarg.'" onchange="'.$jscall.'"'.$disabled.' />';
    $result = &mt('[_1] Local Authentication with argument [_2]',
                  '<label>'.$authtype,'</label>'.$autharg);
    return $result;
}

sub authform_filesystem {
    my %in = (
              formname => 'document.cu',
              kerb_def_dom => 'MSU.EDU',
              @_,
              );
    my ($fsyscheck,$result,$authtype,$autharg,$jscall,$disabled);
    my ($authnum,%can_assign) = &get_assignable_auth($in{'domain'});
    if ($in{'readonly'}) {
        $disabled = ' disabled="disabled"';
    }
    if (defined($in{'curr_authtype'})) {
        if ($in{'curr_authtype'} eq 'fsys') {
            if ($can_assign{'fsys'}) {
                $fsyscheck = 'checked="checked" ';
                if (defined($in{'mode'})) {
                    if ($in{'mode'} eq 'modifyuser') {
                        $fsyscheck = '';
                    }
                }
            } else {
                $result = &mt('Currently Filesystem Authenticated.');
                return $result;
            }           
        }
    } else {
        if ($authnum == 1) {
            $authtype = '<input type="hidden" name="login" value="fsys" />';
        }
    }
    if (!$can_assign{'fsys'}) {
        return;
    } elsif ($authtype eq '') {
        if (defined($in{'mode'})) {
            if ($in{'mode'} eq 'modifycourse') {
                if ($authnum == 1) {
                    $authtype = '<input type="radio" name="login" value="fsys"'.$disabled.' />';
                }
            }
        }
    }
    $jscall = "javascript:changed_radio('fsys',$in{'formname'});";
    if ($authtype eq '') {
        $authtype = '<input type="radio" name="login" value="fsys" '.
                    $fsyscheck.' onchange="'.$jscall.'" onclick="'.
                    $jscall.'"'.$disabled.' />';
    }
    $autharg = '<input type="password" size="10" name="fsysarg" value=""'.
               ' onchange="'.$jscall.'"'.$disabled.' />';
    $result = &mt
        ('[_1] Filesystem Authenticated (with initial password [_2])',
         '<label>'.$authtype,'</label>'.$autharg);
    return $result;
}

sub get_assignable_auth {
    my ($dom) = @_;
    if ($dom eq '') {
        $dom = $env{'request.role.domain'};
    }
    my %can_assign = (
                          krb4 => 1,
                          krb5 => 1,
                          int  => 1,
                          loc  => 1,
                     );
    my %domconfig = &Apache::lonnet::get_dom('configuration',['usercreation'],$dom);
    if (ref($domconfig{'usercreation'}) eq 'HASH') {
        if (ref($domconfig{'usercreation'}{'authtypes'}) eq 'HASH') {
            my $authhash = $domconfig{'usercreation'}{'authtypes'};
            my $context;
            if ($env{'request.role'} =~ /^au/) {
                $context = 'author';
            } elsif ($env{'request.role'} =~ /^(dc|dh)/) {
                $context = 'domain';
            } elsif ($env{'request.course.id'}) {
                $context = 'course';
            }
            if ($context) {
                if (ref($authhash->{$context}) eq 'HASH') {
                   %can_assign = %{$authhash->{$context}}; 
                }
            }
        }
    }
    my $authnum = 0;
    foreach my $key (keys(%can_assign)) {
        if ($can_assign{$key}) {
            $authnum ++;
        }
    }
    if ($can_assign{'krb4'} && $can_assign{'krb5'}) {
        $authnum --;
    }
    return ($authnum,%can_assign);
}

sub check_passwd_rules {
    my ($domain,$plainpass) = @_;
    my %passwdconf = &Apache::lonnet::get_passwdconf($domain);
    my ($min,$max,@chars,@brokerule,$warning);
    $min = $Apache::lonnet::passwdmin;
    if (ref($passwdconf{'chars'}) eq 'ARRAY') {
        if ($passwdconf{'min'} =~ /^\d+$/) {
            if ($passwdconf{'min'} > $min) {
                $min = $passwdconf{'min'};
            }
        }
        if ($passwdconf{'max'} =~ /^\d+$/) {
            $max = $passwdconf{'max'};
        }
        @chars = @{$passwdconf{'chars'}};
    }
    if (($min) && (length($plainpass) < $min)) {
        push(@brokerule,'min');
    }
    if (($max) && (length($plainpass) > $max)) {
        push(@brokerule,'max');
    }
    if (@chars) {
        my %rules;
        map { $rules{$_} = 1; } @chars;
        if ($rules{'uc'}) {
            unless ($plainpass =~ /[A-Z]/) {
                push(@brokerule,'uc');
            }
        }
        if ($rules{'lc'}) {
            unless ($plainpass =~ /[a-z]/) {
                push(@brokerule,'lc');
            }
        }
        if ($rules{'num'}) {
            unless ($plainpass =~ /\d/) {
                push(@brokerule,'num');
            }
        }
        if ($rules{'spec'}) {
            unless ($plainpass =~ /[!"#$%&'()*+,\-.\/:;<=>?@[\\\]^_`{|}~]/) {
                push(@brokerule,'spec');
            }
        }
    }
    if (@brokerule) {
        my %rulenames = &Apache::lonlocal::texthash(
            uc   => 'At least one upper case letter',
            lc   => 'At least one lower case letter',
            num  => 'At least one number',
            spec => 'At least one non-alphanumeric',
        );
        $rulenames{'uc'} .= ': ABCDEFGHIJKLMNOPQRSTUVWXYZ';
        $rulenames{'lc'} .= ': abcdefghijklmnopqrstuvwxyz';
        $rulenames{'num'} .= ': 0123456789';
        $rulenames{'spec'} .= ': !&quot;\#$%&amp;\'()*+,-./:;&lt;=&gt;?@[\]^_\`{|}~';
        $rulenames{'min'} = &mt('Minimum password length: [_1]',$min);
        $rulenames{'max'} = &mt('Maximum password length: [_1]',$max);
        $warning = &mt('Password did not satisfy the following:').'<ul>';
        foreach my $rule ('min','max','uc','lc','num','spec') {
            if (grep(/^$rule$/,@brokerule)) {
                $warning .= '<li>'.$rulenames{$rule}.'</li>';
            }
        }
        $warning .= '</ul>';
    }
    if (wantarray) {
        return @brokerule;
    }
    return $warning;
}

###############################################################
##    Get Kerberos Defaults for Domain                 ##
###############################################################
##
## Returns default kerberos version and an associated argument
## as listed in file domain.tab. If not listed, provides
## appropriate default domain and kerberos version.
##
#-------------------------------------------

=pod

=item * &get_kerberos_defaults()

get_kerberos_defaults($target_domain) returns the default kerberos
version and domain. If not found, it defaults to version 4 and the 
domain of the server.

=over 4

($def_version, $def_krb_domain) = &get_kerberos_defaults($target_domain);

=back

=back

=cut

#-------------------------------------------
sub get_kerberos_defaults {
    my $domain=shift;
    my ($krbdef,$krbdefdom);
    my %domdefaults = &Apache::lonnet::get_domain_defaults($domain);
    if (($domdefaults{'auth_def'} =~/^krb(4|5)$/) && ($domdefaults{'auth_arg_def'} ne '')) {
        $krbdef = $domdefaults{'auth_def'};
        $krbdefdom = $domdefaults{'auth_arg_def'};
    } else {
        $ENV{'SERVER_NAME'}=~/(\w+\.\w+)$/;
        my $krbdefdom=$1;
        $krbdefdom=~tr/a-z/A-Z/;
        $krbdef = "krb4";
    }
    return ($krbdef,$krbdefdom);
}


###############################################################
##                Thesaurus Functions                        ##
###############################################################

=pod

=head1 Thesaurus Functions

=over 4

=item * &initialize_keywords()

Initializes the package variable %Keywords if it is empty.  Uses the
package variable $thesaurus_db_file.

=cut

###################################################

sub initialize_keywords {
    return 1 if (scalar keys(%Keywords));
    # If we are here, %Keywords is empty, so fill it up
    #   Make sure the file we need exists...
    if (! -e $thesaurus_db_file) {
        &Apache::lonnet::logthis("Attempt to access $thesaurus_db_file".
                                 " failed because it does not exist");
        return 0;
    }
    #   Set up the hash as a database
    my %thesaurus_db;
    if (! tie(%thesaurus_db,'GDBM_File',
              $thesaurus_db_file,&GDBM_READER(),0640)){
        &Apache::lonnet::logthis("Could not tie \%thesaurus_db to ".
                                 $thesaurus_db_file);
        return 0;
    } 
    #  Get the average number of appearances of a word.
    my $avecount = $thesaurus_db{'average.count'};
    #  Put keywords (those that appear > average) into %Keywords
    while (my ($word,$data)=each (%thesaurus_db)) {
        my ($count,undef) = split /:/,$data;
        $Keywords{$word}++ if ($count > $avecount);
    }
    untie %thesaurus_db;
    # Remove special values from %Keywords.
    foreach my $value ('total.count','average.count') {
        delete($Keywords{$value}) if (exists($Keywords{$value}));
  }
    return 1;
}

###################################################

=pod

=item * &keyword($word)

Returns true if $word is a keyword.  A keyword is a word that appears more 
than the average number of times in the thesaurus database.  Calls 
&initialize_keywords

=cut

###################################################

sub keyword {
    return if (!&initialize_keywords());
    my $word=lc(shift());
    $word=~s/\W//g;
    return exists($Keywords{$word});
}

###############################################################

=pod 

=item * &get_related_words()

Look up a word in the thesaurus.  Takes a scalar argument and returns
an array of words.  If the keyword is not in the thesaurus, an empty array
will be returned.  The order of the words returned is determined by the
database which holds them.

Uses global $thesaurus_db_file.


=cut

###############################################################
sub get_related_words {
    my $keyword = shift;
    my %thesaurus_db;
    if (! -e $thesaurus_db_file) {
        &Apache::lonnet::logthis("Attempt to access $thesaurus_db_file ".
                                 "failed because the file does not exist");
        return ();
    }
    if (! tie(%thesaurus_db,'GDBM_File',
              $thesaurus_db_file,&GDBM_READER(),0640)){
        return ();
    } 
    my @Words=();
    my $count=0;
    if (exists($thesaurus_db{$keyword})) {
	# The first element is the number of times
	# the word appears.  We do not need it now.
	my (undef,@RelatedWords) = (split(/:/,$thesaurus_db{$keyword}));
	my (undef,$mostfrequentcount)=split(/\,/,$RelatedWords[0]);
	my $threshold=$mostfrequentcount/10;
        foreach my $possibleword (@RelatedWords) {
            my ($word,$wordcount)=split(/\,/,$possibleword);
            if ($wordcount>$threshold) {
		push(@Words,$word);
                $count++;
                if ($count>10) { last; }
	    }
        }
    }
    untie %thesaurus_db;
    return @Words;
}

=pod

=back

=cut

# -------------------------------------------------------------- Plaintext name
=pod

=head1 User Name Functions

=over 4

=item * &plainname($uname,$udom,$first)

Takes a users logon name and returns it as a string in
"first middle last generation" form 
if $first is set to 'lastname' then it returns it as
'lastname generation, firstname middlename' if their is a lastname

=cut


###############################################################
sub plainname {
    my ($uname,$udom,$first)=@_;
    return if (!defined($uname) || !defined($udom));
    my %names=&getnames($uname,$udom);
    my $name=&Apache::lonnet::format_name($names{'firstname'},
					  $names{'middlename'},
					  $names{'lastname'},
					  $names{'generation'},$first);
    $name=~s/^\s+//;
    $name=~s/\s+$//;
    $name=~s/\s+/ /g;
    if ($name !~ /\S/) { $name=$uname.':'.$udom; }
    return $name;
}

# -------------------------------------------------------------------- Nickname
=pod

=item * &nickname($uname,$udom)

Gets a users name and returns it as a string as

"&quot;nickname&quot;"

if the user has a nickname or

"first middle last generation"

if the user does not

=cut

sub nickname {
    my ($uname,$udom)=@_;
    return if (!defined($uname) || !defined($udom));
    my %names=&getnames($uname,$udom);
    my $name=$names{'nickname'};
    if ($name) {
       $name='&quot;'.$name.'&quot;'; 
    } else {
       $name=$names{'firstname'}.' '.$names{'middlename'}.' '.
	     $names{'lastname'}.' '.$names{'generation'};
       $name=~s/\s+$//;
       $name=~s/\s+/ /g;
    }
    return $name;
}

sub getnames {
    my ($uname,$udom)=@_;
    return if (!defined($uname) || !defined($udom));
    if ($udom eq 'public' && $uname eq 'public') {
	return ('lastname' => &mt('Public'));
    }
    my $id=$uname.':'.$udom;
    my ($names,$cached)=&Apache::lonnet::is_cached_new('namescache',$id);
    if ($cached) {
	return %{$names};
    } else {
	my %loadnames=&Apache::lonnet::get('environment',
                    ['firstname','middlename','lastname','generation','nickname'],
					 $udom,$uname);
	&Apache::lonnet::do_cache_new('namescache',$id,\%loadnames);
	return %loadnames;
    }
}

# -------------------------------------------------------------------- getemails

=pod

=item * &getemails($uname,$udom)

Gets a user's email information and returns it as a hash with keys:
notification, critnotification, permanentemail

For notification and critnotification, values are comma-separated lists 
of e-mail addresses; for permanentemail, value is a single e-mail address.
 

=cut


sub getemails {
    my ($uname,$udom)=@_;
    if ($udom eq 'public' && $uname eq 'public') {
	return;
    }
    if (!$udom) { $udom=$env{'user.domain'}; }
    if (!$uname) { $uname=$env{'user.name'}; }
    my $id=$uname.':'.$udom;
    my ($names,$cached)=&Apache::lonnet::is_cached_new('emailscache',$id);
    if ($cached) {
	return %{$names};
    } else {
	my %loadnames=&Apache::lonnet::get('environment',
                    			   ['notification','critnotification',
					    'permanentemail'],
					   $udom,$uname);
	&Apache::lonnet::do_cache_new('emailscache',$id,\%loadnames);
	return %loadnames;
    }
}

sub flush_email_cache {
    my ($uname,$udom)=@_;
    if (!$udom)  { $udom =$env{'user.domain'}; }
    if (!$uname) { $uname=$env{'user.name'};   }
    return if ($udom eq 'public' && $uname eq 'public');
    my $id=$uname.':'.$udom;
    &Apache::lonnet::devalidate_cache_new('emailscache',$id);
}

# -------------------------------------------------------------------- getlangs

=pod

=item * &getlangs($uname,$udom)

Gets a user's language preference and returns it as a hash with key:
language.

=cut


sub getlangs {
    my ($uname,$udom) = @_;
    if (!$udom)  { $udom =$env{'user.domain'}; }
    if (!$uname) { $uname=$env{'user.name'};   }
    my $id=$uname.':'.$udom;
    my ($langs,$cached)=&Apache::lonnet::is_cached_new('userlangs',$id);
    if ($cached) {
        return %{$langs};
    } else {
        my %loadlangs=&Apache::lonnet::get('environment',['languages'],
                                           $udom,$uname);
        &Apache::lonnet::do_cache_new('userlangs',$id,\%loadlangs);
        return %loadlangs;
    }
}

sub flush_langs_cache {
    my ($uname,$udom)=@_;
    if (!$udom)  { $udom =$env{'user.domain'}; }
    if (!$uname) { $uname=$env{'user.name'};   }
    return if ($udom eq 'public' && $uname eq 'public');
    my $id=$uname.':'.$udom;
    &Apache::lonnet::devalidate_cache_new('userlangs',$id);
}

# ------------------------------------------------------------------ Screenname

=pod

=item * &screenname($uname,$udom)

Gets a users screenname and returns it as a string

=cut

sub screenname {
    my ($uname,$udom)=@_;
    if ($uname eq $env{'user.name'} &&
	$udom eq $env{'user.domain'}) {return $env{'environment.screenname'};}
    my %names=&Apache::lonnet::get('environment',['screenname'],$udom,$uname);
    return $names{'screenname'};
}


# ------------------------------------------------------------- Confirm Wrapper
=pod

=item * &confirmwrapper($message)

Wrap messages about completion of operation in box

=cut

sub confirmwrapper {
    my ($message)=@_;
    if ($message) {
        return "\n".'<div class="LC_confirm_box">'."\n"
               .$message."\n"
               .'</div>'."\n";
    } else {
        return $message;
    }
}

# ------------------------------------------------------------- Message Wrapper

sub messagewrapper {
    my ($link,$username,$domain,$subject,$text)=@_;
    return 
        '<a href="/adm/email?compose=individual&amp;'.
        'recname='.$username.'&amp;recdom='.$domain.
	'&amp;subject='.&escape($subject).'&amp;text='.&escape($text).'" '.
        'title="'.&mt('Send message').'">'.$link.'</a>';
}

# --------------------------------------------------------------- Notes Wrapper

sub noteswrapper {
    my ($link,$un,$do)=@_;
    return 
"<a href='/adm/email?recordftf=retrieve&amp;recname=$un&amp;recdom=$do'>$link</a>";
}

# ------------------------------------------------------------- Aboutme Wrapper

sub aboutmewrapper {
    my ($link,$username,$domain,$target,$class)=@_;
    if (!defined($username)  && !defined($domain)) {
        return;
    }
    return '<a href="/adm/'.$domain.'/'.$username.'/aboutme"'.
	($target?' target="'.$target.'"':'').($class?' class="'.$class.'"':'').' title="'.&mt("View this user's personal information page").'">'.$link.'</a>';
}

# ------------------------------------------------------------ Syllabus Wrapper

sub syllabuswrapper {
    my ($linktext,$coursedir,$domain)=@_;
    return qq{<a href="/public/$domain/$coursedir/syllabus">$linktext</a>};
}

# -----------------------------------------------------------------------------

sub aboutme_on {
    my ($uname,$udom)=@_;
    unless ($uname) { $uname=$env{'user.name'}; }
    unless ($udom)  { $udom=$env{'user.domain'}; }
    return if ($udom eq 'public' && $uname eq 'public');
    my $hashkey=$uname.':'.$udom;
    my ($aboutme,$cached)=&Apache::lonnet::is_cached_new('aboutme',$hashkey);
    if ($cached) {
        return $aboutme;
    }
    $aboutme = &Apache::lonnet::usertools_access($uname,$udom,'aboutme');
    &Apache::lonnet::do_cache_new('aboutme',$hashkey,$aboutme,3600);
    return $aboutme;
}

sub devalidate_aboutme_cache {
    my ($uname,$udom)=@_;
    if (!$udom)  { $udom =$env{'user.domain'}; }
    if (!$uname) { $uname=$env{'user.name'};   }
    return if ($udom eq 'public' && $uname eq 'public');
    my $id=$uname.':'.$udom;
    &Apache::lonnet::devalidate_cache_new('aboutme',$id);
}

sub track_student_link {
    my ($linktext,$sname,$sdom,$target,$start,$only_body) = @_;
    my $link ="/adm/trackstudent?";
    my $title = 'View recent activity';
    if (defined($sname) && $sname !~ /^\s*$/ &&
        defined($sdom)  && $sdom  !~ /^\s*$/) {
        $link .= "selected_student=$sname:$sdom";
        $title .= ' of this student';
    } 
    if (defined($target) && $target !~ /^\s*$/) {
        $target = qq{target="$target"};
    } else {
        $target = '';
    }
    if ($start) { $link.='&amp;start='.$start; }
    if ($only_body) { $link .= '&amp;only_body=1'; }
    $title = &mt($title);
    $linktext = &mt($linktext);
    return qq{<a href="$link" title="$title" $target>$linktext</a>}.
	&help_open_topic('View_recent_activity');
}

sub slot_reservations_link {
    my ($linktext,$sname,$sdom,$target) = @_;
    my $link ="/adm/slotrequest?command=showresv&amp;origin=aboutme";
    my $title = 'View slot reservation history';
    if (defined($sname) && $sname !~ /^\s*$/ &&
        defined($sdom)  && $sdom  !~ /^\s*$/) {
        $link .= "&amp;uname=$sname&amp;udom=$sdom";
        $title .= ' of this student';
    }
    if (defined($target) && $target !~ /^\s*$/) {
        $target = qq{target="$target"};
    } else {
        $target = '';
    }
    $title = &mt($title);
    $linktext = &mt($linktext);
    return qq{<a href="$link" title="$title" $target>$linktext</a>};
# FIXME uncomment when help item created: &help_open_topic('Slot_Reservation_History');

}

# ===================================================== Display a student photo


sub student_image_tag {
    my ($domain,$user)=@_;
    my $imgsrc=&Apache::lonnet::studentphoto($domain,$user,'jpg');
    if (($imgsrc) && ($imgsrc ne '/adm/lonKaputt/lonlogo_broken.gif')) {
	return '<img src="'.$imgsrc.'" align="right" />';
    } else {
	return '';
    }
}

=pod

=back

=head1 Access .tab File Data

=over 4

=item * &languageids() 

returns list of all language ids

=cut

sub languageids {
    return sort(keys(%language));
}

=pod

=item * &languagedescription() 

returns description of a specified language id

=cut

sub languagedescription {
    my $code=shift;
    return  ($supported_language{$code}?'* ':'').
            $language{$code}.
	    ($supported_language{$code}?' ('.&mt('interface available').')':'');
}

=pod

=item * &plainlanguagedescription

Returns both the plain language description (e.g. 'Creoles and Pidgins, English-based (Other)')
and the language character encoding (e.g. ISO) separated by a ' - ' string.

=cut

sub plainlanguagedescription {
    my $code=shift;
    return $language{$code};
}

=pod

=item * &supportedlanguagecode

Returns the supported language code (e.g. sptutf maps to pt) given a language
code.

=cut

sub supportedlanguagecode {
    my $code=shift;
    return $supported_language{$code};
}

=pod

=item * &latexlanguage()

Given a language key code returns the correspondnig language to use
to select the correct hyphenation on LaTeX printouts.  This is undef if there
is no supported hyphenation for the language code.

=cut

sub latexlanguage {
    my $code = shift;
    return $latex_language{$code};
}

=pod

=item * &latexhyphenation()

Same as above but what's supplied is the language as it might be stored
in the metadata.

=cut

sub latexhyphenation {
    my $key = shift;
    return $latex_language_bykey{$key};
}

=pod

=item * &copyrightids() 

returns list of all copyrights

=cut

sub copyrightids {
    return sort(keys(%cprtag));
}

=pod

=item * &copyrightdescription() 

returns description of a specified copyright id

=cut

sub copyrightdescription {
    return &mt($cprtag{shift(@_)});
}

=pod

=item * &source_copyrightids() 

returns list of all source copyrights

=cut

sub source_copyrightids {
    return sort(keys(%scprtag));
}

=pod

=item * &source_copyrightdescription() 

returns description of a specified source copyright id

=cut

sub source_copyrightdescription {
    return &mt($scprtag{shift(@_)});
}

=pod

=item * &filecategories() 

returns list of all file categories

=cut

sub filecategories {
    return sort(keys(%category_extensions));
}

=pod

=item * &filecategorytypes() 

returns list of file types belonging to a given file
category

=cut

sub filecategorytypes {
    my ($cat) = @_;
    return @{$category_extensions{lc($cat)}};
}

=pod

=item * &fileembstyle() 

returns embedding style for a specified file type

=cut

sub fileembstyle {
    return $fe{lc(shift(@_))};
}

sub filemimetype {
    return $fm{lc(shift(@_))};
}


sub filecategoryselect {
    my ($name,$value)=@_;
    return &select_form($value,$name,
                        {'' => &mt('Any category'), map { $_,$_ } sort(keys(%category_extensions))});
}

=pod

=item * &filedescription() 

returns description for a specified file type

=cut

sub filedescription {
    my $file_description = $fd{lc(shift())};
    $file_description =~ s:([\[\]]):~$1:g;
    return &mt($file_description);
}

=pod

=item * &filedescriptionex() 

returns description for a specified file type with
extra formatting

=cut

sub filedescriptionex {
    my $ex=shift;
    my $file_description = $fd{lc($ex)};
    $file_description =~ s:([\[\]]):~$1:g;
    return '.'.$ex.' '.&mt($file_description);
}

# End of .tab access
=pod

=back

=cut

# ------------------------------------------------------------------ File Types
sub fileextensions {
    return sort(keys(%fe));
}

# ----------------------------------------------------------- Display Languages
# returns a hash with all desired display languages
#

sub display_languages {
    my %languages=();
    foreach my $lang (&Apache::lonlocal::preferred_languages()) {
	$languages{$lang}=1;
    }
    &get_unprocessed_cgi($ENV{'QUERY_STRING'},['displaylanguage']);
    if ($env{'form.displaylanguage'}) {
	foreach my $lang (split(/\s*(\,|\;|\:)\s*/,$env{'form.displaylanguage'})) {
	    $languages{$lang}=1;
        }
    }
    return %languages;
}

sub languages {
    my ($possible_langs) = @_;
    my @preferred_langs = &Apache::lonlocal::preferred_languages();
    if (!ref($possible_langs)) {
	if( wantarray ) {
	    return @preferred_langs;
	} else {
	    return $preferred_langs[0];
	}
    }
    my %possibilities = map { $_ => 1 } (@$possible_langs);
    my @preferred_possibilities;
    foreach my $preferred_lang (@preferred_langs) {
	if (exists($possibilities{$preferred_lang})) {
	    push(@preferred_possibilities, $preferred_lang);
	}
    }
    if( wantarray ) {
	return @preferred_possibilities;
    }
    return $preferred_possibilities[0];
}

sub user_lang {
    my ($touname,$toudom,$fromcid) = @_;
    my @userlangs;
    if (($fromcid ne '') && ($env{'course.'.$fromcid.'.languages'} ne '')) {
        @userlangs=(@userlangs,split(/\s*(\,|\;|\:)\s*/,
                    $env{'course.'.$fromcid.'.languages'}));
    } else {
        my %langhash = &getlangs($touname,$toudom);
        if ($langhash{'languages'} ne '') {
            @userlangs = split(/\s*(\,|\;|\:)\s*/,$langhash{'languages'});
        } else {
            my %domdefs = &Apache::lonnet::get_domain_defaults($toudom);
            if ($domdefs{'lang_def'} ne '') {
                @userlangs = ($domdefs{'lang_def'});
            }
        }
    }
    my @languages=&Apache::lonlocal::get_genlanguages(@userlangs);
    my $user_lh = Apache::localize->get_handle(@languages);
    return $user_lh;
}


###############################################################
##               Student Answer Attempts                     ##
###############################################################

=pod

=head1 Alternate Problem Views

=over 4

=item * &get_previous_attempt($symb, $username, $domain, $course,
    $getattempt, $regexp, $gradesub, $usec, $identifier)

Return string with previous attempt on problem. Arguments:

=over 4

=item * $symb: Problem, including path

=item * $username: username of the desired student

=item * $domain: domain of the desired student

=item * $course: Course ID

=item * $getattempt: Leave blank for all attempts, otherwise put
    something

=item * $regexp: if string matches this regexp, the string will be
    sent to $gradesub

=item * $gradesub: routine that processes the string if it matches $regexp

=item * $usec: section of the desired student

=item * $identifier: counter for student (multiple students one problem) or
    problem (one student; whole sequence).

=back

The output string is a table containing all desired attempts, if any.

=cut

sub get_previous_attempt {
  my ($symb,$username,$domain,$course,$getattempt,$regexp,$gradesub,$usec,$identifier)=@_;
  my $prevattempts='';
  no strict 'refs';
  if ($symb) {
    my (%returnhash)=
      &Apache::lonnet::restore($symb,$course,$domain,$username);
    if ($returnhash{'version'}) {
      my %lasthash=();
      my $version;
      for ($version=1;$version<=$returnhash{'version'};$version++) {
        foreach my $key (reverse(sort(split(/\:/,$returnhash{$version.':keys'})))) {
            if ($key =~ /\.rawrndseed$/) {
                my ($id) = ($key =~ /^(.+)\.rawrndseed$/);
                $lasthash{$id.'.rndseed'} = $returnhash{$version.':'.$key};
            } else {
                $lasthash{$key}=$returnhash{$version.':'.$key};
            }
        }
      }
      $prevattempts=&start_data_table().&start_data_table_header_row();
      $prevattempts.='<th>'.&mt('History').'</th>';
      my (%typeparts,%lasthidden,%regraded,%hidestatus);
      my $showsurv=&Apache::lonnet::allowed('vas',$env{'request.course.id'});
      foreach my $key (sort(keys(%lasthash))) {
	my ($ign,@parts) = split(/\./,$key);
	if ($#parts > 0) {
	  my $data=$parts[-1];
          next if ($data eq 'foilorder');
	  pop(@parts);
          $prevattempts.='<th>'.&mt('Part ').join('.',@parts).'<br />'.$data.'&nbsp;</th>';
          if ($data eq 'type') {
              unless ($showsurv) {
                  my $id = join(',',@parts);
                  $typeparts{$ign.'.'.$id} = $lasthash{$key};
                  if (($lasthash{$key} eq 'anonsurvey') || ($lasthash{$key} eq 'anonsurveycred')) {
                      $lasthidden{$ign.'.'.$id} = 1;
                  }
              }
              if ($identifier ne '') {
                  my $id = join(',',@parts);
                  if (&Apache::lonnet::EXT("resource.$id.problemstatus",$symb,
                                               $domain,$username,$usec,undef,$course) =~ /^no/) {
                      $hidestatus{$ign.'.'.$id} = 1;
                  }
              }
          } elsif ($data eq 'regrader') {
              if (($identifier ne '') && (@parts)) {
                  my $id = join(',',@parts);
                  $regraded{$ign.'.'.$id} = 1;
              }
          } 
	} else {
	  if ($#parts == 0) {
	    $prevattempts.='<th>'.$parts[0].'</th>';
	  } else {
	    $prevattempts.='<th>'.$ign.'</th>';
	  }
	}
      }
      $prevattempts.=&end_data_table_header_row();
      if ($getattempt eq '') {
        my (%solved,%resets,%probstatus);
        if (($identifier ne '') && (keys(%regraded) > 0)) {
            for ($version=1;$version<=$returnhash{'version'};$version++) {
                foreach my $id (keys(%regraded)) {
                    if (($returnhash{$version.':'.$id.'.regrader'}) &&
                        ($returnhash{$version.':'.$id.'.tries'} eq '') &&
                        ($returnhash{$version.':'.$id.'.award'} eq '')) {
                        push(@{$resets{$id}},$version);
                    }
                }
            }
        }
	for ($version=1;$version<=$returnhash{'version'};$version++) {
            my (@hidden,@unsolved);
            if (%typeparts) {
                foreach my $id (keys(%typeparts)) {
                    if (($returnhash{$version.':'.$id.'.type'} eq 'anonsurvey') ||
                        ($returnhash{$version.':'.$id.'.type'} eq 'anonsurveycred')) {
                        push(@hidden,$id);
                    } elsif ($identifier ne '') {
                        unless (($returnhash{$version.':'.$id.'.type'} eq 'survey') ||
                                ($returnhash{$version.':'.$id.'.type'} eq 'surveycred') ||
                                ($hidestatus{$id})) {
                            next if ((ref($resets{$id}) eq 'ARRAY') && grep(/^\Q$version\E$/,@{$resets{$id}}));
                            if ($returnhash{$version.':'.$id.'.solved'} eq 'correct_by_student') {
                                push(@{$solved{$id}},$version);
                            } elsif (($returnhash{$version.':'.$id.'.solved'} ne '') &&
                                     (ref($solved{$id}) eq 'ARRAY')) {
                                my $skip;
                                if (ref($resets{$id}) eq 'ARRAY') {
                                    foreach my $reset (@{$resets{$id}}) {
                                        if ($reset > $solved{$id}[-1]) {
                                            $skip=1;
                                            last;
                                        }
                                    }
                                }
                                unless ($skip) {
                                    my ($ign,$partslist) = split(/\./,$id,2);
                                    push(@unsolved,$partslist);
                                }
                            }
                        }
                    }
                }
            }
            $prevattempts.=&start_data_table_row().
                           '<td>'.&mt('Transaction [_1]',$version);
            if (@unsolved) {
                $prevattempts .= '<span class="LC_nobreak"><label>'.
                                 '<input type="checkbox" name="HIDE'.$identifier.'" value="'.$version.':'.join('_',@unsolved).'" />'.
                                 &mt('Hide').'</label></span>';
            }
            $prevattempts .= '</td>';
            if (@hidden) {
                foreach my $key (sort(keys(%lasthash))) {
                    next if ($key =~ /\.foilorder$/);
                    my $hide;
                    foreach my $id (@hidden) {
                        if ($key =~ /^\Q$id\E/) {
                            $hide = 1;
                            last;
                        }
                    }
                    if ($hide) {
                        my ($id,$data) = ($key =~ /^(.+)\.([^.]+)$/);
                        if (($data eq 'award') || ($data eq 'awarddetail')) {
                            my $value = &format_previous_attempt_value($key,
                                             $returnhash{$version.':'.$key});
                            $prevattempts.='<td>'.$value.'&nbsp;</td>';
                        } else {
                            $prevattempts.='<td>&nbsp;</td>';
                        }
                    } else {
                        if ($key =~ /\./) {
                            my $value = $returnhash{$version.':'.$key};
                            if ($key =~ /\.rndseed$/) {
                                my ($id) = ($key =~ /^(.+)\.rndseed$/);
                                if (exists($returnhash{$version.':'.$id.'.rawrndseed'})) {
                                    $value = $returnhash{$version.':'.$id.'.rawrndseed'};
                                }
                            }
                            $prevattempts.='<td>'.&format_previous_attempt_value($key,$value).
                                           '&nbsp;</td>';
                        } else {
                            $prevattempts.='<td>&nbsp;</td>';
                        }
                    }
                }
            } else {
	        foreach my $key (sort(keys(%lasthash))) {
                    next if ($key =~ /\.foilorder$/);
                    my $value = $returnhash{$version.':'.$key};
                    if ($key =~ /\.rndseed$/) {
                        my ($id) = ($key =~ /^(.+)\.rndseed$/);
                        if (exists($returnhash{$version.':'.$id.'.rawrndseed'})) {
                            $value = $returnhash{$version.':'.$id.'.rawrndseed'};
                        }
                    }
                    $prevattempts.='<td>'.&format_previous_attempt_value($key,$value).
                                   '&nbsp;</td>';
	        }
            }
	    $prevattempts.=&end_data_table_row();
	 }
      }
      my @currhidden = keys(%lasthidden);
      $prevattempts.=&start_data_table_row().'<td>'.&mt('Current').'</td>';
      foreach my $key (sort(keys(%lasthash))) {
          next if ($key =~ /\.foilorder$/);
          if (%typeparts) {
              my $hidden;
              foreach my $id (@currhidden) {
                  if ($key =~ /^\Q$id\E/) {
                      $hidden = 1;
                      last;
                  }
              }
              if ($hidden) {
                  my ($id,$data) = ($key =~ /^(.+)\.([^.]+)$/);
                  if (($data eq 'award') || ($data eq 'awarddetail')) {
                      my $value = &format_previous_attempt_value($key,$lasthash{$key});
                      if ($key =~/$regexp$/ && (defined &$gradesub)) {
                          $value = &$gradesub($value);
                      }
                      $prevattempts.='<td>'.$value.'&nbsp;</td>';
                  } else {
                      $prevattempts.='<td>&nbsp;</td>';
                  }
              } else {
                  my $value = &format_previous_attempt_value($key,$lasthash{$key});
                  if ($key =~/$regexp$/ && (defined &$gradesub)) {
                      $value = &$gradesub($value);
                  }
                  $prevattempts.='<td>'.$value.'&nbsp;</td>';
              }
          } else {
	      my $value = &format_previous_attempt_value($key,$lasthash{$key});
	      if ($key =~/$regexp$/ && (defined &$gradesub)) {
                  $value = &$gradesub($value);
              }
	      $prevattempts.='<td>'.$value.'&nbsp;</td>';
          }
      }
      $prevattempts.= &end_data_table_row().&end_data_table();
    } else {
      $prevattempts=
	  &start_data_table().&start_data_table_row().
	  '<td>'.&mt('Nothing submitted - no attempts.').'</td>'.
	  &end_data_table_row().&end_data_table();
    }
  } else {
    $prevattempts=
	  &start_data_table().&start_data_table_row().
	  '<td>'.&mt('No data.').'</td>'.
	  &end_data_table_row().&end_data_table();
  }
}

sub format_previous_attempt_value {
    my ($key,$value) = @_;
    if (($key =~ /timestamp/) || ($key=~/duedate/)) {
	$value = &Apache::lonlocal::locallocaltime($value);
    } elsif (ref($value) eq 'ARRAY') {
	$value = '('.join(', ', @{ $value }).')';
    } elsif ($key =~ /answerstring$/) {
        my %answers = &Apache::lonnet::str2hash($value);
        my @anskeys = sort(keys(%answers));
        if (@anskeys == 1) {
            my $answer = $answers{$anskeys[0]};
            if ($answer =~ m{\0}) {
                $answer =~ s{\0}{,}g;
            }
            my $tag_internal_answer_name = 'INTERNAL';
            if ($anskeys[0] eq $tag_internal_answer_name) {
                $value = $answer; 
            } else {
                $value = $anskeys[0].'='.$answer;
            }
        } else {
            foreach my $ans (@anskeys) {
                my $answer = $answers{$ans};
                if ($answer =~ m{\0}) {
                    $answer =~ s{\0}{,}g;
                }
                $value .=  $ans.'='.$answer.'<br />';;
            } 
        }
    } else {
	$value = &unescape($value);
    }
    return $value;
}


sub relative_to_absolute {
    my ($url,$output)=@_;
    my $parser=HTML::TokeParser->new(\$output);
    my $token;
    my $thisdir=$url;
    my @rlinks=();
    while ($token=$parser->get_token) {
	if ($token->[0] eq 'S') {
	    if ($token->[1] eq 'a') {
		if ($token->[2]->{'href'}) {
		    $rlinks[$#rlinks+1]=$token->[2]->{'href'};
		}
	    } elsif ($token->[1] eq 'img' || $token->[1] eq 'embed' ) {
		$rlinks[$#rlinks+1]=$token->[2]->{'src'};
	    } elsif ($token->[1] eq 'base') {
		$thisdir=$token->[2]->{'href'};
	    }
	}
    }
    $thisdir=~s-/[^/]*$--;
    foreach my $link (@rlinks) {
	unless (($link=~/^https?\:\/\//i) ||
		($link=~/^\//) ||
		($link=~/^javascript:/i) ||
		($link=~/^mailto:/i) ||
		($link=~/^\#/)) {
	    my $newlocation=&Apache::lonnet::hreflocation($thisdir,$link);
	    $output=~s/(\"|\'|\=\s*)\Q$link\E(\"|\'|\s|\>)/$1$newlocation$2/;
	}
    }
# -------------------------------------------------- Deal with Applet codebases
    $output=~s/(\<applet[^\>]+)(codebase\=[^\S\>]+)*([^\>]*)\>/$1.($2?$2:' codebase="'.$thisdir.'"').$3.'>'/gei;
    return $output;
}

=pod

=item * &get_student_view()

show a snapshot of what student was looking at

=cut

sub get_student_view {
  my ($symb,$username,$domain,$courseid,$target,$moreenv) = @_;
  my ($map,$id,$feedurl) = &Apache::lonnet::decode_symb($symb);
  my (%form);
  my @elements=('symb','courseid','domain','username');
  foreach my $element (@elements) {
      $form{'grade_'.$element}=eval '$'.$element #'
  }
  if (defined($moreenv)) {
      %form=(%form,%{$moreenv});
  }
  if (defined($target)) { $form{'grade_target'} = $target; }
  $feedurl=&Apache::lonnet::clutter($feedurl);
  my ($userview,$response)=&Apache::lonnet::ssi_body($feedurl,%form);
  $userview=~s/\<body[^\>]*\>//gi;
  $userview=~s/\<\/body\>//gi;
  $userview=~s/\<html\>//gi;
  $userview=~s/\<\/html\>//gi;
  $userview=~s/\<head\>//gi;
  $userview=~s/\<\/head\>//gi;
  $userview=~s/action\s*\=/would_be_action\=/gi;
  $userview=&relative_to_absolute($feedurl,$userview);
  if (wantarray) {
     return ($userview,$response);
  } else {
     return $userview;
  }
}

sub get_student_view_with_retries {
  my ($symb,$retries,$username,$domain,$courseid,$target,$moreenv) = @_;

    my $ok = 0;                 # True if we got a good response.
    my $content;
    my $response;

    # Try to get the student_view done. within the retries count:
    
    do {
         ($content, $response) = &get_student_view($symb,$username,$domain,$courseid,$target,$moreenv);
         $ok      = $response->is_success;
         if (!$ok) {
            &Apache::lonnet::logthis("Failed get_student_view_with_retries on $symb: ".$response->is_success.', '.$response->code.', '.$response->message);
         }
         $retries--;
    } while (!$ok && ($retries > 0));
    
    if (!$ok) {
       $content = '';          # On error return an empty content.
    }
    if (wantarray) {
       return ($content, $response);
    } else {
       return $content;
    }
}

sub css_links {
    my ($currsymb,$level) = @_;
    my ($links,@symbs,%cssrefs,%httpref);
    if ($level eq 'map') {
        my $navmap = Apache::lonnavmaps::navmap->new();
        if (ref($navmap)) {
            my ($map,undef,$url)=&Apache::lonnet::decode_symb($currsymb);
            my @resources = $navmap->retrieveResources($map,sub { $_[0]->is_problem() },0,0);
            foreach my $res (@resources) {
                if (ref($res) && $res->symb()) {
                    push(@symbs,$res->symb());
                }
            }
        }
    } else {
        @symbs = ($currsymb);
    }
    foreach my $symb (@symbs) {
        my $css_href = &Apache::lonnet::EXT('resource.0.cssfile',$symb);
        if ($css_href =~ /\S/) {
            unless ($css_href =~ m{https?://}) {
                my $url = (&Apache::lonnet::decode_symb($symb))[-1];
                my $proburl =  &Apache::lonnet::clutter($url);
                my ($probdir) = ($proburl =~ m{(.+)/[^/]+$});
                unless ($css_href =~ m{^/}) {
                    $css_href = &Apache::lonnet::hreflocation($probdir,$css_href);
                }
                if ($css_href =~ m{^/(res|uploaded)/}) {
                    unless (($httpref{'httpref.'.$css_href}) ||
                            (&Apache::lonnet::is_on_map($css_href))) {
                        my $thisurl = $proburl;
                        if ($env{'httpref.'.$proburl}) {
                            $thisurl = $env{'httpref.'.$proburl};
                        }
                        $httpref{'httpref.'.$css_href} = $thisurl;
                    }
                }
            }
            $cssrefs{$css_href} = 1;
        }
    }
    if (keys(%httpref)) {
        &Apache::lonnet::appenv(\%httpref);
    }
    if (keys(%cssrefs)) {
        foreach my $css_href (keys(%cssrefs)) {
            next unless ($css_href =~ m{^(/res/|/uploaded/|https?://)});
            $links .= '<link rel="stylesheet" type="text/css" href="'.$css_href.'" />'."\n";
        }
    }
    return $links;
}

=pod

=item * &get_student_answers() 

show a snapshot of how student was answering problem

=cut

sub get_student_answers {
  my ($symb,$username,$domain,$courseid,%form) = @_;
  my ($map,$id,$feedurl) = &Apache::lonnet::decode_symb($symb);
  my (%moreenv);
  my @elements=('symb','courseid','domain','username');
  foreach my $element (@elements) {
    $moreenv{'grade_'.$element}=eval '$'.$element #'
  }
  $moreenv{'grade_target'}='answer';
  %moreenv=(%form,%moreenv);
  $feedurl = &Apache::lonnet::clutter($feedurl);
  my $userview=&Apache::lonnet::ssi($feedurl,%moreenv);
  return $userview;
}

=pod

=item * &submlink()

Inputs: $text $uname $udom $symb $target

Returns: A link to grades.pm such as to see the SUBM view of a student

=cut

###############################################
sub submlink {
    my ($text,$uname,$udom,$symb,$target)=@_;
    if (!($uname && $udom)) {
	(my $cursymb, my $courseid,$udom,$uname)=
	    &Apache::lonnet::whichuser($symb);
	if (!$symb) { $symb=$cursymb; }
    }
    if (!$symb) { $symb=&Apache::lonnet::symbread(); }
    $symb=&escape($symb);
    if ($target) { $target=" target=\"$target\""; }
    return
        '<a href="/adm/grades?command=submission'.
        '&amp;symb='.$symb.
        '&amp;student='.$uname.
        '&amp;userdom='.$udom.'"'.
        $target.'>'.$text.'</a>';
}
##############################################

=pod

=item * &pgrdlink()

Inputs: $text $uname $udom $symb $target

Returns: A link to grades.pm such as to see the PGRD view of a student

=cut

###############################################
sub pgrdlink {
    my $link=&submlink(@_);
    $link=~s/(&command=submission)/$1&showgrading=yes/;
    return $link;
}
##############################################

=pod

=item * &pprmlink()

Inputs: $text $uname $udom $symb $target

Returns: A link to parmset.pm such as to see the PPRM view of a
student and a specific resource

=cut

###############################################
sub pprmlink {
    my ($text,$uname,$udom,$symb,$target)=@_;
    if (!($uname && $udom)) {
	(my $cursymb, my $courseid,$udom,$uname)=
	    &Apache::lonnet::whichuser($symb);
	if (!$symb) { $symb=$cursymb; }
    }
    if (!$symb) { $symb=&Apache::lonnet::symbread(); }
    $symb=&escape($symb);
    if ($target) { $target="target=\"$target\""; }
    return '<a href="/adm/parmset?command=set&amp;'.
	'symb='.$symb.'&amp;uname='.$uname.
	'&amp;udom='.$udom.'" '.$target.'>'.$text.'</a>';
}
##############################################

=pod

=back

=cut

###############################################


sub timehash {
    my ($thistime) = @_;
    my $timezone = &Apache::lonlocal::gettimezone();
    my $dt = DateTime->from_epoch(epoch => $thistime)
                     ->set_time_zone($timezone);
    my $wday = $dt->day_of_week();
    if ($wday == 7) { $wday = 0; }
    return ( 'second' => $dt->second(),
             'minute' => $dt->minute(),
             'hour'   => $dt->hour(),
             'day'     => $dt->day_of_month(),
             'month'   => $dt->month(),
             'year'    => $dt->year(),
             'weekday' => $wday,
             'dayyear' => $dt->day_of_year(),
             'dlsav'   => $dt->is_dst() );
}

sub utc_string {
    my ($date)=@_;
    return strftime("%Y%m%dT%H%M%SZ",gmtime($date));
}

sub maketime {
    my %th=@_;
    my ($epoch_time,$timezone,$dt);
    $timezone = &Apache::lonlocal::gettimezone();
    eval {
        $dt = DateTime->new( year   => $th{'year'},
                             month  => $th{'month'},
                             day    => $th{'day'},
                             hour   => $th{'hour'},
                             minute => $th{'minute'},
                             second => $th{'second'},
                             time_zone => $timezone,
                         );
    };
    if (!$@) {
        $epoch_time = $dt->epoch;
        if ($epoch_time) {
            return $epoch_time;
        }
    }
    return POSIX::mktime(
        ($th{'seconds'},$th{'minutes'},$th{'hours'},
         $th{'day'},$th{'month'}-1,$th{'year'}-1900,0,0,-1));
}

#########################################

sub findallcourses {
    my ($roles,$uname,$udom) = @_;
    my %roles;
    if (ref($roles)) { %roles = map { $_ => 1 } @{$roles}; }
    my %courses;
    my $now=time;
    if (!defined($uname)) {
        $uname = $env{'user.name'};
    }
    if (!defined($udom)) {
        $udom = $env{'user.domain'};
    }
    if (($uname ne $env{'user.name'}) || ($udom ne $env{'user.domain'})) {
        my %roleshash = &Apache::lonnet::dump('roles',$udom,$uname);
        if (!%roles) {
            %roles = (
                       cc => 1,
                       co => 1,
                       in => 1,
                       ep => 1,
                       ta => 1,
                       cr => 1,
                       st => 1,
             );
        }
        foreach my $entry (keys(%roleshash)) {
            my ($trole,$tend,$tstart) = split(/_/,$roleshash{$entry});
            if ($trole =~ /^cr/) { 
                next if (!exists($roles{$trole}) && !exists($roles{'cr'}));
            } else {
                next if (!exists($roles{$trole}));
            }
            if ($tend) {
                next if ($tend < $now);
            }
            if ($tstart) {
                next if ($tstart > $now);
            }
            my ($cdom,$cnum,$sec,$cnumpart,$secpart,$role);
            (undef,$cdom,$cnumpart,$secpart) = split(/\//,$entry);
            my $value = $trole.'/'.$cdom.'/';
            if ($secpart eq '') {
                ($cnum,$role) = split(/_/,$cnumpart); 
                $sec = 'none';
                $value .= $cnum.'/';
            } else {
                $cnum = $cnumpart;
                ($sec,$role) = split(/_/,$secpart);
                $value .= $cnum.'/'.$sec;
            }
            if (ref($courses{$cdom.'_'.$cnum}{$sec}) eq 'ARRAY') {
                unless (grep(/^\Q$value\E$/,@{$courses{$cdom.'_'.$cnum}{$sec}})) {
                    push(@{$courses{$cdom.'_'.$cnum}{$sec}},$value);
                }
            } else {
                @{$courses{$cdom.'_'.$cnum}{$sec}} = ($value);
            }
        }
    } else {
        foreach my $key (keys(%env)) {
	    if ( $key=~m{^user\.role\.(\w+)\./($match_domain)/($match_courseid)/?(\w*)$} ||
                 $key=~m{^user\.role\.(cr/$match_domain/$match_username/\w+)\./($match_domain)/($match_courseid)/?(\w*)$}) {
	        my ($role,$cdom,$cnum,$sec) = ($1,$2,$3,$4);
	        next if ($role eq 'ca' || $role eq 'aa');
	        next if (%roles && !exists($roles{$role}));
	        my ($starttime,$endtime)=split(/\./,$env{$key});
                my $active=1;
                if ($starttime) {
		    if ($now<$starttime) { $active=0; }
                }
                if ($endtime) {
                    if ($now>$endtime) { $active=0; }
                }
                if ($active) {
                    my $value = $role.'/'.$cdom.'/'.$cnum.'/';
                    if ($sec eq '') {
                        $sec = 'none';
                    } else {
                        $value .= $sec;
                    }
                    if (ref($courses{$cdom.'_'.$cnum}{$sec}) eq 'ARRAY') {
                        unless (grep(/^\Q$value\E$/,@{$courses{$cdom.'_'.$cnum}{$sec}})) {
                            push(@{$courses{$cdom.'_'.$cnum}{$sec}},$value);
                        }
                    } else {
                        @{$courses{$cdom.'_'.$cnum}{$sec}} = ($value);
                    }
                }
            }
        }
    }
    return %courses;
}

###############################################

sub blockcheck {
    my ($setters,$activity,$clientip,$uname,$udom,$url,$is_course,$symb,$caller) = @_;

    unless ($activity eq 'docs') {
        my ($has_evb,$check_ipaccess);
        my $dom = $env{'user.domain'};
        if ($env{'request.course.id'}) {
            my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
            my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
            my $checkrole = "cm./$cdom/$cnum";
            my $sec = $env{'request.course.sec'};
            if ($sec ne '') {
                $checkrole .= "/$sec";
            }
            if ((&Apache::lonnet::allowed('evb',undef,undef,$checkrole)) &&
                ($env{'request.role'} !~ /^st/)) {
                $has_evb = 1;
            }
            unless ($has_evb) {
                if (($activity eq 'printout') || ($activity eq 'grades') || ($activity eq 'search') ||
                    ($activity eq 'boards') || ($activity eq 'groups') || ($activity eq 'chat')) {
                    if ($udom eq $cdom) {
                        $check_ipaccess = 1;
                    }
                }
            }
        } elsif (($activity eq 'com') || ($activity eq 'port') || ($activity eq 'blogs') ||
                ($activity eq 'about') || ($activity eq 'wishlist') || ($activity eq 'passwd')) {
            my $checkrole;
            if ($env{'request.role.domain'} eq '') {
                $checkrole = "cm./$env{'user.domain'}/";
            } else {
                $checkrole = "cm./$env{'request.role.domain'}/";
            }
            if (($checkrole) && (&Apache::lonnet::allowed('evb',undef,undef,$checkrole))) {
                $has_evb = 1;
            }
        }
        unless ($has_evb || $check_ipaccess) {
            my @machinedoms = &Apache::lonnet::current_machine_domains();
            if (($dom eq 'public') && ($activity eq 'port')) {
                $dom = $udom;
            }
            if (($dom ne '') && (grep(/^\Q$dom\E$/,@machinedoms))) {
                $check_ipaccess = 1;
            } else {
                my $lonhost = $Apache::lonnet::perlvar{'lonHostID'};
                my $internet_names = &Apache::lonnet::get_internet_names($lonhost);
                my $prim = &Apache::lonnet::domain($dom,'primary');
                my $intdom = &Apache::lonnet::internet_dom($prim);
                if (($intdom ne '') && (ref($internet_names) eq 'ARRAY')) {
                    if (grep(/^\Q$intdom\E$/,@{$internet_names})) {
                        $check_ipaccess = 1;
                    }
                }
            }
        }
        if ($check_ipaccess) {
            my ($ipaccessref,$cached)=&Apache::lonnet::is_cached_new('ipaccess',$dom);
            unless (defined($cached)) {
                my %domconfig =
                    &Apache::lonnet::get_dom('configuration',['ipaccess'],$dom);
                $ipaccessref = &Apache::lonnet::do_cache_new('ipaccess',$dom,$domconfig{'ipaccess'},1800);
            }
            if ((ref($ipaccessref) eq 'HASH') && ($clientip)) {
                foreach my $id (keys(%{$ipaccessref})) {
                    if (ref($ipaccessref->{$id}) eq 'HASH') {
                        my $range = $ipaccessref->{$id}->{'ip'};
                        if ($range) {
                            if (&Apache::lonnet::ip_match($clientip,$range)) {
                                if (ref($ipaccessref->{$id}->{'commblocks'}) eq 'HASH') {
                                    if ($ipaccessref->{$id}->{'commblocks'}->{$activity} eq 'on') {
                                        return ('','','',$id,$dom);
                                        last;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        if (($activity eq 'wishlist') || ($activity eq 'annotate')) {
            return ();
        }
    }
    if (defined($udom) && defined($uname)) {
        # If uname and udom are for a course, check for blocks in the course.
        if (($is_course) || (&Apache::lonnet::is_course($udom,$uname))) {
            my ($startblock,$endblock,$triggerblock) =
                &get_blocks($setters,$activity,$udom,$uname,$url,$symb,$caller);
            return ($startblock,$endblock,$triggerblock);
        }
    } else {
        $udom = $env{'user.domain'};
        $uname = $env{'user.name'};
    }

    my $startblock = 0;
    my $endblock = 0;
    my $triggerblock = '';
    my %live_courses;
    unless (($activity eq 'wishlist') || ($activity eq 'annotate')) {
        %live_courses = &findallcourses(undef,$uname,$udom);
    }

    # If uname is for a user, and activity is course-specific, i.e.,
    # boards, chat or groups, check for blocking in current course only.

    if (($activity eq 'boards' || $activity eq 'chat' ||
         $activity eq 'groups' || $activity eq 'printout') &&
        ($env{'request.course.id'})) {
        foreach my $key (keys(%live_courses)) {
            if ($key ne $env{'request.course.id'}) {
                delete($live_courses{$key});
            }
        }
    }

    my $otheruser = 0;
    my %own_courses;
    if ((($uname ne $env{'user.name'})) || ($udom ne $env{'user.domain'})) {
        # Resource belongs to user other than current user.
        $otheruser = 1;
        # Gather courses for current user
        %own_courses = 
            &findallcourses(undef,$env{'user.name'},$env{'user.domain'});
    }

    # Gather active course roles - course coordinator, instructor, 
    # exam proctor, ta, student, or custom role.

    foreach my $course (keys(%live_courses)) {
        my ($cdom,$cnum);
        if ((defined($env{'course.'.$course.'.domain'})) && (defined($env{'course.'.$course.'.num'}))) {
            $cdom = $env{'course.'.$course.'.domain'};
            $cnum = $env{'course.'.$course.'.num'};
        } else {
            ($cdom,$cnum) = split(/_/,$course); 
        }
        my $no_ownblock = 0;
        my $no_userblock = 0;
        if ($otheruser && $activity ne 'com') {
            # Check if current user has 'evb' priv for this
            if (defined($own_courses{$course})) {
                foreach my $sec (keys(%{$own_courses{$course}})) {
                    my $checkrole = 'cm./'.$cdom.'/'.$cnum;
                    if ($sec ne 'none') {
                        $checkrole .= '/'.$sec;
                    }
                    if (&Apache::lonnet::allowed('evb',undef,undef,$checkrole)) {
                        $no_ownblock = 1;
                        last;
                    }
                }
            }
            # if they have 'evb' priv and are currently not playing student
            next if (($no_ownblock) &&
                 ($env{'request.role'} !~ m{^st\./$cdom/$cnum}));
        }
        foreach my $sec (keys(%{$live_courses{$course}})) {
            my $checkrole = 'cm./'.$cdom.'/'.$cnum;
            if ($sec ne 'none') {
                $checkrole .= '/'.$sec;
            }
            if ($otheruser) {
                # Resource belongs to user other than current user.
                # Assemble privs for that user, and check for 'evb' priv.
                my (%allroles,%userroles);
                if (ref($live_courses{$course}{$sec}) eq 'ARRAY') {
                    foreach my $entry (@{$live_courses{$course}{$sec}}) { 
                        my ($trole,$tdom,$tnum,$tsec);
                        if ($entry =~ /^cr/) {
                            ($trole,$tdom,$tnum,$tsec) = 
                                ($entry =~ m|^(cr/$match_domain/$match_username/\w+)\./($match_domain)/($match_username)/?(\w*)$|);
                        } else {
                           ($trole,$tdom,$tnum,$tsec) = split(/\//,$entry);
                        }
                        my ($spec,$area,$trest);
                        $area = '/'.$tdom.'/'.$tnum;
                        $trest = $tnum;
                        if ($tsec ne '') {
                            $area .= '/'.$tsec;
                            $trest .= '/'.$tsec;
                        }
                        $spec = $trole.'.'.$area;
                        if ($trole =~ /^cr/) {
                            &Apache::lonnet::custom_roleprivs(\%allroles,$trole,
                                                              $tdom,$spec,$trest,$area);
                        } else {
                            &Apache::lonnet::standard_roleprivs(\%allroles,$trole,
                                                                $tdom,$spec,$trest,$area);
                        }
                    }
                    my ($author,$adv,$rar) = &Apache::lonnet::set_userprivs(\%userroles,\%allroles);
                    if ($userroles{'user.priv.'.$checkrole} =~ /evb\&([^\:]*)/) {
                        if ($1) {
                            $no_userblock = 1;
                            last;
                        }
                    }
                }
            } else {
                # Resource belongs to current user
                # Check for 'evb' priv via lonnet::allowed().
                if (&Apache::lonnet::allowed('evb',undef,undef,$checkrole)) {
                    $no_ownblock = 1;
                    last;
                }
            }
        }
        # if they have the evb priv and are currently not playing student
        next if (($no_ownblock) &&
                 ($env{'request.role'} !~ m{^st\./\Q$cdom\E/\Q$cnum\E}));
        next if ($no_userblock);

        # Retrieve blocking times and identity of blocker for course
        # of specified user, unless user has 'evb' privilege.
        
        my ($start,$end,$trigger) = 
            &get_blocks($setters,$activity,$cdom,$cnum,$url,$symb,$caller);
        if (($start != 0) && 
            (($startblock == 0) || ($startblock > $start))) {
            $startblock = $start;
            if ($trigger ne '') {
                $triggerblock = $trigger;
            }
        }
        if (($end != 0)  &&
            (($endblock == 0) || ($endblock < $end))) {
            $endblock = $end;
            if ($trigger ne '') {
                $triggerblock = $trigger;
            }
        }
    }
    return ($startblock,$endblock,$triggerblock);
}

sub get_blocks {
    my ($setters,$activity,$cdom,$cnum,$url,$symb,$caller) = @_;
    my $startblock = 0;
    my $endblock = 0;
    my $triggerblock = '';
    my $course = $cdom.'_'.$cnum;
    $setters->{$course} = {};
    $setters->{$course}{'staff'} = [];
    $setters->{$course}{'times'} = [];
    $setters->{$course}{'triggers'} = [];
    my (@blockers,%triggered);
    my $now = time;
    my %commblocks = &Apache::lonnet::get_comm_blocks($cdom,$cnum);
    if ($activity eq 'docs') {
        my ($blocked,$nosymbcache,$noenccheck);
        if (($caller eq 'blockedaccess') || ($caller eq 'blockingstatus')) {
            $blocked = 1;
            $nosymbcache = 1;
            $noenccheck = 1;
        }
        @blockers = &Apache::lonnet::has_comm_blocking('bre',$symb,$url,$nosymbcache,$noenccheck,$blocked,\%commblocks);
        foreach my $block (@blockers) {
            if ($block =~ /^firstaccess____(.+)$/) {
                my $item = $1;
                my $type = 'map';
                my $timersymb = $item;
                if ($item eq 'course') {
                    $type = 'course';
                } elsif ($item =~ /___\d+___/) {
                    $type = 'resource';
                } else {
                    $timersymb = &Apache::lonnet::symbread($item);
                }
                my $start = $env{'course.'.$cdom.'_'.$cnum.'.firstaccess.'.$timersymb};
                my $end = $start + $env{'course.'.$cdom.'_'.$cnum.'.timerinterval.'.$timersymb};
                $triggered{$block} = {
                                       start => $start,
                                       end   => $end,
                                       type  => $type,
                                     };
            }
        }
    } else {
        foreach my $block (keys(%commblocks)) {
            if ($block =~ m/^(\d+)____(\d+)$/) { 
                my ($start,$end) = ($1,$2);
                if ($start <= time && $end >= time) {
                    if (ref($commblocks{$block}) eq 'HASH') {
                        if (ref($commblocks{$block}{'blocks'}) eq 'HASH') {
                            if ($commblocks{$block}{'blocks'}{$activity} eq 'on') {
                                unless(grep(/^\Q$block\E$/,@blockers)) {
                                    push(@blockers,$block);
                                }
                            }
                        }
                    }
                }
            } elsif ($block =~ /^firstaccess____(.+)$/) {
                my $item = $1;
                my $timersymb = $item; 
                my $type = 'map';
                if ($item eq 'course') {
                    $type = 'course';
                } elsif ($item =~ /___\d+___/) {
                    $type = 'resource';
                } else {
                    $timersymb = &Apache::lonnet::symbread($item);
                }
                my $start = $env{'course.'.$cdom.'_'.$cnum.'.firstaccess.'.$timersymb};
                my $end = $start + $env{'course.'.$cdom.'_'.$cnum.'.timerinterval.'.$timersymb}; 
                if ($start && $end) {
                    if (($start <= time) && ($end >= time)) {
                        if (ref($commblocks{$block}) eq 'HASH') {
                            if (ref($commblocks{$block}{'blocks'}) eq 'HASH') {
                                if ($commblocks{$block}{'blocks'}{$activity} eq 'on') {
                                    unless(grep(/^\Q$block\E$/,@blockers)) {
                                        push(@blockers,$block);
                                        $triggered{$block} = {
                                                               start => $start,
                                                               end   => $end,
                                                               type  => $type,
                                                             };
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    foreach my $blocker (@blockers) {
        my ($staff_name,$staff_dom,$title,$blocks) =
            &parse_block_record($commblocks{$blocker});
        push(@{$$setters{$course}{'staff'}},[$staff_name,$staff_dom]);
        my ($start,$end,$triggertype);
        if ($blocker =~ m/^(\d+)____(\d+)$/) {
            ($start,$end) = ($1,$2);
        } elsif (ref($triggered{$blocker}) eq 'HASH') {
            $start = $triggered{$blocker}{'start'};
            $end = $triggered{$blocker}{'end'};
            $triggertype = $triggered{$blocker}{'type'};
        }
        if ($start) {
            push(@{$$setters{$course}{'times'}}, [$start,$end]);
            if ($triggertype) {
                push(@{$$setters{$course}{'triggers'}},$triggertype);
            } else {
                push(@{$$setters{$course}{'triggers'}},0);
            }
            if ( ($startblock == 0) || ($startblock > $start) ) {
                $startblock = $start;
                if ($triggertype) {
                    $triggerblock = $blocker;
                }
            }
            if ( ($endblock == 0) || ($endblock < $end) ) {
               $endblock = $end;
               if ($triggertype) {
                   $triggerblock = $blocker;
               }
            }
        }
    }
    return ($startblock,$endblock,$triggerblock);
}

sub parse_block_record {
    my ($record) = @_;
    my ($setuname,$setudom,$title,$blocks);
    if (ref($record) eq 'HASH') {
        ($setuname,$setudom) = split(/:/,$record->{'setter'});
        $title = &unescape($record->{'event'});
        $blocks = $record->{'blocks'};
    } else {
        my @data = split(/:/,$record,3);
        if (scalar(@data) eq 2) {
            $title = $data[1];
            ($setuname,$setudom) = split(/@/,$data[0]);
        } else {
            ($setuname,$setudom,$title) = @data;
        }
        $blocks = { 'com' => 'on' };
    }
    return ($setuname,$setudom,$title,$blocks);
}

sub blocking_status {
    my ($activity,$clientip,$uname,$udom,$url,$is_course,$symb,$caller) = @_;
    my %setters;

# check for active blocking
    if ($clientip eq '') {
        $clientip = &Apache::lonnet::get_requestor_ip();
    }
    my ($startblock,$endblock,$triggerblock,$by_ip,$blockdom) = 
        &blockcheck(\%setters,$activity,$clientip,$uname,$udom,$url,$is_course,$symb,$caller);
    my $blocked = 0;
    if (($startblock && $endblock) || ($by_ip)) {
        $blocked = 1;
    }

# caller just wants to know whether a block is active
    if (!wantarray) { return $blocked; }

# build a link to a popup window containing the details
    my $querystring  = "?activity=$activity";
# $uname and $udom decide whose portfolio (or information page) the user is trying to look at
    if (($activity eq 'port') || ($activity eq 'about') || ($activity eq 'passwd')) {
        $querystring .= "&amp;udom=$udom"      if ($udom =~ /^$match_domain$/);
        $querystring .= "&amp;uname=$uname"    if ($uname =~ /^$match_username$/);
    } elsif ($activity eq 'docs') {
        my $showurl = &Apache::lonenc::check_encrypt($url);
        $querystring .= '&amp;url='.&HTML::Entities::encode($showurl,'\'&"<>');
        if ($symb) {
            my $showsymb = &Apache::lonenc::check_encrypt($symb);
            $querystring .= '&amp;symb='.&HTML::Entities::encode($showsymb,'\'&"<>');
        }
    }

    my $output .= <<'END_MYBLOCK';
function openWindow(url, wdwName, w, h, toolbar,scrollbar) {
    var options = "width=" + w + ",height=" + h + ",";
    options += "resizable=yes,scrollbars="+scrollbar+",status=no,";
    options += "menubar=no,toolbar="+toolbar+",location=no,directories=no";
    var newWin = window.open(url, wdwName, options);
    newWin.focus();
}
END_MYBLOCK

    $output = Apache::lonhtmlcommon::scripttag($output);
  
    my $popupUrl = "/adm/blockingstatus/$querystring";
    my $text = &mt('Communication Blocked');
    my $class = 'LC_comblock';
    if ($activity eq 'docs') {
        $text = &mt('Content Access Blocked');
        $class = '';
    } elsif ($activity eq 'printout') {
        $text = &mt('Printing Blocked');
    } elsif ($activity eq 'passwd') {
        $text = &mt('Password Changing Blocked');
    } elsif ($activity eq 'grades') {
        $text = &mt('Gradebook Blocked');
    } elsif ($activity eq 'search') {
        $text = &mt('Search Blocked');
    } elsif ($activity eq 'about') {
        $text = &mt('Access to User Information Pages Blocked');
    } elsif ($activity eq 'wishlist') {
        $text = &mt('Access to Stored Links Blocked');
    } elsif ($activity eq 'annotate') {
        $text = &mt('Access to Annotations Blocked');
    }
    $output .= <<"END_BLOCK";
<div class='$class'>
  <a onclick='openWindow("$popupUrl","Blocking Table",600,300,"no","no");return false;' href='/adm/blockingstatus/$querystring'
  title='$text'>
  <img class='LC_noBorder LC_middle' title='$text' src='/res/adm/pages/comblock.png' alt='$text'/></a>
  <a onclick='openWindow("$popupUrl","Blocking Table",600,300,"no","no");return false;' href='/adm/blockingstatus/$querystring' 
  title='$text'>$text</a>
</div>

END_BLOCK

    return ($blocked, $output);
}

###############################################

sub check_ip_acc {
    my ($acc,$clientip)=@_;
    &Apache::lonxml::debug("acc is $acc");
    if (!defined($acc) || $acc =~ /^\s*$/ || $acc =~/^\s*no\s*$/i) {
        return 1;
    }
    my $allowed=0;
    my $ip;
    if (($ENV{'REMOTE_ADDR'} eq '127.0.0.1') ||
        ($ENV{'REMOTE_ADDR'} eq &Apache::lonnet::get_host_ip($Apache::lonnet::perlvar{'lonHostID'}))) {
        $ip = $env{'request.host'} || $ENV{'REMOTE_ADDR'} || $clientip;
    } else {
        my $remote_ip = &Apache::lonnet::get_requestor_ip();
        $ip = $remote_ip || $env{'request.host'} || $clientip;
    }

    my $name;
    foreach my $pattern (split(',',$acc)) {
        $pattern =~ s/^\s*//;
        $pattern =~ s/\s*$//;
        if ($pattern =~ /\*$/) {
            #35.8.*
            $pattern=~s/\*//;
            if ($ip =~ /^\Q$pattern\E/) { $allowed=1; }
        } elsif ($pattern =~ /(\d+\.\d+\.\d+)\.\[(\d+)-(\d+)\]$/) {
            #35.8.3.[34-56]
            my $low=$2;
            my $high=$3;
            $pattern=$1;
            if ($ip =~ /^\Q$pattern\E/) {
                my $last=(split(/\./,$ip))[3];
                if ($last <=$high && $last >=$low) { $allowed=1; }
            }
        } elsif ($pattern =~ /^\*/) {
            #*.msu.edu
            $pattern=~s/\*//;
            if (!defined($name)) {
                use Socket;
                my $netaddr=inet_aton($ip);
                ($name)=gethostbyaddr($netaddr,AF_INET);
            }
            if ($name =~ /\Q$pattern\E$/i) { $allowed=1; }
        } elsif ($pattern =~ /\d+\.\d+\.\d+\.\d+/) {
            #127.0.0.1
            if ($ip =~ /^\Q$pattern\E/) { $allowed=1; }
        } else {
            #some.name.com
            if (!defined($name)) {
                use Socket;
                my $netaddr=inet_aton($ip);
                ($name)=gethostbyaddr($netaddr,AF_INET);
            }
            if ($name =~ /\Q$pattern\E$/i) { $allowed=1; }
        }
        if ($allowed) { last; }
    }
    return $allowed;
}

###############################################

=pod

=head1 Domain Template Functions

=over 4

=item * &determinedomain()

Inputs: $domain (usually will be undef)

Returns: Determines which domain should be used for designs

=cut

###############################################
sub determinedomain {
    my $domain=shift;
    if (! $domain) {
        # Determine domain if we have not been given one
        $domain = &Apache::lonnet::default_login_domain();
        if ($env{'user.domain'}) { $domain=$env{'user.domain'}; }
        if ($env{'request.role.domain'}) { 
            $domain=$env{'request.role.domain'}; 
        }
    }
    return $domain;
}
###############################################

sub devalidate_domconfig_cache {
    my ($udom)=@_;
    &Apache::lonnet::devalidate_cache_new('domainconfig',$udom);
}

# ---------------------- Get domain configuration for a domain
sub get_domainconf {
    my ($udom) = @_;
    my $cachetime=1800;
    my ($result,$cached)=&Apache::lonnet::is_cached_new('domainconfig',$udom);
    if (defined($cached)) { return %{$result}; }

    my %domconfig = &Apache::lonnet::get_dom('configuration',
					     ['login','rolecolors','autoenroll'],$udom);
    my (%designhash,%legacy);
    if (keys(%domconfig) > 0) {
        if (ref($domconfig{'login'}) eq 'HASH') {
            if (keys(%{$domconfig{'login'}})) {
                foreach my $key (keys(%{$domconfig{'login'}})) {
                    if (ref($domconfig{'login'}{$key}) eq 'HASH') {
                        if (($key eq 'loginvia') || ($key eq 'headtag')) {
                            if (ref($domconfig{'login'}{$key}) eq 'HASH') {
                                foreach my $hostname (keys(%{$domconfig{'login'}{$key}})) {
                                    if (ref($domconfig{'login'}{$key}{$hostname}) eq 'HASH') {
                                        if ($key eq 'loginvia') {
                                            if ($domconfig{'login'}{'loginvia'}{$hostname}{'server'}) {
                                                my $server = $domconfig{'login'}{'loginvia'}{$hostname}{'server'};
                                                $designhash{$udom.'.login.loginvia'} = $server;
                                                if ($domconfig{'login'}{'loginvia'}{$hostname}{'serverpath'} eq 'custom') {
                                                    $designhash{$udom.'.login.loginvia_'.$hostname} = $server.':'.$domconfig{'login'}{'loginvia'}{$hostname}{'custompath'};
                                                } else {
                                                    $designhash{$udom.'.login.loginvia_'.$hostname} = $server.':'.$domconfig{'login'}{'loginvia'}{$hostname}{'serverpath'};
                                                }
                                            }
                                        } elsif ($key eq 'headtag') {
                                            if ($domconfig{'login'}{'headtag'}{$hostname}{'url'}) {
                                                $designhash{$udom.'.login.headtag_'.$hostname} = $domconfig{'login'}{'headtag'}{$hostname}{'url'};
                                            }
                                        }
                                        if ($domconfig{'login'}{$key}{$hostname}{'exempt'}) {
                                            $designhash{$udom.'.login.'.$key.'_exempt_'.$hostname} = $domconfig{'login'}{$key}{$hostname}{'exempt'};
                                        }
                                    }
                                }
                            }
                        } elsif ($key eq 'saml') {
                            if (ref($domconfig{'login'}{$key}) eq 'HASH') {
                                foreach my $host (keys(%{$domconfig{'login'}{$key}})) {
                                    if (ref($domconfig{'login'}{$key}{$host}) eq 'HASH') {
                                        $designhash{$udom.'.login.'.$key.'_'.$host} = 1;
                                        foreach my $item ('text','img','alt','url','title','notsso') {
                                            $designhash{$udom.'.login.'.$key.'_'.$item.'_'.$host} = $domconfig{'login'}{$key}{$host}{$item};
                                        }
                                    }
                                }
                            }
                        } else {
                            foreach my $img (keys(%{$domconfig{'login'}{$key}})) {
                                $designhash{$udom.'.login.'.$key.'_'.$img} = 
                                    $domconfig{'login'}{$key}{$img};
                            }
                        }
                    } else {
                        $designhash{$udom.'.login.'.$key}=$domconfig{'login'}{$key};
                    }
                }
            } else {
                $legacy{'login'} = 1;
            }
        } else {
            $legacy{'login'} = 1;
        }
        if (ref($domconfig{'rolecolors'}) eq 'HASH') {
            if (keys(%{$domconfig{'rolecolors'}})) {
                foreach my $role (keys(%{$domconfig{'rolecolors'}})) {
                    if (ref($domconfig{'rolecolors'}{$role}) eq 'HASH') {
                        foreach my $item (keys(%{$domconfig{'rolecolors'}{$role}})) {
                            $designhash{$udom.'.'.$role.'.'.$item}=$domconfig{'rolecolors'}{$role}{$item};
                        }
                    }
                }
            } else {
                $legacy{'rolecolors'} = 1;
            }
        } else {
            $legacy{'rolecolors'} = 1;
        }
        if (ref($domconfig{'autoenroll'}) eq 'HASH') {
            if ($domconfig{'autoenroll'}{'co-owners'}) {
                $designhash{$udom.'.autoassign.co-owners'}=$domconfig{'autoenroll'}{'co-owners'};
            }
        }
        if (keys(%legacy) > 0) {
            my %legacyhash = &get_legacy_domconf($udom);
            foreach my $item (keys(%legacyhash)) {
                if ($item =~ /^\Q$udom\E\.login/) {
                    if ($legacy{'login'}) { 
                        $designhash{$item} = $legacyhash{$item};
                    }
                } else {
                    if ($legacy{'rolecolors'}) {
                        $designhash{$item} = $legacyhash{$item};
                    }
                }
            }
        }
    } else {
        %designhash = &get_legacy_domconf($udom); 
    }
    &Apache::lonnet::do_cache_new('domainconfig',$udom,\%designhash,
				  $cachetime);
    return %designhash;
}

sub get_legacy_domconf {
    my ($udom) = @_;
    my %legacyhash;
    my $designdir=$Apache::lonnet::perlvar{'lonTabDir'}.'/lonDomColors';
    my $designfile =  $designdir.'/'.$udom.'.tab';
    if (-e $designfile) {
        if ( open (my $fh,'<',$designfile) ) {
            while (my $line = <$fh>) {
                next if ($line =~ /^\#/);
                chomp($line);
                my ($key,$val)=(split(/\=/,$line));
                if ($val) { $legacyhash{$udom.'.'.$key}=$val; }
            }
            close($fh);
        }
    }
    if (-e $Apache::lonnet::perlvar{'lonDocRoot'}.'/adm/lonDomLogos/'.$udom.'.gif') {
        $legacyhash{$udom.'.login.domlogo'} = "/adm/lonDomLogos/$udom.gif";
    }
    return %legacyhash;
}

=pod

=item * &domainlogo()

Inputs: $domain (usually will be undef)

Returns: A link to a domain logo, if the domain logo exists.
If the domain logo does not exist, a description of the domain.

=cut

###############################################
sub domainlogo {
    my $domain = &determinedomain(shift);
    my %designhash = &get_domainconf($domain);    
    # See if there is a logo
    if ($designhash{$domain.'.login.domlogo'} ne '') {
        my $imgsrc = $designhash{$domain.'.login.domlogo'};
        if ($imgsrc =~ m{^/(adm|res)/}) {
	    if ($imgsrc =~ m{^/res/}) {
		my $local_name = &Apache::lonnet::filelocation('',$imgsrc);
		&Apache::lonnet::repcopy($local_name);
	    }
	   $imgsrc = &lonhttpdurl($imgsrc);
        }
        my $alttext = $domain;
        if ($designhash{$domain.'.login.alttext_domlogo'} ne '') {
            $alttext = $designhash{$domain.'.login.alttext_domlogo'};
        }
        return '<img src="'.$imgsrc.'" alt="'.$alttext.'" id="lclogindomlogo" />';
    } elsif (defined(&Apache::lonnet::domain($domain,'description'))) {
        return &Apache::lonnet::domain($domain,'description');
    } else {
        return '';
    }
}
##############################################

=pod

=item * &designparm()

Inputs: $which parameter; $domain (usually will be undef)

Returns: value of designparamter $which

=cut


##############################################
sub designparm {
    my ($which,$domain)=@_;
    if (exists($env{'environment.color.'.$which})) {
        return $env{'environment.color.'.$which};
    }
    $domain=&determinedomain($domain);
    my %domdesign;
    unless ($domain eq 'public') {
        %domdesign = &get_domainconf($domain);
    }
    my $output;
    if ($domdesign{$domain.'.'.$which} ne '') {
        $output = $domdesign{$domain.'.'.$which};
    } else {
        $output = $defaultdesign{$which};
    }
    if (($which =~ /^(student|coordinator|author|admin)\.img$/) ||
        ($which =~ /login\.(img|logo|domlogo|login)/)) {
        if ($output =~ m{^/(adm|res)/}) {
            if ($output =~ m{^/res/}) {
                my $local_name = &Apache::lonnet::filelocation('',$output);
                &Apache::lonnet::repcopy($local_name);
            }
            $output = &lonhttpdurl($output);
        }
    }
    return $output;
}

##############################################
=pod

=item * &authorspace()

Inputs: $url (usually will be undef).

Returns: Path to Authoring Space containing the resource or 
         directory being viewed (or for which action is being taken). 
         If $url is provided, and begins /priv/<domain>/<uname>
         the path will be that portion of the $context argument.
         Otherwise the path will be for the author space of the current
         user when the current role is author, or for that of the 
         co-author/assistant co-author space when the current role 
         is co-author or assistant co-author.

=cut

sub authorspace {
    my ($url) = @_;
    if ($url ne '') {
        if ($url =~ m{^(/priv/$match_domain/$match_username/)}) {
           return $1;
        }
    }
    my $caname = '';
    my $cadom = '';
    if ($env{'request.role'} =~ /^(?:ca|aa)/) {
        ($cadom,$caname) =
            ($env{'request.role'}=~/($match_domain)\/($match_username)$/);
    } elsif ($env{'request.role'} =~ m{^au\./($match_domain)/}) {
        $caname = $env{'user.name'};
        $cadom = $env{'user.domain'};
    }
    if (($caname ne '') && ($cadom ne '')) {
        return "/priv/$cadom/$caname/";
    }
    return;
}

##############################################
=pod

=item * &head_subbox()

Inputs: $content (contains HTML code with page functions, etc.)

Returns: HTML div with $content
         To be included in page header

=cut

sub head_subbox {
    my ($content)=@_;
    my $output =
        '<div class="LC_head_subbox">'
       .$content
       .'</div>'
}

##############################################
=pod

=item * &CSTR_pageheader()

Input: (optional) filename from which breadcrumb trail is built.
       In most cases no input as needed, as $env{'request.filename'}
       is appropriate for use in building the breadcrumb trail.

Returns: HTML div with CSTR path and recent box
         To be included on Authoring Space pages

=cut

sub CSTR_pageheader {
    my ($trailfile) = @_;
    if ($trailfile eq '') {
        $trailfile = $env{'request.filename'};
    }

# this is for resources; directories have customtitle, and crumbs
# and select recent are created in lonpubdir.pm

    my $londocroot = $Apache::lonnet::perlvar{'lonDocRoot'};
    my ($udom,$uname,$thisdisfn)=
        ($trailfile =~ m{^\Q$londocroot\E/priv/([^/]+)/([^/]+)(?:|/(.*))$});
    my $formaction = "/priv/$udom/$uname/$thisdisfn";
    $formaction =~ s{/+}{/}g;

    my $parentpath = '';
    my $lastitem = '';
    if ($thisdisfn =~ m-(.+/)([^/]*)$-) {
        $parentpath = $1;
        $lastitem = $2;
    } else {
        $lastitem = $thisdisfn;
    }

    my $output =
         '<div>'
        .&Apache::loncommon::help_open_menu('','',3,'Authoring') #FIXME: Broken? Where is it?
        .'<b>'.&mt('Authoring Space:').'</b> '
        .'<form name="dirs" method="post" action="'.$formaction
        .'" target="_top">' #FIXME lonpubdir: target="_parent"
        .&Apache::lonhtmlcommon::crumbs($uname.'/'.$parentpath,'_top','/priv/'.$udom,undef,undef);

    if ($lastitem) {
        $output .=
             '<span class="LC_filename">'
            .$lastitem
            .'</span>';
    }
    $output .=
         '<br />'
        #FIXME lonpubdir: &Apache::lonhtmlcommon::crumbs($uname.$thisdisfn.'/','_top','/priv','','+1',1)."</b></tt><br />"
        .&Apache::lonhtmlcommon::select_recent('construct','recent','this.form.action=this.form.recent.value;this.form.submit()')
        .'</form>'
        .&Apache::lonmenu::constspaceform()
        .'</div>';

    return $output;
}

###############################################
###############################################

=pod

=back

=head1 HTML Helpers

=over 4

=item * &bodytag()

Returns a uniform header for LON-CAPA web pages.

Inputs: 

=over 4

=item * $title, A title to be displayed on the page.

=item * $function, the current role (can be undef).

=item * $addentries, extra parameters for the <body> tag.

=item * $bodyonly, if defined, only return the <body> tag.

=item * $domain, if defined, force a given domain.

=item * $forcereg, if page should register as content page (relevant for 
            text interface only)

=item * $no_nav_bar, if true, keep the 'what is this' info but remove the
                     navigational links

=item * $bgcolor, used to override the bgcolor on a webpage to a specific value

=item * $no_inline_link, if true and in remote mode, don't show the
         'Switch To Inline Menu' link

=item * $args, optional argument valid values are
            no_auto_mt_title -> prevents &mt()ing the title arg
            use_absolute     -> for external resource or syllabus, this will
                                contain https://<hostname> if server uses
                                https (as per hosts.tab), but request is for http
            hostname         -> hostname, from $r->hostname().

=item * $advtoolsref, optional argument, ref to an array containing
            inlineremote items to be added in "Functions" menu below
            breadcrumbs.

=back

Returns: A uniform header for LON-CAPA web pages.  
If $bodyonly is nonzero, a string containing a <body> tag will be returned.
If $bodyonly is undef or zero, an html string containing a <body> tag and 
other decorations will be returned.

=cut

sub bodytag {
    my ($title,$function,$addentries,$bodyonly,$domain,$forcereg,
        $no_nav_bar,$bgcolor,$no_inline_link,$args,$advtoolsref)=@_;

    my $public;
    if ((($env{'user.name'} eq 'public') && ($env{'user.domain'} eq 'public'))
        || ($env{'user.name'} eq '') && ($env{'user.domain'} eq '')) {
        $public = 1;
    }
    if (!$args->{'no_auto_mt_title'}) { $title = &mt($title); }
    my $httphost = $args->{'use_absolute'};
    my $hostname = $args->{'hostname'};

    $function = &get_users_function() if (!$function);
    my $img =    &designparm($function.'.img',$domain);
    my $font =   &designparm($function.'.font',$domain);
    my $pgbg   = $bgcolor || &designparm($function.'.pgbg',$domain);

    my %design = ( 'style'   => 'margin-top: 0',
		   'bgcolor' => $pgbg,
		   'text'    => $font,
                   'alink'   => &designparm($function.'.alink',$domain),
		   'vlink'   => &designparm($function.'.vlink',$domain),
		   'link'    => &designparm($function.'.link',$domain),);
    @design{keys(%$addentries)} = @$addentries{keys(%$addentries)}; 

 # role and realm
    my ($role,$realm) = split(m{\./},$env{'request.role'},2);
    if ($realm) {
        $realm = '/'.$realm;
    }
    if ($role eq 'ca') {
        my ($rdom,$rname) = ($realm =~ m{^/($match_domain)/($match_username)$});
        $realm = &plainname($rname,$rdom);
    } 
# realm
    my ($cid,$sec);
    if ($env{'request.course.id'}) {
        $cid = $env{'request.course.id'};
        if ($env{'request.course.sec'}) {
            $sec = $env{'request.course.sec'};
        }
    } elsif ($realm =~ m{^/($match_domain)/($match_courseid)(?:|/(\w+))$}) {
        if (&Apache::lonnet::is_course($1,$2)) {
            $cid = $1.'_'.$2;
            $sec = $3;
        }
    }
    if ($cid) {
        if ($env{'request.role'} !~ /^cr/) {
            $role = &Apache::lonnet::plaintext($role,&course_type());
        } elsif ($role =~ m{^cr/($match_domain)/\1-domainconfig/(\w+)$}) {
            if ($env{'request.role.desc'}) {
                $role = $env{'request.role.desc'};
            } else {
                $role = &mt('Helpdesk[_1]','&nbsp;'.$2);
            }
        } else {
            $role = (split(/\//,$role,4))[-1];
        }
        if ($sec) {
            $role .= ('&nbsp;'x2).'-&nbsp;'.&mt('section:').'&nbsp;'.$sec;
        }   
	$realm = $env{'course.'.$cid.'.description'};
    } else {
        $role = &Apache::lonnet::plaintext($role);
    }

    if (!$realm) { $realm='&nbsp;'; }

    my $extra_body_attr = &make_attr_string($forcereg,\%design);

# construct main body tag
    my $bodytag = "<body $extra_body_attr>".
	&Apache::lontexconvert::init_math_support();

    &get_unprocessed_cgi($ENV{'QUERY_STRING'}, ['inhibitmenu']);

    if (($bodyonly) || ($no_nav_bar) || ($env{'form.inhibitmenu'} eq 'yes')) {
        return $bodytag;
    }

    if ($public) {
	undef($role);
    }

    my $titleinfo = '<h1>'.$title.'</h1>';
    #
    # Extra info if you are the DC
    my $dc_info = '';
    if (($env{'user.adv'}) && ($env{'request.course.id'}) &&
        (exists($env{'user.role.dc./'.$env{'course.'.$cid.'.domain'}.'/'}))) {
        $dc_info = $cid.' '.$env{'course.'.$cid.'.internal.coursecode'};
        $dc_info =~ s/\s+$//;
    }

    $role = '<span class="LC_nobreak">('.$role.')</span>' if ($role && !$env{'browser.mobile'});

    if ($env{'request.state'} eq 'construct') { $forcereg=1; }



    my $funclist;
    if (($env{'environment.remote'} eq 'on') && ($env{'request.state'} ne 'construct')) {
        unless ($args->{'switchserver'}) {
            $bodytag .= Apache::lonhtmlcommon::scripttag(Apache::lonmenu::utilityfunctions($httphost), 'start')."\n".
                        Apache::lonmenu::serverform();
            my $forbodytag;
            &Apache::lonmenu::prepare_functions($env{'request.noversionuri'},
                                                $forcereg,$args->{'group'},
                                                $args->{'bread_crumbs'},
                                                $advtoolsref,'','',\$forbodytag);
            unless (ref($args->{'bread_crumbs'}) eq 'ARRAY') {
                $funclist = $forbodytag;
            }
            $bodytag .= Apache::lonhtmlcommon::scripttag('', 'end');
        }
    } else {

        #    if ($env{'request.state'} eq 'construct') {
        #        $titleinfo = &CSTR_pageheader(); #FIXME: Will be removed once all scripts have their own calls
        #    }

        my $need_endlcint;
        unless ($args->{'switchserver'}) {
            $bodytag .= Apache::lonhtmlcommon::scripttag(
                Apache::lonmenu::utilityfunctions($httphost), 'start');
            $need_endlcint = 1;
        }

        if (($env{'environment.remote'} eq 'on') && ($env{'request.state'} eq 'construct')) {
            unless ($env{'form.inhibitmenu'}) {
                $bodytag .= &inline_for_remote($public,$role,$realm,$dc_info,$no_inline_link);
            }
        } else {
            my ($left,$right) = Apache::lonmenu::primary_menu($args->{'links_disabled'});

            if ($env{'request.noversionuri'} =~ m{^/res/adm/pages/}) {
                if ($dc_info) {
                    $dc_info = qq|<span class="LC_cusr_subheading">$dc_info</span>|;
                }
                $bodytag .= qq|<div id="LC_nav_bar">$left $role<br />
                               <em>$realm</em> $dc_info</div>|;
                if ($need_endlcint) {
                    $bodytag .= Apache::lonhtmlcommon::scripttag('', 'end');
                }
                return $bodytag;
            }

            unless ($env{'request.symb'} =~ m/\.page___\d+___/) {
                $bodytag .= qq|<div id="LC_nav_bar">$left $role</div>|;
            }

            $bodytag .= $right;

            if ($dc_info) {
                $dc_info = &dc_courseid_toggle($dc_info);
            }
            $bodytag .= qq|<div id="LC_realm">$realm $dc_info</div>|;
        }

        #if directed to not display the secondary menu, don't.
        if ($args->{'no_secondary_menu'}) {
            if ($need_endlcint) {
                $bodytag .= Apache::lonhtmlcommon::scripttag('', 'end');
            }
            return $bodytag;
        }
        #don't show menus for public users
        if (!$public){
            unless (($env{'environment.remote'} eq 'on') &&
                     ($env{'request.state'} eq 'construct')) {
                $bodytag .= Apache::lonmenu::secondary_menu($httphost,$args->{'links_disabled'});
            }
            $bodytag .= Apache::lonmenu::serverform();
            if ($need_endlcint) {
                $bodytag .= Apache::lonhtmlcommon::scripttag('', 'end');
            }
            if ($env{'request.state'} eq 'construct') {
                $bodytag .= &Apache::lonmenu::innerregister($forcereg,
                                $args->{'bread_crumbs'},'','',$hostname);
            } elsif ($forcereg) {
                $bodytag .= &Apache::lonmenu::innerregister($forcereg,undef,
                                                            $args->{'group'},
                                                            $args->{'hide_buttons'},
                                                            $hostname);
            } else {
                my $forbodytag;
                &Apache::lonmenu::prepare_functions($env{'request.noversionuri'},
                                                    $forcereg,$args->{'group'},
                                                    $args->{'bread_crumbs'},
                                                    $advtoolsref,'',$hostname,
                                                    \$forbodytag);
                unless (ref($args->{'bread_crumbs'}) eq 'ARRAY') {
                    $bodytag .= $forbodytag;
                }
            }
        } else {
            # this is to separate menu from content when there's no secondary
            # menu. Especially needed for public accessible ressources.
            $bodytag .= '<hr style="clear:both" />';
            if ($need_endlcint) {
                $bodytag .= Apache::lonhtmlcommon::scripttag('', 'end');
            }
        }

        return $bodytag;
    }

#
# Top frame rendering, Remote is up
#

    my $linkattr;
    if ($args->{'links_disabled'}) {
        $linkattr = 'class="LCisDisabled" aria-disabled="true"';
    }

    my $help=($no_inline_link?''
              :&top_nav_help('Help',$linkattr));

    # Explicit link to get inline menu
    my $menu= ($no_inline_link?''
               :'<a href="/adm/remote?action=collapse" $linkattr target="_top">'.&mt('Switch to Inline Menu Mode').'</a>');

    if ($dc_info) {
        $dc_info = qq|<span class="LC_cusr_subheading">($dc_info)</span>|;
    }

    my $name = &plainname($env{'user.name'},$env{'user.domain'});
    unless ($public) {
        my $class = 'LC_menubuttons_link';
        if ($args->{'links_disabled'}) {
            $class .= ' LCisDisabled';
        }
        $name = &aboutmewrapper($name,$env{'user.name'},$env{'user.domain'},
                                undef,$class);
    }

    unless ($env{'form.inhibitmenu'}) {
        $bodytag .= &inline_for_remote($public,$role,$realm,$dc_info,$no_inline_link);
    }
    return $bodytag."\n".$funclist;
}

sub inline_for_remote {
    my ($public,$role,$realm,$dc_info,$no_inline_link) = @_;
    my $help=($no_inline_link?''
              :&Apache::loncommon::top_nav_help('Help'));

    # Explicit link to get inline menu
    my $menu= ($no_inline_link?''
               :'<a href="/adm/remote?action=collapse" target="_top">'.&mt('Switch to Inline Menu Mode').'</a>');

    if ($dc_info) {
        $dc_info = qq|<span class="LC_cusr_subheading">($dc_info)</span>|;
    }

    my $name = &plainname($env{'user.name'},$env{'user.domain'});
    unless ($public) {
        $name = &aboutmewrapper($name,$env{'user.name'},$env{'user.domain'},
                                undef,'LC_menubuttons_link');
    }

    return qq|<div id="LC_nav_bar">$name $role</div>
              <ol class="LC_primary_menu LC_floatright LC_right">
              <li>$help</li>
              <li>$menu</li>
              </ol><div id="LC_realm"> $realm $dc_info</div>|;
}

sub dc_courseid_toggle {
    my ($dc_info) = @_;
    return ' <span id="dccidtext" class="LC_cusr_subheading LC_nobreak">'.
           '<a href="javascript:showCourseID();" class="LC_menubuttons_link">'.
           &mt('(More ...)').'</a></span>'.
           '<div id="dccid" class="LC_dccid">'.$dc_info.'</div>';
}

sub make_attr_string {
    my ($register,$attr_ref) = @_;

    if ($attr_ref && !ref($attr_ref)) {
	die("addentries Must be a hash ref ".
	    join(':',caller(1))." ".
	    join(':',caller(0))." ");
    }

    if ($register) {
	my ($on_load,$on_unload);
	foreach my $key (keys(%{$attr_ref})) {
	    if      (lc($key) eq 'onload') {
		$on_load.=$attr_ref->{$key}.';';
		delete($attr_ref->{$key});

	    } elsif (lc($key) eq 'onunload') {
		$on_unload.=$attr_ref->{$key}.';';
		delete($attr_ref->{$key});
	    }
	}
        if ($env{'environment.remote'} eq 'on') {
            $attr_ref->{'onload'}  =
                &Apache::lonmenu::loadevents().  $on_load;
            $attr_ref->{'onunload'}=
                &Apache::lonmenu::unloadevents().$on_unload;
        } else {  
	    $attr_ref->{'onload'}  = $on_load;
	    $attr_ref->{'onunload'}= $on_unload;
        }
    }

    my $attr_string;
    foreach my $attr (sort(keys(%$attr_ref))) {
	$attr_string .= " $attr=\"".$attr_ref->{$attr}.'" ';
    }
    return $attr_string;
}


###############################################
###############################################

=pod

=item * &endbodytag()

Returns a uniform footer for LON-CAPA web pages.

Inputs: 1 - optional reference to an args hash
If in the hash, key for noredirectlink has a value which evaluates to true,
a 'Continue' link is not displayed if the page contains an
internal redirect in the <head></head> section,
i.e., $env{'internal.head.redirect'} exists   

=cut

sub endbodytag {
    my ($args) = @_;
    my $endbodytag;
    unless ((ref($args) eq 'HASH') && ($args->{'notbody'})) {
        $endbodytag='</body>';
    }
    if ( exists( $env{'internal.head.redirect'} ) ) {
        if (!(ref($args) eq 'HASH' && $args->{'noredirectlink'})) {
	    $endbodytag=
	        "<br /><a href=\"$env{'internal.head.redirect'}\">".
	        &mt('Continue').'</a>'.
	        $endbodytag;
        }
    }
    if ((ref($args) eq 'HASH') && ($args->{'dashjs'})) {
        $endbodytag = &Apache::lonhtmlcommon::dash_to_minus_js().$endbodytag;
    }
    return $endbodytag;
}

=pod

=item * &standard_css()

Returns a style sheet

Inputs: (all optional)
            domain         -> force to color decorate a page for a specific
                               domain
            function       -> force usage of a specific rolish color scheme
            bgcolor        -> override the default page bgcolor

=cut

sub standard_css {
    my ($function,$domain,$bgcolor) = @_;
    $function  = &get_users_function() if (!$function);
    my $img    = &designparm($function.'.img',   $domain);
    my $tabbg  = &designparm($function.'.tabbg', $domain);
    my $font   = &designparm($function.'.font',  $domain);
    my $fontmenu = &designparm($function.'.fontmenu', $domain);
#second colour for later usage
    my $sidebg = &designparm($function.'.sidebg',$domain);
    my $pgbg_or_bgcolor =
	         $bgcolor ||
	         &designparm($function.'.pgbg',  $domain);
    my $pgbg   = &designparm($function.'.pgbg',  $domain);
    my $alink  = &designparm($function.'.alink', $domain);
    my $vlink  = &designparm($function.'.vlink', $domain);
    my $link   = &designparm($function.'.link',  $domain);

    my $sans                 = 'Verdana,Arial,Helvetica,sans-serif';
    my $mono                 = 'monospace';
    my $data_table_head      = $sidebg;
    my $data_table_light     = '#FAFAFA';
    my $data_table_dark      = '#E0E0E0';
    my $data_table_darker    = '#CCCCCC';
    my $data_table_highlight = '#FFFF00';
    my $mail_new             = '#FFBB77';
    my $mail_new_hover       = '#DD9955';
    my $mail_read            = '#BBBB77';
    my $mail_read_hover      = '#999944';
    my $mail_replied         = '#AAAA88';
    my $mail_replied_hover   = '#888855';
    my $mail_other           = '#99BBBB';
    my $mail_other_hover     = '#669999';
    my $table_header         = '#DDDDDD';
    my $feedback_link_bg     = '#BBBBBB';
    my $lg_border_color      = '#C8C8C8';
    my $button_hover         = '#BF2317';

    my $border = ($env{'browser.type'} eq 'explorer' ||
      $env{'browser.type'} eq 'safari'     ) ? '0 2px 0 2px'
                                             : '0 3px 0 4px';


    return <<END;

/* needed for iframe to allow 100% height in FF */
body, html { 
    margin: 0;
    padding: 0 0.5%;
    height: 99%; /* to avoid scrollbars */
}

body {
  font-family: $sans;
  line-height:130%;
  font-size:0.83em;
  color:$font;
  background-color: $pgbg_or_bgcolor;
}

a:focus,
a:focus img {
  color: red;
}

form, .inline {
  display: inline;
}

.LC_right {
  text-align:right;
}

.LC_middle {
  vertical-align:middle;
}

.LC_floatleft {
  float: left;
}

.LC_floatright {
  float: right;
}

.LC_400Box {
  width:400px;
}

.LC_iframecontainer {
    width: 98%;
    margin: 0;
    position: fixed;
    top: 8.5em;
    bottom: 0;
}

.LC_iframecontainer iframe{
    border: none;
    width: 100%;
    height: 100%;
}

.LC_filename {
  font-family: $mono;
  white-space:pre;
  font-size: 120%;
}

.LC_fileicon {
  border: none;
  height: 1.3em;
  vertical-align: text-bottom;
  margin-right: 0.3em;
  text-decoration:none;
}

.LC_setting {
  text-decoration:underline;
}

.LC_error {
  color: red;
}

.LC_warning {
  color: darkorange;
}

.LC_diff_removed {
  color: red;
}

.LC_info,
.LC_success,
.LC_diff_added {
  color: green;
}

div.LC_confirm_box {
  background-color: #FAFAFA;
  border: 1px solid $lg_border_color;
  margin-right: 0;
  padding: 5px;
}

div.LC_confirm_box .LC_error img,
div.LC_confirm_box .LC_success img {
  vertical-align: middle;
}

.LC_maxwidth {
  max-width: 100%;
  height: auto;
}

.LC_textsize_mobile {
  \@media only screen and (max-device-width: 480px) {
      -webkit-text-size-adjust:100%; -moz-text-size-adjust:100%; -ms-text-size-adjust:100%;
  }
}

.LC_icon {
  border: none;
  vertical-align: middle;
}

.LC_docs_spacer {
  width: 25px;
  height: 1px;
  border: none;
}

.LC_internal_info {
  color: #999999;
}

.LC_discussion {
  background: $data_table_dark;
  border: 1px solid black;
  margin: 2px;
}

.LC_disc_action_left {
  background: $sidebg;
  text-align: left;
  padding: 4px;
  margin: 2px;
}

.LC_disc_action_right {
  background: $sidebg;
  text-align: right;
  padding: 4px;
  margin: 2px;
}

.LC_disc_new_item {
  background: white;
  border: 2px solid red;
  margin: 4px;
  padding: 4px;
}

.LC_disc_old_item {
  background: white;
  margin: 4px;
  padding: 4px;
}

table.LC_pastsubmission {
  border: 1px solid black;
  margin: 2px;
}

table#LC_menubuttons {
  width: 100%;
  background: $pgbg;
  border: 2px;
  border-collapse: separate;
  padding: 0;
}

table#LC_title_bar a {
  color: $fontmenu;
}

table#LC_title_bar {
  clear: both;
  display: none;
}

table#LC_title_bar,
table.LC_breadcrumbs, /* obsolete? */
table#LC_title_bar.LC_with_remote {
  width: 100%;
  border-color: $pgbg;
  border-style: solid;
  border-width: $border;
  background: $pgbg;
  color: $fontmenu;
  border-collapse: collapse;
  padding: 0;
  margin: 0;
}

ul.LC_breadcrumb_tools_outerlist {
    margin: 0;
    padding: 0;
    position: relative;
    list-style: none;
}
ul.LC_breadcrumb_tools_outerlist li {
    display: inline;
}

.LC_breadcrumb_tools_navigation {
    padding: 0;
    margin: 0;
    float: left;
}
.LC_breadcrumb_tools_tools {
    padding: 0;
    margin: 0;
    float: right;
}

table#LC_title_bar td {
  background: $tabbg;
}

table#LC_menubuttons img {
  border: none;
}

.LC_breadcrumbs_component {
  float: right;
  margin: 0 1em;
}
.LC_breadcrumbs_component img {
  vertical-align: middle;
}

.LC_breadcrumbs_hoverable {
  background: $sidebg;
}

td.LC_table_cell_checkbox {
  text-align: center;
}

.LC_fontsize_small {
  font-size: 70%;
}

#LC_breadcrumbs {
  clear:both;
  background: $sidebg;
  border-bottom: 1px solid $lg_border_color;
  line-height: 2.5em;
  overflow: hidden;
  margin: 0;
  padding: 0;
  text-align: left;
}

.LC_head_subbox, .LC_actionbox {
  clear:both;
  background: #F8F8F8; /* $sidebg; */
  border: 1px solid $sidebg;
  margin: 0 0 10px 0;
  padding: 3px;
  text-align: left;
}

.LC_fontsize_medium {
  font-size: 85%;
}

.LC_fontsize_large {
  font-size: 120%;
}

.LC_menubuttons_inline_text {
  color: $font;
  font-size: 90%;
  padding-left:3px;
}

.LC_menubuttons_inline_text img{
  vertical-align: middle;
}

li.LC_menubuttons_inline_text img {
  cursor:pointer;
  text-decoration: none;
}

.LC_menubuttons_link {
  text-decoration: none;
}

.LC_menubuttons_category {
  color: $font;
  background: $pgbg;
  font-size: larger;
  font-weight: bold;
}

td.LC_menubuttons_text {
  color: $font;
}

.LC_current_location {
  background: $tabbg;
}

td.LC_zero_height {
  line-height: 0;
  cellpadding: 0;
}

table.LC_data_table {
  border: 1px solid #000000;
  border-collapse: separate;
  border-spacing: 1px;
  background: $pgbg;
}

.LC_data_table_dense {
  font-size: small;
}

table.LC_nested_outer {
  border: 1px solid #000000;
  border-collapse: collapse;
  border-spacing: 0;
  width: 100%;
}

table.LC_innerpickbox,
table.LC_nested {
  border: none;
  border-collapse: collapse;
  border-spacing: 0;
  width: 100%;
}

table.LC_data_table tr th,
table.LC_calendar tr th,
table.LC_prior_tries tr th,
table.LC_innerpickbox tr th {
  font-weight: bold;
  background-color: $data_table_head;
  color:$fontmenu;
  font-size:90%;
}

table.LC_innerpickbox tr th,
table.LC_innerpickbox tr td {
  vertical-align: top;
}

table.LC_data_table tr.LC_info_row > td {
  background-color: #CCCCCC;
  font-weight: bold;
  text-align: left;
}

table.LC_data_table tr.LC_odd_row > td {
  background-color: $data_table_light;
  padding: 2px;
  vertical-align: top;
}

table.LC_pick_box tr > td.LC_odd_row {
  background-color: $data_table_light;
  vertical-align: top;
}

table.LC_data_table tr.LC_even_row > td {
  background-color: $data_table_dark;
  padding: 2px;
  vertical-align: top;
}

table.LC_pick_box tr > td.LC_even_row {
  background-color: $data_table_dark;
  vertical-align: top;
}

table.LC_data_table tr.LC_data_table_highlight td {
  background-color: $data_table_darker;
}

table.LC_data_table tr td.LC_leftcol_header {
  background-color: $data_table_head;
  font-weight: bold;
}

table.LC_data_table tr.LC_empty_row td,
table.LC_nested tr.LC_empty_row td {
  font-weight: bold;
  font-style: italic;
  text-align: center;
  padding: 8px;
}

table.LC_data_table tr.LC_empty_row td,
table.LC_data_table tr.LC_footer_row td {
  background-color: $sidebg;
}

table.LC_nested tr.LC_empty_row td {
  background-color: #FFFFFF;
}

table.LC_caption {
}

table.LC_nested tr.LC_empty_row td {
  padding: 4ex
}

table.LC_nested_outer tr th {
  font-weight: bold;
  color:$fontmenu;
  background-color: $data_table_head;
  font-size: small;
  border-bottom: 1px solid #000000;
}

table.LC_nested_outer tr td.LC_subheader {
  background-color: $data_table_head;
  font-weight: bold;
  font-size: small;
  border-bottom: 1px solid #000000;
  text-align: right;
}

table.LC_nested tr.LC_info_row td {
  background-color: #CCCCCC;
  font-weight: bold;
  font-size: small;
  text-align: center;
}

table.LC_nested tr.LC_info_row td.LC_left_item,
table.LC_nested_outer tr th.LC_left_item {
  text-align: left;
}

table.LC_nested td {
  background-color: #FFFFFF;
  font-size: small;
}

table.LC_nested_outer tr th.LC_right_item,
table.LC_nested tr.LC_info_row td.LC_right_item,
table.LC_nested tr.LC_odd_row td.LC_right_item,
table.LC_nested tr td.LC_right_item {
  text-align: right;
}

table.LC_nested tr.LC_odd_row td {
  background-color: #EEEEEE;
}

table.LC_createuser {
}

table.LC_createuser tr.LC_section_row td {
  font-size: small;
}

table.LC_createuser tr.LC_info_row td  {
  background-color: #CCCCCC;
  font-weight: bold;
  text-align: center;
}

table.LC_calendar {
  border: 1px solid #000000;
  border-collapse: collapse;
  width: 98%;
}

table.LC_calendar_pickdate {
  font-size: xx-small;
}

table.LC_calendar tr td {
  border: 1px solid #000000;
  vertical-align: top;
  width: 14%;
}

table.LC_calendar tr td.LC_calendar_day_empty {
  background-color: $data_table_dark;
}

table.LC_calendar tr td.LC_calendar_day_current {
  background-color: $data_table_highlight;
}

table.LC_data_table tr td.LC_mail_new {
  background-color: $mail_new;
}

table.LC_data_table tr.LC_mail_new:hover {
  background-color: $mail_new_hover;
}

table.LC_data_table tr td.LC_mail_read {
  background-color: $mail_read;
}

/*
table.LC_data_table tr.LC_mail_read:hover {
  background-color: $mail_read_hover;
}
*/

table.LC_data_table tr td.LC_mail_replied {
  background-color: $mail_replied;
}

/*
table.LC_data_table tr.LC_mail_replied:hover {
  background-color: $mail_replied_hover;
}
*/

table.LC_data_table tr td.LC_mail_other {
  background-color: $mail_other;
}

/*
table.LC_data_table tr.LC_mail_other:hover {
  background-color: $mail_other_hover;
}
*/

table.LC_data_table tr > td.LC_browser_file,
table.LC_data_table tr > td.LC_browser_file_published {
  background: #AAEE77;
}

table.LC_data_table tr > td.LC_browser_file_locked,
table.LC_data_table tr > td.LC_browser_file_unpublished {
  background: #FFAA99;
}

table.LC_data_table tr > td.LC_browser_file_obsolete {
  background: #888888;
}

table.LC_data_table tr > td.LC_browser_file_modified,
table.LC_data_table tr > td.LC_browser_file_metamodified {
  background: #F8F866;
}

table.LC_data_table tr.LC_browser_folder > td {
  background: #E0E8FF;
}

table.LC_data_table tr > td.LC_roles_is {
  /* background: #77FF77; */
}

table.LC_data_table tr > td.LC_roles_future {
  border-right: 8px solid #FFFF77;
}

table.LC_data_table tr > td.LC_roles_will {
  border-right: 8px solid #FFAA77;
}

table.LC_data_table tr > td.LC_roles_expired {
  border-right: 8px solid #FF7777;
}

table.LC_data_table tr > td.LC_roles_will_not {
  border-right: 8px solid #AAFF77;
}

table.LC_data_table tr > td.LC_roles_selected {
  border-right: 8px solid #11CC55;
}

span.LC_current_location {
  font-size:larger;
  background: $pgbg;
}

span.LC_current_nav_location {
  font-weight:bold;
  background: $sidebg;
}

span.LC_parm_menu_item {
  font-size: larger;
}

span.LC_parm_scope_all {
  color: red;
}

span.LC_parm_scope_folder {
  color: green;
}

span.LC_parm_scope_resource {
  color: orange;
}

span.LC_parm_part {
  color: blue;
}

span.LC_parm_folder,
span.LC_parm_symb {
  font-size: x-small;
  font-family: $mono;
  color: #AAAAAA;
}

ul.LC_parm_parmlist li {
  display: inline-block;
  padding: 0.3em 0.8em;
  vertical-align: top;
  width: 150px;
  border-top:1px solid $lg_border_color;
}

td.LC_parm_overview_level_menu,
td.LC_parm_overview_map_menu,
td.LC_parm_overview_parm_selectors,
td.LC_parm_overview_restrictions  {
  border: 1px solid black;
  border-collapse: collapse;
}

table.LC_parm_overview_restrictions td {
  border-width: 1px 4px 1px 4px;
  border-style: solid;
  border-color: $pgbg;
  text-align: center;
}

table.LC_parm_overview_restrictions th {
  background: $tabbg;
  border-width: 1px 4px 1px 4px;
  border-style: solid;
  border-color: $pgbg;
}

table#LC_helpmenu {
  border: none;
  height: 55px;
  border-spacing: 0;
}

table#LC_helpmenu fieldset legend {
  font-size: larger;
}

table#LC_helpmenu_links {
  width: 100%;
  border: 1px solid black;
  background: $pgbg;
  padding: 0;
  border-spacing: 1px;
}

table#LC_helpmenu_links tr td {
  padding: 1px;
  background: $tabbg;
  text-align: center;
  font-weight: bold;
}

table#LC_helpmenu_links a:link,
table#LC_helpmenu_links a:visited,
table#LC_helpmenu_links a:active {
  text-decoration: none;
  color: $font;
}

table#LC_helpmenu_links a:hover {
  text-decoration: underline;
  color: $vlink;
}

.LC_chrt_popup_exists {
  border: 1px solid #339933;
  margin: -1px;
}

.LC_chrt_popup_up {
  border: 1px solid yellow;
  margin: -1px;
}

.LC_chrt_popup {
  border: 1px solid #8888FF;
  background: #CCCCFF;
}

table.LC_pick_box {
  border-collapse: separate;
  background: white;
  border: 1px solid black;
  border-spacing: 1px;
}

table.LC_pick_box td.LC_pick_box_title {
  background: $sidebg;
  font-weight: bold;
  text-align: left;
  vertical-align: top;
  width: 184px;
  padding: 8px;
}

table.LC_pick_box td.LC_pick_box_value {
  text-align: left;
  padding: 8px;
}

table.LC_pick_box td.LC_pick_box_select {
  text-align: left;
  padding: 8px;
}

table.LC_pick_box td.LC_pick_box_separator {
  padding: 0;
  height: 1px;
  background: black;
}

table.LC_pick_box td.LC_pick_box_submit {
  text-align: right;
}

table.LC_pick_box td.LC_evenrow_value {
  text-align: left;
  padding: 8px;
  background-color: $data_table_light;
}

table.LC_pick_box td.LC_oddrow_value {
  text-align: left;
  padding: 8px;
  background-color: $data_table_light;
}

span.LC_helpform_receipt_cat {
  font-weight: bold;
}

table.LC_group_priv_box {
  background: white;
  border: 1px solid black;
  border-spacing: 1px;
}

table.LC_group_priv_box td.LC_pick_box_title {
  background: $tabbg;
  font-weight: bold;
  text-align: right;
  width: 184px;
}

table.LC_group_priv_box td.LC_groups_fixed {
  background: $data_table_light;
  text-align: center;
}

table.LC_group_priv_box td.LC_groups_optional {
  background: $data_table_dark;
  text-align: center;
}

table.LC_group_priv_box td.LC_groups_functionality {
  background: $data_table_darker;
  text-align: center;
  font-weight: bold;
}

table.LC_group_priv td {
  text-align: left;
  padding: 0;
}

.LC_navbuttons {
  margin: 2ex 0ex 2ex 0ex;
}

.LC_topic_bar {
  font-weight: bold;
  background: $tabbg;
  margin: 1em 0em 1em 2em;
  padding: 3px;
  font-size: 1.2em;
}

.LC_topic_bar span {
  left: 0.5em;
  position: absolute;
  vertical-align: middle;
  font-size: 1.2em;
}

table.LC_course_group_status {
  margin: 20px;
}

table.LC_status_selector td {
  vertical-align: top;
  text-align: center;
  padding: 4px;
}

div.LC_feedback_link {
  clear: both;
  background: $sidebg;
  width: 100%;
  padding-bottom: 10px;
  border: 1px $tabbg solid;
  height: 22px;
  line-height: 22px;
  padding-top: 5px;
}

div.LC_feedback_link img {
  height: 22px;
  vertical-align:middle;
}

div.LC_feedback_link a {
  text-decoration: none;
}

div.LC_comblock {
  display:inline;
  color:$font;
  font-size:90%;
}

div.LC_feedback_link div.LC_comblock {
  padding-left:5px;
}

div.LC_feedback_link div.LC_comblock a {
  color:$font;
}

span.LC_feedback_link {
  /* background: $feedback_link_bg; */
  font-size: larger;
}

span.LC_message_link {
  /* background: $feedback_link_bg; */
  font-size: larger;
  position: absolute;
  right: 1em;
}

table.LC_prior_tries {
  border: 1px solid #000000;
  border-collapse: separate;
  border-spacing: 1px;
}

table.LC_prior_tries td {
  padding: 2px;
}

.LC_answer_correct {
  background: lightgreen;
  color: darkgreen;
  padding: 6px;
}

.LC_answer_charged_try {
  background: #FFAAAA;
  color: darkred;
  padding: 6px;
}

.LC_answer_not_charged_try,
.LC_answer_no_grade,
.LC_answer_late {
  background: lightyellow;
  color: black;
  padding: 6px;
}

.LC_answer_previous {
  background: lightblue;
  color: darkblue;
  padding: 6px;
}

.LC_answer_no_message {
  background: #FFFFFF;
  color: black;
  padding: 6px;
}

.LC_answer_unknown,
.LC_answer_warning {
  background: orange;
  color: black;
  padding: 6px;
}

span.LC_prior_numerical,
span.LC_prior_string,
span.LC_prior_custom,
span.LC_prior_reaction,
span.LC_prior_math {
  font-family: $mono;
  white-space: pre;
}

span.LC_prior_string {
  font-family: $mono;
  white-space: pre;
}

table.LC_prior_option {
  width: 100%;
  border-collapse: collapse;
}

table.LC_prior_rank,
table.LC_prior_match {
  border-collapse: collapse;
}

table.LC_prior_option tr td,
table.LC_prior_rank tr td,
table.LC_prior_match tr td {
  border: 1px solid #000000;
}

.LC_nobreak {
  white-space: nowrap;
}

span.LC_cusr_emph {
  font-style: italic;
}

span.LC_cusr_subheading {
  font-weight: normal;
  font-size: 85%;
}

div.LC_docs_entry_move {
  border: 1px solid #BBBBBB;
  background: #DDDDDD;
  width: 22px;
  padding: 1px;
  margin: 0;
}

table.LC_data_table tr > td.LC_docs_entry_commands,
table.LC_data_table tr > td.LC_docs_entry_parameter {
  font-size: x-small;
}

.LC_docs_entry_parameter {
  white-space: nowrap;
}

.LC_docs_copy {
  color: #000099;
}

.LC_docs_cut {
  color: #550044;
}

.LC_docs_rename {
  color: #009900;
}

.LC_docs_remove {
  color: #990000;
}

.LC_domprefs_email,
.LC_docs_reinit_warn,
.LC_docs_ext_edit {
  font-size: x-small;
}

table.LC_docs_adddocs td,
table.LC_docs_adddocs th {
  border: 1px solid #BBBBBB;
  padding: 4px;
  background: #DDDDDD;
}

table.LC_sty_begin {
  background: #BBFFBB;
}

table.LC_sty_end {
  background: #FFBBBB;
}

table.LC_double_column {
  border-width: 0;
  border-collapse: collapse;
  width: 100%;
  padding: 2px;
}

table.LC_double_column tr td.LC_left_col {
  top: 2px;
  left: 2px;
  width: 47%;
  vertical-align: top;
}

table.LC_double_column tr td.LC_right_col {
  top: 2px;
  right: 2px;
  width: 47%;
  vertical-align: top;
}

div.LC_left_float {
  float: left;
  padding-right: 5%;
  padding-bottom: 4px;
}

div.LC_clear_float_header {
  padding-bottom: 2px;
}

div.LC_clear_float_footer {
  padding-top: 10px;
  clear: both;
}

div.LC_grade_show_user {
/*  border-left: 5px solid $sidebg; */
  border-top: 5px solid #000000;
  margin: 50px 0 0 0;
  padding: 15px 0 5px 10px;
}

div.LC_grade_show_user_odd_row {
/*  border-left: 5px solid #000000; */
}

div.LC_grade_show_user div.LC_Box {
  margin-right: 50px;
}

div.LC_grade_submissions,
div.LC_grade_message_center,
div.LC_grade_info_links {
  margin: 5px;
  width: 99%;
  background: #FFFFFF;
}

div.LC_grade_submissions_header,
div.LC_grade_message_center_header {
  font-weight: bold;
  font-size: large;
}

div.LC_grade_submissions_body,
div.LC_grade_message_center_body {
  border: 1px solid black;
  width: 99%;
  background: #FFFFFF;
}

table.LC_scantron_action {
  width: 100%;
}

table.LC_scantron_action tr th {
  font-weight:bold;
  font-style:normal;
}

.LC_edit_problem_header,
div.LC_edit_problem_footer {
  font-weight: normal;
  font-size:  medium;
  margin: 2px;
  background-color: $sidebg;
}

div.LC_edit_problem_header,
div.LC_edit_problem_header div,
div.LC_edit_problem_footer,
div.LC_edit_problem_footer div,
div.LC_edit_problem_editxml_header,
div.LC_edit_problem_editxml_header div {
  z-index: 100;
}

div.LC_edit_problem_header_title {
  font-weight: bold;
  font-size: larger;
  background: $tabbg;
  padding: 3px;
  margin: 0 0 5px 0;
}

table.LC_edit_problem_header_title {
  width: 100%;
  background: $tabbg;
}

div.LC_edit_actionbar {
    background-color: $sidebg;
    margin: 0;
    padding: 0;
    line-height: 200%;
}

div.LC_edit_actionbar div{
    padding: 0;
    margin: 0;
    display: inline-block;
}

.LC_edit_opt {
  padding-left: 1em;
  white-space: nowrap;
}

.LC_edit_problem_latexhelper{
    text-align: right;
}

#LC_edit_problem_colorful div{
    margin-left: 40px;
}

#LC_edit_problem_codemirror div{
    margin-left: 0px;
}

img.stift {
  border-width: 0;
  vertical-align: middle;
}

table td.LC_mainmenu_col_fieldset {
  vertical-align: top;
}

div.LC_createcourse {
  margin: 10px 10px 10px 10px;
}

.LC_dccid {
  float: right;
  margin: 0.2em 0 0 0;
  padding: 0;
  font-size: 90%;
  display:none;
}

ol.LC_primary_menu a:hover,
ol#LC_MenuBreadcrumbs a:hover,
ol#LC_PathBreadcrumbs a:hover,
ul#LC_secondary_menu a:hover,
.LC_FormSectionClearButton input:hover
ul.LC_TabContent   li:hover a {
  color:$button_hover;
  text-decoration:none;
}

h1 {
  padding: 0;
  line-height:130%;
}

h2,
h3,
h4,
h5,
h6 {
  margin: 5px 0 5px 0;
  padding: 0;
  line-height:130%;
}

.LC_hcell {
  padding:3px 15px 3px 15px;
  margin: 0;
  background-color:$tabbg;
  color:$fontmenu;
  border-bottom:solid 1px $lg_border_color;
}

.LC_Box > .LC_hcell {
  margin: 0 -10px 10px -10px;
}

.LC_noBorder {
  border: 0;
}

.LC_FormSectionClearButton input {
  background-color:transparent;
  border: none;
  cursor:pointer;
  text-decoration:underline;
}

.LC_help_open_topic {
  color: #FFFFFF;
  background-color: #EEEEFF;
  margin: 1px;
  padding: 4px;
  border: 1px solid #000033;
  white-space: nowrap;
  /* vertical-align: middle; */
}

dl,
ul,
div,
fieldset {
  margin: 10px 10px 10px 0;
  /* overflow: hidden; */
}

article.geogebraweb div {
    margin: 0;
}

fieldset > legend {
  font-weight: bold;
  padding: 0 5px 0 5px;
}

#LC_nav_bar {
  float: left;
  background-color: $pgbg_or_bgcolor;
  margin: 0 0 2px 0;
}

#LC_realm {
  margin: 0.2em 0 0 0;
  padding: 0;
  font-weight: bold;
  text-align: center;
  background-color: $pgbg_or_bgcolor;
}

#LC_nav_bar em {
  font-weight: bold;
  font-style: normal;
}

ol.LC_primary_menu {
  margin: 0;
  padding: 0;
}

ol#LC_PathBreadcrumbs {
  margin: 0;
}

ol.LC_primary_menu li {
  color: RGB(80, 80, 80);
  vertical-align: middle;
  text-align: left;
  list-style: none;
  position: relative;
  float: left;
  z-index: 100; /* will be displayed above codemirror and underneath the help-layer */
  line-height: 1.5em;
}

ol.LC_primary_menu li a, 
ol.LC_primary_menu li p {
  display: block;
  margin: 0;
  padding: 0 5px 0 10px;
  text-decoration: none;
}

ol.LC_primary_menu li p span.LC_primary_menu_innertitle {
  display: inline-block;
  width: 95%;
  text-align: left;
}

ol.LC_primary_menu li p span.LC_primary_menu_innerarrow {
  display: inline-block;
  width: 5%;
  float: right;
  text-align: right;
  font-size: 70%;
}

ol.LC_primary_menu ul {
  display: none;
  width: 15em;
  background-color: $data_table_light;
  position: absolute;
  top: 100%;
}

ol.LC_primary_menu ul ul {
  left: 100%;
  top: 0;
}

ol.LC_primary_menu li:hover > ul, ol.LC_primary_menu li.hover > ul {
  display: block;
  position: absolute;
  margin: 0;
  padding: 0;
  z-index: 2;
}

ol.LC_primary_menu li:hover li, ol.LC_primary_menu li.hover li {
/* First Submenu -> size should be smaller than the menu title of the whole menu */
  font-size: 90%;
  vertical-align: top;
  float: none;
  border-left: 1px solid black;
  border-right: 1px solid black;
/* A dark bottom border to visualize different menu options;
overwritten in the create_submenu routine for the last border-bottom of the menu */
  border-bottom: 1px solid $data_table_dark;
}

ol.LC_primary_menu li li p:hover {
  color:$button_hover;
  text-decoration:none;
  background-color:$data_table_dark;
}

ol.LC_primary_menu li li a:hover {
   color:$button_hover;
   background-color:$data_table_dark;
}

/* Font-size equal to the size of the predecessors*/
ol.LC_primary_menu li:hover li li {
  font-size: 100%;
}

ol.LC_primary_menu li img {
  vertical-align: bottom;
  height: 1.1em;
  margin: 0.2em 0 0 0;
}

ol.LC_primary_menu a {
  color: RGB(80, 80, 80);
  text-decoration: none;
}

ol.LC_primary_menu a.LC_new_message {
  font-weight:bold;
  color: darkred;
}

ol.LC_docs_parameters {
  margin-left: 0;
  padding: 0;
  list-style: none;
}

ol.LC_docs_parameters li {
  margin: 0;
  padding-right: 20px;
  display: inline;
}

ol.LC_docs_parameters li:before {
  content: "\\002022 \\0020";
}

li.LC_docs_parameters_title {
  font-weight: bold;
}

ol.LC_docs_parameters li.LC_docs_parameters_title:before {
  content: "";
}

ul#LC_secondary_menu {
  clear: right;
  color: $fontmenu;
  background: $tabbg;
  list-style: none;
  padding: 0;
  margin: 0;
  width: 100%;
  text-align: left;
  float: left;
}

ul#LC_secondary_menu li {
  font-weight: bold;
  line-height: 1.8em;
  border-right: 1px solid black;
  float: left;
}

ul#LC_secondary_menu li.LC_hoverable:hover, ul#LC_secondary_menu li.hover {
  background-color: $data_table_light;
}

ul#LC_secondary_menu li a {
  padding: 0 0.8em;
}

ul#LC_secondary_menu li ul {
  display: none;
}

ul#LC_secondary_menu li:hover ul, ul#LC_secondary_menu li.hover ul {
  display: block;
  position: absolute;
  margin: 0;
  padding: 0;
  list-style:none;
  float: none;
  background-color: $data_table_light;
  z-index: 2;
  margin-left: -1px;
}

ul#LC_secondary_menu li ul li {
  font-size: 90%;
  vertical-align: top;
  border-left: 1px solid black;
  border-right: 1px solid black;
  background-color: $data_table_light;
  list-style:none;
  float: none;
}

ul#LC_secondary_menu li ul li:hover, ul#LC_secondary_menu li ul li.hover {
  background-color: $data_table_dark;
}

ul.LC_TabContent {
  display:block;
  background: $sidebg;
  border-bottom: solid 1px $lg_border_color;
  list-style:none;
  margin: -1px -10px 0 -10px;
  padding: 0;
}

ul.LC_TabContent li,
ul.LC_TabContentBigger li {
  float:left;
}

ul#LC_secondary_menu li a {
  color: $fontmenu;
  text-decoration: none;
}

ul.LC_TabContent {
  min-height:20px;
}

ul.LC_TabContent li {
  vertical-align:middle;
  padding: 0 16px 0 10px;
  background-color:$tabbg;
  border-bottom:solid 1px $lg_border_color;
  border-left: solid 1px $font;
}

ul.LC_TabContent .right {
  float:right;
}

ul.LC_TabContent li a,
ul.LC_TabContent li {
  color:rgb(47,47,47);
  text-decoration:none;
  font-size:95%;
  font-weight:bold;
  min-height:20px;
}

ul.LC_TabContent li a:hover,
ul.LC_TabContent li a:focus {
  color: $button_hover;
  background:none;
  outline:none;
}

ul.LC_TabContent li:hover {
  color: $button_hover;
  cursor:pointer;
}

ul.LC_TabContent li.active {
  color: $font;
  background:#FFFFFF url(/adm/lonIcons/open.gif) no-repeat scroll right center;
  border-bottom:solid 1px #FFFFFF;
  cursor: default;
}

ul.LC_TabContent li.active a {
  color:$font;
  background:#FFFFFF;
  outline: none;
}

ul.LC_TabContent li.goback {
  float: left;
  border-left: none;
}

#maincoursedoc {
  clear:both;
}

ul.LC_TabContentBigger {
  display:block;
  list-style:none;
  padding: 0;
}

ul.LC_TabContentBigger li {
  vertical-align:bottom;
  height: 30px;
  font-size:110%;
  font-weight:bold;
  color: #737373;
}

ul.LC_TabContentBigger li.active {
  position: relative;
  top: 1px;
}

ul.LC_TabContentBigger li a {
  background:url('/adm/lonIcons/tabbgleft.gif') left bottom no-repeat;
  height: 30px;
  line-height: 30px;
  text-align: center;
  display: block;
  text-decoration: none;
  outline: none;  
}

ul.LC_TabContentBigger li.active a {
  background:url('/adm/lonIcons/tabbgleft.gif') left top no-repeat;
  color:$font;
}

ul.LC_TabContentBigger li b {
  background: url('/adm/lonIcons/tabbgright.gif') no-repeat right bottom;
  display: block;
  float: left;
  padding: 0 30px;
  border-bottom: 1px solid $lg_border_color;
}

ul.LC_TabContentBigger li:hover b {
  color:$button_hover;
}

ul.LC_TabContentBigger li.active b {
  background:url('/adm/lonIcons/tabbgright.gif') right top no-repeat;
  color:$font;
  border: 0;
}


ul.LC_CourseBreadcrumbs {
  background: $sidebg;
  height: 2em;
  padding-left: 10px;
  margin: 0;
  list-style-position: inside;
}

ol#LC_MenuBreadcrumbs,
ol#LC_PathBreadcrumbs {
  padding-left: 10px;
  margin: 0;
  height: 2.5em;  /* equal to #LC_breadcrumbs line-height */
}

ol#LC_MenuBreadcrumbs li,
ol#LC_PathBreadcrumbs li,
ul.LC_CourseBreadcrumbs li {
  display: inline;
  white-space: normal;  
}

ol#LC_MenuBreadcrumbs li a,
ul.LC_CourseBreadcrumbs li a {
  text-decoration: none;
  font-size:90%;
}

ol#LC_MenuBreadcrumbs h1 {
  display: inline;
  font-size: 90%;
  line-height: 2.5em;
  margin: 0;
  padding: 0;
}

ol#LC_PathBreadcrumbs li a {
  text-decoration:none;
  font-size:100%;
  font-weight:bold;
}

.LC_Box {
  border: solid 1px $lg_border_color;
  padding: 0 10px 10px 10px;
}

.LC_DocsBox {
  border: solid 1px $lg_border_color;
  padding: 0 0 10px 10px;
}

.LC_AboutMe_Image {
  float:left;
  margin-right:10px;
}

.LC_Clear_AboutMe_Image {
  clear:left;
}

dl.LC_ListStyleClean dt {
  padding-right: 5px;
  display: table-header-group;
}

dl.LC_ListStyleClean dd {
  display: table-row;
}

.LC_ListStyleClean,
.LC_ListStyleSimple,
.LC_ListStyleNormal,
.LC_ListStyleSpecial {
  /* display:block; */
  list-style-position: inside;
  list-style-type: none;
  overflow: hidden;
  padding: 0;
}

.LC_ListStyleSimple li,
.LC_ListStyleSimple dd,
.LC_ListStyleNormal li,
.LC_ListStyleNormal dd,
.LC_ListStyleSpecial li,
.LC_ListStyleSpecial dd {
  margin: 0;
  padding: 5px 5px 5px 10px;
  clear: both;
}

.LC_ListStyleClean li,
.LC_ListStyleClean dd {
  padding-top: 0;
  padding-bottom: 0;
}

.LC_ListStyleSimple dd,
.LC_ListStyleSimple li {
  border-bottom: solid 1px $lg_border_color;
}

.LC_ListStyleSpecial li,
.LC_ListStyleSpecial dd {
  list-style-type: none;
  background-color: RGB(220, 220, 220);
  margin-bottom: 4px;
}

table.LC_SimpleTable {
  margin:5px;
  border:solid 1px $lg_border_color;
}

table.LC_SimpleTable tr {
  padding: 0;
  border:solid 1px $lg_border_color;
}

table.LC_SimpleTable thead {
  background:rgb(220,220,220);
}

div.LC_columnSection {
  display: block;
  clear: both;
  overflow: hidden;
  margin: 0;
}

div.LC_columnSection>* {
  float: left;
  margin: 10px 20px 10px 0;
  overflow:hidden;
}

table em {
  font-weight: bold;
  font-style: normal;
}

table.LC_tableBrowseRes,
table.LC_tableOfContent {
  border:none;
  border-spacing: 1px;
  padding: 3px;
  background-color: #FFFFFF;
  font-size: 90%;
}

table.LC_tableOfContent {
  border-collapse: collapse;
}

table.LC_tableBrowseRes a,
table.LC_tableOfContent a {
  background-color: transparent;
  text-decoration: none;
}

table.LC_tableOfContent img {
  border: none;
  height: 1.3em;
  vertical-align: text-bottom;
  margin-right: 0.3em;
}

a#LC_content_toolbar_firsthomework {
  background-image:url(/res/adm/pages/open-first-problem.gif);
}

a#LC_content_toolbar_everything {
  background-image:url(/res/adm/pages/show-all.gif);
}

a#LC_content_toolbar_uncompleted {
  background-image:url(/res/adm/pages/show-incomplete-problems.gif);
}

#LC_content_toolbar_clearbubbles {
  background-image:url(/res/adm/pages/mark-discussionentries-read.gif);
}

a#LC_content_toolbar_changefolder {
  background : url(/res/adm/pages/close-all-folders.gif) top center ;
}

a#LC_content_toolbar_changefolder_toggled {
  background-image:url(/res/adm/pages/open-all-folders.gif);
}

a#LC_content_toolbar_edittoplevel {
  background-image:url(/res/adm/pages/edittoplevel.gif);
}

ul#LC_toolbar li a:hover {
  background-position: bottom center;
}

ul#LC_toolbar {
  padding: 0;
  margin: 2px;
  list-style:none;
  position:relative;
  background-color:white;
  overflow: auto;
}

ul#LC_toolbar li {
  border:1px solid white;
  padding: 0;
  margin: 0;
  float: left;
  display:inline;
  vertical-align:middle;
  white-space: nowrap;
}


a.LC_toolbarItem {
  display:block;
  padding: 0;
  margin: 0;
  height: 32px;
  width: 32px;
  color:white;
  border: none;
  background-repeat:no-repeat;
  background-color:transparent;
}

ul.LC_funclist {
    margin: 0;
    padding: 0.5em 1em 0.5em 0;
}

ul.LC_funclist > li:first-child {
    font-weight:bold; 
    margin-left:0.8em;
}

ul.LC_funclist + ul.LC_funclist {
    /* 
       left border as a seperator if we have more than
       one list 
    */
    border-left: 1px solid $sidebg;
    /* 
       this hides the left border behind the border of the 
       outer box if element is wrapped to the next 'line' 
    */
    margin-left: -1px;
}

ul.LC_funclist li {
  display: inline;
  white-space: nowrap;
  margin: 0 0 0 25px;
  line-height: 150%;
}

.LC_hidden {
  display: none;
}

.LCmodal-overlay {
		position:fixed;
		top:0;
		right:0;
		bottom:0;
		left:0;
		height:100%;
		width:100%;
		margin:0;
		padding:0;
		background:#999;
		opacity:.75;
		filter: alpha(opacity=75);
		-moz-opacity: 0.75;
		z-index:101;
}

* html .LCmodal-overlay {   
		position: absolute;
		height: expression(document.body.scrollHeight > document.body.offsetHeight ? document.body.scrollHeight : document.body.offsetHeight + 'px');
}

.LCmodal-window {
		position:fixed;
		top:50%;
		left:50%;
		margin:0;
		padding:0;
		z-index:102;
	}

* html .LCmodal-window {
		position:absolute;
}

.LCclose-window {
		position:absolute;
		width:32px;
		height:32px;
		right:8px;
		top:8px;
		background:transparent url('/res/adm/pages/process-stop.png') no-repeat scroll right top;
		text-indent:-99999px;
		overflow:hidden;
		cursor:pointer;
}

.LCisDisabled {
  cursor: not-allowed;
  opacity: 0.5;
}

a[aria-disabled="true"] {
  color: currentColor;
  display: inline-block;  /* For IE11/ MS Edge bug */
  pointer-events: none;
  text-decoration: none;
}

pre.LC_wordwrap {
  white-space: pre-wrap;
  white-space: -moz-pre-wrap;
  white-space: -pre-wrap;
  white-space: -o-pre-wrap;
  word-wrap: break-word;
}

/*
  styles used by TTH when "Default set of options to pass to tth/m
  when converting TeX" in course settings has been set

  option passed: -t

*/

td div.comp { margin-top: -0.6ex; margin-bottom: -1ex;}
td div.comb { margin-top: -0.6ex; margin-bottom: -.6ex;}
td div.hrcomp { line-height: 0.9; margin-top: -0.8ex; margin-bottom: -1ex;}
td div.norm {line-height:normal;}

/*
  option passed -y3
*/

span.roman {font-family: serif; font-style: normal; font-weight: normal;}
span.overacc2 {position: relative;  left: .8em; top: -1.2ex;}
span.overacc1 {position: relative;  left: .6em; top: -1.2ex;}

#LC_minitab_header {
  float:left;
  width:100%;
  background:#DAE0D2 url("/res/adm/pages/minitabmenu_bg.gif") repeat-x bottom;
  font-size:93%;
  line-height:normal;
  margin: 0.5em 0 0.5em 0;
}
#LC_minitab_header ul {
  margin:0;
  padding:10px 10px 0;
  list-style:none;
}
#LC_minitab_header li {
  float:left;
  background:url("/res/adm/pages/minitabmenu_left.gif") no-repeat left top;
  margin:0;
  padding:0 0 0 9px;
}
#LC_minitab_header a {
  display:block;
  background:url("/res/adm/pages/minitabmenu_right.gif") no-repeat right top;
  padding:5px 15px 4px 6px;
}
#LC_minitab_header #LC_current_minitab {
  background-image:url("/res/adm/pages/minitabmenu_left_on.gif");
}
#LC_minitab_header #LC_current_minitab a {
  background-image:url("/res/adm/pages/minitabmenu_right_on.gif");
  padding-bottom:5px;
}


END
}

=pod

=item * &headtag()

Returns a uniform footer for LON-CAPA web pages.

Inputs: $title - optional title for the head
        $head_extra - optional extra HTML to put inside the <head>
        $args - optional arguments
            force_register - if is true call registerurl so the remote is 
                             informed
            redirect       -> array ref of
                                   1- seconds before redirect occurs
                                   2- url to redirect to
                                   3- whether the side effect should occur
                           (side effect of setting 
                               $env{'internal.head.redirect'} to the url 
                               redirected too)
                                   4- whether encrypt check should be skipped
            domain         -> force to color decorate a page for a specific
                               domain
            function       -> force usage of a specific rolish color scheme
            bgcolor        -> override the default page bgcolor
            no_auto_mt_title
                           -> prevent &mt()ing the title arg

=cut

sub headtag {
    my ($title,$head_extra,$args) = @_;
    
    my $function = $args->{'function'} || &get_users_function();
    my $domain   = $args->{'domain'}   || &determinedomain();
    my $bgcolor  = $args->{'bgcolor'}  || &designparm($function.'.pgbg',$domain);
    my $httphost = $args->{'use_absolute'};
    my $url = join(':',$env{'user.name'},$env{'user.domain'},
		   $Apache::lonnet::perlvar{'lonVersion'},
		   #time(),
		   $env{'environment.color.timestamp'},
		   $function,$domain,$bgcolor);

    $url = '/adm/css/'.&escape($url).'.css';

    my $result =
	'<head>'.
	&font_settings($args);

    my $inhibitprint;
    if ($args->{'print_suppress'}) {
        $inhibitprint = &print_suppression();
    }

    if (!$args->{'frameset'} && !$args->{'switchserver'}) {
	$result .= &Apache::lonhtmlcommon::htmlareaheaders();
    }
    if ($args->{'force_register'}) {
        $result .= &Apache::lonmenu::registerurl(1);
    }
    if (!$args->{'no_nav_bar'} 
	&& !$args->{'only_body'}
	&& !$args->{'frameset'}
	&& !$args->{'switchserver'}) {
	$result .= &help_menu_js($httphost);
        $result.=&modal_window();
        $result.=&togglebox_script();
        $result.=&wishlist_window();
        $result.=&LCprogressbarUpdate_script();
    } else {
        if ($args->{'add_modal'}) {
           $result.=&modal_window();
        }
        if ($args->{'add_wishlist'}) {
           $result.=&wishlist_window();
        }
        if ($args->{'add_togglebox'}) {
           $result.=&togglebox_script();
        }
        if ($args->{'add_progressbar'}) {
           $result.=&LCprogressbarUpdate_script();
        }
    }
    if (ref($args->{'redirect'})) {
	my ($time,$url,$inhibit_continue,$skip_enc_check) = @{$args->{'redirect'}};
        if (!$skip_enc_check) {
            $url = &Apache::lonenc::check_encrypt($url);
        }
	if (!$inhibit_continue) {
	    $env{'internal.head.redirect'} = $url;
	}
	$result.=<<ADDMETA
<meta http-equiv="pragma" content="no-cache" />
<meta http-equiv="Refresh" content="$time; url=$url" />
ADDMETA
    } else {
        unless (($args->{'frameset'}) || ($args->{'js_ready'}) || ($args->{'only_body'}) || ($args->{'no_nav_bar'})) {
            my $requrl = $env{'request.uri'};
            if ($requrl eq '') {
                $requrl = $ENV{'REQUEST_URI'};
                $requrl =~ s/\?.+$//;
            }
            unless (($requrl =~ m{^/adm/(?:switchserver|login|authenticate|logout|groupsort|cleanup|helper|slotrequest|grades)(\?|$)}) ||
                    (($requrl =~ m{^/res/}) && (($env{'form.submitted'} eq 'scantron') ||
                     ($env{'form.grade_symb'}) || ($Apache::lonhomework::scantronmode)))) {
                my $dom_in_use = $Apache::lonnet::perlvar{'lonDefDomain'};
                unless (&Apache::lonnet::allowed('mau',$dom_in_use)) {
                    my %domdefs = &Apache::lonnet::get_domain_defaults($dom_in_use);
                    my $lonhost = $Apache::lonnet::perlvar{'lonHostID'};
                    my ($offload,$offloadoth);
                    if (ref($domdefs{'offloadnow'}) eq 'HASH') {
                        if ($domdefs{'offloadnow'}{$lonhost}) {
                            $offload = 1;
                            if (($env{'user.domain'} ne '') && ($env{'user.domain'} ne $dom_in_use) &&
                                (!(($env{'user.name'} eq 'public') && ($env{'user.domain'} eq 'public')))) {
                                unless (&Apache::lonnet::shared_institution($env{'user.domain'})) {
                                    $offloadoth = 1;
                                    $dom_in_use = $env{'user.domain'};
                                }
                            }
                        }
                    }
                    unless ($offload) {
                        if (ref($domdefs{'offloadoth'}) eq 'HASH') {
                            if ($domdefs{'offloadoth'}{$lonhost}) {
                                if (($env{'user.domain'} ne '') && ($env{'user.domain'} ne $dom_in_use) &&
                                    (!(($env{'user.name'} eq 'public') && ($env{'user.domain'} eq 'public')))) {
                                    unless (&Apache::lonnet::shared_institution($env{'user.domain'})) {
                                        $offload = 1;
                                        $offloadoth = 1;
                                        $dom_in_use = $env{'user.domain'};
                                    }
                                }
                            }
                        }
                    }
                    if ($offload) {
                        my $newserver = &Apache::lonnet::spareserver(undef,30000,undef,1,$dom_in_use);
                        if (($newserver eq '') && ($offloadoth)) {
                            my @domains = &Apache::lonnet::current_machine_domains();
                            if (($dom_in_use ne '') && (!grep(/^\Q$dom_in_use\E$/,@domains))) { 
                                ($newserver) = &Apache::lonnet::choose_server($dom_in_use);
                            }
                        }
                        if (($newserver) && ($newserver ne $lonhost)) {
                            my $numsec = 5;
                            my $timeout = $numsec * 1000;
                            my ($newurl,$locknum,%locks,$msg);
                            if ($env{'request.role.adv'}) {
                                ($locknum,%locks) = &Apache::lonnet::get_locks();
                            }
                            my $disable_submit = 0;
                            if ($requrl =~ /$LONCAPA::assess_re/) {
                                $disable_submit = 1;
                            }
                            if ($locknum) {
                                my @lockinfo = sort(values(%locks));
                                $msg = &mt('Once the following tasks are complete:')." \n".
                                       join(", ",sort(values(%locks)))."\n";
                                if (&show_course()) {
                                    $msg .= &mt('your session will be transferred to a different server, after you click "Courses".');
                                } else {
                                    $msg .= &mt('your session will be transferred to a different server, after you click "Roles".');
                                }
                            } else {
                                if (($requrl =~ m{^/res/}) && ($env{'form.submitted'} =~ /^part_/)) {
                                    $msg = &mt('Your LON-CAPA submission has been recorded')."\n";
                                }
                                $msg .= &mt('Your current LON-CAPA session will be transferred to a different server in [quant,_1,second].',$numsec);
                                $newurl = '/adm/switchserver?otherserver='.$newserver;
                                if (($env{'request.role'}) && ($env{'request.role'} ne 'cm')) {
                                    $newurl .= '&role='.$env{'request.role'};
                                }
                                if ($env{'request.symb'}) {
                                    my $shownsymb = &Apache::lonenc::check_encrypt($env{'request.symb'});
                                    if ($shownsymb =~ m{^/enc/}) {
                                        my $reqdmajor = 2;
                                        my $reqdminor = 11;
                                        my $reqdsubminor = 3;
                                        my $newserverrev = &Apache::lonnet::get_server_loncaparev('',$newserver);
                                        my $remoterev = &Apache::lonnet::get_server_loncaparev(undef,$newserver);
                                        my ($major,$minor,$subminor) = ($remoterev =~ /^\'?(\d+)\.(\d+)\.(\d+|)[\w.\-]+\'?$/);
                                        if (($major eq '' && $minor eq '') ||
                                            (($reqdmajor > $major) || (($reqdmajor == $major) && ($reqdminor > $minor)) ||
                                            (($reqdmajor == $major) && ($reqdminor == $minor) && (($subminor eq '') ||
                                             ($reqdsubminor > $subminor))))) {
                                            undef($shownsymb);
                                        }
                                    }
                                    if ($shownsymb) {
                                        &js_escape(\$shownsymb);
                                        $newurl .= '&symb='.$shownsymb;
                                    }
                                } else {
                                    my $shownurl = &Apache::lonenc::check_encrypt($requrl);
                                    &js_escape(\$shownurl);
                                    $newurl .= '&origurl='.$shownurl;
                                }
                            }
                            &js_escape(\$msg);
                            $result.=<<OFFLOAD
<meta http-equiv="pragma" content="no-cache" />
<script type="text/javascript">
// <![CDATA[
function LC_Offload_Now() {
    var dest = "$newurl";
    if (dest != '') {
        window.location.href="$newurl";
    }
}
\$(document).ready(function () {
    window.alert('$msg');
    if ($disable_submit) {
        \$(".LC_hwk_submit").prop("disabled", true);
        \$( ".LC_textline" ).prop( "readonly", "readonly");
    }
    setTimeout('LC_Offload_Now()', $timeout);
});
// ]]>
</script>
OFFLOAD
                        }
                    }
                }
            }
        }
    }
    if (!defined($title)) {
	$title = 'The LearningOnline Network with CAPA';
    }
    if (!$args->{'no_auto_mt_title'}) { $title = &mt($title); }
    if ($title =~ /^LON-CAPA\s+/) {
        $result .= '<title> '.$title.'</title>';
    } else {
        $result .= '<title> LON-CAPA '.$title.'</title>';
    }
    $result .= "\n".'<link rel="stylesheet" type="text/css" href="'.$url.'"';
    if (!$args->{'frameset'}) {
        $result .= ' /';
    }
    $result .= '>'
        .$inhibitprint
	.$head_extra;
    my $clientmobile;
    if (($env{'user.name'} eq '') && ($env{'user.domain'} eq '')) {
        (undef,undef,undef,undef,undef,undef,$clientmobile) = &decode_user_agent();
    } else {
        $clientmobile = $env{'browser.mobile'};
    }
    if ($clientmobile) {
        $result .= '
<meta name="viewport" content="width=device-width, initial-scale=1.0, user-scalable=0, minimum-scale=1.0, maximum-scale=1.0">
<meta name="apple-mobile-web-app-capable" content="yes" />';
    }
    $result .= '<meta name="google" content="notranslate" />'."\n";
    return $result.'</head>';
}

=pod

=item * &font_settings()

Returns neccessary <meta> to set the proper encoding

Inputs: optional reference to HASH -- $args passed to &headtag()

=cut

sub font_settings {
    my ($args) = @_;
    my $headerstring='';
    if ((!$env{'browser.mathml'} && $env{'browser.unicode'}) ||
        ((ref($args) eq 'HASH') && ($args->{'browser.unicode'}))) {
	$headerstring.=
	    '<meta http-equiv="Content-Type" content="text/html; charset=utf-8"';
        if (!$args->{'frameset'}) {
            $headerstring.= ' /';
        }
        $headerstring .= '>'."\n";
    }
    return $headerstring;
}

=pod

=item * &print_suppression()

In course context returns css which causes the body to be blank when media="print",
if printout generation is unavailable for the current resource.

This could be because:

(a) printstartdate is in the future

(b) printenddate is in the past

(c) there is an active exam block with "printout"
functionality blocked

Users with pav, pfo or evb privileges are exempt.

Inputs: none

=cut


sub print_suppression {
    my $noprint;
    if ($env{'request.course.id'}) {
        my $scope = $env{'request.course.id'};
        if ((&Apache::lonnet::allowed('pav',$scope)) ||
            (&Apache::lonnet::allowed('pfo',$scope))) {
            return;
        }
        if ($env{'request.course.sec'} ne '') {
            $scope .= "/$env{'request.course.sec'}";
            if ((&Apache::lonnet::allowed('pav',$scope)) ||
                (&Apache::lonnet::allowed('pfo',$scope))) {
                return;
            }
        }
        my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
        my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
        my $clientip = &Apache::lonnet::get_requestor_ip();
        my $blocked = &blocking_status('printout',$clientip,$cnum,$cdom,undef,1);
        if ($blocked) {
            my $checkrole = "cm./$cdom/$cnum";
            if ($env{'request.course.sec'} ne '') {
                $checkrole .= "/$env{'request.course.sec'}";
            }
            unless ((&Apache::lonnet::allowed('evb',undef,undef,$checkrole)) &&
                    ($env{'request.role'} !~ m{^st\./$cdom/$cnum})) {
                $noprint = 1;
            }
        }
        unless ($noprint) {
            my $symb = &Apache::lonnet::symbread();
            if ($symb ne '') {
                my $navmap = Apache::lonnavmaps::navmap->new();
                if (ref($navmap)) {
                    my $res = $navmap->getBySymb($symb);
                    if (ref($res)) {
                        if (!$res->resprintable()) {
                            $noprint = 1;
                        }
                    }
                }
            }
        }
        if ($noprint) {
            return <<"ENDSTYLE";
<style type="text/css" media="print">
    body { display:none }
</style>
ENDSTYLE
        }
    }
    return;
}

=pod

=item * &xml_begin()

Returns the needed doctype and <html>

Inputs: none

=cut

sub xml_begin {
    my ($is_frameset) = @_;
    my $output='';

    if ($env{'browser.mathml'}) {
	$output='<?xml version="1.0"?>'
            #.'<?xml-stylesheet type="text/css" href="/adm/MathML/mathml.css"?>'."\n"
#            .'<!DOCTYPE html SYSTEM "/adm/MathML/mathml.dtd" '
            
#	    .'<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd" [<!ENTITY mathns "http://www.w3.org/1998/Math/MathML">] >'
	    .'<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.1 plus MathML 2.0 plus SVG 1.1//EN" "http://www.w3.org/2002/04/xhtml-math-svg/xhtml-math-svg.dtd">'
            .'<html xmlns:math="http://www.w3.org/1998/Math/MathML" ' 
	    .'xmlns="http://www.w3.org/1999/xhtml">';
    } elsif ($is_frameset) {
        $output='<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.01 Frameset//EN" "http://www.w3.org/TR/html4/frameset.dtd">'."\n".
                '<html>'."\n";
    } else {
	$output='<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">'."\n".
                '<html xmlns="http://www.w3.org/1999/xhtml" xml:lang="en" lang="en">'."\n";
    }
    return $output;
}

=pod

=item * &start_page()

Returns a complete <html> .. <body> section for LON-CAPA web pages.

Inputs:

=over 4

$title - optional title for the page

$head_extra - optional extra HTML to incude inside the <head>

$args - additional optional args supported are:

=over 8

             only_body      -> is true will set &bodytag() onlybodytag
                                    arg on
             no_nav_bar     -> is true will set &bodytag() no_nav_bar arg on
             add_entries    -> additional attributes to add to the  <body>
             domain         -> force to color decorate a page for a 
                                    specific domain
             function       -> force usage of a specific rolish color
                                    scheme
             redirect       -> see &headtag()
             bgcolor        -> override the default page bg color
             js_ready       -> return a string ready for being used in 
                                    a javascript writeln
             html_encode    -> return a string ready for being used in 
                                    a html attribute
             force_register -> if is true will turn on the &bodytag()
                                    $forcereg arg
             frameset       -> if true will start with a <frameset>
                                    rather than <body>
             skip_phases    -> hash ref of 
                                    head -> skip the <html><head> generation
                                    body -> skip all <body> generation
             no_inline_link -> if true and in remote mode, don't show the
                                    'Switch To Inline Menu' link
             no_auto_mt_title -> prevent &mt()ing the title arg
             bread_crumbs ->             Array containing breadcrumbs
             bread_crumbs_component ->  if exists show it as headline else show only the breadcrumbs
             bread_crumbs_style -> breadcrumbs are contained within <div id="LC_breadcrumbs">,
                                   and &standard_css() contains CSS for #LC_breadcrumbs, if you want
                                   to override those values, or add to them, specify the value to
                                   include in the style attribute to include in the div tag by using
                                   bread_crumbs_style (e.g., overflow: visible)
             bread_crumbs_nomenu -> if true will pass false as the value of $menulink
                                    to lonhtmlcommon::breadcrumbs
             group          -> includes the current group, if page is for a
                               specific group
             use_absolute   -> for request for external resource or syllabus, this
                               will contain https://<hostname> if server uses
                               https (as per hosts.tab), but request is for http
             hostname       -> hostname, originally from $r->hostname(), (optional).
             links_disabled -> Links in primary and secondary menus are disabled
                               (Can enable them once page has loaded - see lonroles.pm
                               for an example).

=back

=back

=cut

sub start_page {
    my ($title,$head_extra,$args) = @_;
    #&Apache::lonnet::logthis("start_page ".join(':',caller(0)));

    $env{'internal.start_page'}++;
    my ($result,@advtools);

    if (! exists($args->{'skip_phases'}{'head'}) ) {
        $result .= &xml_begin($args->{'frameset'}) . &headtag($title, $head_extra, $args);
    }
    
    if (! exists($args->{'skip_phases'}{'body'}) ) {
	if ($args->{'frameset'}) {
	    my $attr_string = &make_attr_string($args->{'force_register'},
						$args->{'add_entries'});
	    $result .= "\n<frameset $attr_string>\n";
        } else {
            $result .=
                &bodytag($title, 
                         $args->{'function'},       $args->{'add_entries'},
                         $args->{'only_body'},      $args->{'domain'},
                         $args->{'force_register'}, $args->{'no_nav_bar'},
                         $args->{'bgcolor'},        $args->{'no_inline_link'},
                         $args,                     \@advtools);
        }
    }

    if ($args->{'js_ready'}) {
		$result = &js_ready($result);
    }
    if ($args->{'html_encode'}) {
		$result = &html_encode($result);
    }

    # Preparation for new and consistent functionlist at top of screen
    # if ($args->{'functionlist'}) {
    #            $result .= &build_functionlist();
    #}

    # Don't add anything more if only_body wanted or in const space
    return $result if    $args->{'only_body'} 
                      || $env{'request.state'} eq 'construct';

    #Breadcrumbs
    if (exists($args->{'bread_crumbs'}) or exists($args->{'bread_crumbs_component'})) {
		&Apache::lonhtmlcommon::clear_breadcrumbs();
		#if any br links exists, add them to the breadcrumbs
		if (exists($args->{'bread_crumbs'}) and ref($args->{'bread_crumbs'}) eq 'ARRAY') {         
			foreach my $crumb (@{$args->{'bread_crumbs'}}){
				&Apache::lonhtmlcommon::add_breadcrumb($crumb);
			}
		}
                # if @advtools array contains items add then to the breadcrumbs
                if (@advtools > 0) {
                    &Apache::lonmenu::advtools_crumbs(@advtools);
                }
                my $menulink;
                # if arg: bread_crumbs_nomenu is true pass 0 as $menulink item.
                if (exists($args->{'bread_crumbs_nomenu'})) {
                    $menulink = 0;
                } else {
                    undef($menulink);
                }
		#if bread_crumbs_component exists show it as headline else show only the breadcrumbs
		if(exists($args->{'bread_crumbs_component'})){
			$result .= &Apache::lonhtmlcommon::breadcrumbs($args->{'bread_crumbs_component'},
                                                                       '',$menulink,'',
                                                                       $args->{'bread_crumbs_style'});
		}else{
			$result .= &Apache::lonhtmlcommon::breadcrumbs('','',$menulink,'',
                                                                       $args->{'bread_crumbs_style'});
		}
    } elsif (($env{'environment.remote'} eq 'on') &&
             ($env{'form.inhibitmenu'} ne 'yes') &&
             ($env{'request.noversionuri'} =~ m{^/res/}) &&
             ($env{'request.noversionuri'} !~ m{^/res/adm/pages/})) {
        $result .= '<div style="padding:0;margin:0;clear:both"><hr /></div>';
    }
    return $result;
}

sub end_page {
    my ($args) = @_;
    $env{'internal.end_page'}++;
    my $result;
    if ($args->{'discussion'}) {
	my ($target,$parser);
	if (ref($args->{'discussion'})) {
	    ($target,$parser) =($args->{'discussion'}{'target'},
				$args->{'discussion'}{'parser'});
	}
	$result .= &Apache::lonxml::xmlend($target,$parser);
    }
    if ($args->{'frameset'}) {
	$result .= '</frameset>';
    } else {
	$result .= &endbodytag($args);
    }
    unless ($args->{'notbody'}) {
        $result .= "\n</html>";
    }

    if ($args->{'js_ready'}) {
	$result = &js_ready($result);
    }

    if ($args->{'html_encode'}) {
	$result = &html_encode($result);
    }

    return $result;
}

sub wishlist_window {
    return(<<'ENDWISHLIST');
<script type="text/javascript">
// <![CDATA[
// <!-- BEGIN LON-CAPA Internal
function set_wishlistlink(title, path) {
    if (!title) {
        title = document.title;
        title = title.replace(/^LON-CAPA /,'');
    }
    title = encodeURIComponent(title);
    title = title.replace("'","\\\'");
    if (!path) {
        path = location.pathname;
    }
    path = encodeURIComponent(path);
    path = path.replace("'","\\\'");
    Win = window.open('/adm/wishlist?mode=newLink&setTitle='+title+'&setPath='+path,
                      'wishlistNewLink','width=560,height=350,scrollbars=0');
}
// END LON-CAPA Internal -->
// ]]>
</script>
ENDWISHLIST
}

sub modal_window {
    return(<<'ENDMODAL');
<script type="text/javascript">
// <![CDATA[
// <!-- BEGIN LON-CAPA Internal
var modalWindow = {
	parent:"body",
	windowId:null,
	content:null,
	width:null,
	height:null,
	close:function()
	{
	        $(".LCmodal-window").remove();
	        $(".LCmodal-overlay").remove();
	},
	open:function()
	{
		var modal = "";
		modal += "<div class=\"LCmodal-overlay\"></div>";
		modal += "<div id=\"" + this.windowId + "\" class=\"LCmodal-window\" style=\"width:" + this.width + "px; height:" + this.height + "px; margin-top:-" + (this.height / 2) + "px; margin-left:-" + (this.width / 2) + "px;\">";
		modal += this.content;
		modal += "</div>";	

		$(this.parent).append(modal);

		$(".LCmodal-window").append("<a class=\"LCclose-window\"></a>");
		$(".LCclose-window").click(function(){modalWindow.close();});
		$(".LCmodal-overlay").click(function(){modalWindow.close();});
	}
};
	var openMyModal = function(source,width,height,scrolling,transparency,style)
	{
                source = source.replace(/'/g,"&#39;");
		modalWindow.windowId = "myModal";
		modalWindow.width = width;
		modalWindow.height = height;
		modalWindow.content = "<iframe width='"+width+"' height='"+height+"' frameborder='0' scrolling='"+scrolling+"' allowtransparency='"+transparency+"' src='" + source + "' style='"+style+"'></iframe>";
		modalWindow.open();
	};
// END LON-CAPA Internal -->
// ]]>
</script>
ENDMODAL
}

sub modal_link {
    my ($link,$linktext,$width,$height,$target,$scrolling,$title,$transparency,$style)=@_;
    unless ($width) { $width=480; }
    unless ($height) { $height=400; }
    unless ($scrolling) { $scrolling='yes'; }
    unless ($transparency) { $transparency='true'; }

    my $target_attr;
    if (defined($target)) {
        $target_attr = 'target="'.$target.'"';
    }
    return <<"ENDLINK";
<a href="$link" $target_attr title="$title" onclick="javascript:openMyModal('$link',$width,$height,'$scrolling','$transparency','$style'); return false;">$linktext</a>
ENDLINK
}

sub modal_adhoc_script {
    my ($funcname,$width,$height,$content,$possmathjax)=@_;
    my $mathjax;
    if ($possmathjax) {
        $mathjax = <<'ENDJAX';
               if (typeof MathJax == 'object') {
                   MathJax.Hub.Queue(["Typeset",MathJax.Hub]);
               }
ENDJAX
    }
    return (<<ENDADHOC);
<script type="text/javascript">
// <![CDATA[
        var $funcname = function()
        {
                modalWindow.windowId = "myModal";
                modalWindow.width = $width;
                modalWindow.height = $height;
                modalWindow.content = '$content';
                modalWindow.open();
                $mathjax
        };  
// ]]>
</script>
ENDADHOC
}

sub modal_adhoc_inner {
    my ($funcname,$width,$height,$content,$possmathjax)=@_;
    my $innerwidth=$width-20;
    $content=&js_ready(
               &start_page('Dialog',undef,{'only_body'=>1,'bgcolor'=>'#FFFFFF'}).
                 &start_scrollbox($width.'px',$innerwidth.'px',$height.'px','myModal','#FFFFFF',undef,1).
                 $content.
                 &end_scrollbox().
                 &end_page()
             );
    return &modal_adhoc_script($funcname,$width,$height,$content,$possmathjax);
}

sub modal_adhoc_window {
    my ($funcname,$width,$height,$content,$linktext,$possmathjax)=@_;
    return &modal_adhoc_inner($funcname,$width,$height,$content,$possmathjax).
           "<a href=\"javascript:$funcname();void(0);\">".$linktext."</a>";
}

sub modal_adhoc_launch {
    my ($funcname,$width,$height,$content)=@_;
    return &modal_adhoc_inner($funcname,$width,$height,$content).(<<ENDLAUNCH);
<script type="text/javascript">
// <![CDATA[
$funcname();
// ]]>
</script>
ENDLAUNCH
}

sub modal_adhoc_close {
    return (<<ENDCLOSE);
<script type="text/javascript">
// <![CDATA[
modalWindow.close();
// ]]>
</script>
ENDCLOSE
}

sub togglebox_script {
   return(<<ENDTOGGLE);
<script type="text/javascript"> 
// <![CDATA[
function LCtoggleDisplay(id,hidetext,showtext) {
   link = document.getElementById(id + "link").childNodes[0];
   with (document.getElementById(id).style) {
      if (display == "none" ) {
          display = "inline";
          link.nodeValue = hidetext;
        } else {
          display = "none";
          link.nodeValue = showtext;
       }
   }
}
// ]]>
</script>
ENDTOGGLE
}

sub start_togglebox {
    my ($id,$heading,$headerbg,$hidetext,$showtext)=@_;
    unless ($heading) { $heading=''; } else { $heading.=' '; }
    unless ($showtext) { $showtext=&mt('show'); }
    unless ($hidetext) { $hidetext=&mt('hide'); }
    unless ($headerbg) { $headerbg='#FFFFFF'; }
    return &start_data_table().
           &start_data_table_header_row().
           '<td bgcolor="'.$headerbg.'">'.$heading.
           '[<a id="'.$id.'link" href="javascript:LCtoggleDisplay(\''.$id.'\',\''.$hidetext.'\',\''.
           $showtext.'\')">'.$showtext.'</a>]</td>'.
           &end_data_table_header_row().
           '<tr id="'.$id.'" style="display:none""><td>';
}

sub end_togglebox {
    return '</td></tr>'.&end_data_table();
}

sub LCprogressbar_script {
   my ($id,$number_to_do)=@_;
   if ($number_to_do) {
       return(<<ENDPROGRESS);
<script type="text/javascript">
// <![CDATA[
\$('#progressbar$id').progressbar({
  value: 0,
  change: function(event, ui) {
    var newVal = \$(this).progressbar('option', 'value');
    \$('.pblabel', this).text(LCprogressTxt);
  }
});
// ]]>
</script>
ENDPROGRESS
   } else {
       return(<<ENDPROGRESS);
<script type="text/javascript">
// <![CDATA[
\$('#progressbar$id').progressbar({
  value: false,
  create: function(event, ui) {
    \$('.ui-widget-header', this).css({'background':'#F0F0F0'});
    \$('.ui-progressbar-overlay', this).css({'margin':'0'});
  }
});
// ]]>
</script>
ENDPROGRESS
   }
}

sub LCprogressbarUpdate_script {
   return(<<ENDPROGRESSUPDATE);
<style type="text/css">
.ui-progressbar { position:relative; }
.progress-label {position: absolute; width: 100%; text-align: center; top: 1px; font-weight: bold; text-shadow: 1px 1px 0 #fff;margin: 0; line-height: 200%; }
.pblabel { position: absolute; width: 100%; text-align: center; line-height: 1.9em; }
</style>
<script type="text/javascript">
// <![CDATA[
var LCprogressTxt='---';

function LCupdateProgress(percent,progresstext,id,maxnum) {
   LCprogressTxt=progresstext;
   if ((maxnum == '') || (maxnum == undefined) || (maxnum == null)) {
       \$('#progressbar'+id).find('.progress-label').text(LCprogressTxt);
   } else if (percent === \$('#progressbar'+id).progressbar( "value" )) {
       \$('#progressbar'+id).find('.pblabel').text(LCprogressTxt);
   } else {
       \$('#progressbar'+id).progressbar('value',percent);
   }
}
// ]]>
</script>
ENDPROGRESSUPDATE
}

my $LClastpercent;
my $LCidcnt;
my $LCcurrentid;

sub LCprogressbar {
    my ($r,$number_to_do,$preamble)=@_;
    $LClastpercent=0;
    $LCidcnt++;
    $LCcurrentid=$$.'_'.$LCidcnt;
    my ($starting,$content);
    if ($number_to_do) {
        $starting=&mt('Starting');
        $content=(<<ENDPROGBAR);
$preamble
  <div id="progressbar$LCcurrentid">
    <span class="pblabel">$starting</span>
  </div>
ENDPROGBAR
    } else {
        $starting=&mt('Loading...');
        $LClastpercent='false';
        $content=(<<ENDPROGBAR);
$preamble
  <div id="progressbar$LCcurrentid">
      <div class="progress-label">$starting</div>
  </div>
ENDPROGBAR
    }
    &r_print($r,$content.&LCprogressbar_script($LCcurrentid,$number_to_do));
}

sub LCprogressbarUpdate {
    my ($r,$val,$text,$number_to_do)=@_;
    if ($number_to_do) {
        unless ($val) { 
            if ($LClastpercent) {
                $val=$LClastpercent;
            } else {
                $val=0;
            }
        }
        if ($val<0) { $val=0; }
        if ($val>100) { $val=0; }
        $LClastpercent=$val;
        unless ($text) { $text=$val.'%'; }
    } else {
        $val = 'false';
    }
    $text=&js_ready($text);
    &r_print($r,<<ENDUPDATE);
<script type="text/javascript">
// <![CDATA[
LCupdateProgress($val,'$text','$LCcurrentid','$number_to_do');
// ]]>
</script>
ENDUPDATE
}

sub LCprogressbarClose {
    my ($r)=@_;
    $LClastpercent=0;
    &r_print($r,<<ENDCLOSE);
<script type="text/javascript">
// <![CDATA[
\$("#progressbar$LCcurrentid").hide('slow'); 
// ]]>
</script>
ENDCLOSE
}

sub r_print {
    my ($r,$to_print)=@_;
    if ($r) {
      $r->print($to_print);
      $r->rflush();
    } else {
      print($to_print);
    }
}

sub html_encode {
    my ($result) = @_;

    $result = &HTML::Entities::encode($result,'<>&"');
    
    return $result;
}

sub js_ready {
    my ($result) = @_;

    $result =~ s/[\n\r]/ /xmsg;
    $result =~ s/\\/\\\\/xmsg;
    $result =~ s/'/\\'/xmsg;
    $result =~ s{</}{<\\/}xmsg;
    
    return $result;
}

sub validate_page {
    if (  exists($env{'internal.start_page'})
	  &&     $env{'internal.start_page'} > 1) {
	&Apache::lonnet::logthis('start_page called multiple times '.
				 $env{'internal.start_page'}.' '.
				 $ENV{'request.filename'});
    }
    if (  exists($env{'internal.end_page'})
	  &&     $env{'internal.end_page'} > 1) {
	&Apache::lonnet::logthis('end_page called multiple times '.
				 $env{'internal.end_page'}.' '.
				 $env{'request.filename'});
    }
    if (     exists($env{'internal.start_page'})
	&& ! exists($env{'internal.end_page'})) {
	&Apache::lonnet::logthis('start_page called without end_page '.
				 $env{'request.filename'});
    }
    if (   ! exists($env{'internal.start_page'})
	&&   exists($env{'internal.end_page'})) {
	&Apache::lonnet::logthis('end_page called without start_page'.
				 $env{'request.filename'});
    }
}


sub start_scrollbox {
    my ($outerwidth,$width,$height,$id,$bgcolor,$cursor,$needjsready) = @_;
    unless ($outerwidth) { $outerwidth='520px'; }
    unless ($width) { $width='500px'; }
    unless ($height) { $height='200px'; }
    my ($table_id,$div_id,$tdcol);
    if ($id ne '') {
        $table_id = ' id="table_'.$id.'"';
        $div_id = ' id="div_'.$id.'"';
    }
    if ($bgcolor ne '') {
        $tdcol = "background-color: $bgcolor;";
    }
    my $nicescroll_js;
    if ($env{'browser.mobile'}) {
        $nicescroll_js = &nicescroll_javascript('div_'.$id,$cursor,$needjsready);
    }
    return <<"END";
$nicescroll_js

<table style="width: $outerwidth; border: 1px solid none;"$table_id><tr><td style="width: $width;$tdcol">
<div style="overflow:auto; width:$width; height:$height;"$div_id>
END
}

sub end_scrollbox {
    return '</div></td></tr></table>';
}

sub nicescroll_javascript {
    my ($id,$cursor,$needjsready,$framecheck,$location) = @_;
    my %options;
    if (ref($cursor) eq 'HASH') {
        %options = %{$cursor};
    }
    unless ($options{'railalign'} =~ /^left|right$/) {
        $options{'railalign'} = 'left';
    }
    unless ($options{'cursorcolor'} =~ /^\#\w+$/) {
        my $function  = &get_users_function();
        $options{'cursorcolor'} = &designparm($function.'.sidebg',$env{'request.role.domain'});
        unless ($options{'cursorcolor'} =~ /^\#\w+$/) {
            $options{'cursorcolor'} = '#00F';
        }
    }
    if ($options{'cursoropacity'} =~ /^[\d.]+$/) {
        unless ($options{'cursoropacity'} >= 0.0 && $options{'cursoropacity'} <=1.0) {
            $options{'cursoropacity'}='1.0';
        }
    } else {
        $options{'cursoropacity'}='1.0';
    }
    if ($options{'cursorfixedheight'} eq 'none') {
        delete($options{'cursorfixedheight'});
    } else {
        unless ($options{'cursorfixedheight'} =~ /^\d+$/) { $options{'cursorfixedheight'}='50'; }
    }
    unless ($options{'railoffset'} =~ /^{[\w\:\d\-,]+}$/) {
        delete($options{'railoffset'});
    }
    my @niceoptions;
    while (my($key,$value) = each(%options)) {
        if ($value =~ /^\{.+\}$/) {
            push(@niceoptions,$key.':'.$value);
        } else {
            push(@niceoptions,$key.':"'.$value.'"');
        }
    }
    my $nicescroll_js = '
$(document).ready(
      function() {
          $("#'.$id.'").niceScroll({'.join(',',@niceoptions).'});
      }
);
';
    if ($framecheck) {
        $nicescroll_js .= '
function expand_div(caller) {
    if (top === self) {
        document.getElementById("'.$id.'").style.width = "auto";
        document.getElementById("'.$id.'").style.height = "auto";
    } else {
        try {
            if (parent.frames) {
                if (parent.frames.length > 1) {
                    var framesrc = parent.frames[1].location.href;
                    var currsrc = framesrc.replace(/\#.*$/,"");
                    if ((caller == "search") || (currsrc == "'.$location.'")) {
                        document.getElementById("'.$id.'").style.width = "auto";
                        document.getElementById("'.$id.'").style.height = "auto";
                    }
                }
            }
        } catch (e) {
            return;
        }
    }
    return;
}
';
    }
    if ($needjsready) {
        $nicescroll_js = '
<script type="text/javascript">'."\n".$nicescroll_js."\n</script>\n";
    } else {
        $nicescroll_js = &Apache::lonhtmlcommon::scripttag($nicescroll_js);
    }
    return $nicescroll_js;
}

sub simple_error_page {
    my ($r,$title,$msg,$args) = @_;
    if (ref($args) eq 'HASH') {
        if (!$args->{'no_auto_mt_msg'}) { $msg = &mt($msg); }
    } else {
        $msg = &mt($msg);
    }

    my $page =
	&Apache::loncommon::start_page($title).
	'<p class="LC_error">'.$msg.'</p>'.
	&Apache::loncommon::end_page();
    if (ref($r)) {
	$r->print($page);
	return;
    }
    return $page;
}

{
    my @row_count;

    sub start_data_table_count {
        unshift(@row_count, 0);
        return;
    }

    sub end_data_table_count {
        shift(@row_count);
        return;
    }

    sub start_data_table {
	my ($add_class,$id) = @_;
	my $css_class = (join(' ','LC_data_table',$add_class));
        my $table_id;
        if (defined($id)) {
            $table_id = ' id="'.$id.'"';
        }
	&start_data_table_count();
	return '<table class="'.$css_class.'"'.$table_id.'>'."\n";
    }

    sub end_data_table {
	&end_data_table_count();
	return '</table>'."\n";;
    }

    sub start_data_table_row {
	my ($add_class, $id) = @_;
	$row_count[0]++;
	my $css_class = ($row_count[0] % 2)?'LC_odd_row':'LC_even_row';
	$css_class = (join(' ',$css_class,$add_class)) unless ($add_class eq '');
        $id = (' id="'.$id.'"') unless ($id eq '');
        return  '<tr class="'.$css_class.'"'.$id.'>'."\n";
    }
    
    sub continue_data_table_row {
	my ($add_class, $id) = @_;
	my $css_class = ($row_count[0] % 2)?'LC_odd_row':'LC_even_row';
	$css_class = (join(' ',$css_class,$add_class)) unless ($add_class eq '');
        $id = (' id="'.$id.'"') unless ($id eq '');
        return  '<tr class="'.$css_class.'"'.$id.'>'."\n";
    }

    sub end_data_table_row {
	return '</tr>'."\n";;
    }

    sub start_data_table_empty_row {
#	$row_count[0]++;
	return  '<tr class="LC_empty_row" >'."\n";;
    }

    sub end_data_table_empty_row {
	return '</tr>'."\n";;
    }

    sub start_data_table_header_row {
	return  '<tr class="LC_header_row">'."\n";;
    }

    sub end_data_table_header_row {
	return '</tr>'."\n";;
    }

    sub data_table_caption {
        my $caption = shift;
        return "<caption class=\"LC_caption\">$caption</caption>";
    }
}

=pod

=item * &inhibit_menu_check($arg)

Checks for a inhibitmenu state and generates output to preserve it

Inputs:         $arg - can be any of
                     - undef - in which case the return value is a string 
                               to add  into arguments list of a uri
                     - 'input' - in which case the return value is a HTML
                                 <form> <input> field of type hidden to
                                 preserve the value
                     - a url - in which case the return value is the url with
                               the neccesary cgi args added to preserve the
                               inhibitmenu state
                     - a ref to a url - no return value, but the string is
                                        updated to include the neccessary cgi
                                        args to preserve the inhibitmenu state

=cut

sub inhibit_menu_check {
    my ($arg) = @_;
    &get_unprocessed_cgi($ENV{'QUERY_STRING'}, ['inhibitmenu']);
    if ($arg eq 'input') {
	if ($env{'form.inhibitmenu'}) {
	    return '<input type="hidden" name="inhibitmenu" value="'.$env{'form.inhibitmenu'}.'" />';
	} else {
	    return
	}
    }
    if ($env{'form.inhibitmenu'}) {
	if (ref($arg)) {
	    $$arg .= '?inhibitmenu='.$env{'form.inhibitmenu'};
	} elsif ($arg eq '') {
	    $arg .= 'inhibitmenu='.$env{'form.inhibitmenu'};
	} else {
	    $arg .= '?inhibitmenu='.$env{'form.inhibitmenu'};
	}
    }
    if (!ref($arg)) {
	return $arg;
    }
}

###############################################

=pod

=back

=head1 User Information Routines

=over 4

=item * &get_users_function()

Used by &bodytag to determine the current users primary role.
Returns either 'student','coordinator','admin', or 'author'.

=cut

###############################################
sub get_users_function {
    my $function = 'norole';
    if ($env{'request.role'}=~/^(st)/) {
        $function='student';
    }
    if ($env{'request.role'}=~/^(cc|co|in|ta|ep)/) {
        $function='coordinator';
    }
    if ($env{'request.role'}=~/^(su|dc|ad|li)/) {
        $function='admin';
    }
    if (($env{'request.role'}=~/^(au|ca|aa)/) ||
        ($ENV{'REQUEST_URI'}=~ m{/^(/priv)})) {
        $function='author';
    }
    return $function;
}

###############################################

=pod

=item * &show_course()

Used by lonmenu.pm and lonroles.pm to determine whether to use the word
'Courses' or 'Roles' in inline navigation and on screen displaying user's roles.

Inputs:
None

Outputs:
Scalar: 1 if 'Course' to be used, 0 otherwise.

=cut

###############################################
sub show_course {
    my $course = !$env{'user.adv'};
    if (!$env{'user.adv'}) {
        foreach my $env (keys(%env)) {
            next if ($env !~ m/^user\.priv\./);
            if ($env !~ m/^user\.priv\.(?:st|cm)/) {
                $course = 0;
                last;
            }
        }
    }
    return $course;
}

###############################################

=pod

=item * &check_user_status()

Determines current status of supplied role for a
specific user. Roles can be active, previous or future.

Inputs: 
user's domain, user's username, course's domain,
course's number, optional section ID.

Outputs:
role status: active, previous or future. 

=cut

sub check_user_status {
    my ($udom,$uname,$cdom,$crs,$role,$sec) = @_;
    my %userinfo = &Apache::lonnet::dump('roles',$udom,$uname);
    my @uroles = keys(%userinfo);
    my $srchstr;
    my $active_chk = 'none';
    my $now = time;
    if (@uroles > 0) {
        if (($role eq 'cc') || ($role eq 'co') || ($sec eq '') || (!defined($sec))) {
            $srchstr = '/'.$cdom.'/'.$crs.'_'.$role;
        } else {
            $srchstr = '/'.$cdom.'/'.$crs.'/'.$sec.'_'.$role;
        }
        if (grep/^\Q$srchstr\E$/,@uroles) {
            my $role_end = 0;
            my $role_start = 0;
            $active_chk = 'active';
            if ($userinfo{$srchstr} =~ m/^\Q$role\E_(\d+)/) {
                $role_end = $1;
                if ($userinfo{$srchstr} =~ m/^\Q$role\E_\Q$role_end\E_(\d+)$/) {
                    $role_start = $1;
                }
            }
            if ($role_start > 0) {
                if ($now < $role_start) {
                    $active_chk = 'future';
                }
            }
            if ($role_end > 0) {
                if ($now > $role_end) {
                    $active_chk = 'previous';
                }
            }
        }
    }
    return $active_chk;
}

###############################################

=pod

=item * &get_sections()

Determines all the sections for a course including
sections with students and sections containing other roles.
Incoming parameters: 

1. domain
2. course number 
3. reference to array containing roles for which sections should 
be gathered (optional).
4. reference to array containing status types for which sections 
should be gathered (optional).

If the third argument is undefined, sections are gathered for any role. 
If the fourth argument is undefined, sections are gathered for any status.
Permissible values are 'active' or 'future' or 'previous'.
 
Returns section hash (keys are section IDs, values are
number of users in each section), subject to the
optional roles filter, optional status filter 

=cut

###############################################
sub get_sections {
    my ($cdom,$cnum,$possible_roles,$possible_status) = @_;
    if (!defined($cdom) || !defined($cnum)) {
        my $cid =  $env{'request.course.id'};

	return if (!defined($cid));

        $cdom = $env{'course.'.$cid.'.domain'};
        $cnum = $env{'course.'.$cid.'.num'};
    }

    my %sectioncount;
    my $now = time;

    my $check_students = 1;
    my $only_students = 0;
    if (ref($possible_roles) eq 'ARRAY') {
        if (grep(/^st$/,@{$possible_roles})) {
            if (@{$possible_roles} == 1) {
                $only_students = 1;
            }
        } else {
            $check_students = 0;
        }
    }

    if ($check_students) {
	my ($classlist) = &Apache::loncoursedata::get_classlist($cdom,$cnum);
	my $sec_index = &Apache::loncoursedata::CL_SECTION();
	my $status_index = &Apache::loncoursedata::CL_STATUS();
        my $start_index = &Apache::loncoursedata::CL_START();
        my $end_index = &Apache::loncoursedata::CL_END();
        my $status;
	while (my ($student,$data) = each(%$classlist)) {
	    my ($section,$stu_status,$start,$end) = ($data->[$sec_index],
				                     $data->[$status_index],
                                                     $data->[$start_index],
                                                     $data->[$end_index]);
            if ($stu_status eq 'Active') {
                $status = 'active';
            } elsif ($end < $now) {
                $status = 'previous';
            } elsif ($start > $now) {
                $status = 'future';
            } 
	    if ($section ne '-1' && $section !~ /^\s*$/) {
                if ((!defined($possible_status)) || (($status ne '') && 
                    (grep/^\Q$status\E$/,@{$possible_status}))) { 
		    $sectioncount{$section}++;
                }
	    }
	}
    }
    if ($only_students) {
        return %sectioncount;
    }
    my %courseroles = &Apache::lonnet::dump('nohist_userroles',$cdom,$cnum);
    foreach my $user (sort(keys(%courseroles))) {
	if ($user !~ /^(\w{2})/) { next; }
	my ($role) = ($user =~ /^(\w{2})/);
	if ($possible_roles && !(grep(/^$role$/,@$possible_roles))) { next; }
	my ($section,$status);
	if ($role eq 'cr' &&
	    $user =~ m-^$role/[^/]*/[^/]*/[^/]*:[^:]*:[^:]*:(\w+)-) {
	    $section=$1;
	}
	if ($user =~ /^$role:[^:]*:[^:]*:(\w+)/) { $section=$1; }
	if (!defined($section) || $section eq '-1') { next; }
        my ($end,$start) = ($courseroles{$user} =~ /^([^:]*):([^:]*)$/);
        if ($end == -1 && $start == -1) {
            next; #deleted role
        }
        if (!defined($possible_status)) { 
            $sectioncount{$section}++;
        } else {
            if ((!$end || $end >= $now) && (!$start || $start <= $now)) {
                $status = 'active';
            } elsif ($end < $now) {
                $status = 'future';
            } elsif ($start > $now) {
                $status = 'previous';
            }
            if (($status ne '') && (grep/^\Q$status\E$/,@{$possible_status})) {
                $sectioncount{$section}++;
            }
        }
    }
    return %sectioncount;
}

###############################################

=pod

=item * &get_course_users()

Retrieves usernames:domains for users in the specified course
with specific role(s), and access status. 

Incoming parameters:
1. course domain
2. course number
3. access status: users must have - either active, 
previous, future, or all.
4. reference to array of permissible roles
5. reference to array of section restrictions (optional)
6. reference to results object (hash of hashes).
7. reference to optional userdata hash
8. reference to optional statushash
9. flag if privileged users (except those set to unhide in
   course settings) should be excluded    
Keys of top level results hash are roles.
Keys of inner hashes are username:domain, with 
values set to access type.
Optional userdata hash returns an array with arguments in the 
same order as loncoursedata::get_classlist() for student data.

Optional statushash returns

Entries for end, start, section and status are blank because
of the possibility of multiple values for non-student roles.

=cut

###############################################

sub get_course_users {
    my ($cdom,$cnum,$types,$roles,$sections,$users,$userdata,$statushash,$hidepriv) = @_;
    my %idx = ();
    my %seclists;

    $idx{udom} = &Apache::loncoursedata::CL_SDOM();
    $idx{uname} =  &Apache::loncoursedata::CL_SNAME();
    $idx{end} = &Apache::loncoursedata::CL_END();
    $idx{start} = &Apache::loncoursedata::CL_START();
    $idx{id} = &Apache::loncoursedata::CL_ID();
    $idx{section} = &Apache::loncoursedata::CL_SECTION();
    $idx{fullname} = &Apache::loncoursedata::CL_FULLNAME();
    $idx{status} = &Apache::loncoursedata::CL_STATUS();

    if (grep(/^st$/,@{$roles})) {
        my ($classlist,$keylist)=&Apache::loncoursedata::get_classlist($cdom,$cnum);
        my $now = time;
        foreach my $student (keys(%{$classlist})) {
            my $match = 0;
            my $secmatch = 0;
            my $section = $$classlist{$student}[$idx{section}];
            my $status = $$classlist{$student}[$idx{status}];
            if ($section eq '') {
                $section = 'none';
            }
            if ((ref($sections) eq 'ARRAY') && (@{$sections} > 0)) {
                if (grep(/^all$/,@{$sections})) {
                    $secmatch = 1;
                } elsif ($$classlist{$student}[$idx{section}] eq '') {
                    if (grep(/^none$/,@{$sections})) {
                        $secmatch = 1;
                    }
                } else {  
		    if (grep(/^\Q$section\E$/,@{$sections})) {
		        $secmatch = 1;
                    }
		}
                if (!$secmatch) {
                    next;
                }
            }
            if (defined($$types{'active'})) {
                if ($$classlist{$student}[$idx{status}] eq 'Active') {
                    push(@{$$users{st}{$student}},'active');
                    $match = 1;
                }
            }
            if (defined($$types{'previous'})) {
                if ($$classlist{$student}[$idx{status}] eq 'Expired') {
                    push(@{$$users{st}{$student}},'previous');
                    $match = 1;
                }
            }
            if (defined($$types{'future'})) {
                if ($$classlist{$student}[$idx{status}] eq 'Future') {
                    push(@{$$users{st}{$student}},'future');
                    $match = 1;
                }
            }
            if ($match) {
                push(@{$seclists{$student}},$section);
                if (ref($userdata) eq 'HASH') {
                    $$userdata{$student} = $$classlist{$student};
                }
                if (ref($statushash) eq 'HASH') {
                    $statushash->{$student}{'st'}{$section} = $status;
                }
            }
        }
    }
    if ((@{$roles} > 1) || ((@{$roles} == 1) && ($$roles[0] ne "st"))) {
        my %coursepersonnel = &Apache::lonnet::dump('nohist_userroles',$cdom,$cnum);
        my $now = time;
        my %displaystatus = ( previous => 'Expired',
                              active   => 'Active',
                              future   => 'Future',
                            );
        my (%nothide,@possdoms);
        if ($hidepriv) {
            my %coursehash=&Apache::lonnet::coursedescription($cdom.'_'.$cnum);
            foreach my $user (split(/\s*\,\s*/,$coursehash{'nothideprivileged'})) {
                if ($user !~ /:/) {
                    $nothide{join(':',split(/[\@]/,$user))}=1;
                } else {
                    $nothide{$user} = 1;
                }
            }
            my @possdoms = ($cdom);
            if ($coursehash{'checkforpriv'}) {
                push(@possdoms,split(/,/,$coursehash{'checkforpriv'}));
            }
        }
        foreach my $person (sort(keys(%coursepersonnel))) {
            my $match = 0;
            my $secmatch = 0;
            my $status;
            my ($role,$user,$usec) = ($person =~ /^([^:]*):([^:]+:[^:]+):([^:]*)/);
            $user =~ s/:$//;
            my ($end,$start) = split(/:/,$coursepersonnel{$person});
            if ($end == -1 || $start == -1) {
                next;
            }
            if (($role) && ((grep(/^\Q$role\E$/,@{$roles})) ||
                (grep(/^cr$/,@{$roles}) && $role =~ /^cr\//))) {
                my ($uname,$udom) = split(/:/,$user);
                if ((ref($sections) eq 'ARRAY') && (@{$sections} > 0)) {
                    if (grep(/^all$/,@{$sections})) {
                        $secmatch = 1;
                    } elsif ($usec eq '') {
                        if (grep(/^none$/,@{$sections})) {
                            $secmatch = 1;
                        }
                    } else {
                        if (grep(/^\Q$usec\E$/,@{$sections})) {
                            $secmatch = 1;
                        }
                    }
                    if (!$secmatch) {
                        next;
                    }
                }
                if ($usec eq '') {
                    $usec = 'none';
                }
                if ($uname ne '' && $udom ne '') {
                    if ($hidepriv) {
                        if ((&Apache::lonnet::privileged($uname,$udom,\@possdoms)) &&
                            (!$nothide{$uname.':'.$udom})) {
                            next;
                        }
                    }
                    if ($end > 0 && $end < $now) {
                        $status = 'previous';
                    } elsif ($start > $now) {
                        $status = 'future';
                    } else {
                        $status = 'active';
                    }
                    foreach my $type (keys(%{$types})) { 
                        if ($status eq $type) {
                            if (!grep(/^\Q$type\E$/,@{$$users{$role}{$user}})) {
                                push(@{$$users{$role}{$user}},$type);
                            }
                            $match = 1;
                        }
                    }
                    if (($match) && (ref($userdata) eq 'HASH')) {
                        if (!exists($$userdata{$uname.':'.$udom})) {
			    &get_user_info($udom,$uname,\%idx,$userdata);
                        }
                        if (!grep(/^\Q$usec\E$/,@{$seclists{$uname.':'.$udom}})) {
                            push(@{$seclists{$uname.':'.$udom}},$usec);
                        }
                        if (ref($statushash) eq 'HASH') {
                            $statushash->{$uname.':'.$udom}{$role}{$usec} = $displaystatus{$status};
                        }
                    }
                }
            }
        }
        if (grep(/^ow$/,@{$roles})) {
            if ((defined($cdom)) && (defined($cnum))) {
                my %csettings = &Apache::lonnet::get('environment',['internal.courseowner'],$cdom,$cnum);
                if ( defined($csettings{'internal.courseowner'}) ) {
                    my $owner = $csettings{'internal.courseowner'};
                    next if ($owner eq '');
                    my ($ownername,$ownerdom);
                    if ($owner =~ /^([^:]+):([^:]+)$/) {
                        $ownername = $1;
                        $ownerdom = $2;
                    } else {
                        $ownername = $owner;
                        $ownerdom = $cdom;
                        $owner = $ownername.':'.$ownerdom;
                    }
                    @{$$users{'ow'}{$owner}} = 'any';
                    if (defined($userdata) && 
			!exists($$userdata{$owner})) {
			&get_user_info($ownerdom,$ownername,\%idx,$userdata);
                        if (!grep(/^none$/,@{$seclists{$owner}})) {
                            push(@{$seclists{$owner}},'none');
                        }
                        if (ref($statushash) eq 'HASH') {
                            $statushash->{$owner}{'ow'}{'none'} = 'Any';
                        }
		    }
                }
            }
        }
        foreach my $user (keys(%seclists)) {
            @{$seclists{$user}} = (sort {$a <=> $b} @{$seclists{$user}});
            $$userdata{$user}[$idx{section}] = join(',',@{$seclists{$user}});
        }
    }
    return;
}

sub get_user_info {
    my ($udom,$uname,$idx,$userdata) = @_;
    $$userdata{$uname.':'.$udom}[$$idx{fullname}] = 
	&plainname($uname,$udom,'lastname');
    $$userdata{$uname.':'.$udom}[$$idx{uname}] = $uname;
    $$userdata{$uname.':'.$udom}[$$idx{udom}] = $udom;
    my %idhash =  &Apache::lonnet::idrget($udom,($uname));
    $$userdata{$uname.':'.$udom}[$$idx{id}] = $idhash{$uname}; 
    return;
}

###############################################

=pod

=item * &get_user_quota()

Retrieves quota assigned for storage of user files.
Default is to report quota for portfolio files.

Incoming parameters:
1. user's username
2. user's domain
3. quota name - portfolio, author, or course
   (if no quota name provided, defaults to portfolio).
4. crstype - official, unofficial, textbook or community, if quota name is
   course

Returns:
1. Disk quota (in MB) assigned to student.
2. (Optional) Type of setting: custom or default
   (individually assigned or default for user's 
   institutional status).
3. (Optional) - User's institutional status (e.g., faculty, staff
   or student - types as defined in localenroll::inst_usertypes 
   for user's domain, which determines default quota for user.
4. (Optional) - Default quota which would apply to the user.

If a value has been stored in the user's environment, 
it will return that, otherwise it returns the maximal default
defined for the user's institutional status(es) in the domain.

=cut

###############################################


sub get_user_quota {
    my ($uname,$udom,$quotaname,$crstype) = @_;
    my ($quota,$quotatype,$settingstatus,$defquota);
    if (!defined($udom)) {
        $udom = $env{'user.domain'};
    }
    if (!defined($uname)) {
        $uname = $env{'user.name'};
    }
    if (($udom eq '' || $uname eq '') ||
        ($udom eq 'public') && ($uname eq 'public')) {
        $quota = 0;
        $quotatype = 'default';
        $defquota = 0; 
    } else {
        my $inststatus;
        if ($quotaname eq 'course') {
            if (($env{'course.'.$udom.'_'.$uname.'.num'} eq $uname) &&
                ($env{'course.'.$udom.'_'.$uname.'.domain'} eq $udom)) {
                $quota = $env{'course.'.$udom.'_'.$uname.'.internal.uploadquota'};
            } else {
                my %cenv = &Apache::lonnet::coursedescription("$udom/$uname");
                $quota = $cenv{'internal.uploadquota'};
            }
        } else {
            if ($udom eq $env{'user.domain'} && $uname eq $env{'user.name'}) {
                if ($quotaname eq 'author') {
                    $quota = $env{'environment.authorquota'};
                } else {
                    $quota = $env{'environment.portfolioquota'};
                }
                $inststatus = $env{'environment.inststatus'};
            } else {
                my %userenv = 
                    &Apache::lonnet::get('environment',['portfolioquota',
                                         'authorquota','inststatus'],$udom,$uname);
                my ($tmp) = keys(%userenv);
                if ($tmp !~ /^(con_lost|error|no_such_host)/i) {
                    if ($quotaname eq 'author') {
                        $quota = $userenv{'authorquota'};
                    } else {
                        $quota = $userenv{'portfolioquota'};
                    }
                    $inststatus = $userenv{'inststatus'};
                } else {
                    undef(%userenv);
                }
            }
        }
        if ($quota eq '' || wantarray) {
            if ($quotaname eq 'course') {
                my %domdefs = &Apache::lonnet::get_domain_defaults($udom);
                if (($crstype eq 'official') || ($crstype eq 'unofficial') ||
                    ($crstype eq 'community') || ($crstype eq 'textbook')) {
                    $defquota = $domdefs{$crstype.'quota'};
                }
                if ($defquota eq '') {
                    $defquota = 500;
                }
            } else {
                ($defquota,$settingstatus) = &default_quota($udom,$inststatus,$quotaname);
            }
            if ($quota eq '') {
                $quota = $defquota;
                $quotatype = 'default';
            } else {
                $quotatype = 'custom';
            }
        }
    }
    if (wantarray) {
        return ($quota,$quotatype,$settingstatus,$defquota);
    } else {
        return $quota;
    }
}

###############################################

=pod

=item * &default_quota()

Retrieves default quota assigned for storage of user portfolio files,
given an (optional) user's institutional status.

Incoming parameters:

1. domain
2. (Optional) institutional status(es).  This is a : separated list of 
   status types (e.g., faculty, staff, student etc.)
   which apply to the user for whom the default is being retrieved.
   If the institutional status string in undefined, the domain
   default quota will be returned.
3.  quota name - portfolio, author, or course
   (if no quota name provided, defaults to portfolio).

Returns:

1. Default disk quota (in MB) for user portfolios in the domain.
2. (Optional) institutional type which determined the value of the
   default quota.

If a value has been stored in the domain's configuration db,
it will return that, otherwise it returns 20 (for backwards 
compatibility with domains which have not set up a configuration
db file; the original statically defined portfolio quota was 20 MB). 

If the user's status includes multiple types (e.g., staff and student),
the largest default quota which applies to the user determines the
default quota returned.

=cut

###############################################


sub default_quota {
    my ($udom,$inststatus,$quotaname) = @_;
    my ($defquota,$settingstatus);
    my %quotahash = &Apache::lonnet::get_dom('configuration',
                                            ['quotas'],$udom);
    my $key = 'defaultquota';
    if ($quotaname eq 'author') {
        $key = 'authorquota';
    }
    if (ref($quotahash{'quotas'}) eq 'HASH') {
        if ($inststatus ne '') {
            my @statuses = map { &unescape($_); } split(/:/,$inststatus);
            foreach my $item (@statuses) {
                if (ref($quotahash{'quotas'}{$key}) eq 'HASH') {
                    if ($quotahash{'quotas'}{$key}{$item} ne '') {
                        if ($defquota eq '') {
                            $defquota = $quotahash{'quotas'}{$key}{$item};
                            $settingstatus = $item;
                        } elsif ($quotahash{'quotas'}{$key}{$item} > $defquota) {
                            $defquota = $quotahash{'quotas'}{$key}{$item};
                            $settingstatus = $item;
                        }
                    }
                } elsif ($key eq 'defaultquota') {
                    if ($quotahash{'quotas'}{$item} ne '') {
                        if ($defquota eq '') {
                            $defquota = $quotahash{'quotas'}{$item};
                            $settingstatus = $item;
                        } elsif ($quotahash{'quotas'}{$item} > $defquota) {
                            $defquota = $quotahash{'quotas'}{$item};
                            $settingstatus = $item;
                        }
                    }
                }
            }
        }
        if ($defquota eq '') {
            if (ref($quotahash{'quotas'}{$key}) eq 'HASH') {
                $defquota = $quotahash{'quotas'}{$key}{'default'};
            } elsif ($key eq 'defaultquota') {
                $defquota = $quotahash{'quotas'}{'default'};
            }
            $settingstatus = 'default';
            if ($defquota eq '') {
                if ($quotaname eq 'author') {
                    $defquota = 500;
                }
            }
        }
    } else {
        $settingstatus = 'default';
        if ($quotaname eq 'author') {
            $defquota = 500;
        } else {
            $defquota = 20;
        }
    }
    if (wantarray) {
        return ($defquota,$settingstatus);
    } else {
        return $defquota;
    }
}

###############################################

=pod

=item * &excess_filesize_warning()

Returns warning message if upload of file to authoring space, or copying
of existing file within authoring space will cause quota for the authoring
space to be exceeded.

Same, if upload of a file directly to a course/community via Course Editor
will cause quota for uploaded content for the course to be exceeded.

Inputs: 7 
1. username or coursenum
2. domain
3. context ('author' or 'course')
4. filename of file for which action is being requested
5. filesize (kB) of file
6. action being taken: copy or upload.
7. quotatype (in course context -- official, unofficial, community or textbook).

Returns: 1 scalar: HTML to display containing warning if quota would be exceeded,
         otherwise return null.

=back

=cut

sub excess_filesize_warning {
    my ($uname,$udom,$context,$filename,$filesize,$action,$quotatype) = @_;
    my $current_disk_usage = 0;
    my $disk_quota = &get_user_quota($uname,$udom,$context,$quotatype); #expressed in MB
    if ($context eq 'author') {
        my $authorspace = $Apache::lonnet::perlvar{'lonDocRoot'}."/priv/$udom/$uname";
        $current_disk_usage = &Apache::lonnet::diskusage($udom,$uname,$authorspace);
    } else {
        foreach my $subdir ('docs','supplemental') {
            $current_disk_usage += &Apache::lonnet::diskusage($udom,$uname,"userfiles/$subdir",1);
        }
    }
    $disk_quota = int($disk_quota * 1000);
    if (($current_disk_usage + $filesize) > $disk_quota) {
        return '<p class="LC_warning">'.
                &mt("Unable to $action [_1]. (size = [_2] kilobytes). Disk quota will be exceeded.",
                    '<span class="LC_filename">'.$filename.'</span>',$filesize).'</p>'.
               '<p>'.&mt('Disk quota is [_1] kilobytes. Your current disk usage is [_2] kilobytes.',
                            $disk_quota,$current_disk_usage).
               '</p>';
    }
    return;
}

###############################################


sub get_secgrprole_info {
    my ($cdom,$cnum,$needroles,$type)  = @_;
    my %sections_count = &get_sections($cdom,$cnum);
    my @sections =  (sort {$a <=> $b} keys(%sections_count));
    my %curr_groups = &Apache::longroup::coursegroups($cdom,$cnum);
    my @groups = sort(keys(%curr_groups));
    my $allroles = [];
    my $rolehash;
    my $accesshash = {
                     active => 'Currently has access',
                     future => 'Will have future access',
                     previous => 'Previously had access',
                  };
    if ($needroles) {
        $rolehash = {'all' => 'all'};
        my %user_roles = &Apache::lonnet::dump('nohist_userroles',$cdom,$cnum);
	if (&Apache::lonnet::error(%user_roles)) {
	    undef(%user_roles);
	}
        foreach my $item (keys(%user_roles)) {
            my ($role)=split(/\:/,$item,2);
            if ($role eq 'cr') { next; }
            if ($role =~ /^cr/) {
                $$rolehash{$role} = (split('/',$role))[3];
            } else {
                $$rolehash{$role} = &Apache::lonnet::plaintext($role,$type);
            }
        }
        foreach my $key (sort(keys(%{$rolehash}))) {
            push(@{$allroles},$key);
        }
        push (@{$allroles},'st');
        $$rolehash{'st'} = &Apache::lonnet::plaintext('st',$type);
    }
    return (\@sections,\@groups,$allroles,$rolehash,$accesshash);
}

sub user_picker {
    my ($dom,$srch,$forcenewuser,$caller,$cancreate,$usertype,$context,$fixeddom,$noinstd) = @_;
    my $currdom = $dom;
    my @alldoms = &Apache::lonnet::all_domains();
    if (@alldoms == 1) {
        my %domsrch = &Apache::lonnet::get_dom('configuration',
                                               ['directorysrch'],$alldoms[0]);
        my $domdesc = &Apache::lonnet::domain($alldoms[0],'description');
        my $showdom = $domdesc;
        if ($showdom eq '') {
            $showdom = $dom;
        }
        if (ref($domsrch{'directorysrch'}) eq 'HASH') {
            if ((!$domsrch{'directorysrch'}{'available'}) &&
                ($domsrch{'directorysrch'}{'lcavailable'} eq '0')) {
                return (&mt('LON-CAPA directory search is not available in domain: [_1]',$showdom),0);
            }
        }
    }
    my %curr_selected = (
                        srchin => 'dom',
                        srchby => 'lastname',
                      );
    my $srchterm;
    if ((ref($srch) eq 'HASH') && ($env{'form.origform'} ne 'crtusername')) {
        if ($srch->{'srchby'} ne '') {
            $curr_selected{'srchby'} = $srch->{'srchby'};
        }
        if ($srch->{'srchin'} ne '') {
            $curr_selected{'srchin'} = $srch->{'srchin'};
        }
        if ($srch->{'srchtype'} ne '') {
            $curr_selected{'srchtype'} = $srch->{'srchtype'};
        }
        if ($srch->{'srchdomain'} ne '') {
            $currdom = $srch->{'srchdomain'};
        }
        $srchterm = $srch->{'srchterm'};
    }
    my %html_lt=&Apache::lonlocal::texthash(
                    'usr'       => 'Search criteria',
                    'doma'      => 'Domain/institution to search',
                    'uname'     => 'username',
                    'lastname'  => 'last name',
                    'lastfirst' => 'last name, first name',
                    'crs'       => 'in this course',
                    'dom'       => 'in selected LON-CAPA domain', 
                    'alc'       => 'all LON-CAPA',
                    'instd'     => 'in institutional directory for selected domain',
                    'exact'     => 'is',
                    'contains'  => 'contains',
                    'begins'    => 'begins with',
                                       );
    my %js_lt=&Apache::lonlocal::texthash(
                    'youm'      => "You must include some text to search for.",
                    'thte'      => "The text you are searching for must contain at least two characters when using a 'begins' type search.",
                    'thet'      => "The text you are searching for must contain at least three characters when using a 'contains' type search.",
                    'yomc'      => "You must choose a domain when using an institutional directory search.",
                    'ymcd'      => "You must choose a domain when using a domain search.",
                    'whus'      => "When using searching by last,first you must include a comma as separator between last name and first name.",
                    'whse'      => "When searching by last,first you must include at least one character in the first name.",
                     'thfo'     => "The following need to be corrected before the search can be run:",
                                       );
    &html_escape(\%html_lt);
    &js_escape(\%js_lt);
    my $domform;
    my $allow_blank = 1;
    if ($fixeddom) {
        $allow_blank = 0;
        $domform = &select_dom_form($currdom,'srchdomain',$allow_blank,1,undef,[$currdom]);
    } else {
        $domform = &select_dom_form($currdom,'srchdomain',$allow_blank,1);
    }
    my $srchinsel = ' <select name="srchin">';

    my @srchins = ('crs','dom','alc','instd');

    foreach my $option (@srchins) {
        # FIXME 'alc' option unavailable until 
        #       loncreateuser::print_user_query_page()
        #       has been completed.
        next if ($option eq 'alc');
        next if (($option eq 'crs') && ($env{'form.form'} eq 'requestcrs'));  
        next if ($option eq 'crs' && !$env{'request.course.id'});
        next if (($option eq 'instd') && ($noinstd));
        if ($curr_selected{'srchin'} eq $option) {
            $srchinsel .= ' 
   <option value="'.$option.'" selected="selected">'.$html_lt{$option}.'</option>';
        } else {
            $srchinsel .= '
   <option value="'.$option.'">'.$html_lt{$option}.'</option>';
        }
    }
    $srchinsel .= "\n  </select>\n";

    my $srchbysel =  ' <select name="srchby">';
    foreach my $option ('lastname','lastfirst','uname') {
        if ($curr_selected{'srchby'} eq $option) {
            $srchbysel .= '
   <option value="'.$option.'" selected="selected">'.$html_lt{$option}.'</option>';
        } else {
            $srchbysel .= '
   <option value="'.$option.'">'.$html_lt{$option}.'</option>';
         }
    }
    $srchbysel .= "\n  </select>\n";

    my $srchtypesel = ' <select name="srchtype">';
    foreach my $option ('begins','contains','exact') {
        if ($curr_selected{'srchtype'} eq $option) {
            $srchtypesel .= '
   <option value="'.$option.'" selected="selected">'.$html_lt{$option}.'</option>';
        } else {
            $srchtypesel .= '
   <option value="'.$option.'">'.$html_lt{$option}.'</option>';
        }
    }
    $srchtypesel .= "\n  </select>\n";

    my ($newuserscript,$new_user_create);
    my $context_dom = $env{'request.role.domain'};
    if ($context eq 'requestcrs') {
        if ($env{'form.coursedom'} ne '') { 
            $context_dom = $env{'form.coursedom'};
        }
    }
    if ($forcenewuser) {
        if (ref($srch) eq 'HASH') {
            if ($srch->{'srchby'} eq 'uname' && $srch->{'srchtype'} eq 'exact' && $srch->{'srchin'} eq 'dom' && $srch->{'srchdomain'} eq $context_dom) {
                if ($cancreate) {
                    $new_user_create = '<p> <input type="submit" name="forcenew" value="'.&HTML::Entities::encode(&mt('Make new user "[_1]"',$srchterm),'<>&"').'" onclick="javascript:setSearch(\'1\','.$caller.');" /> </p>';
                } else {
                    my $helplink = 'javascript:helpMenu('."'display'".')';
                    my %usertypetext = (
                        official   => 'institutional',
                        unofficial => 'non-institutional',
                    );
                    $new_user_create = '<p class="LC_warning">'
                                      .&mt("You are not authorized to create new $usertypetext{$usertype} users in this domain.")
                                      .' '
                                      .&mt('Please contact the [_1]helpdesk[_2] for assistance.'
                                          ,'<a href="'.$helplink.'">','</a>')
                                      .'</p><br />';
                }
            }
        }

        $newuserscript = <<"ENDSCRIPT";

function setSearch(createnew,callingForm) {
    if (createnew == 1) {
        for (var i=0; i<callingForm.srchby.length; i++) {
            if (callingForm.srchby.options[i].value == 'uname') {
                callingForm.srchby.selectedIndex = i;
            }
        }
        for (var i=0; i<callingForm.srchin.length; i++) {
            if ( callingForm.srchin.options[i].value == 'dom') {
		callingForm.srchin.selectedIndex = i;
            }
        }
        for (var i=0; i<callingForm.srchtype.length; i++) {
            if (callingForm.srchtype.options[i].value == 'exact') {
                callingForm.srchtype.selectedIndex = i;
            }
        }
        for (var i=0; i<callingForm.srchdomain.length; i++) {
            if (callingForm.srchdomain.options[i].value == '$context_dom') {
                callingForm.srchdomain.selectedIndex = i;
            }
        }
    }
}
ENDSCRIPT

    }

    my $output = <<"END_BLOCK";
<script type="text/javascript">
// <![CDATA[
function validateEntry(callingForm) {

    var checkok = 1;
    var srchin;
    for (var i=0; i<callingForm.srchin.length; i++) {
	if ( callingForm.srchin[i].checked ) {
	    srchin = callingForm.srchin[i].value;
	}
    }

    var srchtype = callingForm.srchtype.options[callingForm.srchtype.selectedIndex].value;
    var srchby = callingForm.srchby.options[callingForm.srchby.selectedIndex].value;
    var srchdomain = callingForm.srchdomain.options[callingForm.srchdomain.selectedIndex].value;
    var srchterm =  callingForm.srchterm.value;
    var srchin = callingForm.srchin.options[callingForm.srchin.selectedIndex].value;
    var msg = "";

    if (srchterm == "") {
        checkok = 0;
        msg += "$js_lt{'youm'}\\n";
    }

    if (srchtype== 'begins') {
        if (srchterm.length < 2) {
            checkok = 0;
            msg += "$js_lt{'thte'}\\n";
        }
    }

    if (srchtype== 'contains') {
        if (srchterm.length < 3) {
            checkok = 0;
            msg += "$js_lt{'thet'}\\n";
        }
    }
    if (srchin == 'instd') {
        if (srchdomain == '') {
            checkok = 0;
            msg += "$js_lt{'yomc'}\\n";
        }
    }
    if (srchin == 'dom') {
        if (srchdomain == '') {
            checkok = 0;
            msg += "$js_lt{'ymcd'}\\n";
        }
    }
    if (srchby == 'lastfirst') {
        if (srchterm.indexOf(",") == -1) {
            checkok = 0;
            msg += "$js_lt{'whus'}\\n";
        }
        if (srchterm.indexOf(",") == srchterm.length -1) {
            checkok = 0;
            msg += "$js_lt{'whse'}\\n";
        }
    }
    if (checkok == 0) {
        alert("$js_lt{'thfo'}\\n"+msg);
        return;
    }
    if (checkok == 1) {
        callingForm.submit();
    }
}

$newuserscript

// ]]>
</script>

$new_user_create

END_BLOCK

    $output .= &Apache::lonhtmlcommon::start_pick_box().
               &Apache::lonhtmlcommon::row_title($html_lt{'doma'}).
               $domform.
               &Apache::lonhtmlcommon::row_closure().
               &Apache::lonhtmlcommon::row_title($html_lt{'usr'}).
               $srchbysel.
               $srchtypesel. 
               '<input type="text" size="15" name="srchterm" value="'.$srchterm.'" />'.
               $srchinsel.
               &Apache::lonhtmlcommon::row_closure(1). 
               &Apache::lonhtmlcommon::end_pick_box().
               '<br />';
    return ($output,1);
}

sub user_rule_check {
    my ($usershash,$checks,$alerts,$rulematch,$inst_results,$curr_rules,$got_rules) = @_;
    my ($response,%inst_response);
    if (ref($usershash) eq 'HASH') {
        if (keys(%{$usershash}) > 1) {
            my (%by_username,%by_id,%userdoms);
            my $checkid;
            if (ref($checks) eq 'HASH') {
                if ((!defined($checks->{'username'})) && (defined($checks->{'id'}))) {
                    $checkid = 1;
                }
            }
            foreach my $user (keys(%{$usershash})) {
                my ($uname,$udom) = split(/:/,$user);
                if ($checkid) {
                    if (ref($usershash->{$user}) eq 'HASH') {
                        if ($usershash->{$user}->{'id'} ne '') {
                            $by_id{$udom}{$usershash->{$user}->{'id'}} = $uname;
                            $userdoms{$udom} = 1;
                            if (ref($inst_results) eq 'HASH') {
                                $inst_results->{$uname.':'.$udom} = {};
                            }
                        }
                    }
                } else {
                    $by_username{$udom}{$uname} = 1;
                    $userdoms{$udom} = 1;
                    if (ref($inst_results) eq 'HASH') {
                        $inst_results->{$uname.':'.$udom} = {};
                    }
                }
            }
            foreach my $udom (keys(%userdoms)) {
                if (!$got_rules->{$udom}) {
                    my %domconfig = &Apache::lonnet::get_dom('configuration',
                                                             ['usercreation'],$udom);
                    if (ref($domconfig{'usercreation'}) eq 'HASH') {
                        foreach my $item ('username','id') {
                            if (ref($domconfig{'usercreation'}{$item.'_rule'}) eq 'ARRAY') {
                                $$curr_rules{$udom}{$item} =
                                    $domconfig{'usercreation'}{$item.'_rule'};
                            }
                        }
                    }
                    $got_rules->{$udom} = 1;
                }
            }
            if ($checkid) {
                foreach my $udom (keys(%by_id)) {
                    my ($outcome,$results) = &Apache::lonnet::get_multiple_instusers($udom,$by_id{$udom},'id');
                    if ($outcome eq 'ok') {
                        foreach my $id (keys(%{$by_id{$udom}})) {
                            my $uname = $by_id{$udom}{$id};
                            $inst_response{$uname.':'.$udom} = $outcome;
                        }
                        if (ref($results) eq 'HASH') {
                            foreach my $uname (keys(%{$results})) {
                                if (exists($inst_response{$uname.':'.$udom})) {
                                    $inst_response{$uname.':'.$udom} = $outcome;
                                    $inst_results->{$uname.':'.$udom} = $results->{$uname};
                                }
                            }
                        }
                    }
                }
            } else {
                foreach my $udom (keys(%by_username)) {
                    my ($outcome,$results) = &Apache::lonnet::get_multiple_instusers($udom,$by_username{$udom});
                    if ($outcome eq 'ok') {
                        foreach my $uname (keys(%{$by_username{$udom}})) {
                            $inst_response{$uname.':'.$udom} = $outcome;
                        }
                        if (ref($results) eq 'HASH') {
                            foreach my $uname (keys(%{$results})) {
                                $inst_results->{$uname.':'.$udom} = $results->{$uname};
                            }
                        }
                    }
                }
            }
        } elsif (keys(%{$usershash}) == 1) {
            my $user = (keys(%{$usershash}))[0];
            my ($uname,$udom) = split(/:/,$user);
            if (($udom ne '') && ($uname ne '')) {
                if (ref($usershash->{$user}) eq 'HASH') {
                    if (ref($checks) eq 'HASH') {
                        if (defined($checks->{'username'})) {
                            ($inst_response{$user},%{$inst_results->{$user}}) =
                                &Apache::lonnet::get_instuser($udom,$uname);
                        } elsif (defined($checks->{'id'})) {
                            if ($usershash->{$user}->{'id'} ne '') {
                                ($inst_response{$user},%{$inst_results->{$user}}) =
                                    &Apache::lonnet::get_instuser($udom,undef,
                                                                  $usershash->{$user}->{'id'});
                            } else {
                                ($inst_response{$user},%{$inst_results->{$user}}) =
                                    &Apache::lonnet::get_instuser($udom,$uname);
                            }
                        }
                    } else {
                       ($inst_response{$user},%{$inst_results->{$user}}) =
                            &Apache::lonnet::get_instuser($udom,$uname);
                       return;
                    }
                    if (!$got_rules->{$udom}) {
                        my %domconfig = &Apache::lonnet::get_dom('configuration',
                                                                 ['usercreation'],$udom);
                        if (ref($domconfig{'usercreation'}) eq 'HASH') {
                            foreach my $item ('username','id') {
                                if (ref($domconfig{'usercreation'}{$item.'_rule'}) eq 'ARRAY') {
                                   $$curr_rules{$udom}{$item} =
                                       $domconfig{'usercreation'}{$item.'_rule'};
                                }
                            }
                        }
                        $got_rules->{$udom} = 1;
                    }
                }
            } else {
                return;
            }
        } else {
            return;
        }
        foreach my $user (keys(%{$usershash})) {
            my ($uname,$udom) = split(/:/,$user);
            next if (($udom eq '') || ($uname eq ''));
            my $id;
            if (ref($inst_results) eq 'HASH') {
                if (ref($inst_results->{$user}) eq 'HASH') {
                    $id = $inst_results->{$user}->{'id'};
                }
            }
            if ($id eq '') {
                if (ref($usershash->{$user})) {
                    $id = $usershash->{$user}->{'id'};
                }
            }
            foreach my $item (keys(%{$checks})) {
                if (ref($$curr_rules{$udom}) eq 'HASH') {
                    if (ref($$curr_rules{$udom}{$item}) eq 'ARRAY') {
                        if (@{$$curr_rules{$udom}{$item}} > 0) {
                            my %rule_check = &Apache::lonnet::inst_rulecheck($udom,$uname,$id,$item,
                                                                             $$curr_rules{$udom}{$item});
                            foreach my $rule (@{$$curr_rules{$udom}{$item}}) {
                                if ($rule_check{$rule}) {
                                    $$rulematch{$user}{$item} = $rule;
                                    if ($inst_response{$user} eq 'ok') {
                                        if (ref($inst_results) eq 'HASH') {
                                            if (ref($inst_results->{$user}) eq 'HASH') {
                                                if (keys(%{$inst_results->{$user}}) == 0) {
                                                    $$alerts{$item}{$udom}{$uname} = 1;
                                                } elsif ($item eq 'id') {
                                                    if ($inst_results->{$user}->{'id'} eq '') {
                                                        $$alerts{$item}{$udom}{$uname} = 1;
                                                    }
                                                }
                                            }
                                        }
                                    }
                                    last;
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    return;
}

sub user_rule_formats {
    my ($domain,$domdesc,$curr_rules,$check) = @_;
    my %text = ( 
                 'username' => 'Usernames',
                 'id'       => 'IDs',
               );
    my $output;
    my ($rules,$ruleorder) = &Apache::lonnet::inst_userrules($domain,$check);
    if ((ref($rules) eq 'HASH') && (ref($ruleorder) eq 'ARRAY')) {
        if (@{$ruleorder} > 0) {
            $output = '<br />'.
                      &mt($text{$check}.' with the following format(s) may [_1]only[_2] be used for verified users at [_3]:',
                          '<span class="LC_cusr_emph">','</span>',$domdesc).
                      ' <ul>';
            foreach my $rule (@{$ruleorder}) {
                if (ref($curr_rules) eq 'ARRAY') {
                    if (grep(/^\Q$rule\E$/,@{$curr_rules})) {
                        if (ref($rules->{$rule}) eq 'HASH') {
                            $output .= '<li>'.$rules->{$rule}{'name'}.': '.
                                        $rules->{$rule}{'desc'}.'</li>';
                        }
                    }
                }
            }
            $output .= '</ul>';
        }
    }
    return $output;
}

sub instrule_disallow_msg {
    my ($checkitem,$domdesc,$count,$mode) = @_;
    my $response;
    my %text = (
                  item   => 'username',
                  items  => 'usernames',
                  match  => 'matches',
                  do     => 'does',
                  action => 'a username',
                  one    => 'one',
               );
    if ($count > 1) {
        $text{'item'} = 'usernames';
        $text{'match'} ='match';
        $text{'do'} = 'do';
        $text{'action'} = 'usernames',
        $text{'one'} = 'ones';
    }
    if ($checkitem eq 'id') {
        $text{'items'} = 'IDs';
        $text{'item'} = 'ID';
        $text{'action'} = 'an ID';
        if ($count > 1) {
            $text{'item'} = 'IDs';
            $text{'action'} = 'IDs';
        }
    }
    $response = &mt("The $text{'item'} you chose $text{'match'} the format of $text{'items'} defined for [_1], but the $text{'item'} $text{'do'} not exist in the institutional directory.",'<span class="LC_cusr_emph">'.$domdesc.'</span>').'<br />';
    if ($mode eq 'upload') {
        if ($checkitem eq 'username') {
            $response .= &mt("You will need to modify your upload file so it will include $text{'action'} with a different format --  $text{'one'} that will not conflict with 'official' institutional $text{'items'}.");
        } elsif ($checkitem eq 'id') {
            $response .= &mt("Either upload a file which includes $text{'action'} with a different format --  $text{'one'} that will not conflict with 'official' institutional $text{'items'}, or when associating fields with data columns, omit an association for the Student/Employee ID field.");
        }
    } elsif ($mode eq 'selfcreate') {
        if ($checkitem eq 'id') {
            $response .= &mt("You must either choose $text{'action'} with a different format --  $text{'one'} that will not conflict with 'official' institutional $text{'items'}, or leave the ID field blank.");
        }
    } else {
        if ($checkitem eq 'username') {
            $response .= &mt("You must choose $text{'action'} with a different format --  $text{'one'} that will not conflict with 'official' institutional $text{'items'}.");
        } elsif ($checkitem eq 'id') {
            $response .= &mt("You must either choose $text{'action'} with a different format --  $text{'one'} that will not conflict with 'official' institutional $text{'items'}, or leave the ID field blank.");
        }
    }
    return $response;
}

sub personal_data_fieldtitles {
    my %fieldtitles = &Apache::lonlocal::texthash (
                        id => 'Student/Employee ID',
                        permanentemail => 'E-mail address',
                        lastname => 'Last Name',
                        firstname => 'First Name',
                        middlename => 'Middle Name',
                        generation => 'Generation',
                        gen => 'Generation',
                        inststatus => 'Affiliation',
                   );
    return %fieldtitles;
}

sub sorted_inst_types {
    my ($dom) = @_;
    my ($usertypes,$order);
    my %domdefaults = &Apache::lonnet::get_domain_defaults($dom);
    if (ref($domdefaults{'inststatus'}) eq 'HASH') {
        $usertypes = $domdefaults{'inststatus'}{'inststatustypes'};
        $order = $domdefaults{'inststatus'}{'inststatusorder'};
    } else {
        ($usertypes,$order) = &Apache::lonnet::retrieve_inst_usertypes($dom);
    }
    my $othertitle = &mt('All users');
    if ($env{'request.course.id'}) {
        $othertitle  = &mt('Any users');
    }
    my @types;
    if (ref($order) eq 'ARRAY') {
        @types = @{$order};
    }
    if (@types == 0) {
        if (ref($usertypes) eq 'HASH') {
            @types = sort(keys(%{$usertypes}));
        }
    }
    if (keys(%{$usertypes}) > 0) {
        $othertitle = &mt('Other users');
    }
    return ($othertitle,$usertypes,\@types);
}

sub get_institutional_codes {
    my ($cdom,$crs,$settings,$allcourses,$LC_code) = @_;
# Get complete list of course sections to update
    my @currsections = ();
    my @currxlists = ();
    my (%unclutteredsec,%unclutteredlcsec);
    my $coursecode = $$settings{'internal.coursecode'};
    my $crskey = $crs.':'.$coursecode;
    @{$unclutteredsec{$crskey}} = ();
    @{$unclutteredlcsec{$crskey}} = ();

    if ($$settings{'internal.sectionnums'} ne '') {
        @currsections = split(/,/,$$settings{'internal.sectionnums'});
    }

    if ($$settings{'internal.crosslistings'} ne '') {
        @currxlists = split(/,/,$$settings{'internal.crosslistings'});
    }

    if (@currxlists > 0) {
        foreach my $xl (@currxlists) {
            if ($xl =~ /^([^:]+):(\w*)$/) {
                unless (grep/^$1$/,@{$allcourses}) {
                    push(@{$allcourses},$1);
                    $$LC_code{$1} = $2;
                }
            }
        }
    }

    if (@currsections > 0) {
        foreach my $sec (@currsections) {
            if ($sec =~ m/^(\w+):(\w*)$/ ) {
                my $instsec = $1;
                my $lc_sec = $2;
                unless (grep/^\Q$instsec\E$/,@{$unclutteredsec{$crskey}}) {
                    push(@{$unclutteredsec{$crskey}},$instsec);
                    push(@{$unclutteredlcsec{$crskey}},$lc_sec);
                }
            }
        }
    }

    if (@{$unclutteredsec{$crskey}} > 0) {
        my %formattedsec = &Apache::lonnet::auto_instsec_reformat($cdom,'clutter',\%unclutteredsec);
        if ((ref($formattedsec{$crskey}) eq 'ARRAY') && (ref($unclutteredlcsec{$crskey}) eq 'ARRAY')) {
            for (my $i=0; $i<@{$formattedsec{$crskey}}; $i++) {
                my $sec = $coursecode.$formattedsec{$crskey}[$i];
                unless (grep/^\Q$sec\E$/,@{$allcourses}) {
                    push(@{$allcourses},$sec);
                    $$LC_code{$sec} = $unclutteredlcsec{$crskey}[$i];
                }
            }
        }
    }
    return;
}

sub get_standard_codeitems {
    return ('Year','Semester','Department','Number','Section');
}

=pod

=head1 Slot Helpers

=over 4

=item * sorted_slots()

Sorts an array of slot names in order of an optional sort key,
default sort is by slot start time (earliest first). 

Inputs:

=over 4

slotsarr  - Reference to array of unsorted slot names.

slots     - Reference to hash of hash, where outer hash keys are slot names.

sortkey   - Name of key in inner hash to be sorted on (e.g., starttime).

=back

Returns:

=over 4

sorted   - An array of slot names sorted by a specified sort key 
           (default sort key is start time of the slot).

=back

=cut


sub sorted_slots {
    my ($slotsarr,$slots,$sortkey) = @_;
    if ($sortkey eq '') {
        $sortkey = 'starttime';
    }
    my @sorted;
    if ((ref($slotsarr) eq 'ARRAY') && (ref($slots) eq 'HASH')) {
        @sorted =
            sort {
                     if (ref($slots->{$a}) && ref($slots->{$b})) {
                         return $slots->{$a}{$sortkey} <=> $slots->{$b}{$sortkey}
                     }
                     if (ref($slots->{$a})) { return -1;}
                     if (ref($slots->{$b})) { return 1;}
                     return 0;
                 } @{$slotsarr};
    }
    return @sorted;
}

=pod

=item * get_future_slots()

Inputs:

=over 4

cnum - course number

cdom - course domain

now - current UNIX time

symb - optional symb

=back

Returns:

=over 4

sorted_reservable - ref to array of student_schedulable slots currently 
                    reservable, ordered by end date of reservation period.

reservable_now - ref to hash of student_schedulable slots currently
                 reservable.

    Keys in inner hash are:
    (a) symb: either blank or symb to which slot use is restricted.
    (b) endreserve: end date of reservation period.
    (c) uniqueperiod: start,end dates when slot is to be uniquely
        selected.

sorted_future - ref to array of student_schedulable slots reservable in
                the future, ordered by start date of reservation period.

future_reservable - ref to hash of student_schedulable slots reservable
                    in the future.

    Keys in inner hash are:
    (a) symb: either blank or symb to which slot use is restricted.
    (b) startreserve:  start date of reservation period.
    (c) uniqueperiod: start,end dates when slot is to be uniquely
        selected.

=back

=cut

sub get_future_slots {
    my ($cnum,$cdom,$now,$symb) = @_;
    my (%reservable_now,%future_reservable,@sorted_reservable,@sorted_future);
    my %slots = &Apache::lonnet::get_course_slots($cnum,$cdom);
    foreach my $slot (keys(%slots)) {
        next unless($slots{$slot}->{'type'} eq 'schedulable_student');
        if ($symb) {
            next if (($slots{$slot}->{'symb'} ne '') && 
                     ($slots{$slot}->{'symb'} ne $symb));
        }
        if (($slots{$slot}->{'starttime'} > $now) &&
            ($slots{$slot}->{'endtime'} > $now)) {
            if (($slots{$slot}->{'allowedsections'}) || ($slots{$slot}->{'allowedusers'})) {
                my $userallowed = 0;
                if ($slots{$slot}->{'allowedsections'}) {
                    my @allowed_sec = split(',',$slots{$slot}->{'allowedsections'});
                    if (!defined($env{'request.role.sec'})
                        && grep(/^No section assigned$/,@allowed_sec)) {
                        $userallowed=1;
                    } else {
                        if (grep(/^\Q$env{'request.role.sec'}\E$/,@allowed_sec)) {
                            $userallowed=1;
                        }
                    }
                    unless ($userallowed) {
                        if (defined($env{'request.course.groups'})) {
                            my @groups = split(/:/,$env{'request.course.groups'});
                            foreach my $group (@groups) {
                                if (grep(/^\Q$group\E$/,@allowed_sec)) {
                                    $userallowed=1;
                                    last;
                                }
                            }
                        }
                    }
                }
                if ($slots{$slot}->{'allowedusers'}) {
                    my @allowed_users = split(',',$slots{$slot}->{'allowedusers'});
                    my $user = $env{'user.name'}.':'.$env{'user.domain'};
                    if (grep(/^\Q$user\E$/,@allowed_users)) {
                        $userallowed = 1;
                    }
                }
                next unless($userallowed);
            }
            my $startreserve = $slots{$slot}->{'startreserve'};
            my $endreserve = $slots{$slot}->{'endreserve'};
            my $symb = $slots{$slot}->{'symb'};
            my $uniqueperiod;
            if (ref($slots{$slot}->{'uniqueperiod'}) eq 'ARRAY') {
                $uniqueperiod = join(',',@{$slots{$slot}->{'uniqueperiod'}});
            }
            if (($startreserve < $now) &&
                (!$endreserve || $endreserve > $now)) {
                my $lastres = $endreserve;
                if (!$lastres) {
                    $lastres = $slots{$slot}->{'starttime'};
                }
                $reservable_now{$slot} = {
                                           symb       => $symb,
                                           endreserve => $lastres,
                                           uniqueperiod => $uniqueperiod,   
                                         };
            } elsif (($startreserve > $now) &&
                     (!$endreserve || $endreserve > $startreserve)) {
                $future_reservable{$slot} = {
                                              symb         => $symb,
                                              startreserve => $startreserve,
                                              uniqueperiod => $uniqueperiod,
                                            };
            }
        }
    }
    my @unsorted_reservable = keys(%reservable_now);
    if (@unsorted_reservable > 0) {
        @sorted_reservable = 
            &sorted_slots(\@unsorted_reservable,\%reservable_now,'endreserve');
    }
    my @unsorted_future = keys(%future_reservable);
    if (@unsorted_future > 0) {
        @sorted_future =
            &sorted_slots(\@unsorted_future,\%future_reservable,'startreserve');
    }
    return (\@sorted_reservable,\%reservable_now,\@sorted_future,\%future_reservable);
}

=pod

=back

=head1 HTTP Helpers

=over 4

=item * &get_unprocessed_cgi($query,$possible_names)

Modify the %env hash to contain unprocessed CGI form parameters held in
$query.  The parameters listed in $possible_names (an array reference),
will be set in $env{'form.name'} if they do not already exist.

Typically called with $ENV{'QUERY_STRING'} as the first parameter.  
$possible_names is an ref to an array of form element names.  As an example:
get_unprocessed_cgi($ENV{'QUERY_STRING'},['uname','udom']);
will result in $env{'form.uname'} and $env{'form.udom'} being set.

=cut

sub get_unprocessed_cgi {
  my ($query,$possible_names)= @_;
  # $Apache::lonxml::debug=1;
  foreach my $pair (split(/&/,$query)) {
    my ($name, $value) = split(/=/,$pair);
    $name = &unescape($name);
    if (!defined($possible_names) || (grep {$_ eq $name} @$possible_names)) {
      $value =~ tr/+/ /;
      $value =~ s/%([a-fA-F0-9][a-fA-F0-9])/pack("C",hex($1))/eg;
      unless (defined($env{'form.'.$name})) { &add_to_env('form.'.$name,$value) };
    }
  }
}

=pod

=item * &cacheheader() 

returns cache-controlling header code

=cut

sub cacheheader {
    unless ($env{'request.method'} eq 'GET') { return ''; }
    my $date=strftime("%a, %d %b %Y %H:%M:%S GMT",gmtime);
    my $output .='<meta HTTP-EQUIV="Expires" CONTENT="'.$date.'" />
                <meta HTTP-EQUIV="Cache-control" CONTENT="no-cache" />
                <meta HTTP-EQUIV="Pragma" CONTENT="no-cache" />';
    return $output;
}

=pod

=item * &no_cache($r) 

specifies header code to not have cache

=cut

sub no_cache {
    my ($r) = @_;
    if ($ENV{'REQUEST_METHOD'} ne 'GET' &&
	$env{'request.method'} ne 'GET') { return ''; }
    my $date=strftime("%a, %d %b %Y %H:%M:%S GMT",gmtime(time));
    $r->no_cache(1);
    $r->header_out("Expires" => $date);
    $r->header_out("Pragma" => "no-cache");
}

sub content_type {
    my ($r,$type,$charset) = @_;
    if ($r) {
	#  Note that printout.pl calls this with undef for $r.
	&no_cache($r);
    }
    if ($env{'browser.mathml'} && $type eq 'text/html') { $type='text/xml'; }
    unless ($charset) {
	$charset=&Apache::lonlocal::current_encoding;
    }
    if ($charset) { $type.='; charset='.$charset; }
    if ($r) {
	$r->content_type($type);
    } else {
	print("Content-type: $type\n\n");
    }
}

=pod

=item * &add_to_env($name,$value) 

adds $name to the %env hash with value
$value, if $name already exists, the entry is converted to an array
reference and $value is added to the array.

=cut

sub add_to_env {
  my ($name,$value)=@_;
  if (defined($env{$name})) {
    if (ref($env{$name})) {
      #already have multiple values
      push(@{ $env{$name} },$value);
    } else {
      #first time seeing multiple values, convert hash entry to an arrayref
      my $first=$env{$name};
      undef($env{$name});
      push(@{ $env{$name} },$first,$value);
    }
  } else {
    $env{$name}=$value;
  }
}

=pod

=item * &get_env_multiple($name) 

gets $name from the %env hash, it seemlessly handles the cases where multiple
values may be defined and end up as an array ref.

returns an array of values

=cut

sub get_env_multiple {
    my ($name) = @_;
    my @values;
    if (defined($env{$name})) {
        # exists is it an array
        if (ref($env{$name})) {
            @values=@{ $env{$name} };
        } else {
            $values[0]=$env{$name};
        }
    }
    return(@values);
}

sub ask_for_embedded_content {
    my ($actionurl,$state,$allfiles,$codebase,$args)=@_;
    my (%subdependencies,%dependencies,%mapping,%existing,%newfiles,%pathchanges,
        %currsubfile,%unused,$rem);
    my $counter = 0;
    my $numnew = 0;
    my $numremref = 0;
    my $numinvalid = 0;
    my $numpathchg = 0;
    my $numexisting = 0;
    my $numunused = 0;
    my ($output,$upload_output,$toplevel,$url,$udom,$uname,$getpropath,$cdom,$cnum,
        $fileloc,$filename,$delete_output,$modify_output,$title,$symb,$path,$navmap);
    my $heading = &mt('Upload embedded files');
    my $buttontext = &mt('Upload');

    if ($env{'request.course.id'}) {
        if ($actionurl eq '/adm/dependencies') {
            $navmap = Apache::lonnavmaps::navmap->new();
        }
        $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
        $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
    }
    if (($actionurl eq '/adm/portfolio') ||
        ($actionurl eq '/adm/coursegrp_portfolio')) {
        my $current_path='/';
        if ($env{'form.currentpath'}) {
            $current_path = $env{'form.currentpath'};
        }
        if ($actionurl eq '/adm/coursegrp_portfolio') {
            $udom = $cdom;
            $uname = $cnum;
            $url = '/userfiles/groups/'.$env{'form.group'}.'/portfolio';
        } else {
            $udom = $env{'user.domain'};
            $uname = $env{'user.name'};
            $url = '/userfiles/portfolio';
        }
        $toplevel = $url.'/';
        $url .= $current_path;
        $getpropath = 1;
    } elsif (($actionurl eq '/adm/upload') || ($actionurl eq '/adm/testbank') ||
             ($actionurl eq '/adm/imsimport')) { 
        my ($udom,$uname,$rest) = ($args->{'current_path'} =~ m{/priv/($match_domain)/($match_username)/?(.*)$});
        $url = $Apache::lonnet::perlvar{'lonDocRoot'}."/priv/$udom/$uname/";
        $toplevel = $url;
        if ($rest ne '') {
            $url .= $rest;
        }
    } elsif ($actionurl eq '/adm/coursedocs') {
        if (ref($args) eq 'HASH') {
            $url = $args->{'docs_url'};
            $toplevel = $url;
            if ($args->{'context'} eq 'paste') {
                ($cdom,$cnum) = ($url =~ m{^\Q/uploaded/\E($match_domain)/($match_courseid)/});
                ($path) =
                    ($toplevel =~ m{^(\Q/uploaded/$cdom/$cnum/\E(?:docs|supplemental)/(?:default|\d+)/\d+)/});
                $fileloc = &Apache::lonnet::filelocation('',$toplevel);
                $fileloc =~ s{^/}{};
            }
        }
    } elsif ($actionurl eq '/adm/dependencies') {
        if ($env{'request.course.id'} ne '') {
            if (ref($args) eq 'HASH') {
                $url = $args->{'docs_url'};
                $title = $args->{'docs_title'};
                $toplevel = $url;
                unless ($toplevel =~ m{^/}) {
                    $toplevel = "/$url";
                }
                ($rem) = ($toplevel =~ m{^(.+/)[^/]+$});
                if ($toplevel =~ m{^(\Q/uploaded/$cdom/$cnum/portfolio/syllabus\E)}) {
                    $path = $1;
                } else {
                    ($path) =
                        ($toplevel =~ m{^(\Q/uploaded/$cdom/$cnum/\E(?:docs|supplemental)/(?:default|\d+)/\d+)/});
                }
                if ($toplevel=~/^\/*(uploaded|editupload)/) {
                    $fileloc = $toplevel;
                    $fileloc=~ s/^\s*(\S+)\s*$/$1/;
                    my ($udom,$uname,$fname) =
                        ($fileloc=~ m{^/+(?:uploaded|editupload)/+($match_domain)/+($match_name)/+(.*)$});
                    $fileloc = propath($udom,$uname).'/userfiles/'.$fname;
                } else {
                    $fileloc = &Apache::lonnet::filelocation('',$toplevel);
                }
                $fileloc =~ s{^/}{};
                ($filename) = ($fileloc =~ m{.+/([^/]+)$});
                $heading = &mt('Status of dependencies in [_1]',"$title ($filename)");
            }
        }
    } elsif ($actionurl eq "/public/$cdom/$cnum/syllabus") {
        $udom = $cdom;
        $uname = $cnum;
        $url = "/uploaded/$cdom/$cnum/portfolio/syllabus";
        $toplevel = $url;
        $path = $url;
        $fileloc = &Apache::lonnet::filelocation('',$toplevel).'/';
        $fileloc =~ s{^/}{};
    }
    foreach my $file (keys(%{$allfiles})) {
        my $embed_file;
        if (($path eq "/uploaded/$cdom/$cnum/portfolio/syllabus") && ($file =~ m{^\Q$path/\E(.+)$})) {
            $embed_file = $1;
        } else {
            $embed_file = $file;
        }
        my ($absolutepath,$cleaned_file);
        if ($embed_file =~ m{^\w+://}) {
            $cleaned_file = $embed_file;
            $newfiles{$cleaned_file} = 1;
            $mapping{$cleaned_file} = $embed_file;
        } else {
            $cleaned_file = &clean_path($embed_file);
            if ($embed_file =~ m{^/}) {
                $absolutepath = $embed_file;
            }
            if ($cleaned_file =~ m{/}) {
                my ($path,$fname) = ($cleaned_file =~ m{^(.+)/([^/]*)$});
                $path = &check_for_traversal($path,$url,$toplevel);
                my $item = $fname;
                if ($path ne '') {
                    $item = $path.'/'.$fname;
                    $subdependencies{$path}{$fname} = 1;
                } else {
                    $dependencies{$item} = 1;
                }
                if ($absolutepath) {
                    $mapping{$item} = $absolutepath;
                } else {
                    $mapping{$item} = $embed_file;
                }
            } else {
                $dependencies{$embed_file} = 1;
                if ($absolutepath) {
                    $mapping{$cleaned_file} = $absolutepath;
                } else {
                    $mapping{$cleaned_file} = $embed_file;
                }
            }
        }
    }
    my $dirptr = 16384;
    foreach my $path (keys(%subdependencies)) {
        $currsubfile{$path} = {};
        if (($actionurl eq '/adm/portfolio') ||
            ($actionurl eq '/adm/coursegrp_portfolio')) { 
            my ($sublistref,$listerror) =
                &Apache::lonnet::dirlist($url.$path,$udom,$uname,$getpropath);
            if (ref($sublistref) eq 'ARRAY') {
                foreach my $line (@{$sublistref}) {
                    my ($file_name,$rest) = split(/\&/,$line,2);
                    $currsubfile{$path}{$file_name} = 1;
                }
            }
        } elsif (($actionurl eq '/adm/upload') || ($actionurl eq '/adm/testbank')) {
            if (opendir(my $dir,$url.'/'.$path)) {
                my @subdir_list = grep(!/^\./,readdir($dir));
                map {$currsubfile{$path}{$_} = 1;} @subdir_list;
            }
        } elsif (($actionurl eq '/adm/dependencies') ||
                 (($actionurl eq '/adm/coursedocs') && (ref($args) eq 'HASH') &&
                  ($args->{'context'} eq 'paste')) ||
                 ($actionurl eq "/public/$cdom/$cnum/syllabus")) {
            if ($env{'request.course.id'} ne '') {
                my $dir;
                if ($actionurl eq "/public/$cdom/$cnum/syllabus") {
                    $dir = $fileloc;
                } else {
                    ($dir) = ($fileloc =~ m{^(.+/)[^/]+$});
                }
                if ($dir ne '') {
                    my ($sublistref,$listerror) =
                        &Apache::lonnet::dirlist($dir.$path,$cdom,$cnum,$getpropath,undef,'/');
                    if (ref($sublistref) eq 'ARRAY') {
                        foreach my $line (@{$sublistref}) {
                            my ($file_name,$dom,undef,$testdir,undef,undef,undef,undef,$size,
                                undef,$mtime)=split(/\&/,$line,12);
                            unless (($testdir&$dirptr) ||
                                    ($file_name =~ /^\.\.?$/)) {
                                $currsubfile{$path}{$file_name} = [$size,$mtime];
                            }
                        }
                    }
                }
            }
        }
        foreach my $file (keys(%{$subdependencies{$path}})) {
            if (exists($currsubfile{$path}{$file})) {
                my $item = $path.'/'.$file;
                unless ($mapping{$item} eq $item) {
                    $pathchanges{$item} = 1;
                }
                $existing{$item} = 1;
                $numexisting ++;
            } else {
                $newfiles{$path.'/'.$file} = 1;
            }
        }
        if ($actionurl eq '/adm/dependencies') {
            foreach my $path (keys(%currsubfile)) {
                if (ref($currsubfile{$path}) eq 'HASH') {
                    foreach my $file (keys(%{$currsubfile{$path}})) {
                         unless ($subdependencies{$path}{$file}) {
                             next if (($rem ne '') &&
                                      (($env{"httpref.$rem"."$path/$file"} ne '') ||
                                       (ref($navmap) &&
                                       (($navmap->getResourceByUrl($rem."$path/$file") ne '') ||
                                        (($file =~ /^(.*\.s?html?)\.bak$/i) &&
                                         ($navmap->getResourceByUrl($rem."$path/$1")))))));
                             $unused{$path.'/'.$file} = 1; 
                         }
                    }
                }
            }
        }
    }
    my %currfile;
    if (($actionurl eq '/adm/portfolio') ||
        ($actionurl eq '/adm/coursegrp_portfolio')) {
        my ($dirlistref,$listerror) =
            &Apache::lonnet::dirlist($url,$udom,$uname,$getpropath);
        if (ref($dirlistref) eq 'ARRAY') {
            foreach my $line (@{$dirlistref}) {
                my ($file_name,$rest) = split(/\&/,$line,2);
                $currfile{$file_name} = 1;
            }
        }
    } elsif (($actionurl eq '/adm/upload') || ($actionurl eq '/adm/testbank')) {
        if (opendir(my $dir,$url)) {
            my @dir_list = grep(!/^\./,readdir($dir));
            map {$currfile{$_} = 1;} @dir_list;
        }
    } elsif (($actionurl eq '/adm/dependencies') ||
             (($actionurl eq '/adm/coursedocs') && (ref($args) eq 'HASH') &&
              ($args->{'context'} eq 'paste')) ||
             ($actionurl eq "/public/$cdom/$cnum/syllabus")) {
        if ($env{'request.course.id'} ne '') {
            my ($dir) = ($fileloc =~ m{^(.+/)[^/]+$});
            if ($dir ne '') {
                my ($dirlistref,$listerror) =
                    &Apache::lonnet::dirlist($dir,$cdom,$cnum,$getpropath,undef,'/');
                if (ref($dirlistref) eq 'ARRAY') {
                    foreach my $line (@{$dirlistref}) {
                        my ($file_name,$dom,undef,$testdir,undef,undef,undef,undef,
                            $size,undef,$mtime)=split(/\&/,$line,12);
                        unless (($testdir&$dirptr) ||
                                ($file_name =~ /^\.\.?$/)) {
                            $currfile{$file_name} = [$size,$mtime];
                        }
                    }
                }
            }
        }
    }
    foreach my $file (keys(%dependencies)) {
        if (exists($currfile{$file})) {
            unless ($mapping{$file} eq $file) {
                $pathchanges{$file} = 1;
            }
            $existing{$file} = 1;
            $numexisting ++;
        } else {
            $newfiles{$file} = 1;
        }
    }
    foreach my $file (keys(%currfile)) {
        unless (($file eq $filename) ||
                ($file eq $filename.'.bak') ||
                ($dependencies{$file})) {
            if ($actionurl eq '/adm/dependencies') {
                unless ($toplevel =~ m{^\Q/uploaded/$cdom/$cnum/portfolio/syllabus\E}) {
                    next if (($rem ne '') &&
                             (($env{"httpref.$rem".$file} ne '') ||
                              (ref($navmap) &&
                              (($navmap->getResourceByUrl($rem.$file) ne '') ||
                               (($file =~ /^(.*\.s?html?)\.bak$/i) &&
                                ($navmap->getResourceByUrl($rem.$1)))))));
                }
            }
            $unused{$file} = 1;
        }
    }
    if (($actionurl eq '/adm/coursedocs') && (ref($args) eq 'HASH') &&
        ($args->{'context'} eq 'paste')) {
        $counter = scalar(keys(%existing));
        $numpathchg = scalar(keys(%pathchanges));
        return ($output,$counter,$numpathchg,\%existing);
    } elsif (($actionurl eq "/public/$cdom/$cnum/syllabus") &&
             (ref($args) eq 'HASH') && ($args->{'context'} eq 'rewrites')) {
        $counter = scalar(keys(%existing));
        $numpathchg = scalar(keys(%pathchanges));
        return ($output,$counter,$numpathchg,\%existing,\%mapping);
    }
    foreach my $embed_file (sort {lc($a) cmp lc($b)} keys(%newfiles)) {
        if ($actionurl eq '/adm/dependencies') {
            next if ($embed_file =~ m{^\w+://});
        }
        $upload_output .= &start_data_table_row().
                          '<td valign="top"><img src="'.&icon($embed_file).'" />&nbsp;'.
                          '<span class="LC_filename">'.$embed_file.'</span>';
        unless ($mapping{$embed_file} eq $embed_file) {
            $upload_output .= '<br /><span class="LC_info" style="font-size:smaller;">'.
                              &mt('changed from: [_1]',$mapping{$embed_file}).'</span>';
        }
        $upload_output .= '</td>';
        if ($args->{'ignore_remote_references'} && $embed_file =~ m{^\w+://}) { 
            $upload_output.='<td align="right">'.
                            '<span class="LC_info LC_fontsize_medium">'.
                            &mt("URL points to web address").'</span>';
            $numremref++;
        } elsif ($args->{'error_on_invalid_names'}
            && $embed_file ne &Apache::lonnet::clean_filename($embed_file,{'keep_path' => 1,})) {
            $upload_output.='<td align="right"><span class="LC_warning">'.
                            &mt('Invalid characters').'</span>';
            $numinvalid++;
        } else {
            $upload_output .= '<td>'.
                              &embedded_file_element('upload_embedded',$counter,
                                                     $embed_file,\%mapping,
                                                     $allfiles,$codebase,'upload');
            $counter ++;
            $numnew ++;
        }
        $upload_output .= '</td>'.&Apache::loncommon::end_data_table_row()."\n";
    }
    foreach my $embed_file (sort {lc($a) cmp lc($b)} keys(%existing)) {
        if ($actionurl eq '/adm/dependencies') {
            my ($size,$mtime) = &get_dependency_details(\%currfile,\%currsubfile,$embed_file);
            $modify_output .= &start_data_table_row().
                              '<td><a href="'.$path.'/'.$embed_file.'" style="text-decoration:none;">'.
                              '<img src="'.&icon($embed_file).'" border="0" />'.
                              '&nbsp;<span class="LC_filename">'.$embed_file.'</span></a></td>'.
                              '<td>'.$size.'</td>'.
                              '<td>'.$mtime.'</td>'.
                              '<td><label><input type="checkbox" name="mod_upload_dep" '.
                              'onclick="toggleBrowse('."'$counter'".')" id="mod_upload_dep_'.
                              $counter.'" value="'.$counter.'" />'.&mt('Yes').'</label>'.
                              '<div id="moduploaddep_'.$counter.'" style="display:none;">'.
                              &embedded_file_element('upload_embedded',$counter,
                                                     $embed_file,\%mapping,
                                                     $allfiles,$codebase,'modify').
                              '</div></td>'.
                              &end_data_table_row()."\n";
            $counter ++;
        } else {
            $upload_output .= &start_data_table_row().
                              '<td valign="top"><img src="'.&icon($embed_file).'" />&nbsp;'.
                              '<span class="LC_filename">'.$embed_file.'</span></td>'.
                              '<td align="right"><span class="LC_info LC_fontsize_medium">'.&mt('Already exists').'</span></td>'.
                              &Apache::loncommon::end_data_table_row()."\n";
        }
    }
    my $delidx = $counter;
    foreach my $oldfile (sort {lc($a) cmp lc($b)} keys(%unused)) {
        my ($size,$mtime) = &get_dependency_details(\%currfile,\%currsubfile,$oldfile);
        $delete_output .= &start_data_table_row().
                          '<td><img src="'.&icon($oldfile).'" />'.
                          '&nbsp;<span class="LC_filename">'.$oldfile.'</span></td>'.
                          '<td>'.$size.'</td>'.
                          '<td>'.$mtime.'</td>'.
                          '<td><label><input type="checkbox" name="del_upload_dep" '.
                          ' value="'.$delidx.'" />'.&mt('Yes').'</label>'.
                          &embedded_file_element('upload_embedded',$delidx,
                                                 $oldfile,\%mapping,$allfiles,
                                                 $codebase,'delete').'</td>'.
                          &end_data_table_row()."\n"; 
        $numunused ++;
        $delidx ++;
    }
    if ($upload_output) {
        $upload_output = &start_data_table().
                         $upload_output.
                         &end_data_table()."\n";
    }
    if ($modify_output) {
        $modify_output = &start_data_table().
                         &start_data_table_header_row().
                         '<th>'.&mt('File').'</th>'.
                         '<th>'.&mt('Size (KB)').'</th>'.
                         '<th>'.&mt('Modified').'</th>'.
                         '<th>'.&mt('Upload replacement?').'</th>'.
                         &end_data_table_header_row().
                         $modify_output.
                         &end_data_table()."\n";
    }
    if ($delete_output) {
        $delete_output = &start_data_table().
                         &start_data_table_header_row().
                         '<th>'.&mt('File').'</th>'.
                         '<th>'.&mt('Size (KB)').'</th>'.
                         '<th>'.&mt('Modified').'</th>'.
                         '<th>'.&mt('Delete?').'</th>'.
                         &end_data_table_header_row().
                         $delete_output.
                         &end_data_table()."\n";
    }
    my $applies = 0;
    if ($numremref) {
        $applies ++;
    }
    if ($numinvalid) {
        $applies ++;
    }
    if ($numexisting) {
        $applies ++;
    }
    if ($counter || $numunused) {
        $output = '<form name="upload_embedded" action="'.$actionurl.'"'.
                  ' method="post" enctype="multipart/form-data">'."\n".
                  $state.'<h3>'.$heading.'</h3>'; 
        if ($actionurl eq '/adm/dependencies') {
            if ($numnew) {
                $output .= '<h4>'.&mt('Missing dependencies').'</h4>'.
                           '<p>'.&mt('The following files need to be uploaded.').'</p>'."\n".
                           $upload_output.'<br />'."\n";
            }
            if ($numexisting) {
                $output .= '<h4>'.&mt('Uploaded dependencies (in use)').'</h4>'.
                           '<p>'.&mt('Upload a new file to replace the one currently in use.').'</p>'."\n".
                           $modify_output.'<br />'."\n";
                           $buttontext = &mt('Save changes');
            }
            if ($numunused) {
                $output .= '<h4>'.&mt('Unused files').'</h4>'.
                           '<p>'.&mt('The following uploaded files are no longer used.').'</p>'."\n".
                           $delete_output.'<br />'."\n";
                           $buttontext = &mt('Save changes');
            }
        } else {
            $output .= $upload_output.'<br />'."\n";
        }
        $output .= '<input type ="hidden" name="number_embedded_items" value="'.
                   $counter.'" />'."\n";
        if ($actionurl eq '/adm/dependencies') { 
            $output .= '<input type ="hidden" name="number_newemb_items" value="'.
                       $numnew.'" />'."\n";
        } elsif ($actionurl eq '') {
            $output .=  '<input type="hidden" name="phase" value="three" />';
        }
    } elsif ($applies) {
        $output = '<b>'.&mt('Referenced files').'</b>:<br />';
        if ($applies > 1) {
            $output .=  
                &mt('No dependencies need to be uploaded, as one of the following applies to each reference:').'<ul>';
            if ($numremref) {
                $output .= '<li>'.&mt('reference is to a URL which points to another server').'</li>'."\n";
            }
            if ($numinvalid) {
                $output .= '<li>'.&mt('reference is to file with a name containing invalid characters').'</li>'."\n";
            }
            if ($numexisting) {
                $output .= '<li>'.&mt('reference is to an existing file at the specified location').'</li>'."\n";
            }
            $output .= '</ul><br />';
        } elsif ($numremref) {
            $output .= '<p>'.&mt('None to upload, as all references are to URLs pointing to another server.').'</p>';
        } elsif ($numinvalid) {
            $output .= '<p>'.&mt('None to upload, as all references are to files with names containing invalid characters.').'</p>';
        } elsif ($numexisting) {
            $output .= '<p>'.&mt('None to upload, as all references are to existing files.').'</p>';
        }
        $output .= $upload_output.'<br />';
    }
    my ($pathchange_output,$chgcount);
    $chgcount = $counter;
    if (keys(%pathchanges) > 0) {
        foreach my $embed_file (sort {lc($a) cmp lc($b)} keys(%pathchanges)) {
            if ($counter) {
                $output .= &embedded_file_element('pathchange',$chgcount,
                                                  $embed_file,\%mapping,
                                                  $allfiles,$codebase,'change');
            } else {
                $pathchange_output .= 
                    &start_data_table_row().
                    '<td><input type ="checkbox" name="namechange" value="'.
                    $chgcount.'" checked="checked" /></td>'.
                    '<td>'.$mapping{$embed_file}.'</td>'.
                    '<td>'.$embed_file.
                    &embedded_file_element('pathchange',$numpathchg,$embed_file,
                                           \%mapping,$allfiles,$codebase,'change').
                    '</td>'.&end_data_table_row();
            }
            $numpathchg ++;
            $chgcount ++;
        }
    }
    if (($counter) || ($numunused)) {
        if ($numpathchg) {
            $output .= '<input type ="hidden" name="number_pathchange_items" value="'.
                       $numpathchg.'" />'."\n";
        }
        if (($actionurl eq '/adm/upload') || ($actionurl eq '/adm/testbank') || 
            ($actionurl eq '/adm/imsimport')) {
            $output .= '<input type="hidden" name="phase" value="three" />'."\n";
        } elsif ($actionurl eq '/adm/portfolio' || $actionurl eq '/adm/coursegrp_portfolio') {
            $output .= '<input type="hidden" name="action" value="upload_embedded" />';
        } elsif ($actionurl eq '/adm/dependencies') {
            $output .= '<input type="hidden" name="action" value="process_changes" />';
        }
        $output .= '<input type ="submit" value="'.$buttontext.'" />'."\n".'</form>'."\n";
    } elsif ($numpathchg) {
        my %pathchange = ();
        $output .= &modify_html_form('pathchange',$actionurl,$state,\%pathchange,$pathchange_output);
        if (($actionurl eq '/adm/portfolio') || ($actionurl eq '/adm/coursegrp_portfolio')) {
            $output .= '<p>'.&mt('or').'</p>'; 
        }
    }
    return ($output,$counter,$numpathchg);
}

=pod

=item * clean_path($name)

Performs clean-up of directories, subdirectories and filename in an
embedded object, referenced in an HTML file which is being uploaded
to a course or portfolio, where
"Upload embedded images/multimedia files if HTML file" checkbox was
checked.

Clean-up is similar to replacements in lonnet::clean_filename()
except each / between sub-directory and next level is preserved.

=cut

sub clean_path {
    my ($embed_file) = @_;
    $embed_file =~s{^/+}{};
    my @contents;
    if ($embed_file =~ m{/}) {
        @contents = split(/\//,$embed_file);
    } else {
        @contents = ($embed_file);
    }
    my $lastidx = scalar(@contents)-1;
    for (my $i=0; $i<=$lastidx; $i++) {
        $contents[$i]=~s{\\}{/}g;
        $contents[$i]=~s/\s+/\_/g;
        $contents[$i]=~s{[^/\w\.\-]}{}g;
        if ($i == $lastidx) {
            $contents[$i]=~s/\.(\d+)(?=\.)/_$1/g;
        }
    }
    if ($lastidx > 0) {
        return join('/',@contents);
    } else {
        return $contents[0];
    }
}

sub embedded_file_element {
    my ($context,$num,$embed_file,$mapping,$allfiles,$codebase,$type) = @_;
    return unless ((ref($mapping) eq 'HASH') && (ref($allfiles) eq 'HASH') &&
                   (ref($codebase) eq 'HASH'));
    my $output;
    if (($context eq 'upload_embedded') && ($type ne 'delete')) {
       $output = '<input name="embedded_item_'.$num.'" type="file" value="" />'."\n";
    }
    $output .= '<input name="embedded_orig_'.$num.'" type="hidden" value="'.
               &escape($embed_file).'" />';
    unless (($context eq 'upload_embedded') && 
            ($mapping->{$embed_file} eq $embed_file)) {
        $output .='
        <input name="embedded_ref_'.$num.'" type="hidden" value="'.&escape($mapping->{$embed_file}).'" />';
    }
    my $attrib;
    if (ref($allfiles->{$mapping->{$embed_file}}) eq 'ARRAY') {
        $attrib = &escape(join(':',@{$allfiles->{$mapping->{$embed_file}}}));
    }
    $output .=
        "\n\t\t".
        '<input name="embedded_attrib_'.$num.'" type="hidden" value="'.
        $attrib.'" />';
    if (exists($codebase->{$mapping->{$embed_file}})) {
        $output .=
            "\n\t\t".
            '<input name="codebase_'.$num.'" type="hidden" value="'.
            &escape($codebase->{$mapping->{$embed_file}}).'" />';
    }
    return $output;
}

sub get_dependency_details {
    my ($currfile,$currsubfile,$embed_file) = @_;
    my ($size,$mtime,$showsize,$showmtime);
    if ((ref($currfile) eq 'HASH') && (ref($currsubfile))) {
        if ($embed_file =~ m{/}) {
            my ($path,$fname) = split(/\//,$embed_file);
            if (ref($currsubfile->{$path}{$fname}) eq 'ARRAY') {
                ($size,$mtime) = @{$currsubfile->{$path}{$fname}};
            }
        } else {
            if (ref($currfile->{$embed_file}) eq 'ARRAY') {
                ($size,$mtime) = @{$currfile->{$embed_file}};
            }
        }
        $showsize = $size/1024.0;
        $showsize = sprintf("%.1f",$showsize);
        if ($mtime > 0) {
            $showmtime = &Apache::lonlocal::locallocaltime($mtime);
        }
    }
    return ($showsize,$showmtime);
}

sub ask_embedded_js {
    return <<"END";
<script type="text/javascript"">
// <![CDATA[
function toggleBrowse(counter) {
    var chkboxid = document.getElementById('mod_upload_dep_'+counter);
    var fileid = document.getElementById('embedded_item_'+counter);
    var uploaddivid = document.getElementById('moduploaddep_'+counter);
    if (chkboxid.checked == true) {
        uploaddivid.style.display='block';
    } else {
        uploaddivid.style.display='none';
        fileid.value = '';
    }
}
// ]]>
</script>

END
}

sub upload_embedded {
    my ($context,$dirpath,$uname,$udom,$dir_root,$url_root,$group,$disk_quota,
        $current_disk_usage,$hiddenstate,$actionurl) = @_;
    my (%pathchange,$output,$modifyform,$footer,$returnflag);
    for (my $i=0; $i<$env{'form.number_embedded_items'}; $i++) {
        next if (!exists($env{'form.embedded_item_'.$i.'.filename'}));
        my $orig_uploaded_filename =
            $env{'form.embedded_item_'.$i.'.filename'};
        foreach my $type ('orig','ref','attrib','codebase') {
            if ($env{'form.embedded_'.$type.'_'.$i} ne '') {
                $env{'form.embedded_'.$type.'_'.$i} =
                    &unescape($env{'form.embedded_'.$type.'_'.$i});
            }
        }
        my ($path,$fname) =
            ($env{'form.embedded_orig_'.$i} =~ m{(.*/)([^/]*)});
        # no path, whole string is fname
        if (!$fname) { $fname = $env{'form.embedded_orig_'.$i} };
        $fname = &Apache::lonnet::clean_filename($fname);
        # See if there is anything left
        next if ($fname eq '');

        # Check if file already exists as a file or directory.
        my ($state,$msg);
        if ($context eq 'portfolio') {
            my $port_path = $dirpath;
            if ($group ne '') {
                $port_path = "groups/$group/$port_path";
            }
            ($state,$msg) = &check_for_upload($env{'form.currentpath'}.$path,
                                              $fname,$group,'embedded_item_'.$i,
                                              $dir_root,$port_path,$disk_quota,
                                              $current_disk_usage,$uname,$udom);
            if ($state eq 'will_exceed_quota'
                || $state eq 'file_locked') {
                $output .= $msg;
                next;
            }
        } elsif (($context eq 'author') || ($context eq 'testbank')) {
            ($state,$msg) = &check_for_existing($path,$fname,'embedded_item_'.$i);
            if ($state eq 'exists') {
                $output .= $msg;
                next;
            }
        }
        # Check if extension is valid
        if (($fname =~ /\.(\w+)$/) &&
            (&Apache::loncommon::fileembstyle($1) eq 'hdn')) {
            $output .= &mt('Invalid file extension ([_1]) - reserved for internal use.',$1)
                      .' '.&mt('Rename the file with a different extension and re-upload.').'<br />';
            next;
        } elsif (($fname =~ /\.(\w+)$/) &&
                 (!defined(&Apache::loncommon::fileembstyle($1)))) {
            $output .= &mt('Unrecognized file extension ([_1]) - rename the file with a proper extension and re-upload.',$1).'<br />';
            next;
        } elsif ($fname=~/\.(\d+)\.(\w+)$/) {
            $output .= &mt('Filename not allowed - rename the file to remove the number immediately before the file extension([_1]) and re-upload.',$2).'<br />';
            next;
        }
        $env{'form.embedded_item_'.$i.'.filename'}=$fname;
        my $subdir = $path;
        $subdir =~ s{/+$}{};
        if ($context eq 'portfolio') {
            my $result;
            if ($state eq 'existingfile') {
                $result=
                    &Apache::lonnet::userfileupload('embedded_item_'.$i,'existingfile',
                                                    $dirpath.$env{'form.currentpath'}.$subdir);
            } else {
                $result=
                    &Apache::lonnet::userfileupload('embedded_item_'.$i,'',
                                                    $dirpath.
                                                    $env{'form.currentpath'}.$subdir);
                if ($result !~ m|^/uploaded/|) {
                    $output .= '<span class="LC_error">'
                               .&mt('An error occurred ([_1]) while trying to upload [_2] for embedded element [_3].'
                               ,$result,$orig_uploaded_filename,$env{'form.embedded_orig_'.$i})
                               .'</span><br />';
                    next;
                } else {
                    $output .= &mt('Uploaded [_1]','<span class="LC_filename">'.
                               $path.$fname.'</span>').'<br />';     
                }
            }
        } elsif (($context eq 'coursedoc') || ($context eq 'syllabus')) {
            my $extendedsubdir = $dirpath.'/'.$subdir;
            $extendedsubdir =~ s{/+$}{};
            my $result =
                &Apache::lonnet::userfileupload('embedded_item_'.$i,$context,$extendedsubdir);
            if ($result !~ m|^/uploaded/|) {
                $output .= '<span class="LC_error">'
                           .&mt('An error occurred ([_1]) while trying to upload [_2] for embedded element [_3].'
                           ,$result,$orig_uploaded_filename,$env{'form.embedded_orig_'.$i})
                           .'</span><br />';
                    next;
            } else {
                $output .= &mt('Uploaded [_1]','<span class="LC_filename">'.
                           $path.$fname.'</span>').'<br />';
                if ($context eq 'syllabus') {
                    &Apache::lonnet::make_public_indefinitely($result);
                }
            }
        } else {
# Save the file
            my $target = $env{'form.embedded_item_'.$i};
            my $fullpath = $dir_root.$dirpath.'/'.$path;
            my $dest = $fullpath.$fname;
            my $url = $url_root.$dirpath.'/'.$path.$fname;
            my @parts=split(/\//,"$dirpath/$path");
            my $count;
            my $filepath = $dir_root;
            foreach my $subdir (@parts) {
                $filepath .= "/$subdir";
                if (!-e $filepath) {
                    mkdir($filepath,0770);
                }
            }
            my $fh;
            if (!open($fh,'>'.$dest)) {
                &Apache::lonnet::logthis('Failed to create '.$dest);
                $output .= '<span class="LC_error">'.
                           &mt('An error occurred while trying to upload [_1] for embedded element [_2].',
                               $orig_uploaded_filename,$env{'form.embedded_orig_'.$i}).
                           '</span><br />';
            } else {
                if (!print $fh $env{'form.embedded_item_'.$i}) {
                    &Apache::lonnet::logthis('Failed to write to '.$dest);
                    $output .= '<span class="LC_error">'.
                              &mt('An error occurred while writing the file [_1] for embedded element [_2].',
                                  $orig_uploaded_filename,$env{'form.embedded_orig_'.$i}).
                              '</span><br />';
                } else {
                    $output .= &mt('Uploaded [_1]','<span class="LC_filename">'.
                               $url.'</span>').'<br />';
                    unless ($context eq 'testbank') {
                        $footer .= &mt('View embedded file: [_1]',
                                       '<a href="'.$url.'">'.$fname.'</a>').'<br />';
                    }
                }
                close($fh);
            }
        }
        if ($env{'form.embedded_ref_'.$i}) {
            $pathchange{$i} = 1;
        }
    }
    if ($output) {
        $output = '<p>'.$output.'</p>';
    }
    $output .= &modify_html_form('upload_embedded',$actionurl,$hiddenstate,\%pathchange);
    $returnflag = 'ok';
    my $numpathchgs = scalar(keys(%pathchange));
    if ($numpathchgs > 0) {
        if ($context eq 'portfolio') {
            $output .= '<p>'.&mt('or').'</p>';
        } elsif ($context eq 'testbank') {
            $output .=  '<p>'.&mt('Or [_1]continue[_2] the testbank import without modifying the reference(s).',
                                  '<a href="javascript:document.testbankForm.submit();">','</a>').'</p>';
            $returnflag = 'modify_orightml';
        }
    }
    return ($output.$footer,$returnflag,$numpathchgs);
}

sub modify_html_form {
    my ($context,$actionurl,$hiddenstate,$pathchange,$pathchgtable) = @_;
    my $end = 0;
    my $modifyform;
    if ($context eq 'upload_embedded') {
        return unless (ref($pathchange) eq 'HASH');
        if ($env{'form.number_embedded_items'}) {
            $end += $env{'form.number_embedded_items'};
        }
        if ($env{'form.number_pathchange_items'}) {
            $end += $env{'form.number_pathchange_items'};
        }
        if ($end) {
            for (my $i=0; $i<$end; $i++) {
                if ($i < $env{'form.number_embedded_items'}) {
                    next unless($pathchange->{$i});
                }
                $modifyform .=
                    &start_data_table_row().
                    '<td><input type ="checkbox" name="namechange" value="'.$i.'" '.
                    'checked="checked" /></td>'.
                    '<td>'.$env{'form.embedded_ref_'.$i}.
                    '<input type="hidden" name="embedded_ref_'.$i.'" value="'.
                    &escape($env{'form.embedded_ref_'.$i}).'" />'.
                    '<input type="hidden" name="embedded_codebase_'.$i.'" value="'.
                    &escape($env{'form.embedded_codebase_'.$i}).'" />'.
                    '<input type="hidden" name="embedded_attrib_'.$i.'" value="'.
                    &escape($env{'form.embedded_attrib_'.$i}).'" /></td>'.
                    '<td>'.$env{'form.embedded_orig_'.$i}.
                    '<input type="hidden" name="embedded_orig_'.$i.'" value="'.
                    &escape($env{'form.embedded_orig_'.$i}).'" /></td>'.
                    &end_data_table_row();
            }
        }
    } else {
        $modifyform = $pathchgtable;
        if (($actionurl eq '/adm/upload') || ($actionurl eq '/adm/testbank')) {
            $hiddenstate .= '<input type="hidden" name="phase" value="four" />';
        } elsif (($actionurl eq '/adm/portfolio') || ($actionurl eq '/adm/coursegrp_portfolio')) {
            $hiddenstate .= '<input type="hidden" name="action" value="modify_orightml" />';
        }
    }
    if ($modifyform) {
        if ($actionurl eq '/adm/dependencies') {
            $hiddenstate .= '<input type="hidden" name="action" value="modifyhrefs" />';
        }
        return '<h3>'.&mt('Changes in content of HTML file required').'</h3>'."\n".
               '<p>'.&mt('Changes need to be made to the reference(s) used for one or more of the dependencies, if your HTML file is to work correctly:').'<ol>'."\n".
               '<li>'.&mt('For consistency between the reference(s) and the location of the corresponding stored file within LON-CAPA.').'</li>'."\n".
               '<li>'.&mt('To change absolute paths to relative paths, or replace directory traversal via "../" within the original reference.').'</li>'."\n".
               '</ol></p>'."\n".'<p>'.
               &mt('LON-CAPA can make the required changes to your HTML file.').'</p>'."\n".
               '<form method="post" name="refchanger" action="'.$actionurl.'">'.
               &start_data_table()."\n".
               &start_data_table_header_row().
               '<th>'.&mt('Change?').'</th>'.
               '<th>'.&mt('Current reference').'</th>'.
               '<th>'.&mt('Required reference').'</th>'.
               &end_data_table_header_row()."\n".
               $modifyform.
               &end_data_table().'<br />'."\n".$hiddenstate.
               '<input type="submit" name="pathchanges" value="'.&mt('Modify HTML file').'" />'.
               '</form>'."\n";
    }
    return;
}

sub modify_html_refs {
    my ($context,$dirpath,$uname,$udom,$dir_root,$url) = @_;
    my $container;
    if ($context eq 'portfolio') {
        $container = $env{'form.container'};
    } elsif ($context eq 'coursedoc') {
        $container = $env{'form.primaryurl'};
    } elsif ($context eq 'manage_dependencies') {
        (undef,undef,$container) = &Apache::lonnet::decode_symb($env{'form.symb'});
        $container = "/$container";
    } elsif ($context eq 'syllabus') {
        $container = $url;
    } else {
        $container = $Apache::lonnet::perlvar{'lonDocRoot'}.$env{'form.filename'};
    }
    my (%allfiles,%codebase,$output,$content);
    my @changes = &get_env_multiple('form.namechange');
    unless ((@changes > 0)  || ($context eq 'syllabus')) {
        if (wantarray) {
            return ('',0,0); 
        } else {
            return;
        }
    }
    if (($context eq 'portfolio') || ($context eq 'coursedoc') || 
        ($context eq 'manage_dependencies') || ($context eq 'syllabus')) {
        unless ($container =~ m{^/uploaded/\Q$udom\E/\Q$uname\E/}) {
            if (wantarray) {
                return ('',0,0);
            } else {
                return;
            }
        } 
        $content = &Apache::lonnet::getfile($container);
        if ($content eq '-1') {
            if (wantarray) {
                return ('',0,0);
            } else {
                return;
            }
        }
    } else {
        unless ($container =~ /^\Q$dir_root\E/) {
            if (wantarray) {
                return ('',0,0);
            } else {
                return;
            }
        } 
        if (open(my $fh,'<',$container)) {
            $content = join('', <$fh>);
            close($fh);
        } else {
            if (wantarray) {
                return ('',0,0);
            } else {
                return;
            }
        }
    }
    my ($count,$codebasecount) = (0,0);
    my $mm = new File::MMagic;
    my $mime_type = $mm->checktype_contents($content);
    if ($mime_type eq 'text/html') {
        my $parse_result = 
            &Apache::lonnet::extract_embedded_items($container,\%allfiles,
                                                    \%codebase,\$content);
        if ($parse_result eq 'ok') {
            foreach my $i (@changes) {
                my $orig = &unescape($env{'form.embedded_orig_'.$i});
                my $ref = &unescape($env{'form.embedded_ref_'.$i});
                if ($allfiles{$ref}) {
                    my $newname =  $orig;
                    my ($attrib_regexp,$codebase);
                    $attrib_regexp = &unescape($env{'form.embedded_attrib_'.$i});
                    if ($attrib_regexp =~ /:/) {
                        $attrib_regexp =~ s/\:/|/g;
                    }
                    if ($content =~ m{($attrib_regexp\s*=\s*['"]?)\Q$ref\E(['"]?)}) {
                        my $numchg = ($content =~ s{($attrib_regexp\s*=\s*['"]?)\Q$ref\E(['"]?)}{$1$newname$2}gi);
                        $count += $numchg;
                        $allfiles{$newname} = $allfiles{$ref};
                        delete($allfiles{$ref});
                    }
                    if ($env{'form.embedded_codebase_'.$i} ne '') {
                        $codebase = &unescape($env{'form.embedded_codebase_'.$i});
                        my $numchg = ($content =~ s/(codebase\s*=\s*["']?)\Q$codebase\E(["']?)/$1.$2/i); #' stupid emacs
                        $codebasecount ++;
                    }
                }
            }
            my $skiprewrites;
            if ($count || $codebasecount) {
                my $saveresult;
                if (($context eq 'portfolio') || ($context eq 'coursedoc') || 
                    ($context eq 'manage_dependencies') || ($context eq 'syllabus')) {
                    my $url = &Apache::lonnet::store_edited_file($container,$content,$udom,$uname,\$saveresult);
                    if ($url eq $container) {
                        my ($fname) = ($container =~ m{/([^/]+)$});
                        $output = '<p>'.&mt('Updated [quant,_1,reference] in [_2].',
                                            $count,'<span class="LC_filename">'.
                                            $fname.'</span>').'</p>';
                    } else {
                         $output = '<p class="LC_error">'.
                                   &mt('Error: update failed for: [_1].',
                                   '<span class="LC_filename">'.
                                   $container.'</span>').'</p>';
                    }
                    if ($context eq 'syllabus') {
                        unless ($saveresult eq 'ok') {
                            $skiprewrites = 1;
                        }
                    }
                } else {
                    if (open(my $fh,'>',$container)) {
                        print $fh $content;
                        close($fh);
                        $output = '<p>'.&mt('Updated [quant,_1,reference] in [_2].',
                                  $count,'<span class="LC_filename">'.
                                  $container.'</span>').'</p>';
                    } else {
                         $output = '<p class="LC_error">'.
                                   &mt('Error: could not update [_1].',
                                   '<span class="LC_filename">'.
                                   $container.'</span>').'</p>';
                    }
                }
            }
            if (($context eq 'syllabus') && (!$skiprewrites)) {
                my ($actionurl,$state);
                $actionurl = "/public/$udom/$uname/syllabus";
                my ($ignore,$num,$numpathchanges,$existing,$mapping) =
                    &ask_for_embedded_content($actionurl,$state,\%allfiles,
                                              \%codebase,
                                              {'context' => 'rewrites',
                                               'ignore_remote_references' => 1,});
                if (ref($mapping) eq 'HASH') {
                    my $rewrites = 0;
                    foreach my $key (keys(%{$mapping})) {
                        next if ($key =~ m{^https?://});
                        my $ref = $mapping->{$key};
                        my $newname = "/uploaded/$udom/$uname/portfolio/syllabus/$key";
                        my $attrib;
                        if (ref($allfiles{$mapping->{$key}}) eq 'ARRAY') {
                            $attrib = join('|',@{$allfiles{$mapping->{$key}}});
                        }
                        if ($content =~ m{($attrib\s*=\s*['"]?)\Q$ref\E(['"]?)}) {
                            my $numchg = ($content =~ s{($attrib\s*=\s*['"]?)\Q$ref\E(['"]?)}{$1$newname$2}gi);
                            $rewrites += $numchg;
                        }
                    }
                    if ($rewrites) {
                        my $saveresult;
                        my $url = &Apache::lonnet::store_edited_file($container,$content,$udom,$uname,\$saveresult);
                        if ($url eq $container) {
                            my ($fname) = ($container =~ m{/([^/]+)$});
                            $output .= '<p>'.&mt('Rewrote [quant,_1,link] as [quant,_1,absolute link] in [_2].',
                                            $count,'<span class="LC_filename">'.
                                            $fname.'</span>').'</p>';
                        } else {
                            $output .= '<p class="LC_error">'.
                                       &mt('Error: could not update links in [_1].',
                                       '<span class="LC_filename">'.
                                       $container.'</span>').'</p>';

                        }
                    }
                }
            }
        } else {
            &logthis('Failed to parse '.$container.
                     ' to modify references: '.$parse_result);
        }
    }
    if (wantarray) {
        return ($output,$count,$codebasecount);
    } else {
        return $output;
    }
}

sub check_for_existing {
    my ($path,$fname,$element) = @_;
    my ($state,$msg);
    if (-d $path.'/'.$fname) {
        $state = 'exists';
        $msg = &mt('Unable to upload [_1]. A directory by that name was found in [_2].','<span class="LC_filename">'.$fname.'</span>',$path);
    } elsif (-e $path.'/'.$fname) {
        $state = 'exists';
        $msg = &mt('Unable to upload [_1]. A file by that name was found in [_2].','<span class="LC_filename">'.$fname.'</span>',$path);
    }
    if ($state eq 'exists') {
        $msg = '<span class="LC_error">'.$msg.'</span><br />';
    }
    return ($state,$msg);
}

sub check_for_upload {
    my ($path,$fname,$group,$element,$portfolio_root,$port_path,
        $disk_quota,$current_disk_usage,$uname,$udom) = @_;
    my $filesize = length($env{'form.'.$element});
    if (!$filesize) {
        my $msg = '<span class="LC_error">'.
                  &mt('Unable to upload [_1]. (size = [_2] bytes)', 
                      '<span class="LC_filename">'.$fname.'</span>',
                      $filesize).'<br />'.
                  &mt('Either the file you attempted to upload was empty, or your web browser was unable to read its contents.').'<br />'.
                  '</span>';
        return ('zero_bytes',$msg);
    }
    $filesize =  $filesize/1000; #express in k (1024?)
    my $getpropath = 1;
    my ($dirlistref,$listerror) =
         &Apache::lonnet::dirlist($portfolio_root.$path,$udom,$uname,$getpropath);
    my $found_file = 0;
    my $locked_file = 0;
    my @lockers;
    my $navmap;
    if ($env{'request.course.id'}) {
        $navmap = Apache::lonnavmaps::navmap->new();
    }
    if (ref($dirlistref) eq 'ARRAY') {
        foreach my $line (@{$dirlistref}) {
            my ($file_name,$rest)=split(/\&/,$line,2);
            if ($file_name eq $fname){
                $file_name = $path.$file_name;
                if ($group ne '') {
                    $file_name = $group.$file_name;
                }
                $found_file = 1;
                if (&Apache::lonnet::is_locked($file_name,$udom,$uname,\@lockers) eq 'true') {
                    foreach my $lock (@lockers) {
                        if (ref($lock) eq 'ARRAY') {
                            my ($symb,$crsid) = @{$lock};
                            if ($crsid eq $env{'request.course.id'}) {
                                if (ref($navmap)) {
                                    my $res = $navmap->getBySymb($symb);
                                    foreach my $part (@{$res->parts()}) { 
                                        my ($slot_status,$slot_time,$slot_name)=$res->check_for_slot($part);
                                        unless (($slot_status == $res->RESERVED) ||
                                                ($slot_status == $res->RESERVED_LOCATION)) {
                                            $locked_file = 1;
                                        }
                                    }
                                } else {
                                    $locked_file = 1;
                                }
                            } else {
                                $locked_file = 1;
                            }
                        }
                   }
                } else {
                    my @info = split(/\&/,$rest);
                    my $currsize = $info[6]/1000;
                    if ($currsize < $filesize) {
                        my $extra = $filesize - $currsize;
                        if (($current_disk_usage + $extra) > $disk_quota) {
                            my $msg = '<p class="LC_warning">'.
                                      &mt('Unable to upload [_1]. (size = [_2] kilobytes). Disk quota will be exceeded if existing (smaller) file with same name (size = [_3] kilobytes) is replaced.',
                                          '<span class="LC_filename">'.$fname.'</span>',$filesize,$currsize).'</p>'.
                                      '<p>'.&mt('Disk quota is [_1] kilobytes. Your current disk usage is [_2] kilobytes.',
                                                   $disk_quota,$current_disk_usage).'</p>';
                            return ('will_exceed_quota',$msg);
                        }
                    }
                }
            }
        }
    }
    if (($current_disk_usage + $filesize) > $disk_quota){
        my $msg = '<p class="LC_warning">'.
                &mt('Unable to upload [_1]. (size = [_2] kilobytes). Disk quota will be exceeded.','<span class="LC_filename">'.$fname.'</span>',$filesize).'</p>'.
                  '<p>'.&mt('Disk quota is [_1] kilobytes. Your current disk usage is [_2] kilobytes.',$disk_quota,$current_disk_usage).'</p>';
        return ('will_exceed_quota',$msg);
    } elsif ($found_file) {
        if ($locked_file) {
            my $msg = '<p class="LC_warning">';
            $msg .= &mt('Unable to upload [_1]. A locked file by that name was found in [_2].','<span class="LC_filename">'.$fname.'</span>','<span class="LC_filename">'.$port_path.$env{'form.currentpath'}.'</span>');
            $msg .= '</p>';
            $msg .= &mt('You will be able to rename or delete existing [_1] after a grade has been assigned.','<span class="LC_filename">'.$fname.'</span>');
            return ('file_locked',$msg);
        } else {
            my $msg = '<p class="LC_error">';
            $msg .= &mt(' A file by that name: [_1] was found in [_2].','<span class="LC_filename">'.$fname.'</span>',$port_path.$env{'form.currentpath'});
            $msg .= '</p>';
            return ('existingfile',$msg);
        }
    }
}

sub check_for_traversal {
    my ($path,$url,$toplevel) = @_;
    my @parts=split(/\//,$path);
    my $cleanpath;
    my $fullpath = $url;
    for (my $i=0;$i<@parts;$i++) {
        next if ($parts[$i] eq '.');
        if ($parts[$i] eq '..') {
            $fullpath =~ s{([^/]+/)$}{};
        } else {
            $fullpath .= $parts[$i].'/';
        }
    }
    if ($fullpath =~ /^\Q$url\E(.*)$/) {
        $cleanpath = $1;
    } elsif ($fullpath =~ /^\Q$toplevel\E(.*)$/) {
        my $curr_toprel = $1;
        my @parts = split(/\//,$curr_toprel);
        my ($url_toprel) = ($url =~ /^\Q$toplevel\E(.*)$/);
        my @urlparts = split(/\//,$url_toprel);
        my $doubledots;
        my $startdiff = -1;
        for (my $i=0; $i<@urlparts; $i++) {
            if ($startdiff == -1) {
                unless ($urlparts[$i] eq $parts[$i]) {
                    $startdiff = $i;
                    $doubledots .= '../';
                }
            } else {
                $doubledots .= '../';
            }
        }
        if ($startdiff > -1) {
            $cleanpath = $doubledots;
            for (my $i=$startdiff; $i<@parts; $i++) {
                $cleanpath .= $parts[$i].'/';
            }
        }
    }
    $cleanpath =~ s{(/)$}{};
    return $cleanpath;
}

sub is_archive_file {
    my ($mimetype) = @_;
    if (($mimetype eq 'application/octet-stream') ||
        ($mimetype eq 'application/x-stuffit') ||
        ($mimetype =~ m{^application/(x\-)?(compressed|tar|zip|tgz|gz|gtar|gzip|gunzip|bz|bz2|bzip2)})) {
        return 1;
    }
    return;
}

sub decompress_form {
    my ($mimetype,$archiveurl,$action,$noextract,$hiddenelements,$dirlist) = @_;
    my %lt = &Apache::lonlocal::texthash (
        this => 'This file is an archive file.',
        camt => 'This file is a Camtasia archive file.',
        itsc => 'Its contents are as follows:',
        youm => 'You may wish to extract its contents.',
        extr => 'Extract contents',
        auto => 'LON-CAPA can process the files automatically, or you can decide how each should be handled.',
        proa => 'Process automatically?',
        yes  => 'Yes',
        no   => 'No',
        fold => 'Title for folder containing movie',
        movi => 'Title for page containing embedded movie', 
    );
    my $fileloc = &Apache::lonnet::filelocation(undef,$archiveurl);
    my ($is_camtasia,$topdir,%toplevel,@paths);
    my $info = &list_archive_contents($fileloc,\@paths);
    if (@paths) {
        foreach my $path (@paths) {
            $path =~ s{^/}{};
            if ($path =~ m{^([^/]+)/$}) {
                $topdir = $1;
            }
            if ($path =~ m{^([^/]+)/}) {
                $toplevel{$1} = $path;
            } else {
                $toplevel{$path} = $path;
            }
        }
    }
    if ($mimetype =~ m{^application/(x\-)?(compressed|zip)}) {
        my @camtasia6 = ("$topdir/","$topdir/index.html",
                        "$topdir/media/",
                        "$topdir/media/$topdir.mp4",
                        "$topdir/media/FirstFrame.png",
                        "$topdir/media/player.swf",
                        "$topdir/media/swfobject.js",
                        "$topdir/media/expressInstall.swf");
        my @camtasia8_1 = ("$topdir/","$topdir/$topdir.html",
                         "$topdir/$topdir.mp4",
                         "$topdir/$topdir\_config.xml",
                         "$topdir/$topdir\_controller.swf",
                         "$topdir/$topdir\_embed.css",
                         "$topdir/$topdir\_First_Frame.png",
                         "$topdir/$topdir\_player.html",
                         "$topdir/$topdir\_Thumbnails.png",
                         "$topdir/playerProductInstall.swf",
                         "$topdir/scripts/",
                         "$topdir/scripts/config_xml.js",
                         "$topdir/scripts/handlebars.js",
                         "$topdir/scripts/jquery-1.7.1.min.js",
                         "$topdir/scripts/jquery-ui-1.8.15.custom.min.js",
                         "$topdir/scripts/modernizr.js",
                         "$topdir/scripts/player-min.js",
                         "$topdir/scripts/swfobject.js",
                         "$topdir/skins/",
                         "$topdir/skins/configuration_express.xml",
                         "$topdir/skins/express_show/",
                         "$topdir/skins/express_show/player-min.css",
                         "$topdir/skins/express_show/spritesheet.png");
        my @camtasia8_4 = ("$topdir/","$topdir/$topdir.html",
                         "$topdir/$topdir.mp4",
                         "$topdir/$topdir\_config.xml",
                         "$topdir/$topdir\_controller.swf",
                         "$topdir/$topdir\_embed.css",
                         "$topdir/$topdir\_First_Frame.png",
                         "$topdir/$topdir\_player.html",
                         "$topdir/$topdir\_Thumbnails.png",
                         "$topdir/playerProductInstall.swf",
                         "$topdir/scripts/",
                         "$topdir/scripts/config_xml.js",
                         "$topdir/scripts/techsmith-smart-player.min.js",
                         "$topdir/skins/",
                         "$topdir/skins/configuration_express.xml",
                         "$topdir/skins/express_show/",
                         "$topdir/skins/express_show/spritesheet.min.css",
                         "$topdir/skins/express_show/spritesheet.png",
                         "$topdir/skins/express_show/techsmith-smart-player.min.css");
        my @diffs = &compare_arrays(\@paths,\@camtasia6);
        if (@diffs == 0) {
            $is_camtasia = 6;
        } else {
            @diffs = &compare_arrays(\@paths,\@camtasia8_1);
            if (@diffs == 0) {
                $is_camtasia = 8;
            } else {
                @diffs = &compare_arrays(\@paths,\@camtasia8_4);
                if (@diffs == 0) {
                    $is_camtasia = 8;
                }
            }
        }
    }
    my $output;
    if ($is_camtasia) {
        $output = <<"ENDCAM";
<script type="text/javascript" language="Javascript">
// <![CDATA[

function camtasiaToggle() {
    for (var i=0; i<document.uploaded_decompress.autoextract_camtasia.length; i++) {
        if (document.uploaded_decompress.autoextract_camtasia[i].checked) {
            if (document.uploaded_decompress.autoextract_camtasia[i].value == $is_camtasia) {
                document.getElementById('camtasia_titles').style.display='block';
            } else {
                document.getElementById('camtasia_titles').style.display='none';
            }
        }
    }
    return;
}

// ]]>
</script>
<p>$lt{'camt'}</p>
ENDCAM
    } else {
        $output = '<p>'.$lt{'this'};
        if ($info eq '') {
            $output .= ' '.$lt{'youm'}.'</p>'."\n";
        } else {
            $output .= ' '.$lt{'itsc'}.'</p>'."\n".
                       '<div><pre>'.$info.'</pre></div>';
        }
    }
    $output .= '<form name="uploaded_decompress" action="'.$action.'" method="post">'."\n";
    my $duplicates;
    my $num = 0;
    if (ref($dirlist) eq 'ARRAY') {
        foreach my $item (@{$dirlist}) {
            if (ref($item) eq 'ARRAY') {
                if (exists($toplevel{$item->[0]})) {
                    $duplicates .= 
                        &start_data_table_row().
                        '<td><label><input type="radio" name="archive_overwrite_'.$num.'" '.
                        'value="0" checked="checked" />'.&mt('No').'</label>'.
                        '&nbsp;<label><input type="radio" name="archive_overwrite_'.$num.'" '.
                        'value="1" />'.&mt('Yes').'</label>'.
                        '<input type="hidden" name="archive_overwrite_name_'.$num.'" value="'.$item->[0].'" /></td>'."\n".
                        '<td>'.$item->[0].'</td>';
                    if ($item->[2]) {
                        $duplicates .= '<td>'.&mt('Directory').'</td>';
                    } else {
                        $duplicates .= '<td>'.&mt('File').'</td>';
                    }
                    $duplicates .= '<td>'.$item->[3].'</td>'.
                                   '<td>'.
                                   &Apache::lonlocal::locallocaltime($item->[4]).
                                   '</td>'.
                                   &end_data_table_row();
                    $num ++;
                }
            }
        }
    }
    my $itemcount;
    if (@paths > 0) {
        $itemcount = scalar(@paths);
    } else {
        $itemcount = 1;
    }
    if ($is_camtasia) {
        $output .= $lt{'auto'}.'<br />'.
                   '<span class="LC_nobreak">'.$lt{'proa'}.'<label>'.
                   '<input type="radio" name="autoextract_camtasia" value="'.$is_camtasia.'" onclick="javascript:camtasiaToggle();" checked="checked" />'.
                   $lt{'yes'}.'</label>&nbsp;<label>'.
                   '<input type="radio" name="autoextract_camtasia" value="0" onclick="javascript:camtasiaToggle();" />'.
                   $lt{'no'}.'</label></span><br />'.
                   '<div id="camtasia_titles" style="display:block">'.
                   &Apache::lonhtmlcommon::start_pick_box().
                   &Apache::lonhtmlcommon::row_title($lt{'fold'}).
                   '<input type="textbox" name="camtasia_foldername" value="'.$env{'form.comment'}.'" />'."\n".
                   &Apache::lonhtmlcommon::row_closure().
                   &Apache::lonhtmlcommon::row_title($lt{'movi'}).
                   '<input type="textbox" name="camtasia_moviename" value="" />'."\n".
                   &Apache::lonhtmlcommon::row_closure(1).
                   &Apache::lonhtmlcommon::end_pick_box().
                   '</div>';
    }
    $output .= 
        '<input type="hidden" name="archive_overwrite_total" value="'.$num.'" />'.
        '<input type="hidden" name="archive_itemcount" value="'.$itemcount.'" />'.
        "\n";
    if ($duplicates ne '') {
        $output .= '<p><span class="LC_warning">'.
                   &mt('Warning: decompression of the archive will overwrite the following items which already exist:').'</span><br />'.  
                   &start_data_table().
                   &start_data_table_header_row().
                   '<th>'.&mt('Overwrite?').'</th>'.
                   '<th>'.&mt('Name').'</th>'.
                   '<th>'.&mt('Type').'</th>'.
                   '<th>'.&mt('Size').'</th>'.
                   '<th>'.&mt('Last modified').'</th>'.
                   &end_data_table_header_row().
                   $duplicates.
                   &end_data_table().
                   '</p>';
    }
    $output .= '<input type="hidden" name="archiveurl" value="'.$archiveurl.'" />'."\n";
    if (ref($hiddenelements) eq 'HASH') {
        foreach my $hidden (sort(keys(%{$hiddenelements}))) {
            $output .= '<input type="hidden" name="'.$hidden.'" value="'.$hiddenelements->{$hidden}.'" />'."\n";
        }
    }
    $output .= <<"END";
<br />
<input type="submit" name="decompress" value="$lt{'extr'}" />
</form>
$noextract
END
    return $output;
}

sub decompression_utility {
    my ($program) = @_;
    my @utilities = ('tar','gunzip','bunzip2','unzip'); 
    my $location;
    if (grep(/^\Q$program\E$/,@utilities)) { 
        foreach my $dir ('/bin/','/usr/bin/','/usr/local/bin/','/sbin/',
                         '/usr/sbin/') {
            if (-x $dir.$program) {
                $location = $dir.$program;
                last;
            }
        }
    }
    return $location;
}

sub list_archive_contents {
    my ($file,$pathsref) = @_;
    my (@cmd,$output);
    my $needsregexp;
    if ($file =~ /\.zip$/) {
        @cmd = (&decompression_utility('unzip'),"-l");
        $needsregexp = 1;
    } elsif (($file =~ m/\.tar\.gz$/) ||
             ($file =~ /\.tgz$/)) {
        @cmd = (&decompression_utility('tar'),"-ztf");
    } elsif ($file =~ /\.tar\.bz2$/) {
        @cmd = (&decompression_utility('tar'),"-jtf");
    } elsif ($file =~ m|\.tar$|) {
        @cmd = (&decompression_utility('tar'),"-tf");
    }
    if (@cmd) {
        undef($!);
        undef($@);
        if (open(my $fh,"-|", @cmd, $file)) {
            while (my $line = <$fh>) {
                $output .= $line;
                chomp($line);
                my $item;
                if ($needsregexp) {
                    ($item) = ($line =~ /^\s*\d+\s+[\d\-]+\s+[\d:]+\s*(.+)$/); 
                } else {
                    $item = $line;
                }
                if ($item ne '') {
                    unless (grep(/^\Q$item\E$/,@{$pathsref})) {
                        push(@{$pathsref},$item);
                    } 
                }
            }
            close($fh);
        }
    }
    return $output;
}

sub decompress_uploaded_file {
    my ($file,$dir) = @_;
    &Apache::lonnet::appenv({'cgi.file' => $file});
    &Apache::lonnet::appenv({'cgi.dir' => $dir});
    my $result = &Apache::lonnet::ssi_body('/cgi-bin/decompress.pl');
    my ($handle) = ($env{'user.environment'} =~m{/([^/]+)\.id$});
    my $lonidsdir = $Apache::lonnet::perlvar{'lonIDsDir'};
    &Apache::lonnet::transfer_profile_to_env($lonidsdir,$handle,1);
    my $decompressed = $env{'cgi.decompressed'};
    &Apache::lonnet::delenv('cgi.file');
    &Apache::lonnet::delenv('cgi.dir');
    &Apache::lonnet::delenv('cgi.decompressed');
    return ($decompressed,$result);
}

sub process_decompression {
    my ($docudom,$docuname,$file,$destination,$dir_root,$hiddenelem) = @_;
    unless (($dir_root eq '/userfiles') && ($destination =~ m{^(docs|supplemental)/(default|\d+)/\d+$})) {
        return '<p class="LC_error">'.&mt('Not extracted.').'<br />'.
               &mt('Unexpected file path.').'</p>'."\n";
    }
    unless (($docudom =~ /^$match_domain$/) && ($docuname =~ /^$match_courseid$/)) {
        return '<p class="LC_error">'.&mt('Not extracted.').'<br />'.
               &mt('Unexpected course context.').'</p>'."\n";
    }
    unless ($file eq &Apache::lonnet::clean_filename($file)) {
        return '<p class="LC_error">'.&mt('Not extracted.').'<br />'.
               &mt('Filename contained unexpected characters.').'</p>'."\n";
    }
    my ($dir,$error,$warning,$output);
    if ($file !~ /\.(zip|tar|bz2|gz|tar.gz|tar.bz2|tgz)$/i) {
        $error = &mt('Filename not a supported archive file type.').
                 '<br />'.&mt('Filename should end with one of: [_1].',
                              '.zip, .tar, .bz2, .gz, .tar.gz, .tar.bz2, .tgz');
    } else {
        my $docuhome = &Apache::lonnet::homeserver($docuname,$docudom);
        if ($docuhome eq 'no_host') {
            $error = &mt('Could not determine home server for course.');
        } else {
            my @ids=&Apache::lonnet::current_machine_ids();
            my $currdir = "$dir_root/$destination";
            if (grep(/^\Q$docuhome\E$/,@ids)) {
                $dir = &LONCAPA::propath($docudom,$docuname).
                       "$dir_root/$destination";
            } else {
                $dir = $Apache::lonnet::perlvar{'lonDocRoot'}.
                       "$dir_root/$docudom/$docuname/$destination";
                unless (&Apache::lonnet::repcopy_userfile("$dir/$file") eq 'ok') {
                    $error = &mt('Archive file not found.');
                }
            }
            my (@to_overwrite,@to_skip);
            if ($env{'form.archive_overwrite_total'} > 0) {
                my $total = $env{'form.archive_overwrite_total'};
                for (my $i=0; $i<$total; $i++) {
                    if ($env{'form.archive_overwrite_'.$i} == 1) {
                        push(@to_overwrite,$env{'form.archive_overwrite_name_'.$i});
                    } elsif ($env{'form.archive_overwrite_'.$i} == 0) {
                        push(@to_skip,$env{'form.archive_overwrite_name_'.$i});
                    }
                }
            }
            my $numskip = scalar(@to_skip);
            my $numoverwrite = scalar(@to_overwrite);
            if (($numskip) && (!$numoverwrite)) {
                $warning = &mt('All items in the archive file already exist, and no overwriting of existing files has been requested.');         
            } elsif ($dir eq '') {
                $error = &mt('Directory containing archive file unavailable.');
            } elsif (!$error) {
                my ($decompressed,$display);
                if (($numskip) || ($numoverwrite)) {
                    my $tempdir = time.'_'.$$.int(rand(10000));
                    mkdir("$dir/$tempdir",0755);
                    if (&File::Copy::move("$dir/$file","$dir/$tempdir/$file")) {
                        ($decompressed,$display) =
                            &decompress_uploaded_file($file,"$dir/$tempdir");
                        foreach my $item (@to_skip) {
                            if (($item ne '') && ($item !~ /\.\./)) {
                                if (-f "$dir/$tempdir/$item") {
                                    unlink("$dir/$tempdir/$item");
                                } elsif (-d "$dir/$tempdir/$item") {
                                    &File::Path::remove_tree("$dir/$tempdir/$item",{ safe => 1 });
                                }
                            }
                        }
                        foreach my $item (@to_overwrite) {
                            if ((-e "$dir/$tempdir/$item") && (-e "$dir/$item")) {
                                if (($item ne '') && ($item !~ /\.\./)) {
                                    if (-f "$dir/$item") {
                                        unlink("$dir/$item");
                                    } elsif (-d "$dir/$item") {
                                        &File::Path::remove_tree("$dir/$item",{ safe => 1 });
                                    }
                                    &File::Copy::move("$dir/$tempdir/$item","$dir/$item");
                                }
                            }
                        }
                        if (&File::Copy::move("$dir/$tempdir/$file","$dir/$file")) {
                            &File::Path::remove_tree("$dir/$tempdir",{ safe => 1 });
                        }
                    }
                } else {
                    ($decompressed,$display) = 
                        &decompress_uploaded_file($file,$dir);
                }
                if ($decompressed eq 'ok') {
                    $output = '<p class="LC_info">'.
                              &mt('Files extracted successfully from archive.').
                              '</p>'."\n";
                    my ($warning,$result,@contents);
                    my ($newdirlistref,$newlisterror) =
                        &Apache::lonnet::dirlist($currdir,$docudom,
                                                 $docuname,1);
                    my (%is_dir,%changes,@newitems);
                    my $dirptr = 16384;
                    if (ref($newdirlistref) eq 'ARRAY') {
                        foreach my $dir_line (@{$newdirlistref}) {
                            my ($item,undef,undef,$testdir)=split(/\&/,$dir_line,5);
                            unless (($item =~ /^\.+$/) || ($item eq $file)) { 
                                push(@newitems,$item);
                                if ($dirptr&$testdir) {
                                    $is_dir{$item} = 1;
                                }
                                $changes{$item} = 1;
                            }
                        }
                    }
                    if (keys(%changes) > 0) {
                        foreach my $item (sort(@newitems)) {
                            if ($changes{$item}) {
                                push(@contents,$item);
                            }
                        }
                    }
                    if (@contents > 0) {
                        my $wantform;
                        unless ($env{'form.autoextract_camtasia'}) {
                            $wantform = 1;
                        }
                        my (%children,%parent,%dirorder,%titles);
                        my ($count,$datatable) = &get_extracted($docudom,$docuname,
                                                                $currdir,\%is_dir,
                                                                \%children,\%parent,
                                                                \@contents,\%dirorder,
                                                                \%titles,$wantform);
                        if ($datatable ne '') {
                            $output .= &archive_options_form('decompressed',$datatable,
                                                             $count,$hiddenelem);
                            my $startcount = 6;
                            $output .= &archive_javascript($startcount,$count,
                                                           \%titles,\%children);
                        }
                        if ($env{'form.autoextract_camtasia'}) {
                            my $version = $env{'form.autoextract_camtasia'};
                            my %displayed;
                            my $total = 1;
                            $env{'form.archive_directory'} = [];
                            foreach my $i (sort { $a <=> $b } keys(%dirorder)) {
                                my $path = join('/',map { $titles{$_}; } @{$dirorder{$i}});
                                $path =~ s{/$}{};
                                my $item;
                                if ($path ne '') {
                                    $item = "$path/$titles{$i}";
                                } else {
                                    $item = $titles{$i};
                                }
                                $env{'form.archive_content_'.$i} = "$dir_root/$destination/$item";
                                if ($item eq $contents[0]) {
                                    push(@{$env{'form.archive_directory'}},$i);
                                    $env{'form.archive_'.$i} = 'display';
                                    $env{'form.archive_title_'.$i} = $env{'form.camtasia_foldername'};
                                    $displayed{'folder'} = $i;
                                } elsif ((($item eq "$contents[0]/index.html") && ($version == 6)) ||
                                         (($item eq "$contents[0]/$contents[0]".'.html') && ($version == 8))) {
                                    $env{'form.archive_'.$i} = 'display';
                                    $env{'form.archive_title_'.$i} = $env{'form.camtasia_moviename'};
                                    $displayed{'web'} = $i;
                                } else {
                                    if ((($item eq "$contents[0]/media") && ($version == 6)) ||
                                        ((($item eq "$contents[0]/scripts") || ($item eq "$contents[0]/skins") ||
                                             ($item eq "$contents[0]/skins/express_show")) && ($version == 8))) {
                                        push(@{$env{'form.archive_directory'}},$i);
                                    }
                                    $env{'form.archive_'.$i} = 'dependency';
                                }
                                $total ++;
                            }
                            for (my $i=1; $i<$total; $i++) {
                                next if ($i == $displayed{'web'});
                                next if ($i == $displayed{'folder'});
                                $env{'form.archive_dependent_on_'.$i} = $displayed{'web'};
                            }
                            $env{'form.phase'} = 'decompress_cleanup';
                            $env{'form.archivedelete'} = 1;
                            $env{'form.archive_count'} = $total-1;
                            $output .=
                                &process_extracted_files('coursedocs',$docudom,
                                                         $docuname,$destination,
                                                         $dir_root,$hiddenelem);
                        }
                    } else {
                        $warning = &mt('No new items extracted from archive file.');
                    }
                } else {
                    $output = $display;
                    $error = &mt('An error occurred during extraction from the archive file.');
                }
            }
        }
    }
    if ($error) {
        $output .= '<p class="LC_error">'.&mt('Not extracted.').'<br />'.
                   $error.'</p>'."\n";
    }
    if ($warning) {
        $output .= '<p class="LC_warning">'.$warning.'</p>'."\n";
    }
    return $output;
}

sub get_extracted {
    my ($docudom,$docuname,$currdir,$is_dir,$children,$parent,$contents,$dirorder,
        $titles,$wantform) = @_;
    my $count = 0;
    my $depth = 0;
    my $datatable;
    my @hierarchy;
    return unless ((ref($is_dir) eq 'HASH') && (ref($children) eq 'HASH') &&
                   (ref($parent) eq 'HASH') && (ref($contents) eq 'ARRAY') &&
                   (ref($dirorder) eq 'HASH') && (ref($titles) eq 'HASH'));
    foreach my $item (@{$contents}) {
        $count ++;
        @{$dirorder->{$count}} = @hierarchy;
        $titles->{$count} = $item;
        &archive_hierarchy($depth,$count,$parent,$children);
        if ($wantform) {
            $datatable .= &archive_row($is_dir->{$item},$item,
                                       $currdir,$depth,$count);
        }
        if ($is_dir->{$item}) {
            $depth ++;
            push(@hierarchy,$count);
            $parent->{$depth} = $count;
            $datatable .=
                &recurse_extracted_archive("$currdir/$item",$docudom,$docuname,
                                           \$depth,\$count,\@hierarchy,$dirorder,
                                           $children,$parent,$titles,$wantform);
            $depth --;
            pop(@hierarchy);
        }
    }
    return ($count,$datatable);
}

sub recurse_extracted_archive {
    my ($currdir,$docudom,$docuname,$depth,$count,$hierarchy,$dirorder,
        $children,$parent,$titles,$wantform) = @_;
    my $result='';
    unless ((ref($depth)) && (ref($count)) && (ref($hierarchy) eq 'ARRAY') &&
            (ref($children) eq 'HASH') && (ref($parent) eq 'HASH') &&
            (ref($dirorder) eq 'HASH')) {
        return $result;
    }
    my $dirptr = 16384;
    my ($newdirlistref,$newlisterror) =
        &Apache::lonnet::dirlist($currdir,$docudom,$docuname,1);
    if (ref($newdirlistref) eq 'ARRAY') {
        foreach my $dir_line (@{$newdirlistref}) {
            my ($item,undef,undef,$testdir)=split(/\&/,$dir_line,5);
            unless ($item =~ /^\.+$/) {
                $$count ++;
                @{$dirorder->{$$count}} = @{$hierarchy};
                $titles->{$$count} = $item;
                &archive_hierarchy($$depth,$$count,$parent,$children);

                my $is_dir;
                if ($dirptr&$testdir) {
                    $is_dir = 1;
                }
                if ($wantform) {
                    $result .= &archive_row($is_dir,$item,$currdir,$$depth,$$count);
                }
                if ($is_dir) {
                    $$depth ++;
                    push(@{$hierarchy},$$count);
                    $parent->{$$depth} = $$count;
                    $result .=
                        &recurse_extracted_archive("$currdir/$item",$docudom,
                                                   $docuname,$depth,$count,
                                                   $hierarchy,$dirorder,$children,
                                                   $parent,$titles,$wantform);
                    $$depth --;
                    pop(@{$hierarchy});
                }
            }
        }
    }
    return $result;
}

sub archive_hierarchy {
    my ($depth,$count,$parent,$children) =@_;
    if ((ref($parent) eq 'HASH') && (ref($children) eq 'HASH')) {
        if (exists($parent->{$depth})) {
             $children->{$parent->{$depth}} .= $count.':';
        }
    }
    return;
}

sub archive_row {
    my ($is_dir,$item,$currdir,$depth,$count) = @_;
    my ($name) = ($item =~ m{([^/]+)$});
    my %choices = &Apache::lonlocal::texthash (
                                       'display'    => 'Add as file',
                                       'dependency' => 'Include as dependency',
                                       'discard'    => 'Discard',
                                      );
    if ($is_dir) {
        $choices{'display'} = &mt('Add as folder'); 
    }
    my $output = &start_data_table_row().'<td align="right">'.$count.'</td>'."\n";
    my $offset = 0;
    foreach my $action ('display','dependency','discard') {
        $offset ++;
        if ($action ne 'display') {
            $offset ++;
        }  
        $output .= '<td><span class="LC_nobreak">'.
                   '<label><input type="radio" name="archive_'.$count.
                   '" id="archive_'.$action.'_'.$count.'" value="'.$action.'"';
        my $text = $choices{$action};
        if ($is_dir) {
            $output .= ' onclick="javascript:propagateCheck(this.form,'."'$count'".');"';
            if ($action eq 'display') {
                $text = &mt('Add as folder');
            }
        } else {
            $output .= ' onclick="javascript:dependencyCheck(this.form,'."$count,$offset".');"';

        }
        $output .= ' />&nbsp;'.$choices{$action}.'</label></span>';
        if ($action eq 'dependency') {
            $output .= '<div id="arc_depon_'.$count.'" style="display:none;">'."\n".
                       &mt('Used by:').'&nbsp;<select name="archive_dependent_on_'.$count.'" '.
                       'onchange="propagateSelect(this.form,'."$count,$offset".')">'."\n".
                       '<option value=""></option>'."\n".
                       '</select>'."\n".
                       '</div>';
        } elsif ($action eq 'display') {
            $output .= '<div id="arc_title_'.$count.'" style="display:none;">'."\n".
                       &mt('Title:').'&nbsp;<input type="text" name="archive_title_'.$count.'" id="archive_title_'.$count.'" />'."\n".
                       '</div>';
        }
        $output .= '</td>';
    }
    $output .= '<td><input type="hidden" name="archive_content_'.$count.'" value="'.
               &HTML::Entities::encode("$currdir/$item",'"<>&').'" />'.('&nbsp;' x 2);
    for (my $i=0; $i<$depth; $i++) {
        $output .= ('<img src="/adm/lonIcons/whitespace1.gif" class="LC_docs_spacer" alt="" />' x2)."\n";
    }
    if ($is_dir) {
        $output .= '<img src="/adm/lonIcons/navmap.folder.open.gif" alt="" />&nbsp;'."\n".
                   '<input type="hidden" name="archive_directory" value="'.$count.'" />'."\n";
    } else {
        $output .= '<input type="hidden" name="archive_file" value="'.$count.'" />'."\n";
    }
    $output .= '&nbsp;'.$name.'</td>'."\n".
               &end_data_table_row();
    return $output;
}

sub archive_options_form {
    my ($form,$display,$count,$hiddenelem) = @_;
    my %lt = &Apache::lonlocal::texthash(
               perm => 'Permanently remove archive file?',
               hows => 'How should each extracted item be incorporated in the course?',
               cont => 'Content actions for all',
               addf => 'Add as folder/file',
               incd => 'Include as dependency for a displayed file',
               disc => 'Discard',
               no   => 'No',
               yes  => 'Yes',
               save => 'Save',
    );
    my $output = <<"END";
<form name="$form" method="post" action="">
<p><span class="LC_nobreak">$lt{'perm'}&nbsp;
<label>
  <input type="radio" name="archivedelete" value="0" checked="checked" />$lt{'no'}
</label>
&nbsp;
<label>
  <input type="radio" name="archivedelete" value="1" />$lt{'yes'}</label>
</span>
</p>
<input type="hidden" name="phase" value="decompress_cleanup" />
<br />$lt{'hows'}
<div class="LC_columnSection">
  <fieldset>
    <legend>$lt{'cont'}</legend>
    <input type="button" value="$lt{'addf'}" onclick="javascript:checkAll(document.$form,'display');" /> 
    &nbsp;&nbsp;<input type="button" value="$lt{'incd'}" onclick="javascript:checkAll(document.$form,'dependency');" />
    &nbsp;&nbsp;<input type="button" value="$lt{'disc'}" onclick="javascript:checkAll(document.$form,'discard');" />
  </fieldset>
</div>
END
    return $output.
           &start_data_table()."\n".
           $display."\n".
           &end_data_table()."\n".
           '<input type="hidden" name="archive_count" value="'.$count.'" />'.
           $hiddenelem.
           '<br /><input type="submit" name="archive_submit" value="'.$lt{'save'}.'" />'.
           '</form>';
}

sub archive_javascript {
    my ($startcount,$numitems,$titles,$children) = @_;
    return unless ((ref($titles) eq 'HASH') && (ref($children) eq 'HASH'));
    my $maintitle = $env{'form.comment'};
    my $scripttag = <<START;
<script type="text/javascript">
// <![CDATA[

function checkAll(form,prefix) {
    var idstr =  new RegExp("^archive_"+prefix+"_\\\\d+\$");
    for (var i=0; i < form.elements.length; i++) {
        var id = form.elements[i].id;
        if ((id != '') && (id != undefined)) {
            if (idstr.test(id)) {
                if (form.elements[i].type == 'radio') {
                    form.elements[i].checked = true;
                    var nostart = i-$startcount;
                    var offset = nostart%7;
                    var count = (nostart-offset)/7;    
                    dependencyCheck(form,count,offset);
                }
            }
        }
    }
}

function propagateCheck(form,count) {
    if (count > 0) {
        var startelement = $startcount + ((count-1) * 7);
        for (var j=1; j<6; j++) {
            if ((j != 2) && (j != 4)) {
                var item = startelement + j; 
                if (form.elements[item].type == 'radio') {
                    if (form.elements[item].checked) {
                        containerCheck(form,count,j);
                        break;
                    }
                }
            }
        }
    }
}

numitems = $numitems
var titles = new Array(numitems);
var parents = new Array(numitems);
for (var i=0; i<numitems; i++) {
    parents[i] = new Array;
}
var maintitle = '$maintitle';

START

    foreach my $container (sort { $a <=> $b } (keys(%{$children}))) {
        my @contents = split(/:/,$children->{$container});
        for (my $i=0; $i<@contents; $i ++) {
            $scripttag .= 'parents['.$container.']['.$i.'] = '.$contents[$i]."\n";
        }
    }

    foreach my $key (sort { $a <=> $b } (keys(%{$titles}))) {
        $scripttag .= "titles[$key] = '".$titles->{$key}."';\n";
    }

    $scripttag .= <<END;

function containerCheck(form,count,offset) {
    if (count > 0) {
        dependencyCheck(form,count,offset);
        var item = (offset+$startcount)+7*(count-1);
        form.elements[item].checked = true;
        if(Object.prototype.toString.call(parents[count]) === '[object Array]') {
            if (parents[count].length > 0) {
                for (var j=0; j<parents[count].length; j++) {
                    containerCheck(form,parents[count][j],offset);
                }
            }
        }
    }
}

function dependencyCheck(form,count,offset) {
    if (count > 0) {
        var chosen = (offset+$startcount)+7*(count-1);
        var depitem = $startcount + ((count-1) * 7) + 4;
        var currtype = form.elements[depitem].type;
        if (form.elements[chosen].value == 'dependency') {
            document.getElementById('arc_depon_'+count).style.display='block'; 
            form.elements[depitem].options.length = 0;
            form.elements[depitem].options[0] = new Option('Select','',true,true);
            for (var i=1; i<=numitems; i++) {
                if (i == count) {
                    continue;
                }
                var startelement = $startcount + (i-1) * 7;
                for (var j=1; j<6; j++) {
                    if ((j != 2) && (j!= 4)) {
                        var item = startelement + j;
                        if (form.elements[item].type == 'radio') {
                            if (form.elements[item].checked) {
                                if (form.elements[item].value == 'display') {
                                    var n = form.elements[depitem].options.length;
                                    form.elements[depitem].options[n] = new Option(titles[i],i,false,false);
                                }
                            }
                        }
                    }
                }
            }
        } else {
            document.getElementById('arc_depon_'+count).style.display='none';
            form.elements[depitem].options.length = 0;
            form.elements[depitem].options[0] = new Option('Select','',true,true);
        }
        titleCheck(form,count,offset);
    }
}

function propagateSelect(form,count,offset) {
    if (count > 0) {
        var item = (1+offset+$startcount)+7*(count-1);
        var picked = form.elements[item].options[form.elements[item].selectedIndex].value; 
        if (Object.prototype.toString.call(parents[count]) === '[object Array]') {
            if (parents[count].length > 0) {
                for (var j=0; j<parents[count].length; j++) {
                    containerSelect(form,parents[count][j],offset,picked);
                }
            }
        }
    }
}

function containerSelect(form,count,offset,picked) {
    if (count > 0) {
        var item = (offset+$startcount)+7*(count-1);
        if (form.elements[item].type == 'radio') {
            if (form.elements[item].value == 'dependency') {
                if (form.elements[item+1].type == 'select-one') {
                    for (var i=0; i<form.elements[item+1].options.length; i++) {
                        if (form.elements[item+1].options[i].value == picked) {
                            form.elements[item+1].selectedIndex = i;
                            break;
                        }
                    }
                }
                if (Object.prototype.toString.call(parents[count]) === '[object Array]') {
                    if (parents[count].length > 0) {
                        for (var j=0; j<parents[count].length; j++) {
                            containerSelect(form,parents[count][j],offset,picked);
                        }
                    }
                }
            }
        }
    }
}

function titleCheck(form,count,offset) {
    if (count > 0) {
        var chosen = (offset+$startcount)+7*(count-1);
        var depitem = $startcount + ((count-1) * 7) + 2;
        var currtype = form.elements[depitem].type;
        if (form.elements[chosen].value == 'display') {
            document.getElementById('arc_title_'+count).style.display='block';
            if ((count==1) && ((parents[count].length > 0) || (numitems == 1))) {
                document.getElementById('archive_title_'+count).value=maintitle;
            }
        } else {
            document.getElementById('arc_title_'+count).style.display='none';
            if (currtype == 'text') { 
                document.getElementById('archive_title_'+count).value='';
            }
        }
    }
    return;
}

// ]]>
</script>
END
    return $scripttag;
}

sub process_extracted_files {
    my ($context,$docudom,$docuname,$destination,$dir_root,$hiddenelem) = @_;
    my $numitems = $env{'form.archive_count'};
    return if ((!$numitems) || ($numitems =~ /\D/));
    my @ids=&Apache::lonnet::current_machine_ids();
    my ($prefix,$pathtocheck,$dir,$ishome,$error,$warning,%toplevelitems,%is_dir,
        %folders,%containers,%mapinner,%prompttofetch);
    my $docuhome = &Apache::lonnet::homeserver($docuname,$docudom);
    if (grep(/^\Q$docuhome\E$/,@ids)) {
        $prefix = &LONCAPA::propath($docudom,$docuname);
        $pathtocheck = "$dir_root/$destination";
        $dir = $dir_root;
        $ishome = 1;
    } else {
        $prefix = $Apache::lonnet::perlvar{'lonDocRoot'};
        $pathtocheck = "$dir_root/$docudom/$docuname/$destination";
        $dir = "$dir_root/$docudom/$docuname";
    }
    my $currdir = "$dir_root/$destination";
    (my $docstype,$mapinner{'0'}) = ($destination =~ m{^(docs|supplemental)/(\w+)/});
    if ($env{'form.folderpath'}) {
        my @items = split('&',$env{'form.folderpath'});
        $folders{'0'} = $items[-2];
        if ($env{'form.folderpath'} =~ /\:1$/) {
            $containers{'0'}='page';
        } else {
            $containers{'0'}='sequence';
        }
    }
    my @archdirs = &get_env_multiple('form.archive_directory');
    if ($numitems) {
        for (my $i=1; $i<=$numitems; $i++) {
            my $path = $env{'form.archive_content_'.$i};
            if ($path =~ m{^\Q$pathtocheck\E/([^/]+)$}) {
                my $item = $1;
                $toplevelitems{$item} = $i;
                if (grep(/^\Q$i\E$/,@archdirs)) {
                    $is_dir{$item} = 1;
                }
            }
        }
    }
    my ($output,%children,%parent,%titles,%dirorder,$result);
    if (keys(%toplevelitems) > 0) {
        my @contents = sort(keys(%toplevelitems));
        (my $count,undef) = &get_extracted($docudom,$docuname,$currdir,\%is_dir,\%children,
                                           \%parent,\@contents,\%dirorder,\%titles);
    }
    my (%referrer,%orphaned,%todelete,%todeletedir,%newdest,%newseqid);
    if ($numitems) {
        for (my $i=1; $i<=$numitems; $i++) {
            next if ($env{'form.archive_'.$i} eq 'dependency');
            my $path = $env{'form.archive_content_'.$i};
            if ($path =~ /^\Q$pathtocheck\E/) {
                if ($env{'form.archive_'.$i} eq 'discard') {
                    if ($prefix ne '' && $path ne '') {
                        if (-e $prefix.$path) {
                            if ((@archdirs > 0) && 
                                (grep(/^\Q$i\E$/,@archdirs))) {
                                $todeletedir{$prefix.$path} = 1;
                            } else {
                                $todelete{$prefix.$path} = 1;
                            }
                        }
                    }
                } elsif ($env{'form.archive_'.$i} eq 'display') {
                    my ($docstitle,$title,$url,$outer);
                    ($title) = ($path =~ m{/([^/]+)$});
                    $docstitle = $env{'form.archive_title_'.$i};
                    if ($docstitle eq '') {
                        $docstitle = $title;
                    }
                    $outer = 0;
                    if (ref($dirorder{$i}) eq 'ARRAY') {
                        if (@{$dirorder{$i}} > 0) {
                            foreach my $item (reverse(@{$dirorder{$i}})) {
                                if ($env{'form.archive_'.$item} eq 'display') {
                                    $outer = $item;
                                    last;
                                }
                            }
                        }
                    }
                    my ($errtext,$fatal) = 
                        &LONCAPA::map::mapread('/uploaded/'.$docudom.'/'.$docuname.
                                               '/'.$folders{$outer}.'.'.
                                               $containers{$outer});
                    next if ($fatal);
                    if ((@archdirs > 0) && (grep(/^\Q$i\E$/,@archdirs))) {
                        if ($context eq 'coursedocs') {
                            $mapinner{$i} = time;
                            $folders{$i} = 'default_'.$mapinner{$i};
                            $containers{$i} = 'sequence';
                            my $url = '/uploaded/'.$docudom.'/'.$docuname.'/'.
                                      $folders{$i}.'.'.$containers{$i};
                            my $newidx = &LONCAPA::map::getresidx();
                            $LONCAPA::map::resources[$newidx]=
                                $docstitle.':'.$url.':false:normal:res';
                            push(@LONCAPA::map::order,$newidx);
                            my ($outtext,$errtext) =
                                &LONCAPA::map::storemap('/uploaded/'.$docudom.'/'.
                                                        $docuname.'/'.$folders{$outer}.
                                                        '.'.$containers{$outer},1,1);
                            $newseqid{$i} = $newidx;
                            unless ($errtext) {
                                $result .=  '<li>'.&mt('Folder: [_1] added to course',
                                                       &HTML::Entities::encode($docstitle,'<>&"'))..
                                            '</li>'."\n";
                            }
                        }
                    } else {
                        if ($context eq 'coursedocs') {
                            my $newidx=&LONCAPA::map::getresidx();
                            my $url = '/uploaded/'.$docudom.'/'.$docuname.'/'.
                                      $docstype.'/'.$mapinner{$outer}.'/'.$newidx.'/'.
                                      $title;
                            if (($outer !~ /\D/) &&
                                (($mapinner{$outer} eq 'default') || ($mapinner{$outer} !~ /\D/)) &&
                                ($newidx !~ /\D/)) {
                                if (!-e "$prefix$dir/$docstype/$mapinner{$outer}") {
                                    mkdir("$prefix$dir/$docstype/$mapinner{$outer}",0755);
                                }
                                if (!-e "$prefix$dir/$docstype/$mapinner{$outer}/$newidx") {
                                    mkdir("$prefix$dir/$docstype/$mapinner{$outer}/$newidx");
                                }
                                if (-e "$prefix$dir/$docstype/$mapinner{$outer}/$newidx") {
                                    if (rename("$prefix$path","$prefix$dir/$docstype/$mapinner{$outer}/$newidx/$title")) {
                                        $newdest{$i} = "$prefix$dir/$docstype/$mapinner{$outer}/$newidx";
                                        unless ($ishome) {
                                            my $fetch = "$newdest{$i}/$title";
                                            $fetch =~ s/^\Q$prefix$dir\E//;
                                            $prompttofetch{$fetch} = 1;
                                        }
                                   }
                                }
                                $LONCAPA::map::resources[$newidx]=
                                    $docstitle.':'.$url.':false:normal:res';
                                push(@LONCAPA::map::order, $newidx);
                                my ($outtext,$errtext)=
                                    &LONCAPA::map::storemap('/uploaded/'.$docudom.'/'.
                                                            $docuname.'/'.$folders{$outer}.
                                                            '.'.$containers{$outer},1,1);
                                unless ($errtext) {
                                    if (-e "$prefix$dir/$docstype/$mapinner{$outer}/$newidx/$title") {
                                        $result .= '<li>'.&mt('File: [_1] added to course',
                                                              &HTML::Entities::encode($docstitle,'<>&"')).
                                                   '</li>'."\n";
                                    }
                                }
                            } else {
                                $warning .= &mt('Item extracted from archive: [_1] has unexpected path.',
                                                &HTML::Entities::encode($path,'<>&"')).'<br />';
                            }
                        }
                    }
                }
            } else {
                $warning .= &mt('Item extracted from archive: [_1] has unexpected path.',
                                &HTML::Entities::encode($path,'<>&"')).'<br />';
            }
        }
        for (my $i=1; $i<=$numitems; $i++) {
            next unless ($env{'form.archive_'.$i} eq 'dependency');
            my $path = $env{'form.archive_content_'.$i};
            if ($path =~ /^\Q$pathtocheck\E/) {
                my ($title) = ($path =~ m{/([^/]+)$});
                $referrer{$i} = $env{'form.archive_dependent_on_'.$i};
                if ($env{'form.archive_'.$referrer{$i}} eq 'display') {
                    if (ref($dirorder{$i}) eq 'ARRAY') {
                        my ($itemidx,$fullpath,$relpath);
                        if (ref($dirorder{$referrer{$i}}) eq 'ARRAY') {
                            my $container = $dirorder{$referrer{$i}}->[-1];
                            for (my $j=0; $j<@{$dirorder{$i}}; $j++) {
                                if ($dirorder{$i}->[$j] eq $container) {
                                    $itemidx = $j;
                                }
                            }
                        }
                        if ($itemidx eq '') {
                            $itemidx =  0;
                        }
                        if (grep(/^\Q$referrer{$i}\E$/,@archdirs)) {
                            if ($mapinner{$referrer{$i}}) {
                                $fullpath = "$prefix$dir/$docstype/$mapinner{$referrer{$i}}";
                                for (my $j=$itemidx; $j<@{$dirorder{$i}}; $j++) {
                                    if (grep(/^\Q$dirorder{$i}->[$j]\E$/,@archdirs)) {
                                        unless (defined($newseqid{$dirorder{$i}->[$j]})) {
                                            $fullpath .= '/'.$titles{$dirorder{$i}->[$j]};
                                            $relpath .= '/'.$titles{$dirorder{$i}->[$j]};
                                            if (!-e $fullpath) {
                                                mkdir($fullpath,0755);
                                            }
                                        }
                                    } else {
                                        last;
                                    }
                                }
                            }
                        } elsif ($newdest{$referrer{$i}}) {
                            $fullpath = $newdest{$referrer{$i}};
                            for (my $j=$itemidx; $j<@{$dirorder{$i}}; $j++) {
                                if ($env{'form.archive_'.$dirorder{$i}->[$j]} eq 'discard') {
                                    $orphaned{$i} = $env{'form.archive_'.$dirorder{$i}->[$j]};
                                    last;
                                } elsif (grep(/^\Q$dirorder{$i}->[$j]\E$/,@archdirs)) {
                                    unless (defined($newseqid{$dirorder{$i}->[$j]})) {
                                        $fullpath .= '/'.$titles{$dirorder{$i}->[$j]};
                                        $relpath .= '/'.$titles{$dirorder{$i}->[$j]};
                                        if (!-e $fullpath) {
                                            mkdir($fullpath,0755);
                                        }
                                    }
                                } else {
                                    last;
                                }
                            }
                        }
                        if ($fullpath ne '') {
                            if (-e "$prefix$path") {
                                unless (rename("$prefix$path","$fullpath/$title")) {
                                     $warning .= &mt('Failed to rename dependency').'<br />';
                                }
                            }
                            if (-e "$fullpath/$title") {
                                my $showpath;
                                if ($relpath ne '') {
                                    $showpath = "$relpath/$title";
                                } else {
                                    $showpath = "/$title";
                                }
                                $result .= '<li>'.&mt('[_1] included as a dependency',
                                                      &HTML::Entities::encode($showpath,'<>&"')).
                                           '</li>'."\n";
                                unless ($ishome) {
                                    my $fetch = "$fullpath/$title";
                                    $fetch =~ s/^\Q$prefix$dir\E//;
                                    $prompttofetch{$fetch} = 1;
                                }
                            }
                        }
                    }
                } elsif ($env{'form.archive_'.$referrer{$i}} eq 'discard') {
                    $warning .= &mt('[_1] is a dependency of [_2], which was discarded.',
                                    &HTML::Entities::encode($path,'<>&"'),
                                    &HTML::Entities::encode($env{'form.archive_content_'.$referrer{$i}},'<>&"')).
                                '<br />';
                }
            } else {
                $warning .= &mt('Item extracted from archive: [_1] has unexpected path.',
                                &HTML::Entities::encode($path)).'<br />';
            }
        }
        if (keys(%todelete)) {
            foreach my $key (keys(%todelete)) {
                unlink($key);
            }
        }
        if (keys(%todeletedir)) {
            foreach my $key (keys(%todeletedir)) {
                rmdir($key);
            }
        }
        foreach my $dir (sort(keys(%is_dir))) {
            if (($pathtocheck ne '') && ($dir ne ''))  {
                &cleanup_empty_dirs($prefix."$pathtocheck/$dir");
            }
        }
        if ($result ne '') {
            $output .= '<ul>'."\n".
                       $result."\n".
                       '</ul>';
        }
        unless ($ishome) {
            my $replicationfail;
            foreach my $item (keys(%prompttofetch)) {
                my $fetchresult= &Apache::lonnet::reply('fetchuserfile:'.$item,$docuhome);
                unless ($fetchresult eq 'ok') {
                    $replicationfail .= '<li>'.$item.'</li>'."\n";
                }
            }
            if ($replicationfail) {
                $output .= '<p class="LC_error">'.
                           &mt('Course home server failed to retrieve:').'<ul>'.
                           $replicationfail.
                           '</ul></p>';
            }
        }
    } else {
        $warning = &mt('No items found in archive.');
    }
    if ($error) {
        $output .= '<p class="LC_error">'.&mt('Not extracted.').'<br />'.
                   $error.'</p>'."\n";
    }
    if ($warning) {
        $output .= '<p class="LC_warning">'.$warning.'</p>'."\n";
    }
    return $output;
}

sub cleanup_empty_dirs {
    my ($path) = @_;
    if (($path ne '') && (-d $path)) {
        if (opendir(my $dirh,$path)) {
            my @dircontents = grep(!/^\./,readdir($dirh));
            my $numitems = 0;
            foreach my $item (@dircontents) {
                if (-d "$path/$item") {
                    &cleanup_empty_dirs("$path/$item");
                    if (-e "$path/$item") {
                        $numitems ++;
                    }
                } else {
                    $numitems ++;
                }
            }
            if ($numitems == 0) {
                rmdir($path);
            }
            closedir($dirh);
        }
    }
    return;
}

=pod

=item * &get_folder_hierarchy()

Provides hierarchy of names of folders/sub-folders containing the current
item,

Inputs: 3
     - $navmap - navmaps object

     - $map - url for map (either the trigger itself, or map containing
                           the resource, which is the trigger).

     - $showitem - 1 => show title for map itself; 0 => do not show.

Outputs: 1 @pathitems - array of folder/subfolder names.

=cut

sub get_folder_hierarchy {
    my ($navmap,$map,$showitem) = @_;
    my @pathitems;
    if (ref($navmap)) {
        my $mapres = $navmap->getResourceByUrl($map);
        if (ref($mapres)) {
            my $pcslist = $mapres->map_hierarchy();
            if ($pcslist ne '') {
                my @pcs = split(/,/,$pcslist);
                foreach my $pc (@pcs) {
                    if ($pc == 1) {
                        push(@pathitems,&mt('Main Content'));
                    } else {
                        my $res = $navmap->getByMapPc($pc);
                        if (ref($res)) {
                            my $title = $res->compTitle();
                            $title =~ s/\W+/_/g;
                            if ($title ne '') {
                                push(@pathitems,$title);
                            }
                        }
                    }
                }
            }
            if ($showitem) {
                if ($mapres->{ID} eq '0.0') {
                    push(@pathitems,&mt('Main Content'));
                } else {
                    my $maptitle = $mapres->compTitle();
                    $maptitle =~ s/\W+/_/g;
                    if ($maptitle ne '') {
                        push(@pathitems,$maptitle);
                    }
                }
            }
        }
    }
    return @pathitems;
}

=pod

=item * &get_turnedin_filepath()

Determines path in a user's portfolio file for storage of files uploaded
to a specific essayresponse or dropbox item.

Inputs: 3 required + 1 optional.
$symb is symb for resource, $uname and $udom are for current user (required).
$caller is optional (can be "submission", if routine is called when storing
an upoaded file when "Submit Answer" button was pressed).

Returns array containing $path and $multiresp. 
$path is path in portfolio.  $multiresp is 1 if this resource contains more
than one file upload item.  Callers of routine should append partid as a 
subdirectory to $path in cases where $multiresp is 1.

Called by: homework/essayresponse.pm and homework/structuretags.pm

=cut

sub get_turnedin_filepath {
    my ($symb,$uname,$udom,$caller) = @_;
    my ($map,$resid,$resurl)=&Apache::lonnet::decode_symb($symb);
    my $turnindir;
    my %userhash = &Apache::lonnet::userenvironment($udom,$uname,'turnindir');
    $turnindir = $userhash{'turnindir'};
    my ($path,$multiresp);
    if ($turnindir eq '') {
        if ($caller eq 'submission') {
            $turnindir = &mt('turned in');
            $turnindir =~ s/\W+/_/g;
            my %newhash = (
                            'turnindir' => $turnindir,
                          );
            &Apache::lonnet::put('environment',\%newhash,$udom,$uname);
        }
    }
    if ($turnindir ne '') {
        $path = '/'.$turnindir.'/';
        my ($multipart,$turnin,@pathitems);
        my $navmap = Apache::lonnavmaps::navmap->new();
        if (defined($navmap)) {
            my $mapres = $navmap->getResourceByUrl($map);
            if (ref($mapres)) {
                my $pcslist = $mapres->map_hierarchy();
                if ($pcslist ne '') {
                    foreach my $pc (split(/,/,$pcslist)) {
                        my $res = $navmap->getByMapPc($pc);
                        if (ref($res)) {
                            my $title = $res->compTitle();
                            $title =~ s/\W+/_/g;
                            if ($title ne '') {
                                if (($pc > 1) && (length($title) > 12)) {
                                    $title = substr($title,0,12);
                                }
                                push(@pathitems,$title);
                            }
                        }
                    }
                }
                my $maptitle = $mapres->compTitle();
                $maptitle =~ s/\W+/_/g;
                if ($maptitle ne '') {
                    if (length($maptitle) > 12) {
                        $maptitle = substr($maptitle,0,12);
                    }
                    push(@pathitems,$maptitle);
                }
                unless ($env{'request.state'} eq 'construct') {
                    my $res = $navmap->getBySymb($symb);
                    if (ref($res)) {
                        my $partlist = $res->parts();
                        my $totaluploads = 0;
                        if (ref($partlist) eq 'ARRAY') {
                            foreach my $part (@{$partlist}) {
                                my @types = $res->responseType($part);
                                my @ids = $res->responseIds($part);
                                for (my $i=0; $i < scalar(@ids); $i++) {
                                    if ($types[$i] eq 'essay') {
                                        my $partid = $part.'_'.$ids[$i];
                                        if (&Apache::lonnet::EXT("resource.$partid.uploadedfiletypes") ne '') {
                                            $totaluploads ++;
                                        }
                                    }
                                }
                            }
                            if ($totaluploads > 1) {
                                $multiresp = 1;
                            }
                        }
                    }
                }
            } else {
                return;
            }
        } else {
            return;
        }
        my $restitle=&Apache::lonnet::gettitle($symb);
        $restitle =~ s/\W+/_/g;
        if ($restitle eq '') {
            $restitle = ($resurl =~ m{/[^/]+$});
            if ($restitle eq '') {
                $restitle = time;
            }
        }
        if (length($restitle) > 12) {
            $restitle = substr($restitle,0,12);
        }
        push(@pathitems,$restitle);
        $path .= join('/',@pathitems);
    }
    return ($path,$multiresp);
}

=pod

=back

=head1 CSV Upload/Handling functions

=over 4

=item * &upfile_store($r)

Store uploaded file, $r should be the HTTP Request object,
needs $env{'form.upfile'}
returns $datatoken to be put into hidden field

=cut

sub upfile_store {
    my $r=shift;
    $env{'form.upfile'}=~s/\r/\n/gs;
    $env{'form.upfile'}=~s/\f/\n/gs;
    $env{'form.upfile'}=~s/\n+/\n/gs;
    $env{'form.upfile'}=~s/\n+$//gs;

    my $datatoken = &valid_datatoken($env{'user.name'}.'_'.$env{'user.domain'}.
                                     '_enroll_'.$env{'request.course.id'}.'_'.
                                     time.'_'.$$);
    return if ($datatoken eq '');

    {
        my $datafile = $r->dir_config('lonDaemons').
                           '/tmp/'.$datatoken.'.tmp';
        if ( open(my $fh,'>',$datafile) ) {
            print $fh $env{'form.upfile'};
            close($fh);
        }
    }
    return $datatoken;
}

=pod

=item * &load_tmp_file($r,$datatoken)

Load uploaded file from tmp, $r should be the HTTP Request object,
$datatoken is the name to assign to the temporary file.
sets $env{'form.upfile'} to the contents of the file

=cut

sub load_tmp_file {
    my ($r,$datatoken) = @_;
    return if ($datatoken eq '');
    my @studentdata=();
    {
        my $studentfile = $r->dir_config('lonDaemons').
                              '/tmp/'.$datatoken.'.tmp';
        if ( open(my $fh,'<',$studentfile) ) {
            @studentdata=<$fh>;
            close($fh);
        }
    }
    $env{'form.upfile'}=join('',@studentdata);
}

sub valid_datatoken {
    my ($datatoken) = @_;
    if ($datatoken =~ /^$match_username\_$match_domain\_enroll_(|$match_domain\_$match_courseid)\_\d+_\d+$/) {
        return $datatoken;
    }
    return;
}

=pod

=item * &upfile_record_sep()

Separate uploaded file into records
returns array of records,
needs $env{'form.upfile'} and $env{'form.upfiletype'}

=cut

sub upfile_record_sep {
    if ($env{'form.upfiletype'} eq 'xml') {
    } else {
	my @records;
	foreach my $line (split(/\n/,$env{'form.upfile'})) {
	    if ($line=~/^\s*$/) { next; }
	    push(@records,$line);
	}
	return @records;
    }
}

=pod

=item * &record_sep($record)

Separate a record into fields $record should be an item from the upfile_record_sep(), needs $env{'form.upfiletype'}

=cut

sub takeleft {
    my $index=shift;
    return substr('0000'.$index,-4,4);
}

sub record_sep {
    my $record=shift;
    my %components=();
    if ($env{'form.upfiletype'} eq 'xml') {
    } elsif ($env{'form.upfiletype'} eq 'space') {
        my $i=0;
        foreach my $field (split(/\s+/,$record)) {
            $field=~s/^(\"|\')//;
            $field=~s/(\"|\')$//;
            $components{&takeleft($i)}=$field;
            $i++;
        }
    } elsif ($env{'form.upfiletype'} eq 'tab') {
        my $i=0;
        foreach my $field (split(/\t/,$record)) {
            $field=~s/^(\"|\')//;
            $field=~s/(\"|\')$//;
            $components{&takeleft($i)}=$field;
            $i++;
        }
    } else {
        my $separator=',';
        if ($env{'form.upfiletype'} eq 'semisv') {
            $separator=';';
        }
        my $i=0;
# the character we are looking for to indicate the end of a quote or a record 
        my $looking_for=$separator;
# do not add the characters to the fields
        my $ignore=0;
# we just encountered a separator (or the beginning of the record)
        my $just_found_separator=1;
# store the field we are working on here
        my $field='';
# work our way through all characters in record
        foreach my $character ($record=~/(.)/g) {
            if ($character eq $looking_for) {
               if ($character ne $separator) {
# Found the end of a quote, again looking for separator
                  $looking_for=$separator;
                  $ignore=1;
               } else {
# Found a separator, store away what we got
                  $components{&takeleft($i)}=$field;
	          $i++;
                  $just_found_separator=1;
                  $ignore=0;
                  $field='';
               }
               next;
            }
# single or double quotation marks after a separator indicate beginning of a quote
# we are now looking for the end of the quote and need to ignore separators
            if ((($character eq '"') || ($character eq "'")) && ($just_found_separator))  {
               $looking_for=$character;
               next;
            }
# ignore would be true after we reached the end of a quote
            if ($ignore) { next; }
            if (($just_found_separator) && ($character=~/\s/)) { next; }
            $field.=$character;
            $just_found_separator=0; 
        }
# catch the very last entry, since we never encountered the separator
        $components{&takeleft($i)}=$field;
    }
    return %components;
}

######################################################
######################################################

=pod

=item * &upfile_select_html()

Return HTML code to select a file from the users machine and specify 
the file type.

=cut

######################################################
######################################################
sub upfile_select_html {
    my %Types = (
                 csv   => &mt('CSV (comma separated values, spreadsheet)'),
                 semisv => &mt('Semicolon separated values'),
                 space => &mt('Space separated'),
                 tab   => &mt('Tabulator separated'),
#                 xml   => &mt('HTML/XML'),
                 );
    my $Str = '<input type="file" name="upfile" size="50" />'.
        '<br />'.&mt('Type').': <select name="upfiletype">';
    foreach my $type (sort(keys(%Types))) {
        $Str .= '<option value="'.$type.'" >'.$Types{$type}."</option>\n";
    }
    $Str .= "</select>\n";
    return $Str;
}

sub get_samples {
    my ($records,$toget) = @_;
    my @samples=({});
    my $got=0;
    foreach my $rec (@$records) {
	my %temp = &record_sep($rec);
	if (! grep(/\S/, values(%temp))) { next; }
	if (%temp) {
	    $samples[$got]=\%temp;
	    $got++;
	    if ($got == $toget) { last; }
	}
    }
    return \@samples;
}

######################################################
######################################################

=pod

=item * &csv_print_samples($r,$records)

Prints a table of sample values from each column uploaded $r is an
Apache Request ref, $records is an arrayref from
&Apache::loncommon::upfile_record_sep

=cut

######################################################
######################################################
sub csv_print_samples {
    my ($r,$records) = @_;
    my $samples = &get_samples($records,5);

    $r->print(&mt('Samples').'<br />'.&start_data_table().
              &start_data_table_header_row());
    foreach my $sample (sort({$a <=> $b} keys(%{ $samples->[0] }))) { 
        $r->print('<th>'.&mt('Column [_1]',($sample+1)).'</th>'); }
    $r->print(&end_data_table_header_row());
    foreach my $hash (@$samples) {
	$r->print(&start_data_table_row());
	foreach my $sample (sort({$a <=> $b} keys(%{ $samples->[0] }))) {
	    $r->print('<td>');
	    if (defined($$hash{$sample})) { $r->print($$hash{$sample}); }
	    $r->print('</td>');
	}
	$r->print(&end_data_table_row());
    }
    $r->print(&end_data_table().'<br />'."\n");
}

######################################################
######################################################

=pod

=item * &csv_print_select_table($r,$records,$d)

Prints a table to create associations between values and table columns.

$r is an Apache Request ref,
$records is an arrayref from &Apache::loncommon::upfile_record_sep,
$d is an array of 2 element arrays (internal name, displayed name,defaultcol)

=cut

######################################################
######################################################
sub csv_print_select_table {
    my ($r,$records,$d) = @_;
    my $i=0;
    my $samples = &get_samples($records,1);
    $r->print(&mt('Associate columns with student attributes.')."\n".
	      &start_data_table().&start_data_table_header_row().
              '<th>'.&mt('Attribute').'</th>'.
              '<th>'.&mt('Column').'</th>'.
              &end_data_table_header_row()."\n");
    foreach my $array_ref (@$d) {
	my ($value,$display,$defaultcol)=@{ $array_ref };
	$r->print(&start_data_table_row().'<td>'.$display.'</td>');

	$r->print('<td><select name="f'.$i.'"'.
		  ' onchange="javascript:flip(this.form,'.$i.');">');
	$r->print('<option value="none"></option>');
	foreach my $sample (sort({$a <=> $b} keys(%{ $samples->[0] }))) {
	    $r->print('<option value="'.$sample.'"'.
                      ($sample eq $defaultcol ? ' selected="selected" ' : '').
                      '>'.&mt('Column [_1]',($sample+1)).'</option>');
	}
	$r->print('</select></td>'.&end_data_table_row()."\n");
	$i++;
    }
    $r->print(&end_data_table());
    $i--;
    return $i;
}

######################################################
######################################################

=pod

=item * &csv_samples_select_table($r,$records,$d)

Prints a table of sample values from the upload and can make associate samples to internal names.

$r is an Apache Request ref,
$records is an arrayref from &Apache::loncommon::upfile_record_sep,
$d is an array of 2 element arrays (internal name, displayed name)

=cut

######################################################
######################################################
sub csv_samples_select_table {
    my ($r,$records,$d) = @_;
    my $i=0;
    #
    my $max_samples = 5;
    my $samples = &get_samples($records,$max_samples);
    $r->print(&start_data_table().
              &start_data_table_header_row().'<th>'.
              &mt('Field').'</th><th>'.&mt('Samples').'</th>'.
              &end_data_table_header_row());

    foreach my $key (sort(keys(%{ $samples->[0] }))) {
	$r->print(&start_data_table_row().'<td><select name="f'.$i.'"'.
		  ' onchange="javascript:flip(this.form,'.$i.');">');
	foreach my $option (@$d) {
	    my ($value,$display,$defaultcol)=@{ $option };
	    $r->print('<option value="'.$value.'"'.
                      ($i eq $defaultcol ? ' selected="selected" ':'').'>'.
                      $display.'</option>');
	}
	$r->print('</select></td><td>');
	foreach my $line (0..($max_samples-1)) {
	    if (defined($samples->[$line]{$key})) { 
		$r->print($samples->[$line]{$key}."<br />\n"); 
	    }
	}
	$r->print('</td>'.&end_data_table_row());
	$i++;
    }
    $r->print(&end_data_table());
    $i--;
    return($i);
}

######################################################
######################################################

=pod

=item * &clean_excel_name($name)

Returns a replacement for $name which does not contain any illegal characters.

=cut

######################################################
######################################################
sub clean_excel_name {
    my ($name) = @_;
    $name =~ s/[:\*\?\/\\]//g;
    if (length($name) > 31) {
        $name = substr($name,0,31);
    }
    return $name;
}

=pod

=item * &check_if_partid_hidden($id,$symb,$udom,$uname)

Returns either 1 or undef

1 if the part is to be hidden, undef if it is to be shown

Arguments are:

$id the id of the part to be checked
$symb, optional the symb of the resource to check
$udom, optional the domain of the user to check for
$uname, optional the username of the user to check for

=cut

sub check_if_partid_hidden {
    my ($id,$symb,$udom,$uname) = @_;
    my $hiddenparts=&Apache::lonnet::EXT('resource.0.hiddenparts',
					 $symb,$udom,$uname);
    my $truth=1;
    #if the string starts with !, then the list is the list to show not hide
    if ($hiddenparts=~s/^\s*!//) { $truth=undef; }
    my @hiddenlist=split(/,/,$hiddenparts);
    foreach my $checkid (@hiddenlist) {
	if ($checkid =~ /^\s*\Q$id\E\s*$/) { return $truth; }
    }
    return !$truth;
}


############################################################
############################################################

=pod

=back 

=head1 cgi-bin script and graphing routines

=over 4

=item * &get_cgi_id()

Inputs: none

Returns an id which can be used to pass environment variables
to various cgi-bin scripts.  These environment variables will
be removed from the users environment after a given time by
the routine &Apache::lonnet::transfer_profile_to_env.

=cut

############################################################
############################################################
my $uniq=0;
sub get_cgi_id {
    $uniq=($uniq+1)%100000;
    return (time.'_'.$$.'_'.$uniq);
}

############################################################
############################################################

=pod

=item * &DrawBarGraph()

Facilitates the plotting of data in a (stacked) bar graph.
Puts plot definition data into the users environment in order for 
graph.png to plot it.  Returns an <img> tag for the plot.
The bars on the plot are labeled '1','2',...,'n'.

Inputs:

=over 4

=item $Title: string, the title of the plot

=item $xlabel: string, text describing the X-axis of the plot

=item $ylabel: string, text describing the Y-axis of the plot

=item $Max: scalar, the maximum Y value to use in the plot
If $Max is < any data point, the graph will not be rendered.

=item $colors: array ref holding the colors to be used for the data sets when
they are plotted.  If undefined, default values will be used.

=item $labels: array ref holding the labels to use on the x-axis for the bars.

=item @Values: An array of array references.  Each array reference holds data
to be plotted in a stacked bar chart.

=item If the final element of @Values is a hash reference the key/value
pairs will be added to the graph definition.

=back

Returns:

An <img> tag which references graph.png and the appropriate identifying
information for the plot.

=cut

############################################################
############################################################
sub DrawBarGraph {
    my ($Title,$xlabel,$ylabel,$Max,$colors,$labels,@Values)=@_;
    #
    if (! defined($colors)) {
        $colors = ['#33ff00', 
                  '#0033cc', '#990000', '#aaaa66', '#663399', '#ff9933',
                  '#66ccff', '#ff9999', '#cccc33', '#660000', '#33cc66',
                  ]; 
    }
    my $extra_settings = {};
    if (ref($Values[-1]) eq 'HASH') {
        $extra_settings = pop(@Values);
    }
    #
    my $identifier = &get_cgi_id();
    my $id = 'cgi.'.$identifier;        
    if (! @Values || ref($Values[0]) ne 'ARRAY') {
        return '';
    }
    #
    my @Labels;
    if (defined($labels)) {
        @Labels = @$labels;
    } else {
        for (my $i=0;$i<@{$Values[0]};$i++) {
            push(@Labels,$i+1);
        }
    }
    #
    my $NumBars = scalar(@{$Values[0]});
    if ($NumBars < scalar(@Labels)) { $NumBars = scalar(@Labels); }
    my %ValuesHash;
    my $NumSets=1;
    foreach my $array (@Values) {
        next if (! ref($array));
        $ValuesHash{$id.'.data.'.$NumSets++} = 
            join(',',@$array);
    }
    #
    my ($height,$width,$xskip,$bar_width) = (200,120,1,15);
    if ($NumBars < 3) {
        $width = 120+$NumBars*32;
        $xskip = 1;
        $bar_width = 30;
    } elsif ($NumBars < 5) {
        $width = 120+$NumBars*20;
        $xskip = 1;
        $bar_width = 20;
    } elsif ($NumBars < 10) {
        $width = 120+$NumBars*15;
        $xskip = 1;
        $bar_width = 15;
    } elsif ($NumBars <= 25) {
        $width = 120+$NumBars*11;
        $xskip = 5;
        $bar_width = 8;
    } elsif ($NumBars <= 50) {
        $width = 120+$NumBars*8;
        $xskip = 5;
        $bar_width = 4;
    } else {
        $width = 120+$NumBars*8;
        $xskip = 5;
        $bar_width = 4;
    }
    #
    $Max = 1 if ($Max < 1);
    if ( int($Max) < $Max ) {
        $Max++;
        $Max = int($Max);
    }
    $Title  = '' if (! defined($Title));
    $xlabel = '' if (! defined($xlabel));
    $ylabel = '' if (! defined($ylabel));
    $ValuesHash{$id.'.title'}    = &escape($Title);
    $ValuesHash{$id.'.xlabel'}   = &escape($xlabel);
    $ValuesHash{$id.'.ylabel'}   = &escape($ylabel);
    $ValuesHash{$id.'.y_max_value'} = $Max;
    $ValuesHash{$id.'.NumBars'}  = $NumBars;
    $ValuesHash{$id.'.NumSets'}  = $NumSets;
    $ValuesHash{$id.'.PlotType'} = 'bar';
    $ValuesHash{$id.'.Colors'}   = join(',',@{$colors});
    $ValuesHash{$id.'.height'}   = $height;
    $ValuesHash{$id.'.width'}    = $width;
    $ValuesHash{$id.'.xskip'}    = $xskip;
    $ValuesHash{$id.'.bar_width'} = $bar_width;
    $ValuesHash{$id.'.labels'} = join(',',@Labels);
    #
    # Deal with other parameters
    while (my ($key,$value) = each(%$extra_settings)) {
        $ValuesHash{$id.'.'.$key} = $value;
    }
    #
    &Apache::lonnet::appenv(\%ValuesHash);
    return '<img src="/cgi-bin/graph.png?'.$identifier.'" border="1" />';
}

############################################################
############################################################

=pod

=item * &DrawXYGraph()

Facilitates the plotting of data in an XY graph.
Puts plot definition data into the users environment in order for 
graph.png to plot it.  Returns an <img> tag for the plot.

Inputs:

=over 4

=item $Title: string, the title of the plot

=item $xlabel: string, text describing the X-axis of the plot

=item $ylabel: string, text describing the Y-axis of the plot

=item $Max: scalar, the maximum Y value to use in the plot
If $Max is < any data point, the graph will not be rendered.

=item $colors: Array ref containing the hex color codes for the data to be 
plotted in.  If undefined, default values will be used.

=item $Xlabels: Array ref containing the labels to be used for the X-axis.

=item $Ydata: Array ref containing Array refs.  
Each of the contained arrays will be plotted as a separate curve.

=item %Values: hash indicating or overriding any default values which are 
passed to graph.png.  
Possible values are: width, xskip, x_ticks, x_tick_offset, among others.

=back

Returns:

An <img> tag which references graph.png and the appropriate identifying
information for the plot.

=cut

############################################################
############################################################
sub DrawXYGraph {
    my ($Title,$xlabel,$ylabel,$Max,$colors,$Xlabels,$Ydata,%Values)=@_;
    #
    # Create the identifier for the graph
    my $identifier = &get_cgi_id();
    my $id = 'cgi.'.$identifier;
    #
    $Title  = '' if (! defined($Title));
    $xlabel = '' if (! defined($xlabel));
    $ylabel = '' if (! defined($ylabel));
    my %ValuesHash = 
        (
         $id.'.title'  => &escape($Title),
         $id.'.xlabel' => &escape($xlabel),
         $id.'.ylabel' => &escape($ylabel),
         $id.'.y_max_value'=> $Max,
         $id.'.labels'     => join(',',@$Xlabels),
         $id.'.PlotType'   => 'XY',
         );
    #
    if (defined($colors) && ref($colors) eq 'ARRAY') {
        $ValuesHash{$id.'.Colors'}   = join(',',@{$colors});
    }
    #
    if (! ref($Ydata) || ref($Ydata) ne 'ARRAY') {
        return '';
    }
    my $NumSets=1;
    foreach my $array (@{$Ydata}){
        next if (! ref($array));
        $ValuesHash{$id.'.data.'.$NumSets++} = join(',',@$array);
    }
    $ValuesHash{$id.'.NumSets'} = $NumSets-1;
    #
    # Deal with other parameters
    while (my ($key,$value) = each(%Values)) {
        $ValuesHash{$id.'.'.$key} = $value;
    }
    #
    &Apache::lonnet::appenv(\%ValuesHash);
    return '<img src="/cgi-bin/graph.png?'.$identifier.'" border="1" />';
}

############################################################
############################################################

=pod

=item * &DrawXYYGraph()

Facilitates the plotting of data in an XY graph with two Y axes.
Puts plot definition data into the users environment in order for 
graph.png to plot it.  Returns an <img> tag for the plot.

Inputs:

=over 4

=item $Title: string, the title of the plot

=item $xlabel: string, text describing the X-axis of the plot

=item $ylabel: string, text describing the Y-axis of the plot

=item $colors: Array ref containing the hex color codes for the data to be 
plotted in.  If undefined, default values will be used.

=item $Xlabels: Array ref containing the labels to be used for the X-axis.

=item $Ydata1: The first data set

=item $Min1: The minimum value of the left Y-axis

=item $Max1: The maximum value of the left Y-axis

=item $Ydata2: The second data set

=item $Min2: The minimum value of the right Y-axis

=item $Max2: The maximum value of the left Y-axis

=item %Values: hash indicating or overriding any default values which are 
passed to graph.png.  
Possible values are: width, xskip, x_ticks, x_tick_offset, among others.

=back

Returns:

An <img> tag which references graph.png and the appropriate identifying
information for the plot.

=cut

############################################################
############################################################
sub DrawXYYGraph {
    my ($Title,$xlabel,$ylabel,$colors,$Xlabels,$Ydata1,$Min1,$Max1,
                                        $Ydata2,$Min2,$Max2,%Values)=@_;
    #
    # Create the identifier for the graph
    my $identifier = &get_cgi_id();
    my $id = 'cgi.'.$identifier;
    #
    $Title  = '' if (! defined($Title));
    $xlabel = '' if (! defined($xlabel));
    $ylabel = '' if (! defined($ylabel));
    my %ValuesHash = 
        (
         $id.'.title'  => &escape($Title),
         $id.'.xlabel' => &escape($xlabel),
         $id.'.ylabel' => &escape($ylabel),
         $id.'.labels' => join(',',@$Xlabels),
         $id.'.PlotType' => 'XY',
         $id.'.NumSets' => 2,
         $id.'.two_axes' => 1,
         $id.'.y1_max_value' => $Max1,
         $id.'.y1_min_value' => $Min1,
         $id.'.y2_max_value' => $Max2,
         $id.'.y2_min_value' => $Min2,
         );
    #
    if (defined($colors) && ref($colors) eq 'ARRAY') {
        $ValuesHash{$id.'.Colors'}   = join(',',@{$colors});
    }
    #
    if (! ref($Ydata1) || ref($Ydata1) ne 'ARRAY' ||
        ! ref($Ydata2) || ref($Ydata2) ne 'ARRAY'){
        return '';
    }
    my $NumSets=1;
    foreach my $array ($Ydata1,$Ydata2){
        next if (! ref($array));
        $ValuesHash{$id.'.data.'.$NumSets++} = join(',',@$array);
    }
    #
    # Deal with other parameters
    while (my ($key,$value) = each(%Values)) {
        $ValuesHash{$id.'.'.$key} = $value;
    }
    #
    &Apache::lonnet::appenv(\%ValuesHash);
    return '<img src="/cgi-bin/graph.png?'.$identifier.'" border="1" />';
}

############################################################
############################################################

=pod

=back 

=head1 Statistics helper routines?  

Bad place for them but what the hell.

=over 4

=item * &chartlink()

Returns a link to the chart for a specific student.  

Inputs:

=over 4

=item $linktext: The text of the link

=item $sname: The students username

=item $sdomain: The students domain

=back

=back

=cut

############################################################
############################################################
sub chartlink {
    my ($linktext, $sname, $sdomain) = @_;
    my $link = '<a href="/adm/statistics?reportSelected=student_assessment'.
        '&amp;SelectedStudent='.&escape($sname.':'.$sdomain).
        '&amp;chartoutputmode='.HTML::Entities::encode('html, with all links').
       '">'.$linktext.'</a>';
}

#######################################################
#######################################################

=pod

=head1 Course Environment Routines

=over 4

=item * &restore_course_settings()

=item * &store_course_settings()

Restores/Store indicated form parameters from the course environment.
Will not overwrite existing values of the form parameters.

Inputs: 
a scalar describing the data (e.g. 'chart', 'problem_analysis')

a hash ref describing the data to be stored.  For example:
   
%Save_Parameters = ('Status' => 'scalar',
    'chartoutputmode' => 'scalar',
    'chartoutputdata' => 'scalar',
    'Section' => 'array',
    'Group' => 'array',
    'StudentData' => 'array',
    'Maps' => 'array');

Returns: both routines return nothing

=back

=cut

#######################################################
#######################################################
sub store_course_settings {
    return &store_settings($env{'request.course.id'},@_);
}

sub store_settings {
    # save to the environment
    # appenv the same items, just to be safe
    my $udom  = $env{'user.domain'};
    my $uname = $env{'user.name'};
    my ($context,$prefix,$Settings) = @_;
    my %SaveHash;
    my %AppHash;
    while (my ($setting,$type) = each(%$Settings)) {
        my $basename = join('.','internal',$context,$prefix,$setting);
        my $envname = 'environment.'.$basename;
        if (exists($env{'form.'.$setting})) {
            # Save this value away
            if ($type eq 'scalar' &&
                (! exists($env{$envname}) || 
                 $env{$envname} ne $env{'form.'.$setting})) {
                $SaveHash{$basename} = $env{'form.'.$setting};
                $AppHash{$envname}   = $env{'form.'.$setting};
            } elsif ($type eq 'array') {
                my $stored_form;
                if (ref($env{'form.'.$setting})) {
                    $stored_form = join(',',
                                        map {
                                            &escape($_);
                                        } sort(@{$env{'form.'.$setting}}));
                } else {
                    $stored_form = 
                        &escape($env{'form.'.$setting});
                }
                # Determine if the array contents are the same.
                if ($stored_form ne $env{$envname}) {
                    $SaveHash{$basename} = $stored_form;
                    $AppHash{$envname}   = $stored_form;
                }
            }
        }
    }
    my $put_result = &Apache::lonnet::put('environment',\%SaveHash,
                                          $udom,$uname);
    if ($put_result !~ /^(ok|delayed)/) {
        &Apache::lonnet::logthis('unable to save form parameters, '.
                                 'got error:'.$put_result);
    }
    # Make sure these settings stick around in this session, too
    &Apache::lonnet::appenv(\%AppHash);
    return;
}

sub restore_course_settings {
    return &restore_settings($env{'request.course.id'},@_);
}

sub restore_settings {
    my ($context,$prefix,$Settings) = @_;
    while (my ($setting,$type) = each(%$Settings)) {
        next if (exists($env{'form.'.$setting}));
        my $envname = 'environment.internal.'.$context.'.'.$prefix.
            '.'.$setting;
        if (exists($env{$envname})) {
            if ($type eq 'scalar') {
                $env{'form.'.$setting} = $env{$envname};
            } elsif ($type eq 'array') {
                $env{'form.'.$setting} = [ 
                                           map { 
                                               &unescape($_); 
                                           } split(',',$env{$envname})
                                           ];
            }
        }
    }
}

#######################################################
#######################################################

=pod

=head1 Domain E-mail Routines  

=over 4

=item * &build_recipient_list()

Build recipient lists for following types of e-mail:
(a) Error Reports, (b) Package Updates, (c) lonstatus warnings/errors
(d) Help requests, (e) Course requests needing approval, (f) loncapa
module change checking, student/employee ID conflict checks, as
generated by lonerrorhandler.pm, CHECKRPMS, loncron,
lonsupportreq.pm, loncoursequeueadmin.pm, searchcat.pl respectively.

Inputs:
defmail (scalar - email address of default recipient),
mailing type (scalar: errormail, packagesmail, helpdeskmail,
requestsmail, updatesmail, or idconflictsmail).

defdom (domain for which to retrieve configuration settings),

origmail (scalar - email address of recipient from loncapa.conf,
i.e., predates configuration by DC via domainprefs.pm

$requname username of requester (if mailing type is helpdeskmail)

$requdom domain of requester (if mailing type is helpdeskmail)

$reqemail e-mail address of requester (if mailing type is helpdeskmail)

Returns: comma separated list of addresses to which to send e-mail.

=back

=cut

############################################################
############################################################
sub build_recipient_list {
    my ($defmail,$mailing,$defdom,$origmail,$requname,$requdom,$reqemail) = @_;
    my @recipients;
    my ($otheremails,$lastresort,$allbcc,$addtext);
    my %domconfig =
        &Apache::lonnet::get_dom('configuration',['contacts'],$defdom);
    if (ref($domconfig{'contacts'}) eq 'HASH') {
        if (exists($domconfig{'contacts'}{$mailing})) {
            if (ref($domconfig{'contacts'}{$mailing}) eq 'HASH') {
                my @contacts = ('adminemail','supportemail');
                foreach my $item (@contacts) {
                    if ($domconfig{'contacts'}{$mailing}{$item}) {
                        my $addr = $domconfig{'contacts'}{$item}; 
                        if (!grep(/^\Q$addr\E$/,@recipients)) {
                            push(@recipients,$addr);
                        }
                    }
                }
                $otheremails = $domconfig{'contacts'}{$mailing}{'others'};
                if ($mailing eq 'helpdeskmail') {
                    if ($domconfig{'contacts'}{$mailing}{'bcc'}) {
                        my @bccs = split(/,/,$domconfig{'contacts'}{$mailing}{'bcc'});
                        my @ok_bccs;
                        foreach my $bcc (@bccs) {
                            $bcc =~ s/^\s+//g;
                            $bcc =~ s/\s+$//g;
                            if ($bcc =~ m/^[^\@]+\@[^\@]+$/) {
                                if (!(grep(/^\Q$bcc\E$/,@ok_bccs))) {
                                    push(@ok_bccs,$bcc);
                                }
                            }
                        }
                        if (@ok_bccs > 0) {
                            $allbcc = join(', ',@ok_bccs);
                        }
                    }
                    $addtext = $domconfig{'contacts'}{$mailing}{'include'};
                }
            }
        } elsif ($origmail ne '') {
            $lastresort = $origmail;
        }
        if ($mailing eq 'helpdeskmail') {
            if ((ref($domconfig{'contacts'}{'overrides'}) eq 'HASH') &&
                (keys(%{$domconfig{'contacts'}{'overrides'}}))) {
                my ($inststatus,$inststatus_checked);
                if (($env{'user.name'} ne '') && ($env{'user.domain'} ne '') &&
                    ($env{'user.domain'} ne 'public')) {
                    $inststatus_checked = 1;
                    $inststatus = $env{'environment.inststatus'};
                }
                unless ($inststatus_checked) {
                    if (($requname ne '') && ($requdom ne '')) {
                        if (($requname =~ /^$match_username$/) &&
                            ($requdom =~ /^$match_domain$/) &&
                            (&Apache::lonnet::domain($requdom))) {
                            my $requhome = &Apache::lonnet::homeserver($requname,
                                                                      $requdom);
                            unless ($requhome eq 'no_host') {
                                my %userenv = &Apache::lonnet::userenvironment($requdom,$requname,'inststatus');
                                $inststatus = $userenv{'inststatus'};
                                $inststatus_checked = 1;
                            }
                        }
                    }
                }
                unless ($inststatus_checked) {
                    if ($reqemail =~ /^[^\@]+\@[^\@]+$/) {
                        my %srch = (srchby     => 'email',
                                    srchdomain => $defdom,
                                    srchterm   => $reqemail,
                                    srchtype   => 'exact');
                        my %srch_results = &Apache::lonnet::usersearch(\%srch);
                        foreach my $uname (keys(%srch_results)) {
                            if (ref($srch_results{$uname}{'inststatus'}) eq 'ARRAY') {
                                $inststatus = join(',',@{$srch_results{$uname}{'inststatus'}});
                                $inststatus_checked = 1;
                                last;
                            }
                        }
                        unless ($inststatus_checked) {
                            my ($dirsrchres,%srch_results) = &Apache::lonnet::inst_directory_query(\%srch);
                            if ($dirsrchres eq 'ok') {
                                foreach my $uname (keys(%srch_results)) {
                                    if (ref($srch_results{$uname}{'inststatus'}) eq 'ARRAY') {
                                        $inststatus = join(',',@{$srch_results{$uname}{'inststatus'}});
                                        $inststatus_checked = 1;
                                        last;
                                    }
                                }
                            }
                        }
                    }
                }
                if ($inststatus ne '') {
                    foreach my $status (split(/\:/,$inststatus)) {
                        if (ref($domconfig{'contacts'}{'overrides'}{$status}) eq 'HASH') {
                            my @contacts = ('adminemail','supportemail');
                            foreach my $item (@contacts) {
                                if ($domconfig{'contacts'}{'overrides'}{$status}{$item}) {
                                    my $addr = $domconfig{'contacts'}{'overrides'}{$status};
                                    if (!grep(/^\Q$addr\E$/,@recipients)) {
                                        push(@recipients,$addr);
                                    }
                                }
                            }
                            $otheremails = $domconfig{'contacts'}{'overrides'}{$status}{'others'};
                            if ($domconfig{'contacts'}{'overrides'}{$status}{'bcc'}) {
                                my @bccs = split(/,/,$domconfig{'contacts'}{'overrides'}{$status}{'bcc'});
                                my @ok_bccs;
                                foreach my $bcc (@bccs) {
                                    $bcc =~ s/^\s+//g;
                                    $bcc =~ s/\s+$//g;
                                    if ($bcc =~ m/^[^\@]+\@[^\@]+$/) {
                                        if (!(grep(/^\Q$bcc\E$/,@ok_bccs))) {
                                            push(@ok_bccs,$bcc);
                                        }
                                    }
                                }
                                if (@ok_bccs > 0) {
                                    $allbcc = join(', ',@ok_bccs);
                                }
                            }
                            $addtext = $domconfig{'contacts'}{'overrides'}{$status}{'include'};
                            last;
                        }
                    }
                }
            }
        }
    } elsif ($origmail ne '') {
        $lastresort = $origmail;
    }
    if (($mailing eq 'helpdeskmail') && ($lastresort ne '')) {
        unless (grep(/^\Q$defdom\E$/,&Apache::lonnet::current_machine_domains())) {
            my $lonhost = $Apache::lonnet::perlvar{'lonHostID'};
            my $machinedom = $Apache::lonnet::perlvar{'lonDefDomain'};
            my %what = (
                          perlvar => 1,
                       );
            my $primary = &Apache::lonnet::domain($defdom,'primary');
            if ($primary) {
                my $gotaddr;
                my ($result,$returnhash) =
                    &Apache::lonnet::get_remote_globals($primary,{ perlvar => 1 });
                if (($result eq 'ok') && (ref($returnhash) eq 'HASH')) {
                    if ($returnhash->{'lonSupportEMail'} =~ /^[^\@]+\@[^\@]+$/) {
                        $lastresort = $returnhash->{'lonSupportEMail'};
                        $gotaddr = 1;
                    }
                }
                unless ($gotaddr) {
                    my $uintdom = &Apache::lonnet::internet_dom($primary);
                    my $intdom = &Apache::lonnet::internet_dom($lonhost);
                    unless ($uintdom eq $intdom) {
                        my %domconfig =
                            &Apache::lonnet::get_dom('configuration',['contacts'],$machinedom);
                        if (ref($domconfig{'contacts'}) eq 'HASH') {
                            if (ref($domconfig{'contacts'}{'otherdomsmail'}) eq 'HASH') {
                                my @contacts = ('adminemail','supportemail');
                                foreach my $item (@contacts) {
                                    if ($domconfig{'contacts'}{'otherdomsmail'}{$item}) {
                                        my $addr = $domconfig{'contacts'}{$item};
                                        if (!grep(/^\Q$addr\E$/,@recipients)) {
                                            push(@recipients,$addr);
                                        }
                                    }
                                }
                                if ($domconfig{'contacts'}{'otherdomsmail'}{'others'}) {
                                    $otheremails = $domconfig{'contacts'}{'otherdomsmail'}{'others'};
                                }
                                if ($domconfig{'contacts'}{'otherdomsmail'}{'bcc'}) {
                                    my @bccs = split(/,/,$domconfig{'contacts'}{'otherdomsmail'}{'bcc'});
                                    my @ok_bccs;
                                    foreach my $bcc (@bccs) {
                                        $bcc =~ s/^\s+//g;
                                        $bcc =~ s/\s+$//g;
                                        if ($bcc =~ m/^[^\@]+\@[^\@]+$/) {
                                            if (!(grep(/^\Q$bcc\E$/,@ok_bccs))) {
                                                push(@ok_bccs,$bcc);
                                            }
                                        }
                                    }
                                    if (@ok_bccs > 0) {
                                        $allbcc = join(', ',@ok_bccs);
                                    }
                                }
                                $addtext = $domconfig{'contacts'}{'otherdomsmail'}{'include'};
                            }
                        }
                    }
                }
            }
        }
    }
    if (defined($defmail)) {
        if ($defmail ne '') {
            push(@recipients,$defmail);
        }
    }
    if ($otheremails) {
        my @others;
        if ($otheremails =~ /,/) {
            @others = split(/,/,$otheremails);
        } else {
            push(@others,$otheremails);
        }
        foreach my $addr (@others) {
            if (!grep(/^\Q$addr\E$/,@recipients)) {
                push(@recipients,$addr);
            }
        }
    }
    if ($mailing eq 'helpdeskmail') {
        if ((!@recipients) && ($lastresort ne '')) {
            push(@recipients,$lastresort);
        }
    } elsif ($lastresort ne '') {
        if (!grep(/^\Q$lastresort\E$/,@recipients)) {
            push(@recipients,$lastresort);
        }
    }
    my $recipientlist = join(',',@recipients);
    if (wantarray) {
        return ($recipientlist,$allbcc,$addtext);
    } else {
        return $recipientlist;
    }
}

############################################################
############################################################

=pod

=head1 Course Catalog Routines

=over 4

=item * &gather_categories()

Converts category definitions - keys of categories hash stored in  
coursecategories in configuration.db on the primary library server in a 
domain - to an array.  Also generates javascript and idx hash used to 
generate Domain Coordinator interface for editing Course Categories.

Inputs:

categories (reference to hash of category definitions).

cats (reference to array of arrays/hashes which encapsulates hierarchy of
      categories and subcategories).

idx (reference to hash of counters used in Domain Coordinator interface for 
      editing Course Categories).

jsarray (reference to array of categories used to create Javascript arrays for
         Domain Coordinator interface for editing Course Categories).

Returns: nothing

Side effects: populates cats, idx and jsarray. 

=cut

sub gather_categories {
    my ($categories,$cats,$idx,$jsarray) = @_;
    my %counters;
    my $num = 0;
    foreach my $item (keys(%{$categories})) {
        my ($cat,$container,$depth) = map { &unescape($_); } split(/:/,$item);
        if ($container eq '' && $depth == 0) {
            $cats->[$depth][$categories->{$item}] = $cat;
        } else {
            $cats->[$depth]{$container}[$categories->{$item}] = $cat;
        }
        my ($escitem,$tail) = split(/:/,$item,2);
        if ($counters{$tail} eq '') {
            $counters{$tail} = $num;
            $num ++;
        }
        if (ref($idx) eq 'HASH') {
            $idx->{$item} = $counters{$tail};
        }
        if (ref($jsarray) eq 'ARRAY') {
            push(@{$jsarray->[$counters{$tail}]},$item);
        }
    }
    return;
}

=pod

=item * &extract_categories()

Used to generate breadcrumb trails for course categories.

Inputs:

categories (reference to hash of category definitions).

cats (reference to array of arrays/hashes which encapsulates hierarchy of
      categories and subcategories).

trails (reference to array of breacrumb trails for each category).

allitems (reference to hash - key is category key 
         (format: escaped(name):escaped(parent category):depth in hierarchy).

idx (reference to hash of counters used in Domain Coordinator interface for
      editing Course Categories).

jsarray (reference to array of categories used to create Javascript arrays for
         Domain Coordinator interface for editing Course Categories).

subcats (reference to hash of arrays containing all subcategories within each 
         category, -recursive)

maxd (reference to hash used to hold max depth for all top-level categories).

Returns: nothing

Side effects: populates trails and allitems hash references.

=cut

sub extract_categories {
    my ($categories,$cats,$trails,$allitems,$idx,$jsarray,$subcats,$maxd) = @_;
    if (ref($categories) eq 'HASH') {
        &gather_categories($categories,$cats,$idx,$jsarray);
        if (ref($cats->[0]) eq 'ARRAY') {
            for (my $i=0; $i<@{$cats->[0]}; $i++) {
                my $name = $cats->[0][$i];
                my $item = &escape($name).'::0';
                my $trailstr;
                if ($name eq 'instcode') {
                    $trailstr = &mt('Official courses (with institutional codes)');
                } elsif ($name eq 'communities') {
                    $trailstr = &mt('Communities');
                } else {
                    $trailstr = $name;
                }
                if ($allitems->{$item} eq '') {
                    push(@{$trails},$trailstr);
                    $allitems->{$item} = scalar(@{$trails})-1;
                }
                my @parents = ($name);
                if (ref($cats->[1]{$name}) eq 'ARRAY') {
                    for (my $j=0; $j<@{$cats->[1]{$name}}; $j++) {
                        my $category = $cats->[1]{$name}[$j];
                        if (ref($subcats) eq 'HASH') {
                            push(@{$subcats->{$item}},&escape($category).':'.&escape($name).':1');
                        }
                        &recurse_categories($cats,2,$category,$trails,$allitems,\@parents,$subcats,$maxd);
                    }
                } else {
                    if (ref($subcats) eq 'HASH') {
                        $subcats->{$item} = [];
                    }
                    if (ref($maxd) eq 'HASH') {
                        $maxd->{$name} = 1;
                    }
                }
            }
        }
    }
    return;
}

=pod

=item * &recurse_categories()

Recursively used to generate breadcrumb trails for course categories.

Inputs:

cats (reference to array of arrays/hashes which encapsulates hierarchy of
      categories and subcategories).

depth (current depth in hierarchy of categories and sub-categories - 0 indexed).

category (current course category, for which breadcrumb trail is being generated).

trails (reference to array of breadcrumb trails for each category).

allitems (reference to hash - key is category key
         (format: escaped(name):escaped(parent category):depth in hierarchy).

parents (array containing containers directories for current category, 
         back to top level). 

Returns: nothing

Side effects: populates trails and allitems hash references

=cut

sub recurse_categories {
    my ($cats,$depth,$category,$trails,$allitems,$parents,$subcats,$maxd) = @_;
    my $shallower = $depth - 1;
    if (ref($cats->[$depth]{$category}) eq 'ARRAY') {
        for (my $k=0; $k<@{$cats->[$depth]{$category}}; $k++) {
            my $name = $cats->[$depth]{$category}[$k];
            my $item = &escape($category).':'.&escape($parents->[-1]).':'.$shallower;
            my $trailstr = join(' &raquo; ',(@{$parents},$category));
            if ($allitems->{$item} eq '') {
                push(@{$trails},$trailstr);
                $allitems->{$item} = scalar(@{$trails})-1;
            }
            my $deeper = $depth+1;
            push(@{$parents},$category);
            if (ref($subcats) eq 'HASH') {
                my $subcat = &escape($name).':'.$category.':'.$depth;
                for (my $j=@{$parents}; $j>=0; $j--) {
                    my $higher;
                    if ($j > 0) {
                        $higher = &escape($parents->[$j]).':'.
                                  &escape($parents->[$j-1]).':'.$j;
                    } else {
                        $higher = &escape($parents->[$j]).'::'.$j;
                    }
                    push(@{$subcats->{$higher}},$subcat);
                }
            }
            &recurse_categories($cats,$deeper,$name,$trails,$allitems,$parents,
                                $subcats,$maxd);
            pop(@{$parents});
        }
    } else {
        my $item = &escape($category).':'.&escape($parents->[-1]).':'.$shallower;
        my $trailstr = join(' &raquo; ',(@{$parents},$category));
        if ($allitems->{$item} eq '') {
            push(@{$trails},$trailstr);
            $allitems->{$item} = scalar(@{$trails})-1;
        }
        if (ref($maxd) eq 'HASH') {
            if ($depth > $maxd->{$parents->[0]}) {
                $maxd->{$parents->[0]} = $depth;
            }
        }
    }
    return;
}

=pod

=item * &assign_categories_table()

Create a datatable for display of hierarchical categories in a domain,
with checkboxes to allow a course to be categorized. 

Inputs:

cathash - reference to hash of categories defined for the domain (from
          configuration.db)

currcat - scalar with an & separated list of categories assigned to a course. 

type    - scalar contains course type (Course or Community).

disabled - scalar (optional) contains disabled="disabled" if input elements are
           to be readonly (e.g., Domain Helpdesk role viewing course settings).

Returns: $output (markup to be displayed) 

=cut

sub assign_categories_table {
    my ($cathash,$currcat,$type,$disabled) = @_;
    my $output;
    if (ref($cathash) eq 'HASH') {
        my (@cats,@trails,%allitems,%idx,@jsarray,%maxd,@path,$maxdepth);
        &extract_categories($cathash,\@cats,\@trails,\%allitems,\%idx,\@jsarray,\%maxd);
        $maxdepth = scalar(@cats);
        if (@cats > 0) {
            my $itemcount = 0;
            if (ref($cats[0]) eq 'ARRAY') {
                my @currcategories;
                if ($currcat ne '') {
                    @currcategories = split('&',$currcat);
                }
                my $table;
                for (my $i=0; $i<@{$cats[0]}; $i++) {
                    my $parent = $cats[0][$i];
                    next if ($parent eq 'instcode');
                    if ($type eq 'Community') {
                        next unless ($parent eq 'communities');
                    } else {
                        next if ($parent eq 'communities');
                    }
                    my $css_class = $itemcount%2?' class="LC_odd_row"':'';
                    my $item = &escape($parent).'::0';
                    my $checked = '';
                    if (@currcategories > 0) {
                        if (grep(/^\Q$item\E$/,@currcategories)) {
                            $checked = ' checked="checked"';
                        }
                    }
                    my $parent_title = $parent;
                    if ($parent eq 'communities') {
                        $parent_title = &mt('Communities');
                    }
                    $table .= '<tr '.$css_class.'><td><span class="LC_nobreak">'.
                              '<input type="checkbox" name="usecategory" value="'.
                              $item.'"'.$checked.$disabled.' />'.$parent_title.'</span>'.
                              '<input type="hidden" name="catname" value="'.$parent.'" /></td>';
                    my $depth = 1;
                    push(@path,$parent);
                    $table .= &assign_category_rows($itemcount,\@cats,$depth,$parent,\@path,\@currcategories,$disabled);
                    pop(@path);
                    $table .= '</tr><tr><td colspan="'.$maxdepth.'" class="LC_row_separator"></td></tr>';
                    $itemcount ++;
                }
                if ($itemcount) {
                    $output = &Apache::loncommon::start_data_table().
                              $table.
                              &Apache::loncommon::end_data_table();
                }
            }
        }
    }
    return $output;
}

=pod

=item * &assign_category_rows()

Create a datatable row for display of nested categories in a domain,
with checkboxes to allow a course to be categorized,called recursively.

Inputs:

itemcount - track row number for alternating colors

cats - reference to array of arrays/hashes which encapsulates hierarchy of
      categories and subcategories.

depth - current depth in hierarchy of categories and sub-categories - 0 indexed.

parent - parent of current category item

path - Array containing all categories back up through the hierarchy from the
       current category to the top level.

currcategories - reference to array of current categories assigned to the course

disabled - scalar (optional) contains disabled="disabled" if input elements are
           to be readonly (e.g., Domain Helpdesk role viewing course settings).

Returns: $output (markup to be displayed).

=cut

sub assign_category_rows {
    my ($itemcount,$cats,$depth,$parent,$path,$currcategories,$disabled) = @_;
    my ($text,$name,$item,$chgstr);
    if (ref($cats) eq 'ARRAY') {
        my $maxdepth = scalar(@{$cats});
        if (ref($cats->[$depth]) eq 'HASH') {
            if (ref($cats->[$depth]{$parent}) eq 'ARRAY') {
                my $numchildren = @{$cats->[$depth]{$parent}};
                my $css_class = $itemcount%2?' class="LC_odd_row"':'';
                $text .= '<td><table class="LC_data_table">';
                for (my $j=0; $j<$numchildren; $j++) {
                    $name = $cats->[$depth]{$parent}[$j];
                    $item = &escape($name).':'.&escape($parent).':'.$depth;
                    my $deeper = $depth+1;
                    my $checked = '';
                    if (ref($currcategories) eq 'ARRAY') {
                        if (@{$currcategories} > 0) {
                            if (grep(/^\Q$item\E$/,@{$currcategories})) {
                                $checked = ' checked="checked"';
                            }
                        }
                    }
                    $text .= '<tr><td><span class="LC_nobreak"><label>'.
                             '<input type="checkbox" name="usecategory" value="'.
                             $item.'"'.$checked.$disabled.' />'.$name.'</label></span>'.
                             '<input type="hidden" name="catname" value="'.$name.'" />'.
                             '</td><td>';
                    if (ref($path) eq 'ARRAY') {
                        push(@{$path},$name);
                        $text .= &assign_category_rows($itemcount,$cats,$deeper,$name,$path,$currcategories,$disabled);
                        pop(@{$path});
                    }
                    $text .= '</td></tr>';
                }
                $text .= '</table></td>';
            }
        }
    }
    return $text;
}

=pod

=back

=cut

############################################################
############################################################


sub commit_customrole {
    my ($udom,$uname,$url,$three,$four,$five,$start,$end,$context) = @_;
    my $output = &mt('Assigning custom role').' "'.$five.'" by '.$four.':'.$three.' in '.$url.
                         ($start?', '.&mt('starting').' '.localtime($start):'').
                         ($end?', ending '.localtime($end):'').': <b>'.
              &Apache::lonnet::assigncustomrole(
                 $udom,$uname,$url,$three,$four,$five,$end,$start,undef,undef,$context).
                 '</b><br />';
    return $output;
}

sub commit_standardrole {
    my ($udom,$uname,$url,$three,$start,$end,$one,$two,$sec,$context,$credits) = @_;
    my ($output,$logmsg,$linefeed);
    if ($context eq 'auto') {
        $linefeed = "\n";
    } else {
        $linefeed = "<br />\n";
    }  
    if ($three eq 'st') {
        my $result = &commit_studentrole(\$logmsg,$udom,$uname,$url,$three,$start,$end,
                                         $one,$two,$sec,$context,$credits);
        if (($result =~ /^error/) || ($result eq 'not_in_class') || 
            ($result eq 'unknown_course') || ($result eq 'refused')) {
            $output = $logmsg.' '.&mt('Error: ').$result."\n"; 
        } else {
            $output = $logmsg.$linefeed.&mt('Assigning').' '.$three.' in '.$url.
               ($start?', '.&mt('starting').' '.localtime($start):'').
               ($end?', '.&mt('ending').' '.localtime($end):'').': ';
            if ($context eq 'auto') {
                $output .= $result.$linefeed.&mt('Add to classlist').': ok';
            } else {
               $output .= '<b>'.$result.'</b>'.$linefeed.
               &mt('Add to classlist').': <b>ok</b>';
            }
            $output .= $linefeed;
        }
    } else {
        $output = &mt('Assigning').' '.$three.' in '.$url.
               ($start?', '.&mt('starting').' '.localtime($start):'').
               ($end?', '.&mt('ending').' '.localtime($end):'').': ';
        my $result = &Apache::lonnet::assignrole($udom,$uname,$url,$three,$end,$start,'','',$context);
        if ($context eq 'auto') {
            $output .= $result.$linefeed;
        } else {
            $output .= '<b>'.$result.'</b>'.$linefeed;
        }
    }
    return $output;
}

sub commit_studentrole {
    my ($logmsg,$udom,$uname,$url,$three,$start,$end,$one,$two,$sec,$context,
        $credits) = @_;
    my ($result,$linefeed,$oldsecurl,$newsecurl);
    if ($context eq 'auto') {
        $linefeed = "\n";
    } else {
        $linefeed = '<br />'."\n";
    }
    if (defined($one) && defined($two)) {
        my $cid=$one.'_'.$two;
        my $oldsec=&Apache::lonnet::getsection($udom,$uname,$cid);
        my $secchange = 0;
        my $expire_role_result;
        my $modify_section_result;
        if ($oldsec ne '-1') { 
            if ($oldsec ne $sec) {
                $secchange = 1;
                my $now = time;
                my $uurl='/'.$cid;
                $uurl=~s/\_/\//g;
                if ($oldsec) {
                    $uurl.='/'.$oldsec;
                }
                $oldsecurl = $uurl;
                $expire_role_result = 
                    &Apache::lonnet::assignrole($udom,$uname,$uurl,'st',$now,
                                                '','','',$context);
                if ($env{'request.course.sec'} ne '') { 
                    if ($expire_role_result eq 'refused') {
                        my @roles = ('st');
                        my @statuses = ('previous');
                        my @roledoms = ($one);
                        my $withsec = 1;
                        my %roleshash = 
                            &Apache::lonnet::get_my_roles($uname,$udom,'userroles',
                                              \@statuses,\@roles,\@roledoms,$withsec);
                        if (defined ($roleshash{$two.':'.$one.':st:'.$oldsec})) {
                            my ($oldstart,$oldend) = 
                                split(':',$roleshash{$two.':'.$one.':st:'.$oldsec});
                            if ($oldend > 0 && $oldend <= $now) {
                                $expire_role_result = 'ok';
                            }
                        }
                    }
                }
                $result = $expire_role_result;
            }
        }
        if (($expire_role_result eq 'ok') || ($secchange == 0)) {
            $modify_section_result = 
                &Apache::lonnet::modify_student_enrollment($udom,$uname,undef,undef,
                                                           undef,undef,undef,$sec,
                                                           $end,$start,'','',$cid,
                                                           '',$context,$credits);
            if ($modify_section_result =~ /^ok/) {
                if ($secchange == 1) {
                    if ($sec eq '') {
                        $$logmsg .= &mt('Section for [_1] switched from (possibly expired) old section: [_2] to student role without a section.',$uname,$oldsec).$linefeed;
                    } else {
                        $$logmsg .= &mt('Section for [_1] switched from (possibly expired) old section: [_2] to new section: [_3].',$uname,$oldsec,$sec).$linefeed;
                    }
                } elsif ($oldsec eq '-1') {
                    if ($sec eq '') {
                        $$logmsg .= &mt('New student role without a section for [_1] in course [_2].',$uname,$cid).$linefeed;
                    } else {
                        $$logmsg .= &mt('New student role for [_1] in section [_2] in course [_3].',$uname,$sec,$cid).$linefeed;
                    }
                } else {
                    if ($sec eq '') {
                        $$logmsg .= &mt('Student [_1] assigned to course [_2] without a section.',$uname,$cid).$linefeed;
                    } else {
                        $$logmsg .= &mt('Student [_1] assigned to section [_2] in course [_3].',$uname,$sec,$cid).$linefeed;
                    }
                }
            } else {
                if ($secchange) {       
                    $$logmsg .= &mt('Error when attempting section change for [_1] from old section "[_2]" to new section: "[_3]" in course [_4] -error:',$uname,$oldsec,$sec,$cid).' '.$modify_section_result.$linefeed;
                } else {
                    $$logmsg .= &mt('Error when attempting to modify role for [_1] for section: "[_2]" in course [_3] -error:',$uname,$sec,$cid).' '.$modify_section_result.$linefeed;
                }
            }
            $result = $modify_section_result;
        } elsif ($secchange == 1) {
            if ($oldsec eq '') {
                $$logmsg .= &mt('Error when attempting to expire existing role without a section for [_1] in course [_2] -error: ',$uname,$cid).' '.$expire_role_result.$linefeed;
            } else {
                $$logmsg .= &mt('Error when attempting to expire existing role for [_1] in section [_2] in course [_3] -error: ',$uname,$oldsec,$cid).' '.$expire_role_result.$linefeed;
            }
            if ($expire_role_result eq 'refused') {
                my $newsecurl = '/'.$cid;
                $newsecurl =~ s/\_/\//g;
                if ($sec ne '') {
                    $newsecurl.='/'.$sec;
                }
                if (&Apache::lonnet::allowed('cst',$newsecurl) && !(&Apache::lonnet::allowed('cst',$oldsecurl))) {
                    if ($sec eq '') {
                        $$logmsg .= &mt('Although your current role has privileges to add students to section "[_1]", you do not have privileges to modify existing enrollments unaffiliated with any section.',$sec).$linefeed;
                    } else {
                        $$logmsg .= &mt('Although your current role has privileges to add students to section "[_1]", you do not have privileges to modify existing enrollments in other sections.',$sec).$linefeed;
                    }
                }
            }
        }
    } else {
        $$logmsg .= &mt('Incomplete course id defined.').$linefeed.&mt('Addition of user [_1] from domain [_2] to course [_3], section [_4] not completed.',$uname,$udom,$one.'_'.$two,$sec).$linefeed;
        $result = "error: incomplete course id\n";
    }
    return $result;
}

sub show_role_extent {
    my ($scope,$context,$role) = @_;
    $scope =~ s{^/}{};
    my @courseroles = &Apache::lonuserutils::roles_by_context('course',1);
    push(@courseroles,'co');
    my @authorroles = &Apache::lonuserutils::roles_by_context('author');
    if (($context eq 'course') || (grep(/^\Q$role\E/,@courseroles))) {
        $scope =~ s{/}{_};
        return '<span class="LC_cusr_emph">'.$env{'course.'.$scope.'.description'}.'</span>';
    } elsif (($context eq 'author') || (grep(/^\Q$role\E/,@authorroles))) {
        my ($audom,$auname) = split(/\//,$scope);
        return &mt('[_1] Author Space','<span class="LC_cusr_emph">'.
                   &Apache::loncommon::plainname($auname,$audom).'</span>');
    } else {
        $scope =~ s{/$}{};
        return &mt('Domain: [_1]','<span class="LC_cusr_emph">'.
                   &Apache::lonnet::domain($scope,'description').'</span>');
    }
}

############################################################
############################################################

sub check_clone {
    my ($args,$linefeed) = @_;
    my $cloneid='/'.$args->{'clonedomain'}.'/'.$args->{'clonecourse'};
    my ($clonecrsudom,$clonecrsunum)= &LONCAPA::split_courseid($cloneid);
    my $clonehome=&Apache::lonnet::homeserver($clonecrsunum,$clonecrsudom);
    my $clonemsg;
    my $can_clone = 0;
    my $lctype = lc($args->{'crstype'});
    if ($lctype ne 'community') {
        $lctype = 'course';
    }
    if ($clonehome eq 'no_host') {
        if ($args->{'crstype'} eq 'Community') {
            $clonemsg = &mt('No new community created.').$linefeed.&mt('A new community could not be cloned from the specified original - [_1] - because it is a non-existent community.',$args->{'clonecourse'}.':'.$args->{'clonedomain'});
        } else {
            $clonemsg = &mt('No new course created.').$linefeed.&mt('A new course could not be cloned from the specified original - [_1] - because it is a non-existent course.',$args->{'clonecourse'}.':'.$args->{'clonedomain'});
        }     
    } else {
	my %clonedesc = &Apache::lonnet::coursedescription($cloneid,{'one_time' => 1});
        if ($args->{'crstype'} eq 'Community') {
            if ($clonedesc{'type'} ne 'Community') {
                 $clonemsg = &mt('No new community created.').$linefeed.&mt('A new community could not be cloned from the specified original - [_1] - because it is a course not a community.',$args->{'clonecourse'}.':'.$args->{'clonedomain'});
                return ($can_clone, $clonemsg, $cloneid, $clonehome);
            }
        }
	if (($env{'request.role.domain'} eq $args->{'clonedomain'}) &&
            (&Apache::lonnet::allowed('ccc',$env{'request.role.domain'}))) {
	    $can_clone = 1;
	} else {
	    my %clonehash = &Apache::lonnet::get('environment',['cloners','internal.coursecode'],
						 $args->{'clonedomain'},$args->{'clonecourse'});
            if ($clonehash{'cloners'} eq '') {
                my %domdefs = &Apache::lonnet::get_domain_defaults($args->{'course_domain'});
                if ($domdefs{'canclone'}) {
                    unless ($domdefs{'canclone'} eq 'none') {
                        if ($domdefs{'canclone'} eq 'domain') {
                            if ($args->{'ccdomain'} eq $args->{'clonedomain'}) {
                                $can_clone = 1;
                            }
                        } elsif (($clonehash{'internal.coursecode'}) && ($args->{'crscode'}) &&
                                 ($args->{'clonedomain'} eq  $args->{'course_domain'})) {
                            if (&Apache::lonnet::default_instcode_cloning($args->{'clonedomain'},$domdefs{'canclone'},
                                                                          $clonehash{'internal.coursecode'},$args->{'crscode'})) {
                                $can_clone = 1;
                            }
                        }
                    }
                }
            } else {
	        my @cloners = split(/,/,$clonehash{'cloners'});
                if (grep(/^\*$/,@cloners)) {
                    $can_clone = 1;
                } elsif (grep(/^\*\:\Q$args->{'ccdomain'}\E$/,@cloners)) {
                    $can_clone = 1;
                } elsif (grep(/^\Q$args->{'ccuname'}\E:\Q$args->{'ccdomain'}\E$/,@cloners)) {
                    $can_clone = 1;
                }
                unless ($can_clone) {
                    if (($clonehash{'internal.coursecode'}) && ($args->{'crscode'}) &&
                        ($args->{'clonedomain'} eq  $args->{'course_domain'})) {
                        my (%gotdomdefaults,%gotcodedefaults);
                        foreach my $cloner (@cloners) {
                            if (($cloner ne '*') && ($cloner !~ /^\*\:$match_domain$/) &&
                                ($cloner !~ /^$match_username\:$match_domain$/) && ($cloner ne '')) {
                                my (%codedefaults,@code_order);
                                if (ref($gotcodedefaults{$args->{'clonedomain'}}) eq 'HASH') {
                                    if (ref($gotcodedefaults{$args->{'clonedomain'}}{'defaults'}) eq 'HASH') {
                                        %codedefaults = %{$gotcodedefaults{$args->{'clonedomain'}}{'defaults'}};
                                    }
                                    if (ref($gotcodedefaults{$args->{'clonedomain'}}{'order'}) eq 'ARRAY') {
                                        @code_order = @{$gotcodedefaults{$args->{'clonedomain'}}{'order'}};
                                    }
                                } else {
                                    &Apache::lonnet::auto_instcode_defaults($args->{'clonedomain'},
                                                                            \%codedefaults,
                                                                            \@code_order);
                                    $gotcodedefaults{$args->{'clonedomain'}}{'defaults'} = \%codedefaults;
                                    $gotcodedefaults{$args->{'clonedomain'}}{'order'} = \@code_order;
                                }
                                if (@code_order > 0) {
                                    if (&Apache::lonnet::check_instcode_cloning(\%codedefaults,\@code_order,
                                                                                $cloner,$clonehash{'internal.coursecode'},
                                                                                $args->{'crscode'})) {
                                        $can_clone = 1;
                                        last;
                                    }
                                }
                            }
                        }
                    }
                }
            }
            unless ($can_clone) {
                my $ccrole = 'cc';
                if ($args->{'crstype'} eq 'Community') {
                    $ccrole = 'co';
                }
                my %roleshash =
                    &Apache::lonnet::get_my_roles($args->{'ccuname'},
                                                  $args->{'ccdomain'},
                                                  'userroles',['active'],[$ccrole],
                                                  [$args->{'clonedomain'}]);
                if ($roleshash{$args->{'clonecourse'}.':'.$args->{'clonedomain'}.':'.$ccrole}) {
                    $can_clone = 1;
                } elsif (&Apache::lonnet::is_course_owner($args->{'clonedomain'},$args->{'clonecourse'},
                                                          $args->{'ccuname'},$args->{'ccdomain'})) {
                    $can_clone = 1;
                }
            }
            unless ($can_clone) {
                if ($args->{'crstype'} eq 'Community') {
                    $clonemsg = &mt('No new community created.').$linefeed.&mt('The new community could not be cloned from the existing community because the new community owner ([_1]) does not have cloning rights in the existing community ([_2]).',$args->{'ccuname'}.':'.$args->{'ccdomain'},$clonedesc{'description'});
                } else {
                    $clonemsg = &mt('No new course created.').$linefeed.&mt('The new course could not be cloned from the existing course because the new course owner ([_1]) does not have cloning rights in the existing course ([_2]).',$args->{'ccuname'}.':'.$args->{'ccdomain'},$clonedesc{'description'});
	        }
	    }
        }
    }
    return ($can_clone, $clonemsg, $cloneid, $clonehome);
}

sub construct_course {
    my ($args,$logmsg,$courseid,$crsudom,$crsunum,$udom,$uname,$context,
        $cnum,$category,$coderef) = @_;
    my $outcome;
    my $linefeed =  '<br />'."\n";
    if ($context eq 'auto') {
        $linefeed = "\n";
    }

#
# Are we cloning?
#
    my ($can_clone, $clonemsg, $cloneid, $clonehome);
    if (($args->{'clonecourse'}) && ($args->{'clonedomain'})) {
	($can_clone, $clonemsg, $cloneid, $clonehome) = &check_clone($args,$linefeed);
	if ($context ne 'auto') {
            if ($clonemsg ne '') {
	        $clonemsg = '<span class="LC_error">'.$clonemsg.'</span>';
            }
	}
	$outcome .= $clonemsg.$linefeed;

        if (!$can_clone) {
	    return (0,$outcome);
	}
    }

#
# Open course
#
    my $crstype = lc($args->{'crstype'});
    my %cenv=();
    $$courseid=&Apache::lonnet::createcourse($args->{'course_domain'},
                                             $args->{'cdescr'},
                                             $args->{'curl'},
                                             $args->{'course_home'},
                                             $args->{'nonstandard'},
                                             $args->{'crscode'},
                                             $args->{'ccuname'}.':'.
                                             $args->{'ccdomain'},
                                             $args->{'crstype'},
                                             $cnum,$context,$category);

    # Note: The testing routines depend on this being output; see 
    # Utils::Course. This needs to at least be output as a comment
    # if anyone ever decides to not show this, and Utils::Course::new
    # will need to be suitably modified.
    $outcome .= &mt('New LON-CAPA [_1] ID: [_2]',$crstype,$$courseid).$linefeed;
    if ($$courseid =~ /^error:/) {
        return (0,$outcome);
    }

#
# Check if created correctly
#
    ($$crsudom,$$crsunum)= &LONCAPA::split_courseid($$courseid);
    my $crsuhome=&Apache::lonnet::homeserver($$crsunum,$$crsudom);
    if ($crsuhome eq 'no_host') {
        $outcome .= &mt('Course creation failed, unrecognized course home server.').$linefeed;
        return (0,$outcome);
    }
    $outcome .= &mt('Created on').': '.$crsuhome.$linefeed;

#
# Do the cloning
#   
    if ($can_clone && $cloneid) {
	$clonemsg = &mt('Cloning [_1] from [_2]',$crstype,$clonehome);
	if ($context ne 'auto') {
	    $clonemsg = '<span class="LC_success">'.$clonemsg.'</span>';
	}
	$outcome .= $clonemsg.$linefeed;
	my %oldcenv=&Apache::lonnet::dump('environment',$$crsudom,$$crsunum);
# Copy all files
	&Apache::lonclonecourse::copycoursefiles($cloneid,$$courseid,$args->{'datemode'},$args->{'dateshift'});
# Restore URL
	$cenv{'url'}=$oldcenv{'url'};
# Restore title
	$cenv{'description'}=$oldcenv{'description'};
# Restore creation date, creator and creation context.
        $cenv{'internal.created'}=$oldcenv{'internal.created'};
        $cenv{'internal.creator'}=$oldcenv{'internal.creator'};
        $cenv{'internal.creationcontext'}=$oldcenv{'internal.creationcontext'};
# Mark as cloned
	$cenv{'clonedfrom'}=$cloneid;
# Need to clone grading mode
        my %newenv=&Apache::lonnet::get('environment',['grading'],$$crsudom,$$crsunum);
        $cenv{'grading'}=$newenv{'grading'};
# Do not clone these environment entries
        &Apache::lonnet::del('environment',
                  ['default_enrollment_start_date',
                   'default_enrollment_end_date',
                   'question.email',
                   'policy.email',
                   'comment.email',
                   'pch.users.denied',
                   'plc.users.denied',
                   'hidefromcat',
                   'checkforpriv',
                   'categories'],
                   $$crsudom,$$crsunum);
        if ($args->{'textbook'}) {
            $cenv{'internal.textbook'} = $args->{'textbook'};
        }
    }

#
# Set environment (will override cloned, if existing)
#
    my @sections = ();
    my @xlists = ();
    if ($args->{'crstype'}) {
        $cenv{'type'}=$args->{'crstype'};
    }
    if ($args->{'crsid'}) {
        $cenv{'courseid'}=$args->{'crsid'};
    }
    if ($args->{'crscode'}) {
        $cenv{'internal.coursecode'}=$args->{'crscode'};
    }
    if ($args->{'crsquota'} ne '') {
        $cenv{'internal.coursequota'}=$args->{'crsquota'};
    } else {
        $cenv{'internal.coursequota'}=$args->{'crsquota'} = 20;
    }
    if ($args->{'ccuname'}) {
        $cenv{'internal.courseowner'} = $args->{'ccuname'}.
                                        ':'.$args->{'ccdomain'};
    } else {
        $cenv{'internal.courseowner'} = $args->{'curruser'};
    }
    if ($args->{'defaultcredits'}) {
        $cenv{'internal.defaultcredits'} = $args->{'defaultcredits'};
    }
    my @badclasses = (); # Used to accumulate sections/crosslistings that did not pass classlist access check for course owner.
    my @oklcsecs = (); # Used to accumulate LON-CAPA sections for validated institutional sections.
    if ($args->{'crssections'}) {
        $cenv{'internal.sectionnums'} = '';
        if ($args->{'crssections'} =~ m/,/) {
            @sections = split/,/,$args->{'crssections'};
        } else {
            $sections[0] = $args->{'crssections'};
        }
        if (@sections > 0) {
            foreach my $item (@sections) {
                my ($sec,$gp) = split/:/,$item;
                my $class = $args->{'crscode'}.$sec;
                my $addcheck = &Apache::lonnet::auto_new_course($$crsunum,$$crsudom,$class,$cenv{'internal.courseowner'});
                $cenv{'internal.sectionnums'} .= $item.',';
                if ($addcheck eq 'ok') {
                    unless (grep(/^\Q$gp\E$/,@oklcsecs)) {
                        push(@oklcsecs,$gp);
                    }
                } else {
                    push(@badclasses,$class);
                }
            }
            $cenv{'internal.sectionnums'} =~ s/,$//;
        }
    }
# do not hide course coordinator from staff listing, 
# even if privileged
    $cenv{'nothideprivileged'}=$args->{'ccuname'}.':'.$args->{'ccdomain'};
# add course coordinator's domain to domains to check for privileged users
# if different to course domain
    if ($$crsudom ne $args->{'ccdomain'}) {
        $cenv{'checkforpriv'} = $args->{'ccdomain'};
    }
# add crosslistings
    if ($args->{'crsxlist'}) {
        $cenv{'internal.crosslistings'}='';
        if ($args->{'crsxlist'} =~ m/,/) {
            @xlists = split/,/,$args->{'crsxlist'};
        } else {
            $xlists[0] = $args->{'crsxlist'};
        }
        if (@xlists > 0) {
            foreach my $item (@xlists) {
                my ($xl,$gp) = split/:/,$item;
                my $addcheck =  &Apache::lonnet::auto_new_course($$crsunum,$$crsudom,$xl,$cenv{'internal.courseowner'});
                $cenv{'internal.crosslistings'} .= $item.',';
                if ($addcheck eq 'ok') {
                    unless (grep(/^\Q$gp\E$/,@oklcsecs)) {
                        push(@oklcsecs,$gp);
                    }
                } else {
                    push(@badclasses,$xl);
                }
            }
            $cenv{'internal.crosslistings'} =~ s/,$//;
        }
    }
    if ($args->{'autoadds'}) {
        $cenv{'internal.autoadds'}=$args->{'autoadds'};
    }
    if ($args->{'autodrops'}) {
        $cenv{'internal.autodrops'}=$args->{'autodrops'};
    }
# check for notification of enrollment changes
    my @notified = ();
    if ($args->{'notify_owner'}) {
        if ($args->{'ccuname'} ne '') {
            push(@notified,$args->{'ccuname'}.':'.$args->{'ccdomain'});
        }
    }
    if ($args->{'notify_dc'}) {
        if ($uname ne '') { 
            push(@notified,$uname.':'.$udom);
        }
    }
    if (@notified > 0) {
        my $notifylist;
        if (@notified > 1) {
            $notifylist = join(',',@notified);
        } else {
            $notifylist = $notified[0];
        }
        $cenv{'internal.notifylist'} = $notifylist;
    }
    if (@badclasses > 0) {
        my %lt=&Apache::lonlocal::texthash(
                'tclb' => 'The courses listed below were included as sections or crosslistings affiliated with your new LON-CAPA course.',
                'howi' => 'However, if automated course roster updates are enabled for this class, these particular sections/crosslistings are not guaranteed to contribute towards enrollment.',
                'itis' => 'It is possible that rights to access enrollment for these classes will be available through assignment of co-owners.',
        );
        my $badclass_msg = $lt{'tclb'}.$linefeed.$lt{'howi'}.$linefeed.
                           &mt('That is because the user identified as the course owner ([_1]) does not have rights to access enrollment in these classes, as determined by the policies of your institution on access to official classlists',$cenv{'internal.courseowner'}).$linefeed.$lt{'itis'};
        if ($context eq 'auto') {
            $outcome .= $badclass_msg.$linefeed;
        } else {
            $outcome .= '<div class="LC_warning">'.$badclass_msg.$linefeed.'<ul>'."\n";
        }
        foreach my $item (@badclasses) {
            if ($context eq 'auto') {
                $outcome .= " - $item\n";
            } else {
                $outcome .= "<li>$item</li>\n";
            }
        }
        if ($context eq 'auto') {
            $outcome .= $linefeed;
        } else {
            $outcome .= "</ul><br /><br /></div>\n";
        }
    }
    if ($args->{'no_end_date'}) {
        $args->{'endaccess'} = 0;
    }
#  If an official course with institutional sections is created by cloning
#  an existing course, section-specific hiding of course totals in student's
#  view of grades as copied from cloned course, will be checked for valid
#  sections.
    if (($can_clone && $cloneid) &&
        ($cenv{'internal.coursecode'} ne '') &&
        ($cenv{'grading'} eq 'standard') &&
        ($cenv{'hidetotals'} ne '') &&
        ($cenv{'hidetotals'} ne 'all')) {
        my @hidesecs;
        my $deletehidetotals;
        if (@oklcsecs) {
            foreach my $sec (split(/,/,$cenv{'hidetotals'})) {
                if (grep(/^\Q$sec$/,@oklcsecs)) {
                    push(@hidesecs,$sec);
                }
            }
            if (@hidesecs) {
                $cenv{'hidetotals'} = join(',',@hidesecs);
            } else {
                $deletehidetotals = 1;
            }
        } else {
            $deletehidetotals = 1;
        }
        if ($deletehidetotals) {
            delete($cenv{'hidetotals'});
            &Apache::lonnet::del('environment',['hidetotals'],$$crsudom,$$crsunum);
        }
    }
    $cenv{'internal.autostart'}=$args->{'enrollstart'};
    $cenv{'internal.autoend'}=$args->{'enrollend'};
    $cenv{'default_enrollment_start_date'}=$args->{'startaccess'};
    $cenv{'default_enrollment_end_date'}=$args->{'endaccess'};
    if ($args->{'showphotos'}) {
      $cenv{'internal.showphotos'}=$args->{'showphotos'};
    }
    $cenv{'internal.authtype'} = $args->{'authtype'};
    $cenv{'internal.autharg'} = $args->{'autharg'}; 
    if ( ($cenv{'internal.authtype'} =~ /^krb/) && ($cenv{'internal.autoadds'} == 1)) {
        if (! defined($cenv{'internal.autharg'}) || $cenv{'internal.autharg'}  eq '') {
            my $krb_msg = &mt('As you did not include the default Kerberos domain to be used for authentication in this class, the institutional data used by the automated enrollment process must include the Kerberos domain for each new student'); 
            if ($context eq 'auto') {
                $outcome .= $krb_msg;
            } else {
                $outcome .= '<span class="LC_error">'.$krb_msg.'</span>';
            }
            $outcome .= $linefeed;
        }
    }
    if (($args->{'ccdomain'}) && ($args->{'ccuname'})) {
       if ($args->{'setpolicy'}) {
           $cenv{'policy.email'}=$args->{'ccuname'}.':'.$args->{'ccdomain'};
       }
       if ($args->{'setcontent'}) {
           $cenv{'question.email'}=$args->{'ccuname'}.':'.$args->{'ccdomain'};
       }
       if ($args->{'setcomment'}) {
           $cenv{'comment.email'}=$args->{'ccuname'}.':'.$args->{'ccdomain'};
       }
    }
    if ($args->{'reshome'}) {
	$cenv{'reshome'}=$args->{'reshome'}.'/';
	$cenv{'reshome'}=~s/\/+$/\//;
    }
#
# course has keyed access
#
    if ($args->{'setkeys'}) {
       $cenv{'keyaccess'}='yes';
    }
# if specified, key authority is not course, but user
# only active if keyaccess is yes
    if ($args->{'keyauth'}) {
	my ($user,$domain) = split(':',$args->{'keyauth'});
	$user = &LONCAPA::clean_username($user);
	$domain = &LONCAPA::clean_username($domain);
	if ($user ne '' && $domain ne '') {
	    $cenv{'keyauth'}=$user.':'.$domain;
	}
    }

#
#  generate and store uniquecode (available to course requester), if course should have one.
#
    if ($args->{'uniquecode'}) {
        my ($code,$error) = &make_unique_code($$crsudom,$$crsunum);
        if ($code) {
            $cenv{'internal.uniquecode'} = $code;
            my %crsinfo =
                &Apache::lonnet::courseiddump($$crsudom,'.',1,'.','.',$$crsunum,undef,undef,'.');
            if (ref($crsinfo{$$crsudom.'_'.$$crsunum}) eq 'HASH') {
                $crsinfo{$$crsudom.'_'.$$crsunum}{'uniquecode'} = $code;
                my $putres = &Apache::lonnet::courseidput($$crsudom,\%crsinfo,$crsuhome,'notime');
            }
            if (ref($coderef)) {
                $$coderef = $code;
            }
        }
    }

    if ($args->{'disresdis'}) {
        $cenv{'pch.roles.denied'}='st';
    }
    if ($args->{'disablechat'}) {
        $cenv{'plc.roles.denied'}='st';
    }

    # Record we've not yet viewed the Course Initialization Helper for this 
    # course
    $cenv{'course.helper.not.run'} = 1;
    #
    # Use new Randomseed
    #
    $cenv{'rndseed'}=&Apache::lonnet::latest_rnd_algorithm_id();;
    $cenv{'receiptalg'}=&Apache::lonnet::latest_receipt_algorithm_id();;
    #
    # The encryption code and receipt prefix for this course
    #
    $cenv{'internal.encseed'}=$Apache::lonnet::perlvar{'lonReceipt'}.$$.time.int(rand(9999));
    $cenv{'internal.encpref'}=100+int(9*rand(99));
    #
    # By default, use standard grading
    if (!defined($cenv{'grading'})) { $cenv{'grading'} = 'standard'; }

    $outcome .= $linefeed.&mt('Setting environment').': '.                 
          &Apache::lonnet::put('environment',\%cenv,$$crsudom,$$crsunum).$linefeed;
#
# Open all assignments
#
    if ($args->{'openall'}) {
       my $opendate = time;
       if ($args->{'openallfrom'} =~ /^\d+$/) {
           $opendate = $args->{'openallfrom'};
       }
       my $storeunder=$$crsudom.'_'.$$crsunum.'.0.opendate';
       my %storecontent = ($storeunder         => $opendate,
                           $storeunder.'.type' => 'date_start');
       $outcome .= &mt('All assignments open starting [_1]',
                       &Apache::lonlocal::locallocaltime($opendate)).': '.
                   &Apache::lonnet::cput
                       ('resourcedata',\%storecontent,$$crsudom,$$crsunum).$linefeed;
   }
#
# Set first page
#
    unless (($args->{'nonstandard'}) || ($args->{'firstres'} eq 'blank')
	    || ($cloneid)) {
	use LONCAPA::map;
	$outcome .= &mt('Setting first resource').': ';

	my $map = '/uploaded/'.$$crsudom.'/'.$$crsunum.'/default.sequence';
        my ($errtext,$fatal)=&LONCAPA::map::mapread($map);

        $outcome .= ($fatal?$errtext:'read ok').' - ';
        my $title; my $url;
        if ($args->{'firstres'} eq 'syl') {
	    $title=&mt('Syllabus');
            $url='/public/'.$$crsudom.'/'.$$crsunum.'/syllabus';
        } else {
            $title=&mt('Table of Contents');
            $url='/adm/navmaps';
        }

        $LONCAPA::map::resources[1]=$title.':'.$url.':false:start:res';
	(my $outtext,$errtext) = &LONCAPA::map::storemap($map,1);

	if ($errtext) { $fatal=2; }
        $outcome .= ($fatal?$errtext:'write ok').$linefeed;
    }

    return (1,$outcome);
}

sub make_unique_code {
    my ($cdom,$cnum) = @_;
    # get lock on uniquecodes db
    my $lockhash = {
                      $cnum."\0".'uniquecodes' => $env{'user.name'}.
                                                  ':'.$env{'user.domain'},
                   };
    my $tries = 0;
    my $gotlock = &Apache::lonnet::newput_dom('uniquecodes',$lockhash,$cdom);
    my ($code,$error);

    while (($gotlock ne 'ok') && ($tries<3)) {
        $tries ++;
        sleep 1;
        $gotlock = &Apache::lonnet::newput_dom('uniquecodes',$lockhash,$cdom);
    }
    if ($gotlock eq 'ok') {
        my %currcodes = &Apache::lonnet::dump_dom('uniquecodes',$cdom);
        my $gotcode;
        my $attempts = 0;
        while ((!$gotcode) && ($attempts < 100)) {
            $code = &generate_code();
            if (!exists($currcodes{$code})) {
                $gotcode = 1;
                unless (&Apache::lonnet::newput_dom('uniquecodes',{ $code => $cnum },$cdom) eq 'ok') {
                    $error = 'nostore';
                }
            }
            $attempts ++;
        }
        my @del_lock = ($cnum."\0".'uniquecodes');
        my $dellockoutcome = &Apache::lonnet::del_dom('uniquecodes',\@del_lock,$cdom);
    } else {
        $error = 'nolock';
    }
    return ($code,$error);
}

sub generate_code {
    my $code;
    my @letts = qw(B C D G H J K M N P Q R S T V W X Z);
    for (my $i=0; $i<6; $i++) {
        my $lettnum = int (rand 2);
        my $item = '';
        if ($lettnum) {
            $item = $letts[int( rand(18) )];
        } else {
            $item = 1+int( rand(8) );
        }
        $code .= $item;
    }
    return $code;
}

############################################################
############################################################

#SD
# only Community and Course, or anything else?
sub course_type {
    my ($cid) = @_;
    if (!defined($cid)) {
        $cid = $env{'request.course.id'};
    }
    if (defined($env{'course.'.$cid.'.type'})) {
        return $env{'course.'.$cid.'.type'};
    } else {
        return 'Course';
    }
}

sub group_term {
    my $crstype = &course_type();
    my %names = (
                  'Course' => 'group',
                  'Community' => 'group',
                );
    return $names{$crstype};
}

sub course_types {
    my @types = ('official','unofficial','community','textbook');
    my %typename = (
                         official   => 'Official course',
                         unofficial => 'Unofficial course',
                         community  => 'Community',
                         textbook   => 'Textbook course',
                   );
    return (\@types,\%typename);
}

sub icon {
    my ($file)=@_;
    my $curfext = lc((split(/\./,$file))[-1]);
    my $iconname=$Apache::lonnet::perlvar{'lonIconsURL'}.'/unknown.gif';
    my $embstyle = &Apache::loncommon::fileembstyle($curfext);
    if (!(!defined($embstyle) || $embstyle eq 'unk' || $embstyle eq 'hdn')) {
	if (-e  $Apache::lonnet::perlvar{'lonDocRoot'}.'/'.
	          $Apache::lonnet::perlvar{'lonIconsURL'}.'/'.
	            $curfext.".gif") {
	    $iconname=$Apache::lonnet::perlvar{'lonIconsURL'}.'/'.
		$curfext.".gif";
	}
    }
    return &lonhttpdurl($iconname);
} 

sub lonhttpdurl {
#
# Had been used for "small fry" static images on separate port 8080.
# Modify here if lightweight http functionality desired again.
# Currently eliminated due to increasing firewall issues.
#
    my ($url)=@_;
    return $url;
}

sub connection_aborted {
    my ($r)=@_;
    $r->print(" ");$r->rflush();
    my $c = $r->connection;
    return $c->aborted();
}

#    Escapes strings that may have embedded 's that will be put into
#    strings as 'strings'.
sub escape_single {
    my ($input) = @_;
    $input =~ s/\\/\\\\/g;	# Escape the \'s..(must be first)>
    $input =~ s/\'/\\\'/g;	# Esacpe the 's....
    return $input;
}

#  Same as escape_single, but escape's "'s  This 
#  can be used for  "strings"
sub escape_double {
    my ($input) = @_;
    $input =~ s/\\/\\\\/g;	# Escape the /'s..(must be first)>
    $input =~ s/\"/\\\"/g;	# Esacpe the "s....
    return $input;
}
 
#   Escapes the last element of a full URL.
sub escape_url {
    my ($url)   = @_;
    my @urlslices = split(/\//, $url,-1);
    my $lastitem = &escape(pop(@urlslices));
    return &HTML::Entities::encode(join('/',@urlslices),"'").'/'.$lastitem;
}

sub compare_arrays {
    my ($arrayref1,$arrayref2) = @_;
    my (@difference,%count);
    @difference = ();
    %count = ();
    if ((ref($arrayref1) eq 'ARRAY') && (ref($arrayref2) eq 'ARRAY')) {
        foreach my $element (@{$arrayref1}, @{$arrayref2}) { $count{$element}++; }
        foreach my $element (keys(%count)) {
            if ($count{$element} == 1) {
                push(@difference,$element);
            }
        }
    }
    return @difference;
}

sub lon_status_items {
    my %defaults = (
                     E         => 100,
                     W         => 4,
                     N         => 1,
                     U         => 5,
                     threshold => 200,
                     sysmail   => 2500,
                   );
    my %names = (
                   E => 'Errors',
                   W => 'Warnings',
                   N => 'Notices',
                   U => 'Unsent',
                );
    return (\%defaults,\%names);
}

# -------------------------------------------------------- Initialize user login
sub init_user_environment {
    my ($r, $username, $domain, $authhost, $form, $args) = @_;
    my $lonids=$Apache::lonnet::perlvar{'lonIDsDir'};

    my $public=($username eq 'public' && $domain eq 'public');

# See if old ID present, if so, remove

    my ($filename,$cookie,$userroles,$firstaccenv,$timerintenv);
    my $now=time;

    if ($public) {
	my $max_public=100;
	my $oldest;
	my $oldest_time=0;
	for(my $next=1;$next<=$max_public;$next++) {
	    if (-e $lonids."/publicuser_$next.id") {
		my $mtime=(stat($lonids."/publicuser_$next.id"))[9];
		if ($mtime<$oldest_time || !$oldest_time) {
		    $oldest_time=$mtime;
		    $oldest=$next;
		}
	    } else {
		$cookie="publicuser_$next";
		last;
	    }
	}
	if (!$cookie) { $cookie="publicuser_$oldest"; }
    } else {
	# if this isn't a robot, kill any existing non-robot sessions
	if (!$args->{'robot'}) {
	    opendir(DIR,$lonids);
	    while ($filename=readdir(DIR)) {
		if ($filename=~/^$username\_\d+\_$domain\_$authhost\.id$/) {
                    if (tie(my %oldenv,'GDBM_File',"$lonids/$filename",
                            &GDBM_READER(),0640)) {
                        my $linkedfile;
                        if (exists($oldenv{'user.linkedenv'})) {
                            $linkedfile = $oldenv{'user.linkedenv'};
                        }
                        untie(%oldenv);
                        if (unlink("$lonids/$filename")) {
                            if ($linkedfile =~ /^[a-f0-9]+_linked$/) {
                                if (-l "$lonids/$linkedfile.id") {
                                    unlink("$lonids/$linkedfile.id");
                                }
                            }
                        }
                    } else {
                        unlink($lonids.'/'.$filename);
                    }
		}
	    }
	    closedir(DIR);
# If there is a undeleted lockfile for the user's paste buffer remove it.
            my $namespace = 'nohist_courseeditor';
            my $lockingkey = 'paste'."\0".'locked_num';
            my %lockhash = &Apache::lonnet::get($namespace,[$lockingkey],
                                                $domain,$username);
            if (exists($lockhash{$lockingkey})) {
                my $delresult = &Apache::lonnet::del($namespace,[$lockingkey],$domain,$username);
                unless ($delresult eq 'ok') {
                    &Apache::lonnet::logthis("Failed to delete paste buffer locking key in $namespace for ".$username.":".$domain." Result was: $delresult");
                }
            }
	}
# Give them a new cookie
	my $id = ($args->{'robot'} ? 'robot'.$args->{'robot'}
		                   : $now.$$.int(rand(10000)));
	$cookie="$username\_$id\_$domain\_$authhost";
    
# Initialize roles

	($userroles,$firstaccenv,$timerintenv) = 
            &Apache::lonnet::rolesinit($domain,$username,$authhost);
    }
# ------------------------------------ Check browser type and MathML capability

    my ($httpbrowser,$clientbrowser,$clientversion,$clientmathml,$clientunicode,
        $clientos,$clientmobile,$clientinfo,$clientosversion) = &decode_user_agent($r);

# ------------------------------------------------------------- Get environment

    my %userenv = &Apache::lonnet::dump('environment',$domain,$username);
    my ($tmp) = keys(%userenv);
    if ($tmp !~ /^(con_lost|error|no_such_host)/i) {
    } else {
	undef(%userenv);
    }
    if (($userenv{'interface'}) && (!$form->{'interface'})) {
	$form->{'interface'}=$userenv{'interface'};
    }
    if ($userenv{'texengine'} eq 'ttm') { $clientmathml=1; }

# --------------- Do not trust query string to be put directly into environment
    foreach my $option ('interface','localpath','localres') {
        $form->{$option}=~s/[\n\r\=]//gs;
    }
# --------------------------------------------------------- Write first profile

    {
        my $ip = &Apache::lonnet::get_requestor_ip();
	my %initial_env = 
	    ("user.name"          => $username,
	     "user.domain"        => $domain,
	     "user.home"          => $authhost,
	     "browser.type"       => $clientbrowser,
	     "browser.version"    => $clientversion,
	     "browser.mathml"     => $clientmathml,
	     "browser.unicode"    => $clientunicode,
	     "browser.os"         => $clientos,
             "browser.mobile"     => $clientmobile,
             "browser.info"       => $clientinfo,
             "browser.osversion"  => $clientosversion,
	     "server.domain"      => $Apache::lonnet::perlvar{'lonDefDomain'},
	     "request.course.fn"  => '',
	     "request.course.uri" => '',
	     "request.course.sec" => '',
	     "request.role"       => 'cm',
	     "request.role.adv"   => $env{'user.adv'},
	     "request.host"       => $ip,);

        if ($form->{'localpath'}) {
	    $initial_env{"browser.localpath"}  = $form->{'localpath'};
	    $initial_env{"browser.localres"}   = $form->{'localres'};
        }
	
	if ($form->{'interface'}) {
	    $form->{'interface'}=~s/\W//gs;
	    $initial_env{"browser.interface"} = $form->{'interface'};
	    $env{'browser.interface'}=$form->{'interface'};
	}

        if ($form->{'iptoken'}) {
            my $lonhost = $r->dir_config('lonHostID');
            $initial_env{"user.noloadbalance"} = $lonhost;
            $env{'user.noloadbalance'} = $lonhost;
        }

        if ($form->{'noloadbalance'}) {
            my @hosts = &Apache::lonnet::current_machine_ids();
            my $hosthere = $form->{'noloadbalance'};
            if (grep(/^\Q$hosthere\E$/,@hosts)) {
                $initial_env{"user.noloadbalance"} = $hosthere;
                $env{'user.noloadbalance'} = $hosthere;
            }
        }

        unless ($domain eq 'public') {
            my %is_adv = ( is_adv => $env{'user.adv'} );
            my %domdef = &Apache::lonnet::get_domain_defaults($domain);

            foreach my $tool ('aboutme','blog','webdav','portfolio') {
                $userenv{'availabletools.'.$tool} = 
                    &Apache::lonnet::usertools_access($username,$domain,$tool,'reload',
                                                      undef,\%userenv,\%domdef,\%is_adv);
            }

            foreach my $crstype ('official','unofficial','community','textbook') {
                $userenv{'canrequest.'.$crstype} =
                    &Apache::lonnet::usertools_access($username,$domain,$crstype,
                                                      'reload','requestcourses',
                                                      \%userenv,\%domdef,\%is_adv);
            }

            $userenv{'canrequest.author'} =
                &Apache::lonnet::usertools_access($username,$domain,'requestauthor',
                                                  'reload','requestauthor',
                                                  \%userenv,\%domdef,\%is_adv);
            my %reqauthor = &Apache::lonnet::get('requestauthor',['author_status','author'],
                                                 $domain,$username);
            my $reqstatus = $reqauthor{'author_status'};
            if ($reqstatus eq 'approval' || $reqstatus eq 'approved') {
                if (ref($reqauthor{'author'}) eq 'HASH') {
                    $userenv{'requestauthorqueued'} = $reqstatus.':'.
                                                      $reqauthor{'author'}{'timestamp'};
                }
            }
        }

	$env{'user.environment'} = "$lonids/$cookie.id";

	if (tie(my %disk_env,'GDBM_File',"$lonids/$cookie.id",
		 &GDBM_WRCREAT(),0640)) {
	    &_add_to_env(\%disk_env,\%initial_env);
	    &_add_to_env(\%disk_env,\%userenv,'environment.');
	    &_add_to_env(\%disk_env,$userroles);
            if (ref($firstaccenv) eq 'HASH') {
                &_add_to_env(\%disk_env,$firstaccenv);
            }
            if (ref($timerintenv) eq 'HASH') {
                &_add_to_env(\%disk_env,$timerintenv);
            }
	    if (ref($args->{'extra_env'})) {
		&_add_to_env(\%disk_env,$args->{'extra_env'});
	    }
	    untie(%disk_env);
	} else {
	    &Apache::lonnet::logthis("<span style=\"color:blue;\">WARNING: ".
			   'Could not create environment storage in lonauth: '.$!.'</span>');
	    return 'error: '.$!;
	}
    }
    $env{'request.role'}='cm';
    $env{'request.role.adv'}=$env{'user.adv'};
    $env{'browser.type'}=$clientbrowser;

    return $cookie;

}

sub _add_to_env {
    my ($idf,$env_data,$prefix) = @_;
    if (ref($env_data) eq 'HASH') {
        while (my ($key,$value) = each(%$env_data)) {
	    $idf->{$prefix.$key} = $value;
	    $env{$prefix.$key}   = $value;
        }
    }
}

# --- Get the symbolic name of a problem and the url
sub get_symb {
    my ($request,$silent) = @_;
    (my $url=$env{'form.url'}) =~ s-^https?\://($ENV{'SERVER_NAME'}|$ENV{'HTTP_HOST'})--;
    my $symb=($env{'form.symb'} ne '' ? $env{'form.symb'} : (&Apache::lonnet::symbread($url)));
    if ($symb eq '') {
        if (!$silent) {
            if (ref($request)) { 
                $request->print("Unable to handle ambiguous references:$url:.");
            }
            return ();
        }
    }
    &Apache::lonenc::check_decrypt(\$symb);
    return ($symb);
}

# --------------------------------------------------------------Get annotation

sub get_annotation {
    my ($symb,$enc) = @_;

    my $key = $symb;
    if (!$enc) {
        $key =
            &Apache::lonnet::clutter((&Apache::lonnet::decode_symb($symb))[2]);
    }
    my %annotation=&Apache::lonnet::get('nohist_annotations',[$key]);
    return $annotation{$key};
}

sub clean_symb {
    my ($symb,$delete_enc) = @_;

    &Apache::lonenc::check_decrypt(\$symb);
    my $enc = $env{'request.enc'};
    if ($delete_enc) {
        delete($env{'request.enc'});
    }

    return ($symb,$enc);
}

############################################################
############################################################

=pod

=head1 Routines for building display used to search for courses


=over 4

=item * &build_filters()

Create markup for a table used to set filters to use when selecting
courses in a domain.  Used by lonpickcourse.pm, lonmodifycourse.pm
and quotacheck.pl


Inputs:

filterlist - anonymous array of fields to include as potential filters

crstype - course type

roleelement - fifth arg in selectcourse_link() populates fifth arg in javascript: opencrsbrowser() function, used
              to pop-open a course selector (will contain "extra element").

multelement - if multiple course selections will be allowed, this will be a hidden form element: name: multiple; value: 1

filter - anonymous hash of criteria and their values

action - form action

numfiltersref - ref to scalar (count of number of elements in institutional codes -- e.g., 4 for year, semester, department, and number)

caller - caller context (e.g., set to 'modifycourse' when routine is called from lonmodifycourse.pm)

cloneruname - username of owner of new course who wants to clone

clonerudom - domain of owner of new course who wants to clone

typeelem - text to use for left column in row containing course type (i.e., Course, Community or Course/Community)

codetitlesref - reference to array of titles of components in institutional codes (official courses)

codedom - domain

formname - value of form element named "form".

fixeddom - domain, if fixed.

prevphase - value to assign to form element named "phase" when going back to the previous screen

cnameelement - name of form element in form on opener page which will receive title of selected course

cnumelement - name of form element in form on opener page which will receive courseID  of selected course

cdomelement - name of form element in form on opener page which will receive domain of selected course

setroles - includes access constraint identifier when setting a roles-based condition for acces to a portfolio file

clonetext - hidden form elements containing list of courses cloneable by intended course owner when DC creates a course

clonewarning - warning message about missing information for intended course owner when DC creates a course


Returns: $output - HTML for display of search criteria, and hidden form elements.


Side Effects: None

=cut

# ---------------------------------------------- search for courses based on last activity etc.

sub build_filters {
    my ($filterlist,$crstype,$roleelement,$multelement,$filter,$action,
        $numtitlesref,$caller,$cloneruname,$clonerudom,$typeelement,
        $codetitlesref,$codedom,$formname,$fixeddom,$prevphase,
        $cnameelement,$cnumelement,$cdomelement,$setroles,
        $clonetext,$clonewarning) = @_;
    my ($list,$jscript);
    my $onchange = 'javascript:updateFilters(this)';
    my ($domainselectform,$sincefilterform,$createdfilterform,
        $ownerdomselectform,$persondomselectform,$instcodeform,
        $typeselectform,$instcodetitle);
    if ($formname eq '') {
        $formname = $caller;
    }
    foreach my $item (@{$filterlist}) {
        unless (($item eq 'descriptfilter') || ($item eq 'instcodefilter') ||
                ($item eq 'sincefilter') || ($item eq 'createdfilter')) {
            if ($item eq 'domainfilter') {
                $filter->{$item} = &LONCAPA::clean_domain($filter->{$item});
            } elsif ($item eq 'coursefilter') {
                $filter->{$item} = &LONCAPA::clean_courseid($filter->{$item});
            } elsif ($item eq 'ownerfilter') {
                $filter->{$item} = &LONCAPA::clean_username($filter->{$item});
            } elsif ($item eq 'ownerdomfilter') {
                $filter->{'ownerdomfilter'} =
                    &LONCAPA::clean_domain($filter->{$item});
                $ownerdomselectform = &select_dom_form($filter->{'ownerdomfilter'},
                                                       'ownerdomfilter',1);
            } elsif ($item eq 'personfilter') {
                $filter->{$item} = &LONCAPA::clean_username($filter->{$item});
            } elsif ($item eq 'persondomfilter') {
                $persondomselectform = &select_dom_form($filter->{'persondomfilter'},
                                                        'persondomfilter',1);
            } else {
                $filter->{$item} =~ s/\W//g;
            }
            if (!$filter->{$item}) {
                $filter->{$item} = '';
            }
        }
        if ($item eq 'domainfilter') {
            my $allow_blank = 1;
            if ($formname eq 'portform') {
                $allow_blank=0;
            } elsif ($formname eq 'studentform') {
                $allow_blank=0;
            }
            if ($fixeddom) {
                $domainselectform = '<input type="hidden" name="domainfilter"'.
                                    ' value="'.$codedom.'" />'.
                                    &Apache::lonnet::domain($codedom,'description');
            } else {
                $domainselectform = &select_dom_form($filter->{$item},
                                                     'domainfilter',
                                                      $allow_blank,'',$onchange);
            }
        } else {
            $list->{$item} = &HTML::Entities::encode($filter->{$item},'<>&"');
        }
    }

    # last course activity filter and selection
    $sincefilterform = &timebased_select_form('sincefilter',$filter);

    # course created filter and selection
    if (exists($filter->{'createdfilter'})) {
        $createdfilterform = &timebased_select_form('createdfilter',$filter);
    }

    my %lt = &Apache::lonlocal::texthash(
                'cac' => "$crstype Activity",
                'ccr' => "$crstype Created",
                'cde' => "$crstype Title",
                'cdo' => "$crstype Domain",
                'ins' => 'Institutional Code',
                'inc' => 'Institutional Categorization',
                'cow' => "$crstype Owner/Co-owner",
                'cop' => "$crstype Personnel Includes",
                'cog' => 'Type',
             );

    if (($formname eq 'ccrs') || ($formname eq 'requestcrs')) {
        my $typeval = 'Course';
        if ($crstype eq 'Community') {
            $typeval = 'Community';
        }
        $typeselectform = '<input type="hidden" name="type" value="'.$typeval.'" />';
    } else {
        $typeselectform =  '<select name="type" size="1"';
        if ($onchange) {
            $typeselectform .= ' onchange="'.$onchange.'"';
        }
        $typeselectform .= '>'."\n";
        foreach my $posstype ('Course','Community') {
            $typeselectform.='<option value="'.$posstype.'"'.
                ($posstype eq $crstype ? ' selected="selected" ' : ''). ">".&mt($posstype)."</option>\n";
        }
        $typeselectform.="</select>";
    }

    my ($cloneableonlyform,$cloneabletitle);
    if (exists($filter->{'cloneableonly'})) {
        my $cloneableon = '';
        my $cloneableoff = ' checked="checked"';
        if ($filter->{'cloneableonly'}) {
            $cloneableon = $cloneableoff;
            $cloneableoff = '';
        }
        $cloneableonlyform = '<span class="LC_nobreak"><label><input type="radio" name="cloneableonly" value="1" '.$cloneableon.'/>&nbsp;'.&mt('Required').'</label>'.('&nbsp;'x3).'<label><input type="radio" name="cloneableonly" value="" '.$cloneableoff.' />&nbsp;'.&mt('No restriction').'</label></span>';
        if ($formname eq 'ccrs') {
            $cloneabletitle = &mt('Cloneable for [_1]',$cloneruname.':'.$clonerudom);
        } else {
            $cloneabletitle = &mt('Cloneable by you');
        }
    }
    my $officialjs;
    if ($crstype eq 'Course') {
        if (exists($filter->{'instcodefilter'})) {
#            if (($fixeddom) || ($formname eq 'requestcrs') ||
#                ($formname eq 'modifycourse') || ($formname eq 'filterpicker')) {
            if ($codedom) {
                $officialjs = 1;
                ($instcodeform,$jscript,$$numtitlesref) =
                    &Apache::courseclassifier::instcode_selectors($codedom,'filterpicker',
                                                                  $officialjs,$codetitlesref);
                if ($jscript) {
                    $jscript = '<script type="text/javascript">'."\n".
                               '// <![CDATA['."\n".
                               $jscript."\n".
                               '// ]]>'."\n".
                               '</script>'."\n";
                }
            }
            if ($instcodeform eq '') {
                $instcodeform =
                    '<input type="text" name="instcodefilter" size="10" value="'.
                    $list->{'instcodefilter'}.'" />';
                $instcodetitle = $lt{'ins'};
            } else {
                $instcodetitle = $lt{'inc'};
            }
            if ($fixeddom) {
                $instcodetitle .= '<br />('.$codedom.')';
            }
        }
    }
    my $output = qq|
<form method="post" name="filterpicker" action="$action">
<input type="hidden" name="form" value="$formname" />
|;
    if ($formname eq 'modifycourse') {
        $output .= '<input type="hidden" name="phase" value="courselist" />'."\n".
                   '<input type="hidden" name="prevphase" value="'.
                   $prevphase.'" />'."\n";
    } elsif ($formname eq 'quotacheck') {
        $output .= qq|
<input type="hidden" name="sortby" value="" />
<input type="hidden" name="sortorder" value="" />
|;
    } else {
        my $name_input;
        if ($cnameelement ne '') {
            $name_input = '<input type="hidden" name="cnameelement" value="'.
                          $cnameelement.'" />';
        }
        $output .= qq|
<input type="hidden" name="cnumelement" value="$cnumelement" />
<input type="hidden" name="cdomelement" value="$cdomelement" />
$name_input
$roleelement
$multelement
$typeelement
|;
        if ($formname eq 'portform') {
            $output .= '<input type="hidden" name="setroles" value="'.$setroles.'" />'."\n";
        }
    }
    if ($fixeddom) {
        $output .= '<input type="hidden" name="fixeddom" value="'.$fixeddom.'" />'."\n";
    }
    $output .= "<br />\n".&Apache::lonhtmlcommon::start_pick_box();
    if ($sincefilterform) {
        $output .= &Apache::lonhtmlcommon::row_title($lt{'cac'})
                  .$sincefilterform
                  .&Apache::lonhtmlcommon::row_closure();
    }
    if ($createdfilterform) {
        $output .= &Apache::lonhtmlcommon::row_title($lt{'ccr'})
                  .$createdfilterform
                  .&Apache::lonhtmlcommon::row_closure();
    }
    if ($domainselectform) {
        $output .= &Apache::lonhtmlcommon::row_title($lt{'cdo'})
                  .$domainselectform
                  .&Apache::lonhtmlcommon::row_closure();
    }
    if ($typeselectform) {
        if (($formname eq 'ccrs') || ($formname eq 'requestcrs')) {
            $output .= $typeselectform;
        } else {
            $output .= &Apache::lonhtmlcommon::row_title($lt{'cog'})
                      .$typeselectform
                      .&Apache::lonhtmlcommon::row_closure();
        }
    }
    if ($instcodeform) {
        $output .= &Apache::lonhtmlcommon::row_title($instcodetitle)
                  .$instcodeform
                  .&Apache::lonhtmlcommon::row_closure();
    }
    if (exists($filter->{'ownerfilter'})) {
        $output .= &Apache::lonhtmlcommon::row_title($lt{'cow'}).
                   '<table><tr><td>'.&mt('Username').'<br />'.
                   '<input type="text" name="ownerfilter" size="20" value="'.
                   $list->{'ownerfilter'}.'" /></td><td>'.&mt('Domain').'<br />'.
                   $ownerdomselectform.'</td></tr></table>'.
                   &Apache::lonhtmlcommon::row_closure();
    }
    if (exists($filter->{'personfilter'})) {
        $output .= &Apache::lonhtmlcommon::row_title($lt{'cop'}).
                   '<table><tr><td>'.&mt('Username').'<br />'.
                   '<input type="text" name="personfilter" size="20" value="'.
                   $list->{'personfilter'}.'" /></td><td>'.&mt('Domain').'<br />'.
                   $persondomselectform.'</td></tr></table>'.
                   &Apache::lonhtmlcommon::row_closure();
    }
    if (exists($filter->{'coursefilter'})) {
        $output .= &Apache::lonhtmlcommon::row_title(&mt('LON-CAPA course ID'))
                  .'<input type="text" name="coursefilter" size="25" value="'
                  .$list->{'coursefilter'}.'" />'
                  .&Apache::lonhtmlcommon::row_closure();
    }
    if ($cloneableonlyform) {
        $output .= &Apache::lonhtmlcommon::row_title($cloneabletitle).
                   $cloneableonlyform.&Apache::lonhtmlcommon::row_closure();
    }
    if (exists($filter->{'descriptfilter'})) {
        $output .= &Apache::lonhtmlcommon::row_title($lt{'cde'})
                  .'<input type="text" name="descriptfilter" size="40" value="'
                  .$list->{'descriptfilter'}.'" />'
                  .&Apache::lonhtmlcommon::row_closure(1);
    }
    $output .= &Apache::lonhtmlcommon::end_pick_box().'<p>'.$clonetext."\n".
               '<input type="hidden" name="updater" value="" />'."\n".
               '<input type="submit" name="gosearch" value="'.
               &mt('Search').'" /></p>'."\n".'</form>'."\n".'<hr />'."\n";
    return $jscript.$clonewarning.$output;
}

=pod

=item * &timebased_select_form()

Create markup for a dropdown list used to select a time-based
filter e.g., Course Activity, Course Created, when searching for courses
or communities

Inputs:

item - name of form element (sincefilter or createdfilter)

filter - anonymous hash of criteria and their values

Returns: HTML for a select box contained a blank, then six time selections,
         with value set in incoming form variables currently selected.

Side Effects: None

=cut

sub timebased_select_form {
    my ($item,$filter) = @_;
    if (ref($filter) eq 'HASH') {
        $filter->{$item} =~ s/[^\d-]//g;
        if (!$filter->{$item}) { $filter->{$item}=-1; }
        return &select_form(
                            $filter->{$item},
                            $item,
                            {      '-1' => '',
                                '86400' => &mt('today'),
                               '604800' => &mt('last week'),
                              '2592000' => &mt('last month'),
                              '7776000' => &mt('last three months'),
                             '15552000' => &mt('last six months'),
                             '31104000' => &mt('last year'),
                    'select_form_order' =>
                           ['-1','86400','604800','2592000','7776000',
                            '15552000','31104000']});
    }
}

=pod

=item * &js_changer()

Create script tag containing Javascript used to submit course search form
when course type or domain is changed, and also to hide 'Searching ...' on
page load completion for page showing search result.

Inputs: None

Returns: markup containing updateFilters() and hideSearching() javascript functions.

Side Effects: None

=cut

sub js_changer {
    return <<ENDJS;
<script type="text/javascript">
// <![CDATA[
function updateFilters(caller) {
    if (typeof(caller) != "undefined") {
        document.filterpicker.updater.value = caller.name;
    }
    document.filterpicker.submit();
}

function hideSearching() {
    if (document.getElementById('searching')) {
        document.getElementById('searching').style.display = 'none';
    }
    return;
}

// ]]>
</script>

ENDJS
}

=pod

=item * &search_courses()

Process selected filters form course search form and pass to lonnet::courseiddump
to retrieve a hash for which keys are courseIDs which match the selected filters.

Inputs:

dom - domain being searched

type - course type ('Course' or 'Community' or '.' if any).

filter - anonymous hash of criteria and their values

numtitles - for institutional codes - number of categories

cloneruname - optional username of new course owner

clonerudom - optional domain of new course owner

domcloner - optional "domcloner" flag; has value=1 if user has ccc priv in domain being filtered by,
            (used when DC is using course creation form)

codetitles - reference to array of titles of components in institutional codes (official courses).

cc_clone - escaped comma separated list of courses for which course cloner has active CC role
           (and so can clone automatically)

reqcrsdom - domain of new course, where search_courses is used to identify potential courses to clone

reqinstcode - institutional code of new course, where search_courses is used to identify potential
              courses to clone

Returns: %courses - hash of courses satisfying search criteria, keys = course IDs, values are corresponding colon-separated escaped description, institutional code, owner and type.


Side Effects: None

=cut


sub search_courses {
    my ($dom,$type,$filter,$numtitles,$cloneruname,$clonerudom,$domcloner,$codetitles,
        $cc_clone,$reqcrsdom,$reqinstcode) = @_;
    my (%courses,%showcourses,$cloner);
    if (($filter->{'ownerfilter'} ne '') ||
        ($filter->{'ownerdomfilter'} ne '')) {
        $filter->{'combownerfilter'} = $filter->{'ownerfilter'}.':'.
                                       $filter->{'ownerdomfilter'};
    }
    foreach my $item ('descriptfilter','coursefilter','combownerfilter') {
        if (!$filter->{$item}) {
            $filter->{$item}='.';
        }
    }
    my $now = time;
    my $timefilter =
       ($filter->{'sincefilter'}==-1?1:$now-$filter->{'sincefilter'});
    my ($createdbefore,$createdafter);
    if (($filter->{'createdfilter'} ne '') && ($filter->{'createdfilter'} !=-1)) {
        $createdbefore = $now;
        $createdafter = $now-$filter->{'createdfilter'};
    }
    my ($instcodefilter,$regexpok);
    if ($numtitles) {
        if ($env{'form.official'} eq 'on') {
            $instcodefilter =
                &Apache::courseclassifier::instcode_search_str($dom,$numtitles,$codetitles);
            $regexpok = 1;
        } elsif ($env{'form.official'} eq 'off') {
            $instcodefilter = &Apache::courseclassifier::instcode_search_str($dom,$numtitles,$codetitles);
            unless ($instcodefilter eq '') {
                $regexpok = -1;
            }
        }
    } else {
        $instcodefilter = $filter->{'instcodefilter'};
    }
    if ($instcodefilter eq '') { $instcodefilter = '.'; }
    if ($type eq '') { $type = '.'; }

    if (($clonerudom ne '') && ($cloneruname ne '')) {
        $cloner = $cloneruname.':'.$clonerudom;
    }
    %courses = &Apache::lonnet::courseiddump($dom,
                                             $filter->{'descriptfilter'},
                                             $timefilter,
                                             $instcodefilter,
                                             $filter->{'combownerfilter'},
                                             $filter->{'coursefilter'},
                                             undef,undef,$type,$regexpok,undef,undef,
                                             undef,undef,$cloner,$cc_clone,
                                             $filter->{'cloneableonly'},
                                             $createdbefore,$createdafter,undef,
                                             $domcloner,undef,$reqcrsdom,$reqinstcode);
    if (($filter->{'personfilter'} ne '') && ($filter->{'persondomfilter'} ne '')) {
        my $ccrole;
        if ($type eq 'Community') {
            $ccrole = 'co';
        } else {
            $ccrole = 'cc';
        }
        my %rolehash = &Apache::lonnet::get_my_roles($filter->{'personfilter'},
                                                     $filter->{'persondomfilter'},
                                                     'userroles',undef,
                                                     [$ccrole,'in','ad','ep','ta','cr'],
                                                     $dom);
        foreach my $role (keys(%rolehash)) {
            my ($cnum,$cdom,$courserole) = split(':',$role);
            my $cid = $cdom.'_'.$cnum;
            if (exists($courses{$cid})) {
                if (ref($courses{$cid}) eq 'HASH') {
                    if (ref($courses{$cid}{roles}) eq 'ARRAY') {
                        if (!grep(/^\Q$courserole\E$/,@{$courses{$cid}{roles}})) {
                            push(@{$courses{$cid}{roles}},$courserole);
                        }
                    } else {
                        $courses{$cid}{roles} = [$courserole];
                    }
                    $showcourses{$cid} = $courses{$cid};
                }
            }
        }
        %courses = %showcourses;
    }
    return %courses;
}

=pod

=back

=head1 Routines for version requirements for current course.

=over 4

=item * &check_release_required()

Compares required LON-CAPA version with version on server, and
if required version is newer looks for a server with the required version.

Looks first at servers in user's owen domain; if none suitable, looks at
servers in course's domain are permitted to host sessions for user's domain.

Inputs:

$loncaparev - Version on current server (format: Major.Minor.Subrelease-datestamp)

$courseid - Course ID of current course

$rolecode - User's current role in course (for switchserver query string).

$required - LON-CAPA version needed by course (format: Major.Minor).


Returns:

$switchserver - query string tp append to /adm/switchserver call (if
                current server's LON-CAPA version is too old.

$warning - Message is displayed if no suitable server could be found.

=cut

sub check_release_required {
    my ($loncaparev,$courseid,$rolecode,$required) = @_;
    my ($switchserver,$warning);
    if ($required ne '') {
        my ($reqdmajor,$reqdminor) = ($required =~ /^(\d+)\.(\d+)$/);
        my ($major,$minor) = ($loncaparev =~ /^\'?(\d+)\.(\d+)\.[\w.\-]+\'?$/);
        if ($reqdmajor ne '' && $reqdminor ne '') {
            my $otherserver;
            if (($major eq '' && $minor eq '') ||
                (($reqdmajor > $major) || (($reqdmajor == $major) && ($reqdminor > $minor)))) {
                my ($userdomserver) = &Apache::lonnet::choose_server($env{'user.domain'},undef,$required,1);
                my $switchlcrev =
                    &Apache::lonnet::get_server_loncaparev($env{'user.domain'},
                                                           $userdomserver);
                my ($swmajor,$swminor) = ($switchlcrev =~ /^\'?(\d+)\.(\d+)\.[\w.\-]+\'?$/);
                if (($swmajor eq '' && $swminor eq '') || ($reqdmajor > $swmajor) ||
                    (($reqdmajor == $swmajor) && ($reqdminor > $swminor))) {
                    my $cdom = $env{'course.'.$courseid.'.domain'};
                    if ($cdom ne $env{'user.domain'}) {
                        my ($coursedomserver,$coursehostname) = &Apache::lonnet::choose_server($cdom,undef,$required,1);
                        my $serverhomeID = &Apache::lonnet::get_server_homeID($coursehostname);
                        my $serverhomedom = &Apache::lonnet::host_domain($serverhomeID);
                        my %defdomdefaults = &Apache::lonnet::get_domain_defaults($serverhomedom);
                        my %udomdefaults = &Apache::lonnet::get_domain_defaults($env{'user.domain'});
                        my $remoterev = &Apache::lonnet::get_server_loncaparev($serverhomedom,$coursedomserver);
                        my $canhost =
                            &Apache::lonnet::can_host_session($env{'user.domain'},
                                                              $coursedomserver,
                                                              $remoterev,
                                                              $udomdefaults{'remotesessions'},
                                                              $defdomdefaults{'hostedsessions'});

                        if ($canhost) {
                            $otherserver = $coursedomserver;
                        } else {
                            $warning = &mt('Requires LON-CAPA version [_1].',$env{'course.'.$courseid.'.internal.releaserequired'}).'<br />'. &mt("No suitable server could be found amongst servers in either your own domain or in the course's domain.");
                        }
                    } else {
                        $warning = &mt('Requires LON-CAPA version [_1].',$env{'course.'.$courseid.'.internal.releaserequired'}).'<br />'.&mt("No suitable server could be found amongst servers in your own domain (which is also the course's domain).");
                    }
                } else {
                    $otherserver = $userdomserver;
                }
            }
            if ($otherserver ne '') {
                $switchserver = 'otherserver='.$otherserver.'&amp;role='.$rolecode;
            }
        }
    }
    return ($switchserver,$warning);
}

=pod

=item * &check_release_result()

Inputs:

$switchwarning - Warning message if no suitable server found to host session.

$switchserver - query string to append to /adm/switchserver containing lonHostID
                and current role.

Returns: HTML to display with information about requirement to switch server.
         Either displaying warning with link to Roles/Courses screen or
         display link to switchserver.

=cut

sub check_release_result {
    my ($switchwarning,$switchserver) = @_;
    my $output = &start_page('Selected course unavailable on this server').
                 '<p class="LC_warning">';
    if ($switchwarning) {
        $output .= $switchwarning.'<br /><a href="/adm/roles">';
        if (&show_course()) {
            $output .= &mt('Display courses');
        } else {
            $output .= &mt('Display roles');
        }
        $output .= '</a>';
    } elsif ($switchserver) {
        $output .= &mt('This course requires a newer version of LON-CAPA than is installed on this server.').
                   '<br />'.
                   '<a href="/adm/switchserver?'.$switchserver.'">'.
                   &mt('Switch Server').
                   '</a>';
    }
    $output .= '</p>'.&end_page();
    return $output;
}

=pod

=item * &needs_coursereinit()

Determine if course contents stored for user's session needs to be
refreshed, because content has changed since "Big Hash" last tied.

Check for change is made if time last checked is more than 10 minutes ago
(by default).

Inputs:

$loncaparev - Version on current server (format: Major.Minor.Subrelease-datestamp)

$interval (optional) - Time which may elapse (in s) between last check for content
                       change in current course. (default: 600 s).

Returns: an array; first element is:

=over 4

'switch' - if content updates mean user's session
           needs to be switched to a server running a newer LON-CAPA version

'update' - if course session needs to be refreshed (i.e., Big Hash needs to be reloaded)
           on current server hosting user's session

''       - if no action required.

=back

If first item element is 'switch':

second item is $switchwarning - Warning message if no suitable server found to host session.

third item is $switchserver - query string to append to /adm/switchserver containing lonHostID
                              and current role.

otherwise: no other elements returned.

=back

=cut

sub needs_coursereinit {
    my ($loncaparev,$interval) = @_;
    return() unless ($env{'request.course.id'} && $env{'request.course.tied'});
    my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
    my $now = time;
    if ($interval eq '') {
        $interval = 600;
    }
    if (($now-$env{'request.course.timechecked'})>$interval) {
        my $lastchange = &Apache::lonnet::get_coursechange($cdom,$cnum);
        &Apache::lonnet::appenv({'request.course.timechecked'=>$now});
        if ($lastchange > $env{'request.course.tied'}) {
            my %curr_reqd_hash = &Apache::lonnet::userenvironment($cdom,$cnum,'internal.releaserequired');
            if ($curr_reqd_hash{'internal.releaserequired'} ne '') {
                my $required = $env{'course.'.$cdom.'_'.$cnum.'.internal.releaserequired'};
                if ($curr_reqd_hash{'internal.releaserequired'} ne $required) {
                    &Apache::lonnet::appenv({'course.'.$cdom.'_'.$cnum.'.internal.releaserequired' =>
                                             $curr_reqd_hash{'internal.releaserequired'}});
                    my ($switchserver,$switchwarning) =
                        &check_release_required($loncaparev,$cdom.'_'.$cnum,$env{'request.role'},
                                                $curr_reqd_hash{'internal.releaserequired'});
                    if ($switchwarning ne '' || $switchserver ne '') {
                        return ('switch',$switchwarning,$switchserver);
                    }
                }
            }
            return ('update');
        }
    }
    return ();
}

sub update_content_constraints {
    my ($cdom,$cnum,$chome,$cid) = @_;
    my %curr_reqd_hash = &Apache::lonnet::userenvironment($cdom,$cnum,'internal.releaserequired');
    my ($reqdmajor,$reqdminor) = split(/\./,$curr_reqd_hash{'internal.releaserequired'});
    my %checkresponsetypes;
    foreach my $key (keys(%Apache::lonnet::needsrelease)) {
        my ($item,$name,$value) = split(/:/,$key);
        if ($item eq 'resourcetag') {
            if ($name eq 'responsetype') {
                $checkresponsetypes{$value} = $Apache::lonnet::needsrelease{$key}
            }
        }
    }
    my $navmap = Apache::lonnavmaps::navmap->new();
    if (defined($navmap)) {
        my %allresponses;
        foreach my $res ($navmap->retrieveResources(undef,sub { $_[0]->is_problem() },1,0)) {
            my %responses = $res->responseTypes();
            foreach my $key (keys(%responses)) {
                next unless(exists($checkresponsetypes{$key}));
                $allresponses{$key} += $responses{$key};
            }
        }
        foreach my $key (keys(%allresponses)) {
            my ($major,$minor) = split(/\./,$checkresponsetypes{$key});
            if (($major > $reqdmajor) || ($major == $reqdmajor && $minor > $reqdminor)) {
                ($reqdmajor,$reqdminor) = ($major,$minor);
            }
        }
        undef($navmap);
    }
    unless (($reqdmajor eq '') && ($reqdminor eq '')) {
        &Apache::lonnet::update_released_required($reqdmajor.'.'.$reqdminor,$cdom,$cnum,$chome,$cid);
    }
    return;
}

sub allmaps_incourse {
    my ($cdom,$cnum,$chome,$cid) = @_;
    if ($cdom eq '' || $cnum eq '' || $chome eq '' || $cid eq '') {
        $cid = $env{'request.course.id'};
        $cdom = $env{'course.'.$cid.'.domain'};
        $cnum = $env{'course.'.$cid.'.num'};
        $chome = $env{'course.'.$cid.'.home'};
    }
    my %allmaps = ();
    my $lastchange =
        &Apache::lonnet::get_coursechange($cdom,$cnum);
    if ($lastchange > $env{'request.course.tied'}) {
        my ($furl,$ferr) = &Apache::lonuserstate::readmap("$cdom/$cnum");
        unless ($ferr) {
            &update_content_constraints($cdom,$cnum,$chome,$cid);
        }
    }
    my $navmap = Apache::lonnavmaps::navmap->new();
    if (defined($navmap)) {
        foreach my $res ($navmap->retrieveResources(undef,sub { $_[0]->is_map() },1,0,1)) {
            $allmaps{$res->src()} = 1;
        }
    }
    return \%allmaps;
}

sub parse_supplemental_title {
    my ($title) = @_;

    my ($foldertitle,$renametitle);
    if ($title =~ /&amp;&amp;&amp;/) {
        $title = &HTML::Entites::decode($title);
    }
    if ($title =~ m/^(\d+)___&&&___($match_username)___&&&___($match_domain)___&&&___(.*)$/) {
        $renametitle=$4;
        my ($time,$uname,$udom) = ($1,$2,$3);
        $foldertitle=&Apache::lontexconvert::msgtexconverted($4);
        my $name =  &plainname($uname,$udom);
        $name = &HTML::Entities::encode($name,'"<>&\'');
        $renametitle = &HTML::Entities::encode($renametitle,'"<>&\'');
        $title='<i>'.&Apache::lonlocal::locallocaltime($time).'</i> '.
            $name.': <br />'.$foldertitle;
    }
    if (wantarray) {
        return ($title,$foldertitle,$renametitle);
    }
    return $title;
}

sub recurse_supplemental {
    my ($cnum,$cdom,$suppmap,$numfiles,$errors) = @_;
    if ($suppmap) {
        my ($errtext,$fatal) = &LONCAPA::map::mapread('/uploaded/'.$cdom.'/'.$cnum.'/'.$suppmap);
        if ($fatal) {
            $errors ++;
        } else {
            my @order = @LONCAPA::map::order;
            if (@order > 0) {
                my @resources = @LONCAPA::map::resources;
                my @resparms = @LONCAPA::map::resparms;
                foreach my $idx (@order) {
                    my ($title,$src,$ext,$type,$status)=split(/\:/,$resources[$idx]);
                    if (($src ne '') && ($status eq 'res')) {
                        if ($src =~ m{^\Q/uploaded/$cdom/$cnum/\E(supplemental_\d+\.sequence)$}) {
                            ($numfiles,$errors) = &recurse_supplemental($cnum,$cdom,$1,$numfiles,$errors);
                        } else {
                            $numfiles ++;
                        }
                    }
                }
            }
        }
    }
    return ($numfiles,$errors);
}

sub symb_to_docspath {
    my ($symb,$navmapref) = @_;
    return unless ($symb && ref($navmapref));
    my ($mapurl,$id,$resurl) = &Apache::lonnet::decode_symb($symb);
    if ($resurl=~/\.(sequence|page)$/) {
        $mapurl=$resurl;
    } elsif ($resurl eq 'adm/navmaps') {
        $mapurl=$env{'course.'.$env{'request.course.id'}.'.url'};
    }
    my $mapresobj;
    unless (ref($$navmapref)) {
        $$navmapref = Apache::lonnavmaps::navmap->new();
    }
    if (ref($$navmapref)) {
        $mapresobj = $$navmapref->getResourceByUrl($mapurl);
    }
    $mapurl=~s{^.*/([^/]+)\.(\w+)$}{$1};
    my $type=$2;
    my $path;
    if (ref($mapresobj)) {
        my $pcslist = $mapresobj->map_hierarchy();
        if ($pcslist ne '') {
            foreach my $pc (split(/,/,$pcslist)) {
                next if ($pc <= 1);
                my $res = $$navmapref->getByMapPc($pc);
                if (ref($res)) {
                    my $thisurl = $res->src();
                    $thisurl=~s{^.*/([^/]+)\.\w+$}{$1};
                    my $thistitle = $res->title();
                    $path .= '&'.
                             &Apache::lonhtmlcommon::entity_encode($thisurl).'&'.
                             &escape($thistitle).
                             ':'.$res->randompick().
                             ':'.$res->randomout().
                             ':'.$res->encrypted().
                             ':'.$res->randomorder().
                             ':'.$res->is_page();
                }
            }
        }
        $path =~ s/^\&//;
        my $maptitle = $mapresobj->title();
        if ($mapurl eq 'default') {
            $maptitle = 'Main Content';
        }
        $path .= (($path ne '')? '&' : '').
                 &Apache::lonhtmlcommon::entity_encode($mapurl).'&'.
                 &escape($maptitle).
                 ':'.$mapresobj->randompick().
                 ':'.$mapresobj->randomout().
                 ':'.$mapresobj->encrypted().
                 ':'.$mapresobj->randomorder().
                 ':'.$mapresobj->is_page();
    } else {
        my $maptitle = &Apache::lonnet::gettitle($mapurl);
        my $ispage = (($type eq 'page')? 1 : '');
        if ($mapurl eq 'default') {
            $maptitle = 'Main Content';
        }
        $path = &Apache::lonhtmlcommon::entity_encode($mapurl).'&'.
                &escape($maptitle).':::::'.$ispage;
    }
    unless ($mapurl eq 'default') {
        $path = 'default&'.
                &escape('Main Content').
                ':::::&'.$path;
    }
    return $path;
}

sub captcha_display {
    my ($context,$lonhost,$defdom) = @_;
    my ($output,$error);
    my ($captcha,$pubkey,$privkey,$version) =
        &get_captcha_config($context,$lonhost,$defdom);
    if ($captcha eq 'original') {
        $output = &create_captcha();
        unless ($output) {
            $error = 'captcha';
        }
    } elsif ($captcha eq 'recaptcha') {
        $output = &create_recaptcha($pubkey,$version);
        unless ($output) {
            $error = 'recaptcha';
        }
    }
    return ($output,$error,$captcha,$version);
}

sub captcha_response {
    my ($context,$lonhost,$defdom) = @_;
    my ($captcha_chk,$captcha_error);
    my ($captcha,$pubkey,$privkey,$version) = &get_captcha_config($context,$lonhost,$defdom);
    if ($captcha eq 'original') {
        ($captcha_chk,$captcha_error) = &check_captcha();
    } elsif ($captcha eq 'recaptcha') {
        $captcha_chk = &check_recaptcha($privkey,$version);
    } else {
        $captcha_chk = 1;
    }
    return ($captcha_chk,$captcha_error);
}

sub get_captcha_config {
    my ($context,$lonhost,$dom_in_effect) = @_;
    my ($captcha,$pubkey,$privkey,$version,$hashtocheck);
    my $hostname = &Apache::lonnet::hostname($lonhost);
    my $serverhomeID = &Apache::lonnet::get_server_homeID($hostname);
    my $serverhomedom = &Apache::lonnet::host_domain($serverhomeID);
    if ($context eq 'usercreation') {
        my %domconfig = &Apache::lonnet::get_dom('configuration',[$context],$serverhomedom);
        if (ref($domconfig{$context}) eq 'HASH') {
            $hashtocheck = $domconfig{$context}{'cancreate'};
            if (ref($hashtocheck) eq 'HASH') {
                if ($hashtocheck->{'captcha'} eq 'recaptcha') {
                    if (ref($hashtocheck->{'recaptchakeys'}) eq 'HASH') {
                        $pubkey = $hashtocheck->{'recaptchakeys'}{'public'};
                        $privkey = $hashtocheck->{'recaptchakeys'}{'private'};
                    }
                    if ($privkey && $pubkey) {
                        $captcha = 'recaptcha';
                        $version = $hashtocheck->{'recaptchaversion'};
                        if ($version ne '2') {
                            $version = 1;
                        }
                    } else {
                        $captcha = 'original';
                    }
                } elsif ($hashtocheck->{'captcha'} ne 'notused') {
                    $captcha = 'original';
                }
            }
        } else {
            $captcha = 'captcha';
        }
    } elsif ($context eq 'login') {
        my %domconfhash = &Apache::loncommon::get_domainconf($serverhomedom);
        if ($domconfhash{$serverhomedom.'.login.captcha'} eq 'recaptcha') {
            $pubkey = $domconfhash{$serverhomedom.'.login.recaptchakeys_public'};
            $privkey = $domconfhash{$serverhomedom.'.login.recaptchakeys_private'};
            if ($privkey && $pubkey) {
                $captcha = 'recaptcha';
                $version = $domconfhash{$serverhomedom.'.login.recaptchaversion'};
                if ($version ne '2') {
                    $version = 1;
                }
            } else {
                $captcha = 'original';
            }
        } elsif ($domconfhash{$serverhomedom.'.login.captcha'} eq 'original') {
            $captcha = 'original';
        }
    } elsif ($context eq 'passwords') {
        if ($dom_in_effect) {
            my %passwdconf = &Apache::lonnet::get_passwdconf($dom_in_effect);
            if ($passwdconf{'captcha'} eq 'recaptcha') {
                if (ref($passwdconf{'recaptchakeys'}) eq 'HASH') {
                    $pubkey = $passwdconf{'recaptchakeys'}{'public'};
                    $privkey = $passwdconf{'recaptchakeys'}{'private'};
                }
                if ($privkey && $pubkey) {
                    $captcha = 'recaptcha';
                    $version = $passwdconf{'recaptchaversion'};
                    if ($version ne '2') {
                        $version = 1;
                    }
                } else {
                    $captcha = 'original';
                }
            } elsif ($passwdconf{'captcha'} ne 'notused') {
                $captcha = 'original';
            }
        }
    }
    return ($captcha,$pubkey,$privkey,$version);
}

sub create_captcha {
    my %captcha_params = &captcha_settings();
    my ($output,$maxtries,$tries) = ('',10,0);
    while ($tries < $maxtries) {
        $tries ++;
        my $captcha = Authen::Captcha->new (
                                           output_folder => $captcha_params{'output_dir'},
                                           data_folder   => $captcha_params{'db_dir'},
                                          );
        my $md5sum = $captcha->generate_code($captcha_params{'numchars'});

        if (-e $Apache::lonnet::perlvar{'lonCaptchaDir'}.'/'.$md5sum.'.png') {
            $output = '<input type="hidden" name="crypt" value="'.$md5sum.'" />'."\n".
                      '<span class="LC_nobreak">'.
                      &mt('Type in the letters/numbers shown below').'&nbsp;'.
                      '<input type="text" size="5" name="code" value="" autocomplete="new-password" />'.
                      '</span><br />'.
                      '<img src="'.$captcha_params{'www_output_dir'}.'/'.$md5sum.'.png" alt="captcha" />';
            last;
        }
    }
    if ($output eq '') {
        &Apache::lonnet::logthis("Failed to create Captcha code after $tries attempts.");
    }
    return $output;
}

sub captcha_settings {
    my %captcha_params = (
                           output_dir     => $Apache::lonnet::perlvar{'lonCaptchaDir'},
                           www_output_dir => "/captchaspool",
                           db_dir         => $Apache::lonnet::perlvar{'lonCaptchaDb'},
                           numchars       => '5',
                         );
    return %captcha_params;
}

sub check_captcha {
    my ($captcha_chk,$captcha_error);
    my $code = $env{'form.code'};
    my $md5sum = $env{'form.crypt'};
    my %captcha_params = &captcha_settings();
    my $captcha = Authen::Captcha->new(
                      output_folder => $captcha_params{'output_dir'},
                      data_folder   => $captcha_params{'db_dir'},
                  );
    $captcha_chk = $captcha->check_code($code,$md5sum);
    my %captcha_hash = (
                        0       => 'Code not checked (file error)',
                       -1      => 'Failed: code expired',
                       -2      => 'Failed: invalid code (not in database)',
                       -3      => 'Failed: invalid code (code does not match crypt)',
    );
    if ($captcha_chk != 1) {
        $captcha_error = $captcha_hash{$captcha_chk}
    }
    return ($captcha_chk,$captcha_error);
}

sub create_recaptcha {
    my ($pubkey,$version) = @_;
    if ($version >= 2) {
        return '<div class="g-recaptcha" data-sitekey="'.$pubkey.'"></div>'.
               '<div style="padding:0;clear:both;margin:0;border:0"></div>';
    } else {
        my $use_ssl;
        if ($ENV{'SERVER_PORT'} == 443) {
            $use_ssl = 1;
        }
        my $captcha = Captcha::reCAPTCHA->new;
        return $captcha->get_options_setter({theme => 'white'})."\n".
               $captcha->get_html($pubkey,undef,$use_ssl).
               &mt('If the text is hard to read, [_1] will replace them.',
                   '<img src="/res/adm/pages/refresh.gif" alt="reCAPTCHA refresh" />').
               '<br /><br />';
     }
}

sub check_recaptcha {
    my ($privkey,$version) = @_;
    my $captcha_chk;
    my $ip = &Apache::lonnet::get_requestor_ip(); 
    if ($version >= 2) {
        my $ua = LWP::UserAgent->new;
        $ua->timeout(10);
        my %info = (
                     secret   => $privkey,
                     response => $env{'form.g-recaptcha-response'},
                     remoteip => $ip,
                   );
        my $response = $ua->post('https://www.google.com/recaptcha/api/siteverify',\%info);
        if ($response->is_success)  {
            my $data = JSON::DWIW->from_json($response->decoded_content);
            if (ref($data) eq 'HASH') {
                if ($data->{'success'}) {
                    $captcha_chk = 1;
                }
            }
        }
    } else {
        my $captcha = Captcha::reCAPTCHA->new;
        my $captcha_result =
            $captcha->check_answer(
                                    $privkey,
                                    $ip,
                                    $env{'form.recaptcha_challenge_field'},
                                    $env{'form.recaptcha_response_field'},
                                  );
        if ($captcha_result->{is_valid}) {
            $captcha_chk = 1;
        }
    }
    return $captcha_chk;
}

sub emailusername_info {
    my @fields = ('firstname','lastname','institution','web','location','officialemail','id');
    my %titles = &Apache::lonlocal::texthash (
                     lastname      => 'Last Name',
                     firstname     => 'First Name',
                     institution   => 'School/college/university',
                     location      => "School's city, state/province, country",
                     web           => "School's web address",
                     officialemail => 'E-mail address at institution (if different)',
                     id            => 'Student/Employee ID',
                 );
    return (\@fields,\%titles);
}

sub cleanup_html {
    my ($incoming) = @_;
    my $outgoing;
    if ($incoming ne '') {
        $outgoing = $incoming;
        $outgoing =~ s/;/&#059;/g;
        $outgoing =~ s/\#/&#035;/g;
        $outgoing =~ s/\&/&#038;/g;
        $outgoing =~ s/</&#060;/g;
        $outgoing =~ s/>/&#062;/g;
        $outgoing =~ s/\(/&#040/g;
        $outgoing =~ s/\)/&#041;/g;
        $outgoing =~ s/"/&#034;/g;
        $outgoing =~ s/'/&#039;/g;
        $outgoing =~ s/\$/&#036;/g;
        $outgoing =~ s{/}{&#047;}g;
        $outgoing =~ s/=/&#061;/g;
        $outgoing =~ s/\\/&#092;/g
    }
    return $outgoing;
}

# Checks for critical messages and returns a redirect url if one exists.
# $interval indicates how often to check for messages.
sub critical_redirect {
    my ($interval) = @_;
    unless (($env{'user.domain'} ne '') && ($env{'user.name'} ne '')) {
        return ();
    }
    if ((time-$env{'user.criticalcheck.time'})>$interval) {
        my @what=&Apache::lonnet::dump('critical', $env{'user.domain'},
                                        $env{'user.name'});
        &Apache::lonnet::appenv({'user.criticalcheck.time'=>time});
        my $redirecturl;
        if ($what[0]) {
            if (($what[0] ne 'con_lost') && ($what[0] ne 'no_such_host') && ($what[0]!~/^error\:/)) {
                $redirecturl='/adm/email?critical=display';
                my $url=&Apache::lonnet::absolute_url().$redirecturl;
                return (1, $url);
            }
        }
    }
    return ();
}

# Use:
#   my $answer=reply("encrypt:passwd:$udom:$uname:$upass",$tryserver);
#
##################################################
#          password associated functions         #
##################################################
sub des_keys {
    # Make a new key for DES encryption.
    # Each key has two parts which are returned separately.
    # Please note:  Each key must be passed through the &hex function
    # before it is output to the web browser.  The hex versions cannot
    # be used to decrypt.
    my @hexstr=('0','1','2','3','4','5','6','7',
                '8','9','a','b','c','d','e','f');
    my $lkey='';
    for (0..7) {
        $lkey.=$hexstr[rand(15)];
    }
    my $ukey='';
    for (0..7) {
        $ukey.=$hexstr[rand(15)];
    }
    return ($lkey,$ukey);
}

sub des_decrypt {
    my ($key,$cyphertext) = @_;
    my $keybin=pack("H16",$key);
    my $cypher;
    if ($Crypt::DES::VERSION>=2.03) {
        $cypher=new Crypt::DES $keybin;
    } else {
        $cypher=new DES $keybin;
    }
    my $plaintext='';
    my $cypherlength = length($cyphertext);
    my $numchunks = int($cypherlength/32);
    for (my $j=0; $j<$numchunks; $j++) {
        my $start = $j*32;
        my $cypherblock = substr($cyphertext,$start,32);
        my $chunk =
            $cypher->decrypt(unpack("a8",pack("H16",substr($cypherblock,0,16))));
        $chunk .=
            $cypher->decrypt(unpack("a8",pack("H16",substr($cypherblock,16,16))));
        $chunk=substr($chunk,1,ord(substr($chunk,0,1)) );
        $plaintext .= $chunk;
    }
    return $plaintext;
}

sub is_nonframeable {
    my ($url,$absolute,$hostname,$ip,$nocache) = @_;
    my ($remprotocol,$remhost) = ($url =~ m{^(https?)\://(([a-z0-9]+(-[a-z0-9]+)*\.)+[a-z]{2,})}i);
    return if (($remprotocol eq '') || ($remhost eq ''));

    $remprotocol = lc($remprotocol);
    $remhost = lc($remhost);
    my $remport = 80;
    if ($remprotocol eq 'https') {
        $remport = 443;
    }
    my ($result,$cached) = &Apache::lonnet::is_cached_new('noiframe',$remhost.':'.$remport);
    if ($cached) {
        unless ($nocache) {
            if ($result) {
                return 1;
            } else {
                return 0;
            }
        }
    }
    my $uselink;
    my $request = new HTTP::Request('HEAD',$url);
    my $ua = LWP::UserAgent->new;
    $ua->timeout(5);
    my $response=$ua->request($request);
    if ($response->is_success()) {
        my $secpolicy = lc($response->header('content-security-policy'));
        my $xframeop = lc($response->header('x-frame-options'));
        $secpolicy =~ s/^\s+|\s+$//g;
        $xframeop =~ s/^\s+|\s+$//g;
        if (($secpolicy ne '') || ($xframeop ne '')) {
            my $remotehost = $remprotocol.'://'.$remhost;
            my ($origin,$protocol,$port);
            if ($ENV{'SERVER_PORT'} =~/^\d+$/) {
                $port = $ENV{'SERVER_PORT'};
            } else {
                $port = 80;
            }
            if ($absolute eq '') {
                $protocol = 'http:';
                if ($port == 443) {
                    $protocol = 'https:';
                }
                $origin = $protocol.'//'.lc($hostname);
            } else {
                $origin = lc($absolute);
                ($protocol,$hostname) = ($absolute =~ m{^(https?:)//([^/]+)$});
            }
            if (($secpolicy) && ($secpolicy =~ /\Qframe-ancestors\E([^;]*)(;|$)/)) {
                my $framepolicy = $1;
                $framepolicy =~ s/^\s+|\s+$//g;
                my @policies = split(/\s+/,$framepolicy);
                if (@policies) {
                    if (grep(/^\Q'none'\E$/,@policies)) {
                        $uselink = 1;
                    } else {
                        $uselink = 1;
                        if ((grep(/^\Q*\E$/,@policies)) || (grep(/^\Q$protocol\E$/,@policies)) ||
                                (($origin ne '') && (grep(/^\Q$origin\E$/,@policies))) ||
                                (($ip ne '') && (grep(/^\Q$ip\E$/,@policies)))) {
                            undef($uselink);
                        }
                        if ($uselink) {
                            if (grep(/^\Q'self'\E$/,@policies)) {
                                if (($origin ne '') && ($remotehost eq $origin)) {
                                    undef($uselink);
                                }
                            }
                        }
                        if ($uselink) {
                            my @possok;
                            if ($ip ne '') {
                                push(@possok,$ip);
                            }
                            my $hoststr = '';
                            foreach my $part (reverse(split(/\./,$hostname))) {
                                if ($hoststr eq '') {
                                    $hoststr = $part;
                                } else {
                                    $hoststr = "$part.$hoststr";
                                }
                                if ($hoststr eq $hostname) {
                                    push(@possok,$hostname);
                                } else {
                                    push(@possok,"*.$hoststr");
                                }
                            }
                            if (@possok) {
                                foreach my $poss (@possok) {
                                    last if (!$uselink);
                                    foreach my $policy (@policies) {
                                        if ($policy =~ m{^(\Q$protocol\E//|)\Q$poss\E(\Q:$port\E|)$}) {
                                            undef($uselink);
                                            last;
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            } elsif ($xframeop ne '') {
                $uselink = 1;
                my @policies = split(/\s*,\s*/,$xframeop);
                if (@policies) {
                    unless (grep(/^deny$/,@policies)) {
                        if ($origin ne '') {
                            if (grep(/^sameorigin$/,@policies)) {
                                if ($remotehost eq $origin) {
                                    undef($uselink);
                                }
                            }
                            if ($uselink) {
                                foreach my $policy (@policies) {
                                    if ($policy =~ /^allow-from\s*(.+)$/) {
                                        my $allowfrom = $1;
                                        if (($allowfrom ne '') && ($allowfrom eq $origin)) {
                                            undef($uselink);
                                            last;
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    if ($nocache) {
        if ($cached) {
            my $devalidate;
            if ($uselink && !$result) {
                $devalidate = 1;
            } elsif (!$uselink && $result) {
                $devalidate = 1;
            }
            if ($devalidate) {
                &Apache::lonnet::devalidate_cache_new('noiframe',$remhost.':'.$remport);
            }
        }
    } else {
        if ($uselink) {
            $result = 1;
        } else {
            $result = 0;
        }
        &Apache::lonnet::do_cache_new('noiframe',$remhost.':'.$remport,$result,3600);
    }
    return $uselink;
}

1;
__END__;

