# The LearningOnline Network with CAPA
# handler for DC-only modifiable course settings
#
# $Id: lonmodifycourse.pm,v 1.79.2.10 2024/12/30 00:34:19 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
package Apache::lonmodifycourse;

use strict;
use Apache::Constants qw(:common :http);
use Apache::lonnet;
use Apache::loncommon;
use Apache::lonhtmlcommon;
use Apache::lonlocal;
use Apache::lonuserutils;
use Apache::loncreateuser;
use Apache::lonpickcourse;
use lib '/home/httpd/lib/perl';
use LONCAPA qw(:DEFAULT :match);

my $registered_cleanup;
my $modified_dom;

sub get_dc_settable {
    my ($type,$cdom) = @_;
    if ($type eq 'Community') {
        return ('courseowner','selfenrollmgrdc','selfenrollmgrcc');
    } else {
        my @items = ('courseowner','coursecode','authtype','autharg','selfenrollmgrdc',
                     'selfenrollmgrcc','mysqltables');
        if (&showcredits($cdom)) {
            push(@items,'defaultcredits');
        }
        my %passwdconf = &Apache::lonnet::get_passwdconf($cdom);
        if (($passwdconf{'crsownerchg'}) && ($type ne 'Placement')) {
            push(@items,'nopasswdchg');
        }
        return @items;
    }
}

sub autoenroll_keys {
    my $internals = ['coursecode','courseowner','authtype','autharg','defaultcredits',
                     'autoadds','autodrops','autostart','autoend','sectionnums',
                     'crosslistings','co-owners','autodropfailsafe'];
    my $accessdates = ['default_enrollment_start_date','default_enrollment_end_date'];
    return ($internals,$accessdates);
}

sub catalog_settable {
    my ($confhash,$type) = @_;
    my @settable;
    if (ref($confhash) eq 'HASH') {
        if ($type eq 'Community') {
            if ($confhash->{'togglecatscomm'} ne 'comm') {
                push(@settable,'togglecats');
            }
            if ($confhash->{'categorizecomm'} ne 'comm') {
                push(@settable,'categorize');
            }
        } else {
            if ($confhash->{'togglecats'} ne 'crs') {
                push(@settable,'togglecats');
            }
            if ($confhash->{'categorize'} ne 'crs') {
                push(@settable,'categorize');
            }
        }
    } else {
        push(@settable,('togglecats','categorize'));
    }
    return @settable;
}

sub get_enrollment_settings {
    my ($cdom,$cnum) = @_;
    my ($internals,$accessdates) = &autoenroll_keys();
    my @items;
    if ((ref($internals) eq 'ARRAY') && (ref($accessdates) eq 'ARRAY')) { 
        @items = map { 'internal.'.$_; } (@{$internals});
        push(@items,@{$accessdates});
    }
    push(@items,'internal.nopasswdchg');
    my %settings = &Apache::lonnet::get('environment',\@items,$cdom,$cnum);
    my %enrollvar;
    $enrollvar{'autharg'} = '';
    $enrollvar{'authtype'} = '';
    foreach my $item (keys(%settings)) {
        if ($item =~ m/^internal\.(.+)$/) {
            my $type = $1;
            if ( ($type eq "autoadds") || ($type eq "autodrops") ) {
                if ($settings{$item} == 1) {
                    $enrollvar{$type} = "ON";
                } else {
                    $enrollvar{$type} = "OFF";
                }
            } elsif ( ($type eq "autostart") || ($type eq "autoend") ) {
                if ( ($type eq "autoend") && ($settings{$item} == 0) ) {
                    $enrollvar{$type} = &mt('No end date');
                } else {
                    $enrollvar{$type} = &Apache::lonlocal::locallocaltime($settings{$item});
                }
            } elsif (($type eq 'sectionnums') || ($type eq 'co-owners')) {
                $enrollvar{$type} = $settings{$item};
                $enrollvar{$type} =~ s/,/, /g;
            } elsif ($type eq "authtype"
                     || $type eq "autharg"    || $type eq "coursecode"
                     || $type eq "crosslistings" || $type eq "selfenrollmgr"
                     || $type eq "autodropfailsafe" || $type eq 'nopasswdchg') {
                $enrollvar{$type} = $settings{$item};
            } elsif ($type eq 'defaultcredits') {
                if (&showcredits($cdom)) {
                    $enrollvar{$type} = $settings{$item};
                }
            } elsif ($type eq 'courseowner') {
                if ($settings{$item} =~ /^[^:]+:[^:]+$/) {
                    $enrollvar{$type} = $settings{$item};
                } else {
                    if ($settings{$item} ne '') {
                        $enrollvar{$type} = $settings{$item}.':'.$cdom;
                    }
                }
            }
        } elsif ($item =~ m/^default_enrollment_(start|end)_date$/) {
            my $type = $1;
            if ( ($type eq 'end') && ($settings{$item} == 0) ) {
                $enrollvar{$item} = &mt('No end date');
            } elsif ( ($type eq 'start') && ($settings{$item} eq '') ) {
                $enrollvar{$item} = 'When enrolled';
            } else {
                $enrollvar{$item} = &Apache::lonlocal::locallocaltime($settings{$item});
            }
        }
    }
    return %enrollvar;
}

sub print_course_search_page {
    my ($r,$dom,$domdesc) = @_;
    my $action = '/adm/modifycourse';
    my $type = $env{'form.type'};
    if (!defined($env{'form.type'})) {
        $type = 'Course';
    }
    &print_header($r,$type);
    my ($filterlist,$filter) = &get_filters($dom);
    my ($numtitles,$cctitle,$dctitle,@codetitles);
    my $ccrole = 'cc';
    if ($type eq 'Community') {
        $ccrole = 'co';
    }
    $cctitle = &Apache::lonnet::plaintext($ccrole,$type);
    $dctitle = &Apache::lonnet::plaintext('dc');
    $r->print(&Apache::loncommon::js_changer());
    if ($type eq 'Community') {
        $r->print('<h3>'.&mt('Search for a community in the [_1] domain',$domdesc).'</h3>');
    } else {
        $r->print('<h3>'.&mt('Search for a course in the [_1] domain',$domdesc).'</h3>');
    }
    $r->print(&Apache::loncommon::build_filters($filterlist,$type,undef,undef,$filter,$action,
                                                \$numtitles,'modifycourse',undef,undef,undef,
                                                \@codetitles,$dom));

    my ($actiontext,$roleoption,$settingsoption);
    if ($type eq 'Community') {
        $actiontext = &mt('Actions available after searching for a community:');
    } else {
        $actiontext = &mt('Actions available after searching for a course:');
    }
    if (&Apache::lonnet::allowed('ccc',$dom)) {
       if ($type eq 'Community') {
           $roleoption = &mt('Enter the community with the role of [_1]',$cctitle);
           $settingsoption = &mt('View or modify community settings which only a [_1] may modify.',$dctitle);
       } else {
           $roleoption = &mt('Enter the course with the role of [_1]',$cctitle);
           $settingsoption = &mt('View or modify course settings which only a [_1] may modify.',$dctitle);
       }
    } elsif (&Apache::lonnet::allowed('rar',$dom)) {
        my ($roles_by_num,$description,$accessref,$accessinfo) = &Apache::lonnet::get_all_adhocroles($dom);
        if ((ref($roles_by_num) eq 'ARRAY') && (ref($description) eq 'HASH')) {
            if (@{$roles_by_num} > 1) {
                if ($type eq 'Community') {
                    $roleoption = &mt('Enter the community with one of the available ad hoc roles');
                } else {
                    $roleoption = &mt('Enter the course with one of the available ad hoc roles.');
                }
            } else {
                my $rolename = $description->{$roles_by_num->[0]};
                if ($type eq 'Community') {
                    $roleoption = &mt('Enter the community with the ad hoc role of: [_1]',$rolename);
                } else {
                    $roleoption = &mt('Enter the course with the ad hoc role of: [_1]',$rolename);
                }
            }
        }
        if ($type eq 'Community') {
            $settingsoption = &mt('View community settings which only a [_1] may modify.',$dctitle);
        } else {
            $settingsoption = &mt('View course settings which only a [_1] may modify.',$dctitle);
        }
    }
    $r->print($actiontext.'<ul>');
    if ($roleoption) {
        $r->print('<li>'.$roleoption.'</li>'."\n");
    }
    $r->print('<li>'.$settingsoption.'</li>'."\n".'</ul>');
    return;
}

sub print_course_selection_page {
    my ($r,$dom,$domdesc,$permission) = @_;
    my $type = $env{'form.type'};
    if (!defined($type)) {
        $type = 'Course';
    }
    &print_header($r,$type);

    if ($permission->{'adhocrole'} eq 'custom') {
        my %lt = &Apache::lonlocal::texthash(
            title    => 'Ad hoc role selection',
            preamble => 'Please choose an ad hoc role in the course.',
            cancel   => 'Click "OK" to enter the course, or "Cancel" to choose a different course.',
        );
        my %jslt = &Apache::lonlocal::texthash (
            none => 'You are not eligible to use an ad hoc role for the selected course',
            ok   => 'OK',
            exit => 'Cancel',
        );
        &js_escape(\%jslt);
        $r->print(<<"END");
<script type="text/javascript">
// <![CDATA[
\$(document).ready(function(){
    \$( "#LC_adhocrole_chooser" ).dialog({ autoOpen: false });
});

function gochoose(cname,cdom,cdesc) {
    document.courselist.pickedcourse.value = cdom+'_'+cname;
    \$("#LC_choose_adhoc").empty();
    var pickedaction = \$('input[name=phase]:checked', '#LCcoursepicker').val();
    if (pickedaction == 'adhocrole') {
        var http = new XMLHttpRequest();
        var url = "/adm/pickcourse";
        var params = "cid="+cdom+"_"+cname+"&context=adhoc";
        http.open("POST", url, true);
        http.setRequestHeader("Content-type", "application/x-www-form-urlencoded");
        http.onreadystatechange = function() {
            if (http.readyState == 4 && http.status == 200) {
                var data = \$.parseJSON(http.responseText);
                var len = data.roles.length;
                if (len == '' || len == null || len == 0) {
                    alert('$jslt{none}');
                } else {
                    if (len == 1) {
                        \$( '[name="adhocrole"]' )[0].value = data.roles[0].name;
                        document.courselist.submit();
                    } else {
                        var str = '';
                        \$("#LC_choose_adhoc").empty();
                        for (var i=0; i<data.roles.length; i++) {
                            \$("<label><input type='radio' value='"+data.roles[i].name+"' name='LC_get_role' id='LC_get_role_"+i+"' />"+data.roles[i].desc+"</label><span>&nbsp;&nbsp;</span>")
                            .appendTo("#LC_choose_adhoc");
                        }
                        \$( "#LC_adhocrole_chooser" ).toggle( true );
                        \$( "#LC_get_role_0").prop("checked", true);
                        \$( "#LC_adhocrole_chooser" ).dialog({ autoOpen: false });
                        \$( "#LC_adhocrole_chooser" ).dialog("open");
                        \$( "#LC_adhocrole_chooser" ).dialog({
                            height: 400,
                            width: 500,
                            modal: true,
                            resizable: false,
                            buttons: [
                                  {
                                   text: "$jslt{'ok'}",
                                   click: function() {
                                            var rolename = \$('input[name=LC_get_role]:checked', '#LChelpdeskpicker').val();
                                            \$( '[name="adhocrole"]' )[0].value = rolename;
                                            document.courselist.submit();
                                        }
                                  },
                                  {
                                   text: "$jslt{'exit'}",
                                   click: function() {
                                        \$("#LC_adhocrole_chooser").dialog( "close" );
                                       }
                                  }
                            ],
                        });
                        \$( "#LC_adhocrole_chooser" ).find( "form" ).on( "submit", function( event ) {
                            event.preventDefault();
                            var rolename = \$('input[name=LC_get_role]:checked', '#LChelpdeskpicker').val()
                            \$( '[name="adhocrole"]' )[0].value = rolename;
                            document.courselist.submit();
                            \$("#LC_adhocrole_chooser").dialog( "close" );
                        });
                    }
                }
            }
        }
        http.send(params);
    } else {
        document.courselist.submit();
    }
    return;
}
// ]]>
</script>

<div id="LC_adhocrole_chooser" title="$lt{'title'}" style="display:none">
  <p>$lt{'preamble'}</p>
  <form name="LChelpdeskadhoc" id="LChelpdeskpicker" action="">
    <div id="LC_choose_adhoc">
    </div>
    <input type="hidden" name="adhocrole" id="LCadhocrole" value="" />
    <input type="submit" tabindex="-1" style="position:absolute; top:-1000px" />
  </form>
  <p>$lt{'cancel'}</p>
</div>
END
    } elsif ($permission->{'adhocrole'} eq 'coord') {
        $r->print(<<"END");
<script type="text/javascript">
// <![CDATA[

function gochoose(cname,cdom,cdesc) {
    document.courselist.pickedcourse.value = cdom+'_'+cname;
    document.courselist.submit();
    return;
}

// ]]>
</script>
END
    }

# Criteria for course search
    my ($filterlist,$filter) = &get_filters();
    my $action = '/adm/modifycourse';
    my $dctitle = &Apache::lonnet::plaintext('dc');
    my ($numtitles,@codetitles);
    $r->print(&Apache::loncommon::js_changer());
    $r->print(&mt('Revise your search criteria for this domain').' ('.$domdesc.').<br />');
    $r->print(&Apache::loncommon::build_filters($filterlist,$type,undef,undef,$filter,$action,
                                                \$numtitles,'modifycourse',undef,undef,undef,
                                                \@codetitles,$dom,$env{'form.form'}));
    my %courses = &Apache::loncommon::search_courses($dom,$type,$filter,$numtitles,
                                                     undef,undef,undef,\@codetitles);
    &Apache::lonpickcourse::display_matched_courses($r,$type,0,$action,undef,undef,undef,
                                                    $dom,undef,%courses);
    return;
}

sub get_filters {
    my ($dom) = @_;
    my @filterlist = ('descriptfilter','instcodefilter','ownerfilter',
                      'ownerdomfilter','coursefilter','sincefilter');
    # created filter
    my $loncaparev = &Apache::lonnet::get_server_loncaparev($dom);
    if ($loncaparev ne 'unknown_cmd') {
        push(@filterlist,'createdfilter');
    }
    my %filter;
    foreach my $item (@filterlist) {
        $filter{$item} = $env{'form.'.$item};
    }
    return (\@filterlist,\%filter);
}

sub print_modification_menu {
    my ($r,$cdesc,$domdesc,$dom,$type,$cid,$coursehash,$permission) = @_;
    &print_header($r,$type);
    my ($ccrole,$categorytitle,$setquota_text,$setuploadquota_text,$cdom,$cnum);
    if (ref($coursehash) eq 'HASH') {
        $cdom = $coursehash->{'domain'};
        $cnum = $coursehash->{'num'};
    } else {
         ($cdom,$cnum) = split(/_/,$cid);
    }
    if ($type eq 'Community') {
        $ccrole = 'co';
    } else {
        $ccrole = 'cc';
    }
    my %linktext;
    if ($permission->{'setparms'} eq 'edit') {
        %linktext = (
                      'setquota'      => 'View/Modify quotas for group portfolio files, and for uploaded content',
                      'setanon'       => 'View/Modify responders threshold for anonymous survey submissions display',
                      'selfenroll'    => 'View/Modify Self-Enrollment configuration',
                      'setpostsubmit' => 'View/Modify submit button behavior, post-submission',
                    );
    } else {
        %linktext = (
                      'setquota'      => 'View quotas for group portfolio files, and for uploaded content',
                      'setanon'       => 'View responders threshold for anonymous survey submissions display',
                      'selfenroll'    => 'View Self-Enrollment configuration',
                      'setpostsubmit' => 'View submit button behavior, post-submission',
                    );
    }
    if ($type eq 'Community') {
        if ($permission->{'setparms'} eq 'edit') {
            $categorytitle = 'View/Modify Community Settings';
            $linktext{'setparms'} = 'View/Modify community owner';
            $linktext{'catsettings'} = 'View/Modify catalog settings for community';
        } else {
            $categorytitle = 'View Community Settings';
            $linktext{'setparms'} = 'View community owner';
            $linktext{'catsettings'} = 'View catalog settings for community';
        }
        $setquota_text = &mt('Total disk space allocated for storage of portfolio files in all groups in a community.');
        $setuploadquota_text = &mt('Disk space allocated for storage of content uploaded directly to a community via Content Editor.');
    } else {
        if ($permission->{'setparms'} eq 'edit') {
            $categorytitle = 'View/Modify Course Settings';
            $linktext{'catsettings'} = 'View/Modify catalog settings for course';
            if (($type ne 'Placement') && (&showcredits($dom))) {
                $linktext{'setparms'} = 'View/Modify course owner, institutional code, default authentication, credits, self-enrollment and table lifetime';
            } else {
                $linktext{'setparms'} = 'View/Modify course owner, institutional code, default authentication, self-enrollment and table lifetime';
            }
        } else {
            $categorytitle = 'View Course Settings';
            $linktext{'catsettings'} = 'View catalog settings for course';
            if (($type ne 'Placement') && (&showcredits($dom))) {
                $linktext{'setparms'} = 'View course owner, institutional code, default authentication, credits, self-enrollment and table lifetime';
            } else {
                $linktext{'setparms'} = 'View course owner, institutional code, default authentication, self-enrollment and table lifetime';
            }
        }
        $setquota_text = &mt('Total disk space allocated for storage of portfolio files in all groups in a course.');
        $setuploadquota_text = &mt('Disk space allocated for storage of content uploaded directly to a course via Content Editor.');
    }
    my $anon_text = &mt('Responder threshold required to display anonymous survey submissions.');
    my $postsubmit_text = &mt('Override defaults for submit button behavior post-submission for this specific course.'); 
    my $mysqltables_text = &mt('Override default for lifetime of "temporary" MySQL tables containing student performance data.');
    $linktext{'viewparms'} = 'Display current settings for automated enrollment';

    my %domconf = &Apache::lonnet::get_dom('configuration',['coursecategories'],$dom);
    my @additional_params = &catalog_settable($domconf{'coursecategories'},$type);

    sub manage_selfenrollment {
        my ($cdom,$cnum,$type,$coursehash,$permission) = @_;
        if ($permission->{'selfenroll'}) {
            my ($managed_by_cc,$managed_by_dc) = &Apache::lonuserutils::selfenrollment_administration($cdom,$cnum,$type,$coursehash);
            if (ref($managed_by_dc) eq 'ARRAY') {
                if (@{$managed_by_dc}) {
                    return 1;
                }
            }
        }
        return 0;
    }

    sub phaseurl {
        my $phase = shift;
        return "javascript:changePage(document.menu,'$phase')"
    }
    my @menu =
        ({  categorytitle => $categorytitle,
        items => [
            {
                linktext => $linktext{'setparms'},
                url => &phaseurl('setparms'),
                permission => $permission->{'setparms'},
                #help => '',
                icon => 'crsconf.png',
                linktitle => ''
            },
            {
                linktext => $linktext{'setquota'},
                url => &phaseurl('setquota'),
                permission => $permission->{'setquota'},
                #help => '',
                icon => 'groupportfolioquota.png',
                linktitle => ''
            },
            {
                linktext => $linktext{'setanon'},
                url => &phaseurl('setanon'),
                permission => $permission->{'setanon'},
                #help => '',
                icon => 'anonsurveythreshold.png',
                linktitle => ''
            },
            {
                linktext => $linktext{'catsettings'},
                url => &phaseurl('catsettings'),
                permission => (($permission->{'catsettings'}) && (@additional_params > 0)),
                #help => '',
                icon => 'ccatconf.png',
                linktitle => ''
            },
            {
                linktext => $linktext{'viewparms'},
                url => &phaseurl('viewparms'),
                permission => ($permission->{'viewparms'} && ($type ne 'Community')),
                #help => '',
                icon => 'roles.png',
                linktitle => ''
            },
            {
                linktext => $linktext{'selfenroll'},
                icon => 'self_enroll.png',
                #help => 'Course_Self_Enrollment',
                url => &phaseurl('selfenroll'),
                permission => &manage_selfenrollment($cdom,$cnum,$type,$coursehash,$permission),
                linktitle => 'Configure user self-enrollment.',
            },
            {
                linktext => $linktext{'setpostsubmit'},
                icon => 'emblem-readonly.png',
                #help => '',
                url => &phaseurl('setpostsubmit'),
                permission => $permission->{'setpostsubmit'},
                linktitle => '',
            },
        ]
        },
        );

    my $menu_html =
        '<h3>'
       .&mt('View/Modify settings for: [_1]',
                '<span class="LC_nobreak">'.$cdesc.'</span>')
       .'</h3>'."\n".'<p>';
    if ($type eq 'Community') {
        $menu_html .= &mt('Although almost all community settings can be modified by a Coordinator, the following may only be set or modified by a Domain Coordinator:');
    } else {
        $menu_html .= &mt('Although almost all course settings can be modified by a Course Coordinator, the following may only be set or modified by a Domain Coordinator:');
    }
    $menu_html .= '</p>'."\n".'<ul>';
    if ($type eq 'Community') {
        $menu_html .= '<li>'.&mt('Community owner (permitted to assign Coordinator roles in the community).').'</li>'."\n".
                      '<li>'.&mt('Override defaults for who configures self-enrollment for this specific community').'</li>'."\n";
    } else {
        $menu_html .=  '<li>'.&mt('Course owner (permitted to assign Course Coordinator roles in the course).').'</li>'."\n".
                       '<li>'.&mt("Institutional code and default authentication (both required for auto-enrollment of students from institutional datafeeds).").'</li>'."\n";
        if (&showcredits($dom)) {
            $menu_html .= '<li>'.&mt('Default credits earned by student on course completion.').'</li>'."\n";
        }
        $menu_html .= ' <li>'.&mt('Override defaults for who configures self-enrollment for this specific course.').'</li>'."\n";
    }
    $menu_html .= '<li>'.$mysqltables_text.'</li>'."\n".
                  '<li>'.$setquota_text.'</li>'."\n".
                  '<li>'.$setuploadquota_text.'</li>'."\n".
                  '<li>'.$anon_text.'</li>'."\n".
                  '<li>'.$postsubmit_text.'</li>'."\n";
    my ($categories_link_start,$categories_link_end);
    if ($permission->{'catsettings'} eq 'edit') {
        $categories_link_start = '<a href="/adm/domainprefs?actions=coursecategories&amp;phase=display">';
        $categories_link_end = '</a>';
    }
    foreach my $item (@additional_params) {
        if ($type eq 'Community') {
            if ($item eq 'togglecats') {
                $menu_html .= '  <li>'.&mt('Hiding/unhiding a community from the catalog (although can be [_1]configured[_2] to be modifiable by a Coordinator in community context).',$categories_link_start,$categories_link_end).'</li>'."\n";
            } elsif ($item eq 'categorize') {
                $menu_html .= '  <li>'.&mt('Manual cataloging of a community (although can be [_1]configured[_2] to be modifiable by a Coordinator in community context).',$categories_link_start,$categories_link_end).'</li>'."\n";
            }
        } else {
            if ($item eq 'togglecats') {
                $menu_html .= '  <li>'.&mt('Hiding/unhiding a course from the course catalog (although can be [_1]configured[_2] to be modifiable by a Course Coordinator in course context).',$categories_link_start,$categories_link_end).'</li>'."\n";
            } elsif ($item eq 'categorize') {
                $menu_html .= '  <li>'.&mt('Manual cataloging of a course (although can be [_1]configured[_2] to be modifiable by a Course Coordinator in course context).',$categories_link_start,$categories_link_end).'</li>'."\n";
            }
        }
    }
    $menu_html .=
        ' </ul>'
       .'<form name="menu" method="post" action="/adm/modifycourse">'
       ."\n"
       .&hidden_form_elements();
    
    $r->print($menu_html);
    $r->print(&Apache::lonhtmlcommon::generate_menu(@menu));
    $r->print('</form>');
    return;
}

sub print_adhocrole_selected {
    my ($r,$type,$permission) = @_;
    &print_header($r,$type);
    my ($cdom,$cnum) = split(/_/,$env{'form.pickedcourse'});
    my ($newrole,$selectrole);
    if ($permission->{'adhocrole'} eq 'coord') {
        if ($type eq 'Community') {
            $newrole = "co./$cdom/$cnum";
        } else {
            $newrole = "cc./$cdom/$cnum";
        }
        $selectrole = 1;
    } elsif ($permission->{'adhocrole'} eq 'custom') {
        my ($okroles,$description) = &Apache::lonnet::get_my_adhocroles($env{'form.pickedcourse'},1);
        if (ref($okroles) eq 'ARRAY') {
            my $possrole = $env{'form.adhocrole'};
            if (($possrole ne '') && (grep(/^\Q$possrole\E$/,@{$okroles}))) {
                my $confname = &Apache::lonnet::get_domainconfiguser($cdom);
                $newrole = "cr/$cdom/$confname/$possrole./$cdom/$cnum";
                $selectrole = 1;
            }
        }
    }
    if ($selectrole) {
        $r->print('<form name="adhocrole" method="post" action="/adm/roles">
<input type="hidden" name="selectrole" value="'.$selectrole.'" />
<input type="hidden" name="newrole" value="'.$newrole.'" />
</form>');
    } else {
        $r->print('<form name="ccrole" method="post" action="/adm/modifycourse">'.
                  '</form>');
    }
    return;
}

sub print_settings_display {
    my ($r,$cdom,$cnum,$cdesc,$type,$permission) = @_;
    my %enrollvar = &get_enrollment_settings($cdom,$cnum);
    my %longtype = &course_settings_descrip($type);
    my %lt = &Apache::lonlocal::texthash(
            'valu' => 'Current value',
            'cour' => 'Current settings are:',
            'cose' => "Settings which control auto-enrollment using classlists from your institution's student information system fall into two groups:",
            'dcon' => 'Modifiable only by Domain Coordinator',
            'back' => 'Pick another action',
    );
    my $ccrole = 'cc';
    if ($type eq 'Community') {
       $ccrole = 'co';
    }
    my $cctitle = &Apache::lonnet::plaintext($ccrole,$type);
    my $dctitle = &Apache::lonnet::plaintext('dc');
    my @modifiable_params = &get_dc_settable($type,$cdom);
    my ($internals,$accessdates) = &autoenroll_keys();
    my @items;
    if ((ref($internals) eq 'ARRAY') && (ref($accessdates) eq 'ARRAY')) {
        @items =  (@{$internals},@{$accessdates});
    }
    my $disp_table = &Apache::loncommon::start_data_table()."\n".
                     &Apache::loncommon::start_data_table_header_row()."\n".
                     "<th>&nbsp;</th>\n".
                     "<th>$lt{'valu'}</th>\n".
                     "<th>$lt{'dcon'}</th>\n".
                     &Apache::loncommon::end_data_table_header_row()."\n";
    foreach my $item (@items) {
        my $shown = $enrollvar{$item};
        if ($item eq 'crosslistings') {
            my (@xlists,@lcsecs);
            foreach my $entry (split(/,/,$enrollvar{$item})) {
                my ($xlist,$lc_sec) = split(/:/,$entry);
                push(@xlists,$xlist);
                push(@lcsecs,$lc_sec);
            }
            if (@xlists) {
                my $crskey = $cnum.':'.$enrollvar{'coursecode'};
                my %reformatted =
                    &Apache::lonnet::auto_instsec_reformat($cdom,'declutter',
                                                           {$crskey => \@xlists});
                if (ref($reformatted{$crskey}) eq 'ARRAY') {
                    my @show;
                    my @xlcodes = @{$reformatted{$crskey}};
                    for (my $i=0; $i<@xlcodes; $i++) {
                        push(@show,$xlcodes[$i].':'.$lcsecs[$i]);
                    }
                    if (@show) {
                        $shown = join(',',@show);
                    }
                }
            }
        }
        $disp_table .= &Apache::loncommon::start_data_table_row()."\n".
                       "<td><b>$longtype{$item}</b></td>\n".
                       "<td>$shown</td>\n";
        if (grep(/^\Q$item\E$/,@modifiable_params)) {
            $disp_table .= '<td align="right">'.&mt('Yes').'</td>'."\n";
        } else {
            $disp_table .= '<td align="right">'.&mt('No').'</td>'."\n";
        }
        $disp_table .= &Apache::loncommon::end_data_table_row()."\n";
    }
    $disp_table .= &Apache::loncommon::end_data_table()."\n";
    &print_header($r,$type);
    my ($enroll_link_start,$enroll_link_end,$setparms_link_start,$setparms_link_end);
    if (&Apache::lonnet::allowed('ccc',$cdom)) {
        my $newrole = $ccrole.'./'.$cdom.'/'.$cnum;
        my $escuri = &HTML::Entities::encode('/adm/roles?selectrole=1&'.$newrole.
                                             '=1&destinationurl=/adm/populate','&<>"');
        $enroll_link_start = '<a href="'.$escuri.'">';
        $enroll_link_end = '</a>';
    }
    if ($permission->{'setparms'}) {
        $setparms_link_start = '<a href="javascript:changePage(document.viewparms,'."'setparms'".');">';
        $setparms_link_end = '</a>';
    }
    $r->print('<h3>'.&mt('Current automated enrollment settings for:').
              ' <span class="LC_nobreak">'.$cdesc.'</span></h3>'.
              '<form action="/adm/modifycourse" method="post" name="viewparms">'."\n".
              '<p>'.$lt{'cose'}.'<ul>'.
              '<li>'.&mt('Settings modifiable by a [_1] via the [_2]Automated Enrollment Manager[_3] in a course.',
                         $cctitle,$enroll_link_start,$enroll_link_end).'</li>');
    if (&showcredits($cdom)) {
        $r->print('<li>'.&mt('Settings modifiable by a [_1] via [_2]View/Modify course owner, institutional code, default authentication, credits, and self-enrollment[_3].',$dctitle,$setparms_link_start,$setparms_link_end)."\n");
    } else {
        $r->print('<li>'.&mt('Settings modifiable by a [_1] via [_2]View/Modify course owner, institutional code, default authentication, and self-enrollment[_3].',$dctitle,$setparms_link_start,$setparms_link_end)."\n");
    }
    $r->print('</li></ul></p>'.
              '<p>'.$lt{'cour'}.'</p><p>'.$disp_table.'</p><p>'.
              '<a href="javascript:changePage(document.viewparms,'."'menu'".')">'.$lt{'back'}.'</a>'."\n".
              &hidden_form_elements().
              '</p></form>'
    );
}

sub print_setquota {
    my ($r,$cdom,$cnum,$cdesc,$type,$readonly) = @_;
    my $lctype = lc($type);
    my $headline = &mt("Set disk space quotas for $lctype: [_1]",
                     '<span class="LC_nobreak">'.$cdesc.'</span>');
    my %lt = &Apache::lonlocal::texthash(
                'gpqu' => 'Disk space for storage of group portfolio files',
                'upqu' => 'Disk space for storage of content directly uploaded to course via Content Editor',
                'modi' => 'Save',
                'back' => 'Pick another action',
    );
    my %staticdefaults = (
                           coursequota   => 20,
                           uploadquota   => 500,
                         );
    my %settings = &Apache::lonnet::get('environment',['internal.coursequota','internal.uploadquota','internal.coursecode'],
                                        $cdom,$cnum);
    my $coursequota = $settings{'internal.coursequota'};
    my $uploadquota = $settings{'internal.uploadquota'};
    if ($coursequota eq '') {
        $coursequota = $staticdefaults{'coursequota'};
    }
    if ($uploadquota eq '') {
        my %domdefs = &Apache::lonnet::get_domain_defaults($cdom);
        my $quotatype = &Apache::lonuserutils::get_extended_type($cdom,$cnum,$type,\%settings);
        $uploadquota = $domdefs{$quotatype.'quota'};
        if ($uploadquota eq '') {
            $uploadquota = $staticdefaults{'uploadquota'};
        }
    }
    &print_header($r,$type);
    my $hidden_elements = &hidden_form_elements();
    my $porthelpitem = &Apache::loncommon::help_open_topic('Modify_Course_Quota');
    my $uploadhelpitem = &Apache::loncommon::help_open_topic('Modify_Course_Upload_Quota');
    my ($disabled,$submit);
    if ($readonly) {
        $disabled = ' disabled="disabled"';
    } else {
        $submit = '<input type="submit" value="'.$lt{'modi'}.'" />';
    }
    $r->print(<<ENDDOCUMENT);
<form action="/adm/modifycourse" method="post" name="setquota" onsubmit="return verify_quota();">
<h3>$headline</h3>
<p><span class="LC_nobreak">
$porthelpitem $lt{'gpqu'}: <input type="text" size="4" name="coursequota" value="$coursequota" $disabled /> MB
</span>
<br />
<span class="LC_nobreak">
$uploadhelpitem $lt{'upqu'}: <input type="text" size="4" name="uploadquota" value="$uploadquota" $disabled /> MB
</span>
</p>
<p>
$submit
</p>
$hidden_elements
<a href="javascript:changePage(document.setquota,'menu')">$lt{'back'}</a>
</form>
ENDDOCUMENT
    return;
}

sub print_set_anonsurvey_threshold {
    my ($r,$cdom,$cnum,$cdesc,$type,$readonly) = @_;
    my %lt = &Apache::lonlocal::texthash(
                'resp' => 'Responder threshold for anonymous survey submissions display:',
                'sufa' => 'Anonymous survey submissions displayed when responders exceeds',
                'modi' => 'Save',
                'back' => 'Pick another action',
    );
    my %settings = &Apache::lonnet::get('environment',['internal.anonsurvey_threshold'],$cdom,$cnum);
    my $threshold = $settings{'internal.anonsurvey_threshold'};
    if ($threshold eq '') {
        my %domconfig = 
            &Apache::lonnet::get_dom('configuration',['coursedefaults'],$cdom);
        if (ref($domconfig{'coursedefaults'}) eq 'HASH') {
            $threshold = $domconfig{'coursedefaults'}{'anonsurvey_threshold'};
            if ($threshold eq '') {
                $threshold = 10;
            }
        } else {
            $threshold = 10;
        }
    }
    &print_header($r,$type);
    my $hidden_elements = &hidden_form_elements();
    my ($disabled,$submit);
    if ($readonly) {
        $disabled = ' disabled="disabled"';
    } else {
        $submit = '<input type="submit" value="'.$lt{'modi'}.'" />';
    }
    my $helpitem = &Apache::loncommon::help_open_topic('Modify_Anonsurvey_Threshold');
    $r->print(<<ENDDOCUMENT);
<form action="/adm/modifycourse" method="post" name="setanon" onsubmit="return verify_anon_threshold();">
<h3>$lt{'resp'} <span class="LC_nobreak">$cdesc</span></h3>
<p>
$helpitem $lt{'sufa'}: <input type="text" size="4" name="threshold" value="$threshold" $disabled /> &nbsp;&nbsp;&nbsp;&nbsp;
$submit
</p>
$hidden_elements
<a href="javascript:changePage(document.setanon,'menu')">$lt{'back'}</a>
</form>
ENDDOCUMENT
    return;
}

sub print_postsubmit_config {
    my ($r,$cdom,$cnum,$cdesc,$type,$readonly) = @_;
    my %lt = &Apache::lonlocal::texthash (
                'conf' => 'Configure submit button behavior after student makes a submission',
                'disa' => 'Disable submit button/keypress following student submission',
                'nums' => 'Number of seconds submit is disabled',
                'modi' => 'Save',
                'back' => 'Pick another action',
                'yes'  => 'Yes',
                'no'   => 'No',
    );
    my %settings = &Apache::lonnet::get('environment',['internal.postsubmit','internal.postsubtimeout',
                                                       'internal.coursecode','internal.textbook'],$cdom,$cnum);
    my $postsubmit = $settings{'internal.postsubmit'};
    if ($postsubmit eq '') {
        my %domconfig =
            &Apache::lonnet::get_dom('configuration',['coursedefaults'],$cdom);
        $postsubmit = 1; 
        if (ref($domconfig{'coursedefaults'}) eq 'HASH') {
            if (ref($domconfig{'coursedefaults'}{'postsubmit'}) eq 'HASH') {
                if ($domconfig{'coursedefaults'}{'postsubmit'}{'client'} eq 'off') {
                    $postsubmit = 0; 
                }
            }
        }
    }
    my ($checkedon,$checkedoff,$display);
    if ($postsubmit) {
        $checkedon = 'checked="checked"';
        $display = 'block';
    } else {
        $checkedoff = 'checked="checked"';
        $display = 'none';
    }
    my $postsubtimeout = $settings{'internal.postsubtimeout'};
    my $default = &domain_postsubtimeout($cdom,$type,\%settings);
    my $zero = &mt('(Enter 0 to disable until next page reload, or leave blank to use the domain default: [_1])',$default);
    if ($postsubtimeout eq '') {
        $postsubtimeout = $default;
    }
    &print_header($r,$type);
    my $hidden_elements = &hidden_form_elements();
    my ($disabled,$submit);
    if ($readonly) {
        $disabled = ' disabled="disabled"';
    } else {
        $submit = '<input type="submit" value="'.$lt{'modi'}.'" />';
    }
    my $helpitem = &Apache::loncommon::help_open_topic('Modify_Postsubmit_Config');
    $r->print(<<ENDDOCUMENT);
<form action="/adm/modifycourse" method="post" name="setpostsubmit" onsubmit="return verify_postsubmit();">
<h3>$lt{'conf'} <span class="LC_nobreak">($cdesc)</span></h3>
<p>
$helpitem $lt{'disa'}: 
<label><input type="radio" name="postsubmit" $checkedon onclick="togglePostsubmit('studentsubmission');" value="1" $disabled />
$lt{'yes'}</label>&nbsp;&nbsp;
<label><input type="radio" name="postsubmit" $checkedoff onclick="togglePostsubmit('studentsubmission');" value="0" $disabled />
$lt{'no'}</label>
<div id="studentsubmission" style="display: $display">
$lt{'nums'} <input type="text" name="postsubtimeout" value="$postsubtimeout" $disabled /><br />
$zero</div>
<br />     
$submit
</p>
$hidden_elements
<a href="javascript:changePage(document.setpostsubmit,'menu')">$lt{'back'}</a>
</form>
ENDDOCUMENT
    return;
}

sub domain_postsubtimeout {
    my ($cdom,$type,$settings) = @_;
    return unless (ref($settings) eq 'HASH'); 
    my $lctype = lc($type);
    unless ($type eq 'Community') {
        $lctype = 'unofficial';
        if ($settings->{'internal.coursecode'}) {
            $lctype = 'official';
        } elsif ($settings->{'internal.textbook'}) {
            $lctype = 'textbook';
        }
    }
    my %domconfig =
        &Apache::lonnet::get_dom('configuration',['coursedefaults'],$cdom);
    my $postsubtimeout = 60;
    if (ref($domconfig{'coursedefaults'}) eq 'HASH') {
        if (ref($domconfig{'coursedefaults'}{'postsubmit'}) eq 'HASH') {
            if (ref($domconfig{'coursedefaults'}{'postsubmit'}{'timeout'}) eq 'HASH') {
                if ($domconfig{'coursedefaults'}{'postsubmit'}{'timeout'}{$lctype} ne '') {
                    $postsubtimeout = $domconfig{'coursedefaults'}{'postsubmit'}{'timeout'}{$lctype};
                }
            }
        }
    }
    return $postsubtimeout;
}

sub print_catsettings {
    my ($r,$cdom,$cnum,$cdesc,$type,$readonly) = @_;
    &print_header($r,$type);
    my %lt = &Apache::lonlocal::texthash(
                                         'back'    => 'Pick another action',
                                         'catset'  => 'Catalog Settings for Course',
                                         'visi'    => 'Visibility in Course/Community Catalog',
                                         'exclude' => 'Exclude from course catalog:',
                                         'categ'   => 'Categorize Course',
                                         'assi'    => 'Assign one or more categories and/or subcategories to this course.'
                                        );
    if ($type eq 'Community') {
        $lt{'catset'} = &mt('Catalog Settings for Community');
        $lt{'exclude'} = &mt('Exclude from course catalog');
        $lt{'categ'} = &mt('Categorize Community');
        $lt{'assi'} = &mt('Assign one or more subcategories to this community.');
    }
    $r->print('<form action="/adm/modifycourse" method="post" name="catsettings">'.
              '<h3>'.$lt{'catset'}.' <span class="LC_nobreak">'.$cdesc.'</span></h3>');
    my %domconf = &Apache::lonnet::get_dom('configuration',['coursecategories'],$cdom);
    my @cat_params = &catalog_settable($domconf{'coursecategories'},$type);
    if (@cat_params > 0) {
        my $disabled;
        if ($readonly) {
            $disabled = ' disabled="disabled"';
        }
        my %currsettings = 
            &Apache::lonnet::get('environment',['hidefromcat','categories'],$cdom,$cnum);
        if (grep(/^togglecats$/,@cat_params)) {
            my $excludeon = '';
            my $excludeoff = ' checked="checked" ';
            if ($currsettings{'hidefromcat'} eq 'yes') {
                $excludeon = $excludeoff;
                $excludeoff = ''; 
            }
            $r->print('<br /><h4>'.$lt{'visi'}.'</h4>'.
                      $lt{'exclude'}.
                      '&nbsp;<label><input name="hidefromcat" type="radio" value="yes" '.$excludeon.$disabled.' />'.&mt('Yes').'</label>&nbsp;&nbsp;&nbsp;<label><input name="hidefromcat" type="radio" value="" '.$excludeoff.$disabled.' />'.&mt('No').'</label><br /><p>');
            if ($type eq 'Community') {
                $r->print(&mt("If a community has been categorized using at least one of the categories defined for communities in the domain, it will be listed in the domain's publicly accessible Course/Community Catalog, unless excluded."));
            } else {
                $r->print(&mt("Unless excluded, a course will be listed in the domain's publicly accessible Course/Community Catalog, if at least one of the following applies").':<ul>'.
                          '<li>'.&mt('Auto-cataloging is enabled and the course is assigned an institutional code.').'</li>'.
                          '<li>'.&mt('The course has been categorized using at least one of the course categories defined for the domain.').'</li></ul>');
            }
            $r->print('</ul></p>');
        }
        if (grep(/^categorize$/,@cat_params)) {
            $r->print('<br /><h4>'.$lt{'categ'}.'</h4>');
            if (ref($domconf{'coursecategories'}) eq 'HASH') {
                my $cathash = $domconf{'coursecategories'}{'cats'};
                if (ref($cathash) eq 'HASH') {
                    $r->print($lt{'assi'}.'<br /><br />'.
                              &Apache::loncommon::assign_categories_table($cathash,
                                                     $currsettings{'categories'},$type,$disabled));
                } else {
                    $r->print(&mt('No categories defined for this domain'));
                }
            } else {
                $r->print(&mt('No categories defined for this domain'));
            }
            unless ($type eq 'Community') { 
                $r->print('<p>'.&mt('If auto-cataloging based on institutional code is enabled in the domain, a course will continue to be listed in the catalog of official courses, in addition to receiving a listing under any manually assigned categor(ies).').'</p>');
            }
        }
        unless ($readonly) {
            $r->print('<p><input type="button" name="chgcatsettings" value="'.
                      &mt('Save').'" onclick="javascript:changePage(document.catsettings,'."'processcat'".');" /></p>');
        }
    } else {
        $r->print('<span class="LC_warning">');
        if ($type eq 'Community') {
            $r->print(&mt('Catalog settings in this domain are set in community context via "Community Configuration".'));
        } else {
            $r->print(&mt('Catalog settings in this domain are set in course context via "Course Configuration".'));
        }
        $r->print('</span><br /><br />'."\n".
                  '<a href="javascript:changePage(document.catsettings,'."'menu'".');">'.
                  $lt{'back'}.'</a>');
    }
    $r->print(&hidden_form_elements().'</form>'."\n");
    return;
}

sub print_course_modification_page {
    my ($r,$cdom,$cnum,$cdesc,$crstype,$readonly) = @_;
    my %lt=&Apache::lonlocal::texthash(
            'actv' => "Active",
            'inac' => "Inactive",
            'ownr' => "Owner",
            'name' => "Name",
            'unme' => "Username:Domain",
            'stus' => "Status",
            'nocc' => 'There is currently no owner set for this course.',
            'gobt' => "Save",
            'sett' => 'Setting',
            'domd' => 'Domain default',
            'whom' => 'Who configures',  
    );
    my ($ownertable,$ccrole,$javascript_validations,$authenitems,$ccname,$disabled);
    my %enrollvar = &get_enrollment_settings($cdom,$cnum);
    my %settings = &Apache::lonnet::get('environment',['internal.coursecode','internal.textbook',
                                                       'internal.selfenrollmgrdc','internal.selfenrollmgrcc',
                                                       'internal.mysqltables'],$cdom,$cnum);
    my $type = &Apache::lonuserutils::get_extended_type($cdom,$cnum,$crstype,\%settings);
    my @specific_managebydc = split(/,/,$settings{'internal.selfenrollmgrdc'});
    my @specific_managebycc = split(/,/,$settings{'internal.selfenrollmgrcc'});
    my %domdefaults = &Apache::lonnet::get_domain_defaults($cdom);
    my %passwdconf = &Apache::lonnet::get_passwdconf($cdom);
    my @default_managebydc = split(/,/,$domdefaults{$type.'selfenrolladmdc'});
    if ($crstype eq 'Community') {
        $ccrole = 'co';
        $lt{'nocc'} = &mt('There is currently no owner set for this community.');
    } else {
        $ccrole ='cc';
        ($javascript_validations,$authenitems) = &gather_authenitems($cdom,\%enrollvar,$readonly);
    }
    $ccname = &Apache::lonnet::plaintext($ccrole,$crstype);
    if ($readonly) {
       $disabled = ' disabled="disabled"';
    }
    my %roleshash = &Apache::lonnet::get_my_roles($cnum,$cdom,'','',[$ccrole]);
    my (@local_ccs,%cc_status,%pname);
    foreach my $item (keys(%roleshash)) {
        my ($uname,$udom) = split(/:/,$item);
        if (!grep(/^\Q$uname\E:\Q$udom\E$/,@local_ccs)) {
            push(@local_ccs,$uname.':'.$udom);
            $pname{$uname.':'.$udom} = &Apache::loncommon::plainname($uname,$udom);
            $cc_status{$uname.':'.$udom} = $lt{'actv'};
        }
    }
    if (($enrollvar{'courseowner'} ne '') && 
        (!grep(/^$enrollvar{'courseowner'}$/,@local_ccs))) {
        push(@local_ccs,$enrollvar{'courseowner'});
        my ($owneruname,$ownerdom) = split(/:/,$enrollvar{'courseowner'});
        $pname{$enrollvar{'courseowner'}} = 
                         &Apache::loncommon::plainname($owneruname,$ownerdom);
        my $active_cc = &Apache::loncommon::check_user_status($ownerdom,$owneruname,
                                                              $cdom,$cnum,$ccrole);
        if ($active_cc eq 'active') {
            $cc_status{$enrollvar{'courseowner'}} = $lt{'actv'};
        } else {
            $cc_status{$enrollvar{'courseowner'}} = $lt{'inac'};
        }
    }
    @local_ccs = sort(@local_ccs);
    if (@local_ccs == 0) {
        $ownertable = $lt{'nocc'};
    } else {
        my $numlocalcc = scalar(@local_ccs);
        $ownertable = '<input type="hidden" name="numlocalcc" value="'.$numlocalcc.'" />'.
                      &Apache::loncommon::start_data_table()."\n".
                      &Apache::loncommon::start_data_table_header_row()."\n".
                      '<th>'.$lt{'ownr'}.'</th>'.
                      '<th>'.$lt{'name'}.'</th>'.
                      '<th>'.$lt{'unme'}.'</th>'.
                      '<th>'.$lt{'stus'}.'</th>'.
                      &Apache::loncommon::end_data_table_header_row()."\n";
        foreach my $cc (@local_ccs) {
            $ownertable .= &Apache::loncommon::start_data_table_row()."\n";
            if ($cc eq $enrollvar{'courseowner'}) {
                  $ownertable .= '<td><input type="radio" name="courseowner" value="'.$cc.'" checked="checked"'.$disabled.' /></td>'."\n";
            } else {
                $ownertable .= '<td><input type="radio" name="courseowner" value="'.$cc.'"'.$disabled.' /></td>'."\n";
            }
            $ownertable .= 
                 '<td>'.$pname{$cc}.'</td>'."\n".
                 '<td>'.$cc.'</td>'."\n".
                 '<td>'.$cc_status{$cc}.' '.$ccname.'</td>'."\n".
                 &Apache::loncommon::end_data_table_row()."\n";
        }
        $ownertable .= &Apache::loncommon::end_data_table();
    }
    &print_header($r,$crstype,$javascript_validations);
    my $dctitle = &Apache::lonnet::plaintext('dc');
    my $mainheader = &modifiable_only_title($crstype);
    my $hidden_elements = &hidden_form_elements();
    $r->print('<form action="/adm/modifycourse" method="post" name="'.$env{'form.phase'}.'">'."\n".
              '<h3>'.$mainheader.' <span class="LC_nobreak">'.$cdesc.'</span></h3><p>'.
              &Apache::lonhtmlcommon::start_pick_box());
    if ($crstype eq 'Community') {
        $r->print(&Apache::lonhtmlcommon::row_title(
                  &Apache::loncommon::help_open_topic('Modify_Community_Owner').
                  '&nbsp;'.&mt('Community Owner'))."\n".
                  $ownertable."\n".&Apache::lonhtmlcommon::row_closure());
    } else {
        $r->print(&Apache::lonhtmlcommon::row_title(
                      &Apache::loncommon::help_open_topic('Modify_Course_Instcode').
                      '&nbsp;'.&mt('Course Code'))."\n".
                  '<input type="text" size="15" name="coursecode" value="'.$enrollvar{'coursecode'}.'"'.$disabled.' />'.
                  &Apache::lonhtmlcommon::row_closure());
        if (&showcredits($cdom)) {
            $r->print(&Apache::lonhtmlcommon::row_title(
                          &Apache::loncommon::help_open_topic('Modify_Course_Credithours').
                          '&nbsp;'.&mt('Credits (students)'))."\n".
                      '<input type="text" size="3" name="defaultcredits" value="'.$enrollvar{'defaultcredits'}.'"'.$disabled.' />'.
                      &Apache::lonhtmlcommon::row_closure());
        }
        $r->print(&Apache::lonhtmlcommon::row_title(
                      &Apache::loncommon::help_open_topic('Modify_Course_Defaultauth').
                      '&nbsp;'.&mt('Default Authentication method'))."\n".
                  $authenitems."\n".
                  &Apache::lonhtmlcommon::row_closure().
                  &Apache::lonhtmlcommon::row_title(
                      &Apache::loncommon::help_open_topic('Modify_Course_Owner').
                      '&nbsp;'.&mt('Course Owner'))."\n".
                  $ownertable."\n".&Apache::lonhtmlcommon::row_closure());
        if (($passwdconf{'crsownerchg'}) && ($type ne 'Placement')) {
            my $checked;
            if ($enrollvar{'nopasswdchg'}) {
                $checked = ' checked="checked"';
            }
            $r->print(&Apache::lonhtmlcommon::row_title(
                         &Apache::loncommon::help_open_topic('Modify_Course_Chgpasswd').
                         '&nbsp;'.&mt('Changing passwords (internal)'))."\n".
                         '<label><input type="checkbox" value="1" name="nopasswdchg"'.$checked.$disabled.' />'.
                         &mt('Disable changing password for users with student role by course owner').'<label>'."\n".
                         &Apache::lonhtmlcommon::row_closure());
        }
    }
    my ($cctitle,$rolename,$currmanages,$ccchecked,$dcchecked,$defaultchecked);
    my ($selfenrollrows,$selfenrolltitles) = &Apache::lonuserutils::get_selfenroll_titles();
    if ($type eq 'Community') {
        $cctitle = &mt('Community personnel');
    } else {
        $cctitle = &mt('Course personnel');
    }

    $r->print(&Apache::lonhtmlcommon::row_title(
              &Apache::loncommon::help_open_topic('Modify_Course_Selfenrolladmin').
                  '&nbsp;'.&mt('Self-enrollment configuration')).
              &Apache::loncommon::start_data_table()."\n".
              &Apache::loncommon::start_data_table_header_row()."\n".
              '<th>'.$lt{'sett'}.'</th>'.
              '<th>'.$lt{'domd'}.'</th>'.
              '<th>'.$lt{'whom'}.'</th>'.
              &Apache::loncommon::end_data_table_header_row()."\n");
    my %optionname;
    $optionname{''} = &mt('Use domain default'); 
    $optionname{'0'} = $dctitle;
    $optionname{'1'} = $cctitle;
    foreach my $item (@{$selfenrollrows}) {
        my %checked;
        my $default = $cctitle;
        if (grep(/^\Q$item\E$/,@default_managebydc)) {
            $default = $dctitle;
        }
        if (grep(/^\Q$item\E$/,@specific_managebydc)) {
            $checked{'0'} = ' checked="checked"';
        } elsif (grep(/^\Q$item\E$/,@specific_managebycc)) {
            $checked{'1'} = ' checked="checked"';
        } else {
            $checked{''} = ' checked="checked"';
        } 
        $r->print(&Apache::loncommon::start_data_table_row()."\n".
                 '<td>'.$selfenrolltitles->{$item}.'</td>'."\n".
                 '<td>'.&mt('[_1] configures',$default).'</td>'."\n".
                 '<td>');
        foreach my $option ('','0','1') {  
            $r->print('<span class="LC_nobreak"><label>'.
                      '<input type="radio" name="selfenrollmgr_'.$item.'" '.
                      'value="'.$option.'"'.$checked{$option}.$disabled.' />'.
                      $optionname{$option}.'</label></span><br />');
        }
        $r->print('</td>'."\n".
                  &Apache::loncommon::end_data_table_row()."\n");
    }
    $r->print(&Apache::loncommon::end_data_table()."\n".
              '<br />'.&Apache::lonhtmlcommon::row_closure().
              &Apache::lonhtmlcommon::row_title(
              &Apache::loncommon::help_open_topic('Modify_Course_Table_Lifetime').
              '&nbsp;'.&mt('"Temporary" Tables Lifetime (s)'))."\n".
              '<input type="text" size="10" name="mysqltables" value="'.$settings{'internal.mysqltables'}.'"'.$disabled.' />'.
              &Apache::lonhtmlcommon::row_closure(1).
              &Apache::lonhtmlcommon::end_pick_box().'</p><p>'.$hidden_elements);
    unless ($readonly) {
        $r->print('<input type="button" onclick="javascript:changePage(this.form,'."'processparms'".');');
        if ($crstype eq 'Community') {
            $r->print('this.form.submit();"');
        } else {
            $r->print('javascript:verify_message(this.form);"');
        }
        $r->print(' value="'.$lt{'gobt'}.'" />');
    }
    $r->print('</p></form>');
    return;
}

sub print_selfenrollconfig {
    my ($r,$type,$cdesc,$coursehash,$readonly) = @_;
    return unless(ref($coursehash) eq 'HASH');
    my $cnum = $coursehash->{'num'};
    my $cdom = $coursehash->{'domain'};
    my %currsettings = &get_selfenroll_settings($coursehash);
    &print_header($r,$type);
    $r->print('<h3>'.&mt('Self-enrollment with a student role in: [_1]',
              '<span class="LC_nobreak">'.$cdesc.'</span>').'</h3>'."\n");
    &Apache::loncreateuser::print_selfenroll_menu($r,'domain',$env{'form.pickedcourse'},
                                                  $cdom,$cnum,\%currsettings,
                                                  &hidden_form_elements(),$readonly);
    return;
}

sub modify_selfenrollconfig {
    my ($r,$type,$cdesc,$coursehash) = @_;
    return unless(ref($coursehash) eq 'HASH');
    my $cnum = $coursehash->{'num'};
    my $cdom = $coursehash->{'domain'};
    my %currsettings = &get_selfenroll_settings($coursehash);
    &print_header($r,$type);
    $r->print('<h3>'.&mt('Self-enrollment with a student role in: [_1]',
             '<span class="LC_nobreak">'.$cdesc.'</span>').'</h3>'."\n");
    $r->print('<form action="/adm/modifycourse" method="post" name="selfenroll">'."\n".
              &hidden_form_elements().'<br />');
    &Apache::loncreateuser::update_selfenroll_config($r,$env{'form.pickedcourse'},
                                                     $cdom,$cnum,'domain',$type,\%currsettings);
    $r->print('</form>');
    return;
}

sub get_selfenroll_settings {
    my ($coursehash) = @_;
    my %currsettings;
    if (ref($coursehash) eq 'HASH') {
        %currsettings = (
            selfenroll_types              => $coursehash->{'internal.selfenroll_types'},
            selfenroll_registered         => $coursehash->{'internal.selfenroll_registered'},
            selfenroll_section            => $coursehash->{'internal.selfenroll_section'},
            selfenroll_notifylist         => $coursehash->{'internal.selfenroll_notifylist'},
            selfenroll_approval           => $coursehash->{'internal.selfenroll_approval'},
            selfenroll_limit              => $coursehash->{'internal.selfenroll_limit'},
            selfenroll_cap                => $coursehash->{'internal.selfenroll_cap'},
            selfenroll_start_date         => $coursehash->{'internal.selfenroll_start_date'},
            selfenroll_end_date           => $coursehash->{'internal.selfenroll_end_date'},
            selfenroll_start_access       => $coursehash->{'internal.selfenroll_start_access'},
            selfenroll_end_access         => $coursehash->{'internal.selfenroll_end_access'},
            default_enrollment_start_date => $coursehash->{'default_enrollment_start_date'},
            default_enrollment_end_date   => $coursehash->{'default_enrollment_end_date'},
            uniquecode                    => $coursehash->{'internal.uniquecode'},
        );
    }
    return %currsettings;
}

sub modifiable_only_title {
    my ($type) = @_;
    my $dctitle = &Apache::lonnet::plaintext('dc');
    if ($type eq 'Community') {
        return &mt('Community settings modifiable only by [_1] for:',$dctitle);
    } else {
        return &mt('Course settings modifiable only by [_1] for:',$dctitle);
    }
}

sub gather_authenitems {
    my ($cdom,$enrollvar,$readonly) = @_;
    my ($krbdef,$krbdefdom)=&Apache::loncommon::get_kerberos_defaults($cdom);
    my $curr_authtype = '';
    my $curr_authfield = '';
    if (ref($enrollvar) eq 'HASH') {
        if ($enrollvar->{'authtype'} =~ /^krb/) {
            $curr_authtype = 'krb';
        } elsif ($enrollvar->{'authtype'} eq 'internal' ) {
            $curr_authtype = 'int';
        } elsif ($enrollvar->{'authtype'} eq 'localauth' ) {
            $curr_authtype = 'loc';
        }
    }
    unless ($curr_authtype eq '') {
        $curr_authfield = $curr_authtype.'arg';
    }
    my $javascript_validations = 
        &Apache::lonuserutils::javascript_validations('modifycourse',$krbdefdom,
                                                      $curr_authtype,$curr_authfield);
    my %param = ( formname => 'document.'.$env{'form.phase'},
           kerb_def_dom => $krbdefdom,
           kerb_def_auth => $krbdef,
           mode => 'modifycourse',
           curr_authtype => $curr_authtype,
           curr_autharg => $enrollvar->{'autharg'},
           readonly => $readonly,
        );
    my (%authform,$authenitems);
    $authform{'krb'} = &Apache::loncommon::authform_kerberos(%param);
    $authform{'int'} = &Apache::loncommon::authform_internal(%param);
    $authform{'loc'} = &Apache::loncommon::authform_local(%param);
    foreach my $item ('krb','int','loc') {
        if ($authform{$item} ne '') {
            $authenitems .= $authform{$item}.'<br />';
        }
    }
    return($javascript_validations,$authenitems);
}

sub modify_course {
    my ($r,$cdom,$cnum,$cdesc,$domdesc,$type) = @_;
    my %longtype = &course_settings_descrip($type);
    my @items = ('internal.courseowner','description','internal.co-owners',
                 'internal.pendingco-owners','internal.selfenrollmgrdc',
                 'internal.selfenrollmgrcc','internal.mysqltables');
    my ($selfenrollrows,$selfenrolltitles) = &Apache::lonuserutils::get_selfenroll_titles();
    unless ($type eq 'Community') {
        push(@items,('internal.coursecode','internal.authtype','internal.autharg',
                     'internal.sectionnums','internal.crosslistings'));
        if (&showcredits($cdom)) {  
            push(@items,'internal.defaultcredits');
        }
        my %passwdconf = &Apache::lonnet::get_passwdconf($cdom);
        if ($passwdconf{'crsownerchg'}) {
            push(@items,'internal.nopasswdchg');
        }
    }
    my %settings = &Apache::lonnet::get('environment',\@items,$cdom,$cnum);
    my $description = $settings{'description'};
    my ($ccrole,$response,$chgresponse,$nochgresponse,$reply,%currattr,%newattr,
        %cenv,%changed,@changes,@nochanges,@sections,@xlists,@warnings);
    my @modifiable_params = &get_dc_settable($type,$cdom);
    foreach my $param (@modifiable_params) {
        $currattr{$param} = $settings{'internal.'.$param};
    }
    if ($type eq 'Community') {
        %changed = ( owner  => 0 );
        $ccrole = 'co';
    } else {
        %changed = ( code  => 0,
                     owner => 0,
                     passwd => 0,
                   );
        $ccrole = 'cc';
        unless ($settings{'internal.sectionnums'} eq '') {
            if ($settings{'internal.sectionnums'} =~ m/,/) {
                @sections = split/,/,$settings{'internal.sectionnums'};
            } else {
                $sections[0] = $settings{'internal.sectionnums'};
            }
        }
        unless ($settings{'internal.crosslistings'} eq '') {
            if ($settings{'internal.crosslistings'} =~ m/,/) {
                @xlists = split/,/,$settings{'internal.crosslistings'};
            } else {
                $xlists[0] = $settings{'internal.crosslistings'};
            }
        }
        if ($env{'form.login'} eq 'krb') {
            $newattr{'authtype'} = $env{'form.login'};
            $newattr{'authtype'} .= $env{'form.krbver'};
            $newattr{'autharg'} = $env{'form.krbarg'};
        } elsif ($env{'form.login'} eq 'int') {
            $newattr{'authtype'} ='internal';
            if ((defined($env{'form.intarg'})) && ($env{'form.intarg'})) {
                $newattr{'autharg'} = $env{'form.intarg'};
            }
        } elsif ($env{'form.login'} eq 'loc') {
            $newattr{'authtype'} = 'localauth';
            if ((defined($env{'form.locarg'})) && ($env{'form.locarg'})) {
                $newattr{'autharg'} = $env{'form.locarg'};
            }
        }
        if ( $newattr{'authtype'}=~ /^krb/) {
            if ($newattr{'autharg'}  eq '') {
                push(@warnings,
                           &mt('As you did not include the default Kerberos domain'
                          .' to be used for authentication in this class, the'
                          .' institutional data used by the automated'
                          .' enrollment process must include the Kerberos'
                          .' domain for each new student.'));
            }
        }

        if ( exists($env{'form.coursecode'}) ) {
            $newattr{'coursecode'}=$env{'form.coursecode'};
            unless ( $newattr{'coursecode'} eq $currattr{'coursecode'} ) {
                $changed{'code'} = 1;
            }
        }
        if ( exists($env{'form.mysqltables'}) ) {
            $newattr{'mysqltables'} = $env{'form.mysqltables'};
            $newattr{'mysqltables'} =~ s/\D+//g;
        }
        if ($type ne 'Placement') {
            if (&showcredits($cdom) && exists($env{'form.defaultcredits'})) {
                $newattr{'defaultcredits'}=$env{'form.defaultcredits'};
                $newattr{'defaultcredits'} =~ s/[^\d\.]//g;
            }
            if (grep(/^nopasswdchg$/,@modifiable_params)) {
                if ($env{'form.nopasswdchg'}) {
                    $newattr{'nopasswdchg'} = 1;
                    unless ($currattr{'nopasswdchg'}) {
                        $changed{'passwd'} = 1;
                    }
                } elsif ($currattr{'nopasswdchg'}) {
                    $changed{'passwd'} = 1;
                }
            }
        }
    }

    my @newmgrdc = ();
    my @newmgrcc = ();
    my @currmgrdc = split(/,/,$currattr{'selfenrollmgrdc'});
    my @currmgrcc = split(/,/,$currattr{'selfenrollmgrcc'});

    foreach my $item (@{$selfenrollrows}) {
        if ($env{'form.selfenrollmgr_'.$item} eq '0') {
            push(@newmgrdc,$item);
        } elsif ($env{'form.selfenrollmgr_'.$item} eq '1') {
            push(@newmgrcc,$item);
        }
    }

    $newattr{'selfenrollmgrdc'}=join(',',@newmgrdc);
    $newattr{'selfenrollmgrcc'}=join(',',@newmgrcc);

    my $cctitle;
    if ($type eq 'Community') {
        $cctitle = &mt('Community personnel');
    } else {
        $cctitle = &mt('Course personnel');
    }
    my $dctitle = &Apache::lonnet::plaintext('dc');

    if ( exists($env{'form.courseowner'}) ) {
        $newattr{'courseowner'}=$env{'form.courseowner'};
        unless ( $newattr{'courseowner'} eq $currattr{'courseowner'} ) {
            $changed{'owner'} = 1;
        } 
    }

    if ($changed{'owner'} || $changed{'code'} || $changed{'passwd'}) {
        my %crsinfo = &Apache::lonnet::courseiddump($cdom,'.',1,'.','.',$cnum,
                                                    undef,undef,'.');
        if (ref($crsinfo{$env{'form.pickedcourse'}}) eq 'HASH') {
            if ($changed{'code'}) {
                $crsinfo{$env{'form.pickedcourse'}}{'inst_code'} = $env{'form.coursecode'};
            }
            if ($changed{'owner'}) {
                $crsinfo{$env{'form.pickedcourse'}}{'owner'} = $env{'form.courseowner'};
            }
            if ($changed{'passwd'}) {
                if ($env{'form.nopasswdchg'}) {
                    $crsinfo{$env{'form.pickedcourse'}}{'nopasswdchg'} = 1;
                } else {
                    delete($crsinfo{'nopasswdchg'});
                }
            }
            my $chome = &Apache::lonnet::homeserver($cnum,$cdom);
            my $putres = &Apache::lonnet::courseidput($cdom,\%crsinfo,$chome,'notime');
            if (($putres eq 'ok') && (($changed{'owner'} || $changed{'code'}))) {
                &update_coowners($cdom,$cnum,$chome,\%settings,\%newattr);
                if ($changed{'code'}) {
                    &Apache::lonnet::devalidate_cache_new('instcats',$cdom);
                    # Update cache of self-cataloging courses on institution's server(s).
                    if (&Apache::lonnet::shared_institution($cdom)) {
                        unless ($registered_cleanup) {
                            my $handlers = $r->get_handlers('PerlCleanupHandler');
                            $r->set_handlers('PerlCleanupHandler' => [\&devalidate_remote_instcats,@{$handlers}]);
                            $registered_cleanup=1;
                            $modified_dom = $cdom;
                        }
                    }
                }
            }
        }
    }
    foreach my $param (@modifiable_params) {
        if ($currattr{$param} eq $newattr{$param}) {
            push(@nochanges,$param);
        } else {
            $cenv{'internal.'.$param} = $newattr{$param};
            push(@changes,$param);
        }
    }
    if (@changes > 0) {
        $chgresponse = &mt('The following settings have been changed:').'<br/><ul>';
    }
    if (@nochanges > 0) {
        $nochgresponse = &mt('The following settings remain unchanged:').'<br/><ul>';
    }
    if (@changes > 0) {
        my $putreply = &Apache::lonnet::put('environment',\%cenv,$cdom,$cnum);
        if ($putreply !~ /^ok$/) {
            $response = '<p class="LC_error">'.
                        &mt('There was a problem processing your requested changes.').'<br />';
            if ($type eq 'Community') {
                $response .= &mt('Settings for this community have been left unchanged.');
            } else {
                $response .= &mt('Settings for this course have been left unchanged.');
            }
            $response .= '<br/>'.&mt('Error: ').$putreply.'</p>';
        } else {
            if ($env{'course.'.$cdom.'_'.$cnum.'.description'} ne '') {
                my %newenv;
                map { $newenv{'course.'.$cdom.'_'.$cnum.'.internal.'.$_} = $newattr{$_}; } @changes;   
                &Apache::lonnet::appenv(\%newenv);
            }
            foreach my $attr (@modifiable_params) {
                if (grep/^\Q$attr\E$/,@changes) {
                    my $shown = $newattr{$attr};
                    if ($attr eq 'selfenrollmgrdc') {
                        $shown = &selfenroll_config_status(\@newmgrdc,$selfenrolltitles);
                    } elsif ($attr eq 'selfenrollmgrcc') {
                        $shown = &selfenroll_config_status(\@newmgrcc,$selfenrolltitles);
                    } elsif (($attr eq 'defaultcredits') && ($shown eq '')) {
                        $shown = &mt('None');
                    } elsif (($attr eq 'mysqltables') && ($shown eq '')) {
                        $shown = &mt('domain default');
                    } elsif ($attr eq 'nopasswdchg') {
                        if ($shown) {
                            $shown = &mt('Yes');
                        } else {
                            $shown = &mt('No');
                        }
                    }
                    $chgresponse .= '<li>'.&mt('[_1] now set to: [_2]',$longtype{$attr},$shown).'</li>';
                } else {
                    my $shown = $currattr{$attr};
                    if ($attr eq 'selfenrollmgrdc') {
                        $shown = &selfenroll_config_status(\@currmgrdc,$selfenrolltitles);
                    } elsif ($attr eq 'selfenrollmgrcc') {
                        $shown = &selfenroll_config_status(\@currmgrcc,$selfenrolltitles);
                    } elsif (($attr eq 'defaultcredits') && ($shown eq '')) {
                        $shown = &mt('None');
                    } elsif (($attr eq 'mysqltables') && ($shown eq '')) {
                        $shown = &mt('domain default');
                    } elsif ($attr eq 'nopasswdchg') {
                        if ($shown) {
                            $shown = &mt('Yes');
                        } else {
                            $shown = &mt('No');
                        }
                    }
                    $nochgresponse .= '<li>'.&mt('[_1] still set to: [_2]',$longtype{$attr},$shown).'</li>';
                }
            }
            if (($type ne 'Community') && ($changed{'code'} || $changed{'owner'})) {
                if ( $newattr{'courseowner'} eq '') {
	            push(@warnings,&mt('There is no owner associated with this LON-CAPA course.').
                                   '<br />'.&mt('If automated enrollment at your institution requires validation of course owners, automated enrollment will fail.'));
                } else {
                    my %crsenv = &Apache::lonnet::get('environment',['internal.co-owners'],$cdom,$cnum);
                    my $coowners = $crsenv{'internal.co-owners'};
	            if (@sections > 0) {
                        if ($changed{'code'}) {
	                    foreach my $sec (@sections) {
		                if ($sec =~ m/^(.+):/) {
                                    my $instsec = $1;
		                    my $inst_course_id = $newattr{'coursecode'}.$1;
                                    my $course_check = &Apache::lonnet::auto_validate_courseID($cnum,$cdom,$inst_course_id);
			            if ($course_check eq 'ok') {
                                        my $outcome = &Apache::lonnet::auto_new_course($cnum,$cdom,$inst_course_id,$newattr{'courseowner'},$coowners);
			                unless ($outcome eq 'ok') {
                               
				            push(@warnings,&mt('If automatic enrollment is enabled for "[_1]", automated enrollment may fail for "[_2]" - section: [_3] for the following reason: "[_4]".',$description,$newattr{'coursecode'},$instsec,$outcome).'<br/>');
			                }
			            } else {
                                        push(@warnings,&mt('If automatic enrollment is enabled for "[_1]", automated enrollment may fail for "[_2]" - section: [_3] for the following reason: "[_4]".',$description,$newattr{'coursecode'},$instsec,$course_check));
			            }
		                } else {
			            push(@warnings,&mt('If automatic enrollment is enabled for "[_1]", automated enrollment may fail for "[_2]" - section: [_3], because this is not a valid section entry.',$description,$newattr{'coursecode'},$sec));
		                }
		            }
	                } elsif ($changed{'owner'}) {
                            foreach my $sec (@sections) {
                                if ($sec =~ m/^(.+):/) {
                                    my $instsec = $1;
                                    my $inst_course_id = $newattr{'coursecode'}.$instsec;
                                    my $outcome = &Apache::lonnet::auto_new_course($cnum,$cdom,$inst_course_id,$newattr{'courseowner'},$coowners);
                                    unless ($outcome eq 'ok') {
                                        push(@warnings,&mt('If automatic enrollment is enabled for "[_1]", automated enrollment may fail for "[_2]" - section: [_3] for the following reason: "[_4]".',$description,$newattr{'coursecode'},$instsec,$outcome));
                                    }
                                } else {
                                    push(@warnings,&mt('If automatic enrollment is enabled for "[_1]", automated enrollment may fail for "[_2]" - section: [_3], because this is not a valid section entry.',$description,$newattr{'coursecode'},$sec));
                                }
                            }
                        }
	            } else {
	                push(@warnings,&mt('As no section numbers are currently listed for "[_1]", automated enrollment will not occur for any sections of institutional course code: "[_2]".',$description,$newattr{'coursecode'}));
	            }
	            if ( (@xlists > 0) && ($changed{'owner'}) ) {
	                foreach my $xlist (@xlists) {
		            if ($xlist =~ m/^(.+):/) {
                                my $instxlist = $1;
                                my $outcome = &Apache::lonnet::auto_new_course($cnum,$cdom,$instxlist,$newattr{'courseowner'},$coowners);
		                unless ($outcome eq 'ok') {
			            push(@warnings,&mt('If automatic enrollment is enabled for "[_1]", automated enrollment may fail for crosslisted class "[_2]" for the following reason: "[_3]".',$description,$instxlist,$outcome));
		                }
		            }
	                }
	            }
                }
            }
        }
    } else {
        foreach my $attr (@modifiable_params) {
            my $shown = $currattr{$attr};
            if ($attr eq 'selfenrollmgrdc') {
                $shown = &selfenroll_config_status(\@currmgrdc,$selfenrolltitles);
            } elsif ($attr eq 'selfenrollmgrcc') {
                $shown = &selfenroll_config_status(\@currmgrcc,$selfenrolltitles);
            } elsif (($attr eq 'defaultcredits') && ($shown eq '')) {
                $shown = &mt('None');
            } elsif (($attr eq 'mysqltables') && ($shown eq '')) {
                $shown = &mt('domain default');
            }
            $nochgresponse .= '<li>'.&mt('[_1] still set to: [_2]',$longtype{$attr},$shown).'</li>';
        }
    }

    if (@changes > 0) {
        $chgresponse .= "</ul><br/><br/>";
    }
    if (@nochanges > 0) {
        $nochgresponse .=  "</ul><br/><br/>";
    }
    my ($warning,$numwarnings);
    my $numwarnings = scalar(@warnings); 
    if ($numwarnings) {
        $warning = &mt('The following [quant,_1,warning was,warnings were] generated when applying your changes to automated enrollment:',$numwarnings).'<p><ul>';
        foreach my $warn (@warnings) {
            $warning .= '<li><span class="LC_warning">'.$warn.'</span></li>';
        }
        $warning .= '</ul></p>';
    }
    if ($response) {
        $reply = $response;
    } else {
        $reply = $chgresponse.$nochgresponse.$warning;
    }
    &print_header($r,$type);
    my $mainheader = &modifiable_only_title($type);
    $reply = '<h3>'.$mainheader.' <span class="LC_nobreak">'.$cdesc.'</span></h3>'."\n".
             '<p>'.$reply.'</p>'."\n".
             '<form action="/adm/modifycourse" method="post" name="processparms">'.
             &hidden_form_elements();
    my @actions =
        ('<a href="javascript:changePage(document.processparms,'."'menu'".')">'.
                 &mt('Pick another action').'</a>');
    if ($numwarnings) {
        my $newrole = $ccrole.'./'.$cdom.'/'.$cnum;
        my $escuri = &HTML::Entities::encode('/adm/roles?selectrole=1&'.$newrole.
                                             '=1&destinationurl=/adm/populate','&<>"');

        push(@actions, '<a href="'.$escuri.'">'.
                  &mt('Go to Automated Enrollment Manager for course').'</a>');
    }
    $reply .= &Apache::lonhtmlcommon::actionbox(\@actions).'</form>';
    $r->print($reply);
    return;
}

sub selfenroll_config_status {
    my ($items,$selfenrolltitles) = @_;
    my $shown;
    if ((ref($items) eq 'ARRAY') && (ref($selfenrolltitles) eq 'HASH')) {
        if (@{$items} > 0) {
            $shown = '<ul>';
            foreach my $item (@{$items}) {
                $shown .= '<li>'.$selfenrolltitles->{$item}.'</li>';
            }
            $shown .= '</ul>';
        } else {
            $shown = &mt('None');
        }
    }
    return $shown;
}

sub update_coowners {
    my ($cdom,$cnum,$chome,$settings,$newattr) = @_;
    return unless ((ref($settings) eq 'HASH') && (ref($newattr) eq 'HASH'));
    my %designhash = &Apache::loncommon::get_domainconf($cdom);
    my (%cchash,$autocoowners);
    if ($designhash{$cdom.'.autoassign.co-owners'}) {
        $autocoowners = 1;
        %cchash = &Apache::lonnet::get_my_roles($cnum,$cdom,undef,undef,['cc']);
    }
    if ($settings->{'internal.courseowner'} ne $newattr->{'courseowner'}) {
        my $oldowner_to_coowner;
        my @types = ('co-owners');
        if (($newattr->{'coursecode'}) && ($autocoowners)) {
            my $oldowner = $settings->{'internal.courseowner'};
            if ($cchash{$oldowner.':cc'}) {
                my ($result,$desc) = &Apache::lonnet::auto_validate_instcode($cnum,$cdom,$newattr->{'coursecode'},$oldowner);
                if ($result eq 'valid') {
                    if ($settings->{'internal.co-owner'}) {
                        my @current = split(',',$settings->{'internal.co-owners'});
                        unless (grep(/^\Q$oldowner\E$/,@current)) {
                            $oldowner_to_coowner = 1;
                        }
                    } else {
                        $oldowner_to_coowner = 1;
                    }
                }
            }
        } else {
            push(@types,'pendingco-owners');
        }
        foreach my $type (@types) {
            if ($settings->{'internal.'.$type}) {
                my @current = split(',',$settings->{'internal.'.$type});
                my $newowner = $newattr->{'courseowner'};
                my @newvalues = ();
                if (($newowner ne '') && (grep(/^\Q$newowner\E$/,@current))) {
                    foreach my $person (@current) {
                        unless ($person eq $newowner) {
                            push(@newvalues,$person);
                        }
                    }
                } else {
                    @newvalues = @current;
                }
                if ($oldowner_to_coowner) {
                    push(@newvalues,$settings->{'internal.courseowner'});
                    @newvalues = sort(@newvalues);
                }
                my $newownstr = join(',',@newvalues);
                if ($newownstr ne $settings->{'internal.'.$type}) {
                    if ($type eq 'co-owners') {
                        my $deleted = '';
                        unless (@newvalues) {
                            $deleted = 1;
                        }
                        &Apache::lonnet::store_coowners($cdom,$cnum,$chome,
                                                        $deleted,@newvalues);
                    } else {
                        my $pendingcoowners;
                        my $cid = $cdom.'_'.$cnum;
                        if (@newvalues) {
                            $pendingcoowners = join(',',@newvalues);
                            my %pendinghash = (
                                'internal.pendingco-owners' => $pendingcoowners,
                            );
                            my $putresult = &Apache::lonnet::put('environment',\%pendinghash,$cdom,$cnum);
                            if ($putresult eq 'ok') {
                                if ($env{'course.'.$cid.'.num'} eq $cnum) {
                                    &Apache::lonnet::appenv({'course.'.$cid.'.internal.pendingco-owners' => $pendingcoowners});
                                }
                            }
                        } else {
                            my $delresult = &Apache::lonnet::del('environment',['internal.pendingco-owners'],$cdom,$cnum);
                            if ($delresult eq 'ok') {
                                if ($env{'course.'.$cid.'.internal.pendingco-owners'}) {
                                    &Apache::lonnet::delenv('course.'.$cid.'.internal.pendingco-owners');
                                }
                            }
                        }
                    }
                } elsif ($oldowner_to_coowner) {
                    &Apache::lonnet::store_coowners($cdom,$cnum,$chome,'',
                                         $settings->{'internal.courseowner'});

                }
            } elsif ($oldowner_to_coowner) {
                &Apache::lonnet::store_coowners($cdom,$cnum,$chome,'',
                                     $settings->{'internal.courseowner'});
            }
        }
    }
    if ($settings->{'internal.coursecode'} ne $newattr->{'coursecode'}) {
        if ($newattr->{'coursecode'} ne '') {
            my %designhash = &Apache::loncommon::get_domainconf($cdom);
            if ($designhash{$cdom.'.autoassign.co-owners'}) {
                my @newcoowners = ();
                if ($settings->{'internal.co-owners'}) {
                    my @currcoown = split(',',$settings->{'internal.co-owners'});
                    my ($updatecoowners,$delcoowners);
                    foreach my $person (@currcoown) {
                        my ($result,$desc) = &Apache::lonnet::auto_validate_instcode($cnum,$cdom,$newattr->{'coursecode'},$person);
                        if ($result eq 'valid') {
                            push(@newcoowners,$person);
                        }
                    }
                    foreach my $item (sort(keys(%cchash))) {
                        my ($uname,$udom,$urole) = split(':',$item);
                        next if ($uname.':'.$udom eq $newattr->{'courseowner'});
                        unless (grep(/^\Q$uname\E:\Q$udom\E$/,@newcoowners)) {
                            my ($result,$desc) = &Apache::lonnet::auto_validate_instcode($cnum,$cdom,$newattr->{'coursecode'},$uname.':'.$udom);
                            if ($result eq 'valid') {
                                push(@newcoowners,$uname.':'.$udom);
                            }
                        }
                    }
                    if (@newcoowners) {
                        my $coowners = join(',',sort(@newcoowners));
                        unless ($coowners eq $settings->{'internal.co-owners'}) {
                            $updatecoowners = 1;
                        }
                    } else {
                        $delcoowners = 1;
                    }
                    if ($updatecoowners || $delcoowners) {
                        &Apache::lonnet::store_coowners($cdom,$cnum,$chome,
                                                        $delcoowners,@newcoowners);
                    }
                } else {
                    foreach my $item (sort(keys(%cchash))) {
                        my ($uname,$udom,$urole) = split(':',$item);
                        push(@newcoowners,$uname.':'.$udom);
                    }
                    if (@newcoowners) {
                        &Apache::lonnet::store_coowners($cdom,$cnum,$chome,'',
                                                        @newcoowners);
                    }
                }
            }
        }
    }
    return;
}

sub modify_quota {
    my ($r,$cdom,$cnum,$cdesc,$domdesc,$type) = @_;
    &print_header($r,$type);
    my $lctype = lc($type);
    my $headline = &mt("Disk space quotas for $lctype: [_1]",
                     '<span class="LC_nobreak">'.$cdesc.'</span>');
    $r->print('<form action="/adm/modifycourse" method="post" name="processquota">'."\n".
              '<h3>'.$headline.'</h3>');
    my %oldsettings = &Apache::lonnet::get('environment',['internal.coursequota','internal.uploadquota'],$cdom,$cnum);
    my %staticdefaults = (
                           coursequota   => 20,
                           uploadquota   => 500,
                         );
    my %default;
    $default{'coursequota'} = $staticdefaults{'coursequota'};
    my %domdefs = &Apache::lonnet::get_domain_defaults($cdom);
    $default{'uploadquota'} = $domdefs{'uploadquota'};
    if ($default{'uploadquota'} eq '') {
        $default{'uploadquota'} = $staticdefaults{'uploadquota'};
    }
    my (%cenv,%showresult);
    foreach my $item ('coursequota','uploadquota') {
        if ($env{'form.'.$item} ne '') {
            my $newquota = $env{'form.'.$item};
            if ($newquota =~ /^\s*(\d+\.?\d*|\.\d+)\s*$/) {
                $newquota = $1;
                if ($oldsettings{'internal.'.$item} == $newquota) {
                    if ($item eq 'coursequota') {
                        $r->print(&mt('The disk space allocated for group portfolio files remains unchanged as [_1] MB.',$newquota).'<br />');
                    } else {
                        $r->print(&mt('The disk space allocated for files uploaded via the Content Editor remains unchanged as [_1] MB.',$newquota).'<br />');
                    }
                } else {
                    $cenv{'internal.'.$item} = $newquota;
                    $showresult{$item} = 1;
                }
            } else {
                if ($item eq 'coursequota') { 
                    $r->print(&mt('The proposed group portfolio quota contained invalid characters, so the quota is unchanged.').'<br />');
                } else {
                    $r->print(&mt('The proposed quota for content uploaded via the Content Editor contained invalid characters, so the quota is unchanged.').'<br />');

                }
            }
        }
    }
    if (keys(%cenv)) {
        my $putreply = &Apache::lonnet::put('environment',\%cenv,$cdom,
                                            $cnum);
        foreach my $key (sort(keys(%showresult))) {
            if (($oldsettings{'internal.'.$key} eq '') && 
                ($env{'form.'.$key} == $default{$key})) {
                if ($key eq 'uploadquota') {
                    if ($type eq 'Community') {
                        $r->print(&mt('The disk space allocated for files uploaded to this community via the Content Editor is the default quota for this domain: [_1] MB.',
                                      $default{$key}).'<br />');
                    } else {
                        $r->print(&mt('The disk space allocated for files uploaded to this course via the Content Editor is the default quota for this domain: [_1] MB.',
                                      $default{$key}).'<br />');
                    }
                } else { 
                    if ($type eq 'Community') {
                        $r->print(&mt('The disk space allocated for group portfolio files in this community is the default quota for this domain: [_1] MB.',
                                      $default{$key}).'<br />');
                    } else {
                        $r->print(&mt('The disk space allocated for group portfolio files in this course is the default quota for this domain: [_1] MB.',
                                      $default{$key}).'<br />');
                    }
                }
                delete($showresult{$key});
            }
        }
        if ($putreply eq 'ok') {
            my %updatedsettings = &Apache::lonnet::get('environment',['internal.coursequota','internal.uploadquota'],$cdom,$cnum);
            if ($showresult{'coursequota'}) {
                $r->print(&mt('The disk space allocated for group portfolio files is now: [_1] MB.',
                              '<b>'.$updatedsettings{'internal.coursequota'}.'</b>').'<br />');
                my $usage = &Apache::longroup::sum_quotas($cdom.'_'.$cnum);
                if ($usage >= $updatedsettings{'internal.coursequota'}) {
                    my $newoverquota;
                    if ($usage < $oldsettings{'internal.coursequota'}) {
                        $newoverquota = 'now';
                    }
                    $r->print('<p>');
                    if ($type eq 'Community') {
                        $r->print(&mt("Disk usage $newoverquota exceeds the quota for this community.").' '.
                                  &mt('Upload of new portfolio files and assignment of a non-zero MB quota to new groups in the community will not be possible until some files have been deleted, and total usage is below community quota.'));
                    } else {
                        $r->print(&mt("Disk usage $newoverquota exceeds the quota for this course.").' '.
                                  &mt('Upload of new portfolio files and assignment of a non-zero MB quota to new groups in the course will not be possible until some files have been deleted, and total usage is below course quota.'));
                    }
                    $r->print('</p>');
                }
            }
            if ($showresult{'uploadquota'}) {
                $r->print(&mt('The disk space allocated for content uploaded directly via the Content Editor is now: [_1] MB.',
                              '<b>'.$updatedsettings{'internal.uploadquota'}.'</b>').'<br />');
            }
        } else {
            $r->print(&mt('An error occurred storing the quota(s) for group portfolio files and/or uploaded content: ').
                      $putreply);
        }
    }
    $r->print('<p>'.
              '<a href="javascript:changePage(document.processquota,'."'menu'".')">'.
              &mt('Pick another action').'</a>');
    $r->print(&hidden_form_elements().'</form>');
    return;
}

sub modify_anonsurvey_threshold {
    my ($r,$cdom,$cnum,$cdesc,$domdesc,$type) = @_;
    &print_header($r,$type);
    $r->print('<form action="/adm/modifycourse" method="post" name="processthreshold">'."\n".
              '<h3>'.&mt('Responder threshold required for display of anonymous survey submissions:').
              ' <span class="LC_nobreak">'.$cdesc.'</span></h3><br />');
    my %oldsettings = &Apache::lonnet::get('environment',['internal.anonsurvey_threshold'],$cdom,$cnum);
    my %domconfig =
        &Apache::lonnet::get_dom('configuration',['coursedefaults'],$cdom);
    my $defaultthreshold; 
    if (ref($domconfig{'coursedefaults'}) eq 'HASH') {
        $defaultthreshold = $domconfig{'coursedefaults'}{'anonsurvey_threshold'};
        if ($defaultthreshold eq '') {
            $defaultthreshold = 10;
        }
    } else {
        $defaultthreshold = 10;
    }
    if ($env{'form.threshold'} eq '') {
        $r->print(&mt('The proposed responder threshold for display of anonymous survey submissions was blank, so the threshold is unchanged.'));
    } else {
        my $newthreshold = $env{'form.threshold'};
        if ($newthreshold =~ /^\s*(\d+)\s*$/) {
            $newthreshold = $1;
            if ($oldsettings{'internal.anonsurvey_threshold'} eq $env{'form.threshold'}) {
                $r->print(&mt('Responder threshold for anonymous survey submissions display remains unchanged: [_1].',$env{'form.threshold'}));
            } else {
                my %cenv = (
                           'internal.anonsurvey_threshold' => $env{'form.threshold'},
                           );
                my $putreply = &Apache::lonnet::put('environment',\%cenv,$cdom,
                                                    $cnum);
                if ($putreply eq 'ok') {
                    if ($env{'course.'.$cdom.'_'.$cnum.'.description'} ne '') {
                        &Apache::lonnet::appenv(
                           {'course.'.$cdom.'_'.$cnum.'.internal.anonsurvey_threshold' => $env{'form.threshold'}});
                    }
                }
                if (($oldsettings{'internal.anonsurvey_threshold'} eq '') &&
                    ($env{'form.threshold'} == $defaultthreshold)) {
                    $r->print(&mt('The responder threshold for display of anonymous survey submissions is the default for this domain: [_1].',$defaultthreshold));
                } else {
                    if ($putreply eq 'ok') {
                        my %updatedsettings = &Apache::lonnet::get('environment',['internal.anonsurvey_threshold'],$cdom,$cnum);
                        $r->print(&mt('The responder threshold for display of anonymous survey submissions is now: [_1].','<b>'.$updatedsettings{'internal.anonsurvey_threshold'}.'</b>'));
                    } else {
                        $r->print(&mt('An error occurred storing the responder threshold for anonymous submissions display: ').
                                  $putreply);
                    }
                }
            }
        } else {
            $r->print(&mt('The proposed responder threshold for display of anonymous submissions contained invalid characters, so the threshold is unchanged.'));
        }
    }
    $r->print('<p>'.
              '<a href="javascript:changePage(document.processthreshold,'."'menu'".')">'.
              &mt('Pick another action').'</a></p>');
    $r->print(&hidden_form_elements().'</form>');
    return;
}

sub modify_postsubmit_config {
    my ($r,$cdom,$cnum,$cdesc,$domdesc,$type) = @_;
    &print_header($r,$type);
    my %lt = &Apache::lonlocal::texthash(
                subb => 'Submit button behavior after student makes a submission:',
                unch => 'Post submission behavior of the Submit button is unchanged.',
                erro => 'An error occurred when saving your proposed changes.',
                inva => 'An invalid response was recorded.',
                pick => 'Pick another action',
             );
    $r->print('<form action="/adm/modifycourse" method="post" name="processpostsubmit">'."\n".
              '<h3>'.$lt{'subb'}.' <span class="LC_nobreak">('.$cdesc.')</span></h3><br />');
    my %oldsettings = 
        &Apache::lonnet::get('environment',['internal.postsubmit','internal.postsubtimeout','internal.coursecode','internal.textbook'],$cdom,$cnum);
    my $postsubmit = $env{'form.postsubmit'};
    if ($postsubmit eq '1') {
        my $postsubtimeout = $env{'form.postsubtimeout'};
        $postsubtimeout =~ s/[^\d\.]+//g;
        if (($oldsettings{'internal.postsubmit'} eq $postsubmit) && ($oldsettings{'internal.postsubtimeout'} eq $postsubtimeout)) {
            $r->print($lt{'unch'}); 
        } else {
            my %cenv = (
                         'internal.postsubmit' => $postsubmit,
                       );
            if ($postsubtimeout eq '') {
                my $putreply = &Apache::lonnet::put('environment',\%cenv,$cdom,$cnum);
                if ($putreply eq 'ok') {
                    my $defaulttimeout = &domain_postsubtimeout($cdom,$type,\%oldsettings);
                    $r->print(&mt('The proposed duration for disabling the Submit button post-submission was blank, so the domain default of [quant,_1,second] will be used.',$defaulttimeout));
                    if (exists($oldsettings{'internal.postsubtimeout'})) {
                        &Apache::lonnet::del('environment',['internal.postsubtimeout'],$cdom,$cnum);
                    }
                } else {
                    $r->print($lt{'erro'});
                }
            } else { 
                $cenv{'internal.postsubtimeout'} = $postsubtimeout;
                my $putreply = &Apache::lonnet::put('environment',\%cenv,$cdom,$cnum);
                if ($putreply eq 'ok') {
                    if ($postsubtimeout eq '0') {
                        $r->print(&mt('Submit button will be disabled after student submission until page is reloaded.')); 
                    } else {
                        $r->print(&mt('Submit button will be disabled after student submission for [quant,_1,second].',$postsubtimeout));
                    }
                } else {
                    $r->print($lt{'erro'});
                }
            }
        }
    } elsif ($postsubmit eq '0') {
        if ($oldsettings{'internal.postsubmit'} eq $postsubmit) {
            $r->print($lt{'unch'});
        } else {
            if (exists($oldsettings{'internal.postsubtimeout'})) {
                &Apache::lonnet::del('environment',['internal.postsubtimeout'],$cdom,$cnum);  
            }
            my %cenv = (
                         'internal.postsubmit' => $postsubmit,
                       );
            my $putreply = &Apache::lonnet::put('environment',\%cenv,$cdom,$cnum);
            if ($putreply eq 'ok') {
                $r->print(&mt('Submit button will not be disabled after student submission'));
            } else {
                $r->print($lt{'erro'});
            }
        }
    } else {
        $r->print($lt{'inva'}.' '.$lt{'unch'});
    }
    $r->print('<p>'.
              '<a href="javascript:changePage(document.processpostsubmit,'."'menu'".')">'.
              &mt('Pick another action').'</a></p>');
    $r->print(&hidden_form_elements().'</form>');
    return;
}

sub modify_catsettings {
    my ($r,$cdom,$cnum,$cdesc,$domdesc,$type) = @_;
    &print_header($r,$type);
    my ($ccrole,%desc);
    if ($type eq 'Community') {
        $desc{'hidefromcat'} = &mt('Excluded from community catalog');
        $desc{'categories'} = &mt('Assigned categories for this community');
        $ccrole = 'co';
    } else {
        $desc{'hidefromcat'} = &mt('Excluded from course catalog');
        $desc{'categories'} = &mt('Assigned categories for this course');
        $ccrole = 'cc';
    }
    $r->print('
<form action="/adm/modifycourse" method="post" name="processcat">
<h3>'.&mt('Category settings').'</h3>');
    my %domconf = &Apache::lonnet::get_dom('configuration',['coursecategories'],$cdom);
    my @cat_params = &catalog_settable($domconf{'coursecategories'},$type);
    if (@cat_params > 0) {
        my (%cenv,@changes,@nochanges);
        my %currsettings =
            &Apache::lonnet::get('environment',['hidefromcat','categories'],$cdom,$cnum);
        my (@newcategories,%showitem); 
        if (grep(/^togglecats$/,@cat_params)) {
            if ($currsettings{'hidefromcat'} ne $env{'form.hidefromcat'}) {
                push(@changes,'hidefromcat');
                $cenv{'hidefromcat'} = $env{'form.hidefromcat'};
            } else {
                push(@nochanges,'hidefromcat');
            }
            if ($env{'form.hidefromcat'} eq 'yes') {
                $showitem{'hidefromcat'} = '"'.&mt('Yes')."'";
            } else {
                $showitem{'hidefromcat'} = '"'.&mt('No').'"';
            }
        }
        if (grep(/^categorize$/,@cat_params)) {
            my (@cats,@trails,%allitems,%idx,@jsarray);
            if (ref($domconf{'coursecategories'}) eq 'HASH') {
                my $cathash = $domconf{'coursecategories'}{'cats'};
                if (ref($cathash) eq 'HASH') {
                    &Apache::loncommon::extract_categories($cathash,\@cats,\@trails,
                                                           \%allitems,\%idx,\@jsarray);
                }
            }
            @newcategories =  &Apache::loncommon::get_env_multiple('form.usecategory');
            if (@newcategories == 0) {
                $showitem{'categories'} = '"'.&mt('None').'"';
            } else {
                $showitem{'categories'} = '<ul>';
                foreach my $item (@newcategories) {
                    $showitem{'categories'} .= '<li>'.$trails[$allitems{$item}].'</li>';
                }
                $showitem{'categories'} .= '</ul>';
            }
            my $catchg = 0;
            if ($currsettings{'categories'} ne '') {
                my @currcategories = split('&',$currsettings{'categories'});
                foreach my $cat (@currcategories) {
                    if (!grep(/^\Q$cat\E$/,@newcategories)) {
                        $catchg = 1;
                        last;
                    }
                }
                if (!$catchg) {
                    foreach my $cat (@newcategories) {
                        if (!grep(/^\Q$cat\E$/,@currcategories)) {
                            $catchg = 1;
                            last;                     
                        } 
                    } 
                }
            } else {
                if (@newcategories > 0) {
                    $catchg = 1;
                }
            }
            if ($catchg) {
                $cenv{'categories'} = join('&',@newcategories);
                push(@changes,'categories');
            } else {
                push(@nochanges,'categories');
            }
            if (@changes > 0) {
                my $putreply = &Apache::lonnet::put('environment',\%cenv,$cdom,$cnum);
                if ($putreply eq 'ok') {
                    if ($env{'course.'.$cdom.'_'.$cnum.'.description'} ne '') {
                        my %newenvhash;
                        foreach my $item (@changes) {
                            $newenvhash{'course.'.$cdom.'_'.$cnum.'.'.$item} = $cenv{$item};
                        }
                        &Apache::lonnet::appenv(\%newenvhash);
                    }
                    my %crsinfo = &Apache::lonnet::courseiddump($cdom,'.',1,'.','.',
                                                                $cnum,undef,undef,'.');
                    if (ref($crsinfo{$env{'form.pickedcourse'}}) eq 'HASH') {
                        if (grep(/^hidefromcat$/,@changes)) {
                            $crsinfo{$env{'form.pickedcourse'}}{'hidefromcat'} = $env{'form.hidefromcat'};
                        }
                        if (grep(/^categories$/,@changes)) {
                            $crsinfo{$env{'form.pickedcourse'}}{'categories'} = $cenv{'categories'};
                        }
                        my $chome = &Apache::lonnet::homeserver($cnum,$cdom);
                        my $putres = &Apache::lonnet::courseidput($cdom,\%crsinfo,$chome,'notime');
                    }
                    $r->print(&mt('The following changes occurred:').'<ul>');
                    foreach my $item (@changes) {
                        $r->print('<li>'.&mt('[_1] now set to: [_2]',$desc{$item},$showitem{$item}).'</li>');
                    }
                    $r->print('</ul><br />');
                }
            }
            if (@nochanges > 0) {
                $r->print(&mt('The following were unchanged:').'<ul>');
                foreach my $item (@nochanges) {
                    $r->print('<li>'.&mt('[_1] still set to: [_2]',$desc{$item},$showitem{$item}).'</li>');
                }
                $r->print('</ul>');
            }
        }
    } else {
        my $newrole = $ccrole.'./'.$cdom.'/'.$cnum;
        my $escuri = &HTML::Entities::encode('/adm/roles?selectrole=1&'.$newrole.
                                             '=1&destinationurl=/adm/courseprefs','&<>"');
        if ($type eq 'Community') {
            $r->print(&mt('Category settings for communities in this domain should be modified in community context (via "[_1]Community Configuration[_2]").','<a href="$escuri">','</a>').'<br />');
        } else {
            $r->print(&mt('Category settings for courses in this domain should be modified in course context (via "[_1]Course Configuration[_2]").','<a href="$escuri">','</a>').'<br />');
        }
    }
    $r->print('<br />'."\n".
              '<a href="javascript:changePage(document.processcat,'."'menu'".')">'.
              &mt('Pick another action').'</a>');
    $r->print(&hidden_form_elements().'</form>');
    return;
}

sub print_header {
    my ($r,$type,$javascript_validations) = @_;
    my $phase = "start";
    if ( exists($env{'form.phase'}) ) {
        $phase = $env{'form.phase'};
    }
    my $js = qq|

function changePage(formname,newphase) {
    formname.phase.value = newphase;
    if (newphase == 'processparms') {
        return;
    }
    formname.submit();
}

|;
    if ($phase eq 'setparms') {
	$js .= $javascript_validations;
    } elsif ($phase eq 'courselist') {
        $js .= <<"ENDJS";
function hide_searching() {
    if (document.getElementById('searching')) {
        document.getElementById('searching').style.display = 'none';
    }
    return;
}

ENDJS
    } elsif ($phase eq 'setquota') {
        my $invalid = &mt('The quota you entered contained invalid characters.');
        my $alert = &mt('You must enter a number');
        &js_escape(\$invalid);
        &js_escape(\$alert);
        my $regexp = '/^\s*(\d+\.?\d*|\.\d+)\s*$/';
        $js .= <<"ENDSCRIPT";

function verify_quota() {
    var newquota = document.setquota.coursequota.value; 
    var num_reg = $regexp;
    if (num_reg.test(newquota)) {
        changePage(document.setquota,'processquota');
    } else {
        alert("$invalid\\n$alert");
        return false;
    }
    return true;
}

ENDSCRIPT
    } elsif ($phase eq 'setanon') {
        my $invalid = &mt('The responder threshold you entered is invalid.');
        my $alert = &mt('You must enter a positive integer.');
        &js_escape(\$invalid);
        &js_escape(\$alert);
        my $regexp = ' /^\s*\d+\s*$/';
        $js .= <<"ENDSCRIPT";

function verify_anon_threshold() {
    var newthreshold = document.setanon.threshold.value;
    var num_reg = $regexp;
    if (num_reg.test(newthreshold)) {
        if (newthreshold > 0) {
            changePage(document.setanon,'processthreshold');
        } else {
            alert("$invalid\\n$alert");
            return false;
        }
    } else {
        alert("$invalid\\n$alert");
        return false;
    }
    return true;
}

ENDSCRIPT
    } elsif ($phase eq 'setpostsubmit') {
        my $invalid = &mt('The choice entered for disabling the submit button is invalid.');
        my $invalidtimeout = &mt('The timeout you entered for disabling the submit button is invalid.');
        my $alert = &mt('Enter one of: a positive integer, 0 (for no timeout), or leave blank to use domain default');
        &js_escape(\$invalid);
        &js_escape(\$invalidtimeout);
        &js_escape(\$alert);
        my $regexp = ' /^\s*\d+\s*$/';

        $js .= <<"ENDSCRIPT"; 

function verify_postsubmit() {
    var optionsElement = document.setpostsubmit.postsubmit;
    var verified = '';
    if (optionsElement.length) {
        var currval;
        for (var i=0; i<optionsElement.length; i++) {
            if (optionsElement[i].checked) {
               currval = optionsElement[i].value;
            }
        }
        if (currval == 1) {
            var newtimeout = document.setpostsubmit.postsubtimeout.value;
            if (newtimeout == '') {
                verified = 'ok';
            } else {
                var num_reg = $regexp;
                if (num_reg.test(newtimeout)) {
                    if (newtimeout>= 0) {
                        verified = 'ok';
                    } else {
                        alert("$invalidtimeout\\n$alert");
                        return false;
                    }
                } else {
                    alert("$invalid\\n$alert");
                    return false;
                }
            }
        } else {
            if (currval == 0) {
               verified = 'ok'; 
            } else {
               alert('$invalid');
               return false;
            }
        }
        if (verified == 'ok') {
            changePage(document.setpostsubmit,'processpostsubmit');
            return true;
        }
    }
    return false;
}

function togglePostsubmit(caller) {
    var optionsElement = document.setpostsubmit.postsubmit;
    if (document.getElementById(caller)) {
        var divitem = document.getElementById(caller);
        var optionsElement = document.setpostsubmit.postsubmit; 
        if (optionsElement.length) {
            var currval;
            for (var i=0; i<optionsElement.length; i++) {
                if (optionsElement[i].checked) {
                   currval = optionsElement[i].value;
                }
            }
            if (currval == 1) {
                divitem.style.display = 'block';
            } else {
                divitem.style.display = 'none';
            }
        }
    }
    return;
}

ENDSCRIPT

    }
    my $starthash;
    if ($env{'form.phase'} eq 'adhocrole') {
        $starthash = {
           add_entries => {'onload' => "javascript:document.adhocrole.submit();"},
                     };
    } elsif ($phase eq 'courselist') {
        $starthash = {
           add_entries => {'onload' => "hide_searching(); courseSet(document.filterpicker.official, 'load');"},
                     };
    }
    $r->print(&Apache::loncommon::start_page('View/Modify Course/Community Settings',
					     &Apache::lonhtmlcommon::scripttag($js),
                                             $starthash));
    my $bread_text = "View/Modify Courses/Communities";
    if ($type eq 'Community') {
        $bread_text = 'Community Settings';
    } else {
        $bread_text = 'Course Settings';
    }
    $r->print(&Apache::lonhtmlcommon::breadcrumbs($bread_text));
    return;
}

sub print_footer {
    my ($r) = @_;
    $r->print('<br />'.&Apache::loncommon::end_page());
    return;
}

sub check_course {
    my ($dom,$domdesc) = @_;
    my ($ok_course,$description,$instcode);
    my %coursehash;
    if ($env{'form.pickedcourse'} =~ /^$match_domain\_$match_courseid$/) {
        my %args;
        unless ($env{'course.'.$env{'form.pickedcourse'}.'.description'}) {
            %args = (
                      'one_time'      => 1,
                      'freshen_cache' => 1,
                    );
        }
        %coursehash =
           &Apache::lonnet::coursedescription($env{'form.pickedcourse'},\%args);
        my $cnum = $coursehash{'num'};
        my $cdom = $coursehash{'domain'};
        $description = $coursehash{'description'};
        $instcode = $coursehash{'internal.coursecode'};
        if ($instcode) {
            $description .= " ($instcode)";
        }
        if (($cdom eq $dom) && ($cnum =~ /^$match_courseid$/)) {
            my %courseIDs = &Apache::lonnet::courseiddump($cdom,'.',1,'.','.',
                                                          $cnum,undef,undef,'.');
            if ($courseIDs{$cdom.'_'.$cnum}) {
                $ok_course = 'ok';
            }
        }
    }
    return ($ok_course,$description,\%coursehash);
}

sub course_settings_descrip {
    my ($type) = @_;
    my %longtype;
    if ($type eq 'Community') {
         %longtype = &Apache::lonlocal::texthash(
                      'courseowner'      => "Username:domain of community owner",
                      'co-owners'        => "Username:domain of each co-owner",
                      'selfenrollmgrdc'  => "Community-specific self-enrollment configuration by Domain Coordinator",
                      'selfenrollmgrcc'  => "Community-specific self-enrollment configuration by Community personnel",
                      'mysqltables'      => '"Temporary" student performance tables lifetime (seconds)',
         );
    } else {
         %longtype = &Apache::lonlocal::texthash(
                      'authtype' => 'Default authentication method',
                      'autharg'  => 'Default authentication parameter',
                      'autoadds' => 'Automated adds',
                      'autodrops' => 'Automated drops',
                      'autostart' => 'Date of first automated enrollment',
                      'autoend' => 'Date of last automated enrollment',
                      'default_enrollment_start_date' => 'Date of first student access',
                      'default_enrollment_end_date' => 'Date of last student access',
                      'coursecode' => 'Official course code',
                      'courseowner' => "Username:domain of course owner",
                      'co-owners'   => "Username:domain of each co-owner",
                      'notifylist' => 'Course Coordinators to be notified of enrollment changes',
                      'sectionnums' => 'Course section number:LON-CAPA section',
                      'crosslistings' => 'Crosslisted class:LON-CAPA section',
                      'defaultcredits' => 'Credits',
                      'autodropfailsafe' => "Failsafe section enrollment count",
                      'selfenrollmgrdc'  => "Course-specific self-enrollment configuration by Domain Coordinator",
                      'selfenrollmgrcc'  => "Course-specific self-enrollment configuration by Course personnel",
                      'mysqltables'      => '"Temporary" student performance tables lifetime (seconds)',
                      'nopasswdchg' => 'Disable changing password for users with student role by course owner',
         );
    }
    return %longtype;
}

sub hidden_form_elements {
    my $hidden_elements = 
      &Apache::lonhtmlcommon::echo_form_input(['gosearch','updater','coursecode',
          'prevphase','numlocalcc','courseowner','login','coursequota','intarg',
          'locarg','krbarg','krbver','counter','hidefromcat','usecategory',
          'threshold','postsubmit','postsubtimeout','defaultcredits','uploadquota',
          'selfenrollmgrdc','selfenrollmgrcc','action','state','currsec_st',
          'sections','newsec','mysqltables','nopasswdchg'],
          ['^selfenrollmgr_','^selfenroll_'])."\n".
          '<input type="hidden" name="prevphase" value="'.$env{'form.phase'}.'" />';
    return $hidden_elements;
}

sub showcredits {
    my ($dom) = @_;
    my %domdefaults = &Apache::lonnet::get_domain_defaults($dom);
    if ($domdefaults{'officialcredits'} || $domdefaults{'unofficialcredits'} || $domdefaults{'textbookcredits'}) {
        return 1;
    }
}

sub get_permission {
    my ($dom) = @_;
    my ($allowed,%permission);
    my %passwdconf = &Apache::lonnet::get_passwdconf($dom);
    if (&Apache::lonnet::allowed('ccc',$dom)) {
        $allowed = 1;
        %permission = (
            setquota          => 'edit',
            processquota      => 'edit',
            setanon           => 'edit',
            processthreshold  => 'edit',
            setpostsubmit     => 'edit',
            processpostsubmit => 'edit',
            viewparms         => 'view',
            setparms          => 'edit',
            processparms      => 'edit',
            catsettings       => 'edit',
            processcat        => 'edit',
            selfenroll        => 'edit',
            adhocrole         => 'coord',
        );
        if ($passwdconf{'crsownerchg'}) {
            $permission{passwdchg} = 'edit';
        }
    } elsif (&Apache::lonnet::allowed('rar',$dom)) {
        $allowed = 1;
        %permission = (
            setquota      => 'view',
            viewparms     => 'view',
            setanon       => 'view',
            setpostsubmit => 'view',
            setparms      => 'view',
            catsettings   => 'view',
            selfenroll    => 'view',
            adhocrole     => 'custom',
        );
        if ($passwdconf{'crsownerchg'}) {
            $permission{passwdchg} = 'view';
        }
    }
    return ($allowed,\%permission);
}

sub devalidate_remote_instcats {
    if ($modified_dom ne '') {
        my %servers = &Apache::lonnet::internet_dom_servers($modified_dom);
        my %thismachine;
        map { $thismachine{$_} = 1; } &Apache::lonnet::current_machine_ids();
        if (keys(%servers)) {
            foreach my $server (keys(%servers)) {
                next if ($thismachine{$server});
                &Apache::lonnet::remote_devalidate_cache($server,['instcats:'.$modified_dom]);
            }
        }
        $modified_dom = '';
    }
    return OK;
}

sub handler {
    my $r = shift;
    if ($r->header_only) {
        &Apache::loncommon::content_type($r,'text/html');
        $r->send_http_header;
        return OK;
    }

    $registered_cleanup=0;
    $modified_dom = '';

    my $dom = $env{'request.role.domain'};
    my $domdesc = &Apache::lonnet::domain($dom,'description');
    my ($allowed,$permission) = &get_permission($dom);
    if ($allowed) {
        &Apache::loncommon::content_type($r,'text/html');
        $r->send_http_header;

        &Apache::lonhtmlcommon::clear_breadcrumbs();

        my $phase = $env{'form.phase'};
        if ($env{'form.updater'}) {
            $phase = '';
        }
        if ($phase eq '') {
            &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>"/adm/modifycourse",
              text=>"Course/Community search"});
            &print_course_search_page($r,$dom,$domdesc);
        } else {
            my $firstform = $phase;
            if ($phase eq 'courselist') {
                $firstform = 'filterpicker';
            }
            my $choose_text;
            my $type = $env{'form.type'};
            if ($type eq '') {
                $type = 'Course';
            }
            if ($type eq 'Community') {
                $choose_text = "Choose a community";
            } else {
                $choose_text = "Choose a course";
            } 
            &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>"javascript:changePage(document.$firstform,'')",
              text=>"Course/Community search"},
              {href=>"javascript:changePage(document.$phase,'courselist')",
              text=>$choose_text});
            if ($phase eq 'courselist') {
                &print_course_selection_page($r,$dom,$domdesc,$permission);
            } else {
                my ($checked,$cdesc,$coursehash) = &check_course($dom,$domdesc);
                if ($checked eq 'ok') {
                    my $enter_text;
                    if ($type eq 'Community') {
                        $enter_text = 'Enter community';
                    } else {
                        $enter_text = 'Enter course';
                    }
                    if ($phase eq 'menu') {
                        &Apache::lonhtmlcommon::add_breadcrumb
                        ({href=>"javascript:changePage(document.$phase,'menu')",
                          text=>"Pick action"});
                        &print_modification_menu($r,$cdesc,$domdesc,$dom,$type,
                                                 $env{'form.pickedcourse'},$coursehash,
                                                 $permission);
                    } elsif ($phase eq 'adhocrole') {
                        &Apache::lonhtmlcommon::add_breadcrumb
                         ({href=>"javascript:changePage(document.$phase,'adhocrole')",
                           text=>$enter_text});
                        &print_adhocrole_selected($r,$type,$permission);
                    } else {
                        &Apache::lonhtmlcommon::add_breadcrumb
                        ({href=>"javascript:changePage(document.$phase,'menu')",
                          text=>"Pick action"});
                        my ($cdom,$cnum) = split(/_/,$env{'form.pickedcourse'});
                        my ($readonly,$linktext);
                        if ($permission->{$phase} eq 'view') {
                           $readonly = 1;
                        }
                        if (($phase eq 'setquota') && ($permission->{'setquota'})) {
                            if ($permission->{'setquota'} eq 'view') {
                                $linktext = 'Set quota';
                            } else {
                                $linktext = 'Display quota';
                            }
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'$phase')",
                              text=>$linktext});
                            &print_setquota($r,$cdom,$cnum,$cdesc,$type,$readonly);
                        } elsif (($phase eq 'processquota') && ($permission->{'processquota'})) { 
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'setquota')",
                              text=>"Set quota"});
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'$phase')",
                              text=>"Result"});
                            &modify_quota($r,$cdom,$cnum,$cdesc,$domdesc,$type);
                        } elsif (($phase eq 'setanon') && ($permission->{'setanon'})) {
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'$phase')",
                              text=>"Threshold for anonymous submissions display"});
                            &print_set_anonsurvey_threshold($r,$cdom,$cnum,$cdesc,$type,$readonly);
                        } elsif (($phase eq 'processthreshold') && ($permission->{'processthreshold'})) {
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'setanon')",
                              text=>"Threshold for anonymous submissions display"});
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'$phase')",
                              text=>"Result"});
                            &modify_anonsurvey_threshold($r,$cdom,$cnum,$cdesc,$domdesc,$type);
                        } elsif (($phase eq 'setpostsubmit') && ($permission->{'setpostsubmit'})) {
                            if ($permission->{'setpostsubmit'} eq 'view') {
                                $linktext = 'Submit button behavior post-submission';
                            } else {
                                $linktext = 'Configure submit button behavior post-submission';
                            }
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'$phase')",
                              text=>$linktext});
                            &print_postsubmit_config($r,$cdom,$cnum,$cdesc,$type,$readonly);
                        } elsif (($phase eq 'processpostsubmit') && ($permission->{'processpostsubmit'})) {
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'$phase')",
                              text=>"Result"});
                            &modify_postsubmit_config($r,$cdom,$cnum,$cdesc,$domdesc,$type);
                        } elsif (($phase eq 'viewparms') && ($permission->{'viewparms'})) {
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'viewparms')",
                              text=>"Display settings"});
                            &print_settings_display($r,$cdom,$cnum,$cdesc,$type,$permission);
                        } elsif (($phase eq 'setparms') && ($permission->{'setparms'})) {
                            if ($permission->{'setparms'} eq 'view') {
                                $linktext = 'Display settings';
                            } else {
                                $linktext = 'Change settings';
                            }
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'$phase')",
                              text=>$linktext});
                            &print_course_modification_page($r,$cdom,$cnum,$cdesc,$type,$readonly);
                        } elsif (($phase eq 'processparms') && ($permission->{'processparms'})) {
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'setparms')",
                              text=>"Change settings"});
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'$phase')",
                              text=>"Result"});
                            &modify_course($r,$cdom,$cnum,$cdesc,$domdesc,$type);
                        } elsif (($phase eq 'catsettings') && ($permission->{'catsettings'})) {
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'$phase')",
                              text=>"Catalog settings"});
                            &print_catsettings($r,$cdom,$cnum,$cdesc,$type,$readonly);
                        } elsif (($phase eq 'processcat') && ($permission->{'processcat'})) {
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'catsettings')",
                              text=>"Catalog settings"});
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'$phase')",
                              text=>"Result"});
                            &modify_catsettings($r,$cdom,$cnum,$cdesc,$domdesc,$type);
                        } elsif (($phase eq 'selfenroll') && ($permission->{'selfenroll'})) {
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href => "javascript:changePage(document.$phase,'$phase')",
                              text => "Self-enrollment settings"});
                            if (!exists($env{'form.state'})) {
                                &print_selfenrollconfig($r,$type,$cdesc,$coursehash,$readonly);
                            } elsif ($env{'form.state'} eq 'done') {
                                &Apache::lonhtmlcommon::add_breadcrumb 
                                ({href=>"javascript:changePage(document.$phase,'$phase')",
                                  text=>"Result"});
                                &modify_selfenrollconfig($r,$type,$cdesc,$coursehash);
                            }
                        }
                    }
                } else {
                    $r->print('<span class="LC_error">');
                    if ($type eq 'Community') {
                        $r->print(&mt('The community you selected is not a valid community in this domain'));
                    } else {
                        $r->print(&mt('The course you selected is not a valid course in this domain'));
                    }
                    $r->print(" ($domdesc)</span>");
                }
            }
        }
        &print_footer($r);
    } else {
        $env{'user.error.msg'}=
        "/adm/modifycourse:ccc:0:0:Cannot modify course/community settings";
        return HTTP_NOT_ACCEPTABLE;
    }
    return OK;
}

1;
__END__
