# The LearningOnline Network with CAPA
# Helpdesk request form
#
# $Id: lonsupportreq.pm,v 1.79.2.12 2024/07/15 17:13:06 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#

package Apache::lonsupportreq;

use strict;
use MIME::Types;
use MIME::Lite;
use CGI::Cookie();
use Apache::Constants qw(:common);
use Apache::loncommon();
use Apache::lonhtmlcommon;
use Apache::lonnet;
use Apache::lonlocal;
use Apache::lonacc();
use Apache::lonauth();
use Apache::courseclassifier;
use LONCAPA qw(:DEFAULT :match);
use HTML::Entities;

sub handler {
    my ($r) = @_;
    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;

    if ($r->header_only) {
        return OK;
    }
    if ($r->uri eq '/adm/helpdesk') {
	&Apache::lonlocal::get_language_handle($r);
    }

    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},['origurl','function']);
    if ($r->uri eq '/adm/helpdesk') {
        &Apache::lonacc::get_posted_cgi($r);
    }
    my $function;
    if ($env{'form.function'}) {
        if (($env{'form.function'} eq 'norole')  ||
            ($env{'form.function'} eq 'student') ||
            ($env{'form.function'} eq 'admin')   ||
            ($env{'form.function'} eq 'author')) {
            $function = $env{'form.function'};
        }
    }
    my $origurl = $env{'form.origurl'};
    if ($origurl =~ m{^https?://[^/]+(.*)$}) {
        $origurl = $1;
    }
    $origurl =~ s/(`)//g;
    $origurl =~ s/\$/\(\$\)/g;
    my $command = $env{'form.command'};

    if ($command eq 'process') {
        &print_request_receipt($r,$origurl,$function);
    } else {
        &print_request_form($r,$origurl,$function);
    }
    return OK;
}

sub print_request_form {
    my ($r,$origurl,$function) = @_;
    my ($os,$browser,$bversion,$uname,$udom,$uhome,$urole,$usec,$email,$cid,
        $cdom,$cnum,$ctitle,$ccode,$sectionlist,$lastname,$firstname,$server,
        $formname,$public,$homeserver,$knownuser,$captcha_form,$captcha_error,
        $captcha,$recaptcha_version,$extra_validations,%groupid);
    $function = &Apache::loncommon::get_users_function() if (!$function);
    $ccode = '';
    $os = $env{'browser.os'};
    $browser = $env{'browser.type'};
    $bversion = $env{'browser.version'};
    if (($env{'user.name'} eq 'public') && ($env{'user.domain'} eq 'public')) {
        $public = 1;
    } else {
        if (($env{'user.name'} ne '') && ($env{'user.domain'} ne '')) {
            $homeserver = &Apache::lonnet::homeserver($env{'user.name'},
                                                      $env{'user.domain'});
            if ($homeserver eq 'no_host') {
                undef($homeserver); 
            } else {
                $uname = $env{'user.name'};
                $udom = $env{'user.domain'};
            }
        }
    }
    if (($env{'user.name'} =~ /^$match_username$/) &&
        ($env{'user.domain'} =~ /^$match_domain$/) && (!$public)) {
        $knownuser = 1;
    } else {
        my $lonhost = $r->dir_config('lonHostID');
        ($captcha_form,$captcha_error,$captcha,$recaptcha_version) =
            &Apache::loncommon::captcha_display('login',$lonhost);
    }
    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},['codedom',
                                                 'useremail','useraccount']);

    if ($homeserver) {
        $uhome = $env{'user.home'};
        $urole = $env{'request.role'};
        $usec = $env{'request.course.sec'};
        $cid = $env{'request.course.id'};
    }
    if ($cid) {
        $cdom = $env{'course.'.$cid.'.domain'};
        $cnum = $env{'course.'.$cid.'.num'};
    }
    if ($cdom && $cnum) {
        my %csettings = &Apache::lonnet::get('environment',['description','internal.coursecode','internal.sectionnums'],$cdom,$cnum);
        $ctitle = $csettings{'description'};
        $ccode = $csettings{'internal.coursecode'};
        $sectionlist = $csettings{'internal.sectionnums'};
        my @sections = split(/,/,$sectionlist);
        foreach my $section (@sections) {
           my ($sec,$grp) = split(/:/,$section);
           $groupid{$sec} = $grp;
        }
    }

    $formname = 'logproblem';
    my $codedom = &get_domain();
    my %helpform;
    my %domconfig =
         &Apache::lonnet::get_dom('configuration',['contacts'],$codedom);
    if (ref($domconfig{'contacts'}) eq 'HASH') {
        if (ref($domconfig{'contacts'}{'helpform'}) eq 'HASH') {
            %helpform = %{$domconfig{'contacts'}{'helpform'}};
        }
    }

    my %codes;
    my @codetitles;
    my %cat_titles;
    my %cat_order;
    my %idlist;
    my %idnums;
    my %idlist_titles;
    my $totcodes = 0;
    my $jscript = '';
    my $loaditems = qq|
function initialize_codes() {
    return;
}
    |;
    unless ($helpform{'course'} eq 'no') {
        my $instcats = &Apache::lonnet::get_dom_instcats($codedom);
        if (ref($instcats) eq 'HASH') {
            if ((ref($instcats->{'codetitles'}) eq 'ARRAY') && (ref($instcats->{'codes'}) eq 'HASH') &&
                (ref($instcats->{'cat_titles'}) eq 'HASH') && (ref($instcats->{'cat_order'}) eq 'HASH')) {
                %codes = %{$instcats->{'codes'}};
                @codetitles = @{$instcats->{'codetitles'}};
                %cat_titles = %{$instcats->{'cat_titles'}};
                %cat_order = %{$instcats->{'cat_order'}};
                $totcodes = scalar(keys(%codes));
            }
            if ($totcodes > 0) {
                my $numtypes = @codetitles;
                &Apache::courseclassifier::build_code_selections(\%codes,\@codetitles,\%cat_titles,\%cat_order,\%idlist,\%idnums,\%idlist_titles);
                my ($scripttext,$longtitles) = &Apache::courseclassifier::javascript_definitions(\@codetitles,\%idlist,\%idlist_titles,\%idnums,\%cat_titles);
                my $longtitles_str = join('","',@{$longtitles});
                my $allidlist = $idlist{$codetitles[0]};
                my $knowncodes;
                if (($ccode ne '') && ($cnum ne '')) {
                    my (%singlecodes,@singlecodetitles,%singlecat_titles,%singlecat_order);
                    my $crscode_reply = &Apache::lonnet::auto_instcode_format($cnum,$codedom,{ $cnum => $ccode },\%singlecodes,
                                                                              \@singlecodetitles,\%singlecat_titles,\%singlecat_order);
                    if (ref($singlecodes{$cnum}) eq 'HASH') {
                        $knowncodes = $singlecodes{$cnum};
                    }
                }
                $jscript .= &Apache::courseclassifier::courseset_js_start($formname,$longtitles_str,$allidlist);
                $jscript .= $scripttext;
                $jscript .= &Apache::courseclassifier::javascript_code_selections($formname,\@codetitles,$knowncodes,$sectionlist);
                $loaditems = '';
            }
        }
    }
    my %missing = &Apache::lonlocal::texthash(
                         username   => 'You must include a name.',
                         phone      => 'You must include a phone.',
    );
    &js_escape(\%missing);
    if (keys(%helpform)) {
        foreach my $key (keys(%helpform)) {
            if (($helpform{$key} eq 'req') && ($missing{$key} ne '')) {
                next if (($key eq 'username') && (($knownuser) ||
                                                  (($env{'user.name'} =~ /^$match_username$/) && (!$public))));
                $extra_validations .= <<"END";

    if (document.$formname.$key.value == '') {
        alert("$missing{$key}");
        return;
    }

END
            }
        }
    }
    my $machine = &Apache::lonnet::absolute_url();
    my $sourceurl = $machine.$origurl;
    $server = $machine.&Apache::loncommon::cleanup_html($origurl);
    $server =~ s/\?.*$//;
    my %js_lt = &Apache::lonlocal::texthash (
                  email => 'The e-mail address you entered',
                  notv => 'is not a valid e-mail address',
                  rsub => 'You must include a subject',
                  rdes => 'You must include a description',
    );
    my %html_lt = &Apache::lonlocal::texthash (
                  name => 'Name',
                  subm => 'Submit Request',
                  emad => 'Your e-mail address',
                  emac => 'Cc',
                  unme => 'username',
                  doma => 'domain',
                  entu => 'Enter the username you use to log-in to LON-CAPA',
                  chdo => 'Choose your LON-CAPA domain',
                  entr => 'Enter the username you use to log-in to LON-CAPA, and your domain.',
                  urlp => 'URL of page',
                  phon => 'Phone',
                  crsd => 'Course Details',
                  enin => 'Enter institutional course code',
                  pick => 'Pick',
                  enct => 'Enter course title',
                  secn => 'Section Number',
                  sele => 'Select',
                  titl => 'Title',
                  lsec => 'LON-CAPA sec',
                  subj => 'Subject',
                  detd => 'Detailed Description',
                  opfi => 'Optional file upload',
                  uplf => 'Upload a file (e.g., a screenshot) relevant to your help request',
                  fini => 'Finish',
                  clfm => 'Clear Form',
    );
    &js_escape(\%js_lt);
    &html_escape(\%html_lt);
    my $scripttag = (<<"END");
function validate() {
    if (validmail(document.logproblem.email) == false) {
        alert("$js_lt{'email'}: "+document.logproblem.email.value+" $js_lt{'notv'}.");
        return;
    }
    if (document.logproblem.subject.value == '') {
        alert("$js_lt{'rsub'}.");
        return;
    }
    if (document.logproblem.description.value == '') {
        alert("$js_lt{'rdes'}.");
        return;
    }
    $extra_validations
    document.logproblem.submit();
}

END
    $scripttag .= &Apache::lonhtmlcommon::javascript_valid_email();

    if ($homeserver) {
        if ($env{'environment.permanentemail'}) {
            $email = $env{'environment.permanentemail'};
        } elsif ($env{'environment.critnotification'}) {
            $email = $env{'environment.critnotification'};
        } elsif ($env{'environment.notification'}) {
            $email = $env{'environment.notification'};
        }
        if ($env{'environment.lastname'}) {
            $lastname = $env{'environment.lastname'};
        }
        if ($env{'environment.firstname'}) {
            $firstname = $env{'environment.firstname'};
        }
    }
    if ($env{'form.origurl'} eq '/adm/createaccount') {
        if ($email eq '') {
            if ($env{'form.useremail'} =~ /^[^\@]+\@[^\@]+$/) {
                $email = &HTML::Entities::encode($env{'form.useremail'},'"<>&');
            }
        }
        if ($uname eq '') {
            if ($env{'form.useraccount'} =~ /^$match_username$/) {
                $uname = &HTML::Entities::encode($env{'form.useraccount'},'"<>&');
            }
        }
    }
    my $details_title;
    if ($codedom) {
        $details_title = '<br />('.$codedom.')';
    }

    my $js = <<"ENDJS";
<script type="text/javascript">
// <![CDATA[
$scripttag
$jscript
$loaditems
// ]]>
</script>
ENDJS
    if ($knownuser) {
        $js .="\n".'<script type="text/javascript" src="/res/adm/includes/file_upload.js"></script>';
    } elsif ($recaptcha_version >=2) {
        $js.= "\n".'<script src="https://www.google.com/recaptcha/api.js"></script>'."\n";
    }
    my %add_entries = (
                       style    => "margin-top:0px;margin-bottom:0px;",
                       onload   => "initialize_codes();",
                      );

    
    $r->print(&Apache::loncommon::start_page('Support Request',$js,
				       { 'function'    => $function,
					 'add_entries' => \%add_entries,
					 'only_body'   => 1,}));
    if ($r->uri eq '/adm/helpdesk') {
        &print_header($r,$origurl);
    }
    my @css = ('LC_evenrow_value','LC_oddrow_value');
    my $num = 1;
    my $i = $num%2;
    my $formtype;
    if ($homeserver) {
        $formtype = ' enctype="multipart/form-data"';
    }
    my $topsubmit = '&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<input type="button" value="'.$html_lt{'subm'}.'" onclick="validate()" />&nbsp;';
    my $shownsubmit;
    $r->print('<form method="post" action="" name="logproblem"'.$formtype.'>'."\n");
    my $output = &Apache::lonhtmlcommon::start_pick_box().
                 &Apache::lonhtmlcommon::row_headline().
                 '<span class="LC_info">'.
                 &mt('(All fields marked with * are required.)').
                 '</span>'.
                 &Apache::lonhtmlcommon::row_closure();
    unless ($helpform{'username'} eq 'no') {
        my ($reqd,$namefield,$fullname);
        if ((defined($lastname) && $lastname ne '') && (defined($firstname) && $firstname ne '')) {
            $fullname = "$firstname $lastname";
            $namefield = $fullname.'<input type="hidden" name="username" value="'.&HTML::Entities::encode($fullname,'"<>&').'" />'."\n";
        } else {
            if (defined($firstname) && $firstname ne '') {
                $fullname = $firstname;
            } elsif (defined($lastname) && $lastname ne '') {
                $fullname = " $lastname";
            }
            $namefield = '<input type="text" size="20" name="username" value="'.&HTML::Entities::encode($fullname,'"<>&').'" />'."\n";
            if ($helpform{'username'} eq 'req') {
                $reqd = '<span class="LC_info">*</span>';
            }
        }
        $output .= &Apache::lonhtmlcommon::row_title($html_lt{'name'}.$reqd,undef,$css[$num])."\n".$namefield.
                   $topsubmit.
                   &Apache::lonhtmlcommon::row_closure()."\n";
        $shownsubmit = 1;
        $num ++;
        $i = $num%2;
    }
    $output .= &Apache::lonhtmlcommon::row_title(
                   '<span title="'.&mt('required').'">'.
                   $html_lt{'emad'}.' <span class="LC_info">*</span></span>'
                  ,undef,$css[$i]).
               '<input type="text" size="20" name="email" value="'.
               &HTML::Entities::encode($email,'"<>&').'" />'."\n";
    unless ($shownsubmit) {
        $output .= $topsubmit;
    }
    $output .= &Apache::lonhtmlcommon::row_closure();
    $num ++;
    $i = $num%2;
    if ($knownuser) {
        if ($homeserver) {
            my $ip = &Apache::lonnet::get_requestor_ip($r);
            my %setters;
            my ($startblock,$endblock,$triggerblock,$by_ip,$blockdom) =
                &Apache::loncommon::blockcheck(\%setters,'com',$ip);
            unless (($helpform{'cc'} eq 'no') || ($by_ip)) {
                $output .= &Apache::lonhtmlcommon::row_title($html_lt{'emac'},undef,$css[$i]).
                           '<input type="text" size="50" name="cc" value="" /><br />'."\n".
                           &Apache::lonhtmlcommon::row_closure();
                $num ++;
                $i = $num%2;
            }
        }
    }
    unless ($helpform{'user'} eq 'no') {
        $output .= &Apache::lonhtmlcommon::row_title("$html_lt{'unme'}/$html_lt{'doma'}",undef,$css[$i]);
        my $udom_input = '<input type="hidden" name="udom" value="'.
                         &HTML::Entities::encode($udom,'"<>&').'" />'."\n";
        my $uname_input = '<input type="hidden" name="uname" value="'.
                         &HTML::Entities::encode($uname,'"<>&').'" />'."\n";
        if ($knownuser) {
            $output .= '<i>'.$html_lt{'unme'}.'</i>:&nbsp;'.$uname.'&nbsp;&nbsp;<i>'.$html_lt{'doma'}.'</i>:&nbsp;'.$udom.$udom_input.$uname_input;
        } else {
            my $udomform = '';
            my $unameform = '';
            if (($env{'user.domain'} =~ /^$match_domain$/) && (!$public)) {
                $output .= $html_lt{'entu'};
            } elsif (($env{'user.name'} =~ /^$match_username$/) && (!$public)) { 
                $output .= $html_lt{'chdo'};
            } else {
                $output .= $html_lt{'entr'};
            }
            $output .= '<br />'."\n";
            if (!$public) {
                if ($env{'user.domain'} =~ /^$match_domain$/) {
                    $udomform = '<i>'.$html_lt{'doma'}.'</i>:&nbsp;'.$udom.$udom_input;
                } elsif ($env{'user.name'} =~ /^$match_username$/) {
                    $unameform = '<i>'.$html_lt{'unme'}.'</i>:&nbsp;'.$uname.'&nbsp;&nbsp;'.$uname_input;
                }
            }
            if ($udomform eq '') {
                $udomform = '<i>'.$html_lt{'doma'}.'</i>:&nbsp;';
                $udomform .= &Apache::loncommon::select_dom_form($codedom,'udom')."\n";
            }
            if ($unameform eq '') {
                $unameform= '<i>'.$html_lt{'unme'}.'</i>:&nbsp;<input type="text" size="20" name="uname" value="'.$uname.'" />&nbsp;&nbsp;';
            }
            $output .= $unameform.$udomform;
        }
        $output .= &Apache::lonhtmlcommon::row_closure();
        $num ++;
    }
    $i = $num%2;
    $output .= &Apache::lonhtmlcommon::row_title("$html_lt{'urlp'}",undef,$css[$i]).
               $server."\n".'<input type="hidden" name="sourceurl" value="'.
               &HTML::Entities::encode($sourceurl,'"<>&').'" />'."\n".
               &Apache::lonhtmlcommon::row_closure();
    unless ($helpform{'phone'} eq 'no') {
        my $reqd;
        if ($helpform{'phone'} eq 'req') {
            $reqd = '<span class="LC_info">*</span>';
        }
        $output .= &Apache::lonhtmlcommon::row_title($html_lt{'phon'}.$reqd,undef,'LC_evenrow_value').
                   '<input type="text" size="15" name="phone" /><br />'."\n".
                   &Apache::lonhtmlcommon::row_closure();
        $num ++;
        $i = $num%2;
    }
    unless ($helpform{'course'} eq 'no') {
        $output .= &Apache::lonhtmlcommon::row_title("$html_lt{'crsd'}$details_title",undef,$css[$i]);
        if ($totcodes > 0) {
            my $numtitles = @codetitles;
            if ($numtitles == 0) {
                $output .= $html_lt{'enin'}.':&nbsp;
                  <input type="text" name="coursecode" size="15" value="'.&HTML::Entities::encode($ccode,'\'"&<>').'" />'."\n";
            } else {
                my @standardnames = &Apache::loncommon::get_standard_codeitems();
                my $lasttitle = $numtitles;
                if ($numtitles > 4) {
                    $lasttitle = 4;
                }
                my $onchange;
                if ($sectionlist) {
                    $onchange = 'toggleSecVis()';
                }
                $output .= '<table><tr><td>'.$codetitles[0].'<br />'."\n".
                      '<select name="'.$standardnames[0].'" onchange="courseSet('."'$codetitles[0]'".');'.$onchange.'">'."\n".
                      ' <option value="-1">'.$html_lt{'sele'}."</option>\n";
                my @items = ();
                my @longitems = ();
                if ($idlist{$codetitles[0]} =~ /","/) {
                    @items = split(/","/,$idlist{$codetitles[0]});
                } else {
                    $items[0] = $idlist{$codetitles[0]};
                }
                if (defined($idlist_titles{$codetitles[0]})) {
                    if ($idlist_titles{$codetitles[0]} =~ /","/) {
                        @longitems = split(/","/,$idlist_titles{$codetitles[0]});
                    } else {
                        $longitems[0] = $idlist_titles{$codetitles[0]};
                    }
                    for (my $i=0; $i<@longitems; $i++) {
                        if ($longitems[$i] eq '') {
                            $longitems[$i] = $items[$i];
                        }
                    }
                } else {
                    @longitems = @items;
                }
                for (my $i=0; $i<@items; $i++) {
                    $output .= ' <option value="'.$items[$i].'">'.$longitems[$i].'</option>'."\n";
                }
                $output .= '</select></td>';
                for (my $i=1; $i<$numtitles; $i++) {
                    $output .= '<td>'.$codetitles[$i].'<br />'."\n".
                     '<select name="'.$standardnames[$i].'" onchange="courseSet('."'$codetitles[$i]'".');'.$onchange.'">'."\n".
                     '<option value="-1">&lt;-'.$html_lt{'pick'}.' '.$codetitles[$i-1].'</option>'."\n".
                     '</select>'."\n".
                     '</td>'."\n";
                }
                $output .= '</tr></table>';
                if ($numtitles > 4) {
                    $output .= '<br /><br />'.$codetitles[$numtitles].'<br />'."\n".
                          '<select name="'.$standardnames[$numtitles].'" onchange="courseSet('."'$codetitles[$numtitles]'".');'.$onchange.'">'."\n".
                          '<option value="-1">&lt;-'.$html_lt{'pick'}.' '.$codetitles[$numtitles-1].'</option>'."\n".
                          '</select>'."\n";
                }
            }
        } else {
            $output .= $html_lt{'enin'}.':&nbsp;
                  <input type="text" name="coursecode" size="15" value="" />'."\n";
        }
        my $reqd;
        if ($helpform{'course'} eq 'req') {
             $reqd = '<span class="LC_info">*</span>';
        }
        $output .= '<br />'.$html_lt{'enct'}.$reqd.':&nbsp;
                   <input type="text" name="title" size="25" value="'.
                   &HTML::Entities::encode($ctitle,'"<>&').'" />'."\n";
        $output .= &Apache::lonhtmlcommon::row_closure();
        $num ++;
        $i = $num%2;
    }
    unless ($helpform{'section'} eq 'no') {
        $output .= &Apache::lonhtmlcommon::row_title($html_lt{'secn'},undef,$css[$i]);
        if ($sectionlist) {
            $output .= "<div id=\"LC_helpdesk_sectionlist\"><select name=\"sectionsel\">\n".
                       "  <option value=\"\" selected=\"selected\">$html_lt{'sele'}</option>\n";
            foreach my $id (sort(keys(%groupid))) {
                if ($id eq $groupid{$id} || $groupid{$id} eq '') {
                    $output .= "  <option value=".
                               &HTML::Entities::encode($id,'"<>&').
                               " >$id</option>\n";
                } else {
                    $output .= "  <option value=".
                               &HTML::Entities::encode($id,'"<>&').
                               " >$id - ($html_lt{'lsec'}: $groupid{$id})</option>\n";
                }
            }
            $output .= '</select></div>'."\n".
                       '<div id="LC_helpdesk_section" style="display:none">'.
                       '<input type="text" name="sectiontxt" size="10" /></div>'."\n";
        } else {
            $output .= '<input type="text" name="section" size="10" />'."\n";
        }
        $output .= &Apache::lonhtmlcommon::row_closure();
        $num ++;
        $i = $num%2;
    }
    $output .= &Apache::lonhtmlcommon::row_title(
                   '<span title="'.&mt('required').'">'.
                   $html_lt{'subj'}.' <span class="LC_info">*</span></span>'
                  ,undef,'LC_oddrow_value').
               '<input type="text" size="40" name="subject" />'."\n".
               &Apache::lonhtmlcommon::row_closure().
               &Apache::lonhtmlcommon::row_title(
                   '<span title="'.&mt('required').'">'.
                   $html_lt{'detd'}.' <span class="LC_info">*</span></span>'
                  ,undef,'LC_evenrow_value').
               '<textarea rows="10" cols="45" name="description" style="word-wrap:normal;">'.
               '</textarea>'."\n".
               &Apache::lonhtmlcommon::row_closure();
    $num ++;
    $i = $num%2; 
    if ($knownuser) {
        if ($homeserver) {
            unless ($helpform{'screenshot'} eq 'no') {
                my $max = 1048576;
                my $showmax = 1.00;
                if ($helpform{'maxsize'} =~ /^\d+\.\d*$/) {
                    $max *= $helpform{'maxsize'};
                    $showmax = $helpform{'maxsize'};
                }
                $showmax = ' ('.sprintf("%.2f",$showmax).' '.&mt('MB max.').')';
                $output .= &Apache::lonhtmlcommon::row_title($html_lt{'opfi'},undef,$css[$i])
                    .' <input type="file" name="screenshot" class="LC_flUpload" size="20" />'
                    .'<input type="hidden" id="LC_free_space" value="'.$max.'" />'
                    .'<br />'."\n".$html_lt{'uplf'}.$showmax."\n"
                    .&Apache::lonhtmlcommon::row_closure();
                $num ++;
                $i = $num%2;
            }
        }
    } else {
        if ($captcha_form) {
            $output .= &Apache::lonhtmlcommon::row_title(
                           '<span title="'.&mt('required').'">'.
                           &mt('Validation').
                           ' <span class="LC_info">*</span></span>'
                      ,undef,$css[$i]).
                       $captcha_form."\n".
                       &Apache::lonhtmlcommon::row_closure();
            $num ++;
            $i = $num%2;
        }
    }
    $output .= &Apache::lonhtmlcommon::row_title($html_lt{'fini'},undef,$css[$i]);
    $output .= <<END;
             <table border="0" cellpadding="8" cellspacing="0">
              <tr>
               <td>
                <input type="hidden" name="command" value="process" />
                <input type="button" value="$html_lt{'subm'}" onclick="validate()" /> &nbsp;
               </td>
               <td>&nbsp;</td>
               <td>
                <input type="reset" value="$html_lt{'clfm'}" />
               </td>
              </tr>
             </table>
END
    $output .= &Apache::lonhtmlcommon::row_closure(1);
    $output .= &Apache::lonhtmlcommon::end_pick_box();
    $r->print(<<END);
$output
</form>
<br />
END
    $r->print(&Apache::loncommon::end_page());
    return;
}

sub print_request_receipt {
    my ($r,$url,$function) = @_;
    my $public;
    if (($env{'user.name'} eq 'public') && ($env{'user.domain'} eq 'public')) {
        $public = 1;
    }
    my $lonhost = $r->dir_config('lonHostID');
    unless (($env{'user.name'} =~ /^$match_username$/) && (!$public)) {
        my ($captcha_chk,$captcha_error) = 
            &Apache::loncommon::captcha_response('login',$lonhost);
        if ($captcha_chk != 1) {
            $r->print(&Apache::loncommon::start_page('Support request failed',undef,
                                       {'function'    => $function,
                                        'add_entries' => {
                                            topmargin    => "0",
                                            marginheight => "0",
                                        },
                                        'only_body'   => 1,}));
            if ($r->uri eq '/adm/helpdesk') {
                &print_header($r,$url,'process');
            }
            $r->print(
                '<h2>'.&mt('Support request failed').'</h2>'.
                      &Apache::lonhtmlcommon::confirm_success(
                        &mt('Validation of the code you entered failed.'),1).
                '<br /><br />'.
                &Apache::lonhtmlcommon::actionbox([
                    &mt('[_1]Go back[_2] and try again',
                        '<a href="javascript:history.go(-1)">','</a>')]).
                      &Apache::loncommon::end_page());
            return;
        }
    }
    my $ip = &Apache::lonnet::get_requestor_ip($r);
    my %setters;
    my ($startblock,$endblock,$triggerblock,$by_ip,$blockdom) =
        &Apache::loncommon::blockcheck(\%setters,'com',$ip);
    my @ENVvars = ('HTTP_HOST','HTTP_USER_AGENT','SERVER_ADDR','SERVER_NAME');
    my @envvars = ('browser.os','browser.type','browser.version','user.home','request.role');
    my @loncvars = ('user.name','user.domain','request.course.sec','request.course.id');
    my @cookievars;
    if ($ENV{'SERVER_PORT'} == 443) {
        @cookievars = ('lonLinkID');
    } else {
        @cookievars = ('lonID');
    }

    my $admin = $Apache::lonnet::perlvar{'lonAdminMail'};
    my $origmail = $Apache::lonnet::perlvar{'lonSupportEMail'};
    my $defdom = &get_domain();
    my $from = $admin;
    my %helpform;
    my %domconfig =
         &Apache::lonnet::get_dom('configuration',['contacts'],$defdom);
    if (ref($domconfig{'contacts'}) eq 'HASH') {
        if (ref($domconfig{'contacts'}{'helpform'}) eq 'HASH') {
            %helpform = %{$domconfig{'contacts'}{'helpform'}};
        }
    }
    my $reporttime = &Apache::lonlocal::locallocaltime(time);
    my %formvars = (
                     email       => 1,
                     sourceurl   => 1,
                     subject     => 1,
                     description => 1,
                   );
    unless ($helpform{'username'} eq 'no') {
        $formvars{'username'} = 1;
    }
    unless ($helpform{'user'} eq 'no') {
        $formvars{'uname'} = 1;
        $formvars{'udom'} = 1;
    }
    unless ($helpform{'phone'} eq 'no') {
        $formvars{'phone'} = 1;
    }
    unless ($helpform{'section'} eq 'no') {
        $formvars{'section'} = 1;
    }
    unless ($helpform{'course'} eq 'no') {
        $formvars{'course'} = 1;
    }
    unless (($helpform{'cc'} eq 'no') || ($by_ip)) {
        $formvars{'cc'} = 1;
    }
    unless ($helpform{'screenshot'} eq 'no') {
        $formvars{'screenshot'} = 1;
    }
    my $coursecode;
    if ($formvars{'course'}) {
        $coursecode = $env{'form.coursecode'};
        if ($coursecode eq '') {
            my $totcodes = 0;
            my $instcats = &Apache::lonnet::get_dom_instcats($defdom);
            if (ref($instcats) eq 'HASH') {
                if (ref($instcats->{'codes'}) eq 'HASH') {
                    $totcodes = scalar(keys(%{$instcats->{'codes'}}));
                }
            }
            my @standardnames = &Apache::loncommon::get_standard_codeitems();
            if ($totcodes > 0) {
                my $noregexps = 1;
                $coursecode = 
                    &Apache::courseclassifier::instcode_from_selectors($defdom,$noregexps);
            }
            if ($coursecode eq '') {
                foreach my $item (@standardnames) {
                    if ((defined($env{'form.'.$item})) && ($env{'form.'.$item} ne '-1')) {
                        $coursecode .= $env{'form.'.$item};
                    }
                }
            }
        }
    }
    my %lt = &Apache::lonlocal::texthash (
                 username    => 'Name',
                 email       => 'E-mail',
                 cc          => 'Cc',
                 user        => 'Username/domain',
                 phone       => 'Phone',
                 crsi        => 'Course Information',
                 subject     => 'Subject',
                 description => 'Description',
                 sourceurl   => 'URL',
                 date        => 'Date/Time',
                 secn        => 'Section',
                 warn        => 'Warning: Problem with support e-mail address',
                 your        => 'Your support request contained the following information',
                 belo        => 'Below is your support request which could not be sent',
                 ifyk        => 'If you know an IT support address for your institution, you could copy this information and send it to that email address.',
                 sect        => 'section',
                 info        => 'Information supplied',
                 adin        => 'Additional information recorded',
    );

    my (@ok_ccs,@bad_ccs,$badccmsg,$okcclist,$homeserver);
    unless ($public) {
        if (($env{'user.name'} ne '') && ($env{'user.domain'} ne '')) {
            $homeserver = &Apache::lonnet::homeserver($env{'user.name'},
                                                      $env{'user.domain'});
        }
    }
    my ($cid,$cdom,$cnum,$sectionlist,$ccode);
    if ($homeserver) {
        $cid = $env{'request.course.id'};
    }
    if ($cid) {
        $cdom = $env{'course.'.$cid.'.domain'};
        $cnum = $env{'course.'.$cid.'.num'};
    }
    if ($cdom && $cnum) {
        my %csettings = &Apache::lonnet::get('environment',['description','internal.coursecode','internal.sectionnums'],$cdom,$cnum);
        $ccode = $csettings{'internal.coursecode'};
        $sectionlist = $csettings{'internal.sectionnums'};
    }
    if (($homeserver) && (defined($env{'form.cc'})) && ($formvars{'cc'})) {
        my @ccs;
        if ($env{'form.cc'} =~ /,/) {
            @ccs = split(/,/,$env{'form.cc'});
        } else {
            $env{'form.cc'} =~ s/^\s+//;
            $env{'form.cc'} =~ s/\s+$//;
            @ccs = $env{'form.cc'};
        }
        foreach my $cc (@ccs) {
            $cc =~ s/^\s+//g;
            $cc =~ s/\s+$//g;
            if ($cc =~ m/^[^\@]+\@[^\@]+$/) {
                if (!(grep(/^\Q$cc\E$/,@ok_ccs))) {
                    push(@ok_ccs,$cc);
                }
            } elsif ($cc ne '') {
                if (!(grep(/^\Q$cc\E$/,@bad_ccs))) {
                    push(@bad_ccs,$cc);
                }
            }
        }
        if (@ok_ccs > 0) {
           $okcclist = join(', ',@ok_ccs); 
        } 
        if (@bad_ccs == 1) {
            if ($bad_ccs[0] ne '') {
                $badccmsg .= '<br />'.&mt('The following Cc e-mail address is invalid: ').&Apache::loncommon::cleanup_html($bad_ccs[0]);
            }
        } elsif (@bad_ccs > 1) {
            $badccmsg .= '<br />'.&mt('The following Cc e-mail addresses are invalid: '). &Apache::loncommon::cleanup_html(join(', ',@bad_ccs));
        }
    }
    $env{'form.user'} = "'".$env{'form.uname'}.':'.$env{'form.udom'}."'";
    if ($formvars{'course'}) {
        $env{'form.crsi'} = $env{'form.title'};
        if ($coursecode ne '') {
            $env{'form.crsi'} .= ' - '.$coursecode;
        }
    }
    if ($formvars{'section'}) {
        my $section;
        if ($sectionlist) {
            if ($coursecode eq $ccode) {
                $section = $env{'form.sectionsel'}
            } else {
                $section = $env{'form.sectiontxt'};
            }
        } else {
            $section = $env{'form.section'};
        }
        $env{'form.crsi'} .= ' - '.$lt{'sect'}.': '.$section;
    }
    my $supportmsg;
    if ($formvars{'username'}) {
        $supportmsg .= "$lt{'username'}: $env{'form.username'}\n";
    }
    $supportmsg .= "$lt{'email'}: $env{'form.email'}\n";
    if (($homeserver) && (defined($env{'form.cc'})) && ($formvars{'cc'})) {
        $supportmsg .= "$lt{'cc'}: $okcclist\n";
    }
    if ($formvars{'user'}) {
        $supportmsg .= "$lt{'user'}: $env{'form.user'}\n";
    }
    if ($formvars{'phone'}) {
        $supportmsg .= "$lt{'phone'}: $env{'form.phone'}\n";
    }
    if ($formvars{'course'}) {
        $supportmsg .= "$lt{'crsi'}: $env{'form.crsi'}\n";
    }
    $supportmsg .= "$lt{'subject'}: $env{'form.subject'}
$lt{'description'}: $env{'form.description'}
$lt{'sourceurl'}: $env{'form.sourceurl'}
$lt{'date'}: $reporttime
";

    my $displaymsg;
    foreach my $item ('username','email','cc','user','phone','crsi','subject','description','sourceurl') {
        if ($env{'form.'.$item} ne '') {
            if ($item eq 'description') {
                my $descrip = $env{'form.description'};
                $descrip =  &Apache::loncommon::cleanup_html($descrip);
                $descrip =~ s|[\n\r\f]|<br />|g;
                $displaymsg .= 
                    '<span class="LC_helpform_receipt_cat">'.
                    "$lt{$item}</span>: $descrip<br />\n";
            } elsif ($item eq 'sourceurl') {
                my $showurl = $env{'form.sourceurl'};
                $showurl =~ s/\?.*$//;
                $showurl =  &Apache::loncommon::cleanup_html($showurl);
                $displaymsg .= 
                    '<span class="LC_helpform_receipt_cat">'.
                    "$lt{$item}</span>: $showurl<br />\n";
            } elsif ($item eq 'cc') {
                next if ($helpform{'cc'} eq 'no');
                if ($okcclist) {
                    my $showcclist = &Apache::loncommon::cleanup_html($okcclist);
                    $displaymsg .=
                        '<span class="LC_helpform_receipt_cat">'.
                        "$lt{$item}</span>: $showcclist<br />\n";
                }
            } else {
                if (($item eq 'username') || ($item eq 'user') || ($item eq 'phone')) {
                    next if ($helpform{$item} eq 'no');
                }
                next if (($item eq 'crsi') && ($helpform{'course'} eq 'no'));
                my $showitem = $env{'form.'.$item};
                $showitem = &Apache::loncommon::cleanup_html($showitem);
                $displaymsg .= 
                    '<span class="LC_helpform_receipt_cat">'.
                    "$lt{$item}</span>: $showitem<br />\n";
            }
        }
    }

    my ($requname,$requdom,$reqemail);
    foreach my $field ('uname','udom','email') {
        $env{'form.'.$field} =~ s/^\s+//;
        $env{'form.'.$field} =~ s/\s+$//;
    }
    if ($env{'form.uname'} =~ /^$match_username$/) {
        $requname = $env{'form.uname'};
    }
    if ($env{'form.udom'} =~ /^$match_domain$/) {
        $requdom = $env{'form.udom'};
    }
    if ($env{'form.email'} =~ m/^[^\@]+\@[^\@]+$/) {
        $reqemail = $env{'form.email'};
    }

    my $dom_in_effect;
    unless ($env{'user.domain'} eq 'public') {
        $dom_in_effect = $env{'user.domain'};
    }
    if ($dom_in_effect eq '') {
        $dom_in_effect = $requdom;
    }
    if ($dom_in_effect eq '') {
        $dom_in_effect = $defdom;
    }

    $displaymsg .= '<span class="LC_helpform_receipt_cat">'.
                   $lt{'date'}.'</span>: '.$reporttime.'<br />'."\n";

    my $start_page = 
	&Apache::loncommon::start_page('Support request recorded',undef,
				       {'function'    => $function,
					'add_entries' => {
					    topmargin    => "0",
					    marginheight => "0",
					},
					'only_body'   => 1,});

    $r->print(<<"END");
$start_page
<form name="logproblem" action="">
<input type="hidden" name="command" value="result" />
</form>
END
    if ($r->uri eq '/adm/helpdesk') {
        &print_header($r,$url,'process');
    }
    my $bad_email = 0;
    my ($to,$bcc,$addtext) =
        &Apache::loncommon::build_recipient_list(undef,'helpdeskmail',
                                                 $dom_in_effect,$origmail,
                                                 $requname,$requdom,
                                                 $reqemail);
    if ($to =~ /,/) {
        my @ok_email; 
        foreach my $email (split(/,/,$to)) {
            if ($email =~ m/^[^\@]+\@[^\@]+$/) {
                if (!grep(/^\Q$email\E$/,@ok_email)) {
                    push(@ok_email,$email);
                }
            }
        }
        if (@ok_email > 0) {
            $to = join(',',@ok_email);
        } elsif ($admin =~ m/^[^\@]+\@[^\@]+$/) {
            $to = $admin;
        } else {
            $bad_email = 1;
        }
    } elsif ($to !~ m/^[^\@]+\@[^\@]+$/) {
        if ($admin =~ m/^[^\@]+\@[^\@]+$/) {
            $to = $admin;
        } else {
            $bad_email = 1;
        }
    }

    my $message;
    if (!$bad_email) {
        $message = &mt('A support request has been sent to [_1]','<tt>'.$to.'</tt>');
    } else {
        $message = $lt{'warn'}.'<br />';
        if ($to eq '') {
            $message .= &mt('For this LON-CAPA server no e-mail address has been configured to receive support requests, so your support request has [_1]not[_2] been sent to the LON-CAPA support staff or administrator at your institution.','<b>','</b>');
        } else {
            $message .= &mt('As the e-mail address provided for this LON-CAPA server ([_1]) does not appear to be a valid e-mail address, your support request has [_2]not[_3] been sent to the LON-CAPA support staff or administrator at your institution.','<tt>'.$to.'</tt>','<b>','</b>');
        }
    }
    $r->print(&Apache::loncommon::confirmwrapper(&Apache::lonhtmlcommon::confirm_success($message,$bad_email)));

    if ($reqemail ne '') {
        $from = $reqemail;
    }

    if (defined($env{'form.cc'})) {
        if ($badccmsg) {
            $displaymsg .= $badccmsg;
        }
    }

    my $subject = $env{'form.subject'};
    $subject =~ s/(`)/'/g;
    $subject =~ s/\$/\(\$\)/g;
    $supportmsg =~ s/(`)/'/g;
    $supportmsg =~ s/\$/\(\$\)/g;
    $displaymsg =~ s/(`)/'/g;
    $displaymsg =~ s/\$/\(\$\)/g;
    my $fname;

    my $attachmentpath = '';
    my $showsize = '';

    if ((defined($env{'user.name'})) && (!$public)) {
        if ($homeserver && $env{'form.screenshot.filename'}) {
            unless ($helpform{'screenshot'} eq 'no') {
                my $attachmentsize = length($env{'form.screenshot'});
                my $max = 1048576;
                my $showmax = 1.00;
                if ($helpform{'maxsize'} =~ /^\d+\.\d*$/) {
                    $max *= $helpform{'maxsize'};
                    $showmax = $helpform{'maxsize'};
                }
                $showmax = '('.sprintf("%.2f",$showmax).' MB)';
                $showsize = $attachmentsize/1048576;
                $showsize = '('.sprintf("%.2f",$showsize).' MB)';
                if ($attachmentsize > $max) {
                    $displaymsg .= '<br /><span class="LC_warning">'.
                                   &mt('The uploaded screenshot file [_1] included with your request exceeded the maximum allowed size [_2], and has therefore been discarded.',$showsize,$showmax).'</span>';
                } else {
                    $attachmentpath=&Apache::lonnet::userfileupload('screenshot',undef,'helprequests');
                }
            }
        }
    }

    my %cookies;
    my $cookie=CGI::Cookie->parse($r->header_in('Cookie'));
    if ($$cookie{'lonID'} =~ /lonID=($LONCAPA::handle_re);/) {
        $cookies{'lonID'} = $1;
    }
    if ($$cookie{'lonLinkID'} =~ /lonLinkID=([a-f0-9]+_linked);/) {
        $cookies{'lonLinkID'} = $1;
    }
    if ($attachmentpath =~ m-/([^/]+)$-) {
        $fname = $1;
        $displaymsg .= '<br />'
                      .&mt('An uploaded screenshot file [_1] was included in the request sent by [_2].'
                          ,'<span class="LC_filename">'.$fname.'</span>&nbsp;'.$showsize,
                          ,$env{'user.name'}.':'.$env{'user.domain'}
                       );
        $supportmsg .= "\n";
        foreach my $var (@cookievars) {
            $supportmsg .= "$var: $cookies{$var}\n";
        }
        foreach my $var(@ENVvars) {
            $supportmsg .= "$var: $ENV{$var}\n";
        }
        $supportmsg .= "REMOTE_ADDR: $ip\n";
        foreach my $var (@envvars) {
            $supportmsg .= "$var: $env{$var}\n";
        }
    }

    my $cc_string;
    if ($homeserver) {
        if (@ok_ccs > 0) {
            $cc_string = join(', ',@ok_ccs);
        }
    }

    my $attachment_text = '';
    unless ($homeserver && $attachmentpath) {
        foreach my $var (@cookievars) {
            $attachment_text .= "$var: $cookies{$var}\n";
        }
        foreach my $var (@ENVvars) {
            $attachment_text .= "$var: $ENV{$var}\n";
        }
        $attachment_text .= "REMOTE_ADDR: $ip\n";
        foreach my $var (@envvars) {
            $attachment_text .= "$var: $env{$var}\n";
        }
        foreach my $var (@loncvars) {
            $attachment_text .= "$var: $env{$var}\n";
        }
    }

    if ($addtext) {
        my ($addloc,$addstr) = split(/:/,$addtext,2);
        $addstr = &unescape($addstr);
        if ($addloc eq 's') {
            $subject = $addstr.' '.$subject;
        } elsif ($addloc eq 'b') {
            $supportmsg = $addstr."\n".$supportmsg;
        }
    }

    unless ($bad_email) {
        my $msg = MIME::Lite->new(
                     From    => $from,
                     To      => $to,
                     Subject => $subject,
                     Type    =>'TEXT',
                     Data    => $supportmsg,
                     );
        if ($homeserver) {
            if (@ok_ccs > 0) {
                my $cc_string = join(', ',@ok_ccs);
                $msg->add("Cc" => $cc_string);
            }
        }
        if ($bcc ne '') {
            $msg->add("Bcc" => $bcc);
        }
        $msg->attr("content-type"         => "text/plain");
        $msg->attr("content-type.charset" => "UTF-8");

        if ($homeserver && $attachmentpath) {
            my ($type, $encoding) = MIME::Types::by_suffix($attachmentpath);
            $msg->attach(Type     => $type,
                         Path     => $attachmentpath,
                         Filename => $fname
                         );

        } elsif ($attachment_text ne '') {
            $msg->attach(Type => 'TEXT',
                         Data => $attachment_text);
        }

        ### Send it:
        $msg->send('sendmail');
    }

    if ($attachmentpath =~ m|$Apache::lonnet::perlvar{'lonDaemons'}/tmp/helprequests/(\d+)/[^/]+|) {
        unlink($attachmentpath);
    }
    if ($bad_email) {
        $r->print('<b>'.$lt{'belo'}.'</b><br />'.
                  $lt{'ifyk'}.'<br /><br />'."\n");
    } else {
        $r->print('<b>'.$lt{'your'}.'</b>:<br /><br />'."\n");
    }
    $r->print('<div style="width:620px;">'.
              &Apache::lonhtmlcommon::start_pick_box().
              &Apache::lonhtmlcommon::row_title($lt{'info'},undef,'LC_oddrow_value')."\n".$displaymsg."\n".
              &Apache::lonhtmlcommon::row_closure().
              &Apache::lonhtmlcommon::row_title($lt{'adin'},undef,'LC_evenrow_value'));
    my $envmsg;
    foreach my $var (@cookievars) {
        if ($cookies{$var} ne '') {
            $envmsg.= '<span class="LC_helpform_receipt_cat">'.
                      $var.'</span>:&nbsp;'.$cookies{$var}.', ';
        }
    }
    foreach my $var (@ENVvars) {
        if ($ENV{$var} ne '') {
            $envmsg .= '<span class="LC_helpform_receipt_cat">'.
                       $var.'</span>:&nbsp;'.$ENV{$var}.', ';
        }
    }
    if ($ip ne '') {
        $envmsg .= '<span class="LC_helpform_receipt_cat">'.
                   'REMOTE_ADDR</span>:&nbsp;'.$ip.', ';
    }
    foreach my $var (@envvars) {
        if ($env{$var} ne '') { 
            $envmsg .= '<span class="LC_helpform_receipt_cat">'.
                       $var.'</span>:&nbsp;'.$env{$var}.', ';
        }
    }
    $envmsg =~ s/, $//;
    $r->print($envmsg."\n".
              &Apache::lonhtmlcommon::row_closure(1)."\n".
              &Apache::lonhtmlcommon::end_pick_box().
              "</div>\n".
              &Apache::loncommon::end_page());
}

sub print_header {
    my ($r,$origurl,$command) = @_;
    my $location=&Apache::loncommon::lonhttpdurl("/adm");
    my ($component_url);
    my $helpdesk_link = '<a href="javascript:validate()">';
    if ($command eq 'process') {
        $helpdesk_link = '<a href="/adm/helpdesk">';
    }
    my %lt = &Apache::lonlocal::texthash (
                                           login    => 'Log-in help',
                                           ask      => 'Ask helpdesk',
                                           getst    => 'Getting started guide',
                                           back     => 'Back to last location',
                                           headline => 'help/support',
                                           stud     => 'Students',
                                           ifyo     => 'If your problem is still unresolved, the form below can be used to send a question to the LON-CAPA helpdesk.',
                                           cont     => 'Contact your instructor instead.',
                                         );
    my ($getstartlink,$reviewtext);
    if (-e $Apache::lonnet::perlvar{'lonDocRoot'}.'/adm/gettingstarted.html') {
        $getstartlink = qq|<td align="center">&nbsp;<b><a href="/adm/gettingstarted.html">$lt{'getst'}</a></td>|;
        $reviewtext = &mt('Please review the information in "Log-in help" and the "Getting started" guide if you are unable to log-in.');
    } else {
        $reviewtext = &mt('Please review the information in "Log-in help" if you are unable to log-in.');
    }
    my $linkback;
    if ($origurl eq '') {
        $linkback = 'javascript:history.go(-1)';
    } else {
        $linkback = &HTML::Entities::encode($origurl,'"<>&');
    }
    my $loginhelp = &Apache::lonauth::loginhelpdisplay();
    if ($loginhelp eq '') {
        $loginhelp = '/adm/loginproblems.html';
    }
    $r->print(<<"END");
<table width="620" border="0" cellspacing="0" cellpadding="0" style="height: 55px;">
 <tr>
   <td width="5" height="50">&nbsp;</td>
   <td height="50">
    <fieldset>
      <legend>
        <img src="$location/lonIcons/minilogo.gif" height="20" width="29" alt="logo" style="vertical-align:bottom" />
        LON-CAPA $lt{'headline'}
      </legend>
 <table id="LC_helpmenu_links">
   <tr>
    <td align="center"><span class="LC_nobreak"><img src="$location/help/help.png" border="0" alt="($lt{'login'})" style="vertical-align:middle" />&nbsp;<b><a href="$loginhelp">$lt{'login'}</a></b>&nbsp;</span></td>
    <td align="center"><span class="LC_nobreak">&nbsp;<b>$helpdesk_link<img src="$location/lonIcons/helpdesk.gif" border="0" alt="($lt{'ask'})" style="vertical-align:middle" />&nbsp;$lt{'ask'}</a></b>&nbsp;</span></td>$getstartlink
    <td align="center"><span class="LC_nobreak">&nbsp;<b><a href="$linkback" target="_top"><img src="$location/lonIcons/move_up.gif" border="0" alt="($lt{'back'})" style="vertical-align:middle" />&nbsp;$lt{'back'}</a></b>&nbsp;</span></td>
   </tr>
 </table>
</fieldset>
  </td>
  <td width="5">&nbsp;</td>
 </tr>
 <tr>
  <td colspan="3" height="5">&nbsp;</td>
 </tr>
END
    if  ($command ne 'process') {
        my $stuwarn = &mt('Do [_1]not[_2] use this form to ask about course content.',
                          '<b>','</b>');
        $r->print(<<"END");
 <tr>
  <td colspan="3">$reviewtext 
  $lt{'ifyo'}
<p class="LC_info">
<b>$lt{'stud'}:</b> 
$stuwarn $lt{'cont'}
</p><br />
  </td>
 </tr>
END
    }
    $r->print('
</table>');
    return;
}

sub get_domain {
    my $codedom;
    if (exists($env{'form.codedom'})) {
        if (&Apache::lonnet::domain($env{'form.codedom'}) ne '') {
            return $env{'form.codedom'};
        }
    }
    if ($env{'request.course.id'}) {
        $codedom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    } elsif ($env{'request.role.domain'}) {
        $codedom = $env{'request.role.domain'};
    } else {
        $codedom = &Apache::lonnet::default_login_domain();
    }
    return $codedom;
}

1;
