/* 
The LearningOnline Network with CAPA
JavaScript functions handling file uploading

$Id: file_upload.js,v 1.4 2023/07/11 22:24:30 raeburn Exp $

Copyright Michigan State University Board of Trustees

This file is part of the LearningOnline Network with CAPA (LON-CAPA).

LON-CAPA is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

LON-CAPA is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with LON-CAPA; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/home/httpd/html/adm/gpl.txt

http://www.lon-capa.org/
*/



/*
This function accepts a file input element and the universal part of the id 
used for the hidden input element containing maximum upload size permitted.
If the file(s) is too large, an alert is shown and the input is cleared.

INPUT:
    fileInput -
        <input type="file" class="LC_flUpload" />
        Using the class "LC_flUpload" is needed to use the event handlers below.
    sizeItem -
        <input type="hidden" id="PREFIXsizeItemSUFFIX" value="$maxsize" /> 

    The PREFIX is empty unless the resource is within a composite page.
    
    The SUFFIX is empty in cases where there will only ever be one file upload
    input element in a web page. Otherwise it will contain a unique identifier,
    so different maximum sizes can exist for each upload element.
 
    The value assigned to the hidden element is the maximum upload size in bytes.

    It is either calculated from quota and disk usage (e.g., upload to a course,
    authoring space or portfolio space), or is set by a parameter (e.g., upload
    to a dropbox, essayresponse or externalresponse item), or is hard-coded 
    (e.g., upload to the help request form, or an attachment to a discussion post
    or feedback message).

*/

function checkUploadSize (fileInput,sizeItem) {
    try {
        var maxSize = getMaxSize(fileInput,sizeItem);
        var fileSize = 0;
        if ('files' in fileInput) {
            if (fileInput.files.length > 0) {
                for (var i = 0; i < fileInput.files.length; i++) {
                    fileSize += fileInput.files[i].size;
                }
                if (fileSize > maxSize) {
                    alert("File(s) too large to be attached");
                    clearFileInput(fileInput);
                }
            }
        } else { alert("no files selected for upload");}
    } catch (e) { alert("Error is: " + e); }
}

/* 
This function clears the contents of a file input element.

INPUT:
    ctrl -
        <input type="file" />
*/
function clearFileInput(ctrl) {
    try {
        ctrl.value = null;
    } catch(e) { }
    if (ctrl.value) {
        ctrl.parentNode.replaceChild(ctrl.cloneNode(true), ctrl);
    }
}

/*
This function retrieves the allowed maximum file size for a file input element
INPUT:
    fileInput -
        <input type="file" />
    sizeItem -
        <input type="hidden" id="PREFIXsizeItemSUFFIX" />

    For upload to a dropbox, essayresponse or externalresponse item,
    the PREFIX and SUFFIX are extracted from the id of the file upload 
    element, by separating the id into parts before and after HWFILE.

    The PREFIX is empty unless the resource is within a composite page.

    For upload to "IMS upload" in Main Content or "Upload File" in 
    Supplemental Content in the Course Editor, the SUFFIX is extracted 
    from the id of the file upload element, by splitting the id on
    underscore, and using the second element.

*/
function getMaxSize (fileInput,sizeItem) {
    var maxSize = 0;
    var sizeId = sizeItem;
    var uploadId;
    try {
        if ($(fileInput).hasClass("LC_hwkfile")) {
            uploadId = $(fileInput).attr('id');
            var re = /^(|\w*)HWFILE(.+)$/;
            var found = uploadId.match(re);
            if (found.length == 3) {
                sizeId = found[1]+sizeItem+'_'+found[2];
            }
        } else if ($(fileInput).hasClass("LC_uploaddoc")) {
            uploadId = $(fileInput).attr('id');
            var re = /^uploaddoc(.+)$/;
            var found = uploadId.match(re);
            if (found.length == 2) {
                sizeId = sizeItem+'_'+found[1]; 
            }
        }
        if ( $("#"+sizeId).length) {
            if ( $("#"+sizeId).val() > 0) {
                maxSize = $("#"+sizeId).val();
            }
        }
    }
    catch(e) { }
    return maxSize;
}

/*
This block adds event listeners to file upload elements.  It looks for input
elements with class="LC_flUpload".

    <input type="file" class="LC_flUpload" />

It also looks for a hidden element with an id containing: "LC_free_space",
which contains the maximum allowable upload size (bytes).

    <input type="hidden" id="*LC_free_space*" value="$free_space" />

The * before LC_free_space and the * after LC_free_space are PREFIX and SUFFIX.

When the contents of the input element change, the function checkUploadSize()
checks if it is allowed based on size.
*/
$( document ).ready(function() {
    var upload_elements = $( ".LC_flUpload" );
    for (var i=0; i<upload_elements.length; i++) {
        if (getMaxSize(upload_elements[i],'LC_free_space')) {
            upload_elements[i].addEventListener( "change", function(){
                checkUploadSize(this,'LC_free_space');
            });
        }
    }
});
