# The LearningOnline Network with CAPA
# Page Handler
#
# $Id: lonpage.pm,v 1.111.2.16 2024/07/03 02:38:40 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
###




package Apache::lonpage;

use strict;
use Apache::Constants qw(:common :http);
use Apache::lonnet;
use Apache::loncommon();
use Apache::lonhtmlcommon;
use Apache::lonxml();
use Apache::lonlocal;
use Apache::lonmenu;
use Apache::lonhomework;
use Apache::lonenc();
use HTML::TokeParser;
use HTML::Entities();
use GDBM_File;
use Apache::lonsequence;
use lib '/home/httpd/lib/perl/';
use LONCAPA;
 

# -------------------------------------------------------------- Module Globals
my %hash;
my @rows;

# ------------------------------------------------------------------ Euclid gcd

sub euclid {
    my ($e,$f)=@_;
    my $a; my $b; my $r;
    if ($e>$f) { $b=$e; $r=$f; } else { $r=$e; $b=$f; }
    while ($r!=0) {
	$a=$b; $b=$r;
        $r=$a%$b;
    }
    return $b;
}

# ------------------------------------------------------------ Build page table

sub tracetable {
    my ($sofar,$rid,$beenhere)=@_;
    my $further=$sofar;
    my $randomout=0;
    unless ($env{'request.role.adv'}) {
        $randomout = $hash{'randomout_'.$rid};
    }
    unless ($beenhere=~/\&$rid\&/) {
        $beenhere.=$rid.'&';
        unless ($randomout) {
            if (defined($hash{'is_map_'.$rid})) {
                if ((defined($hash{'map_start_'.$hash{'src_'.$rid}})) &&
                    (defined($hash{'map_finish_'.$hash{'src_'.$rid}}))) {
                    my $frid=$hash{'map_finish_'.$hash{'src_'.$rid}};
	            $sofar=
                       &tracetable($sofar,$hash{'map_start_'.$hash{'src_'.$rid}},
                       '&'.$frid.$beenhere);
                    $sofar++;
                    if ($hash{'src_'.$frid}) {
                        my $brepriv=&Apache::lonnet::allowed('bre',$hash{'src_'.$frid});
                        if (($brepriv eq '2') || ($brepriv eq 'F')) {
                            if (defined($rows[$sofar])) {
                                $rows[$sofar].='&'.$frid;
                            } else {
                                $rows[$sofar]=$frid;
                            }
	                }
	            }
	        }
            } else {
                $sofar++;
                if ($hash{'src_'.$rid}) {
                    my ($mapid,$resid)=split(/\./,$rid);
                    my $symb = &Apache::lonnet::encode_symb($hash{'map_id_'.$mapid},$resid,$hash{'src_'.$rid});
                    my $brepriv=&Apache::lonnet::allowed('bre',$hash{'src_'.$rid},$symb);
                    if (($brepriv eq '2') || ($brepriv eq 'F')) {
                        if (defined($rows[$sofar])) {
                            $rows[$sofar].='&'.$rid;
                        } else {
                            $rows[$sofar]=$rid;
                        }
	            }
                }
            }
        }

        if (defined($hash{'to_'.$rid})) {
	    my $mincond=1;
            my $next='';
            foreach (split(/\,/,$hash{'to_'.$rid})) {
                my $thiscond=
      &Apache::lonnet::directcondval($hash{'condid_'.$hash{'undercond_'.$_}});
                if ($thiscond>=$mincond) {
		    if ($next) {
		        $next.=','.$_.':'.$thiscond;
                    } else {
                        $next=$_.':'.$thiscond;
		    }
                    if ($thiscond>$mincond) { $mincond=$thiscond; }
	        }
            }
            foreach (split(/\,/,$next)) {
                my ($linkid,$condval)=split(/\:/,$_);
                if ($condval>=$mincond) {
                    my $now=&tracetable($sofar,$hash{'goesto_'.$linkid},$beenhere);
                    if ($now>$further) { $further=$now; }
	        }
            }
        }
    }
    return $further;
}

# ================================================================ Main Handler

sub handler {
  my $r=shift;

# ------------------------------------------- Set document type for header only

  if ($r->header_only) {
       if ($env{'browser.mathml'}) {
           &Apache::loncommon::content_type($r,'text/xml');
       } else {
           &Apache::loncommon::content_type($r,'text/html'); 
       }
       $r->send_http_header;
       return OK;
   }
  
   &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},
                                          ['forceselect','launch']);
  my $number_of_columns = 1;
  my $requrl=$r->uri;  
  my $target = $env{'form.grade_target'};

# Short term solution: define target as 'tex_answer' when retrieving answers
# for resources in a .page when generating printouts.
# A better long-term fix would be to modify the way problem rendering, and 
# answer rendering are retrieved for individual resources when printing a .page,
# so rendered problem and answer are sequential for individual resources in 
# the .page
#
  if ($target eq 'answer') {
      if ($env{'form.answer_output_mode'} eq 'tex') {
          $target = 'tex_answer';
      }
  }
#  &Apache::lonnet::logthis("Got a target of $target");
  if ($target eq 'meta') {
      &Apache::loncommon::content_type($r,'text/html');
      $r->send_http_header;
      return OK;
  }
# ----------------------------------------------------------------- Tie db file
  if (($env{'request.course.fn'}) && (!$env{'form.forceselect'})) {
      my $fn=$env{'request.course.fn'};
      if (-e "$fn.db") {
          my %buttonshide;
          my $hostname = $r->hostname();
          my $lonhost = $r->dir_config('lonHostID');
          my $ip = &Apache::lonnet::get_host_ip($lonhost);
          if (tie(%hash,'GDBM_File',"$fn.db",&GDBM_READER(),0640)) {
# ------------------------------------------------------------------- Hash tied
              my $firstres=$hash{'map_start_'.$requrl};
              my $lastres=$hash{'map_finish_'.$requrl};
              if (($firstres) && ($lastres)) {
# ------------------------------------------------------------- Countdown Timer
                  my $now = time;
                  my ($pagefirstaccess,%hastimeleft,%countdowndisp);
                  my ($pagesymb,$courseid,$domain,$name)=&Apache::lonnet::whichuser();
                  if ($pagesymb && ($courseid ne '') && ($domain ne '') && ($name ne '')) {
                      my %times=&Apache::lonnet::get('firstaccesstimes',
                                                     [$courseid."\0".$pagesymb],
                                                     $domain,$name);
                      if ($times{$courseid."\0".$pagesymb} =~ /^\d+$/) {
                          $pagefirstaccess = $times{$courseid."\0".$pagesymb};
                      }
                  }
# ----------------------------------------------------------------- Render page

                  @rows=();

                  &tracetable(0,$firstres,'&');

# ------------------------------------------------------------ Add to symb list

                  my $i;
                  my %symbhash=();
                  for ($i=0;$i<=$#rows;$i++) {
		     if ($rows[$i]) {
                        my @colcont=split(/\&/,$rows[$i]);
                        foreach my $rid (@colcont) {
			    my ($mapid,$resid)=split(/\./,$rid);
			    $symbhash{$hash{'src_'.$rid}}=
				[$hash{'src_'.$rid},$resid];
		        }
		     }
		  }
                  &Apache::lonnet::symblist($requrl,%symbhash);

# ------------------------------------------------------------------ Page parms

                  my $j;
                  my $lcm=1;
                  my $contents=0;
                  my $nforms=0;
                  my $nuploads=0;
                  my $ntimers=0;
                  my $hasnumresp;
                  my %turninpaths;
                  my %multiresps;
                  my $turninparent;
                  
                  my %ssibody=();
                  my %ssibgcolor=();
                  my %ssitext=();
                  my %ssilink=();
                  my %ssivlink=();
                  my %ssialink=();
                  my %cssrefs=();
                  my %httpref=();
     
                  my %cellemb=();
                  my %cellexternal=();

                  my $allscript='';
                  my $allmeta='';

                  my $isxml=0;
                  my $xmlheader='';
                  my $xmlbody='';

# --------------------------------------------- Get SSI output, post parameters

                  for ($i=0;$i<=$#rows;$i++) {
		     if ($rows[$i]) {
		      $contents++;
                      my @colcont=split(/\&/,$rows[$i]);
                      $lcm*=($#colcont+1)/euclid($lcm,($#colcont+1));
                      foreach (@colcont) {
                          my $src=$hash{'src_'.$_};
                          my $plainsrc = $src;
                          my $anchor;
                          if ($hash{'ext_'.$_} eq 'true:') {
                              $cellexternal{$_}=($hash{'ext_'.$_} eq 'true:');
                              $src =~ s{^/ext/}{http://};
                              $src =~ s{http://https://}{https://};
                              if ($src =~ /(\#[^#]+)$/) {
                                  $anchor = $1;
                                  $src =~ s/\#[^#]+$//;
                              }
                          }
                          my $unencsrc = $src;
                          my ($extension)=($src=~/\.(\w+)$/);
			  if ($hash{'encrypted_'.$_}) {
			      $src=&Apache::lonenc::encrypted($src);
			  }
                          my ($mapid,$resid)=split(/\./,$_);
                          my $symb=&Apache::lonnet::encode_symb($hash{'map_id_'.$mapid},$resid,$plainsrc);
                          unless ($env{'request.role.adv'}) {
                              $buttonshide{$symb} = &Apache::lonnet::EXT("resource.0.buttonshide",$symb);
                          }
                          $cellemb{$_}=
			      &Apache::loncommon::fileembstyle($extension);
                          if ($cellexternal{$_}) {
                              if (($target eq 'tex') || ($target eq 'tex_answer')) {
                                  my $shown = $src.$anchor;
                                  if (($hash{'encrypted_'.$_}) && (!$env{'request.role.adv'})) {
                                      $shown = &mt('URL not shown (encrypted)');
                                  }
                                  my $title=&Apache::lonnet::gettitle($symb);
                                  $title = &Apache::lonxml::latex_special_symbols($title);
                                  $shown = &Apache::lonxml::latex_special_symbols($shown);
                                  $ssibody{$_} = ' \strut \\\\ \textit{'.$title.'} \strut \\\\ '.$shown.'\\\\';
                              } else {
                                  my $showsrc = $src;
                                  my ($is_pdf,$title,$linktext);
                                  if ($unencsrc =~ /\.pdf$/i) {
                                      $is_pdf = 1;
                                  }
                                  if (($hash{'encrypted_'.$_}) && ($symb)) {
                                      $title=&Apache::lonnet::gettitle(&Apache::lonenc::encrypted($symb));
                                  } else {
                                      $title=&Apache::lonnet::gettitle($symb);
                                  }
                                  if ($env{'browser.mobile'}) {
                                      if ($is_pdf) {
                                          $linktext = &mt('Link to PDF (for mobile devices)');
                                          $ssibody{$_} = &create_extlink($unencsrc,$anchor,$title,$linktext);
                                      } else {
                                          $linktext = &mt('Link to resource');
                                          $ssibody{$_} = &create_extlink($unencsrc,$anchor,$title,$linktext);
                                      }
                                  } else {
                                      my $absolute = $env{'request.use_absolute'};
                                      my $uselink = &Apache::loncommon::is_nonframeable($unencsrc,$absolute,$hostname,$ip);
                                      if (($uselink) || (($ENV{'SERVER_PORT'} == 443) && ($unencsrc =~ m{^http://}))) {
                                          $linktext = &mt('Link to resource');
                                          $ssibody{$_} =  &create_extlink($unencsrc,$anchor,$title,$linktext);
                                      } else {
                                          if (($hash{'encrypted_'.$_}) && ($symb) && (!$env{'request.role.adv'})) {
                                              $showsrc .= '?symb='.&Apache::lonenc::encrypted($symb);
                                          } elsif ($anchor) {
                                              $showsrc .= $anchor;
                                          }
                                          $ssibody{$_} = <<ENDEXT;
<iframe src="$showsrc" width="100%" height="300px">No iframe support!</iframe>
ENDEXT
                                      }
                                  }
                              }
                          } elsif ($cellemb{$_} eq 'ssi') {
# --------------------------------------------------------- This is an SSI cell
			      my $prefix='p_'.$_.'_';
                              my $idprefix='p_'.join('_',($mapid,$resid,''));
                              my %posthash=('request.prefix' => $prefix,
					    'LONCAPA_INTERNAL_no_discussion' => 'true',
					    'symb' => $symb);
			      if (($env{'form.grade_target'} eq 'tex') ||
                                 ($env{'form.answer_output_mode'} eq 'tex')) {
				  $posthash{'grade_target'}=$env{'form.grade_target'};
				  $posthash{'textwidth'}=$env{'form.textwidth'};
				  $posthash{'problem_split'}=$env{'form.problem_split'};
				  $posthash{'latex_type'}=$env{'form.latex_type'};
				  $posthash{'rndseed'}=$env{'form.rndseed'};
                                  $posthash{'answer_output_mode'} = $env{'form.answer_output_mode'};
			      }
			      my $submitted=$env{'form.all_submit_pressed'};
			      if (!$submitted) {
				  foreach my $key (keys(%env)) {
				      if ($key=~/^\Qform.$prefix\Esubmit_(.+)_pressed$/) {
                                          if ($env{$key}) {
                                              $submitted=1;
                                              last;
                                          }
				      }
                                  }
			      }
                              if ($submitted) {
				  foreach my $key (keys(%env)) {
				      if ($key=~/^\Qform.$prefix\E/) {
					  my $name=$key;
					  $name=~s/^\Qform.$prefix\E//;
					  $posthash{$name}=$env{$key};
                                      }
				  }
				  if ($env{'form.all_submit_pressed'}) {
				      $posthash{'all_submit'}='yes';
				  }
			      } elsif ($env{'form.'.$prefix.'markaccess'} eq 'yes') {
		                  $posthash{'markaccess'} = $env{'form.'.$prefix.'markaccess'};
		              }
                              if ($env{'environment.remote'} eq 'on') {
                                  $posthash{'inhibitmenu'} = 'yes';
                              }
                              my $output=Apache::lonnet::ssi($src,%posthash);
			      $output=~s|//(\s*<!--)? BEGIN LON-CAPA Internal.+?// END LON-CAPA Internal\s*(-->)?\s||gs;
                              if (($target eq 'tex') || ($target eq 'tex_answer')) {
				  $output =~ s/^([^&]+)\\begin\{document}//;
				  $output =~ s/\\end\{document}//;
#				  $output = '\parbox{\minipagewidth}{ '.$output.' }';
                                  #some additional cleanup necessary for LateX (due to limitations of table environment 
				  $output =~ s/(\\vskip\s*\d+mm)\s*(\\\\)+/$1/g;
			      }
                              my $matheditor;
                              if ($output =~ /\Qjavascript:LC_mathedit_HWVAL_\E/) {
                                  $matheditor = 'dragmath';
                              } elsif ($output =~ /LCmathField/) {
                                  $matheditor = 'lcmath';
                              }
                              my $parser=HTML::TokeParser->new(\$output);
                              my $token;
                              my $thisdir=$src;
                              my $bodydef=0;
                              my $thisxml=0;
                              my @rlinks=();
                              my @css_hrefs=();
                              if ($output=~/\?xml/) {
                                 $isxml=1;
                                 $thisxml=1;
                                 $output=~
         /((?:\<(?:\?xml|\!DOC|html)[^\>]*(?:\>|\>\]\>)\s*)+)\<body[^\>]*\>/si;
                                 $xmlheader=$1;
			      }
                              while ($token=$parser->get_token) {
				if ($token->[0] eq 'S') {
                                  if ($token->[1] eq 'a') {
				      if ($token->[2]->{'href'}) {
                                         $rlinks[$#rlinks+1]=
					     $token->[2]->{'href'};
				      }
				  } elsif ($token->[1] eq 'img') {
                                         $rlinks[$#rlinks+1]=
					     $token->[2]->{'src'};
				  } elsif ($token->[1] eq 'embed') {
                                         $rlinks[$#rlinks+1]=
					     $token->[2]->{'src'};
				  } elsif ($token->[1] eq 'base') {
				      $thisdir=$token->[2]->{'href'};
				  } elsif ($token->[1] eq 'body') {
				      $bodydef=1;
                                      $ssibgcolor{$_}=$token->[2]->{'bgcolor'};
                                      $ssitext{$_}=$token->[2]->{'text'};
                                      $ssilink{$_}=$token->[2]->{'link'};
                                      $ssivlink{$_}=$token->[2]->{'vlink'};
                                      $ssialink{$_}=$token->[2]->{'alink'};
                                      if ($thisxml) {
					  $xmlbody=$token->[4];
                                      }
                                  } elsif ($token->[1] eq 'meta') {
				    if ($token->[4] !~ m:/>$:) {
				      $allmeta.="\n".$token->[4].'</meta>';
				    } else {
				      $allmeta.="\n".$token->[4];
				    }
                                  } elsif (($token->[1] eq 'script') &&
                                           ($bodydef==0)) {
				      $allscript.="\n\n"
                                                .$parser->get_text('/script');
                                  } elsif (($token->[1] eq 'link') &&
                                           ($bodydef==0)) {
                                      if (($token->[2]->{'href'} !~ m{^/adm/}) &&
                                          ($token->[2]->{'rel'} eq 'stylesheet')) {
                                              $css_hrefs[$#css_hrefs+1]=
                                                  $token->[2]->{'href'};

                                      }
                                  }
			        }
			      }
                              if ($output=~/\<body[^\>]*\>(.*)/si) {
                                 $output=$1; 
                              }
                              $output=~s/\<\/body\>.*//si;
                              if ($output=~/\<form/si) {
                                  my $hastimer; 
				  $nforms++;
                                  $output=~s/\<form[^\>]*\>//gsi;
                                  $output=~s/\<\/form[^\>]*\>//gsi;
                                  if ($output=~/\<input[^\>]+name\s*=\s*[\'\"]*HWFILE/) {
                                      $nuploads++;
                                  }
                                  if ($output=~/\<input[^\>]+name\s*=\s*[\'\"]*accessbutton/) {
                                      $ntimers++;
                                      $hastimer = 1;
                                  }
                                  unless ($hasnumresp) {
                                      if ($output=~/\<input[^\>]+class\s*=\s*['"]*[^'">]*LC_numresponse_text\W/) {
                                          $hasnumresp = 1;
                                      }
                                  }
                                  $output=~
				      s/\<((?:input|select|button|textarea)[^\>]+)name\s*\=\s*[\'\"]*([^\'\"]+)[\'\"]*([^\>]*)\>/\<$1 name="$prefix$2" $3\>/gsi;
                                  $output=~
                                      s/\<((?:input|select|button|textarea)[^\>]+)id\s*\=\s*[\'\"]*([^\'\"]+)[\'\"]*([^\>]*)\>/\<$1 id="$idprefix$2" $3\>/gsi;
                                  $output=~
                                      s/(\Qthis.form.elements['\E)(HW(?:VAL|CHK)_[^']+\'\]\.(?:value=\'|checked))/$1$prefix$2/gsi;
                                  if ($hastimer) {
                                      $output=~
                                          s/\<(input[^\>]+name=\Q"$prefix\Eaccessbutton"[^\>]+)(?:\Qdocument.markaccess.submit();\E)([^\>]*)\>/\<$1pageTimer(this.form,'$prefix')$2\>/gsi;
                                      $output=~  s/\<(input[^\>]+name=\Q"$prefix\Emarkaccess"[^\>]+value=["'])(?:yes)(['"][^\>]*)\>/\<$1$2\>/gsi;
                                  }
                                  if ($matheditor eq 'dragmath') {
                                      $output=~
                                          s/(\Qjavascript:LC_mathedit_\E)(HWVAL_)([^'"]+?)(\(['"]*)(\QHWVAL_\E\3['"]\)\;void\(0\)\;)/$1$idprefix$2$3$4$idprefix$5/g;
                                      $output=~
                                          s/(function\s+LC_mathedit_)(HWVAL_)([^'"]+?)(\s+\(LCtextline\))/$1$idprefix$2$3$4/g;
                                  } elsif ($matheditor eq 'lcmath') {
                                      $output=~
                                          s/(var\s+LCmathField\s+=\s+document\.getElementById\(['"])([^'"]+?)(['"]\)\;)/$1$idprefix$2$3/g;
                                  }
                                  $output=~
                                      s/(\Q<div id="msg_\E)(\Qsubmit_\E)([^"]*)(\Q" style="display:none">\E)/<input type="hidden" name="$prefix$2$3_pressed" id="$idprefix$2$3_pressed" value="" \/>$1$idprefix$2$3$4/g;
                                  $output=~
                                      s/(\Q<td class="LC_status_\E)(\Qsubmit_\E)([^\"]*)(\s*[^\"]*"\>)/$1$idprefix$2$3$4/g;
                                  if ($nuploads) {
                                       ($turninpaths{$prefix},$multiresps{$prefix}) = 
                                           &Apache::loncommon::get_turnedin_filepath($symb,$env{'user.name'},$env{'user.domain'});
                                       if ($turninparent eq '') {
                                           $turninparent = $turninpaths{$prefix};
                                           $turninparent =~ s{(/[^/]+)$}{}; 
                                       }
                                  }
                                  $output=~
                                      s/\<((?:input|select)[^\>]+\Qjavascript:setSubmittedPart\E)\(\s*[\'\"]([^\'\"]+)[\'\"]*\s*\)/\<$1('$2','$prefix')/gsi;
                                  $output=~
                                      s/\<(input[^\>]+\Qonfocus=\"javascript:disableAutoComplete\E)\(\'([^\']+)\'\)(;\")/\<$1('$idprefix$2')$3/gsi;
                                  unless ($hastimer) {
                                      if ($plainsrc =~ /$LONCAPA::assess_re/) {
                                          %Apache::lonhomework::history =
                                              &Apache::lonnet::restore($symb,$courseid,$domain,$name);
                                          my $type = 'problem';
                                          if ($extension eq 'task') {
                                              $type = 'Task';
                                          }
                                          my ($status,$accessmsg,$slot_name,$slot) =
                                              &Apache::lonhomework::check_slot_access('0',$type,$symb);
                                          undef(%Apache::lonhomework::history);
                                          my $probstatus = &Apache::lonnet::EXT("resource.0.problemstatus",$symb);
                                          if (($status eq 'CAN_ANSWER') || (($status eq 'CANNOT_ANSWER') && 
                                              (($probstatus eq 'no') || ($probstatus eq 'no_feedback_ever')))) {
                                              my ($slothastime,$timerhastime);
                                              if ($slot_name ne '') {
                                                  if (ref($slot) eq 'HASH') {
                                                      if (($slot->{'starttime'} < $now) &&
                                                          ($slot->{'endtime'} > $now)) {
                                                          $slothastime = $now - $slot->{'endtime'};
                                                      }
                                                  }
                                              }
                                              my $duedate = &Apache::lonnet::EXT("resource.0.duedate",$symb);
                                              my @interval=&Apache::lonnet::EXT("resource.0.interval",$symb);
                                              if (@interval > 1) {
                                                  my $first_access=&Apache::lonnet::get_first_access($interval[1],$symb);
                                                  if ($first_access > 0) {
                                                      my $timeremains = $first_access+$interval[0] - $now;
                                                      if ($timeremains > 0) {
                                                          $timerhastime = $timeremains;
                                                      }
                                                  }
                                              }
                                              if (($duedate && $duedate > $now) ||
                                                  (!$duedate && $timerhastime > 0) ||
                                                  ($slot_name ne '' && $slothastime)) {
                                                  if ((@interval > 1 && $timerhastime) ||
                                                      ($type eq 'Task' && $slothastime)) {
                                                      $countdowndisp{$symb} = 'inline';
                                                      if ((@interval > 1) && ($timerhastime)) {
                                                          $hastimeleft{$symb} = $timerhastime;
                                                      } else {
                                                          $hastimeleft{$symb} = $slothastime;
                                                      }
                                                  } else {
                                                      $hastimeleft{$symb} = $duedate - $now;
                                                      $countdowndisp{$symb} = 'none';
                                                  }
                                              }
                                          }
                                      }
                                  }
                              }
                              $thisdir=~s/\/[^\/]*$//;
			      foreach (@rlinks) {
				  unless (($_=~/^https?\:\/\//i) ||
					  ($_=~/^\//) ||
					  ($_=~/^javascript:/i) ||
					  ($_=~/^mailto:/i) ||
					  ($_=~/^\#/)) {
				      my $newlocation=
				    &Apache::lonnet::hreflocation($thisdir,$_);
                     $output=~s/(\"|\'|\=\s*)$_(\"|\'|\s|\>)/$1$newlocation$2/;
				  }
			      }
                              foreach my $css_href (@css_hrefs) {
                                  next if ($css_href eq '');
                                  unless ($css_href =~ m{https?://}) {
                                      my $proburl = &Apache::lonnet::clutter($plainsrc);
                                      unless ($css_href =~ m{^/}) {
                                          my $probdir = $proburl;
                                          $probdir=~s/\/[^\/]*$//;
                                          $css_href = &Apache::lonnet::hreflocation($probdir,$css_href);
                                      }
                                      if ($css_href =~ m{^/(res|uploaded)/}) {
                                          unless (($env{'httpref.'.$css_href}) ||
                                                  ($httpref{'httpref.'.$css_href}) ||
                                                  (&Apache::lonnet::is_on_map($css_href))) {
                                              if ($env{'httpref.'.$proburl}) {
                                                  $proburl = $env{'httpref.'.$proburl};
                                              }
                                              $httpref{'httpref.'.$css_href} = $proburl;
                                          }
                                      }
                                  }
                                  $cssrefs{$css_href} = 1;
                              }
# -------------------------------------------------- Deal with Applet codebases
  $output=~s/(\<applet[^\>]+)(codebase\=[^\S\>]+)*([^\>]*)\>/$1.($2?$2:' codebase="'.$thisdir.'"').$3.'>'/gei;
			      $ssibody{$_}=$output;
# ---------------------------------------------------------------- End SSI cell
                          }
                      }
                     }
                  }
                  unless ($contents) {
                      &Apache::loncommon::content_type($r,'text/html');
                      $r->send_http_header;
                      $r->print(&Apache::loncommon::start_page(undef,undef,
							       {'force_register' => 1}));
                      $r->print(&mt('This page is either empty or it only contains resources that are currently hidden').'. ');
                      $r->print('<br /><br />'.&mt('Please use the LON-CAPA navigation arrows to move to another item in the course').
				&Apache::loncommon::end_page());
                  } else {
# ------------------------------------------------------------------ Build page

# ---------------------------------------------------------------- Send headers
		      unless (($target eq 'tex') || ($target eq 'tex_answer')) {
			  if ($isxml) {
			      &Apache::loncommon::content_type($r,'text/xml');
			  } else {
			      &Apache::loncommon::content_type($r,'text/html');
			  }
			  $r->send_http_header;
# ------------------------------------------------------------------------ Head
			  if ($allscript) {
			      $allscript = 
				  "\n".'<script type="text/javascript">'."\n".
				  $allscript.
				  "\n</script>\n";
			  }
                          if (($nforms) && ($nuploads)) {
                              $allscript .= &Apache::lonhtmlcommon::file_submissionchk_js(\%turninpaths,\%multiresps).
                                            '<script type="text/javascript" '.
                                            'src="/res/adm/includes/file_upload.js"></script>';
                          }
                          if (($nforms) && (&Apache::lonhtmlcommon::htmlareabrowser())) {
                              my %textarea_args = (
                                  dragmath => 'math',
                              );
                              $allscript .= &Apache::lonhtmlcommon::htmlareaselectactive(\%textarea_args);
                          }
                          if ($ntimers) {
                              $allscript .= '<script type="text/javascript">'."\n".
                                            '// <![CDATA['."\n".
                                            'function pageTimer(form,prefix) {'."\n".
                                            "   form.elements[prefix+'markaccess'].value = 'yes';\n".
                                            "   form.submit();\n".
                                            '}'."\n".
                                            '// ]]>'.
                                            "\n</script>\n";
                          }
                          &Apache::lonhtmlcommon::clear_breadcrumb_tools();
                          if (keys(%hastimeleft)) {
                              my (%uniquetimes,%uniquedisplays);
                              foreach my $item (values(%hastimeleft)) {
                                  if (exists($uniquetimes{$item})) {
                                      $uniquetimes{$item} ++; 
                                  } else {
                                      $uniquetimes{$item} = 1;
                                  }
                              }
                              if (keys(%uniquetimes) == 1) {
                                  my (%uniquedisplays,%uniquedones,$currdisp);
                                  if (keys(%countdowndisp)) {
                                      foreach my $item (values(%countdowndisp)) {
                                          if (exists($uniquedisplays{$item})) {
                                              $uniquedisplays{$item} ++;
                                          } else {
                                              $uniquedisplays{$item} = 1;
                                          }
                                      }
                                      my @countdowndisplay = keys(%uniquedisplays);
                                      if (scalar(@countdowndisplay) == 1) {
                                          $currdisp = $countdowndisplay[0];
                                      }
                                  }
                                  &add_countdown_timer($currdisp);
                              }
                          }
                          my $pagebuttonshide;
                          if (keys(%buttonshide)) {
                              my %uniquebuttonhide;
                              foreach my $item (values(%buttonshide)) {
                                  if (exists($uniquebuttonhide{$item})) {
                                      $uniquebuttonhide{$item} ++;
                                  } else {
                                      $uniquebuttonhide{$item} = 1;
                                  }
                              }
                              if (keys(%uniquebuttonhide) == 1) {
                                  if (lc((keys(%uniquebuttonhide))[0]) eq 'yes') {
                                      $pagebuttonshide = 'yes';
                                  }
                              }
                          }
                          if (keys(%cssrefs)) {
                              my $links;
                              if (keys(%cssrefs)) {
                                  foreach my $css_href (keys(%cssrefs)) {
                                      next unless ($css_href =~ m{^(/res/|/uploaded/|https?://)});
                                      $links .= '<link rel="stylesheet" type="text/css" href="'.$css_href.'" />'."\n";
                                  }
                              }
                              if ($links) {
                                  if (keys(%httpref)) {
                                      &Apache::lonnet::appenv(\%httpref);
                                  }
                                  $allscript .= "\n$links";
                              }
                          }
# ------------------------------------------------------------------ Start body
			  $r->print(&Apache::loncommon::start_page(undef,$allscript,
								   {'force_register' => 1,
								    'bgcolor'        => '#ffffff',
								    'hide_buttons'   => $pagebuttonshide}));
# ------------------------------------------------------------------ Start form
			  if ($nforms) {
			      my $fmtag = '<form name="lonhomework" method="post"  enctype="multipart/form-data"';
                              if ($nuploads) {
                                  my $multi;
                                  if ($nuploads > 1) {
                                      $multi = 1;
                                  }
                                  $fmtag .= 'onsubmit="return file_submission_check(this,'."'$turninparent','$multi'".');"';
                              }
                              $fmtag .= ' action="'.
					&Apache::lonenc::check_encrypt($requrl)
					.'" id="LC_page">';
                              $r->print($fmtag);
			  }
		      } elsif (($target eq 'tex') || ($target eq 'tex_answer')) {
			  #  I think this is not needed as the header
			  # will be put in for each of the page parts
			  # by the londefdef.pm now that we are opening up
			  # the parts of a page.
			  #$r->print('\documentclass{article}
                          #       \newcommand{\keephidden}[1]{}           
                          #       \usepackage[dvips]{graphicx}
                          #       \usepackage{epsfig}
                          #       \usepackage{calc}
                          #       \usepackage{longtable}
                          #       \begin{document}');
		      }
# ----------------------------------------------------------------- Start table
		      if (($target eq 'tex') || ($target eq 'tex_answer')) {
#			 #  $r->print('\begin{longtable}INSERTTHEHEADOFLONGTABLE\endfirsthead\endhead ');
			  if ($number_of_columns le $lcm) {$number_of_columns=$lcm;};
		      } else {
			  $r->print('<table width="100%" cols="'.$lcm.'" border="0">');
		      }
# generate rows
                      for ($i=0;$i<=$#rows;$i++) {
			if ($rows[$i]) {
			    unless (($target eq 'tex') || ($target eq 'tex_answer')) {
				$r->print("\n<tr>");
			    }
                          my @colcont=split(/\&/,$rows[$i]);
                          my $avespan=$lcm/($#colcont+1);
                          for ($j=0;$j<=$#colcont;$j++) {
                              my $rid=$colcont[$j];
			      my $metainfo =&get_buttons(\%hash,$rid,\%buttonshide,$hostname).'<br />';
			    unless (($target eq 'tex') || ($target eq 'tex_answer')) {
				$r->print('<td colspan="'.$avespan.'"');
			    }
                              if (($cellemb{$rid} eq 'ssi') || ($cellexternal{$rid})) {
				  unless (($target eq 'tex') || ($target eq 'tex_answer')) {
				      if ($ssibgcolor{$rid}) {
					  $r->print(' bgcolor="'.
						    $ssibgcolor{$rid}.'"');
				      }
				      $r->print('>'.$metainfo.'<font');
		    
				      if ($ssitext{$rid}) {
					  $r->print(' text="'.$ssitext{$rid}.'"');
				      }
				      if ($ssilink{$rid}) {
					  $r->print(' link="'.$ssilink{$rid}.'"');
				      }
				      if ($ssitext{$rid}) {
					  $r->print(' vlink="'.$ssivlink{$rid}.'"');
				      }
				      if ($ssialink{$rid}) {
					  $r->print(' alink="'.$ssialink{$rid}.'"');
				      }             
				      $r->print('>');
				  }
                                  $r->print($ssibody{$rid});
				  unless (($target eq 'tex') || ($target eq 'tex_answer')) {
				      $r->print('</font>');
                                  }
                                  if ($env{'course.'.
                                      $env{'request.course.id'}.
                                      '.pageseparators'} eq 'yes') {
                                      unless (($target eq 'tex') || ($target eq 'tex_answer')) {
                                          $r->print('<hr />');
                                      } 
				  }
			      } elsif ($cellemb{$rid} eq 'img') {
                                  $r->print('>'.$metainfo.'<img src="'.
                                    $hash{'src_'.$rid}.'" />');
			      } elsif ($cellemb{$rid} eq 'emb') {
                                  $r->print('>'.$metainfo.'<embed src="'.
                                    $hash{'src_'.$rid}.'"></embed>');
                              } elsif (&Apache::lonnet::declutter($hash{'src_'.$rid}) !~/\.(sequence|page)$/) {
                                  $r->print($metainfo.'<b>'.$hash{'title_'.$rid}.'</b><br />');
                                  unless ($cellemb{$rid} eq 'wrp') {
                                      $r->print(&mt('It is recommended that you use an up-to-date virus scanner before handling this file.'));
                                  }
                                  $r->print('</p><p><table>'.
                                            &Apache::londocs::entryline(0,
                                                                        &mt("Click to download or use your browser's Save Link function"),
                                                                        '/'.&Apache::lonnet::declutter($hash{'src_'.$rid})).
                                                                        '</table></p><br />');
                              }
			      unless (($target eq 'tex') || ($target eq 'tex_answer')) {
				  $r->print('</td>');
			      } else {
#                                  for (my $incol=1;$incol<=$avespan;$incol++) {
#				      $r->print(' & ');
#				  }
			      }
                          }
			      unless (($target eq 'tex') || ($target eq 'tex_answer')) {
				  $r->print('</tr>');
			      } else {
#				  $r->print('REMOVETHEHEADOFLONGTABLE\\\\');
			      }
		        }
                      }
		      unless (($target eq 'tex') || ($target eq 'tex_answer')) {
			  $r->print("\n</table>");
		      } else {
#			  $r->print('\end{longtable}\strut');
		      }
# ---------------------------------------------------------------- Submit, etc.
                      if ($nforms) {
                          my $class;
                          if ($nforms > 1) {
                              $class = ' class="LC_hwk_submit"';
                              if ($ntimers) {
                                  $nforms = 1;
                                  $class = '';
                              }
                          }
                          $r->print(
	                  '<input name="all_submit" value="'.&mt('Submit All').'" type="'.
			  (($nforms>1)?'submit':'hidden').'"'.$class.' id="all_submit" />'.
                          '<input type="hidden" name="all_submit_pressed" '.
                          'id="all_submit_pressed" value="" />'.
                          '<div id="msg_all_submit" style="display:none">'.
                          &mt('Processing your submission ...').'</div></form>');
                      }
		      unless (($target eq 'tex') || ($target eq 'tex_answer')) {
                          my $args = {'discussion' => 1};
                          if ($hasnumresp) {
                              $args->{'dashjs'} = 1;
                          }
			  $r->print(&Apache::loncommon::end_page($args));
		      } else {
			  $r->print('\end{document}'.$number_of_columns);
		      }
		      &Apache::lonnet::symblist($requrl,%symbhash);
		      my ($map,$id,$url)=&Apache::lonnet::decode_symb(&Apache::lonnet::symbread());
		      &Apache::lonnet::symblist($map,'last_known'=>[$url,$id]);
# -------------------------------------------------------------------- End page
                  }                  
# ------------------------------------------------------------- End render page
              } else {
                  if ($hash{'map_type_'.$hash{'map_pc_'.$requrl}} eq 'none') {
                      &Apache::loncommon::content_type($r,'text/html');
                      $r->send_http_header;
                      $r->print(&Apache::loncommon::start_page(undef,undef,
                                                               {'force_register' => 1,}));
                      my $crstype = &Apache::loncommon::course_type();
                      if (&Apache::lonnet::allowed('mdc',$env{'request.course.id'})) {
                          $r->print('<span class="LC_warning">'.&mt('Missing composite page file.').'</span><br />'.
                                    &mt("You may want to use the $crstype Editor to remove this item."));
                      } else {
                          if ($crstype eq 'Placement') {
                              $r->print('<span class="LC_warning">'.&mt('Missing page').'</span>');
                          } else {
                              $r->print('<span class="LC_info">'.
                                        &mt('This resource was unavailable when your '.lc($crstype).' session was loaded').'<br />'.
                                        &mt("Please use 'Contents' to list items available in the $crstype.").'</span>');
                          }
                      }
                      $r->print(&Apache::loncommon::end_page());
                  } else {
                      &Apache::loncommon::content_type($r,'text/html');
                      $r->send_http_header;
                      &Apache::lonsequence::viewmap($r,$requrl);
                  }
              }
# ------------------------------------------------------------------ Untie hash
              unless (untie(%hash)) {
                   &Apache::lonnet::logthis("<font color=blue>WARNING: ".
                       "Could not untie coursemap $fn (browse).</font>"); 
              }
# -------------------------------------------------------------------- All done
	      return OK;
# ----------------------------------------------- Errors, hash could no be tied
          }
      } 
  }
  &Apache::loncommon::content_type($r,'text/html');
  $r->send_http_header;
  &Apache::lonsequence::viewmap($r,$requrl);
  return OK; 
}

sub get_buttons {
    my ($hash,$rid,$buttonshide,$hostname) = @_;

    my ($mapid,$resid)=split(/\./,$rid);
    my $symb=&Apache::lonnet::encode_symb($hash->{'map_id_'.$mapid},
					  $resid,
					  $hash->{'src_'.$rid});
    my ($aname,$shownsymb);
    if (($hash->{'encrypted_'.$rid}) && (!$env{'request.role.adv'})) {
        $aname = 'LC_'.$rid;
        $shownsymb = &Apache::lonenc::encrypted($symb);
    } else {
        $shownsymb = $symb;
        my $dispsymb = $symb;
        if ($symb =~ /\#([^\#]+)$/) {
            my $escan = &escape('#');
            $dispsymb =~ s/#([^\#]+)$/$escan$1/;
        }
        $aname = &escape($dispsymb);
    }
    my $metainfo = '<a name="'.$aname.'"></a>';
    unless ($env{'request.role.adv'}) {
        if ($buttonshide->{$symb} eq 'yes') {
            return $metainfo;
        }
    }
    my $crs_sec = $env{'request.course.id'} . (($env{'request.course.sec'} ne '')
                                               ? "/$env{'request.course.sec'}"
                                               : '');
    my $esrc=&Apache::lonnet::declutter($hash->{'src_'.$rid});
    if ($hash->{'encrypted_'.$rid}) {
	$esrc=&Apache::lonenc::encrypted($esrc);
    }
    if ($hash->{'src_'.$rid} !~ m-^/uploaded/-
        && $hash->{'src_'.$rid} !~ m{^/ext/}
	&& !$env{'request.enc'}
	&& ($env{'request.role.adv'}
	    || !$hash->{'encrypted_'.$rid})) { 
	$metainfo .= '<a href="'.$hash->{'src_'.$rid}.'.meta'.'" '.
	    'target="LONcatInfo">'.
            '<img src="/res/adm/pages/catalog.png" class="LC_icon"'.
            ' alt="'.&mt('Show Metadata').'"'.
            ' title="'.&mt('Show Metadata').'" />'.
	    '</a>';
    }
    if (($hash->{'src_'.$rid} !~ m{^/uploaded/}) &&
        ($hash->{'src_'.$rid} !~ m{^/ext/})) {
        $metainfo .= '<a href="/adm/evaluate?postdata='.
	    &escape($esrc).
	    '" target="LONcatInfo">'.
            '<img src="/res/adm/pages/eval.png" class="LC_icon"'.
            ' alt="'.&mt('Provide my evaluation of this resource').'"'.
            ' title="'.&mt('Provide my evaluation of this resource').'" />'.
	    '</a>';
    }
    if (($hash->{'src_'.$rid}=~/$LONCAPA::assess_re/) &&
	($hash->{'src_'.$rid} !~ m-^/uploaded/-)) {

	if ((&Apache::lonnet::allowed('mgr',$crs_sec)) ||
            (&Apache::lonnet::allowed('vgr',$crs_sec))) {
	    $metainfo.=
		'<a href="/adm/grades?symb='.&escape($symb).
#               '&command=submission" target="LONcatInfo">'.
		'&command=submission">'.
                '<img src="/adm/lonMisc/subm_button.png" class="LC_icon"'.
                ' alt="'.&mt('View Submissions for a Student or a Group of Students').'"'.
                ' title="'.&mt('View Submissions for a Student or a Group of Students').'" />'.
		'</a>';
        }
        if (&Apache::lonnet::allowed('mgr',$crs_sec)) {
            $metainfo.=
		'<a href="/adm/grades?symb='.&escape($symb).
#               '&command=gradingmenu" target="LONcatInfo">'.
		'&command=gradingmenu">'.
                '<img src="/res/adm/pages/pgrd.png" class="LC_icon"'.
                ' alt="'.&mt('Content Grades').'"'.
                ' title="'.&mt('Content Grades').'" />'.
		'</a>';
	}
	if ((&Apache::lonnet::allowed('opa',$crs_sec)) ||
            (&Apache::lonnet::allowed('vpa',$crs_sec))) {
	    $metainfo.=
		'<a href="/adm/parmset?symb='.&escape($symb).
#               '" target="LONcatInfo">'.
		'" >'.
                '<img src="/adm/lonMisc/pprm_button.png" class="LC_icon"'.
                ' alt="'.&mt('Content Settings').'"'.
                ' title="'.&mt('Content Settings').'" />'.
		'</a>';
	}
    }
    if ($env{'request.course.id'}) {
        my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
        my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
        my $file=&Apache::lonnet::declutter($hash->{'src_'.$rid});
        my $editbutton = '';
        if (&Apache::lonnet::allowed('mdc',$env{'request.course.id'})) {
            my ($cfile,$home,$switchserver,$forceedit,$forceview) =
                &Apache::lonnet::can_edit_resource($file,$cnum,$cdom,$hash->{'src_'.$rid},$symb);
            if ($cfile ne '') {
                my $jscall = &Apache::lonhtmlcommon::jump_to_editres($cfile,$home,$switchserver,
                                                                     $forceedit,1,$symb,$shownsymb,
                                                                     undef,&escape($env{'form.title'}),
                                                                     $hostname);
                if ($jscall) {
                    $editbutton = 1;
                    my $icon = 'pcstr.png';
                    my $label = &mt('Edit');
                    my $title = &mt('Edit this resource');
                    my $pic = '<img src="'.&Apache::loncommon::lonhttpdurl('/res/adm/pages/'.$icon).'"'.
                              ' class="LC_icon" alt="'.$label.'" title="'.$title.'" />';
                    $metainfo .= '&nbsp;<a href="javascript:'.$jscall.';">'.$pic.'</a>';
                }
            }
        }
        if ((!$editbutton) && ($file=~/$LONCAPA::assess_re/)) {
            my $url = &Apache::lonnet::clutter($file);
            my $viewsrcbutton;
            if ((&Apache::lonnet::allowed('cre','/')) &&
                (&Apache::lonnet::metadata($url,'sourceavail') eq 'open')) {
                $viewsrcbutton = 1;
            } elsif (&Apache::lonnet::allowed('vxc',$crs_sec)) {
                if ($url =~ m{^\Q/res/$cdom/\E($LONCAPA::match_username)/}) {
                    my $auname = $1;
                    if (($env{'request.course.adhocsrcaccess'} ne '') &&
                        (grep(/^\Q$auname\E$/,split(/,/,$env{'request.course.adhocsrcaccess'})))) {
                        $viewsrcbutton = 1;
                    } elsif ((&Apache::lonnet::metadata($url,'sourceavail') eq 'open') &&
                             (&Apache::lonnet::allowed('bre',$crs_sec))) {
                        $viewsrcbutton = 1;
                    }
                }
            }
            if ($viewsrcbutton) {
                my $icon = 'pcstr.png';
                my $label = &mt('View Source');
                my $title = &mt('View source code');
                my $jsrid = $rid;
                $jsrid =~ s/\./_/g;
                my $showurl = &escape(&Apache::lonenc::check_encrypt($url));
                my $pic = '<img src="'.&Apache::loncommon::lonhttpdurl('/res/adm/pages/'.$icon).'"'.
                          ' class="LC_icon" alt="'.$label.'" title="'.$title.'" />';
                $metainfo .= '&nbsp;<a href="javascript:open_source_'.$jsrid.'();">'.$pic.'</a>'."\n".
                             '<script type="text/javascript">'."\n".
                             "function open_source_$jsrid() {\n".
                             "  sourcewin=window.open('/adm/source?inhibitmenu=yes&viewonly=1&filename=$showurl','LONsource',".
                             "'height=500,width=600,resizable=yes,location=no,menubar=no,toolbar=no,scrollbars=yes');\n".
                             "}\n".
                             "</script>\n";
            }
        }
    }
    return $metainfo;
}

sub add_countdown_timer {
    my ($currdisp) = @_;
    my ($collapse,$expand,$alttxt,$title);
    if ($currdisp eq 'inline') {
        $collapse = '&#9658;&nbsp;';
    } else {
        $expand = '&#9668;&nbsp;';
    }
    unless ($env{'environment.icons'} eq 'iconsonly') {
        $alttxt = &mt('Timer');
        $title = $alttxt.'&nbsp;';
    }
    my $desc = &mt('Countdown to due date/time');
    my $output = <<END;
<a href="javascript:toggleCountdown();" class="LC_menubuttons_link">
<span id="ddcountcollapse" class="LC_menubuttons_inline_text">
$collapse
</span></a>
<span id="duedatecountdown" class="LC_menubuttons_inline_text" style="display: $currdisp;"></span>
<a href="javascript:toggleCountdown();" class="LC_menubuttons_link">
<span id="ddcountexpand" class="LC_menubuttons_inline_text" >$expand</span>
<img src="/res/adm/pages/timer.png" title="$desc" class="LC_icon" alt="$alttxt" /><span class="LC_menubuttons_inline_text">$title</span></a>
END
    &Apache::lonhtmlcommon::add_breadcrumb_tool('tools',$output);
    return;
}

sub create_extlink {
    my ($url,$anchor,$title,$linktext) = @_;
    my $shownlink;
    unless ($title eq '') {
        $shownlink = '<span style="font-weight:bold;">'.$title.'</span><br />';
    }
    my $dest = &HTML::Entities::encode($url.$anchor,'&<>"');
    $shownlink .= '<a href="'.$dest.'">'.$linktext.'</a>';
    return $shownlink;
}

1;
__END__


=head1 NAME

Apache::lonpage - Page Handler

=head1 SYNOPSIS

Invoked by /etc/httpd/conf/srm.conf:

 <LocationMatch "^/res/.*\.page$>
 SetHandler perl-script
 PerlHandler Apache::lonpage
 </LocationMatch>

=head1 INTRODUCTION

This module renders a .page resource.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 HANDLER SUBROUTINE

This routine is called by Apache and mod_perl.

=over 4

=item *

set document type for header only

=item *

tie db file

=item *

render page

=item *

add to symb list

=item *

page parms

=item *

Get SSI output, post parameters

=item *

SSI cell rendering

=item *

Deal with Applet codebases

=item *

Build page

=item *

send headers

=item *

start body

=item *

start form

=item *

start table

=item *

submit element, etc, render page, untie hash

=back

=head1 OTHER SUBROUTINES

=over 4

=item *

euclid() : Euclid's method for determining the greatest common denominator.

=item *

tracetable() : Build page table.

=back

=cut


