# The LearningOnline Network
# Cookie Based Access Handler
#
# $Id: lonacc.pm,v 1.103 2006/12/11 14:06:04 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
###

package Apache::lonacc;

use strict;
use Apache::Constants qw(:common :http :methods);
use Apache::File;
use Apache::lonnet;
use Apache::loncommon();
use Apache::lonlocal;
use Apache::restrictedaccess();
use Apache::blockedaccess(); 
use CGI::Cookie();
use Fcntl qw(:flock);
use LONCAPA;

sub cleanup {
    my ($r)=@_;
    if (! $r->is_initial_req()) { return DECLINED; }
    &Apache::lonnet::save_cache();
    &Apache::lontexconvert::jsMath_reset();
    return OK;
}

sub goodbye {
    my ($r)=@_;
    &Apache::lonnet::goodbye();
    return DONE;
}

###############################################

sub get_posted_cgi {
    my ($r) = @_;

    my $buffer;
    if ($r->header_in('Content-length')) {
	$r->read($buffer,$r->header_in('Content-length'),0);
    }
    unless ($buffer=~/^(\-+\w+)\s+Content\-Disposition\:\s*form\-data/si) {
	my @pairs=split(/&/,$buffer);
	my $pair;
	foreach $pair (@pairs) {
	    my ($name,$value) = split(/=/,$pair);
	    $value =~ tr/+/ /;
	    $value =~ s/%([a-fA-F0-9][a-fA-F0-9])/pack("C",hex($1))/eg;
	    $name  =~ tr/+/ /;
	    $name  =~ s/%([a-fA-F0-9][a-fA-F0-9])/pack("C",hex($1))/eg;
	    &Apache::loncommon::add_to_env("form.$name",$value);
	}
    } else {
	my $contentsep=$1;
	my @lines = split (/\n/,$buffer);
	my $name='';
	my $value='';
	my $fname='';
	my $fmime='';
	my $i;
	for ($i=0;$i<=$#lines;$i++) {
	    if ($lines[$i]=~/^$contentsep/) {
		if ($name) {
		    chomp($value);
		    if ($fname) {
			$env{"form.$name.filename"}=$fname;
			$env{"form.$name.mimetype"}=$fmime;
		    } else {
			$value=~s/\s+$//s;
		    }
		    &Apache::loncommon::add_to_env("form.$name",$value);
		}
		if ($i<$#lines) {
		    $i++;
		    $lines[$i]=~
		/Content\-Disposition\:\s*form\-data\;\s*name\=\"([^\"]+)\"/i;
		    $name=$1;
		    $value='';
		    if ($lines[$i]=~/filename\=\"([^\"]+)\"/i) {
			$fname=$1;
			if 
                            ($lines[$i+1]=~/Content\-Type\:\s*([\w\-\/]+)/i) {
				$fmime=$1;
				$i++;
			    } else {
				$fmime='';
			    }
		    } else {
			$fname='';
			$fmime='';
		    }
		    $i++;
		}
	    } else {
		$value.=$lines[$i]."\n";
	    }
	}
    }
#
# Digested POSTed values
#
# Remember the way this was originally done (GET or POST)
#
    $env{'request.method'}=$ENV{'REQUEST_METHOD'};
#
# There may also be stuff in the query string
# Tell subsequent handlers that this was GET, not POST, so they can access query string.
# Also, unset POSTed content length to cover all tracks.
#

    $r->method_number(M_GET);

    $r->method('GET');
    $r->headers_in->unset('Content-length');
}

# handle the case of the single sign on user, at this point $r->user 
# will be set and valid now need to find the loncapa user info and possibly
# balance them
# returns OK if it was a SSO and user was handled
#         undef if not SSO or no means to hanle the user
sub sso_login {
    my ($r,$lonid,$handle) = @_;

    my $lonidsdir=$r->dir_config('lonIDsDir');
    if (!($r->user 
	  && (!defined($env{'user.name'}) && !defined($env{'user.domain'}))
	  && (!$lonid || !-e "$lonidsdir/$handle.id" || $handle eq ''))) {
	# not an SSO case or already logged in
	return undef;
    }

    my ($user) = ($r->user =~ m/([a-zA-Z0-9_\-@.]*)/);

    my $domain = $r->dir_config('lonDefDomain');
    my $home=&Apache::lonnet::homeserver($user,$domain);
    if ($home !~ /(con_lost|no_host|no_such_host)/) {
	if ($r->dir_config("lonBalancer") eq 'yes') {
	    # login but immeaditly go to switch server to find us a new 
	    # machine
	    &Apache::lonauth::success($r,$user,$domain,$home,'noredirect');
	    $r->internal_redirect('/adm/switchserver');
	    $r->set_handlers('PerlHandler'=> undef);
	} else {
	    # need to login them in, so generate the need data that
	    # migrate expects to do login
	    my %info=('ip'        => $r->connection->remote_ip(),
		      'domain'    => $domain,
		      'username'  => $user,
		      'server'    => $r->dir_config('lonHostID'),
		      'sso.login' => 1
		      );
	    my $token = 
		&Apache::lonnet::tmpput(\%info,
					$r->dir_config('lonHostID'));
	    $env{'form.token'} = $token;
	    $r->internal_redirect('/adm/migrateuser');
	    $r->set_handlers('PerlHandler'=> undef);
	}
	return OK;
    } elsif (defined($r->dir_config('lonSSOUserUnknownRedirect'))) {
	$r->internal_redirect($r->dir_config('lonSSOUserUnknownRedirect'));
	$r->set_handlers('PerlHandler'=> undef);
	return OK;
    }
    return undef;
}

sub handler {
    my $r = shift;
    my $requrl=$r->uri;
    my %cookies=CGI::Cookie->parse($r->header_in('Cookie'));
    my $lonid=$cookies{'lonID'};
    my $cookie;
    my $lonidsdir=$r->dir_config('lonIDsDir');

    my $handle;
    if ($lonid) {
	$handle=&LONCAPA::clean_handle($lonid->value);
    }

    my $result = &sso_login($r,$lonid,$handle);
    if (defined($result)) {
	return $result
    }


    if ($r->dir_config("lonBalancer") eq 'yes') {
	$r->set_handlers('PerlResponseHandler'=>
			 [\&Apache::switchserver::handler]);
    }
    
    if ($handle eq '') {
	$r->log_reason("Cookie $handle not valid", $r->filename); 
    } elsif ((-e "$lonidsdir/$handle.id") && ($handle ne '')) {

# ------------------------------------------------------ Initialize Environment

	&Apache::lonnet::transfer_profile_to_env($lonidsdir,$handle);

# --------------------------------------------------------- Initialize Language

	&Apache::lonlocal::get_language_handle($r);

    }

# -------------------------------------------------- Should be a valid user now
    if ($env{'user.name'} ne '' && $env{'user.domain'} ne '') {
# -------------------------------------------------------------- Resource State

	if ($requrl=~/^\/+(res|uploaded)\//) {
	    $env{'request.state'} = "published";
	} else {
	    $env{'request.state'} = 'unknown';
	}
	$env{'request.filename'} = $r->filename;
	$env{'request.noversionuri'} = &Apache::lonnet::deversion($requrl);
# -------------------------------------------------------- Load POST parameters

	&Apache::lonacc::get_posted_cgi($r);

# ---------------------------------------------------------------- Check access
	my $now = time;
	if ($requrl !~ m{^/(?:adm|public|prtspool)/}
	    || $requrl =~ /^\/adm\/.*\/(smppg|bulletinboard)(\?|$ )/x) {
	    my $access=&Apache::lonnet::allowed('bre',$requrl);
	    if ($access eq '1') {
		$env{'user.error.msg'}="$requrl:bre:0:0:Choose Course";
		return HTTP_NOT_ACCEPTABLE; 
	    }
	    if ($access eq 'A') {
		&Apache::restrictedaccess::setup_handler($r);
		return OK;
	    }
            if ($access eq 'B') {
                &Apache::blockedaccess::setup_handler($r);
                return OK;
            }
	    if (($access ne '2') && ($access ne 'F')) {
		$env{'user.error.msg'}="$requrl:bre:1:1:Access Denied";
		return HTTP_NOT_ACCEPTABLE; 
	    }
	}
	if ($requrl =~ m|^/prtspool/|) {
	    my $start='/prtspool/'.$env{'user.name'}.'_'.
		$env{'user.domain'};
	    if ($requrl !~ /^\Q$start\E/) {
		$env{'user.error.msg'}="$requrl:bre:1:1:Access Denied";
		return HTTP_NOT_ACCEPTABLE;
	    }
	}
	if ($env{'user.name'} eq 'public' && 
	    $env{'user.domain'} eq 'public' &&
	    $requrl !~ m{^/+(res|public|uploaded)/} &&
	    $requrl !~ m{^/adm/[^/]+/[^/]+/aboutme/portfolio$ }x &&
	    $requrl !~ m{^/+adm/(help|logout|restrictedaccess|randomlabel\.png)}) {
	    $env{'request.querystring'}=$r->args;
	    $env{'request.firsturl'}=$requrl;
	    return FORBIDDEN;
	}
# ------------------------------------------------------------- This is allowed
	if ($env{'request.course.id'}) {
	    &Apache::lonnet::countacc($requrl);
	    $requrl=~/\.(\w+)$/;
	    if ((&Apache::loncommon::fileembstyle($1) eq 'ssi') ||
		($requrl=~/^\/adm\/.*\/(aboutme|navmaps|smppg|bulletinboard)(\?|$ )/x) ||
		($requrl=~/^\/adm\/wrapper\//) ||
		($requrl=~m|^/adm/coursedocs/showdoc/|) ||
		($requrl=~m|\.problem/smpedit$|) ||
		($requrl=~/^\/public\/.*\/syllabus$/)) {
# ------------------------------------- This is serious stuff, get symb and log
		my $query=$r->args;
		my $symb;
		if ($query) {
		    &Apache::loncommon::get_unprocessed_cgi($query,['symb']);
		}
		if ($env{'form.symb'}) {
		    $symb=&Apache::lonnet::symbclean($env{'form.symb'});
		    if ($requrl =~ m|^/adm/wrapper/|
			|| $requrl =~ m|^/adm/coursedocs/showdoc/|) {
			my ($map,$mid,$murl)=&Apache::lonnet::decode_symb($symb);
			&Apache::lonnet::symblist($map,$murl => [$murl,$mid],
						  'last_known' =>[$murl,$mid]);
		    } elsif ((&Apache::lonnet::symbverify($symb,$requrl)) ||
			     (($requrl=~m|(.*)/smpedit$|) &&
			      &Apache::lonnet::symbverify($symb,$1))) {
			my ($map,$mid,$murl)=&Apache::lonnet::decode_symb($symb);
			&Apache::lonnet::symblist($map,$murl => [$murl,$mid],
						  'last_known' =>[$murl,$mid]);
		    } else {
			$r->log_reason('Invalid symb for '.$requrl.': '.
				       $symb);
			$env{'user.error.msg'}=
			    "$requrl:bre:1:1:Invalid Access";
			return HTTP_NOT_ACCEPTABLE; 
		    }
		} else {
		    $symb=&Apache::lonnet::symbread($requrl);
		    if (&Apache::lonnet::is_on_map($requrl) && $symb &&
			!&Apache::lonnet::symbverify($symb,$requrl)) {
			$r->log_reason('Invalid symb for '.$requrl.': '.$symb);
			$env{'user.error.msg'}=
			    "$requrl:bre:1:1:Invalid Access";
			return HTTP_NOT_ACCEPTABLE; 
		    }
		    if ($symb) {
			my ($map,$mid,$murl)=
			    &Apache::lonnet::decode_symb($symb);
			&Apache::lonnet::symblist($map,$murl =>[$murl,$mid],
						  'last_known' =>[$murl,$mid]);
		    }
		}
		$env{'request.symb'}=$symb;
		&Apache::lonnet::courseacclog($symb);
	    } else {
# ------------------------------------------------------- This is other content
		&Apache::lonnet::courseacclog($requrl);    
	    }
	}
	return OK;
    }
# -------------------------------------------- See if this is a public resource
    if ($requrl=~m|^/+adm/+help/+|) {
 	return OK;
    }
# ------------------------------------ See if this is a viewable portfolio file
    if (&Apache::lonnet::is_portfolio_url($requrl)) {
	my $access=&Apache::lonnet::allowed('bre',$requrl);
	if ($access eq 'A') {
	    &Apache::restrictedaccess::setup_handler($r);
	    return OK;
	}
	if (($access ne '2') && ($access ne 'F')) {
	    $env{'user.error.msg'}="$requrl:bre:1:1:Access Denied";
	    return HTTP_NOT_ACCEPTABLE;
	}
    }

# -------------------------------------------------------------- Not authorized
    $requrl=~/\.(\w+)$/;
#    if ((&Apache::loncommon::fileembstyle($1) eq 'ssi') ||
#        ($requrl=~/^\/adm\/(roles|logout|email|menu|remote)/) ||
#        ($requrl=~m|^/prtspool/|)) {
# -------------------------- Store where they wanted to go and get login screen
	$env{'request.querystring'}=$r->args;
	$env{'request.firsturl'}=$requrl;
       return FORBIDDEN;
#   } else {
# --------------------------------------------------------------------- Goodbye
#       return HTTP_BAD_REQUEST;
#   }
}

1;
__END__

=head1 NAME

Apache::lonacc - Cookie Based Access Handler

=head1 SYNOPSIS

Invoked (for various locations) by /etc/httpd/conf/srm.conf:

 PerlAccessHandler       Apache::lonacc

=head1 INTRODUCTION

This module enables cookie based authentication and is used
to control access for many different LON-CAPA URIs.

Whenever the client sends the cookie back to the server, 
this cookie is handled by either lonacc.pm or loncacc.pm
(see srm.conf for what is invoked when).  If
the cookie is missing or invalid, the user is re-challenged
for login information.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

=head1 HANDLER SUBROUTINE

This routine is called by Apache and mod_perl.

=over 4

=item *

transfer profile into environment

=item *

load POST parameters

=item *

check access

=item *

if allowed, get symb, log, generate course statistics if applicable

=item *

otherwise return error

=item *

see if public resource

=item *

store attempted access

=back

=cut
