# The LearningOnline Network with CAPA
# handler for DC-only modifiable course settings
#
# $Id: lonmodifycourse.pm,v 1.28 2006/11/14 20:38:21 raeburn Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
package Apache::lonmodifycourse;

use strict;
use Apache::Constants qw(:common :http);
use Apache::lonnet;
use Apache::loncommon;
use Apache::lonhtmlcommon;
use Apache::lonlocal;
use Apache::londropadd;
use Apache::lonpickcourse;
use LONCAPA::Enrollment;
use lib '/home/httpd/lib/perl';
use LONCAPA;

sub get_dc_settable {
    return ('courseowner','coursecode','authtype','autharg');
}

sub get_enrollment_settings {
    my ($cdom,$cnum) = @_;
    my %settings = &Apache::lonnet::dump('environment',$cdom,$cnum);
    my %enrollvar;
    $enrollvar{'autharg'} = '';
    $enrollvar{'authtype'} = '';
    my %lt=&Apache::lonlocal::texthash(
            'noen' => "No end date",
    );
    foreach my $item (keys %settings) {
        if ($item =~ m/^internal\.(.+)$/) {
            my $type = $1;
            if ( ($type eq "autoadds") || ($type eq "autodrops") ) {
                if ($settings{$item} == 1) {
                    $enrollvar{$type} = "ON";
                } else {
                    $enrollvar{$type} = "OFF";
                }
            } elsif ( ($type eq "autostart") || ($type eq "autoend") ) {
                if ( ($type eq "autoend") && ($settings{$item} == 0) ) {
                    $enrollvar{$type} = $lt{'noen'};
                } else {
                    $enrollvar{$type} = localtime($settings{$item});
                }
            } elsif ($type eq "sectionums") {
                $enrollvar{$type} = $settings{$item};
                $enrollvar{$type} =~ s/,/, /g;
            } elsif ($type eq "authtype"
                     || $type eq "autharg"    || $type eq "coursecode"
                     || $type eq "crosslistings") {
                $enrollvar{$type} = $settings{$item};
            } elsif ($type eq 'courseowner') {
                if ($settings{$item} =~ /^[^:]+:[^:]+$/) {
                    $enrollvar{$type} = $settings{$item};
                } else {
                    if ($settings{$item} ne '') {
                        $enrollvar{$type} = $settings{$item}.':'.$cdom;
                    }
                }
            }
        } elsif ($item =~ m/^default_enrollment_(start|end)_date$/) {
            my $type = $1;
            if ( ($type eq 'end') && ($settings{$item} == 0) ) {
                $enrollvar{$item} = $lt{'noen'};
            } elsif ( ($type eq 'start') && ($settings{$item} eq '') ) {
                $enrollvar{$item} = 'When enrolled';
            } else {
                $enrollvar{$item} = localtime($settings{$item});
            }
        }
    }
    return %enrollvar;
}

sub print_course_search_page {
    my ($r,$dom,$domdesc) = @_;
    &print_header($r);
    my $filterlist = ['descriptfilter',
                      'instcodefilter','ownerfilter',
                      'ownerdomfilter','coursefilter'];
    my $filter = {};
    my $type = 'Course';
    my $action = '/adm/modifycourse';
    my $cctitle = &Apache::lonnet::plaintext('cc',$type);
    my $dctitle = &Apache::lonnet::plaintext('dc');
    my %lt=&Apache::lonlocal::texthash(
                    'some' => "Certain settings which control auto-enrollment of students from your institution's student information system.",
                    'crqo' => 'The total disk space allocated for storage of portfolio files in all groups in a course.',
                    'tmod' => 'To view or modify these settings use the criteria below to select a course from this domain.',
    );
    $r->print('<h3>'.
              &mt('Course settings which only a [_1] may modify.'
                  ,$dctitle).'</h3>'.
              &mt('Although almost all course settings can be modified by a [_1], a number of settings exist which only a [_2] may change:',$cctitle,$dctitle).'
<ul>
  <li>'.$lt{'some'}.'</li>
  <li>'.$lt{'crqo'}.'</li>
</ul>'.
$lt{'tmod'}.' ('.$domdesc.')
<br /><br />
    ');
    $r->print(&Apache::lonpickcourse::build_filters($filterlist,$type,
                             undef,undef,$filter,$action,'modifycourse'));
}

sub print_course_selection_page {
    my ($r,$dom,$domdesc) = @_;
    &print_header($r);

# Criteria for course search 
    my $filterlist = ['descriptfilter',
                      'instcodefilter','ownerfilter',
                      'ownerdomfilter','coursefilter'];
    my %filter;
    my $type = $env{'form.type'};
    my $action = '/adm/modifycourse';
    my $dctitle = &Apache::lonnet::plaintext('dc');
    $r->print(&mt('Revise your search criteria for this domain').' ('.$domdesc.').<br /><br />');
    $r->print(&Apache::lonpickcourse::build_filters($filterlist,$type,
                                       undef,undef,\%filter,$action));
    $filter{'domainfilter'} = $dom;
    my %courses = &Apache::lonpickcourse::search_courses($r,$type,0,
                                                         \%filter);
    if (keys(%courses) > 0) {
        $r->print(&mt("Click a 'Select' button to view or modify settings for a [_1] which may only be modified by a [_2] in this domain.",lc($type),$dctitle).'<br /><br />');
    }

    &Apache::lonpickcourse::display_matched_courses($r,$type,0,$action,
                                                    %courses);
    return;
}

sub print_modification_menu {
    my ($r,$cdesc) = @_;
    &print_header($r,$cdesc);
    my @menu =
        (
          { text  => 'Modify quota for group portfolio files',
            phase => 'setquota',
            },
          { text  => 'Display current settings for automated enrollment',
            phase => 'viewparms',
            },
          { text => 'Modify institutional code, course owner and/or default authentication',
            phase => 'setparms',
            }
         );
    my $menu_html = '<h3>'.&mt('View/Modify settings for: ').$cdesc.'</h3>'."\n".
                    '<form name="menu" method="post" action="/adm/modifycourse" />'."\n".
                     &hidden_form_elements();
    foreach my $menu_item (@menu) {
        $menu_html.='<p>';
        $menu_html.='<font size="+1">';
        $menu_html.=
                qq|<a href="javascript:changePage(document.menu,'$menu_item->{'phase'}')">|;
        $menu_html.= &mt($menu_item->{'text'}).'</a></font>';
        $menu_html.='</p>';
    }
    
    $r->print($menu_html);
    return;
}

sub print_settings_display {
    my ($r,$cdom,$cnum,$cdesc,$type) = @_;
    my %enrollvar = &get_enrollment_settings($cdom,$cnum);
    my %longtype = &course_settings_descrip();
    my %lt = &Apache::lonlocal::texthash(
            'cset' => "Course setting",
            'valu' => "Current value",
            'caes' => 'Current automated enrollment settings for ',
            'cour' => "Course settings that control automated enrollment in this LON-CAPA course are currently:",
            'cose' => "Course settings for LON-CAPA courses that control auto-enrollment based on classlist data available from your institution's student information system fall into two groups:",
            'dcon' => "Modifiable by DC only",
            'back' => "Back to options page",
    );

    my @bgcolors = ('#eeeeee','#cccccc');
    my $cctitle = &Apache::lonnet::plaintext('cc',$type);
    my $dctitle = &Apache::lonnet::plaintext('dc');
    my @modifiable_params = &get_dc_settable();

    my $disp_table = &Apache::loncommon::start_data_table()."\n".
                     &Apache::loncommon::start_data_table_header_row()."\n".
                     "<th>$lt{'cset'}</th>\n".
                     "<th>$lt{'valu'}</th>\n".
                     "<th>$lt{'dcon'}</th>\n".
                     &Apache::loncommon::end_data_table_header_row()."\n";
    foreach my $key (sort(keys(%enrollvar))) {
        $disp_table .= &Apache::loncommon::start_data_table_row()."\n".
                 "<td>$longtype{$key}</td>\n".
                 "<td>$enrollvar{$key}</td>\n";
        if (grep(/^\Q$key\E$/,@modifiable_params)) {
            $disp_table .= '<td>'.&mt('Yes').'</td>'."\n"; 
        } else {
            $disp_table .= '<td>'.&mt('No').'</td>'."\n";
        }
        $disp_table .= &Apache::loncommon::end_data_table_row()."\n";
    }
    $disp_table .= &Apache::loncommon::end_data_table()."\n";
    &print_header($r,$cdesc);
    $r->print('
<h3>'.$lt{'caes'}.'</h3>
<p>
<form action="/adm/modifycourse" method="post" name="viewparms">
'.$lt{'cose'}.'<ul><li>'.&mt('Settings that can be modified by a [_1] using the <a href="/adm/populate">Automated Enrollment Manager</a>.',$cctitle).'</li><li>'.&mt('Settings that may only be modified by a [_1] from this menu.',$dctitle).'</li></ul>
</p><p>
'.$lt{'cour'}.'
</p><p>
'.$disp_table.'
</p><p>
<a href="javascript:changePage(document.viewparms,'."'menu'".')">'.$lt{'back'}.'</a>&nbsp;&nbsp;&nbsp;&nbsp;
<a href="javascript:changePage(document.viewparms,'."'setparms'".')">'.&mt('Modify [_1]-only settings',$dctitle).'</a>'."\n".
&hidden_form_elements().    
'</form>');
}

sub print_setquota {
    my ($r,$cdom,$cnum,$cdesc,$type) = @_;
    my $dctitle = &Apache::lonnet::plaintext('dc');
    my $cctitle = &Apache::lonnet::plaintext('cc',$type);
    my $subdiv = &mt('Although a [_1] will assign the disk quota for each individual group, the size of the quota is constrained by the total disk space allocated by the [_2] for portfolio files in a course.',$cctitle,$dctitle);
    my %lt = &Apache::lonlocal::texthash(
                'cquo' => 'Disk space for storage of group portfolio files',
                'gpqu' => 'Course portfolio files disk space',
                'each' => 'Each course group can be assigned a quota for portfolio files uploaded to the group.',
                'modi' => 'Modify quota',
                'back' => "Back to options page",
    );
    my %settings = &Apache::lonnet::get('environment',['internal.coursequota'],$cdom,$cnum);
    my $coursequota = $settings{'internal.coursequota'};
    if ($coursequota eq '') {
        $coursequota = 20;
    }
    &print_header($r,$cdesc);
    my $hidden_elements = &hidden_form_elements();
    $r->print(<<ENDDOCUMENT);
<form action="/adm/modifycourse" method="post" name="setquota">
<h3>$lt{'cquo'}.</h3>
<p>
$lt{'each'}<br />
$subdiv
</p><p>
$lt{'gpqu'}: <input type="text" size="4" name="coursequota" value="$coursequota" /> Mb &nbsp;&nbsp;&nbsp;&nbsp;
<input type="button" onClick="javascript:verify_quota(this.form)" value="$lt{'modi'}" />
</p>
$hidden_elements
<a href="javascript:changePage(document.setquota,'menu')">$lt{'back'}</a>
</form>
ENDDOCUMENT
    return;
}

sub print_course_modification_page {
    my ($r,$cdom,$cnum,$cdesc,$domdesc) = @_;
    my %longtype = &course_settings_descrip();
    my %enrollvar = &get_enrollment_settings($cdom,$cnum);
    my $ownertable;
    my %lt=&Apache::lonlocal::texthash(
            'actv' => "Active",
            'inac' => "Inactive",
            'ccor' => "Course Coordinator",
            'noen' => "No end date",
            'ownr' => "Owner",
            'name' => "Name",
            'unme' => "Username:Domain",
            'stus' => "Status",
            'cquo' => "Disk space for storage of group portfolio files",
            'gpqu' => "Course portfolio files disk space",
            'each' => "Each course group can be assigned a quota for portfolio files uploaded to the group.",
            'cose' => "Course settings for LON-CAPA courses that control automated student enrollment based on classlist data available from your institution's student information system fall into two groups: (a) settings that can be modified by a Course Coordinator using the ",
            'aenm' => "Automated Enrollment Manager",
            'andb' => " and (b) settings that may only be modified by a Domain Coordinator via this page.",
            'caes' => 'Current automated enrollment settings',
            'cour' => "Course settings that control automated enrollment in this LON-CAPA course
are currently:",
            'nocc' => "There is currently no course owner set for this course. In addition, no active course coordinators from this LON-CAPA domain were found, so you will not be able assign a course owner.  If you wish to assign a course owner, it is recommended that you use the 'User Roles' screen to add a course coordinator with a LON-CAPA account in this domain to the course.",    
            'ccus' => "A course coordinator can use the 'Automated Enrollment Manager' to change
all settings except course code, course owner, and default authentication method for students added to your course (who are also new to the LON-CAPA system at this domain).",
            'ccod' => "Course Code",
            'ccus' => "The course code is used during automated enrollment to map the internal LON-CAPA course ID for this course to the corresponding course section ID(s) used by the office responsible for providing official class lists for courses at your institution.",
            'cown' => "Course Owner",
            'cous' => "The course owner is used in the retrieval of class lists from your institution's student information system when access to class lists data incorporates validation of instructor status.",
            'tabl' => 'The table below contains a list of active course coordinators in this course, who are from this domain',
            'usrd' => 'Use the radio buttons to select a different course owner.',
            'deam' => "Default Authentication method",
            'deus' => "The default authentication method, and default authentication parameter (domain, initial password or argument) are used when automatic enrollment of students in a course requires addition of new user accounts in your domain, and the class list file contains empty entries for the &lt;authtype&gt; and &lt;autharg&gt; properties for the new student. If you choose 'internally authenticated', and leave the initial password field empty, the automated enrollment process will create a randomized password for each new student account that it adds to your LON-CAPA domain.",
            'gobt' => "Modify settings",
    );
    my @bgcolors = ('#eeeeee','#cccccc');

    my @coursepersonnel = &Apache::lonnet::getkeys('nohist_userroles',$cdom,$cnum);
    my @local_ccs = ();
    my %cc_status = ();
    my %pname = ();
    my $active_cc;
    foreach my $person (@coursepersonnel) {
        my ($role,$user) = split(/:/,$person,2);
        $user =~ s/:$//;
        if (($role eq 'cc') && ($user ne ''))  {
            if (!grep(/^\Q$user\E$/,@local_ccs)) {
                my ($ccname,$ccdom) = split(/:/,$user);
                $active_cc = 
                   &Apache::loncommon::check_user_status($ccdom,$ccname,$cdom,
                                                         $cnum,'cc');
                if ($active_cc eq 'active') {
                    push(@local_ccs,$user);
                    $pname{$user} = &Apache::loncommon::plainname($ccname,$ccdom);
                    $cc_status{$user} = $lt{'actv'};
                }
            }
        }
    }
    if ( (!grep(/^$enrollvar{'courseowner'}$/,@local_ccs)) && 
             ($enrollvar{'courseowner'} ne '') )  {
        my ($owneruname,$ownerdom) = split(/:/,$enrollvar{'courseowner'});
        push(@local_ccs,$enrollvar{'courseowner'});
        $pname{$enrollvar{'courseowner'}} = 
                         &Apache::loncommon::plainname($owneruname,$ownerdom);
        $active_cc = &Apache::loncommon::check_user_status($ownerdom,$owneruname,
                                                           $cdom,$cnum,'cc');
        if ($active_cc eq 'active') {
            $cc_status{$enrollvar{'courseowner'}} = $lt{'actv'};
        } else {
            $cc_status{$enrollvar{'courseowner'}} = $lt{'inac'};
        }
    }
    my $numlocalcc = @local_ccs;

    my $helplink=&Apache::loncommon::help_open_topic('Modify_Course',&mt("Help on Modifying Courses"));
    my ($krbdef,$krbdefdom)=&Apache::loncommon::get_kerberos_defaults($cdom);
    my $curr_authtype = '';
    my $curr_authfield = '';
    if ($enrollvar{'authtype'} =~ /^krb/) {
        $curr_authtype = 'krb';
    } elsif ($enrollvar{'authtype'} eq 'internal' ) {
        $curr_authtype = 'int';
    } elsif ($enrollvar{'authtype'} eq 'localauth' ) {
        $curr_authtype = 'loc';
    }
    unless ($curr_authtype eq '') {
        $curr_authfield = $curr_authtype.'arg';
    } 
    my $javascript_validations=&Apache::londropadd::javascript_validations('modifycourse',$krbdefdom,$curr_authtype,$curr_authfield);
    my %param = ( formname => 'document.cmod',
	   kerb_def_dom => $krbdefdom,
	   kerb_def_auth => $krbdef,
           mode => 'modifycourse',
           curr_authtype => $curr_authtype,
           curr_autharg => $enrollvar{'autharg'} 
	);
    my $krbform = &Apache::loncommon::authform_kerberos(%param);
    my $intform = &Apache::loncommon::authform_internal(%param);
    my $locform = &Apache::loncommon::authform_local(%param);

    if ($numlocalcc == 0) {
        $ownertable = $lt{'nocc'};
    }

    if ($numlocalcc > 0) {
        @local_ccs = sort @local_ccs;
        $ownertable = qq(
            <input type="hidden" name="numlocalcc" value="$numlocalcc" />
            <table>
             <tr>
               <td>$lt{'tabl'} - $cdom ($domdesc). $lt{'usrd'}
              </td>
             </tr>
             <tr><td>&nbsp;</td></tr>
             <tr>
              <td>).
            &Apache::loncommon::start_data_table()."\n".
            &Apache::loncommon::start_data_table_header_row()."\n".
                       "<th>$lt{'ownr'}</th>\n".
                       "<th>$lt{'name'}</th>\n".
                       "<th>$lt{'unme'}</th>\n".
                       "<th>$lt{'stus'}</th>\n".
            &Apache::loncommon::end_data_table_header_row()."\n";
        foreach my $cc (@local_ccs) {
            $ownertable .= &Apache::loncommon::start_data_table_row()."\n";
            if ($cc eq $enrollvar{'courseowner'}) {
                  $ownertable .= '<td><input type="radio" name="courseowner" value="'.$cc.'" checked="checked" /></td>'."\n";
            } else {
                $ownertable .= '<td><input type="radio" name="courseowner" value="'.$cc.'" /></td>'."\n";
            }
            $ownertable .= 
                 '<td>'.$pname{$cc}.'</td>'."\n".
                 '<td>'.$cc.'</td>'."\n".
                 '<td>'.$cc_status{$cc}.' '.$lt{'ccor'}.'</td>'."\n".
                 &Apache::loncommon::end_data_table_row()."\n";
        }
        $ownertable .= &Apache::loncommon::end_data_table()."
              </td>
             </tr>
            </table>";
    }
    &print_header($r,$cdesc,$javascript_validations);
    my $type = $env{'form.type'};
    my $dctitle = &Apache::lonnet::plaintext('dc');
    my $cctitle = &Apache::lonnet::plaintext('cc',$type);
    my $mainheader = &mt('Course settings modifiable by [_1] only.',$dctitle);
    my $hidden_elements = &hidden_form_elements();
    $r->print(<<ENDDOCUMENT);
<form action="/adm/modifycourse" method="post" name="setparms">
<h3>$mainheader</h3>
</p><p>
<table width="100%" cellspacing="6" cellpadding="6">
 <tr>
  <td colspan="2">Use the appropriate text boxes and radio buttons below to change some or all of the four automated enrollment settings that may only be changed by a Domain Coordinator. Click the <b>"$lt{'gobt'}"</b> button to save your changes.</td>
 </tr>
 <tr>
  <td width="50%" valign="top">
   <b>$lt{'ccod'}:</b>&nbsp;&nbsp;
    <input type="text" size="10" name="coursecode" value="$enrollvar{'coursecode'}"/><br/><br/>
    $lt{'ccus'}
  </td>
  <td width="50%" valign="top" rowspan="2">
   <b>$lt{'cown'}:</b><br/><br/>
   $ownertable
   <br/><br/>
   $lt{'cous'}
  </td>
 </tr>
 <tr>
  <td width="50%" valign="top">
   <b>$lt{'deam'}:</b><br/><br/>
   $krbform
   <br/>
   $intform
   <br/>
   $locform
   <br/>
   <br/>
   $lt{'deus'}.
   </td>
   <td width="50%">&nbsp;</td>
 </tr>
</table>
<table width="90%" cellpadding="5" cellspacing="0">
 <tr>
  <td align="left">
   $hidden_elements
   <input type="button" onClick="javascript:changePage(this.form,'processparms');javascript:verify_message(this.form)" value="$lt{'gobt'}" />
  </td>
 </tr>
</table>
</form>
<br/>
<br/>
ENDDOCUMENT
    return;
}

sub modify_course {
    my ($r,$cdom,$cnum,$cdesc,$domdesc) = @_;
    my %longtype = &course_settings_descrip();
    my %settings = &Apache::lonnet::get('environment',['internal.courseowner','internal.coursecode','internal.authtype','internal.autharg','internal.sectionnums','internal.crosslistings','description'],$cdom,$cnum);
    my %currattr = ();
    my %newattr = ();
    my %cenv = ();
    my $response;
    my $chgresponse;
    my $nochgresponse;
    my $warning;
    my $reply;
    my @changes = ();
    my @nochanges = ();
    my @sections = ();
    my @xlists = ();
    my $changecode = 0;
    my $changeowner = 0;
    unless ($settings{'internal.sectionnums'} eq '') {
        if ($settings{'internal.sectionnums'} =~ m/,/) {
            @sections = split/,/,$settings{'internal.sectionnums'};
        } else {
            $sections[0] = $settings{'internal.sectionnums'};
        }
    }
    unless ($settings{'internal.crosslistings'} eq'') {
        if ($settings{'internal.crosslistings'} =~ m/,/) {
            @xlists = split/,/,$settings{'internal.crosslistings'};
        } else {
            $xlists[0] = $settings{'internal.crosslistings'};
        }
    }

    my @modifiable_params = &get_dc_settable();
    foreach my $param (@modifiable_params) {
        my $attr = 'internal.'.$param;
        $currattr{$param} = $settings{$attr};
    }

    my $description = $settings{'description'};
    my %cenv = ();

    if ($env{'form.login'} eq 'krb') {
        $newattr{'authtype'} = $env{'form.login'};
        $newattr{'authtype'} .= $env{'form.krbver'};
        $newattr{'autharg'} = $env{'form.krbarg'};
    } elsif ($env{'form.login'} eq 'int') {
        $newattr{'authtype'} ='internal';
        if ((defined($env{'form.intarg'})) && ($env{'form.intarg'})) {
            $newattr{'autharg'} = $env{'form.intarg'};
        }
    } elsif ($env{'form.login'} eq 'loc') {
        $newattr{'authtype'} = 'localauth';
        if ((defined($env{'form.locarg'})) && ($env{'form.locarg'})) {
            $newattr{'autharg'} = $env{'form.locarg'};
        }
    }
    if ( $newattr{'authtype'}=~ /^krb/) {
        if ($newattr{'autharg'}  eq '') {
            $warning = qq(<font color="red" size="+1">).
	    &mt("As you did not include the default Kerberos domain to be used for authentication in this class, the institutional data used by the automated enrollment process must include the Kerberos domain for each new student").qq(</font></p>);
        }
    }

    if ( exists($env{'form.courseowner'}) ) {
        $newattr{'courseowner'}=$env{'form.courseowner'};
        unless ( $newattr{'courseowner'} eq $currattr{'courseowner'} ) {
            $changeowner = 1;
        } 
    }
													      
    if ( exists($env{'form.coursecode'}) ) {
        $newattr{'coursecode'}=$env{'form.coursecode'};
        unless ( $newattr{'coursecode'} eq $currattr{'coursecode'} ) {
            $changecode = 1;
        }
    }
    if ($changeowner == 1 || $changecode == 1) {
        my $courseid_entry = &escape($cdom.'_'.$cnum).'='.&escape($description).':'.&escape($env{'form.coursecode'}).':'.&escape($env{'form.courseowner'});
        &Apache::lonnet::courseidput($cdom,$courseid_entry,&Apache::lonnet::homeserver($cnum,$cdom));
    }

    foreach my $param (@modifiable_params) {
        if ($currattr{$param} eq $newattr{$param}) {
            push(@nochanges,$param);
        } else {
            my $attr = 'internal.'.$param;
            $cenv{$attr} = $newattr{$param};
            push(@changes,$param);
        }
    }

    if (@changes > 0) {
        $chgresponse = &mt("The following automated enrollment parameters have been changed:<br/><ul>");
    }
    if (@nochanges > 0) { 
        $nochgresponse = &mt("The following automated enrollment parameters remain unchanged:<br/><ul>");
    }
    if (@changes > 0) { 
        my $putreply = &Apache::lonnet::put('environment',\%cenv,$cdom,$cnum);
        if ($putreply !~ /^ok$/) {
            $response = &mt("There was a problem processing your requested changes. The automated enrollment settings for this course have been left unchanged.<br/>");
        } else {
            foreach my $attr (@modifiable_params) {
                if (grep/^$attr$/,@changes) {
	            $chgresponse .= "<li>$longtype{$attr} ".&mt("now set to \"").$newattr{$attr}."\".</li>";
                } else {
	            $nochgresponse .= "<li>$longtype{$attr} ".&mt("still set to \"").$currattr{$attr}."\".</li>";
                }
            }
            if ($changecode || $changeowner) {
                if ( $newattr{'courseowner'} eq '') {
	            $warning .= &mt("There is no owner associated with this LON-CAPA course.  If automated enrollment in LON-CAPA courses at your institution requires validation of course owners, automated enrollment will fail for this course.<br/>");
                } else {
	            if (@sections > 0) {
                        if ($changecode) {
	                    foreach my $sec (@sections) {
		                if ($sec =~ m/^(.+):/) {
		                    my $inst_course_id = $newattr{'coursecode'}.$1;
                                    my $course_check = &Apache::lonnet::auto_validate_courseID($cnum,$cdom,$inst_course_id);
			            if ($course_check eq 'ok') {
                                        my $outcome = &Apache::lonnet::auto_new_course($cnum,$cdom,$inst_course_id,$newattr{'courseowner'});
			                unless ($outcome eq 'ok') { 
				            $warning .= &mt("If automatic enrollment is enabled for LON-CAPA course: ").$description.&mt(", automated enrollment may fail for ").$newattr{'coursecode'}.&mt(" - section $1 for the following reason: $outcome.<br/>");
			                }
			            } else {
			                $warning .= &mt("If automatic enrollment is enabled for LON-CAPA course: ").$description.&mt(", automated enrollment may fail for ").$newattr{'coursecode'}.&mt(" - section $1 for the following reason: $course_check.<br/>");
			            }
		                } else {
			            $warning .= &mt("If automatic enrollment is enabled for LON-CAPA course: ").$description.&mt(", automated enrollment may fail for ").$newattr{'coursecode'}.&mt(" - section $sec because this is not a valid section entry.<br/>");
		                }
		            }
	                } elsif ($changeowner) {
                            foreach my $sec (@sections) {
                                if ($sec =~ m/^(.+):/) {
                                    my $inst_course_id = $newattr{'coursecode'}.$1;
                                    my $outcome = &Apache::lonnet::auto_new_course($cnum,$cdom,$inst_course_id,$newattr{'courseowner'});
                                    unless ($outcome eq 'ok') {
                                        $warning .= &mt("If automatic enrollment is enabled for LON-CAPA course: ").$description.&mt(", automated enrollment may fail for ").$newattr{'coursecode'}.&mt(" - section $1 for the following reason: $outcome.<br/>");
                                    }
                                } else {
                                    $warning .= &mt("If automatic enrollment is enabled for LON-CAPA course: ").$description.&mt(", automated enrollment may fail for ").$newattr{'coursecode'}.&mt(" - section $sec because this is not a valid section entry.<br/>");
                                }
                            }
                        }
	            } else {
	                $warning .= &mt("As no section numbers are currently listed for LON-CAPA course: ").$description.&mt(", automated enrollment will not occur for any sections of coursecode: ").$newattr{'coursecode'}."<br/>";
	            }
	            if ( (@xlists > 0) && ($changeowner) ) {
	                foreach my $xlist (@xlists) {
		            if ($xlist =~ m/^(.+):/) {
                                my $outcome = &Apache::lonnet::auto_new_course($cnum,$cdom,$1,$newattr{'courseowner'});
		                unless ($outcome eq 'ok') {
			            $warning .= &mt("If automatic enrollment is enabled for LON-CAPA course: ").$description.&mt(", automated enrollment may fail for crosslisted class: ").$1.&mt(" for the following reason: $outcome.<br/>");
		                }
		            }
	                }
	            }
                }
            }
        }
    } else {
        foreach my $attr (@modifiable_params) {
            $nochgresponse .= "<li>$longtype{$attr} ".&mt("still set to")." \"".$currattr{$attr}."\".</li>";
        }
    }

    if (@changes > 0) {
        $chgresponse .= "</ul><br/><br/>";
    }
    if (@nochanges > 0) {
        $nochgresponse .=  "</ul><br/><br/>";
    }
    unless ($warning eq '') {
        $warning = &mt("The following warning messages were generated as a result of applying the changes you specified to course settings that can affect the automated enrollment process:<br/><br/>").$warning;
    }
    if ($response eq '') {
        $reply = $chgresponse.$nochgresponse.$warning;
    } else {
        $reply = $response;
    }
    &print_header($r,$cdesc);
    $reply = '<h3>'.
              &mt('Result of Changes to Automated Enrollment Settings.').
             '</h3>'."\n".
             '<table><tr><td>'.$reply.'</td></tr></table>'.
             '<form action="/adm/modifycourse" method="post" name="processparms">'.
             &hidden_form_elements().
             '<a href="javascript:changePage(document.processparms,'."'menu'".')">'.&mt('Back to options page').'</a>
             </form>';
    $r->print($reply);
    return;
}

sub modify_quota {
    my ($r,$cdom,$cnum,$cdesc,$domdesc) = @_;
    &print_header($r,$cdesc);

    my %lt = &Apache::lonlocal::texthash(
                                         'back' => 'Back to options page',
                                        );
    $r->print('
<form action="/adm/modifycourse" method="post" name="processquota">
<h3>'.&mt('Disk space for storage of group portfolio files for [_1]',$cdesc).
             '</h3>');
    my %oldsettings = &Apache::lonnet::get('environment',['internal.coursequota'],$cdom,$cnum);
    my $defaultquota = 20;
    if ($env{'form.coursequota'} ne '') {
        my $newquota = $env{'form.coursequota'};
        if ($newquota =~ /^\s*(\d+\.?\d*|\.\d+)\s*$/) {
            $newquota = $1;
            if ($oldsettings{'internal.coursequota'} eq $env{'form.coursequota'}) {
                $r->print(&mt('The disk space allocated for group portfolio files remains unchanged as ').$env{'form.coursequota'}.' Mb');
            } else {
                my %cenv = (
                           'internal.coursequota' => $env{'form.coursequota'},
                           );
                my $putreply = &Apache::lonnet::put('environment',\%cenv,$cdom,
                                                    $cnum);
                if (($oldsettings{'internal.coursequota'} eq '') && 
                    ($env{'form.coursequota'} == $defaultquota)) {
                    $r->print(&mt('The disk space allocated for group portfolio files in this course is the default quota for this domain:').' '.$defaultquota.' Mb');
                } else {
                    if ($putreply eq 'ok') {
                        my %updatedsettings = &Apache::lonnet::get('environment',['internal.coursequota'],$cdom,$cnum);
                        $r->print(&mt('The disk space allocated for group portfolio files is now:').' '.$updatedsettings{'internal.coursequota'}.' Mb.');
                        my $usage = &Apache::longroup::sum_quotas($cdom.'_'.$cnum);
                        if ($usage >= $updatedsettings{'internal.coursequota'}) {
                            my $newoverquota;
                            if ($usage < $oldsettings{'internal.coursequota'}) {
                                $newoverquota = 'now';
                            }
                            $r->print('<br /><br />'.
      &mt('Disk usage [_1] exceeds the quota for this course.',$newoverquota).' '.
      &mt('Upload of new portfolio files and assignment of a non-zero Mb quota to new groups in the course will not be possible until some files have been deleted, and total usage is below course quota.'));
                        }
                    } else {
                        $r->print(&mt('There was an error'));
                    }
                }
            }
        } else {
            $r->print(&mt('The new quota requested contained invalid characters, so the quota is unchanged.'));
        }
    }
    $r->print(qq|<br /><br />
<a href="javascript:changePage(document.processquota,'menu')">$lt{'back'}</a>|);
    $r->print(&hidden_form_elements().'</form>');
    return;
}

sub print_header {
    my ($r,$cdesc,$javascript_validations) = @_;
    my $phase = "start";
    if ( exists($env{'form.phase'}) ) {
        $phase = $env{'form.phase'};
    }
    my $js = qq|
<script type="text/javascript">
function changePage(formname,newphase) {
    formname.phase.value = newphase;
    if (newphase == 'processparms') {
        return;
    }
    formname.submit();
}
</script>
|;
    if ($phase eq 'setparms') {
	$js .= qq|
<script  type="text/javascript">
$javascript_validations
</script>
|;
    } elsif ($phase eq 'courselist') {
        $js .= qq|
<script type="text/javascript">
function gochoose(cname,cdom,cdesc) {
    document.courselist.pickedcourse.value = cdom+'_'+cname;
    document.courselist.submit();
}
</script>
|;
    } elsif ($phase eq 'setquota') {
        $js .= <<'ENDSCRIPT';
<script type="text/javascript">
function verify_quota(formname) {
    var newquota = formname.coursequota.value; 
    var num_reg = /^\s*(\d+\.?\d*|\.\d+)\s*$/;
    if (num_reg.test(newquota)) {
        changePage(formname,'processquota');
    } else {
        alert("The quota you entered contained invalid characters.\nYou must enter a number");
    }
    return;
}
</script>
ENDSCRIPT
    }
    $r->print(&Apache::loncommon::start_page('View/Modify Course Settings',
					     $js));
    my $bread_text = "View/Modify Courses";
    if ($cdesc ne '') {
        $bread_text = "Course Settings: $cdesc"; 
    } 
    $r->print(&Apache::lonhtmlcommon::breadcrumbs($bread_text));
    return;
}

sub print_footer {
    my ($r) = @_;
    $r->print('<br />'.&Apache::loncommon::end_page());
    return;
}

sub check_course {
    my ($r,$dom,$domdesc) = @_;
    my ($ok_course,$description,$instcode,$owner);
    if (defined($env{'form.pickedcourse'})) {
        my ($cdom,$cnum) = split(/_/,$env{'form.pickedcourse'});
        if ($cdom eq $dom) {
            my %courseIDs = &Apache::lonnet::courseiddump($cdom,'.',1,'.','.',
                                                          $cnum,undef,undef,'.');
            if (keys(%courseIDs) > 0) {
                $ok_course = 'ok';
                ($description,$instcode,$owner) = 
                                       split(/:/,$courseIDs{$cdom.'_'.$cnum});
                $description = &unescape($description);
                $instcode = &unescape($instcode);
                if ($instcode) {
                    $description .= " ($instcode)";
                } 
            }
        }
    }

    if ($ok_course ne 'ok') {
        $r->print('<br/>'.&mt('The LON-CAPA course selected was not a valid course for the [_1] domain',$domdesc));
    }
    return ($ok_course,$description);
}

sub course_settings_descrip {
    my %longtype = &Apache::lonlocal::texthash(
                      'authtype' => 'Default authentication method',
                      'autharg'  => 'Default authentication parameter',
                      'autoadds' => 'Automated adds',
                      'autodrops' => 'Automated drops',
                      'autostart' => 'Date of first automated enrollment',
                      'autoend' => 'Date of last automated enrollment',
                      'default_enrollment_start_date' => 'Date of first student access',
                      'default_enrollment_end_date' => 'Date of last student access',
                      'coursecode' => 'Official course code',
                      'courseowner' => "Username:domain of course owner",
                      'notifylist' => 'Course Coordinators to be notified of enrollment changes',
                      'sectionnums' => 'Course section number(:LON-CAPA section)',
                      'crosslistings' => 'Crosslisted class(:LON-CAPA section)',
                   );
    return %longtype;
}

sub hidden_form_elements {
    my $hidden_elements = 
      &Apache::lonhtmlcommon::echo_form_input(['gosearch','coursecode',
                                               'numlocalcc','courseowner',
                                               'login','coursequota','intarg',                                               'locarg','krbarg','krbver']);
    return $hidden_elements;
}

sub handler {
    my $r = shift;
    if ($r->header_only) {
        &Apache::loncommon::content_type($r,'text/html');
        $r->send_http_header;
        return OK;
    }
    my $dom = $env{'request.role.domain'};
    my $domdesc = $Apache::lonnet::domaindescription{$dom}; 
    if (&Apache::lonnet::allowed('ccc',$dom)) {
        &Apache::loncommon::content_type($r,'text/html');
        $r->send_http_header;

        &Apache::lonhtmlcommon::clear_breadcrumbs();

        my $phase = $env{'form.phase'};
        &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>"/adm/modifycourse",
              text=>"Course search"});
        if ($phase eq '') {
            &print_course_search_page($r,$dom,$domdesc);
        } else {
            &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>"javascript:changePage(document.$phase,'courselist')",
              text=>"Choose a course"});
            if ($phase eq 'courselist') {
                &print_course_selection_page($r,$dom,$domdesc);
            } else {
                &Apache::lonhtmlcommon::add_breadcrumb
                ({href=>"javascript:changePage(document.$phase,'menu')",
                  text=>"Pick action"});
                my ($checked,$cdesc) = &check_course($r,$dom,$domdesc);
                my $type = $env{'form.type'};
                if ($checked eq 'ok') {
                    if ($phase eq 'menu') {
                        &print_modification_menu($r,$cdesc);
                    } else {
                        my ($cdom,$cnum) = split(/_/,$env{'form.pickedcourse'});
                        if ($phase eq 'setquota') {
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'$phase')",
                              text=>"Set quota"});
                            &print_setquota($r,$cdom,$cnum,$cdesc,$type)
                        } elsif ($phase eq 'processquota') { 
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'setquota')",
                              text=>"Set quota"});
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'$phase')",
                              text=>"Result"});
                            &modify_quota($r,$cdom,$cnum,$cdesc,$domdesc);
                        } elsif ($phase eq 'viewparms') {  
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'viewparms')",
                              text=>"Display settings"});
                            &print_settings_display($r,$cdom,$cnum,$cdesc,$type);
                        } elsif ($phase eq 'setparms') {
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'$phase')",
                              text=>"Change settings"});
                            &print_course_modification_page($r,$cdom,$cnum,$cdesc,$domdesc);
                        } elsif ($phase eq 'processparms') {
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'setparms')",
                              text=>"Change settings"});
                            &Apache::lonhtmlcommon::add_breadcrumb
                            ({href=>"javascript:changePage(document.$phase,'$phase')",
                              text=>"Result"});
                            &modify_course($r,$cdom,$cnum,$cdesc,$domdesc);
                        }
                    }
                } else {
                    $r->print(&mt('The course you selected is not a valid course in this domain')." ($domdesc)"); 
                }
            }
        }
        &print_footer($r);
    } else {
        $env{'user.error.msg'}=
        "/adm/modifycourse:ccc:0:0:Cannot modify course settings";
        return HTTP_NOT_ACCEPTABLE;
    }
    return OK;
}

1;
__END__
