# The LearningOnline Network with CAPA
# Handler to set parameters for assessments
#
# $Id: lonparmset.pm,v 1.350.2.3 2006/12/14 20:50:25 albertel Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
###################################################################
###################################################################

=pod

=head1 NAME

lonparmset - Handler to set parameters for assessments and course

=head1 SYNOPSIS

lonparmset provides an interface to setting course parameters. 

=head1 DESCRIPTION

This module sets coursewide and assessment parameters.

=head1 INTERNAL SUBROUTINES

=over 4

=cut

###################################################################
###################################################################

package Apache::lonparmset;

use strict;
use Apache::lonnet;
use Apache::Constants qw(:common :http REDIRECT);
use Apache::lonhtmlcommon();
use Apache::loncommon;
use GDBM_File;
use Apache::lonhomework;
use Apache::lonxml;
use Apache::lonlocal;
use Apache::lonnavmaps;
use Apache::longroup;
use Apache::lonrss;
use LONCAPA qw(:DEFAULT :match);

# --- Caches local to lonparmset

my $parmhashid;
my %parmhash;
my $symbsid;
my %symbs;
my $rulesid;
my %rules;

# --- end local caches

##################################################
##################################################

=pod

=item parmval

Figure out a cascading parameter.

Inputs:  $what - a parameter spec (incluse part info and name I.E. 0.weight)
         $id   - a bighash Id number
         $def  - the resource's default value   'stupid emacs

Returns:  A list, the first item is the index into the remaining list of items of parm valuse that is the active one, the list consists of parm values at the 14 possible levels

14- General Course
13- Map or Folder level in course
12- resource default
11- map default
10- resource level in course
9 - General for section
8 - Map or Folder level for section
7 - resource level in section
6 - General for group
5 - Map or Folder level for group
4 - resource level in group
3 - General for specific student
2 - Map or Folder level for specific student
1 - resource level for specific student

=cut

##################################################
sub parmval {
    my ($what,$id,$def,$uname,$udom,$csec,$cgroup,$courseopt)=@_;
    return &parmval_by_symb($what,&symbcache($id),$def,$uname,$udom,$csec,
                                                           $cgroup,$courseopt);
}

sub parmval_by_symb {
    my ($what,$symb,$def,$uname,$udom,$csec,$cgroup,$courseopt)=@_;
# load caches
    &cacheparmhash();

    my $useropt;
    if ($uname ne '' && $udom ne '') {
	$useropt = &Apache::lonnet::get_userresdata($uname,$udom);
    }

    my $result='';
    my @outpar=();
# ----------------------------------------------------- Cascading lookup scheme
    my $map=(&Apache::lonnet::decode_symb($symb))[0];    
    $map = &Apache::lonnet::deversion($map);

    my $symbparm=$symb.'.'.$what;
    my $mapparm=$map.'___(all).'.$what;

    my $grplevel=$env{'request.course.id'}.'.['.$cgroup.'].'.$what;
    my $grplevelr=$env{'request.course.id'}.'.['.$cgroup.'].'.$symbparm;
    my $grplevelm=$env{'request.course.id'}.'.['.$cgroup.'].'.$mapparm;

    my $seclevel=$env{'request.course.id'}.'.['.$csec.'].'.$what;
    my $seclevelr=$env{'request.course.id'}.'.['.$csec.'].'.$symbparm;
    my $seclevelm=$env{'request.course.id'}.'.['.$csec.'].'.$mapparm;

    my $courselevel=$env{'request.course.id'}.'.'.$what;
    my $courselevelr=$env{'request.course.id'}.'.'.$symbparm;
    my $courselevelm=$env{'request.course.id'}.'.'.$mapparm;


# --------------------------------------------------------- first, check course

    if (defined($$courseopt{$courselevel})) {
	$outpar[14]=$$courseopt{$courselevel};
	$result=14;
    }

    if (defined($$courseopt{$courselevelm})) {
	$outpar[13]=$$courseopt{$courselevelm};
	$result=13;
    }

# ------------------------------------------------------- second, check default

    if (defined($def)) { $outpar[12]=$def; $result=12; }

# ------------------------------------------------------ third, check map parms

    my $thisparm=$parmhash{$symbparm};
    if (defined($thisparm)) { $outpar[11]=$thisparm; $result=11; }

    if (defined($$courseopt{$courselevelr})) {
	$outpar[10]=$$courseopt{$courselevelr};
	$result=10;
    }

# ------------------------------------------------------ fourth, back to course
    if ($csec ne '') {
        if (defined($$courseopt{$seclevel})) {
	    $outpar[9]=$$courseopt{$seclevel};
	    $result=9;
	}
        if (defined($$courseopt{$seclevelm})) {
	    $outpar[8]=$$courseopt{$seclevelm};
	    $result=8;
	}

        if (defined($$courseopt{$seclevelr})) {
	    $outpar[7]=$$courseopt{$seclevelr};
	    $result=7;
	}
    }
# ------------------------------------------------------ fifth, check course group
    if ($cgroup ne '') {
        if (defined($$courseopt{$grplevel})) {
            $outpar[6]=$$courseopt{$grplevel};
            $result=6;
        }
        if (defined($$courseopt{$grplevelm})) {
            $outpar[5]=$$courseopt{$grplevelm};
            $result=5;
        }
        if (defined($$courseopt{$grplevelr})) {
            $outpar[4]=$$courseopt{$grplevelr};
            $result=4;
        }
    }

# ---------------------------------------------------------- fifth, check user

    if ($uname ne '') {
	if (defined($$useropt{$courselevel})) {
	    $outpar[3]=$$useropt{$courselevel};
	    $result=3;
	}

	if (defined($$useropt{$courselevelm})) {
	    $outpar[2]=$$useropt{$courselevelm};
	    $result=2;
	}

	if (defined($$useropt{$courselevelr})) {
	    $outpar[1]=$$useropt{$courselevelr};
	    $result=1;
	}
    }
    return ($result,@outpar);
}

sub resetparmhash {
    $parmhashid='';
}

sub cacheparmhash {
    if ($parmhashid eq  $env{'request.course.fn'}) { return; }
    my %parmhashfile;
    if (tie(%parmhashfile,'GDBM_File',
	      $env{'request.course.fn'}.'_parms.db',&GDBM_READER(),0640)) {
	%parmhash=%parmhashfile;
	untie %parmhashfile;
	$parmhashid=$env{'request.course.fn'};
    }
}

sub resetsymbcache {
    $symbsid='';
}

sub symbcache {
    my $id=shift;
    if ($symbsid ne $env{'request.course.id'}) {
	%symbs=();
    }
    unless ($symbs{$id}) {
	my $navmap = Apache::lonnavmaps::navmap->new();
	if ($id=~/\./) {
	    my $resource=$navmap->getById($id);
	    $symbs{$id}=$resource->symb();
	} else {
	    my $resource=$navmap->getByMapPc($id);
	    $symbs{$id}=&Apache::lonnet::declutter($resource->src());
	}
	$symbsid=$env{'request.course.id'};
    }
    return $symbs{$id};
}

sub resetrulescache {
    $rulesid='';
}

sub rulescache {
    my $id=shift;
    if ($rulesid ne $env{'request.course.id'}
	&& !defined($rules{$id})) {
	my $dom = $env{'course.'.$env{'request.course.id'}.'.domain'};
	my $crs = $env{'course.'.$env{'request.course.id'}.'.num'};
	%rules=&Apache::lonnet::dump('parmdefactions',$dom,$crs);
	$rulesid=$env{'request.course.id'};
    }
    return $rules{$id};
}

sub preset_defaults {
    my $type=shift;
    if (&rulescache($type.'_action') eq 'default') {
# yes, there is something
	return (&rulescache($type.'_hours'),
		&rulescache($type.'_min'),
		&rulescache($type.'_sec'),
		&rulescache($type.'_value'));
    } else {
# nothing there or something else
	return ('','','','','');
    }
}

##################################################

sub date_sanity_info {
   my $checkdate=shift;
   unless ($checkdate) { return ''; }
   my $result='';
   my $crsprefix='course.'.$env{'request.course.id'}.'.';
   if ($env{$crsprefix.'default_enrollment_end_date'}) {
      if ($checkdate>$env{$crsprefix.'default_enrollment_end_date'}) {
         $result.='<br />'.&mt('After course enrollment end!');
      }
   }
   if ($env{$crsprefix.'default_enrollment_start_date'}) {
      if ($checkdate<$env{$crsprefix.'default_enrollment_start_date'}) {
         $result.='<br />'.&mt('Before course enrollment start!');
      }
   }
   return $result;
}
##################################################
##################################################
#
# Store a parameter by ID
#
# Takes
# - resource id
# - name of parameter
# - level
# - new value
# - new type
# - username
# - userdomain

sub storeparm {
    my ($sresid,$spnam,$snum,$nval,$ntype,$uname,$udom,$csec,$cgroup)=@_;
    &storeparm_by_symb(&symbcache($sresid),$spnam,$snum,$nval,$ntype,$uname,$udom,$csec,'',$cgroup);
}

#
# Store a parameter by symb
#
# Takes
# - symb
# - name of parameter
# - level
# - new value
# - new type
# - username
# - userdomain

my %recstack;
sub storeparm_by_symb {
    my ($symb,$spnam,$snum,$nval,$ntype,$uname,$udom,$csec,$recflag,$cgroup)=@_;
    unless ($recflag) {
# first time call
	%recstack=();
	$recflag=1;
    }
# store parameter
    &storeparm_by_symb_inner
	($symb,$spnam,$snum,$nval,$ntype,$uname,$udom,$csec,$cgroup);
# don't do anything if parameter was reset
    unless ($nval) { return; }
    my ($prefix,$parm)=($spnam=~/^(.*[\_\.])([^\_\.]+)$/);
# remember that this was set
    $recstack{$parm}=1;
# what does this trigger?
    foreach my $triggered (split(/\:/,&rulescache($parm.'_triggers'))) {
# don't backfire
       unless ((!$triggered) || ($recstack{$triggered})) {
	   my $action=&rulescache($triggered.'_action');
	   my ($whichaction,$whichparm)=($action=~/^(.*\_)([^\_]+)$/);
# set triggered parameter on same level
	   my $newspnam=$prefix.$triggered;
	   my $newvalue='';
	   my $active=1;
	   if ($action=~/^when\_setting/) {
# are there restrictions?
	       if (&rulescache($triggered.'_triggervalue')=~/\w/) {
		   $active=0;
		   foreach my $possiblevalue (split(/\s*\,\s*/,&rulescache($triggered.'_triggervalue'))) {
		       if (lc($possiblevalue) eq lc($nval)) { $active=1; }
		   }
	       }
	       $newvalue=&rulescache($triggered.'_value');
	   } else {
	       my $totalsecs=((&rulescache($triggered.'_days')*24+&rulescache($triggered.'_hours'))*60+&rulescache($triggered.'_min'))*60+&rulescache($triggered.'_sec');
	       if ($action=~/^later\_than/) {
		   $newvalue=$nval+$totalsecs;
	       } else {
		   $newvalue=$nval-$totalsecs;
	       }
	   }
	   if ($active) {
	       &storeparm_by_symb($symb,$newspnam,$snum,$newvalue,&rulescache($triggered.'_type'),
				   $uname,$udom,$csec,$recflag,$cgroup);
	   }
       }
    }
    return '';
}

sub log_parmset {
    return &Apache::lonnet::instructor_log('parameterlog',@_);
}

sub storeparm_by_symb_inner {
# ---------------------------------------------------------- Get symb, map, etc
    my ($symb,$spnam,$snum,$nval,$ntype,$uname,$udom,$csec,$cgroup)=@_;
# ---------------------------------------------------------- Construct prefixes
    $spnam=~s/\_([^\_]+)$/\.$1/;
    my $map=(&Apache::lonnet::decode_symb($symb))[0];    
    $map = &Apache::lonnet::deversion($map);

    my $symbparm=$symb.'.'.$spnam;
    my $mapparm=$map.'___(all).'.$spnam;

    my $grplevel=$env{'request.course.id'}.'.['.$cgroup.'].'.$spnam;
    my $grplevelr=$env{'request.course.id'}.'.['.$cgroup.'].'.$symbparm;
    my $grplevelm=$env{'request.course.id'}.'.['.$cgroup.'].'.$mapparm;

    my $seclevel=$env{'request.course.id'}.'.['.$csec.'].'.$spnam;
    my $seclevelr=$env{'request.course.id'}.'.['.$csec.'].'.$symbparm;
    my $seclevelm=$env{'request.course.id'}.'.['.$csec.'].'.$mapparm;
    
    my $courselevel=$env{'request.course.id'}.'.'.$spnam;
    my $courselevelr=$env{'request.course.id'}.'.'.$symbparm;
    my $courselevelm=$env{'request.course.id'}.'.'.$mapparm;
    
    my $storeunder='';
    if (($snum==14) || ($snum==3)) { $storeunder=$courselevel; }
    if (($snum==13) || ($snum==2)) { $storeunder=$courselevelm; }
    if (($snum==10) || ($snum==1)) { $storeunder=$courselevelr; }
    if ($snum==9) { $storeunder=$seclevel; }
    if ($snum==8) { $storeunder=$seclevelm; }
    if ($snum==7) { $storeunder=$seclevelr; }
    if ($snum==6) { $storeunder=$grplevel; }
    if ($snum==5) { $storeunder=$grplevelm; }
    if ($snum==4) { $storeunder=$grplevelr; }

    
    my $delete;
    if ($nval eq '') { $delete=1;}
    my %storecontent = ($storeunder         => $nval,
			$storeunder.'.type' => $ntype);
    my $reply='';
    if ($snum>3) {
# ---------------------------------------------------------------- Store Course
#
	my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
	my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
# Expire sheets
	&Apache::lonnet::expirespread('','','studentcalc');
	if (($snum==10) || ($snum==7) || ($snum==4)) {
	    &Apache::lonnet::expirespread('','','assesscalc',$symb);
	} elsif (($snum==11) || ($snum==8) || ($snum==5)) {
	    &Apache::lonnet::expirespread('','','assesscalc',$map);
	} else {
	    &Apache::lonnet::expirespread('','','assesscalc');
	}
# Store parameter
	if ($delete) {
	    $reply=&Apache::lonnet::del
		('resourcedata',[keys(%storecontent)],$cdom,$cnum);
            &log_parmset(\%storecontent,1);
	} else {
	    $reply=&Apache::lonnet::cput
		('resourcedata',\%storecontent,$cdom,$cnum);
	    &log_parmset(\%storecontent);
	}
	&Apache::lonnet::devalidatecourseresdata($cnum,$cdom);
    } else {
# ------------------------------------------------------------------ Store User
#
# Expire sheets
	&Apache::lonnet::expirespread($uname,$udom,'studentcalc');
	if ($snum==1) {
	    &Apache::lonnet::expirespread
		($uname,$udom,'assesscalc',$symb);
	} elsif ($snum==2) {
	    &Apache::lonnet::expirespread
		($uname,$udom,'assesscalc',$map);
	} else {
	    &Apache::lonnet::expirespread($uname,$udom,'assesscalc');
	}
# Store parameter
	if ($delete) {
	    $reply=&Apache::lonnet::del
		('resourcedata',[keys(%storecontent)],$udom,$uname);
	    &log_parmset(\%storecontent,1,$uname,$udom);
	} else {
	    $reply=&Apache::lonnet::cput
		('resourcedata',\%storecontent,$udom,$uname);
	    &log_parmset(\%storecontent,0,$uname,$udom);
	}
	&Apache::lonnet::devalidateuserresdata($uname,$udom);
    }
    
    if ($reply=~/^error\:(.*)/) {
	return "<span class=\"LC_error\">Write Error: $1</span>";
    }
    return '';
}

##################################################
##################################################

=pod

=item valout

Format a value for output.

Inputs:  $value, $type, $editable

Returns: $value, formatted for output.  If $type indicates it is a date,
localtime($value) is returned.
$editable will return an icon to click on

=cut

##################################################
##################################################
sub valout {
    my ($value,$type,$editable)=@_;
    my $result = '';
    # Values of zero are valid.
    if (! $value && $value ne '0') {
	if ($editable) {
	    $result = '<span class="LC_clickhere">*</span>';
	} else {
	    $result='&nbsp;';
	}
    } else {
        if ($type eq 'date_interval') {
            my ($sec,$min,$hour,$mday,$mon,$year)=gmtime($value);
            $year=$year-70;
            $mday--;
            if ($year) {
		$result.=$year.' yrs ';
            }
            if ($mon) {
		$result.=$mon.' mths ';
            }
            if ($mday) {
		$result.=$mday.' days ';
            }
            if ($hour) {
		$result.=$hour.' hrs ';
            }
            if ($min) {
		$result.=$min.' mins ';
            }
            if ($sec) {
		$result.=$sec.' secs ';
            }
            $result=~s/\s+$//;
        } elsif (&isdateparm($type)) {
            $result = localtime($value).&date_sanity_info($value);
        } else {
            $result = $value;
        }
    }
    return $result;
}

##################################################
##################################################

=pod

=item plink

Produces a link anchor.

Inputs: $type,$dis,$value,$marker,$return,$call

Returns: scalar with html code for a link which will envoke the 
javascript function 'pjump'.

=cut

##################################################
##################################################
sub plink {
    my ($type,$dis,$value,$marker,$return,$call)=@_;
    my $winvalue=$value;
    unless ($winvalue) {
	if (&isdateparm($type)) {
            $winvalue=$env{'form.recent_'.$type};
        } else {
            $winvalue=$env{'form.recent_'.(split(/\_/,$type))[0]};
        }
    }
    my ($parmname)=((split(/\&/,$marker))[1]=~/\_([^\_]+)$/);
    my ($hour,$min,$sec,$val)=&preset_defaults($parmname);
    unless (defined($winvalue)) { $winvalue=$val; }
    return '<table width="100%"><tr valign="top" align="right"><td><a name="'.$marker.'" /></td></tr><tr><td align="center">'.
	'<a href="javascript:pjump('."'".$type."','".$dis."','".$winvalue."','"
	    .$marker."','".$return."','".$call."','".$hour."','".$min."','".$sec."'".');">'.
		&valout($value,$type,1).'</a></td></tr></table>';
}

sub page_js {

    my $selscript=&Apache::loncommon::studentbrowser_javascript();
    my $pjump_def = &Apache::lonhtmlcommon::pjump_javascript_definition();

    return(<<ENDJS);
<script type="text/javascript">

    function pclose() {
        parmwin=window.open("/adm/rat/empty.html","LONCAPAparms",
                 "height=350,width=350,scrollbars=no,menubar=no");
        parmwin.close();
    }

    $pjump_def

    function psub() {
        pclose();
        if (document.parmform.pres_marker.value!='') {
            document.parmform.action+='#'+document.parmform.pres_marker.value;
            var typedef=new Array();
            typedef=document.parmform.pres_type.value.split('_');
           if (document.parmform.pres_type.value!='') {
            if (typedef[0]=='date') {
                eval('document.parmform.recent_'+
                     document.parmform.pres_type.value+
		     '.value=document.parmform.pres_value.value;');
            } else {
                eval('document.parmform.recent_'+typedef[0]+
		     '.value=document.parmform.pres_value.value;');
            }
	   }
            document.parmform.submit();
        } else {
            document.parmform.pres_value.value='';
            document.parmform.pres_marker.value='';
        }
    }

    function openWindow(url, wdwName, w, h, toolbar,scrollbar) {
        var options = "width=" + w + ",height=" + h + ",";
        options += "resizable=yes,scrollbars="+scrollbar+",status=no,";
        options += "menubar=no,toolbar="+toolbar+",location=no,directories=no";
        var newWin = window.open(url, wdwName, options);
        newWin.focus();
    }
</script>
$selscript
ENDJS

}
sub startpage {
    my ($r) = @_;

    my %loaditems = ('onunload' => "pclose()",
		     'onload'   => "group_or_section('cgroup')",);

    my $start_page = 
	&Apache::loncommon::start_page('Set/Modify Course Parameters',
				       &page_js(),
				       {'add_entries' => \%loaditems,});
    my $breadcrumbs = 
	&Apache::lonhtmlcommon::breadcrumbs('Table Mode Parameter Setting','Table_Mode');
    $r->print(<<ENDHEAD);
$start_page
$breadcrumbs
<form method="post" action="/adm/parmset?action=settable" name="parmform">
<input type="hidden" value='' name="pres_value" />
<input type="hidden" value='' name="pres_type" />
<input type="hidden" value='' name="pres_marker" />
<input type="hidden" value='1' name="prevvisit" />
ENDHEAD
}


sub print_row {
    my ($r,$which,$part,$name,$symbp,$rid,$default,$defaulttype,$display,$defbgone,
	$defbgtwo,$defbgthree,$parmlev,$uname,$udom,$csec,$cgroup,$usersgroups)=@_;
    my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
    my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $courseopt=&Apache::lonnet::get_courseresdata($cnum,$cdom);
# get the values for the parameter in cascading order
# empty levels will remain empty
    my ($result,@outpar)=&parmval($$part{$which}.'.'.$$name{$which},
	  $rid,$$default{$which},$uname,$udom,$csec,$cgroup,$courseopt);
# get the type for the parameters
# problem: these may not be set for all levels
    my ($typeresult,@typeoutpar)=&parmval($$part{$which}.'.'.
                                          $$name{$which}.'.type',$rid,
		 $$defaulttype{$which},$uname,$udom,$csec,$cgroup,$courseopt);
# cascade down manually
    my $cascadetype=$$defaulttype{$which};
    for (my $i=14;$i>0;$i--) {
	 if ($typeoutpar[$i]) { 
            $cascadetype=$typeoutpar[$i];
	} else {
            $typeoutpar[$i]=$cascadetype;
        }
    }
    my $parm=$$display{$which};

    if ($parmlev eq 'full') {
        $r->print('<td bgcolor='.$defbgtwo.' align="center">'
                  .$$part{$which}.'</td>');
    } else {    
        $parm=~s|\[.*\]\s||g;
    }
    my $automatic=&rulescache(($which=~/\_([^\_]+)$/)[0].'_triggers');
    if ($automatic) {
	$parm.='<span class="LC_warning"><br />'.&mt('Automatically sets').' '.join(', ',split(/\:/,$automatic)).'</span>';
    }
    $r->print('<td bgcolor='.$defbgone.'>'.$parm.'</td>');
   
    my $thismarker=$which;
    $thismarker=~s/^parameter\_//;
    my $mprefix=$rid.'&'.$thismarker.'&';
    my $effective_parm = &valout($outpar[$result],$typeoutpar[$result]);
    my ($othergrp,$grp_parm,$controlgrp);

    if ($parmlev eq 'general') {

        if ($uname) {
            &print_td($r,3,$defbgone,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
        } elsif ($cgroup) {
            &print_td($r,6,$defbgthree,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
        } elsif ($csec) {
            &print_td($r,9,$defbgtwo,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display); 
        } else {
            &print_td($r,14,$defbgone,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display); 
        }
    } elsif ($parmlev eq 'map') {

        if ($uname) {
            &print_td($r,2,$defbgone,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
        } elsif ($cgroup) {
            &print_td($r,5,$defbgthree,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
        } elsif ($csec) {
            &print_td($r,8,$defbgtwo,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
        } else {
            &print_td($r,13,$defbgone,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
        }
    } else {
        if ($uname) {
            if (@{$usersgroups} > 1) {
                my ($coursereply,$grp_parm,$controlgrp);
                ($coursereply,$othergrp,$grp_parm,$controlgrp) =
                    &print_usergroups($r,$$part{$which}.'.'.$$name{$which},
                       $rid,$cgroup,$defbgone,$usersgroups,$result,$courseopt);
                if ($coursereply && $result > 3) {
                    if (defined($controlgrp)) {
                        if ($cgroup ne $controlgrp) {
                            $effective_parm = $grp_parm;
                            $result = 0;
                        }
                    }
                }
            }
        }

        &print_td($r,14,$defbgone,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);

	&print_td($r,13,$defbgone,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
	&print_td($r,12,'#FFDDDD',$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
	&print_td($r,11,'#FFDDDD',$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
	&print_td($r,10,$defbgone,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
	
	if ($csec) {
	    &print_td($r,9,$defbgtwo,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
	    &print_td($r,8,$defbgtwo,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
	    &print_td($r,7,$defbgtwo,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
	}

        if ($cgroup) {
            &print_td($r,6,$defbgthree,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
            &print_td($r,5,$defbgthree,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
            &print_td($r,4,$defbgthree,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
        }
     
	if ($uname) {
            if ($othergrp) {
                $r->print($othergrp);
            }
	    &print_td($r,3,$defbgone,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
	    &print_td($r,2,$defbgone,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
	    &print_td($r,1,$defbgone,$result,\@outpar,$mprefix,$_,\@typeoutpar,$display);
	}

    } # end of $parmlev if/else
    $r->print('<td bgcolor="#CCCCFF" align="center">'.$effective_parm.'</td>');

    if ($parmlev eq 'full') {
        my $sessionval=&Apache::lonnet::EXT('resource.'.$$part{$which}.
                                        '.'.$$name{$which},$$symbp{$rid});
        my $sessionvaltype=$typeoutpar[$result];
        if (!defined($sessionvaltype)) { $sessionvaltype=$$defaulttype{$which}; }
        $r->print('<td bgcolor="#999999" align="center"><font color="#FFFFFF">'.
                  &valout($sessionval,$sessionvaltype).'&nbsp;'.
                  '</font></td>');
    }
    $r->print('</tr>');
    $r->print("\n");
}

sub print_td {
    my ($r,$which,$defbg,$result,$outpar,$mprefix,$value,$typeoutpar,$display)=@_;
    $r->print('<td bgcolor='.(($result==$which)?'"#AAFFAA"':$defbg).
              ' align="center">');
    if ($which<11 || $which > 12) {
	$r->print(&plink($$typeoutpar[$which],
			 $$display{$value},$$outpar[$which],
			 $mprefix."$which",'parmform.pres','psub'));
    } else {
	$r->print(&valout($$outpar[$which],$$typeoutpar[$which]));
    }
    $r->print('</td>'."\n");
}

sub print_usergroups {
    my ($r,$what,$rid,$cgroup,$defbg,$usersgroups,$result,$courseopt) = @_;
    my $courseid = $env{'request.course.id'};
    my $output;
    my $symb = &symbcache($rid);
    my $symbparm=$symb.'.'.$what;
    my $map=(&Apache::lonnet::decode_symb($symb))[0];
    my $mapparm=$map.'___(all).'.$what;
    my ($coursereply,$resultitem,$resultgroup,$resultlevel,$resulttype) =
          &parm_control_group($courseid,$usersgroups,$symbparm,$mapparm,$what,
                                                                   $courseopt);
    my $bgcolor = $defbg;
    my $grp_parm;
    if (($coursereply) && ($cgroup ne $resultgroup)) { 
        if ($result > 3) {
            $bgcolor = '"#AAFFAA"';
            $grp_parm = &valout($coursereply,$resulttype);
        }
        $grp_parm = &valout($coursereply,$resulttype);
        $output = '<td bgcolor='.$bgcolor.' align="center">';
        if ($resultgroup && $resultlevel) {
            $output .= '<small><b>'.$resultgroup.'</b> ('.$resultlevel.'): </small>'.$grp_parm;
        } else {
            $output .= '&nbsp;';
        }
        $output .= '</td>';
    } else {
        $output .= '<td bgcolor='.$bgcolor.'>&nbsp;</td>';
    }
    return ($coursereply,$output,$grp_parm,$resultgroup);
}

sub parm_control_group {
    my ($courseid,$usersgroups,$symbparm,$mapparm,$what,$courseopt) = @_;
    my ($coursereply,$resultitem,$resultgroup,$resultlevel,$resulttype);
    my $grpfound = 0;
    my @levels = ($symbparm,$mapparm,$what);
    my @levelnames = ('resource','map/folder','general');
    foreach my $group (@{$usersgroups}) {
        if ($grpfound) { last; }
        for (my $i=0; $i<@levels; $i++) {
            my $item = $courseid.'.['.$group.'].'.$levels[$i];
            if (defined($$courseopt{$item})) {
                $coursereply = $$courseopt{$item};
                $resultitem = $item;
                $resultgroup = $group;
                $resultlevel = $levelnames[$i];
                $resulttype = $$courseopt{$item.'.type'};
                $grpfound = 1;
                last;
            }
        }
    }
    return($coursereply,$resultitem,$resultgroup,$resultlevel,$resulttype);
}

=pod

=item B<extractResourceInformation>: Given the course data hash, extractResourceInformation extracts lots of information about the course's resources into a variety of hashes.

Input: See list below:

=over 4

=item B<ids>: An array that will contain all of the ids in the course.

=item B<typep>: hash, id->type, where "type" contains the extension of the file, thus, I<problem exam quiz assess survey form>.

=item B<keyp>: hash, id->key list, will contain a comma separated list of the meta-data keys available for the given id

=item B<allparms>: hash, name of parameter->display value (what is the display value?)

=item B<allparts>: hash, part identification->text representation of part, where the text representation is "[Part $part]"

=item B<allkeys>: hash, full key to part->display value (what's display value?)

=item B<allmaps>: hash, ???

=item B<fcat>: ???

=item B<defp>: hash, ???

=item B<mapp>: ??

=item B<symbp>: hash, id->full sym?

=back

=cut

sub extractResourceInformation {
    my $ids = shift;
    my $typep = shift;
    my $keyp = shift;
    my $allparms = shift;
    my $allparts = shift;
    my $allmaps = shift;
    my $mapp = shift;
    my $symbp = shift;
    my $maptitles=shift;
    my $uris=shift;
    my $keyorder=shift;
    my $defkeytype=shift;

    my $keyordercnt=100;

    my $navmap = Apache::lonnavmaps::navmap->new();
    my @allres=$navmap->retrieveResources(undef,undef,1,undef,1);
    foreach my $resource (@allres) {
	my $id=$resource->id();
        my ($mapid,$resid)=split(/\./,$id);
	if ($mapid eq '0') { next; }
	$$ids[$#$ids+1]=$id;
	my $srcf=$resource->src();
	$srcf=~/\.(\w+)$/;
	$$typep{$id}=$1;
	$$keyp{$id}='';
        $$uris{$id}=$srcf;
	foreach (split(/\,/,&Apache::lonnet::metadata($srcf,'allpossiblekeys'))) {
	    if ($_=~/^parameter\_(.*)/) {
		my $key=$_;
# Hidden parameters
		if (&Apache::lonnet::metadata($srcf,$key.'.hidden') eq 'parm') {
		    next;
		}
		my $display= &Apache::lonnet::metadata($srcf,$key.'.display');
		my $name=&Apache::lonnet::metadata($srcf,$key.'.name');
		my $part= &Apache::lonnet::metadata($srcf,$key.'.part');
#
# allparms is a hash of parameter names
#
		my $parmdis = $display;
		$parmdis =~ s/\[Part.*$//g;
                $$allparms{$name}=$parmdis;
		$$defkeytype{$name}=&Apache::lonnet::metadata($srcf,$key.'.type');
#
# allparts is a hash of all parts
#
		$$allparts{$part} = "Part: $part";
#
# Remember all keys going with this resource
#
		if ($$keyp{$id}) {
		    $$keyp{$id}.=','.$key;
		} else {
		    $$keyp{$id}=$key;
		}
#
# Put in order
# 
                unless ($$keyorder{$key}) {
                    $$keyorder{$key}=$keyordercnt;
                    $keyordercnt++;
		}

	    }
	}
	$$mapp{$id}=
	    &Apache::lonnet::declutter($resource->enclosing_map_src());
	$$mapp{$mapid}=$$mapp{$id};
	$$allmaps{$mapid}=$$mapp{$id};
	if ($mapid eq '1') {
	    $$maptitles{$mapid}='Main Course Documents';
	} else {
	    $$maptitles{$mapid}=&Apache::lonnet::gettitle(&Apache::lonnet::clutter($$mapp{$id}));
	}
	$$maptitles{$$mapp{$id}}=$$maptitles{$mapid};
	$$symbp{$id}=&Apache::lonnet::encode_symb($$mapp{$id},$resid,$srcf);
	$$symbp{$mapid}=$$mapp{$id}.'___(all)';
    }
}


##################################################
##################################################

sub isdateparm {
    my $type=shift;
    return (($type=~/^date/) && (!($type eq 'date_interval')));
}

sub parmmenu {
    my ($r,$allparms,$pscat,$keyorder)=@_;
    my $tempkey;
    $r->print(<<ENDSCRIPT);
<script type="text/javascript">
    function checkall(value, checkName) {
	for (i=0; i<document.forms.parmform.elements.length; i++) {
            ele = document.forms.parmform.elements[i];
            if (ele.name == checkName) {
                document.forms.parmform.elements[i].checked=value;
            }
        }
    }

    function checkthis(thisvalue, checkName) {
	for (i=0; i<document.forms.parmform.elements.length; i++) {
            ele = document.forms.parmform.elements[i];
            if (ele.name == checkName) {
		if (ele.value == thisvalue) {
		    document.forms.parmform.elements[i].checked=true;
		}
            }
        }
    }

    function checkdates() {
	checkthis('duedate','pscat');
 	checkthis('opendate','pscat');
	checkthis('answerdate','pscat');
    }

    function checkdisset() {
	checkthis('discussend','pscat');
 	checkthis('discusshide','pscat');
    }

    function checkcontdates() {
	checkthis('contentopen','pscat');
 	checkthis('contentclose','pscat');
    }
 

    function checkvisi() {
	checkthis('hiddenresource','pscat');
 	checkthis('encrypturl','pscat');
	checkthis('problemstatus','pscat');
	checkthis('contentopen','pscat');
	checkthis('opendate','pscat');
    }

    function checkparts() {
	checkthis('hiddenparts','pscat');
	checkthis('display','pscat');
	checkthis('ordered','pscat');
    }

    function checkstandard() {
        checkall(false,'pscat');
	checkdates();
	checkthis('weight','pscat');
	checkthis('maxtries','pscat');
    }

</script>
ENDSCRIPT
    $r->print();
    $r->print("\n<table id=\"LC_parm_overview_parm_menu\"><tr>");
    my $cnt=0;
    foreach $tempkey (&keysindisplayorder($allparms,$keyorder)) {
	$r->print("\n<td><label><input type='checkbox' name='pscat' ");
	$r->print('value="'.$tempkey.'"');
	if ($$pscat[0] eq "all" || grep $_ eq $tempkey, @{$pscat}) {
	    $r->print(' checked');
	}
	$r->print('>'.$$allparms{$tempkey}.'</label></td>');
 	$cnt++;
        if ($cnt==3) {
	    $r->print("</tr>\n<tr>");
	    $cnt=0;
	}
    }
    $r->print('
</tr><tr id=\"LC_parm_overview_parm_menu_selectors\"><td>
<a href="javascript:checkall(true, \'pscat\')">Select&nbsp;All</a><br />
<a href="javascript:checkstandard()">Select&nbsp;Common&nbsp;Only</a>
</td><td>
<a href="javascript:checkdates()">Add&nbsp;Problem&nbsp;Dates</a>
<a href="javascript:checkcontdates()">Add&nbsp;Content&nbsp;Dates</a><br />
<a href="javascript:checkdisset()">Add&nbsp;Discussion&nbsp;Settings</a>
<a href="javascript:checkvisi()">Add&nbsp;Visibilities</a><br />
<a href="javascript:checkparts()">Add&nbsp;Part&nbsp;Parameters</a>
</td><td>
<a href="javascript:checkall(false, \'pscat\')">Unselect&nbsp;All</a>
</td>
');
    $r->print('</tr></table>');
}

sub partmenu {
    my ($r,$allparts,$psprt)=@_;
    $r->print('<select multiple name="psprt" size="8">');
    $r->print('<option value="all"');
    $r->print(' selected') unless (@{$psprt});
    $r->print('>'.&mt('All Parts').'</option>');
    my %temphash=();
    foreach (@{$psprt}) { $temphash{$_}=1; }
    foreach my $tempkey (sort {
	if ($a==$b) { return ($a cmp $b) } else { return ($a <=> $b); }
    } keys(%{$allparts})) {
	unless ($tempkey =~ /\./) {
	    $r->print('<option value="'.$tempkey.'"');
	    if ($$psprt[0] eq "all" ||  $temphash{$tempkey}) {
		$r->print(' selected');
	    }
	    $r->print('>'.$$allparts{$tempkey}.'</option>');
	}
    }
    $r->print('</select>');
}

sub usermenu {
    my ($r,$uname,$id,$udom,$csec,$cgroup,$parmlev,$usersgroups)=@_;
    my $chooseopt=&Apache::loncommon::select_dom_form($udom,'udom').' '.
        &Apache::loncommon::selectstudent_link('parmform','uname','udom');
    my $selscript=&Apache::loncommon::studentbrowser_javascript();
    my %lt=&Apache::lonlocal::texthash(
		    'se'    => "Section",
                    'gr'    => "Group",
		    'fu'    => "For User",
		    'oi'    => "or ID",
		    'ad'    => "at Domain"
				       );
    my $sections='';
    my %sectionhash = &Apache::loncommon::get_sections();

    my $groups;
    my %grouphash = &Apache::longroup::coursegroups();

    if (%sectionhash) {
        $sections=$lt{'se'}.': <select name="csec"';
        if (%grouphash && $parmlev ne 'full') {
            $sections .= qq| onchange="group_or_section('csec')" |;
        }
        $sections .= '>';
	foreach my $section ('',sort keys %sectionhash) {
	    $sections.='<option value="'.$section.'" '.
		($section eq $csec?'selected="selected"':'').'>'.$section.
                                                              '</option>';
        }
        $sections.='</select>';
    }
    if (%sectionhash && %grouphash && $parmlev ne 'full') {
        $sections .= '&nbsp;or&nbsp;';
        $sections .= qq|
<script type="text/javascript">
function group_or_section(caller) {
   if (caller == "cgroup") {
       if (document.parmform.cgroup.selectedIndex != 0) {
           document.parmform.csec.selectedIndex = 0;
       }
   } else {
       if (document.parmform.csec.selectedIndex != 0) {
           document.parmform.cgroup.selectedIndex = 0;
       }
   }
}
</script>
|;
    } else {
        $sections .= qq|
<script type="text/javascript">
function group_or_section(caller) {
    return;
}
</script>
|;
    } 

    if (%grouphash) {
        $groups=$lt{'gr'}.': <select name="cgroup"';
        if (%sectionhash && $env{'form.action'} eq 'settable') {
            $groups .= qq| onchange="group_or_section('cgroup')" |;
        }
        $groups .= '>';
        foreach my $grp ('',sort keys %grouphash) {
            $groups.='<option value="'.$grp.'" ';
            if ($grp eq $cgroup) {
                unless ((defined($uname)) && ($grp eq '')) {
                    $groups .=  'selected="selected" ';
                }
            } elsif (!defined($cgroup)) {
                if (@{$usersgroups} == 1) {
                    if ($grp eq $$usersgroups[0]) {
                        $groups .=  'selected="selected" ';
                    }
                }
            }
            $groups .= '>'.$grp.'</option>';
        }
        $groups.='</select>';
    }
    $r->print(<<ENDMENU);
<b>
$sections
$groups
<br />
$lt{'fu'} 
<input type="text" value="$uname" size="12" name="uname" />
$lt{'oi'}
<input type="text" value="$id" size="12" name="id" /> 
$lt{'ad'}
$chooseopt
</b>
ENDMENU
}

sub displaymenu {
    my ($r,$allparms,$allparts,$pscat,$psprt,$keyorder)=@_;
    $r->print('<table border="1"><tr><th>'.&mt('Select Parameters to View').'</th><th>'.
	     &mt('Select Parts to View').'</th></tr><tr><td>');  
    &parmmenu($r,$allparms,$pscat,$keyorder);
    $r->print('</td><td>');
    &partmenu($r,$allparts,$psprt);
    $r->print('</td></tr></table>');
}

sub mapmenu {
    my ($r,$allmaps,$pschp,$maptitles)=@_;
    $r->print('<b>'.&mt('Select Enclosing Map or Folder').'</b> ');
    $r->print('<select name="pschp">');
    $r->print('<option value="all">'.&mt('All Maps or Folders').'</option>');
    foreach (sort {$$allmaps{$a} cmp $$allmaps{$b}} keys %{$allmaps}) {
	$r->print('<option value="'.$_.'"');
	if (($pschp eq $_)) { $r->print(' selected'); }
	$r->print('>'.$$maptitles{$_}.($$allmaps{$_}!~/^uploaded/?' ['.$$allmaps{$_}.']':'').'</option>');
    }
    $r->print("</select>");
}

sub levelmenu {
    my ($r,$alllevs,$parmlev)=@_;
    $r->print('<b>'.&mt('Select Parameter Level').
	      &Apache::loncommon::help_open_topic('Course_Parameter_Levels').'</b> ');
    $r->print('<select name="parmlev">');
    foreach (reverse sort keys %{$alllevs}) {
	$r->print('<option value="'.$$alllevs{$_}.'"');
	if ($parmlev eq $$alllevs{$_}) {
	    $r->print(' selected'); 
	}
	$r->print('>'.$_.'</option>');
    }
    $r->print("</select>");
}


sub sectionmenu {
    my ($r,$selectedsections)=@_;
    my %sectionhash = &Apache::loncommon::get_sections();
    return if (!%sectionhash);

    $r->print('<select name="Section" multiple="true" size="8" >');
    foreach my $s ('all',sort keys %sectionhash) {
	$r->print('    <option value="'.$s.'"');
	foreach (@{$selectedsections}) {
	    if ($s eq $_) {
		$r->print(' selected');
		last;
	    }
	}
	$r->print('>'.$s."</option>\n");
    }
    $r->print("</select>\n");
}

sub groupmenu {
    my ($r,$selectedgroups)=@_;
    my %grouphash = &Apache::longroup::coursegroups();
    return if (!%grouphash);

    $r->print('<select name="Group" multiple="true" size="8" >');
    foreach my $group (sort(keys(%grouphash))) {
	$r->print('    <option value="'.$group.'"');
	foreach (@{$selectedgroups}) {
	    if ($group eq $_) {
		$r->print(' selected');
		last;
	    }
	}
	$r->print('>'.$group."</option>\n");
    }
    $r->print("</select>\n");
}


sub keysplit {
    my $keyp=shift;
    return (split(/\,/,$keyp));
}

sub keysinorder {
    my ($name,$keyorder)=@_;
    return sort {
	$$keyorder{$a} <=> $$keyorder{$b};
    } (keys %{$name});
}

sub keysinorder_bytype {
    my ($name,$keyorder)=@_;
    return sort {
	my $ta=(split('_',$a))[-1];
	my $tb=(split('_',$b))[-1];
	if ($$keyorder{'parameter_0_'.$ta} == $$keyorder{'parameter_0_'.$tb}) {
	    return ($a cmp $b);
	}
	$$keyorder{'parameter_0_'.$ta} <=> $$keyorder{'parameter_0_'.$tb};
    } (keys %{$name});
}

sub keysindisplayorder {
    my ($name,$keyorder)=@_;
    return sort {
	$$keyorder{'parameter_0_'.$a} <=> $$keyorder{'parameter_0_'.$b};
    } (keys %{$name});
}

sub sortmenu {
    my ($r,$sortorder)=@_;
    $r->print('<br /><label><input type="radio" name="sortorder" value="realmstudent"');
    if ($sortorder eq 'realmstudent') {
       $r->print(' checked="on"');
    }
    $r->print(' />'.&mt('Sort by realm first, then student (group/section)'));
    $r->print('</label><br /><label><input type="radio" name="sortorder" value="studentrealm"');
    if ($sortorder eq 'studentrealm') {
       $r->print(' checked="on"');
    }
    $r->print(' />'.&mt('Sort by student (group/section) first, then realm').
	      '</label>');
}

sub standardkeyorder {
    return ('parameter_0_opendate' => 1,
	    'parameter_0_duedate' => 2,
	    'parameter_0_answerdate' => 3,
	    'parameter_0_interval' => 4,
	    'parameter_0_weight' => 5,
	    'parameter_0_maxtries' => 6,
	    'parameter_0_hinttries' => 7,
	    'parameter_0_contentopen' => 8,
	    'parameter_0_contentclose' => 9,
	    'parameter_0_type' => 10,
	    'parameter_0_problemstatus' => 11,
	    'parameter_0_hiddenresource' => 12,
	    'parameter_0_hiddenparts' => 13,
	    'parameter_0_display' => 14,
	    'parameter_0_ordered' => 15,
	    'parameter_0_tol' => 16,
	    'parameter_0_sig' => 17,
	    'parameter_0_turnoffunit' => 18,
            'parameter_0_discussend' => 19,
            'parameter_0_discusshide' => 20);
}

##################################################
##################################################

=pod

=item assessparms

Show assessment data and parameters.  This is a large routine that should
be simplified and shortened... someday.

Inputs: $r

Returns: nothing

Variables used (guessed by Jeremy):

=over 4

=item B<pscat>: ParameterS CATegories? ends up a list of the types of parameters that exist, e.g., tol, weight, acc, opendate, duedate, answerdate, sig, maxtries, type.

=item B<psprt>: ParameterS PaRTs? a list of the parts of a problem that we are displaying? Used to display only selected parts?

=item B<allmaps>:

=back

=cut

##################################################
##################################################
sub assessparms {

    my $r=shift;

    my @ids=();
    my %symbp=();
    my %mapp=();
    my %typep=();
    my %keyp=();
    my %uris=();
    my %maptitles=();

# -------------------------------------------------------- Variable declaration

    my %allmaps=();
    my %alllevs=();

    my $uname;
    my $udom;
    my $uhome;
    my $csec;
    my $cgroup;
    my @usersgroups = ();
 
    my $coursename=$env{'course.'.$env{'request.course.id'}.'.description'};

    $alllevs{'Resource Level'}='full';
    $alllevs{'Map/Folder Level'}='map';
    $alllevs{'Course Level'}='general';

    my %allparms;
    my %allparts;
#
# Order in which these parameters will be displayed
#
    my %keyorder=&standardkeyorder();

    @ids=();
    %symbp=();
    %typep=();

    my $message='';

    $csec=$env{'form.csec'};
    $cgroup=$env{'form.cgroup'};

    if      ($udom=$env{'form.udom'}) {
    } elsif ($udom=$env{'request.role.domain'}) {
    } elsif ($udom=$env{'user.domain'}) {
    } else {
	$udom=$r->dir_config('lonDefDomain');
    }

    my @pscat=&Apache::loncommon::get_env_multiple('form.pscat');
    my $pschp=$env{'form.pschp'};
    my @psprt=&Apache::loncommon::get_env_multiple('form.psprt');
    if (!@psprt) { $psprt[0]='0'; }

    my $pssymb='';
    my $parmlev='';
 
    unless ($env{'form.parmlev'}) {
        $parmlev = 'map';
    } else {
        $parmlev = $env{'form.parmlev'};
    }

# ----------------------------------------------- Was this started from grades?

    if (($env{'form.command'} eq 'set') && ($env{'form.url'})
	&& (!$env{'form.dis'})) {
	my $url=$env{'form.url'};
	$url=~s-^http://($ENV{'SERVER_NAME'}|$ENV{'HTTP_HOST'})--;
	$pssymb=&Apache::lonnet::symbread($url);
	if (!@pscat) { @pscat=('all'); }
	$pschp='';
        $parmlev = 'full';
    } elsif ($env{'form.symb'}) {
	$pssymb=$env{'form.symb'};
	if (!@pscat) { @pscat=('all'); }
	$pschp='';
        $parmlev = 'full';
    } else {
	$env{'form.url'}='';
    }

    my $id=$env{'form.id'};
    if (($id) && ($udom)) {
	$uname=(&Apache::lonnet::idget($udom,$id))[1];
	if ($uname) {
	    $id='';
	} else {
	    $message=
		'<span class="LC_error">'.&mt("Unknown ID")." '$id' ".
		&mt('at domain')." '$udom'</span>";
	}
    } else {
	$uname=$env{'form.uname'};
    }
    unless ($udom) { $uname=''; }
    $uhome='';
    if ($uname) {
	$uhome=&Apache::lonnet::homeserver($uname,$udom);
        if ($uhome eq 'no_host') {
	    $message=
		'<span class="LC_error">'.&mt("Unknown user")." '$uname' ".
		&mt("at domain")." '$udom'</span>";
	    $uname='';
        } else {
	    $csec=&Apache::lonnet::getsection($udom,$uname,
					      $env{'request.course.id'});
            
	    if ($csec eq '-1') {
		$message='<span class="LC_error">'.
		    &mt("User")." '$uname' ".&mt("at domain")." '$udom' ".
		    &mt("not in this course")."</span>";
		$uname='';
		$csec=$env{'form.csec'};
                $cgroup=$env{'form.cgroup'};
	    } else {
		my %name=&Apache::lonnet::userenvironment($udom,$uname,
		      ('firstname','middlename','lastname','generation','id'));
		$message="\n<p>\n".&mt("Full Name").": ".
		    $name{'firstname'}.' '.$name{'middlename'}.' '
			.$name{'lastname'}.' '.$name{'generation'}.
			    "<br />\n".&mt('ID').": ".$name{'id'}.'<p>';
	    }
            @usersgroups = &Apache::lonnet::get_users_groups(
                                       $udom,$uname,$env{'request.course.id'});
            if (@usersgroups > 0) {
                unless (grep(/^\Q$cgroup\E$/,@usersgroups)) {
                    $cgroup = $usersgroups[0];
                }
            }
        }
    }

    unless ($csec) { $csec=''; }
    unless ($cgroup) { $cgroup=''; }

# --------------------------------------------------------- Get all assessments
    &extractResourceInformation(\@ids, \%typep,\%keyp, \%allparms, \%allparts, \%allmaps, 
				\%mapp, \%symbp,\%maptitles,\%uris,
				\%keyorder);

    $mapp{'0.0'} = '';
    $symbp{'0.0'} = '';

# ---------------------------------------------------------- Anything to store?
    if ($env{'form.pres_marker'}) {
        my @markers=split(/\&\&\&/,$env{'form.pres_marker'});
        my @values=split(/\&\&\&/,$env{'form.pres_value'});
        my @types=split(/\&\&\&/,$env{'form.pres_type'});
	for (my $i=0;$i<=$#markers;$i++) {
	    $message.=&storeparm(split(/\&/,$markers[$i]),
				 $values[$i],
				 $types[$i],
				 $uname,$udom,$csec,$cgroup);
	}
# ---------------------------------------------------------------- Done storing
	$message.='<h3>'.&mt('Changes can take up to 10 minutes before being active for all students.').&Apache::loncommon::help_open_topic('Caching').'</h3>';
    }
#----------------------------------------------- if all selected, fill in array
    if ($pscat[0] eq "all") {@pscat = (keys %allparms);}
    if (!@pscat) { @pscat=('duedate','opendate','answerdate','weight','maxtries') }; 
    if ($psprt[0] eq "all" || !@psprt) {@psprt = (keys %allparts);}
# ------------------------------------------------------------------ Start page

    &startpage($r);

    foreach ('tolerance','date_default','date_start','date_end',
	     'date_interval','int','float','string') {
	$r->print('<input type="hidden" value="'.
		  $env{'form.recent_'.$_}.'" name="recent_'.$_.'" />');
    }
                        
    if (!$pssymb) {
        $r->print('<table border="1"><tr><td>');
        &levelmenu($r,\%alllevs,$parmlev);
	if ($parmlev ne 'general') {
            $r->print('<td>');
	    &mapmenu($r,\%allmaps,$pschp,\%maptitles);
	    $r->print('</td>');
	}
        $r->print('</td></tr></table>');
	&displaymenu($r,\%allparms,\%allparts,\@pscat,\@psprt,\%keyorder);
    } else {
        my ($map,$id,$resource)=&Apache::lonnet::decode_symb($pssymb);
	my $title = &Apache::lonnet::gettitle($pssymb);
        $r->print(&mt('Specific Resource: [_1] ([_2])',$title,$resource).
                  '<input type="hidden" value="'.$pssymb.'" name="symb" />'.
		  '<br /><label><b>'.&mt('Show all parts').': <input type="checkbox" name="psprt" value="all"'.
		  ($env{'form.psprt'}?' checked="checked"':'').' /></b></label><br />');
    }
    &usermenu($r,$uname,$id,$udom,$csec,$cgroup,$parmlev,\@usersgroups);    

    $r->print('<p>'.$message.'</p>');

    $r->print('<br /><input type="submit" name="dis" value="'.&mt("Update Parameter Display").'" />');

    my @temp_pscat;
    map {
        my $cat = $_;
        push(@temp_pscat, map { $_.'.'.$cat } @psprt);
    } @pscat;

    @pscat = @temp_pscat;

    if (($env{'form.prevvisit'}) || ($pschp) || ($pssymb)) {
# ----------------------------------------------------------------- Start Table
        my @catmarker=map { tr|.|_|; 'parameter_'.$_; } @pscat;
        my $csuname=$env{'user.name'};
        my $csudom=$env{'user.domain'};

        if ($parmlev eq 'full') {
           my $coursespan=$csec?8:5;
           my $userspan=3;
           if ($cgroup ne '') {
              $coursespan += 3;
           } 
      
           $r->print('<p><table border=2>');
           $r->print('<tr><td colspan=5></td>');
           $r->print('<th colspan='.($coursespan).'>'.&mt('Any User').'</th>');
           if ($uname) {
               if (@usersgroups > 1) {
                   $userspan ++;
               }
               $r->print('<th colspan="'.$userspan.'" rowspan="2">');
               $r->print(&mt("User")." $uname ".&mt('at Domain')." $udom</th>");
           }
	   my %lt=&Apache::lonlocal::texthash(
				  'pie'    => "Parameter in Effect",
				  'csv'    => "Current Session Value",
                                  'at'     => 'at',
                                  'rl'     => "Resource Level",
				  'ic'     => 'in Course',
				  'aut'    => "Assessment URL and Title",
				  'type'   => 'Type',
				  'emof'   => "Enclosing Map or Folder",
				  'part'   => 'Part',
                                  'pn'     => 'Parameter Name',
				  'def'    => 'default',
				  'femof'  => 'from Enclosing Map or Folder',
				  'gen'    => 'general',
				  'foremf' => 'for Enclosing Map or Folder',
				  'fr'     => 'for Resource'
					      );
           $r->print(<<ENDTABLETWO);
<th rowspan=3>$lt{'pie'}</th>
<th rowspan=3>$lt{'csv'}<br />($csuname $lt{'at'} $csudom)</th>
</tr><tr><td colspan=5></td><th colspan=2>$lt{'ic'}</th><th colspan=2>$lt{'rl'}</th>
<th colspan=1>$lt{'ic'}</th>

ENDTABLETWO
           if ($csec) {
                $r->print("<th colspan=3>".
			  &mt("in Section")." $csec</th>");
           }
           if ($cgroup) {
                $r->print("<th colspan=3>".
                          &mt("in Group")." $cgroup</th>");
           }
           $r->print(<<ENDTABLEHEADFOUR);
</tr><tr><th>$lt{'aut'}</th><th>$lt{'type'}</th>
<th>$lt{'emof'}</th><th>$lt{'part'}</th><th>$lt{'pn'}</th>
<th>$lt{'gen'}</th><th>$lt{'foremf'}</th>
<th>$lt{'def'}</th><th>$lt{'femof'}</th><th>$lt{'fr'}</th>
ENDTABLEHEADFOUR

           if ($csec) {
               $r->print('<th>'.&mt('general').'</th><th>'.&mt('for Enclosing Map or Folder').'</th><th>'.&mt('for Resource').'</th>');
           }

           if ($cgroup) {
               $r->print('<th>'.&mt('general').'</th><th>'.&mt('for Enclosing Map or Folder').'</th><th>'.&mt('for Resource').'</th>');
           }

           if ($uname) {
               if (@usersgroups > 1) {
                   $r->print('<th>'.&mt('Control by other group?').'</th>');
               }
               $r->print('<th>'.&mt('general').'</th><th>'.&mt('for Enclosing Map or Folder').'</th><th>'.&mt('for Resource').'</th>');
           }

           $r->print('</tr>');

           my $defbgone='';
           my $defbgtwo='';
           my $defbgthree = '';

           foreach (@ids) {

                my $rid=$_;
                my ($inmapid)=($rid=~/\.(\d+)$/);

                if ((!$pssymb && 
		     (($pschp eq 'all') || ($allmaps{$pschp} eq $mapp{$rid})))
		    ||
		    ($pssymb && $pssymb eq $symbp{$rid})) {
# ------------------------------------------------------ Entry for one resource
                    if ($defbgone eq '"#E0E099"') {
                        $defbgone='"#E0E0DD"';
                    } else {
                        $defbgone='"#E0E099"';
                    }
                    if ($defbgtwo eq '"#FFFF99"') {
                        $defbgtwo='"#FFFFDD"';
                    } else {
                        $defbgtwo='"#FFFF99"';
                    }
                    if ($defbgthree eq '"#FFBB99"') {
                        $defbgthree='"#FFBBDD"';
                    } else {
                        $defbgthree='"#FFBB99"';
                    }

                    my $thistitle='';
                    my %name=   ();
                    undef %name;
                    my %part=   ();
                    my %display=();
                    my %type=   ();
                    my %default=();
                    my $uri=&Apache::lonnet::declutter($uris{$rid});

                    foreach (&keysplit($keyp{$rid})) {
                        my $tempkeyp = $_;
                        if (grep $_ eq $tempkeyp, @catmarker) {
                          $part{$_}=&Apache::lonnet::metadata($uri,$_.'.part');
                          $name{$_}=&Apache::lonnet::metadata($uri,$_.'.name');
                          $display{$_}=&Apache::lonnet::metadata($uri,$_.'.display');
                          unless ($display{$_}) { $display{$_}=''; }
                          $display{$_}.=' ('.$name{$_}.')';
                          $default{$_}=&Apache::lonnet::metadata($uri,$_);
                          $type{$_}=&Apache::lonnet::metadata($uri,$_.'.type');
                          $thistitle=&Apache::lonnet::metadata($uri,$_.'.title');
                        }
                    }
                    my $totalparms=scalar keys %name;
                    if ($totalparms>0) {
                        my $firstrow=1;
			my $title=&Apache::lonnet::gettitle($symbp{$rid});
                        $r->print('<tr><td bgcolor='.$defbgone.
                             ' rowspan='.$totalparms.
                             '><tt><font size=-1>'.
                             join(' / ',split(/\//,$uri)).
                             '</font></tt><p><b>'.
                             "<a href=\"javascript:openWindow('".
				  &Apache::lonnet::clutter($uri).'?symb='.
				  &escape($symbp{$rid}).
                             "', 'metadatafile', '450', '500', 'no', 'yes');\"".
                             " target=\"_self\">$title");

                        if ($thistitle) {
                            $r->print(' ('.$thistitle.')');
                        }
                        $r->print('</a></b></td>');
                        $r->print('<td bgcolor='.$defbgtwo.
                                      ' rowspan='.$totalparms.'>'.$typep{$rid}.
                                      '</td>');

                        $r->print('<td bgcolor='.$defbgone.
                                      ' rowspan='.$totalparms.
                                      '>'.$maptitles{$mapp{$rid}}.'</td>');

                        foreach (&keysinorder_bytype(\%name,\%keyorder)) {
                            unless ($firstrow) {
                                $r->print('<tr>');
                            } else {
                                undef $firstrow;
                            }
                            &print_row($r,$_,\%part,\%name,\%symbp,$rid,\%default,
                                       \%type,\%display,$defbgone,$defbgtwo,
                                       $defbgthree,$parmlev,$uname,$udom,$csec,
                                                            $cgroup,\@usersgroups);
                        }
                    }
                }
            } # end foreach ids
# -------------------------------------------------- End entry for one resource
            $r->print('</table>');
        } # end of  full
#--------------------------------------------------- Entry for parm level map
        if ($parmlev eq 'map') {
            my $defbgone = '"E0E099"';
            my $defbgtwo = '"FFFF99"';
            my $defbgthree = '"FFBB99"';

            my %maplist;

            if ($pschp eq 'all') {
                %maplist = %allmaps; 
            } else {
                %maplist = ($pschp => $mapp{$pschp});
            }

#-------------------------------------------- for each map, gather information
            my $mapid;
	    foreach $mapid (sort {$maplist{$a} cmp $maplist{$b}} keys %maplist) {
                my $maptitle = $maplist{$mapid};

#-----------------------  loop through ids and get all parameter types for map
#-----------------------------------------          and associated information
                my %name = ();
                my %part = ();
                my %display = ();
                my %type = ();
                my %default = ();
                my $map = 0;

#		$r->print("Catmarker: @catmarker<br />\n");
               
                foreach (@ids) {
                  ($map)=(/([\d]*?)\./);
                  my $rid = $_;
        
#                  $r->print("$mapid:$map:   $rid <br /> \n");

                  if ($map eq $mapid) {
                    my $uri=&Apache::lonnet::declutter($uris{$rid});
#                    $r->print("Keys: $keyp{$rid} <br />\n");

#--------------------------------------------------------------------
# @catmarker contains list of all possible parameters including part #s
# $fullkeyp contains the full part/id # for the extraction of proper parameters
# $tempkeyp contains part 0 only (no ids - ie, subparts)
# When storing information, store as part 0
# When requesting information, request from full part
#-------------------------------------------------------------------
                    foreach (&keysplit($keyp{$rid})) {
                      my $tempkeyp = $_;
                      my $fullkeyp = $tempkeyp;
                      $tempkeyp =~ s/_\w+_/_0_/;
                      
                      if ((grep $_ eq $fullkeyp, @catmarker) &&(!$name{$tempkeyp})) {
                        $part{$tempkeyp}="0";
                        $name{$tempkeyp}=&Apache::lonnet::metadata($uri,$fullkeyp.'.name');
                        $display{$tempkeyp}=&Apache::lonnet::metadata($uri,$fullkeyp.'.display');
                        unless ($display{$tempkeyp}) { $display{$tempkeyp}=''; }
                        $display{$tempkeyp}.=' ('.$name{$tempkeyp}.')';
                        $display{$tempkeyp} =~ s/_\w+_/_0_/;
                        $default{$tempkeyp}=&Apache::lonnet::metadata($uri,$fullkeyp);
                        $type{$tempkeyp}=&Apache::lonnet::metadata($uri,$fullkeyp.'.type');
                      }
                    } # end loop through keys
                  }
                } # end loop through ids
                                 
#---------------------------------------------------- print header information
                my $foldermap=&mt($maptitle=~/^uploaded/?'Folder':'Map');
                my $showtitle=$maptitles{$maptitle}.($maptitle!~/^uploaded/?' ['.$maptitle.']':'');
                $r->print(<<ENDMAPONE);
<center><h4>
Set Defaults for All Resources in $foldermap<br />
<font color="red"><i>$showtitle</i></font><br />
Specifically for
ENDMAPONE
                if ($uname) {
		    my $person=&Apache::loncommon::plainname($uname,$udom);
                    $r->print(&mt("User")." <font color=\"red\"><i>$uname \($person\) </i></font> ".
                        &mt('in')." \n");
                } else {
                    $r->print("<font color=\"red\"><i>".&mt('all').'</i></font> '.&mt('users in')." \n");
                }
                if ($cgroup) {
                    $r->print(&mt("Group")." <font color=\"red\"><i>$cgroup".
                              "</i></font> ".&mt('of')." \n");
                    $csec = '';
                } elsif ($csec) {
                    $r->print(&mt("Section")." <font color=\"red\"><i>$csec".
                              "</i></font> ".&mt('of')." \n");
                }
                $r->print("<font color=\"red\"><i>$coursename</i></font><br />");
                $r->print("</h4>\n");
#---------------------------------------------------------------- print table
                $r->print('<p><table border="2">');
                $r->print('<tr><th>'.&mt('Parameter Name').'</th>');
                $r->print('<th>'.&mt('Default Value').'</th>');
                $r->print('<th>'.&mt('Parameter in Effect').'</th></tr>');

	        foreach (&keysinorder(\%name,\%keyorder)) {
                    $r->print('<tr>');
                    &print_row($r,$_,\%part,\%name,\%symbp,$mapid,\%default,
                           \%type,\%display,$defbgone,$defbgtwo,$defbgthree,
                           $parmlev,$uname,$udom,$csec,$cgroup);
                }
                $r->print("</table></center>");
            } # end each map
        } # end of $parmlev eq map
#--------------------------------- Entry for parm level general (Course level)
        if ($parmlev eq 'general') {
            my $defbgone = '"E0E099"';
            my $defbgtwo = '"FFFF99"';
            my $defbgthree = '"FFBB99"';

#-------------------------------------------- for each map, gather information
            my $mapid="0.0";
#-----------------------  loop through ids and get all parameter types for map
#-----------------------------------------          and associated information
            my %name = ();
            my %part = ();
            my %display = ();
            my %type = ();
            my %default = ();
               
            foreach (@ids) {
                my $rid = $_;
        
                my $uri=&Apache::lonnet::declutter($uris{$rid});

#--------------------------------------------------------------------
# @catmarker contains list of all possible parameters including part #s
# $fullkeyp contains the full part/id # for the extraction of proper parameters
# $tempkeyp contains part 0 only (no ids - ie, subparts)
# When storing information, store as part 0
# When requesting information, request from full part
#-------------------------------------------------------------------
                foreach (&keysplit($keyp{$rid})) {
                  my $tempkeyp = $_;
                  my $fullkeyp = $tempkeyp;
                  $tempkeyp =~ s/_\w+_/_0_/;
                  if ((grep $_ eq $fullkeyp, @catmarker) &&(!$name{$tempkeyp})) {
                    $part{$tempkeyp}="0";
                    $name{$tempkeyp}=&Apache::lonnet::metadata($uri,$fullkeyp.'.name');
                    $display{$tempkeyp}=&Apache::lonnet::metadata($uri,$fullkeyp.'.display');
                    unless ($display{$tempkeyp}) { $display{$tempkeyp}=''; }
                    $display{$tempkeyp}.=' ('.$name{$tempkeyp}.')';
                    $display{$tempkeyp} =~ s/_\w+_/_0_/;
                    $default{$tempkeyp}=&Apache::lonnet::metadata($uri,$fullkeyp);
                    $type{$tempkeyp}=&Apache::lonnet::metadata($uri,$fullkeyp.'.type');
                  }
                } # end loop through keys
            } # end loop through ids
                                 
#---------------------------------------------------- print header information
	    my $setdef=&mt("Set Defaults for All Resources in Course");
            $r->print(<<ENDMAPONE);
<center><h4>$setdef
<font color="red"><i>$coursename</i></font><br />
ENDMAPONE
            if ($uname) {
		my $person=&Apache::loncommon::plainname($uname,$udom);
                $r->print(" ".&mt("User")."<font color=\"red\"> <i>$uname \($person\) </i></font> \n");
            } else {
                $r->print("<i><font color=\"red\"> ".&mt("ALL")."</i> ".&mt("USERS")."</font> \n");
            }
            
            if ($csec) {$r->print(&mt("Section")."<font color=\"red\"> <i>$csec</i></font>\n")};
            if ($cgroup) {$r->print(&mt("Group")."<font color=\"red\"> <i>$cgroup</i></font>\n")};
            $r->print("</h4>\n");
#---------------------------------------------------------------- print table
            $r->print('<p><table border="2">');
            $r->print('<tr><th>'.&mt('Parameter Name').'</th>');
            $r->print('<th>'.&mt('Default Value').'</th>');
            $r->print('<th>'.&mt('Parameter in Effect').'</th></tr>');

	    foreach (&keysinorder(\%name,\%keyorder)) {
                $r->print('<tr>');
                &print_row($r,$_,\%part,\%name,\%symbp,$mapid,\%default,
                       \%type,\%display,$defbgone,$defbgtwo,$defbgthree,
                                   $parmlev,$uname,$udom,$csec,$cgroup);
            }
            $r->print("</table></center>");
        } # end of $parmlev eq general
    }
    $r->print('</form>'.&Apache::loncommon::end_page());
} # end sub assessparms


##################################################
##################################################

=pod

=item crsenv

Show and set course data and parameters.  This is a large routine that should
be simplified and shortened... someday.

Inputs: $r

Returns: nothing

=cut

##################################################
##################################################
sub crsenv {
    my $r=shift;
    my $setoutput='';

    my $breadcrumbs = 
	&Apache::lonhtmlcommon::breadcrumbs('Edit Course Environment');
    my $dom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $crs = $env{'course.'.$env{'request.course.id'}.'.num'};

    #
    # Go through list of changes
    foreach (keys %env) {
        next if ($_!~/^form\.(.+)\_setparmval$/);
        my $name  = $1;
        my $value = $env{'form.'.$name.'_value'};
        if ($name eq 'newp') {
            $name = $env{'form.newp_name'};
        }
        if ($name eq 'url') {
            $value=~s/^\/res\///;
            my $bkuptime=time;
            my @tmp = &Apache::lonnet::get
                ('environment',['url'],$dom,$crs);
            $setoutput.=&mt('Backing up previous URL').': '.
                &Apache::lonnet::put
                ('environment',
                 {'top level map backup '.$bkuptime => $tmp[1] },
                 $dom,$crs).
                     '<br />';
        }
        #
        # Deal with modified default spreadsheets
        if ($name =~ /^spreadsheet_default_(classcalc|
                                            studentcalc|
                                            assesscalc)$/x) {
            my $sheettype = $1; 
            if ($sheettype eq 'classcalc') {
                # no need to do anything since viewing the sheet will
                # cause it to be updated. 
            } elsif ($sheettype eq 'studentcalc') {
                # expire all the student spreadsheets
                &Apache::lonnet::expirespread('','','studentcalc');
            } else {
                # expire all the assessment spreadsheets 
                #    this includes non-default spreadsheets, but better to
                #    be safe than sorry.
                &Apache::lonnet::expirespread('','','assesscalc');
                # expire all the student spreadsheets
                &Apache::lonnet::expirespread('','','studentcalc');
            }
        }
        #
        # Deal with the enrollment dates
        if ($name =~ /^default_enrollment_(start|end)_date$/) {
            $value=&Apache::lonhtmlcommon::get_date_from_form($name.'_value');
        }
        # Get existing cloners
        my @oldcloner = ();
        if ($name eq 'cloners') {
            my %clonenames=&Apache::lonnet::dump('environment',$dom,$crs,'cloners');
            if ($clonenames{'cloners'} =~ /,/) {
                @oldcloner = split/,/,$clonenames{'cloners'};
            } else {
                $oldcloner[0] = $clonenames{'cloners'};
            }
        }
        #
        # Let the user know we made the changes
        if ($name && defined($value)) {
            my $failed_cloners;
            if ($name eq 'cloners') {
                $value =~ s/\s//g;
                $value =~ s/^,//;
                $value =~ s/,$//;
                # check requested clones are valid users.
                $failed_cloners = &check_cloners(\$value,\@oldcloner);
            }
            my $put_result = &Apache::lonnet::put('environment',
                                                  {$name=>$value},$dom,$crs);
            if ($put_result eq 'ok') {
                $setoutput.=&mt('Set').' <b>'.$name.'</b> '.&mt('to').' <b>'.$value.'</b>.<br />';
                if ($name eq 'cloners') {
                    &change_clone($value,\@oldcloner);
                }
                # Flush the course logs so course description is immediately updated
                if ($name eq 'description' && defined($value)) {
                    &Apache::lonnet::flushcourselogs();
                }
            } else {
                $setoutput.=&mt('Unable to set').' <b>'.$name.'</b> '.&mt('to').
		    ' <b>'.$value.'</b> '.&mt('due to').' '.$put_result.'.<br />';
            }
            if (($name eq 'cloners') && ($failed_cloners)) {
                $setoutput.= &mt('Unable to include').' - <b>'.$failed_cloners.'</b>, '.
                 &mt('reason').' - '.&mt('LON-CAPA user(s) do(es) not exist').
                 '.<br />'.&mt('Please ').
                 ' <a href="/adm/createuser">'.
                 &mt('add the user(s)').'</a>, '.
                 &mt('and then return to the ').
                 '<a href="/admparmset?action=crsenv">'.
                 &mt('Course Parameters page').'</a> '.
                 &mt('to add the new user(s) to the list of possible cloners').
                 '.<br />';
            }
        }
    }

    my $start_table     =&Apache::loncommon::start_data_table();
    my $start_header_row=&Apache::loncommon::start_data_table_header_row();
    my $end_header_row  =&Apache::loncommon::end_data_table_header_row();
# ------------------------- Re-init course environment entries for this session

    &Apache::lonnet::coursedescription($env{'request.course.id'},
				       {'freshen_cache' => 1});

# -------------------------------------------------------- Get parameters again

    my %values=&Apache::lonnet::dump('environment',$dom,$crs);
    my $SelectStyleFile=&mt('Select Style File');
    my $SelectSpreadsheetFile=&mt('Select Spreadsheet File');
    my $output='';
    if (! exists($values{'con_lost'})) {
        my %descriptions=
	    ('url'            => '<b>'.&mt('Top Level Map').'</b> '.
                                 '<a href="javascript:openbrowser'.
                                 "('envform','url','sequence')\">".
                                 &mt('Select Map').'</a><br /><span class="LC_warning"> '.
                                 &mt('Modification may make assessment data inaccessible').
                                 '</span>',
             'description'    => '<b>'.&mt('Course Description').'</b>',
             'courseid'       => '<b>'.&mt('Course ID or number').
                                 '</b><br />'.
                                 '('.&mt('internal').', '.&mt('optional').')',
             'cloners'        => '<b>'.&mt('Users allowed to clone course').'</b><br /><tt>(user:domain,user:domain)</tt><br />'.&mt('Users with active Course Coordinator role in the course automatically have the right to clone it, and can be omitted from list.'),
             'grading'        => '<b>'.&mt('Grading').'</b><br />'.
                                 '<tt>"standard", "external", or "spreadsheet"</tt> '.&Apache::loncommon::help_open_topic('GradingOptions'),
             'default_xml_style' => '<b>'.&mt('Default XML Style File').'</b> '.
                    '<a href="javascript:openbrowser'.
                    "('envform','default_xml_style'".
                    ",'sty')\">$SelectStyleFile</a><br />",
             'question.email' => '<b>'.&mt('Feedback Addresses for Resource Content Question').
                                 '</b><br />(<tt>user:domain,'.
                                 'user:domain(section;section;...;*;...),...</tt>)',
             'comment.email'  => '<b>'.&mt('Feedback Addresses for Course Content Comments').'</b><br />'.
                                 '(<tt>user:domain,user:domain(section;section;...;*;...),...</tt>)',
             'policy.email'   => '<b>'.&mt('Feedback Addresses for Course Policy').'</b>'.
                                 '<br />(<tt>user:domain,user:domain(section;section;...;*;...),...</tt>)',
             'hideemptyrows'  => '<b>'.&mt('Hide Empty Rows in Spreadsheets').'</b><br />'.
                                 '('.&mt('"[_1]" for default hiding','<tt>yes</tt>').')',
             'pageseparators'  => '<b>'.&mt('Visibly Separate Items on Pages').'</b><br />'.
                                 '('.&mt('"[_1]" for visible separation','<tt>yes</tt>').', '.
                                 &mt('changes will not show until next login').')',
             'student_classlist_view' => '<b>'.&mt('Allow students to view classlist.').'</b>'.&mt('("all":students can view all sections,"section":students can only view their own section.blank or "disabled" prevents student view.'),

             'plc.roles.denied'=> '<b>'.&mt('Disallow live chatroom use for Roles').
                                  '</b><br />"<tt>st</tt>": '.
                                  &mt('student').', "<tt>ta</tt>": '.
                                  'TA, "<tt>in</tt>": '.
                                  &mt('instructor').';<br /><tt>'.&mt('role,role,...').'</tt>) '.
	       Apache::loncommon::help_open_topic("Course_Disable_Discussion"),
             'plc.users.denied' => 
                          '<b>'.&mt('Disallow live chatroom use for Users').'</b><br />'.
                                 '(<tt>user:domain,user:domain,...</tt>)',

             'pch.roles.denied'=> '<b>'.&mt('Disallow Resource Discussion for Roles').
                                  '</b><br />"<tt>st</tt>": '.
                                  'student, "<tt>ta</tt>": '.
                                  'TA, "<tt>in</tt>": '.
                                  'instructor;<br /><tt>role,role,...</tt>) '.
	       Apache::loncommon::help_open_topic("Course_Disable_Discussion"),
             'pch.users.denied' => 
                          '<b>'.&mt('Disallow Resource Discussion for Users').'</b><br />'.
                                 '(<tt>user:domain,user:domain,...</tt>)',
             'spreadsheet_default_classcalc' 
                 => '<b>'.&mt('Default Course Spreadsheet').'</b> '.
                    '<a href="javascript:openbrowser'.
                    "('envform','spreadsheet_default_classcalc'".
                    ",'spreadsheet')\">$SelectSpreadsheetFile</a><br />",
             'spreadsheet_default_studentcalc' 
                 => '<b>'.&mt('Default Student Spreadsheet').'</b> '.
                    '<a href="javascript:openbrowser'.
                    "('envform','spreadsheet_default_calc'".
                    ",'spreadsheet')\">$SelectSpreadsheetFile</a><br />",
             'spreadsheet_default_assesscalc' 
                 => '<b>'.&mt('Default Assessment Spreadsheet').'</b> '.
                    '<a href="javascript:openbrowser'.
                    "('envform','spreadsheet_default_assesscalc'".
                    ",'spreadsheet')\">$SelectSpreadsheetFile</a><br />",
	     'allow_limited_html_in_feedback'
	         => '<b>'.&mt('Allow limited HTML in discussion posts').'</b><br />'.
	            '('.&mt('Set value to "[_1]" to allow',"<tt>yes</tt>").')',
             'allow_discussion_post_editing'
                 => '<b>'.&mt('Allow users with specified roles to edit/delete their own discussion posts').'</b><br />"<tt>st</tt>": '.
                                  &mt('student').', "<tt>ta</tt>": '.
                                  'TA, "<tt>in</tt>": '.
                                  &mt('instructor').';&nbsp;(<tt>'.&mt('role:section,role:section,..., e.g., st:001,st:002,in,cc would permit students in sections 001 and 002 and instructors in any section, and course coordinators to edit their own posts.').'</tt>)<br />'.
                    '('.&mt('or set value to "[_1]" to allow all roles',"<tt>yes</tt>").')',
	     'rndseed'
	         => '<b>'.&mt('Randomization algorithm used').'</b> <br />'.
                    '<span class="LC_error">'.&mt('Modifying this will make problems').' '.
                    &mt('have different numbers and answers').'</span>',
	     'receiptalg'
	         => '<b>'.&mt('Receipt algorithm used').'</b> <br />'.
                    &mt('This controls how receipt numbers are generated.'),
             'suppress_tries'
                 => '<b>'.&mt('Suppress number of tries in printing').'</b><br />'.
                    ' ('.&mt('"[_1]" to suppress, anything else to not suppress','<tt>yes</tt>').')',
             'problem_stream_switch'
                 => '<b>'.&mt('Allow problems to be split over pages').'</b><br />'.
                    ' ('.&mt('"[_1]" if allowed, anything else if not','<tt>yes</tt>').')',
             'default_paper_size' 
                 => '<b>'.&mt('Default paper type').'</b><br />'.
                    ' ('.&mt('supported types').': Letter [8 1/2x11 in], Legal [8 1/2x14 in],'. 
                    ' Tabloid [11x17 in], Executive [7 1/2x10 in], A2 [420x594 mm],'. 
                    ' A3 [297x420 mm], A4 [210x297 mm], A5 [148x210 mm], A6 [105x148 mm])',
	     'print_header_format'
	         => '<b>Print header format; substitutions</b>:  %n student name %c course id %a assignment note, numbers after the % limit the field size',
             'anonymous_quiz'
                 => '<b>'.&mt('Anonymous quiz/exam').'</b><br />'.
                    ' (<tt><b>'.&mt('yes').'</b> '.&mt('to avoid print students names').' </tt>)',
             'default_enrollment_start_date' => '<b>'.&mt('Default beginning date for student access.').'</b>',
             'default_enrollment_end_date'   => '<b>'.&mt('Default ending date for student access.').'</b>',
             'nothideprivileged'   => '<b>'.&mt('Privileged users that should not be hidden on staff listings').'</b>'.
                                 '<br />(<tt>user:domain,user:domain,...</tt>)',
             'languages' => '<b>'.&mt('Languages used').'</b>',
             'disable_receipt_display'
                 => '<b>'.&mt('Disable display of problem receipts').'</b><br />'.
                    ' ('.&mt('"[_1]" to disable, anything else if not','<tt>yes</tt>').')',
	     'task_messages'
	         => '<b>'.&mt('Send message to student when clicking Done on Tasks. [_1] to send a message only to student, [_2] to send message to student and add record to user information page for instructors. Leave blank to disable.','<tt>only_student</tt>','<tt>student_and_user_notes_screen</tt>').'</b>',
	     'disablesigfigs'
	         => '<b>'.&mt('Disable checking of Significant Figures').'</b><br />'.
                    ' ('.&mt('"[_1]" to disable, anything else if not','<tt>yes</tt>').')',
	     'disableexampointprint'
	         => '<b>'.&mt('Disable automatically printing point values onto exams.').'</b><br />'.
                    ' ('.&mt('"[_1]" to disable, anything else if not','<tt>yes</tt>').')',
             'externalsyllabus'
                 => '<b>'.&mt('URL of Syllabus (not using internal handler)').'</b>',
	     'tthoptions'
	         => '<b>'.&mt('Default set of options to pass to tth/m when converting tex').'</b>'
             ); 
        my @Display_Order = ('url','description','courseid','cloners','grading',
                             'externalsyllabus',
                             'default_xml_style','pageseparators',
                             'question.email','comment.email','policy.email',
                             'student_classlist_view',
                             'plc.roles.denied','plc.users.denied',
                             'pch.roles.denied','pch.users.denied',
                             'allow_limited_html_in_feedback',
                             'allow_discussion_post_editing',
                             'languages',
			     'nothideprivileged',
                             'rndseed',
                             'receiptalg',
                             'problem_stream_switch',
			     'suppress_tries',
                             'default_paper_size',
			     'print_header_format',
                             'disable_receipt_display',
                             'spreadsheet_default_classcalc',
                             'spreadsheet_default_studentcalc',
                             'spreadsheet_default_assesscalc', 
                             'hideemptyrows',
                             'default_enrollment_start_date',
                             'default_enrollment_end_date',
			     'tthoptions',
			     'disablesigfigs',
			     'disableexampointprint',
			     'task_messages'
                             );
	foreach my $parameter (sort(keys(%values))) {
            unless (($parameter =~ m/^internal\./)||($parameter =~ m/^metadata\./)) {
                if (! $descriptions{$parameter}) {
                    $descriptions{$parameter}=$parameter;
                    push(@Display_Order,$parameter);
                }
            }
	}
	
        foreach my $parameter (@Display_Order) {
            my $description = $descriptions{$parameter};
            # onchange is javascript to automatically check the 'Set' button.
            my $onchange = 'onFocus="javascript:window.document.forms'.
                "['envform'].elements['".$parameter."_setparmval']".
                '.checked=true;"';
            $output .= &Apache::loncommon::start_data_table_row().
		'<td>'.$description.'</td>';
            if ($parameter =~ /^default_enrollment_(start|end)_date$/) {
                $output .= '<td>'.
                    &Apache::lonhtmlcommon::date_setter('envform',
                                                        $parameter.'_value',
                                                        $values{$parameter},
                                                        $onchange).
                                                        '</td>';
            } else {
                $output .= '<td>'.
                    &Apache::lonhtmlcommon::textbox($parameter.'_value',
                                                    $values{$parameter},
                                                    40,$onchange).'</td>';
            }
            $output .= '<td>'.
                &Apache::lonhtmlcommon::checkbox($parameter.'_setparmval').
                '</td>';
            $output .= &Apache::loncommon::end_data_table_row()."\n";
	}
        my $onchange = 'onFocus="javascript:window.document.forms'.
            '[\'envform\'].elements[\'newp_setparmval\']'.
            '.checked=true;"';
	$output.=&Apache::loncommon::start_data_table_row().
	    '<td><i>'.&mt('Create New Environment Variable').'</i><br />'.
	    '<input type="text" size=40 name="newp_name" '.
                $onchange.' /></td><td>'.
            '<input type="text" size=40 name="newp_value" '.
                $onchange.' /></td><td>'.
	    '<input type="checkbox" name="newp_setparmval" /></td>'.
	    &Apache::loncommon::end_data_table_row()."\n";
    }
    my %lt=&Apache::lonlocal::texthash(
		    'par'   => 'Parameter',
		    'val'   => 'Value',
		    'set'   => 'Set',
		    'sce'   => 'Set Course Environment'
				       );

    my $Parameter=&mt('Parameter');
    my $Value=&mt('Value');
    my $Set=&mt('Set');
    my $browse_js=
	'<script type="text/javascript" language="Javascript">'.
	&Apache::loncommon::browser_and_searcher_javascript('parmset').
	'</script>';
    
    my $start_page = 
	&Apache::loncommon::start_page('Set Course Environment',
				       $browse_js);
    my $end_page = 
	&Apache::loncommon::end_page();
    my $end_table=&Apache::loncommon::end_data_table();
    $r->print(<<ENDENV);
$start_page
$breadcrumbs
<form method="post" action="/adm/parmset?action=crsenv" name="envform">
$setoutput
$start_table
$start_header_row
<th>$lt{'par'}</th><th>$lt{'val'}</th><th>$lt{'set'}?</th>
$end_header_row
$output
$end_table
<input type="submit" name="crsenv" value="$lt{'sce'}">
</form>
$end_page
ENDENV
}
##################################################
# Overview mode
##################################################
my $tableopen;

sub tablestart {
    if ($tableopen) {
	return '';
    } else {
	$tableopen=1;
	return &Apache::loncommon::start_data_table().'<tr><th>'.&mt('Parameter').'</th><th>'.
	    &mt('Delete').'</th><th>'.&mt('Set to ...').'</th></tr>';
    }
}

sub tableend {
    if ($tableopen) {
	$tableopen=0;
	return &Apache::loncommon::end_data_table();
    } else {
	return'';
    }
}

sub readdata {
    my ($crs,$dom)=@_;
# Read coursedata
    my $resourcedata=&Apache::lonnet::get_courseresdata($crs,$dom);
# Read userdata

    my $classlist=&Apache::loncoursedata::get_classlist();
    foreach (keys %$classlist) {
        if ($_=~/^($match_username)\:($match_domain)$/) {
	    my ($tuname,$tudom)=($1,$2);
	    my $useropt=&Apache::lonnet::get_userresdata($tuname,$tudom);
            foreach my $userkey (keys %{$useropt}) {
		if ($userkey=~/^$env{'request.course.id'}/) {
                    my $newkey=$userkey;
		    $newkey=~s/^($env{'request.course.id'}\.)/$1\[useropt\:$tuname\:$tudom\]\./;
		    $$resourcedata{$newkey}=$$useropt{$userkey};
		}
	    }
	}
    }
    return $resourcedata;
}


# Setting

sub storedata {
    my ($r,$crs,$dom)=@_;
# Set userlevel immediately
# Do an intermediate store of course level
    my $olddata=&readdata($crs,$dom);
    my %newdata=();
    undef %newdata;
    my @deldata=();
    undef @deldata;
    foreach (keys %env) {
	if ($_=~/^form\.([a-z]+)\_(.+)$/) {
	    my $cmd=$1;
	    my $thiskey=$2;
	    my ($tuname,$tudom)=&extractuser($thiskey);
	    my $tkey=$thiskey;
            if ($tuname) {
		$tkey=~s/\.\[useropt\:$tuname\:$tudom\]\./\./;
	    }
	    if ($cmd eq 'set') {
		my $data=$env{$_};
                my $typeof=$env{'form.typeof_'.$thiskey};
 		if ($$olddata{$thiskey} ne $data) { 
		    if ($tuname) {
			if (&Apache::lonnet::put('resourcedata',{$tkey=>$data,
								 $tkey.'.type' => $typeof},
						 $tudom,$tuname) eq 'ok') {
			    &log_parmset({$tkey=>$data,$tkey.'.type' => $typeof},0,$tuname,$tudom);
			    $r->print('<br />'.&mt('Stored modified parameter for').' '.
				      &Apache::loncommon::plainname($tuname,$tudom));
			} else {
			    $r->print('<div class="LC_error">'.
				      &mt('Error storing parameters').'</div>');
			}
			&Apache::lonnet::devalidateuserresdata($tuname,$tudom);
		    } else {
			$newdata{$thiskey}=$data;
 			$newdata{$thiskey.'.type'}=$typeof; 
                   } 
		}
	    } elsif ($cmd eq 'del') {
		if ($tuname) {
		    if (&Apache::lonnet::del('resourcedata',[$tkey],$tudom,$tuname) eq 'ok') {
			    &log_parmset({$tkey=>''},1,$tuname,$tudom);
			$r->print('<br />'.&mt('Deleted parameter for').' '.&Apache::loncommon::plainname($tuname,$tudom));
		    } else {
			$r->print('<div class="LC_error">'.
				  &mt('Error deleting parameters').'</div>');
		    }
		    &Apache::lonnet::devalidateuserresdata($tuname,$tudom);
		} else {
		    push (@deldata,$thiskey,$thiskey.'.type');
		}
	    } elsif ($cmd eq 'datepointer') {
		my $data=&Apache::lonhtmlcommon::get_date_from_form($env{$_});
                my $typeof=$env{'form.typeof_'.$thiskey};
		if (defined($data) and $$olddata{$thiskey} ne $data) { 
		    if ($tuname) {
			if (&Apache::lonnet::put('resourcedata',{$tkey=>$data,
								 $tkey.'.type' => $typeof},
						 $tudom,$tuname) eq 'ok') {
			    &log_parmset({$tkey=>$data,$tkey.'.type' => $typeof},0,$tuname,$tudom);
			    $r->print('<br />'.&mt('Stored modified date for').' '.&Apache::loncommon::plainname($tuname,$tudom));
			} else {
			    $r->print('<div class="LC_error">'.
				      &mt('Error storing parameters').'</div>');
			}
			&Apache::lonnet::devalidateuserresdata($tuname,$tudom);
		    } else {
			$newdata{$thiskey}=$data;
			$newdata{$thiskey.'.type'}=$typeof; 
		    }
		}
	    }
	}
    }
# Store all course level
    my $delentries=$#deldata+1;
    my @newdatakeys=keys %newdata;
    my $putentries=$#newdatakeys+1;
    if ($delentries) {
	if (&Apache::lonnet::del('resourcedata',\@deldata,$dom,$crs) eq 'ok') {
	    my %loghash=map { $_ => '' } @deldata;
	    &log_parmset(\%loghash,1);
	    $r->print('<h2>'.&mt('Deleted [_1] parameter(s)</h2>',$delentries));
	} else {
	    $r->print('<div class="LC_error">'.
		      &mt('Error deleting parameters').'</div>');
	}
	&Apache::lonnet::devalidatecourseresdata($crs,$dom);
    }
    if ($putentries) {
	if (&Apache::lonnet::put('resourcedata',\%newdata,$dom,$crs) eq 'ok') {
			    &log_parmset(\%newdata,0);
	    $r->print('<h3>'.&mt('Stored [_1] parameter(s)',$putentries/2).'</h3>');
	} else {
	    $r->print('<div class="LC_error">'.
		      &mt('Error storing parameters').'</div>');
	}
	&Apache::lonnet::devalidatecourseresdata($crs,$dom);
    }
}

sub extractuser {
    my $key=shift;
    return ($key=~/^$env{'request.course.id'}.\[useropt\:($match_username)\:($match_domain)\]\./);
}

sub listdata {
    my ($r,$resourcedata,$listdata,$sortorder)=@_;
# Start list output

    my $oldsection='';
    my $oldrealm='';
    my $oldpart='';
    my $pointer=0;
    $tableopen=0;
    my $foundkeys=0;
    my %keyorder=&standardkeyorder();
    foreach my $thiskey (sort {
	if ($sortorder eq 'realmstudent') {
	    my ($astudent,$arealm)=($a=~/^\Q$env{'request.course.id'}\E\.\[([^\.]+)\]\.(.+)\.[^\.]+$/);
	    my ($bstudent,$brealm)=($b=~/^\Q$env{'request.course.id'}\E\.\[([^\.]+)\]\.(.+)\.[^\.]+$/);
	    if (!defined($astudent)) {
		($arealm)=($a=~/^\Q$env{'request.course.id'}\E\.(.+)$/);
	    }
	    if (!defined($bstudent)) {
		($brealm)=($b=~/^\Q$env{'request.course.id'}\E\.(.+)$/);
	    }
	    $arealm=~s/\.type//;
	    my ($ares, $aparm) = ($arealm=~/^(.*)\.(.*)$/);
	    $aparm=$keyorder{'parameter_0_'.$aparm};
	    $brealm=~s/\.type//;
	    my ($bres, $bparm) = ($brealm=~/^(.*)\.(.*)$/);
	    $bparm=$keyorder{'parameter_0_'.$bparm};	   
	    if ($ares eq $bres) {
		if (defined($aparm) && defined($bparm)) {
		    ($aparm <=> $bparm);
		} elsif (defined($aparm)) {
		    -1;
		} elsif (defined($bparm)) {
		    1;
		} else {
		    ($arealm cmp $brealm) || ($astudent cmp $bstudent);
		}
	    } else {
		($arealm cmp $brealm) || ($astudent cmp $bstudent);
	    }
	} else {
	    $a cmp $b;
	}
    } keys %{$listdata}) {
	 
	if ($$listdata{$thiskey.'.type'}) {
            my $thistype=$$listdata{$thiskey.'.type'};
            if ($$resourcedata{$thiskey.'.type'}) {
		$thistype=$$resourcedata{$thiskey.'.type'};
	    }
	    my ($middle,$part,$name)=
		($thiskey=~/^$env{'request.course.id'}\.(?:(.+)\.)*([\w\s]+)\.(\w+)$/);
	    my $section=&mt('All Students');
	    if ($middle=~/^\[(.*)\]/) {
		my $issection=$1;
		if ($issection=~/^useropt\:($match_username)\:($match_domain)/) {
		    $section=&mt('User').": ".&Apache::loncommon::plainname($1,$2);
		} else {
		    $section=&mt('Group/Section').': '.$issection;
		}
		$middle=~s/^\[(.*)\]//;
	    }
	    $middle=~s/\.+$//;
	    $middle=~s/^\.+//;
	    my $realm='<span class="LC_parm_scope_all">'.&mt('All Resources').'</span>';
	    if ($middle=~/^(.+)\_\_\_\(all\)$/) {
		$realm='<span class="LC_parm_scope_folder">'.&mt('Folder/Map').': '.&Apache::lonnet::gettitle($1).' <br /><span class="LC_parm_folder">('.$1.')</span></span>';
	    } elsif ($middle) {
		my ($map,$id,$url)=&Apache::lonnet::decode_symb($middle);
		$realm='<span class="LC_parm_scope_resource">'.&mt('Resource').': '.&Apache::lonnet::gettitle($middle).' <br /><span class="LC_parm_symb">('.$url.' in '.$map.' id: '.$id.')</span></span>';
	    }
	    if ($sortorder eq 'realmstudent') {
		if ($realm ne $oldrealm) {
		    $r->print(&tableend()."\n<hr /><h1>$realm</h1>");
		    $oldrealm=$realm;
		    $oldsection='';
		}
		if ($section ne $oldsection) {
		    $r->print(&tableend()."\n<h2>$section</h2>");
		    $oldsection=$section;
		    $oldpart='';
		}
	    } else {
		if ($section ne $oldsection) {
		    $r->print(&tableend()."\n<hr /><h1>$section</h1>");
		    $oldsection=$section;
		    $oldrealm='';
		}
		if ($realm ne $oldrealm) {
		    $r->print(&tableend()."\n<h2>$realm</h2>");
		    $oldrealm=$realm;
		    $oldpart='';
		}
	    }
	    if ($part ne $oldpart) {
		$r->print(&tableend().
			  "\n<span class=\"LC_parm_part\">".&mt('Part').": $part</span>");
		$oldpart=$part;
	    }
#
# Ready to print
#
	    $r->print(&tablestart().
		      &Apache::loncommon::start_data_table_row().
		      '<td><b>'.&standard_parameter_names($name).
		      '</b></td><td><input type="checkbox" name="del_'.
		      $thiskey.'" /></td><td>');
	    $foundkeys++;
	    if (&isdateparm($thistype)) {
		my $jskey='key_'.$pointer;
		$pointer++;
		$r->print(
			  &Apache::lonhtmlcommon::date_setter('parmform',
							      $jskey,
						      $$resourcedata{$thiskey},
							      '',1,'','').
'<input type="hidden" name="datepointer_'.$thiskey.'" value="'.$jskey.'" />'.
&date_sanity_info($$resourcedata{$thiskey})
			  );
	    } elsif ($thistype eq 'string_yesno') {
		my $showval;
		if (defined($$resourcedata{$thiskey})) {
		    $showval=$$resourcedata{$thiskey};
		}
		$r->print('<label><input type="radio" name="set_'.$thiskey.
			  '" value="yes"');
		if ($showval eq 'yes') {
		    $r->print(' checked="checked"');
		}
                $r->print(' />'.&mt('Yes').'</label> ');
		$r->print('<label><input type="radio" name="set_'.$thiskey.
			  '" value="no"');
		if ($showval eq 'no') {
		    $r->print(' checked="checked"');
		}
                $r->print(' />'.&mt('No').'</label>');
	    } else {
		my $showval;
		if (defined($$resourcedata{$thiskey})) {
		    $showval=$$resourcedata{$thiskey};
		}
		$r->print('<input type="text" name="set_'.$thiskey.'" value="'.
			  $showval.'">');
	    }
	    $r->print('<input type="hidden" name="typeof_'.$thiskey.'" value="'.
		      $thistype.'">');
	    $r->print('</td>'.&Apache::loncommon::end_data_table_row());
	}
    }
    return $foundkeys;
}

sub newoverview {
    my ($r) = @_;

    my $dom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $crs = $env{'course.'.$env{'request.course.id'}.'.num'};
    my $start_page = &Apache::loncommon::start_page('Set Parameters');
    my $breadcrumbs = &Apache::lonhtmlcommon::breadcrumbs('Overview');
    $r->print(<<ENDOVER);
$start_page
$breadcrumbs
<form method="post" action="/adm/parmset?action=newoverview" name="parmform">
ENDOVER
    my @ids=();
    my %typep=();
    my %keyp=();
    my %allparms=();
    my %allparts=();
    my %allmaps=();
    my %mapp=();
    my %symbp=();
    my %maptitles=();
    my %uris=();
    my %keyorder=&standardkeyorder();
    my %defkeytype=();

    my %alllevs=();
    $alllevs{'Resource Level'}='full';
    $alllevs{'Map/Folder Level'}='map';
    $alllevs{'Course Level'}='general';

    my $csec=$env{'form.csec'};
    my $cgroup=$env{'form.cgroup'};

    my @pscat=&Apache::loncommon::get_env_multiple('form.pscat');
    my $pschp=$env{'form.pschp'};
    my @psprt=&Apache::loncommon::get_env_multiple('form.psprt');
    if (!@psprt) { $psprt[0]='0'; }

    my @selected_sections = 
	&Apache::loncommon::get_env_multiple('form.Section');
    @selected_sections = ('all') if (! @selected_sections);
    foreach (@selected_sections) {
        if ($_ eq 'all') {
            @selected_sections = ('all');
        }
    }
    my @selected_groups =
        &Apache::loncommon::get_env_multiple('form.Group');

    my $pssymb='';
    my $parmlev='';
 
    unless ($env{'form.parmlev'}) {
        $parmlev = 'map';
    } else {
        $parmlev = $env{'form.parmlev'};
    }

    &extractResourceInformation(\@ids, \%typep,\%keyp, \%allparms, \%allparts, \%allmaps, 
				\%mapp, \%symbp,\%maptitles,\%uris,
				\%keyorder,\%defkeytype);

# Menu to select levels, etc

    $r->print('<table id="LC_parm_overview_scope">
               <tr><td class="LC_parm_overview_level_menu">');
    &levelmenu($r,\%alllevs,$parmlev);
    if ($parmlev ne 'general') {
	$r->print('<td class="LC_parm_overview_map_menu">');
	&mapmenu($r,\%allmaps,$pschp,\%maptitles);
	$r->print('</td>');
    }
    $r->print('</td></tr></table>');

    $r->print('<table id="LC_parm_overview_controls">
               <tr><td class="LC_parm_overview_parm_selectors">');  
    &parmmenu($r,\%allparms,\@pscat,\%keyorder);
    $r->print('</td><td class="LC_parm_overview_restrictions">
                <table class="LC_parm_overview_restrictions">'.
              '<tr><th>'.&mt('Parts').'</th><th>'.&mt('Section(s)').
              '</th><th>'.&mt('Group(s)').'</th></tr><tr><td>');
    &partmenu($r,\%allparts,\@psprt);
    $r->print('</td><td>');
    &sectionmenu($r,\@selected_sections);
    $r->print('</td><td>');
    &groupmenu($r,\@selected_groups);
    $r->print('</td></tr></table>');
    $r->print('</td></tr></table>');
 
    my $sortorder=$env{'form.sortorder'};
    unless ($sortorder) { $sortorder='realmstudent'; }
    &sortmenu($r,$sortorder);

    $r->print('<p><input type="submit" name="dis" value="'.&mt('Display').'" /></p>');

# Build the list data hash from the specified parms

    my $listdata;
    %{$listdata}=();

    foreach my $cat (@pscat) {
        &secgroup_lister($cat,$pschp,$parmlev,$listdata,\@psprt,\@selected_sections,\%defkeytype,\%allmaps,\@ids,\%symbp);
        &secgroup_lister($cat,$pschp,$parmlev,$listdata,\@psprt,\@selected_groups,\%defkeytype,\%allmaps,\@ids,\%symbp);
    }

    if (($env{'form.store'}) || ($env{'form.dis'})) {

	if ($env{'form.store'}) { &storedata($r,$crs,$dom); }

# Read modified data

	my $resourcedata=&readdata($crs,$dom);

# List data

	&listdata($r,$resourcedata,$listdata,$sortorder);
    }
    $r->print(&tableend().
	     ((($env{'form.store'}) || ($env{'form.dis'}))?'<p><input type="submit" name="store" value="'.&mt('Store').'" /></p>':'').
	      '</form>'.&Apache::loncommon::end_page());
}

sub secgroup_lister {
    my ($cat,$pschp,$parmlev,$listdata,$psprt,$selections,$defkeytype,$allmaps,$ids,$symbp) = @_;
    foreach my $item (@{$selections}) {
        foreach my $part (@{$psprt}) {
            my $rootparmkey=$env{'request.course.id'};
            if (($item ne 'all') && ($item ne 'none') && ($item)) {
                $rootparmkey.='.['.$item.']';
            }
            if ($parmlev eq 'general') {
# course-level parameter
                my $newparmkey=$rootparmkey.'.'.$part.'.'.$cat;
                $$listdata{$newparmkey}=1;
                $$listdata{$newparmkey.'.type'}=$$defkeytype{$cat};
            } elsif ($parmlev eq 'map') {
# map-level parameter
                foreach my $mapid (keys %{$allmaps}) {
                    if (($pschp ne 'all') && ($pschp ne $mapid)) { next; }
                    my $newparmkey=$rootparmkey.'.'.$$allmaps{$mapid}.'___(all).'.$part.'.'.$cat;
                    $$listdata{$newparmkey}=1;
                    $$listdata{$newparmkey.'.type'}=$$defkeytype{$cat};
                }
            } else {
# resource-level parameter
                foreach my $rid (@{$ids}) {
                    my ($map,$resid,$url)=&Apache::lonnet::decode_symb($$symbp{$rid});
                    if (($pschp ne 'all') && ($$allmaps{$pschp} ne $map)) { next; }
                    my $newparmkey=$rootparmkey.'.'.$$symbp{$rid}.'.'.$part.'.'.$cat;
                    $$listdata{$newparmkey}=1;
                    $$listdata{$newparmkey.'.type'}=$$defkeytype{$cat};
                }
            }
        }
    }
}

sub overview {
    my ($r) = @_;
    my $dom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $crs = $env{'course.'.$env{'request.course.id'}.'.num'};

    my $start_page=&Apache::loncommon::start_page('Modify Parameters');
    my $breadcrumbs = &Apache::lonhtmlcommon::breadcrumbs('Overview');
    $r->print(<<ENDOVER);
$start_page
$breadcrumbs
<form method="post" action="/adm/parmset?action=setoverview" name="parmform">
ENDOVER
# Store modified

    &storedata($r,$crs,$dom);

# Read modified data

    my $resourcedata=&readdata($crs,$dom);


    my $sortorder=$env{'form.sortorder'};
    unless ($sortorder) { $sortorder='realmstudent'; }
    &sortmenu($r,$sortorder);

# List data

    my $foundkeys=&listdata($r,$resourcedata,$resourcedata,$sortorder);

    $r->print(&tableend().'<p>'.
	($foundkeys?'<input type="submit" value="'.&mt('Modify Parameters').'" />':&mt('There are no parameters.')).'</p></form>'.
	      &Apache::loncommon::end_page());
}

sub clean_parameters {
    my ($r) = @_;
    my $dom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $crs = $env{'course.'.$env{'request.course.id'}.'.num'};

    my $start_page=&Apache::loncommon::start_page('Clean Parameters');
    my $breadcrumbs = &Apache::lonhtmlcommon::breadcrumbs('Clean');
    $r->print(<<ENDOVER);
$start_page
$breadcrumbs
<form method="post" action="/adm/parmset?action=cleanparameters" name="parmform">
ENDOVER
# Store modified

    &storedata($r,$crs,$dom);

# Read modified data

    my $resourcedata=&readdata($crs,$dom);

# List data

    $r->print('<h3>'.
	      &mt('These parameters refer to resources that do not exist.').
	      '</h3>'.
	      '<input type="submit" value="'.&mt('Delete Checked Parameters').'" />'.'<br />'.
	      '<br />');
    $r->print(&Apache::loncommon::start_data_table().
	      '<tr>'.
	      '<th>'.&mt('Delete').'</th>'.
	      '<th>'.&mt('Parameter').'</th>'.
	      '</tr>');
    foreach my $thiskey (sort(keys(%{$resourcedata}))) {
	next if (!exists($resourcedata->{$thiskey.'.type'})
		 && $thiskey=~/\.type$/);
	my %data = &parse_key($thiskey);
	if (exists($data{'realm_exists'})
	    && !$data{'realm_exists'}) {
	    $r->print(&Apache::loncommon::start_data_table_row().
		      '<tr>'.
		      '<td><input type="checkbox" name="del_'.$thiskey.'" /></td>'		      );
	    
	    $r->print('<td>');
	    $r->print(&mt('Parameter: "[_1]" with value: "[_2]"',
			  &standard_parameter_names($data{'parameter_name'}),
			  $resourcedata->{$thiskey}));
	    $r->print('<br />');
	    if ($data{'scope_type'} eq 'all') {
		$r->print(&mt('All users'));
	    } elsif ($data{'scope_type'} eq 'user') {
		$r->print(&mt('User: [_1]',join(':',@{$data{'scope'}})));
	    } elsif ($data{'scope_type'} eq 'section') {
		$r->print(&mt('Section: [_1]',$data{'scope'}));
	    } elsif ($data{'scope_type'} eq 'group') {
		$r->print(&mt('Group: [_1]',$data{'scope'}));
	    }
	    $r->print('<br />');
	    if ($data{'realm_type'} eq 'all') {
		$r->print(&mt('All Resources'));
	    } elsif ($data{'realm_type'} eq 'folder') {
		$r->print(&mt('Folder: [_1]'),$data{'realm'});
	    } elsif ($data{'realm_type'} eq 'symb') {
		my ($map,$resid,$url) =
		    &Apache::lonnet::decode_symb($data{'realm'});
		$r->print(&mt('Resource: [_1] <br />&nbsp;&nbsp;&nbsp;with ID: [_2] <br />&nbsp;&nbsp;&nbsp;in folder [_3]',
			      $url,$resid,$map));
	    }
	    $r->print(&mt('Part: [_1]',$data{'parameter_part'}));
	    $r->print('</td></tr>');
	
	}
    }
    $r->print(&Apache::loncommon::end_data_table().'<p>'.
	      '<input type="submit" value="'.&mt('Delete Checked Parameters').'" />'.
	      '</p></form>'.
	      &Apache::loncommon::end_page());
}

sub parse_key {
    my ($key) = @_;
    my %data;
    my ($middle,$part,$name)=
	($key=~/^$env{'request.course.id'}\.(?:(.+)\.)*([\w\s]+)\.(\w+)$/);
    $data{'scope_type'} = 'all';
    if ($middle=~/^\[(.*)\]/) {
       	$data{'scope'} = $1;
	if ($data{'scope'}=~/^useropt\:($match_username)\:($match_domain)/) {
	    $data{'scope_type'} = 'user';
	    $data{'scope'} = [$1,$2];
	} else {
	    #FIXME check for group scope
	    $data{'scope_type'} = 'section';
	}
	$middle=~s/^\[(.*)\]//;
    }
    $middle=~s/\.+$//;
    $middle=~s/^\.+//;
    $data{'realm_type'}='all';
    if ($middle=~/^(.+)\_\_\_\(all\)$/) {
	$data{'realm'} = $1;
	$data{'realm_type'} = 'folder';
	$data{'realm_title'} = &Apache::lonnet::gettitle($data{'realm'});
	($data{'realm_exists'}) = &Apache::lonnet::is_on_map($data{'realm'});
	&Apache::lonnet::logthis($1." siad ".	$data{'realm_exists'} );
    } elsif ($middle) {
	$data{'realm'} = $middle;
	$data{'realm_type'} = 'symb';
	$data{'realm_title'} = &Apache::lonnet::gettitle($data{'realm'});
	my ($map,$resid,$url) = &Apache::lonnet::decode_symb($data{'realm'});
	$data{'realm_exists'} = &Apache::lonnet::symbverify($data{'realm'},$url);
    }
    
    $data{'parameter_part'} = $part;
    $data{'parameter_name'} = $name;

    return %data;
}

##################################################
##################################################

=pod

=item check_cloners

Checks if new users included in list of allowed cloners
are valid users.  Replaces supplied list with 
cleaned list containing only users with valid usernames
and domains.

Inputs: $clonelist, $oldcloner 
where $clonelist is ref to array of requested cloners,
and $oldcloner is ref to array of currently allowed
cloners.

Returns: string - comma separated list of requested
cloners (username:domain) who do not exist in system.

=item change_clone

Modifies the list of courses a user can clone (stored
in the user's environment.db file), called when a
change is made to the list of users allowed to clone
a course.

Inputs: $action,$cloner
where $action is add or drop, and $cloner is identity of 
user for whom cloning ability is to be changed in course. 

=cut
                                                                                            
##################################################
##################################################

sub extract_cloners {
    my ($clonelist,$allowclone) = @_;
    if ($clonelist =~ /,/) {
        @{$allowclone} = split/,/,$clonelist;
    } else {
        $$allowclone[0] = $clonelist;
    }
}


sub check_cloners {
    my ($clonelist,$oldcloner) = @_;
    my ($clean_clonelist,$disallowed);
    my @allowclone = ();
    &extract_cloners($$clonelist,\@allowclone);
    foreach my $currclone (@allowclone) {
        if (!grep/^$currclone$/,@$oldcloner) {
            my ($uname,$udom) = split/:/,$currclone;
            if ($uname && $udom) {
                if (&Apache::lonnet::homeserver($uname,$udom) eq 'no_host') {
                    $disallowed .= $currclone.',';   
                } else {
                    $clean_clonelist .= $currclone.',';
                }
            }
        } else {
            $clean_clonelist .= $currclone.',';
        }
    }
    if ($disallowed) {
        $disallowed =~ s/,$//;
    }
    if ($clean_clonelist) {
        $clean_clonelist =~ s/,$//;
    }
    $$clonelist = $clean_clonelist;
    return $disallowed;
}  

sub change_clone {
    my ($clonelist,$oldcloner) = @_;
    my ($uname,$udom);
    my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
    my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $clone_crs = $cnum.':'.$cdom;
    
    if ($cnum && $cdom) {
        my @allowclone;
        &extract_cloners($clonelist,\@allowclone);
        foreach my $currclone (@allowclone) {
            if (!grep/^$currclone$/,@$oldcloner) {
                ($uname,$udom) = split/:/,$currclone;
                if ($uname && $udom) {
                    unless (&Apache::lonnet::homeserver($uname,$udom) eq 'no_host') {
                        my %currclonecrs = &Apache::lonnet::dump('environment',$udom,$uname,'cloneable');
                        if ($currclonecrs{'cloneable'} !~ /\Q$clone_crs\E/) {
                            if ($currclonecrs{'cloneable'} eq '') {
                                $currclonecrs{'cloneable'} = $clone_crs;
                            } else {
                                $currclonecrs{'cloneable'} .= ','.$clone_crs;
                            }
                            &Apache::lonnet::put('environment',\%currclonecrs,$udom,$uname);
                        }
                    }
                }
            }
        }
        foreach my $oldclone (@$oldcloner) {
            if (!grep/^$oldclone$/,@allowclone) {
                ($uname,$udom) = split/:/,$oldclone;
                if ($uname && $udom) {
                    unless (&Apache::lonnet::homeserver($uname,$udom) eq 'no_host') {
                        my %currclonecrs = &Apache::lonnet::dump('environment',$udom,$uname,'cloneable');
                        my %newclonecrs = ();
                        if ($currclonecrs{'cloneable'} =~ /\Q$clone_crs\E/) {
                            if ($currclonecrs{'cloneable'} =~ /,/) {
                                my @currclonecrs = split/,/,$currclonecrs{'cloneable'};
                                foreach (@currclonecrs) {
                                    unless ($_ eq $clone_crs) {
                                        $newclonecrs{'cloneable'} .= $_.',';
                                    }
                                }
                                $newclonecrs{'cloneable'} =~ s/,$//;
                            } else {
                                $newclonecrs{'cloneable'} = '';
                            }
                            &Apache::lonnet::put('environment',\%newclonecrs,$udom,$uname);
                        }
                    }
                }
            }
        }
    }
}


##################################################
##################################################

=pod

=item * header

Output html header for page

=cut

##################################################
##################################################
sub header {
    return &Apache::loncommon::start_page('Parameter Manager');
}
##################################################
##################################################
sub print_main_menu {
    my ($r,$parm_permission)=@_;
    #
    $r->print(<<ENDMAINFORMHEAD);
<form method="post" enctype="multipart/form-data"
      action="/adm/parmset" name="studentform">
ENDMAINFORMHEAD
#
    my $cnum = $env{'course.'.$env{'request.course.id'}.'.num'};
    my $cdom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $vgr  = &Apache::lonnet::allowed('vgr',$env{'request.course.id'});

    my @menu =
        ( { divider=>'Settings for Your Course',
	  },
          { text => 'Set Course Environment',
	    action => 'crsenv',
            permission => $parm_permission,
            help => 'Course_Environment',
            },
          { text => 'Set Portfolio Metadata',
	    action => 'setrestrictmeta',
            permission => $parm_permission,
            },
	  { text => 'Manage Course Slots',
	    url => '/adm/slotrequest?command=showslots',
	    permission => $vgr,
            },
          { text => 'Set Parameter Setting Default Actions',
            action => 'setdefaults',
            permission => $parm_permission,
            },          
	  { divider => 'New and Existing Parameter Settings for Your Resources',
	    },
          { text => 'Set/Modify Resource Parameters - Helper Mode',
            url => '/adm/helper/parameter.helper',
            permission => $parm_permission,
            help => 'Parameter_Helper',
            },
 	  { text => 'Set/Modify Resource Parameters - Overview Mode',
            action => 'newoverview',
            permission => $parm_permission,
            help => 'Parameter_Overview',
            },
          { text => 'Set/Modify Resource Parameters - Table Mode',
            action => 'settable',
            permission => $parm_permission,
            help => 'Table_Mode',
            },
           { divider => 'Existing Parameter Settings for Your Resources',
	  },
	  { text => 'Modify Resource Parameters - Overview Mode',
            action => 'setoverview',
            permission => $parm_permission,
            help => 'Parameter_Overview',
 	    },          
	  { text => 'Parameter Change Log and Course Blog Posting/User Notification',
            action => 'parameterchangelog',
            permission => $parm_permission,
            },
          );
    my $menu_html = '';
    foreach my $menu_item (@menu) {
	if ($menu_item->{'divider'}) {
	    $menu_html .= '<h3>'.&mt($menu_item->{'divider'}).'</h3>';
	    next;
	}
        next if (! $menu_item->{'permission'});
        $menu_html.='<p>';
        $menu_html.='<span class="LC_parm_menu_item">';
        if (exists($menu_item->{'url'})) {
            $menu_html.=qq{<a href="$menu_item->{'url'}">};
        } else {
            $menu_html.=
                qq{<a href="/adm/parmset?action=$menu_item->{'action'}">};
        }
        $menu_html.= &mt($menu_item->{'text'}).'</a></span>';
        if (exists($menu_item->{'help'})) {
            $menu_html.=
                &Apache::loncommon::help_open_topic($menu_item->{'help'});
        }
        $menu_html.='</p>'.$/;
    }
    $r->print($menu_html);
    return;
}
### Set portfolio metadata
sub output_row {
    my ($r, $field_name, $field_text, $added_flag) = @_;
    my $row_class;
    my $output;
    my $options=$env{'course.'.$env{'request.course.id'}.'.metadata.'.$field_name.'.options'};
    my $values=$env{'course.'.$env{'request.course.id'}.'.metadata.'.$field_name.'.values'};
    if (!defined($options)) {
        $options = 'active,stuadd';
        $values = '';
    }
    if (!($options =~ /deleted/)) {
        $output = &Apache::loncommon::start_data_table_row();
        $output .= '<td><span class="LC_metadata"><strong>'.$field_text.':</strong></span></td>';
        # $output .= '<td><strong>'.$field_text.':</strong></td>';
        $output .= '<td><span class="LC_metadata"><input name="'.$field_name.'_values" type="text" value="'.$values.'" size="80" /></span></td>';
        $output .= &Apache::loncommon::end_data_table_row();
        my @options= ( ['active', 'Show to student'],
		   ['onlyone','Student may select only one choice'],
		   ['stuadd', 'Student may type choices']);
        if ($added_flag) {
            push @options,['deleted', 'Delete Metadata Field'];
        }
        foreach my $opt (@options) {
	    my $checked = ($options =~ m/$opt->[0]/) ? ' checked="checked" ' : '' ;
	    $output .= &Apache::loncommon::continue_data_table_row();
	    $output .= '<td colspan="2">'.('&nbsp;' x 5).'<span class="LC_metadata"><label><input type="checkbox" name="'.
	        $field_name.'_'.$opt->[0].'" value="yes"'.$checked.' />'.
	        &mt($opt->[1]).'</label></span> </td>';
	    $output .= &Apache::loncommon::end_data_table_row();
	}
	
    }
    return ($output);
}
sub order_meta_fields {
    my ($r)=@_;
    my $idx = 1;
    my $dom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $crs = $env{'course.'.$env{'request.course.id'}.'.num'};
    $r->print(&Apache::loncommon::start_page('Order Metadata Fields'));
    &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>"/adm/parmset?action=setrestrictmeta",
              text=>"Restrict Metadata"},
             {text=>"Order Metadata"});
    $r->print(&Apache::lonhtmlcommon::breadcrumbs('Order Metadata'));
    if ($env{'form.storeorder'}) {
        my $newpos = $env{'form.newpos'} - 1;
        my $currentpos = $env{'form.currentpos'} - 1;
        my @neworder = ();
        my @oldorder = split /,/,$env{'course.'.$env{'request.course.id'}.'.metadata.addedorder'};
        my $i;
        if ($newpos > $currentpos) {
        # moving stuff up
            for ($i=0;$i<$currentpos;$i++) {
        	$neworder[$i]=$oldorder[$i];
            }
            for ($i=$currentpos;$i<$newpos;$i++) {
        	$neworder[$i]=$oldorder[$i+1];
            }
            $neworder[$newpos]=$oldorder[$currentpos];
            for ($i=$newpos+1;$i<=$#oldorder;$i++) {
        	$neworder[$i]=$oldorder[$i];
            }
        } else {
        # moving stuff down
    	    for ($i=0;$i<$newpos;$i++) {
    	        $neworder[$i]=$oldorder[$i];
    	    }
    	    $neworder[$newpos]=$oldorder[$currentpos];
    	    for ($i=$newpos+1;$i<$currentpos+1;$i++) {
    	        $neworder[$i]=$oldorder[$i-1];
    	    }
    	    for ($i=$currentpos+1;$i<=$#oldorder;$i++) {
    	        $neworder[$i]=$oldorder[$i];
    	    }
        }
	my $ordered_fields = join ",", @neworder;
        my $put_result = &Apache::lonnet::put('environment',
                           {'metadata.addedorder'=>$ordered_fields},$dom,$crs); 	
	&Apache::lonnet::appenv('course.'.$env{'request.course.id'}.'.metadata.addedorder' => $ordered_fields);
    }
    my $fields = &get_added_meta_fieldnames();
    my $ordered_fields;
    my @fields_in_order = split /,/,$env{'course.'.$env{'request.course.id'}.'.metadata.addedorder'};
    if (!@fields_in_order) {
        # no order found, pick sorted order then create metadata.addedorder key.
        foreach my $key (sort keys %$fields) {
            push @fields_in_order, $key;
            $ordered_fields = join ",", @fields_in_order;
        }
        my $put_result = &Apache::lonnet::put('environment',
                            {'metadata.addedorder'=>$ordered_fields},$dom,$crs); 
    } 
    $r->print('<table>');
    my $num_fields = scalar(@fields_in_order);
    foreach my $key (@fields_in_order) {
        $r->print('<tr><td>');
        $r->print('<form method="post" action="">');
        $r->print('<select name="newpos" onChange="this.form.submit()">');
        for (my $i = 1;$i le $num_fields;$i ++) {
            if ($i eq $idx) {
                $r->print('<option value="'.$i.'"  SELECTED>('.$i.')</option>');
            } else {
                $r->print('<option value="'.$i.'">'.$i.'</option>');
            }
        }
        $r->print('</select></td><td>');
        $r->print('<input type="hidden" name="currentpos" value="'.$idx.'" />');
        $r->print('<input type="hidden" name="storeorder" value="true" />');
        $r->print('</form>');
        $r->print($$fields{$key}.'</td></tr>');
        $idx ++;
    }
    $r->print('</table>');
    return 'ok';
}
sub addmetafield {
    my ($r)=@_;
    $r->print(&Apache::loncommon::start_page('Add Metadata Field'));
    $r->print(&Apache::lonhtmlcommon::breadcrumbs('Add Metadata Field'));
    my $dom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $crs = $env{'course.'.$env{'request.course.id'}.'.num'};
    if (exists($env{'form.undelete'})) {
        my @meta_fields = &Apache::loncommon::get_env_multiple('form.undelete');
        foreach my $meta_field(@meta_fields) {
            my $options = $env{'course.'.$env{'request.course.id'}.'.metadata.'.$meta_field.'.options'};
            $options =~ s/deleted//;
            $options =~ s/,,/,/;
            my $put_result = &Apache::lonnet::put('environment',
                                        {'metadata.'.$meta_field.'.options'=>$options},$dom,$crs);
                                        
            $r->print('Undeleted Metadata Field <strong>'.$env{'course.'.$env{'request.course.id'}.'.metadata.'.$meta_field.'.added'}."</strong> with result ".$put_result.'<br />');
        }
        $r->print('<form action="" method="post">');
        $r->print('<input type="hidden" name="action" value="setrestrictmeta" />');
        $r->print('<input type="submit" value="Continue" />');
    } elsif (exists($env{'form.fieldname'})) {
        my $meta_field = $env{'form.fieldname'};
        my $display_field = $env{'form.fieldname'};
        $meta_field =~ s/\W/_/g;
        $meta_field =~ tr/A-Z/a-z/;
        my $put_result = &Apache::lonnet::put('environment',
                            {'metadata.'.$meta_field.'.values'=>"",
                             'metadata.'.$meta_field.'.added'=>"$display_field",
                             'metadata.'.$meta_field.'.options'=>""},$dom,$crs);
        $r->print('Added new Metadata Field '.$env{'form.fieldname'}." with result ".$put_result);
    } else {
        my $fields = &get_deleted_meta_fieldnames();
        if ($fields) {
            $r->print('You may undelete previously deleted fields.<br />Check those you wish to undelete and click Undelete.<br />');
            $r->print('<form method="post" action="">');
            foreach my $key(keys(%$fields)) {
                $r->print('<input type="checkbox" name="undelete" value="'.$key.'" />'.$$fields{$key}.'<br /');
            }
            $r->print('<input type="submit" name="undelete" value="Undelete" />');
            $r->print('</form>');
        }
        $r->print('<hr /><strong>Or</strong> you may enter a new metadata field name.<form method="post" action="/adm/parmset?action=addmetadata"');
        $r->print('<input type="text" name="fieldname" /><br />');
        $r->print('<input type="submit" value="Add Metadata Field" />');
        $r->print('</form>');
    }
}
sub setrestrictmeta {
    my ($r)=@_;
    my $next_meta;
    my $output;
    my $item_num;
    my $put_result;
    $r->print(&Apache::loncommon::start_page('Restrict Metadata'));
    $r->print(&Apache::lonhtmlcommon::breadcrumbs('Restrict Metadata'));
    my $dom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $crs = $env{'course.'.$env{'request.course.id'}.'.num'};
    my $key_base = $env{'course.'.$env{'request.course.id'}.'.'};
    my $save_field = '';
    if ($env{'form.restrictmeta'}) {
        foreach my $field (sort(keys(%env))) {
            if ($field=~m/^form.(.+)_(.+)$/) {
                my $options;
                my $meta_field = $1;
                my $meta_key = $2;
                if ($save_field ne $meta_field) {
                    $save_field = $meta_field;
            	    if ($env{'form.'.$meta_field.'_stuadd'}) {
            	        $options.='stuadd,';
            	    } 
            	    if ($env{'form.'.$meta_field.'_onlyone'}) {
            	        $options.='onlyone,';
            	    } 
            	    if ($env{'form.'.$meta_field.'_active'}) {
            	        $options.='active,';
            	    }
            	    if ($env{'form.'.$meta_field.'_deleted'}) {
            	        $options.='deleted,';
            	    }
            	    
                    my $name = $save_field;
                     $put_result = &Apache::lonnet::put('environment',
                                                  {'metadata.'.$meta_field.'.options'=>$options,
                                                   'metadata.'.$meta_field.'.values'=>$env{'form.'.$meta_field.'_values'},
                                                   },$dom,$crs);
                }
            }
        }
    }
    &Apache::lonnet::coursedescription($env{'request.course.id'},
				       {'freshen_cache' => 1});
    # Get the default metadata fields
    my %metadata_fields = &Apache::lonmeta::fieldnames('portfolio');
    # Now get possible added metadata fields
    my $added_metadata_fields = &get_added_meta_fieldnames(\%metadata_fields);
    my $row_alt = 1;
    $output .= &Apache::loncommon::start_data_table();
    foreach my $field (sort(keys(%metadata_fields))) {
        if ($field ne 'courserestricted') {
            $row_alt = $row_alt ? 0 : 1;
	    $output.= &output_row($r, $field, $metadata_fields{$field});
	}
    }
    my $added_flag = 1;
    foreach my $field (sort(keys(%$added_metadata_fields))) {
        $row_alt = $row_alt ? 0 : 1;
        $output.= &output_row($r, $field, $$added_metadata_fields{$field},$added_flag, $row_alt);
    }
    $output .= &Apache::loncommon::end_data_table();
    $r->print(<<ENDenv);       
        <form method="post" action="/adm/parmset?action=setrestrictmeta" name="form">
        $output
        <input type="submit" name="restrictmeta" value="Update Metadata Restrictions" />
        </form><br />
        <form method="post" action="/adm/parmset?action=addmetadata" name="form1">
        <input type="submit" name="restrictmeta" value="Add a Metadata Field" />
        </form>
        <br />
        <form method="post" action="/adm/parmset?action=ordermetadata" name="form2">
        <input type="submit" name="restrictmeta" value="Order Metadata Fields" />
        </form>
ENDenv
    $r->print(&Apache::loncommon::end_page());
    return 'ok';
}
##################################################
sub get_added_meta_fieldnames {
    my %fields;
    foreach my $key(%env) {
        if ($key =~ m/\.metadata\.(.+)\.added$/) {
            my $field_name = $1;
            my ($display_field_name) = $env{$key};
            $fields{$field_name} = $display_field_name;
        }
    }
    return \%fields;
}
sub get_deleted_meta_fieldnames {
    my %fields;
    my ($default_fields) = @_;
    foreach my $key(%env) {
        if ($key =~ m/\.metadata\.(.+)\.added$/) {
            my $field_name = $1;
            if ($env{'course.'.$env{'request.course.id'}.'.metadata.'.$field_name.'.options'} =~ m/deleted/) {
                my ($display_field_name) = $env{$key};
                $fields{$field_name} = $display_field_name;
            }
        }
    }
    return \%fields;
}
sub defaultsetter {
    my ($r) = @_;

    my $start_page = 
	&Apache::loncommon::start_page('Parameter Setting Default Actions');
    my $breadcrumbs = &Apache::lonhtmlcommon::breadcrumbs('Defaults');
    $r->print(<<ENDDEFHEAD);
$start_page
$breadcrumbs
<form method="post" action="/adm/parmset?action=setdefaults" name="defaultform">
ENDDEFHEAD

    my $dom = $env{'course.'.$env{'request.course.id'}.'.domain'};
    my $crs = $env{'course.'.$env{'request.course.id'}.'.num'};
    my @ids=();
    my %typep=();
    my %keyp=();
    my %allparms=();
    my %allparts=();
    my %allmaps=();
    my %mapp=();
    my %symbp=();
    my %maptitles=();
    my %uris=();
    my %keyorder=&standardkeyorder();
    my %defkeytype=();

    &extractResourceInformation(\@ids, \%typep,\%keyp, \%allparms, \%allparts, \%allmaps, 
				\%mapp, \%symbp,\%maptitles,\%uris,
				\%keyorder,\%defkeytype);
    if ($env{'form.storerules'}) {
	my %newrules=();
	my @delrules=();
	my %triggers=();
	foreach my $key (keys(%env)) {
            if ($key=~/^form\.(\w+)\_action$/) {
		my $tempkey=$1;
		my $action=$env{$key};
                if ($action) {
		    $newrules{$tempkey.'_action'}=$action;
		    if ($action ne 'default') {
			my ($whichaction,$whichparm)=($action=~/^(.*\_)([^\_]+)$/);
			$triggers{$whichparm}.=$tempkey.':';
		    }
		    $newrules{$tempkey.'_type'}=$defkeytype{$tempkey};
		    if (&isdateparm($defkeytype{$tempkey})) {
			$newrules{$tempkey.'_days'}=$env{'form.'.$tempkey.'_days'};
			$newrules{$tempkey.'_hours'}=$env{'form.'.$tempkey.'_hours'};
			$newrules{$tempkey.'_min'}=$env{'form.'.$tempkey.'_min'};
			$newrules{$tempkey.'_sec'}=$env{'form.'.$tempkey.'_sec'};
		    } else {
			$newrules{$tempkey.'_value'}=$env{'form.'.$tempkey.'_value'};
			$newrules{$tempkey.'_triggervalue'}=$env{'form.'.$tempkey.'_triggervalue'};
		    }
		} else {
		    push(@delrules,$tempkey.'_action');
		    push(@delrules,$tempkey.'_type');
		    push(@delrules,$tempkey.'_hours');
		    push(@delrules,$tempkey.'_min');
		    push(@delrules,$tempkey.'_sec');
		    push(@delrules,$tempkey.'_value');
		}
	    }
	}
	foreach my $key (keys %allparms) {
	    $newrules{$key.'_triggers'}=$triggers{$key};
	}
	&Apache::lonnet::put('parmdefactions',\%newrules,$dom,$crs);
	&Apache::lonnet::del('parmdefactions',\@delrules,$dom,$crs);
	&resetrulescache();
    }
    my %lt=&Apache::lonlocal::texthash('days' => 'Days',
				       'hours' => 'Hours',
				       'min' => 'Minutes',
				       'sec' => 'Seconds',
				       'yes' => 'Yes',
				       'no' => 'No');
    my @standardoptions=('','default');
    my @standarddisplay=('',&mt('Default value when manually setting'));
    my @dateoptions=('','default');
    my @datedisplay=('',&mt('Default value when manually setting'));
    foreach my $tempkey (&keysindisplayorder(\%allparms,\%keyorder)) {
	unless ($tempkey) { next; }
	push @standardoptions,'when_setting_'.$tempkey;
	push @standarddisplay,&mt('Automatically set when setting ').$tempkey;
	if (&isdateparm($defkeytype{$tempkey})) {
	    push @dateoptions,'later_than_'.$tempkey;
	    push @datedisplay,&mt('Automatically set later than ').$tempkey;
	    push @dateoptions,'earlier_than_'.$tempkey;
	    push @datedisplay,&mt('Automatically set earlier than ').$tempkey;
	} 
    }
$r->print(&mt('Manual setting rules apply to all interfaces.').'<br />'.
	  &mt('Automatic setting rules apply to table mode interfaces only.'));
    $r->print("\n".&Apache::loncommon::start_data_table().
	      &Apache::loncommon::start_data_table_header_row().
	      "<th>".&mt('Rule for parameter').'</th><th>'.
	      &mt('Action').'</th><th>'.&mt('Value').'</th>'.
	      &Apache::loncommon::end_data_table_header_row());
    foreach my $tempkey (&keysindisplayorder(\%allparms,\%keyorder)) {
	unless ($tempkey) { next; }
	$r->print("\n".&Apache::loncommon::start_data_table_row().
		  "<td>".$allparms{$tempkey}."\n<br />(".$tempkey.')</td><td>');
	my $action=&rulescache($tempkey.'_action');
	$r->print('<select name="'.$tempkey.'_action">');
	if (&isdateparm($defkeytype{$tempkey})) {
	    for (my $i=0;$i<=$#dateoptions;$i++) {
		if ($dateoptions[$i]=~/\_$tempkey$/) { next; }
		$r->print("\n<option value='$dateoptions[$i]'".
			  ($dateoptions[$i] eq $action?' selected="selected"':'').
			  ">$datedisplay[$i]</option>");
	    }
	} else {
	    for (my $i=0;$i<=$#standardoptions;$i++) {
		if ($standardoptions[$i]=~/\_$tempkey$/) { next; }
		$r->print("\n<option value='$standardoptions[$i]'".
			  ($standardoptions[$i] eq $action?' selected="selected"':'').
			  ">$standarddisplay[$i]</option>");
	    }
	}
	$r->print('</select>');
	unless (&isdateparm($defkeytype{$tempkey})) {
	    $r->print("\n<br />".&mt('Triggering value(s) of other parameter (optional, comma-separated):').
		      '<input type="text" size="20" name="'.$tempkey.'_triggervalue" value="'.&rulescache($tempkey.'_triggervalue').'" />');
	}
	$r->print("\n</td><td>\n");

        if (&isdateparm($defkeytype{$tempkey})) {
	    my $days=&rulescache($tempkey.'_days');
	    my $hours=&rulescache($tempkey.'_hours');
	    my $min=&rulescache($tempkey.'_min');
	    my $sec=&rulescache($tempkey.'_sec');
	    $r->print(<<ENDINPUTDATE);
<input name="$tempkey\_days" type="text" size="4" value="$days" />$lt{'days'}<br />
<input name="$tempkey\_hours" type="text" size="4" value="$hours" />$lt{'hours'}<br />
<input name="$tempkey\_min" type="text" size="4" value="$min" />$lt{'min'}<br />
<input name="$tempkey\_sec" type="text" size="4" value="$sec" />$lt{'sec'}
ENDINPUTDATE
	} elsif ($defkeytype{$tempkey} eq 'string_yesno') {
            my $yeschecked='';
            my $nochecked='';
            if (&rulescache($tempkey.'_value') eq 'yes') { $yeschecked='checked="checked"'; }
            if (&rulescache($tempkey.'_value') eq 'no') { $nochecked='checked="checked"'; }

	    $r->print(<<ENDYESNO);
<label><input type="radio" name="$tempkey\_value" value="yes" $yeschecked /> $lt{'yes'}</label><br />
<label><input type="radio" name="$tempkey\_value" value="no" $nochecked /> $lt{'no'}</label>
ENDYESNO
        } else {
	    $r->print('<input type="text" size="20" name="'.$tempkey.'_value" value="'.&rulescache($tempkey.'_value').'" />');
	}
        $r->print('</td>'.&Apache::loncommon::end_data_table_row());
    }
    $r->print(&Apache::loncommon::end_data_table().
	      "\n<input type='submit' name='storerules' value='".
	      &mt('Store Rules')."' /></form>\n".
	      &Apache::loncommon::end_page());
    return;
}

sub components {
    my ($key,$uname,$udom,$exeuser,$exedomain,$typeflag)=@_;

    if ($typeflag) {
	$key=~s/\.type$//;
    }

    my ($middle,$part,$name)=
	($key=~/^$env{'request.course.id'}\.(?:(.+)\.)*([\w\s]+)\.(\w+)$/);
    my $issection;

    my $section=&mt('All Students');
    if ($middle=~/^\[(.*)\]/) {
	$issection=$1;
	$section=&mt('Group/Section').': '.$issection;
	$middle=~s/^\[(.*)\]//;
    }
    $middle=~s/\.+$//;
    $middle=~s/^\.+//;
    if ($uname) {
	$section=&mt('User').": ".&Apache::loncommon::plainname($uname,$udom);
	$issection='';
    }
    my $realm='<span class="LC_parm_scope_all">'.&mt('All Resources').'</span>';
    my $realmdescription=&mt('all resources'); 
    if ($middle=~/^(.+)\_\_\_\(all\)$/) {
	$realm='<span class="LC_parm_scope_folder">'.&mt('Folder/Map').': '.&Apache::lonnet::gettitle($1).' <span class="LC_parm_folder"><br />('.$1.')</span></span>';
 	$realmdescription=&mt('folder').' '.&Apache::lonnet::gettitle($1);
   } elsif ($middle) {
	my ($map,$id,$url)=&Apache::lonnet::decode_symb($middle);
	$realm='<span class="LC_parm_scope_resource">'.&mt('Resource').': '.&Apache::lonnet::gettitle($middle).' <br /><span class="LC_parm_symb">('.$url.' in '.$map.' id: '.$id.')</span></span>';
	$realmdescription=&mt('resource').' '.&Apache::lonnet::gettitle($middle);
    }
    my $what=$part.'.'.$name;
    return ($realm,$section,$name,$part,
	    $what,$middle,$uname,$udom,$issection,$realmdescription);
}

my %standard_parms;
sub load_parameter_names {
    open(my $config,"<$Apache::lonnet::perlvar{'lonTabDir'}/packages.tab");
    while (my $configline=<$config>) {
	if ($configline !~ /\S/ || $configline=~/^\#/) { next; }
	chomp($configline);
	my ($short,$plain)=split(/:/,$configline);
	my (undef,$name,$type)=split(/\&/,$short,3);
	if ($type eq 'display') {
	    $standard_parms{$name} = $plain;
	}
    }
    close($config);
    $standard_parms{'int_pos'}      = 'Positive Integer';
    $standard_parms{'int_zero_pos'} = 'Positive Integer or Zero';
    %standard_parms=&Apache::lonlocal::texthash(%standard_parms);	
}

sub standard_parameter_names {
    my ($name)=@_;
    if (!%standard_parms) {
	&load_parameter_names();
    }
    if ($standard_parms{$name}) {
	return $standard_parms{$name}; 
    } else { 
	return $name; 
    }
}

#
# Parameter Change Log
#


sub parm_change_log {
    my ($r)=@_;
    $r->print(&Apache::loncommon::start_page('Parameter Change Log'));
    $r->print(&Apache::lonhtmlcommon::breadcrumbs('Parameter Change Log'));

    my %parmlog=&Apache::lonnet::dump('nohist_parameterlog',
				      $env{'course.'.$env{'request.course.id'}.'.domain'},
				      $env{'course.'.$env{'request.course.id'}.'.num'});

    if ((keys(%parmlog))[0]=~/^error\:/) { undef(%parmlog); }

    $r->print('<form action="/adm/parmset?action=parameterchangelog"
                     method="post" name="parameterlog">');
    
    my %saveable_parameters = ('show' => 'scalar',);
    &Apache::loncommon::store_course_settings('parameter_log',
                                              \%saveable_parameters);
    &Apache::loncommon::restore_course_settings('parameter_log',
                                                \%saveable_parameters);
    $r->print(&Apache::loncommon::display_filter().
              '<label>'.&Apache::lonhtmlcommon::checkbox('includetypes',$env{'form.includetypes'},'1').
	      ' '.&mt('Include parameter types').'</label>'.
	      '<input type="submit" value="'.&mt('Display').'" /></form>');

    my $courseopt=&Apache::lonnet::get_courseresdata($env{'course.'.$env{'request.course.id'}.'.num'},
						     $env{'course.'.$env{'request.course.id'}.'.domain'});
    $r->print(&Apache::loncommon::start_data_table().&Apache::loncommon::start_data_table_header_row().
	      '<th>'.&mt('Time').'</th><th>'.&mt('User').'</th><th>'.&mt('Extent').'</th><th>'.&mt('Users').'</th><th>'.
	      &mt('Parameter').'</th><th>'.&mt('Part').'</th><th>'.&mt('New Value').'</th><th>'.&mt('Announce').'</th>'.
	      &Apache::loncommon::end_data_table_header_row());
    my $shown=0;
    my $folder='';
    if ($env{'form.displayfilter'} eq 'currentfolder') {
	my $last='';
	if (tie(my %hash,'GDBM_File',$env{'request.course.fn'}.'_symb.db',
		&GDBM_READER(),0640)) {
	    $last=$hash{'last_known'};
	    untie(%hash);
	}
	if ($last) { ($folder) = &Apache::lonnet::decode_symb($last); }
    }
    foreach my $id (sort { $parmlog{$b}{'exe_time'}<=>$parmlog{$a}{'exe_time'} } (keys(%parmlog))) {
        my @changes=keys(%{$parmlog{$id}{'logentry'}});
	my $count = 0;
	my $time =
	    &Apache::lonlocal::locallocaltime($parmlog{$id}{'exe_time'});
	my $plainname = 
	    &Apache::loncommon::plainname($parmlog{$id}{'exe_uname'},
					  $parmlog{$id}{'exe_udom'});
	my $about_me_link = 
	    &Apache::loncommon::aboutmewrapper($plainname,
					       $parmlog{$id}{'exe_uname'},
					       $parmlog{$id}{'exe_udom'});
	my $send_msg_link='';
	if ((($parmlog{$id}{'exe_uname'} ne $env{'user.name'}) 
	     || ($parmlog{$id}{'exe_udom'} ne $env{'user.domain'}))) {
	    $send_msg_link ='<br />'.
		&Apache::loncommon::messagewrapper(&mt('Send message'),
						   $parmlog{$id}{'exe_uname'},
						   $parmlog{$id}{'exe_udom'});
	}
	my $row_start=&Apache::loncommon::start_data_table_row();
	my $makenewrow=0;
	my %istype=();
	my $output;
	foreach my $changed (reverse(sort(@changes))) {
            my $value=$parmlog{$id}{'logentry'}{$changed};
	    my $typeflag = ($changed =~/\.type$/ &&
			    !exists($parmlog{$id}{'logentry'}{$changed.'.type'}));
            my ($realm,$section,$parmname,$part,$what,$middle,$uname,$udom,$issection,$realmdescription)=
		&components($changed,$parmlog{$id}{'uname'},$parmlog{$id}{'udom'},undef,undef,$typeflag);
	    if ($env{'form.displayfilter'} eq 'currentfolder') {
		if ($folder) {
		    if ($middle!~/^\Q$folder\E/) { next; }
		}
	    }
	    if ($typeflag) {
		$istype{$parmname}=$value; 
		if (!$env{'form.includetypes'}) { next; } 
	    }
	    $count++;
	    if ($makenewrow) {
		$output .= $row_start;
	    } else {
		$makenewrow=1;
	    }
	    $output .='<td>'.$realm.'</td><td>'.$section.'</td><td>'.
		      &standard_parameter_names($parmname).'</td><td>'.
		      ($part?&mt('Part: [_1]',$part):&mt('All Parts')).'</td><td>';
	    my $stillactive=0;
	    if ($parmlog{$id}{'delflag'}) {
		$output .= &mt('Deleted');
	    } else {
		if ($typeflag) {
		    $output .= &mt('Type: [_1]',&standard_parameter_names($value));
		} else {
		    my ($level,@all)=&parmval_by_symb($what,$middle,&Apache::lonnet::metadata($middle,$what),
						      $uname,$udom,$issection,$issection,$courseopt);
		    if (&isdateparm($istype{$parmname})) {
			$output .= &Apache::lonlocal::locallocaltime($value);
		    } else {
			$output .= $value;
		    }
		    if ($value ne $all[$level]) {
			$output .= '<br /><span class="LC_warning">'.&mt('Not active anymore').'</span>';
		    } else {
			$stillactive=1;
		    }
		}
	    }
	    $output .= '</td><td>';
	    if ($stillactive) {
		my $title=&mt('Changed [_1]',&standard_parameter_names($parmname));
                my $description=&mt('Changed [_1] for [_2] to [_3]',&standard_parameter_names($parmname),$realmdescription,
				    (&isdateparm($istype{$parmname})?&Apache::lonlocal::locallocaltime($value):$value));
		if (($uname) && ($udom)) {
		    $output .= 
			&Apache::loncommon::messagewrapper('Notify User',
							   $uname,$udom,$title,
							   $description);
		} else {
		    $output .= 
			&Apache::lonrss::course_blog_link($id,$title,
							  $description);
		}
	    }
	    $output .= '</td>'.&Apache::loncommon::end_data_table_row();
	}
        if ($env{'form.displayfilter'} eq 'containing') {
	    my $wholeentry=$about_me_link.':'.
		$parmlog{$id}{'exe_uname'}.':'.$parmlog{$id}{'exe_udom'}.':'.
		$output;
	    if ($wholeentry!~/\Q$env{'form.containingphrase'}\E/i) { next; }        
	}
        if ($count) {
	    $r->print($row_start.'<td rowspan="'.$count.'">'.$time.'</td>
                       <td rowspan="'.$count.'">'.$about_me_link.
		  '<br /><tt>'.$parmlog{$id}{'exe_uname'}.
			          ':'.$parmlog{$id}{'exe_udom'}.'</tt>'.
		  $send_msg_link.'</td>'.$output);
	    $shown++;
	}
	if (!($env{'form.show'} eq &mt('all') 
	      || $shown<=$env{'form.show'})) { last; }
    }
    $r->print(&Apache::loncommon::end_data_table());
    $r->print(&Apache::loncommon::end_page());
}

sub check_for_course_info {
    my $navmap = Apache::lonnavmaps::navmap->new();
    return 1 if ($navmap);
    return 0;
}

##################################################
##################################################

=pod

=item * handler

Main handler.  Calls &assessparms and &crsenv subroutines.

=cut
##################################################
##################################################
#    use Data::Dumper;


sub handler {
    my $r=shift;

    if ($r->header_only) {
	&Apache::loncommon::content_type($r,'text/html');
	$r->send_http_header;
	return OK;
    }
    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},
					    ['action','state',
                                             'pres_marker',
                                             'pres_value',
                                             'pres_type',
                                             'udom','uname','symb','serial']);


    &Apache::lonhtmlcommon::clear_breadcrumbs();
    &Apache::lonhtmlcommon::add_breadcrumb({href=>"/adm/parmset",
					    text=>"Parameter Manager",
					    faq=>10,
					    bug=>'Instructor Interface',
                                            help => 'Parameter_Manager'});

# ----------------------------------------------------- Needs to be in a course
    my $parm_permission =
	(&Apache::lonnet::allowed('opa',$env{'request.course.id'}) ||
	 &Apache::lonnet::allowed('opa',$env{'request.course.id'}.'/'.
				  $env{'request.course.sec'}));
    my $exists = &check_for_course_info();

    if ($env{'request.course.id'} &&  $parm_permission && $exists) {

        # Start Page
        &Apache::loncommon::content_type($r,'text/html');
        $r->send_http_header;


        # id numbers can change on re-ordering of folders

        &resetsymbcache();

        #
        # Main switch on form.action and form.state, as appropriate
        #
        # Check first if coming from someone else headed directly for
        #  the table mode
        if ((($env{'form.command'} eq 'set') && ($env{'form.url'})
	     && (!$env{'form.dis'})) || ($env{'form.symb'})) {
            &Apache::lonhtmlcommon::add_breadcrumb({help=>'Problem_Parameters',
						    text=>"Problem Parameters"});
	    &assessparms($r);

        } elsif (! exists($env{'form.action'})) {
            $r->print(&header());
            $r->print(&Apache::lonhtmlcommon::breadcrumbs('Parameter Manager'));
            &print_main_menu($r,$parm_permission);
        } elsif ($env{'form.action'} eq 'crsenv' && $parm_permission) {
            &Apache::lonhtmlcommon::add_breadcrumb({href=>'/adm/parmset?action=crsenv',
						    text=>"Course Environment"});
            &crsenv($r); 
        } elsif ($env{'form.action'} eq 'setoverview' && $parm_permission) {
            &Apache::lonhtmlcommon::add_breadcrumb({href=>'/adm/parmset?action=setoverview',
						    text=>"Overview Mode"});
	    &overview($r);
	} elsif ($env{'form.action'} eq 'addmetadata' && $parm_permission) {
            &Apache::lonhtmlcommon::add_breadcrumb({href=>'/adm/parmset?action=addmetadata',
						    text=>"Add Metadata Field"});
	    &addmetafield($r);
	} elsif ($env{'form.action'} eq 'ordermetadata' && $parm_permission) {
            &Apache::lonhtmlcommon::add_breadcrumb({href=>'/adm/parmset?action=addmetadata',
						    text=>"Add Metadata Field"});
	    &order_meta_fields($r);
        } elsif ($env{'form.action'} eq 'setrestrictmeta' && $parm_permission) {
            &Apache::lonhtmlcommon::add_breadcrumb({href=>'/adm/parmset?action=setrestrictmeta',
						    text=>"Restrict Metadata"});
	    &setrestrictmeta($r);
        } elsif ($env{'form.action'} eq 'newoverview' && $parm_permission) {
            &Apache::lonhtmlcommon::add_breadcrumb({href=>'/adm/parmset?action=setoverview',
						    text=>"Overview Mode"});
	    &newoverview($r);
        }  elsif ($env{'form.action'} eq 'setdefaults' && $parm_permission) {
            &Apache::lonhtmlcommon::add_breadcrumb({href=>'/adm/parmset?action=setdefaults',
						    text=>"Set Defaults"});
	    &defaultsetter($r);
	} elsif ($env{'form.action'} eq 'settable' && $parm_permission) {
            &Apache::lonhtmlcommon::add_breadcrumb({href=>'/adm/parmset?action=settable',
						    text=>"Table Mode",
						    help => 'Course_Setting_Parameters'});
	    &assessparms($r);
        } elsif ($env{'form.action'} eq 'parameterchangelog' && $parm_permission) {
            &Apache::lonhtmlcommon::add_breadcrumb({href=>'/adm/parmset?action=settable',
						    text=>"Parameter Change Log"});
	    &parm_change_log($r);
        } elsif ($env{'form.action'} eq 'cleanparameters' && $parm_permission) {
            &Apache::lonhtmlcommon::add_breadcrumb({href=>'/adm/parmset?action=cleanparameters',
						    text=>"Clean Parameters"});
	    &clean_parameters($r);
	}       
    } else {
# ----------------------------- Not in a course, or not allowed to modify parms
	if ($exists) {
	    $env{'user.error.msg'}=
		"/adm/parmset:opa:0:0:Cannot modify assessment parameters";
	} else {
	    $env{'user.error.msg'}=
		"/adm/parmset::0:1:Course environment gone, reinitialize the course";
	}
	return HTTP_NOT_ACCEPTABLE;
    }
    return OK;
}

1;
__END__

=pod

=back

=cut



