# The LearningOnline Network with CAPA
# Search Catalog
#
# $Id: lonsearchcat.pm,v 1.278 2006/12/06 22:22:37 albertel Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
###############################################################################
###############################################################################

=pod 

=head1 NAME

lonsearchcat - LONCAPA Search Interface

=head1 SYNOPSIS

Search interface to LON-CAPAs digital library

=head1 DESCRIPTION

This module enables searching for a distributed browseable catalog.

This is part of the LearningOnline Network with CAPA project
described at http://www.lon-capa.org.

lonsearchcat presents the user with an interface to search the LON-CAPA
digital library.  lonsearchcat also initiates the execution of a search
by sending the search parameters to LON-CAPA servers.  The progress of 
search (on a server basis) is displayed to the user in a separate window.

=head1 Internals

=over 4

=cut

###############################################################################
###############################################################################

package Apache::lonsearchcat;

use strict;
use Apache::Constants qw(:common :http);
use Apache::lonnet;
use Apache::File();
use CGI qw(:standard);
use Text::Query;
use GDBM_File;
use Apache::loncommon();
use Apache::lonmysql();
use Apache::lonmeta;
use Apache::lonhtmlcommon;
use Apache::lonlocal;
use LONCAPA::lonmetadata();
use HTML::Entities();
use Parse::RecDescent;
use Apache::lonnavmaps;
use Apache::lonindexer();
use LONCAPA;

######################################################################
######################################################################
##
## Global variables
##
######################################################################
######################################################################
my %groupsearch_db;  # Database hash used to save values for the 
                     # groupsearch RAT interface.
my %persistent_db;   # gdbm hash which holds data which is supposed to
                     # persist across calls to lonsearchcat.pm

# The different view modes and associated functions

my %Views = ("detailed" => \&detailed_citation_view,
             "detailedpreview" => \&detailed_citation_preview,
	     "summary"  => \&summary_view,
             "summarypreview" => \&summary_preview,
	     "fielded"  => \&fielded_format_view,
	     "xml"      => \&xml_sgml_view,
	     "compact"  => \&compact_view);

######################################################################
######################################################################
sub handler {
    my $r = shift;
#    &set_defaults();
    #
    # set form defaults
    #
    my $hidden_fields;# Hold all the hidden fields used to keep track
                      # of the search system state
    my $importbutton; # button to take the selected results and go to group 
                      # sorting
    my $diropendb;    # The full path to the (temporary) search database file.
                      # This is set and used in &handler() and is also used in 
                      # &output_results().

    my $loaderror=&Apache::lonnet::overloaderror($r);
    if ($loaderror) { return $loaderror; }
    #
    my $closebutton;  # button that closes the search window 
                      # This button is different for the RAT compared to
                      # normal invocation.
    #
    &Apache::loncommon::content_type($r,'text/html');
    $r->send_http_header;
    return OK if $r->header_only;
    ##
    ## Prevent caching of the search interface window.  Hopefully this means
    ## we will get the launch=1 passed in a little more.
    &Apache::loncommon::no_cache($r);
    ## 
    ## Pick up form fields passed in the links.
    ##
    &Apache::loncommon::get_unprocessed_cgi($ENV{'QUERY_STRING'},
             ['catalogmode','launch','acts','mode','form','element','pause',
              'phase','persistent_db_id','table','start','show',
              'cleargroupsort','titleelement','area']);
    ##
    ## The following is a trick - we wait a few seconds if asked to so
    ##     the daemon running the search can get ahead of the daemon
    ##     printing the results.  We only need (theoretically) to do
    ##     this once, so the pause indicator is deleted
    ##
    if (exists($env{'form.pause'})) {
        sleep(1);
        delete($env{'form.pause'});
    }
    ##
    ## Initialize global variables
    ##
    my $domain  = $r->dir_config('lonDefDomain');
    $diropendb= "/home/httpd/perl/tmp/".
        "$env{'user.domain'}_$env{'user.name'}_sel_res.db";
    #
    # set the name of the persistent database
    #          $env{'form.persistent_db_id'} can only have digits in it.
    if (! exists($env{'form.persistent_db_id'}) ||
        ($env{'form.persistent_db_id'} =~ /\D/) ||
        ($env{'form.launch'} eq '1')) {
        $env{'form.persistent_db_id'} = time;
    }

    my $persistent_db_file = "/home/httpd/perl/tmp/".
        &escape($domain).
            '_'.&escape($env{'user.name'}).
                '_'.$env{'form.persistent_db_id'}.'_persistent_search.db';
    ##
    &Apache::lonhtmlcommon::clear_breadcrumbs();

    my @allowed_searches = ('portfolio');
    if (&Apache::lonnet::allowed('bre',$env{'request.role.domain'})) {
        push(@allowed_searches,'res');
    } 
    if (exists($env{'request.course.id'}) && $env{'request.course.id'} ne '') {
        push(@allowed_searches,'course');
    }
    my $crumb_text = 'Portfolio Search';
    if (@allowed_searches == 3) {
        $crumb_text = 'Course, Portfolio and Catalog Search';
    } elsif (@allowed_searches ==2) {
        if (grep(/^res$/,@allowed_searches)) {
            $crumb_text = 'Portfolio and Catalog Search'; 
        } elsif (grep(/^course$/,@allowed_searches)) {
            $crumb_text = 'Portfolio and Course Search';
        }
    }
    &Apache::lonhtmlcommon::add_breadcrumb
       ({href=>'/adm/searchcat?'.
               'catalogmode='.$env{'form.catalogmode'}.
               '&launch='.$env{'form.launch'}.
               '&mode='.$env{'form.mode'},
              text=>"$crumb_text",
              target=>'_top',
              bug=>'Searching',});
    #
    if ($env{'form.phase'} !~ m/(basic|adv|course)_search/) {
        if (! &get_persistent_form_data($persistent_db_file)) {
            if ($env{'form.phase'} =~ /(run_search|results)/) {
                &Apache::lonnet::logthis('lonsearchcat:'.
                                         'Unable to recover data from '.
                                         $persistent_db_file);
		my $msg =
		    'We were unable to retrieve data describing your search. '.
		    'This is a serious error and has been logged. '.
		    'Please alert your LON-CAPA administrator.';
		&Apache::loncommon::simple_error_page($r,'Search Error',
						      $msg);
		return OK;
            }
        }
    } else {
        &clean_up_environment();
    }
    ##
    ## Clear out old values from groupsearch database
    ##
    untie %groupsearch_db if (tied(%groupsearch_db));
    if (($env{'form.cleargroupsort'} eq '1') || 
        (($env{'form.launch'} eq '1') && 
         ($env{'form.catalogmode'} eq 'import'))) {
	if (tie(%groupsearch_db,'GDBM_File',$diropendb,&GDBM_WRCREAT(),0640)) {
	    &start_fresh_session();
	    untie %groupsearch_db;
            delete($env{'form.cleargroupsort'});
	} else {
            # This is a stupid error to give to the user.  
            # It really tells them nothing.
	    my $msg = 'Unable to tie hash to db file.';
	    &Apache::loncommon::simple_error_page($r,'Search Error',
						  $msg);
	    return OK;
	}
    }
    ##
    ## Configure hidden fields
    ##
    $hidden_fields = '<input type="hidden" name="persistent_db_id" value="'.
        $env{'form.persistent_db_id'}.'" />'."\n";
    if (exists($env{'form.catalogmode'})) {
        $hidden_fields .= &hidden_field('catalogmode');
    }
    if (exists($env{'form.form'})) {
        $hidden_fields .= &hidden_field('form');
    }
    if (exists($env{'form.element'})) {
        $hidden_fields .= &hidden_field('element');
    }
    if (exists($env{'form.titleelement'})) {
        $hidden_fields .= &hidden_field('titleelement');
    }
    if (exists($env{'form.mode'})) {
        $hidden_fields .= &hidden_field('mode');
    }
    if (exists($env{'form.area'})) {
        $hidden_fields .= &hidden_field('area');
    }
    ##
    ## Configure dynamic components of interface
    ##
    if ($env{'form.catalogmode'} eq 'interactive') {
        $closebutton="<input type='button' name='close' value='CLOSE' ";
        if ($env{'form.phase'} =~ /(results|run_search)/) {
	    $closebutton .="onClick='parent.close()'";
        } else {
            $closebutton .="onClick='self.close()'";
        }
        $closebutton .=">\n";
    } elsif ($env{'form.catalogmode'} eq 'import') {
        $closebutton="<input type='button' name='close' value='CLOSE' ";
        if ($env{'form.phase'} =~ /(results|run_search)/) {
	    $closebutton .="onClick='parent.close()'";
        } else {
            $closebutton .="onClick='self.close()'";
        }
        $closebutton .= ">";
        $importbutton=<<END;
<input type='button' name='import' value='IMPORT'
onClick='javascript:select_group()'>
END
    } else {
        $closebutton = '';
        $importbutton = '';
    }
    ##
    ## Sanity checks on form elements
    ##
    if (!defined($env{'form.viewselect'})) {
	$env{'form.viewselect'} ="summary";
    }
    $env{'form.phase'} = 'disp_basic' if (! exists($env{'form.phase'}));
    $env{'form.show'} = 20 if (! exists($env{'form.show'}));
    #
    $env{'form.searchmode'} = 'basic' if (! exists($env{'form.searchmode'}));
    if ($env{'form.phase'} eq 'adv_search' ||
        $env{'form.phase'} eq 'disp_adv') {
        $env{'form.searchmode'} = 'advanced';
    } elsif ($env{'form.phase'} eq 'course_search') {
        $env{'form.searchmode'} = 'course_search';
    }
    #
    if ($env{'form.searchmode'} eq 'advanced') {
        my $srchtype = 'Catalog';
        if ($env{'form.area'} eq 'portfolio') {
            $srchtype = 'Portfolio';
        }
        &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>'/adm/searchcat?phase=disp_adv&'.
                  'catalogmode='.$env{'form.catalogmode'}.
                  '&launch='.$env{'form.launch'}.
                  '&mode='.$env{'form.mode'},
                  text=>"Advanced $srchtype Search",
                  bug=>'Searching',});
    } elsif ($env{'form.searchmode'} eq 'course search') {
        &Apache::lonhtmlcommon::add_breadcrumb
            ({href=>'/adm/searchcat?phase=disp_adv&'.
                  'catalogmode='.$env{'form.catalogmode'}.
                  '&launch='.$env{'form.launch'}.
                  '&mode='.$env{'form.mode'},
                  text=>"Course Search",
                  bug=>'Searching',});
    }
    ##
    ## Switch on the phase
    ##
    if ($env{'form.phase'} eq 'disp_basic') {
        &print_basic_search_form($r,$closebutton,$hidden_fields);
    } elsif ($env{'form.phase'} eq 'disp_adv') {
        &print_advanced_search_form($r,$closebutton,$hidden_fields);
    } elsif ($env{'form.phase'} eq 'results') {
        &display_results($r,$importbutton,$closebutton,$diropendb,
                         $env{'form.area'});
    } elsif ($env{'form.phase'} =~ /^(sort|run_search)$/) {
        my ($query,$customquery,$customshow,$libraries,$pretty_string) =
            &get_persistent_data($persistent_db_file,
                 ['query','customquery','customshow',
                  'libraries','pretty_string']);
        if ($env{'form.phase'} eq 'sort') {
            &print_sort_form($r,$pretty_string);
        } elsif ($env{'form.phase'} eq 'run_search') {
            &run_search($r,$query,$customquery,$customshow,
                        $libraries,$pretty_string,$env{'form.area'});
        }
    } elsif ($env{'form.phase'} eq 'course_search') {
        &course_search($r);
    } elsif(($env{'form.phase'} eq 'basic_search') ||
            ($env{'form.phase'} eq 'adv_search')) {
        #
        # We are running a search, try to parse it
        my ($query,$customquery,$customshow,$libraries) = 
            (undef,undef,undef,undef);
        my $pretty_string;
        if ($env{'form.phase'} eq 'basic_search') {
            ($query,$pretty_string,$libraries) = 
                &parse_basic_search($r,$closebutton,$hidden_fields);
            return OK if (! defined($query));
            &make_persistent({ basicexp => $env{'form.basicexp'}},
                             $persistent_db_file);
        } else {                      # Advanced search
            ($query,$customquery,$customshow,$libraries,$pretty_string) 
                = &parse_advanced_search($r,$closebutton,$hidden_fields);
            return OK if (! defined($query));
        }
        &make_persistent({ query => $query,
                           customquery => $customquery,
                           customshow => $customshow,
                           libraries => $libraries,
                           pretty_string => $pretty_string },
                         $persistent_db_file);
        #
        # Set up table
        if (! defined(&create_results_table($env{'form.area'}))) {
	    my $errorstring=&Apache::lonmysql::get_error();
            &Apache::lonnet::logthis('lonsearchcat.pm: Unable to create '.
                                     'needed table.  lonmysql error:'.
                                     $errorstring);

	    my $msg = 
		'Unable to create table in which to store search results. '.
		'The search has been aborted.';
	    &Apache::loncommon::simple_error_page($r,'Search Error',
						  $msg);
	    return OK;
        }
        delete($env{'form.launch'});
        if (! &make_form_data_persistent($r,$persistent_db_file)) {
	    my $msg=
		'Unable to properly store search information. '.
		'The search has been aborted.';
	    &Apache::loncommon::simple_error_page($r,'Search Error',
						  $msg);
	    return OK;
	}
        ##
        ## Print out the frames interface
        ##
        if (defined($query)) {
            &print_frames_interface($r);
        }
    }
    return OK;
} 

#
# The mechanism used to store values away and retrieve them does not
# handle the case of missing environment variables being significant.
#
# This routine sets non existant checkbox form elements to ''.
#
sub clean_up_environment {
    if ($env{'form.phase'} eq 'basic_search') {
        if (! exists($env{'form.related'})) {
            $env{'form.related'} = '';
        }
        if (! exists($env{'form.domains'})) {
            $env{'form.domains'} = '';
        }
    } elsif ($env{'form.phase'} eq 'adv_search') {
        foreach my $field ('title','keywords','notes',
                           'abstract','standards','mime') {
            if (! exists($env{'form.'.$field.'_related'})) {
                $env{'form.'.$field.'_related'} = '';
            }
        }
    } elsif ($env{'form.phase'} eq 'course_search') {
        if (! exists($env{'form.crsrelated'})) {
            $env{'form.crsrelated'} = '';
        }
    }
}

sub hidden_field {
    my ($name,$value) = @_;
    if (! defined($value)) {
        $value = $env{'form.'.$name};
    }
    return '<input type="hidden" name="'.$name.'" value="'.$value.'" />'.$/;
}

######################################################################
######################################################################
##
##   Course Search
##
######################################################################
######################################################################
{   # Scope the course search to avoid global variables
#
# Variables For course search
my %alreadyseen;
my %hash;
my $totalfound;

sub make_symb {
    my ($id)=@_;
    my ($mapid,$resid)=split(/\./,$id);
    my $map=$hash{'map_id_'.$mapid};
    my $res=$hash{'src_'.$id};
    my $symb=&Apache::lonnet::encode_symb($map,$resid,$res);
    return $symb;
}

sub course_search {
    my $r=shift;
    my $pretty_search_string = '<b>'.$env{'form.courseexp'}.'</b>';
    my $search_string = $env{'form.courseexp'};
    my @New_Words;
    undef(%alreadyseen);
    if ($env{'form.crsrelated'}) {
        ($search_string,@New_Words) = &related_version($env{'form.courseexp'});
        if (@New_Words) {
            $pretty_search_string .= ' '.&mt("with related words").": <b>@New_Words</b>.";
        } else {
            $pretty_search_string .= ' '.&mt('with no related words').".";
        }
    }
    my $fulltext=$env{'form.crsfulltext'};
    my $discuss=$env{'form.crsdiscuss'};
    my @allwords=($search_string,@New_Words);
    $totalfound=0;
    $r->print(&Apache::loncommon::start_page('Course Search').
	      '<hr /><center><font size="+2" face="arial">'.
	      $pretty_search_string.'</font></center>'.
	      '<hr /><b>'.&mt('Course content').':</b><br />');
    $r->rflush();
# ======================================================= Go through the course
    my $c=$r->connection;
    if (tie(%hash,'GDBM_File',$env{'request.course.fn'}.".db",
            &GDBM_READER(),0640)) {
        foreach (sort(keys(%hash))) {
            if ($c->aborted()) { last; }
            if (($_=~/^src\_(.+)$/)) {
		if ($hash{'randomout_'.$1} & !$env{'request.role.adv'}) {
		    next; 
		}
		my $symb=&make_symb($1);
                &checkonthis($r,$1,$hash{$_},0,&Apache::lonnet::gettitle($symb),
			     $fulltext,$symb,@allwords);
            }
        }
        untie(%hash);
    }
    unless ($totalfound) {
	$r->print('<p>'.&mt('No matches found in resources').'.</p>');
    }

# Check discussions if requested
    if ($discuss) {
        my $totaldiscussions = 0;
        $r->print('<br /><br /><b>'.&mt('Discussion postings').':</b><br />'); 
        my $navmap = Apache::lonnavmaps::navmap->new();
        my @allres=$navmap->retrieveResources();
        my %discussiontime = &Apache::lonnet::dump('discussiontimes',
                               $env{'course.'.$env{'request.course.id'}.'.domain'},
                               $env{'course.'.$env{'request.course.id'}.'.num'});
        foreach my $resource (@allres) {
            my $result = '';
            my $applies = 0;
            my $symb = $resource->symb();
            my $ressymb = $symb;
            if ($symb =~ m#(___adm/$LONCAPA::domain_re/$LONCAPA::username_re)/(\d+)/bulletinboard$#) {
                $ressymb = 'bulletin___'.$2.$1.'/'.$2.'/bulletinboard';
                unless ($ressymb =~ m#bulletin___\d+___adm/wrapper#) {
                    $ressymb=~s#(bulletin___\d+___)#$1adm/wrapper/#;
                }
            }
            if (defined($discussiontime{$ressymb})) { 
                my %contrib = &Apache::lonnet::restore($ressymb,$env{'request.course.id'},
                     $env{'course.'.$env{'request.course.id'}.'.domain'},
                     $env{'course.'.$env{'request.course.id'}.'.num'});
                if ($contrib{'version'}) {
                    for (my $id=1;$id<=$contrib{'version'};$id++) {
                        unless (($contrib{'hidden'}=~/\.$id\./) || ($contrib{'deleted'}=~/\.$id\./)) { 
                            if ($contrib{$id.':subject'}) {
                                $result .= $contrib{$id.':subject'};
                            }
                            if ($contrib{$id.':message'}) {
                                $result .= $contrib{$id.':message'};
                            }
                            if ($contrib{$id,':attachmenturl'}) {
                                if ($contrib{$id,':attachmenturl'} =~ m-/([^/]+)$-) {
                                    $result .= $1;
                                }
                            }
                            $applies = &checkwords($result,$applies,@allwords);
                        }
                    }
                }
            }
# Does this discussion apply?
            if ($applies) {
                my ($map,$ind,$url)=&Apache::lonnet::decode_symb($ressymb);
                my $disctype = &mt('resource');
                if ($url =~ m#/bulletinboard$#) {
                    if ($url =~m#^adm/wrapper/adm/.*/bulletinboard$#) {
                        $url =~s#^adm/wrapper##;
                    }
                    $disctype = &mt('bulletin board');
                } else {
                    $url = '/res/'.$url;
                }
                if ($url =~ /\?/) {
                    $url .= '&symb=';
                } else {
                    $url .= '?symb=';
                }
                $url .= &escape($resource->symb());
                my $title = $resource->compTitle();
                $r->print('<br /><a href="'.$url.'" target="cat">'.
                     ($title?$title:$url).'</a>&nbsp;&nbsp;-&nbsp;'.$disctype.'<br />');
                $totaldiscussions++;
            } else {
                $r->print(' .');
            }
        }
        unless ($totaldiscussions) {
            $r->print('<p>'.&mt('No matches found in postings').'.</p>');
        }
    }
 
# =================================================== Done going through course
    $r->print(&Apache::loncommon::end_page());
}

# =============================== This pulls up a resource and its dependencies

sub checkonthis {
    my ($r,$id,$url,$level,$title,$fulltext,$symb,@allwords)=@_;
    $alreadyseen{$id}=1;
    if (&Apache::loncommon::connection_aborted($r)) { return; }
    $r->rflush();
    
    my $result=$title.' ';
    if ($env{'request.role.adv'} || !$hash{'encrypted_'.$id}) {
	$result.=&Apache::lonnet::metadata($url,'title').' '.
	    &Apache::lonnet::metadata($url,'subject').' '.
	    &Apache::lonnet::metadata($url,'abstract').' '.
	    &Apache::lonnet::metadata($url,'keywords');
    }
    my ($extension)=($url=~/\.(\w+)$/);
    if (&Apache::loncommon::fileembstyle($extension) eq 'ssi' &&
	($url) && ($fulltext)) {
	$result.=&Apache::lonnet::ssi_body($url.'?symb='.&escape($symb));
    }
    $result=~s/\s+/ /gs;
    my $applies = 0;
    $applies = &checkwords($result,$applies,@allwords);
# Does this resource apply?
    if ($applies) {
       $r->print('<br />');
       for (my $i=0;$i<=$level*5;$i++) {
           $r->print('&nbsp;');
       }
       my $href=$url;
       if ($hash{'encrypted_'.$id} && !$env{'request.role.adv'}) {
	   $href=&Apache::lonenc::encrypted($href)
	       .'?symb='.&Apache::lonenc::encrypted($symb);
       } else {
	   $href.='?symb='.&escape($symb);
       }
       $r->print('<a href="'.$href.'" target="cat">'.($title?$title:$url).
		 '</a><br />');
       $totalfound++;
    } elsif ($fulltext) {
       $r->print(' .');
    }
    $r->rflush();
# Check also the dependencies of this one
    my $dependencies=
                &Apache::lonnet::metadata($url,'dependencies');
    foreach (split(/\,/,$dependencies)) {
       if (($_=~/^\/res\//) && (!$alreadyseen{$id})) { 
          &checkonthis($r,$id,$_,$level+1,'',$fulltext,undef,@allwords);
       }
    }
}

sub checkwords {
    my ($result,$applies,@allwords) = @_;
    foreach (@allwords) {
        if ($_=~/\w/) {
            if ($result=~/$_/si) {
                $applies++;
            }
        }
    }
    return $applies;
}

sub untiehash {
    if (tied(%hash)) {
        untie(%hash);
    }
}

} # End of course search scoping


######################################################################
######################################################################

=pod 

=item &print_basic_search_form() 

Prints the form for the basic search.  Sorry the name is so cryptic.

=cut

######################################################################
######################################################################
sub print_basic_search_form {
    my ($r,$closebutton,$hidden_fields) = @_;
    my $result = ($env{'form.catalogmode'} ne 'import');
    my $bread_crumb =
        &Apache::lonhtmlcommon::breadcrumbs('Searching','Search_Basic',
					    $env{'form.catalogmode'} ne 'import');
    my $scrout = &Apache::loncommon::start_page('Search').$bread_crumb;
# Search form for resource space 
    if (&Apache::lonnet::allowed('bre',$env{'request.role.domain'})) {
        $scrout .= &setup_basic_search($r,'res',$hidden_fields,$closebutton);
        $scrout .= '<hr /><br />';
    }
# Search form for accessible portfolio files
    $scrout.= &setup_basic_search($r,'portfolio',$hidden_fields,$closebutton);
    if ($env{'request.course.id'}) {
	my %lt=&Apache::lonlocal::texthash('srch' => 'Search',
                                           'header' => 'Course Search',
	 'note' => 'Enter terms or phrases, then press "Search" below',
	 'use' => 'use related words',
	 'full' =>'fulltext search (time consuming)',
         'disc' => 'search discussion postings (resources and bulletin boards)',
					   );
        $scrout.=(<<ENDCOURSESEARCH);
<form name="loncapa_search" method="post" action="/adm/searchcat">
<center>
<hr />
<h1>$lt{'header'}</h1>    
<input type="hidden" name="phase" value="course_search" />
$hidden_fields
<p>
$lt{'note'}.
</p>
<p>
<table>
<tr><td>
ENDCOURSESEARCH
        $scrout.='&nbsp;'.
            &Apache::lonhtmlcommon::textbox('courseexp',
                                  $env{'form.courseexp'},40);
        my $crscheckbox = 
            &Apache::lonhtmlcommon::checkbox('crsfulltext',
                                   $env{'form.crsfulltext'});
        my $relcheckbox = 
            &Apache::lonhtmlcommon::checkbox('crsrelated',
				   $env{'form.crsrelated'});
        my $discheckbox = 
            &Apache::lonhtmlcommon::checkbox('crsdiscuss',
                                   $env{'form.crsrelated'});
        $scrout.=(<<ENDENDCOURSE);
</td></tr>
<tr><td><label>$relcheckbox $lt{'use'}</label></td><td></td></tr>
<tr><td><label>$crscheckbox $lt{'full'}</label></td><td></td></tr>
<tr><td><label>$discheckbox $lt{'disc'}</label></td><td></td></tr>
</table><p>
&nbsp;<input type="submit" name="coursesubmit" value='$lt{'srch'}' />
</p>
</center>
</form>
ENDENDCOURSE
    }
    $scrout .= &Apache::loncommon::end_page();
    $r->print($scrout);
    return;
}

sub setup_basic_search {
    my ($r,$area,$hidden_fields,$closebutton) = @_;
    # Define interface components
    my %lt = &Apache::lonlocal::texthash (
                              res => 'LON-CAPA Catalog Search',
                              portfolio => 'Portfolio Search',
    );
    my ($userelatedwords,$onlysearchdomain,$inclext,$adv_search_link,$scrout);
    $userelatedwords = '<label>'.&mt('[_1] use related words',
      &Apache::lonhtmlcommon::checkbox('related',$env{'form.related'},'related')).
                       '</label>';
    $onlysearchdomain = '<label>'.&mt('[_1] only search domain [_2]',
      &Apache::lonhtmlcommon::checkbox('domains',$env{'form.domains'},
                                       $r->dir_config('lonDefDomain'))).
                        '</label>';
    if ($area eq 'res') {
        $inclext= '<label>'.&mt('[_1] include external resources',
             &Apache::lonhtmlcommon::checkbox('inclext',$env{'form.inclext'})).
                  '</label>';
    }
    $adv_search_link = '<a href="/adm/searchcat?'.
                       'phase=disp_adv&'.
                       'catalogmode='.$env{'form.catalogmode'}.
                       '&launch='.$env{'form.launch'}.
                       '&mode='.$env{'form.mode'}.
                       '&area='.$area.
                       '">'.&mt('Advanced Search').'</a>';
    #
    $scrout.='<form name="loncapa_search" method="post" '.
             'action="/adm/searchcat">'.
             '<input type="hidden" name="phase" value="basic_search" />'.
             $hidden_fields;
             if (!exists($env{'form.area'})) {
                 $scrout .= '<input type="hidden" name="area" value="'.$area.'" />';
             }
    #
    $scrout .= '<center>'.$/;
#    if ($env{'request.course.id'}) {
        $scrout .= '<h1>'.$lt{$area}.'</h1>';
#    } else {
        # No need to tell them they are searching
#        $scrout.= ('<br />'x2);
#    }
    $scrout.='<table>'.
             '<tr><td align="center" valign="top">'.
             &Apache::lonhtmlcommon::textbox('basicexp',
                                             $env{'form.basicexp'},50).
             '<br />'.
            '<font size="-1">'.&searchhelp().'</font>'.'</td>'.
            '<td><font size="-1">'.
            '<nobr>'.('&nbsp;'x3).$adv_search_link.'</nobr>'.'<br />'.
            '<nobr>'.('&nbsp;'x1).$userelatedwords.'</nobr>'.'<br />'.
            '<nobr>'.('&nbsp;'x1).$onlysearchdomain.'</nobr>'.'<br />'.
            '<nobr>'.('&nbsp;'x1).$inclext.'</nobr>'.'<br />'.
             '</font></td>'.
            '</tr>'.$/;
    #
    $scrout .= '<tr><td align="center" colspan="2">'.
               '<font size="-1">'.
               '<input type="submit" name="basicsubmit" '.
               'value="'.&mt('Search').'" />'.
               ('&nbsp;'x2).$closebutton.('&nbsp;'x2). &viewoptions().
               '</font>'.
               '</td></tr>'.$/;
    $scrout .= '</table>'.$/.'</center>'.'</form>';
    return $scrout;
} 

######################################################################
######################################################################

=pod 

=item &advanced_search_form() 

Prints the advanced search form.

=cut

######################################################################
######################################################################
sub print_advanced_search_form{
    my ($r,$closebutton,$hidden_fields) = @_;
    my $bread_crumb = 
        &Apache::lonhtmlcommon::breadcrumbs('Searching','Search_Advanced',
					    $env{'form.catalogmode'} ne 'import');
    my %lt=&Apache::lonlocal::texthash('srch' => 'Search',
				       'reset' => 'Reset',
				       'help' => 'Help');
    my $advanced_buttons=<<"END";
<input type="submit" name="advancedsubmit" value='$lt{"srch"}' />
<input type="reset" name="reset" value='$lt{"reset"}' />
$closebutton
END
    my $srchtype = 'Catalog';
    if ($env{'form.area'} eq 'portfolio') {
        $srchtype = 'Portfolio';
    }
    my $scrout= &Apache::loncommon::start_page("Advanced $srchtype Search");
    $scrout .= <<"ENDHEADER";
$bread_crumb
<form method="post" action="/adm/searchcat" name="advsearch">
<p>
$advanced_buttons
ENDHEADER
    $scrout.=('&nbsp;'x2).&viewoptions().'</p>'.$hidden_fields. 
        '<input type="hidden" name="phase" value="adv_search" />';
    my %fields=&Apache::lonmeta::fieldnames();
    #
    $scrout .= '<h3>'.&mt('Standard [_1] Metadata',$srchtype).'</h3>';
    $scrout .= "<table>\n";
    $scrout .= '<tr><td>&nbsp;</td><td colspan="2"><font size="-1">'.
        ('&nbsp;'x2).&searchhelp()."</font></td></tr>\n";
    my %related_word_search = 
        ('title'    => 1,
         'author'   => 0,
         'owner'    => 0,
         'authorspace'  => 0,
         'modifyinguser'=> 0,
         'keywords' => 1,
         'notes'    => 1,
         'abstract' => 1,
         'standards'=> 1,
         'mime'     => 1,
	 'subject'  => 1,
         );
    #
    foreach my $field ('title','author','subject','owner','authorspace',
		       'modifyinguser','keywords','notes','abstract',
		       'standards','mime') {
	$scrout.='<tr><td align="right">'.&titlefield($fields{$field}).'</td><td>'.
	    &Apache::lonmeta::prettyinput($field,
                                          $env{'form.'.$field},
                                          $field,
                                          'advsearch',
					  $related_word_search{$field},
                                          '</td><td align="left">',
                                          $env{'form.'.$field.'_related'},
                                          50);
        if ($related_word_search{$field}) {
            $scrout .= 'related words';
        } else {
            $scrout .= '</td><td>&nbsp;';
        }
        $scrout .= '</td></tr>'.$/;
    }
    foreach my $field ('lowestgradelevel','highestgradelevel') {
	$scrout.='<tr>'.
            '<td align="right">'.&titlefield($fields{$field}).'</td>'.
            '<td colspan="2">'.
	    &Apache::lonmeta::prettyinput($field,
                                          $env{'form.'.$field},
                                          $field,
                                          'advsearch',
					  0).
                                          '</td></tr>'.$/;
    }
    $scrout.='<tr><td align="right">'.
	&titlefield(&mt('MIME Type Category')).'</td><td colspan="2">'. 
	    &Apache::loncommon::filecategoryselect('category',
						   $env{'form.category'}).
	    '</td></tr>'.$/;
    $scrout.='<tr><td align="right" valign="top">'.
	&titlefield(&mt('Domains')).'</td><td colspan="2">'. 
	    &Apache::loncommon::domain_select('domains',
						   $env{'form.domains'},1).
						   '<br /><label>';
            if ($env{'form.area'} ne 'portfolio') {
                $scrout .= &mt('[_1] include external resources',
                           &Apache::lonhtmlcommon::checkbox
                           ('inclext',$env{'form.inclext'})).'</label>'
            }
     $scrout .= '</td></tr>'.$/;
    #
    # Misc metadata
    if ($env{'form.area'} ne 'portfolio') {
        $scrout.='<tr><td align="right" valign="top">'.
	         &titlefield(&mt('Copyright/Distribution')).
                 '</td><td colspan="2">'.
                 &Apache::lonmeta::selectbox('copyright',
                                             $env{'form.copyright'},
                                \&Apache::loncommon::copyrightdescription,
                                       ( undef,
                                        &Apache::loncommon::copyrightids)
                                ).'</td></tr>'.$/;
    }
    $scrout.='<tr><td align="right" valign="top">'.
	&titlefield(&mt('Language')).'</td><td colspan="2">'.
        &Apache::lonmeta::selectbox('language',
                                    $env{'form.language'},
                                    \&Apache::loncommon::languagedescription,
                                    ('any',&Apache::loncommon::languageids)
                                    ).'</td></tr>';
    $scrout .= "</table>\n";

    
    if ($env{'form.area'} eq 'portfolio') {
        # Added fields
        $scrout .= '<h3>'.&mt('Custom Metadata fields').'</h3>';
        $scrout .= "<table>\n";
        $scrout .= '<tr><td>&nbsp;</td><td align="center">'.
                   &mt('Field Name').'</td>'.'<td align="center">'.
                   &mt('Field Value(s)').'</td></tr>'.
                   '<tr><td>'.&mt('1: ').
                   '</td><td align="center">'.
                   '<input type="text" name="addedfield_0" size="10" /></td>'.
                   '<td align="center"><input type="text" '.
                   'name="addedvalues_0" size="15" /></td></tr>';
        for (my $j=1; $j<=$env{'form.numaddedfields'}; $j++) {
            my $num = $j+1;
            $scrout .= '<tr><td>'.&mt('Custom metadata [_1]: ',$num).
                       '</td><td align="center">'.
                       '<input type="text" name="addedfield_'.$j.
                       '" size="10" /></td>'.
                       '<td align="center"><input type="text" '.
                       'name="addedvalues_'.$j.'" size="15" /></td></tr>';
        }
        my $numadded = 1 + $env{'form.numaddedfields'};
        $scrout .= '<tr><td colspan="3">&nbsp;</td></tr>'.
                   '<tr><td align="left" colspan="3">'.
                   '<input type="button" name="newfield" '.
                   'value="Additional custom field/value" '.
                   'onclick="javascript:additional_metadata()" />'.
                   '<input type="hidden" name="numaddedfelds" value="'.
                   $numadded.'" /></td></tr></table>';
    } else {
        #
        # Dynamic metadata
        $scrout .= '<h3>'.&mt('Problem Statistics').'</h3>';
        $scrout .= "<table>\n";
        $scrout .= '<tr><td>&nbsp;</td><td align="center">'.
                   &mt('Minimum').'</td>'.'<td align="center">'.
                   &mt('Maximum').'</td></tr>'."\n";
        foreach my $statistic 
            ({ name=>'count',
               description=>'Network-wide number of accesses (hits)',},
             { name=>'stdno',
               description=>
               'Total number of students who have worked on this problem',},
             { name => 'avetries',
               description=>'Average number of tries till solved',},
             { name => 'difficulty',
               description=>'Degree of difficulty',},
             { name => 'disc',
               description=>'Degree of discrimination'}) {
              $scrout .= '<tr><td align="right">'.
                         &titlefield(&mt($statistic->{'description'})).
                         '</td><td align="center">'.
                         '<input type="text" name="'.$statistic->{'name'}.
                         '_min" value="" size="6" /></td><td align="center">'.
                         '<input type="text" name="'.$statistic->{'name'}.
                         '_max" value="" size="6" /></td></tr>'.$/;
        }
        $scrout .= "</table>\n";
        $scrout .= '<h3>'.&mt('Evaluation Data').'</h3>';
        $scrout .= "<table>\n";
        $scrout .= '<tr><td>&nbsp;</td><td align="center">'.
                   &mt('Minimum').'</td>'.'<td align="center">'.
                   &mt('Maximum').'</td></tr>'."\n";
        foreach my $evaluation
            ( { name => 'clear',
                description => 'Material presented in clear way'},
              { name =>'depth',
                description => 'Material covered with sufficient depth'},
              { name => 'helpful',
                description => 'Material is helpful'},
              { name => 'correct',
                description => 'Material appears to be correct'},
              { name => 'technical',
                description => 'Resource is technically correct'}){
            $scrout .= '<tr><td align="right">'.
                       &titlefield(&mt($evaluation->{'description'})).
                       '</td><td align="center">'.
                       '<input type="text" name="'.
                       $evaluation->{'name'}.'_min" value="" size="6" />'.
                       '</td><td align="center"><input type="text" name="'.
                       $evaluation->{'name'}.'_max" value="" size="6" />'.
                       '</td></tr>'.$/;
        }
        $scrout .= "</table>\n";
    }
    #
    # Creation/Modification date limits
    $scrout .= '<h3>'.&mt('Creation and Modification dates').'</h3>';
    $scrout .= "\n<table>\n";
    my $cafter = 
        &Apache::lonhtmlcommon::date_setter('advsearch',         # formname
                                            'creationdate1', # fieldname
                                            0,           # current value
                                            '',          # special 
                                            1,           # includeempty
                                            '',          # state
                                            1,           # no_hh_mm_ss
                                            );
    my $cbefore = 
        &Apache::lonhtmlcommon::date_setter('advsearch',         # formname
                                            'creationdate2', # fieldname
                                            0,           # current value
                                            '',          # special 
                                            1,           # includeempty
                                            '',          # state
                                            1,           # no_hh_mm_ss
                                            );
    $scrout .= &mt('<tr><td align="right">Created between</td>'.
                   '<td>[_1]</td></tr>'.
                   '<tr><td align="right">and </td>'.
                   '<td>[_2]</td></tr>',$cafter,$cbefore);
    my $lafter = 
        &Apache::lonhtmlcommon::date_setter('advsearch',
                                            'revisiondate1', 
                                            0,           # current value
                                            '',          # special 
                                            1,           # includeempty
                                            '',          # state
                                            1,           # no_hh_mm_ss
                                            );
    my $lbefore = 
        &Apache::lonhtmlcommon::date_setter('advsearch',
                                            'revisiondate2',
                                            0,           # current value
                                            '',          # special 
                                            1,           # includeempty
                                            '',          # state
                                            1,           # no_hh_mm_ss
                                            );
    $scrout .= &mt('<tr><td align="right">Last modified between </td>'.
                   '<td>[_1]</td></tr>'.
                   '<tr><td align="right">and</td>'.
                   '<td>[_2]</td></tr>',$lafter,$lbefore);
    $scrout.="</table>\n";
    $scrout.=<<ENDDOCUMENT;
$advanced_buttons
</form>
ENDDOCUMENT
    $scrout .= &Apache::loncommon::end_page();
    $r->print($scrout);
    return;
}

######################################################################
######################################################################

=pod 

=item &titlefield()

Inputs: title text

Outputs: titletext with font wrapper

=cut

######################################################################
######################################################################
sub titlefield {
    my $title=shift;
    return $title;
}

######################################################################
######################################################################

=pod 

=item viewoptiontext()

Inputs: codename for view option

Outputs: displayed text

=cut

######################################################################
######################################################################
sub viewoptiontext {
    my $code=shift;
    my %desc=&Apache::lonlocal::texthash
        ('detailed' => "Detailed Citation View",
         'xml' => 'XML/SGML',
         'compact' => 'Compact View',
         'fielded' => 'Fielded Format',
         'summary' => 'Summary View',
         'summarypreview' => 'Summary Preview',
         'detailedpreview' => 'Detailed Citation Preview');
    return $desc{$code};
}

######################################################################
######################################################################

=pod 

=item viewoptions()

Inputs: none

Outputs: text for box with view options

=cut

######################################################################
######################################################################
sub viewoptions {
    my $scrout;
    if (! defined($env{'form.viewselect'})) { 
        $env{'form.viewselect'}='detailed'; 
    }
    $scrout.=&Apache::lonmeta::selectbox('viewselect',
			$env{'form.viewselect'},
			\&viewoptiontext,
			sort(keys(%Views)));
    $scrout.= '&nbsp;&nbsp;';
    my $countselect = &Apache::lonmeta::selectbox('show',
                                                  $env{'form.show'},
                                                  undef,
                                                  (10,20,50,100,1000,10000));
    $scrout .= ('&nbsp;'x2).&mt('[_1] Records per Page',$countselect).
        '</nobr>'.$/;
    return $scrout;
}

######################################################################
######################################################################

=pod 

=item searchhelp()

Inputs: none

Outputs: return little blurb on how to enter searches

=cut

######################################################################
######################################################################
sub searchhelp {
    return &mt('Enter words and quoted phrases');
}

######################################################################
######################################################################

=pod 

=item &get_persistent_form_data()

Inputs: filename of database

Outputs: returns undef on database errors.

This function is the reverse of &make_persistent() for form data.
Retrieve persistent data from %persistent_db.  Retrieved items will have their
values unescaped.  If a form value already exists in $env, it will not be
overwritten.  Form values that are array references may have values appended
to them.

=cut

######################################################################
######################################################################
sub get_persistent_form_data {
    my $filename = shift;
    return 0 if (! -e $filename);
    return undef if (! tie(%persistent_db,'GDBM_File',$filename,
                           &GDBM_READER(),0640));
    #
    # These make sure we do not get array references printed out as 'values'.
    my %arrays_allowed = ('form.domains'=>1);
    #
    # Loop through the keys, looking for 'form.'
    foreach my $name (keys(%persistent_db)) {
        next if ($name !~ /^form./);
        # Kludgification begins!
        if ($name eq 'form.domains' && 
            $env{'form.searchmode'} eq 'basic' &&
            $env{'form.phase'} ne 'disp_basic') {
            next;
        }
        # End kludge (hopefully)
        next if (exists($env{$name}));
        my @values = map { 
            &unescape($_);
        } split(',',$persistent_db{$name});
        next if (@values <1);
        if ($arrays_allowed{$name}) {
            $env{$name} = [@values];
        } else {
            $env{$name} = $values[0] if ($values[0]);
        }
    }
    untie (%persistent_db);
    return 1;
}

######################################################################
######################################################################

=pod 

=item &get_persistent_data()

Inputs: filename of database, ref to array of values to recover.

Outputs: array of values.  Returns undef on error.

This function is the reverse of &make_persistent();
Retrieve persistent data from %persistent_db.  Retrieved items will have their
values unescaped.  If the item contains commas (before unescaping), the
returned value will be an array pointer. 

=cut

######################################################################
######################################################################
sub get_persistent_data {
    my $filename = shift;
    my @Vars = @{shift()};
    my @Values;   # Return array
    return undef if (! -e $filename);
    return undef if (! tie(%persistent_db,'GDBM_File',$filename,
                           &GDBM_READER(),0640));
    foreach my $name (@Vars) {
        if (! exists($persistent_db{$name})) {
            push @Values, undef;
            next;
        }
        my @values = map { 
            &unescape($_);
        } split(',',$persistent_db{$name});
        if (@values <= 1) {
            push @Values,$values[0];
        } else {
            push @Values,\@values;
        }
    }
    untie (%persistent_db);
    return @Values;
}

######################################################################
######################################################################

=pod 

=item &make_persistent() 

Inputs: Hash of values to save, filename of persistent database.

Store variables away to the %persistent_db.
Values will be escaped.  Values that are array pointers will have their
elements escaped and concatenated in a comma separated string.  

=cut

######################################################################
######################################################################
sub make_persistent {
    my %save = %{shift()};
    my $filename = shift;
    return undef if (! tie(%persistent_db,'GDBM_File',
                           $filename,&GDBM_WRCREAT(),0640));
    foreach my $name (keys(%save)) {
        my @values = (ref($save{$name}) ? @{$save{$name}} : ($save{$name}));
        # We handle array references, but not recursively.
        my $store = join(',', map { &escape($_); } @values );
        $persistent_db{$name} = $store;
    }
    untie(%persistent_db);
    return 1;
}

######################################################################
######################################################################

=pod 

=item &make_form_data_persistent() 

Inputs: filename of persistent database.

Store most form variables away to the %persistent_db.
Values will be escaped.  Values that are array pointers will have their
elements escaped and concatenated in a comma separated string.  

=cut

######################################################################
######################################################################
sub make_form_data_persistent {
    my $r = shift;
    my $filename = shift;
    my %save;
    foreach (keys(%env)) {
        next if (!/^form/ || /submit/);
        $save{$_} = $env{$_};
    }
    return &make_persistent(\%save,$filename);
}

######################################################################
######################################################################

=pod 

=item &parse_advanced_search()

Parse advanced search form and return the following:

=over 4

=item $query Scalar containing an SQL query.

=item $customquery Scalar containing a custom query.

=item $customshow Scalar containing commands to show custom metadata.

=item $libraries_to_query Reference to array of domains to search.

=back

=cut

######################################################################
######################################################################
sub parse_advanced_search {
    my ($r,$closebutton,$hidden_fields)=@_;
    my @BasicFields = ('title','author','subject','keywords','url','version',
                       'notes','abstract','extension','owner','authorspace',
#                       'custommetadata','customshow',
                       'modifyinguser','standards','mime');
    my @StatsFields = &statfields();
    my @EvalFields = &evalfields();
    my $fillflag=0;
    my $pretty_search_string = "";
    # Clean up fields for safety
    for my $field (@BasicFields,
                   'creationdatestart_month','creationdatestart_day',
		   'creationdatestart_year','creationdateend_month',
		   'creationdateend_day','creationdateend_year',
		   'lastrevisiondatestart_month','lastrevisiondatestart_day',
		   'lastrevisiondatestart_year','lastrevisiondateend_month',
		   'lastrevisiondateend_day','lastrevisiondateend_year') {
	$env{'form.'.$field}=~s/[^\w\/\s\(\)\=\-\"\']//g;
    }
    foreach ('mode','form','element') {
	# is this required?  Hmmm.
	next if (! exists($env{'form.'.$_}));
	$env{'form.'.$_}=&unescape($env{'form.'.$_});
	$env{'form.'.$_}=~s/[^\w\/\s\(\)\=\-\"\']//g;
    }
    # Preprocess the category form element.
    $env{'form.category'} = 'any' if (! defined($env{'form.category'}) ||
                                      ref($env{'form.category'}));
    #
    # Check to see if enough information was filled in
    foreach my $field (@BasicFields) {
	if (&filled($env{'form.'.$field})) {
	    $fillflag++;
	}
    }
    foreach my $field (@StatsFields,@EvalFields) {
        if (&filled($env{'form.'.$field.'_max'})) {
            $fillflag++;
        }
        if (&filled($env{'form.'.$field.'_min'})) {
            $fillflag++;
        }
    }

    for my $field ('lowestgradelevel','highestgradelevel') {
        if ( $env{'form.'.$field} =~ /^\d+$/ &&
             $env{'form.'.$field} > 0) {
            $fillflag++;
        }
    }
    if (! $fillflag) {
	&output_blank_field_error($r,$closebutton,
                                  'phase=disp_adv',$hidden_fields);
	return ;
    }
    # Turn the form input into a SQL-based query
    my $query='';
    my @queries;
    my $font = '<font color="#800000" face="helvetica">';
    # Evaluate logical expression AND/OR/NOT phrase fields.
    foreach my $field (@BasicFields) {
	next if (!defined($env{'form.'.$field}) || $env{'form.'.$field} eq '');
        my ($error,$SQLQuery) = 
            &process_phrase_input($env{'form.'.$field},
                                  $env{'form.'.$field.'_related'},$field);
        if (defined($error)) {
            &output_unparsed_phrase_error($r,$closebutton,'phase=disp_adv',
                                         $hidden_fields,$field);
            return;
        } else {
            $pretty_search_string .= 
                $font.$field.'</font>: '.$env{'form.'.$field};
            if ($env{'form.'.$field.'_related'}) {
                my @Words = 
                    &Apache::loncommon::get_related_words
                    ($env{'form.'.$field});
                if (@Words) {
                    $pretty_search_string.= ' with related words: '.
                        join(', ',@Words[0..4]);
                } else {
                    $pretty_search_string.= ' with related words.';
                }
            }
            $pretty_search_string .= '<br />';
            push (@queries,$SQLQuery);
        }
    }
    #
    # Make the 'mime' from 'form.category' and 'form.extension'
    #
    my $searchphrase;
    if (exists($env{'form.category'})    && 
        $env{'form.category'} !~ /^\s*$/ &&
        $env{'form.category'} ne 'any')     {
        my @extensions = &Apache::loncommon::filecategorytypes
                                                   ($env{'form.category'});
        if (scalar(@extensions) > 0) {
            $searchphrase = join(' OR ',@extensions);
        }
    }
    if (defined($searchphrase)) {
        my ($error,$SQLsearch) = &process_phrase_input($searchphrase,0,'mime');
        push @queries,$SQLsearch;
        $pretty_search_string .=$font.'mime</font> contains <b>'.
            $searchphrase.'</b><br />';
    }
    #
    # Evaluate option lists
    if ($env{'form.lowestgradelevel'}        &&
        $env{'form.lowestgradelevel'} ne '0' &&
        $env{'form.lowestgradelevel'} =~ /^\d+$/) {
	push(@queries,
             '(lowestgradelevel>='.$env{'form.lowestgradelevel'}.')');
        $pretty_search_string.="lowestgradelevel>=".
            $env{'form.lowestgradelevel'}."<br />\n";
    }
    if ($env{'form.highestgradelevel'}        &&
        $env{'form.highestgradelevel'} ne '0' &&
        $env{'form.highestgradelevel'} =~ /^\d+$/) {
	push(@queries,
             '(highestgradelevel<='.$env{'form.highestgradelevel'}.')');
        $pretty_search_string.="highestgradelevel<=".
            $env{'form.highestgradelevel'}."<br />\n";
    }
    if ($env{'form.language'} and $env{'form.language'} ne 'any') {
	push @queries,"(language like \"$env{'form.language'}\")";
        $pretty_search_string.=$font."language</font>= ".
            &Apache::loncommon::languagedescription($env{'form.language'}).
                "<br />\n";
    }
    if ($env{'form.copyright'} and $env{'form.copyright'} ne 'any') {
	push @queries,"(copyright like \"$env{'form.copyright'}\")";
        $pretty_search_string.=$font."copyright</font> = ".
            &Apache::loncommon::copyrightdescription($env{'form.copyright'}).
                "<br />\n";
    }
    if ($env{'form.area'} eq 'portfolio') {
        #
        # Added metadata fields
        for (my $i=0; $i<$env{'form.numaddedfields'} ; $i++) {
            if (($env{'form.addedfield_'.$i} ne '') && 
                ($env{'form.addedvalue_'.$i} ne '')) {
                my $stuff = 1; #FIXME 
            }
        }
    } else {
        #
        # Statistics
        foreach my $field (@StatsFields,@EvalFields) {
            my ($min,$max);
            if (exists($env{'form.'.$field.'_min'}) && 
                $env{'form.'.$field.'_min'} ne '') {
                $min = $env{'form.'.$field.'_min'};
            }
            if (exists($env{'form.'.$field.'_max'}) &&
                $env{'form.'.$field.'_max'} ne '') {
                $max = $env{'form.'.$field.'_max'};
            }
            next if (! defined($max) && ! defined($min));
            if (defined($min) && defined($max)) {
                ($min,$max) = sort {$a <=>$b} ($min,$max);
            }
            if (defined($min) && $min =~ /^(\d+\.\d+|\d+|\.\d+)$/) {
                push(@queries,'('.$field.'>'.$min.')');
                $pretty_search_string.=$font.$field.'</font>&gt;'.$min.'<br />';
            }
            if (defined($max) && $max =~ /^(\d+\.\d+|\d+|\.\d+)$/) {
                push(@queries,'('.$field.'<'.$max.')');
                $pretty_search_string.=$font.$field.'</font>&lt;'.$max.'<br />';
            }
        }
    }
    #
    # Evaluate date windows
    my $cafter =
        &Apache::lonhtmlcommon::get_date_from_form('creationdate1');
    my $cbefore = 
        &Apache::lonhtmlcommon::get_date_from_form('creationdate2');
    if ($cafter > $cbefore) {
        my $tmp = $cafter;
        $cafter = $cbefore;
        $cbefore = $tmp;
    }
    my $mafter = 
        &Apache::lonhtmlcommon::get_date_from_form('revisiondate1');
    my $mbefore =
        &Apache::lonhtmlcommon::get_date_from_form('revisiondate2');
    if ($mafter > $mbefore) {
        my $tmp = $mafter;
        $mafter = $mbefore;
        $mbefore = $tmp;
    }
    my ($datequery,$error,$prettydate)=&build_date_queries($cafter,$cbefore,
                                                           $mafter,$mbefore);
    if (defined($error)) {
        &output_date_error($r,$error,$closebutton,$hidden_fields);
    } elsif (defined($datequery)) {
        # Here is where you would set up pretty_search_string to output
        # date query information.
        $pretty_search_string .= '<br />'.$prettydate.'<br />';
	push @queries,$datequery;
    }
    #
    # Process form information for custom metadata querying
    my $customquery=undef;
    ##
    ## The custom metadata search was removed q long time ago mostly 
    ## because I was unable to figureout exactly how it worked and could
    ## not imagine people actually using it.  MH
    ##
    # if ($env{'form.custommetadata'}) {
    #    $pretty_search_string .=$font."Custom Metadata Search</font>: <b>".
    #    $env{'form.custommetadata'}."</b><br />\n";
    #    $customquery=&build_custommetadata_query('custommetadata',
    #                                             $env{'form.custommetadata'});
    # }
    my $customshow=undef;
    # if ($env{'form.customshow'}) {
    # $pretty_search_string .=$font."Custom Metadata Display</font>: <b>".
    #                         $env{'form.customshow'}."</b><br />\n";
    #    $customshow=$env{'form.customshow'};
    #    $customshow=~s/[^\w\s]//g;
    #    my @fields=split(/\s+/,$customshow);
    #    $customshow=join(" ",@fields);
    # }
    ##
    ## Deal with restrictions to given domains
    ## 
    my ($libraries_to_query,$pretty_domains_string) = &parse_domain_restrictions();
    if ($pretty_domains_string) {
       $pretty_search_string .= $pretty_domains_string."<br />\n";
    }
    #
    if (@queries) {
        if ($env{'form.area'} eq 'portfolio') {
            $query ="SELECT pm.*,pa.keynum,pa.scope FROM portfolio_metadata pm, portfolio_access pa  WHERE (pm.url = pa.url AND (pa.start < NOW() AND (pa.end IS NULL OR pa.end > NOW())) AND (".join(') AND (',@queries).'))';
        } else {
	    $query="SELECT * FROM metadata WHERE (".join(") AND (",@queries).')';
        }
    } elsif ($customquery) {
        $query = '';
    }
    #&Apache::lonnet::logthis('advanced query = '.$/.$query);
    return ($query,$customquery,$customshow,$libraries_to_query,
            $pretty_search_string);
}

sub parse_domain_restrictions {
    my $libraries_to_query = undef;
    # $env{'form.domains'} can be either a scalar or an array reference.
    # We need an array.
    if (! exists($env{'form.domains'}) || $env{'form.domains'} eq '') {
        return (undef,'',undef);
    }
    my @allowed_domains = &Apache::loncommon::get_env_multiple('form.domains');
    #
    my %domain_hash = ();
    my $pretty_domains_string;
    foreach (@allowed_domains) {
        $domain_hash{$_}++;
    }
    if ($domain_hash{'any'}) {
        $pretty_domains_string = &mt("in all LON-CAPA domains.");
    } else {
        if (@allowed_domains > 1) {
            $pretty_domains_string = &mt("in LON-CAPA domains:");
        } else {
            $pretty_domains_string = &mt("in LON-CAPA domain ");
        }
        foreach (sort @allowed_domains) {
            $pretty_domains_string .= "<b>".$_."</b> ";
        }
        foreach (keys(%Apache::lonnet::libserv)) {
            if (exists($domain_hash{$Apache::lonnet::hostdom{$_}})) {
                push @$libraries_to_query,$_;
            }
        }
    }
    return ($libraries_to_query,
            $pretty_domains_string);
}

######################################################################
######################################################################

=pod 

=item &parse_basic_search() 

Parse the basic search form and return a scalar containing an sql query.

=cut

######################################################################
######################################################################
sub parse_basic_search {
    my ($r,$closebutton)=@_;
    #
    # Clean up fields for safety
    for my $field ('basicexp') {
	$env{"form.$field"}=~s/[^\w\s\'\"\!\(\)\-]//g;
    }
    foreach ('mode','form','element') {
	# is this required?  Hmmm.
	next unless (exists($env{"form.$_"}));
	$env{"form.$_"}=&unescape($env{"form.$_"});
	$env{"form.$_"}=~s/[^\w\/\s\(\)\=\-\"\']//g;
    }
    my ($libraries_to_query,$pretty_domains_string) = &parse_domain_restrictions();
    #
    # Check to see if enough of a query is filled in
    my $search_string = $env{'form.basicexp'};
    if (! &filled($search_string)) {
	&output_blank_field_error($r,$closebutton,'phase=disp_basic');
	return OK;
    }
    my $pretty_search_string=$search_string;
    my @Queries;
    my @fields = ('title','author','subject','notes','abstract','keywords');
    my $searchfield;
    if ($env{'form.area'} eq 'portfolio') {
        $searchfield = 'concat_ws(" ",pm.'.join(',pm.',@fields).')';
    } else {
        $searchfield = 'concat_ws(" ",'.join(',',@fields).')';
    }
    my ($error,$SQLQuery) = &process_phrase_input($search_string,
                                                    $env{'form.related'},
                                                    $searchfield);
    if ($error) {
        &output_unparsed_phrase_error($r,$closebutton,'phase=disp_basic',
                                      '','basicexp');
        return;
    }
    push(@Queries,$SQLQuery);
    #foreach my $q (@Queries) {
    #    &Apache::lonnet::logthis('    '.$q);
    #}
    my $final_query;
    if ($env{'form.area'} eq 'portfolio') {
        $final_query = 'SELECT pm.*,pa.keynum,pa.scope FROM portfolio_metadata pm, portfolio_access pa  WHERE (pm.url = pa.url AND (pa.start < NOW() AND (pa.end IS NULL OR pa.end > NOW())) AND '.join(" AND ",@Queries).')';
    } else {
        $final_query = 'SELECT * FROM metadata WHERE '.join(" AND ",@Queries);
    }
    #
    if ($env{'form.related'}) {
	$pretty_search_string.=' '.&mt('(including related words)');
    }
    if (defined($pretty_domains_string) && $pretty_domains_string ne '') {
        $pretty_search_string .= ' '.$pretty_domains_string;
    }
    $pretty_search_string .= "<br />\n";
    $pretty_search_string =~ s:^<br /> and ::;
    &Apache::lonnet::logthis('simple search final query = '.$/.$final_query);
    return ($final_query,$pretty_search_string,
            $libraries_to_query);
}


###############################################################
###############################################################

my @Phrases;

sub concat {
    my ($item) = @_;
    my $results = '';
    foreach (@$item) {
        if (ref($_) eq 'ARRAY') {
            $results .= join(' ',@$_);
        }
    }
    return $results;
}

sub process_phrase_input {
    my ($phrase,$related,$field)=@_;
    #&Apache::lonnet::logthis('phrase = :'.$phrase.':');
    my $grammar = <<'ENDGRAMMAR';
    searchphrase:
        expression /^\Z/ {
            # &Apache::lonsearchcat::print_item(\@item,0);
            [@item];
        }
    expression:
        phrase(s)   {
            [@item];
        }
    phrase:
        orword {
            [@item];
        }
      | andword {
            [@item];
        }
      | minusword {
            unshift(@::Phrases,$item[1]->[0]);
            unshift(@::Phrases,$item[1]->[1]);
            [@item];
        }
      | word {
            unshift(@::Phrases,$item[1]);
            [@item];
        } 
    #
    orword:
        word 'OR' phrase {
            unshift(@::Phrases,'OR');
            unshift(@::Phrases,$item[1]);
            [@item];
        }
        | word 'or' phrase {
            unshift(@::Phrases,'OR');
            unshift(@::Phrases,$item[1]);
            [@item];
        }    
        | minusword 'OR' phrase {
            unshift(@::Phrases,'OR');
            unshift(@::Phrases,$item[1]->[0]);
            unshift(@::Phrases,$item[1]->[1]);
            [@item];
        }
        | minusword 'or' phrase {
            unshift(@::Phrases,'OR');
            unshift(@::Phrases,$item[1]->[0]);
            unshift(@::Phrases,$item[1]->[1]);
            [@item];
        }    
    andword:
        word phrase {
            unshift(@::Phrases,'AND');
            unshift(@::Phrases,$item[1]);
            [@item];
        }
        | minusword phrase {
            unshift(@::Phrases,'AND');
            unshift(@::Phrases,$item[1]->[0]);
            unshift(@::Phrases,$item[1]->[1]);
            [@item];
        }
    #
    minusword:
        '-' word {
            [$item[2],'NOT'];
        }
    word:
        "'" term(s) "'" {
          &Apache::lonsearchcat::concat(\@item);
        }
      | '"' term(s) '"' {
          &Apache::lonsearchcat::concat(\@item);
        }
      | term {
            $item[1];
        }
    term:
        /[\w\Q:!@#$%^&*()+_=|{}<>,.;\\\/?\E]+/ {
            $item[1];
        }
ENDGRAMMAR
    #
    # The end result of parsing the phrase with the grammar is an array
    # @::Phrases.
    # $phrase = "gene splicing" or cat -> "gene splicing","OR","cat"
    # $phrase = "genetic engineering" -dna ->
    #                      "genetic engineering","AND","NOT","dna"
    # $phrase = cat or dog -poodle -> "cat","OR","dog","AND","NOT","poodle"
    undef(@::Phrases);
    my $p = new Parse::RecDescent($grammar);
    if (! defined($p->searchphrase($phrase))) {
        &Apache::lonnet::logthis('lonsearchcat:unable to process:'.$phrase);
        return 'Unable to process phrase '.$phrase;
    }
    #
    # Go through the phrases and make sense of them.  
    # Apply modifiers NOT OR and AND to the phrases.
    my @NewPhrases;
    while(@::Phrases) {
        my $phrase = shift(@::Phrases);
        # &Apache::lonnet::logthis('phrase = '.$phrase);
        my $phrasedata;
        if ($phrase =~ /^(NOT|OR|AND)$/) {
            if ($phrase eq 'OR') {
                $phrasedata->{'or'}++;
                if (! @::Phrases) { $phrasedata = undef; last; }
                $phrase = shift(@::Phrases);
            } elsif ($phrase eq 'AND') {
                $phrasedata->{'and'}++;
                if (! @::Phrases) { $phrasedata = undef; last; }
                $phrase = shift(@::Phrases);
            }
            if ($phrase eq 'NOT') {
                $phrasedata->{'negate'}++;
                if (! @::Phrases) { $phrasedata = undef; last; }
                $phrase = shift(@::Phrases);
            }
        }
        $phrasedata->{'phrase'} = $phrase;
        if ($related) {
            my @NewWords;
            (undef,@NewWords) = &related_version($phrasedata->{'phrase'});
            $phrasedata->{'related_words'} = \@NewWords;
        }
        push(@NewPhrases,$phrasedata);
    }
    #
    # Actually build the sql query from the phrases
    my $SQLQuery;
    foreach my $phrase (@NewPhrases) {
        my $query;
        if ($phrase->{'negate'}) {
            $query .= $field.' NOT LIKE "%'.$phrase->{'phrase'}.'%"';
        } else {
            $query .= $field.' LIKE "%'.$phrase->{'phrase'}.'%"';
        }
        foreach my $related (@{$phrase->{'related_words'}}) {
            if ($phrase->{'negate'}) {
                $query .= ' AND '.$field.' NOT LIKE "%'.$related.'%"';
            } else {
                $query .= ' OR '.$field.' LIKE "%'.$related.'%"';
            }
        }
        if ($SQLQuery) {
            if ($phrase->{'or'}) {
                $SQLQuery .= ' OR ('.$query.')';
            } else {
                $SQLQuery .= ' AND ('.$query.')';
            }
        } else {
            $SQLQuery = '('.$query.')';
        }
    }
    #
    # &Apache::lonnet::logthis("SQLQuery = $SQLQuery");
    #
    return undef,$SQLQuery;
}

######################################################################
######################################################################

=pod 

=item &related_version()

Modifies an input string to include related words.  Words in the string
are replaced with parenthesized lists of 'OR'd words.  For example
"torque" is replaced with "(torque OR word1 OR word2 OR ...)".  

Note: Using this twice on a string is probably silly.

=cut

######################################################################
######################################################################
sub related_version {
    my ($word) = @_;
    return (undef) if (lc($word) =~ /\b(or|and|not)\b/);
    my @Words = &Apache::loncommon::get_related_words($word);
    # Only use 4 related words
    @Words = ($#Words>4? @Words[0..4] : @Words);
    my $result = join " OR ", ($word,@Words);
    return $result,sort(@Words);
}


######################################################################
######################################################################

=pod 

=item &build_custommetadata_query() 

Constructs a custom metadata query using a rather heinous regular
expression.

=cut

######################################################################
######################################################################
sub build_custommetadata_query {
    my ($field_name,$logic_statement)=@_;
    my $q=new Text::Query('abc',
			  -parse => 'Text::Query::ParseAdvanced',
			  -build => 'Text::Query::BuildAdvancedString');
    $q->prepare($logic_statement);
    my $matchexp=${$q}{'-parse'}{'-build'}{'matchstring'};
    # quick fix to change literal into xml tag-matching
    # will eventually have to write a separate builder module
    # wordone=wordtwo becomes\<wordone\>[^\<] *wordtwo[^\<]*\<\/wordone\>
    $matchexp =~ s/(\w+)\\=([\w\\\+]+)?# wordone=wordtwo is changed to 
                 /\\<$1\\>?#           \<wordone\>
                   \[\^\\<\]?#        [^\<]         
                   \*$2\[\^\\<\]?#           *wordtwo[^\<]
                   \*\\<\\\/$1\\>?#                        *\<\/wordone\>
                   /g;
    return $matchexp;
}


######################################################################
######################################################################

=pod 

=item &build_date_queries() 

Builds a SQL logic query to check time/date entries.
Also reports errors (check for /^Incorrect/).

=cut

######################################################################
######################################################################
sub build_date_queries {
    my ($cafter,$cbefore,$mafter,$mbefore) = @_;
    my ($result,$error,$pretty_string);
    #
    # Verify the input
    if (! defined($cafter) && ! defined($cbefore) &&
        ! defined($mafter) && ! defined($mbefore)) {
        # This is an okay situation, so return undef for the error
        return (undef,undef,undef);
    }
    if ((defined($cafter)  && ! defined($cbefore)) ||
        (defined($cbefore) && ! defined($cafter))) {
        # This is bad, so let them know
        $error = &mt('Incorrect entry for the creation date.  '.
                    'You must specify both the beginning and ending dates.');
    }
    if (! defined($error) && 
        ((defined($mafter)  && ! defined($mbefore)) ||
        (defined($mbefore) && ! defined($mafter)))) {
        # This is also bad, so let them know
        $error = &mt('Incorrect entry for the last revision date.  '.
                     'You must specify both the beginning and ending dates.');
    }
    if (! defined($error)) {
        #
        # Build the queries
        my @queries;
        if (defined($cbefore) && defined($cafter)) {
            my (undef,undef,undef,$caday,$camon,$cayear) = localtime($cafter);
            my (undef,undef,undef,$cbday,$cbmon,$cbyear) = localtime($cbefore);
            # Correct for year being relative to 1900
            $cayear+=1900; $cbyear+=1900;
            my $cquery=
                '(creationdate BETWEEN '.
                "'".$cayear.'-'.$camon.'-'.$caday."'".
                ' AND '.
                "'".$cbyear.'-'.$cbmon.'-'.$cbday." 23:59:59')";
            $pretty_string .= '<br />' if (defined($pretty_string));
            $pretty_string .= 
                &mt('created between [_1] and [_2]',
                    &Apache::lonlocal::locallocaltime($cafter),
                    &Apache::lonlocal::locallocaltime($cbefore+24*60*60-1));
            push(@queries,$cquery);
            $pretty_string =~ s/ 00:00:00//g;
        }
        if (defined($mbefore) && defined($mafter)) {
            my (undef,undef,undef,$maday,$mamon,$mayear) = localtime($mafter);
            my (undef,undef,undef,$mbday,$mbmon,$mbyear) = localtime($mbefore);
            # Correct for year being relative to 1900
            $mayear+=1900; $mbyear+=1900;
            my $mquery=
                '(lastrevisiondate BETWEEN '.
                "'".$mayear.'-'.$mamon.'-'.$maday."'".
                ' AND '.
                "'".$mbyear.'-'.$mbmon.'-'.$mbday." 23:59:59')";
            push(@queries,$mquery);
            $pretty_string .= '<br />' if (defined($pretty_string));
            $pretty_string .= 
                &mt('last revised between [_1] and [_2]',
                    &Apache::lonlocal::locallocaltime($mafter),
                    &Apache::lonlocal::locallocaltime($mbefore+24*60*60-1));
            $pretty_string =~ s/ 00:00:00//g;
        }
        if (@queries) {
            $result .= join(" AND ",@queries);
        }
    }
    return ($result,$error,$pretty_string);
}

######################################################################
######################################################################

=pod

=item &copyright_check()

Inputs: $Metadata, a hash pointer of metadata for a resource.

Returns: 1 if the resource is available to the user making the query, 
         0 otherwise.

=cut

######################################################################
######################################################################
sub copyright_check {
    my $Metadata = shift;
    # Check copyright tags and skip results the user cannot use
    my (undef,undef,$resdom,$resname) = split('/',
                                              $Metadata->{'url'});
    # Check for priv
    if (($Metadata->{'copyright'} eq 'priv') && 
        (($env{'user.name'} ne $resname) &&
         ($env{'user.domain'} ne $resdom))) {
        return 0;
    }
    # Check for domain
    if (($Metadata->{'copyright'} eq 'domain') &&
        ($env{'user.domain'} ne $resdom)) {
        return 0;
    }
    return 1;
}

######################################################################
######################################################################

=pod

=item &ensure_db_and_table()

Ensure we can get lonmysql to connect to the database and the table we
need exists.

Inputs: $r, table id

Returns: undef on error, 1 if the table exists.

=cut

######################################################################
######################################################################
sub ensure_db_and_table {
    my ($r,$table) = @_;
    ##
    ## Sanity check the table id.
    ##
    if (! defined($table) || $table eq '' || $table =~ /\D/ ) {
        $r->print("Unable to retrieve search results.  ".
                  "Unable to determine the table results were stored in.  ".
		  &Apache::loncommon::end_page());
        return undef;
    }
    ##
    ## Make sure we can connect and the table exists.
    ##
    my $connection_result = &Apache::lonmysql::connect_to_db();
    if (!defined($connection_result)) {
        $r->print("Unable to connect to the MySQL database where your results".
                  " are stored.".
		  &Apache::loncommon::end_page());
        &Apache::lonnet::logthis("lonsearchcat: unable to get lonmysql to".
                                 " connect to database.");
        &Apache::lonnet::logthis(&Apache::lonmysql::get_error());
        return undef;
    }
    my $table_check = &Apache::lonmysql::check_table($table);
    if (! defined($table_check)) {
        $r->print("A MySQL error has occurred.</form>".
		  &Apache::loncommon::end_page());
        &Apache::lonnet::logthis("lonmysql was unable to determine the status".
                                 " of table ".$table);
        return undef;
    } elsif (! $table_check) {
        $r->print("The table of results could not be found.");
        &Apache::lonnet::logthis("The user requested a table, ".$table.
                                 ", that could not be found.");
        return undef;
    }
    return 1;
}

######################################################################
######################################################################

=pod

=item &print_sort_form()

The sort feature is not implemented at this time.  This form just prints 
a link to change the search query.

=cut

######################################################################
######################################################################
sub print_sort_form {
    my ($r,$pretty_query_string) = @_;

    ##
    my %SortableFields=&Apache::lonlocal::texthash( 
         id        => 'Default',
         title     => 'Title',
         author    => 'Author',
         subject   => 'Subject',
         url       => 'URL',
         version   => 'Version Number',
         mime      => 'Mime type',
         lang      => 'Language',
         owner     => 'Owner/Publisher',
         copyright => 'Copyright',
         hostname  => 'Host',
         creationdate     => 'Creation Date',
         lastrevisiondate => 'Revision Date'
     );
    ##
    my $table = $env{'form.table'};
    return if (! &ensure_db_and_table($r,$table));
    ##
    ## Get the number of results 
    ##
    my $total_results = &Apache::lonmysql::number_of_rows($table);
    if (! defined($total_results)) {
        $r->print("A MySQL error has occurred.</form>".
		  &Apache::loncommon::end_page());
        &Apache::lonnet::logthis("lonmysql was unable to determine the number".
                                 " of rows in table ".$table);
        &Apache::lonnet::logthis(&Apache::lonmysql::get_error());
        return;
    }
    my $js =<<END;
<script type="text/javascript">
    function change_sort() {
        var newloc = "/adm/searchcat?phase=results";
        newloc += "&persistent_db_id=$env{'form.persistent_db_id'}";
        newloc += "&sortby=";
        newloc += document.forms.statusform.elements.sortby.value;
        parent.resultsframe.location= newloc;
    }
</script>
END

    my $start_page = &Apache::loncommon::start_page('Results',$js,
						    {'no_title' => 1});
    my $breadcrumbs=
        &Apache::lonhtmlcommon::breadcrumbs('Searching','Searching',
					    $env{'form.catalogmode'} ne 'import');

    my $result = <<END;
$start_page
$breadcrumbs
<form name="statusform" action="" method="post" target="_top">
<input type="hidden" name="catalogmode" value="import" />
<input type="hidden" name="acts" value="" />
END

#<h2>Sort Results</h2>
#Sort by: <select size="1" name="sortby" onchange="javascript:change_sort();">
#    $env{'form.sortby'} = 'id' if (! defined($env{'form.sortby'}));
#    foreach (keys(%SortableFields)) {
#        $result.="<option name=\"$_\"";
#        if ($_ eq $env{'form.sortby'}) {
#            $result.=" selected ";
#        }
#        $result.=" >$SortableFields{$_}</option>\n";
#    }
#    $result.="</select>\n";
    my $revise = &revise_button();
    $result.=<<END;
<p>
There are $total_results matches to your query. $revise
</p><p>
Search: $pretty_query_string
</p>
</form>
END
    $r->print($result.&Apache::loncommon::end_page());
    return;
}

#####################################################################
#####################################################################

=pod

=item MySQL Table Description

MySQL table creation requires a precise description of the data to be
stored.  The use of the correct types to hold data is vital to efficient
storage and quick retrieval of records.  The columns must be described in
the following format:

=cut

#####################################################################
#####################################################################
#
# These should probably be scoped but I don't have time right now...
#
my @Datatypes;
my @Fullindicies;
    
######################################################################
######################################################################

=pod

=item &create_results_table()

Creates the table of search results by calling lonmysql.  Stores the
table id in $env{'form.table'}

Inputs: search area - either res or portfolio 

Returns: the identifier of the table on success, undef on error.

=cut

######################################################################
######################################################################
sub set_up_table_structure {
    my ($tabletype) = @_;
    my ($datatypes,$fullindicies) = 
        &LONCAPA::lonmetadata::describe_metadata_storage($tabletype);
    # Copy the table description before modifying it...
    @Datatypes = @{$datatypes};
    unshift(@Datatypes,{name => 'id',  
        type => 'MEDIUMINT',
        restrictions => 'UNSIGNED NOT NULL',
        primary_key  => 'yes',
        auto_inc     => 'yes' });
    @Fullindicies = @{$fullindicies};
    return;
}

sub create_results_table {
    my ($area) = @_;
    if ($area eq 'portfolio') {
        &set_up_table_structure('portfolio_search');
    } else {
        &set_up_table_structure('metadata');
    }
    my $table = &Apache::lonmysql::create_table
        ( { columns => \@Datatypes,
            FULLTEXT => [{'columns' => \@Fullindicies},],
        } );
    if (defined($table)) {
        $env{'form.table'} = $table;
        return $table;
    } 
    return undef; # Error...
}

######################################################################
######################################################################

=pod

=item Search Status update functions

Each of the following functions changes the values of one of the
input fields used to display the search status to the user.  The names
should be explanatory.

Inputs: Apache request handler ($r), text to display.

Returns: Nothing.

=over 4

=item &update_count_status()

=item &update_status()

=item &update_seconds()

=back

=cut

######################################################################
######################################################################
sub update_count_status {
    my ($r,$text) = @_;
    $text =~ s/\'/\\\'/g;
    $r->print
        ("<script>document.statusform.count.value = ' $text'</script>\n");
    $r->rflush();
}

sub update_status {
    my ($r,$text) = @_;
    $text =~ s/\'/\\\'/g;
    $r->print
        ("<script>document.statusform.status.value = ' $text'</script>\n");
    $r->rflush();
}

{
    my $max_time  = 300;  # seconds for the search to complete
    my $start_time = 0;
    my $last_time = 0;

sub reset_timing {
    $start_time = 0;
    $last_time = 0;
}

sub time_left {
    if ($start_time == 0) {
        $start_time = time;
    }
    my $time_left = $max_time - (time - $start_time);
    $time_left = 0 if ($time_left < 0);
    return $time_left;
}

sub update_seconds {
    my ($r) = @_;
    my $time = &time_left();
    if (($last_time-$time) > 0) {
        $r->print("<script>".
                  "document.statusform.seconds.value = '$time'".
                  "</script>\n");
        $r->rflush();
    }
    $last_time = $time;
}

}

######################################################################
######################################################################

=pod

=item &revise_button()

Inputs: None

Returns: html string for a 'revise search' button.

=cut

######################################################################
######################################################################
sub revise_button {
    my $revise_phase = 'disp_basic';
    $revise_phase = 'disp_adv' if ($env{'form.searchmode'} eq 'advanced');
    my $newloc = '/adm/searchcat'.
        '?persistent_db_id='.$env{'form.persistent_db_id'}.
            '&cleargroupsort=1'.
            '&phase='.$revise_phase;
    my $result = qq{<input type="button" value="Revise search" name="revise"} .
        qq{ onClick="parent.location='$newloc';" /> };
    return $result;
}

######################################################################
######################################################################

=pod

=item &run_search()

Executes a search query by sending it the the other servers and putting the
results into MySQL.

=cut

######################################################################
######################################################################
sub run_search {
    my ($r,$query,$customquery,$customshow,$serverlist,
        $pretty_string,$area) = @_;
    my $tabletype = 'metadata';
    if ($area eq 'portfolio') {
        $tabletype = 'portfolio_search';
    }
    my $connection = $r->connection;
    #
    # Print run_search header
    #
    my $start_page = &Apache::loncommon::start_page('Search Status',undef,
						    {'no_title' => 1});
    my $breadcrumbs =
	&Apache::lonhtmlcommon::breadcrumbs('Searching','Searching',
					    $env{'form.catalogmode'} ne 'import');
    $r->print(<<END);
$start_page
$breadcrumbs
<form name="statusform" action="" method="post">
<input type="hidden" name="acts" value="" />
END
    # Remove leading and trailing <br />
    $pretty_string =~ s:^\s*<br />::i;
    $pretty_string =~ s:(<br />)*\s*$::im;
    my @Lines = split("<br />",$pretty_string);
    # I keep getting blank items at the end of the list, hence the following:
    while ($Lines[-1] =~ /^\s*$/ && @Lines) {
        pop(@Lines);
    }
    if (@Lines > 2) {
        $pretty_string = join '<br />',(@Lines[0..2],'....<br />');
    }
    $r->print(&mt("Search: [_1]",$pretty_string));
    $r->rflush();
    #
    # Determine the servers we need to contact.
    my @Servers_to_contact;
    if (defined($serverlist)) {
        if (ref($serverlist) eq 'ARRAY') {
            @Servers_to_contact = @$serverlist;
        } else {
            @Servers_to_contact = ($serverlist);
        }
    } else {
        @Servers_to_contact = sort(keys(%Apache::lonnet::libserv));
    }
    my %Server_status;
    #
    # Check on the mysql table we will use to store results.
    my $table =$env{'form.table'};
    if (! defined($table) || $table eq '' || $table =~ /\D/ ) {
        $r->print("Unable to determine table id to store search results in.".
                  "The search has been aborted.".
		  &Apache::loncommon::end_page());
        return;
    }
    my $table_status = &Apache::lonmysql::check_table($table);
    if (! defined($table_status)) {
        $r->print("Unable to determine status of table.".
		  &Apache::loncommon::end_page());
        &Apache::lonnet::logthis("Bogus table id of $table for ".
                                 "$env{'user.name'} @ $env{'user.domain'}");
        &Apache::lonnet::logthis("lonmysql error = ".
                                 &Apache::lonmysql::get_error());
        return;
    }
    if (! $table_status) {
        &Apache::lonnet::logthis("lonmysql error = ".
                                 &Apache::lonmysql::get_error());
        &Apache::lonnet::logthis("lonmysql debug = ".
                                 &Apache::lonmysql::get_debug());
        &Apache::lonnet::logthis('table status = "'.$table_status.'"');
        $r->print("The table id,$table, we tried to use is invalid.".
                  "The search has been aborted.".
		  &Apache::loncommon::end_page());
        return;
    }
    ##
    ## Prepare for the big loop.
    my $hitcountsum;
    my %matches;
    my $server; 
    my $status;
    my $revise = &revise_button();
    $r->print(<<END);
<table>
<tr><th>Status</th><th>Total Matches</th><th>Time Remaining</th><th></th></tr>
<tr>
<td><input type="text" name="status"  value="" size="50" /></td>
<td><input type="text" name="count"   value="" size="10" /></td>
<td><input type="text" name="seconds" value="" size="8" /></td>
<td>$revise</td>
</tr>
</table>
</form>
END
    $r->rflush();
    &reset_timing();
    &update_seconds($r);
    &update_status($r,&mt('contacting [_1]',$Servers_to_contact[0]));
    while (&time_left() &&
           ((@Servers_to_contact) || keys(%Server_status))) {
        &update_seconds($r);
        #
        # Send out a search request
        if (@Servers_to_contact) {
            # Contact one server
            my $server = shift(@Servers_to_contact);
            &update_status($r,&mt('contacting [_1]',$server));
            my $reply=&Apache::lonnet::metadata_query($query,$customquery,
                                                      $customshow,[$server]);
            ($server) = keys(%$reply);
            $Server_status{$server} = $reply->{$server};
        } else {
            # wait a sec. to give time for files to be written
            # This sleep statement is here instead of outside the else 
            # block because we do not want to pause if we have servers
            # left to contact.  
            if (scalar (keys(%Server_status))) {
                &update_status($r,
                       &mt('waiting on [_1]',join(' ',keys(%Server_status))));
            }
            sleep(1); 
        }
        #
        # Loop through the servers we have contacted but do not
        # have results from yet, looking for results.
        foreach my $server (keys(%Server_status)) {
            last if ($connection->aborted());
            &update_seconds($r);
            my $status = $Server_status{$server};
            if ($status eq 'con_lost') {
                delete ($Server_status{$server});
                next;
            }
            $status=~s|/||g;
       	    my $datafile=$r->dir_config('lonDaemons').'/tmp/'.$status;
            if (-e $datafile && ! -e "$datafile.end") {
                &update_status($r,&mt('Receiving results from [_1]',$server));
                next;
            }
            last if ($connection->aborted());
            if (-e "$datafile.end") {
                &update_status($r,&mt('Reading results from [_1]',$server));
                if (-z "$datafile") {
                    delete($Server_status{$server});
                    next;
                }
                my $fh;
                if (!($fh=Apache::File->new($datafile))) { 
                    $r->print("Unable to open search results file for ".
                                  "server $server.  Omitting from search");
                    delete($Server_status{$server}); 
                   next;
                }
                # Read in the whole file.
                while (my $result = <$fh>) {
                    last if ($connection->aborted());
                    #
                    # Records are stored one per line
                    chomp($result);
                    next if (! $result);
                    #
                    # Parse the result.
                    my %Fields = &parse_raw_result($result,$server,$tabletype);
                    $Fields{'hostname'} = $server;
                    #
                    # Skip if external and we did not want that
                    next if ((! $env{'form.inclext'}) && ($Fields{'url'}=~/^\/ext\//));
                    # Skip based on copyright
                    next if (! &copyright_check(\%Fields));

                    if ($area eq 'portfolio') {
                        next if (defined($matches{$Fields{'url'}}));
                        # Skip if inaccessible
                        next if (!&Apache::lonnet::portfolio_access($Fields{'url'}));
                        $matches{$Fields{'url'}} = 1; 
                    }
                    #
                    # Store the result in the mysql database
                    my $result = &Apache::lonmysql::store_row($table,\%Fields);
                    if (! defined($result)) {
                        $r->print(&Apache::lonmysql::get_error());
                    }
                    #
                    $hitcountsum ++;
                    &update_seconds($r);
                    if ($hitcountsum % 50 == 0) {
                        &update_count_status($r,$hitcountsum);
                    }
                }
                $fh->close();
                # $server is only deleted if the results file has been 
                # found and (successfully) opened.  This may be a bad idea.
                delete($Server_status{$server});
            }
            last if ($connection->aborted());
            &update_count_status($r,$hitcountsum);
        }
        last if ($connection->aborted());
        &update_seconds($r);
    }
    &update_status($r,&mt('Search Complete [_1]',$server));
    &update_seconds($r);
    #
    &Apache::lonmysql::disconnect_from_db(); # This is unneccessary
    #
    # We have run out of time or run out of servers to talk to and
    # results to get, so let the client know the top frame needs to be
    # loaded from /adm/searchcat
    $r->print(&Apache::loncommon::end_page());
#    if ($env{'form.catalogmode'} ne 'import') {
        $r->print("<script>".
                      "window.location='/adm/searchcat?".
                      "phase=sort&".
                      "persistent_db_id=$env{'form.persistent_db_id'}';".
                  "</script>");
#    }
    return;
}

######################################################################
######################################################################

=pod

=item &prev_next_buttons()

Returns html for the previous and next buttons on the search results page.

=cut

######################################################################
######################################################################
sub prev_next_buttons {
    my ($current_min,$show,$total,$parms) = @_;
    return '' if ($show eq 'all'); # No links if you get them all at once.
    #
    # Create buttons
    my $buttons = '<input type="submit" name="prev" value="'.&mt('Prev').'" ';
    $buttons .= '/>';
    $buttons .= '&nbsp;'x3;
    $buttons .= '<input type="submit" name="reload" '.
        'value="'.&mt('Reload').'" />';
    $buttons .= '&nbsp;'x3;
    $buttons .= '<input type="submit" name="next" value="'.&mt('Next').'" ';
    $buttons .= '/>';
    return $buttons;
}

######################################################################
######################################################################

=pod

=item &display_results()

Prints the results out for selection and perusal.

=cut

######################################################################
######################################################################
sub display_results {
    my ($r,$importbutton,$closebutton,$diropendb,$area) = @_;
    my $connection = $r->connection;
    $r->print(&search_results_header($importbutton,$closebutton));
    ##
    ## Set viewing function
    ##
    my $viewfunction = $Views{$env{'form.viewselect'}};
    if (!defined($viewfunction)) {
        $r->print("Internal Error - Bad view selected.\n");
        $r->rflush();
        return;
    }
    ##
    ## $checkbox_num is a count of the number of checkboxes output on the 
    ## page this is used only during catalogmode=import.
    my $checkbox_num = 0;
    ##
    ## Get the catalog controls setup
    ##
    my $action = "/adm/searchcat?phase=results";
    ##
    ## Deal with import by opening the import db file.
    if ($env{'form.catalogmode'} eq 'import') {
        if (! tie(%groupsearch_db,'GDBM_File',$diropendb,
                  &GDBM_WRCREAT(),0640)) {
            $r->print('Unable to store import results.</form>'.
		      &Apache::loncommon::end_page());
            $r->rflush();
            return;
        } 
    }
    ##
    ## Prepare the table for querying
    my $table = $env{'form.table'};
    return if (! &ensure_db_and_table($r,$table));
    ##
    ## Get the number of results 
    my $total_results = &Apache::lonmysql::number_of_rows($table);
    if (! defined($total_results)) {
        $r->print("A MySQL error has occurred.</form>".
		  &Apache::loncommon::end_page());
        &Apache::lonnet::logthis("lonmysql was unable to determine the number".
                                 " of rows in table ".$table);
        &Apache::lonnet::logthis(&Apache::lonmysql::get_error());
        return;
    }
    ##
    ## Determine how many results we need to get
    $env{'form.start'} = 1  if (! exists($env{'form.start'}));
    $env{'form.show'}  = 20 if (! exists($env{'form.show'}));
    if (exists($env{'form.prev'})) {
        $env{'form.start'} -= $env{'form.show'};
    } elsif (exists($env{'form.next'})) {
        $env{'form.start'} += $env{'form.show'};
    }
    $env{'form.start'} = 1 if ($env{'form.start'}<1);
    $env{'form.start'} = $total_results if ($env{'form.start'}>$total_results);
    my $min = $env{'form.start'};
    my $max;
    if ($env{'form.show'} eq 'all') {
        $max = $total_results ;
    } else {
        $max = $min + $env{'form.show'} - 1;
        $max = $total_results if ($max > $total_results);
    }
    ##
    ## Output form elements
    $r->print(&hidden_field('table').
              &hidden_field('phase').
              &hidden_field('persistent_db_id').
              &hidden_field('start').
              &hidden_field('area')
              );
    #
    # Build sorting selector
    my @fields = 
        (
         {key=>'default' },
         {key=>'title' },
         {key =>'author' },
         {key =>'subject'},
         {key =>'url',desc=>'URL'},
         {key =>'keywords'},
         {key =>'language'},
         {key =>'creationdate'},
         {key =>'lastrevisiondate'},
         {key =>'owner'},
         {key =>'copyright'},
         {key =>'authorspace'},
         {key =>'lowestgradelevel'},
         {key =>'highestgradelevel'},
         {key =>'standards',desc=>'Standards'},
        );
    if ($area eq 'portfolio') {
        push(@fields,
       (
         {key => 'scope'},
         {key => 'keynum'},
       ));
    } else {
        push(@fields,
       (
         {key =>'count',desc=>'Number of accesses'},
         {key =>'stdno',desc=>'Students Attempting'},
         {key =>'avetries',desc=>'Average Number of Tries'},
         {key =>'difficulty',desc=>'Mean Degree of Difficulty'},
         {key =>'disc',desc=>'Mean Degree of Discrimination'},
         {key =>'clear',desc=>'Evaluation: Clear'},
         {key =>'technical',desc=>'Evaluation: Technically Correct'},
         {key =>'correct',desc=>'Evaluation: Material is Correct'},
         {key =>'helpful',desc=>'Evaluation: Material is Helpful'},
         {key =>'depth',desc=>'Evaluation: Material has Depth'},
       ));
    }
    my %fieldnames = &Apache::lonmeta::fieldnames();
    my @field_order;
    foreach my $field_data (@fields) {
        push(@field_order,$field_data->{'key'});
        if (! exists($field_data->{'desc'})) {
            $field_data->{'desc'}=$fieldnames{$field_data->{'key'}};
        } else {
            if (! defined($field_data->{'desc'})) {
                $field_data->{'desc'} = ucfirst($field_data->{'key'});
            }
            $field_data->{'desc'} = &mt($field_data->{'desc'});
        }
    }
    my %sort_fields = map {$_->{'key'},$_->{'desc'}} @fields;
    $sort_fields{'select_form_order'} = \@field_order;
    $env{'form.sortorder'} = 'desc' if (! exists($env{'form.sortorder'}));
    if (! exists($env{'form.sortfield'})) {
        if ($area eq 'portfolio') {
            $env{'form.sortfield'} = 'owner';
        } else {
            $env{'form.sortfield'} = 'count';
        }
    }
    if (! exists($env{'form.sortorder'})) {
	if ($env{'form.sortfield'}=~/^(count|stdno|disc|clear|technical|correct|helpful)$/) {
	    $env{'form.sortorder'}='desc';
	} else {
	    $env{'form.sortorder'}='asc';
	}
    }
    my $sortform = &mt('Sort by [_1] [_2]',
                       &Apache::loncommon::select_form($env{'form.sortfield'},
                                                      'sortfield',
                                                      %sort_fields),
                       &Apache::loncommon::select_form($env{'form.sortorder'},
                                                      'sortorder',
                                                      (asc =>&mt('Ascending'),
                                                       desc=>&mt('Descending')
                                                       ))
                       );
    ##
    ## Output links (if necessary) for 'prev' and 'next' pages.
    $r->print
        ('<table width="100%"><tr><td width="25%" align="right">'.
         '<nobr>'.$sortform.'</nobr>'.
         '</td><td width="25%" align="right">'.
         &prev_next_buttons($min,$env{'form.show'},$total_results).
         '</td><td align="right">'.
         &viewoptions().'</td></tr></table>'
         );
    if ($total_results == 0) {
        $r->print('<meta HTTP-EQUIV="Refresh" CONTENT="2" />'.
                  '<h3>'.&mt('There are currently no results').'.</h3>'.
                  "</form>".
		  &Apache::loncommon::end_page());
        return;
    } else {
        $r->print('<center>'.
                  mt('Results [_1] to [_2] out of [_3]',
                     $min,$max,$total_results).
                  "</center>\n");
    }
    ##
    ## Get results from MySQL table
    my $sort_command  = 'id>='.$min.' AND id<='.$max;
    my $order;
    if (exists($env{'form.sortorder'})) {
        if ($env{'form.sortorder'} eq 'asc') {
            $order = 'ASC';
        } elsif ($env{'form.sortorder'} eq 'desc') {
            $order = 'DESC';
        } else {
            $order = '';
        }
    } else {
        $order = '';
    }
    if ($env{'form.sortfield'} ne 'default' && 
        exists($sort_fields{$env{'form.sortfield'}})) {
        $sort_command = $env{'form.sortfield'}.' IS NOT NULL '.
            'ORDER BY '.$env{'form.sortfield'}.' '.$order.
            '  LIMIT '.($min-1).','.($max-$min+1);
    }
    my @Results = &Apache::lonmysql::get_rows($table,$sort_command);
    ##
    ## Loop through the results and output them.
    my $tabletype = 'metadata';
    if ($area eq 'portfolio') {
        $tabletype = 'portfolio_search';
    }
    foreach my $row (@Results) {
        if ($connection->aborted()) {
            &cleanup();
            return;
        }
        my %Fields = %{&parse_row($tabletype,@$row)};
        my $output="<p>\n";
        if (! defined($Fields{'title'}) || $Fields{'title'} eq '') {
            $Fields{'title'} = 'Untitled';
        }
        my $prefix=&catalogmode_output($Fields{'title'},$Fields{'url'},
                                       $Fields{'id'},$checkbox_num++);
        # Render the result into html
        $output.= &$viewfunction($prefix,%Fields);
        # Print them out as they come in.
        $r->print($output);
        $r->rflush();
    }
    if (@Results < 1) {
        $r->print(&mt("There were no results matching your query"));
    } else {
        $r->print
            ('<center>'.
             &prev_next_buttons($min,$env{'form.show'},$total_results,
                                "table=".$env{'form.table'}.
                                "&phase=results".
                                "&persistent_db_id=".
                                $env{'form.persistent_db_id'})
             ."</center>\n"
             );
    }
    $r->print("</form>".&Apache::loncommon::end_page());
    $r->rflush();
    untie %groupsearch_db if (tied(%groupsearch_db));
    return;
}

######################################################################
######################################################################

=pod

=item &catalogmode_output($title,$url,$fnum,$checkbox_num)

Returns html needed for the various catalog modes.  Gets inputs from
$env{'form.catalogmode'}.  Stores data in %groupsearch_db.

=cut

######################################################################
######################################################################
sub catalogmode_output {
    my $output = '';
    my ($title,$url,$fnum,$checkbox_num) = @_;
    if ($env{'form.catalogmode'} eq 'interactive') {
        $title=~ s/\'/\\\'/g;
        if ($env{'form.catalogmode'} eq 'interactive') {
            $output.=<<END 
<font size='-1'><INPUT TYPE="button" NAME="returnvalues" VALUE="SELECT"
onClick="javascript:select_data('$title','$url')">
</font>
END
        }
    } elsif ($env{'form.catalogmode'} eq 'import') {
        $groupsearch_db{"pre_${fnum}_link"}=$url;
        $groupsearch_db{"pre_${fnum}_title"}=$title;
        $output.=<<END;
<font size='-1'>
<input type="checkbox" name="returnvalues" value="SELECT"
onClick="javascript:queue($checkbox_num,$fnum)" />
</font>
END
    }
    return $output;
}
######################################################################
######################################################################

=pod

=item &parse_row()

Parse a row returned from the database.

=cut

######################################################################
######################################################################
sub parse_row {
    my ($tabletype,@Row) = @_;
    my %Fields;
    if (! scalar(@Datatypes)) {
        &set_up_table_structure($tabletype);
    }
    for (my $i=0;$i<=$#Row;$i++) {
        $Fields{$Datatypes[$i]->{'name'}}=&unescape($Row[$i]);
    }
    $Fields{'language'} = 
        &Apache::loncommon::languagedescription($Fields{'language'});
    $Fields{'copyrighttag'} =
        &Apache::loncommon::copyrightdescription($Fields{'copyright'});
    $Fields{'mimetag'} =
        &Apache::loncommon::filedescription($Fields{'mime'});
    return \%Fields;
}

###########################################################
###########################################################

=pod

=item &parse_raw_result()

Takes a line from the file of results and parse it.  Returns a hash 
with keys according to column labels

In addition, the following tags are set by calling the appropriate 
lonnet function: 'language', 'copyrighttag', 'mimetag'.

The 'title' field is set to "Untitled" if the title field is blank.

'abstract' and 'keywords' are truncated to 200 characters.

=cut

###########################################################
###########################################################
sub parse_raw_result {
    my ($result,$hostname,$tabletype) = @_;
    # conclude from self to others regarding fields
    my %Fields=&LONCAPA::lonmetadata::metadata_col_to_hash
        ($tabletype,
         map {
            &unescape($_);
         } (split(/\,/,$result)) );
    return %Fields;
}

###########################################################
###########################################################

=pod

=item &handle_custom_fields()

=cut

###########################################################
###########################################################
sub handle_custom_fields {
    my @results = @{shift()};
    my $customshow='';
    my $extrashow='';
    my @customfields;
    if ($env{'form.customshow'}) {
        $customshow=$env{'form.customshow'};
        $customshow=~s/[^\w\s]//g;
        my @fields=map {
            "<font color=\"#008000\">$_:</font><!-- $_ -->";
        } split(/\s+/,$customshow);
        @customfields=split(/\s+/,$customshow);
        if ($customshow) {
            $extrashow="<ul><li>".join("</li><li>",@fields)."</li></ul>\n";
        }
    }
    my $customdata='';
    my %customhash;
    foreach my $result (@results) {
        if ($result=~/^(custom\=.*)$/) { # grab all custom metadata
            my $tmp=$result;
            $tmp=~s/^custom\=//;
            my ($k,$v)=map {&unescape($_);
                        } split(/\,/,$tmp);
            $customhash{$k}=$v;
        }
    }
    return ($extrashow,\@customfields,\%customhash);
}

######################################################################
######################################################################

=pod

=item &search_results_header()

Output the proper html headers and javascript code to deal with different 
calling modes.

Takes most inputs directly from %env, except $mode.  

=over 4

=item $mode is either (at this writing) 'Basic' or 'Advanced'

=back

The following environment variables are checked:

=over 4

=item 'form.catalogmode' 

Checked for 'interactive' and 'import'.

=item 'form.mode'

Checked for existance & 'edit' mode.

=item 'form.form'

Contains the name of the form that has the input fields to set

=item 'form.element'

the name of the input field to put the URL into

=item 'form.titleelement'

the name of the input field to put the title into

=back

=cut

######################################################################
######################################################################
sub search_results_header {
    my ($importbutton,$closebutton) = @_;

    my $js;
    # output beginning of search page
    # conditional output of script functions dependent on the mode in
    # which the search was invoked
    if ($env{'form.catalogmode'} eq 'interactive'){
	if (! exists($env{'form.mode'}) || $env{'form.mode'} ne 'edit') {
            $js.=<<SCRIPT;
<script type="text/javascript">
    function select_data(title,url) {
	changeTitle(title);
	changeURL(url);
	parent.close();
    }
    function changeTitle(val) {
	if (parent.opener.inf.document.forms.resinfo.elements.t) {
	    parent.opener.inf.document.forms.resinfo.elements.t.value=val;
	}
    }
    function changeURL(val) {
	if (parent.opener.inf.document.forms.resinfo.elements.u) {
	    parent.opener.inf.document.forms.resinfo.elements.u.value=val;
	}
    }
</script>
SCRIPT
        } elsif ($env{'form.mode'} eq 'edit') {
            my $form = $env{'form.form'};
            my $element = $env{'form.element'};
            my $titleelement = $env{'form.titleelement'};
	    my $changetitle;
	    if (!$titleelement) {
		$changetitle='function changeTitle(val) {}';
	    } else {
		    $changetitle=<<END;
function changeTitle(val) {
    if (parent.targetwin.document) {
        parent.targetwin.document.forms["$form"].elements["$titleelement"].value=val;
    } else {
	var url = 'forms[\"$form\"].elements[\"$titleelement\"].value';
        alert("Unable to transfer data to "+url);
    }
}
END
            }

            $js.=<<SCRIPT;
<script type="text/javascript">
function select_data(title,url) {
    changeURL(url);
    changeTitle(title);
    parent.close();
}
$changetitle
function changeURL(val) {
    if (parent.targetwin.document) {
        parent.targetwin.document.forms["$form"].elements["$element"].value=val;
    } else {
	var url = 'forms[\"$form\"].elements[\"$element\"].value';
        alert("Unable to transfer data to "+url);
    }
}
</script>
SCRIPT
        }
    }
    $js.=<<SCRIPT if $env{'form.catalogmode'} eq 'import';
<script type="text/javascript">
    function queue(checkbox_num,val) {
        if (document.forms.results.returnvalues.length != "undefined" &&
            typeof(document.forms.results.returnvalues.length) == "number") {
            if (document.forms.results.returnvalues[checkbox_num].checked) {
                parent.statusframe.document.forms.statusform.elements.acts.value +='1a'+val+'b';
            } else {
                parent.statusframe.document.forms.statusform.elements.acts.value +='0a'+val+'b';
            }
        } else {
            if (document.forms.results.returnvalues.checked) {
                parent.statusframe.document.forms.statusform.elements.acts.value +='1a'+val+'b';
            } else {
                parent.statusframe.document.forms.statusform.elements.acts.value +='0a'+val+'b';
            }
        }
    }
    function select_group() {
	parent.window.location=
    "/adm/groupsort?mode=$env{'form.mode'}&catalogmode=import&acts="+
	    parent.statusframe.document.forms.statusform.elements.acts.value;
    }
</script>
SCRIPT

    my $start_page  = &Apache::loncommon::start_page(undef,$js,
						     {'only_body' =>1});
    my $result=<<END;
$start_page
<form name="results" method="post" action="/adm/searchcat">
$importbutton
END
    return $result;
}

sub results_link {
    my $basic_link   = "/adm/searchcat?"."&table=".$env{'form.table'}.
        "&persistent_db_id=".$env{'form.persistent_db_id'};
    my $results_link = $basic_link."&phase=results".
        "&pause=1"."&start=1";
    return $results_link;
}

######################################################################
######################################################################
sub print_frames_interface {
    my $r = shift;
    my $basic_link = "/adm/searchcat?"."&table=".$env{'form.table'}.
        "&persistent_db_id=".$env{'form.persistent_db_id'};
    my $run_search_link = $basic_link."&phase=run_search";
    my $results_link = &results_link();
    my $js = <<JS;
<script type="text/javascript">
var targetwin = opener;
var queue = '';
</script>
JS

    my $start_page =
        &Apache::loncommon::start_page('LON-CAPA Digital Library Search Results',
				       $js,
				       {'frameset'    => 1,
					'add_entries' => {
					    'rows' => "150,*",},});
    my $end_page =
        &Apache::loncommon::end_page({'frameset' => 1});

    my $result = <<"ENDFRAMES";
$start_page
    <frame name="statusframe"  src="$run_search_link">
    <frame name="resultsframe" src="$results_link">
$end_page
ENDFRAMES

    $r->print($result);
    return;
}

######################################################################
######################################################################

sub has_stat_data {
    my ($values) = @_;
    if ( (defined($values->{'count'})      && $values->{'count'}      ne '') ||
         (defined($values->{'stdno'})      && $values->{'stdno'}      ne '') ||
         (defined($values->{'disc'})       && $values->{'disc'}       ne '') ||
         (defined($values->{'avetries'})   && $values->{'avetries'}   ne '') ||
         (defined($values->{'difficulty'}) && $values->{'difficulty'} ne '')) {
        return 1;
    }
    return 0;
}

sub statfields {
    return ('count','stdno','disc','avetries','difficulty');
}

sub has_eval_data {
    my ($values) = @_;
    if ( (defined($values->{'clear'})     && $values->{'clear'}     ne '') ||
         (defined($values->{'technical'}) && $values->{'technical'} ne '') ||
         (defined($values->{'correct'})   && $values->{'correct'}   ne '') ||
         (defined($values->{'helpful'})   && $values->{'helpful'}   ne '') ||
         (defined($values->{'depth'})     && $values->{'depth'}     ne '')) {
        return 1;
    }
    return 0;
}

sub evalfields { 
    return ('clear','technical','correct','helpful','depth');
}

######################################################################
######################################################################

=pod 

=item Metadata Viewing Functions

Output is a HTML-ified string.

Input arguments are title, author, subject, url, keywords, version,
notes, short abstract, mime, language, creation date,
last revision date, owner, copyright, hostname, and
extra custom metadata to show.

=over 4

=item &detailed_citation_view() 

=cut

######################################################################
######################################################################
sub detailed_citation_view {
    my ($prefix,%values) = @_;
    my $result;
    my $jumpurl=$values{'url'};
    $jumpurl=~s|^/ext/|http://|;
    $result .= '<b>'.$prefix.
        '<img src="'.&Apache::loncommon::icon($values{'url'}).'" />'.'&nbsp;'.
        '<a href="'.$jumpurl.'" '.
        'target="preview">'.$values{'title'}."</a></b>\n";
    $result .= "<p>\n";
    $result .= '<b>'.$values{'author'}.'</b>,'.
        ' <i>'.$values{'owner'}.'</i><br />';
    foreach my $field 
        (
         { name=>'url',
           translate => '<b>URL:</b>&nbsp;[_1]',
           special => 'url link',},
         { name=>'subject',
           translate => '<b>Subject:</b>&nbsp;[_1]',},
         { name=>'keywords',
           translate => '<b>Keywords:</b>&nbsp;[_1]',},
         { name=>'notes',
           translate => '<b>Notes:</b>&nbsp;[_1]',},
         { name=>'mimetag',
           translate => '<b>MIME Type:</b>&nbsp;[_1]',},
         { name=>'standards',
           translate => '<b>Standards:</b>[_1]',},
         { name=>'copyrighttag',
           translate => '<b>Copyright/Distribution:</b>&nbsp;[_1]',},
         { name=>'count',
           format => "%d",
           translate => '<b>Access Count:</b>&nbsp;[_1]',},
         { name=>'stdno',
           format => "%d",
           translate => '<b>Number of Students:</b>&nbsp;[_1]',},
         { name=>'avetries',
           format => "%.2f",
           translate => '<b>Average Tries:</b>&nbsp;[_1]',},
         { name=>'disc',
           format => "%.2f",
           translate => '<b>Degree of Discrimination:</b>&nbsp;[_1]',},
         { name=>'difficulty',
           format => "%.2f",
           translate => '<b>Degree of Difficulty:</b>&nbsp;[_1]',},
         { name=>'clear',
           format => "%.2f",
           translate => '<b>Clear:</b>&nbsp;[_1]',},
         { name=>'depth',
           format => "%.2f",
           translate => '<b>Depth:</b>&nbsp;[_1]',},
         { name=>'helpful',
           format => "%.2f",
           translate => '<b>Helpful:</b>&nbsp;[_1]',},
         { name=>'correct',
           format => "%.2f",
           translate => '<b>Correct:</b>&nbsp;[_1]',},
         { name=>'technical',
           format => "%.2f",
           translate => '<b>Technical:</b>&nbsp;[_1]',},
         { name=>'comefrom_list',
           type => 'list',
           translate => 'Resources that lead up to this resource in maps',},
         { name=>'goto_list',
           type => 'list',
           translate => 'Resources that follow this resource in maps',},
         { name=>'sequsage_list',
           type => 'list',
           translate => 'Resources using or importing resource',},
         ) {
        next if (! exists($values{$field->{'name'}}) ||
                 $values{$field->{'name'}} eq '');
        if (exists($field->{'type'}) && $field->{'type'} eq 'list') {
            $result .= '<b>'.&mt($field->{'translate'}).'</b>';
            foreach my $item (split(',',$values{$field->{'name'}})){
                $result .= 
		  &Apache::lonhtmlcommon::crumbs(&Apache::lonnet::clutter($item),
		  'preview',
		  '',
		  (($env{'form.catalogmode'} eq 'import')?'parent.statusframe.document.forms.statusform':''),2,0,1);
            }
        } elsif (exists($field->{'format'}) && $field->{'format'} ne ''){
            $result.= &mt($field->{'translate'},
                          sprintf($field->{'format'},
                                  $values{$field->{'name'}}))."<br />\n";
        } else {
            if ($field->{'special'} eq 'url link') {
               if ($jumpurl=~/^http\:\/\//) {
                 $result.='<tt>'.$jumpurl.'</tt>';
               } else {
                 $result .=
		  &Apache::lonhtmlcommon::crumbs($jumpurl,
		  'preview',
		  '',
		  (($env{'form.catalogmode'} eq 'import')?'parent.statusframe.document.forms.statusform':''),3,0,1);
               }
            } else {
                $result.= &mt($field->{'translate'},
                              $values{$field->{'name'}});
            }
            $result .= "<br />\n";
        }
    }
    $result .= "</p>";
    if (exists($values{'extrashow'}) && $values{'extrashow'} ne '') {
        $result .= '<p>'.$values{'extrashow'}.'</p>';
    }
    if (exists($values{'shortabstract'}) && $values{'shortabstract'} ne '') {
        $result .= '<p>'.$values{'shortabstract'}.'</p>';
    }
    $result .= '<hr align="left" width="200" noshade />'."\n";
    return $result;
}

sub detailed_citation_preview {
    my ($prefix,%values)=@_;
    return '<table><tr><td>'.
           &detailed_citation_view($prefix,%values).
           '</td><td>'.
           &Apache::lonindexer::showpreview($values{'url'}).
           '</td></tr></table><hr />';
}


######################################################################
######################################################################

=pod 

=item &summary_view() 

=cut
######################################################################
######################################################################
sub summary_view {
    my ($prefix,%values) = @_;
    my $icon=&Apache::loncommon::icon($values{'url'});
    my $result=qq{$prefix<img src="$icon" />};
    if (exists($env{'form.sortfield'}) && 
        $env{'form.sortfield'} !~ /^(default|
                                     author|
                                     url|
                                     title|
                                     owner|
                                     lastrevisiondate|
                                     copyright)$/x) {
        my $tmp = $values{$env{'form.sortfield'}};
        if (! defined($tmp)) { $tmp = 'undefined'; }
        $result .= '&nbsp;'.$tmp.'&nbsp;';
    }
    my $jumpurl=$values{'url'};
    my $link;
    if ($jumpurl=~m|^/ext/|) {
	$jumpurl=~s|^/ext/|http://|;
	$link='<br /><tt>'.$jumpurl.'</tt>';
    } else {
	$link=&Apache::lonhtmlcommon::crumbs($jumpurl,
		  'preview',
		  '',
		  (($env{'form.catalogmode'} eq 'import')?'parent.statusframe.document.forms.statusform':''),2,0,1);
    }
    $result.=<<END;
<a href="$jumpurl" 
   target='preview'>$values{'title'}</a>$link
$values{'author'}, $values{'owner'} -- $values{'lastrevisiondate'}<br />
$values{'copyrighttag'}<br />
$values{'extrashow'}
</p>
<hr align='left' width='200' noshade />
END
    return $result;
}

sub summary_preview {
    my ($prefix,%values)=@_;
    return '<table><tr><td>'.
           &summary_view($prefix,%values).
           '</td><td>'.
           &Apache::lonindexer::showpreview($values{'url'}).
           '</td></tr></table><hr />';
}

######################################################################
######################################################################

=pod 

=item &compact_view() 

=cut

######################################################################
######################################################################
sub compact_view {
    my ($prefix,%values) = @_;
    my $jumpurl=$values{'url'};
    my $link;
    if ($jumpurl=~m|^/ext/|) {
	$jumpurl=~s|^/ext/|http://|;
	$link='<tt>'.$jumpurl.'</tt>';
    } else {
        $link=&Apache::lonhtmlcommon::crumbs($jumpurl,
		  'preview',
		  '',
		  (($env{'form.catalogmode'} eq 'import')?'parent.statusframe.document.forms.statusform':''),1,1,1).' ';
    }
    my $result = 
        $prefix.'<img src="'.&Apache::loncommon::icon($values{'url'}).'">';
    if (exists($env{'form.sortfield'}) && 
        $env{'form.sortfield'} !~ /^(default|author|url|title)$/) {
        my $tmp = $values{$env{'form.sortfield'}};
        if (! defined($tmp)) { $tmp = 'undefined'; }
        $result .= '&nbsp;'.$tmp.'&nbsp;';
    }
    $result.='&nbsp;<a href="'.$jumpurl.'" target="preview">'.
        $values{'title'}.'</a>'.('&nbsp;'x2).$link.
        '<b>'.$values{'author'}.'</b> ('.$values{'domain'}.')<br />';
    return $result;
}


######################################################################
######################################################################

=pod 

=item &fielded_format_view() 

=cut

######################################################################
######################################################################
sub fielded_format_view {
    my ($prefix,%values) = @_;
    my $icon=&Apache::loncommon::icon($values{'url'});
    my %Translated = &Apache::lonmeta::fieldnames();
    my $jumpurl=$values{'url'};
    $jumpurl=~s|^/ext/|http://|;

    my $result=<<END;
$prefix <img src="$icon" />
<dl>
<dt>URL:</dt>
    <dd><a href="$jumpurl" 
         target='preview'>$values{'url'}</a></dd>
END
    foreach my $field ('title','author','domain','subject','keywords','notes',
                       'mimetag','language','creationdate','lastrevisiondate',
                       'owner','copyrighttag','hostname','abstract') {
        $result .= (' 'x4).'<dt>'.$Translated{$field}.'</dt>'."\n".
            (' 'x8).'<dd>'.$values{$field}.'</dd>'."\n";
    }
    if (&has_stat_data(\%values)) {
        foreach my $field (&statfields()) {
            $result .= (' 'x4).'<dt>'.$Translated{$field}.'</dt>'."\n".
                (' 'x8).'<dd>'.$values{$field}.'</dd>'."\n";
        }
    }
    if (&has_eval_data(\%values)) {
        foreach my $field (&evalfields()) {
            $result .= (' 'x4).'<dt>'.$Translated{$field}.'</dt>'."\n".
                (' 'x8).'<dd>'.$values{$field}.'</dd>'."\n";
        }
    }
    $result .= "</dl>\n";
    $result .= $values{'extrashow'};
    $result .= '<hr align="left" width="200" noshade />'."\n";
    return $result;
}

######################################################################
######################################################################

=pod 

=item &xml_sgml_view() 

=back 

=cut

######################################################################
######################################################################
sub xml_sgml_view {
    my ($prefix,%values) = @_;
    my $xml = '<LonCapaResource>'."\n";
    # The usual suspects
    foreach my $field ('url','title','author','subject','keywords','notes','domain') {
        $xml .= qq{<$field>$values{$field}</$field>}."\n";
    }
    #
    $xml .= "<mimeInfo>\n";
    foreach my $field ('mime','mimetag') {
        $xml .= qq{<$field>$values{$field}</$field>}."\n";
    }
    $xml .= "</mimeInfo>\n";
    #
    $xml .= "<languageInfo>\n";
    foreach my $field ('language','languagetag') {
        $xml .= qq{<$field>$values{$field}</$field>}."\n";
    }
    $xml .= "</languageInfo>\n";
    #
    foreach my $field ('creationdate','lastrevisiondate','owner') {
        $xml .= qq{<$field>$values{$field}</$field>}."\n";
    }
    #
    $xml .= "<copyrightInfo>\n";
    foreach my $field ('copyright','copyrighttag') {
        $xml .= qq{<$field>$values{$field}</$field>}."\n";
    }
    $xml .= "</copyrightInfo>\n";
    $xml .= qq{<repositoryLocation>$values{'hostname'}</repositoryLocation>}.
        "\n";
    $xml .= qq{<shortabstract>$values{'shortabstract'}</shortabstract>}."\n";
    #
    if (&has_stat_data(\%values)){
        $xml .= "<problemstatistics>\n";
        foreach my $field (&statfields()) {
            $xml .= qq{<$field>$values{$field}</$field>}."\n";            
        }
        $xml .= "</problemstatistics>\n";
    }
    #
    if (&has_eval_data(\%values)) {
        $xml .= "<evaluation>\n";
        foreach my $field (&evalfields) {
            $xml .= qq{<$field>$values{$field}</$field>}."\n";            
        }
        $xml .= "</evaluation>\n";
    }    
    #
    $xml .= "</LonCapaResource>\n";
    $xml = &HTML::Entities::encode($xml,'<>&');
    my $result=<<END;
$prefix
<pre>
$xml
</pre>
$values{'extrashow'}
<hr align='left' width='200' noshade />
END
    return $result;
}

######################################################################
######################################################################

=pod 

=item &filled() see if field is filled.

=cut

######################################################################
######################################################################
sub filled {
    my ($field)=@_;
    if ($field=~/\S/ && $field ne 'any') {
        return 1;
    } else {
        return 0;
    }
}

######################################################################
######################################################################

=pod 

=item &output_unparsed_phrase_error()

=cut

######################################################################
######################################################################
sub output_unparsed_phrase_error {
    my ($r,$closebutton,$parms,$hidden_fields,$field)=@_;
    my $errorstring;
    if ($field eq 'basicexp') {
        $errorstring = &mt('Unable to understand the search phrase <i>[_1]</i>.  Please modify your search.',$env{'form.basicexp'});
    } else {
        $errorstring = &mt('Unable to understand the search phrase <b>[_1]</b>:<i>[_2]</i>.',$field,$env{'form.'.$field});
    }
    my $heading = &mt('Unparsed Field');
    my $revise  = &mt('Revise search request');
    # make query information persistent to allow for subsequent revision
    my $start_page = &Apache::loncommon::start_page('Search');
    my $end_page   = &Apache::loncommon::end_page();
    $r->print(<<ENDPAGE);
$start_page
<form method="post" action="/adm/searchcat">
$hidden_fields
$closebutton
<hr />
<h2>$heading</h2>
<p>
$errorstring
</p>
<p>
<a href="/adm/searchcat?$parms&persistent_db_id=$env{'form.persistent_db_id'}">$revise</a>
</p>
$end_page
ENDPAGE
}

######################################################################
######################################################################

=pod 

=item &output_blank_field_error()

Output a complete page that indicates the user has not filled in enough
information to do a search.

Inputs: $r (Apache request handle), $closebutton, $parms.

Returns: nothing

$parms is extra information to include in the 'Revise search request' link.

=cut

######################################################################
######################################################################
sub output_blank_field_error {
    my ($r,$closebutton,$parms,$hidden_fields)=@_;
    my $errormsg = &mt('You did not fill in enough information for the search to be started.  You need to fill in relevant fields on the search page in order for a query to be processed.');
    my $revise = &mt('Revise Search Request');
    my $heading = &mt('Unactionable Search Queary');
    my $start_page = &Apache::loncommon::start_page('Search');
    my $end_page   = &Apache::loncommon::end_page();
    $r->print(<<ENDPAGE);
$start_page
<form method="post" action="/adm/searchcat">
$hidden_fields
$closebutton
<hr />
<h2>$heading</h2>
<p>
$errormsg
</p>
<p>
<a href="/adm/searchcat?$parms&persistent_db_id=$env{'form.persistent_db_id'}">$revise</a>&nbsp;
</p>
$end_page
ENDPAGE
    return;
}

######################################################################
######################################################################

=pod 

=item &output_date_error()

Output a full html page with an error message.

Inputs: 

    $r, the request pointer.
    $message, the error message for the user.
    $closebutton, the specialized close button needed for groupsearch.

=cut

######################################################################
######################################################################
sub output_date_error {
    my ($r,$message,$closebutton,$hidden_fields)=@_;
    # make query information persistent to allow for subsequent revision
    my $start_page = &Apache::loncommon::start_page('Search');
    my $end_page   = &Apache::loncommon::end_page();
    $r->print(<<RESULTS);
$start_page
<form method="post" action="/adm/searchcat">
$hidden_fields
<input type='button' value='Revise search request'
onClick='this.form.submit();' />
$closebutton
<hr />
<h3>Error</h3>
<p>
$message
</p>
$end_page
RESULTS
}

######################################################################
######################################################################

=pod 

=item &start_fresh_session()

Cleans the global %groupsearch_db by removing all fields which begin with
'pre_' or 'store'.

=cut

######################################################################
######################################################################
sub start_fresh_session {
    delete $groupsearch_db{'mode_catalog'};
    foreach (keys %groupsearch_db) {
        if ($_ =~ /^pre_/) {
            delete $groupsearch_db{$_};
        }
        if ($_ =~ /^store/) {
	    delete $groupsearch_db{$_};
	}
    }
}

1;

sub cleanup {
    if (tied(%groupsearch_db)) {
        unless (untie(%groupsearch_db)) {
	  &Apache::lonnet::logthis('Failed cleanup searchcat: groupsearch_db');
        }
    }
    &untiehash();
    &Apache::lonmysql::disconnect_from_db();
    return OK;
}

__END__

=pod

=back 

=cut
