#!/usr/bin/perl -w
#
# The LearningOnline Network
#
# dump_db.pl - dump a GDBM database to standard output, unescaping if asked to.
#
# $Id: dump_db.pl,v 1.7 2006/08/08 18:20:50 albertel Exp $
#
# Copyright Michigan State University Board of Trustees
#
# This file is part of the LearningOnline Network with CAPA (LON-CAPA).
#
# LON-CAPA is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# LON-CAPA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with LON-CAPA; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
# /home/httpd/html/adm/gpl.txt
#
# http://www.lon-capa.org/
#
#################################################
use strict;
use Getopt::Long;
use GDBM_File;
use Data::Dumper;
use Storable qw(thaw);
use lib '/home/httpd/lib/perl/';
use LONCAPA;
use LONCAPA::Configuration;
use Cwd;

#
# Options
my ($unesc,$help,$localize_times) = (0,0,0);
GetOptions("unescape" => \$unesc,
           "u"        => \$unesc,
           "t"        => \$localize_times,
           "help"     => \$help);

#
# Help them out if they ask for it
if ($help) {
    print <<END;
dump_db.pl - dump GDBM_File databases to stdout.  
Specify the database filenames on the command line.
Specify --unescape to have all the keys and values unescaped from every
database.
Options:
   --help     Display this help.
   --unescape Unescape the keys and values before printing them out.
   -u        Same as --unescape
   -t        Localize times when possible (human readable times)
Examples: 
    dump_db.pl mydata.db
    dump_db.pl mydata.db yourdata.db ourdata.db theirdata.db
    dump_db.pl --unescape \*db
END
    exit;
}

my  %perlvar=%{&LONCAPA::Configuration::read_conf('loncapa.conf')};

#
# Loop through ARGV getting files.
while (my $fname = shift) {
    $fname = &Cwd::abs_path($fname);
    my $dbref;
    if ($fname =~ m/^\Q$perlvar{'lonUsersDir'}\E/) {
	$dbref=&LONCAPA::locking_hash_tie($fname,&GDBM_READER());
    } else {
	if (tie(my %db,'GDBM_File',$fname,&GDBM_READER(),0640)) {
	    $dbref = \%db;
	}
    }

    if (!$dbref) {
        warn "Unable to tie to $fname";
        next;
    }
    while (my ($key,$value) = each(%$dbref)) {
        if ($value =~ s/^__FROZEN__//) {
            #$value = thaw(&unescape($value));
        }
        if ($unesc) {
            $key = &unescape($key);
            $value = &unescape($value) if (! ref($value));
        }
        if ($localize_times && ! ref($value)) {
            $value =~ s/([0-9]{10,10})/localtime($1)/ge;
        }
        print "$key = ".(ref($value)?Dumper($value):$value)."\n";
    }
    &LONCAPA::locking_hash_untie($dbref);
}
exit;

